! Code Aspects, version 2.0.
! Michel Fioc, 2014-4-15. 

module mod_simul_catalogs

  use mod_constants, only : real_type, pi, two_pi, half_pi, force_ctp_frac
  implicit none
  logical, save :: initialize_Gaussian_alea ! Used only in `simul_catalogs` and
!& `Gaussian_alea`.

!______________________________________________________________________
! Variables used in `example_simul`.

! Cardinal number of simulations with the parameters below; ordinal number
! of the current simulation:
  integer :: n_simul, i_simul

! Type of simulation:
  logical :: one_to_one ! `.true.` <=> one-to-one simulation;
!& `.false.` <=> several-to-one simulation.

! Input value of $f$ for the simulation:
  real(kind=real_type) :: input_f_u

! Common value of `semi_axis_a_u`, etc. :
  real(kind=real_type) :: semi_axis_a_u, semi_axis_b_u, &
       semi_axis_a_p, semi_axis_b_p

! Number of associations rejected because...
  integer :: n_unavailable ! ... there is no available counterpart anymore.
  integer :: n_side_effects ! ... of side effects.

! Effective values of $f$ and $f'$ resulting from the simulation:
  real(kind=real_type) :: eff_f_u, eff_f_p

! Measure of the deviation with respect to a one-to-one simulation:
  real(kind=real_type) :: oto_deviation

  logical :: output_catalogs ! `.true.` if catalogs are output for the first of 
!& a sequence of simulations. `.false.` otherwise.

  integer, dimension(:), pointer :: seed => null() ! Seeds used to generate 
!& random numbers.

!______________________________________________________________________

  private
  public :: & ! Procedures: {
       simul_catalogs, write_catalogs, seed_format, & ! }
       ! Data: {
       one_to_one, input_f_u, semi_axis_a_u, semi_axis_b_u, &
       semi_axis_a_p, semi_axis_b_p, output_catalogs, &
       n_unavailable, n_side_effects, eff_f_u, eff_f_p, oto_deviation, &
       seed, n_simul, i_simul ! }

contains
!======================================================================

  subroutine simul_catalogs(n_u, n_p, input_f_u, semi_axis_a_u, &
       semi_axis_b_u, semi_axis_a_p, semi_axis_b_p, one_to_one, &
       coord_u, coord_p, n_unavailable, n_side_effects, &
       eff_f_u, eff_f_p, oto_deviation, seed, area_S)

    use mod_heap_index, only : heap_index
    use mod_types, only : coord_struct
    use mod_variables, only : area_S_tmp => area_S
    implicit none
    integer, intent(in) :: n_u, n_p
    real(kind=real_type), intent(in) :: input_f_u, &
         semi_axis_a_u, semi_axis_b_u, &
         semi_axis_a_p, semi_axis_b_p
    logical, intent(in) :: one_to_one
    type(coord_struct), dimension(:), pointer :: coord_u, coord_p
    integer, intent(out) :: n_unavailable, n_side_effects
    real(kind=real_type), intent(out) :: eff_f_u, eff_f_p, oto_deviation
    integer, dimension(:), pointer :: seed
    real(kind=real_type), intent(inout), optional :: area_S ! If absent,
!& the default value in "mod_variables.f90" (4 `pi`, i.e. all-sky) will be used,
!& thanks to the trick with `area_S_tmp`. 
!& If present and not in ]0, 4 `pi`], `area_S` is set to 4 `pi`.
!& If present and in ]0, 4 `pi`], a spherical cap covering a solid angle 
!& `area_S` and centered on the North pole is used.
!......................................................................

! True right ascension and declination of $K'$-sources:
    real(kind=real_type), dimension(n_p) :: alpha_0_p, delta_0_p

! True position angles of the positional uncertainty ellipses of $K$- and
! $K'$-sources:
    real(kind=real_type) :: beta_0_u, beta_0_p
! (Counted eastward, between the direction of the North at the true position
! and the major axis. Drawn randomly and uniformly in [0, `pi`[.)

! Southernest declination of the spherical cap.
    real(kind=real_type) :: delta_min

    integer, dimension(n_u) :: index_alpha_u ! Used to order $K$-sources by
!& increasing right ascension.

    integer, dimension(n_p) :: index_alpha_p ! The same for $K'$-sources.

    integer, dimension(n_p) :: n_ctp_p ! Number of true $K$-counterparts
!& to a $K'$-source.

    integer :: i, j ! Indices used for $K$- and $K'$-sources.
    integer :: size ! Size of the array of seeds for the generator of random 
!& numbers.

    logical :: association ! Set to `.true.` to search for a $K'$-counterpart 
!& to a $K$-source (may fail on the side of the spherical cap).
!______________________________________________________________________

    if (present(area_S)) then
       if (area_S <= 0 .or. area_S > 4*pi) then
          area_S = 4*pi
          write(*,"(a/)") "Warning: `area_S` set to 4 `pi`."
       endif
       area_S_tmp = area_S
    endif

    delta_min = arc_sin(1-area_S_tmp/two_pi)

! To have reproducible simulations if needed.
    initialize_Gaussian_alea = .true. ! Otherwise, two simulations with the
!& same input parameters -- including the seed for the generation of random
!& numbers -- will give different results if the second random number generated 
!& in `Gaussian_alea` has not been used.

    call random_seed(size = size)
    if (associated(seed)) deallocate(seed)
    allocate(seed(size))
    call random_seed(get = seed)

! Allocations: {
    if (associated(coord_u)) then
       deallocate(coord_u)
    endif
    allocate(coord_u(n_u))
    coord_u(:)%semi_axis_a = semi_axis_a_u
    coord_u(:)%semi_axis_b = semi_axis_b_u

    if (associated(coord_p)) then
       deallocate(coord_p)
    endif
    allocate(coord_p(n_p))
    coord_p(:)%semi_axis_a = semi_axis_a_p
    coord_p(:)%semi_axis_b = semi_axis_b_p
! }

! Generate catalog $K'$ randomly and uniformly on the surface of area `area_S`:
    do j = 1, n_p
! Observed positions:
       coord_p(j)%delta = arc_sin(1-uniform_alea()*area_S_tmp/two_pi)
       coord_p(j)%alpha = two_pi*uniform_alea()
       coord_p(j)%beta = pi*uniform_alea()
! True positions:
       call position(coord_p(j)%alpha, coord_p(j)%delta, &
            coord_p(j)%semi_axis_a, coord_p(j)%semi_axis_b, coord_p(j)%beta, &
            alpha_0_p(j), delta_0_p(j), beta_0_p)
    enddo

! Generate catalog $K$ from $K'$:
    j = 0
    n_ctp_p = 0
    eff_f_u = 0
    n_unavailable = 0
    n_side_effects = 0
    do i = 1, n_u
       if (force_ctp_frac) then
          association = (i <= nint(input_f_u*n_u)) ! Strict enforcement.
       else
          association = (uniform_alea() <= input_f_u) ! Lax enforcement.
       endif
       if (one_to_one .and. j+1 > n_p) then
          association = .false. ! No $K'$-source available for association.
          n_unavailable = n_unavailable+1
       endif

       if (association) then
          if (one_to_one) then
             j = j+1 
          else
             j = ceiling(uniform_alea()*n_p)
          endif
          beta_0_u = pi*uniform_alea()
          call position(alpha_0_p(j), delta_0_p(j), &
               coord_u(i)%semi_axis_a, coord_u(i)%semi_axis_b, beta_0_u, &
               coord_u(i)%alpha, coord_u(i)%delta, coord_u(i)%beta)
          if (coord_u(i)%delta < delta_min) then ! $K$-source outside of the sky
!& area: rejected. Create a randomly distributed unassociated source instead. 
             association = .false.
             n_side_effects = n_side_effects+1
          else ! $K$-source within the sky area: accepted.
             eff_f_u = eff_f_u+1
             n_ctp_p(j) = n_ctp_p(j)+1
          endif
       endif
       if (.not.association) then ! $K$-source without $K'$-counterpart. 
!& No need to use subroutine `position` then.
          coord_u(i)%delta =  arc_sin(1-uniform_alea()*area_S_tmp/two_pi)
          coord_u(i)%alpha = two_pi*uniform_alea()
          coord_u(i)%beta = pi*uniform_alea()
       endif
    enddo

    eff_f_u = eff_f_u/n_u
    eff_f_p = real(count(n_ctp_p(1:n_p) >= 1))/n_p
    
    if (eff_f_p > 0) then
       oto_deviation = (n_u*eff_f_u)/(n_p*eff_f_p)
    else
       oto_deviation = 1
    endif

! Order sources by increasing right ascension.

    call heap_index(coord_u(:)%alpha, index_alpha_u)
    coord_u(:) = coord_u(index_alpha_u(:))

    call heap_index(coord_p(:)%alpha, index_alpha_p)
    coord_p(:) = coord_p(index_alpha_p(:))

  end subroutine simul_catalogs

!======================================================================

  subroutine position(alpha_0, delta_0, semi_axis_a, semi_axis_b, beta_0, &
       alpha, delta, beta)

! Compute the observed position and position angle from the true ones (or the 
! converse). See Appendix B in file "Read_me.pdf" for explanations.
    implicit none
    real(kind=real_type), intent(in) :: alpha_0, delta_0, &
         beta_0, semi_axis_a, semi_axis_b
    real(kind=real_type), intent(out) :: alpha, delta, beta
!......................................................................
    real(kind=real_type) :: diff_major, diff_minor, epsilon, psi
    real(kind=real_type) :: cos_gamma, sin_gamma, gamma, gamma_0
    real(kind=real_type) :: cos_delta_cos_diff_alpha, cos_delta_sin_diff_alpha
!______________________________________________________________________

    diff_major = semi_axis_a*Gaussian_alea()
    diff_minor = semi_axis_b*Gaussian_alea()
    psi = sqrt(diff_major**2 + diff_minor**2)

    if (psi == 0) then
       alpha = alpha_0
       delta = delta_0
       beta = beta_0
       return
    endif

    epsilon = atan2(diff_minor, diff_major)
! (Valid only if `semi_axis_a` << `pi`. Otherwise, the usual [planar]
! normal law is not appropriate on the sphere. There exist generalizations of
! the normal law to the sphere, such as the Kent (1982) distribution, but
! they are not stable (i.e., the sum of two independent variables drawn
! from Kent distributions does not follow a Kent distribution).)

    gamma_0 = half_pi - beta_0 + epsilon
    delta = arc_sin(sin(gamma_0)*sin(psi)*cos(delta_0) + &
         sin(delta_0)*cos(psi))

    if (delta_0 == 0) then
       alpha = modulo(alpha_0+atan2(sin(psi)*cos(gamma_0), cos(psi)), two_pi)
    else
       cos_delta_cos_diff_alpha = (cos(delta_0)*sin(delta) &
            - sin(gamma_0)*sin(psi)) / sin(delta_0)
       cos_delta_sin_diff_alpha = cos(gamma_0)*sin(psi)
       alpha = modulo(alpha_0+atan2(cos_delta_sin_diff_alpha, &
            cos_delta_cos_diff_alpha), two_pi)
    endif

    cos_gamma = cos(delta_0)*sin(alpha-alpha_0)/sin(psi)
    sin_gamma = (cos(delta_0)*sin(delta)*cos(alpha-alpha_0) - &
         sin(delta_0)*cos(delta))/sin(psi)
    gamma = atan2(sin_gamma, cos_gamma)
    beta = modulo(beta_0 + gamma_0 - gamma, pi)

  end subroutine position

!======================================================================

  function arc_sin(x)

! Robust version of `asin` in case $|x| > 1$ for numerical reasons.
    implicit none
    real(kind=real_type) :: arc_sin
    real(kind=real_type), intent(in) :: x
!______________________________________________________________________

    if (x < -1) then
       arc_sin = -half_pi
    else if (x > 1) then
       arc_sin = half_pi
    else
       arc_sin = asin(x)
    endif

  end function arc_sin

!======================================================================

  function uniform_alea() ! Random number uniformly distributed in [0, 1[.

    implicit none
    real(kind=real_type) :: uniform_alea
!______________________________________________________________________

    call random_number(uniform_alea)

  end function uniform_alea

!======================================================================

  function Gaussian_alea()

! Draws a random number following a Gaussian distribution of mean 0 and 
! variance 1.

    implicit none
    real(kind=real_type) :: Gaussian_alea
!......................................................................
    real(kind=real_type), dimension(2) :: alea
    real(kind=real_type) :: x
    real(kind=real_type), save :: next
    logical, save :: exist_next = .false.
!______________________________________________________________________

    if (initialize_Gaussian_alea) then
       exist_next = .false.
       initialize_Gaussian_alea = .false.
    endif

    if (exist_next) then
       Gaussian_alea = next
       exist_next = .false.
    else
       do
          call random_number(alea(:))
          alea(:) = 2*alea(:) - 1
          x = sum(alea(:)**2)
          if (x > 0 .and. x < 1) exit
       enddo
       x = sqrt(-2*log(x)/x)
       Gaussian_alea = alea(1)*x
       next = alea(2)*x
       exist_next = .true.
    endif

  end function Gaussian_alea

!======================================================================

  function seed_format(seed) ! Output format for the array of seeds used
!& by the generator of random numbers.

    integer, dimension(:), intent(in) :: seed
    character(len=13+8*(size(seed)-1)) :: seed_format
!______________________________________________________________________

    seed_format = "('(/'i0"//repeat(",', ',i0", size(seed)-1)//"'/)')"

  end function seed_format

!======================================================================

  subroutine write_catalogs(K_u_file, K_p_file, n_u, n_p, &
       semi_axis_a_u, semi_axis_b_u, semi_axis_a_p, semi_axis_b_p, &
       one_to_one, area_S, seed, coord_u, coord_p)

! Write simulated catalogs to the files `K_u_file` and `K_p_file`.
! (Renamed if already existing.)
    use mod_files
    use mod_types, only : coord_struct
    implicit none
    character(len=*), intent(inout) :: K_u_file, K_p_file
    integer, intent(in) :: n_u, n_p
    real(kind=real_type), intent(in) :: semi_axis_a_u, semi_axis_b_u, &
         semi_axis_a_p, semi_axis_b_p
    logical, intent(in) :: one_to_one
    real(kind=real_type), intent(in) :: area_S
    integer, dimension(:), intent(in) :: seed
    type(coord_struct), dimension(:), intent(in) :: coord_u, coord_p
!......................................................................
    logical :: exist_u, exist_p
    character(len=10) :: date, time, zone
    character(len=132) :: suffix
    integer :: i, unit
!______________________________________________________________________

    inquire(file=K_u_file, exist=exist_u)
    inquire(file=K_p_file, exist=exist_p)
    if (exist_u) write(*,"(3a/)") "Catalog $K$: file """, trim(K_u_file), &
         """ already exists."
    if (exist_p) write(*,"(3a/)") "Catalog $K'$: file """, trim(K_p_file), &
         """ already exists."
    if (exist_u .or. exist_p) then
       call date_and_time(date=date, time=time, zone=zone)
       suffix = "."//date(1:4)//"~"//date(5:6)//"~"//date(7:8)//"_" &
            //time(1:2)//":"//time(3:4)//":"//time(5:10)//"_"//zone(1:3)//":" &
            //zone(4:5)
       write(*,"(5a)",advance="no") "Files """, trim(K_u_file), """ and """, &
            trim(K_p_file), """ renamed """
       K_u_file = trim(K_u_file)//suffix
       K_p_file = trim(K_p_file)//suffix
       write(*,"(4a/)") trim(K_u_file), """ and """, trim(K_p_file), &
            """, respectively."
    endif

! Output $K$.
    unit = new_unit()
    open(unit=unit, file=K_u_file, status="new")
    write(unit,"(a)") "! Code Aspects, version 2.0."
    write(unit,"(a/)") "! Michel Fioc, 2014-4-15."
    write(unit,"(a)") "! Catalog $K$."
    write(unit,"(3a/)") "! Twin catalog: $K'$ = """, trim(K_p_file),"""."
    write(unit,"(a)") "! Inputs of the simulation: "
    write(unit,"(a,i0)") "!   `n_u` = ", n_u
    write(unit,"(a,i0)") "!   `n_p` = ", n_p
    write(unit,"(a,es12.5)") "!   `semi_axis_a_u` = ", semi_axis_a_u
    write(unit,"(a,es12.5)") "!   `semi_axis_b_u` = ", semi_axis_b_u
    write(unit,"(a,es12.5)") "!   `semi_axis_a_p` = ", semi_axis_a_p
    write(unit,"(a,es12.5)") "!   `semi_axis_b_p` = ", semi_axis_b_p
    write(unit,"(a,l1)") "!   `one_to_one` = ", one_to_one
    write(unit,"(a,es12.5)") "!   `area_S` = ", area_S
    write(unit,"(a)", advance="no") "!   `seed` = "
    write(unit,seed_format(seed)) seed
    write(unit,"(/a)") "! Contents: "
    write(unit,"(a)") "! * First line: "
    write(unit,"(a)") "!   -- number of sources $n$;"
    write(unit,"(a/)") "!   -- surface area $S$;"
    write(unit,"(a)") "! * Following lines: sources ordered by increasing &
         &right ascension."
    write(unit,"(a)") "!   For each of them,"
    write(unit,"(a)") "!   -- right ascension $\alpha$;"
    write(unit,"(a)") "!   -- declination $\delta$;"
    write(unit,"(a)") "!   -- semi-major axis $a$ of the positional &
         &uncertainty ellipse;"
    write(unit,"(a)") "!   -- semi-minor axis $b$;"
    write(unit,"(a)") "!   -- position angle $\beta$ of the positional &
         &uncertainty ellipse."
    write(unit,"(a/)") "!   (All the angles are in radians.)"
    write(unit,*) n_u, area_S
    do i = 1, n_u
       write(unit,"(2(f10.7,tr1),2(es12.5,tr1),f10.7)") coord_u(i)%alpha, &
            coord_u(i)%delta, coord_u(i)%semi_axis_a, coord_u(i)%semi_axis_b, &
            coord_u(i)%beta
    enddo

! Output $K'$.
    unit = new_unit()
    open(unit=unit, file=K_p_file, status="new")
    write(unit,"(a)") "! Code Aspects, version 2.0."
    write(unit,"(a/)") "! Michel Fioc, 2014-4-15."
    write(unit,"(a)") "! Catalog $K'$."
    write(unit,"(3a/)") "! Twin catalog: $K$ = """, trim(K_u_file),"""."
    write(unit,"(a)") "! Inputs of the simulation: "
    write(unit,"(a,i0)") "!   `n_u` = ", n_u
    write(unit,"(a,i0)") "!   `n_p` = ", n_p
    write(unit,"(a,es12.5)") "!   `semi_axis_a_u` = ", semi_axis_a_u
    write(unit,"(a,es12.5)") "!   `semi_axis_b_u` = ", semi_axis_b_u
    write(unit,"(a,es12.5)") "!   `semi_axis_a_p` = ", semi_axis_a_p
    write(unit,"(a,es12.5)") "!   `semi_axis_b_p` = ", semi_axis_b_p
    write(unit,"(a,l1)") "!   `one_to_one` = ", one_to_one
    write(unit,"(a,es12.5)") "!   `area_S` = ", area_S
    write(unit,"(a)", advance="no") "!   `seed` = "
    write(unit,seed_format(seed)) seed
    write(unit,"(/a)") "! Contents: "
    write(unit,"(a)") "! * First line: "
    write(unit,"(a)") "!   -- number of sources $n'$;"
    write(unit,"(a/)") "!   -- surface area $S$;"
    write(unit,"(a)") "! * Following lines: sources ordered by increasing &
         &right ascension."
    write(unit,"(a)") "!   For each of them,"
    write(unit,"(a)") "!   -- right ascension $\alpha'$;"
    write(unit,"(a)") "!   -- declination $\delta'$;"
    write(unit,"(a)") "!   -- semi-major axis $a'$ of the positional &
         &uncertainty ellipse;"
    write(unit,"(a)") "!   -- semi-minor axis $b'$ ;"
    write(unit,"(a)") "!   -- position angle $\beta'$ of the positional &
         &uncertainty ellipse."
    write(unit,"(a/)") "!   (All the angles are in radians.)"
    write(unit,*) n_p, area_S
    do i = 1, n_p
       write(unit,"(2(f10.7,tr1),2(es12.5,tr1),f10.7)") coord_p(i)%alpha, &
            coord_p(i)%delta, coord_p(i)%semi_axis_a, coord_p(i)%semi_axis_b, &
            coord_p(i)%beta
    enddo

  end subroutine write_catalogs

end module mod_simul_catalogs
