! Code Aspects, version 2.0.
! Michel Fioc, 2014-4-15. 

program example_simul

! Simulate catalogs and analyze them.
! Run either with
!         "example_simul < input_example_simul.dat"
! or with
!         "example_simul"
! and answer the questions.

  use mod_Aspects! , only : general_preliminaries, oto_preliminaries, &
!& sto_analysis, ots_analysis, oto_analysis, &
!& func_ln_sto_Lh, func_ln_ots_Lh, compute_ln_oto_Lh
  use mod_simul_catalogs! , only : simul_catalogs, write_catalogs, &
!& one_to_one, input_f_u, semi_axis_a_u, semi_axis_b_u, &
!& semi_axis_a_p, semi_axis_b_p, output_catalogs, &
!& n_unavailable, n_side_effects, eff_f_u, eff_f_p, oto_deviation, &
!& seed, seed_format, &
!& n_simul, i_simul
  use mod_variables! , only : area_S, K_u_file, K_p_file, n_u, n_p, &
!& coord_u, coord_p, assoc_data_u, assoc_data_p, &
!& sto_f_u, ots_f_p, sto_f_p, ots_f_u, oto_f_u, oto_f_p, &
!& std_dev_sto_f_u, std_dev_ots_f_p, std_dev_oto_f_u, &
!& max_ln_sto_Lh, max_ln_ots_Lh, max_ln_oto_Lh
  
  implicit none

!######################################################################
! Read the parameters of the simulation: {

  write(*,"(/a)", advance="no") "Number of simulations: "
  read(*,*) n_simul

  write(*,"(/a)", advance="no") "Number of sources in catalog $K$: "
  read(*,*) n_u

  write(*,"(/a)", advance="no") "Number of sources in catalog $K'$: "
  read(*,*) n_p

  write(*,"(/a)", advance="no") "Input fraction of $K$-sources with a &
       &counterpart: "
  read(*,*) input_f_u

  write(*,"(/a)", advance="no") "Semi-major axis (in radians) common &
       &to all $K$-sources: "
  read(*,*) semi_axis_a_u
  
  write(*,"(/a)", advance="no") "Semi-minor axis (in radians) common &
       &to all $K$-sources: "
  read(*,*) semi_axis_b_u

  write(*,"(/a)", advance="no") "Semi-major axis (in radians) common &
       &to all $K'$-sources: "
  read(*,*) semi_axis_a_p

  write(*,"(/a)", advance="no") "Semi-minor axis (in radians) common &
       &to all $K'$-sources: "
  read(*,*) semi_axis_b_p

  write(*,"(/a)", advance="no") "Type of association (`T` for one-to-one, &
       &`F` for several-to-one): "
  read(*,*) one_to_one

  write(*,"(/a)", advance="no") "Area (in steradians) covered by the catalogs &
       &(set to 4 `pi` if not in ]0, 4 `pi`]): "
  read(*,*) area_S

  write(*,"(/a)", advance="no") "Write simulated catalogs for the first &
       &simulation &(`T` for true, `F` for false)? "
  read(*,*) output_catalogs

  if (output_catalogs) then
     write(*,"(/a)", advance="no") "Output file containing the simulated &
          &coordinates and positional uncertainties of $K$-sources for the &
          &first simulation:"
     read(*,*) K_u_file
     
     write(*,"(/a)", advance="no") "Output file containing the simulated &
          &coordinates and positional uncertainties of $K'$-sources for the &
          &first simulation:"
     read(*,*) K_p_file
  endif
  write(*,"(//)")
! }

  do i_simul = 1, n_simul

!######################################################################
! Run simulation `i_simul`.

     write(*,"(a/)") repeat("#", 70)
     write(*,"(a,i0,a,i0/)") "Simulation ", i_simul, " / ", n_simul

!++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! Return observed {coordinates and positional uncertainty parameters}
! and the effective values of $f$ and $f'$.

     call simul_catalogs(n_u = n_u, n_p = n_p, input_f_u = input_f_u, &
          semi_axis_a_u = semi_axis_a_u, semi_axis_b_u = semi_axis_b_u, &
          semi_axis_a_p = semi_axis_a_p, semi_axis_b_p = semi_axis_b_p, &
          one_to_one = one_to_one, &
          coord_u = coord_u, coord_p = coord_p, &
          n_unavailable = n_unavailable, n_side_effects = n_side_effects, &
          eff_f_u = eff_f_u, eff_f_p = eff_f_p, oto_deviation = oto_deviation, &
          seed = seed, &
          area_S = area_S & ! Optional. Set to 4 `pi` 
!& if absent (through "mod_variables.f90") or not in ]0, 4 `pi`].
          )

! Note: In the simulation, the position angles `beta_0_u` and `beta_0_p`
! of the positional uncertainty ellipses are drawn randomly and uniformly in
! [0, `pi`[. Note that these angles are counted eastward from the direction
! of the North *at the true position* of the source. Angles `beta_u`
! (= $\beta$) and `beta_p` (= $\beta'$) relative to the direction of the
! North *at the observed position* of the source are then derived.

!++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! Array of seeds used to generate random numbers in simulation `i_simul`:
     write(*,"(a)", advance="no") "`seed` = "
     write(*,seed_format(seed)) seed

! (In case of failure, it may be useful to run exactly the same simulation to 
! detect where the problem comes from. To run the same simulation exactly, type
!    `call random_seed(put = seed)`
! just before `call simul_catalogs([...])`. Do not forget the leading `(/`, the
! ending `/)` and the commas separating the elements of `seed`.)

!++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! Nnumbers of associations rejected, effective fractions of $K$- and 
! $K'$-sources with a counterpart, and deviation with respect to a one-to-one
! simulation:
     write(*,"(a,i0/)") "`n_unavailable` = ", n_unavailable
     write(*,"(a,i0/)") "`n_side_effects` = ", n_side_effects
     write(*,"(/a,f8.5/)") "`eff_f_u` = ", eff_f_u
     write(*,"(a,f8.5/)") "`eff_f_p` = ", eff_f_p
     write(*,"(a,f8.5/)") "`oto_deviation` = ", oto_deviation

!++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! Output the catalogs for the first simulation if requested.

     if (output_catalogs .and. i_simul == 1) then
        call write_catalogs(K_u_file = K_u_file, K_p_file = K_p_file, &
             n_u = n_u, n_p = n_p, &
             semi_axis_a_u = semi_axis_a_u, semi_axis_b_u = semi_axis_b_u, &
             semi_axis_a_p = semi_axis_a_p, semi_axis_b_p = semi_axis_b_p, &
             one_to_one = one_to_one, area_S = area_S, seed = seed, &
             coord_u = coord_u, coord_p = coord_p)
     endif

!######################################################################
! General preliminaries.

! Required for all types (s:o, o:s and o:o) of computations.
! Must be run any time the coordinates or the positional uncertainties
! are changed.

! Compute "*_main" fields of `assoc_data_u` and `assoc_data_p`.
     call general_preliminaries(area_S = area_S, coord_1 = coord_u, &
          coord_2 = coord_p, &
          assoc_data_1 = assoc_data_u, assoc_data_2 = assoc_data_p)

!######################################################################
! Several-to-one analysis (a $K'$-source may be the counterpart of several
! $K$-sources, but any $K$-source has at most one counterpart in $K'$).

     write(*,"(3a/)") repeat("+", 20), " Several-to-one computations ", &
          repeat("+", 21)

! Compute `sto_f_u`, `std_dev_sto_f_u` and `sto_f_p`.
     call sto_analysis(assoc_data_1 = assoc_data_u, &
          assoc_data_2 = assoc_data_p, sto_f_1 = sto_f_u, &
          std_dev_sto_f_1 = std_dev_sto_f_u, sto_f_2 = sto_f_p &
! , start_f_1 = 0.5_real_type &
          )
! (Argument `start_f_1` (= $f_0$, starting value for iteration), commented out
! here, is optional; if present, must be of type `real_type` and in ]0, 1[.
! For a series of simulations with the same input parameters, one may want to
! set `start_f_1` to the mean value obtained for `sto_f_u` in previous
! simulations: this will accelerate the convergence.)

     write(*,"(a,f8.5,a,f8.5/)") "`sto_f_u` = ", sto_f_u, " +/- ", &
          std_dev_sto_f_u
     write(*,"(a,f8.5/)") "`sto_f_p` = ", sto_f_p

! Compute `max_ln_sto_Lh` = `func_ln_sto_Lh` at `f_u` = `sto_f_u`.
     max_ln_sto_Lh = func_ln_sto_Lh( &
          assoc_data_1 = assoc_data_u, assoc_data_2 = assoc_data_p, &
          f_1 = sto_f_u)
     write(*,"(a,es12.5/)") "`max_ln_sto_Lh` = ", max_ln_sto_Lh

!######################################################################
! One-to-several analysis (a $K$-source may be the counterpart of several
! $K'$-sources, but any $K'$-source has at most one counterpart in $K$).

     write(*,"(3a/)") repeat("+", 20), " One-to-several computations ", &
          repeat("+", 21)

! Compute `ots_f_p`, `std_dev_ots_f_p` and `ots_f_u`.
     call ots_analysis(assoc_data_1 = assoc_data_u, &
          assoc_data_2 = assoc_data_p, ots_f_2 = ots_f_p, &
          std_dev_ots_f_2 = std_dev_ots_f_p, ots_f_1 = ots_f_u &
! , start_f_2 = 0.5_real_type &
          )
! (Argument `start_f_2` (= $f_0$, starting value for iteration), commented out
! here, is optional; if present, must be of type `real_type` and in ]0, 1[.
! For a series of simulations with the same input parameters, one may want to
! set `start_f_2` to the mean value obtained for `ots_f_p` in previous
! simulations: this will accelerate the convergence.)

     write(*,"(a,f8.5,a,f8.5/)") "`ots_f_p` = ", ots_f_p, " +/- ", &
          std_dev_ots_f_p
     write(*,"(a,f8.5/)") "`ots_f_u` = ", ots_f_u

! Compute `max_ln_ots_Lh` = `func_ln_ots_Lh` at `f_p` = `ots_f_p`.
     max_ln_ots_Lh = func_ln_ots_Lh( &
          assoc_data_1 = assoc_data_u, assoc_data_2 = assoc_data_p, &
          f_2 = ots_f_p)
     write(*,"(a,es12.5/)") "`max_ln_ots_Lh` = ", max_ln_ots_Lh

!######################################################################
! One-to-one analysis (any $K$-source has at most one counterpart in $K'$
! and conversely) starting from catalog $K$.

     write(*,"(3a/)") repeat("+", 22), " One-to-one computations ", &
          repeat("+", 23)

! Specific preliminaries required (in addition to, and after, the general
! preliminaries detailed supra) for the one-to-one analysis and the
! computation of one-to-one probabilities.
! Must be run any time the coordinates or the positional uncertainties
! are changed.

! Compute "*ngb" and "*_any" fields of `assoc_data_u`.
     call oto_preliminaries(coord_1 = coord_u, n_2 = n_p, &
          assoc_data_1 = assoc_data_u)

! Compute `oto_f_u` and `std_dev_oto_f_u`.
     call oto_analysis(assoc_data_1 = assoc_data_u, n_2 = n_p, &
          oto_f_1 = oto_f_u, std_dev_oto_f_1 = std_dev_oto_f_u, &
          oto_f_2 = oto_f_p &
          , start_f_1 = sto_f_u &
          )
! (Argument `start_f_1` (= $f_0$, starting value for iteration) is optional
! and may be commented out; if present, must be of type `real_type` and in
! ]0, 1[.
! For a series of simulations with the same input parameters, one may want to
! set `start_f_1` to the mean value obtained for `oto_f_u` in previous
! simulations: this will accelerate the convergence.)

     write(*,"(a,f8.5,a,f8.5/,a/)") "`oto_f_u` = ", oto_f_u, " +/- ", &
          std_dev_oto_f_u, "(computed from $K'$-counterparts to $K$-sources)"

     write(*,"(a,f8.5/,a/)") "`oto_f_p` = ", oto_f_p

! Compute `max_ln_oto_Lh` = {`ln_oto_Lh` at `f_u` = `oto_f_u`}.
     call compute_ln_oto_Lh(assoc_data_1 = assoc_data_u, &
          assoc_data_2 = assoc_data_p &
!          , n_grid = n_grid, f_grid = f_grid, ln_oto_Lh_grid = ln_oto_Lh_grid &
          , f_1 = oto_f_u, ln_oto_Lh = max_ln_oto_Lh &
          )
! At least one of the lines "n_grid = [...]" and "f_1 = [...]" above must be 
! present.
!
! If the line "n_grid = [...]" is present, `compute_ln_oto_Lh` creates for 
! some given integer `n_grid`
! * a grid `f_grid` of `n_grid` equidistant values of `f_1` from 0 to the 
!   maximal possible value of `f_1`,
! * a grid `ln_oto_Lh_grid` containing the corresponding values of `ln_oto_Lh`.
!
! If the line "f_1 = [...]" is present, `compute_ln_oto_Lh` computes the value 
! of `ln_oto_Lh` at some given real `f_1` (typically `oto_f_u`, the maximum 
! likelihood estimator of `f_1`).
     write(*,"(a,es12.5/,a/)") "`max_ln_oto_Lh` = ", max_ln_oto_Lh, &
          "(computed from $K'$-counterparts to $K$-sources)"

  end do

end program example_simul
