! Code Aspects, version 2.0.1.
! Michel Fioc, 2014-4-29. 

program example_read

! Read catalogs, analyze them and output probabilities of association.
! Run either with
!         "example_read < input_example_read.dat"
! or with
!         "example_read"
! and answer the questions.

  use mod_Aspects! , only : general_preliminaries, oto_preliminaries, &
!& sto_analysis, ots_analysis, oto_analysis, &
!& func_ln_sto_Lh, func_ln_ots_Lh, compute_ln_oto_Lh, func_ln_oto_Lh_alt, &
!& compute_sto_prob, compute_ots_prob, compute_oto_prob
  use mod_variables! , only : area_S, K_u_file, K_p_file, n_u, n_p, &
!& coord_u, coord_p, assoc_data_u, assoc_data_p, &
!& sto_f_u, ots_f_p, sto_f_p, ots_f_u, oto_f_u, oto_f_p, &
!& std_dev_sto_f_u, std_dev_ots_f_p, std_dev_oto_f_u, &
!& max_ln_sto_Lh, max_ln_ots_Lh, max_ln_oto_Lh, &
!& n_grid, i_grid, f_grid, ln_oto_Lh_grid, &
!& sto_prob_u_file, sto_prob_p_file, ots_prob_u_file, &
!& ots_prob_p_file, oto_prob_u_file, oto_prob_p_file, &
!& sto_prob_u, ots_prob_u, oto_prob_u, sto_prob_p, ots_prob_p, oto_prob_p,  
!& skip_checks, oto_prob_u_file_sw, oto_prob_p_file_sw, &
!& oto_prob_u_sw, oto_prob_p_sw, &
!& oto_f_u_sw, oto_f_p_sw, std_dev_oto_f_p_sw, &
!& max_ln_oto_Lh_alt, max_ln_oto_Lh_sw, max_ln_oto_Lh_alt_sw
  use mod_read_catalog!, only : read_catalog, area_S_u, area_S_p
  use mod_output_prob!, only : output_sto_prob, output_ots_prob, output_oto_prob

  implicit none

!######################################################################
! Read the names of the input files containing the catalogs and of the files
! into which the probabilities of association are output.

  write(*,"(/a)", advance="no") "Input file containing catalog $K$: "
  read(*,*) K_u_file

  write(*,"(/a)", advance="no") "Input file containing catalog $K'$: "
  read(*,*) K_p_file

  write(*,"(/a)", advance="no") "Output file containing the several-to-one &
       &$K$-$K'$ probabilities of $K$-sources: "
  read(*,*) sto_prob_u_file

  write(*,"(/a)", advance="no") "Output file containing the several-to-one &
       &$K$-$K'$ probabilities of $K'$-sources: "
  read(*,*) sto_prob_p_file

  write(*,"(/a)", advance="no") "Output file containing the one-to-several &
       &$K$-$K'$ probabilities of $K$-sources: "
  read(*,*) ots_prob_u_file

  write(*,"(/a)", advance="no") "Output file containing the one-to-several &
       &$K$-$K'$ probabilities of $K'$-sources: "
  read(*,*) ots_prob_p_file

  write(*,"(/a)", advance="no") "Output file containing the one-to-one &
       &$K$-$K'$ probabilities of $K$-sources: "
  read(*,*) oto_prob_u_file
  
  write(*,"(/a)", advance="no") "Output file containing the one-to-one &
       &$K$-$K'$ probabilities of $K'$-sources: "
  read(*,*) oto_prob_p_file

  write(*,"(/a)", advance="no") "Skip checks of one-to-one computations &
       &(`T`/`F`)? "
  read(*,*) skip_checks
  write(*,"(/)")

!######################################################################
! Read catalogs and the area of the surface they cover.

  call read_catalog(K_u_file, n_u, area_S_u, coord_u)
  call read_catalog(K_p_file, n_p, area_S_p, coord_p)

  if (area_S_u /= area_S_p) then
     write(*,"(a)") "Both catalogs must cover the same surface!"
     stop
  else ! The surface *area* covered by both catalogs is the same (which
!& does not mean that they cover the same *surface*, unless the area is
!& 4 `pi` or 0).
     area_S = area_S_u
  endif

!++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! Note: case of unknown positional uncertainty parameters.

! The values of the semi-major and semi-minor axes, and of the position angle
! of the positional uncertainty ellipse are required hereafter.
! If these quantities are unknown for one or the other catalog, or for both,
! one can do the following:
!
!   For the catalog(s) with unknown positional uncertainty, take the same
!   semi-major axis for all sources, set the semi-minor axis to the same value
!   and the position angle to whatever value (it does not matter);
!   then, compute the likelihood with the functions `func_ln_sto_Lh`,
!   `func_ln_ots_Lh` or the subroutine `compute_ln_oto_Lh`.
!   Do this for a series of values of the semi-major axis, and adopt
!   the value maximizing the likelihood.
!
!   If the positional uncertainties are unknown in both catalogs, one
!   can set the semi-major (and semi-minor) axes to 0 in one of them: only
!   `sqrt(coord_u%semi_axis_a**2 + coord_p%semi_axis_a**2)` will matter.

!######################################################################
! General preliminaries.

! Required for all types (s:o, o:s and o:o) of computations.
! Must be run any time the coordinates or the positional uncertainties
! are changed.

! Compute `assoc_data_u` and `assoc_data_p`.

  call general_preliminaries(area_S = area_S, coord_1 = coord_u, &
       coord_2 = coord_p, assoc_data_1 = assoc_data_u, &
       assoc_data_2 = assoc_data_p)

!######################################################################
! Several-to-one analysis (a $K'$-source may be the counterpart of several
! $K$-sources, but any $K$-source has at most one counterpart in $K'$).

! Compute `sto_f_u`, `std_dev_sto_f_u` and `sto_f_p`.
  call sto_analysis(assoc_data_1 = assoc_data_u, assoc_data_2 = assoc_data_p, &
       sto_f_1 = sto_f_u, std_dev_sto_f_1 = std_dev_sto_f_u, sto_f_2 = sto_f_p &
! , start_f_1 = 0.5_real_type &
       )
! (Argument `start_f_1` (= $f_0$: starting value for the iteration), commented 
! out here, is optional; if present, must be of type `real_type` and in ]0, 1[.
! For a series of simulations with the same input parameters, one may want to
! set `start_f_1` to the mean value obtained for `sto_f_u` in previous
! simulations: this will accelerate the convergence.)

  write(*,"(/3a/)") repeat("+", 15), &
       " Fraction of sources with a counterpart ", repeat("+", 15)
  write(*,"(a,f8.5,a,f8.5/)") "`sto_f_u` = ", sto_f_u, " +/- ", std_dev_sto_f_u
  write(*,"(a,f8.5/)") "`sto_f_p` = ", sto_f_p

!######################################################################
! One-to-several analysis (a $K$-source may be the counterpart of several
! $K'$-sources, but any $K'$-source has at most one counterpart in $K$).

! Compute `ots_f_p`, `std_dev_ots_f_p` and `ots_f_u`.
  call ots_analysis(assoc_data_1 = assoc_data_u, assoc_data_2 = assoc_data_p, &
       ots_f_1 = ots_f_u, ots_f_2 = ots_f_p, &
       std_dev_ots_f_2 = std_dev_ots_f_p  &
! , start_f_2 = 0.5_real_type &
       )
! (Argument `start_f_2` (= $f_0$: starting value for the iteration), commented 
! out here, is optional; if present, must be of type `real_type` and in ]0, 1[.
! For a series of simulations with the same input parameters, one may want to
! set `start_f_2` to the mean value obtained for `ots_f_p` in previous
! simulations: this will accelerate the convergence.)

  write(*,"(a,f8.5,a,f8.5/)") "`ots_f_p` = ", ots_f_p, " +/- ", std_dev_ots_f_p
  write(*,"(a,f8.5/)") "`ots_f_u` = ", ots_f_u

!######################################################################
! One-to-one analysis (any $K$-source has at most one counterpart in $K'$
! and conversely). Analysis starting from catalog $K$.

! Specific preliminaries required (in addition to, and after, the general
! preliminaries; see `general_preliminaries` supra) for the one-to-one 
! analysis and the computation of one-to-one probabilities.
! Must be run any time the coordinates or the positional uncertainties
! are changed.

! Compute `*ngb` and `*_any` fields of `assoc_data_u`.
  call oto_preliminaries(coord_1 = coord_u, n_2 = n_p, &
       assoc_data_1 = assoc_data_u)

! Compute `oto_f_u` and `std_dev_oto_f_u`.
  call oto_analysis(assoc_data_1 = assoc_data_u, n_2 = n_p, &
       oto_f_1 = oto_f_u, std_dev_oto_f_1 = std_dev_oto_f_u, oto_f_2 = oto_f_p &
       , start_f_1 = sto_f_u &
       )
! (Argument `start_f_1` (= $f_0$, starting value for iteration) is optional;
! here, we start with the several-to-one estimate.
! For a series of simulations with the same input parameters, one may want to
! set `start_f_1` to the mean value obtained for `oto_f_u` in previous
! simulations: this will accelerate the convergence.)

  write(*,"(a,f8.5,a,f8.5/,a/)") "`oto_f_u` = ", oto_f_u, " +/- ", &
       std_dev_oto_f_u, "(computed from $K'$-counterparts to $K$-sources)"
  write(*,"(a,f8.5/,a/)") "`oto_f_p`: ", oto_f_p

!######################################################################
! Choice of model.

  write(*,"(/3a/)") repeat("+", 20), " Choice of association model ", &
       repeat("+", 21)

  write(*,"(a,es10.4/)") "`n_u*sto_f_u` = ", n_u*sto_f_u
  write(*,"(a,es10.4/)") "`n_p*ots_f_p` = ", n_p*ots_f_p

  write(*,"(a)") "Recommendation:"
  write(*,"(a)") "* if `n_u*sto_f_u` ~= `n_p*ots_f_p`, &
       &use the one-to-one association model;"
  write(*,"(a)") "* else,"
  write(*,"(a)") "  -- if `n_u*sto_f_u` > `n_p*ots_f_p`, &
       &use the several-to-one association model;"
  write(*,"(a)") "  -- if `n_u*sto_f_u` < `n_p*ots_f_p`, &
       &use the one-to-several association model."
  write(*,"(a/)") "(Compare `max_ln_sto_Lh`, `max_ln_ots_Lh` and &
       &`max_ln_oto_Lh` below to confirm.)"

! Compute `max_ln_sto_Lh` = `func_ln_sto_Lh`(`f_u` = `sto_f_u`).
  max_ln_sto_Lh = func_ln_sto_Lh(assoc_data_1 = assoc_data_u, &
       assoc_data_2 = assoc_data_p, f_1 = sto_f_u)
  write(*,"(a,es12.5/)") "`max_ln_sto_Lh` = ", max_ln_sto_Lh

! Compute `max_ln_ots_Lh` = `func_ln_ots_Lh`(`f_p` = `ots_f_p`).
  max_ln_ots_Lh = func_ln_ots_Lh(assoc_data_1 = assoc_data_u, &
       assoc_data_2 = assoc_data_p, f_2 = ots_f_p)
  write(*,"(a,es12.5/)") "`max_ln_ots_Lh` = ", max_ln_ots_Lh

! Compute `max_ln_oto_Lh` = value of `ln_oto_Lh` for `f_u` = `oto_f_u`.
  call compute_ln_oto_Lh(assoc_data_1 = assoc_data_u, &
       assoc_data_2 = assoc_data_p, f_1 = oto_f_u, ln_oto_Lh = max_ln_oto_Lh)
  write(*,"(a,es12.5/)") "`max_ln_oto_Lh` = ", max_ln_oto_Lh

!######################################################################
! Probabilities of association.

  write(*,"(/3a/)") repeat("+", 20), " Probabilities of association ", &
       repeat("+", 20)

  write(*,"(a)") "Recommendation:"
  write(*,"(a)") "* for several-to-one probabilities, &
       &use subroutine `compute_sto_prob` with `f_1` = `sto_f_u`;"
  write(*,"(a)") "* for one-to-several probabilities, &
       &use subroutine `compute_ots_prob` with `f_2` = `ots_f_p`;"
  write(*,"(a/)") "* for one-to-one probabilities, &
       &use subroutine `compute_oto_prob` with `f_1` = `oto_f_u`."

!++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! Several-to-one probabilities.

  call compute_sto_prob(assoc_data_1 = assoc_data_u, n_2 = n_p, f_1 = sto_f_u, &
       sto_prob_1 = sto_prob_u, sto_prob_2 = sto_prob_p)

  call output_sto_prob(K_u_file = K_u_file, K_p_file = K_p_file, &
       f_u = sto_f_u, sto_prob_u_file = sto_prob_u_file, &
       sto_prob_p_file = sto_prob_p_file, sto_prob_u = sto_prob_u, &
       sto_prob_p = sto_prob_p)

  write(*,"(5a/)") "Several-to-one probabilities of association written &
       &in files """, trim(sto_prob_u_file), """ and """, &
       trim(sto_prob_p_file), """."

!++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! One-to-several probabilities.

  call compute_ots_prob(n_1 = n_u, assoc_data_2 = assoc_data_p, f_2 = ots_f_p, &
       ots_prob_1 = ots_prob_u, ots_prob_2 = ots_prob_p)

  call output_ots_prob(K_u_file = K_u_file, K_p_file = K_p_file, &
       f_p = ots_f_p, ots_prob_u_file = ots_prob_u_file, &
       ots_prob_p_file = ots_prob_p_file, ots_prob_u = ots_prob_u, &
       ots_prob_p = ots_prob_p)

  write(*,"(5a/)") "One-to-several probabilities of association written &
       &in files """, trim(ots_prob_u_file), """ and """, &
       trim(ots_prob_p_file), """."

!++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! One-to-one probabilities.

  call compute_oto_prob(assoc_data_1 = assoc_data_u, n_2 = n_p, &
       f_1 = oto_f_u, oto_prob_1 = oto_prob_u, oto_prob_2 = oto_prob_p)

  call output_oto_prob(K_u_file = K_u_file, K_p_file = K_p_file, f = oto_f_u, &
       oto_prob_u_file = oto_prob_u_file, oto_prob_p_file = oto_prob_p_file, &
       oto_prob_u = oto_prob_u, oto_prob_p = oto_prob_p)

  write(*,"(5a/)") "One-to-one probabilities of association written &
       &in files """, trim(oto_prob_u_file), """ and """, &
       trim(oto_prob_p_file), """."

!######################################################################
! For illustration: compute a grid of `n_grid` values of $(f, \ln L_{o:o})$.

  n_grid = 10
  call compute_ln_oto_Lh(assoc_data_1 = assoc_data_u, &
       assoc_data_2 = assoc_data_p &
       , n_grid = n_grid, f_grid = f_grid, ln_oto_Lh_grid = ln_oto_Lh_grid &
!      , f_1 = ..., ln_oto_Lh = ... &
       )
! At least one of the lines "n_grid = [...]" and "f_1 = [...]" above must be 
! present.
!
! If the line "n_grid = [...]" is present, `compute_ln_oto_Lh` creates for 
! some given integer `n_grid`
! * a grid `f_grid` of `n_grid` equidistant values of `f_1` from 0 to the 
! maximal possible value of `f_1`,
! * a grid `ln_oto_Lh_grid` containing the corresponding values of `ln_oto_Lh`.
!
! If the line "f_1 = [...]" is present, `compute_ln_oto_Lh` computes the value 
! of `ln_oto_Lh` at some given real `f_1` (typically `oto_f_u`, the maximum 
! likelihood estimator of `f_1`).

  write(*,"(3a/)") repeat("+", 20), " Grid of (`f_u`, `ln_oto_Lh`) ", &
       repeat("+", 20)
  write(*,"(a,f8.5,a,es12.5,a)") ("(", f_grid(i_grid), ", ", &
       ln_oto_Lh_grid(i_grid), ")", i_grid = 1, n_grid)
  write(*,"(/)")

!######################################################################
! Checks of one-to-one computations. May be skipped.

  if (skip_checks) stop
  
  write(*,"(3a/)") repeat("+", 17), " Checks of one-to-one computations ", &
       repeat("+", 18)

! Alternative (and inefficient) computation of `max_ln_oto_Lh`.
  write(*,"(a)") "Alternative procedure:"
  max_ln_oto_Lh_alt = func_ln_oto_Lh_alt(assoc_data_1 = assoc_data_u, &
       assoc_data_2 = assoc_data_p, f_1 = oto_f_u)
  write(*,"(a,es12.5/)") "`max_ln_oto_Lh` = ", max_ln_oto_Lh_alt

!++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! Checks with $K$ and $K'$ swapped.

  write(*,"(3a/)") repeat("-", 15), " Computations with $K$ and $K'$ &
       &swapped ", repeat("-", 15)

  call oto_preliminaries(coord_1 = coord_p, n_2 = n_u, &
       assoc_data_1 = assoc_data_p)

  call oto_analysis(assoc_data_1 = assoc_data_p, n_2 = n_u, &
       oto_f_1 = oto_f_p_sw, std_dev_oto_f_1 = std_dev_oto_f_p_sw, &
       oto_f_2 = oto_f_u_sw &
       , start_f_1 = ots_f_p &
       )
  write(*,"(a,f8.5,a,f8.5/)") "`oto_f_p` = ", oto_f_p_sw, " +/- ", &
       std_dev_oto_f_p_sw
  write(*,"(a,f8.5/)") "`oto_f_u` = ", oto_f_u_sw

! Compute `max_ln_oto_Lh` = value of `ln_oto_Lh` for `f_p` = `oto_f_p`.
  call compute_ln_oto_Lh(assoc_data_1 = assoc_data_p, &
       assoc_data_2 = assoc_data_u, f_1 = oto_f_p, ln_oto_Lh = max_ln_oto_Lh_sw)
  write(*,"(a)") "Computation of `max_ln_oto_Lh` with $K$ and $K'$ swapped:"
  write(*,"(a,es12.5/,a/)") "`max_ln_oto_Lh` = ", max_ln_oto_Lh_sw

! Alternative (and inefficient) computation of `max_ln_oto_Lh`.
  write(*,"(a)") "Alternative procedure:"
  max_ln_oto_Lh_alt_sw = func_ln_oto_Lh_alt(assoc_data_1 = assoc_data_p, &
       assoc_data_2 = assoc_data_u, f_1 = oto_f_p)
  write(*,"(a,es12.5/)") "`max_ln_oto_Lh` = ", max_ln_oto_Lh_alt_sw

! Swap $K$ and $K'$ and compute o:o probabilities anew.
! Within numerical errors, one should obtain the same values in the new files 
! as in the previous ones.

  call compute_oto_prob(assoc_data_1 = assoc_data_p, n_2 = n_u, &
       f_1 = oto_f_p, oto_prob_1 = oto_prob_p_sw, oto_prob_2 = oto_prob_u_sw)

  oto_prob_u_file_sw = trim(oto_prob_u_file)//"_sw"
  oto_prob_p_file_sw = trim(oto_prob_p_file)//"_sw"
  call output_oto_prob(K_u_file = K_u_file, K_p_file = K_p_file, f = oto_f_p, &
       oto_prob_u_file = oto_prob_u_file_sw, &
       oto_prob_p_file = oto_prob_p_file_sw, &
       oto_prob_u = oto_prob_u_sw, oto_prob_p = oto_prob_p_sw, swapped = .true.)

  write(*,"(a)") "One-to-one probabilities of association computed with $K$ &
       &and $K'$ swapped."
  write(*,"(5a)") "Written in files """, trim(oto_prob_u_file_sw), &
       """ and """, trim(oto_prob_p_file_sw), """."
  write(*,"(5a)") "Compare them to those in """, trim(oto_prob_u_file), &
       """ and """, trim(oto_prob_p_file), """."

end program example_read
