///////////////////////////////////////////////////////////////////////////////////////////////
//
//	The KEPLERH program solves the Kepler equation for hyperbolic Keplerian orbits. The code is 
//	collected in the following modules: Keplerh.c, polinomios25.c polinomios25Q.c and hyperk.h.
//  The module main.c is a driver to check good operation of the program
//	Copyright (C) 2018 by the UNIVERSIDAD POLITECNICA DE MADRID (UPM)  
//	AUthors: Virginia Raposo-Pulido and Jesus Pelaez
//
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <https://www.gnu.org/licenses/>.
//
//////////////////////////////////////////////////////////////////////////////////////////////////

#include "hyperk.h"

/*************************************************************************************/
//
//This file contains the 25 polynomials PPHi(e, M) that provide the seed (S = sinh(H)) 
//of the modified Newton-Rapson method that solves the hyperbolic Kepler equation.
//The polynomials in this module are grade 5.
//
/*************************************************************************************/




double PPH0_ke (double e, double x)
{
  double t80;
  t80 = x * x;
  return((0.1e1 / (e - 0.1e1) + ((0.125619031568129558537380271410332022e1 + (-0.386896297445922329942791253526031698e1 + (0.397458463649162035319156450269314720e1 - 0.136187300221481648906471060024501584e1 * e) * e) * e) / (-0.753709874513349572842363902433619830e1 + (0.533644703178604011903435655253666572e2 + (-0.161926955103827310067286221499564231e3 + (0.272966483185125737342108674783906395e3 + (-0.276087334554369700721547069525986956e3 + (0.167544674481392071366040635772371738e3 + (-0.564856332122875304429233576403064543e2 + 0.816139363123982706168741160854905003e1 * e) * e) * e) * e) * e) * e) * e) + ((-0.958688217924823935501782842000181705e-4 + (-0.510121027425997692040825072704324965e-2 + (-0.164741666448638963333205206206751994e-1 + 0.220780757484086884956517410243543940e-1 * e) * e) * e) / (0.753709874513349572842363902433619829e1 + (-0.609519169783737701972277800942650357e2 + (0.215647900761431692125272092573189541e3 + (-0.435975112406404674084898863736474596e3 + (0.550877237315339524214194941686440662e3 + (-0.445476275938404291166795839088250074e3 + (0.225149508247949125762430962753541289e3 + (-0.650243515910619529228042795155324044e2 + 0.821591184439085054140512639701441960e1 * e) * e) * e) * e) * e) * e) * e) * e) + (0.252674257989487261929593560802687026e-1 + (0.181156144842423602994829135698156169e0 + (-0.421277152858462836797440502550471456e0 + 0.214548459711471257973372417227719640e0 * e) * e) * e) / (-0.301483949805339829136945560973447932e1 + (0.274157454555548556816447978652653657e2 + (-0.110802790927857499178963701893328363e3 + (0.261225417553932260103325655073133759e3 + (-0.395905867604755066389384777273332807e3 + (0.400013352079265784684681827805794974e3 + (-0.269440628547842955871503875019778378e3 + (0.116671144565895074989564934132360149e3 + (-0.294698507720984166784906571885258136e2 + 0.330831769595936095049525210814559361e1 * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * t80) * x);
}
double PPH1_ke (double e, double x)
{
  double t142;
  double t43;
  double t45;
  t43 = 0.1e1 / (-0.238616181075569079907279719182764268e1 + (0.275335046629707935784776320531701710e2 + (-0.144404347605108829868184376500440285e3 + (0.454391460150503644373651037094430466e3 + (-0.953166298731923767348891393434001460e3 + (0.139953943135848347406023034456708312e4 + (-0.146775471427404348587374473986230463e4 + (0.109944556297142040921759641375598407e4 + (-0.576463711085188909450697471538009066e3 + (0.201492520488408993640339382050770354e3 + (-0.422549047335956582404149780258192066e2 + 0.402765860882902908758861926713430584e1 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e);
  t45 = 0.161106344353161163503544770685372234e2 * e;
  t142 = 0.322212688706322327007089541370744467e1 * e;
  return((0.441681431719762304482474427403630828e-6 + (0.210600067264282750473533248753425210e-4 + (-0.783908468734597172411928424264364364e-4 + (-0.182677241818020378169948260414258184e-3 + (0.160917036523310634848317011586842978e-2 + (-0.419356078693553844081620751253818388e-2 + (0.596499119680092068638877427446379436e-2 + (-0.514621565468353590650182294170244496e-2 + (0.269988206058691284098003332874180506e-2 + (-0.796389608234277980000603932072960320e-3 + 0.101688827765993520938641307467846188e-3 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * t43 + ((0.954468599874300963478237381060534106e1 + (-0.100587486924494057487179383629089804e3 + (0.477023011493003462894342570427622880e3 + (-0.134055873962193595575227041922717069e4 + (0.247224741379262656775025484211196854e4 + (-0.312627830151316121823625602363481434e4 + (0.274526473482406838132837482427681640e4 + (-0.165297037681733777476968460486844184e4 + (0.653122399374886975350003580055156448e3 + (-0.152917975041714564921669264357637946e3 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.954464724302276319629118876731057070e1 + (0.110134018651883174313910528212680683e3 + (-0.577617390420435319472737506001761138e3 + (0.181756584060201457749460414837772186e4 + (-0.381266519492769506939556557373600583e4 + (0.559815772543393389624092137826833248e4 + (-0.587101885709617394349497895944921852e4 + (0.439778225188568163687038565502393629e4 + (-0.230585484434075563780278988615203627e4 + (0.805970081953635974561357528203081416e3 + (-0.169019618934382632961659912103276826e3 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.427513027921542490462253860670661972e-4 + (0.211809155515240211603487596398432339e-2 + (-0.343639060082082472399128169078482778e-2 + (-0.263456496890794627043907598660744495e-1 + (0.105956941158480254731075669853793467e0 + (-0.168156045957631643274017615356144260e0 + (0.138051446720670290344493912596450072e0 + (-0.582845735493737619454790926132199942e-1 + 0.100534290306820672113739896258204380e-1 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.119308090537784539953639859591382134e1 + (0.137667523314853967892388160265850855e2 + (-0.722021738025544149340921882502201422e2 + (0.227195730075251822186825518547215232e3 + (-0.476583149365961883674445696717000730e3 + (0.699769715679241737030115172283541561e3 + (-0.733877357137021742936872369931152315e3 + (0.549722781485710204608798206877992034e3 + (-0.288231855542594454725348735769004533e3 + (0.100746260244204496820169691025385177e3 + (-0.211274523667978291202074890129096033e2 + 0.201382930441451454379430963356715292e1 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.318461770589434056215336106999447265e0 + (-0.238303988488018227538459089756741503e1 + (0.774132523659477032714993498568266667e1 + (-0.141154616195493316065894264114485701e2 + (0.155628254837319779963126995848032127e2 + (-0.103510185811609959634482849089719684e2 + (0.383778698958924693915417105401497981e1 - 0.610879387955369194694186182791099658e0 * e) * e) * e) * e) * e) * e) * e) / (-0.190892944860455263925823775346211414e1 + (0.220268037303766348627821056425361368e2 + (-0.115523478084087063894547501200352228e3 + (0.363513168120402915498920829675544372e3 + (-0.762533038985539013879113114747201169e3 + (0.111963154508678677924818427565366650e4 + (-0.117420377141923478869899579188984370e4 + (0.879556450377136327374077131004787257e3 + (-0.461170968868151127560557977230407252e3 + (0.161194016390727194912271505640616284e3 + (-0.338039237868765265923319824206553654e2 + t142) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.894879784547056742126937948435161388e-3 + (0.458642255079603342465232987057286800e-1 + (0.157079192952402224486295878817970611e-1 + (-0.555953277679028219255495287735324531e0 + (0.110088206581843767338087653178896169e1 + (-0.831378374839105181373684877346963669e0 + 0.223981944333084336327985781628871179e0 * e) * e) * e) * e) * e) * e) * t43 + (0.166408600868465672227361686101077701e-1 + (0.109067828060139926486683547249842732e0 + (-0.478874700133663131798068897440184237e0 + (0.609088830872959881995975355726874746e0 + (-0.301405675043546555675840074552598332e0 + 0.454960657539946913684117166910634097e-1 * e) * e) * e) * e) * e) / (-0.190892944860455263925823775346211414e1 + (0.220268037303766348627821056425361367e2 + (-0.115523478084087063894547501200352228e3 + (0.363513168120402915498920829675544372e3 + (-0.762533038985539013879113114747201169e3 + (0.111963154508678677924818427565366650e4 + (-0.117420377141923478869899579188984371e4 + (0.879556450377136327374077131004787256e3 + (-0.461170968868151127560557977230407253e3 + (0.161194016390727194912271505640616283e3 + (-0.338039237868765265923319824206553653e2 + t142) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH2_ke (double e, double x)
{
  double t45;
  t45 = 0.117658478199243300997225315957872238e2 * e;
  return((0.103410951603657825056575644542441055e-4 + (0.523580762459535053619393179545049645e-3 + (-0.178271021513901359493381689140590216e-2 + (-0.572460799194923854525965726679171592e-2 + (0.433312319207436206554428376937265071e-1 + (-0.108407310008766068649887547845215840e0 + (0.149068145524211342281365521080259248e0 + (-0.123852786394552472729765560829429736e0 + (0.620725377313505632231416797241517778e-1 + (-0.172918417692314286398583666277049691e-1 + 0.205341937552578475178183049356804313e-2 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.121621179511730431325840442830693298e1 + (0.151383635658768880361281973914494554e2 + (-0.856397604533250800376720617447756236e2 + (0.290651596961601051454336396343674509e3 + (-0.657549215830237037124159479383893950e3 + (0.104119345354702756144703810374046908e4 + (-0.117748712196784009806423875122927831e4 + (0.951048161144004362857436895214552161e3 + (-0.537646189276421752930647838149929248e3 + (0.202604500106144032352413751061008520e3 + (-0.458039151295317402947022561416315512e2 + 0.470633912796973203988901263831488949e1 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.304084368706776845461161320942534096e1 + (-0.347892116031137873465201460886420077e2 + (0.179255682304441586772470982289484931e3 + (-0.547543932398890622757872963631369430e3 + (0.109763851022842982909266305842240649e4 + (-0.150863999844185408759306154543012898e4 + (0.143963467074929287048531431770880632e4 + (-0.941797722493692513215241777227645960e3 + (0.404244640067127247913962576544079152e3 + (-0.102809329907754127195569200848403678e3 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.304052948779326078314601107076733245e1 + (0.378459089146922200903204934786236386e2 + (-0.214099401133312700094180154361939059e3 + (0.726628992404002628635840990859186276e3 + (-0.164387303957559259281039869845973488e4 + (0.260298363386756890361759525935117270e4 + (-0.294371780491960024516059687807319578e4 + (0.237762040286001090714359223803638040e4 + (-0.134411547319105438232661959537482312e4 + (0.506511250265360080881034377652521301e3 + (-0.114509787823829350736755640354078878e3 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.155801166662432705622487874987754581e-2 + (0.813574645439041438270648986216925014e-1 + (-0.112212376342327926398054025449616695e0 + (-0.112824312838704039378881544185822701e1 + (0.431326856288550521570128845897129201e1 + (-0.664991790247573686190739047687040028e1 + (0.528271743172911867357911607271295280e1 + (-0.213619865939595883732310660275818985e1 + 0.347670143354271137118470855421429579e0 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.304052948779326078314601107076733245e1 + (0.378459089146922200903204934786236386e2 + (-0.214099401133312700094180154361939059e3 + (0.726628992404002628635840990859186272e3 + (-0.164387303957559259281039869845973487e4 + (0.260298363386756890361759525935117269e4 + (-0.294371780491960024516059687807319578e4 + (0.237762040286001090714359223803638040e4 + (-0.134411547319105438232661959537482312e4 + (0.506511250265360080881034377652521300e3 + (-0.114509787823829350736755640354078878e3 + 0.117658478199243300997225315957872237e2 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.204297478304816088737054669959015282e0 + (-0.162787788383417427879761462435178087e1 + (0.617113700309075579958763569917196044e1 + (-0.136944593990205608960446529973085339e2 + (0.184544950847552465796113835790972696e2 + (-0.147756105940785585952290790747519429e2 + (0.644679039051529422594404886140240722e1 - 0.117876653285711194085336362938062339e1 * e) * e) * e) * e) * e) * e) * e) / (-0.121621179511730431325840442830693298e1 + (0.151383635658768880361281973914494555e2 + (-0.856397604533250800376720617447756236e2 + (0.290651596961601051454336396343674509e3 + (-0.657549215830237037124159479383893951e3 + (0.104119345354702756144703810374046908e4 + (-0.117748712196784009806423875122927831e4 + (0.951048161144004362857436895214552162e3 + (-0.537646189276421752930647838149929247e3 + (0.202604500106144032352413751061008519e3 + (-0.458039151295317402947022561416315512e2 + 0.470633912796973203988901263831488948e1 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.107936848616708890432526585648471530e-1 + (0.576265390693283273048661313422361125e0 + (0.250889875573621677304921365622620455e0 + (-0.750382817323140310821304974967794082e1 + (0.150098299777437009056664972615859633e2 + (-0.114442925033946150990557011246159786e2 + 0.309988286056944858748974652921855012e1 * e) * e) * e) * e) * e) * e) / (-0.608105897558652156629202214153466488e1 + (0.756918178293844401806409869572472773e2 + (-0.428198802266625400188360308723878118e3 + (0.145325798480800525727168198171837255e4 + (-0.328774607915118518562079739691946975e4 + (0.520596726773513780723519051870234540e4 + (-0.588743560983920049032119375614639156e4 + (0.475524080572002181428718447607276081e4 + (-0.268823094638210876465323919074964624e4 + (0.101302250053072016176206875530504260e4 + (-0.229019575647658701473511280708157756e3 + 0.235316956398486601994450631915744475e2 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + (0.114367655483379337982625259547471716e-1 + (0.105802758797803092560192125826549331e0 + (-0.298582129630590643862033800562524865e0 + (0.113632411222050075062743470486955423e0 + (0.182528240539457775058982237637390183e0 - 0.114202078761065382208870212230288675e0 * e) * e) * e) * e) * e) / (-0.121621179511730431325840442830693298e1 + (0.151383635658768880361281973914494555e2 + (-0.856397604533250800376720617447756235e2 + (0.290651596961601051454336396343674510e3 + (-0.657549215830237037124159479383893950e3 + (0.104119345354702756144703810374046908e4 + (-0.117748712196784009806423875122927831e4 + (0.951048161144004362857436895214552161e3 + (-0.537646189276421752930647838149929248e3 + (0.202604500106144032352413751061008520e3 + (-0.458039151295317402947022561416315513e2 + 0.470633912796973203988901263831488950e1 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH3_ke (double e, double x)
{
  double t150;
  double t45;
  double t66;
  t45 = 0.125657651593564888932066539034618416e2 * e;
  t66 = (-0.109958583273762588107072106809450635e3 + t45) * e;
  t150 = (0.414683587586558221603230592144868883e4 + (-0.260793898085218366134885466229108655e4 + (0.109320264637919085407165606831567633e4 + (-0.274896458184406470267680267023626588e3 + 0.314144128983912222330166347586546038e2 * e) * e) * e) * e) * e;
  return((0.985439324685591071531518660820753175e-4 + (0.543817645338768327750865369480841092e-2 + (-0.161014967705027119371132132918254655e-1 + (-0.794916907995924809687373315864561419e-1 + (0.510476086732636796203673206126545750e0 + (-0.119335039594895394319076150784836328e1 + (0.151173305992445774429410929069077447e1 + (-0.110768899404176603456451991746760472e1 + (0.449011184155063535246536217260536926e0 + (-0.826696853599279243644084915816286849e-1 + 0.254529148625229188482789989128188418e-2 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.100363440830359617273821161475179279e1 + (0.139053652598139607073066113146512367e2 + (-0.875548563107291885975536854954577147e2 + (0.330706723121788531045706884136672386e3 + (-0.832584423139276429905207105689373463e3 + (0.146698390260569176747560511072345293e4 + (-0.184590007529440313991883131573163943e4 + (0.165873435034623288641292236857947553e4 + (-0.104317559234087346453954186491643462e4 + (0.437281058551676341628662427326270533e3 + (-0.109958583273762588107072106809450635e3 + 0.125657651593564888932066539034618415e2 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.100447140191978901622006317831494204e1 + (-0.128550012955225889630303319366765258e2 + (0.745611808775026163507045200591352690e2 + (-0.256799415376361139395769684922929148e3 + (0.580070409565270968984111670761819178e3 + (-0.897060979476403857010474880150650764e3 + (0.961920763205049148677389855234241954e3 + (-0.706677275317541060181453216422250414e3 + (0.340717707722124292393409534535724470e3 + (-0.974476224726818889707675384516237178e2 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.100363440830359617273821161475179279e1 + (0.139053652598139607073066113146512367e2 + (-0.875548563107291885975536854954577146e2 + (0.330706723121788531045706884136672386e3 + (-0.832584423139276429905207105689373464e3 + (0.146698390260569176747560511072345294e4 + (-0.184590007529440313991883131573163943e4 + (0.165873435034623288641292236857947554e4 + (-0.104317559234087346453954186491643462e4 + (0.437281058551676341628662427326270534e3 + t66) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.145554323951536768524233320269039871e-1 + (0.820168496933522026732985908988722665e0 + (-0.846444943212557512950812292792169130e0 + (-0.133680744306468277583821905463180293e2 + (0.477745068266959349878700579597646490e2 + (-0.691549338606637755152254910887030645e2 + (0.494262656952113184913081213435161614e2 + (-0.164143719836033875746716323418786548e2 + 0.174805813383347951689589749231912340e1 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.501817204151798086369105807375896394e1 + (0.695268262990698035365330565732561837e2 + (-0.437774281553645942987768427477288574e3 + (0.165353361560894265522853442068336193e4 + (-0.416292211569638214952603552844686733e4 + (0.733491951302845883737802555361726470e4 + (-0.922950037647201569959415657865819716e4 + (0.829367175173116443206461184289737771e4 + (-0.521587796170436732269770932458217312e4 + (0.218640529275838170814331213663135268e4 + (-0.549792916368812940535360534047253177e3 + 0.628288257967824444660332695173092078e2 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.431297204470433726767215745754005370e0 + (-0.337782754345778137036924288204879180e1 + (0.173590256026025392389243358237807758e2 + (-0.531624394585902084412564167344532555e2 + (0.908784999472588507726748583392122855e2 + (-0.853284059552665111589737611123853925e2 + (0.410856067593938952364467164153842535e2 - 0.788428707161600181370425805907677295e1 * e) * e) * e) * e) * e) * e) * e) / (-0.250908602075899043184552903687948197e1 + (0.347634131495349017682665282866280919e2 + (-0.218887140776822971493884213738644287e3 + (0.826766807804471327614267210341680968e3 + (-0.208146105784819107476301776422343366e4 + (0.366745975651422941868901277680863235e4 + (-0.461475018823600784979707828932909857e4 + t150) * e) * e) * e) * e) * e) * e) + ((0.125353421442181927774144578620928568e-1 + (0.710672578821082371073631403768405420e0 + (0.408806352921085598860806968195893355e0 + (-0.103355435437807065047738268170107954e2 + (0.209137974762788404682980389868827082e2 + (-0.159505733435407351327013269363079914e2 + 0.423137234424519292239195101655159900e1 * e) * e) * e) * e) * e) * e) / (-0.250908602075899043184552903687948196e1 + (0.347634131495349017682665282866280917e2 + (-0.218887140776822971493884213738644287e3 + (0.826766807804471327614267210341680966e3 + (-0.208146105784819107476301776422343367e4 + (0.366745975651422941868901277680863233e4 + (-0.461475018823600784979707828932909856e4 + t150) * e) * e) * e) * e) * e) * e) + (0.105052571665650529168947047468345850e-1 + (0.138242350756741555631283546310314423e0 + (-0.226592915633803388686172887936684430e0 + (-0.377681348355561459506718303074219109e0 + (0.791413557593421172606776597404133192e0 - 0.326824708839103910518682711753492696e0 * e) * e) * e) * e) * e) / (-0.100363440830359617273821161475179279e1 + (0.139053652598139607073066113146512367e2 + (-0.875548563107291885975536854954577148e2 + (0.330706723121788531045706884136672386e3 + (-0.832584423139276429905207105689373464e3 + (0.146698390260569176747560511072345294e4 + (-0.184590007529440313991883131573163943e4 + (0.165873435034623288641292236857947554e4 + (-0.104317559234087346453954186491643462e4 + (0.437281058551676341628662427326270532e3 + t66) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH4_ke (double e, double x)
{
  double t45;
  double t70;
  t45 = 0.476178514889444235658256852377661324e3 * e;
  t70 = (-0.266177549004829560765146744124755464e5 + (0.127310810035916640143248500700408054e5 + (-0.365249282448337056860881156475898467e4 + t45) * e) * e) * e;
  return((0.135198519133824042371112634507878287e-2 + (0.831539726473753349445050437788634190e-1 + (-0.200945291692304854662966068684404056e0 + (-0.167443556582244380296544758864966256e1 + (0.931169319293923304751693448465909050e1 + (-0.196634621375716499443157598063771122e2 + (0.203915575184103854864621766370453272e2 + (-0.845287984842083881363541170138172685e1 + (-0.228903832282403075764684689244750120e1 + (0.338299008267717186236433027268048587e1 - 0.889904829612789905191874689835953935e0 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.222201085068570698389694521573712690e1 + (0.351498275063426437215717324340990034e2 + (-0.252671524946284582526907232412656717e3 + (0.108948043903814806901909164265501595e4 + (-0.313090664524921359206674757838666206e4 + (0.629647768828964667514782596310404909e4 + (-0.904224019061972445715350392843290389e4 + (0.927267968260930502383319474776122202e4 + (-0.665443872512073901912866860311888661e4 + (0.318277025089791600358121251751020136e4 + (-0.913123206120842642152202891189746169e3 + 0.119044628722361058914564213094415331e3 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.892342069071682597844628139118378948e1 + (-0.129521488065112035654399210632923118e3 + (0.875762711650262388841937057446737476e3 + (-0.352407463213650590738801505788303804e4 + (0.923820904974540871140808034488676548e4 + (-0.164639114743420914232480833979943798e5 + (0.202690398049394336148314392546084192e5 + (-0.171041684621579512864521892556864532e5 + (0.951176455913749291608890817150759516e4 + (-0.315819446893078930135251063749644138e4 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.888804340274282793558778086294850760e1 + (0.140599310025370574886286929736396013e3 + (-0.101068609978513833010762892965062687e4 + (0.435792175615259227607636657062006381e4 + (-0.125236265809968543682669903135466482e5 + (0.251859107531585867005913038524161964e5 + (-0.361689607624788978286140157137316156e5 + (0.370907187304372200953327789910448880e5 + t70) * e) * e) * e) * e) * e) * e) * e) + ((0.948827801452782917465869801831510146e-1 + (0.589010442717807136063614036040195604e1 + (-0.350793432825464970816674360698718010e1 + (-0.118441966654287413445083532814168148e3 + (0.393662227336133985546964232183581762e3 + (-0.502983007519416517348883344092628816e3 + (0.249801527593636232224189608458171584e3 + (0.692777660474796385474564740803718490e1 - 0.314878629446322983797129204029305364e2 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.888804340274282793558778086294850760e1 + (0.140599310025370574886286929736396013e3 + (-0.101068609978513833010762892965062687e4 + (0.435792175615259227607636657062006381e4 + (-0.125236265809968543682669903135466482e5 + (0.251859107531585867005913038524161963e5 + (-0.361689607624788978286140157137316156e5 + (0.370907187304372200953327789910448881e5 + (-0.266177549004829560765146744124755463e5 + (0.127310810035916640143248500700408054e5 + (-0.365249282448337056860881156475898466e4 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.201668093469067975946333613761787395e0 + (-0.116647001788103398542009776688936372e1 + (0.113617860605125070156424609921306762e2 + (-0.513673989369863975848039483287517225e2 + (0.105805785010584755751068374900541908e3 + (-0.107517861009818589476817713271683898e3 + (0.514396421720416866400391674580498860e2 - 0.872952863588008667092650269201745390e1 * e) * e) * e) * e) * e) * e) * e) / (-0.111100542534285349194847260786856345e1 + (0.175749137531713218607858662170495017e2 + (-0.126335762473142291263453616206328359e3 + (0.544740219519074034509545821327507977e3 + (-0.156545332262460679603337378919333103e4 + (0.314823884414482333757391298155202454e4 + (-0.452112009530986222857675196421645194e4 + (0.463633984130465251191659737388061101e4 + (-0.332721936256036950956433430155944330e4 + (0.159138512544895800179060625875510067e4 + (-0.456561603060421321076101445594873084e3 + 0.595223143611805294572821065472076656e2 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.974557899732034338590563287665154905e-1 + (0.597277377737022117111704644732834295e1 + (0.455960322225042085380238325309127720e1 + (-0.101060720605570133561527618205019708e3 + (0.206745730046969290628688292037028212e3 + (-0.153429078795939661908290345302024639e3 + 0.364873755105945183211337648520495105e2 * e) * e) * e) * e) * e) * e) / (-0.888804340274282793558778086294850759e1 + (0.140599310025370574886286929736396013e3 + (-0.101068609978513833010762892965062687e4 + (0.435792175615259227607636657062006381e4 + (-0.125236265809968543682669903135466482e5 + (0.251859107531585867005913038524161963e5 + (-0.361689607624788978286140157137316155e5 + (0.370907187304372200953327789910448880e5 + (-0.266177549004829560765146744124755465e5 + (0.127310810035916640143248500700408054e5 + (-0.365249282448337056860881156475898468e4 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + (0.106220288475751054899699943786673682e0 + (0.193411169630462961122455721155045916e1 + (-0.153255170898760341190147337075544526e1 + (-0.118700199457783231179748845636173870e2 + (0.183245158947221976887783807803144314e2 - 0.621082740200023790275669859398896310e1 * e) * e) * e) * e) * e) / (-0.888804340274282793558778086294850759e1 + (0.140599310025370574886286929736396013e3 + (-0.101068609978513833010762892965062687e4 + (0.435792175615259227607636657062006381e4 + (-0.125236265809968543682669903135466482e5 + (0.251859107531585867005913038524161964e5 + (-0.361689607624788978286140157137316155e5 + (0.370907187304372200953327789910448881e5 + t70) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH5_ke (double e, double x)
{
  double t43;
  double t65;
  t43 = 0.1e1 / (-0.895481551266453833065880977135438670e1 + (0.164927218951642751117442361559129651e3 + (-0.138023792107674795419277426121461718e4 + (0.692812039701420970608674426242884805e4 + (-0.231757734542115248477844084289715412e5 + (0.542499930160567174593084180295314256e5 + (-0.906746113551297541381498750481818455e5 + (0.108216202979498216255903913314760021e6 + (-0.903744230334546821905146644723877268e5 + (0.502986297254114300465549587041788128e5 + (-0.167906173464267876287681988384154302e5 + 0.254684875933351202659973100306714015e4 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e);
  t65 = 0.814991602986723848511913920981484845e3 * e;
  return((0.236162668533245426718894531007280034e-1 + (0.164874746170575453681608115276329436e1 + (-0.295654601930416155386531368087177682e1 + (-0.463928542808212870083760345005687993e2 + (0.232141663605202557658813518174054496e3 + (-0.424306754222459978304834879125206836e3 + (0.239437501975415489847776235992054000e3 + (0.291115363001752245655329267341565439e3 + (-0.550663356968829204335483822665437190e3 + (0.331719570833132416608410731926445295e3 - 0.717182830718716149239183334758869641e2 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * t43 + ((0.290575226301698339679792971925373376e1 + (-0.471014390265174627259272374365649388e2 + (0.389274846362525180701367850692588050e3 + (-0.190244826450106644720215807284932614e4 + (0.589640006499850594134060659511060056e4 + (-0.121919966416125718688723556701746176e5 + (0.173307700017966665853474776449021033e5 + (-0.170170735368479803604534921802239728e5 + (0.111797229374763521565239658109754526e5 + (-0.445524343441576008447825423098995732e4 + 0.814991602986723848511913920981484844e3 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.286554096405265226581081912683340375e1 + (0.527767100645256803575815556989214883e2 + (-0.441676134744559345341687763588677496e3 + (0.221699852704454710594775816397723138e4 + (-0.741624750534768795129101069727089322e4 + (0.173599977651381495869786937694500562e5 + (-0.290158756336415213242079600154181905e5 + (0.346291849534394292018892522607232066e5 + (-0.289198153707054983009646926311640725e5 + (0.160955615121316576148975867853372201e5 + (-0.537299755085657204120582362829293764e4 + t65) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.877511870468004502088868954595109685e-1 + (0.610840922140590520274803686000597705e1 + (-0.500957611333397254972142508251875995e0 + (-0.157958701368055039614015480267006549e3 + (0.492836844380975379811684939236276074e3 + (-0.496585773256642501115981316628420178e3 + (-0.555541877254928054302305653164243715e2 + (0.387295861891084833076801077869864755e3 - 0.176325639614808519067666130129565399e3 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.286554096405265226581081912683340375e1 + (0.527767100645256803575815556989214883e2 + (-0.441676134744559345341687763588677496e3 + (0.221699852704454710594775816397723139e4 + (-0.741624750534768795129101069727089322e4 + (0.173599977651381495869786937694500562e5 + (-0.290158756336415213242079600154181905e5 + (0.346291849534394292018892522607232066e5 + (-0.289198153707054983009646926311640726e5 + (0.160955615121316576148975867853372201e5 + (-0.537299755085657204120582362829293766e4 + 0.814991602986723848511913920981484846e3 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.288467075982569283235183552588717078e0 + (-0.223379444381799223385531848524573330e-1 + (0.235710209954847005879197470422842790e2 + (-0.156952490123967924913726665632408085e3 + (0.366255150586408560486762273658671812e3 + (-0.373771916032937925121102533669380512e3 + (0.148844246501291885919195503812363471e3 - 0.741535091582477674327906952240420770e1 * e) * e) * e) * e) * e) * e) * e) / (-0.143277048202632613290540956341670187e1 + (0.263883550322628401787907778494607440e2 + (-0.220838067372279672670843881794338748e3 + (0.110849926352227355297387908198861568e4 + (-0.370812375267384397564550534863544661e4 + (0.867999888256907479348934688472502806e4 + (-0.145079378168207606621039800077090952e5 + (0.173145924767197146009446261303616032e5 + (-0.144599076853527491504823463155820363e5 + (0.804778075606582880744879339266861001e4 + (-0.268649877542828602060291181414646882e4 + 0.407495801493361924255956960490742423e3 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.186446033882843396299430449373950979e0 + (0.125551429972995948332340641633502043e2 + (0.125872307835002686525775154829643006e2 + (-0.257529136321390565459465830082557532e3 + (0.532389391527852701542151130614966837e3 + (-0.362756863321714811182481120852494779e3 + 0.549542193273562773433654465219326292e2 * e) * e) * e) * e) * e) * e) * t43 + (0.398718696901387324196827181910604436e-1 + (0.972974954051201078542599771121925996e0 + (-0.140547957928664325582841284845408839e0 + (-0.950978253728914420013161312479198414e1 + (0.131454522776230027873853532038009635e2 - 0.292330691164190609872267852478371066e1 * e) * e) * e) * e) * e) / (-0.286554096405265226581081912683340375e1 + (0.527767100645256803575815556989214882e2 + (-0.441676134744559345341687763588677496e3 + (0.221699852704454710594775816397723138e4 + (-0.741624750534768795129101069727089322e4 + (0.173599977651381495869786937694500562e5 + (-0.290158756336415213242079600154181905e5 + (0.346291849534394292018892522607232066e5 + (-0.289198153707054983009646926311640725e5 + (0.160955615121316576148975867853372200e5 + (-0.537299755085657204120582362829293765e4 + t65) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH6_ke (double e, double x)
{
  return((0.153478012977841977537507513737089107e-1 + (0.123406454597342336477744427746503731e1 + (-0.136112430610706932097316790253426371e1 + (-0.488693722489189572308482112906599999e2 + (0.230013225225315786472457954801533027e3 + (-0.343744056124565194934739644795298613e3 + (-0.144646616882807653343177436530917856e3 + (0.110202785432386470586913390986257527e4 + (-0.142076512117659767752888989544601215e4 + (0.789462399761461481717809435166174739e3 - 0.162692489002542362881232065676180477e3 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.169290558549733768555738598775950540e1 + (0.368302880629264375647229337987464289e2 + (-0.364065386614969738610177149318611420e3 + (0.215838770174622976735801154640884184e4 + (-0.852732423039730100863983295015582084e4 + (0.235732041500587992003506153921739361e5 + (-0.465286960063593912512356066905765165e5 + (0.655721688161506875579932616758052951e5 + (-0.646607410638312564595852580981123637e5 + (0.424908049122324706385449460613410429e5 + (-0.167465322787143684055989441001084889e5 + 0.299886241382136701862882961539499043e4 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.704683644109159811013563444650504600e1 + (-0.118521163365162280595044692976158634e3 + (0.131052357203774143000266338323611085e4 + (-0.814359894861346137666592273245606500e4 + (0.299019514951916188457585784878942559e5 + (-0.706748458119890284938750618234880938e5 + (0.114776538757314768597397572283808388e6 + (-0.132232115014512753549524069936205418e6 + (0.105629027386408687285272023050753649e6 + (-0.524296919035969277713118009119369138e5 + 0.119954496552854680745153184615799618e5 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.677162234198935074222954395103802160e1 + (0.147321152251705750258891735194985715e3 + (-0.145626154645987895444070859727444569e4 + (0.863355080698491906943204618563536738e4 + (-0.341092969215892040345593318006232833e5 + (0.942928166002351968014024615686957443e5 + (-0.186114784025437565004942426762306066e6 + (0.262288675264602750231973046703221181e6 + (-0.258642964255325025838341032392449455e6 + (0.169963219648929882554179784245364171e6 + (-0.669861291148574736223957764004339554e5 + 0.119954496552854680745153184615799617e5 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.252981855979077592702561563961460148e0 + (0.200394650028138810332715783214712416e2 + (0.100559574265679558529219037173299230e2 + (-0.688815201519593454907550697822600730e3 + (0.206484916261297658542123490073580682e4 + (-0.129045800765909026274868150750804250e4 + (-0.284071665383126591870682132737318338e4 + (0.470845029313453033608849053282256405e4 - 0.200139422904954539821378295812934615e4 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.338581117099467537111477197551901080e1 + (0.736605761258528751294458675974928575e2 + (-0.728130773229939477220354298637222843e3 + (0.431677540349245953471602309281768371e4 + (-0.170546484607946020172796659003116417e5 + (0.471464083001175984007012307843478723e5 + (-0.930573920127187825024712133811530331e5 + (0.131144337632301375115986523351610591e6 + (-0.129321482127662512919170516196224727e6 + (0.849816098244649412770898921226820854e5 + (-0.334930645574287368111978882002169775e5 + 0.599772482764273403725765923078998087e4 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.322198331285668814570374587910981466e0 + (0.348802524339954627247755593470175658e1 + (0.390308072033573728948974816766806572e2 + (-0.366344965354633854184308916400436526e3 + (0.931363259865723182124176195537863494e3 + (-0.882418627668072159040489560622910938e3 + (0.140119348383678337603033798079253370e3 + 0.145635245058581816149393580707988410e3 * e) * e) * e) * e) * e) * e) * e) / (-0.135432446839787014844590879020760433e1 + (0.294642304503411500517783470389971431e2 + (-0.291252309291975790888141719454889137e3 + (0.172671016139698381388640923712707348e4 + (-0.682185938431784080691186636012465668e4 + (0.188585633200470393602804923137391489e5 + (-0.372229568050875130009884853524612132e5 + (0.524577350529205500463946093406442362e5 + (-0.517285928510650051676682064784898911e5 + (0.339926439297859765108359568490728343e5 + (-0.133972258229714947244791552800867911e5 + 0.239908993105709361490306369231599234e4 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.195008664682676659101375799856287462e0 + (0.146400749708917149042058369002730009e2 + (0.189938022572644821292530633772635778e2 + (-0.378549352588069032070694373092134331e3 + (0.792597567561657451621028762170501250e3 + (-0.440528539745773584211350158121213051e3 - 0.479365498516003204658938281019668290e2 * e) * e) * e) * e) * e) * e) / (-0.541729787359148059378363516083041728e1 + (0.117856921801364600207113388155988572e3 + (-0.116500923716790316355256687781955655e4 + (0.690684064558793525554563694850829390e4 + (-0.272874375372713632276474654404986267e5 + (0.754342532801881574411219692549565957e5 + (-0.148891827220350052003953941409844853e6 + (0.209830940211682200185578437362576944e6 + (-0.206914371404260020670672825913959564e6 + (0.135970575719143906043343827396291337e6 + (-0.535889032918859788979166211203471643e5 + 0.959635972422837445961225476926396940e4 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + (0.445194084890815450207551305123013718e-1 + (0.141748383293319102310646791866242444e1 + (0.574637677053335262607410329095904180e0 + (-0.202931195587572394983838358953981488e2 + (0.261195973259711714627195910710015985e2 - 0.716845134398580674579335753947701112e-2 * e) * e) * e) * e) * e) / (-0.270864893679574029689181758041520864e1 + (0.589284609006823001035566940779942862e2 + (-0.582504618583951581776283438909778272e3 + (0.345342032279396762777281847425414696e4 + (-0.136437187686356816138237327202493133e5 + (0.377171266400940787205609846274782978e5 + (-0.744459136101750260019769707049224264e5 + (0.104915470105841100092789218681288472e6 + (-0.103457185702130010335336412956979782e6 + (0.679852878595719530216719136981456686e5 + (-0.267944516459429894489583105601735820e5 + 0.479817986211418722980612738463198468e4 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH7_ke (double e, double x)
{
  return((0.155546028446408459927642621589078743e0 + (0.145682360347026030238724636047681344e2 + (-0.506612115890619938573206698159708153e1 + (-0.815397832829250261018499145319004850e3 + (0.376944686722640072789143606732994481e4 + (-0.422121450803828130132552218106559194e4 + (-0.111208592538154900524274225951579661e5 + (0.393350260419860893128488221053496316e5 + (-0.480448207532162232076892860887399731e5 + (0.261276395153653294756227703552654230e5 - 0.493431728721111804400891741742807006e4 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.586897430800638442694352252635709195e1 + (0.152398189219754859721423953366428134e3 + (-0.179796479966669380007965232336891068e4 + (0.127215552991382934847636587771356729e5 + (-0.599810845312984654851525374514176838e5 + (0.197875826504555746823887899965073091e6 + (-0.466068089000285081474902976074793555e6 + (0.783761902122814386257600775670976773e6 + (-0.922195756109611167311298593358177646e6 + (0.723064342232678819201453645730557913e6 + (-0.340007193333370938171357941246003136e6 + 0.726411388656647309575079095955218035e5 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.517690985773698750547525052073038859e1 + (-0.725391651075638785805789653517026664e2 + (0.134277642737938858241422123697336012e4 + (-0.111775218914260906628036595533805905e5 + (0.477741784181343223473000636100512199e5 + (-0.124200798900295692162814257737224197e6 + (0.223858817633756819156376654349754640e6 + (-0.305508952269283749351214091426452010e6 + (0.309649304499693823211461637735169993e6 + (-0.199325281329284210349051780904007146e6 + 0.581129110925317847660063276764174434e5 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.469517944640510754155481802108567355e1 + (0.121918551375803887777139162693142508e3 + (-0.143837183973335504006372185869512855e4 + (0.101772442393106347878109270217085383e5 + (-0.479848676250387723881220299611341471e5 + (0.158300661203644597459110319972058473e6 + (-0.372854471200228065179922380859834844e6 + (0.627009521698251509006080620536781419e6 + (-0.737756604887688933849038874686542118e6 + (0.578451473786143055361162916584446331e6 + (-0.272005754666696750537086352996802508e6 + 0.581129110925317847660063276764174432e5 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.191138194057247302205766651460672639e0 + (0.174366755601796561216802380906345691e2 + (0.201530084419686020448604655885712189e2 + (-0.816771731988695423192366659456325450e3 + (0.243114215747589271857095356688735066e4 + (-0.330997548525437168586601453455177294e3 + (-0.812345930123825608576994651546585391e4 + (0.121285822184865039775310410247413894e5 - 0.543578612263150748509608228512459566e4 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.117379486160127688538870450527141839e1 + (0.304796378439509719442847906732856268e2 + (-0.359592959933338760015930464673782136e3 + (0.254431105982765869695273175542713458e4 + (-0.119962169062596930970305074902835367e5 + (0.395751653009111493647775799930146182e5 + (-0.932136178000570162949805952149587109e5 + (0.156752380424562877251520155134195355e6 + (-0.184439151221922233462259718671635530e6 + (0.144612868446535763840290729146111583e6 + (-0.680014386666741876342715882492006273e5 + 0.145282277731329461915015819191043608e5 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.705703548620220807444349152662010488e0 + (0.193124190095022358637761784110390762e2 + (0.127617157614114287406420455368748588e3 + (-0.160021992713032797223131483128483452e4 + (0.435921689732760787519960624133528878e4 + (-0.343639639066294310067278144601321144e4 + (-0.165548305445536843176285169093371170e4 + 0.240393786766394550918562387463265892e4 * e) * e) * e) * e) * e) * e) * e) / (-0.234758972320255377077740901054283678e1 + (0.609592756879019438885695813465712537e2 + (-0.719185919866677520031860929347564273e3 + (0.508862211965531739390546351085426918e4 + (-0.239924338125193861940610149805670736e5 + (0.791503306018222987295551599860292366e5 + (-0.186427235600114032589961190429917423e6 + (0.313504760849125754503040310268390711e6 + (-0.368878302443844466924519437343271059e6 + (0.289225736893071527680581458292223166e6 + (-0.136002877333348375268543176498401254e6 + 0.290564555462658923830031638382087215e5 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.273831877480181361452578837916938181e0 + (0.232146451263068225774492291723297632e2 + (0.384215603920067400607886781067390906e2 + (-0.783599674590867519896800069731902369e3 + (0.167329456564944265106519098957946450e4 + (-0.564001727367207761701977404179675942e3 - 0.638954061566046938256730766017287862e3 * e) * e) * e) * e) * e) * e) / (-0.469517944640510754155481802108567356e1 + (0.121918551375803887777139162693142508e3 + (-0.143837183973335504006372185869512855e4 + (0.101772442393106347878109270217085383e5 + (-0.479848676250387723881220299611341471e5 + (0.158300661203644597459110319972058473e6 + (-0.372854471200228065179922380859834843e6 + (0.627009521698251509006080620536781418e6 + (-0.737756604887688933849038874686542118e6 + (0.578451473786143055361162916584446333e6 + (-0.272005754666696750537086352996802509e6 + 0.581129110925317847660063276764174429e5 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + (0.460536530917070265529357853653789524e-1 + (0.187508966364566409322478550463667715e1 + (0.172858264068217108028129532780125519e1 + (-0.385293248882752299009236933991819420e2 + (0.468435104533223922990966693630074880e2 + 0.189563170902210245945738048720130315e2 * e) * e) * e) * e) * e) / (-0.234758972320255377077740901054283678e1 + (0.609592756879019438885695813465712535e2 + (-0.719185919866677520031860929347564270e3 + (0.508862211965531739390546351085426915e4 + (-0.239924338125193861940610149805670736e5 + (0.791503306018222987295551599860292365e5 + (-0.186427235600114032589961190429917422e6 + (0.313504760849125754503040310268390710e6 + (-0.368878302443844466924519437343271059e6 + (0.289225736893071527680581458292223166e6 + (-0.136002877333348375268543176498401254e6 + 0.290564555462658923830031638382087216e5 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH8_ke (double e, double x)
{
  double t21;
  double t45;
  t21 = 0.145998077721168370003075392813962762e6 * e;
  t45 = 0.583992310884673480012301571255851048e6 * e;
  return((0.107067339777942492686647998536984166e0 + (0.117818676812905157029615487861863290e2 + (0.565664609045508702035190539549375232e1 + (-0.935154943871953142804563842472371012e3 + (0.441230389524638690350067683989774412e4 + (-0.312110772019384064648591302951920842e4 + (-0.272410131471232869203252479618553352e5 + (0.854728707156820843660441893155845225e5 + (-0.105317368114897126442556707167184980e6 + (0.553481709190742675880568538862991638e5 - 0.775032292301196365812663819831677292e4 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.155700410996199628000198171180416332e1 + (0.486096236502302827278561680826682183e2 + (-0.689485894651875263710107966789738601e3 + (0.586506709830107829399854060755960541e4 + (-0.332447035850142749102977831007523102e5 + (0.131844813261058238343765191702166827e6 + (-0.373309112980792260450822957988141972e6 + (0.754638619083356449156627985996773262e6 + (-0.106733406347442028726864226255442207e7 + (0.100591959461960772513504983632793257e7 + (-0.568551987056128305240893692642518590e6 + t21) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.768617808453676094801451816361069664e1 + (-0.300353230649687287995754690572875448e2 + (0.277213544083312949762708621927009300e4 + (-0.324079760424995073199553201928541622e5 + (0.156094462270401929664536085104582889e6 + (-0.418879489361083898168629459906398073e6 + (0.787593017711809625797650284622160479e6 + (-0.129667375454502715713929110941626913e7 + (0.179283701934299133762577868194734189e7 + (-0.156149101783699871940020684816881141e7 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.622801643984798512000792684721665328e1 + (0.194438494600921130911424672330672873e3 + (-0.275794357860750105484043186715895441e4 + (0.234602683932043131759941624302384217e5 + (-0.132978814340057099641191132403009241e6 + (0.527379253044232953375060766808667306e6 + (-0.149323645192316904180329183195256788e7 + (0.301855447633342579662651194398709304e7 + (-0.426933625389768114907456905021768827e7 + (0.402367837847843090054019934531173028e7 + (-0.227420794822451322096357477057007437e7 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.254303942135814421823282439688735710e1 + (0.269751599757532397561311866674771065e3 + (0.508967736384118953137740446732799740e3 + (-0.175331259555596120392254280654364526e5 + (0.537503906477928369561571986977440860e5 + (0.235341662481102298128692407636799379e5 + (-0.334133069799657270649381677721278875e6 + (0.518600959928545122372349305911827405e6 - 0.255114378269303055955060483812687805e6 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.778502054980998140000990855902081660e1 + (0.243048118251151413639280840413341091e3 + (-0.344742947325937631855053983394869301e4 + (0.293253354915053914699927030377980272e5 + (-0.166223517925071374551488915503761551e6 + (0.659224066305291191718825958510834133e6 + (-0.186654556490396130225411478994070985e7 + (0.377319309541678224578313992998386630e7 + (-0.533667031737210143634321131277211034e7 + (0.502959797309803862567524918163966285e7 + (-0.284275993528064152620446846321259296e7 + 0.729990388605841850015376964069813810e6 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.626973598165496885720579362242684100e0 + (0.305771975850898048636766805388124280e2 + (0.168401727877412911691496315602384856e3 + (-0.269970700767566149977293127111409892e4 + (0.787355874040970265440731797451069710e4 + (-0.420054404251636673875640614440067755e4 + (-0.104375826477087109717560968515997091e5 + 0.106090735369937984692071835831205052e5 * e) * e) * e) * e) * e) * e) * e) / (-0.155700410996199628000198171180416332e1 + (0.486096236502302827278561680826682182e2 + (-0.689485894651875263710107966789738602e3 + (0.586506709830107829399854060755960543e4 + (-0.332447035850142749102977831007523103e5 + (0.131844813261058238343765191702166827e6 + (-0.373309112980792260450822957988141972e6 + (0.754638619083356449156627985996773262e6 + (-0.106733406347442028726864226255442207e7 + (0.100591959461960772513504983632793257e7 + (-0.568551987056128305240893692642518591e6 + t21) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.112275208507233499894546354147513465e0 + (0.108690711358213930195708281786764409e2 + (0.226701102667643887464822423468044339e2 + (-0.493279522399880560802639973992837238e3 + (0.108746296945027046818513779592546292e4 + (0.460904536887988741007144998516629223e1 - 0.104423790511477531785967772921699588e4 * e) * e) * e) * e) * e) * e) / (-0.124560328796959702400158536944333066e1 + (0.388876989201842261822849344661345746e2 + (-0.551588715721500210968086373431790882e3 + (0.469205367864086263519883248604768433e4 + (-0.265957628680114199282382264806018482e5 + (0.105475850608846590675012153361733462e6 + (-0.298647290384633808360658366390513577e6 + (0.603710895266685159325302388797418610e6 + (-0.853867250779536229814913810043537654e6 + (0.804735675695686180108039869062346054e6 + (-0.454841589644902644192714954114014874e6 + 0.116798462176934696002460314251170210e6 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + (0.734474194203862024677653290847712242e-1 + (0.376930753994555053450563163025082618e1 + (0.546064348148726259964162972541136292e1 + (-0.111203422295173034813695700757150697e3 + (0.129408243861641562351230330230144864e3 + 0.141500381882201839685686857596285307e3 * e) * e) * e) * e) * e) / (-0.311400821992399256000396342360832664e1 + (0.972192473004605654557123361653364365e2 + (-0.137897178930375052742021593357947720e4 + (0.117301341966021565879970812151192108e5 + (-0.664894071700285498205955662015046205e5 + (0.263689626522116476687530383404333653e6 + (-0.746618225961584520901645915976283940e6 + (0.150927723816671289831325597199354652e7 + (-0.213466812694884057453728452510884414e7 + (0.201183918923921545027009967265586514e7 + (-0.113710397411225661048178738528503718e7 + 0.291996155442336740006150785627925524e6 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH9_ke (double e, double x)
{
  double t45;
  t45 = 0.734729819723249193738136649936752177e7 * e;
  return((0.497653659611058795057950889591942790e0 + (0.647644909665710035767893139667708595e2 + (0.904670881981218475088416658948423460e2 + (-0.731199108484487255939992966839706140e4 + (0.363670601318193810186575500321347112e5 + (-0.921542991761723031527863780180438345e4 + (-0.388044056746531823630086058704095858e6 + (0.122110528862101176502820792562751030e7 + (-0.154881906325360079348218288276243810e7 + (0.743491678335004385867911726157719750e6 - 0.504216015249141405030541192928285565e4 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.305579163092106801775997068758416615e1 + (0.115283282760207073373812776058682685e3 + (-0.197592408919881575406658632289926549e4 + (0.203099721066072844233948913278183408e5 + (-0.139104423231128052200819352897987910e6 + (0.666583103484241004151683239914466114e6 + (-0.228046414098876725553724173354991618e7 + (0.556990385165342720612796579580809480e7 + (-0.951817430322794913221869778468617598e7 + (0.108380648393933299589130228596676967e8 + (-0.740089799953474394450289174610951654e7 + 0.229603068663515373043167703105235055e7 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.146150595655815848476920684087988527e2 + (0.258756576355917195336742679765180594e3 + (0.730230675219077574058958624217296963e4 + (-0.122604814586045981261681025978591653e6 + (0.643709607824802126613054559103434502e6 + (-0.162527689749808057032039944149102616e7 + (0.271673171360583392587084160321002022e7 + (-0.568714128199164490015111326418700868e7 + (0.124667669188031822008693106398278683e8 + (-0.153478835328666029209530388515914581e8 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.977853321894741765683190620026933168e1 + (0.368906504832662634796200883387784593e3 + (-0.632295708543621041301307623327764961e4 + (0.649919107411433101548636522490186904e5 + (-0.445134154339609767042621929273561314e6 + (0.213306593114957121328538636772629157e7 + (-0.729748525116405521771917354735973178e7 + (0.178236923252909670596094905465859033e8 + (-0.304581577703294372230998329109957632e8 + (0.346818074860586558685216731509366296e8 + (-0.236828735985111806224092535875504529e8 + 0.734729819723249193738136649936752178e7 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.300792689038893072813404337017645058e1 + (0.373851389147975303599384774436959260e3 + (0.101259333113455354567218487415718652e4 + (-0.341529764844741987882162009908087630e5 + (0.111651705868614055855779039086840583e6 + (0.119912731216990831770923427520939409e6 + (-0.114529076108327761028098379717456027e7 + (0.194266881399629047762493162999868216e7 - 0.107002340282180526777069633575863481e7 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.488926660947370882841595310013466584e1 + (0.184453252416331317398100441693892297e3 + (-0.316147854271810520650653811663882480e4 + (0.324959553705716550774318261245093453e5 + (-0.222567077169804883521310964636780658e6 + (0.106653296557478560664269318386314579e7 + (-0.364874262558202760885958677367986591e7 + (0.891184616264548352980474527329295170e7 + (-0.152290788851647186115499164554978817e8 + (0.173409037430293279342608365754683148e8 + (-0.118414367992555903112046267937752265e8 + 0.367364909861624596869068324968376088e7 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.137527822193060038256629705474106068e1 + (0.101648248230921380290317863734136715e3 + (0.543064998157409508661298860612574640e3 + (-0.108166073520610626620210818446926818e5 + (0.340638917852190177161508700981113485e5 + (-0.595537572111949134917578260151941440e4 + (-0.981997466892774446874241344734511445e5 + 0.972382884989300761214643235786061980e5 * e) * e) * e) * e) * e) * e) * e) / (-0.244463330473685441420797655006733293e1 + (0.922266262081656586990502208469461483e2 + (-0.158073927135905260325326905831941240e4 + (0.162479776852858275387159130622546726e5 + (-0.111283538584902441760655482318390329e6 + (0.533266482787392803321346591931572893e6 + (-0.182437131279101380442979338683993294e7 + (0.445592308132274176490237263664647584e7 + (-0.761453944258235930577495822774894078e7 + (0.867045187151466396713041828773415737e7 + (-0.592071839962779515560231339688761322e7 + 0.183682454930812298434534162484188044e7 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.164345429647639197288693838171644781e0 + (0.183427412416785587096465024321153459e2 + (0.477481097007990226726615679475767344e2 + (-0.114616167526893526403171430032610213e4 + (0.265191772305865593676692216562328545e4 + (0.131560832295356200494145988890916119e4 - 0.513630813128658550049547819760463512e4 * e) * e) * e) * e) * e) * e) / (-0.122231665236842720710398827503366646e1 + (0.461133131040828293495251104234730741e2 + (-0.790369635679526301626634529159706201e3 + (0.812398884264291376935795653112733631e4 + (-0.556417692924512208803277411591951643e5 + (0.266633241393696401660673295965786446e6 + (-0.912185656395506902214896693419966470e6 + (0.222796154066137088245118631832323792e7 + (-0.380726972129117965288747911387447040e7 + (0.433522593575733198356520914386707869e7 + (-0.296035919981389757780115669844380661e7 + 0.918412274654061492172670812420940222e6 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + (0.278702330451235054871151485830326401e0 + (0.178477047332117705291185260126373412e2 + (0.360410480621878629403647335146209458e2 + (-0.761059807878326477895271672689767931e3 + (0.863964569368907462240932506056916998e3 + 0.188825865298228041992758390024914746e4 * e) * e) * e) * e) * e) / (-0.977853321894741765683190620026933168e1 + (0.368906504832662634796200883387784593e3 + (-0.632295708543621041301307623327764960e4 + (0.649919107411433101548636522490186904e5 + (-0.445134154339609767042621929273561315e6 + (0.213306593114957121328538636772629157e7 + (-0.729748525116405521771917354735973178e7 + (0.178236923252909670596094905465859033e8 + (-0.304581577703294372230998329109957631e8 + (0.346818074860586558685216731509366295e8 + (-0.236828735985111806224092535875504529e8 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH10_ke (double e, double x)
{
  double t45;
  t45 = 0.192619730597892657760975987359546318e8 * e;
  return((0.172651934939717494214187831123662995e1 + (0.267043089683169704159719510072942820e3 + (0.647932074615349607081163216401512752e3 + (-0.430123159499561151975325312100518445e5 + (0.231495218264250570292195165438156222e6 + (0.584772200638899066651303953095016730e5 + (-0.393512353835681848780227266962403738e7 + (0.130542401739695438003145203918471046e8 + (-0.171802877389866092134303965699611364e8 + (0.663673787221178896795464040760990562e7 + 0.253468825218803087430266298436627602e7 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.481020702155677316384422611977636140e1 + (0.220049804219064175095757530521559544e3 + (-0.457333548715236781387244374403402785e4 + (0.569998621460646922726340816784224570e5 + (-0.473369855331307222475444611335560129e6 + (0.275044290183259814780389751345066331e7 + (-0.114091617383857547278434891155654529e8 + (0.337873254271349083247251699831320152e8 + (-0.700049671861668748632773546295054187e8 + (0.966472457710174637690695908967068739e8 + (-0.800162378357801012982563523550694130e8 + 0.300968329059207277751524980249291122e8 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.610588227844633221453770057074656712e1 + (0.320222017074818422390526544341932194e3 + (0.423914580789996201526233694012383136e4 + (-0.100692014486632116728956170783123886e6 + (0.565214631595787043927677277359629676e6 + (-0.117109815508999777279813547952733586e7 + (0.738833855517516742186583384707435536e6 + (-0.303357970840312765015168682793856640e7 + (0.173622766036043637190302081470121856e8 + (-0.316527173115101130336032512677108608e8 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.307853249379633482486030471665687130e1 + (0.140831874700201072061284819533798108e3 + (-0.292693471177751540087836399618177782e4 + (0.364799117734814030544858122741903726e5 + (-0.302956707412036622384284551254758484e6 + (0.176028345717286281459449440860842452e7 + (-0.730186351256688302581983303396188984e7 + (0.216238882733663413278241087892044897e8 + (-0.448031789991467999124975069628834678e8 + (0.618542372934511768122045381738923993e8 + (-0.512103922148992648308840655072444245e8 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.169739001524367642476795097251206293e1 + (0.248699216043221302464621118597798684e3 + (0.905572388117118704369811961747639188e3 + (-0.322263150403863079677155918853913388e5 + (0.115721805000558659057904594208600818e6 + (0.203625823566624908294331315197011370e6 + (-0.183200734857918794275894218197478690e7 + (0.347985927487623563824262743373681176e7 - 0.217218298135909974905103091458507844e7 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.153926624689816741243015235832843565e1 + (0.704159373501005360306424097668990540e2 + (-0.146346735588875770043918199809088891e4 + (0.182399558867407015272429061370951863e5 + (-0.151478353706018311192142275627379242e6 + (0.880141728586431407297247204304212258e6 + (-0.365093175628344151290991651698094492e7 + (0.108119441366831706639120543946022448e8 + (-0.224015894995733999562487534814417339e8 + (0.309271186467255884061022690869461997e8 + (-0.256051961074496324154420327536222122e8 + 0.963098652989463288804879936797731586e7 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.124031592429423283822340182980987781e1 + (0.126757902706957482146614896306462314e3 + (0.713014290801751808711812856158032755e3 + (-0.173686189496735443326087146550006960e5 + (0.598445576083996447918930323310359840e5 + (0.176823337220629215328473785682269076e5 + (-0.318508770726688250425310436514582387e6 + 0.334887614092401198801948821390382064e6 * e) * e) * e) * e) * e) * e) * e) / (-0.153926624689816741243015235832843565e1 + (0.704159373501005360306424097668990540e2 + (-0.146346735588875770043918199809088891e4 + (0.182399558867407015272429061370951862e5 + (-0.151478353706018311192142275627379242e6 + (0.880141728586431407297247204304212259e6 + (-0.365093175628344151290991651698094492e7 + (0.108119441366831706639120543946022448e8 + (-0.224015894995733999562487534814417339e8 + (0.309271186467255884061022690869461996e8 + (-0.256051961074496324154420327536222122e8 + 0.963098652989463288804879936797731585e7 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.375468390898733421697169689724499322e0 + (0.487121047496476527148571775152779234e2 + (0.157087996402461064289192530382991501e3 + (-0.426873699848584994688179191933596469e4 + (0.105629452302978374986594057332060029e5 + (0.123013937730004935548379845875812851e5 - 0.370625947144192933525107940828507750e5 * e) * e) * e) * e) * e) * e) / (-0.192408280862270926553769044791054456e1 + (0.880199216876256700383030122086238177e2 + (-0.182933419486094712554897749761361114e4 + (0.227999448584258769090536326713689829e5 + (-0.189347942132522888990177844534224052e6 + (0.110017716073303925912155900538026532e7 + (-0.456366469535430189113739564622618116e7 + (0.135149301708539633298900679932528061e8 + (-0.280019868744667499453109418518021674e8 + (0.386588983084069855076278363586827497e8 + (-0.320064951343120405193025409420277652e8 + 0.120387331623682911100609992099716449e8 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + (0.531979934842384970786744867487061802e-1 + (0.422171262159649169385110964701079702e1 + (0.112437282795614803068501763947298260e2 + (-0.262176573368817268929444558413409056e3 + (0.297924233339635548077093396472938520e3 + 0.113304407761041148634799342321796833e4 * e) * e) * e) * e) * e) / (-0.153926624689816741243015235832843565e1 + (0.704159373501005360306424097668990539e2 + (-0.146346735588875770043918199809088891e4 + (0.182399558867407015272429061370951862e5 + (-0.151478353706018311192142275627379242e6 + (0.880141728586431407297247204304212258e6 + (-0.365093175628344151290991651698094492e7 + (0.108119441366831706639120543946022448e8 + (-0.224015894995733999562487534814417339e8 + (0.309271186467255884061022690869461997e8 + (-0.256051961074496324154420327536222122e8 + 0.963098652989463288804879936797731588e7 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH11_ke (double e, double x)
{
  double t43;
  t43 = 0.1e1 / (-0.477671233845080182496034318614960642e1 + (0.265597948213750075247633651907678333e3 + (-0.670920028760996820065553192642459772e4 + (0.101634467434638956722420204989673926e6 + (-0.102587125845747815233092123401938679e7 + (0.724461785716055644732272077844989935e7 + (-0.365243664153021995249213283037482678e8 + (0.131460230066790107146028520168918024e9 + (-0.331037012291467579295395515766923002e9 + (0.555443387408659450483988216926133818e9 + (-0.558890281614305096097500756172312330e9 + 0.255483478429220979183599378288418299e9 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e);
  return((0.356562455462900006530740214167110050e1 + (0.657916243193569916212745111067637222e3 + (0.236816896744465315082598608460600054e4 + (-0.151615425742722408676742728091691211e6 + (0.901656440900449375497900875727934230e6 + (0.745898982934737979226668686101529776e6 + (-0.234322910156644780459249874701996128e8 + (0.842827175769805784667785516162286096e8 + (-0.115625828421671622888780556311680571e9 + (0.253726608704388603376546723980524912e8 + 0.513749759607659967854133379978189844e8 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * t43 + ((0.273412128752897243397490070213450984e2 + (0.267827191118149452195300872682221189e4 + (0.266823564033180767641709588784183388e5 + (-0.859209936109265960405642125828972856e6 + (0.515348757481306284444620529584070506e7 + (-0.684986620708101051959517258244945744e7 + (-0.242737631662584835617291546332009548e8 + (0.337219913505781065476923229580313475e8 + (0.233593852180909452267738990183046081e9 + (-0.666081557115484257181543596421314788e9 + 0.510966956858441958367198756576836596e9 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.955342467690160364992068637229921286e1 + (0.531195896427500150495267303815356664e3 + (-0.134184005752199364013110638528491954e5 + (0.203268934869277913444840409979347850e6 + (-0.205174251691495630466184246803877357e7 + (0.144892357143211128946454415568997987e8 + (-0.730487328306043990498426566074965355e8 + (0.262920460133580214292057040337836048e9 + (-0.662074024582935158590791031533846002e9 + (0.111088677481731890096797643385226764e10 + (-0.111778056322861019219500151234462466e10 + 0.510966956858441958367198756576836595e9 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.907398821295222923237870972988541340e1 + (0.157488629104050812685746926365818502e4 + (0.742025297374404084901570526296370720e4 + (-0.291390001296994254209574160241377102e6 + (0.117642741737421296210338528185042786e7 + (0.293869672235216741515958166553435372e7 + (-0.276301439039777740467194906967009056e8 + (0.596063253104450890382462818463693216e8 - 0.425164178054235490822413202400312104e8 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.477671233845080182496034318614960641e1 + (0.265597948213750075247633651907678331e3 + (-0.670920028760996820065553192642459770e4 + (0.101634467434638956722420204989673926e6 + (-0.102587125845747815233092123401938679e7 + (0.724461785716055644732272077844989935e7 + (-0.365243664153021995249213283037482678e8 + (0.131460230066790107146028520168918024e9 + (-0.331037012291467579295395515766923002e9 + (0.555443387408659450483988216926133820e9 + (-0.558890281614305096097500756172312331e9 + 0.255483478429220979183599378288418300e9 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.278793438152513410859337556211923969e1 + (0.373090215626802236292516756018624122e3 + (0.231710852575320534464632078249995116e4 + (-0.685206574813675547210084266614417888e5 + (0.262042242342310085836082030835400579e6 + (0.232656236479883935688368047945660624e6 + (-0.236557881499245707824611619762665380e7 + 0.273948019894826241078974807297250068e7 * e) * e) * e) * e) * e) * e) * e) / (-0.238835616922540091248017159307480320e1 + (0.132798974106875037623816825953839166e3 + (-0.335460014380498410032776596321229885e4 + (0.508172337173194783612101024948369627e5 + (-0.512935629228739076165460617009693392e6 + (0.362230892858027822366136038922494967e7 + (-0.182621832076510997624606641518741340e8 + (0.657301150333950535730142600844590121e8 + (-0.165518506145733789647697757883461501e9 + (0.277721693704329725241994108463066910e9 + (-0.279445140807152548048750378086156166e9 + 0.127741739214610489591799689144209149e9 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.132394239041289669874524753596788678e1 + (0.201053240395800736549369160013558737e3 + (0.798834449170335826556074241950469440e3 + (-0.250556843224472141934452813640593448e5 + (0.675896211507059291539028028604820845e5 + (0.137610531775091662812438809715801062e6 - 0.402315552225567697287444433091789019e6 * e) * e) * e) * e) * e) * e) * t43 + (0.200654545692588401870144995395488879e0 + (0.196402175881469089036754531741154645e2 + (0.669717675468963838363096635540500125e2 + (-0.178804653421734772502658067204111312e4 + (0.209721930488505043268656907471264740e4 + 0.126895633037591896177504006369614635e5 * e) * e) * e) * e) * e) / (-0.477671233845080182496034318614960642e1 + (0.265597948213750075247633651907678332e3 + (-0.670920028760996820065553192642459770e4 + (0.101634467434638956722420204989673926e6 + (-0.102587125845747815233092123401938679e7 + (0.724461785716055644732272077844989935e7 + (-0.365243664153021995249213283037482678e8 + (0.131460230066790107146028520168918024e9 + (-0.331037012291467579295395515766923002e9 + (0.555443387408659450483988216926133820e9 + (-0.558890281614305096097500756172312330e9 + 0.255483478429220979183599378288418298e9 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH12_ke (double e, double x)
{
  double t105;
  double t25;
  double t43;
  t25 = (0.128292952393022028051688483115552415e10 + (-0.157142538348665746049186034641850761e10 + 0.874442293221495371483915601033422726e9 * e) * e) * e;
  t43 = 0.1e1 / (-0.187895568271949507364746141208188607e1 + (0.127187326172428550555575106191962049e3 + (-0.391127655037883850891444158690092364e4 + (0.721297447038740887884294169354767016e5 + (-0.886316457638283267864458871156732104e6 + (0.761957869465466443836082093478020335e7 + (-0.467643218825825740643618940194817176e8 + (0.204898959658861649188955075584321601e9 + (-0.628105028695609148384585504363505276e9 + t25) * e) * e) * e) * e) * e) * e) * e) * e);
  t105 = 0.874442293221495371483915601033422725e9 * e;
  return((0.278083713962880705116722856720669141e1 + (0.613865687369327857420399776954644881e3 + (0.304156336246446431545723980270719586e4 + (-0.202957263748981338848245007909118335e6 + (0.135584661949349610349653339375146298e7 + (0.202348110089398738569988533548165210e7 + (-0.526755723254774834260243307316936558e8 + (0.209428047937610077213216864816860092e9 + (-0.300988519657985703768286899159376939e9 + (-0.149670426895373227846322400626809290e8 + 0.299000647729040805801068782702804220e9 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * t43 + ((0.103031538850822969535094228645861562e2 + (0.155058819392057529361713212630709850e4 + (0.143419610508137001066396054785921165e5 + (-0.597238011352070691115776630620040578e6 + (0.387783554436238298672905432113484809e7 + (-0.132249370538043202754096688531141841e7 + (-0.580332629097746879282787343547975695e8 + (0.132271974947489663635779925913451084e9 + (0.236762013303979053969513456123399906e9 + (-0.113925583672227838125502134085599111e10 + 0.109305286652686921435489450129177841e10 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.234869460339936884205932676510235759e1 + (0.158984157715535688194468882739952561e3 + (-0.488909568797354813614305198362615456e4 + (0.901621808798426109855367711693458770e5 + (-0.110789557204785408483057358894591513e7 + (0.952447336831833054795102616847525420e7 + (-0.584554023532282175804523675243521470e8 + (0.256123699573577061486193844480402001e9 + (-0.785131285869511435480731880454381596e9 + (0.160366190491277535064610603894440519e10 + (-0.196428172935832182561482543302313451e10 + 0.109305286652686921435489450129177840e10 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.595218394898397573389884161407305512e1 + (0.122845204703791099212811515439502740e4 + (0.731330502803219345686205730436668009e4 + (-0.326159794169039813320261037105646120e6 + (0.150647848398153069094910645740195405e7 + (0.498291400840007649314273778512442300e7 + (-0.512930106533080711341988147214438292e8 + (0.126817212037367472482372548106829032e9 - 0.103997063794977217831365053479132627e9 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.187895568271949507364746141208188607e1 + (0.127187326172428550555575106191962049e3 + (-0.391127655037883850891444158690092364e4 + (0.721297447038740887884294169354767016e5 + (-0.886316457638283267864458871156732103e6 + (0.761957869465466443836082093478020334e7 + (-0.467643218825825740643618940194817176e8 + (0.204898959658861649188955075584321601e9 + (-0.628105028695609148384585504363505276e9 + (0.128292952393022028051688483115552415e10 + (-0.157142538348665746049186034641850760e10 + t105) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.318443584795598780942797073301803875e1 + (0.540089943313051456687802790041302931e3 + (0.381179858225385792920372518449379042e4 + (-0.136459096893803699318723572319006474e6 + (0.587131762059727270266496814547890346e6 + (0.950580176429704467556225272551519781e6 + (-0.854408089308915398485238813319826038e7 + 0.111127147294179476211442054793975746e8 * e) * e) * e) * e) * e) * e) * e) / (-0.187895568271949507364746141208188607e1 + (0.127187326172428550555575106191962048e3 + (-0.391127655037883850891444158690092363e4 + (0.721297447038740887884294169354767016e5 + (-0.886316457638283267864458871156732104e6 + (0.761957869465466443836082093478020335e7 + (-0.467643218825825740643618940194817176e8 + (0.204898959658861649188955075584321601e9 + (-0.628105028695609148384585504363505278e9 + t25) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.726953893704559480652606763258868591e0 + (0.129981858491311999092853730398241069e3 + (0.633816793422105459067014562423221509e3 + (-0.232767191949496509712722559978205919e5 + (0.696845473955691832972418799578450755e5 + (0.218882592838109350945678817767333158e6 - 0.673199183084917723106101221486067662e6 * e) * e) * e) * e) * e) * e) * t43 + (0.960877722234696662095200063713347876e-1 + (0.115633000693358903533425943470072820e2 + (0.496193723871242322568226962467744792e2 + (-0.155120934442713223182909156128461804e4 + (0.193955416600509296168737029825469022e4 + 0.174648533111170842352371577715206196e5 * e) * e) * e) * e) * e) / (-0.187895568271949507364746141208188607e1 + (0.127187326172428550555575106191962049e3 + (-0.391127655037883850891444158690092364e4 + (0.721297447038740887884294169354767017e5 + (-0.886316457638283267864458871156732103e6 + (0.761957869465466443836082093478020334e7 + (-0.467643218825825740643618940194817175e8 + (0.204898959658861649188955075584321601e9 + (-0.628105028695609148384585504363505276e9 + (0.128292952393022028051688483115552415e10 + (-0.157142538348665746049186034641850761e10 + t105) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH13_ke (double e, double x)
{
  return((0.129381610284230446052500983112729052e2 + (0.342441312491350047959409442690004108e4 + (0.223909220897861088887233926379037760e5 + (-0.162856851058980346643521657391359090e7 + (0.123809618823724370651737099524684179e8 + (0.281678529753623567880799860792691995e8 + (-0.710060480516354259195910925959448821e9 + (0.316684697489519010326057830663991061e10 + (-0.480148416349919193677752174088947095e10 + (-0.217748154079716336845734406175483434e10 + 0.941554158933815319064464110489708415e10 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.458904870142121294746473264600339470e1 + (0.378572910958471472235400673736632033e3 + (-0.141880182040957432670182868616266577e5 + (0.318869563527716014326559707211918887e6 + (-0.477508255917574524551781597152932242e7 + (0.500281408608999787869184895762292474e8 + (-0.374185807756642434933400908550787825e9 + (0.199801900861849305426002298534833900e10 + (-0.746409653921812392827805258459255005e10 + (0.185793626328864079328989769853095642e11 + (-0.277333751125374620229263302952112496e11 + 0.188069763456636188921519952397325104e11 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.127711626623066038646683377841197166e2 + (0.265781212633026472747447917029940440e4 + (0.256604257483771666534771688577691185e5 + (-0.133309639375666107136813677974915591e7 + (0.954586656112075737492808278878939240e7 + (0.804834057957759808714045123433957895e7 + (-0.294843896909994677470136266930040924e9 + (0.851319370207967018455123343619878010e9 + (0.680420321336346493118027449099666645e9 + (-0.631021914549760097442896326422455475e10 + 0.752279053826544755686079809589300420e10 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.183561948056848517898589305840135788e1 + (0.151429164383388588894160269494652813e3 + (-0.567520728163829730680731474465066308e4 + (0.127547825411086405730623882884767555e6 + (-0.191003302367029809820712638861172897e7 + (0.200112563443599915147673958304916989e8 + (-0.149674323102656973973360363420315130e9 + (0.799207603447397221704009194139335603e9 + (-0.298563861568724957131122103383702002e10 + (0.743174505315456317315959079412382566e10 + (-0.110933500450149848091705321180844998e11 + 0.752279053826544755686079809589300418e10 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.943829393483032517299163678460381575e1 + (0.232353750683759592216473589858337838e4 + (0.172113836322987694003228415662911808e5 + (-0.888681477107596473251940658786300138e6 + (0.475533584178895048278757486148849576e7 + (0.200651643860018558308346604683056857e8 + (-0.231744927186435950252463422641688566e9 + (0.660785081828945274003949659809210758e9 - 0.625950417322073117752324356164033802e9 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.183561948056848517898589305840135789e1 + (0.151429164383388588894160269494652813e3 + (-0.567520728163829730680731474465066307e4 + (0.127547825411086405730623882884767555e6 + (-0.191003302367029809820712638861172897e7 + (0.200112563443599915147673958304916990e8 + (-0.149674323102656973973360363420315129e9 + (0.799207603447397221704009194139335603e9 + (-0.298563861568724957131122103383702002e10 + (0.743174505315456317315959079412382568e10 + (-0.110933500450149848091705321180844999e11 + 0.752279053826544755686079809589300417e10 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.360192786379611229214936047392604266e1 + (0.759181293829408021965316756627398154e3 + (0.620086527113363616728707016605741804e4 + (-0.268596128615773974607467434242427420e6 + (0.131569419520023655038795237504038268e7 + (0.330868032863098270834573893005741560e7 + (-0.298745370849771674782157447258851340e8 + 0.441627728596332714551041383260739520e8 * e) * e) * e) * e) * e) * e) * e) / (-0.146849558445478814318871444672108630e1 + (0.121143331506710871115328215595722251e3 + (-0.454016582531063784544585179572053047e4 + (0.102038260328869124584499106307814044e6 + (-0.152802641893623847856570111088938317e7 + (0.160090050754879932118139166643933593e8 + (-0.119739458482125579178688290736252104e9 + (0.639366082757917777363207355311468483e9 + (-0.238851089254979965704897682706961602e10 + (0.594539604252365053852767263529906055e10 + (-0.887468003601198784733642569446759990e10 + 0.601823243061235804548863847671440335e10 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.781919465159404430546232246062086635e0 + (0.165435248328245713145272086137754760e3 + (0.987311130011416945093698882449364060e3 + (-0.428996344136361861035587580646018585e5 + (0.144773065103784035913346203764955708e6 + (0.654383373096719524709286990410339135e6 - 0.220073301828383679526735805279132841e7 * e) * e) * e) * e) * e) * e) / (-0.146849558445478814318871444672108630e1 + (0.121143331506710871115328215595722251e3 + (-0.454016582531063784544585179572053046e4 + (0.102038260328869124584499106307814044e6 + (-0.152802641893623847856570111088938317e7 + (0.160090050754879932118139166643933591e8 + (-0.119739458482125579178688290736252104e9 + (0.639366082757917777363207355311468482e9 + (-0.238851089254979965704897682706961602e10 + (0.594539604252365053852767263529906054e10 + (-0.887468003601198784733642569446759991e10 + 0.601823243061235804548863847671440334e10 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + (0.915216413027518681748540634964402696e-1 + (0.135119230178486419642225858939999826e2 + (0.721975929244057943900260670779301216e2 + (-0.268103921463832881483940940993107792e4 + (0.368461313285248439745164048925802548e4 + 0.468740383632118650459966856853287278e5 * e) * e) * e) * e) * e) / (-0.146849558445478814318871444672108630e1 + (0.121143331506710871115328215595722251e3 + (-0.454016582531063784544585179572053048e4 + (0.102038260328869124584499106307814044e6 + (-0.152802641893623847856570111088938318e7 + (0.160090050754879932118139166643933592e8 + (-0.119739458482125579178688290736252104e9 + (0.639366082757917777363207355311468484e9 + (-0.238851089254979965704897682706961602e10 + (0.594539604252365053852767263529906056e10 + (-0.887468003601198784733642569446759988e10 + 0.601823243061235804548863847671440336e10 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH14_ke (double e, double x)
{
  double t147;
  double t149;
  t147 = (0.107857735670403076200832244215151553e12 + (-0.196347461151955489414800067395300487e12 + 0.162383616445418761641973110994759661e12 * e) * e) * e;
  t149 = (-0.355299216855201606400862758992454186e11 + t147) * e;
  return((0.115860633325119464265570631130881183e2 + (0.368306854188490531664819396901357474e4 + (0.309632613333191146387221189420684030e5 + (-0.252564398565678378443214976102576710e7 + (0.220750972658097878614514511085117212e8 + (0.708171577318516836211112286469290066e8 + (-0.185545009592150016393904485477858968e10 + (0.939389784939053494820637820032571313e10 + (-0.151689830709704897246881117680862066e11 + (-0.155496135944033493586422720520547834e11 + 0.545767590710009085193802210445995101e11 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.223167747484735449705667404556233714e1 + (0.224529261702933228874219249049636553e3 + (-0.102626239586536050963300920479271694e5 + (0.281294747314733816013869621457606751e6 + (-0.513735911188885208810665527468612310e7 + (0.656421042406988334415773933197241071e8 + (-0.598774276872494042397385778501597736e9 + (0.389925584798517147516745029404673683e10 + (-0.177649608427600803200431379496227094e11 + (0.539288678352015381004161221075757768e11 + (-0.981737305759777447074000336976502436e11 + 0.811918082227093808209865554973798304e11 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.199711160424893987172583494863112912e2 + (0.541812349340303749845478987224429309e4 + (0.583473588353827493136094430016703971e5 + (-0.370038293377869387050192063057562246e7 + (0.297650485437356996585249061978010939e8 + (0.653804898472897234530074012368099744e8 + (-0.162249653443700942515062892326344560e10 + (0.570801195503988784140285385753632347e10 + (0.175487679217006834795155197629787008e10 + (-0.440698319478845992282583075773321347e11 + 0.649534465781675046567892443979038643e11 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.178534197987788359764533923644986972e1 + (0.179623409362346583099375399239709242e3 + (-0.821009916692288407706407363834173561e4 + (0.225035797851787052811095697166085401e6 + (-0.410988728951108167048532421974889847e7 + (0.525136833925590667532619146557792856e8 + (-0.479019421497995233917908622801278189e9 + (0.311940467838813718013396023523738946e10 + (-0.142119686742080642560345103596981675e11 + (0.431430942681612304803328976860606211e11 + (-0.785389844607821957659200269581201949e11 + 0.649534465781675046567892443979038644e11 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.455120082795063577168933660575421578e2 + (0.133981754191391154827721721437604713e5 + (0.122245082616361194926696274683786002e6 + (-0.740495323303993262730252569479042815e7 + (0.463167921992850174759757877005602841e8 + (0.244291198046132160078223349315360096e9 + (-0.321063805288220738045732161338624514e10 + (0.106060941311330394816980003900250309e11 - 0.116501562648563201522615381707102978e11 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.557919368711838624264168511390584287e1 + (0.561323154257333072185548122624091384e3 + (-0.256565598966340127408252301198179238e5 + (0.703236868286834540034674053644016876e6 + (-0.128433977797221302202666381867153077e8 + (0.164105260601747083603943483299310268e9 + (-0.149693569218123510599346444625399434e10 + (0.974813961996292868791862573511684203e10 + (-0.444124021069002008001078448740567732e11 + (0.134822169588003845251040305268939441e12 + (-0.245434326439944361768500084244125609e12 + 0.202979520556773452052466388743449576e12 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.157480423846622649287272392399474864e2 + (0.407661254767347955749782560958239060e4 + (0.389995613921411133059291457378611262e5 + (-0.204495830239391581724825947691017452e7 + (0.115168145526769632199685743722323697e8 + (0.416003972414459784985556209452705092e8 + (-0.399395010959733160846613968939828558e9 + 0.676451146777177689343670274726389664e9 * e) * e) * e) * e) * e) * e) * e) / (-0.446335494969470899411334809112467429e1 + (0.449058523405866457748438498099273105e3 + (-0.205252479173072101926601840958543390e5 + (0.562589494629467632027739242915213500e6 + (-0.102747182237777041762133105493722462e8 + (0.131284208481397666883154786639448214e9 + (-0.119754855374498808479477155700319547e10 + (0.779851169597034295033490058809347362e10 + t149) * e) * e) * e) * e) * e) * e) * e) + ((0.323268823052745606821905700976516906e1 + (0.812714223795775311744163759341658658e3 + (0.592518109743265401954468408516289808e4 + (-0.306923138797745066570206926786757392e6 + (0.118288739043647533687198063832033994e7 + (0.736621649074653420178571715812819292e7 - 0.276813943760635285059060892998425612e8 * e) * e) * e) * e) * e) * e) / (-0.446335494969470899411334809112467429e1 + (0.449058523405866457748438498099273106e3 + (-0.205252479173072101926601840958543391e5 + (0.562589494629467632027739242915213502e6 + (-0.102747182237777041762133105493722462e8 + (0.131284208481397666883154786639448214e9 + (-0.119754855374498808479477155700319547e10 + (0.779851169597034295033490058809347362e10 + (-0.355299216855201606400862758992454187e11 + t147) * e) * e) * e) * e) * e) * e) * e) * e) + (0.339255760857281792970304166404408440e0 + (0.613577523804064307732828570146012484e2 + (0.405522673528207406258574149828480716e3 + (-0.180559985042755189247748290170180701e5 + (0.280112412272071597379435453455746270e5 + 0.483552527852256190725379461581319408e6 * e) * e) * e) * e) * e) / (-0.446335494969470899411334809112467429e1 + (0.449058523405866457748438498099273106e3 + (-0.205252479173072101926601840958543390e5 + (0.562589494629467632027739242915213500e6 + (-0.102747182237777041762133105493722462e8 + (0.131284208481397666883154786639448214e9 + (-0.119754855374498808479477155700319547e10 + (0.779851169597034295033490058809347365e10 + t149) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH15_ke (double e, double x)
{
  double t106;
  double t43;
  double t45;
  t43 = 0.1e1 / (-0.422672555302849936548341942196728931e1 + (0.518886148645480346264867941328315094e3 + (-0.289389734672659125841830546228679971e5 + (0.967856496143396470535356234672979447e6 + (-0.215681539695862996505306915288139596e8 + (0.336262400427105190037021441452128747e9 + (-0.374266341632368216984813128949688527e10 + (0.297385814656714707312036504454322604e11 + (-0.165318975621575414789287321429642166e12 + (0.612348617286979460714259107567796978e12 + (-0.136016242148332896355797223240106254e13 + 0.137253927389641664618298525930000127e13 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e);
  t45 = 0.274507854779283329236597051860000254e13 * e;
  t106 = (-0.544064968593331585423188892960425016e12 + 0.549015709558566658473194103720000508e12 * e) * e;
  return((0.392465370144160155361940680065755881e2 + (0.150047059359553548125191190875250821e5 + (0.159451612158586070744868649477379357e6 + (-0.148698453536059486167398984223197668e8 + (0.150637324331824171883450178383033098e9 + (0.652363164040146644973026079462521358e9 + (-0.184794704566052899090984561980479780e11 + (0.107246495768162343458447558459047941e12 + (-0.186627300366598580860958267228658327e12 + (-0.335186452350772580989633239268608806e12 + 0.117046556990408706607022406926164804e13 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * t43 + ((0.152145506309434429105711076293581948e3 + (0.520561110797163408715055099120858190e5 + (0.648790572765657141968025654060922095e6 + (-0.496781059254407377379784755555427384e8 + (0.455948563943476832122852344240857884e9 + (0.169806194227414309508860957119208856e10 + (-0.405197665218484384592955487919858212e11 + (0.171537049786182590581538194346933854e12 + (-0.763119850395930570774161446081958605e10 + (-0.151208523176796334774701198434082364e13 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.845345110605699873096683884393457861e1 + (0.103777229729096069252973588265663019e4 + (-0.578779469345318251683661092457359944e5 + (0.193571299228679294107071246934595889e7 + (-0.431363079391725993010613830576279191e8 + (0.672524800854210380074042882904257495e9 + (-0.748532683264736433969626257899377054e10 + (0.594771629313429414624073008908645207e11 + (-0.330637951243150829578574642859284332e12 + (0.122469723457395892142851821513559396e13 + (-0.272032484296665792711594446480212508e13 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.214583790904164807523761079812807575e2 + (0.756959834226155944834701633152040228e4 + (0.845031712591969239321449313769427388e5 + (-0.606128955009340501012645682728853256e7 + (0.445967651219239507407006064796386480e8 + (0.290872912854436401231230058149331535e9 + (-0.438762529095742630951098798744740578e10 + (0.168508826540126014412115693946348395e11 - 0.215313984560820022151999629005690334e11 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.169069022121139974619336776878691572e1 + (0.207554459458192138505947176531326037e3 + (-0.115755893869063650336732218491471989e5 + (0.387142598457358588214142493869191779e6 + (-0.862726158783451986021227661152558384e7 + (0.134504960170842076014808576580851500e9 + (-0.149706536652947286793925251579875411e10 + (0.118954325862685882924814601781729042e11 + (-0.661275902486301659157149285718568665e11 + (0.244939446914791784285703643027118791e12 + t106) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.851745461526237339062480153539846242e1 + (0.268903403870728258908009005001684006e4 + (0.303801535354944518411206191557757990e5 + (-0.192990945971527210579345043857843789e7 + (0.125958969536792483852901024368080516e8 + (0.624894811926819977226537947874148444e8 + (-0.657769702808022879955997956770288742e9 + 0.128379470980250583908492620568438253e10 * e) * e) * e) * e) * e) * e) * e) / (-0.169069022121139974619336776878691572e1 + (0.207554459458192138505947176531326038e3 + (-0.115755893869063650336732218491471989e5 + (0.387142598457358588214142493869191781e6 + (-0.862726158783451986021227661152558384e7 + (0.134504960170842076014808576580851499e9 + (-0.149706536652947286793925251579875410e10 + (0.118954325862685882924814601781729041e11 + (-0.661275902486301659157149285718568666e11 + (0.244939446914791784285703643027118790e12 + t106) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.412347938492127191501480908905297277e1 + (0.123617209935311852840633602283005758e4 + (0.109964387021103656844996354941438021e5 + (-0.682792696981036037204039854869206138e6 + (0.303712842052344059433581342977106012e7 + (0.253148425519587683957066066699968198e8 - 0.107737863355615837096108499519897900e9 * e) * e) * e) * e) * e) * e) * t43 + (0.156803717984814758182152911397009760e0 + (0.347074913444454670572908821119967963e2 + (0.282516383772553417497917440514032487e3 + (-0.151759631483791346115918884748177215e5 + (0.271567345814316802440651049808496750e5 + 0.617026393449414638534238913180198884e6 * e) * e) * e) * e) * e) / (-0.169069022121139974619336776878691572e1 + (0.207554459458192138505947176531326037e3 + (-0.115755893869063650336732218491471989e5 + (0.387142598457358588214142493869191779e6 + (-0.862726158783451986021227661152558384e7 + (0.134504960170842076014808576580851499e9 + (-0.149706536652947286793925251579875411e10 + (0.118954325862685882924814601781729041e11 + (-0.661275902486301659157149285718568666e11 + (0.244939446914791784285703643027118791e12 + (-0.544064968593331585423188892960425017e12 + 0.549015709558566658473194103720000509e12 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH16_ke (double e, double x)
{
  double t43;
  double t45;
  double t68;
  double t72;
  t43 = 0.1e1 / (-0.399462403086558316460832896587539486e1 + (0.598566981465892378481348718967764608e3 + (-0.407466576509692219798838450427618186e5 + (0.166336438763319479280301764459645961e7 + (-0.452435147005983249360743590674964355e8 + (0.860969898124275793954737865841768029e9 + (-0.116965046061682813913260371951892989e11 + (0.113438533610106229585156550273628255e12 + (-0.769710579683718517003335658670570928e12 + (0.347990389728510913722615123969336715e13 + (-0.943457630692192628411530777689131838e13 + 0.116203414921211291985525259790206385e14 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e);
  t45 = 0.232406829842422583971050519580412770e14 * e;
  t68 = (0.695980779457021827445230247938673430e13 + (-0.188691526138438525682306155537826368e14 + t45) * e) * e;
  t72 = (0.226877067220212459170313100547256509e12 + (-0.153942115936743703400667131734114185e13 + t68) * e) * e;
  return((0.646800515728539090421847412687274948e2 + (0.297734243816151508240872370771530964e5 + (0.395356043748464303700727131797858716e6 + (-0.427287434204552996011721317480341197e8 + (0.504939952241591907062466408276786177e9 + (0.287364911320982072813921409691140626e10 + (-0.901982574654216154271449868282388774e11 + (0.604880672939875199967380206550079466e12 + (-0.114961040588567274082627745557854407e13 + (-0.321226701415736139296284185425420583e13 + 0.120804278420196062883992627688170569e14 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * t43 + ((0.229878762891062733756286174664084365e3 + (0.973642289155186248120048985030685840e5 + (0.143319898316712169238102206558210600e7 + (-0.131886500556465930534781029583911361e9 + (0.139821430696658015144549861396193305e10 + (0.763487735080433267294802529591724928e10 + (-0.193637216579608051895932638150386430e12 + (0.981349679523920366468471580490480736e12 + (-0.349236816172645149849946992903349406e12 + (-0.104207560957304105996443215741583806e14 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.798924806173116632921665793175078974e1 + (0.119713396293178475696269743793552922e4 + (-0.814933153019384439597676900855236371e5 + (0.332672877526638958560603528919291921e7 + (-0.904870294011966498721487181349928707e8 + (0.172193979624855158790947573168353605e10 + (-0.233930092123365627826520743903785977e11 + t72) * e) * e) * e) * e) * e) * e) + ((0.775665446259540684544881274541835604e2 + (0.328433509066917851095698867105292684e5 + (0.446630775702757735827807159754369000e6 + (-0.381885642520520310299066312304120684e8 + (0.332002181936503704630874935166300416e9 + (0.266382469289043036209131329057665992e10 + (-0.463712258867140783146783730314188656e11 + (0.207622234763133730913811253445174588e12 - 0.309431675379190267320691606570889004e12 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.399462403086558316460832896587539486e1 + (0.598566981465892378481348718967764608e3 + (-0.407466576509692219798838450427618186e5 + (0.166336438763319479280301764459645961e7 + (-0.452435147005983249360743590674964355e8 + (0.860969898124275793954737865841768030e9 + (-0.116965046061682813913260371951892989e11 + (0.113438533610106229585156550273628255e12 + (-0.769710579683718517003335658670570926e12 + (0.347990389728510913722615123969336714e13 + (-0.943457630692192628411530777689131837e13 + 0.116203414921211291985525259790206384e14 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.570198665018544885718234292779499116e2 + (0.218643384208272857182846946808672038e5 + (0.293444793803031482584293847952951405e6 + (-0.226024062923277534487722753698489160e8 + (0.172045229983955890737201521555323090e9 + (0.114006209697365383934068614041481190e10 + (-0.134030311272431923080501779994691386e11 + 0.302779132342776838803091155446518085e11 * e) * e) * e) * e) * e) * e) * e) / (-0.798924806173116632921665793175078972e1 + (0.119713396293178475696269743793552922e4 + (-0.814933153019384439597676900855236372e5 + (0.332672877526638958560603528919291922e7 + (-0.904870294011966498721487181349928708e8 + (0.172193979624855158790947573168353606e10 + (-0.233930092123365627826520743903785978e11 + t72) * e) * e) * e) * e) * e) * e) + ((0.520587682486745386438640109706142989e1 + (0.186657147388502990334883262838418429e4 + (0.202440384220660266384610444569581601e5 + (-0.151276345714152673560517782088393329e7 + (0.783146482484446397097005614968382119e7 + (0.856566186788113732140270144702246079e8 - 0.416600300440995174305522359212900574e9 * e) * e) * e) * e) * e) * e) * t43 + (0.904413505754670889805394077767713360e0 + (0.244835069053037834410211022560225452e3 + (0.244765957369138363647891000812439342e4 + (-0.159275246271450574706014271687710912e6 + (0.334272686940182802332266690724610009e6 + 0.977409564724239356337892671203383736e7 * e) * e) * e) * e) * e) / (-0.798924806173116632921665793175078974e1 + (0.119713396293178475696269743793552922e4 + (-0.814933153019384439597676900855236371e5 + (0.332672877526638958560603528919291921e7 + (-0.904870294011966498721487181349928707e8 + (0.172193979624855158790947573168353605e10 + (-0.233930092123365627826520743903785978e11 + (0.226877067220212459170313100547256509e12 + (-0.153942115936743703400667131734114186e13 + t68) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH17_ke (double e, double x)
{
  double t43;
  double t45;
  double t65;
  t43 = 0.1e1 / (-0.150807594491852264484963899488437514e1 + (0.275882657680058687371180955432788040e3 + (-0.229280730002255643771196271412629291e5 + (0.114268483156177922799050713167045096e7 + (-0.379453421398922161124725697523578712e8 + (0.881561784191820108008143315394636629e9 + (-0.146212030477583253620021865394071001e11 + (0.173120865338479198080989773766734786e12 + (-0.143409406196036249577589402117743967e13 + (0.791550040289841531102945946451213575e13 + (-0.261995626097485242628898644296053951e14 + 0.393958627720626740201797287382784442e14 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e);
  t45 = 0.630333804353002784322875659812455104e14 * e;
  t65 = 0.630333804353002784322875659812455105e14 * e;
  return((0.416463091863157216347585376789435620e2 + (0.231028081541810490250140074039079314e5 + (0.380276884286456838420392383505043579e6 + (-0.481049408614500276731796510199446000e8 + (0.666572872393856742473192682267882050e9 + (0.489771402558570962105907273947432855e10 + (-0.173169339703612515877383210679422685e12 + (0.135074676089819858354305360659554617e13 + (-0.284349891651497361766844822393184694e13 + (-0.114983311706953433398209123905633391e14 + 0.484838681838687393061414586674277653e14 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * t43 + ((0.109909616075430396370526353015663600e3 + (0.570277726291982046229199358567419300e5 + (0.100254044370193589343764548339575795e7 + (-0.110741054348602706028719693506978069e9 + (0.136879755821518050155499946138660380e10 + (0.102159783496501009656867617339171049e11 + (-0.287514238901087111928758300671639996e12 + (0.174014375860671877329949345262881208e13 + (-0.116467796425026360680775011466624157e13 + (-0.230489170008882989237973474788828224e14 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.241292151186963623175942239181500022e1 + (0.441412252288093899793889528692460865e3 + (-0.366849168003609030033914034260206865e5 + (0.182829573049884676478481141067272155e7 + (-0.607125474238275457799561116037725939e8 + (0.141049885470691217281302930463141861e10 + (-0.233939248764133205792034984630513600e11 + (0.276993384541566716929583638026775658e12 + (-0.229455049913657999324143043388390346e13 + (0.126648006446374644976471351432194171e14 + (-0.419193001755976388206237830873686320e14 + t65) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.706365364926839632770207567437416850e2 + (0.359519269155681740912199194849525299e5 + (0.593859462969544788282424683918282985e6 + (-0.608222957120449995746711118744120231e8 + (0.626818898054734604386819546547496171e9 + (0.617447570203041585461852084250697074e10 + (-0.124493181797680843190215968226960909e12 + (0.651317478049878894729471541374055995e12 - 0.113483947132258383482244485939395570e13 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.241292151186963623175942239181500022e1 + (0.441412252288093899793889528692460865e3 + (-0.366849168003609030033914034260206865e5 + (0.182829573049884676478481141067272155e7 + (-0.607125474238275457799561116037725941e8 + (0.141049885470691217281302930463141861e10 + (-0.233939248764133205792034984630513600e11 + (0.276993384541566716929583638026775657e12 + (-0.229455049913657999324143043388390346e13 + (0.126648006446374644976471351432194172e14 + (-0.419193001755976388206237830873686321e14 + t65) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.302604389922033296129770053465016250e2 + (0.140601847366623074268424218640817465e5 + (0.225156107720869646996920972059446859e6 + (-0.210434021187659410292023820420069810e8 + (0.187752529873725605755637353974437646e9 + (0.163171471757000266123993043718750415e10 + (-0.216874451801170443096009771970836954e11 + 0.569058919068000915956471307990184652e11 * e) * e) * e) * e) * e) * e) * e) / (-0.301615188983704528969927798976875027e1 + (0.551765315360117374742361910865576079e3 + (-0.458561460004511287542392542825258581e5 + (0.228536966312355845598101426334090193e7 + (-0.758906842797844322249451395047157423e8 + (0.176312356838364021601628663078927325e10 + (-0.292424060955166507240043730788142000e11 + (0.346241730676958396161979547533469572e12 + (-0.286818812392072499155178804235487933e13 + (0.158310008057968306220589189290242715e14 + (-0.523991252194970485257797288592107899e14 + 0.787917255441253480403594574765568879e14 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.260690746319618157956999652242921128e1 + (0.112074670577806567611923774894103017e4 + (0.148131637482628719682357690724701285e5 + (-0.133604471836337173820192587362182830e7 + (0.810311213582919959437764185242404722e7 + (0.114693306763756370470109205477756707e9 - 0.641585602939869848896832142161647022e9 * e) * e) * e) * e) * e) * e) * t43 + (0.333479138456026421635000794788091569e0 + (0.110363052932218493945271502348661366e3 + (0.135255862443198971013021447632597381e4 + (-0.106910989296416277341334309287955031e6 + (0.266403156531981437941437150423338639e6 + 0.986384745858438806132475957105024054e7 * e) * e) * e) * e) * e) / (-0.241292151186963623175942239181500023e1 + (0.441412252288093899793889528692460864e3 + (-0.366849168003609030033914034260206866e5 + (0.182829573049884676478481141067272154e7 + (-0.607125474238275457799561116037725939e8 + (0.141049885470691217281302930463141861e10 + (-0.233939248764133205792034984630513600e11 + (0.276993384541566716929583638026775657e12 + (-0.229455049913657999324143043388390346e13 + (0.126648006446374644976471351432194172e14 + (-0.419193001755976388206237830873686320e14 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH18_ke (double e, double x)
{
  double t79;
  t79 = (-0.159072677922220073472302438024892542e9 + (0.451249955758699167295051959240860436e10 + (-0.913849203721993891068050280072887013e11 + (0.132119697259724359733684110040537917e13 + (-0.133635619929274802632603151276349741e14 + (0.900635627102633362768871828142201960e14 + (-0.363991006162249433699507472714989034e15 + 0.668302066439499291798531638150745270e15 * e) * e) * e) * e) * e) * e) * e) * e;
  return((0.131353675140254140225121771332078128e3 + (0.878813106041476506378045200946441420e5 + (0.178294811407695538468791865912690254e7 + (-0.265954306604369019305118877408131695e9 + (0.433930289329564640624303313811733695e10 + (0.406670785360452172840434713662165780e11 + (-0.163855852018377252086823753090554748e13 + (0.149460103775905827745281469293311232e14 + (-0.353376199908209044121751010219160265e14 + (-0.196697169571373757725670752245655768e15 + 0.952115950584276782907636885540054012e15 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.284412249033346657616179934408689207e1 + (0.635298630522281265225658571481320599e3 + (-0.644687463259501300621235534898555632e5 + (0.392315980370437084588830585751224902e7 + (-0.159072677922220073472302438024892542e9 + (0.451249955758699167295051959240860435e10 + (-0.913849203721993891068050280072887012e11 + (0.132119697259724359733684110040537916e13 + (-0.133635619929274802632603151276349740e14 + (0.900635627102633362768871828142201960e14 + (-0.363991006162249433699507472714989034e15 + 0.668302066439499291798531638150745271e15 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.202808742402354714542665170347516192e3 + (0.128157563994766803527900786382754546e6 + (0.270727160296729010336895239567296081e7 + (-0.359123945378973051198391139622862374e9 + (0.521121822150311983162339332490899572e10 + (0.511338083925438478905973824880088533e11 + (-0.163286306190637489144083386979885874e13 + (0.117882039366211274975376974021676696e14 + (-0.119045415449976656787423192794225209e14 + (-0.199541848613060470433893587342181851e15 + 0.668302066439499291798531638150745269e15 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.284412249033346657616179934408689207e1 + (0.635298630522281265225658571481320602e3 + (-0.644687463259501300621235534898555633e5 + (0.392315980370437084588830585751224901e7 + t79) * e) * e) * e) + ((0.123911497895562233252015120459479074e3 + (0.759021884183663490019924186447687495e5 + (0.152018305381670130347829173102648764e7 + (-0.187162763553761218283249770176331083e9 + (0.229193601478034315777327161905872662e10 + (0.277118262817987975290891121603330109e11 + (-0.648835822457154103344130610493555610e12 + (0.397414289783596099647391003689891862e13 - 0.811189010981760689801152742556512515e13 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.284412249033346657616179934408689207e1 + (0.635298630522281265225658571481320599e3 + (-0.644687463259501300621235534898555631e5 + (0.392315980370437084588830585751224900e7 + t79) * e) * e) * e) + ((0.159259305243267958402399794206149722e2 + (0.895537826077210896374408103663373435e4 + (0.171690116712775124161790010799684638e6 + (-0.194829166148425867930058745671857586e8 + (0.204573794256439921627894546473477171e9 + (0.230290487023450889111932333588283256e10 + (-0.349002312077442003358948868838879862e11 + 0.106674828445506815697084261087357294e12 * e) * e) * e) * e) * e) * e) * e) / (-0.113764899613338663046471973763475683e1 + (0.254119452208912506090263428592528241e3 + (-0.257874985303800520248494213959422254e5 + (0.156926392148174833835532234300489960e7 + (-0.636290711688880293889209752099570166e8 + (0.180499982303479666918020783696344173e10 + (-0.365539681488797556427220112029154806e11 + (0.528478789038897438934736440162151668e12 + (-0.534542479717099210530412605105398963e13 + (0.360254250841053345107548731256880785e14 + (-0.145596402464899773479802989085995614e15 + 0.267320826575799716719412655260298108e15 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.162041622594820324719519745720427378e2 + (0.837084311280112955895217316217555062e4 + (0.134803177961874027771963437336952688e6 + (-0.147062780870644810427420379175419693e8 + (0.105036539287568657943495083759102584e9 + (0.190492972414443649989363747068613779e10 - 0.123154680378089537941351643251444712e11 * e) * e) * e) * e) * e) * e) / (-0.711030622583366644040449836021723018e1 + (0.158824657630570316306414642870330150e4 + (-0.161171865814875325155308883724638909e6 + (0.980789950926092711472076464378062255e7 + (-0.397681694805550183680756095062231354e9 + (0.112812488939674791823762989810215109e11 + (-0.228462300930498472767012570018221753e12 + (0.330299243149310899334210275101344790e13 + (-0.334089049823187006581507878190874350e14 + (0.225158906775658340692217957035550490e15 + (-0.909977515405623584248768681787472588e15 + 0.167075516609874822949632909537686318e16 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + (0.239978431580316854035697746319593941e0 + (0.970615818577427493420134252248510812e2 + (0.145647653168803327303541469897790810e4 + (-0.140097135733790714895056274302472658e6 + (0.418152545884539184658508158187754155e6 + 0.193833256738584559290814733958684545e8 * e) * e) * e) * e) * e) / (-0.142206124516673328808089967204344604e1 + (0.317649315261140632612829285740660300e3 + (-0.322343731629750650310617767449277816e5 + (0.196157990185218542294415292875612450e7 + (-0.795363389611100367361512190124462710e8 + (0.225624977879349583647525979620430218e10 + (-0.456924601860996945534025140036443507e11 + (0.660598486298621798668420550202689585e12 + (-0.668178099646374013163015756381748705e13 + (0.450317813551316681384435914071100980e14 + (-0.181995503081124716849753736357494517e15 + 0.334151033219749645899265819075372635e15 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH19_ke (double e, double x)
{
  double t35;
  double t65;
  t35 = (-0.133343586905141128997484944728400658e9 + (0.461917455221431869318381248917848362e10 + (-0.114233164623892982865292076227558603e12 + (0.201676602909464038152744949185157111e13 + (-0.249103749584240792540201725496505541e14 + (0.205010944531984142689821863461322524e15 + (-0.101178509044006129550269526281573926e16 + 0.226850599791612277032940263040029348e16 * e) * e) * e) * e) * e) * e) * e) * e;
  t65 = 0.907402399166449108131761052160117395e16 * e;
  return((0.813750881327824953590871226255979942e2 + (0.657055357125981727350733954709893332e5 + (0.163635454023162357626131974066961223e7 + (-0.289458773890258579126063066430953178e9 + (0.558005393837699353268644419834327955e10 + (0.661314456424572346952731013076411648e11 + (-0.306221172099056174731308290445280182e13 + (0.328060792632030828085839665191373598e14 + (-0.882586556028723359876330060531787032e14 + (-0.651715242287989008854141943195584522e15 + 0.367642827126774005762945298095891110e16 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.107211232152524294477859247756045091e1 + (0.292443068938463712021732834835062717e3 + (-0.362396397294073065820281541253758714e5 + (0.269303307953145675211671949678083655e7 + t35) * e) * e) * e) + ((0.473291429036827463815907666769759352e3 + (0.363063892234499903557792451120376708e6 + (0.924755402047716339728215255707702768e7 + (-0.147482870737572136343225146378308962e10 + (0.252545356208222129551928702342741022e11 + (0.318358736568890006969036955348203580e12 + (-0.116838110884283668075948206943665513e14 + (0.100555002596058269255034676270301388e15 + (-0.140453401119986763528925662550124291e15 + (-0.221421115110524870386368223195001518e16 + 0.907402399166449108131761052160117396e16 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.428844928610097177911436991024180363e1 + (0.116977227575385484808693133934025087e4 + (-0.144958558917629226328112616501503486e6 + (0.107721323181258270084668779871233462e8 + (-0.533374347620564515989939778913602632e9 + (0.184766982088572747727352499567139345e11 + (-0.456932658495571931461168304910234413e12 + (0.806706411637856152610979796740628445e13 + (-0.996414998336963170160806901986022172e14 + (0.820043778127936570759287453845290099e15 + (-0.404714036176024518201078105126295702e16 + t65) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.137447711845118557841602720231789065e3 + (0.101434854954596552593296467428303652e6 + (0.246062223918513594589851577865740142e7 + (-0.365150745973342197872999212381869048e9 + (0.532280171629935609912168673437266444e10 + (0.790553301136291767240832661896983652e11 + (-0.215364452964873377923946526764859468e13 + (0.154694704110565780727616147005073533e14 - 0.370577396330766304712268253037711084e14 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.214422464305048588955718495512090182e1 + (0.584886137876927424043465669670125433e3 + (-0.724792794588146131640563082507517428e5 + (0.538606615906291350423343899356167311e7 + (-0.266687173810282257994969889456801316e9 + (0.923834910442863738636762497835696725e10 + (-0.228466329247785965730584152455117206e12 + (0.403353205818928076305489898370314222e13 + (-0.498207499168481585080403450993011086e14 + (0.410021889063968285379643726922645051e15 + (-0.202357018088012259100539052563147851e16 + 0.453701199583224554065880526080058696e16 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.519901922498643214489612394964936530e2 + (0.353602396721751591274743340516536592e5 + (0.813776134977719497884711343268571144e6 + (-0.112179730538698822465346193331290215e9 + (0.139072809039366361279508357908980715e10 + (0.201017465040385329646241971998892876e11 + (-0.349547606670897715074795719537381278e12 + 0.124765359162688438722338414656032736e13 * e) * e) * e) * e) * e) * e) * e) / (-0.268028080381310736194648119390112727e1 + (0.731107672346159280054332087087656790e3 + (-0.905990993235182664550703853134396785e5 + (0.673258269882864188029179874195209139e7 + (-0.333358967262852822493712361821001644e9 + (0.115479363805357967329595312229462090e11 + (-0.285582911559732457163230190568896508e12 + (0.504191507273660095381862372962892777e13 + (-0.622759373960601981350504313741263857e14 + (0.512527361329960356724554658653306312e15 + (-0.252946272610015323875673815703934813e16 + 0.567126499479030692582350657600073369e16 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.320415517031905405494225788478052418e1 + (0.199254651249783640351083232632039775e4 + (0.390926807086698455478117061506483455e5 + (-0.516679369863266012071821537360792992e7 + (0.436355673597269575128655668832673705e8 + (0.100667602277993279663577947656144151e10 - 0.755029821173698490526042484806066445e10 * e) * e) * e) * e) * e) * e) / (-0.107211232152524294477859247756045091e1 + (0.292443068938463712021732834835062717e3 + (-0.362396397294073065820281541253758715e5 + (0.269303307953145675211671949678083656e7 + t35) * e) * e) * e) + (0.883742235712886675510575466243481266e0 + (0.436751905966714313472588219163210450e3 + (0.801791323905549110674454202626276936e4 + (-0.939669389580281728758738050032843232e6 + (0.338028977648019851350995269599945908e7 + 0.194626450733415859663562213956557616e9 * e) * e) * e) * e) * e) / (-0.428844928610097177911436991024180363e1 + (0.116977227575385484808693133934025086e4 + (-0.144958558917629226328112616501503485e6 + (0.107721323181258270084668779871233462e8 + (-0.533374347620564515989939778913602631e9 + (0.184766982088572747727352499567139345e11 + (-0.456932658495571931461168304910234413e12 + (0.806706411637856152610979796740628444e13 + (-0.996414998336963170160806901986022171e14 + (0.820043778127936570759287453845290100e15 + (-0.404714036176024518201078105126295703e16 + t65) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH20_ke (double e, double x)
{
  double t45;
  double t65;
  t45 = 0.246491064227810105866415619391050032e17 * e;
  t65 = 0.246491064227810105866415619391050033e17 * e;
  return((0.793760055638700226380563879246346370e3 + (0.773946545727297958101593906430899448e6 + (0.235902650524227085156245776815256926e8 + (-0.497051239334101883793393898526337591e10 + (0.113531195913338585756466812180374430e12 + (0.169096294452958785130580388243236893e13 + (-0.905566621507109154878259102503582312e14 + (0.114350117353571130982679060128676810e16 + (-0.354170599493752658774496568652905703e16 + (-0.337326412090348706120980084097858815e17 + 0.224050927259407109202812675828855715e18 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.646362018425321307699881991027967635e1 + (0.215316407609233191439750907629270357e4 + (-0.325851547614276906480193640980570168e6 + (0.295717891301528896131467818850804182e8 + (-0.178816633428488538534888458572241268e10 + (0.756484257436233097577123969137580847e11 + (-0.228469031389138552443033003258756375e13 + (0.492595150678291788977404960013242573e14 + (-0.743043489119081125800437603937072215e15 + (0.746810196883681021395119125418952959e16 + (-0.450111970313009718175427122738554500e17 + 0.123245532113905052933207809695525016e18 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.218377994545431796368306060326732542e3 + (0.203002166330025266772833194854241690e6 + (0.624661112424346722387841160153359568e7 + (-0.119937571536720101299514972414495827e10 + (0.243296184971286359726173933683081458e11 + (0.388454408640290939338624662091558830e12 + (-0.165151567587448684179880954819144766e14 + (0.169401829205550865488632496601449470e15 + (-0.313023388805382937843938241417268212e15 + (-0.491851300594279850422214007856155280e16 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.129272403685064261539976398205593527e1 + (0.430632815218466382879501815258540714e3 + (-0.651703095228553812960387281961140337e5 + (0.591435782603057792262935637701608366e7 + (-0.357633266856977077069776917144482537e9 + (0.151296851487246619515424793827516170e11 + (-0.456938062778277104886066006517512750e12 + (0.985190301356583577954809920026485147e13 + (-0.148608697823816225160087520787414443e15 + (0.149362039376736204279023825083790592e16 + (-0.900223940626019436350854245477108999e16 + t65) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.482723711138381210018642287930585616e3 + (0.429589975264002313987653789209085516e6 + (0.126148233015135200627872527923083762e8 + (-0.226095215643982911062769324369162778e10 + (0.392876263358573840980696784058543125e11 + (0.717608867778973631657968910700569120e12 + (-0.227821454723861280817237335477863609e14 + (0.192193956130799763100264731568939795e15 - 0.541223537300080783496551682151089795e15 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.517089614740257046159905592822374107e1 + (0.172253126087386553151800726103416286e4 + (-0.260681238091421525184154912784456135e6 + (0.236574313041223116905174255080643347e8 + (-0.143053306742790830827910766857793014e10 + (0.605187405948986478061699175310064681e11 + (-0.182775225111310841954426402607005100e13 + (0.394076120542633431181923968010594059e14 + (-0.594434791295264900640350083149657774e15 + (0.597448157506944817116095300335162371e16 + (-0.360089576250407774540341698190843600e17 + 0.985964256911240423465662477564200133e17 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.863011955714892771738386247646999269e2 + (0.709834989795672509781072943777990000e5 + (0.196523449182349845577976478915263359e7 + (-0.329236676427404854897123431925076809e9 + (0.483188061038067162631843274361083738e10 + (0.891081529120601471729506625993371000e11 + (-0.178657855609803299224774732849374978e13 + 0.746264247183809360186323294393330038e13 * e) * e) * e) * e) * e) * e) * e) / (-0.323181009212660653849940995513983816e1 + (0.107658203804616595719875453814635179e4 + (-0.162925773807138453240096820490285085e6 + (0.147858945650764448065733909425402092e8 + (-0.894083167142442692674442292861206340e9 + (0.378242128718116548788561984568790425e11 + (-0.114234515694569276221516501629378188e13 + (0.246297575339145894488702480006621286e14 + (-0.371521744559540562900218801968536107e15 + (0.373405098441840510697559562709476481e16 + (-0.225055985156504859087713561369277250e17 + 0.616227660569525264666039048477625082e17 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.504312856652367654908978045119009611e1 + (0.378114887964635177336813527453496359e4 + (0.903800843546204796400670051815422819e5 + (-0.144892511980682043011502297045218453e8 + (0.145165417668102823516165245074239702e9 + (0.423732102331166469150006163667693852e10 - 0.369837191971273673349818196638170556e11 * e) * e) * e) * e) * e) * e) / (-0.129272403685064261539976398205593527e1 + (0.430632815218466382879501815258540713e3 + (-0.651703095228553812960387281961140338e5 + (0.591435782603057792262935637701608365e7 + (-0.357633266856977077069776917144482535e9 + (0.151296851487246619515424793827516170e11 + (-0.456938062778277104886066006517512751e12 + (0.985190301356583577954809920026485148e13 + (-0.148608697823816225160087520787414443e15 + (0.149362039376736204279023825083790592e16 + (-0.900223940626019436350854245477109001e16 + t65) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + (0.325335464671742346763810566639234071e0 + (0.196434120338370881324567243502942264e3 + (0.440938785291511077380611734815357860e4 + (-0.630134957990394128452936477099018694e6 + (0.274374227208487118838921118765003926e7 + 0.195159146655038586875381263649785268e9 * e) * e) * e) * e) * e) / (-0.129272403685064261539976398205593527e1 + (0.430632815218466382879501815258540714e3 + (-0.651703095228553812960387281961140335e5 + (0.591435782603057792262935637701608365e7 + (-0.357633266856977077069776917144482536e9 + (0.151296851487246619515424793827516170e11 + (-0.456938062778277104886066006517512751e12 + (0.985190301356583577954809920026485150e13 + (-0.148608697823816225160087520787414444e15 + (0.149362039376736204279023825083790592e16 + (-0.900223940626019436350854245477109000e16 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH21_ke (double e, double x)
{
  double t107;
  double t43;
  t43 = 0.1e1 / (-0.124664570969317721835198687727092356e1 + (0.507182108223502305385466000512383625e3 + (-0.937402446890326234650367154408183862e5 + (0.103897071244573777510516693584885564e8 + (-0.767278348069996813274476996158623369e9 + (0.396427854038019012283727902906299630e11 + (-0.146221339861599910939943038947871820e13 + (0.385028268496454230511915849854306282e14 + (-0.709309368645423184545570199141213867e15 + (0.870664013394258015257906781150308272e16 + (-0.640884433930337952494907213457624472e17 + 0.214313270189182399023348128959683758e18 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e);
  t107 = (-0.640884433930337952494907213457624475e17 + 0.214313270189182399023348128959683759e18 * e) * e;
  return((0.244227548913443793163790789419769674e3 + (0.287710338739076559426664519669920478e6 + (0.107098737483410706630509470131835324e8 + (-0.269730687581568782222013724855518895e10 + (0.731688031287798195626593485279284640e11 + (0.136336282336106852091898912109289349e13 + (-0.848535261408679942148616251964662210e14 + (0.126659699809802809130342847077439336e16 + (-0.456241733425588599222359367544914394e16 + (-0.548492558921769572656581403933582227e17 + 0.432077024259001616109182961153841885e18 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * t43 + ((0.127590402547087593851105248807922656e4 + (0.143603413934466188539766824535984393e7 + (0.534447724959592861126498514238192680e8 + (-0.123710310727465966519295221592703568e11 + (0.298157181908190180611991011614447888e12 + (0.597551833505451860711702544299604440e13 + (-0.295841287823337248559902022223994802e15 + (0.361651384640146251952310351384466398e16 + (-0.861283855747184643967983126938184576e16 + (-0.139935528623996246555678366871396315e18 + 0.857253080756729596093392515838735036e18 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.498658283877270887340794750908369425e1 + (0.202872843289400922154186400204953450e4 + (-0.374960978756130493860146861763273544e6 + (0.415588284978295110042066774339542254e8 + (-0.306911339227998725309790798463449347e10 + (0.158571141615207604913491161162519852e12 + (-0.584885359446399643759772155791487281e13 + (0.154011307398581692204766339941722514e15 + (-0.283723747458169273818228079656485547e16 + (0.348265605357703206103162712460123310e17 + (-0.256353773572135180997962885383049790e18 + 0.857253080756729596093392515838735035e18 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.167940787611534667396508350189300335e3 + (0.180372058051749044819243257876199792e6 + (0.641013168890862037270825657173276800e7 + (-0.138974964885715421400126715501924875e10 + (0.288199197048175626705967709743747362e11 + (0.648242258942970489565695312077735138e12 + (-0.240164332433620726753177088627382385e14 + (0.238279597073156026466209881586641088e15 - 0.789951322804946972673118705241130725e15 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.124664570969317721835198687727092357e1 + (0.507182108223502305385466000512383625e3 + (-0.937402446890326234650367154408183860e5 + (0.103897071244573777510516693584885564e8 + (-0.767278348069996813274476996158623367e9 + (0.396427854038019012283727902906299632e11 + (-0.146221339861599910939943038947871820e13 + (0.385028268496454230511915849854306284e14 + (-0.709309368645423184545570199141213867e15 + (0.870664013394258015257906781150308273e16 + t107) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.455558152014901888519985771385935572e2 + (0.453182952638336447374662783090737475e5 + (0.151207023939661269230574097249254912e7 + (-0.307965128594959425557408156936185615e9 + (0.536219941047139113455449058236452370e10 + (0.125573097091591320593905033085072028e12 + (-0.291426942325748117257539385175819512e13 + 0.142726388560185806327875477010121405e14 * e) * e) * e) * e) * e) * e) * e) / (-0.124664570969317721835198687727092356e1 + (0.507182108223502305385466000512383624e3 + (-0.937402446890326234650367154408183862e5 + (0.103897071244573777510516693584885564e8 + (-0.767278348069996813274476996158623369e9 + (0.396427854038019012283727902906299631e11 + (-0.146221339861599910939943038947871820e13 + (0.385028268496454230511915849854306284e14 + (-0.709309368645423184545570199141213867e15 + (0.870664013394258015257906781150308275e16 + t107) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.632241535515696699071119672778489655e1 + (0.572289930933749031908146116152960217e4 + (0.166670208122088419600652244455070373e6 + (-0.324400743023476356426165292082150308e8 + (0.386581858923506913018313989819637834e9 + (0.142194292660986961460668851851803862e11 - 0.144802036164772352588067217793693436e12 * e) * e) * e) * e) * e) * e) * t43 + (0.766334199442640745005818633517980512e0 + (0.565251004353410089632786621332035841e3 + (0.155088712115780343043662101869885108e5 + (-0.270404387618535107009002121113986312e7 + (0.142931017518524763656315291731375874e8 + 0.125130671543384284989651142268354044e10 * e) * e) * e) * e) * e) / (-0.249329141938635443670397375454184713e1 + (0.101436421644700461077093200102476725e4 + (-0.187480489378065246930073430881636772e6 + (0.207794142489147555021033387169771128e8 + (-0.153455669613999362654895399231724674e10 + (0.792855708076038024567455805812599260e11 + (-0.292442679723199821879886077895743640e13 + (0.770056536992908461023831699708612565e14 + (-0.141861873729084636909114039828242774e16 + (0.174132802678851603051581356230061654e17 + (-0.128176886786067590498981442691524895e18 + 0.428626540378364798046696257919367518e18 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH22_ke (double e, double x)
{
  double t143;
  t143 = 0.284368713621249815982800168625457794e19 * e;
  return((0.226371994331488958273860258131639072e4 + (0.322350977928389664570721463530847492e7 + (0.146311086800969178683336990018121809e9 + (-0.441693535744139301276048740204287395e11 + (0.142578994879259174900505334120915566e13 + (0.331231833444370977696505036455227040e14 + (-0.240502444058831769234742249639813709e16 + (0.425369629973727854448131146908963824e17 + (-0.179730154468387118351240049520880198e18 + (-0.268211099946545044441187843688543200e19 + 0.251936658023109289091676984048892908e20 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.733636642354017895615102841770821082e1 + (0.364530971420712738328615896384355736e4 + (-0.822865917091722127120962985641459665e6 + (0.111388081182791045225686123143159903e9 + (-0.100466374074885633763871370370846193e11 + (0.633963504601563997255107143568540457e12 + (-0.285590073288134000393904422905082352e14 + (0.918453471174183468347115176028735621e15 + (-0.206648556751632277249074825165811996e17 + (0.309798530541015710982914377183099112e18 + (-0.278509813738090972578360313725058776e19 + 0.113747485448499926393120067450183118e20 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.219953173622880398461738968277836375e4 + (0.299619181456620976118929863457147584e7 + (0.134963504233779698022012873257219091e9 + (-0.377101609800669047379077226664224240e11 + (0.108229450530731784589784982778060306e13 + (0.270710540593231716902251351319294766e14 + (-0.156637013698167548750583185057464332e16 + (0.228219899558916025242013886686650404e17 + (-0.688917174836384561221502394939904293e17 + (-0.118700763866678300718547255199885332e19 + 0.888652230066405674946250526954555602e19 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.573153626839076480949299095133453970e1 + (0.284789821422431826819231169050277918e4 + (-0.642863997727907911813252332532390363e6 + (0.870219384240555040825672837055936745e8 + (-0.784893547460044013780245081022235881e10 + (0.495283987969971872855552455912922229e12 + (-0.223117244756354687807737830394595587e14 + (0.717541774354830834646183731272449704e15 + (-0.161444184962212716600839707160790622e17 + (0.242030101985168524205401857174296181e18 + (-0.217585791982883572326843995097702170e19 + 0.888652230066405674946250526954555600e19 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.707206642702216259022644329547032475e3 + (0.917339942185103412358638863730208312e6 + (0.394548237224564969240852175453840203e8 + (-0.103599387068152270578124363874701539e11 + (0.256638025866683988183091537910093675e12 + (0.711761435514824355672939588005879594e13 + (-0.308119645327541156030236212607593148e15 + (0.359974907671575519274850805225531397e16 - 0.140685834110927336537731174471461412e17 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.366818321177008947807551420885410542e1 + (0.182265485710356369164307948192177868e4 + (-0.411432958545861063560481492820729831e6 + (0.556940405913955226128430615715799517e8 + (-0.502331870374428168819356851854230964e10 + (0.316981752300781998627553571784270229e12 + (-0.142795036644067000196952211452541176e14 + (0.459226735587091734173557588014367810e15 + (-0.103324278375816138624537412582905997e17 + (0.154899265270507855491457188591549556e18 + (-0.139254906869045486289180156862529388e19 + 0.568737427242499631965600337250915585e19 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.912129133667143303122872751789354016e2 + (0.109769889478065471979247290551285047e6 + (0.442049562571126929835115816919554686e7 + (-0.109487089174621063385339451858198830e10 + (0.226590966930284561201539125692940409e11 + (0.671378306954293685846319623224855381e12 + (-0.180940343684791017912132594395812394e14 + 0.104075942847015812114960988416618090e15 * e) * e) * e) * e) * e) * e) * e) / (-0.183409160588504473903775710442705271e1 + (0.911327428551781845821539740960889339e3 + (-0.205716479272930531780240746410364917e6 + (0.278470202956977613064215307857899758e8 + (-0.251165935187214084409678425927115482e10 + (0.158490876150390999313776785892135114e12 + (-0.713975183220335000984761057262705880e13 + (0.229613367793545867086778794007183905e15 + (-0.516621391879080693122687062914529989e16 + (0.774496326352539277457285942957747780e17 + (-0.696274534345227431445900784312646942e18 + t143) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.120484716955819071503802878955142527e2 + (0.131817996207639601066120784808685433e5 + (0.467794283931571380471915619565211856e6 + (-0.110622837969837467012712935714858380e9 + (0.157140078843514259434603746740876652e10 + (0.726057822216023289887763771234817622e11 - 0.864588712393503990410716099458729497e12 * e) * e) * e) * e) * e) * e) / (-0.183409160588504473903775710442705270e1 + (0.911327428551781845821539740960889339e3 + (-0.205716479272930531780240746410364916e6 + (0.278470202956977613064215307857899758e8 + (-0.251165935187214084409678425927115483e10 + (0.158490876150390999313776785892135115e12 + (-0.713975183220335000984761057262705880e13 + (0.229613367793545867086778794007183905e15 + (-0.516621391879080693122687062914529990e16 + (0.774496326352539277457285942957747779e17 + (-0.696274534345227431445900784312646940e18 + t143) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + (0.688491731141264100409652956329599248e0 + (0.620345351731950320074179296863217549e3 + (0.207991036813931424508300910847803238e5 + (-0.442603651333039131941926101043985420e7 + (0.284574465916100871571168854541747646e8 + 0.305869585939033034773092172372300736e10 * e) * e) * e) * e) * e) / (-0.183409160588504473903775710442705270e1 + (0.911327428551781845821539740960889338e3 + (-0.205716479272930531780240746410364916e6 + (0.278470202956977613064215307857899760e8 + (-0.251165935187214084409678425927115482e10 + (0.158490876150390999313776785892135114e12 + (-0.713975183220335000984761057262705880e13 + (0.229613367793545867086778794007183905e15 + (-0.516621391879080693122687062914529990e16 + (0.774496326352539277457285942957747780e17 + (-0.696274534345227431445900784312646940e18 + 0.284368713621249815982800168625457792e19 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH23_ke (double e, double x)
{
  double t43;
  double t45;
  t43 = 0.1e1 / (-0.565816665217407857840330702175092613e1 + (0.343376351455436907705065155939611317e4 + (-0.946686563900033920167528083161637073e6 + (0.156515377617383315073734431854337707e9 + (-0.172417019055206009151326086219037759e11 + (0.132881565439852545676297294719071682e13 + (-0.731113194575912358386794392796837642e14 + (0.287170398339746347779696328052639297e16 + (-0.789143248319373575462006596322015197e17 + (0.144491979591710230802625514161583652e19 + (-0.158652080048991597397156254114399109e20 + 0.791385179948318382942527578164526308e20 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e);
  t45 = 0.316554071979327353177011031265810524e20 * e;
  return((0.271805023253166704762946253042827175e4 + (0.468056557587299831421680961602357484e7 + (0.258739609657006854476358825831531158e9 + (-0.938421300012428654838478184085322476e11 + (0.361076896835198892498442928889071424e13 + (0.104317818851417612128377728118993801e15 + (-0.886336524221847236087673190186503465e16 + (0.186109042918086537524895891451030931e18 + (-0.928936369305509017877901093827994765e18 + (-0.169876783524947168115752763337708904e20 + 0.191044813624319769664077476974006224e21 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * t43 + ((0.129084248605054839693030999088378161e4 + (0.212800044577658498192153252636353508e7 + (0.116073884309588925818517106750761486e9 + (-0.391933421916119649750185405357131903e11 + (0.134192231590277898859641461396729456e13 + (0.417358160373984743619327785384129121e14 + (-0.282968501345827175075676171511577924e16 + (0.491462506435466731327310527411139628e17 + (-0.185929669465808751012341530442463251e18 + (-0.346059415319397966562192982155068081e19 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.226326666086963143136132280870037046e1 + (0.137350540582174763082026062375844527e4 + (-0.378674625560013568067011233264654829e6 + (0.626061510469533260294937727417350824e8 + (-0.689668076220824036605304344876151035e10 + (0.531526261759410182705189178876286728e12 + (-0.292445277830364943354717757118735056e14 + (0.114868159335898539111878531221055718e16 + (-0.315657299327749430184802638528806078e17 + (0.577967918366840923210502056646334609e18 + (-0.634608320195966389588625016457596440e19 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.309897753776449020069956039042507738e3 + (0.485796998405188887879234019055089932e6 + (0.252909200145426181323173050986300840e8 + (-0.805058499471104921747701941893587750e10 + (0.238429098060289932860251085069086096e12 + (0.816289970372610871748815482710122264e13 + (-0.413393500210432254765391208354852530e15 + (0.569367927116389065069273218222159320e16 - 0.262649893542480712196914319385093194e17 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.113163333043481571568066140435018522e1 + (0.686752702910873815410130311879222634e3 + (-0.189337312780006784033505616632327414e6 + (0.313030755234766630147468863708675412e8 + (-0.344834038110412018302652172438075518e10 + (0.265763130879705091352594589438143364e12 + (-0.146222638915182471677358878559367528e14 + (0.574340796679492695559392656105278594e15 + (-0.157828649663874715092401319264403039e17 + (0.288983959183420461605251028323167306e18 + (-0.317304160097983194794312508228798220e19 + 0.158277035989663676588505515632905262e20 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.762027802169678451764397290315151291e2 + (0.110978885725492716319546544163679295e6 + (0.540089938128167197289443097583529080e7 + (-0.162715285840898330033613963901484493e10 + (0.400901099770019132706831338180177535e11 + (0.149858785638307758099000189919319327e13 + (-0.470308084291398995526173748778968903e14 + 0.318203668991408573076320868820656408e15 * e) * e) * e) * e) * e) * e) * e) / (-0.113163333043481571568066140435018523e1 + (0.686752702910873815410130311879222635e3 + (-0.189337312780006784033505616632327414e6 + (0.313030755234766630147468863708675412e8 + (-0.344834038110412018302652172438075518e10 + (0.265763130879705091352594589438143364e12 + (-0.146222638915182471677358878559367528e14 + (0.574340796679492695559392656105278596e15 + (-0.157828649663874715092401319264403040e17 + (0.288983959183420461605251028323167305e18 + (-0.317304160097983194794312508228798219e19 + 0.158277035989663676588505515632905261e20 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.479901701577649009751903377047658564e2 + (0.635239934861977082779522665070483811e5 + (0.274731994491653304204996337207567030e7 + (-0.789798731526546174845958292480573814e9 + (0.133975787886844781104377147730589282e11 + (0.775654392890852926903138033567307789e12 - 0.108219976169361775942285272477761120e14 * e) * e) * e) * e) * e) * e) * t43 + (0.103765900702545090864964650232962746e1 + (0.114204636586915660141997136645011079e4 + (0.467839231615184573141167307139847684e5 + (-0.121537434266377347954410057226813713e8 + (0.951798077947501404149612545889622258e8 + 0.125387142153439359366396439362676130e11 * e) * e) * e) * e) * e) / (-0.226326666086963143136132280870037045e1 + (0.137350540582174763082026062375844527e4 + (-0.378674625560013568067011233264654829e6 + (0.626061510469533260294937727417350824e8 + (-0.689668076220824036605304344876151035e10 + (0.531526261759410182705189178876286728e12 + (-0.292445277830364943354717757118735057e14 + (0.114868159335898539111878531221055718e16 + (-0.315657299327749430184802638528806078e17 + (0.577967918366840923210502056646334609e18 + (-0.634608320195966389588625016457596439e19 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
double PPH24_ke (double e, double x)
{
  double t45;
  double t70;
  t45 = 0.107545732524847949074024882522265746e22 * e;
  t70 = (-0.588600540675611693375652956506486585e18 + (0.131630156593519915931677164680890294e20 + (-0.176524197738482954074604008550188202e21 + t45) * e) * e) * e;
  return((0.315311943316514593650572239644638158e4 + (0.656887001820076628072933419885265448e7 + (0.441881146106519979206238872921302390e9 + (-0.192907885990693101519646067624712132e12 + (0.886021052496504754321333320418898670e13 + (0.317720968901444015828315156889503330e15 + (-0.316675933601599077189569039365760258e17 + (0.790685887573011166805468103110583786e18 + (-0.468891170515116715965260949054152748e19 + (-0.104048493068824153647267450798570833e21 + 0.140535684992769769567966935463516349e22 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.426122878030668596020778386847744090e1 + (0.315846851102014142211206801202838453e4 + (-0.106355366921402612902560132608023665e7 + (0.214761887193099162684629756995127548e9 + (-0.288953065312820606452220028528341686e11 + (0.271993708874092860186879280237645712e13 + (-0.182778735579469453694697644136519647e15 + (0.876854661298519228474305032750626896e16 + (-0.294300270337805846687826478253243292e18 + (0.658150782967599579658385823404451467e19 + (-0.882620988692414770373020042750941010e20 + 0.537728662624239745370124412611328731e21 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.716317825281596034059479694637980144e4 + (0.142923873582119418501555057180182108e8 + (0.944367825228513789304179253735116667e9 + (-0.385735970283477910497013636886879072e12 + (0.157783819620241316667050397391836071e14 + (0.608736099034781178086321557277546589e15 + (-0.484571526934407545778093753813541699e17 + (0.100343980355122060538802052953598356e19 + (-0.471964492639239279146014263905711677e19 + (-0.962342534230351785492277860045438320e20 + t45) * e) * e) * e) * e) * e) * e) * e) * e) * e) / (-0.852245756061337192041556773695488178e1 + (0.631693702204028284422413602405676905e4 + (-0.212710733842805225805120265216047330e7 + (0.429523774386198325369259513990255095e9 + (-0.577906130625641212904440057056683372e11 + (0.543987417748185720373758560475291423e13 + (-0.365557471158938907389395288273039292e15 + (0.175370932259703845694861006550125380e17 + t70) * e) * e) * e) * e) * e) * e) * e) + ((0.658490142570222265121866074577169092e3 + (0.124822529293148665377209020364369672e7 + (0.786780376697257274731300445632943876e8 + (-0.303849355367905516649169288777316406e11 + (0.107666120765235935927304610654206901e13 + (0.455479199768156361469061363165198508e14 + (-0.270162516227732827153466718846832906e16 + (0.439137159403651455097543991278609208e17 - 0.239382079920751260736420384642710990e18 * e) * e) * e) * e) * e) * e) * e) * e) / (-0.170449151212267438408311354739097636e1 + (0.126338740440805656884482720481135381e4 + (-0.425421467685610451610240530432094660e6 + (0.859047548772396650738519027980510190e8 + (-0.115581226125128242580888011411336675e11 + (0.108797483549637144074751712095058285e13 + (-0.731114942317877814778790576546078585e14 + (0.350741864519407691389722013100250758e16 + (-0.117720108135122338675130591301297317e18 + (0.263260313187039831863354329361780587e19 + (-0.353048395476965908149208017100376403e20 + 0.215091465049695898148049765044531492e21 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.193361360695673410855153917735438630e3 + (0.340915670772991922763926573537307062e6 + (0.200714958914559523807848641849477229e8 + (-0.735709319811512748585543239171008972e10 + (0.216096087815677785615129248747404299e12 + (0.101670589267920213324776135695981329e14 + (-0.372450042532897074182379523008100700e15 + 0.296830689562949272083302001061190788e16 * e) * e) * e) * e) * e) * e) * e) / (-0.213061439015334298010389193423872045e1 + (0.157923425551007071105603400601419226e4 + (-0.531776834607013064512800663040118326e6 + (0.107380943596549581342314878497563774e9 + (-0.144476532656410303226110014264170843e11 + (0.135996854437046430093439640118822857e13 + (-0.913893677897347268473488220682598234e14 + (0.438427330649259614237152516375313449e16 + (-0.147150135168902923343913239126621647e18 + (0.329075391483799789829192911702225735e19 + (-0.441310494346207385186510021375470505e20 + 0.268864331312119872685062206305664365e21 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + ((0.581602150758445357218910264546626125e2 + (0.932246696163654705160425336880290169e5 + (0.491416080575727199352369382487778022e7 + (-0.171821630083523714554005851152170719e10 + (0.348595868397676294085091452485923362e11 + (0.252371419991052086899347122136794581e13 - 0.413279198272967563374126717245745181e14 * e) * e) * e) * e) * e) * e) / (-0.532653597538335745025972983559680112e1 + (0.394808563877517677764008501503548066e4 + (-0.132944208651753266128200165760029582e7 + (0.268452358991373953355787196243909435e9 + (-0.361191331641025758065275035660427107e11 + (0.339992136092616075233599100297057139e13 + (-0.228473419474336817118372055170649558e15 + (0.109606832662314903559288129093828362e17 + (-0.367875337922257308359783097816554115e18 + (0.822688478709499474572982279255564334e19 + (-0.110327623586551846296627505343867626e21 + 0.672160828280299681712655515764160911e21 * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) * e) + (0.477233403665490097516304584692172874e1 + (0.641567743631487522084596236552506650e4 + (0.321077651799523180334266798733704928e6 + (-0.101844536908105795256345578035817988e9 + (0.972347433419891943449282939330091796e9 + 0.156820230459739533844064221608206267e12 * e) * e) * e) * e) * e) / (-0.852245756061337192041556773695488180e1 + (0.631693702204028284422413602405676905e4 + (-0.212710733842805225805120265216047331e7 + (0.429523774386198325369259513990255095e9 + (-0.577906130625641212904440057056683372e11 + (0.543987417748185720373758560475291424e13 + (-0.365557471158938907389395288273039293e15 + (0.175370932259703845694861006550125379e17 + t70) * e) * e) * e) * e) * e) * e) * e) * x) * x) * x) * x) * x);
}
