/*-----------------------------------------------------------------*/
/*! 
  \file cmultiple.c 
  \brief Example of usage of the ARoME library using the C interface
         This example computes an array of Rossiter-McLaughlin effect
         for different positions of the planet
         It returns the values computed simulating 3 different techniques:
           - the CCF technique
           - the iodine cell technique
           - the weighted mean (like Ohta et al. 2005)

  \author  G. Boue
           EXOEarths, Centro de Astrofisica, Universidade do Porto.

   Copyright (C) 2012, CAUP
   email of the author : gwenael.boue@astro.up.pt

   This work has been supported by the European Research Council/European
   Community under the FP7 through Starting Grant agreement number 239953, as
   well as from Fundacao para a Ciencia e a Tecnologia (FCT) through program
   Ciencia 2007 funded by FCT/MCTES (Portugal) and POPH/FSE (EC), and in the
   form of grants reference PTDC/CTE-AST/098528/2008, SFRH/BPD/71230/2010, and
   SFRH/BPD/81084/2011.


  History:                                                                
*/
/*-----------------------------------------------------------------*/

/*----------------------------------------------------------------*/
/* License of the file :

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
/*----------------------------------------------------------------*/
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "arome.h"


/*************************/
/* main routine          */
/* just a test           */
/*************************/
int main(int argc, char **argv)
{
   const double pi = acos(-1.0);
   const int ncoord = 20;
   double c1, c2, c3, c4, beta0, Vsini, sigma0, zeta, Rp;
   double sma, inc, lambda, anom;
   int Kmax;
   double M_init, M_end;
   double tab_anom[ncoord];
   double tab_x[ncoord], tab_y[ncoord], tab_z[ncoord];
   double tab_f[ncoord], tab_vp[ncoord], tab_betapR[ncoord], tab_betapT[ncoord];
   double tab_v_CCF[ncoord], tab_v_iodine[ncoord], tab_v_mean[ncoord];
   t_arome *parome;
   int status = 0;
   
   /* planet orbital parameters */
   sma = 4.0;               /* stellar radii */
   inc = 86.0*pi/180.0;     /* radian */
   
   /* spin-orbit angle */
   lambda = 30.0*pi/180.0;  /* radian */
   
   /* Mean anomaly */
   M_init = 70.0*pi/180.0;
   M_end  = 110.0*pi/180.0;
   
   /* planet's coordinates */
   for(int k=0; k<ncoord; k++)
   {
     anom = (M_end-M_init)*k/(ncoord-1.0)+M_init;
     tab_anom[k] = anom;
     tab_x[k] = sma*(-cos(lambda)*cos(anom)+sin(lambda)*sin(anom)*cos(inc));
     tab_y[k] = sma*(-sin(lambda)*cos(anom)-cos(lambda)*sin(anom)*cos(inc));
     tab_z[k] = sma*sin(anom)*sin(inc);
   }
   
   /* limb-darkening */
   c1 = 0.701;
   c2 = 0.149;
   c3 = 0.277;
   c4 =-0.297;
   
   /* line profile */
   beta0  =  2.0;  /* width of the non-rotating star */
   Vsini  = 15.0;  /* Vsini */
   sigma0 =  8.0;  /* width of the best Gaussian fit */
   zeta   =  2.0;  /* macro-turbulence parameter */
   Rp     =  0.1;  /* radius of the planet */
   Kmax   =    4;  /* order of expansion for the Iodine cell technique */
   
   /* you can choose to never exit on error */
   arome_set_continue_on_error();
   
   /* allocation of the ARoME structure */
   parome = arome_alloc_nl(c1, c2, c3, c4);
   
   /* or with a quadratic limb-darkening law */
   //parome = arome_alloc_quad(u1, u2);
   
   /* set the lineprofile */
   status += arome_set_lineprofile(beta0, Vsini, sigma0, zeta, Kmax, parome);
   
   /* print error message and exit if status != 0 */
   if (status) exit(EXIT_FAILURE);
   
   /* you can choose to always exit on error */
   arome_set_exit_on_error();
   
   /* set the planet parameters */
   status += arome_set_planet(Rp, parome);
   
   /* initialize the techniques you are going to use in the following */
   /* here we initialize both the CCF and the iodine cell techniques */
   /* should be done once the planet and line profile parameters are set */
   status += arome_init_CCF(parome);
   status += arome_init_iodine(parome);
   
   /* to use the vectorial routines, it is necessary to allocate the necessary space */
   status += arome_malloc(ncoord, parome);
   
   /* compute the flux, vp and betap at given positions (x, y, z) of the planet */
   status += arome_mcalc_fvpbetap(tab_x, tab_y, tab_z, ncoord, parome);
   
   /* you can have access to the computed flux, subplanet vp, betapR, and betapT */
   status += arome_mget_flux(parome, ncoord, tab_f);
   status += arome_mget_vp(parome, ncoord, tab_vp);
   status += arome_mget_betapR(parome, ncoord, tab_betapR);
   status += arome_mget_betapT(parome, ncoord, tab_betapT);
   
   /* you can change the values of the flux, vp, betapR, betapT, e.g., */
   /* status += arome_mset_flux(own_tab_f, ncoord, parome); */
   
   /* get the RM signals */
   status += arome_mget_RM_CCF(parome, ncoord, tab_v_CCF);
   status += arome_mget_RM_iodine(parome, ncoord, tab_v_iodine);
   status += arome_mget_RM_mean(parome, ncoord, tab_v_mean);
   
   /* print the results */
   printf("%9s %8s %8s %8s ", "phase", "x", "y", "z");
   printf("%8s %8s %8s %8s ", "f", "vp", "betapR", "betapT");
   printf("%8s %8s %8s\n", "v_CCF", "v_iodine", "v_mean");
   for(int k=0; k<ncoord; k++)
   {
     printf("%9.6f %+8.4f %+8.4f %+8.4f ", tab_anom[k]/(2.0*pi), tab_x[k], tab_y[k], tab_z[k]);
     printf("%8.6f %+8.4f %8.4f %8.4f ", tab_f[k], tab_vp[k], tab_betapR[k], tab_betapT[k]);
     printf("%+8.5f %+8.5f %+8.5f\n", tab_v_CCF[k], tab_v_iodine[k], tab_v_mean[k]);
   }
   
   /* release memory */
   arome_mfree(parome);
   arome_free(parome);
   
   return 0;
}
