!# This source file is part of code Pégase.3.0.1 (2019-02-21).
!# Copyright: Michel Fioc (Michel.Fioc@iap.fr), Sorbonne université, 
!# Institut d'astrophysique de Paris/CNRS, France.
!# 
!# Pégase.3.0.1 is governed by the CeCILL license under French law and abides 
!# by the rules of distribution of free software. You can use, modify and/or 
!# redistribute this software under the terms of the CeCILL license as circulated 
!# by CEA, CNRS and INRIA at "http://www.cecill.info". The text of this license
!# is also available in French and in English in directory "doc_dir/" of this
!# code.
!# 
!# As a counterpart to the access to the source code and to the rights to copy,
!# modify and redistribute it granted by the license, users are provided only
!# with a limited warranty, and the software's author, the holder of the
!# economic rights, and the successive licensors have only limited
!# liability. 
!# 
!# The fact that you are presently reading this means that you have had
!# knowledge of the CeCILL license and that you accept its terms.
!#====================================================================== 

module mod_remnants_ejec_SN

  use mod_IMF, only : IMF_d_n

  implicit none
  private
  public :: isolated_stars, close_binaries, reset_remnants_ejec_SN

contains
  
!#======================================================================
  
  subroutine reset_remnants_ejec_SN( &
       nb_CCSN_IS, nb_CCSN_CB, nb_SNIa, ejec_tot_IS, &
       ejec_elem_IS, ejec_tot_CB, ejec_elem_CB, ejec_carb_LMW_IS, &
       ejec_sil_LMW_IS, ejec_carb_HMW_IS, ejec_sil_HMW_IS, &
       ejec_carb_CCSN_IS, ejec_sil_CCSN_IS, &
       ejec_carb_LMW_CB, &
       ejec_sil_LMW_CB, ejec_carb_HMW_CB, ejec_sil_HMW_CB, &
       ejec_carb_CCSN_CB, ejec_sil_CCSN_CB, &
       ejec_carb_SNIa, ejec_sil_SNIa, &
       ejec_dust_O_LMW_IS, ejec_dust_O_HMW_IS, &
       ejec_dust_O_CCSN_IS,ejec_dust_O_LMW_CB, &
       ejec_dust_O_HMW_CB, ejec_dust_O_CCSN_CB, &
       ejec_dust_O_SNIa, &
       mass_BHNS_IS, mass_BHNS_CB, &
       mass_WD_IS, mass_WD_CB)

    use mod_types
    
    implicit none
    real(DPR), intent(out) :: nb_CCSN_IS, nb_CCSN_CB, nb_SNIa, &
         ejec_carb_LMW_IS, ejec_sil_LMW_IS, ejec_carb_HMW_IS, &
         ejec_sil_HMW_IS, ejec_carb_CCSN_IS, &
         ejec_sil_CCSN_IS, &
         ejec_carb_LMW_CB, ejec_sil_LMW_CB, ejec_carb_HMW_CB, &
         ejec_sil_HMW_CB, ejec_carb_CCSN_CB, &
         ejec_sil_CCSN_CB, &
         ejec_carb_SNIa, ejec_sil_SNIa, &
         mass_BHNS_IS, mass_BHNS_CB, mass_WD_IS, mass_WD_CB    
    real(DPR), intent(out) :: ejec_dust_O_LMW_IS, ejec_dust_O_HMW_IS, &
         ejec_dust_O_CCSN_IS,ejec_dust_O_LMW_CB, &
         ejec_dust_O_HMW_CB, ejec_dust_O_CCSN_CB, &
         ejec_dust_O_SNIa
    real(DPR), intent(out) :: ejec_tot_IS, ejec_tot_CB
    real(DPR), dimension(0:), intent(out) :: ejec_elem_IS, ejec_elem_CB
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
    
!# Computation of the mass of gas and metals ejected since the
!# previous time-step. The apportionment between "isolated" stars [IS]
!# and stars in close binary systems [CB] is made in spectra.f ("f_SNIa").

    nb_CCSN_IS = 0
    nb_CCSN_CB = 0
    nb_SNIa = 0
    ejec_tot_IS = 0
    ejec_elem_IS(0:) = 0
    ejec_tot_CB = 0
    ejec_elem_CB(0:) = 0
    ejec_carb_LMW_IS = 0
    ejec_sil_LMW_IS = 0
    ejec_carb_HMW_IS = 0
    ejec_sil_HMW_IS = 0
    ejec_carb_CCSN_IS = 0
    ejec_sil_CCSN_IS = 0
    ejec_carb_LMW_CB = 0
    ejec_sil_LMW_CB = 0
    ejec_carb_HMW_CB = 0
    ejec_sil_HMW_CB = 0
    ejec_carb_CCSN_CB = 0
    ejec_sil_CCSN_CB = 0
    ejec_carb_SNIa = 0
    ejec_sil_SNIa = 0

    ejec_dust_O_LMW_IS = 0
    ejec_dust_O_HMW_IS = 0
    ejec_dust_O_CCSN_IS = 0
    ejec_dust_O_LMW_CB = 0
    ejec_dust_O_HMW_CB = 0
    ejec_dust_O_CCSN_CB = 0
    ejec_dust_O_SNIa = 0

    mass_BHNS_IS = 0
    mass_BHNS_CB = 0
    mass_WD_IS = 0
    mass_WD_CB = 0

  end subroutine reset_remnants_ejec_SN

!#======================================================================

  subroutine isolated_stars(l10_mass, d_l10_mass, IMF_file, IMF_dim_slope, IMF_bin_coeff, IMF_bin_bottom_mass, &
       IMF_bin_slope, IMF_mass_min, IMF_mass_max, IMF_norm, ejec_tot_IS, ejec_elem_IS, &
       ejec_carb_LMW_IS, ejec_sil_LMW_IS, ejec_carb_HMW_IS, &
       ejec_sil_HMW_IS, ejec_carb_CCSN_IS, ejec_sil_CCSN_IS, &
       ejec_dust_O_LMW_IS, ejec_dust_O_HMW_IS, ejec_dust_O_CCSN_IS, &
       nb_CCSN_IS, mass_BHNS_IS, mass_WD_IS, dim_mass_yields, mass_yields, ejec_tot, ejec_elem, carb_dust_LMW, &
       sil_dust_LMW, carb_dust_HMW, sil_dust_HMW, carb_dust_CCSN, sil_dust_CCSN, &
       O_dust_LMW, O_dust_HMW, O_dust_CCSN, &
       min_mass_CCSN)

    use mod_types
    use mod_interp, only : bracket, interp_lin_lin
    
    implicit none
    real(DPR), intent(in) :: l10_mass, d_l10_mass
    character(len=*), intent(in) :: IMF_file
    integer, intent(in) :: IMF_dim_slope
    real(DPR), dimension(:), intent(in) :: IMF_bin_coeff, IMF_bin_bottom_mass, IMF_bin_slope
    real(DPR), intent(in) :: IMF_mass_min, IMF_mass_max, IMF_norm
    real(DPR), intent(inout) :: ejec_carb_LMW_IS, ejec_sil_LMW_IS, &
         ejec_carb_HMW_IS, ejec_sil_HMW_IS, &
         ejec_carb_CCSN_IS, ejec_sil_CCSN_IS, &
         ejec_dust_O_LMW_IS, ejec_dust_O_HMW_IS, ejec_dust_O_CCSN_IS, &
         nb_CCSN_IS, mass_BHNS_IS, mass_WD_IS
    real(DPR), intent(inout) :: ejec_tot_IS
    real(DPR), dimension(0:), intent(inout) :: ejec_elem_IS
    real(DPR), dimension(:), intent(in) :: mass_yields, carb_dust_LMW, &
         sil_dust_LMW, carb_dust_HMW, sil_dust_HMW, carb_dust_CCSN, sil_dust_CCSN
    real(DPR), dimension(:), intent(in) :: O_dust_LMW, O_dust_HMW, O_dust_CCSN
    real(DPR), dimension(:,0:), intent(in) :: ejec_elem
    real(DPR), intent(in) :: min_mass_CCSN
    real(DPR), dimension(:), intent(in) :: ejec_tot
    integer, intent(in) :: dim_mass_yields
!#......................................................................
    integer :: i_bracket, i_elem
    real(DPR) :: d_n, mass
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

!# "Isolated" star (single star or member of a loose binary system).

    d_n = IMF_d_n(l10_mass, d_l10_mass, IMF_file, IMF_dim_slope, IMF_bin_coeff, IMF_bin_bottom_mass, IMF_bin_slope, &
         IMF_mass_min, IMF_mass_max, IMF_norm)
    mass = 10**l10_mass
    i_bracket = 0
    call bracket(dim_mass_yields, mass_yields, mass, i_bracket)
    ejec_tot_IS = ejec_tot_IS + d_n*interp_lin_lin(mass_yields(i_bracket), mass_yields(i_bracket+1), &
         ejec_tot(i_bracket), ejec_tot(i_bracket+1), mass)

    do i_elem = 0, size(ejec_elem_IS)-1
       ejec_elem_IS(i_elem) = ejec_elem_IS(i_elem) + d_n*interp_lin_lin(mass_yields(i_bracket), &
            mass_yields(i_bracket+1), ejec_elem(i_bracket, i_elem), ejec_elem(i_bracket+1, i_elem), mass)
    enddo
    ejec_carb_LMW_IS = ejec_carb_LMW_IS &
         + d_n*interp_lin_lin(mass_yields(i_bracket), mass_yields(i_bracket+1), &
         carb_dust_LMW(i_bracket), carb_dust_LMW(i_bracket+1), mass)
    ejec_sil_LMW_IS = ejec_sil_LMW_IS &
         + d_n*interp_lin_lin(mass_yields(i_bracket), &
         mass_yields(i_bracket+1), sil_dust_LMW(i_bracket), sil_dust_LMW(i_bracket+1), mass)
    ejec_carb_HMW_IS = ejec_carb_HMW_IS & 
         + d_n*interp_lin_lin(mass_yields(i_bracket), &
         mass_yields(i_bracket+1), carb_dust_HMW(i_bracket), carb_dust_HMW(i_bracket+1), mass)
    ejec_sil_HMW_IS = ejec_sil_HMW_IS & 
         + d_n*interp_lin_lin(mass_yields(i_bracket), &
         mass_yields(i_bracket+1), sil_dust_HMW(i_bracket), sil_dust_HMW(i_bracket+1), mass)
    ejec_carb_CCSN_IS = ejec_carb_CCSN_IS & 
         + d_n*interp_lin_lin(mass_yields(i_bracket), &
         mass_yields(i_bracket+1), carb_dust_CCSN(i_bracket), carb_dust_CCSN(i_bracket+1), mass)
    ejec_sil_CCSN_IS = ejec_sil_CCSN_IS & 
         + d_n*interp_lin_lin(mass_yields(i_bracket), &
         mass_yields(i_bracket+1), sil_dust_CCSN(i_bracket), sil_dust_CCSN(i_bracket+1), mass)

    ejec_dust_O_LMW_IS = ejec_dust_O_LMW_IS &
         + d_n*interp_lin_lin(mass_yields(i_bracket), &
         mass_yields(i_bracket+1), O_dust_LMW(i_bracket), O_dust_LMW(i_bracket+1), mass)
    ejec_dust_O_HMW_IS = ejec_dust_O_HMW_IS & 
         + d_n*interp_lin_lin(mass_yields(i_bracket), &
         mass_yields(i_bracket+1), O_dust_HMW(i_bracket), O_dust_HMW(i_bracket+1), mass)
    ejec_dust_O_CCSN_IS = ejec_dust_O_CCSN_IS & 
         + d_n*interp_lin_lin(mass_yields(i_bracket), &
         mass_yields(i_bracket+1), O_dust_CCSN(i_bracket), O_dust_CCSN(i_bracket+1), mass)

    if (mass >= min_mass_CCSN) then 
!#     Core-collapse supernova.
       nb_CCSN_IS = nb_CCSN_IS + d_n
       mass_BHNS_IS = mass_BHNS_IS + d_n*( & !# Mass locked in black holes \
            mass &                           !# or neutron stars. 
            -interp_lin_lin(mass_yields(i_bracket), mass_yields(i_bracket+1), ejec_tot(i_bracket), ejec_tot(i_bracket+1), mass))
    else
       mass_WD_IS = mass_WD_IS + d_n*( & !# Mass locked in white dwarfs. 
            mass &
            -interp_lin_lin(mass_yields(i_bracket), mass_yields(i_bracket+1), ejec_tot(i_bracket), ejec_tot(i_bracket+1), mass))
    end if

  end subroutine isolated_stars

!#======================================================================

  subroutine close_binaries(l10_mass, d_l10_mass, IMF_file, IMF_dim_slope, IMF_bin_coeff, IMF_bin_bottom_mass, &
       IMF_bin_slope, IMF_mass_min, IMF_mass_max, IMF_norm, mass_Ch, ejec_tot_CB, ejec_elem_CB, &
       ejec_carb_LMW_CB, ejec_sil_LMW_CB, &
       ejec_carb_HMW_CB, ejec_sil_HMW_CB, &
       ejec_carb_CCSN_CB, ejec_sil_CCSN_CB, &
       ejec_carb_SNIa, ejec_sil_SNIa, &
       ejec_dust_O_LMW_CB, ejec_dust_O_HMW_CB, &
       ejec_dust_O_CCSN_CB, ejec_dust_O_SNIa, &
       nb_CCSN_CB, mass_BHNS_CB, mass_WD_CB, nb_SNIa, dim_mass_yields, mass_yields, ejec_tot, ejec_elem, &
       carb_dust_LMW, sil_dust_LMW, &
       carb_dust_HMW, sil_dust_HMW, carb_dust_CCSN, sil_dust_CCSN, &
       carb_dust_SNIa, sil_dust_SNIa, &
       O_dust_LMW, O_dust_HMW, O_dust_CCSN, O_dust_SNIa, &
       ejec_tot_SNIa, ejec_elem_SNIa, min_mass_CCSN)

    use mod_types
    use mod_SSPs_constants, only : min_mass_SNIa, close_bin_gamma, d_mu0
    use mod_interp, only : bracket, interp_lin_lin
    
    implicit none
    real(DPR), intent(in) :: l10_mass, d_l10_mass
    character(len=*), intent(in) :: IMF_file
    integer, intent(in) :: IMF_dim_slope
    real(DPR), dimension(:), intent(in) :: IMF_bin_coeff, IMF_bin_bottom_mass, IMF_bin_slope
    real(DPR), intent(in) :: IMF_mass_min, IMF_mass_max, IMF_norm, mass_Ch !# Mass of Chandrasekhar.
    real(DPR), intent(inout) :: nb_CCSN_CB, &
         mass_BHNS_CB, mass_WD_CB, nb_SNIa
    real(DPR), intent(inout) :: ejec_tot_CB
    real(DPR), dimension(0:), intent(inout) :: ejec_elem_CB
    real(DPR), intent(inout) :: ejec_carb_LMW_CB, ejec_sil_LMW_CB, &
         ejec_carb_HMW_CB, ejec_sil_HMW_CB, &
         ejec_carb_CCSN_CB, ejec_sil_CCSN_CB, &
         ejec_carb_SNIa, ejec_sil_SNIa
    real(DPR), intent(inout) :: ejec_dust_O_LMW_CB, &
         ejec_dust_O_HMW_CB, ejec_dust_O_CCSN_CB, ejec_dust_O_SNIa
    real(DPR), dimension(:), intent(in) :: mass_yields
    real(DPR), dimension(0:), intent(in) :: ejec_elem_SNIa
    real(DPR), dimension(:,0:), intent(in) :: ejec_elem
    real(DPR), dimension(:), intent(in) :: carb_dust_LMW, sil_dust_LMW, &
         carb_dust_HMW, sil_dust_HMW, carb_dust_CCSN, sil_dust_CCSN
    real(DPR), intent(in) :: carb_dust_SNIa, sil_dust_SNIa
    real(DPR), dimension(:), intent(in) :: O_dust_LMW, &
         O_dust_HMW, O_dust_CCSN
    real(DPR), intent(in) :: O_dust_SNIa
    real(DPR), dimension(:), intent(in) :: ejec_tot
    real(DPR), intent(in) :: ejec_tot_SNIa
    real(DPR), intent(in) :: min_mass_CCSN
    integer, intent(in) :: dim_mass_yields
!#......................................................................
    integer :: i_bracket, i_elem
    integer :: n_mu, i_mu
    real(DPR) :: mass1, mass2, mu_inf, mass2_max, mu_sup, CB_norm, d_mu, mu, &
         CB_mass, d_n, d_n12, mWD1, delta_mass, mass
    real(DPR) :: ejec_tot1, ejec_tot2, ejec_scaling_factor
    real(DPR), dimension(0:size(ejec_elem_CB)-1) :: ejec_elem1, ejec_elem2
    real(DPR) :: ejec_carb_LMW1, ejec_sil_LMW1, ejec_carb_HMW1, &
         ejec_sil_HMW1, ejec_carb_CCSN1, ejec_sil_CCSN1
    real(DPR) :: ejec_carb_LMW2, ejec_sil_LMW2, ejec_carb_HMW2, &
         ejec_sil_HMW2, ejec_carb_CCSN2, ejec_sil_CCSN2
    real(DPR) :: ejec_dust_O_LMW1, ejec_dust_O_HMW1, ejec_dust_O_CCSN1
    real(DPR) :: ejec_dust_O_LMW2, ejec_dust_O_HMW2, ejec_dust_O_CCSN2
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
    
!#     Close binary system.
!#     `IMF_mass_min` <= `mass2` <= `mass1` <= `IMF_mass_max`.
!#     `mu = mass2/(mass1+mass2)`.
!#     `f(mu) = CB_norm*mu**close_bin_gamma`: mass-ratio function.
!#     $\int_{mu_inf}^{mu_sup} f(mu) d_mu = 1$.
!#     The overall mass of the close binary, `CB_mass = mass1+mass2`,
!#     is assumed to be distributed according to the IMF for "isolated" stars.

!# Primary star of a close binary system.     

    mass = 10**l10_mass
    mass1 = mass

    i_bracket = 0
    call bracket(dim_mass_yields, mass_yields, mass1, i_bracket)
    ejec_tot1 = interp_lin_lin(mass_yields(i_bracket), mass_yields(i_bracket+1), ejec_tot(i_bracket), ejec_tot(i_bracket+1), &
         mass1)

    do i_elem = 0, size(ejec_elem_CB)-1
       ejec_elem1(i_elem) = interp_lin_lin(mass_yields(i_bracket), mass_yields(i_bracket+1), ejec_elem(i_bracket, i_elem), &
            ejec_elem(i_bracket+1, i_elem), mass1)
    enddo
    ejec_carb_LMW1 = interp_lin_lin(mass_yields(i_bracket), mass_yields(i_bracket+1), carb_dust_LMW(i_bracket), &
         carb_dust_LMW(i_bracket+1), mass1)
    ejec_sil_LMW1 = interp_lin_lin(mass_yields(i_bracket), mass_yields(i_bracket+1), sil_dust_LMW(i_bracket), &
         sil_dust_LMW(i_bracket+1), mass1)
    ejec_carb_HMW1 = interp_lin_lin(mass_yields(i_bracket), mass_yields(i_bracket+1), carb_dust_HMW(i_bracket), &
         carb_dust_HMW(i_bracket+1), mass1)
    ejec_sil_HMW1 = interp_lin_lin(mass_yields(i_bracket), mass_yields(i_bracket+1), sil_dust_HMW(i_bracket), &
         sil_dust_HMW(i_bracket+1), mass1)
    ejec_carb_CCSN1 = interp_lin_lin(mass_yields(i_bracket), mass_yields(i_bracket+1), carb_dust_CCSN(i_bracket), &
         carb_dust_CCSN(i_bracket+1), mass1)
    ejec_sil_CCSN1 = interp_lin_lin(mass_yields(i_bracket), mass_yields(i_bracket+1), sil_dust_CCSN(i_bracket), &
         sil_dust_CCSN(i_bracket+1), mass1)

    ejec_dust_O_LMW1 = interp_lin_lin(mass_yields(i_bracket), mass_yields(i_bracket+1), O_dust_LMW(i_bracket), &
         O_dust_LMW(i_bracket+1), mass1)
    ejec_dust_O_HMW1 = interp_lin_lin(mass_yields(i_bracket), mass_yields(i_bracket+1), O_dust_HMW(i_bracket), &
         O_dust_HMW(i_bracket+1), mass1)
    ejec_dust_O_CCSN1 = interp_lin_lin(mass_yields(i_bracket), mass_yields(i_bracket+1), O_dust_CCSN(i_bracket), &
         O_dust_CCSN(i_bracket+1), mass1)

    mu_inf = IMF_mass_min/(IMF_mass_min+mass1)
    mass2_max = min(mass1, IMF_mass_max-mass1)
    mu_sup = mass2_max/(mass2_max+mass1)
    if (mu_sup > mu_inf) then
       CB_norm = (close_bin_gamma+1)/ &
            (mu_sup**(close_bin_gamma+1)-mu_inf**(close_bin_gamma+1))
       n_mu = int((mu_sup-mu_inf)/d_mu0+1)
       d_mu = (mu_sup-mu_inf)/n_mu
       do i_mu = 0, n_mu-1
          mu = mu_inf+(i_mu+0.5_DPR)*d_mu
          CB_mass = mass1/(1-mu)

!# Note: `d_l10_mass*d_mu` := d(log10[mass1])*d(mu) = d(log10[CB_mass])*d(mu).

          d_n = IMF_d_n(log10(CB_mass), d_l10_mass, IMF_file, IMF_dim_slope, IMF_bin_coeff, &
               IMF_bin_bottom_mass, IMF_bin_slope, IMF_mass_min, IMF_mass_max, IMF_norm)
          d_n12 = d_n*CB_norm*mu**close_bin_gamma*d_mu
          ejec_tot_CB = ejec_tot_CB + d_n12*ejec_tot1
           ejec_elem_CB(0:) = ejec_elem_CB(0:) + d_n12*ejec_elem1(0:)
          
          ejec_carb_LMW_CB = ejec_carb_LMW_CB &
               + d_n12*ejec_carb_LMW1
          ejec_sil_LMW_CB = ejec_sil_LMW_CB &
               + d_n12*ejec_sil_LMW1
          ejec_carb_HMW_CB = ejec_carb_HMW_CB & 
               + d_n12*ejec_carb_HMW1
          ejec_sil_HMW_CB = ejec_sil_HMW_CB & 
               + d_n12*ejec_sil_HMW1
          ejec_carb_CCSN_CB = ejec_carb_CCSN_CB & 
               + d_n12*ejec_carb_CCSN1
          ejec_sil_CCSN_CB = ejec_sil_CCSN_CB & 
               + d_n12*ejec_sil_CCSN1
          
          ejec_dust_O_LMW_CB = ejec_dust_O_LMW_CB &
               + d_n12*ejec_dust_O_LMW1
          ejec_dust_O_HMW_CB = ejec_dust_O_HMW_CB & 
               + d_n12*ejec_dust_O_HMW1
          ejec_dust_O_CCSN_CB = ejec_dust_O_CCSN_CB & 
               + d_n12*ejec_dust_O_CCSN1

          if (mass1 >= min_mass_CCSN) then
             nb_CCSN_CB = nb_CCSN_CB + d_n12
             mass_BHNS_CB = mass_BHNS_CB + d_n12*(mass1-ejec_tot1)
          else
             mass_WD_CB = mass_WD_CB + d_n12*(mass1-ejec_tot1)
          end if
       end do
    end if

!# Secondary star of a close binary system.

    mass2 = mass
    ejec_tot2 = ejec_tot1
    ejec_elem2(0:) = ejec_elem1(0:)
    ejec_carb_LMW2 = ejec_carb_LMW1
    ejec_sil_LMW2 = ejec_sil_LMW1
    ejec_carb_HMW2 = ejec_carb_HMW1 
    ejec_sil_HMW2 = ejec_sil_HMW1 
    ejec_carb_CCSN2 = ejec_carb_CCSN1 
    ejec_sil_CCSN2 = ejec_sil_CCSN1 

    ejec_dust_O_LMW2 = ejec_dust_O_LMW1
    ejec_dust_O_HMW2 = ejec_dust_O_HMW1 
    ejec_dust_O_CCSN2 = ejec_dust_O_CCSN1 

    mu_inf = mass2/IMF_mass_max
    mu_sup = 0.5_DPR
    if (mu_sup > mu_inf) then
       CB_norm = (close_bin_gamma+1)/ &
            (mu_sup**(close_bin_gamma+1)-mu_inf**(close_bin_gamma+1))
       n_mu = int((mu_sup-mu_inf)/d_mu0+1)
       d_mu = (mu_sup-mu_inf)/n_mu
       do i_mu = 0, n_mu-1
          mu = mu_inf+(i_mu+0.5_DPR)*d_mu
          mass1 = mass2*(1-mu)/mu
          CB_mass = mass1+mass2

!# Note: `d_l10_mass*d_mu` := d(log10[mass2])*d(mu) = d(log10[CB_mass])*d(mu).

          d_n = IMF_d_n(log10(CB_mass), d_l10_mass, IMF_file, IMF_dim_slope, IMF_bin_coeff, &
               IMF_bin_bottom_mass, IMF_bin_slope, IMF_mass_min, IMF_mass_max, IMF_norm)
          d_n12 = d_n*CB_norm*mu**close_bin_gamma*d_mu
          if (CB_mass < min_mass_SNIa .or. mass1 > min_mass_CCSN) then 
!# No SNIa explosion: mass too small or the remnant of the primary 
!# is not a white dwarf. 
             ejec_tot_CB = ejec_tot_CB + d_n12 &
                  *ejec_tot2
             ejec_elem_CB(0:) = ejec_elem_CB(0:) + d_n12 &
                  *ejec_elem2(0:)
             ejec_carb_LMW_CB = ejec_carb_LMW_CB &
                  + d_n12*ejec_carb_LMW2
             ejec_sil_LMW_CB = ejec_sil_LMW_CB &
                  + d_n12*ejec_sil_LMW2
             ejec_carb_HMW_CB = ejec_carb_HMW_CB & 
                  + d_n12*ejec_carb_HMW2
             ejec_sil_HMW_CB = ejec_sil_HMW_CB & 
                  + d_n12*ejec_sil_HMW2
             ejec_carb_CCSN_CB = ejec_carb_CCSN_CB & 
                  + d_n12*ejec_carb_CCSN2
             ejec_sil_CCSN_CB = ejec_sil_CCSN_CB & 
                  + d_n12*ejec_sil_CCSN2

             ejec_dust_O_LMW_CB = ejec_dust_O_LMW_CB &
                  + d_n12*ejec_dust_O_LMW2
             ejec_dust_O_HMW_CB = ejec_dust_O_HMW_CB & 
                  + d_n12*ejec_dust_O_HMW2
             ejec_dust_O_CCSN_CB = ejec_dust_O_CCSN_CB & 
                  + d_n12*ejec_dust_O_CCSN2

             if (mass2 >= min_mass_CCSN) then
                nb_CCSN_CB = nb_CCSN_CB + d_n12
                mass_BHNS_CB = mass_BHNS_CB + d_n12*(mass2-ejec_tot2)
             else
                mass_WD_CB = mass_WD_CB + d_n12*(mass2-ejec_tot2)
             end if
          else          
!# SNIa explosion.
             nb_SNIa = nb_SNIa + d_n12             
             i_bracket = 0
             call bracket(dim_mass_yields, mass_yields, mass1, i_bracket)
!# `ejec_elem1` is not needed. `ejec_tot1` needed just to 
!# compute the mass of the white dwarf remnant of the primary.
             ejec_tot1 = interp_lin_lin(mass_yields(i_bracket), mass_yields(i_bracket+1), ejec_tot(i_bracket), &
                  ejec_tot(i_bracket+1), mass1)
!# Mass of the white dwarf remnant of the primary.
             mWD1 = mass1-ejec_tot1
!# White dwarf destroyed in the explosion.
             mass_WD_CB = mass_WD_CB-d_n12*mWD1

!# Mass ejected by the secondary used by the primary to reach 
!# the Chandrasekhar mass.
             delta_mass = mass_Ch-mWD1 

             ejec_tot_CB = ejec_tot_CB + d_n12*( &
!# Ejecta of the secondary not processed by the primary. 
                  ejec_tot2-delta_mass &
!# Ejecta of the primary (totally destroyed).
                  + ejec_tot_SNIa) 
             
             ejec_scaling_factor = (ejec_tot2-delta_mass)/ejec_tot2
             do i_elem = 0, size(ejec_elem_CB)-1
                ejec_elem_CB(i_elem) = ejec_elem_CB(i_elem) + d_n12*( &
!# Secundary; ejecta scaled to `ejec_tot2-delta_mass`. 
                     ejec_elem2(i_elem)*ejec_scaling_factor &
!# Primary (no H or He remains).
                     +ejec_elem_SNIa(i_elem)) !# W7.
             enddo

!# Dust ejecta of the secondary, scaled to `ejec_tot2-delta_mass`.
             ejec_carb_LMW_CB = ejec_carb_LMW_CB &
                  + d_n12*ejec_carb_LMW2*ejec_scaling_factor
             ejec_sil_LMW_CB = ejec_sil_LMW_CB &
                  + d_n12*ejec_sil_LMW2*ejec_scaling_factor
             ejec_carb_HMW_CB = ejec_carb_HMW_CB & 
                  + d_n12*ejec_carb_HMW2*ejec_scaling_factor
             ejec_sil_HMW_CB = ejec_sil_HMW_CB & 
                  + d_n12*ejec_sil_HMW2*ejec_scaling_factor
             ejec_carb_CCSN_CB = ejec_carb_CCSN_CB & 
                  + d_n12*ejec_carb_CCSN2*ejec_scaling_factor
             ejec_sil_CCSN_CB = ejec_sil_CCSN_CB & 
                  + d_n12*ejec_sil_CCSN2*ejec_scaling_factor

             ejec_dust_O_LMW_CB = ejec_dust_O_LMW_CB &
                  + d_n12*ejec_dust_O_LMW2*ejec_scaling_factor
             ejec_dust_O_HMW_CB = ejec_dust_O_HMW_CB & 
                  + d_n12*ejec_dust_O_HMW2*ejec_scaling_factor
             ejec_dust_O_CCSN_CB = ejec_dust_O_CCSN_CB & 
                  + d_n12*ejec_dust_O_CCSN2*ejec_scaling_factor

!# Dust ejecta from the SNIa exlosion.
             ejec_carb_SNIa = ejec_carb_SNIa &
                  + d_n12*carb_dust_SNIa
             ejec_sil_SNIa = ejec_sil_SNIa &
                  + d_n12*sil_dust_SNIa

             ejec_dust_O_SNIa = ejec_dust_O_SNIa &
                  + d_n12*O_dust_SNIa

          end if
       end do
    end if

  end subroutine close_binaries

end module mod_remnants_ejec_SN
