!# This source file is part of code Pégase.3.0.1 (2019-02-21).
!# Copyright: Michel Fioc (Michel.Fioc@iap.fr), Sorbonne université, 
!# Institut d'astrophysique de Paris/CNRS, France.
!# 
!# Pégase.3.0.1 is governed by the CeCILL license under French law and abides 
!# by the rules of distribution of free software. You can use, modify and/or 
!# redistribute this software under the terms of the CeCILL license as circulated 
!# by CEA, CNRS and INRIA at "http://www.cecill.info". The text of this license
!# is also available in French and in English in directory "doc_dir/" of this
!# code.
!# 
!# As a counterpart to the access to the source code and to the rights to copy,
!# modify and redistribute it granted by the license, users are provided only
!# with a limited warranty, and the software's author, the holder of the
!# economic rights, and the successive licensors have only limited
!# liability. 
!# 
!# The fact that you are presently reading this means that you have had
!# knowledge of the CeCILL license and that you accept its terms.
!#====================================================================== 

module mod_read_dust_transfer_param

  use mod_types
  use mod_analyze_statement, only : get_val, file_name, line_number, &
       error_message, warning_message

  implicit none
  public
  logical, save :: extinction, extinction_SFC, extinction_DISM, &
       dust_emission, dust_emission_SFC, dust_emission_DISM, &
       stoch_heating, stoch_heating_SFC, &
       stoch_heating_DISM, inclin_averaged
  real(CDR), save :: inclination, bulge_tot_ratio, &
       M_sys_spiral, expo_radius, M_sys_spher, core_radius, slab_factor, &
       self_abs_power, self_abs_power_SFC, self_abs_power_DISM, &
       carb_sublim_temp, carb_sublim_temp_SFC, carb_sublim_temp_DISM, &
       sil_sublim_temp, sil_sublim_temp_SFC, sil_sublim_temp_DISM

  character(len=std_string), save :: geometry, grains_file, &
       grains_file_SFC, grains_file_DISM

!# Following parameters are made private to ensure `_SFC` and `_DISM`
!# versions may be used.
  private :: extinction, dust_emission, self_abs_power, &
       stoch_heating, grains_file, carb_sublim_temp, sil_sublim_temp

contains

!#======================================================================

  subroutine reset_dust_transfer

    use mod_spectra_def_param
    implicit none
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    extinction = extinction_def
    extinction_SFC = extinction_SFC_def
    extinction_DISM = extinction_DISM_def

    grains_file = grains_file_def
    grains_file_SFC = grains_file_SFC_def
    grains_file_DISM = grains_file_DISM_def

    bulge_tot_ratio = bulge_tot_ratio_def
    geometry = geometry_def
    inclin_averaged = inclin_averaged_def
    inclination = inclination_def
    M_sys_spiral = M_sys_spiral_def
    expo_radius = expo_radius_def
    M_sys_spher = M_sys_spher_def
    core_radius = core_radius_def
    slab_factor = slab_factor_def

    carb_sublim_temp = carb_sublim_temp_def
    carb_sublim_temp_SFC = carb_sublim_temp_SFC_def
    carb_sublim_temp_DISM = carb_sublim_temp_DISM_def
    sil_sublim_temp = sil_sublim_temp_def
    sil_sublim_temp_SFC = sil_sublim_temp_SFC_def
    sil_sublim_temp_DISM = sil_sublim_temp_DISM_def

    dust_emission = dust_emission_def
    dust_emission_SFC = dust_emission_SFC_def
    dust_emission_DISM = dust_emission_DISM_def
    self_abs_power = self_abs_power_def
    self_abs_power_SFC = self_abs_power_SFC_def
    self_abs_power_DISM = self_abs_power_DISM_def
    stoch_heating = stoch_heating_def
    stoch_heating_SFC = stoch_heating_SFC_def
    stoch_heating_DISM = stoch_heating_DISM_def

  end subroutine reset_dust_transfer

!#======================================================================

  subroutine dust_transfer_check
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    if (self_abs_power < 0) then
       call warning_message("`self_abs_power` < 0. Set to 0.")
       self_abs_power = 0
    endif

    if (self_abs_power > 1) then
       call warning_message("`self_abs_power` > 1. Set to 1.")
       self_abs_power = 1
    endif

    if (self_abs_power_SFC < 0) then
       call warning_message("`self_abs_power_SFC` < 0. Set to 0.")
       self_abs_power_SFC = 0
    endif

    if (self_abs_power_SFC > 1) then
       call warning_message("`self_abs_power_SFC` > 1. Set to 1.")
       self_abs_power_SFC = 1
    endif

    if (self_abs_power_DISM < 0) then
       call warning_message("`self_abs_power_DISM` < 0. Set to 0.")
       self_abs_power_DISM = 0
    endif

    if (self_abs_power_DISM > 1) then
       call warning_message("`self_abs_power_DISM` > 1. Set to 1.")
       self_abs_power_DISM = 1
    endif

  end subroutine dust_transfer_check

!#======================================================================

  subroutine read_extinction(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(extinction, rhs)
    extinction_SFC = extinction
    extinction_DISM = extinction

  end subroutine read_extinction

!#======================================================================

  subroutine read_extinction_SFC(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(extinction_SFC, rhs)

  end subroutine read_extinction_SFC

!#======================================================================

  subroutine read_extinction_DISM(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(extinction_DISM, rhs)

  end subroutine read_extinction_DISM

!#======================================================================

  subroutine read_geometry(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(geometry, rhs)
    if (all(geometry /= (/ &
         "spheroidal", &
         "spiral    ", &
         "slab      "/))) &
         call error_message("The value of `geometry` should be &
         &""spheroidal"", ""spiral"" or ""slab"".")

  end subroutine read_geometry

!#======================================================================
  subroutine read_inclin_averaged(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(inclin_averaged, rhs)

  end subroutine read_inclin_averaged

!#======================================================================

  subroutine read_inclination(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(inclination, rhs)
    call set_inclin_averaged

  end subroutine read_inclination

!#======================================================================

  subroutine read_bulge_tot_ratio(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(bulge_tot_ratio, rhs)
    call set_geometry("spiral")

  end subroutine read_bulge_tot_ratio

!#======================================================================

  subroutine read_M_sys_spiral(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(M_sys_spiral, rhs)
    call set_geometry("spiral")

  end subroutine read_M_sys_spiral

!#======================================================================

  subroutine read_expo_radius(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(expo_radius, rhs)
    call set_geometry("spiral")

  end subroutine read_expo_radius

!#======================================================================

  subroutine read_M_sys_spher(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(M_sys_spher, rhs)
    call set_geometry("spheroidal")

  end subroutine read_M_sys_spher

!#======================================================================

  subroutine read_core_radius(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(core_radius, rhs)
    call set_geometry("spheroidal")

  end subroutine read_core_radius

!#======================================================================

  subroutine read_slab_factor(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(slab_factor, rhs)
    call set_geometry("slab")

  end subroutine read_slab_factor

!#======================================================================

  subroutine read_dust_emission(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(dust_emission, rhs)
    dust_emission_SFC = dust_emission
    dust_emission_DISM = dust_emission

  end subroutine read_dust_emission

!#======================================================================

  subroutine read_dust_emission_SFC(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(dust_emission_SFC, rhs)

  end subroutine read_dust_emission_SFC

!#======================================================================

  subroutine read_dust_emission_DISM(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(dust_emission_DISM, rhs)

  end subroutine read_dust_emission_DISM

!#======================================================================

  subroutine read_self_abs_power(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(self_abs_power, rhs)
    self_abs_power_SFC = self_abs_power
    self_abs_power_DISM = self_abs_power

  end subroutine read_self_abs_power

!#======================================================================

  subroutine read_self_abs_power_SFC(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(self_abs_power_SFC, rhs)

  end subroutine read_self_abs_power_SFC

!#======================================================================

  subroutine read_self_abs_power_DISM(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(self_abs_power_DISM, rhs)

  end subroutine read_self_abs_power_DISM

!#======================================================================

  subroutine read_stoch_heating(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(stoch_heating, rhs)
    stoch_heating_SFC = stoch_heating
    stoch_heating_DISM = stoch_heating

  end subroutine read_stoch_heating

!#======================================================================

  subroutine read_stoch_heating_SFC(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(stoch_heating_SFC, rhs)

  end subroutine read_stoch_heating_SFC

!#======================================================================

  subroutine read_stoch_heating_DISM(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(stoch_heating_DISM, rhs)

  end subroutine read_stoch_heating_DISM

!#======================================================================

  subroutine read_carb_sublim_temp(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(carb_sublim_temp, rhs)
    carb_sublim_temp_SFC = carb_sublim_temp
    carb_sublim_temp_DISM = carb_sublim_temp

  end subroutine read_carb_sublim_temp

!#======================================================================

  subroutine read_carb_sublim_temp_SFC(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(carb_sublim_temp_SFC, rhs)

  end subroutine read_carb_sublim_temp_SFC

!#======================================================================

  subroutine read_carb_sublim_temp_DISM(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(carb_sublim_temp_DISM, rhs)

  end subroutine read_carb_sublim_temp_DISM

!#======================================================================

  subroutine read_sil_sublim_temp(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(sil_sublim_temp, rhs)
    sil_sublim_temp_SFC = sil_sublim_temp
    sil_sublim_temp_DISM = sil_sublim_temp

  end subroutine read_sil_sublim_temp

!#======================================================================

  subroutine read_sil_sublim_temp_SFC(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(sil_sublim_temp_SFC, rhs)

  end subroutine read_sil_sublim_temp_SFC
!#======================================================================

  subroutine read_sil_sublim_temp_DISM(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(sil_sublim_temp_DISM, rhs)

  end subroutine read_sil_sublim_temp_DISM

!#======================================================================

  subroutine read_grains_file(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(grains_file, rhs)

!# ??? Put all possible `grains_file`s in a list and check if the provided `grains_file` is in this list.
    if (all(grains_file /= (/ &
         "ZDA.txt", &
         "LWD.txt", &
         "MRN.txt"/))) &
         call error_message("The value of `grains_file` should be &
         &""ZDA.txt"", ""LWD.txt"" or ""MRN.txt"".")

    grains_file_SFC = grains_file
    grains_file_DISM = grains_file

  end subroutine read_grains_file

!#======================================================================

  subroutine read_grains_file_SFC(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(grains_file_SFC, rhs)
    if (all(grains_file_SFC /= (/ &
         "ZDA.txt", &
         "LWD.txt", &
         "MRN.txt"/))) &
         call error_message("The value of `grains_file_SFC` should be &
         &""ZDA.txt"", ""LWD.txt"" or ""MRN.txt"".")

  end subroutine read_grains_file_SFC

!#======================================================================

  subroutine read_grains_file_DISM(rhs)

    implicit none
    character(len=*), intent(in) :: rhs
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    call get_val(grains_file_DISM, rhs)
    if (all(grains_file_DISM /= (/ &
         "ZDA.txt", &
         "LWD.txt", &
         "MRN.txt"/))) &
         call error_message("The value of `grains_file_DISM` should be &
         &""ZDA.txt"", ""LWD.txt"" or ""MRN.txt"".")

  end subroutine read_grains_file_DISM

!#======================================================================

  subroutine set_inclin_averaged

    implicit none

    inclin_averaged = .false.

  end subroutine set_inclin_averaged

!#======================================================================

  subroutine set_geometry(val_geometry)

    implicit none
    character(len=*), intent(in) :: val_geometry
!#::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

    geometry = val_geometry

  end subroutine set_geometry

end module mod_read_dust_transfer_param
