!# This source file is part of code Pégase.3.0.1 (2019-02-21).
!# Copyright: Michel Fioc (Michel.Fioc@iap.fr), Sorbonne université, 
!# Institut d'astrophysique de Paris/CNRS, France.
!# 
!# Pégase.3.0.1 is governed by the CeCILL license under French law and abides 
!# by the rules of distribution of free software. You can use, modify and/or 
!# redistribute this software under the terms of the CeCILL license as circulated 
!# by CEA, CNRS and INRIA at "http://www.cecill.info". The text of this license
!# is also available in French and in English in directory "doc_dir/" of this
!# code.
!# 
!# As a counterpart to the access to the source code and to the rights to copy,
!# modify and redistribute it granted by the license, users are provided only
!# with a limited warranty, and the software's author, the holder of the
!# economic rights, and the successive licensors have only limited
!# liability. 
!# 
!# The fact that you are presently reading this means that you have had
!# knowledge of the CeCILL license and that you accept its terms.
!#====================================================================== 

module mod_constants

  use mod_types
  implicit none
  private

!# Mathematical constants:
  real(CDR), parameter, public :: pi = 3.1415926535_CDR

!# Unit conversion factors
!# (A = angstroem, um = micrometer, cm = centimeter):
  real(CDR), parameter, public :: in_um_to_in_A = 1.e4_CDR, &
       per_A_to_per_um = in_um_to_in_A, &
       in_A_to_in_um = 1/in_um_to_in_A, &
       per_um_to_per_A = in_A_to_in_um
  real(CDR), parameter, public :: in_um_to_in_cm = 1.e-4_CDR, &
       per_cm_to_per_um = in_um_to_in_cm, &
       in_cm_to_in_um = 1/in_um_to_in_cm, &
       per_um_to_per_cm = in_cm_to_in_um, &
       um_to_cm = in_um_to_in_cm

!# Physical constants:
  real(CDR), parameter, public :: c_um = 2.99792458e14_CDR, &
       c_ang = c_um * in_um_to_in_A, &
       c_cm = c_um * in_um_to_in_cm !# Speed of light.
  real(CDR), parameter, public :: h_Planck = 6.62606876e-27_CDR !# Planck constant \
!# in erg.s.
  real(CDR), parameter, public :: kB = 1.3806503e-16_CDR !# Boltzmann constant \
!# in erg.K-1.
  real(CDR), parameter, public :: H_atom_mass = 1.67353e-24_CDR !# Mass of the \
!# hydrogen atom in g.
  real(CDR), parameter, public :: h_c = h_Planck*c_um
  real(DPR), parameter, public :: parsec = 3.08568e18_DPR !# 1 parsec in cm.
  real(DPR), parameter, public :: area_10pc = 4*pi*(10*parsec)**2 !# Area in cm^2 \
!# of a sphere of radius = 10 pc.

!# 1 km/s/Mpc in Myr^-1 (for `H_0`):
  real(CDR), parameter, public :: H_0_unit = 1e3*365.25*24*3600/(1e6*parsec*1e-2)*1e6

!# Wavelength of the Lyman break:
  real(CDR), parameter, public :: lambda_Lyman_break_A = 911.7643_CDR !# \
!# In angstroems (from the value of R_H given by doc hazy2 of Cloudy, Sec. 4.7).
  real(CDR), parameter, public :: lambda_Lyman_break_um = &
       lambda_Lyman_break_A * in_A_to_in_um !# In micrometers.

!# Astronomical constants:
  real(CDR), parameter, public :: M_sol = 1.989e33_CDR !# Mass of the Sun in g.
  real(CDR), parameter, public :: L_sol = 3.826e33_CDR !# Bolometric luminosity \
!# of the Sun in erg.s-1.
  real(CDR), parameter, public :: Y_prim = 0.23_CDR, Z_prim = 0 !# Primordial \
!# abundances of helium and metals (in mass fraction).

!# Chemical elements:
  integer, parameter, public :: dim_elem = 10
  character(len=std_string), dimension(dim_elem), parameter, public :: &
       elem_id = (/"O ", "C ", "Fe", "He", "N ", "Ne", "Mg", "Si", "S ", &
       "Ca"/)
  real(DPR), parameter, public :: A_C12 = 12._DPR, A_C13 = 13._DPR, &
       A_C14 = 14._DPR, A_O16 = 16._DPR, A_O17 = 17._DPR, A_O18 = 18._DPR, &
       A_Mg24 = 24._DPR, A_Si28 = 28._DPR, A_S32 = 32._DPR, A_Ca40 = 40._DPR, &
       A_Ti48 = 48._DPR, A_Fe56 = 56._DPR !# Atomic masses.

  character(len=std_string), parameter, public :: solar_abundances = "Anders_Grevesse_1989.txt"

  character(len=std_string), parameter, public :: neb_set = "neb_set.txt"

  character(len=std_string), parameter, public :: &
       spectra_list = "list_spectra.txt", &
       grain_temp_list = "list_grain_temp.txt", &
       grain_SED_list = "list_grain_SED.txt"

end module mod_constants
