!*********************************************************************************************************************************************************
!>  Module: ReadXML
!>
!>
!>  This module contains variables and subroutines for reading MAGIX instance file.
!>  Copyright (C) 2012 - 2016  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - function NumberQ:                         function is true, if the string is a number
!>      - subroutine ReadXMLInstanceFile:           read model parameter from molfits file
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    05.02.2015  Initial version
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!---------------------------------------------------------------------------------------------------------------------------------------------------------
Module ReadXML
            
    use GlobalVariables
    use CommentLineVariables

    implicit none

    contains


        !*************************************************************************************************************************************************
        !> function NumberQ
        !>
        !> This function is true, if the string is a number
        !>
        !>
        !> input variables:     string:         the string which has to be analyzed
        !>
        !> output variables:    NumberQ:        is true, if string is a number
        !>
        !>
        !> \author http://rosettacode.org/wiki/Determine_if_a_string_is_numeric#Fortran
        !>
        !> \date 06.02.2015
        !>
        function NumberQ(string)
            !< returns true if the string is a number

            implicit none
            character(len = *), INTENT(IN) :: string                                        !< the string which has to be analyzed
            logical :: NumberQ                                                              !< the function value
            real :: x                                                                       !< working variable
            integer :: e                                                                    !< working variable


            read(string, *, iostat = e) x
            NumberQ = e == 0
            return
        end function NumberQ


        !>************************************************************************************************************************************************
        !> subroutine: ReadXMLInstanceFile
        !>
        !> read MAGIX instance file containing all required user parameters
        !>
        !> input variables:     ok:                 status variable
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 09.07.2012
        !>
        subroutine ReadXMLInstanceFile(ok)

            implicit none
            integer :: ok                                                                   !< status variable
            integer :: NumberOfLines                                                        !< working variable: number of lines in instance file
            integer :: allocstatus, deallocstatus                                           !< working variable: status variables for (de)allocation
            integer :: i                                                                    !< loop variable
            integer :: j, k                                                                 !< working variables
            integer :: sim_size                                                             !< working variable: total number of freq. points
            integer :: io_err                                                               !< status variable for i/o
            real*8 :: RoundOffError                                                         !< working variable: used for length of spectrum
            character(len=1024) :: helpstring                                               !< working variable
            character(len=1024) :: line                                                     !< current line of file


            !< open instance file
            ok = 0
            if (printflag) then
                print '("Reading parameters from file .. ", $)'
            endif
            open(4444, file = trim(adjustl(InstanceFileName)), status = 'unknown')


            !< determine number of lines and number of parameters
            parameternumber = 0
            NumberOfLines = 0
            io_err = 0
            Do While (io_err == 0)
                read(4444,'(A)', iostat = io_err) line
                if (io_err == 0) then
                    NumberOfLines = NumberOfLines + 1
                    line = trim(adjustl(line))


                    !< get name of parameter
                    j = index(line, "<name>")
                    if (j > 0) then
                        parameternumber = parameternumber + 1
                    endif
                endif
            end Do
            parameternumber = parameternumber + 14


            ! Debug:
            !    print*,'NumberOfLines = ', NumberOfLines
            !    print*,'parameternumber = ', parameternumber


            !< (de-)allocate memory
            if (allocated(FitParameterName)) then
                deallocate(FitParameterName, FitParameterValue, paramset, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine ReadXMLInstanceFile!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables FitParameterName and FitParameterValue!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            allocate(FitParameterName(parameternumber), FitParameterValue(parameternumber), paramset(4, parameternumber), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannel = 6, 6
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ReadXMLInstanceFile!")')
                    write(ErrChannel, '(3x,"Can not allocate variables FitParameterName and FitParameterValue!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            FitParameterName = ""
            FitParameterValue = ""
            paramset = 0.d0


            !< go back to the beginning of the file
            rewind 4444


            !< analyze instance file
            k = 0
            Do i = 1, NumberOfLines
                read(4444, '(A)', iostat = io_err) line
                line = trim(adjustl(line))


                !< get name of parameter
                j = index(line, "<name>")
                if (j > 0) then
                    k = k + 1
                    helpstring = line(j + 6:)
                    j = index(helpstring, "</name>")
                    helpstring = helpstring(:j - 1)
                    FitParameterName(k) = trim(adjustl(helpstring))

                    ! Debug:
                    ! print*,'FitParameterName(k) = >>', trim(adjustl(FitParameterName(k))), '<<'
                endif


                !< get value of parameter
                j = index(line, "<value>")
                if (j > 0) then
                    helpstring = line(j + 7:)
                    j = index(helpstring, "</value>")
                    helpstring = helpstring(:j - 1)
                    FitParameterValue(k) = trim(adjustl(helpstring))

                    ! Debug:
                    ! print*,'FitParameterValue(k) = >>', trim(adjustl(FitParameterValue(k))), '<<'


                    !< define paramset variable
                    if (NumberQ(trim(FitParameterValue(k)))) then
                        read(FitParameterValue(k), *) paramset(1, k)


                        !< special handling of column density
                        !    if (trim(adjustl(FitParameterName(k))) == "N_tot") then
                        !        if (paramset(1, k) > 300.d0) then
                        !            paramset(1, k) = 300.d0
                        !        endif
                        !        paramset(1, k) = 10.d0**paramset(1, k)
                        !    endif

                        ! Debug:
                        !    print*,'FitParameterValue(k) = >>', trim(adjustl(FitParameterValue(k))), '<<'
                        !    print*,'paramset(1, k) = ', paramset(1, k)
                    endif
                endif
            end Do
            close(4444)


            !< store "MAGIXImportExpFileNames"
            k = k + 1
            FitParameterName(k) = "MAGIXImportExpFileNames"
            FitParameterValue(k) = ""


            !< store "MAGIXImportStartFrequency"
            k = k + 1
            FitParameterName(k) = "MAGIXImportStartFrequency"
            write(FitParameterValue(k), '(F25.15)') freqmin
            paramset(1, k) = freqmin


            !< store "MAGIXImportEndFrequency"
            k = k + 1
            FitParameterName(k) = "MAGIXImportEndFrequency"
            write(FitParameterValue(k), '(F25.15)') freqmax
            paramset(1, k) = freqmax


            !< store "MAGIXImportStepFrequency"
            k = k + 1
            FitParameterName(k) = "MAGIXImportStepFrequency"
            write(FitParameterValue(k), '(F25.15)') sim_width
            paramset(1, k) = sim_width


            !< store "MAGIXImportTBackFlag"
            k = k + 1
            FitParameterName(k) = "MAGIXImportTBackFlag"
            if (tback_flag) then
                paramset(1, k) = 1.d0
            else
                paramset(1, k) = 0.d0
            endif


            !< store "MAGIXImportBackgroundTemperature"
            k = k + 1
            FitParameterName(k) = "MAGIXImportBackgroundTemperature"
            write(FitParameterValue(k), '(F25.15)') T_back
            paramset(1, k) = T_back


            !< store "MAGIXImportTemperatureSlope"
            k = k + 1
            FitParameterName(k) = "MAGIXImportTemperatureSlope"
            write(FitParameterValue(k), '(F25.15)') T_slope
            paramset(1, k) = T_slope


            !< store "MAGIXImportHydrogenColumnDensity"
            k = k + 1
            FitParameterName(k) = "MAGIXImportHydrogenColumnDensity"
            write(FitParameterValue(k), '(F25.15)') N_H
            paramset(1, k) = N_H


            !< store "MAGIXImportDustBeta"
            k = k + 1
            FitParameterName(k) = "MAGIXImportDustBeta"
            write(FitParameterValue(k), '(F25.15)') beta_dust
            paramset(1, k) = beta_dust


            !< store "MAGIXImportKappa"
            k = k + 1
            FitParameterName(k) = "MAGIXImportKappa"
            write(FitParameterValue(k), '(F25.15)') kappa_1300
            paramset(1, k) = kappa_1300


            !< store "MAGIXImportGlobalvLSR"
            k = k + 1
            FitParameterName(k) = "MAGIXImportGlobalvLSR"
            write(FitParameterValue(k), '(F25.15)') LocalvLSR
            paramset(1, k) = LocalvLSR


            !< store "MAGIXImportTelescopeSize"
            k = k + 1
            FitParameterName(k) = "MAGIXImportTelescopeSize"
            write(FitParameterValue(k), '(F25.15)') telescope_size
            paramset(1, k) = telescope_size


            !< store "MAGIXImportInterferrometerFlag"
            k = k + 1
            FitParameterName(k) = "MAGIXImportInterferrometerFlag"
            if (inter_flag) then
                paramset(1, k) = 1
            else
                paramset(1, k) = 0
            endif


            !< store "MAGIXImportdbFilename"
            k = k + 1
            FitParameterName(k) = "MAGIXImportdbFilename"
            FitParameterValue(k) = trim(adjustl(dbName))
            paramset(1, k) = 0.0


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< create expdatax arrays
            sim_size = int((freqmax - freqmin) / sim_width) + 1                             !< determine number of frequency points
            RoundOffError = dabs((freqmax - freqmin)/sim_width)                             !< check if sim_size is to short due to round off errors
            if (dabs(RoundOffError - real(int(RoundOffError))) > 0.99) then
                sim_size = sim_size + 1
            endif


            !< allocate memory for MAGIX arrays
            if (allocated(expdatax)) then
                deallocate(expdatax, expdatay, expdatae, lengthexpdata, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine ReadXMLInstanceFile!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables expdatax and expdatay!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            allocate(expdatax(1, sim_size, 1), expdatay(1, sim_size, 1), expdatae(1, sim_size, 1), lengthexpdata(1), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannel = 6, 6
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ReadXMLInstanceFile!")')
                    write(ErrChannel, '(3x,"Can not allocate variables expdatax and expdatay!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            expdatax = 0.d0
            expdatay = 0.d0
            expdatae = 0.d0
            lengthexpdata = sim_size
            NumberExpFiles = 1
            NumberFreeParameter = 0


            !< construct expdatax array
            Do i = 1, sim_size
                expdatax(1, i, 1) = freqmin + (i - 1) * sim_width
            end Do


            !< we've done
            if (printflag) then
                print '("done!")'
                print '(" ")'
                print '(" ")'
            endif


            return
        end subroutine ReadXMLInstanceFile
end Module ReadXML
