!*********************************************************************************************************************************************************
!>  Module: CommentLineVariables
!>
!>
!>  This module contains variables and subroutines for reading in parameters from the comment line
!>  Copyright (C) 2012 - 2016  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine getCommandLineArguments:       get command line arguments
!>      - subroutine getArgument:                   save value of the command-line argument
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    28.06.2012  Initial version
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!---------------------------------------------------------------------------------------------------------------------------------------------------------
Module CommentLineVariables

    use GlobalVariables

    integer :: NumberProc                                                                   !< number of threads
    real*8 :: freqmin                                                                       !< first frequency of the simulated spectrum
    real*8 :: freqmax                                                                       !< last frequency of the simulated spectrum
    real*8 :: sim_width                                                                     !< step frequency of the simulated spectrum
    real*8 :: telescope_size                                                                !< size of the telescope
    real*8 :: LocalvLSR                                                                     !< vLSR
    real*8 :: T_back                                                                        !< background temperature
    real*8 :: T_slope                                                                       !< temperature slope
    real*8 :: N_H                                                                           !< hydrogen column density
    real*8 :: beta_dust                                                                     !< beta for dust
    real*8 :: kappa_1300                                                                    !< kappa for dust
    real*8 :: rms_mod                                                                       !< ??
    integer :: PeakShape                                                                    !< shape of peak
    logical :: inter_flag                                                                   !< flag for indicating interferometric data
    logical :: tback_flag                                                                   !< flag indicating that T_Back and T_Slope describes
                                                                                            !< continuum completely
    logical :: iso_flag                                                                     !< flag for using isotopologues or not
    logical :: verbose                                                                      !<
    logical :: old                                                                          !<

    contains

        !>************************************************************************************************************************************************
        !> subroutine: getCommandLineArguments
        !>
        !> get command line arguments
        !>
        !> input variables:     none
        !>
        !> output variables:    ok:             status variable
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 30.05.2012
        !>
        subroutine getCommandLineArguments(ok)
            
            use GlobalVariables

            implicit none
            integer, parameter :: debug = 1                                                 !< for debugging
            integer :: ok                                                                   !< status variable
            integer :: offset                                                               !< offset
            integer :: NumberOfCommandLineArguments                                         !< number of command-line arguments
            character(len=4096) :: CommandLineArgument                                      !< command-line argument


            ok = 0                                                                          !< reset status variable
            NumberOfCommandLineArguments = iargc()                                          !< get number of command-line arguments
            if (NumberOfCommandLineArguments < 1) then                                      !< continue if 
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '("Error in program myXCLASS:")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(2x, "No command-line arguments are specified!")')
                    write(ErrChannel, '(2x, "Please start myXCLASS with command-line arguments!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '("Abort myXCLASS")')
                end Do
                ok = 1
            else
                !< print what you do
                if (printflag) then
                    print '(" ")'
                    print '("Reading command line parameters .. ", $)'
                endif


                !< get freqmin
                freqmin = 0.d0
                call getArgument(1, CommandLineArgument)
                read(CommandLineArgument,*) freqmin

                ! Debug:
                ! print*,'freqmin = ', freqmin


                !< get freqmax
                freqmax = 0.d0
                call getArgument(2, CommandLineArgument)
                read(CommandLineArgument,*) freqmax

                ! Debug:
                ! print*,'freqmax = ', freqmax


                !< get stepsize
                sim_width = 0.d0
                call getArgument(3, CommandLineArgument)
                read(CommandLineArgument,*) sim_width

                ! Debug:
                ! print*,'sim_width = ', sim_width


                !< get telsize
                telescope_size = 0.d0
                call getArgument(4, CommandLineArgument)
                read(CommandLineArgument,*) telescope_size
                if (telescope_size == 0)  then
                    telescope_size = 10.4d0
                endif

                ! Debug:
                ! print*,'telescope_size = ', telescope_size


                !< get inter_flag
                inter_flag = .false.
                call getArgument(5, CommandLineArgument)
                if (CommandLineArgument == "t" .or. CommandLineArgument == "T" .or. CommandLineArgument == "true" .or. CommandLineArgument == "True" &
                    .or. CommandLineArgument == "TRUE" .or. CommandLineArgument == "y" .or. CommandLineArgument == "Y" .or. CommandLineArgument == "yes" &
                    .or. CommandLineArgument == "YES" .or. CommandLineArgument == "Yes") then
                    inter_flag = .true.
                endif

                ! Debug:
                ! print*,'inter_flag = ', inter_flag


                !< get t_back flag
                tback_flag = .false.
                call getArgument(6, CommandLineArgument)
                if (CommandLineArgument == "t" .or. CommandLineArgument == "T" .or. CommandLineArgument == "true" .or. CommandLineArgument == "True" &
                    .or. CommandLineArgument == "TRUE" .or. CommandLineArgument == "y" .or. CommandLineArgument == "Y" .or. CommandLineArgument == "yes" &
                    .or. CommandLineArgument == "YES" .or. CommandLineArgument == "Yes") then
                    tback_flag = .true.
                endif

                ! Debug:
                ! print*,'tback_flag = ', tback_flag


                !< get tBack
                T_back = 0.d0
                call getArgument(7, CommandLineArgument)
                read(CommandLineArgument,*) T_back

                ! Debug:
                ! print*,'T_back = ', T_back


                !< get tslope
                T_slope = 0.d0
                call getArgument(8, CommandLineArgument)
                read(CommandLineArgument,*) T_slope

                ! Debug:
                ! print*,'T_slope = ', T_slope


                !< get N_H flag
                nHFlagCommLine = .false.
                call getArgument(9, CommandLineArgument)
                if (CommandLineArgument == "t" .or. CommandLineArgument == "T" .or. CommandLineArgument == "true" .or. CommandLineArgument == "True" &
                    .or. CommandLineArgument == "TRUE" .or. CommandLineArgument == "y" .or. CommandLineArgument == "Y" .or. CommandLineArgument == "yes" &
                    .or. CommandLineArgument == "YES" .or. CommandLineArgument == "Yes") then
                    nHFlagCommLine = .true.
                endif

                ! Debug:
                ! print*,'nHFlagCommLine = ', nHFlagCommLine


                !< get value of N_H and kappa if N_H flag is true
                offset = 0
                N_H = 0.d0
                beta_dust = 0.d0
                kappa_1300 = 0.d0
                if (nHFlagCommLine) then
                    call getArgument(10, CommandLineArgument)
                    read(CommandLineArgument,*) N_H

                    ! Debug:
                    ! print*,'N_H = ', N_H


                    !< get beta_dust
                    call getArgument(11, CommandLineArgument)
                    read(CommandLineArgument,*) beta_dust

                    ! Debug:
                    ! print*,'beta_dust = ', beta_dust


                    !< get kappa_1300
                    call getArgument(12, CommandLineArgument)
                    read(CommandLineArgument,*) kappa_1300
                    offset = offset + 3

                    ! Debug:
                    ! print*,'kappa_1300 = ', kappa_1300
                endif


                !< get vLSR parameter
                LocalvLSR = 0.d0
                call getArgument(10 + offset, CommandLineArgument)
                read(CommandLineArgument,*) LocalvLSR


                !< get name (and path) of molfits file
                InstanceFileName = ""
                call getArgument(11 + offset, InstanceFileName)

                ! Debug:
                ! print*,'InstanceFileName = ', trim(adjustl(InstanceFileName))


                !< get iso flag
                iso_flag = .false.
                call getArgument(12 + offset, CommandLineArgument)
                if (CommandLineArgument == "t" .or. CommandLineArgument == "T" .or. CommandLineArgument == "true" .or. CommandLineArgument == "True" &
                    .or. CommandLineArgument == "TRUE" .or. CommandLineArgument == "y" .or. CommandLineArgument == "Y" .or. CommandLineArgument == "yes" &
                    .or. CommandLineArgument == "YES" .or. CommandLineArgument == "Yes") then
                    iso_flag = .true.
                endif

                ! Debug:
                ! print*,'iso_flag = ', iso_flag


                !< get number of threads
                NumberProc = 1
                call getArgument(13 + offset, CommandLineArgument)
                read(CommandLineArgument,*) NumberProc


                !< determine flag for output files (the following lines are included to avoid introducing a further command line argument
                if (NumberProc < 0) then
                    AllOutputFilesFlag = .true.                                             !< write intensities and optical depths of each comp. to file
                endif


                ! Debug:
                ! print*,'NumberProc = ', NumberProc


                !< (optional) get path and name of the sqlite database
                if ((13 + offset) <= IARGC()) then                                          !< read in path if path is given
                    dbName = ""
                    call getArgument(14 + offset, dbName)
                    dbName = trim(adjustl(dbName))
                else
                    dbName = "../../../Database/cdms_sqlite.db"
                endif


                !< print some status information to screen
                if (printflag) then
                    print '("done!")'
                    print '(" ")'
                    print '(2x, "Frequency minimum = ",ES25.15)', freqmin
                    print '(2x, "Frequency maximum = ",ES25.15)', freqmax
                    print '(2x, "Frequency stepsize = ",ES25.15)', sim_width
                    print '(2x, "telescope size = ",ES25.15)', telescope_size
                    print '(2x, "Interferrometer flag = ",L1)', inter_flag
                    print '(2x, "Background temperature flag = ",L1)', tback_flag
                    print '(2x, "Background temperature = ",ES25.15)', T_back
                    print '(2x, "Temperature slope = ",ES25.15)', T_slope
                    print '(2x, "N_H flag = ",L1)', nHFlagCommLine
                    if (nHFlagCommLine) then
                        print '(2x, "N_H = ",ES25.15)', N_H
                        print '(2x, "beta dust = ",ES25.15)', beta_dust
                        print '(2x, "kappa = ",ES25.15)', kappa_1300
                    endif
                    print '(2x, "v_LSR = ",ES25.15)', LocalvLSR
                    print '(2x, "Instance filename = ",A)', trim(adjustl(InstanceFileName))
                    print '(2x, "Iso flag = ",L1)', iso_flag
                    print '(2x, "dbName = ",A)', trim(adjustl(dbName))
                    print '(" ")'
                    print '(" ")'

                    ! Debug
                    ! print '(2x, "Number of command line arguments = ", I3)', IARGC()
                endif
            endif
            return
        end subroutine getCommandLineArguments


        !>************************************************************************************************************************************************
        !> subroutine: getArgument
        !>
        !> save value of the command-line argument
        !>
        !> input variables:     NumArg:                 index of the command-line argument
        !>
        !> output variables:    CommandLineArgument:    contents of the NumArg-th argument
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 30.05.2012
        !>
        subroutine getArgument(NumArg, CommandLineArgument)

            implicit none
            integer :: NumArg                                                               !< output variable: index of the command-line argument
            character(len=4096) :: CommandLineArgument                                      !< command-line argument

            CommandLineArgument = ""
            if (IARGC() >= NumArg) then                                                     !< is the number of command line arguments big enough?
                call getarg(NumArg, CommandLineArgument)
            endif
            return
        end subroutine getArgument
end Module CommentLineVariables
