#!/usr/bin/env python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module starts the main program
##  Copyright (C) 2009 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following python modules are used:
##
##      - numpy
##      - string
##      - shutil
##      - sys
##      - os.path
##
##      - MainProgram
##      - LoadExpFile       -> pyfits
##      - MAGIXxmlPACKAGE   -> xml.dom.minidom
##                          -> traceback
##      - FittingEngine     -> PlotData     -> Gnuplot
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    22.06.2009   initial version
##  T. Moeller    16.01.2012   improve documentation of source code
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************

"""
MAGIX:
------

Copyright (C) 2010
I. Physikalisches Institut, Universitaet zu Koeln
Produced for the CATS project


The modeling of astronomical observations requires specialized numerical codes 
and knowledge about how to use them. The program MAGIX provides a framework to
easily interface existing codes to an iterating engine that allows automatic 
minimizations to constrain model parameters and provide error estimates.

MAGIX can be started by simply typing "./magix_start.py" and the command prompt.
In order to get help type "./magix_start.py --help" or read the readme.txt file.

If no screen output is desired, type "./magix_start.py --quiet" followed by the
path of the i/o control file (see below).

If you do not want to plot the resulting fit function or if you have problems
with the installation of the matplotlib package, type "./magix_start.py --noplot"
followed by the path of the i/o control file (see below).

MAGIX looks for the directory "Modules" in the current working directory, which
contains all necessary python and fortran modules. Do not rename or move this
directory.

MAGIX consists of four modules:

- Read experimental data using import filters for .dat and .fits files
- Read model parameters from xml-file defining model ranges, start values
  and fit attributes. (Fit attributes declare, if a model parameter should be
  optimized by the fit process of hold constant.)
- Start the fit process using several fit algorithm (Levenberg-Marquardt).
- Write optimized parameter to xml-file and plot data.

In the following details of the several modules are given together with an
description of the files necessary for MAGIX:
"""

##********************************************************************* load packages ********************************************************************
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import random                                                                               ## import random package


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## get current directory
CurrentDir = os.getcwd() + "/"


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## expand sys.path
modules_path = [CurrentDir + "Modules/magix-parts_python",
                CurrentDir + "Modules/Additional_Packages/bin",
                CurrentDir + "Modules/ErrEstim_INS/bin",
                CurrentDir + "Modules/Interval_Nested-Sampling/bin",
                CurrentDir + "Modules/MCMC/bin",
                CurrentDir + "Modules/MCMC/emcee"]
for NewModulesPath in modules_path:                                                         ## loop over all new module paths
    already_included_flag = "false"
    for entries in sys.path:
        if (entries == NewModulesPath):
            already_included_flag = "true"
            break
    if (already_included_flag == "false"):
        sys.path.append(NewModulesPath)


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## import magix packages
import MainProgram                                                                          ## import package for experimental file
import LoadExpFile                                                                          ## import package for experimental file
import GetParameterSet                                                                      ## import package for reading fit parameters
import FittingEngine                                                                        ## import package containing the fitting engine


# Debug:
# print "sys.argv = >>%s<<" % sys.argv
# sys.exit(0)


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## read commands from command line
printflag = "false"
plotflag = "true"
debugflag = "false"
modelflag = "false"
if (len(sys.argv) > 1):                                                                     ## check, if command line arguments are defined
    for argument in sys.argv[1:]:                                                           ## loop over all command line arguments


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get run flags
        if (argument.startswith('--')):                                                     ## run flags are marked with "--" characters
            option = argument[2:].strip()                                                   ## remove "--" characters and leading and tailing blanks
            option = option.lower()


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## write version number
            if (option == "version"):
                print 'Version 1.0'


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## write help to screen
            elif (option == "help"):
                print __doc__
                print MainProgram.__doc__
                print LoadExpFile.__doc__
                print GetParameterSet.__doc__
                print FittingEngine.__doc__
                sys.exit(0)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## write short help to screen
            elif (option == "help_short"):
                print " "
                print "MAGIX quick help:"
                print "-----------------"
                print " "
                print "You need:"
                print " "
                print "(1.) i/o control file,"
                print "(2.) xml-file containing import settings for observal data,"
                print "(3.) xml-file with model parameters (fit attribute, value, name, upper and lower limit),"
                print "(4.) xml-file including all settings for the fit process."
                print " "
                print "For more help type './magix_start.py --help' or read the pdf-files within the Documentation directory."
                print " "
                sys.exit(0)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get debug flag
            elif (option == "debug"):
                debugflag = "true"


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get model flag
            elif (option.startswith("model")):
                modelflag = "true"
                printflag = "true"
                SplitModel = option.split("=")
                if (len(SplitModel) > 1):
                    modelflag = SplitModel[1]


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get quiet flag
            elif (option == "quiet"):
                plotflag = "saveonly"
                printflag = "false"


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get noplot flag
            elif (option == "noplot"):
                plotflag = "false"


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get plotsaveonly flag
            elif (option == "plotsaveonly"):
                plotflag = "saveonly"


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read path and name of the i/o control file
        else:
            filename_control = argument.strip()
            if not(os.path.exists(filename_control)):
                print " "
                print "\t  Error in command line input!"
                print "\t      The file ",filename_control
                print "\t      does not exists!"
                print " "
                print "\t      Please correct your input and"
                print "\t      restart the program !"
                print " "
                sys.exit(0)

else:
    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## set default setting for i/o control file if no argument is sepcified
    filename_control = "run/io_control.txt"
    printflag = "true"
    plotflag = "true"
    if not(os.path.exists(filename_control)):
        print " "
        print "\t  Error in io_control.txt file!"
        print "\t      The file ",filename_control
        print "\t      does not exists!"
        print " "
        print "\t      Enter the path and the file name of the"
        print "\t      io_control.txt-file at the command line:"
        print "\t      For example:"
        print "\t      ./magix_start.py test/io_control.txt"
        print " "
        sys.exit(0)


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## test, if matplotlib is available
if (plotflag == "true" or plotflag == "saveonly"):
    try:
        import matplotlib
    except ImportError, err:
        plotflag == "false"

# Debug:
# print "printflag = ", printflag
# print "plotflag = ", plotflag
# print "debugflag = ", debugflag
# print "modelflag = ", modelflag
# print "filename_control = ", filename_control


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## determine job id, i.e. create a number to identify job unambiguous
pid = os.getpid()
ranNum = int(random.uniform (1000, 9999))
jobID = str(pid) + str(ranNum)


##--------------------------------------------------------------------------------------------------------------------------------------------------------
## start MAGIX main routine
MAGIXrootDirectory = CurrentDir
ok = MainProgram.main(printflag, plotflag, debugflag, modelflag, filename_control, jobID, MAGIXrootDirectory)
if (ok != 0):
    print "\t Program MAGIX aborted!"

# Debug:
# print "MAGIX finished"


##
##
## program finished
##
##
##--------------------------------------------------------------------------------------------------------------------------------------------------------

