#!/usr/bin/env python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module reads and writes the model parameter to a xml-file.
##  Copyright (C) 2009 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following subroutines and functions are included in this module:
##
##      - class LoadExp:                                class for loading experimental-file
##      - subroutine LoadExp.__init__:                  initialize variables of class LoadExp
##      - subroutine GetMinMaxFitsFile:                 gets min. and max. frequency of fits file
##      - subroutine LoadExp.xml:                       load xml-file containing settings for the import of experimental-file(s)
##      - subroutine LoadExp.LoadFile:                  call import routines for different file formats
##      - subroutine LoadExp.LoadDat:                   load contents of ascii-file(s)
##      - subroutine LoadExp.LoadFits:                  load contents of fits-file(s)
##      - subroutine LoadExp.WriteFile:                 call export routines for different file formats
##      - subroutine LoadExp.WriteDat:                  write contents to ascii-file(s)
##      - subroutine LoadExp.WriteFits:                 write contents to fits-file(s)
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    14.07.2009   initial version
##  T. Moeller    16.01.2012   improve documentation of source code
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************

"""
Read Experimental Data:
-----------------------

The package "LoadExpFile" contains subroutines for loading experimental
files. Only experimental data, which are given in the following file
formats, can be loaded by MAGIX:

- dat-files
- fits-files.

Before the experimental file(s) is (are) loaded, the xml-file containing all 
settings required for the import of the experimental data has to be loaded.
Especially, the number of experimental files, the path and the name for
each experimental file has to be defined in the xml-file. If no xml-file is
specified and the given filename points directly toward a fits or a 
experimental-file the default settings defined below are used and a warning 
message is printed to the screen. By using the xml-descriptions, the user can
specify several (!) ranges of the experimental data which are considered in 
the fitting process. Thus, there is no need to load all experimental data. 

In the following, the expression "x"-column means the columns of an array
defining the "x" position of the experimental data point. For example: For a
function f(x,y,z) the number of "x"-column is 3, for a function f(x1, x2) the
number of "x" columns is 2. Additionally, for dat-files it may be necessary
to specify the number of "y" columns. This means, that for a given 
"x"-position in the experimental data, you can specify several "y"-values. For
example: You measured several spectra under different polarization angles at
the same frequencies. A line in the corresponding dat-file may look like:

100.12,  0.34134,  0.12341,  0.78901,  0.13361

Here, the first column describe the frequency and the other columns the
transmission at different polarization angles. The number of "x" columns
is 1 and the number of "y"-columns is 4.

Note, that for fits-files, the number of "y"-columns is always set to 1!

Declaration of ranges: If only one x-column exists, the min and max of the 
range is a simple scalar. But if the number of x-columns is greater 1 the 
several indicators of the x-columns have to be separated by the "," character.
For example:

    <NumberColumnsX>3</NumberColumnsX>:
            <MinExpRange>0, 0, 0</MinExpRange>
            <MaxExpRange>2000, 100, 20</MaxExpRange>

In addition, for a dat-file import the user can specify the separator 
character separating the columns from each other for each file and the number 
of columns starting form the left column which belong to the x point of each 
experimental data. Furthermore, the user can specify the number of columns 
belonging to the f(x) point of each experimental data.

In case of xml-files, the user has to specify the HDU (header Data Unit) which
should be loaded.


Examples:
---------


Structure of xml-file for fits-file import:
-------------------------------------------

<ExpFiles>
    <NumberExpFiles>2</NumberExpFiles>
    <file>
        <FileNamesExpFiles>one_parameter_free/File3.fits</FileNamesExpFiles>
        <ImportFilter>fits</ImportFilter>
        <NumberHDU>0</NumberHDU>
        <NumberExpRanges>1</NumberExpRanges>
        <MinExpRange>0</MinExpRange>
        <MaxExpRange>1000</MaxExpRange>
    </file>
    <file>
        <FileNamesExpFiles>one_parameter_free/File4.fits</FileNamesExpFiles>
        <ImportFilter>fits</ImportFilter>
        <NumberHDU>0</NumberHDU>
        <NumberExpRanges>2</NumberExpRanges>
        <MinExpRange>0</MinExpRange>
        <MaxExpRange>2000</MaxExpRange>
        <MinExpRange>3130</MinExpRange>
        <MaxExpRange>3200</MaxExpRange>
    </file>
</ExpFiles>



Structure of xml-file for dat-file import:
------------------------------------------

<ExpFiles>
    <NumberExpFiles>2</NumberExpFiles>
    <file>
        <FileNamesExpFiles>one_parameter_free/File1.dat</FileNamesExpFiles>
        <ImportFilter>ascii</ImportFilter>
        <NumberHeaderLines>0</NumberHeaderLines>
        <SeparatorColumns> </SeparatorColumns>
        <NumberColumnsX>1</NumberColumnsX>
        <NumberColumnsY>1</NumberColumnsY>
        <ErrorY>yes</ErrorY>
        <NumberExpRanges>1</NumberExpRanges>
        <MinExpRange>0</MinExpRange>
        <MaxExpRange>1000</MaxExpRange>
    </file>
    <file>
        <FileNamesExpFiles>one_parameter_free/File2.dat</FileNamesExpFiles>
        <ImportFilter>ascii</ImportFilter>
        <NumberHeaderLines>0</NumberHeaderLines>
        <SeparatorColumns> </SeparatorColumns>
        <NumberColumnsX>1</NumberColumnsX>
        <NumberColumnsY>1</NumberColumnsY>
        <ErrorY>no</ErrorY>
        <NumberExpRanges>2</NumberExpRanges>
        <MinExpRange>0</MinExpRange>
        <MaxExpRange>2000</MaxExpRange>
        <MinExpRange>3130</MinExpRange>
        <MaxExpRange>3200</MaxExpRange>
    </file>
</ExpFiles>


IMPORTANT:
----------

- The names of the tags <ExpFiles>, <NumberExpFiles> etc. have
  to be written in the same type as presented in the above example.

- Using the tag <ImportFilter> the user can select a specific
  import filter. Setting the import filter to "automatic", the
  format of the experimental file is chosen by the ending of the
  file name. ".fits" for fits-file, ".dat" for ASCII files.

- Please, do not mix upper and lower case in the name of the tags!!

- The tag <file></file> must not occur more or less than the
  number of experimental files defined in the tag
  <NumberExpFiles></NumberExpFiles> !!!!!

- The number of ranges <NumberExpRanges></NumberExpRanges>
  must always be given! If no range desired (all data should
  be included in the fit process), the number of ranges must
  be set to 0.

- If the number of ranges is set to 0 the tags 
  <MinExpRange></MinExpRange> and 
  <MaxExpRange></MaxExpRange> need not to be set.
  BUT: All other tags have to be set !!!!

- If the user do not want to consider all data (number of ranges
  set to > 1) the tags <MinExpRange></MinExpRange> and
  <MaxExpRange></MaxExpRange> have to occur as many times as
  given by <NumberExpRanges></NumberExpRanges>

- If the number of "x"-columns is greater than 1, the min and max of
  the "x"-columns of the ranges have to be separated by the "," character.
  For example:

    <NumberColumnsX>1</NumberColumnsX>:
            <MinExpRange>0</MinExpRange>
            <MaxExpRange>2000</MaxExpRange>

    <NumberColumnsX>3</NumberColumnsX>:
            <MinExpRange>0, 0, 0</MinExpRange>
            <MaxExpRange>2000, 100, 20</MaxExpRange>

  Although the <NumberColumnsX></NumberColumnsX> tag is ignored in the
  import of fits-file, the content of this tag is defined by the dimension
  of the fits-file. Thus, the settings for the ranges have to be given
  as described above, if the dimension of the experimental file is larger
  than 1.

- Note, that the xml-file differ by some tags for dat and fits
  file import !!

- (fits-file): For the import of fits-file the user has to specify the
  Header Data Unit (HDU) for each fits file.

- (dat-file): The y-column(s) have to be close to the x-column(s).

  Example:    |<- x-columns ->|<- y-column ->|
                212.124  23.314    1210.643216

- (dat-file): The tag <NumberHeaderLines> defines the number of
  header lines which should be ignored in the import of the 
  dat-files.

- (dat-file): The tag <SeparatorColumns> defines the character
  which separate the columns from each other.

- (dat-file): The tag <NumberColumnsX> defines the number of
  columns counted from the left side, which belongs to the
  "x"-points of the experimental data.
  For example: If the user wants to import a dat-file 
               containing 3D data, the first 3 columns define
               the x, y, and z position. For that kind of data
               the tag <NumberColumnsX> has to be set to 3.

- (dat-file): The tag <NumberColumnsY> defines the number of
  columns, which belongs to the "y"-points of the experimental
  data. These "y"-columns have to be close to the "x" values!
  For example: If the user wants to import a dat-file which
               contains at every given "x"-point 4 differnet
               values of the measured function (i.e. that there
               are 4 different dat-files in one dat-file) the
               tag <NumberColumnsY> has to be set to 4.

  Example:    |<- x-columns ->|<- y-column ->|
                212.124  23.314    1210.643216

  Here, <NumberColumnsY> has to be set to 1.

- (dat-file): If the Error tag <ErrorY>yes</ErrorY> is set to
  yes the columns containing the errors have to be close to the
  y-columns. Additionally, the number of these "error" columns
  have to equal to the number of "y"-columns given in the tag
  <NumberColumnsY></NumberColumnsY>

- An example structure of a dat-file is given below:

  Structure of a sample dat-file:
  -------------------------------
                                                                        ONLY RELEVANT IF ErrorY = "YES"
  1           ...    NumberColumnsX   1         ...    NumberColumnsY   1         ...    NumberColumnsY
  100.2313           20.6578          0.5846           1.4218           0.02             0.01
  102.2463           21.7548          0.5947           1.5432           0.03             0.09
  .                  .                .                .                .                .
  .                  .                .                .                .                .
  .                  .                .                .                .                .

"""


##******************************************************************** load packages *********************************************************************
import numpy                                                                                ## load numpy package
import copy                                                                                 ## load copy package
import os                                                                                   ## load os package
import os.path                                                                              ## load package for file management
import string                                                                               ## load string package for manipulating strings
import sys                                                                                  ## load sys package
import math                                                                                 ## load math package
import MAGIXxmlPACKAGE                                                                      ## load MAGIXxmlPACKAGE package


## import pyfits package via astropy
try:
    import astropy.io.fits as pyfits
except ImportError:
    import pyfits
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
class LoadExp:
    ## class for loading experimental-file
    """

    This class contains the function for loading the xml-file and the corresponding exp-file.

    """

    InputFormat = "fits"                                                                    ## format of the experimental file
    NumberExpFiles = 0                                                                      ## number of exp files
    FileNamesExpFiles = []                                                                  ## path and file name of each experimental file
    NumberHeaderLines = []                                                                  ## number of header lines for each experimental-file
    FitsTyp = []                                                                            ## type of fits data
    NumberHDU = []                                                                          ## number of HDU for each fits-file
    SeparatorColumns = []                                                                   ## column separator for each experimental-file
    NumberColumnsX = []                                                                     ## number of columns belonging to the x-positions
    NumberColumnsY = []                                                                     ## number of columns belonging to the y-positions
    ErrorY = []                                                                             ## flag if error of the experimental data is given
    NumberExpRanges = []                                                                    ## number of selected experimental ranges
    MinExpRange = []                                                                        ## min of each experimental range
    MaxExpRange = []                                                                        ## max of each experimental range


    ##****************************************************************************************************************************************************
    ##
    ## initialize class variables
    ##
    def __init__(self, printflag):
        self.FITSHeaderAllLengths = []                                                      ## list containing lengths for all dim. and files
        self.FITSHeaderAllValueOfFirstPoint = []                                            ## list containing all first values of all dim. and files
        self.FITSHeaderAllDimensions = []                                                   ## list containing all dimensions of all obs. data files
        self.FITSHeaderAllCRPIX = []                                                        ## list containing CRPIX for all dim. and files
        self.FITSHeaderAllCRVAL = []                                                        ## list containing CRVAL for all dim. and files
        self.FITSHeaderAllCDELT = []                                                        ## list containing CDELT for all dim. and files
        self.FITSHeaderAllCROT = []                                                         ## list containing CROT for all dim. and files
        self.FITSHeaderAllCUNIT = []                                                        ## list containing CUNIT for all dim. and files
        self.PixelTolerance = []                                                            ## list containing pixel tolerances for all dim. and files
        self.PixelTolScalFactor = 0.1                                                       ## factor used for compute pixel tolerance
        self.TransposeFlag = []                                                             ## list containing transpose flag for all obs. data files
        self.OldFITSCommandFlag = True                                                      ## select FITS command for header update
        self.UseCentralPixelFlag = False                                                    ## select kind of header


        ## print what you do
        if (printflag == "true"):
            print "Import experimental file using settings declared in the xml-file."
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## gets min. and max. frequency of fits file
    ##
    def GetMinMaxFitsFile(self, filename, HDU):
        """

        input parameters:       filename:                   path and name of FITS file
                                HDU:                        number of HDU

        output parameters:      LowestDataPoint:            lowest point of the first axis
                                HighestDataPoint:           highest point of the first axis

        working variables:      k:                          loop counter
                                filename:                   temp variable for file name of the ith fits-file
                                hdulist:                    temp variable containing the whole ith fits-file
                                fitsdata:                   temp variable containing the data of the ith fits-file
                                HDU:                        temp variable for number of HDU in the ith fits-file
                                number_columns:             temp variable containing the total number of columns in the ith fits-file
                                dimension:                  temp variable containing the dimension of the data of the ith fits-file
                                NumDimensions:              number of dimensions
                                fitshead:                   header of the ith fits file

        """


        ## initialize output parameters
        ok = 0
        length = []
        CRPIXList = []
        CRVALList = []
        CDELTList = []
        CROTList = []
        CUNITList = []
        ValueOfFirstPointList = []


        ## open fits-file using pyfits
        hdulist = pyfits.open(filename)

        # Debug:
        # print "\t HDU = ",HDU
        # print hdulist.info()


        ## save data
        ## number of columns in scidata is given by len(scidata)
        fitsdata = hdulist[HDU].data


        ## verify header, fix if necessary and save header
        # hdulist.verify('fix')
        fitshead = hdulist[HDU].header

        # Debug:
        # print "\t fitsdata.shape = ",fitsdata.shape


        ## set array containing dimension of each axis
        dimension = fitsdata.shape
        NumDimensions = len(dimension)


        ## transpose fitsdata array
        fitsdata = numpy.transpose(fitsdata)


        ## check dimension of fits file
        if (NumDimensions > 4):
            print " "
            print "\t Error in subroutine GetMinMaxFitsFile:"
            print "\t   Can not import fits-file with more than 4 dimensions."
            print "\t   Please select other file and restart program."
            print " "
            print "\t   Program aborted"
            return LowestDataPoint, HighestDataPoint


        ## read header information
        ## get number of axes
        try:
            NAXIS = pyfits.getval(filename, 'NAXIS', 0)
        except KeyError, err:
            NAXIS = 1

        # Debug:
        # print '\n\nNAXIS = ',NAXIS


        ## get scale parameter BSCALE
        try:
            BSCALE = pyfits.getval(filename, 'BSCALE', 0)
        except KeyError, err:
            BSCALE = 1

        # Debug:
        # print '\n\nBSCALE = ',BSCALE


        ## get zero parameter BZERO
        try:
            BZERO = pyfits.getval(filename, 'BZERO', 0)
        except KeyError, err:
            BZERO = 0.0

        # Debug:
        # print '\n\nBZERO = ',BZERO


        ## get unit parameter BUNIT
        try:
            BUNIT = pyfits.getval(filename, 'BUNIT', 0)
        except KeyError, err:
            BUNIT = ""

        # Debug:
        # print '\n\nBUNIT = ',BUNIT


        ## get rest frequency RESTFRQ
        try:
            RESTFRQ = pyfits.getval(filename, 'RESTFRQ', 0)
        except KeyError, err:
            RESTFRQ = 0

        # Debug:
        # print '\n\nRESTFRQ = ',RESTFRQ


        ## analyze header
        length = []
        NumDimensions = 0
        for i in xrange(NAXIS):                                                             ## loop over all dimensions
            try:
                lenAxis = pyfits.getval(filename, 'NAXIS' + str(i + 1), 0)                  ## read CRPIX
            except KeyError, err:
                lenAxis = 1
            if (lenAxis > 0 and NumDimensions == 0):                                        ## read only those entries with more than 1 point
                length.append(lenAxis)
                NumDimensions += 1


                ## get header parameters
                try:
                    CRPIX = pyfits.getval(filename, 'CRPIX' + str(i + 1), 0)                ## location of a reference point along axis 1
                except KeyError, err:
                    CRPIX = 1
                CRPIXList.append(CRPIX)
                try:
                    CRVAL = pyfits.getval(filename, 'CRVAL' + str(i + 1), 0)                ## the value of the coordinate at the reference point CRPIX1
                except KeyError, err:
                    CRVAL = 0
                CRVALList.append(CRVAL)
                try:
                    CDELT = pyfits.getval(filename, 'CDELT' + str(i + 1), 0)                ## partial derivative of the coordinate with respect to the
                except KeyError, err:                                                       ## pixel index, evaluated at the reference point CRPIX
                    CDELT = 1
                CDELTList.append(CDELT)
                try:
                    CROT = pyfits.getval(filename, 'CROT' + str(i + 1), 0)                  ## indicate a rotation from a standard coordinate system
                except KeyError, err:
                    CROT = 0
                CROTList.append(CROT)
                try:
                    CUNIT = pyfits.getval(filename, 'CUNIT' + str(i + 1), 0)                ## indicate the unit of the current axis
                except KeyError, err:
                    CUNIT = ""
                CUNITList.append(CUNIT)


                ## calculate value at first pixel
                LowestDataPoint = CRVAL - ((CRPIX - 1) * CDELT)
                HighestDataPoint = LowestDataPoint + ((lenAxis - 1) * CDELT)

                # Debug:
                # print "CRPIX = ", CRPIX
                # print "CRVAL = ", CRVAL
                # print "CDELT = ", CDELT
                # print "CROT = ", CROT
                # print "CUNIT = ", CUNIT
                # print "LowestDataPoint = ", LowestDataPoint
                # print "HighestDataPoint = ", HighestDataPoint


        ## close fits-file
        hdulist.close()


        ## define return parameters
        return LowestDataPoint, HighestDataPoint


    ##****************************************************************************************************************************************************
    def xml(self, xmlfilename, print_flag, XmlFileInOutput):
        ## load xml-file containing settings for the import of experimental-file(s)
        """

        function xml:       load xml-file containing the settings for the import of experimental-file(s)

        input parameter:    xmlfilename:                file name of the xml(experimental)-file including path
                            print_flag:                 flag if screen output is permitted (=true) or not
                            XmlFileInOutput:            path and file name of the xml description of the model
                                                        (necessary for cso to dat-file conversion, myXCLASS required)

        output parameter:   ok:                         status, if everything is ok: ok = 0
                            InFormat:                   format of the experimental files

        required parameter: NumberExpFiles:             number of experimental files
                            FileNamesExpFiles:          path and file name of each experimental file
                            ErrorY:                     error flag for each experimental file
                            NumberExpRanges:            number of selected experimental ranges
                            MinExpRange:                min of each experimental range
                            MaxExpRange:                max of each experimental range

                            NumberHeaderLines:          number of header lines for each dat-file
                            SeparatorColumns:           column separator for each dat-file
                            NumberColumnsX:             number of columns belonging to the x-positions in dat-files
                            NumberColumnsY:             number of columns belonging to the y-positions in dat-files

                            NumberHDU:                  number of HDU for each fits-file
                            FitsFileDimensions:         dimensions for each fit file


        working variables:  i:                          loop counter
                            j:                          loop counter
                            XmlFile:                    contains the whole contents of the xml-file
                            NumHeadLines:               temp variable for reading number of header lines
                            separator:                  temp variable for reading separator character
                            NumberX:                    temp variable for reading number of x columns
                            NumberY:                    temp variable for reading number of y columns
                            error:                      temp variable for reading error flag
                            NumberRange:                temp variable for reading number of ranges
                            MinRange:                   temp variable for reading min of ranges in file i
                            MaxRange:                   temp variable for reading max of ranges in file i
                            MinR:                       temp variable for reading min of range in range j in file i
                            MaxR:                       temp variable for reading max of range in range j in file i
                            printdoneflag:              flag for printing the word done at the end of the function
                            XmlFileCSO:                 path and file name of the xml-description of the model
                            CommandLine:                path and file name of the start script for myXCLASS
                            command_string:             sting containing the command line starting myXCLASS
        """

        ## set status variable
        ok = 0
        printdoneflag = "true"
        


        ## define default settings
        self.ImportFilter = ["automatic"]
        self.InputFormat = "fits"
        InFormat = self.InputFormat
        self.NumberExpFiles = 1
        self.ErrorY = ["no"]
        self.NumberExpRanges = [0]
        self.MinExpRange = [[-1]]
        self.MaxExpRange = [[-1]]

        self.NumberHeaderLines = [0]
        self.SeparatorColumns = [" "]
        self.NumberColumnsX = [1]
        self.NumberColumnsY = [1]

        self.NumberHDU = [0]
        self.FitsFileDimensions = [0]
        CommandLineParameters = " "
        ImportParameterSet = []                                                             ## set of all parameters used by the import of the
                                                                                            ## observal data:
                                                                                            ## First dimension describes the parameter names
                                                                                            ## Second dimension describes the parameter values
        ImportParameterNames = []                                                           ## help array
        ImportParameterValues = []                                                          ## help array



        ## is the defined file a xml, a fits or a dat file
        if not xmlfilename.endswith(".xml"):

            if xmlfilename.endswith(".dat"):                                                ## if filename ends with dat
                self.InputFormat = "dat"
            elif xmlfilename.endswith(".fits"):                                             ## if filename ends with fits
                self.InputFormat = "fits"
            elif xmlfilename.endswith(".cso"):                                              ## if filename ends with fits
                self.InputFormat = "cso"
            # elif xmlfilename.endswith(".???"):                                            ## if filename ends with ???
            #   self.InputFormat = "???"
            self.FileNamesExpFiles = [xmlfilename]                                          ## define name of exp file
            if (print_flag == "true"):
                print " "
                print "\t WARNING: The selected experimental file is not a xml-file"
                print "\t          But a %s-file. The following default settings for" % self.InputFormat
                print "\t          the import of experimental files are used:"
                print " "
                print "\t          Number of files        = ",self.NumberExpFiles
                if (self.InputFormat == "dat" or self.InputFormat == "cso"):
                    print "\t          Number of header lines = ",self.NumberHeaderLines[0]
                    print "\t          Column separator       = >>%s<<" % self.SeparatorColumns[0]
                    print "\t          Number of x-columns    = ",self.NumberColumnsX[0]
                    print "\t          Number of y-columns    = ",self.NumberColumnsY[0]
                elif (self.InputFormat == "fits"):
                    print "\t          Number of HDU          = ",self.NumberHDU[0]
                # elif (self.InputFormat == "???"):
                #   print "\t          ???                    = ",self.???
                print "\t          Error flag             = ",self.ErrorY[0]
                print "\t          Number of ranges       = ",self.NumberExpRanges[0]
                if (self.NumberExpRanges[0] > 0):
                    print "\t          Min of range           = ",self.MinExpRange[0][0]
                    print "\t          Max of range           = ",self.MaxExpRange[0][0]
                print " "
            self.ImportFilter = [self.InputFormat]
            InFormat = self.InputFormat
            ok = 2
            return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet


        ## print what you do
        if (print_flag != "false"):
            print "\t Open xml-file: " + xmlfilename + " for experimental file import."
            print "\t Reading file .."


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## open xml-file file and save content of file to variable XmlFile
        XmlFile = MAGIXxmlPACKAGE.XmlObject(fileName = xmlfilename)


        ## get number of experimental files
        self.NumberExpFiles = 0
        try:
            self.NumberExpFiles = XmlFile.ExpFiles.NumberExpFiles.getValue()
        except AttributeError, err:
            self.NumberExpFiles = XmlFile.ExpFiles.Section.SubSection.NumberExpFiles.getValue()
        if (self.NumberExpFiles < 1):
            print " "
            print " "
            print "\t WARNING:  Number of experimental files < 1:"
            print "\t           Set number of experimental files to 1!"
            printdoneflag = "false"
            self.NumberExpFiles = 1
            ok = 2
            return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet
        ImportParameterNames.append("MAGIXImportNumberExpFiles")                            ## add MAGIXImportNumberExpFiles to list of import parameter
        ImportParameterValues.append(self.NumberExpFiles)                                   ## add number of exp. files to list of import parameter


        ## get path and file name of each experimental file
        self.ImportFilter = []
        self.FileNamesExpFiles = []
        self.NumberHDU = []
        self.FitsFileDimensions = []
        self.NumberHeaderLines = []
        self.FitsHeader = []
        self.SeparatorColumns = []
        self.NumberColumnsX = []
        self.NumberColumnsY = []
        self.ErrorY = []
        self.NumberExpRanges = []
        self.MinExpRange = []
        self.MaxExpRange = []
        csoVariableList =[]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## start loop over each exp files
        ##
        ##
        ## the following tags are used for all formats:
        ##
        ## - FileNamesExpFiles
        ## - ImportFilter
        ## - NumberExpRanges
        ##
        ##
        for i in xrange(self.NumberExpFiles):


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get path and file name of each experimental-file and set InputFormat variable
            if (self.NumberExpFiles == 1):
                try:
                    filen = XmlFile.ExpFiles.file.FileNamesExpFiles.getValue()
                except AttributeError, err:
                    filen = XmlFile.ExpFiles.Section.SubSection.file.FileNamesExpFiles.getValue()
            else:
                try:
                    filen = XmlFile.ExpFiles.file[i].FileNamesExpFiles.getValue()
                except AttributeError, err:
                    filen = XmlFile.ExpFiles.Section.SubSection.file[i].FileNamesExpFiles.getValue()
            if not(os.path.exists(filen)):
                print " "
                print " "
                print "\t ERROR in function LoadExpFile.LoadExp.xml:"
                print "\t   The experimental-file ",filen
                print "\t   given in the xml-file does not exsist!"
                print "\t   Please correct xml-file and restart program!"
                print " "
                ok = 1
                return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet
            ImportParameterNames.append("MAGIXImportExpFileNames")                          ## add MAGIXImportExpFileNames to list of import parameter
            ImportParameterValues.append(filen)                                             ## add name of exp. file to list of import parameter

            # Debug:
            # print "filen = ", filen


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get filter settings for file import
            self.InputFormat = "dat"
            if (self.NumberExpFiles == 1):
                try:
                    FilterSetting = XmlFile.ExpFiles.file.ImportFilter.getValue()
                except AttributeError, err:
                    FilterSetting = XmlFile.ExpFiles.Section.SubSection.file.ImportFilter.getValue()
            else:
                try:
                    FilterSetting = XmlFile.ExpFiles.file[i].ImportFilter.getValue()
                except AttributeError, err:
                    FilterSetting = XmlFile.ExpFiles.Section.SubSection.file[i].ImportFilter.getValue()
            FilterSetting = FilterSetting.lower()

            # Debug:
            # print "FilterSetting = ", FilterSetting


            ## analyze automatic filter setting
            if (FilterSetting == "automatic" or FilterSetting == "auto"):
                if filen.endswith(".dat"):                                                  ## if filename ends with .dat
                    self.InputFormat = "ascii"
                    FilterSetting = "ascii"
                    InFormat = "dat"
                elif filen.endswith(".fits"):                                               ## if filename ends with .fits
                    self.InputFormat = "fits"
                    FilterSetting = "fits"
                    InFormat = "fits"
                elif filen.endswith(".cso"):                                                ## if filename ends with .cso
                    self.InputFormat = "cso"
                    FilterSetting = "cso"
                    InFormat = "dat"
                else:
                    print " "
                    print " "
                    print "\t ERROR in function LoadExpFile.LoadExp.xml:"
                    print "\t   Can not identify the file format for the experimental-file ",filen
                    print "\t   given in the xml-file!"
                    print "\t   Please correct xml-file and restart program!"
                    print " "
                    ok = 1
                    return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet


            ## analyze filter setting
            elif ((FilterSetting == "dat" and FilterSetting != "xclassascii") or FilterSetting == "ascii"):
                FilterSetting = "ascii"
                InFormat = "dat"
            elif (FilterSetting == "fits"):
                FilterSetting = "fits"
                InFormat = "fits"
            elif (FilterSetting == "cso" or FilterSetting == "xclass"):
                FilterSetting = "cso"
            elif (FilterSetting == "xclassascii"):
                FilterSetting = "xclassascii"
            elif (FilterSetting == "xclassfits"):
                FilterSetting = "xclassfits"
            else:
                FilterSetting = "ascii"
                InFormat = "dat"
            self.ImportFilter.append(FilterSetting)
            self.FileNamesExpFiles.append(filen)

            # Debug:
            # print "FilterSetting = ", FilterSetting


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## for xclassascii file read in exp data file to determine min and max frequency
            #    if (FilterSetting == "xclassascii"):
            #        data = numpy.loadtxt(filen, skiprows = 0)
            #        LowestDataPoint = min(data[:,0])
            #        HighestDataPoint = max(data[:,0])


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## for xclassfits file read in exp data file to determine min and max frequency
            if (FilterSetting == "xclassfits"):
                LowestDataPoint, HighestDataPoint = self.GetMinMaxFitsFile(filen, 0)

                # Debug:
                # print "LowestDataPoint, HighestDataPoint = ", LowestDataPoint, HighestDataPoint


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get number of selected ranges
            NumberRange = 1
            if (self.NumberExpFiles == 1):
                try:
                    NumberRange = XmlFile.ExpFiles.file.NumberExpRanges.getValue()
                except AttributeError, err:
                    try:
                        NumberRange = XmlFile.ExpFiles.Section.SubSection.file.NumberExpRanges.getValue()
                    except AttributeError, err:
                        NumberRange = 0
            else:
                try:
                    NumberRange = XmlFile.ExpFiles.file[i].NumberExpRanges.getValue()
                except AttributeError, err:
                    try:
                        NumberRange = XmlFile.ExpFiles.Section.SubSection.file[i].NumberExpRanges.getValue()
                    except AttributeError, err:
                        NumberRange = 0
            if (NumberRange <= 0):
                if ((FilterSetting == "cso" or FilterSetting == "xclassascii" or FilterSetting == "xclassfits") and NumberRange == 0):
                    print " "
                    print " "
                    print "\t WARNING:  Number of ranges in the experimental-files <= 0:"
                    print "\t           Set number of ranges to 1 !"
                    printdoneflag = "false"
                    NumberRange = 0
                elif (NumberRange < 0):
                    print " "
                    print " "
                    print "\t WARNING:  Number of ranges in the experimental-files < 0:"
                    print "\t           Set number of ranges to 0 !"
                    printdoneflag = "false"
                    NumberRange = 0
            self.NumberExpRanges.append(NumberRange)
            ImportParameterNames.append("MAGIXImportNumberRange")                           ## add MAGIXImportNumberRange to list of import parameter
            ImportParameterValues.append(NumberRange)                                       ## add number of ranges to list of import parameter


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get min and max of each selected range only for ASCII and FITS files
            if (NumberRange > 0 and FilterSetting != "cso" and FilterSetting != "xclassascii" and FilterSetting != "xclassfits"):
                MinRange = []
                MaxRange = []
                for j in xrange(NumberRange):


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## get min of the jth range
                    MinR = 0
                    if (self.NumberExpFiles == 1):
                        if (NumberRange == 1):
                            try:
                                MinR = XmlFile.ExpFiles.file.MinExpRange.getValue()
                            except AttributeError, err:
                                try:
                                    MinR = XmlFile.ExpFiles.Section.SubSection.file.MinExpRange.getValue()
                                except AttributeError, err:
                                    MinR = 0
                        else:
                            try:
                                MinR = XmlFile.ExpFiles.file.MinExpRange[j].getValue()
                            except AttributeError, err:
                                try:
                                    MinR = XmlFile.ExpFiles.Section.SubSection.file.MinExpRange[j].getValue()
                                except AttributeError, err:
                                    MinR = 0
                    else:
                        if (NumberRange == 1):
                            try:
                                MinR = XmlFile.ExpFiles.file[i].MinExpRange.getValue()
                            except AttributeError, err:
                                try:
                                    MinR = XmlFile.ExpFiles.Section.SubSection.file[i].MinExpRange.getValue()
                                except AttributeError, err:
                                    MinR = 0
                        else:
                            try:
                                MinR = XmlFile.ExpFiles.file[i].MinExpRange[j].getValue()
                            except AttributeError, err:
                                try:
                                    MinR = XmlFile.ExpFiles.Section.SubSection.file[i].MinExpRange[j].getValue()
                                except AttributeError, err:
                                    MinR = 0
                    line = str(MinR).split(',')
                    MinRange.append(map(float, line))
                    csoVariableList.append(map(float, line))                                ## add start of range to cso variable list
                    ImportParameterNames.append("MAGIXImportMinRange")
                    ImportParameterValues.append(line)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## get max of the jth range
                    MaxR = 0
                    if (self.NumberExpFiles == 1):
                        if (NumberRange == 1):
                            try:
                                MaxR = XmlFile.ExpFiles.file.MaxExpRange.getValue()
                            except AttributeError, err:
                                try:
                                    MaxR = XmlFile.ExpFiles.Section.SubSection.file.MaxExpRange.getValue()
                                except AttributeError, err:
                                    MaxR = 0
                        else:
                            try:
                                MaxR = XmlFile.ExpFiles.file.MaxExpRange[j].getValue()
                            except AttributeError, err:
                                try:
                                    MaxR = XmlFile.ExpFiles.Section.SubSection.file.MaxExpRange[j].getValue()
                                except AttributeError, err:
                                    MaxR = 0
                    else:
                        if (NumberRange == 1):
                            try:
                                MaxR = XmlFile.ExpFiles.file[i].MaxExpRange.getValue()
                            except AttributeError, err:
                                try:
                                    MaxR = XmlFile.ExpFiles.Section.SubSection.file[i].MaxExpRange.getValue()
                                except AttributeError, err:
                                    MaxR = 0
                        else:
                            try:
                                MaxR = XmlFile.ExpFiles.file[i].MaxExpRange[j].getValue()
                            except AttributeError, err:
                                try:
                                    MaxR = XmlFile.ExpFiles.Section.SubSection.file[i].MaxExpRange[j].getValue()
                                except AttributeError, err:
                                    MaxR = 0
                    line = str(MaxR).split(',')
                    MaxRange.append(map(float, line))
                    csoVariableList.append(map(float, line))                                ## add end of range to cso variable list
                    ImportParameterNames.append("MAGIXImportMaxRange")
                    ImportParameterValues.append(line)


                ## save to global variable
                self.MinExpRange.append(MinRange)
                self.MaxExpRange.append(MaxRange)


            ##********************************************************************************************************************************************
            ## for ascii and xclassascii files only
            if (FilterSetting == "ascii" or FilterSetting == "xclassascii"):


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get number of columns belonging to the y-positions for each experimental-file
                error = "no"
                if (self.NumberExpFiles == 1):
                    try:
                        error = XmlFile.ExpFiles.file.ErrorY.getValue()
                    except AttributeError, err:
                        try:
                            error = XmlFile.ExpFiles.Section.SubSection.file.ErrorY.getValue()
                        except AttributeError, err:
                            error = "no"
                else:
                    try:
                        error = XmlFile.ExpFiles.file[i].ErrorY.getValue()
                    except AttributeError, err:
                        try:
                            error = XmlFile.ExpFiles.Section.SubSection.file[i].ErrorY.getValue()
                        except AttributeError, err:
                            error = "no"
                error = str.lower(error)
                if (error != "no" and error != "not"):
                    if (error != "yes"):
                        print " "
                        print " "
                        print "\t WARNING:  Error flag is not yes or no:"
                        print "\t           Set Error flag to no !"
                        printdoneflag = "false"
                        error = "no"
                self.ErrorY.append(error)
                ImportParameterNames.append("MAGIXImportErrorY")
                ImportParameterValues.append(error)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## if self.InputFormat == "ascii", get number of header lines for each experimental-file
                NumHeadLines = 0
                if (self.NumberExpFiles == 1):
                    try:
                        NumHeadLines = XmlFile.ExpFiles.file.NumberHeaderLines.getValue()
                    except AttributeError, err:
                        try:
                            NumHeadLines = XmlFile.ExpFiles.Section.SubSection.file.NumberHeaderLines.getValue()
                        except AttributeError, err:
                            NumHeadLines = 0
                else:
                    try:
                        NumHeadLines = XmlFile.ExpFiles.file[i].NumberHeaderLines.getValue()
                    except AttributeError, err:
                        try:
                            NumHeadLines = XmlFile.ExpFiles.Section.SubSection.file[i].NumberHeaderLines.getValue()
                        except AttributeError, err:
                            NumHeadLines = 0
                if (NumHeadLines < 0):
                    print " "
                    print " "
                    print "\t WARNING:  Number of header lines in the experimental-files < 0:"
                    print "\t           Set number of header lines to 0 !"
                    printdoneflag = "false"
                    self.NumberHeaderLines = 0
                self.NumberHeaderLines.append(NumHeadLines)
                ImportParameterNames.append("MAGIXImportNumHeadLines")
                ImportParameterValues.append(NumHeadLines)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## if self.InputFormat == "ascii", get separator for each experimental-file
                if (self.NumberExpFiles == 1):
                    try:
                        separator = XmlFile.ExpFiles.file.SeparatorColumns.getValue()
                    except AttributeError, err:
                        try:
                            separator = XmlFile.ExpFiles.Section.SubSection.file.SeparatorColumns.getValue()
                        except AttributeError, err:
                            separator = " "
                else:
                    try:
                        separator = XmlFile.ExpFiles.file[i].SeparatorColumns.getValue()
                    except AttributeError, err:
                        try:
                            separator = XmlFile.ExpFiles.Section.SubSection.file[i].SeparatorColumns.getValue()
                        except AttributeError, err:
                            separator = " "
                self.SeparatorColumns.append(separator)
                ImportParameterNames.append("MAGIXImportSeparatorColumns")
                ImportParameterValues.append(separator)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## the following settings are used only for ASCII files
                if (FilterSetting == "ascii"):


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## if self.InputFormat == "ascii", get number of columns belonging to the x-positions for each experimental-file
                    NumberX = 1
                    if (self.NumberExpFiles == 1):
                        try:
                            NumberX = XmlFile.ExpFiles.file.NumberColumnsX.getValue()
                        except AttributeError, err:
                            try:
                                NumberX = XmlFile.ExpFiles.Section.SubSection.file.NumberColumnsX.getValue()
                            except AttributeError, err:
                                NumberX = 1
                    else:
                        try:
                            NumberX = XmlFile.ExpFiles.file[i].NumberColumnsX.getValue()
                        except AttributeError, err:
                            try:
                                NumberX = XmlFile.ExpFiles.Section.SubSection.file[i].NumberColumnsX.getValue()
                            except AttributeError, err:
                                NumberX = 1
                    if (NumberX < 0):
                        print " "
                        print " "
                        print "\t WARNING:  Number of x-columns in the experimental-files < 0:"
                        print "\t           Set number of x-columns to 1 !"
                        printdoneflag = "false"
                        NumberX = 1
                    self.NumberColumnsX.append(NumberX)
                    ImportParameterNames.append("MAGIXImportNumberColumnsX")
                    ImportParameterValues.append(NumberX)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## if self.InputFormat == "ascii", get number of columns belonging to the y-positions for each experimental-file
                    NumberY = 1
                    if (self.NumberExpFiles == 1):
                        try:
                            NumberY = XmlFile.ExpFiles.file.NumberColumnsY.getValue()
                        except AttributeError, err:
                            try:
                                NumberY = XmlFile.ExpFiles.Section.SubSection.file.NumberColumnsY.getValue()
                            except AttributeError, err:
                                NumberY = 1
                    else:
                        try:
                            NumberY = XmlFile.ExpFiles.file[i].NumberColumnsY.getValue()
                        except AttributeError, err:
                            try:
                                NumberY = XmlFile.ExpFiles.Section.SubSection.file[i].NumberColumnsY.getValue()
                            except AttributeError, err:
                                NumberY = 1
                    if (NumberY < 0):
                        print " "
                        print " "
                        print "\t WARNING:  Number of y-columns in the experimental-files < 0:"
                        print "\t           Set number of y-columns to 1 !"
                        printdoneflag = "false"
                        NumberY = 1
                    self.NumberColumnsY.append(NumberY)
                    ImportParameterNames.append("MAGIXImportNumberColumnsY")
                    ImportParameterValues.append(NumberY)



            ##********************************************************************************************************************************************
            ## for fits-files only
            if (FilterSetting == "fits" or FilterSetting == "xclassfits"):

                ## if self.InputFormat == "fits", get number of HDU
                if (self.NumberExpFiles == 1):
                    try:
                        NumHDU = XmlFile.ExpFiles.file.NumberHDU.getValue()
                    except AttributeError, err:
                        try:
                            NumHDU = XmlFile.ExpFiles.Section.SubSection.file.NumberHDU.getValue()
                        except AttributeError, err:
                            NumHDU = 0
                else:
                    try:
                        NumHDU = XmlFile.ExpFiles.file[i].NumberHDU.getValue()
                    except AttributeError, err:
                        try:
                            NumHDU = XmlFile.ExpFiles.Section.SubSection.file[i].NumberHDU.getValue()
                        except AttributeError, err:
                            NumHDU = 0
                self.NumberHDU.append(NumHDU)
                ImportParameterNames.append("MAGIXImportNumberColumnsY")
                ImportParameterValues.append(NumHDU)


            ##********************************************************************************************************************************************
            ## for cso-files and "xclassascii" only
            ##
            ##
            ## read the following tags:
            ##
            ## - iso_flag
            ## - csoStartFrequency
            ## - csoEndFrequency
            ## - csoStepFrequency
            ## - BackgroundTemperature
            ## - TemperatureSlope
            ## - HydrogenColumnDensity
            ## - DustBeta
            ## - Kappa
            ## - TelescopeSize
            ##
            ##
            ## read the following tags only for "xclassascii" and "xclassfits"
            ##
            ## - IsoTableFileName
            ## - MoleculesFile
            ## - NumberProcessors
            ## - dbFilename
            ##
            ##
            if (FilterSetting == "cso" or FilterSetting == "xclassascii" or FilterSetting == "xclassfits"):


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## read in isotopologues flag
                Isotopologues = "false"
                try:
                    Isotopologues = XmlFile.ExpFiles.Isotopologues.getValue()
                except AttributeError, err:
                    try:
                        Isotopologues = XmlFile.ExpFiles.Section.SubSection.Isotopologues.getValue()
                    except AttributeError, err:
                        try:
                            Isotopologues = XmlFile.ExpFiles.Section.SubSection.iso_flag.getValue()
                        except AttributeError, err:
                            try:
                                Isotopologues = XmlFile.ExpFiles.Isotopologues.getValue()
                            except AttributeError, err:
                                try:
                                    Isotopologues = XmlFile.ExpFiles.iso_flag.getValue()
                                except AttributeError, err:
                                    Isotopologues = "false"
                # Debug:
                # print "Isotopologues = ", Isotopologues


                ## check user input
                if (Isotopologues == 1):
                    Isotopologues = "true"
                elif (Isotopologues == 0):
                    Isotopologues = "false"
                Isotopologues = Isotopologues.strip()                                       ## delete leading and tailing blanks
                Isotopologues = Isotopologues.lower()                                       ## convert to lower case
                if (Isotopologues == "yes" or Isotopologues == "y" or Isotopologues == "true" or Isotopologues == "t"):
                    Isotopologues = 1
                elif (Isotopologues == "no" or Isotopologues == "n" or Isotopologues == "false" or Isotopologues == "f"):
                    Isotopologues = 0
                else:
                    print " "
                    print "\t ERROR in function LoadExpFile.LoadExp.xml:"
                    print "\t   The contents of the tag <iso_flag> (<Isotopologues>) is unequal to"
                    print "\t   " + chr(34) + "yes" + chr(34) + ", " + chr(34) + "y" + chr(34) + ", " \
                                  + chr(34) + "no" + chr(34) + ", " + chr(34) + "n" + chr(34) + ", " \
                                  + chr(34) + "true" + chr(34) + ", " + chr(34) + "t" + chr(34) + ", " \
                                  + chr(34) + "false" + chr(34) + ", and " + chr(34) + "f" + chr(34) + "!"
                    print "\t   Please correct xml-file and restart program!"
                    print " "
                    ok = 1
                    return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet
                ImportParameterNames.append("MAGIXImportIsotopologues")
                ImportParameterValues.append(Isotopologues)

                # Debug:
                # print "Isotopologues = ", Isotopologues


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## the tags IsoTableFileName, MoleculesFile, NumberProcessors, and dbFilename are read only for "xclassascii" or "xclassfits" files
                if (FilterSetting == "xclassascii" or FilterSetting == "xclassfits"):


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## read IsoTableFileName
                    IsoTableFileName = " "
                    try:
                        IsoTableFileName = XmlFile.ExpFiles.IsoTableFileName.getValue()
                    except AttributeError, err:
                        try:
                            IsoTableFileName = XmlFile.ExpFiles.Section.SubSection.IsoTableFileName.getValue()
                        except AttributeError, err:
                            IsoTableFileName = " "
                    ImportParameterNames.append("MAGIXImportIsoTableFileName")
                    ImportParameterValues.append(IsoTableFileName)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## read dbFilename
                    dbFilename = ""
                    try:
                        dbFilename = XmlFile.ExpFiles.dbFilename.getValue()
                    except AttributeError, err:
                        try:
                            dbFilename = XmlFile.ExpFiles.Section.SubSection.dbFilename.getValue()
                        except AttributeError, err:
                            dbFilename = str(os.environ.get("myXCLASSdbFilename", "")).strip()
                    if (dbFilename.strip() == ""):
                        CurrentDir = os.getcwd() + "/"
                        dbFilename = CurrentDir + "Fit-Functions/myXCLASS/bin/myNewXCLASS/src/db/xclass.db"
                    ImportParameterNames.append("MAGIXImportdbFilename")
                    ImportParameterValues.append(dbFilename)

                    # Debug:
                    # print "dbFilename = ", dbFilename
                    # sys.exit(0)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## read the following tags for each range:
                ##
                ## - csoStartFrequency
                ## - csoEndFrequency
                ## - csoStepFrequency
                ## - BackgroundTemperature
                ## - TemperatureSlope
                ## - HydrogenColumnDensity
                ## - DustBeta
                ## - Kappa


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## read start frequency
                MinRange = []
                MaxRange = []
                stepsizeRange = []
                for j in xrange(NumberRange):                                               ## loop of all ranges


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## get min of the jth range
                    csoStartFrequency = 0
                    if (self.NumberExpFiles == 1):
                        if (NumberRange == 1):
                            try:
                                csoStartFrequency = XmlFile.ExpFiles.file.FrequencyRange.MinExpRange.getValue()
                            except AttributeError, err:
                                try:
                                    csoStartFrequency = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange.MinExpRange.getValue()
                                except AttributeError, err:
                                    csoStartFrequency = 0
                        else:
                            try:
                                csoStartFrequency = XmlFile.ExpFiles.file.FrequencyRange[j].MinExpRange.getValue()
                            except AttributeError, err:
                                try:
                                    csoStartFrequency = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange[j].MinExpRange.getValue()
                                except AttributeError, err:
                                    csoStartFrequency = 0
                    else:
                        if (NumberRange == 1):
                            try:
                                csoStartFrequency = XmlFile.ExpFiles.file[i].FrequencyRange.MinExpRange.getValue()
                            except AttributeError, err:
                                try:
                                    csoStartFrequency = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange.MinExpRange.getValue()
                                except AttributeError, err:
                                    csoStartFrequency = 0
                        else:
                            try:
                                csoStartFrequency = XmlFile.ExpFiles.file[i].FrequencyRange[j].MinExpRange.getValue()
                            except AttributeError, err:
                                try:
                                    csoStartFrequency = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange[j].MinExpRange.getValue()
                                except AttributeError, err:
                                    csoStartFrequency = 0
                    if (FilterSetting == "xclassascii" or FilterSetting == "xclassfits"):


                        if (FilterSetting == "xclassascii"):
                            if (separator == " "):
                                data = numpy.loadtxt(filen, skiprows = NumHeadLines)
                            else:
                                data = numpy.loadtxt(filen, delimiter = separator, skiprows = NumHeadLines)
                            LowestDataPoint = min(data[:, 0])
                            HighestDataPoint = max(data[:, 0])

                            # Debug:
                            #    print "csoStartFrequency = ", csoStartFrequency
                            #    print "LowestDataPoint = ", LowestDataPoint


                            ## check, if user defined min. frequency of current data point is lower than the lowet data point in the obs. data file
                            if (csoStartFrequency < LowestDataPoint):
                                if (i == 0):
                                    helpstring = str(i + 1) + "st"
                                elif (i == 1):
                                    helpstring = str(i + 1) + "nd"
                                elif (i == 2):
                                    helpstring = str(i + 1) + "rd"
                                else:
                                    helpstring = str(i + 1) + "th"
                                print " "
                                print "\t WARNING:  The defined start frequency is lower than the lowest frequency in the " + helpstring \
                                                    + " exp. data file!"
                                print "\t           Correct start frequency to lowest frequency " + str(LowestDataPoint) + " in the " + helpstring \
                                                    + " exp. data file."
                                print "\t           MinExpRange = ", csoStartFrequency
                                print " "
                                print " "
                                csoStartFrequency = LowestDataPoint
                    if (FilterSetting == "xclassfits"):
                        SplittedElement = str(csoStartFrequency).split(",")
                        csoStartFrequency = float(SplittedElement[-1])
                        MinRange.append(map(float, SplittedElement))
                    else:
                        MinRange.append(csoStartFrequency)
                    ImportParameterNames.append("MAGIXImportStartFrequency")
                    ImportParameterValues.append(csoStartFrequency)

                    # Debug:
                    # print "csoStartFrequency = ", csoStartFrequency


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## get max of the jth range
                    csoEndFrequency = 0
                    if (self.NumberExpFiles == 1):
                        if (NumberRange == 1):
                            try:
                                csoEndFrequency = XmlFile.ExpFiles.file.FrequencyRange.MaxExpRange.getValue()
                            except AttributeError, err:
                                try:
                                    csoEndFrequency = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange.MaxExpRange.getValue()
                                except AttributeError, err:
                                    csoEndFrequency = 1000
                        else:
                            try:
                                csoEndFrequency = XmlFile.ExpFiles.file.FrequencyRange[j].MaxExpRange.getValue()
                            except AttributeError, err:
                                try:
                                    csoEndFrequency = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange[j].MaxExpRange.getValue()
                                except AttributeError, err:
                                    csoEndFrequency = 1000
                    else:
                        if (NumberRange == 1):
                            try:
                                csoEndFrequency = XmlFile.ExpFiles.file[i].FrequencyRange.MaxExpRange.getValue()
                            except AttributeError, err:
                                try:
                                    csoEndFrequency = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange.MaxExpRange.getValue()
                                except AttributeError, err:
                                    csoEndFrequency = 1000
                        else:
                            try:
                                csoEndFrequency = XmlFile.ExpFiles.file[i].FrequencyRange[j].MaxExpRange.getValue()
                            except AttributeError, err:
                                try:
                                    csoEndFrequency = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange[j].MaxExpRange.getValue()
                                except AttributeError, err:
                                    csoEndFrequency = 1000
                    if (FilterSetting == "xclassascii"):
                        if (csoEndFrequency > HighestDataPoint):
                            csoEndFrequency = HighestDataPoint
                            if (i == 0):
                                helpstring = str(i + 1) + "st"
                            elif (i == 1):
                                helpstring = str(i + 1) + "nd"
                            elif (i == 2):
                                helpstring = str(i + 1) + "rd"
                            else:
                                helpstring = str(i + 1) + "th"
                            print " "
                            print "\t WARNING:  The defined end frequency is larger than the highest frequency in the " + helpstring + " exp. data file!"
                            print "\t           Correct end frequency to highest frequency in the " + helpstring + " exp. data file."
                            print "\t           MaxExpRange = ", csoEndFrequency
                            print " "


                    if (FilterSetting == "xclassfits"):
                        SplittedElement = str(csoEndFrequency).split(",")
                        csoEndFrequency = float(SplittedElement[-1])
                        MaxRange.append(map(float, SplittedElement))
                    else:
                        MaxRange.append(csoEndFrequency)
                    ImportParameterNames.append("MAGIXImportEndFrequency")
                    ImportParameterValues.append(csoEndFrequency)

                    # Debug:
                    # print "csoEndFrequency = ", csoEndFrequency


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## read stepsize
                    csoStepFrequency = 1
                    if (self.NumberExpFiles == 1):
                        if (NumberRange == 1):
                            try:
                                csoStepFrequency = XmlFile.ExpFiles.file.FrequencyRange.StepFrequency.getValue()
                            except AttributeError, err:
                                try:
                                    csoStepFrequency = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange.StepFrequency.getValue()
                                except AttributeError, err:
                                    csoStepFrequency = 1
                        else:
                            try:
                                csoStepFrequency = XmlFile.ExpFiles.file.FrequencyRange[j].StepFrequency.getValue()
                            except AttributeError, err:
                                try:
                                    csoStepFrequency = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange[j].StepFrequency.getValue()
                                except AttributeError, err:
                                    csoStepFrequency = 1
                    else:
                        if (NumberRange == 1):
                            try:
                                csoStepFrequency = XmlFile.ExpFiles.file[i].FrequencyRange.StepFrequency.getValue()
                            except AttributeError, err:
                                try:
                                    csoStepFrequency = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange.StepFrequency.getValue()
                                except AttributeError, err:
                                    csoStepFrequency = 1
                        else:
                            try:
                                csoStepFrequency = XmlFile.ExpFiles.file[i].FrequencyRange[j].StepFrequency.getValue()
                            except AttributeError, err:
                                try:
                                    csoStepFrequency = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange[j].StepFrequency.getValue()
                                except AttributeError, err:
                                    csoStepFrequency = 1
                    ImportParameterNames.append("MAGIXImportStepFrequency")
                    ImportParameterValues.append(csoStepFrequency)


                    ## check if stepsize is in a meaningful range
                    if (csoStepFrequency > (csoEndFrequency - csoStartFrequency) and csoStepFrequency > 0.0):
                        print " "
                        print "\t ERROR in function LoadExpFile.LoadExp.xml:"
                        print "\t   The stepsize is larger than frequency range !"
                        print "\t   Please correct the xml-file and restart MAGIX !"
                        print " "
                        print "\t   csoStartFrequency = ", csoStartFrequency
                        print "\t   csoEndFrequency = ", csoEndFrequency
                        print "\t   csoStepFrequency = ", csoStepFrequency
                        print "\t   (csoEndFrequency - csoStartFrequency) = ", (csoEndFrequency - csoStartFrequency)
                        print " "
                        print "\t   File i = ", i
                        print "\t   Range j = ", j
                        print " "
                        ok = 1
                        return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet
                    else:
                        stepsizeRange.append(csoStepFrequency)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## read t_back_flag
                    t_back_flag = "true"
                    if (self.NumberExpFiles == 1):
                        if (NumberRange == 1):
                            try:
                                t_back_flag = XmlFile.ExpFiles.file.FrequencyRange.t_back_flag.getValue()
                            except AttributeError, err:
                                try:
                                    t_back_flag = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange.t_back_flag.getValue()
                                except AttributeError, err:
                                    t_back_flag = "true"
                        else:
                            try:
                                t_back_flag = XmlFile.ExpFiles.file.FrequencyRange[j].t_back_flag.getValue()
                            except AttributeError, err:
                                try:
                                    t_back_flag = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange[j].t_back_flag.getValue()
                                except AttributeError, err:
                                    t_back_flag = "true"
                    else:
                        if (NumberRange == 1):
                            try:
                                t_back_flag = XmlFile.ExpFiles.file[i].FrequencyRange.t_back_flag.getValue()
                            except AttributeError, err:
                                try:
                                    t_back_flag = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange.t_back_flag.getValue()
                                except AttributeError, err:
                                    t_back_flag = "true"
                        else:
                            try:
                                t_back_flag = XmlFile.ExpFiles.file[i].FrequencyRange[j].t_back_flag.getValue()
                            except AttributeError, err:
                                try:
                                    t_back_flag = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange[j].t_back_flag.getValue()
                                except AttributeError, err:
                                    t_back_flag = "true"


                    ## check user input
                    if (t_back_flag == 1):
                        t_back_flag = "true"
                    elif (t_back_flag == 0):
                        t_back_flag = "false"
                    t_back_flag = t_back_flag.strip()                                       ## delete leading and tailing blanks
                    t_back_flag = t_back_flag.lower()                                       ## convert to lower case
                    if (t_back_flag == "yes" or t_back_flag == "y" or t_back_flag == "true" or t_back_flag == "t"):
                        t_back_flag = 1
                    elif (t_back_flag == "no" or t_back_flag == "n" or t_back_flag == "false" or t_back_flag == "f"):
                        t_back_flag = 0
                    else:
                        print " "
                        print "\t ERROR in function LoadExpFile.LoadExp.xml:"
                        print "\t   The contents of the tag <t_back_flag> is unequal to"
                        print "\t   " + chr(34) + "yes" + chr(34) + ", " + chr(34) + "y" + chr(34) + ", " \
                                      + chr(34) + "no" + chr(34) + ", " + chr(34) + "n" + chr(34) + ", " \
                                      + chr(34) + "true" + chr(34) + ", " + chr(34) + "t" + chr(34) + ", " \
                                      + chr(34) + "false" + chr(34) + ", and " + chr(34) + "f" + chr(34) + "!"
                        print "\t   Please correct xml-file and restart program!"
                        print " "
                        ok = 1
                        return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet
                    ImportParameterNames.append("MAGIXImportTBackFlag")
                    ImportParameterValues.append(t_back_flag)

                    # Debug:
                    # print "t_back_flag = ", t_back_flag


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## read BackgroundTemperature
                    BackgroundTemperature = 1
                    if (self.NumberExpFiles == 1):
                        if (NumberRange == 1):
                            try:
                                BackgroundTemperature = XmlFile.ExpFiles.file.FrequencyRange.BackgroundTemperature.getValue()
                            except AttributeError, err:
                                try:
                                    BackgroundTemperature = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange.BackgroundTemperature.getValue()
                                except AttributeError, err:
                                    BackgroundTemperature = -9999.0
                        else:
                            try:
                                BackgroundTemperature = XmlFile.ExpFiles.file.FrequencyRange[j].BackgroundTemperature.getValue()
                            except AttributeError, err:
                                try:
                                    BackgroundTemperature = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange[j].BackgroundTemperature.getValue()
                                except AttributeError, err:
                                    BackgroundTemperature = -9999.0
                    else:
                        if (NumberRange == 1):
                            try:
                                BackgroundTemperature = XmlFile.ExpFiles.file[i].FrequencyRange.BackgroundTemperature.getValue()
                            except AttributeError, err:
                                try:
                                    BackgroundTemperature = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange.BackgroundTemperature.getValue()
                                except AttributeError, err:
                                    BackgroundTemperature = -9999.0
                        else:
                            try:
                                BackgroundTemperature = XmlFile.ExpFiles.file[i].FrequencyRange[j].BackgroundTemperature.getValue()
                            except AttributeError, err:
                                try:
                                    BackgroundTemperature = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange[j].BackgroundTemperature.getValue()
                                except AttributeError, err:
                                    BackgroundTemperature = -9999.0
                    ImportParameterNames.append("MAGIXImportBackgroundTemperature")
                    ImportParameterValues.append(BackgroundTemperature)


                    ## check if BackgroundTemperature is in meaningful range
                    #if (BackgroundTemperature < 1):
                    #    print " "
                    #    print "\t Warning in function LoadExpFile.LoadExp.xml:"
                    #    print "\t   The given background temperature is less than 1 !"
                    #    print "\t   Please correct the xml-file and restart MAGIX !"
                    #    print " "
                    #    ok = 1
                    #    return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## read TemperatureSlope
                    TemperatureSlope = 1
                    if (self.NumberExpFiles == 1):
                        if (NumberRange == 1):
                            try:
                                TemperatureSlope = XmlFile.ExpFiles.file.FrequencyRange.TemperatureSlope.getValue()
                            except AttributeError, err:
                                try:
                                    TemperatureSlope = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange.TemperatureSlope.getValue()
                                except AttributeError, err:
                                    TemperatureSlope = 0
                        else:
                            try:
                                TemperatureSlope = XmlFile.ExpFiles.file.FrequencyRange[j].TemperatureSlope.getValue()
                            except AttributeError, err:
                                try:
                                    TemperatureSlope = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange[j].TemperatureSlope.getValue()
                                except AttributeError, err:
                                    TemperatureSlope = 0
                    else:
                        if (NumberRange == 1):
                            try:
                                TemperatureSlope = XmlFile.ExpFiles.file[i].FrequencyRange.TemperatureSlope.getValue()
                            except AttributeError, err:
                                try:
                                    TemperatureSlope = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange.TemperatureSlope.getValue()
                                except AttributeError, err:
                                    TemperatureSlope = 0
                        else:
                            try:
                                TemperatureSlope = XmlFile.ExpFiles.file[i].FrequencyRange[j].TemperatureSlope.getValue()
                            except AttributeError, err:
                                try:
                                    TemperatureSlope = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange[j].TemperatureSlope.getValue()
                                except AttributeError, err:
                                    TemperatureSlope = 0
                    ImportParameterNames.append("MAGIXImportTemperatureSlope")
                    ImportParameterValues.append(TemperatureSlope)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## read HydrogenColumnDensity
                    HydrogenColumnDensity = 1
                    if (self.NumberExpFiles == 1):
                        if (NumberRange == 1):
                            try:
                                HydrogenColumnDensity = XmlFile.ExpFiles.file.FrequencyRange.HydrogenColumnDensity.getValue()
                            except AttributeError, err:
                                try:
                                    HydrogenColumnDensity = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange.HydrogenColumnDensity.getValue()
                                except AttributeError, err:
                                    HydrogenColumnDensity = -9999.0
                        else:
                            try:
                                HydrogenColumnDensity = XmlFile.ExpFiles.file.FrequencyRange[j].HydrogenColumnDensity.getValue()
                            except AttributeError, err:
                                try:
                                    HydrogenColumnDensity = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange[j].HydrogenColumnDensity.getValue()
                                except AttributeError, err:
                                    HydrogenColumnDensity = -9999.0
                    else:
                        if (NumberRange == 1):
                            try:
                                HydrogenColumnDensity = XmlFile.ExpFiles.file[i].FrequencyRange.HydrogenColumnDensity.getValue()
                            except AttributeError, err:
                                try:
                                    HydrogenColumnDensity = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange.HydrogenColumnDensity.getValue()
                                except AttributeError, err:
                                    HydrogenColumnDensity = -9999.0
                        else:
                            try:
                                HydrogenColumnDensity = XmlFile.ExpFiles.file[i].FrequencyRange[j].HydrogenColumnDensity.getValue()
                            except AttributeError, err:
                                try:
                                    HydrogenColumnDensity = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange[j].HydrogenColumnDensity.getValue()
                                except AttributeError, err:
                                    HydrogenColumnDensity = -9999.0
                    ImportParameterNames.append("MAGIXImportHydrogenColumnDensity")
                    ImportParameterValues.append(HydrogenColumnDensity)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## read DustBeta
                    DustBeta = 1
                    if (self.NumberExpFiles == 1):
                        if (NumberRange == 1):
                            try:
                                DustBeta = XmlFile.ExpFiles.file.FrequencyRange.DustBeta.getValue()
                            except AttributeError, err:
                                try:
                                    DustBeta = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange.DustBeta.getValue()
                                except AttributeError, err:
                                    DustBeta = 1
                        else:
                            try:
                                DustBeta = XmlFile.ExpFiles.file.FrequencyRange[j].DustBeta.getValue()
                            except AttributeError, err:
                                try:
                                    DustBeta = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange[j].DustBeta.getValue()
                                except AttributeError, err:
                                    DustBeta = 1
                    else:
                        if (NumberRange == 1):
                            try:
                                DustBeta = XmlFile.ExpFiles.file[i].FrequencyRange.DustBeta.getValue()
                            except AttributeError, err:
                                try:
                                    DustBeta = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange.DustBeta.getValue()
                                except AttributeError, err:
                                    DustBeta = 1
                        else:
                            try:
                                DustBeta = XmlFile.ExpFiles.file[i].FrequencyRange[j].DustBeta.getValue()
                            except AttributeError, err:
                                try:
                                    DustBeta = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange[j].DustBeta.getValue()
                                except AttributeError, err:
                                    DustBeta = 1
                    ImportParameterNames.append("MAGIXImportDustBeta")
                    ImportParameterValues.append(DustBeta)

                    ## check if DustBeta is in meaningful range
                    #if (DustBeta < 0):
                    #    print " "
                    #    print "\t ERROR in function LoadExpFile.LoadExp.xml:"
                    #    print "\t   The given size of the DustBeta is less than 0 !"
                    #    print "\t   Please correct the xml-file and restart MAGIX !"
                    #    print " "
                    #    ok = 1
                    #    return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## read Kappa
                    Kappa = 1
                    if (self.NumberExpFiles == 1):
                        if (NumberRange == 1):
                            try:
                                Kappa = XmlFile.ExpFiles.file.FrequencyRange.Kappa.getValue()
                            except AttributeError, err:
                                try:
                                    Kappa = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange.Kappa.getValue()
                                except AttributeError, err:
                                    Kappa = 1
                        else:
                            try:
                                Kappa = XmlFile.ExpFiles.file.FrequencyRange[j].Kappa.getValue()
                            except AttributeError, err:
                                try:
                                    Kappa = XmlFile.ExpFiles.Section.SubSection.file.FrequencyRange[j].Kappa.getValue()
                                except AttributeError, err:
                                    Kappa = 1
                    else:
                        if (NumberRange == 1):
                            try:
                                Kappa = XmlFile.ExpFiles.file[i].FrequencyRange.Kappa.getValue()
                            except AttributeError, err:
                                try:
                                    Kappa = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange.Kappa.getValue()
                                except AttributeError, err:
                                    Kappa = 1
                        else:
                            try:
                                Kappa = XmlFile.ExpFiles.file[i].FrequencyRange[j].Kappa.getValue()
                            except AttributeError, err:
                                try:
                                    Kappa = XmlFile.ExpFiles.Section.SubSection.file[i].FrequencyRange[j].Kappa.getValue()
                                except AttributeError, err:
                                    Kappa = 1
                    ImportParameterNames.append("MAGIXImportKappa")
                    ImportParameterValues.append(Kappa)


                ## save to global variable
                self.MinExpRange.append(MinRange)
                self.MaxExpRange.append(MaxRange)


                ##------------------------------------------------------------------------------------------------------------------------------------
                ## read global vLSR parameter
                GlobalvLSR = 0.0
                if (self.NumberExpFiles == 1):
                    try:
                        GlobalvLSR = XmlFile.ExpFiles.file.GlobalvLSR.getValue()
                    except AttributeError, err:
                        try:
                            GlobalvLSR = XmlFile.ExpFiles.Section.SubSection.file.GlobalvLSR.getValue()
                        except AttributeError, err:
                            GlobalvLSR = 0.0
                else:
                    try:
                        GlobalvLSR = XmlFile.ExpFiles.file[i].GlobalvLSR.getValue()
                    except AttributeError, err:
                        try:
                            GlobalvLSR = XmlFile.ExpFiles.Section.SubSection.file[i].GlobalvLSR.getValue()
                        except AttributeError, err:
                            GlobalvLSR = 0.0
                ImportParameterNames.append("MAGIXImportGlobalvLSR")
                ImportParameterValues.append(GlobalvLSR)

                # Debug:
                # print "GlobalvLSR = ", GlobalvLSR


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## read TelescopeSize
                TelescopeSize = 1
                if (self.NumberExpFiles == 1):
                    try:
                        TelescopeSize = XmlFile.ExpFiles.file.TelescopeSize.getValue()
                    except AttributeError, err:
                        try:
                            TelescopeSize = XmlFile.ExpFiles.Section.SubSection.file.TelescopeSize.getValue()
                        except AttributeError, err:
                            TelescopeSize = 0
                else:
                    try:
                        TelescopeSize = XmlFile.ExpFiles.file[i].TelescopeSize.getValue()
                    except AttributeError, err:
                        try:
                            TelescopeSize = XmlFile.ExpFiles.Section.SubSection.file[i].TelescopeSize.getValue()
                        except AttributeError, err:
                            TelescopeSize = 0
                if (TelescopeSize < 0):                                                     ## check if TelescopeSize is in meaningful range
                    print " "
                    print "\t ERROR in function LoadExpFile.LoadExp.xml:"
                    print "\t   The given size of the telescope is less than 0 !"
                    print "\t   Please correct the xml-file and restart MAGIX !"
                    print " "
                    ok = 1
                    return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet
                else:
                    ImportParameterNames.append("MAGIXImportTelescopeSize")
                    ImportParameterValues.append(TelescopeSize)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## read in interferrometer flag
                InterferrometerFlag = "false"
                if (self.NumberExpFiles == 1):
                    try:
                        InterferrometerFlag = XmlFile.ExpFiles.file.Inter_Flag.getValue()
                    except AttributeError, err:
                        try:
                            InterferrometerFlag = XmlFile.ExpFiles.Section.SubSection.file.Inter_Flag.getValue()
                        except AttributeError, err:
                            InterferrometerFlag = "false"
                else:
                    try:
                        InterferrometerFlag = XmlFile.ExpFiles.file[i].Inter_Flag.getValue()
                    except AttributeError, err:
                        try:
                            InterferrometerFlag = XmlFile.ExpFiles.Section.SubSection.file[i].Inter_Flag.getValue()
                        except AttributeError, err:
                            InterferrometerFlag = "false"


                ## check user input
                if (InterferrometerFlag == 1):
                    InterferrometerFlag = "true"
                elif (InterferrometerFlag == 0):
                    InterferrometerFlag = "false"
                InterferrometerFlag = InterferrometerFlag.strip()                           ## delete leading and tailing blanks
                InterferrometerFlag = InterferrometerFlag.lower()                           ## convert to lower case
                if (InterferrometerFlag == "yes" or InterferrometerFlag == "y" or InterferrometerFlag == "true" or InterferrometerFlag == "t"):
                    InterferrometerFlag = 1
                elif (InterferrometerFlag == "no" or InterferrometerFlag == "n" or InterferrometerFlag == "false" or InterferrometerFlag == "f"):
                    InterferrometerFlag = 0
                else:
                    print " "
                    print "\t ERROR in function LoadExpFile.LoadExp.xml:"
                    print "\t   The contents of the tag <Interferrometer_Flag> is unequal to"
                    print "\t   " + chr(34) + "yes" + chr(34) + ", " + chr(34) + "y" + chr(34) + ", " \
                                  + chr(34) + "no" + chr(34) + ", " + chr(34) + "n" + chr(34) + ", " \
                                  + chr(34) + "true" + chr(34) + ", " + chr(34) + "t" + chr(34) + ", " \
                                  + chr(34) + "false" + chr(34) + ", and " + chr(34) + "f" + chr(34) + "!"
                    print "\t   Please correct xml-file and restart program!"
                    print " "
                    ok = 1
                    return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet
                ImportParameterNames.append("MAGIXImportInterferrometerFlag")
                ImportParameterValues.append(InterferrometerFlag)

                # Debug:
                # print "InterferrometerFlag = ", InterferrometerFlag


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## define parameters for ASCII file import
                self.NumberColumnsX.append(1)                                               ## set number of x-columns to 1
                self.NumberColumnsY.append(1)                                               ## set number of y-columns to 1
                self.InputFormat = "dat"                                                    ## set input format to ascii
                InFormat = "dat"


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## transform cso-file to dat-file (myXCLASS is required)
                ## get path of the myXCLASS start script from the xml-description
                if (FilterSetting == "cso"):
                    self.NumberHeaderLines.append(0)                                        ## set number of header lines to zero
                    self.SeparatorColumns.append(" ")                                       ## set separator character to blank
                    self.ErrorY.append("no")
                    if not(os.path.exists(XmlFileInOutput)):                                ## does path and file described in XmlFileInOutput exsist?
                        print " "
                        print "\t Error in subroutine read_control"
                        print "\t      The conversion of cso to dat-file requires myXCLASS"
                        print "\t      and the path to the myXCLASS start script within the"
                        print "\t      xml-description of the model does not exsist."
                        print "\t      Please correct the command line description in the"
                        print "\t      xml-file %s." % XmlFileInOutput
                        print " "
                        ok = 1
                        sys.exit(0)


                    ## define command string
                    XmlFileCSO = MAGIXxmlPACKAGE.XmlObject(fileName = XmlFileInOutput)       ## read in the model description


                    ## get command line of the myXCLASS start script
                    try:
                        PathStartScript = XmlFileCSO.InOutputFile.ModelProgramCall.PathStartScript.getValue()
                    except AttributeError, err:
                        PathStartScript = XmlFileCSO.InOutputFile.Section.ModelProgramCall.PathStartScript.getValue()
                    CommandLine = PathStartScript.strip()


                    ## get input line of the myXCLASS start script
                    try:
                        InputName = XmlFileCSO.InOutputFile.InputFile[0].InputFileName.getValue()
                    except AttributeError, err:
                        InputName = XmlFileCSO.InOutputFile.Section.InputFile[0].InputFileName.getValue()
                    filenew = filen + ".dat"


                    ## get pid of current process
                    pid = os.getpid()


                    ## get current working directory in magix
                    CurrentWorkingDir = os.getcwd()


                    ## start cso-transformation script
                    command_string_org = CommandLine
                    csoFilenameRangeAll = []
                    for j in xrange(NumberRange):                                           ## loop of all ranges
                        MinR = MinRange[j]                                                  ## get start frequency
                        MaxR = MaxRange[j]                                                  ## get end frequency
                        stepFre = stepsizeRange[j]                                          ## get resolution (stepsize)


                        ## define new ascii file name if more than one range is selected
                        if (NumberRange > 1):
                            csoFilenameRange = filen + str(j) + ".dat"
                            csoFilenameRangeAll.append(csoFilenameRange)
                        else:
                            csoFilenameRange = filen + ".dat"


                        ## define command string parameters
                        CommandLineParameters = " data " + str(pid) + " " + str(MinR) + " " + str(MaxR) + " " + str(stepFre) + " " + filen + " " \
                                                + csoFilenameRange + " " + CurrentWorkingDir

                        # Debug:
                        # print '>>>>>',CommandLineParameters,'<<<<'


                        ## save new filename
                        #filen = filenew
                        self.FileNamesExpFiles[i] = filenew


                        ## define command sting
                        command_string = command_string_org + CommandLineParameters

                        # Debug:
                        # print 'command_string = ',command_string
                        # print "csoFilenameRange = ", csoFilenameRange


                        ## execute cso-transformation script
                        os.system(command_string)


                        ## check, if file conversion was successfully
                        if not (os.path.exists(csoFilenameRange)):
                            print " "
                            print " "
                            print "  Error in subroutine LoadExpFile.xml:"
                            print "    The conversion of the xclass file to ASCII file format was not successfully."
                            print " "
                            print "    Please take a look at the .output-file and restart MAGIX."
                            print " "
                            print " "
                            sys.exit(0)


                    ## combine the different ascii-file for each range to one big ascii file
                    if (NumberRange > 1):
                        command_string = "rm -f " + filenew
                        os.system(command_string)                                           ## remove old cso-dat file

                        command_string = "cat"
                        for j in xrange(NumberRange):                                       ## loop of all ranges
                            csoFilenameRange = csoFilenameRangeAll[j]
                            command_string += " " + csoFilenameRange
                        command_string += " >> " + filenew
                        os.system(command_string)                                           ## append all range files to one final ascii-file


                        ## remove range files
                        for j in xrange(NumberRange):                                       ## loop of all ranges
                            csoFilenameRange = csoFilenameRangeAll[j]
                            command_string = "rm -f " + csoFilenameRange
                            os.system(command_string)                                       ## remove jth range file



        ## add model switch to CommandLineParameters
        ## FreqMin[i][j]+' '+FreqMax[i][j]+' '+StepFreq[i][j]+' '+' '+ tBack[i][j]+' '+' '+tSlope[i][j]+' '+telescopeSize
        ## +' '+ModelParamFile+nh+' '+DustBeta+' '+iso
        #CommandLineParametersModel = " model " + str(pid) + CommandLineParametersModel + " " + Isotopologues + " " + InputName + " " + CurrentWorkingDir


        ## print results
        if (print_flag != "false"):
            #if (printdoneflag != "false"):
            #    print "done !"

            print " "
            print "\t Import settings:"
            print " "
            print "\t NumberExpFiles = ",self.NumberExpFiles
            for i in xrange(self.NumberExpFiles):
                print "\t Experimental file number: ",i+1
                print "\t FileNamesExpFiles[%i] = %s" % (i,self.FileNamesExpFiles[i])
                print "\t NumberExpRanges[%i]   = %i" % (i,self.NumberExpRanges[i])
                if (self.NumberExpRanges[i] > 0):
                    for j in xrange(self.NumberExpRanges[i]):
                        print "\t MinExpRange[%i][%i]    = %s" % (i,j,self.MinExpRange[i][j])
                        print "\t MaxExpRange[%i][%i]    = %s" % (i,j,self.MaxExpRange[i][j])

                if (self.InputFormat == "fits"):
                    print "\t NumberHDU[%i]         = %i" % (i,self.NumberHDU[i])
                elif (self.InputFormat == "ascii"):
                    print "\t ErrorY[%i]            = %s" % (i,self.ErrorY[i])
                    print "\t NumberHeaderLines[%i] = %i" % (i,self.NumberHeaderLines[i])
                    print "\t SeparatorColumns[%i]  = >>%s<<" % (i,self.SeparatorColumns[i])
                    print "\t NumberColumnsX[%i]    = %i" % (i,self.NumberColumnsX[i])
                    print "\t NumberColumnsY[%i]    = %i" % (i,self.NumberColumnsY[i])
                # elif (self.InputFormat == "???"):
                #   print "\t "
            print " "


        ## combine import parameter names and values to final output parameter set
        ImportParameterSet.append(ImportParameterNames)
        ImportParameterSet.append(ImportParameterValues)


        return ok, self.ImportFilter, self.NumberExpRanges, self.MinExpRange, self.MaxExpRange, ImportParameterSet
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ##
    ##                                              Load data
    ##
    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


    ##****************************************************************************************************************************************************
    def LoadFile(self, print_flag):
        ## load contents of ascii-file(s)
        """

        function LoadDat:       load ascii files containing experimental data using the setting given in the xml-file

        input parameters:       print_flag:                 flag for printing message to screen

        required parameters:    NumberExpFiles:             number of ascii-files
                                FileNamesExpFiles:          path and file name of each ascii file
                                NumberHeaderLines:          number of header lines for each ascii-file
                                SeparatorColumns:           column separator for each ascii-file
                                NumberColumnsX:             number of columns belonging to the x-positions
                                NumberColumnsY:             number of columns belonging to the y-positions
                                NumberExpRanges:            number of selected experimental ranges
                                MinExpRange:                min of each experimental range
                                MaxExpRange:                max of each experimental range

        output parameters:      ok:                         status of subroutine
                                NumberFiles:                number of ascii-files
                                LengthExpFileDat:           number of lines in the experimental ascii-file
                                ExpDataX:                   experimental data x column
                                ExpDataY:                   experimental data y column
                                ExpDataError:               error of experimental data y column

        working variables:      i:                          loop counter
                                k:                          loop counter
                                FileFormat:                 format of the current data file


        IMPORTANT:      - The function xml() has to be called before the call of LoadDat()

        """
        ## set status to 0
        ok = 0


        ## read experimental data of each ascii-file and define output variables
        NumberFiles = self.NumberExpFiles
        LengthExpFileDat = []
        ExpDataX = []
        ExpDataY = []
        ExpDataError = []
        CounterASCIIFile = (-1)
        CounterFitsFile = (-1)
        for i in xrange(self.NumberExpFiles):


            ## read in ascii files
            FileFormat = self.ImportFilter[i]
            if (FileFormat == "ascii" or FileFormat == "xclassascii" or FileFormat == "cso"):
                CounterASCIIFile += 1
                ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error = self.LoadDat(i, CounterASCIIFile, print_flag)
            elif (FileFormat == "fits" or FileFormat == "xclassfits"):
                CounterFitsFile += 1
                ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error = self.LoadFits(i, CounterFitsFile, print_flag)
            else:
                print " "
                print " "
                print "\t ERROR in subroutine LoadFile:"
                print "\t   Can not import the experimental-file ",filen
                print "\t   LoadFilegiven in the xml-file!"
                print " "
                print "\t   Unknown file format ", FileFormat
                print " "
                print "\t   Please correct xml-file and restart program!"
                print " "
                ok = 1
                return ok, NumberFiles, LengthExpFileDat, ExpDataX, ExpDataY, ExpDataError


            ## copy results to output variables
            ExpDataX.append(exp_data_x)
            ExpDataY.append(exp_data_y)
            ExpDataError.append(exp_data_error)
            LengthExpFileDat.append(lengthexpdata)


        ## define return variables
        return ok, NumberFiles, LengthExpFileDat, ExpDataX, ExpDataY, ExpDataError
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    def LoadDat(self, i, CounterASCIIFile, print_flag):
        ## load contents of ascii-file(s)
        """

        function LoadDat:       load ascii files containing experimental data using the setting given in the xml-file

        input parameters:       i:                          index of experimental data file
                                CounterASCIIFile:           counter for ascii files
                                print_flag:                 flag for printing message to screen

        required parameters:    FileNamesExpFiles:          path and file name of each ascii file
                                NumberHeaderLines:          number of header lines for each ascii-file
                                SeparatorColumns:           column separator for each ascii-file
                                NumberColumnsX:             number of columns belonging to the x-positions
                                NumberColumnsY:             number of columns belonging to the y-positions
                                NumberExpRanges:            number of selected experimental ranges
                                MinExpRange:                min of each experimental range
                                MaxExpRange:                max of each experimental range

        output parameters:      ok:                         status of subroutine
                                lengthexpdata:              number of lines in the experimental ascii-file
                                exp_data_x:                 experimental data x column
                                exp_data_y:                 experimental data y column
                                exp_data_error:             error of experimental data y column

        working variables:      k:                          loop counter
                                filename:                   temp variable for file name of the ith ascii-file
                                number_header_lines:        temp variable for number of header lines in the ith ascii-file
                                seperator:                  temp variable for column separator in the ith ascii-file
                                data:                       temp variable containing the whole ith ascii-file
                                length_exp_file:            temp variable containing the total number of lines in the ith ascii-file (without header lines)
                                number_columns:             temp variable containing the total number of columns in the ith ascii-file
                                NumberX:                    temp variable containing the number of x-columns in the ith ascii-file
                                NumberY:                    temp variable containing the number of y-columns in the ith ascii-file
                                error:                      temp variable containing the error flag
                                NumberRange:                temp variable containing the number of ranges in the ith ascii-file
                                xmn:                        array containing all lower limits of the ranges of the ith ascii-file
                                xmx:                        array containing all upper limits of the ranges of the ith ascii-file
                                exp_data_x:                 temp variable containing experimental data x column of the ith ascii-file
                                exp_data_y:                 temp variable containing experimental data y column of the ith ascii-file
                                xcolumn:                    temp variable containing the x-columns of the kth line in the ith ascii-file
                                ycolumn:                    temp variable containing the y-columns of the lth line in the ith ascii-file
                                errorcolumn:                temp variable containing the error-columns of the lth line in the ith ascii-file
                                copyflag:                   flag for extracting experimental data within selected ranges in the ith ascii-file
                                copyflagcolumn:             flag for extracting experimental data within selected ranges in the ith ascii-file
                                element:                    flag for extracting experimental data within selected ranges in the ith ascii-file


        IMPORTANT:      - The function xml() has to be called before the call of LoadDat()

        """
        ## set status to 0
        ok = 0


        ## reset output variables
        lengthexpdata = -1
        exp_data_x = []
        exp_data_y = []
        exp_data_error = []


        ## set working variables for the ith ascii-file
        filename = self.FileNamesExpFiles[i]
        number_header_lines = self.NumberHeaderLines[CounterASCIIFile]
        seperator = self.SeparatorColumns[CounterASCIIFile]
        NumberX = self.NumberColumnsX[CounterASCIIFile]
        NumberY = self.NumberColumnsY[CounterASCIIFile]
        error = self.ErrorY[CounterASCIIFile]
        NumberRange = self.NumberExpRanges[i]
        if (NumberRange > 0):
            xmn = self.MinExpRange[i]
            xmx = self.MaxExpRange[i]


        ## print what you do
        if (print_flag != "false"):
            print "\t Reading ASCII-file " + filename + " ..",


        ## read ascii-file using numpy function loadtxt
        if (seperator == " "):
            data = numpy.loadtxt(filename, skiprows = number_header_lines)
        else:
            data = numpy.loadtxt(filename, delimiter = seperator, skiprows = number_header_lines)

        # Debug:
        # print " "
        # print " "
        # print "data[1] = ",data[1]
        # print "data.shape = ",data.shape


        ## check if range definition correct
        if (NumberRange > 0):
            if (len(xmn) != NumberRange):
                print " "
                print " "
                print "\t ERROR in function LoadExpFile.LoadDat:"
                print "\t   The dimension of the definition of the content of "
                print "\t   the <MinExpRange></MinExpRange> tag in the xml-file"
                print "\t   for the import settings does not coincident with the"
                print "\t   dimension of the loaded ascii-file!"
                print "\t   Please correct the xml-file and restart program!"
                print " "
                ok = 1
                return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error

            elif (len(xmx) != NumberRange):
                print " "
                print " "
                print "\t ERROR in function LoadExpFile.LoadDat:"
                print "\t   The dimension of the definition of the content of "
                print "\t   the <MaxExpRange></MaxExpRange> tag in the xml-file"
                print "\t   for the import settings does not coincident with the"
                print "\t   dimension of the loaded ascii-file!"
                print "\t   Please correct the xml-file and restart program!"
                print " "
                ok = 1
                return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error


        ## get number of lines and columns of the ith ascii-file and check consistency with xml settings
        length_exp_file = data.shape[0]
        if (len(data.shape) > 1):
            number_columns = data.shape[1]
        else:
            number_columns = 2
        if (NumberX + NumberY > number_columns):
            print " "
            print " "
            print "\t ERROR in function LoadExpFile.LoadExp.LoadDat:"
            print "\t   The number of total number of columns in the"
            print "\t   ascii-file ",filename
            print "\t   is less than the number of x-columns plus number of y-columns"
            print "\t   Please correct xml-file and restart program!"
            print " "
            ok = 1
            return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error


        ## extract experimental data within ranges if NumberRange > 0
        if (NumberRange > 0 and len(data.shape) > 1):
            for k in xrange(length_exp_file):
                xcolumn = data[k,0:NumberX]
                copyflag = "false"                                                          ## set copyflag to false
                for j in xrange(NumberRange):
                    if (NumberX == 1):                                                      ## is there only one column for the x points?
                        if (xcolumn >= xmn[j]):
                            if (xcolumn <= xmx[j]):
                                copyflag = "true"
                    else:                                                                   ## no? Continue?
                        copyflagcolumn = 0
                        for klm in xrange(len(xcolumn)):
                            if (xcolumn[klm] >= xmn[j][klm]):
                                if (xcolumn[klm] <=xmx[j][klm]):
                                    copyflagcolumn += 1
                        if (copyflagcolumn == NumberX):
                            copyflag = "true"
                    if (copyflag == "true"):                                                ## if experimental x point is within a given range ..
                        lengthexpdata += 1                                                  ## increase length counter
                        exp_data_x.append(xcolumn)                                          ## append current x point
                        exp_data_y.append(data[k,NumberX:NumberX + NumberY])

                        # Debug:
                        # print '>>',k,NumberX,NumberX + NumberY
                        # print '>>',data[k,NumberX:NumberX + NumberY]
                        # sys.exit(0)

                        if (error == "yes"):                                                ## if error flag is set to yes get experimental error
                            exp_data_error.append(data[k,NumberX + NumberY:NumberX + NumberY + NumberY])
                        else:                                                               ## otherwise, set error of experimental files to zero
                            exp_data_error.append(data[k,NumberX:NumberX + NumberY] * 0)
                        break


            ## warning if there is no data within the range(s)
            if (lengthexpdata == (-1) and len(data.shape) > 1):
                print " "
                print " "
                print "\t ERROR in function LoadExpFile.LoadExp.LoadDat:"
                print "\t   There is no data within the given ranges!"
                print "\t   Please correct xml-file and restart program!"
                print " "
                ok = 1
                return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error

        else:
            lengthexpdata = length_exp_file - 1                                             ## get number of x-columns
            if (len(data.shape) > 1):
                exp_data_x = data[:,0:NumberX]                                              ## get x-columns
                exp_data_y = data[:,NumberX:NumberX + NumberY]                              ## get y-columns
                if (error == "yes"):                                                        ## if error flag is set to yes get experimental error
                    exp_data_error = data[:,NumberX + NumberY:NumberX + NumberY + NumberY]
                else:                                                                       ## otherwise, define error array and set error to zero
                    exp_data_error = data[:,NumberX:NumberX + NumberY] * 0
            else:
                lengthexpdata = 0
                exp_data_x = []
                exp_data_y = []
                exp_data_error = []
                exp_data_x.append(data[0:NumberX])                                          ## get x-columns
                exp_data_y.append(data[NumberX:NumberX + NumberY])                          ## get y-columns
                if (error == "yes"):                                                        ## if error flag is set to yes get experimental error
                    exp_data_error.append(data[NumberX + NumberY:NumberX + NumberY + NumberY])
                else:                                                                       ## otherwise, define error array and set error to zero
                    exp_data_error.append(data[NumberX:NumberX + NumberY] * 0)


            ## warning if there is no data within the range(s)
            if (lengthexpdata == 0 and len(data.shape) > 1):
                print " "
                print " "
                print "\t ERROR in function LoadExpFile.LoadExp.LoadDat:"
                print "\t   There is no data within the given ranges!"
                print "\t   Please correct xml-file and restart program!"
                print " "
                ok = 1
                return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error


        # Debug:
        # print
        # print
        # print "lengthexpdata = ",lengthexpdata
        # print "exp_data_x[0] = ",exp_data_x[0]
        # print "exp_data_x[lengthexpdata] = ",exp_data_x[lengthexpdata]
        # print "exp_data_y[0] = ",exp_data_y[0]
        # print "exp_data_y[0][lengthexpdata] = ",exp_data_y[lengthexpdata]
        # print '>>>',lengthexpdata,len(exp_data_x)
        # sys.exit(0)


        ## print that you are finished
        if (print_flag != "false"):
            print "done!"


        ## define return variables
        return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error
    ##----------------------------------------------------------------------------------------------------------------------------------------------------



    ##****************************************************************************************************************************************************
    def LoadFits(self, i, CounterFitsFile, print_flag):
        ## load contents of fits-file(s)
        """

        function fits:          load .fits files containing experimental data using the setting given in the xml-file

        input parameters:       i:                          index of experimental data file
                                CounterFitsFile:            counter for fits files
                                print_flag:                 flag for printing message to screen

        required parameters:    NumberExpFiles:             number of fits-files
                                FileNamesExpFiles:          path and file name of each fits-file
                                NumberHDU:                  number of HDU for each fits-file
                                NumberExpRanges:            number of selected experimental ranges
                                MinExpRange:                min of each experimental range
                                MaxExpRange:                max of each experimental range

        output parameters:      ok:                         status of subroutine
                                lengthexpdata:              number of lines in the experimental ascii-file
                                exp_data_x:                 experimental data x column
                                exp_data_y:                 experimental data y column
                                exp_data_error:             error of experimental data y column

        working variables:      k:                          loop counter
                                filename:                   temp variable for file name of the ith fits-file
                                hdulist:                    temp variable containing the whole ith fits-file
                                fitsdata:                   temp variable containing the data of the ith fits-file
                                HDU:                        temp variable for number of HDU in the ith fits-file
                                length_exp_file:            temp variable containing the total number of lines in the ith fits-file (without header lines)
                                number_columns:             temp variable containing the total number of columns in the ith fits-file
                                dimension:                  temp variable containing the dimension of the data of the ith fits-file
                                NumDimensions:              number of dimensions
                                fitshead:                   header of the ith fits file
                                error:                      temp variable containing the error flag
                                NumberRange:                temp variable containing the number of ranges in the ith fits-file
                                xmn:                        array containing all lower limits of the ranges of the ith fits-file
                                xmx:                        array containing all upper limits of the ranges of the ith fits-file
                                exp_data_x:                 temp variable containing experimental data x column of the ith fits-file
                                exp_data_y:                 temp variable containing experimental data y column of the ith fits-file
                                xcolumn:                    temp variable containing the x-columns of the kth line in the ith fits-file
                                copyflag:                   flag for extracting experimental data within selected ranges in the ith fits-file
                                copyflagcolumn:             flag for extracting experimental data within selected ranges in the ith fits-file
                                element:                    flag for extracting experimental data within selected ranges in the ith fits-file


        IMPORTANT:      - The function xml() has to be called before the call of LoadFits()

        """
        ## set status to 0
        ok = 0


        ## construct exp_data array
        exp_data_x = []
        exp_data_y = []
        exp_data_error = []
        lengthexpdata = -1


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## open FITS data file and get range definitions
        filename = self.FileNamesExpFiles[i]
        if (print_flag == "true"):
            print "\t Reading fits file " + filename + " ..",
        hdulist = pyfits.open(filename)
        if (print_flag == "true"):
            print "done!"


        ## get HDU
        HDU = self.NumberHDU[CounterFitsFile]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get number of ranges and range definitions
        NumberRange = self.NumberExpRanges[i]
        if (NumberRange > 0):
            xmn = self.MinExpRange[i]
            xmx = self.MaxExpRange[i]

            # Debug:
            # print "xmn = ",xmn
            # print "xmx = ",xmx
            # print "\t HDU = ",HDU
            # print hdulist.info()


            ## check if range definition correct
            if (len(xmn) != NumberRange):
                print " "
                print " "
                print "\t ERROR in function LoadExpFile.LoadFits:"
                print "\t   The dimension of the definition of the content of "
                print "\t   the <MinExpRange></MinExpRange> tag in the xml-file"
                print "\t   for the import settings does not coincident with the"
                print "\t   dimension of the loaded fits-file!"
                print "\t   Please correct the xml-file and restart program!"
                print " "
                ok = 1
                return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error
            elif (len(xmx) != NumberRange):
                print " "
                print " "
                print "\t ERROR in function LoadExpFile.LoadFits:"
                print "\t   The dimension of the definition of the content of "
                print "\t   the <MaxExpRange></MaxExpRange> tag in the xml-file"
                print "\t   for the import settings does not coincident with the"
                print "\t   dimension of the loaded fits-file!"
                print "\t   Please correct the xml-file and restart program!"
                print " "
                ok = 1
                return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## verify header, fix if necessary and save header
        hdulist.verify('fix')
        fitshead = hdulist[HDU].header
        self.FitsHeader.append(fitshead)


        ## determine fits type (image or table)
        # self.FitsTyp.append("Table")
        self.FitsTyp.append("Image")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get data of FITS file
        fitsdata = hdulist[HDU].data


        ## set array containing dimension of each axis
        dimension = fitsdata.shape
        self.FitsFileDimensions.append(dimension)
        NumDimensions = len(dimension)
        if (NumDimensions > 3):
            print " "
            print "\t Error in subroutine fits:"
            print "\t   Can not import fits-file with more than 4 dimensions."
            print "\t   Please select other file and restart program."
            print " "
            print "\t   Program aborted"
            ok = 1
            return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error

        # Debug:
        #    print "\n\t dimension = ",dimension
        #    print "len(fitsdata[:, 0, 0]) = ", len(fitsdata[:, 0, 0])
        #    print "len(fitsdata[0, :, 0]) = ", len(fitsdata[0, :, 0])
        #    print "len(fitsdata[0, 0, :]) = ", len(fitsdata[0, 0, :])


        ## print what you do
        if (print_flag == "true"):
            print "\t Number of columns = ", NumDimensions
            print "\t Dimension = ", dimension
            sys.stdout.flush()


        ##================================================================================================================================================
        ## get FITS header parameter
        LocalLengthList = []
        LocalValueOfFirstPointList = []
        LocalCRPIXList = []
        LocalCRVALList = []
        LocalCDELTList = []
        LocalCROTList = []
        LocalCUNITList = []
        LocalPixelTolerance = []


        ## read header information
        ## get number of axes
        if (print_flag == "true"):
            print "\t Reading data ..",
            sys.stdout.flush()
        try:
            NAXIS = pyfits.getval(filename, 'NAXIS', 0)
        except KeyError, err:
            NAXIS = 1

        # Debug:
        # print '\n\nNAXIS = ',NAXIS


        ## get scale parameter BSCALE
        try:
            BSCALE = pyfits.getval(filename, 'BSCALE', 0)
        except KeyError, err:
            BSCALE = 1

        # Debug:
        # print '\n\nBSCALE = ',BSCALE


        ## get zero parameter BZERO
        try:
            BZERO = pyfits.getval(filename, 'BZERO', 0)
        except KeyError, err:
            BZERO = 0.0

        # Debug:
        # print '\n\nBZERO = ',BZERO


        ## get unit parameter BUNIT
        try:
            BUNIT = pyfits.getval(filename, 'BUNIT', 0)
        except KeyError, err:
            BUNIT = ""

        # Debug:
        # print '\n\nBUNIT = ',BUNIT


        ## get rest frequency RESTFRQ
        try:
            RESTFRQ = pyfits.getval(filename, 'RESTFRQ', 0)
        except KeyError, err:
            try:
                RESTFRQ = pyfits.getval(filename, 'RESTFREQ', 0)
            except KeyError, err:
                RESTFRQ = 0
        # RESTFRQ = RESTFRQ * 1.e-6

        # Debug:
        # print '\n\nRESTFRQ = ',RESTFRQ


        ## analyze header
        NumDimensions = 0
        for i in xrange(NAXIS):                                                                 ## loop over all dimensions
            try:
                lenAxis = pyfits.getval(filename, 'NAXIS' + str(i + 1), 0)                      ## read CRPIX
            except KeyError, err:
                lenAxis = 1
            if (lenAxis > 0):                                                                   ## read only those entries with more than 1 point
                LocalLengthList.append(lenAxis)
                NumDimensions += 1


                ## get header parameters
                try:
                    CRPIX = pyfits.getval(filename, 'CRPIX' + str(i + 1), 0)                    ## location of a reference point along axis 1
                except KeyError, err:
                    CRPIX = 1
                LocalCRPIXList.append(CRPIX)
                try:
                    CRVAL = pyfits.getval(filename, 'CRVAL' + str(i + 1), 0)                    ## the value of the coordinate at the reference point CRPIX1
                except KeyError, err:
                    CRVAL = 0
                LocalCRVALList.append(CRVAL)
                try:
                    CDELT = pyfits.getval(filename, 'CDELT' + str(i + 1), 0)                    ## partial derivative of the coordinate with respect to the
                except KeyError, err:                                                           ## pixel index, evaluated at the reference point CRPIX
                    CDELT = 1
                LocalCDELTList.append(CDELT)
                LocalPixelTolerance.append(abs(CDELT) * self.PixelTolScalFactor)
                try:
                    CROT = pyfits.getval(filename, 'CROT' + str(i + 1), 0)                      ## indicate a rotation from a standard coordinate system
                except KeyError, err:
                    CROT = 0
                LocalCROTList.append(CROT)
                try:
                    CUNIT = pyfits.getval(filename, 'CUNIT' + str(i + 1), 0)                    ## indicate the unit of the current axis
                except KeyError, err:
                    CUNIT = ""
                LocalCUNITList.append(CUNIT)


                ## calculate value at first pixel
                ValueOfFirstPoint = CRVAL - ((CRPIX - 1) * CDELT)
                LocalValueOfFirstPointList.append(ValueOfFirstPoint)

                # Debug:
                #    print "\n\ni = ", i
                #    print "lenAxis = ", lenAxis
                #    print "CRPIX = ", CRPIX
                #    print "CRVAL = ", CRVAL
                #    print "CDELT = ", CDELT
                #    print "CROT = ", CROT
                #    print "CUNIT = ", CUNIT
                #    print "ValueOfFirstPoint = ", ValueOfFirstPoint

        # Debug:
        # print "\n\n\nlength = ", length


        ## close fits-file
        hdulist.close()

        # Debug:
        # print "xrange(dimension[0]) = ", xrange(dimension[0])
        # print "xmn[0] = ", xmn[0]
        # print "xmx[0] = ", xmx[0]


        ## store FITS header parameter to class variables
        self.FITSHeaderAllLengths.append(LocalLengthList)
        self.FITSHeaderAllValueOfFirstPoint.append(LocalValueOfFirstPointList)
        self.FITSHeaderAllDimensions.append(dimension)
        self.FITSHeaderAllCRPIX.append(LocalCRPIXList)
        self.FITSHeaderAllCRVAL.append(LocalCRVALList)
        self.FITSHeaderAllCDELT.append(LocalCDELTList)
        self.FITSHeaderAllCROT.append(LocalCROTList)
        self.FITSHeaderAllCUNIT.append(LocalCUNITList)
        self.PixelTolerance.append(LocalPixelTolerance)


        ## check if data has to be transposed or not
        LocalTransposeFlag = True
        self.TransposeFlag.append(LocalTransposeFlag)
        fitsdata = numpy.transpose(fitsdata)

        # Debug:
        #    print "\n\t dimension = ",dimension
        #    print "len(fitsdata[:, 0, 0]) = ", len(fitsdata[:, 0, 0])
        #    print "len(fitsdata[0, :, 0]) = ", len(fitsdata[0, :, 0])
        #    print "len(fitsdata[0, 0, :]) = ", len(fitsdata[0, 0, :])
        #    sys.exit(0)


        ##================================================================================================================================================
        ## get FITS data


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine length and first data point of current range in each dimension
        NumRan = max(1, NumberRange)
        for ran in xrange(NumRan):                                                      ## loop over all ranges in the current exp. file
            ListCoord = []
            ListIndies = []
            for DimID in xrange(NumDimensions):                                             ## loop over all dimensions
                LocalCoord = []
                LocalIndies = []

                # Debug:
                # print "\n\n\nDimID = ", DimID
                # print "ran = ", ran
                # print "LocalPixelTolerance[DimID] = ", LocalPixelTolerance[DimID]


                ## special handling if no range is defined
                if (NumberRange == 0):
                    for l in xrange(LocalLengthList[DimID]):
                        llll = LocalValueOfFirstPointList[DimID] + (l * LocalCDELTList[DimID])
                        LocalCoord.append(llll)
                        LocalIndies.append(l)


                ## general case:
                else:
                    if (NumDimensions == 1):
                        for l in xrange(LocalLengthList[DimID]):
                            llll = LocalValueOfFirstPointList[DimID] + (l * LocalCDELTList[DimID])
                            if (abs(xmn[ran] - llll) < LocalPixelTolerance[DimID] or (xmn[ran] <= llll and llll <= xmx[ran]) \
                                or abs(xmx[ran] - llll) < LocalPixelTolerance[DimID]):
                                LocalCoord.append(llll)
                                LocalIndies.append(l)
                    else:
                        for l in xrange(LocalLengthList[DimID]):
                            llll = LocalValueOfFirstPointList[DimID] + (l * LocalCDELTList[DimID])
                            if (abs(xmn[ran][DimID] - llll) < LocalPixelTolerance[DimID] or (xmn[ran][DimID] <= llll and llll <= xmx[ran][DimID]) \
                                or abs(xmx[ran][DimID] - llll) < LocalPixelTolerance[DimID]):
                                LocalCoord.append(llll)
                                LocalIndies.append(l)


                ## store indices for each data point
                ListCoord.append(LocalCoord)
                ListIndies.append(LocalIndies)

            # Debug:
            # print "ListCoord[0] = ", ListCoord[0]
            # print "len(ListCoord[0]) = ", len(ListCoord[0])
            # print "ListCoord[1] = ", ListCoord[1]
            # print "len(ListCoord[1]) = ", len(ListCoord[1])
            # print "ListCoord[2] = ", ListCoord[2]
            # print "len(ListCoord[2]) = ", len(ListCoord[2])


            ## store data
            for ix, xxxx in enumerate(ListCoord[0]):
                x = ListIndies[0][ix]
                if (NumDimensions == 1):
                    lengthexpdata += 1                                                      ## get number of x-columns
                    exp_data_x.append([xxxx])                                               ## get x-columns
                    exp_data_y.append([fitsdata[x]])                                        ## get y-columns
                    exp_data_error.append([0])
                else:
                    for iy, yyyy in enumerate(ListCoord[1]):
                        y = ListIndies[1][iy]
                        if (NumDimensions == 2):
                            lengthexpdata += 1                                              ## get number of x-columns
                            exp_data_x.append([xxxx, yyyy])                                 ## get x-columns
                            exp_data_y.append([fitsdata[x, y]])                             ## get y-columns
                            exp_data_error.append([0])
                        else:
                            for iz, zzzz in enumerate(ListCoord[2]):
                                z = ListIndies[2][iz]
                                if (NumDimensions == 3):
                                    lengthexpdata += 1                                      ## get number of x-columns
                                    exp_data_x.append([xxxx, yyyy, zzzz])                   ## get x-columns
                                    exp_data_y.append([fitsdata[x, y, z]])                  ## get y-columns
                                    exp_data_error.append([0])
                                else:
                                    for it, tttt in enumerate(ListCoord[3]):
                                        t = ListIndies[3][it]
                                        if (NumDimensions == 4):
                                            lengthexpdata += 1                              ## get number of x-columns
                                            exp_data_x.append([xxxx, yyyy, zzzz, tttt])     ## get x-columns
                                            exp_data_y.append([fitsdata[x, y, z, t]])       ## get y-columns
                                            exp_data_error.append([0])
        # Debug:
        #    print "\n\nlengthexpdata = ", lengthexpdata
        #    print "exp_data_x = ", exp_data_x
        #    print "exp_data_y = ", exp_data_y
        #    print "exp_data_error = ", exp_data_error


        ## everything is fine
        if (print_flag == "true"):
            print "done!"


        ## define return parameters
        return ok, lengthexpdata, exp_data_x, exp_data_y, exp_data_error
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    ##
    ##                                              Write data
    ##
    ##++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


    ##****************************************************************************************************************************************************
    def WriteFile(self, ExpInputFormat, print_flag, NumberExpFiles, LengthExpRange, ColumnY, ExpDataX, FitFunctionValues, Chi2SaveFlag, Chi2Values, \
                  FilenameExtension):
        ## write contents to ascii-file(s)
        """

        function WriteFile:     write values of the fit function within the ranges to file

        input parameters:       ExpInputFormat:             defines if output file has ASCII or fits format
                                print_flag:                 flag for printing message to screen
                                alldata:                    if all == true the x and y values are written to file
                                NumberExpFiles:             number of dat-files
                                LengthExpRange:             number of lines in the experimental dat-file
                                ColumnY:                    number of "y"-columns
                                ExpDataX:                   experimental data x column
                                FitFunctionValues:          values of the fit function
                                Chi2SaveFlag:               flag for saving chi^2 values
                                Chi2Values:                 values of chi^2
                                FilenameExtension:          extension of the parameter file name

        output parameters:      ok:                         status of subroutine

        """


        ## set status to 0
        ok = 0


        ## print what you do
        if (print_flag != "false"):
            print "Write values of the fit function to files."


        ## read experimental data of each ascii-file and define output variables
        NumberFiles = self.NumberExpFiles
        CounterASCIIFile = (-1)
        CounterFitsFile = (-1)
        for i in xrange(NumberFiles):
            FileFormat = ExpInputFormat[i].lower()

            # Debug:
            # print "i, FileFormat = ", i, FileFormat


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## write values of the fit function to file
            ## load experimental data using several import functions depending on the file format
            if (FileFormat == "dat" or FileFormat == "ascii" or FileFormat == "xclassascii" or FileFormat == "cso" or FileFormat == "xclass"):
                                                                                            ## if ascii file selected, use ascii-file function to
                                                                                            ##   import experimental data
                CounterASCIIFile += 1
                alldata = "true"                                                            ## write all data x and y values to file
                ok = self.WriteDat(i, CounterASCIIFile, print_flag, alldata, NumberExpFiles, LengthExpRange, ColumnY, ExpDataX, FitFunctionValues, \
                                   Chi2SaveFlag, Chi2Values, FilenameExtension)
            elif (FileFormat == "fits" or FileFormat == "xclassfits"):                      ## if fits file selected, use fits-file function 
                                                                                            ##   to import experimental data
                CounterFitsFile += 1
                ok = self.WriteFits(i, CounterFitsFile, print_flag, NumberExpFiles, LengthExpRange, ColumnY, ExpDataX, FitFunctionValues, \
                                    Chi2SaveFlag, Chi2Values, FilenameExtension)


            ## does an error occur
            if (ok != 0):
                return ok
            if (print_flag != "false"):
                print
                print


        ## define return parameters
        return ok
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    def WriteDat(self, ObsDataFileIndex, CounterASCIIFile, print_flag, alldata, number_files, LengthExpRange, ColumnY, ExpDataX, FitFunctionValues, \
                 Chi2SaveFlag, Chi2Values, FilenameExtension):
        ## write contents to ascii-file(s)
        """

        function WriteDat:      write values of the fit function within the ranges to ascii-file

        input parameters:       ObsDataFileIndex:           index of experimental data file
                                CounterASCIIFile:           counter for ascii files
                                print_flag:                 flag for printing message to screen
                                alldata:                    if all == true the x and y values are written to file
                                number_files:               number of ascii-files
                                LengthExpRange:             number of lines in the experimental ascii-file
                                ColumnY:                    number of "y"-columns
                                ExpDataX:                   experimental data x column
                                FitFunctionValues:          values of the fit function
                                Chi2SaveFlag:               flag for saving chi^2 values
                                Chi2Values:                 values of chi^2
                                FilenameExtension:          extention of the file name

        required parameters:    NumberExpFiles:             number of ascii-files
                                FileNamesExpFiles:          path and file name of each ascii file
                                NumberHeaderLines:          number of header lines for each ascii-file
                                SeparatorColumns:           column separator for each ascii-file
                                NumberColumnsX:             number of columns belonging to the x-positions
                                NumberColumnsY:             number of columns belonging to the y-positions

        output parameters:      ok:                         status of subroutine

        working variables:      j:                          loop counter
                                filename:                   temp variable for file name of the ith ascii-file
                                number_header_lines:        temp variable for number of header lines in the ith 
                                                            ascii-file
                                seperator:                  temp variable for column separator in the ith 
                                                            ascii-file
                                data:                       temp variable containing the whole ith ascii-file
                                length_exp_file:            temp variable containing the total number of lines 
                                                            in the ith ascii-file (without header lines)
                                error:                      temp variable containing the error flag
                                LengthFile:                 temp variable containing the number of lines in the 
                                                            ith ascii-file
                                column:                     temp variable containing the number of "y"-columns 
                                                            in the ith ascii-file
                                line:                       jth line of the ith ascii file


        IMPORTANT:      - The function xml() has to be called before the call of WriteDat()

        """


        ## set status to 0
        ok = 0


        ## set working variables for the ith ascii-file
        filename = self.FileNamesExpFiles[ObsDataFileIndex]
        number_header_lines = self.NumberHeaderLines[CounterASCIIFile]
        seperator = self.SeparatorColumns[CounterASCIIFile]
        NumberX = self.NumberColumnsX[CounterASCIIFile]
        NumberY = self.NumberColumnsY[CounterASCIIFile]
        error = self.ErrorY[CounterASCIIFile]
        LengthFile = LengthExpRange[ObsDataFileIndex]
        column = ColumnY[CounterASCIIFile]


        ## warning if there is no data within the range(s)
        if (LengthFile == 0):
            print " "
            print " "
            print "\t ERROR in function LoadExpFile.LoadExp.WriteDat:"
            print "\t   There is no data within the given ranges!"
            print "\t   Please correct xml-file and restart program!"
            print " "
            ok = 1
            return ok
        else:


            ## modify filename
            if filename.endswith(".fits"):
                filename = filename.replace(".fits", ".dat")
            else:
                fileExt = "." + FilenameExtension + ".dat"
                if filename.endswith(".dat"):
                    filename = filename.replace(".dat", fileExt)
                else:
                    filename = filename.strip()
                    filename = filename + fileExt


            ## print what you do
            if (print_flag != "false"):
                print "\t Writing data to ASCII file " + filename + " ..",

            # Debug:
            # print
            # print 'NumberFiles = ',NumberFiles
            # print 'LengthFile = ',LengthFile
            # print 'column = ',column
            # print 'FitFunctionValues = ', FitFunctionValues
            # print 'column = ', column
            # print 'LengthFile = ', LengthFile


            ## construct output array
            outdat = []
            for j in xrange(LengthFile):
                line = []
                if (alldata == "true"):
                    for k in xrange(len(ExpDataX[ObsDataFileIndex][j])):
                        line.append(ExpDataX[ObsDataFileIndex][j][k])
                for k in xrange(column):
                    line.append(FitFunctionValues[ObsDataFileIndex][j][k])
                outdat.append(line)


            ## write ascii-file using numpy function loadtxt
            outdatFile = open(filename, 'w')
            for line in outdat:
                finalLine = ""
                for columns in line:
                    # finalLine = finalLine + seperator + str(columns)
                    finalLine = finalLine + " " + str(columns)
                outdatFile.write(finalLine + "\n")
            outdatFile.close()

            #if seperator == " ":
            #    numpy.savetxt(filename, outdat)
            #else:
            #    numpy.savetxt(filename, outdat, fmt='%.9f', delimiter=seperator)

            # Debug:
            # print " "
            # print " "
            # print "data[1] = ",data[1]
            # print "data.shape = ",data.shape


            ## save values of chi^2
            if (Chi2SaveFlag == "true"):

                ## modify filename
                fileExt = "." + FilenameExtension + ".dat"
                fileExtOut = "." + FilenameExtension + ".chi2.dat"
                filename = filename.replace(fileExt, fileExtOut)


                ## print what you do
                if (print_flag != "false"):
                    print "done!"
                    print "\t Writing chi^2 values to file " + filename + " ..",


                ## construct output array
                outdat = []
                for j in xrange(LengthFile):
                    line = []
                    if (alldata == "true"):
                        for k in xrange(len(ExpDataX[ObsDataFileIndex][j])):
                            line.append(ExpDataX[ObsDataFileIndex][j][k])
                    for k in xrange(column):
                        line.append(Chi2Values[ObsDataFileIndex][j][k])
                    outdat.append(line)


                ## write ascii-file using numpy function loadtxt
                outdatFile = open(filename, 'w')
                for line in outdat:
                    finalLine = ""
                    for columns in line:
                        # finalLine = finalLine + seperator + str(columns)
                        finalLine = finalLine + " " + str(columns)
                    outdatFile.write(finalLine + "\n")
                outdatFile.close()

                #if (seperator == " "):
                #    numpy.savetxt(filename, outdat)
                #else:
                #    numpy.savetxt(filename, outdat, fmt='%.9f', delimiter=seperator)


            ## print that you are finished
            if (print_flag != "false"):
                print "done!"

        return ok
    ##----------------------------------------------------------------------------------------------------------------------------------------------------



    ##****************************************************************************************************************************************************
    def WriteFits(self, ObsDataFileIndex, CounterFitsFile, print_flag, number_files, LengthExpRange, ColumnY, ExpDataX, FitFunctionValues, Chi2SaveFlag, \
                  Chi2Values, FilenameExtension):
        ## write contents to fits-file(s)
        """

        function WriteFits:     write values of the fit function within the ranges to fits-file

        input parameters:       ObsDataFileIndex:           index of experimental data file
                                CounterFitsFile:            counter for fits files
                                print_flag:                 flag for printing message to screen
                                number_files:               number of fits-files
                                LengthExpRange:             number of lines in the experimental fits-file
                                ColumnY:                    number of "y"-columns
                                ExpDataX:                   experimental data x column
                                FitFunctionValues:          values of the fit function
                                Chi2SaveFlag:               flag for saving chi^2 values
                                Chi2Values:                 values of chi^2
                                FilenameExtension:          extention of the parameter file name

        required parameters:    NumberExpFiles:             number of fits-files
                                FileNamesExpFiles:          path and file name of each fits file
                                FitsHeader:                 header of each fits file
                                HDU:                        HDU number

        output parameters:      ok:                         status of the subroutine

        working variables:      j:                          loop counter
                                filename:                   temp variable for file name of the ith fits-file
                                number_header_lines:        temp variable for number of header lines in the ith fits-file
                                seperator:                  temp variable for column separator in the ith fits-file
                                data:                       temp variable containing the whole ith fits-file
                                length_exp_file:            temp variable containing the total number of lines in the ith fits-file (without header lines)
                                LengthFile:                 temp variable containing the number of lines in the ith fits-file
                                column:                     temp variable containing the number of "y"-columns in the ith fits-file
                                line:                       jth line of the ith fits file
                                HDUNumber:                  HDU number


        IMPORTANT:      - The function xml() has to be called before the call of WriteDat()

        """


        ## set status to 0
        ok = 0


        ## set working variables for the ith fits-file
        filename = self.FileNamesExpFiles[ObsDataFileIndex]
        HDU = self.NumberHDU[CounterFitsFile]
        LengthFile = LengthExpRange[ObsDataFileIndex]
        column = ColumnY[CounterFitsFile]
        LocalFITSHeader = self.FitsHeader[CounterFitsFile]
        NumberRange = self.NumberExpRanges[ObsDataFileIndex]                                               ## read number of ranges
        if (NumberRange > 0):
            xmn = self.MinExpRange[ObsDataFileIndex]
            xmx = self.MaxExpRange[ObsDataFileIndex]

        # Debug:
        #    print "\n\n"
        #    print "ObsDataFileIndex = ", ObsDataFileIndex
        #    print "xmn = ", xmn
        #    print "xmx = ", xmx


        ## get FITS header parameter for current obs. data file
        LocalValueOfFirstPointList = self.FITSHeaderAllValueOfFirstPoint[ObsDataFileIndex]
        LocalDimension = self.FITSHeaderAllDimensions[ObsDataFileIndex]
        LocalLengthList = self.FITSHeaderAllLengths[ObsDataFileIndex]
        LocalCRPIXList = self.FITSHeaderAllCRPIX[ObsDataFileIndex]
        LocalCRVALList = self.FITSHeaderAllCRVAL[ObsDataFileIndex]
        LocalCDELTList = self.FITSHeaderAllCDELT[ObsDataFileIndex]
        LocalCROTList = self.FITSHeaderAllCROT[ObsDataFileIndex]
        LocalPixelTolerance = self.PixelTolerance[ObsDataFileIndex]

        # Debug:
        # print "\n\nfilename = ", filename
        # print "HDU = ", HDU
        # print "LengthFile = ", LengthFile
        # print "column = ", column
        # print "LocalFITSHeader = ", LocalFITSHeader
        # print "NumberRange = ", NumberRange
        # print "ObsDataFileIndex = ", ObsDataFileIndex
        # print "LengthFile = ", LengthFile


        ## warning if there is no data within the range(s)
        if (LengthFile == 0):
            print " "
            print " "
            print "\t ERROR in function LoadExpFile.LoadExp.WriteFits:"
            print "\t   There is no data within the given ranges!"
            print "\t   Please correct xml-file and restart program!"
            print " "
            ok = 1
            return ok
        else:
            NumRan = max(1, NumberRange)

            # Debug:
            # print "NumRan = ", NumRan


            ##============================================================================================================================================
            ## create FITS images for each data range and obs. data file
            k = -1
            for ran in xrange(NumRan):                                                      ## loop over all ranges in the current exp. file


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## modify filename
                if (NumberRange < 2):
                    if filename.endswith(".dat"):
                        filenameNew = filename.replace(".dat", ".fits")
                        if (Chi2SaveFlag == "true"):
                            Chi2Filename = filename.replace(".dat", ".out.chi2.fits")
                    else:
                        fileExt = "." + FilenameExtension + ".fits"
                        filenameNew = filename.replace(".fits", fileExt)
                        if (Chi2SaveFlag == "true"):
                            Chi2Filename = filenameNew.replace(fileExt, "." + FilenameExtension + ".chi2.fits")
                else:
                    # Debug:
                    # print "\n\n\nran, xmn[ran] = ", ran, xmn[ran]
                    # print "filename = ", filename


                    if (isinstance(xmn[ran], list)):
                        xmnString = ""
                        xmxString = ""
                        for DimID in xrange(len(xmn[ran])):
                            xmnString += "_" + str(xmn[ran][DimID])
                            xmxString += "_" + str(xmx[ran][DimID])
                    else:
                        xmnString = "_" + str(xmn[ran])
                        xmxString = "_" + str(xmx[ran])
                    RangeExten = "_" + str(xmnString) + "_-" + str(xmxString)

                    # Debug:
                    # print "RangeExten = ", RangeExten


                    if filename.endswith(".dat"):
                        filenameNew = filename.replace(".dat", RangeExten + ".fits")
                        if (Chi2SaveFlag == "true"):
                            Chi2Filename = filename.replace(".dat", RangeExten + ".out.chi2.fits")
                    else:
                        fileExt = "." + FilenameExtension + "_" + RangeExten + ".fits"
                        filenameNew = filename.replace(".fits", fileExt)
                        if (Chi2SaveFlag == "true"):
                            Chi2Filename = filenameNew.replace(fileExt, "." + FilenameExtension + RangeExten + ".chi2.fits")


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## delete old output file
                if (os.path.exists(filenameNew)):
                    os.remove(filenameNew)
                if (os.path.exists(Chi2Filename)):
                    os.remove(Chi2Filename)


                ## print what you do
                if (print_flag != "false"):
                    print "\t Writing fits-file " + filenameNew + " ..",


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## construct output array
                NumberDim = len(ExpDataX[ObsDataFileIndex][0])

                # Debug:
                #    print "\n\nNumberDim = ", NumberDim
                #    print "xmn = ", xmn
                #    print "xmx = ", xmx
                #    print "filenameNew = ", filenameNew
                #    print "NumberRange = ", NumberRange


                ## determine length and first data point of current range in each dimension
                ListCoord = []
                ListIndies = []
                LengthArray = []
                for DimID in xrange(NumberDim):                                             ## loop over all dimensions
                    LocalCoord = []
                    LocalIndies = []

                    # Debug:
                    # print "\n\n\nDimID = ", DimID
                    # print "LocalLengthList[DimID] = ", LocalLengthList[DimID]
                    # print "LocalPixelTolerance[DimID] = ", LocalPixelTolerance[DimID]


                    ## special handling if no range is defined
                    if (NumberRange == 0):
                        for l in xrange(LocalLengthList[DimID]):
                            llll = LocalValueOfFirstPointList[DimID] + (l * LocalCDELTList[DimID])
                            LocalCoord.append(llll)
                            LocalIndies.append(l)


                    ## general case:
                    else:
                        if (NumberDim == 1):
                            for l in xrange(LocalLengthList[DimID]):
                                llll = LocalValueOfFirstPointList[DimID] + (l * LocalCDELTList[DimID])
                                if (abs(xmn[ran] - llll) < LocalPixelTolerance[DimID] or (xmn[ran] <= llll and llll <= xmx[ran]) \
                                    or abs(xmx[ran] - llll) < LocalPixelTolerance[DimID]):
                                    LocalCoord.append(llll)
                                    LocalIndies.append(l)
                        else:
                            for l in xrange(LocalLengthList[DimID]):
                                llll = LocalValueOfFirstPointList[DimID] + (l * LocalCDELTList[DimID])
                                if (abs(xmn[ran][DimID] - llll) < LocalPixelTolerance[DimID] or (xmn[ran][DimID] <= llll and llll <= xmx[ran][DimID]) \
                                    or abs(xmx[ran][DimID] - llll) < LocalPixelTolerance[DimID]):
                                    LocalCoord.append(llll)
                                    LocalIndies.append(l)


                    ## store indices for each data point
                    ListCoord.append(LocalCoord)
                    ListIndies.append(LocalIndies)
                    LengthArray.append(len(LocalIndies))

                # Debug:
                # print "LengthArray = ", LengthArray
                # print "ListCoord[0] = ", ListCoord[0]
                # print "len(ListCoord[0]) = ", len(ListCoord[0])
                # print "ListCoord[1] = ", ListCoord[1]
                # print "len(ListCoord[1]) = ", len(ListCoord[1])
                # print "ListCoord[2] = ", ListCoord[2]
                # print "len(ListCoord[2]) = ", len(ListCoord[2])


                ## construct array for model function and chi2 values
                outdat = numpy.zeros(LengthArray)
                if (Chi2SaveFlag == "true"):
                    ChiDat = numpy.zeros(LengthArray)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## store data
                for x in xrange(LengthArray[0]):
                    if (NumberDim == 1):
                        k += 1
                        outdat[x] = FitFunctionValues[ObsDataFileIndex][k][0]
                        if (Chi2SaveFlag == "true"):
                            ChiDat[x] = Chi2Values[ObsDataFileIndex][k][0]
                    else:
                        for y in xrange(LengthArray[1]):
                            if (NumberDim == 2):
                                k += 1
                                outdat[x, y] = FitFunctionValues[ObsDataFileIndex][k][0]
                                if (Chi2SaveFlag == "true"):
                                    ChiDat[x, y] = Chi2Values[ObsDataFileIndex][k][0]
                            else:
                                for z in xrange(LengthArray[2]):
                                    if (NumberDim == 3):
                                        k += 1
                                        outdat[x, y, z] = FitFunctionValues[ObsDataFileIndex][k][0]
                                        if (Chi2SaveFlag == "true"):
                                            ChiDat[x, y, z] = Chi2Values[ObsDataFileIndex][k][0]
                                    else:
                                        for t in xrange(LengthArray[3]):
                                            k += 1
                                            outdat[x, y, z, t] = FitFunctionValues[ObsDataFileIndex][k][0]
                                            if (Chi2SaveFlag == "true"):
                                                ChiDat[x, y, z, t] = Chi2Values[ObsDataFileIndex][k][0]


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## write FITS tables
                if (self.FitsTyp[CounterFitsFile] == "Table"):


                    ## create column definition
                    cols = pyfits.Column(name='', format='E')


                    ## create a new binary table HDU object
                    tbhdu = pyfits.new_table(cols)


                    ## create PrimaryHDU object to encapsulate the data outdat
                    HDUNumber = pyfits.PrimaryHDU(outdat)
                    thdulist = pyfits.HDUList([HDUNumber,tbhdu])


                    ## save data and header
                    thdulist.writeto(filenameNew, LocalFITSHeader)


                    ## ChiDat to file
                    if (Chi2SaveFlag == "true"):
                        HDUNumber = pyfits.PrimaryHDU(ChiDat)
                        thdulist = pyfits.HDUList([HDUNumber,tbhdu])
                        thdulist.writeto(Chi2Filename, LocalFITSHeader)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## write FITS images
                elif (self.FitsTyp[CounterFitsFile] == "Image"):
                    NewFITSHeader = copy.deepcopy(LocalFITSHeader)


                    ## change BITPIX if BITPIX != -64
                    if (self.OldFITSCommandFlag):
                        NewFITSHeader.update('bitpix', -32, '')
                    else:
                        NewFITSHeader['bitpix'] = (-32, '')


                    ##************************************************************************************************************************************
                    ## update FITS header
                    for DimID in xrange(NumberDim):                                         ## loop over all dimensions


                        ## update lengths of each axis
                        LocalNumPixels = int(LengthArray[DimID])
                        if (self.OldFITSCommandFlag):
                            NewFITSHeader.update('NAXIS' + str(DimID + 1), LocalNumPixels, '')
                        else:
                            NewFITSHeader['NAXIS' + str(DimID + 1)] = (LocalNumPixels, '')


                        ## update CRPIX of each axis
                        if (self.UseCentralPixelFlag):
                            LocalCRPIX = int(LocalNumPixels / 2.0 + 1.0)
                        else:
                            LocalCRPIX = 1
                        if (self.OldFITSCommandFlag):
                            NewFITSHeader.update('CRPIX' + str(DimID + 1), LocalCRPIX, '')
                        else:
                            NewFITSHeader['CRPIX' + str(DimID + 1)] = (LocalCRPIX, '')


                        ## update CRVAL of each axis
                        LocalCRVAL = ListCoord[DimID][0] + (LocalCRPIX - 1) * LocalCDELTList[DimID]
                        if (self.OldFITSCommandFlag):
                            NewFITSHeader.update('CRVAL' + str(DimID + 1), LocalCRVAL, '')
                        else:
                            NewFITSHeader['CRVAL' + str(DimID + 1)] = (LocalCRVAL, '')


                        ## update CDELT of each axis
                        LocalCDELT = LocalCDELTList[DimID]
                        if (self.OldFITSCommandFlag):
                            NewFITSHeader.update('CDELT' + str(DimID + 1), LocalCDELT, '')
                        else:
                            NewFITSHeader['CDELT' + str(DimID + 1)] = (LocalCDELT, '')


                        ## update CROT of each axis
                        LocalCROT = LocalCROTList[DimID]
                        if (self.OldFITSCommandFlag):
                            NewFITSHeader.update('CROT' + str(DimID + 1), LocalCROT, '')
                        else:
                            NewFITSHeader['CROT' + str(DimID + 1)] = (LocalCROT, '')


                    ##************************************************************************************************************************************
                    ## store data and chi2 values


                    ## save data to new fits file
                    pyfits.writeto(filenameNew, numpy.transpose(outdat), NewFITSHeader)


                    ## ChiDat to file
                    if (Chi2SaveFlag == "true"):
                        pyfits.writeto(Chi2Filename, numpy.transpose(ChiDat), NewFITSHeader)

                else:
                    print " "
                    print " "
                    print "\t ERROR in function LoadExpFile.LoadExp.WriteFits:"
                    print "\t   Unknown fits type!"
                    print "\t   Please correct xml-file and restart program!"
                    print " "
                    ok = 1
                    return ok


                # Debug:
                # print " "
                # print " "
                # print "data[1] = ",data[1]
                # print "data.shape = ",data.shape


                ## print that you are finished
                if (print_flag != "false"):
                    print "done!"
        return ok
##--------------------------------------------------------------------------------------------------------------------------------------------------------

