#!/usr/bin/env python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module reads and writes the model parameter to a xml-file.
##  Copyright (C) 2009 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following subroutines and functions are included in this module:
##
##      - subroutine ReadParameter:                     reads xml-file containing the start values for the fit process
##      - subroutine TrueFalse:                         function return true of false, if val == 0 or val != 0
##      - subroutine WriteParameter:                    writes xml-file containing the start values for the fit process
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    06.22.2009   initial version
##  T. Moeller    16.01.2012   improve documentation of source code
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************

"""
Read model parameter:
---------------------

The package "GetParameterSet" reads and writes the model parameter to a 
xml-file. The names of the model parameters, etc. have to confirm with the
registration file valid for the according model. The names of the model 
parameters must not include the characters "[", "]", and "," (except [[]]).

The tags <uplimit> </uplimit> and <lowlimit> </lowlimit> indicate the upper
and the lower limit of the model parameters, respectively.


Example xml-file (parameter file):
----------------------------------

<?xml version="1.0" encoding="ISO-8859-1"?>
<ModelParameters>
    <NumberParameters>2</NumberParameters>
    <Parameter fit="false">
        <name>EpsilonInfinity</name>
        <value>3.0</value>
        <error> </error>
        <lowlimit>0</lowlimit>
        <uplimit>+inf</uplimit>
    </Parameter>
    <Parameter fit="false">
        <name>NumberOscillators</name>
        <value>1</value>
        <error> </error>
        <lowlimit>0</lowlimit>
        <uplimit>+inf</uplimit>
    </Parameter>
    <Parameter fit="false">
        <name>EigenFrequency</name>
        <value>300.0</value>
        <error> </error>
        <lowlimit>0</lowlimit>
        <uplimit>+inf</uplimit>
    </Parameter>
    <Parameter fit="true">
        <name>PlasmaFrequency</name>
        <value>800.0</value>
        <error> </error>
        <lowlimit>0</lowlimit>
        <uplimit>+inf</uplimit>
    </Parameter>
    <Parameter fit="false">
        <name>Damping</name>
        <value>10.0</value>
        <error> </error>
        <lowlimit>0</lowlimit>
        <uplimit>+inf</uplimit>
    </Parameter>
</ModelParameters>


IMPORTANT:

- The descriptions of each parameter are bracket between the 
  <Parameter></Parameter> tags. These tags have to occur as many times as
  defined in the tag <NumberParameters></NumberParameters>.


- In order to include a parameter into the fitting process set the
  attribute of the tags <Parameter></Parameter> to "true".
  For example: If you want to optimize the value of the Parameter 
               "EpsilonInfinity" set the attribute to "true"

               <Parameter fit="true">
                   <name>EpsilonInfinity</name>
                   <value>3.0</value>
                   <error> </error>
                   <lowlimit>0</lowlimit>
                   <uplimit>+inf</uplimit>
               </Parameter>

               If the value of this parameter should be fixed set
               the attribute to "false":
               <Parameter fit="false">
                   ...

- The tag <error> </error> must occur for every parameter and must have a 
  non-empty value because the value of this tag is replaced by the error
  of the optimized parameter.
  Note, do not remove these tags!


- The number of model parameters defined in this file must be identical with
  the number of parameters defined in the registration file (inlcuding all 
  replications of lines and parameters).


- The names of the model parameter defined in the tags <name></name>
  must be identical with the definitions of the names of the model
  parameter defined in the registration file. Otherwise the program
  stops!
"""


##********************************************************************* load packages ********************************************************************
import numpy                                                                                ## import numpy package
import os.path                                                                              ## load package for file management
import string                                                                               ## load string package
import shutil                                                                               ## import shutil package
import sys                                                                                  ## import sys package
import MAGIXxmlPACKAGE                                                                      ## import xml module from Modules directory
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
def ReadParameter(print_flag, XmlParameterFile):
    ## reads xml-file containing the start values for the fit process
    """

    function ReadParameter: read the xml-file containing the start values of model parameters for the fit process

    input parameters:       print_flag:                 print flag
                            XmlParameterFile:           path and file name of the xml file containing the start values

    output paramters:       ok:                         status of reading
                            FitParametersNumber:        total number of model parameters
                            FitParameters:              values, fit flags, lower and upper limit of each model parameter
                            FitParameterName:           names of the model parameters

    working parameters:     i:                          loop counter
                            printdoneflag:              flag for printing the word "done"
                            XmlFile:                    contents of the xml-file


    Example xml-file (parameter file):
    ----------------------------------

    <?xml version="1.0" encoding="ISO-8859-1"?>
    <ModelParameters>
        <NumberParameters>2</NumberParameters>
        <Parameter fit="false">
            <name>epsilon_infinity</name>
            <value>3.0</value>
            <error> </error>
            <lowlimit>0</lowlimit>
            <uplimit>+inf</uplimit>
        </Parameter>
        <Parameter fit="false">
            <name>number of oscillators</name>
            <value>1</value>
            <error> </error>
            <lowlimit>0</lowlimit>
            <uplimit>+inf</uplimit>
        </Parameter>
        <Parameter fit="false">
            <name>Eigen frequency[[1]]</name>
            <value>300.0</value>
            <error> </error>
            <lowlimit>0</lowlimit>
            <uplimit>+inf</uplimit>
        </Parameter>
        <Parameter fit="true">
            <name>plasma frequency[[1]]</name>
            <value>800.0</value>
            <error> </error>
            <lowlimit>0</lowlimit>
            <uplimit>+inf</uplimit>
        </Parameter>
        <Parameter fit="false">
            <name>damping[[1]]</name>
            <value>10.0</value>
            <error> </error>
            <lowlimit>0</lowlimit>
            <uplimit>+inf</uplimit>
        </Parameter>
    </ModelParameters>

    """

    ## printdoneflag ?
    if (print_flag == "false"):
        printdoneflag = "false"
    else:
        printdoneflag = "true"


    ## set status parameter, reset FitParameter array and reset FitParameterName array
    ok = 0
    FitParametersNumber = 0
    FitParameters = 0
    FitParameterName = " "


    ## read xml-description of the INPUT file
    ## is the defined file a xml-file ?
    if not XmlParameterFile.endswith(".xml"):
        print " "
        print "\t Error in subroutine GetParameterSet.XmlParameterFile!"
        print "\t     The selected file is not a xml-file!"
        print "\t     Please select a .xml and restart the whole program."
        print " "
        ok = 1
        return ok, FitParametersNumber, FitParameters, FitParameterName


    ## print what you do ..
    if (print_flag != "false"):
        print "Read the xml-file containing the start values for the fit process."
        print "\t Open xml-file: " + XmlParameterFile
        print "\t Reading file ..",


    ## open xml-file file and save content of file to variable XmlFile
    XmlFile = MAGIXxmlPACKAGE.XmlObject(fileName = XmlParameterFile)


    ## get number of model parameters
    try:
        FitParametersNumber = XmlFile.ModelParameters.NumberParameters.getValue()
    except AttributeError, err:
        FitParametersNumber = XmlFile.ModelParameters.Section.SubSection.NumberParameters.getValue()


    ## get settings for each model parameter
    FitParameterName = []
    FitParameters = []
    ParameterValues = []
    ParameterFitFlag = []
    ParameterLowLimit = []
    ParameterUpLimit = []
    if (FitParametersNumber == 1):
        try:
            name = XmlFile.ModelParameters.Parameter.name.getValue()
        except AttributeError, err:
            name = XmlFile.ModelParameters.Section.SubSection.Parameter.name.getValue()


        ## test if name include a one of the following commands (XValue, NumberXValue, CurrentExpFile, CurrentYColumn)
        if (string.find(name,"XValue") != -1):
            print " "
            print "\t Error in subroutine GetParameterSet.XmlParameterFile!"
            print "\t     The selected parameter name: ",name
            print "\t     contains the command XValue!"
            print "\t     Please select a another name for the parameter and restart the whole program."
            print " "
            ok = 1
            return ok, FitParametersNumber, FitParameters, FitParameterName

        elif (string.find(name,"NumberXValue") != -1):
            print " "
            print "\t Error in subroutine GetParameterSet.XmlParameterFile!"
            print "\t     The selected parameter name: ",name
            print "\t     contains the command NumberXValue!"
            print "\t     Please select a another name for the parameter and restart the whole program."
            print " "
            ok = 1
            return ok, FitParametersNumber, FitParameters, FitParameterName

        elif (string.find(name,"CurrentExpFile") != -1):
            print " "
            print "\t Error in subroutine GetParameterSet.XmlParameterFile!"
            print "\t     The selected parameter name: ",name
            print "\t     contains the command CurrentExpFile!"
            print "\t     Please select a another name for the parameter and restart the whole program."
            print " "
            ok = 1
            return ok, FitParametersNumber, FitParameters, FitParameterName

        elif (string.find(name,"CurrentYColumn") != -1):
            print " "
            print "\t Error in subroutine GetParameterSet.XmlParameterFile!"
            print "\t     The selected parameter name: ",name
            print "\t     contains the command CurrentYColumn!"
            print "\t     Please select a another name for the parameter and restart the whole program."
            print " "
            ok = 1
            return ok, FitParametersNumber, FitParameters, FitParameterName


        ## read parameter name
        FitParameterName.append(name)
        try:
            fitflag = XmlFile.ModelParameters.Parameter.getAttribute("fit")
        except AttributeError, err:
            fitflag = XmlFile.ModelParameters.Section.SubSection.Parameter.getAttribute("fit")
        ParameterFitFlag.append(fitflag)


        ## read lower limit
        try:
            lowlimit = str(XmlFile.ModelParameters.Parameter.lowlimit.getValue())
        except AttributeError, err:
            lowlimit = str(XmlFile.ModelParameters.Section.SubSection.Parameter.lowlimit.getValue())
        lowlimit = lowlimit.lower()
        lowlimit = lowlimit.strip()
        if (lowlimit == ""):
            lowlimit = 0
        elif (lowlimit == "+inf" or lowlimit == "inf"):
            lowlimit = 1e99
        elif (lowlimit == "-inf"):
            lowlimit = -1e99
        ParameterLowLimit.append(lowlimit)


        ## read upper limit
        try:
            uplimit = str(XmlFile.ModelParameters.Parameter.uplimit.getValue())
        except AttributeError, err:
            uplimit = str(XmlFile.ModelParameters.Section.SubSection.Parameter.uplimit.getValue())
        uplimit = uplimit.lower()
        uplimit = uplimit.strip()
        if (uplimit == ""):
            uplimit = 0
        elif (uplimit == "+inf" or uplimit == "inf"):
            uplimit = 1e99
        elif (uplimit == "-inf"):
            uplimit = -1e99
        ParameterUpLimit.append(uplimit)


        ## read parameter value
        try:
            value = XmlFile.ModelParameters.Parameter.value.getValue()
        except AttributeError, err:
            try:
                value = XmlFile.ModelParameters.Section.SubSection.Parameter.value.getValue()
            except AttributeError, err:
                value = ""
        if (fitflag == 1):
            try:
                value = value.strip()
            except AttributeError, err:
                pass
            if (value == ""):
                value = (float(uplimit) - float(lowlimit)) / 2.0

        # Debug:
        # print "value = ", value

        ParameterValues.append(value)

    else:
        for i in xrange(FitParametersNumber):
            try:
                name = XmlFile.ModelParameters.Parameter[i].name.getValue()
            except AttributeError, err:
                name = XmlFile.ModelParameters.Section.SubSection.Parameter[i].name.getValue()


            ## test if name include one of the following commands (XValue, NumberXValue, CurrentExpFile, CurrentYColumn)
            if (string.find(name,"XValue") != -1):
                print " "
                print "\t Error in subroutine GetParameterSet.XmlParameterFile!"
                print "\t     The selected parameter name: ",name
                print "\t     contains the command XValue!"
                print "\t     Please select a another name for the parameter and restart the whole program."
                print " "
                ok = 1
                return ok, FitParametersNumber, FitParameters, FitParameterName

            elif (string.find(name,"NumberXValue") != -1):
                print " "
                print "\t Error in subroutine GetParameterSet.XmlParameterFile!"
                print "\t     The selected parameter name: ",name
                print "\t     contains the command NumberXValue!"
                print "\t     Please select a another name for the parameter and restart the whole program."
                print " "
                ok = 1
                return ok, FitParametersNumber, FitParameters, FitParameterName

            elif (string.find(name,"CurrentExpFile") != -1):
                print " "
                print "\t Error in subroutine GetParameterSet.XmlParameterFile!"
                print "\t     The selected parameter name: ",name
                print "\t     contains the command CurrentExpFile!"
                print "\t     Please select a another name for the parameter and restart the whole program."
                print " "
                ok = 1
                return ok, FitParametersNumber, FitParameters, FitParameterName

            elif (string.find(name,"CurrentYColumn") != -1):
                print " "
                print "\t Error in subroutine GetParameterSet.XmlParameterFile!"
                print "\t     The selected parameter name: ",name
                print "\t     contains the command CurrentYColumn!"
                print "\t     Please select a another name for the parameter and restart the whole program."
                print " "
                ok = 1
                return ok, FitParametersNumber, FitParameters, FitParameterName


            ## make each parameter name unique
            if (name != ""):
                counter = 1
                for j in FitParameterName:
                    k = j.find("[[")
                    if (k == -1):
                        k = len(j)
                    if (j[:k] == name):
                        counter += 1
                if (string.rfind(name,"[[]]") == (-1)):
                    name = name + "[[" + str(counter) + "]]"            
            # Debug:
            # print name


            ## read parameter name
            FitParameterName.append(name)
            try:
                fitflag = XmlFile.ModelParameters.Parameter[i].getAttribute("fit")
            except AttributeError, err:
                fitflag = XmlFile.ModelParameters.Section.SubSection.Parameter[i].getAttribute("fit")
            ParameterFitFlag.append(fitflag)

            # Debug:
            # print "fitflag = ", fitflag


            ## read lower limit
            try:
                lowlimit = str(XmlFile.ModelParameters.Parameter[i].lowlimit.getValue())
            except AttributeError, err:
                lowlimit = str(XmlFile.ModelParameters.Section.SubSection.Parameter[i].lowlimit.getValue())
            lowlimit = lowlimit.lower()
            lowlimit = lowlimit.strip()
            if (lowlimit == ""):
                lowlimit = 0
            elif (lowlimit == "+inf" or lowlimit == "inf"):
                lowlimit = 1e99
            elif (lowlimit == "-inf"):
                lowlimit = -1e99
            ParameterLowLimit.append(lowlimit)


            ## read upper limit
            try:
                uplimit = str(XmlFile.ModelParameters.Parameter[i].uplimit.getValue())
            except AttributeError, err:
                uplimit = str(XmlFile.ModelParameters.Section.SubSection.Parameter[i].uplimit.getValue())
            uplimit = uplimit.lower()
            uplimit = uplimit.strip()
            if (uplimit == ""):
                uplimit = 0
            elif (uplimit == "+inf" or uplimit == "inf"):
                uplimit = 1e99
            elif (uplimit == "-inf"):
                uplimit = -1e99
            ParameterUpLimit.append(uplimit)


            ## read parameter value
            try:
                value = XmlFile.ModelParameters.Parameter[i].value.getValue()
            except AttributeError, err:
                try:
                    value = XmlFile.ModelParameters.Section.SubSection.Parameter[i].value.getValue()
                except AttributeError, err:
                    value = ""
            if (fitflag == 1):
                try:
                    value = value.strip()
                except AttributeError, err:
                    pass
                if (value == ""):
                    value = (float(uplimit) - float(lowlimit)) / 2.0

            # Debug:
            # print "value = ", value

            ParameterValues.append(value)


    ## construct FitParameters array
    FitParameters.append(ParameterValues)
    FitParameters.append(ParameterFitFlag)
    FitParameters.append(ParameterLowLimit)
    FitParameters.append(ParameterUpLimit)


    ## print: everything is done
    if (print_flag != "false"):
        print "done!"

    ## some debug:
    if (print_flag != "false"):
        print
        print "\t FitParametersNumber = ", FitParametersNumber
        print "\t FitParameters = ", FitParameters
        print "\t FitParameterName = ",FitParameterName
        print


    ## define return parameters
    return ok, FitParametersNumber, FitParameters, FitParameterName
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
def TrueFalse(val):
    ## function return true of false, if val == 0 or val != 0
    """

    function TrueFalse:     return true of false, if val == 0 or val != 0

    input parameters:       val:                        input variable indicating error flag

    output paramters:       str:                        string = "true" or "false"

    """

    if (val == 0):
        str = "false"
    else:
        str = "true"
    return str
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
def WriteParameter(print_flag, XmlParameterFile, FitParametersNumber, FitParameters, FitParameterName):
    ## writes xml-file containing the start values for the fit process
    """

    function WriteParameter: 
                            Write the xml-file containing the start values of model parameters for the fit process

    input parameters:       print_flag:                 print flag
                            XmlParameterFile:           path and file name of the xml file containing the start values
                            FitParameters:              values, fit flags, lower and upper limit of each model parameter
                            FitParameterName:           names of the model parameters

    output paramters:       ok:                         status of reading

    working parameters:     i:                          loop counter
                            printdoneflag:              flag for printing the word "done"
                            XmlFile:                    contents of the xml-file
                            ParamName:                  name of ith parameter


    """

    ## printdoneflag ?
    if (print_flag == "false"):
        printdoneflag = "false"
    else:
        printdoneflag = "true"


    ## set status parameter, reset FitParameter array and reset FitParameterName array
    ok = 0


    ## read xml-description of the INPUT file
    ## is the defined file a xml-file ?
    if not XmlParameterFile.endswith(".xml"):
        print " "
        print "\t Error in subroutine GetParameterSet.XmlParameterFile!"
        print "\t     The selected file is not a xml-file!"
        print "\t     Please select a .xml and restart the whole program."
        print " "
        ok = 1
        return ok


    ## print what you do ..
    if (print_flag != "false"):
        print "Write optimized model parameter to file."
        print "\t Open xml-file: " + XmlParameterFile
        print "\t Writing file ..",


    ## open xml-file file and save content of file to variable XmlFile
    XmlFile = MAGIXxmlPACKAGE.XmlObject(fileName = XmlParameterFile)
    XmlFileOld = XmlFile


    ## set values of the optimized parameters
    if (FitParametersNumber == 1):


        ## remove squared brackets at the end of the parameter name
        ParamName = FitParameterName[1]
        j = ParamName.find("[[]]")
        if (j == (-1)):
            j = ParamName.find("[[")
            if (j != (-1)):
                ParamName = FitParameterName[1][:j]


        ## write name of parameter to xml-file
        try:
            XmlFile.ModelParameters.Parameter.name.setValue(ParamName)
        except AttributeError, err:
            XmlFile.ModelParameters.Section.SubSection.Parameter.name.setValue(ParamName)


        ## write attribute
        try:
            XmlFile.ModelParameters.Parameter.setAttribute("fit",TrueFalse(FitParameters[1][1]))
        except AttributeError, err:
            XmlFile.ModelParameters.Section.SubSection.Parameter.setAttribute("fit",TrueFalse(FitParameters[1][1]))


        ## write parmeter value
        try:
            XmlFile.ModelParameters.Parameter.value.setValue(FitParameters[0][1])
        except AttributeError, err:
            XmlFile.ModelParameters.Section.SubSection.Parameter.value.setValue(FitParameters[0][1])


        ## write error value
        if (len(FitParameters) == 6):
            if (type(FitParameters[5][1]).__name__=='list'):                                ## is entry list or number
                param_num = 'parameter-number = 1;'
                err_left_str = '  error_left = %10.4f;' % float(FitParameters[5][1][0])
                err_right_str = '  error_right = %10.4f;' % float(FitParameters[5][1][1])
                mean_val_str = '  mean_value = %10.4f;' % float(FitParameters[5][1][2])
                err_mean_str = '  standard_dev = %10.4f;' % float(FitParameters[5][1][3])
                all_logZ_str = '  log(evidence) = %10.4f' % float(FitParameters[5][1][4])
                tag_content = param_num + err_left_str + err_right_str + mean_val_str + err_mean_str + all_logZ_str
                try:
                    XmlFile.ModelParameters.Parameter.error.setValue(tag_content)
                except AttributeError, err:
                    XmlFile.ModelParameters.Section.SubSection.Parameter.error.setValue(tag_content)
            else:
                try:
                    XmlFile.ModelParameters.Parameter.error.setValue(str(FitParameters[5][1]))
                except AttributeError, err:
                    XmlFile.ModelParameters.Section.SubSection.Parameter.error.setValue(str(FitParameters[5][1]))
        else:
            try:
                XmlFile.ModelParameters.Parameter.error.setValue(FitParameters[4][1])
            except AttributeError, err:
                XmlFile.ModelParameters.Section.SubSection.Parameter.error.setValue(FitParameters[4][1])


        ## write low limit value
        try:
            XmlFile.ModelParameters.Parameter.lowlimit.setValue(FitParameters[2][1])
        except AttributeError, err:
            XmlFile.ModelParameters.Section.SubSection.Parameter.lowlimit.setValue(FitParameters[2][1])


        ## write upper limit value
        try:
            XmlFile.ModelParameters.Parameter.uplimit.setValue(FitParameters[3][1])
        except AttributeError, err:
            XmlFile.ModelParameters.Section.SubSection.Parameter.uplimit.setValue(FitParameters[3][1])
    else:
        countFreeParam = 0
        for i in xrange(FitParametersNumber):


            ## remove squared brackets at the end of the parameter name
            ParamName = FitParameterName[i]
            j = ParamName.find("[[]]")
            if (j == (-1)):
                j = ParamName.find("[[")
                if (j != (-1)):
                    ParamName = FitParameterName[i][:j]


            ## write name of parameter to xml-file
            try:
                XmlFile.ModelParameters.Parameter[i].name.setValue(ParamName)
            except AttributeError, err:
                XmlFile.ModelParameters.Section.SubSection.Parameter[i].name.setValue(ParamName)


            ## write attribute
            try:
                XmlFile.ModelParameters.Parameter[i].setAttribute("fit",TrueFalse(FitParameters[1][i]))
            except AttributeError, err:
                XmlFile.ModelParameters.Section.SubSection.Parameter[i].setAttribute("fit",TrueFalse(FitParameters[1][i]))


            ## write parmeter value
            try:
                XmlFile.ModelParameters.Parameter[i].value.setValue(FitParameters[0][i])
            except AttributeError, err:
                XmlFile.ModelParameters.Section.SubSection.Parameter[i].value.setValue(FitParameters[0][i])


            ## write error value
            if (len(FitParameters) == 6):
                if (type(FitParameters[5][i]).__name__=='list'):                            ## is entry list or number
                    countFreeParam += 1
                    param_num = 'parameter-number = %5i;' % int(countFreeParam)
                    err_left_str = '  error_left = %10.4f;' % float(FitParameters[5][i][0])
                    err_right_str = '  error_right = %10.4f;' % float(FitParameters[5][i][1])
                    tag_content = param_num + err_left_str + err_right_str
                    if (len(FitParameters[5][i]) > 2):
                        mean_val_str = '  mean_value = %10.4f;' % float(FitParameters[5][i][2])
                        err_mean_str = '  standard_dev = %10.4f;' % float(FitParameters[5][i][3])
                        all_logZ_str = '  log(evidence) = %10.4f' % float(FitParameters[5][i][4])
                        tag_content += mean_val_str + err_mean_str + all_logZ_str
                    try:
                        XmlFile.ModelParameters.Parameter[i].error.setValue(tag_content)
                    except AttributeError, err:
                        XmlFile.ModelParameters.Section.SubSection.Parameter[i].error.setValue(tag_content)
                else:
                    try:
                        XmlFile.ModelParameters.Parameter[i].error.setValue(str(FitParameters[5][i]))
                    except AttributeError, err:
                        XmlFile.ModelParameters.Section.SubSection.Parameter[i].error.setValue(str(FitParameters[5][i]))
            else:
                try:
                    XmlFile.ModelParameters.Parameter[i].error.setValue(FitParameters[4][i])
                except AttributeError, err:
                    XmlFile.ModelParameters.Section.SubSection.Parameter[i].error.setValue(FitParameters[4][i])


            ## write low limit value
            try:
                XmlFile.ModelParameters.Parameter[i].lowlimit.setValue(FitParameters[2][i])
            except AttributeError, err:
                XmlFile.ModelParameters.Section.SubSection.Parameter[i].lowlimit.setValue(FitParameters[2][i])


            ## write upper limit value
            try:
                XmlFile.ModelParameters.Parameter[i].uplimit.setValue(FitParameters[3][i])
            except AttributeError, err:
                XmlFile.ModelParameters.Section.SubSection.Parameter[i].uplimit.setValue(FitParameters[3][i])


    ## save new contents to file
    NewXmlFile = open(XmlParameterFile, 'w+')
    XmlFile.writexml(NewXmlFile, '')
    NewXmlFile.close()


    ## print that you have finished
    if (print_flag != "false"):
        print "done !"

    return ok
##--------------------------------------------------------------------------------------------------------------------------------------------------------


