#!/usr/bin/env python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module reads the parameters controlling the fit process from the fit control file by using the function "read_control_file".
##  Copyright (C) 2009 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following subroutines and functions are included in this module:
##
##      - subroutine ReadFitControlFile:                read fit-control file
##      - subroutine DetermineParameterValue:           determine value of the parameter
##      - subroutine ReadRegistrationFile:              read registration file
##      - subroutine NumberQ:                           is string a number
##      - subroutine LoadScratch:                       write results of the fit process to files
##      - subroutine StoreResults:                      write results of the fit process to files
##      - subroutine StartAlgPackage:                   start algorithm package
##      - subroutine start:                             call of fit algorithms
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    06.22.2009   initial version
##  T. Moeller    16.01.2012   improve documentation of source code
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************

"""
Optimize fit parameters:
------------------------

The package "FittingEngine" reads the parameters controlling the fit process from the fit control file by using the function "read_control_file".

Example for a fit-control file:

<FitControl>
    <!-- settings for fit process -->


    <!-- set number of used algorithms -->
    <NumberOfFitAlgorithms>1</NumberOfFitAlgorithms>


    <algorithm>
        <FitAlgorithm>Levenberg-Marquardt</FitAlgorithm>


        <!-- define method used for Levenberg-Marquardt ("nr" = "Numerical Recepies version" (default) or "minpack" = "MINPACK version") -->
        <MethodLM>nr</MethodLM>


        <!-- define value of the variation (used for the calculation of the gradient (first derivative of chi^2 function)) in percent/100 -->
        <VariationValue>1.e-3</VariationValue>


        <!-- set max. number of iterations -->
        <number_iterations>20</number_iterations>


        <!-- set max. number of processors -->
        <NumberProcessors>8</NumberProcessors>


        <!-- settings for chi^2 -->
        <limit_of_chi2>1e-7</limit_of_chi2>
        <RenormalizedChi2>yes</RenormalizedChi2>
        <DeterminationChi2>default</DeterminationChi2>
        <SaveChi2>yes</SaveChi2>


        <!-- set plot options -->
        <PlotAxisX>Frequency_[Hz]</PlotAxisX>
        <PlotAxisY>Intensity</PlotAxisY>
        <PlotIteration>no</PlotIteration>
    </algorithm>
</FitControl>


Example for a fit-contral file with algorithm-chain:


<FitControl>
    <!-- settings for fit process -->


    <!-- set number of used algorithms -->
    <NumberOfFitAlgorithms>3</NumberOfFitAlgorithms>


    <algorithm>
        <FitAlgorithm>Bees</FitAlgorithm>


        <!-- special settings for bees algorithm -->
        <BestSiteCounter>2</BestSiteCounter>


        <!-- set max. number of iterations -->
        <number_iterations>20</number_iterations>


        <!-- set max. number of processors -->
        <NumberProcessors>8</NumberProcessors>


        <!-- settings for chi^2 -->
        <limit_of_chi2>1e-5</limit_of_chi2>
        <RenormalizedChi2>yes</RenormalizedChi2>
        <DeterminationChi2>default</DeterminationChi2>
        <SaveChi2>yes</SaveChi2>


        <!-- set plot options -->
        <PlotAxisX>Frequency [Hz]</PlotAxisX>
        <PlotAxisY>Intensity</PlotAxisY>
        <PlotIteration>no</PlotIteration>
    </algorithm>

    <algorithm>
        <FitAlgorithm>Levenberg-Marquardt</FitAlgorithm>


        <!-- set max. number of iterations -->
        <number_iterations>50</number_iterations>


        <!-- set max. number of processors -->
        <NumberProcessors>8</NumberProcessors>


        <!-- settings for chi^2 -->
        <limit_of_chi2>1e-6</limit_of_chi2>
        <RenormalizedChi2>yes</RenormalizedChi2>
        <DeterminationChi2>default</DeterminationChi2>
        <SaveChi2>yes</SaveChi2>


        <!-- set plot options -->
        <PlotAxisX>Frequency [Hz]</PlotAxisX>
        <PlotAxisY>Intensity</PlotAxisY>
        <PlotIteration>no</PlotIteration>
    </algorithm>

    <algorithm>
        <FitAlgorithm>errorestim_ins</FitAlgorithm>


        <!-- set max. number of iterations -->
        <number_iterations>10</number_iterations>


        <!-- set max. number of processors -->
        <NumberProcessors>8</NumberProcessors>


        <!-- settings for chi^2 -->
        <limit_of_chi2>1e-5</limit_of_chi2>
        <RenormalizedChi2>yes</RenormalizedChi2>
        <DeterminationChi2>default</DeterminationChi2>
        <SaveChi2>yes</SaveChi2>


        <!-- set plot options -->
        <PlotAxisX>Frequency [Hz]</PlotAxisX>
        <PlotAxisY>Intensity</PlotAxisY>
        <PlotIteration>no</PlotIteration>
    </algorithm>    
</FitControl>



Remarks:


    * The tag <NumberOfFitAlgorithms> defines the number of algorithms which should be used within the fit process. A number greater than 1 defines a
      so-called algorithm-chain (see example).

    * The settings for each algorithm are enclosed by the <algorithm> tag. Note, that these tags have to occur as many times as given by the
      tag <NumberOfFitAlgorithms>.

    * Each algorithm is described by the tags <FitAlgorithm>, <limit_of_chi2>, <number_iterations>, <NumberProcessors>, <DeterminationChi2>,
      <SaveChi2>, <PlotAxisX>, <PlotAxisY> and <PlotIteration>. Depending on the chosen algorithm, a couple of additional tags have to be added.

    * The tag <FitAlgorithm> defines the algorithm, which should be used in the fit process. Note, that the contents of the tag <FitAlgorithm> has to
      be identical with one of the following names of algorithm (It does not matter, if these words are written in lower case or upper case letters):

      "nested-sampling", "levenberg-marquardt", "simulated-annealing", "pso", "bees", "genetic", "interval-ns", "ins", "mcmc", "additionalpackages", 
      "errorestim", "errorestim_ins"

    * The tag <NumberProcessors> defines the number of processors which are used by MAGIX. NOTE, that a value greater than 1 is used only for
      external program which allow parallized working (see tag <ParallelizationPossible> within the registration xml-file).

    * Changing the content of the tag <DeterminationChi2> the user can specify the method which is used for the determination of chi^2. At the
      moment the following options are included in MAGIX:

      - "default" or "difference":

        \chi^2 = \sum_{i=1}^N (y_i^{observe) - y_i^{fit})^2,

        where y_i^{observe) represents the value of the experimental data at the point i and y_i^{fit} the corresponding value of the fit function.

      - ..

      If the tag <DeterminationChi2> is set to "default" or "difference", the content of the tag <SaveChi2> is read, which specifies that the
      difference (y_i^{observe) - y_i^{fit}) is saved to file for all experimental points.


    IMPROTANT: The following options are available only for 2D functions y = f(x) !!!

    * The tags <PlotAxisX> and <PlotAxisY> define the labels for the x- and the y-axis, respectively.
    * The observed data and the fit function are plotted for each iteration step, if the tag <PlotIteration> is set to "yes".



Registration of model:
----------------------

In order to communicate with the external fit function module MAGIX have to write the input file(s) including the current values of the model parameters.
Additionally, the new value of the fit function have to be read from the output file. Thus, xml-descriptions of the input and the output file are
required. In the following some examples and important remarks are given for the xml-description of the input and the output file:


Example (input/output description):
-----------------------------------

<?xml version="1.0" encoding="ISO-8859-1"?>
<InOutputFile>
    <ModelProgramCall>
        <PathStartScript>Fit-Functions/Drude-Lorentz_conv/bin/DrudeLorentzConv.exe</PathStartScript>
        <ExeCommandStartScript>./DrudeLorentzConv.exe</ExeCommandStartScript>
        <ParallelizationPossible>Yes</ParallelizationPossible>
        <InputDataPath>DataIn.dat</InputDataPath>
    </ModelProgramCall>


    <!-- xml-description of the input file for the conventional Drude-Lorentz model -->


    <NumberInputFiles>1</NumberInputFiles>
    <InputFile>
        <title>description of input-file for conventional Drude-Lorentz model</title>
        <InputFileName>in.txt</InputFileName>
        <NumberLines>4</NumberLines>
        <line group="false">
            <NumberParameterLine>1</NumberParameterLine>
            <Parameter group="false">
                <NumberReplicationParameter> </NumberReplicationParameter>
                <Name>NumberXValue1</Name>
                <Format>I5</Format>
                <LeadingString></LeadingString>
                <TrailingString></TrailingString>
            </Parameter>
        </line>
        <line group="false">
            <NumberParameterLine>1</NumberParameterLine>
            <Parameter group="false">
                <Name>EpsilonInfinity</Name>
                <Format>F15.8</Format>
                <NumberReplicationParameter> </NumberReplicationParameter>
                <LeadingString></LeadingString>
                <TrailingString></TrailingString>
            </Parameter>
        </line>
        <line group="false">
            <NumberParameterLine>1</NumberParameterLine>
            <Parameter group="false">
                <NumberReplicationParameter> </NumberReplicationParameter>
                <Name>NumberOscillators</Name>
                <Format>I6</Format>
                <LeadingString></LeadingString>
                <TrailingString></TrailingString>
            </Parameter>
        </line>
        <line group="group1: start, replication = NumberOscillators">
            <NumberParameterLine>3</NumberParameterLine>
            <Parameter group="false">
                <NumberReplicationParameter> </NumberReplicationParameter>
                <Name>EigenFrequency</Name>
                <Format>F20.5</Format>
                <LeadingString></LeadingString>
                <TrailingString></TrailingString>
            </Parameter>
            <Parameter group="false">
                <NumberReplicationParameter> </NumberReplicationParameter>
                <Name>PlasmaFrequency</Name>
                <Format>F20.8</Format>
                <LeadingString></LeadingString>
                <TrailingString></TrailingString>
            </Parameter>
            <Parameter group="false">
                <NumberReplicationParameter> </NumberReplicationParameter>
                <Name>Damping</Name>
                <Format>F20.8</Format>
                <LeadingString></LeadingString>
                <TrailingString></TrailingString>
            </Parameter>
        </line>
    </InputFile>


    <!-- xml-description of the output file for the conventional Drude-Lorentz model -->


    <AllInOneOutputFile>yes</AllInOneOutputFile>
    <NumberOutputFiles>1</NumberOutputFiles>
    <OutputFile>
        <OnlyYColumn>yes</OnlyYColumn>
        <OutputFileName>FitFunctionValues.dat</OutputFileName>
        <OutputFileFormat>ascii</OutputFileFormat>
        <NumberHeaderLines></NumberHeaderLines>
        <CharacterForComments></CharacterForComments>
    </OutputFile>

</InOutputFile>


IMPORTANT:
----------

- The tag <NumberInputFiles>1</NumberInputFiles> gives the number of input-files
  which are described.


- The tags <PathStartScript> and <ExeCommandStartScript> include the path and the
  command to call the start script for the external model function from the command
  prompt.


- The tag <InputFileName></InputFileName> has to include the path and the file
  name of the input file of the fit function module.


- If the fit function program requires the points of the experimental data where
  the fit function should be calculated then define the file name of a file
  containing the points of the experimental data by the tag <InputDataPath></InputDataPath>.


- The tag <NumberLines></NumberLines> defines the total number of lines
  of the input file (without replications)!


- The description of each line is bracket between the <line></line> tags.
  These tags have to occur as many times as defined in the tag
  <NumberLines></NumberLines>. The attribute 'group' indicates, if this
  line is grouped with other lines forming a block.

  For example:  If you want to replicate the following two lines for three
                times the xml-description would look like:

                <line group="group1: start, replication = 3">
                    <NumberParameterLine>1</NumberParameterLine>
                    <Parameter group="false">
                        <NumberReplicationParameter> </NumberReplicationParameter>
                        <Name>ParameterA</Name>
                        <Format>F7.3</Format>
                        <LeadingString> </LeadingString>
                        <TrailingString> </TrailingString>
                    </Parameter>
                </line>
                    <line group="group2: start, replication = 1">
                    <NumberParameterLine>1</NumberParameterLine>
                    <Parameter group="false">
                        <NumberReplicationParameter> </NumberReplicationParameter>
                        <Name>ParameterB</Name>
                        <Format>F2.1</Format>
                        <LeadingString> </LeadingString>
                        <TrailingString></TrailingString>
                    </Parameter>
                </line>

                This description describes an input file of the form:

                ParameterA
                ParameterB
                ParameterA
                ParameterB
                ParameterA
                ParameterB


- The tag <NumberParameterLine></NumberParameterLine> defines the number of
  model parameters without replications defined within one <line></line> block
  and is given only once within one <line></line> block.


- The settings for each parameter are given within the <Parameter> tags.


- The tag <NumberReplicationParameter></NumberReplicationParameter> defines
  the number of replications of the parameter in the current line. The default
  value is 1. Here, the number can also be defined by a parameter by
  inserting the name of this parameter into the tag. (For example:
  <NumberReplicationParameter>ParameterA</NumberReplicationParameter>, where
  the parameter "ParameterA" defines the number of replications.) The tags
  <Parameter></Parameter> contain attributes 'group' like the <line></line>
  tags mentioned above, which indicate grouped parameters. Grouped parameters
  are replicated as many times as given by the tag
  <NumberReplicationParameter></NumberReplicationParameter> of the first
  grouped parameter. Each value of the tag
  <NumberReplicationParameter></NumberReplicationParameter> of a parameter
  with attribute 'group="true"' which is unequal to " " defines the beginning
  of a new block.

  For example:  If you want to replicate the variables 'ParamterA' and
                'ParameterB' for two times the xml-description would looks
                like:

                ...
                <line group="false">
                        <NumberParameterLine>2</NumberParameterLine>
                    <Parameter group="true">
                        <NumberReplicationParameter>2</NumberReplicationParameter>
                        <Name>ParameterA</Name>
                        <Format>F2.1</Format>
                        <LeadingString> </LeadingString>
                        <TrailingString></TrailingString>
                    </Parameter>
                    <Parameter group="true">
                        <NumberReplicationParameter> </NumberReplicationParameter>
                        <Name>ParameterB</Name>
                        <Format>F2.1</Format>
                        <LeadingString> </LeadingString>
                        <TrailingString></TrailingString>
                    </Parameter>
                </line>
                ...

                This xml-description produces the following line of an input file:

                ParameterA ParameterB ParameterA ParameterB

                If you change the value of the tag
                <NumberReplicationParameter> </NumberReplicationParameter>
                of the second parameter to "2" as well

                ...
                    </Parameter>
                    <Parameter group="true">
                        <NumberReplicationParameter>2</NumberReplicationParameter>
                ...

                the line of the input file would have the following from:

                ParameterA ParameterA ParameterB ParameterB


               <NumberReplicationParameter>NumParamColumn</NumberReplicationParameter>
               <NumberParameterLine>1</NumberParameterLine>
               <LeadingString> </LeadingString>
               <ParameterName>TestParameter</ParameterName>
               <ParameterFormat>I5</ParameterFormat>
               <TrailingString> </TrailingString>


- The name of the model parameters of the current line are defined in the tags
  <Name></Name>. The names of the model parameters used in the xml-description
  of the input file MUST be IDENTICAL with the names of the parameters used
  within the parameter file (defined in the i/o control file by the command
  "parameter_file").

  NOTE, if a name of a model parameter occurs more than once in the
  xml-desciption of the input file, the values of these parameters are set
  following the occurance in the the parameter file.

  For example:  The parameter with name "Eigenfrequency" occurs three times
                within the parameter file with values "200", "400" and "700".
                Thus, the input file will include these values in the
                following order:

                ...
                200
                ...
                400
                ...
                700
                ..

  NOTE, that the number of incidences of a parameter within the parameter
  file MUST be IDENTICAL to the total number of occurrences in the input file,
  including all replications!

  For example:  The parameter "param" containing the values "1", "2", "3",
                "4", "5", and "6" may appear in the input file like the
                following:

                param param param
                param param param

                The values of the parameter are set in the following order:

                1 2 3
                4 5 6


- The tags <Format></Format> indicate the format of the parameter value (like
  FORTRAN notation):

  For example: I5 for integers with max. 5 digits, A4 for strings with
               max. 4 characters, F15.8 for real numbers with 8 decimal place
               and max. 15-8-1 integer places.

  NOTE, define enought integer places! During the fitting algorithm, the value
  of the parameter may increase by many magnitudes. If you do not specify
  enough integer places, the value of the parameter is replaced by "********"
  and MAGIX will abort with an error message!


- The tags <LeadingString></LeadingString>, and
  <TrailingString></TrailingString> define the leading and the trailing
  of the parameter string, respectively.

  For example: If model parameter are set by command words in the input
               file of the external fit function program, these command
               words can be insert here in these tags. For example an
               external program set the current frequency in the input
               file by the command word "FREQUENCY =" this command word
               has to be insert in the <LeadingString></LeadingString>
               tag: "<LeadingString>FREQUENCY =</LeadingString>.

  Note, these tags have to occur as many times as given in the tag
  <NumberParameterLine></NumberParameterLine> WITHOUT counting the
  replications!


- If the input file contains a line without a model parameter, the tags
  <NumberParameterLine></NumberParameterLine> has to be set to 0. Then only
  the value of the tag <LeadingString></LeadingString> is read. The other
  tags <Name></Name>,
  <NumberReplicationParameter></NumberReplicationParameter>, and
  <Format></Format> are ignored.

  The command word "CurrentExpFile" print out the index of the current observation
  file. Furthermore, the command word "CurrentYColumn" gives the index of the
  current y-column.


The description of the output file is less complex than the description of the input
file:


- MAGIX can handle more than one output file. The number of output-files produced
  by the external model program is given by the tag
  <NumberOutputFiles></NumberOutputFiles>, where the content has to be always an
  interger > 0. The following settings for each output file has to be embedded by
  the tag <OutputFile></OutputFile>, whereas the tag <OutputFile></OutputFile> has
  to occur as many times as given by the tag <NumberOutputFiles></NumberOutputFiles>.


- The name for each output file has to be given by the tag
  <OutputFileName></OutputFileName>.


- The tag <OutputFileFormat></OutputFileFormat> defines the format of the current
  output file. The user can choose one  of the following commands: "ascii", and
  "dat" (for an ascii-file) and "fits" for a fits-file. Additionally, the command
  "auto" and "automatic" selects the format of the current output file by the ending
  of the file name, i.e. if the file name ends with ".fits", MAGIX expects a
  fits-file otherwise an ascii-file.

  Note, this tag is read only, if the calculation method is set to "AtOnce".


- The tag <OnlyYColumn></OnlyYColumn> defines if the output file(s) include(s)
  only the values of the model function (y-column) without the corresponding
  x-column vaules. If the tag <OnlyYColumn> includes a content unequal to "yes",
  MAGIX reads in the x-column as well as the y-column of the model function and
  checks if the model function values correspond to the given x-column values
  defined in the observation file(s).


- The tag <NumberHeaderLines></NumberHeaderLines> defines the number of header
  lines, which are ignored during the import of the model output file(s). If
  the tag is empty, MAGIX expects no header lines.


- The tag <CharacterForComments></CharacterForComments> defines a character,
  which indicates a comment in the output file(s).



Examples for input files:
-------------------------

  1.)  Simple model input, no replications of parameters or lines:

       100.123
       2.2          // This is a remark
       45
       300.0

       The corresponding xml-description looks like:
       <?xml version="1.0" encoding="ISO-8859-1"?>
       <InOutputFile>
           <ModelProgramCall>
                <PathStartScript>Fit-Functions/Drude-Lorentz_conv/bin/</PathStartScript>
                <ExeCommandStartScript>DrudeLorentzConv.exe</ExeCommandStartScript>
                <ParallelizationPossible>Yes</ParallelizationPossible>
           </ModelProgramCall>
           <NumberInputFiles>1</NumberInputFiles>
           <InputFile>
               <title>description of the input file for conventional Drude-Lorentz model</title>
               <InputFileName>in.txt</InputFileName>
               <NumberLines>4</NumberLines>
                <line group="false">
                    <NumberParameterLine>1</NumberParameterLine>
                    <Parameter group="false">
                        <NumberReplicationParameter></NumberReplicationParameter>
                        <Name>XValue1</Name>
                        <Format>F7.3</Format>
                        <LeadingString></LeadingString>
                        <TrailingString></TrailingString>
                    </Parameter>
                </line>
                <line group="false">
                    <NumberParameterLine>1</NumberParameterLine>
                    <Parameter group="false">
                        <NumberReplicationParameter></NumberReplicationParameter>
                        <Name>ParameterB</Name>
                        <Format>F4.1</Format>
                        <LeadingString></LeadingString>
                        <TrailingString>          // This is a remark</TrailingString>
                    </Parameter>
                </line>
                <line group="false">
                            <NumberParameterLine>1</NumberParameterLine>
                    <Parameter group="false">
                        <NumberReplicationParameter></NumberReplicationParameter>
                        <Name>ParameterC</Name>
                        <Format>I2</Format>
                        <LeadingString></LeadingString>
                        <TrailingString></TrailingString>
                    </Parameter>
                </line>
                <line group="group1: start, replication = NumberOscillators">
                    <NumberParameterLine>3</NumberParameterLine>
                    <Parameter group="false">
                        <NumberReplicationParameter></NumberReplicationParameter>
                        <Name>ParameterD</Name>
                        <Format>F5.1</Format>
                        <LeadingString></LeadingString>
                        <TrailingString></TrailingString>
                    </Parameter>
                </line>
            </InputFile>


           <title>xml-description of the output file of a test model</title>

           <NumberOutputFiles>1</NumberOutputFiles>
           <OutputFile>
               <OnlyYColumn>yes</OnlyYColumn>
               <OutputFileName>FitFunctionValues.dat</OutputFileName>
               <OutputFileFormat>ascii</OutputFileFormat>
               <NumberHeaderLines></NumberHeaderLines>
               <CharacterForComments></CharacterForComments>
           </OutputFile>

        </InOutputFile>



  2.)  Conventional Drude-Lorentz model: (A model where one line have to be
       repeated as many times as given by the parameter "NumberOscillators"):

       100.123
       3.5
       3
       100.0    456.234    12.120
       250.0    106.641     2.127
       403.0    251.577    30.022

       The corresponding xml-description is:

       <?xml version="1.0" encoding="ISO-8859-1"?>
       <InOutputFile>
           <ModelProgramCall>
                <PathStartScript>Fit-Functions/Drude-Lorentz_conv/bin/</PathStartScript>
                <ExeCommandStartScript>DrudeLorentzConv.exe</ExeCommandStartScript>
                <ParallelizationPossible>Yes</ParallelizationPossible>
                <InputDataPath>DataIn.dat</InputDataPath>
           </ModelProgramCall>
           <NumberInputFiles>1</NumberInputFiles>
           <InputFile>
                <title>input description for conventional Drude-Lorentz model</title>
                <CalculationMethod>PointToPoint</CalculationMethod>
                <NumberLines>4</NumberLines>
                <line group="false">
                    <NumberParameterLine>1</NumberParameterLine>
                    <Parameter group="false">
                        <NumberReplicationParameter></NumberReplicationParameter>
                        <Name>XValue1</Name>
                        <Format>F15.8</Format>
                        <LeadingString></LeadingString>
                        <TrailingString></TrailingString>
                    </Parameter>
                </line>
                <line group="false">
                    <NumberParameterLine>1</NumberParameterLine>
                    <Parameter group="false">
                        <NumberReplicationParameter></NumberReplicationParameter>
                        <Name>EpsilonInfinity</Name>
                        <Format>F15.8</Format>
                        <LeadingString></LeadingString>
                        <TrailingString></TrailingString>
                    </Parameter>
                </line>
                <line group="false">
                   <NumberParameterLine>1</NumberParameterLine>
                   <Parameter group="false">
                       <NumberReplicationParameter></NumberReplicationParameter>
                       <Name>NumberOscillators</Name>
                       <Format>I6</Format>
                       <LeadingString></LeadingString>
                       <TrailingString></TrailingString>
                   </Parameter>
               </line>
               <line group="group1: start, replication = NumberOscillators">
                   <NumberParameterLine>3</NumberParameterLine>
                   <Parameter group="false">
                       <NumberReplicationParameter></NumberReplicationParameter>
                       <Name>EigenFrequency</Name>
                       <Format>F8.1</Format>
                       <LeadingString></LeadingString>
                       <TrailingString></TrailingString>
                   </Parameter>
                   <Parameter group="false">
                       <NumberReplicationParameter></NumberReplicationParameter>
                       <Name>PlasmaFrequency</Name>
                       <Format>F11.3</Format>
                       <LeadingString></LeadingString>
                       <TrailingString></TrailingString>
                   </Parameter>
                   <Parameter group="false">
                       <NumberReplicationParameter></NumberReplicationParameter>
                       <Name>Damping</Name>
                       <Format>F11.3</Format>
                       <LeadingString></LeadingString>
                       <TrailingString></TrailingString>
                   </Parameter>
               </line>
           </InputFile>

           <title>xml-description of the output file for the conventional Drude-Lorentz model</title>

           <NumberOutputFiles>1</NumberOutputFiles>
           <OutputFile>
               <OnlyYColumn>yes</OnlyYColumn>
               <OutputFileName>FitFunctionValues.dat</OutputFileName>
               <OutputFileFormat>ascii</OutputFileFormat>
               <NumberHeaderLines></NumberHeaderLines>
               <CharacterForComments></CharacterForComments>
           </OutputFile>

       </InOutputFile>


Documentation of the fit algorithms:
------------------------------------

Detailed descriptions of the several fit algorithms are located in the MAGIX manual
"""


##********************************************************************* load packages ********************************************************************
import numpy                                                                                ## import numpy package
import copy                                                                                 ## import copy package
import os.path                                                                              ## load package for file management
import string                                                                               ## load string package
import shutil                                                                               ## load module for communicating with the system
import sys                                                                                  ## load module containing interpeter specific exports
import multiprocessing                                                                      ## load module multiprocessing
import MAGIXxmlPACKAGE                                                                      ## load MAGIXxmlPACKAGE package
import AdditionalPackages                                                                   ## load additional package (with Scipy etc.)
import LoadExpFile                                                                          ## import package for experimental file
import GetParameterSet                                                                      ## import package for reading fit parameters
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
def ReadFitControlFile(printflag, plotflag, fit_control, KnwonAlgorithms, StarterExecutable, MAGIXrootDirectory):
    ## read fit-control file
    """

    function ReadFitControlFile:
                            Loads the fit control file defined in variable "fit_control" defined in the i/o control file.

    input parameters:       fit_control:                path and file name of the fit control file
                            printflag:                  print flag
                            plotflag:                   plot flag
                            KnwonAlgorithms:            list of known algorithms
                            StarterExecutable:          name of the executable
                            MAGIXrootDirectory:         MAGIX root directory

    output paramters:       ok:                         status of reading contron file
                            UseHeuristic:               flag for using heuristic or not
                            NumberOfFitAlgorithms:      number of algorithms (if number is greater than 1 a algorithm chain is defined !)
                            AllAlgorithmSettings:       settings for each algorithm

    working variables:      XmlFile:                    contents of the xml-file
    """


    ## set default values
    FitAlgorithm = "levenberg-marquardt"
    chilm = 0.0                                                                             ## limit_of_chi2
    numrange = 1                                                                            ## number of fit ranges
    numiter = 1                                                                             ## number of iterations
    DeterminationChi2 = 1                                                                   ## method being used for the determination of chi^2
                                                                                            ## the value is negative if chi^2 is saved to file for each
                                                                                            ## data point
    NumberProcessors = 1                                                                    ## number of processors
    ok = 0
    UseHeuristic = "no"                                                                     ## flag for heuristic
    NumberOfFitAlgorithms = 1                                                               ## number of fit algorithms used in the fit process
    AllAlgorithmSettings = []                                                               ## settings for each algorithm



    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read xml-description of the INPUT file
    ## is the defined file a xml-file ?
    if not fit_control.endswith(".xml"):
        print "\n\n\n\n"
        print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
        print "\t\t\t The selected file is not a xml-file!"
        print "\t\t\t Please select a .xml and restart the whole program."
        print "\n\n\n\n"
        ok = 1
        return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings


    ## print what you do
    if (printflag != "false"):
        print "\t Open fit-control file: " + fit_control                                    ## print what you do
        print "\t Reading fit control parameters ..",


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## open xml-file file and save content of file to variable XmlFile
    XmlFile = MAGIXxmlPACKAGE.XmlObject(fileName = fit_control)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read flag for using heuristic
    #    UseHeuristic = str(XmlFile.FitControl.UseHeuristic.getValue())
    #    UseHeuristic = UseHeuristic.strip()
    #    UseHeuristic = UseHeuristic.lower()


    #    ## check user input
    #    if (UseHeuristic != "yes" and UseHeuristic != "no"):
    #        print "\n\n\n\n"
    #        print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
    #        print "\t\t\t The given user input for <UseHeuristic> is neither yes nor no !"
    #        print "\t\t\t Please correct the .xml-file and restart MAGIX."
    #        print "\n\n\n\n"
    #        ok = 1
    #        return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read flag for using heuristic
    NumberOfFitAlgorithms = XmlFile.FitControl.NumberOfFitAlgorithms.getValue()


    ## check user input
    if (NumberQ(str(NumberOfFitAlgorithms)) != "true"):                                     ## is the number of fit algorithms a number ?
        print "\n\n\n\n"
        print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
        print "\t\t\t The selected number of number of fit algorithms is not a number !"
        print "\t\t\t Please correct the .xml-file and restart MAGIX."
        print "\n\n\n\n"
        ok = 1
        return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
    elif (NumberOfFitAlgorithms < 1 or NumberOfFitAlgorithms > 100):                        ## is the number of fit algorithms in a meaningful range?
        print "\n\n\n\n"
        print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile!"
        print "\t\t\t The selected number of number of fit algorithms is not in a meaningful range !"
        print "\t\t\t NumberOfFitAlgorithms < 1 or NumberOfFitAlgorithms > 100 !"
        print "\t\t\t Please correct the .xml-file and restart MAGIX."
        print "\n\n\n\n"
        ok = 1
        return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings



    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## loop over all algorithms
    AllAlgorithmSettings = []
    for algorithm in xrange(NumberOfFitAlgorithms):                                         ## loop over all algorithms
        AlgorithmSettings = []


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read type of algorithm
        if (NumberOfFitAlgorithms == 1):
            FitAlgorithm = str(XmlFile.FitControl.algorithm.FitAlgorithm.getValue())
        else:
            FitAlgorithm = str(XmlFile.FitControl.algorithm[algorithm].FitAlgorithm.getValue())
        FitAlgorithm = FitAlgorithm.strip()                                                 ## delete leading and tailing blanks
        FitAlgorithm = FitAlgorithm.lower()                                                 ## convert to lower case


        ## check user input
        if not FitAlgorithm in KnwonAlgorithms:                                             ## is the chosen algorithm known?
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
            print "\t\t\t The selected algorithm is not known or implemented !"
            print "\t\t\t The following algorithms are known:"
            print "\t\t\t %s " % KnwonAlgorithms
            print "\t\t\t Please correct the .xml-file and restart MAGIX."
            print "\n\n\n\n"
            ok = 1
            return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
        AlgorithmSettings.append(FitAlgorithm)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read special settings for each algorithm
        if (FitAlgorithm == "nested-sampling"):                                             ## read settings for Nested-Sampling algorithm
            NestedSamplingSettings = []


            ## get number of best sites
            if (NumberOfFitAlgorithms == 1):
                try:
                    NestedSamplingBestSiteCounter = XmlFile.FitControl.algorithm.BestSiteCounter.getValue()
                except AttributeError, err:
                    NestedSamplingBestSiteCounter = 1
            else:
                try:
                    NestedSamplingBestSiteCounter = XmlFile.FitControl.algorithm[algorithm].BestSiteCounter.getValue()
                except AttributeError, err:
                    NestedSamplingBestSiteCounter = 1


            ## check user input
            if (NumberQ(str(NestedSamplingBestSiteCounter)) != "true"):                     ## is the number of best sites a number ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of best sites for the Nested-Sampling algorithm is not a number !"
                print "\t\t\t Please correct the .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            elif (NestedSamplingBestSiteCounter < 1):                                       ## is the number of best sites less than 1 ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of best sites for the Nested-Sampling algorithm is less than 1 !"
                print "\t\t\t Please correct .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            NestedSamplingSettings.append(NestedSamplingBestSiteCounter)


            ## get number of particles
            if (NumberOfFitAlgorithms == 1):
                try:
                    NestedSamplingNumberObjects = XmlFile.FitControl.algorithm.NumberObjects.getValue()
                except AttributeError, err:
                    NestedSamplingNumberObjects = 100
            else:
                try:
                    NestedSamplingNumberObjects = XmlFile.FitControl.algorithm[algorithm].NumberObjects.getValue()
                except AttributeError, err:
                    NestedSamplingNumberObjects = 100


            ## check user input
            if (NumberQ(str(NestedSamplingNumberObjects)) != "true"):                       ## is the number of objects a number ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of objects for the Nested-Sampling algorithm is not a number !"
                print "\t\t\t Please correct the .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            elif (NestedSamplingNumberObjects < 1):                                         ## is the number of objects less than 1 ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of objects for the Nested-Sampling algorithm is less than 1 !"
                print "\t\t\t Please correct .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            NestedSamplingSettings.append(NestedSamplingNumberObjects)


            ## set min-max transition constant to zero
            NestedSamplingMinMaxTransition = 0
            NestedSamplingSettings.append(NestedSamplingMinMaxTransition)
            AlgorithmSettings.append(NestedSamplingSettings)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read special settings for each algorithm
        elif (FitAlgorithm == "levenberg-marquardt"):                                       ## read settings for Levenberg-Marquardt algorithm
            LevenbergMarquardtSettings = []


            ## get version of Levenberg-Marquardt algorithm
            if (NumberOfFitAlgorithms == 1):
                try:
                    MethodLM = str(XmlFile.FitControl.algorithm.MethodLM.getValue())
                except AttributeError, err:
                    MethodLM = "NR"
            else:
                try:
                    MethodLM = str(XmlFile.FitControl.algorithm[algorithm].MethodLM.getValue())
                except AttributeError, err:
                    MethodLM = "NR"
            MethodLM = MethodLM.lower()
            MethodLM = MethodLM.strip()
            if (MethodLM == "minpack"):
                MethodLM = 1
            else:
                MethodLM = 2
            LevenbergMarquardtSettings.append(MethodLM)


            ## get value of variation (in percent) for the calculation of the gradient
            if (NumberOfFitAlgorithms == 1):
                try:
                    VariationValue = str(XmlFile.FitControl.algorithm.VariationValue.getValue())
                except AttributeError, err:
                    VariationValue = 1.e-3
            else:
                try:
                    VariationValue = str(XmlFile.FitControl.algorithm[algorithm].VariationValue.getValue())
                except AttributeError, err:
                    VariationValue = 1.e-3
            LevenbergMarquardtSettings.append(float(VariationValue))
            AlgorithmSettings.append(LevenbergMarquardtSettings)



        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read special settings for each algorithm
        elif (FitAlgorithm == "genetic"):                                                   ## read settings for Levenberg-Marquardt algorithm
            GeneticSettings = []


            ## get flag for using new ranges
            if (NumberOfFitAlgorithms == 1):
                try:
                    UseNewRange = str(XmlFile.FitControl.algorithm.UseNewRange.getValue())
                except AttributeError, err:
                    UseNewRange = "yes"
            else:
                try:
                    UseNewRange = str(XmlFile.FitControl.algorithm[algorithm].UseNewRange.getValue())
                except AttributeError, err:
                    UseNewRange = "yes"
            UseNewRange = UseNewRange.lower()
            UseNewRange = UseNewRange.strip()
            if (UseNewRange != "yes"):
                UseNewRange = 0
            else:
                UseNewRange = 1
            GeneticSettings.append(UseNewRange)


            ## get number of chromosomes
            if (NumberOfFitAlgorithms == 1):
                try:
                    NumberChromosomes = str(XmlFile.FitControl.algorithm.NumberChromosomes.getValue())
                except AttributeError, err:
                    NumberChromosomes = "automatic"
            else:
                try:
                    NumberChromosomes = str(XmlFile.FitControl.algorithm[algorithm].NumberChromosomes.getValue())
                except AttributeError, err:
                    NumberChromosomes = "automatic"
            NumberChromosomes = NumberChromosomes.lower()
            NumberChromosomes = NumberChromosomes.strip()
            if (NumberChromosomes == "automatic" or NumberQ(NumberChromosomes) != "true"):
                NumberChromosomes = 0
            if (int(NumberChromosomes) < 0):
                NumberChromosomes = 0
            GeneticSettings.append(NumberChromosomes)


            ## get number of best sites
            if (NumberOfFitAlgorithms == 1):
                try:
                    GeneticBestSiteCounter = XmlFile.FitControl.algorithm.BestSiteCounter.getValue()
                except AttributeError, err:
                    GeneticBestSiteCounter = 1
            else:
                try:
                    GeneticBestSiteCounter = XmlFile.FitControl.algorithm[algorithm].BestSiteCounter.getValue()
                except AttributeError, err:
                    GeneticBestSiteCounter = 1
            if (NumberQ(str(GeneticBestSiteCounter)) != "true"):                            ## is the number of best sites a number ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of best sites for the Genetic algorithm is not a number !"
                print "\t\t\t Please correct the .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            elif (GeneticBestSiteCounter < 1):                                              ## is the number of best sites less than 1 ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of best sites for the Genetic algorithm is less than 1 !"
                print "\t\t\t Please correct .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            GeneticSettings.append(GeneticBestSiteCounter)
            AlgorithmSettings.append(GeneticSettings)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read special settings for each algorithm
        elif (FitAlgorithm == "errorestim" or FitAlgorithm == "errorestim_ins"):            ## read settings for Error Estimator algorithm
            ErrEstimSettings = []


            ## get method for error estimation
            if (NumberOfFitAlgorithms == 1):
                try:
                    ErrorMethod = str(XmlFile.FitControl.algorithm.ErrorMethod.getValue())
                except AttributeError, err:
                    ErrorMethod = "mcmc"
            else:
                try:
                    ErrorMethod = str(XmlFile.FitControl.algorithm[algorithm].ErrorMethod.getValue())
                except AttributeError, err:
                    ErrorMethod = "mcmc"
            ErrorMethod = ErrorMethod.lower()
            ErrorMethod = ErrorMethod.strip()
            if (ErrorMethod == "mcmc"):
                ErrorMethod = 1
            elif (ErrorMethod.find("fisher") > (-1)):
                ErrorMethod = 2
            else:
                ErrorMethod = 3
            ErrEstimSettings.append(ErrorMethod)


            ## get number of samplers
            NumberMCMCSampler = 0
            if (ErrorMethod == 1):                                                          ## only used for MCMC method
                if (NumberOfFitAlgorithms == 1):
                    try:
                        NumberMCMCSampler = str(XmlFile.FitControl.algorithm.NumberMCMCSampler.getValue())
                    except AttributeError, err:
                        NumberMCMCSampler = 0
                else:
                    try:
                        NumberMCMCSampler = str(XmlFile.FitControl.algorithm[algorithm].NumberMCMCSampler.getValue())
                    except AttributeError, err:
                        NumberMCMCSampler = 0
                if (NumberQ(NumberMCMCSampler) != "true"):
                    NumberMCMCSampler = 0
            ErrEstimSettings.append(NumberMCMCSampler)


            ## get number of iterations for burn-in phase
            NumberBurnInIter = 50
            if (ErrorMethod == 1):                                                          ## only used for MCMC method
                if (NumberOfFitAlgorithms == 1):
                    try:
                        NumberBurnInIter = str(XmlFile.FitControl.algorithm.NumberBurnInIter.getValue())
                    except AttributeError, err:
                        NumberBurnInIter = 50
                else:
                    try:
                        NumberBurnInIter = str(XmlFile.FitControl.algorithm[algorithm].NumberBurnInIter.getValue())
                    except AttributeError, err:
                        NumberBurnInIter = 50
                if (NumberQ(NumberBurnInIter) != "true"):
                    NumberBurnInIter = 50
            ErrEstimSettings.append(NumberBurnInIter)


            ## get multiplicity of standard deviation (1sigma, 2sigma, 3sigma, etc.)
            MultiplicitySigma = 2
            if (ErrorMethod == 1):                                                          ## only used for MCMC method
                if (NumberOfFitAlgorithms == 1):
                    try:
                        MultiplicitySigma = str(XmlFile.FitControl.algorithm.MultiplicitySigma.getValue())
                    except AttributeError, err:
                        MultiplicitySigma = 2
                else:
                    try:
                        MultiplicitySigma = str(XmlFile.FitControl.algorithm[algorithm].MultiplicitySigma.getValue())
                    except AttributeError, err:
                        MultiplicitySigma = 2
                if (NumberQ(MultiplicitySigma) != "true"):
                    MultiplicitySigma = 2
            ErrEstimSettings.append(MultiplicitySigma)


            ## use previous calculated parameter vectors
            UsePrevResults = "false"
            if (ErrorMethod == 1):                                                          ## only used for MCMC method
                if (NumberOfFitAlgorithms == 1):
                    try:
                        UsePrevResults = str(XmlFile.FitControl.algorithm.UsePrevResults.getValue())
                    except AttributeError, err:
                        UsePrevResults = "yes"
                else:
                    try:
                        UsePrevResults = str(XmlFile.FitControl.algorithm[algorithm].UsePrevResults.getValue())
                    except AttributeError, err:
                        UsePrevResults = "yes"
            UsePrevResults = UsePrevResults.lower()
            UsePrevResults = UsePrevResults.strip()
            if (UsePrevResults == "yes" or UsePrevResults == "true" or UsePrevResults == "t" or UsePrevResults == "y"):
                UsePrevResults = 0
            else:
                UsePrevResults = 1
            ErrEstimSettings.append(UsePrevResults)


            ## for Fisher-matrix method, get variation
            VariationValue = 1.e-3
            if (ErrorMethod == 2):                                                          ## only used for Fisher-matrix method
                if (NumberOfFitAlgorithms == 1):
                    try:
                        VariationValue = str(XmlFile.FitControl.algorithm.VariationValue.getValue())
                    except AttributeError, err:
                        VariationValue = 1.e-3
                else:
                    try:
                        VariationValue = str(XmlFile.FitControl.algorithm[algorithm].VariationValue.getValue())
                    except AttributeError, err:
                        VariationValue = 1.e-3
            ErrEstimSettings.append(VariationValue)
            AlgorithmSettings.append(ErrEstimSettings)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read special settings for each algorithm
        elif (FitAlgorithm == "mcmc"):                                                      ## read settings for MCMC algorithm
            MCMCSettings = []


            ## get number of samplers
            NumberMCMCSampler = 0
            if (NumberOfFitAlgorithms == 1):
                try:
                    NumberMCMCSampler = str(XmlFile.FitControl.algorithm.NumberMCMCSampler.getValue())
                except AttributeError, err:
                    NumberMCMCSampler = 0
            else:
                try:
                    NumberMCMCSampler = str(XmlFile.FitControl.algorithm[algorithm].NumberMCMCSampler.getValue())
                except AttributeError, err:
                    NumberMCMCSampler = 0
            if (NumberQ(NumberMCMCSampler) != "true"):
                NumberMCMCSampler = 0
            MCMCSettings.append(NumberMCMCSampler)


            ## get number of iterations for burn-in phase
            NumberBurnInIter = 50
            if (NumberOfFitAlgorithms == 1):
                try:
                    NumberBurnInIter = str(XmlFile.FitControl.algorithm.NumberBurnInIter.getValue())
                except AttributeError, err:
                    NumberBurnInIter = 50
            else:
                try:
                    NumberBurnInIter = str(XmlFile.FitControl.algorithm[algorithm].NumberBurnInIter.getValue())
                except AttributeError, err:
                    NumberBurnInIter = 50
            if (NumberQ(NumberBurnInIter) != "true"):
                NumberBurnInIter = 50
            MCMCSettings.append(NumberBurnInIter)



            ## get number of best sites
            if (NumberOfFitAlgorithms == 1):
                try:
                    MCMCBestSiteCounter = XmlFile.FitControl.algorithm.BestSiteCounter.getValue()
                except AttributeError, err:
                    MCMCBestSiteCounter = 1
            else:
                try:
                    MCMCBestSiteCounter = XmlFile.FitControl.algorithm[algorithm].BestSiteCounter.getValue()
                except AttributeError, err:
                    MCMCBestSiteCounter = 1
            if (NumberQ(str(MCMCBestSiteCounter)) != "true"):                               ## is the number of best sites a number ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of best sites for the MCMC algorithm is not a number !"
                print "\t\t\t Please correct the .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            elif (MCMCBestSiteCounter < 1):                                                 ## is the number of best sites less than 1 ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of best sites for the MCMC algorithm is less than 1 !"
                print "\t\t\t Please correct .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            MCMCSettings.append(MCMCBestSiteCounter)
            AlgorithmSettings.append(MCMCSettings)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read special settings for each algorithm
        elif (FitAlgorithm == "simulated-annealing"):                                       ## read settings for Simulated-Annealing algorithm
            SimulatedAnnealingSettings = []


            ## get version of Simulated-Annealing algorithm
            if (NumberOfFitAlgorithms == 1):
                try:
                    MethodSA = str(XmlFile.FitControl.algorithm.MethodSA.getValue())
                except AttributeError, err:
                    MethodSA = "scipy"
            else:
                try:
                    MethodSA = str(XmlFile.FitControl.algorithm[algorithm].MethodSA.getValue())
                except AttributeError, err:
                    MethodSA = "scipy"
            MethodSA = MethodSA.lower()
            MethodSA = MethodSA.strip()
            if (MethodSA == "nr"):
                MethodSA = 2
            else:
                MethodSA = 1
            SimulatedAnnealingSettings.append(MethodSA)


            ## get schedule for scipy version of Simulated-Annealing algorithm
            if (NumberOfFitAlgorithms == 1):
                try:
                    ScheduleSA = str(XmlFile.FitControl.algorithm.ScheduleSA.getValue())
                except AttributeError, err:
                    ScheduleSA = "fast"
            else:
                try:
                    ScheduleSA = str(XmlFile.FitControl.algorithm[algorithm].ScheduleSA.getValue())
                except AttributeError, err:
                    ScheduleSA = "fast"
            ScheduleSA = ScheduleSA.lower()
            ScheduleSA = ScheduleSA.strip()
            if (ScheduleSA == "cauchy"):
                ScheduleSA = 2
            elif (ScheduleSA == "boltzmann"):
                ScheduleSA = 3
            else:
                ScheduleSA = 1
            SimulatedAnnealingSettings.append(ScheduleSA)


            ## get number of best sites
            if (NumberOfFitAlgorithms == 1):
                SimulatedAnnealingTemperature = XmlFile.FitControl.algorithm.Temperature.getValue()
            else:
                SimulatedAnnealingTemperature = XmlFile.FitControl.algorithm[algorithm].Temperature.getValue()


            ## check user input
            if (NumberQ(str(SimulatedAnnealingTemperature)) != "true"):                     ## is the number of best sites a number ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected temperture for the Simulated-Annealing algorithm is not a number !"
                print "\t\t\t Please correct the .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            elif (SimulatedAnnealingTemperature < 0):                                       ## is the number of best sites less than 1 ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected temperature for the Simulated-Annealing algorithm is negative !"
                print "\t\t\t Please correct .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            SimulatedAnnealingSettings.append(SimulatedAnnealingTemperature)


            ## get eduction coefficient for the temperture for the Simulated-Annealing algorithm
            if (NumberOfFitAlgorithms == 1):
                SimulatedAnnealingTemperatureReductionKoeff = XmlFile.FitControl.algorithm.TemperatureReductionKoeff.getValue()
            else:
                SimulatedAnnealingTemperatureReductionKoeff = XmlFile.FitControl.algorithm[algorithm].TemperatureReductionKoeff.getValue()


            ## check user input
            if (NumberQ(str(SimulatedAnnealingTemperatureReductionKoeff)) != "true"):       ## is the number of best sites a number ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected reduction coefficient for the temperture for the Simulated-Annealing"
                print "\t\t\t algorithm is not a number !"
                print "\t\t\t Please correct the .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            elif (SimulatedAnnealingTemperatureReductionKoeff < 0 or SimulatedAnnealingTemperatureReductionKoeff > 1):  
                                                                                            ## is the number of best sites less than 1 ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected reduction coefficient for the temperture for the Simulated-Annealing"
                print "\t\t\t algorithm is less than 0 or greater than 1 !"
                print "\t\t\t Please correct the .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            SimulatedAnnealingSettings.append(SimulatedAnnealingTemperatureReductionKoeff)


            ## get reduction coefficient for the temperture for the Simulated-Annealing algorithm
            if (NumberOfFitAlgorithms == 1):
                SimulatedAnnealingNumberOfReduction = XmlFile.FitControl.algorithm.NumberOfReduction.getValue()
            else:
                SimulatedAnnealingNumberOfReduction = XmlFile.FitControl.algorithm[algorithm].NumberOfReduction.getValue()


            ## check user input
            if (NumberQ(str(SimulatedAnnealingNumberOfReduction)) != "true"):               ## is the number of best sites a number ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of reductions for the temperture for the Simulated-Annealing"
                print "\t\t\t algorithm is not a number !"
                print "\t\t\t Please correct the .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            elif (SimulatedAnnealingNumberOfReduction < 1):                                 ## is the number of best sites less than 1 ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of reductions for the temperture for the Simulated-Annealing"
                print "\t\t\t algorithm is less than 1 !"
                print "\t\t\t Please correct the .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            SimulatedAnnealingSettings.append(SimulatedAnnealingNumberOfReduction)


            ## get max number of reheating phases for the Simulated-Annealing algorithm
            if (NumberOfFitAlgorithms == 1):
                try:
                    NumberOfReheatingPhases = XmlFile.FitControl.algorithm.NumberOfReheatingPhases.getValue()
                except AttributeError, err:
                    NumberOfReheatingPhases = 3
            else:
                try:
                    NumberOfReheatingPhases = XmlFile.FitControl.algorithm[algorithm].NumberOfReheatingPhases.getValue()
                except AttributeError, err:
                    NumberOfReheatingPhases = 3


            ## check user input
            if (NumberQ(str(NumberOfReheatingPhases)) != "true"):                           ## is the number of reheating phases a number ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of reheating phases for the Simulated-Annealing"
                print "\t\t\t algorithm is not a number !"
                print "\t\t\t Please correct the .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            elif (NumberOfReheatingPhases < 1):                                             ## is the number of reheating phases less than 1 ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of reheating phases for the Simulated-Annealing"
                print "\t\t\t algorithm is less than 1 !"
                print "\t\t\t Please correct the .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            SimulatedAnnealingSettings.append(int(NumberOfReheatingPhases))
            AlgorithmSettings.append(SimulatedAnnealingSettings)



        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read special settings for each algorithm
        elif (FitAlgorithm == "pso"):                                                       ## read settings for Particle-Swarm-Optimization algorithm
            PSOSettings = []


            ## get number of best sites
            if (NumberOfFitAlgorithms == 1):
                try:
                    PSOBestSiteCounter = XmlFile.FitControl.algorithm.BestSiteCounter.getValue()
                except AttributeError, err:
                    PSOBestSiteCounter = 1
            else:
                try:
                    PSOBestSiteCounter = XmlFile.FitControl.algorithm[algorithm].BestSiteCounter.getValue()
                except AttributeError, err:
                    PSOBestSiteCounter = 1


            ## check user input
            if (NumberQ(str(PSOBestSiteCounter)) != "true"):                                ## is the number of best sites a number ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of best sites for the PSO algorithm is not a number !"
                print "\t\t\t Please correct the .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            elif (PSOBestSiteCounter < 1):                                                  ## is the number of best sites less than 1 ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of best sites for the PSO algorithm is less than 1 !"
                print "\t\t\t Please correct .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            PSOSettings.append(PSOBestSiteCounter)


            ## get number of particles (deactivated)
            NumberParticles = 0
            # if (NumberOfFitAlgorithms == 1):
            #     try:
            #         NumberParticles = str(XmlFile.FitControl.algorithm.NumberParticles.getValue())
            #     except AttributeError, err:
            #         NumberParticles = "automatic"
            # else:
            #     try:
            #         NumberParticles = str(XmlFile.FitControl.algorithm[algorithm].NumberParticles.getValue())
            #     except AttributeError, err:
            #         NumberParticles = "automatic"
            # NumberParticles = NumberParticles.lower()
            # NumberParticles = NumberParticles.strip()
            # if (NumberParticles == "automatic" or NumberQ(NumberParticles) != "true"):
            #     NumberParticles = "0"
            # if (int(NumberParticles) < 0):
            #     NumberParticles = "0"
            PSOSettings.append(NumberParticles)
            AlgorithmSettings.append(PSOSettings)



        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read special settings for each algorithm
        elif (FitAlgorithm == "bees"):                                                      ## read settings for bees algorithm
            BeesSettings = []


            ## get number of best sites
            if (NumberOfFitAlgorithms == 1):
                try:
                    BeesBestSiteCounter = XmlFile.FitControl.algorithm.BestSiteCounter.getValue()
                except AttributeError, err:
                    BeesBestSiteCounter = 1
            else:
                try:
                    BeesBestSiteCounter = XmlFile.FitControl.algorithm[algorithm].BestSiteCounter.getValue()
                except AttributeError, err:
                    BeesBestSiteCounter = 1


            ## check user input
            if (NumberQ(str(BeesBestSiteCounter)) != "true"):                               ## is the number of best sites a number ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of best sites for the Bees algorithm is not a number !"
                print "\t\t\t Please correct the .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            elif (BeesBestSiteCounter < 1):                                                 ## is the number of best sites less than 1 ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of best sites for the Bees algorithm is less than 1 !"
                print "\t\t\t Please correct .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            BeesSettings.append(BeesBestSiteCounter)


            ## get number of bees
            if (NumberOfFitAlgorithms == 1):
                try:
                    NumberBees = str(XmlFile.FitControl.algorithm.NumberBees.getValue())
                except AttributeError, err:
                    NumberBees = "automatic"
            else:
                try:
                    NumberBees = str(XmlFile.FitControl.algorithm[algorithm].NumberBees.getValue())
                except AttributeError, err:
                    NumberBees = "automatic"
            NumberBees = NumberBees.lower()
            NumberBees = NumberBees.strip()
            if (NumberBees == "automatic" or NumberQ(NumberBees) != "true"):
                NumberBees = 0
            if (int(NumberBees) < 0):
                NumberBees = 0
            BeesSettings.append(NumberBees)
            AlgorithmSettings.append(BeesSettings)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read special settings for each algorithm
        elif (FitAlgorithm == "interval-ns" or FitAlgorithm == "ins"):                      ## read settings for interval nested-sampling algorithm
            IntervalNSSettings = []
            IntervalNSFlag = 1
            IntervalNSSettings.append(IntervalNSFlag)


            ## get critical bound for volume
            if (NumberOfFitAlgorithms == 1):
                try:
                    vol_bound = str(XmlFile.FitControl.algorithm.vol_bound.getValue())
                    if (vol_bound == ""):
                        vol_bound = 0.0
                    else:
                        vol_bound = float(vol_bound)
                except AttributeError, err:
                    vol_bound = 0.0
            else:
                try:
                    vol_bound = str(XmlFile.FitControl.algorithm[algorithm].vol_bound.getValue())
                    if (vol_bound == ""):
                        vol_bound = 0.0
                    else:
                        vol_bound = float(vol_bound)
                except AttributeError, err:
                    vol_bound = 0.0
            IntervalNSSettings.append(vol_bound)


            ## get critical bound for volume
            if (NumberOfFitAlgorithms == 1):
                try:
                    delta_incl = str(XmlFile.FitControl.algorithm.delta_incl.getValue())
                    if (delta_incl == ""):
                        delta_incl = 0.001
                    else:
                        delta_incl = float(delta_incl)
                except AttributeError, err:
                    delta_incl = 0.001
            else:
                try:
                    delta_incl = str(XmlFile.FitControl.algorithm[algorithm].delta_incl.getValue())
                    if (delta_incl == ""):
                        delta_incl = 0.001
                    else:
                        delta_incl = float(delta_incl)
                except AttributeError, err:
                    delta_incl = 0.001
            IntervalNSSettings.append(delta_incl)


            ## get number of best sites
            if (NumberOfFitAlgorithms == 1):
                try:
                    INSBestSiteCounter = XmlFile.FitControl.algorithm.BestSiteCounter.getValue()
                except AttributeError, err:
                    INSBestSiteCounter = 1
            else:
                try:
                    INSBestSiteCounter = XmlFile.FitControl.algorithm[algorithm].BestSiteCounter.getValue()
                except AttributeError, err:
                    INSBestSiteCounter = 1
            if (NumberQ(str(INSBestSiteCounter)) != "true"):                                ## is the number of best sites a number ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of best sites for the INS algorithm is not a number !"
                print "\t\t\t Please correct the .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            elif (INSBestSiteCounter < 1):                                                  ## is the number of best sites less than 1 ?
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The selected number of best sites for the INS algorithm is less than 1 !"
                print "\t\t\t Please correct .xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
            IntervalNSSettings.append(INSBestSiteCounter)
            AlgorithmSettings.append(IntervalNSSettings)



        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read special settings for each algorithm
        elif (FitAlgorithm == "additionalpackages"):                                        ## read settings for additonalpackages algorithm
            AdditionalPackagesSettings = []


            ## get number of best sites
            if (NumberOfFitAlgorithms == 1):
                minAlgorithm = XmlFile.FitControl.algorithm.minAlgorithm.getValue()
            else:
                minAlgorithm = XmlFile.FitControl.algorithm[algorithm].minAlgorithm.getValue()
            minAlgorithm = minAlgorithm.strip()
            AdditionalPackagesSettings.append(minAlgorithm)
            AlgorithmSettings.append(AdditionalPackagesSettings)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get limit for chi**2 from control file
        if (NumberOfFitAlgorithms == 1):
            chilm = XmlFile.FitControl.algorithm.limit_of_chi2.getValue()                   ## get lower limit (stopping creteria) for chi**2
        else:
            chilm = XmlFile.FitControl.algorithm[algorithm].limit_of_chi2.getValue()        ## get lower limit (stopping creteria) for chi**2


        ## check user input
        if (NumberQ(str(chilm)) != "true"):                                                 ## is chilm a number ?
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
            print "\t\t\t The selected limit for chi**2 is not a number !"
            print "\t\t\t Please correct the .xml-file and restart MAGIX."
            print "\n\n\n\n"
            ok = 1
            return ok
        elif (chilm < 0):                                                                   ## is the number of fit algorithms in a meaningful range?
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
            print "\t\t\t The selected limit for chi**2 is lower than 0."
            print "\t\t\t Please correct this value to a positive value and restart MAGIX."
            print "\n\n\n\n"
            ok = 1
            return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
        AlgorithmSettings.append(chilm)



        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get number of fit ranges from control file
        if (NumberOfFitAlgorithms == 1):
            try:
                numrange = XmlFile.FitControl.algorithm.number_fit_ranges.getValue()
            except AttributeError, err:
                numrange = 0
        else:
            try:
                numrange = XmlFile.FitControl.algorithm[algorithm].number_fit_ranges.getValue()
            except AttributeError, err:
                numrange = 0


        ## check user input
        if (NumberQ(str(numrange)) != "true"):                                              ## is the number of ranges a number ?
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
            print "\t\t\t The selected number of fit ranges is not a number !"
            print "\t\t\t Please correct the .xml-file and restart MAGIX."
            print "\n\n\n\n"
            ok = 1
            return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
        elif (numrange < 0):                                                                ## is the number of ranges a negative number ?
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
            print "\t\t\t The selected number of fit ranges is negative !"
            print "\t\t\t Please correct .xml-file and restart MAGIX."
            print "\n\n\n\n"
            ok = 1
            return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
        AlgorithmSettings.append(numrange)


        ## get start and end of fit range from control file
        xmn = []                                                                            ## start of fit range(s)
        xmx = []                                                                            ## end of fit range(s)
        if (numrange > 0):


            ## loop over ranges
            for CurrentRange in xrange(numrange):


                ## get start of fit range
                if (NumberOfFitAlgorithms == 1):
                    if (numrange == 1):
                        hilf1 = XmlFile.FitControl.algorithm.fit_range_start.getValue()     ## get beginning of fit range from control file
                    else:
                        hilf1 = XmlFile.FitControl.algorithm.fit_range_start[CurrentRange].getValue()
                                                                                            ## get beginning of fit range from control file
                else:
                    if (numrange == 1):
                        hilf1 = XmlFile.FitControl.algorithm[algorithm].fit_range_start.getValue()
                                                                                            ## get beginning of fit range from control file
                    else:
                        hilf1 = XmlFile.FitControl.algorithm[algorithm].fit_range_start[CurrentRange].getValue()
                                                                                            ## get beginning of fit range from control file
                hilf = str(hilf1).strip()
                hilf = hilf.split(',')
                for startRange in hilf:
                    if (NumberQ(startRange) != "true"):                                     ## is the number of ranges a number ?
                        print "\n\n\n\n"
                        print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                        print "\t\t\t The selected start of (a) fit range is not a number !"
                        print "\t\t\t start range: %s " % hilf1
                        print "\t\t\t Please correct the .xml-file and restart MAGIX."
                        print "\n\n\n\n"
                        ok = 1
                        return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
                xmn.append(map(float, hilf))                                                ## append entry to the list of starting points of fit ranges


                ## get end of fit range
                if (NumberOfFitAlgorithms == 1):
                    if (numrange == 1):
                        hilf1 = XmlFile.FitControl.algorithm.fit_range_end.getValue()       ## get end of fit range from control file
                    else:
                        hilf1 = XmlFile.FitControl.algorithm.fit_range_end[CurrentRange].getValue()
                                                                                            ## get end of fit range from control file
                else:
                    if (numrange == 1):
                        hilf1 = XmlFile.FitControl.algorithm[algorithm].fit_range_end.getValue()
                                                                                            ## get end of fit range from control file
                    else:
                        hilf1 = XmlFile.FitControl.algorithm[algorithm].fit_range_end[CurrentRange].getValue()
                                                                                            ## get end of fit range from control file
                hilf = str(hilf1).strip()
                hilf = hilf.split(',')
                for endRange in hilf:
                    if (NumberQ(endRange) != "true"):                                       ## is the number of ranges a number ?
                        print "\n\n\n\n"
                        print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                        print "\t\t\t The selected start of (a) fit range is not a number !"
                        print "\t\t\t end range: %s " % hilf1
                        print "\t\t\t Please correct the .xml-file and restart MAGIX."
                        print "\n\n\n\n"
                        ok = 1
                        return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
                xmx.append(map(float, hilf))


        ## what happens with xmn and xmx when numrange == 0?
        elif (numrange == 0):
            numrange = 0
            xmn = [0]
            xmx = [0]
        AlgorithmSettings.append(xmn)
        AlgorithmSettings.append(xmx)



        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get maximum number of iterations from control file
        if (NumberOfFitAlgorithms == 1):
            numiter = XmlFile.FitControl.algorithm.number_iterations.getValue()
        else:
            numiter = XmlFile.FitControl.algorithm[algorithm].number_iterations.getValue()


        ## check user input
        if (NumberQ(str(numiter)) != "true"):                                               ## is the number of ranges a number ?
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile!"
            print "\t\t\t The selected maximum number of iterations is not a number !"
            print "\t\t\t Please correct the .xml-file and restart MAGIX."
            print "\n\n\n\n"
            ok = 1
            return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
        elif (numiter < 1):                                                                 ## is number of iterations less than 1
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
            print "\t\t\t The selected maximum number of iterations is less than 1 !"
            print "\t\t\t Please correct .xml-file and restart the MAGIX."
            print "\n\n\n\n"
            ok = 1
            return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
        AlgorithmSettings.append(numiter)



        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get number of processors
        if (NumberOfFitAlgorithms == 1):
            try:
                NumberProcessors = XmlFile.FitControl.algorithm.NumberProcessors.getValue()
            except AttributeError, err:
                NumberProcessors = 1
        else:
            try:
                NumberProcessors = XmlFile.FitControl.algorithm[algorithm].NumberProcessors.getValue()
            except AttributeError, err:
                NumberProcessors = 1


        ## check user input
        if (NumberQ(str(NumberProcessors)) != "true"):                                      ## is the number of processors a number ?
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
            print "\t\t\t The selected number of processors is not a number !"
            print "\t\t\t Please correct .xml-file and restart the MAGIX."
            print "\n\n\n\n"
            ok = 1
            return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
        elif (NumberProcessors < 1):                                                        ## is number of processors less than 1
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
            print "\t\t\t The selected number of processors is less than 1 !"
            print "\t\t\t Please correct .xml-file and restart the MAGIX."
            print "\n\n\n\n"
            ok = 1
            return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
        AlgorithmSettings.append(NumberProcessors)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get path and name of MPI host file
        if (NumberOfFitAlgorithms == 1):
            try:
                MPIHostFileName = XmlFile.FitControl.algorithm.MPIHostFileName.getValue()
            except AttributeError, err:
                MPIHostFileName = ""
        else:
            try:
                MPIHostFileName = XmlFile.FitControl.algorithm[algorithm].MPIHostFileName.getValue()
            except AttributeError, err:
                MPIHostFileName = ""


        ## check host file
        MPIHostFileName = MPIHostFileName.strip()
        if (MPIHostFileName != "" and MPIHostFileName != "MPI_HOSTS"):


            ## check, if MPI version of MAGIX is compiled
            MPIFlag = "false"
            currentpath = MAGIXrootDirectory + "/Modules/Levenberg-Marquardt/bin/"
            MPIStarterPath = os.path.dirname(currentpath) + "/"
            if (os.path.exists(MPIStarterPath + "Levenberg-Marquardt__MPI-" + StarterExecutable)):
                MPIFlag = "true"


            ## check existence of host file only for MPI version
            if (MPIHostFileName != ""):
                if (not (os.path.exists(MPIHostFileName)) and MPIFlag == "true"):
                    print "\n\n\n\n"
                    print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                    print "\t\t\t The host file defined in the fit control xml file does not exists!"
                    print "\t\t\t Please correct .xml-file and restart the MAGIX."
                    print "\n\n\n\n"
                    ok = 1
                    return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings


            ## get name of current machine (system hostname)
            import socket
            try:
                SystemHostname = socket.gethostname()
            except ImportError:
                SystemHostname = "localhost"


            ## read in host file
            MPIHostFile = open(MPIHostFileName)
            MPIHostFileContent = MPIHostFile.readlines()
            MPIHostFile.close()
            if (MPIHostFileContent == []):
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                print "\t\t\t The host file defined in the fit control xml file does not exists!"
                print "\t\t\t Please correct .xml-file and restart the MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings


            ## check, if first entry in host file includes current host
            ChangeFlag = True
            NewHostFile = []
            CounterLines = 0
            CurrentHostLine = ""
            for line in MPIHostFileContent:
                i = line.find("#")
                if (i > (-1)):
                    StrippedLine = line[:i].strip()
                else:
                    StrippedLine = line.strip()
                if (StrippedLine != ""):
                    CounterLines += 1
                    if (CounterLines == 1):
                        if (StrippedLine.find(SystemHostname) > (-1) or StrippedLine.find("localhost") > (-1)):
                            ChangeFlag = False
                            break
                    if (StrippedLine.find(SystemHostname) > (-1)):
                        CurrentHostLine = StrippedLine
                    else:
                        NewHostFile.append(StrippedLine)
            if (ChangeFlag):                                                                ## first line in host file does not describe current host


                ## check, if current host is included in host file
                if (CurrentHostLine == ""):
                    print "\n\n\n\n"
                    print "\t\t ERROR in subroutine FittingEngine.ReadFitControlFile !"
                    print "\t\t\t The host file does not contain settings for the current host (" + SystemHostname + ")!"
                    print "\t\t\t Please correct host file and restart the MAGIX."
                    print "\n\n\n\n"

                    # Debug:
                    #    print "MPIHostFileName = ", MPIHostFileName
                    #    print "MPIHostFileContent = ", MPIHostFileContent
                    #    print "\n\n\n"
                else:
                    MPIHostFile = open(MPIHostFileName, 'w')
                    MPIHostFile.write(CurrentHostLine + "\n")
                    for line in NewHostFile:
                        MPIHostFile.write(line + "\n")
                    MPIHostFile.close()
        AlgorithmSettings.append(MPIHostFileName)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get method of determination of chi^2
        if (NumberOfFitAlgorithms == 1):
            try:
                DeterminationChi2 = XmlFile.FitControl.algorithm.DeterminationChi2.getValue()
            except AttributeError, err:
                DeterminationChi2 = "default"
        else:
            try:
                DeterminationChi2 = XmlFile.FitControl.algorithm[algorithm].DeterminationChi2.getValue()
            except AttributeError, err:
                DeterminationChi2 = "default"
        DeterminationChi2 = DeterminationChi2.lower()
        DeterminationChi2 = DeterminationChi2.strip()


        ## check user input
        if ( DeterminationChi2 == "default" ):
            DeterminationChi2 = 1
        elif ( DeterminationChi2 == "new" ):                                                ## not available yet
            DeterminationChi2 = 2
        elif ( DeterminationChi2 == "integration" ):                                        ## not available yet
            DeterminationChi2 = 3
        elif ( DeterminationChi2 == "difference" ):                                         ## not available yet
            DeterminationChi2 = 4
        elif ( DeterminationChi2 == "relative" ):                                           ## not available yet
            DeterminationChi2 = 5
        else:
            DeterminationChi2 = 1



        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get flag for saving chi^2
        if (NumberOfFitAlgorithms == 1):
            try:
                SaveChi2 = str(XmlFile.FitControl.algorithm.SaveChi2.getValue())
            except AttributeError, err:
                SaveChi2 = "yes"
        else:
            try:
                SaveChi2 = str(XmlFile.FitControl.algorithm[algorithm].SaveChi2.getValue())
            except AttributeError, err:
                SaveChi2 = "yes"
        SaveChi2 = SaveChi2.lower()
        SaveChi2 = SaveChi2.strip()


        ## check user input
        if (SaveChi2 == "yes" or SaveChi2 == "y"):
            DeterminationChi2 = -DeterminationChi2
        AlgorithmSettings.append(DeterminationChi2)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get plot format
        PlotOption = []                                                                     ## includes the options for plotting
        if (plotflag == "true"):
            PlotType = "normal"
            PlotOption.append(PlotType)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get label for x-axis
            if (NumberOfFitAlgorithms == 1):
                try:
                    PlotAxisX = str(XmlFile.FitControl.algorithm.PlotAxisX.getValue())
                except AttributeError, err:
                    PlotAxisX = "x-axis"
            else:
                try:
                    PlotAxisX = str(XmlFile.FitControl.algorithm[algorithm].PlotAxisX.getValue())
                except AttributeError, err:
                    PlotAxisX = "x-axis"
            PlotOption.append(PlotAxisX)



            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get label for y-axis
            if (NumberOfFitAlgorithms == 1):
                try:
                    PlotAxisY = str(XmlFile.FitControl.algorithm.PlotAxisY.getValue())
                except AttributeError, err:
                    PlotAxisY = "y-axis"
            else:
                try:
                    PlotAxisY = str(XmlFile.FitControl.algorithm[algorithm].PlotAxisY.getValue())
                except AttributeError, err:
                    PlotAxisY = "y-axis"
            PlotOption.append(PlotAxisY)



            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get label for z-axis
            if (NumberOfFitAlgorithms == 1):
                try:
                    PlotAxisZ = str(XmlFile.FitControl.algorithm.PlotAxisZ.getValue())
                except AttributeError, err:
                    PlotAxisZ = "z-axis"
            else:
                try:
                    PlotAxisZ = str(XmlFile.FitControl.algorithm[algorithm].PlotAxisZ.getValue())
                except AttributeError, err:
                    PlotAxisZ = "z-axis"
            PlotOption.append(PlotAxisZ)



            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get flag for plotting fit function for each iteration step
            if (NumberOfFitAlgorithms == 1):
                try:
                    PlotIteration = str(XmlFile.FitControl.algorithm.PlotIteration.getValue())
                except AttributeError, err:
                    PlotIteration = "no"
            else:
                try:
                    PlotIteration = str(XmlFile.FitControl.algorithm[algorithm].PlotIteration.getValue())
                except AttributeError, err:
                    PlotIteration = "no"
            PlotIteration = PlotIteration.lower()
            PlotOption.append(PlotIteration)
        else:                                                                               ## define pseudo plot options if noplot flag is given
            PlotOption.append("normal")
            PlotOption.append(" ")
            PlotOption.append(" ")
            PlotOption.append(" ")
            PlotOption.append(" ")
        AlgorithmSettings.append(PlotOption)



        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get flag for using renormalized chi**2
        if (NumberOfFitAlgorithms == 1):
            try:
                RenormalizedChi2 = str(XmlFile.FitControl.algorithm.RenormalizedChi2.getValue())
            except AttributeError, err:
                RenormalizedChi2 = "yes"
        else:
            try:
                RenormalizedChi2 = str(XmlFile.FitControl.algorithm[algorithm].RenormalizedChi2.getValue())
            except AttributeError, err:
                RenormalizedChi2 = "yes"
        RenormalizedChi2 = RenormalizedChi2.lower()
        RenormalizedChi2 = RenormalizedChi2.strip()
        if (RenormalizedChi2 == "y" or RenormalizedChi2 == "yes"):
            RenormalizedChi2 = 1
        else:
            RenormalizedChi2 = 0
        AlgorithmSettings.append(RenormalizedChi2)


        ## append all settings for the current algorithm to the final list
        AllAlgorithmSettings.append(AlgorithmSettings)


    ## print: we are finished
    if (printflag != "false"):
        print "done!"


    ## print settings for fit process
    if (printflag != "false"):
    #   print '\t UseHeuristic:             %s' % UseHeuristic
        print ' '
        print '\t NumberOfFitAlgorithms:    %s' % NumberOfFitAlgorithms
        for i in xrange(NumberOfFitAlgorithms):
            print '\t algorithm:                %s' % AllAlgorithmSettings[i][0]
            print '\t Special algorithm settings: %s' % AllAlgorithmSettings[i][1]
            print '\t chilm:                    %s' % AllAlgorithmSettings[i][2]
            print '\t numrange:                 %s' % AllAlgorithmSettings[i][3]
            for j in xrange(AllAlgorithmSettings[i][3]):
                print '\t xmn[i]:                   %s' % AllAlgorithmSettings[i][4][j]
                print '\t xmx[i]:                   %s' % AllAlgorithmSettings[i][5][j]
            print '\t numiter:                  %s' % AllAlgorithmSettings[i][6]
            print '\t NumberProcessors:         %s' % AllAlgorithmSettings[i][7]
            print '\t MPI host file:            %s' % AllAlgorithmSettings[i][8]
            if (AllAlgorithmSettings[i][9] == (-1)):
                print '\t DeterminationChi2:        default'
            else:
                print '\t DeterminationChi2:        something else'
            if (plotflag == "true"):
                print '\t PlotOption:               %s' % AllAlgorithmSettings[i][10]
            if (AllAlgorithmSettings[i][11] == 1):
                print '\t Renormalized Chi^2:       yes'
            else:
                print '\t Renormalized Chi^2:       no'
            print ' '

    return ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
def DetermineParameterValue(FitParameterName, ParamName, parameter_set, LoopFlag, LoopPosition):
    ## determine value of the parameter
    """

    function DetermineParameterValue:
                            determine the value of the parameter ParamName

    input parameters:       FitParameterName:           list including all names of the parameters
                            ParamName:                  name of the parameter
                            parameter_set:              parameter set
                            LoopFlag:                   flag for parameters within a loop

    output parameters:      value:                      value of the parameter ParamName

    working parameters:     i:                          loop counter
                            name:                       name of the ith parameter
                            loop_counter:               counts the occurance of the parameter with name ParamName with the list FitParameterName
                            ValueOfLoop:                contains all parameter values with name ParamName
    """

    loop_counter = 0
    ValueOfLoop = []

    i = -1
    value = -999999
    for name in FitParameterName:
        if (name.find("[[]]") != (-1)):
            k = len(name)
        else:
            k = name.find("[[")
            if (k == (-1)):
                k = len(name)
        i += 1

        ## do something special for parameters within a loop
        if (LoopFlag == "true"):
            if (name[:k] == ParamName):
                loop_counter += 1
                ValueOfLoop.append(parameter_set[0][i])
        else:
            if (name[:k] == ParamName):
                value = parameter_set[0][i]
                break

    if (LoopFlag == "true" and loop_counter > -1):

        # Debug:
        # print "ParamName = ",ParamName
        # print "LoopPosition = ",LoopPosition
        # print "ValueOfLoop = ",ValueOfLoop

        try:
            value = ValueOfLoop[LoopPosition]
        except IndexError, err:
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine DetermineParameterValue!"
            print "\t\t\t Can not find the value of the parameter = ", ParamName
            print "\n\t\t\t ParamName = ",ParamName
            print "\t\t\t LoopPosition = ",LoopPosition
            print "\t\t\t ValueOfLoop = ",ValueOfLoop
            print "\n"
            print "\t\t\t Please correct parameter xml-file and restart MAGIX."
            sys.exit(0)

    return value
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
def ReadRegistrationFile(print_flag, debugflag, NumberParameter, parameter_set, FitParameterName, ParameterValuesOriginal, XmlFileInOutput, JobDir, \
                         NumberExpFiles):
    ## read registration file
    """

    function ReadRegistrationFile:
                            read the xml description of the input and the output file(s) which are required to communicate with the fit
                            function module.

    input parameters:       print_flag:                 print flag
                            debugflag:                  flag for debugging call of external model program
                            NumberParameter:            number of model parameter
                            parameter_set:              parameter set
                            FitParameterName:           array containing the names of the model parameters
                            ParameterValuesOriginal:    array containing the original values of the model parameters
                            XmlFileInOutput:            path and file name of the xml file containing the description of the input
                                                        and the output file
                            JobDir:                     current job directory for MAGIX
                            NumberExpFiles:             number of experimental data files

    output paramters:       ok:                         status of reading
                            CalculationMethod:          method of computation
                            PathStartScript:            path where the start script is located
                            ExeCommandStartScript:      command to execute the start script
                            FitFunctionIn:              path and file name of the input file
                            TotalNumberInputLines:      total number of lines of the input file
                            MaxNumberParameter:         max number of parameters in one line

    working parameters:     i:                          loop counter
                            j:                          loop counter
                            XmlFile:                    contents of the xml file
                            NumberLines:                number of lines in the input/output file
                            NumberReplicationParameter: number of replications of the parameters in the ith line
                            NumberParameterLine:        number of model parameters in a line
                            LeadingString:              leading string
                            ParameterName:              name of the model parameter
                            ParameterFormat:            format of the model parameter
                            TrailingString:             trailing string
                            NameIncluded:               temp variable used in the name check
                            ParameterNameList:          list including all names of the parameter given in the registration file


    Example xml-file (input/output file):
    -------------------------------------

    <?xml version="1.0" encoding="ISO-8859-1"?>
    <InOutputFile>
        <ModelProgramCall>
            <PathStartScript>Fit-Functions/Drude-Lorentz_conv/bin/</PathStartScript>
            <ExeCommandStartScript>DrudeLorentzConv.exe</ExeCommandStartScript>
            <CalculationMethod>AtOnce</CalculationMethod>
            <ParallelizationPossible>Yes</ParallelizationPossible>
        </ModelProgramCall>
        <NumberInputFiles>1</NumberInputFiles>
        <InputFile>
            <InputFileName>Fit-Functions/Drude-Lorentz_conv/temp/in.txt</InputFileName>
            <title>input description for conventional Drude-Lorentz model</title>
            <NumberLines>4</NumberLines>
            <line group="false">
                    <NumberParameterLine>1</NumberParameterLine>
                <Parameter group="false">
                    <NumberReplicationParameter> </NumberReplicationParameter>
                    <Name>XValue1</Name>
                    <Format>F15.8</Format>
                    <LeadingString> </LeadingString>
                    <TrailingString> </TrailingString>
                </Parameter>
            </line>
            <line group="false">
                    <NumberParameterLine>1</NumberParameterLine>
                <Parameter group="false">
                    <Name>EpsilonInfinity</Name>
                    <Format>F15.8</Format>
                    <NumberReplicationParameter> </NumberReplicationParameter>
                    <LeadingString> </LeadingString>
                    <TrailingString> </TrailingString>
                </Parameter>
            </line>
            <line group="false">
                    <NumberParameterLine>1</NumberParameterLine>
                <Parameter group="false">
                    <NumberReplicationParameter> </NumberReplicationParameter>
                    <Name>NumberOscillators</Name>
                    <Format>I6</Format>
                    <LeadingString> </LeadingString>
                    <TrailingString> </TrailingString>
                </Parameter>
            </line>
            <line group="group1: start, replication = NumberOscillators">
                <NumberParameterLine>3</NumberParameterLine>
                <Parameter group="false">
                    <NumberReplicationParameter> </NumberReplicationParameter>
                    <Name>EigenFrequency</Name>
                    <Format>F8.1</Format>
                    <LeadingString> </LeadingString>
                    <TrailingString> </TrailingString>
                </Parameter>
                <Parameter group="false">
                    <NumberReplicationParameter> </NumberReplicationParameter>
                    <Name>PlasmaFrequency</Name>
                    <Format>F11.3</Format>
                    <LeadingString> </LeadingString>
                    <TrailingString> </TrailingString>
                </Parameter>
                <Parameter group="false">
                    <NumberReplicationParameter> </NumberReplicationParameter>
                    <Name>Damping</Name>
                    <Format>F11.3</Format>
                    <LeadingString> </LeadingString>
                    <TrailingString> </TrailingString>
                </Parameter>
            </line>
        </InputFile>
        <OutputFile>
            <title>xml-description of the output file for the conventional Drude-Lorentz model</title>
            <NumberLines>4</NumberLines>
            <NumberColumnsBegin>1</NumberColumnsBegin>
            <NumberColumnsEnd>10</NumberColumnsEnd>
            <CommandWord> </CommandWord>
        </OutputFile>
    </InOutputFile>

    """

    if (print_flag == "false"):
        printdoneflag = "false"
    else:
        printdoneflag = "true"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## set default values of output variables
    ok = 0
    PathStartScript = ""
    ExeCommandStartScript = ""
    CalculationMethod = "atonce"
    ParallelizationPossible = "no"
    NumberInputFiles = 1
    OnlyYColumn = "no"
    MaxInputLines = 0
    MaxParameter = 0


    ## set max number of groups in registration xml-file
    MaxNumberOfGroups = 100


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read xml-description of the INPUT file
    ## is the defined file a xml-file ?
    if not XmlFileInOutput.endswith(".xml"):
        print "\n\n\n\n"
        print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
        print "\t\t\t The selected file is not a xml-file!"
        print "\t\t\t Please select a .xml and restart the whole program."
        print "\n\n\n\n"
        ok = 1
        return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
               MaxInputLines, MaxParameter


    ## print what you do ..
    if (print_flag != "false"):
        print "\t Read the xml-description of the input and the output file of the chosen fit-function module."
        print "\t Open xml-description stored in file: " + XmlFileInOutput
        print "\t The maximum number of groups in the registration file is set to MaxNumberOfGroups = ", MaxNumberOfGroups
        print "\t   If you use more groups in the registration file, please increase the variable MaxNumberOfGroups"
        print "\t   defined in line 2339 of the file Modules/magix-parts_python/FittingEngine.py"
        print "\n\t Reading registration xml-file ..",


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## open xml-file file and save content of file to variable XmlFile
    XmlFile = MAGIXxmlPACKAGE.XmlObject(fileName = XmlFileInOutput)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get full path of start script
    try:
        PathStartScript = XmlFile.InOutputFile.ModelProgramCall.PathStartScript.getValue()
    except AttributeError, err:
        try:
            PathStartScript = XmlFile.InOutputFile.Section.ModelProgramCall.PathStartScript.getValue()
        except AttributeError, err:
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
            print "\t\t\t An error occurs while reading the tag <PathStartScript></PathStartScript> from the registration xml-file."
            print "\t\t\t Please check the registration xml-file and restart MAGIX."
            print "\n\n\n\n"
            ok = 1
            return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
                   MaxInputLines, MaxParameter
    PathStartScript = PathStartScript.strip()
    if (PathStartScript.endswith("/")):
        print "\n\n\n\n"
        print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
        print "\t\t\t The defined path of the start script has to include the name of the start script as well."
        print "\t\t\t Please add the full path of the start script including the name of the start script."
        print "\t\t\t and restart the whole program."
        print "\n\n\n\n"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get execution command of start script
    try:
        ExeCommandStartScript = XmlFile.InOutputFile.ModelProgramCall.ExeCommandStartScript.getValue()
    except AttributeError, err:
        try:
            ExeCommandStartScript = XmlFile.InOutputFile.Section.ModelProgramCall.ExeCommandStartScript.getValue()
        except AttributeError, err:
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
            print "\t\t\t An error occurs while reading the tag <ExeCommandStartScript></ExeCommandStartScript> from the registration xml-file."
            print "\t\t\t Please check the registration xml-file and restart MAGIX."
            print " "
            print " ",
            ok = 1
            return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
                   MaxInputLines, MaxParameter
    ExeCommandStartScript = ExeCommandStartScript.strip()
    if (not ExeCommandStartScript.startswith("./") and not ExeCommandStartScript.startswith("sh") \
        and not ExeCommandStartScript.startswith("python") and not ExeCommandStartScript.startswith("chmod")):
        print " "
        print "\t WARNING:"
        print "\t\t\t The defined execution command for running the start script does not start with ./ or sh"
        print "\t\t\t Note, that MAGIX uses exactly the command defined in the tag <ExeCommandStartScript>"
        print "\t\t\t to execute the start script."
        print " "
        print " ",


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## old statement (NO LONGER USED): get calculation method
    #   CalculationMethod = XmlFile.InOutputFile.ModelProgramCall.CalculationMethod.getValue()
    #   CalculationMethod = CalculationMethod.lower()
    #   if (CalculationMethod != "pointtopoint"):
    #       if (CalculationMethod != "atonce"):
    #           print "\n\n\n\n"
    #           print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
    #           print "\t\t\t The defined calculation method is not known"
    #           print "\t\t\t Please define 'PointToPoint' or 'AtOnce' and"
    #           print "\t\t\t restart the whole program."
    #           print "\n\n\n\n"
    #           ok = 1
    #           return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
    #                  MaxInputLines, MaxParameter


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read path and file name of the input data file
    try:
        InputDataPath = XmlFile.InOutputFile.ModelProgramCall.InputDataPath.getValue()
    except AttributeError, err:
        try:
            InputDataPath = XmlFile.InOutputFile.Section.ModelProgramCall.InputDataPath.getValue()
        except AttributeError, err:
            InputDataPath = " "


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get ParallelizationPossible flag
    try:
        ParallelizationPossible = XmlFile.InOutputFile.ModelProgramCall.ParallelizationPossible.getValue()
    except AttributeError, err:
        try:
            ParallelizationPossible = XmlFile.InOutputFile.Section.ModelProgramCall.ParallelizationPossible.getValue()
        except AttributeError, err:
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
            print "\t\t\t An error occurs while reading the tag <ParallelizationPossible></ParallelizationPossible> from the registration xml-file."
            print "\t\t\t Please check the registration xml-file and restart MAGIX."
            print "\n\n\n\n"
            ok = 1
            return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
                   MaxInputLines, MaxParameter
    ParallelizationPossible = ParallelizationPossible.lower()
    if (ParallelizationPossible != "yes"):
        if (ParallelizationPossible != "no"):
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
            print "\t\t\t The defined content of the tag ParallelizationPossible"
            print "\t\t\t is not known."
            print "\t\t\t Please define 'yes' or 'no' and"
            print "\t\t\t restart the whole program."
            print "\n\n\n\n"
            ok = 1
            return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
                   MaxInputLines, MaxParameter


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get number of input-files description
    try:
        NumberInputFiles = XmlFile.InOutputFile.NumberInputFile.getValue()                  ## read tag <NumberInputFile></NumberInputFile>
    except AttributeError, err:
        try:
            NumberInputFiles = XmlFile.InOutputFile.NumberInputFiles.getValue()             ## read tag <NumberInputFiles></NumberInputFile>
        except AttributeError, err:
            try:
                NumberInputFiles = XmlFile.InOutputFile.Section.NumberInputFile.getValue()  ## read tag <NumberInputFile></NumberInputFile>
            except AttributeError, err:
                NumberInputFiles = XmlFile.InOutputFile.Section.NumberInputFiles.getValue() ## read tag <NumberInputFiles></NumberInputFiles>
    if (NumberInputFiles < 1):
        print "\n\n\n\n"
        print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
        print "\t\t\t The defined number of input-files description in the registration"
        print "\t\t\t file is < 1!"
        print "\t\t\t Please correct .xml-file for input description and"
        print "\t\t\t restart the whole program."
        print "\n\n\n\n"
        ok = 1
        return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
               MaxInputLines, MaxParameter


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## loop over all input-file descriptions
    for InputFileNumber in xrange(NumberInputFiles):


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## set default values for each input-file description
        FitFunctionIn = ""
        TotalNumberInputLines = 0
        MaxNumberParameter = 0


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine string for reading tag from xml-file
        try:
            XmlString = "XmlFile.InOutputFile.InputFile"                                    ## define xml-string
            if (NumberInputFiles > 1):                                                      ## expand xml-string if there are more than 1 input
                XmlString = XmlString + "[" + str(InputFileNumber) + "]"                    ## file specified
            exec('testOut = str(' + XmlString + '.InputFileName.getValue())')
        except AttributeError, err:
            XmlString = "XmlFile.InOutputFile.Section.InputFile"                            ## define xml-string
            if (NumberInputFiles > 1):                                                      ## expand xml-string if there are more than 1 input
                XmlString = XmlString + "[" + str(InputFileNumber) + "]"                    ## file specified


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## construct path and file name for temp registration file
        ExtentionFileName = JobDir.strip()
        ExtentionFileName = ExtentionFileName + "/Registration_Input_" + str(InputFileNumber + 1) + ".txt"
        TempFile = open(ExtentionFileName, 'w')                                             ## open temp-file for each input file


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write value of debug flag to temp file for the first input file
        if (InputFileNumber == 0):
            TempFile.write("debugflag = " + debugflag + "|\n")                              ## write value of debug flag to temp file


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get path and file name of the input file
        exec('FitFunctionIn = str(' + XmlString + '.InputFileName.getValue())')
        TempFile.write("FitFunctionIn = " + FitFunctionIn + "|\n")                          ## write name of input file to temp-file


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get number of lines
        NumberLines = 0
        exec('NumberLines = ' + XmlString + '.NumberLines.getValue()')
        if (NumberLines < 1):
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
            print "\t\t\t The defined number of lines in the input file is < 1!"
            print "\t\t\t Please correct .xml-file for input description and"
            print "\t\t\t restart the whole program."
            print "\n\n\n\n"
            TempFile.close()
            ok = 1
            return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
                   MaxInputLines, MaxParameter


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read lines
        ParameterNameList = []


        ## set variables for group management to default values
        CurrentReplicationGroupLine = 1


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## read xml-description of the xml-file into several variables
        GroupLineFlag = []
        NumberParameterLine = []
        NumberReplicationLine = []
        GroupParameterFlag = []
        NumberReplicationParameter = []
        LeadingString = []
        ParameterName = []
        ParameterFormat = []
        TrailingString = []
        ParamVisible = []
        for i in xrange(NumberLines):                                                       ## loop over all lines in the xml-file


            ##
            XmlStringLine = XmlString + ".line"
            if (NumberLines > 1):
                XmlStringLine = XmlStringLine + "[" + str(i) + "]"

            # Debug:
            # print "i, XmlStringLine = ", i, XmlStringLine


            ## get group flag for the ith line
            exec('GroupLineFlagXML = str(' + XmlStringLine + '.getAttribute("group"))')
            if (GroupLineFlagXML == "0"):
                GroupLineFlagXML = "false"
            GroupLineFlag.append(GroupLineFlagXML)


            ## get number of replications of current line
            NumberReplicationLineXML = 1
            NumberReplicationLine.append(NumberReplicationLineXML)


            ## get number of parameters in the ith line
            exec('NumberParameterLineXML = ' + XmlStringLine + '.NumberParameterLine.getValue()')
            if (str(NumberParameterLineXML) == " " or str(NumberParameterLineXML) == ""):   ## in case of empty entries, set number to 1
                NumberParameterLineXML = 1
            elif (NumberParameterLineXML < 0):
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
                print "\t\t\t The defined number of lines in the xml-file is < 0."
                print "\t\t\t Please correct .xml-file for the input description and"
                print "\t\t\t restart the whole program."
                print "\n\n\n\n"
                TempFile.close()
                ok = 1
                return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
                       MaxInputLines, MaxParameter
            if (MaxNumberParameter < NumberParameterLineXML):
                MaxNumberParameter = NumberParameterLineXML
            NumberParameterLine.append(NumberParameterLineXML)


            ## get descriptions for each parameter
            GroupParameterFlagLine = []
            NumberReplicationParameterLine = []
            LeadingStringLine = []
            ParameterNameLine = []
            ParameterFormatLine = []
            TrailingStringLine = []
            ParamVisibleLine = []
            for j in xrange(NumberParameterLineXML):                                        ## loop over all parameter in the ith line

                if (NumberParameterLineXML == 1):
                    execpart = XmlStringLine + ".Parameter"
                else:
                    execpart = XmlStringLine + ".Parameter[j]"

                exec('GroupParameterFlagXML = str(' + execpart + '.getAttribute("group"))')
                                                                                            ## get group flag of the jth parameter in the ith line
                if (GroupParameterFlagXML == "0"):                                          ## translate "0" to "false"
                    GroupParameterFlagXML = "false"
                else:
                    GroupParameterFlagXML = "true"
                GroupParameterFlagLine.append(GroupParameterFlagXML)

                exec('NumberReplicationParameterXML = ' + execpart + '.NumberReplicationParameter.getValue()')
                                                                                            ## get number of replication of the jth parameter in the ith line
                if (str(NumberReplicationParameterXML) == " " or str(NumberReplicationParameterXML) == ""):   ## in case of empty entries, set number to 1
                    NumberReplicationParameterXML = 1
                elif (NumberReplicationParameterXML < 0):
                    print "\n\n\n\n"
                    print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
                    print "\t\t\t The defined number of model parameters in the %i line " % i
                    print "\t\t\t of the input file is < 0."
                    print "\t\t\t Please correct .xml-file for the input description and"
                    print "\t\t\t restart the whole program."
                    print "\n\n\n\n"
                    TempFile.close()
                    ok = 1
                    return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
                           MaxInputLines, MaxParameter
                NumberReplicationParameterLine.append(NumberReplicationParameterXML)


                ## get leading string of replication of the jth parameter in the ith line
                exec('LeadingStringLine.append(str(' + execpart + '.LeadingString.getValue()))')


                ## get name of replication of the jth parameter in the ith line
                exec('ParameterNameLine.append(str(' + execpart + '.Name.getValue()))')


                ## get format of replication of the jth parameter in the ith line
                exec('ParameterFormatLine.append(str(' + execpart + '.Format.getValue()))')


                ## get trailing string of replication of the jth parameter in the ith line
                exec('TrailingStringLine.append(str(' + execpart + '.TrailingString.getValue()))')


                ## get visible flag of replication of the jth parameter in the ith line
                ParamVisibletag = "t"
                try:
                    exec('ParamVisibletag = ' + execpart + '.Visible.getValue()')

                    # Debug:
                    # print "ParamVisibletag = ", ParamVisibletag


                    ## check if tag value defines a parameter name
                    if (ParamVisibletag.lower() != "t" and ParamVisibletag.lower() != "f"):


                        ## analyze group element
                        CounterGroupPart = 0
                        GroupElementsPart = ParamVisibletag.split(";")
                        ParamVisibletag = ""
                        for GroupPart in GroupElementsPart:                                 ## loop over all command words
                            CounterGroupPart += 1

                            # Debug:
                            # print "GroupPart = ", GroupPart


                            ## get name of parameter and required value for visibility
                            PosEqualSign = GroupPart.find("=")
                            if (PosEqualSign == (-1)):                                      ## if no "=" sign is found -> Error message
                                print "\n\n\n\n"
                                print "\t\t ERROR in subroutine ReadRegistrationFile:"
                                print "\t\t\t There is no value given for the Visible tag"
                                print "\t\t\t The value has to be given in the form"
                                print "\t\t\t ParameterName = Value"
                                print "\t\t\t Please modify the registration file and restart"
                                print "\t\t\t the whole program."
                                print "\n\n\n\n"
                                ok = 1
                                return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, \
                                       NumberOutputFiles, MaxInputLines, MaxParameter
                            else:
                                ParamName = GroupPart[:PosEqualSign - 1].strip()            ## get value of replication
                                value = GroupPart[PosEqualSign + 1:].strip()


                            ## determine value of the parameter
                            loop_counter = 0
                            ValueOfLoop = []
                            i = (-1)
                            ParamValue = -999999
                            OccurranceCounter = 0
                            for name in FitParameterName:
                                i += 1                                                      ## increase counter


                                ## remove index from parmeter name
                                if (name.find("[[]]") != (-1)):
                                    k = len(name)
                                else:
                                    k = name.find("[[")
                                    if (k == (-1)):
                                        k = len(name)


                                ## get value of parameter
                                if (name[:k] == ParamName):
                                    OccurranceCounter += 1
                                    ParamValue = ParameterValuesOriginal[i]

                                    # Debug:
                                    # print "FitParameterName = ", FitParameterName
                                    # print "ParameterValuesOriginal = ", ParameterValuesOriginal
                                    # print "ParamName = ", ParamName
                                    # print "ParamValue = ", ParamValue
                                    # print "value = ", value


                                    ## if parameter value is equal to required value parameter is visible
                                    if (ParamValue == value):
                                        if (OccurranceCounter == 1):
                                            if (CounterGroupPart > 1):
                                                ParamVisibletag += ";t"
                                            else:
                                                ParamVisibletag += "t"
                                        else:
                                            ParamVisibletag += "t"
                                    else:
                                        if (OccurranceCounter == 1):
                                            if (CounterGroupPart > 1):
                                                ParamVisibletag += ";f"
                                            else:
                                                ParamVisibletag += "f"
                                        else:
                                            ParamVisibletag += "f"


                            ## if at least one condition is not fulfilled, set visibility flag to false
                            if (OccurranceCounter == 1):
                                if (ParamVisibletag == "f"):
                                    ParamVisibletag = "f"
                                    break

                            # Debug:
                            # print "ParamVisibletag = ", ParamVisibletag


                        ## be sure, that visibility flag is defined
                        if (ParamVisibletag == ""):
                            ParamVisibletag = "t"


                except AttributeError, err:
                    ParamVisibletag = "t"
                ParamVisibleLine.append(ParamVisibletag)


            ## append contents to the several lists
            NumberReplicationParameter.append(NumberReplicationParameterLine)
            GroupParameterFlag.append(GroupParameterFlagLine)
            LeadingString.append(LeadingStringLine)
            ParameterName.append(ParameterNameLine)
            ParameterFormat.append(ParameterFormatLine)
            TrailingString.append(TrailingStringLine)
            ParamVisible.append(ParamVisibleLine)


        # Debug:
        # print " "
        # print "NumberLines = ", NumberLines
        # for i in xrange(NumberLines):
        #    print " "
        #    print "GroupLineFlag = >>%s<<" % GroupLineFlag[i]
        #    print "NumberParameterLine = >>%s<<" % NumberParameterLine[i]
        #    for j in xrange(NumberParameterLine[i]):
        #        print "     GroupParameterFlag = >>%s<<" % GroupParameterFlag[i][j]
        #        print "     NumberReplicationParameter = >>%s<<" % NumberReplicationParameter[i][j]
        #        print "     LeadingString = >>%s<<" % LeadingString[i][j]
        #        print "     ParameterName = >>%s<<" % ParameterName[i][j]
        #        print "     ParameterFormat = >>%s<<" % ParameterFormat[i][j]
        #        print "     TrailingString = >>%s<<" % TrailingString[i][j]
        #        print "     ParamVisible = >>%s<<" % ParamVisible[i][j]
        #        print "     #############################################################"


        ##----------------------------------------------------------------------------------------------------------------------------------------------------
        ## analysis of the group behaviour
        for GroupIndex in xrange(MaxNumberOfGroups):
            LoopPosition = 0

            ## set name for identification of the current group
            groupName = "group" + str(GroupIndex + 1) + ":"


            ## set group variables
            NumberParameterLine_Group = []
            GroupLineFlag_Group = []
            NumberReplicationLine_Group = []
            GroupParameterFlag_Group = []
            NumberReplicationParameter_Group = []
            LeadingString_Group = []
            ParameterName_Group = []
            ParameterFormat_Group = []
            TrailingString_Group = []
            ParamVisible_Group = []
            Counter_Group = 0
            NumberRepl_Group = 1

            NumberParameterLine_Group_All = []
            GroupLineFlag_Group_All = []
            NumberReplicationLine_Group_All = []
            GroupParameterFlag_Group_All = []
            NumberReplicationParameter_Group_All = []
            LeadingString_Group_All = []
            ParameterName_Group_All = []
            ParameterFormat_Group_All = []
            TrailingString_Group_All = []
            ParamVisible_Group_All = []
            Counter_Group_All = []
            NumberRepl_Group_All = []

            NumberParameterLine_Single = []
            GroupLineFlag_Single = []
            NumberReplicationLine_Single = []
            GroupParameterFlag_Single = []
            NumberReplicationParameter_Single = []
            LeadingString_Single = []
            ParameterName_Single = []
            ParameterFormat_Single = []
            TrailingString_Single = []
            ParamVisible_Single = []
            Counter_Single = 0

            EmptyFlag = "true"
            for i in xrange(NumberLines):                                                   ## loop over all lines in the xml-file
                CurrentGroupIncluded = "false"


                ## analyze group line flag
                GroupLineFlagElements = GroupLineFlag[i].split(";")                         ## get elements of the group line flag
                for elements in GroupLineFlagElements:                                      ## loop over all elements of the group line flag
                    GroupElements = elements.strip()                                        ## remove leading and tailing blanks

                    # Debug:
                    # print "->%s<-" % GroupElements

                    if (GroupElements.lower().find(groupName) != -1):                       ## get index of group
                        CurrentGroupIncluded = "true"


                        ## analyze group element
                        GroupElementsPart = GroupElements.split(",")
                        for GroupPart in GroupElementsPart:                                 ## loop over all command words


                            ## get number of replications of group
                            # if (GroupPart.find("set_replication") != -1):                 ## old version
                            if (GroupPart.lower().find("replication") != (-1)):


                                ## modification: set_replication flag is replaced by "replication = ParameterA" and the tag NumberReplicationLine
                                ## is removed
                                PosEqualSign = GroupPart.find("=")                          ## get position of "=" sign
                                if (PosEqualSign == (-1)):                                  ## if no "=" sign is found -> Error message
                                    print "\n\n\n\n"
                                    print "\t\t ERROR in subroutine ReadRegistrationFile:"
                                    print "\t\t\t There is no value given for the number of replication"
                                    print "\t\t\t The value has to be given in the form"
                                    print "\t\t\t replication = ParameterName"
                                    print "\t\t\t Please modify the registration file and restart"
                                    print "\t\t\t the whole program."
                                    print "\n\n\n\n"
                                    ok = 1
                                    return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, \
                                           NumberOutputFiles, MaxInputLines, MaxParameter
                                else:
                                    NumberRepl_Group = GroupPart[PosEqualSign+1:]           ## get value of replication

                                # Debug:
                                # print " "
                                # print 'NumberRepl_Group = ',NumberRepl_Group
                                # sys.exit(0)


                                if (not str(NumberRepl_Group).isdigit()):                   ## if NumberRepl_Group is not a number determine value
                                    LoopFlag = "true"
                                    NumberRepl_Group = NumberRepl_Group.strip()
                                    NumberRepl_Group = DetermineParameterValue(FitParameterName, NumberRepl_Group, parameter_set, LoopFlag, LoopPosition)
                                    NumberReplicationLine[i] = NumberRepl_Group

                                    # Debug:
                                    # print 'NumberRepl_Group = ',NumberRepl_Group
                                    # sys.exit(0)


                            ## get beginning of current group
                            elif (GroupPart.lower().find("start") != -1):

                                if (EmptyFlag == "false"):


                                    ## append group variable to total group variable
                                    NumberParameterLine_Group_All.append(NumberParameterLine_Group)
                                    GroupLineFlag_Group_All.append(GroupLineFlag_Group)
                                    NumberReplicationLine_Group_All.append(NumberReplicationLine_Group)
                                    GroupParameterFlag_Group_All.append(GroupParameterFlag_Group)
                                    NumberReplicationParameter_Group_All.append(NumberReplicationParameter_Group)
                                    LeadingString_Group_All.append(LeadingString_Group)
                                    ParameterName_Group_All.append(ParameterName_Group)
                                    ParameterFormat_Group_All.append(ParameterFormat_Group)
                                    TrailingString_Group_All.append(TrailingString_Group)
                                    ParamVisible_Group_All.append(ParamVisible_Group)
                                    Counter_Group_All.append(Counter_Group)
                                    NumberRepl_Group_All.append(NumberRepl_Group)


                                    ## the command "set:replication_" indicate the start of each group
                                    ## start with reseting the group variables containing all information for each group member
                                    LoopPosition += 1
                                    NumberParameterLine_Group = []
                                    GroupLineFlag_Group = []
                                    NumberReplicationLine_Group = []
                                    GroupParameterFlag_Group = []
                                    NumberReplicationParameter_Group = []
                                    LeadingString_Group = []
                                    ParameterName_Group = []
                                    ParameterFormat_Group = []
                                    TrailingString_Group = []
                                    ParamVisible_Group = []
                                    Counter_Group = 0
                                    NumberRepl_Group = 1

                                else:
                                    EmptyFlag = "false"


                        ## add line and parameter settings to the current group variable with group name: groupName
                        NumberReplicationLine_Group.append(NumberReplicationLine[i])
                        GroupLineFlag_Group.append(GroupLineFlag[i])
                        NumberParameterLine_Group.append(NumberParameterLine[i])
                        GroupParameterFlag_Group.append(GroupParameterFlag[i])
                        NumberReplicationParameter_Group.append(NumberReplicationParameter[i])
                        LeadingString_Group.append(LeadingString[i])
                        ParameterName_Group.append(ParameterName[i])
                        ParameterFormat_Group.append(ParameterFormat[i])
                        TrailingString_Group.append(TrailingString[i])
                        ParamVisible_Group.append(ParamVisible[i])
                        Counter_Group += 1                                                  ## increase counter of lines for current group



                if (CurrentGroupIncluded == "false"):                                       ## if lines do not belong to the current group


                    ## determine number of parameter per line
                    NumberParam_Single = NumberParameterLine[i]
                    if (not str(NumberParam_Single).isdigit()):                             ## if NumberRepl_Group is not a number determine value
                        LoopFlag = "false"
                        NumberParam_Single = DetermineParameterValue(FitParameterName, NumberParam_Single, parameter_set, LoopFlag, LoopPosition)
                    NumberParameterLine_Single.append(NumberParam_Single)


                    ## determine number of replication of ith line
                    NumberReplLine_Single = NumberReplicationLine[i]
                    if (not str(NumberReplLine_Single).isdigit()):                          ## if NumberRepl_Group is not a number determine value
                        LoopFlag = "false"
                        NumberReplLine_Single = DetermineParameterValue(FitParameterName, NumberReplLine_Single, parameter_set, LoopFlag, LoopPosition)
                    NumberReplicationLine_Single.append(NumberReplLine_Single)


                    ## determine number of replication of each parameter in the ith line
                    NumberReplParam_Single = []
                    for NumberReplParamNum_Single in NumberReplicationParameter[i]:
                        if (not str(NumberReplParamNum_Single).isdigit()):                  ## if NumberRepl_Group is not a number determine value
                            LoopFlag = "false"
                            NumberReplParamNum_Single = DetermineParameterValue(FitParameterName, NumberReplParamNum_Single, parameter_set, LoopFlag, LoopPosition)
                        NumberReplParam_Single.append(NumberReplParamNum_Single)
                    NumberReplicationParameter_Single.append(NumberReplParam_Single)


                    ## copy contents of several files to single variables
                    GroupLineFlag_Single.append(GroupLineFlag[i])
                    GroupParameterFlag_Single.append(GroupParameterFlag[i])
                    LeadingString_Single.append(LeadingString[i])
                    ParameterName_Single.append(ParameterName[i])
                    ParameterFormat_Single.append(ParameterFormat[i])
                    TrailingString_Single.append(TrailingString[i])
                    ParamVisible_Single.append(ParamVisible[i])
                    Counter_Single += 1                                                     ## increase counter for group


            ## append group variable to total group variable
            NumberParameterLine_Group_All.append(NumberParameterLine_Group)
            GroupLineFlag_Group_All.append(GroupLineFlag_Group)
            NumberReplicationLine_Group_All.append(NumberReplicationLine_Group)
            GroupParameterFlag_Group_All.append(GroupParameterFlag_Group)
            NumberReplicationParameter_Group_All.append(NumberReplicationParameter_Group)
            LeadingString_Group_All.append(LeadingString_Group)
            ParameterName_Group_All.append(ParameterName_Group)
            ParameterFormat_Group_All.append(ParameterFormat_Group)
            TrailingString_Group_All.append(TrailingString_Group)
            ParamVisible_Group_All.append(ParamVisible_Group)
            Counter_Group_All.append(Counter_Group)
            NumberRepl_Group_All.append(NumberRepl_Group)


            ##********************************************************************************************************************************************
            ## expand the xml-description by the grouped lines
            ##
            ## example:
            ##
            ##     non-grouped line
            ##     grouped line
            ##     grouped line
            ##     grouped line
            ##     non-grouped line
            ##
            ## the grouped lines have to be replicated two times:
            ##
            ##     non-grouped line
            ##
            ##     grouped line
            ##     grouped line
            ##     grouped line
            ##
            ##     grouped line
            ##     grouped line
            ##     grouped line
            ##
            ##     non-grouped line
            ##
            ##********************************************************************************************************************************************
            flag = "false"
            NumberParameterLine = []
            NumberReplicationLine = []
            GroupLineFlag_copy = []
            GroupParameterFlag = []
            NumberReplicationParameter = []
            LeadingString = []
            ParameterName = []
            ParameterFormat = []
            TrailingString = []
            ParamVisible = []
            counter = 0
            Counter_Single = 0
            gi = (-1)                                                                       ## index for each group
            for i in xrange(NumberLines):                                                   ## loop over all lines in the xml-file

                GroupLineFlagAll = GroupLineFlag[i].lower()
                GroupLineFlagElements = GroupLineFlagAll.split(";")                         ## get elements of the group line flag
                CurrentGroupIncluded = "false"

                for elements in GroupLineFlagElements:                                      ## loop over all elements of the group line flag
                    GroupElements = elements.strip()                                        ## remove leading and tailing blanks
                    GroupElements = GroupElements.lower()                                   ## convert string to lower case
                    if (GroupElements.find(groupName) != (-1)):                             ## get index of group


                        ## analyze group element
                        GroupElementsPart = GroupElements.split(",")
                        for GroupPart in GroupElementsPart:                                 ## loop over all command words
                            CurrentGroupIncluded = "true"
                            if (GroupPart.find("start") != (-1)):
                                gi += 1

                                # Debug:
                                # if (GroupIndex == 1): print 'gi = ',gi
                                # print 'NumberRepl_Group_All[gi] = ',NumberRepl_Group_All[gi]
                                # print 'ParameterName_Group_All[gi] = ',ParameterName_Group_All[gi]
                                # print 'GroupLineFlag_Group_All[gi] = ',GroupLineFlag_Group_All[gi]
                                # print 'Counter_Group_All[gi] = ',Counter_Group_All[gi]

                                for NumberReplications in xrange(int(NumberRepl_Group_All[gi])):    ## loop over all replications
                                    for j in xrange(Counter_Group_All[gi]):                 ## loop over all group lines
                                        NumberParameterLine.append(NumberParameterLine_Group_All[gi][j])
                                        NumberReplicationLine.append(NumberReplicationLine_Group_All[gi][j])
                                        GroupLineFlag_copy.append(GroupLineFlag_Group_All[gi][j])
                                        GroupParameterFlag.append(GroupParameterFlag_Group_All[gi][j])
                                        NumberReplicationParameter.append(NumberReplicationParameter_Group_All[gi][j])
                                        LeadingString.append(LeadingString_Group_All[gi][j])
                                        ParameterName.append(ParameterName_Group_All[gi][j])
                                        ParameterFormat.append(ParameterFormat_Group_All[gi][j])
                                        TrailingString.append(TrailingString_Group_All[gi][j])
                                        ParamVisible.append(ParamVisible_Group_All[gi][j])
                                        counter += 1


                if (CurrentGroupIncluded == "false"):
                    NumberParameterLine.append(NumberParameterLine_Single[Counter_Single])
                    NumberReplicationLine.append(NumberReplicationLine_Single[Counter_Single])
                    GroupLineFlag_copy.append(GroupLineFlag_Single[Counter_Single])
                    GroupParameterFlag.append(GroupParameterFlag_Single[Counter_Single])
                    NumberReplicationParameter.append(NumberReplicationParameter_Single[Counter_Single])
                    LeadingString.append(LeadingString_Single[Counter_Single])
                    ParameterName.append(ParameterName_Single[Counter_Single])
                    ParameterFormat.append(ParameterFormat_Single[Counter_Single])
                    TrailingString.append(TrailingString_Single[Counter_Single])
                    ParamVisible.append(ParamVisible_Single[Counter_Single])
                    Counter_Single += 1
                    counter += 1


            ## copy GroupLineFlag_copy back to GroupLineFlag variable
            GroupLineFlag = GroupLineFlag_copy


            ## set NumberLines to new value
            NumberLines = counter


            # Debug:
            # for i in xrange(NumberLines):
            #    print '>>>',ParameterName[i]
            #    print '>',NumberReplicationLine[i]
            #    print '1>',NumberParameterLine[i],NumberReplicationLine[i],GroupLineFlag[i]
            #    print '2>',GroupParameterFlag[i],NumberReplicationParameter[i]
            #    print '3>',LeadingString[i],ParameterName[i],ParameterFormat[i],TrailingString[i]
            #    print '##################################################################',i


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write values of variables to file
        TotalNumberInputLines = 0
        for i in xrange(NumberLines):
            TempFile.write("LineIndex = " + str(i) + "|\n")
            TempFile.write("NumberParameterLine = " + str(NumberParameterLine[i]) + "|\n")
            TotalNumberInputLines += 1
            for j in xrange(NumberParameterLine[i]):
                TempFile.write("ColumnIndex = " + str(j) + "|\n")
                TempFile.write("LeadingString = " + str(LeadingString[i][j]) + "|\n")
                TempFile.write("ParameterName = " + str(ParameterName[i][j]) + "|\n")
                TempFile.write("ParameterFormat = " + str(ParameterFormat[i][j]) + "|\n")
                TempFile.write("TrailingString = " + str(TrailingString[i][j]) + "|\n")
                TempFile.write("ParamVisible = " + str(ParamVisible[i][j]) + "|\n")

		# Debug:
		# print "LeadingString = " + str(LeadingString[i][j]) + "|\n"


        ##--------------------------------------------------------------------------------------------------------------------------------------------
        ## write commands to temp file
        TempFile.write("InputDataPath = " + str(InputDataPath) + "|\n")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check if names of parameter declared in the "GetParameterSet.py" package are declared in the registration file
        for name in FitParameterName:
            NameIncluded = "false"
            for RegistName in ParameterNameList:
                if (string.find(name,RegistName) != (-1)):

                    if (string.find(name,'[[]]') != (-1)):
                        i = len(name) + 1
                    else:
                        i = string.find(name,'[[')
                        if (i == (-1)):
                            i = len(name) + 1
                    if (name[:i] == RegistName):
                        NameIncluded = "true"

            #if (NameIncluded == "false"):
            #    print "\n\n\n\n"
            #    print "\t\t ERROR in subroutine ReadRegistrationFile:"
            #    print "\t\t\t The name of the parameter '%s' " % name
            #    print "\t\t\t can not be found in the registration file."
            #    print "\t\t\t Please modify the parameter file and restart"
            #    print "\t\t\t the whole program."
            #    print "\n\n\n\n"
            #    ok = 1
            #    break


        if (TotalNumberInputLines > MaxInputLines):
            MaxInputLines = TotalNumberInputLines
        if (MaxNumberParameter > MaxParameter):
            MaxParameter = MaxNumberParameter


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## close temp file
    TempFile.write(" \n")
    TempFile.write(" \n")
    TempFile.write(" \n")
    TempFile.write(" \n")
    TempFile.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ##  output file description
    ##
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get AllInOneOutputFile flag
    try:
        AllInOneOutputFile = XmlFile.InOutputFile.AllInOneOutputFile.getValue()
    except AttributeError, err:
        try:
            AllInOneOutputFile = XmlFile.InOutputFile.Section.AllInOneOutputFile.getValue()
        except AttributeError, err:
            AllInOneOutputFile = "no"
    AllInOneOutputFile = AllInOneOutputFile.lower()
    AllInOneOutputFile = AllInOneOutputFile.strip()
    if (AllInOneOutputFile == "yes" or AllInOneOutputFile == "y"):
        AllInOneOutputFile = "yes"
    elif (AllInOneOutputFile == "no" or AllInOneOutputFile == "none" or AllInOneOutputFile == "n"):
        AllInOneOutputFile = "no"
    else:
        print "\n\n\n\n"
        print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
        print "\t\t\t The tag <AllInOneOutputFile></AllInOneOutputFile> has to be set to"
        print "\t\t\t yes, y, no, none or n."
        print "\n"
        print "\t\t\t Please correct the registration xml-file and restart MAGIX."
        print "\n\n\n\n"
        ok = 1
        return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
               MaxInputLines, MaxParameter


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get path and file name of the output file
    NumOutputFilesEqualNumExpFiles = "false"
    if (AllInOneOutputFile == "yes"):
        NumberOutputFiles = 1
    else:
        try:
            NumberOutputFiles = str(XmlFile.InOutputFile.NumberOutputFiles.getValue())
        except AttributeError, err:
            try:
                NumberOutputFiles = str(XmlFile.InOutputFile.Section.NumberOutputFiles.getValue())
            except AttributeError, err:
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
                print "\t\t\t An error occurs while reading the tag <NumberOutputFiles></NumberOutputFiles> from the registration xml-file."
                print "\t\t\t Please check the registration xml-file and restart MAGIX."
                print "\n\n\n\n"
                ok = 1
                return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
                       MaxInputLines, MaxParameter
    if (NumberOutputFiles == "MAGIXImportNumberExpFiles"):
        NumberOutputFiles = NumberExpFiles
        NumOutputFilesEqualNumExpFiles = "true"
    else:
        NumberOutputFiles = int(NumberOutputFiles)
    if (NumberOutputFiles < 1):
        print "\n\n\n\n"
        print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
        print "\t\t\t The defined number of output-files given in the registration"
        print "\t\t\t file is < 1!"
        print "\n"
        print "\t\t\t Please correct the registration xml-file and restart MAGIX."
        print "\n\n\n\n"
        ok = 1
        return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
               MaxInputLines, MaxParameter


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get path and file name of the output file(s)
    print_inter_warning = "false"
    #if (NumOutputFilesEqualNumExpFiles == "true"):                                          ## if the number of exp. data files is set to
    #    ii = 1                                                                              ## MAGIXImportNumberExpFiles, the loop below has only 1 cycle
    #else:
    #    ii = NumberOutputFiles
    for OutputFileNumberLoop in xrange(NumberOutputFiles):
        if (NumOutputFilesEqualNumExpFiles == "true"):
            OutputFileNumber = 1
        else:
            OutputFileNumber = OutputFileNumberLoop


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine string for reading tag from xml-file
        try:
            XmlString = "XmlFile.InOutputFile.OutputFile"                                   ## define xml-string
            if (NumberOutputFiles > 1 and NumOutputFilesEqualNumExpFiles == "false"):       ## expand xml-string if there are more than 1 input
                XmlString = XmlString + "[" + str(OutputFileNumber) + "]"                   ## file specified
            exec('testOut = str(' + XmlString + '.OutputFileName.getValue())')
        except AttributeError, err:
            XmlString = "XmlFile.InOutputFile.Section.OutputFile"                           ## define xml-string
            if (NumberOutputFiles > 1 and NumOutputFilesEqualNumExpFiles == "false"):       ## expand xml-string if there are more than 1 input
                XmlString = XmlString + "[" + str(OutputFileNumber) + "]"                   ## file specified


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define file name for registration ascii-file
        ExtentionFileName = JobDir.strip()
        ExtentionFileName = ExtentionFileName + "/Registration_Output_" + str(OutputFileNumberLoop + 1) + ".txt"
        TempFile = open(ExtentionFileName, 'w')                                             ## open temp-file for each output file


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get path and file name of the output file
        exec('OutputFile = str(' + XmlString + '.OutputFileName.getValue())')
        TempFile.write("OutputFileName = " + OutputFile + "|\n")                            ## write name of output file to temp-file


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get OnlyYColumn flag
        try:
            exec('OnlyYColumn = str(' + XmlString + '.OnlyYColumn.getValue())')
            OnlyYColumn = OnlyYColumn.lower()
            OnlyYColumn = OnlyYColumn.strip()
        except AttributeError, err:
            OnlyYColumn = "no"


        ## check if user input has the correct form
        if (OnlyYColumn == "yes" or OnlyYColumn == "y"):
            OnlyYColumn = "yes"
        elif (OnlyYColumn == "no" or OnlyYColumn == "none" or OnlyYColumn == "n"):
            OnlyYColumn = "no"
        else:
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
            print "\t\t\t The tag <OnlyYColumn></OnlyYColumn> of output-file %i has to be" % (OutputFileNumber + 1)
            print "\t\t\t set to yes, y, no, none or n."
            print "\n"
            print "\t\t\t Please correct the registration xml-file and restart MAGIX."
            print "\n\n\n\n"
            TempFile.close()
            ok = 1
            return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
                   MaxInputLines, MaxParameter


        ## write value of tag <OnlyYColumn></OnlyYColumn> to temp file
        TempFile.write("OnlyYColumn = " + OnlyYColumn + "|\n")
        TempFile.write("AllInOneOutputFile = " + AllInOneOutputFile + "|\n")
        TempFile.write("NumOutputFilesEqualNumExpFiles = " + NumOutputFilesEqualNumExpFiles + "|\n")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## old statement (NO LONGER USED): if calculation method is point to point
        #   if (CalculationMethod.lower() == "pointtopoint"):


        #    ##-------------------------------------------------------------------------------------------------------------------------------------------
        #    ## read number of lines
        #    NumberLinesOutput = 0
        #    exec('NumberLinesOutput = str(' + XmlString + '.NumberLines.getValue())')
        #    if (NumberLinesOutput == " "):
        #        NumberLinesOutput = 0


        #    ##-------------------------------------------------------------------------------------------------------------------------------------------
        #    ## read number of columns
        #    NumberColumnsBegin = 0
        #    exec('NumberColumnsBegin = str(' + XmlString + '.NumberColumnsBegin.getValue())')
        #    if (NumberColumnsBegin == " "):
        #        NumberColumnsBegin = 0


        #    ##-------------------------------------------------------------------------------------------------------------------------------------------
        #    ## read number of columns
        #    NumberColumnsEnd = 0
        #    exec('NumberColumnsEnd = str(' + XmlString + '.NumberColumnsEnd.getValue())')
        #    if (NumberColumnsBegin == " "):
        #        NumberColumnsBegin = 0


        #    ##-------------------------------------------------------------------------------------------------------------------------------------------
        #    ## read number of columns
        #    CommandWordOutput = 0
        #    exec('CommandWordOutput = str(' + XmlString + '.CommandWord.getValue())')


        #    ##-------------------------------------------------------------------------------------------------------------------------------------------
        #    ## write commands to temp file
        #    TempFile.write("NumberLinesOutput = " + str(NumberLinesOutput) + "|\n")
        #    TempFile.write("NumberColumnsBegin = " + str(NumberColumnsBegin) + "|\n")
        #    TempFile.write("NumberColumnsEnd = " + str(NumberColumnsEnd) + "|\n")
        #    TempFile.write("CommandWordOutput = " + CommandWordOutput + "|\n")


        # elif (CalculationMethod.lower() == "atonce"):


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get output file format flag
        try:
            exec('OutputFileFormat = str(' + XmlString + '.OutputFileFormat.getValue())')   ## get format of the output file
            OutputFileFormat = OutputFileFormat.lower()
            OutputFileFormat = OutputFileFormat.strip()
        except AttributeError, err:
            OutputFileFormat = "ascii"
        if (OutputFileFormat != "ascii" and OutputFileFormat != "dat" and OutputFileFormat != "fits" and OutputFileFormat != "automatic" and \
            OutputFileFormat != "auto"):
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
            print "\t\t\t The user-defined format of output-file %i has to be" % OutputFileNumber
            print "\t\t\t ascii, dat, fits, automatic or auto."
            print "\n"
            print "\t\t\t Please correct the registration xml-file and restart MAGIX."
            print "\n\n\n\n"
            TempFile.close()
            ok = 1
            return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
                   MaxInputLines, MaxParameter
        if (OutputFileFormat == "automatic" or OutputFileFormat == "auto"):
            if (OutputFile.endswith(".fits")):
                OutputFileFormat = "fits"
            else:
                OutputFileFormat = "ascii"
        if (OutputFileFormat == "ascii" or OutputFileFormat == "dat"):
            OutputFileFormatNum = 1
        elif (OutputFileFormat == "fits"):
            OutputFileFormatNum = 2
        TempFile.write("OutputFileFormat = " + str(OutputFileFormatNum) + "|\n")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get number of header lines
        if (OutputFileFormat == "ascii"):
            try:
                exec('NumberHeaderLines = str(' + XmlString + '.NumberHeaderLines.getValue())')
                if (NumberHeaderLines == ""):
                    NumberHeaderLines = "0"
            except AttributeError, err:
                NumberHeaderLines = 0
            TempFile.write("NumberHeaderLines = " + NumberHeaderLines + "|\n")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get character indicating comments
        if (OutputFileFormat == "ascii"):
            try:
                exec('CharacterForComments = str(' + XmlString + '.CharacterForComments.getValue())')
            except AttributeError, err:
                CharacterForComments = ' '
            TempFile.write("CharacterForComments = " + CharacterForComments + "|\n")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get character seperating columns
        ##if (OutputFileFormat == "ascii"):
        ##    exec('CharacterSeperatingColumns = str(' + XmlString + '.CharacterSeperatingColumns.getValue())')
        ##    TempFile.write("CharacterSeperatingColumns = " + CharacterSeperatingColumns + "|\n")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## the following user settings for resampling is read only if OnlyYColumn is set to "no"
        if (string.find(OnlyYColumn,'n') != (-1)):


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## resampling method
            try:
                exec('ResamplingMethod = str(' + XmlString + '.ResamplingMethod.getValue())')


                ## if AllInOneOutputFile is set to yes OnlyYColumn has to be set to yes as well
                if (AllInOneOutputFile == "yes" and OnlyYColumn == "no" and print_inter_warning == "false"):
                    print " "
                    print "\t WARNING: in subroutine FittingEngine.ReadRegistrationFile!"
                    print "\t\t\t The tag <OnlyYColumn></OnlyYColumn> of output-file %i is set to no " % (OutputFileNumber + 1)
                    print "\t\t\t and the tag <AllInOneOutputFile></AllInOneOutputFile> is set to yes."
                    print " "
                    print "\t\t\t NO INTERPOLATION IS POSSIBLE for this configuration!"
                    print " "
                    print_inter_warning = "true"

            except AttributeError, err:
                ResamplingMethod = "simple"
            TempFile.write("ResamplingMethod = " + ResamplingMethod + "|\n")


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## LSR-adjustement value
            try:
                exec('LSRAdjustement = str(' + XmlString + '.LSRAdjustement.getValue())')
                if (LSRAdjustement == ""):
                    LSRAdjustement = "0"


                ## if AllInOneOutputFile is set to yes OnlyYColumn has to be set to yes as well
                if (AllInOneOutputFile == "yes" and OnlyYColumn == "no" and print_inter_warning == "false"):
                    print " "
                    print "\t WARNING: in subroutine FittingEngine.ReadRegistrationFile!"
                    print "\t\t\t The tag <OnlyYColumn></OnlyYColumn> of output-file %i is set to no " % (OutputFileNumber + 1)
                    print "\t\t\t and the tag <AllInOneOutputFile></AllInOneOutputFile> is set to yes."
                    print " "
                    print "\t\t\t NO INTERPOLATION IS POSSIBLE for this configuration!"
                    print " "
                    print_inter_warning = "true"

            except AttributeError, err:
                LSRAdjustement = "0"
            TempFile.write("LSRAdjustement = " + LSRAdjustement + "|\n")


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## fit flag for LSR-adjustement value
            try:
                exec('LSRAdjustementFlag = str(' + XmlString + '.LSRAdjustementFitFlag.getValue())')
                LSRAdjustementFlag = LSRAdjustementFlag.lower()
                LSRAdjustementFlag = LSRAdjustementFlag.strip()


                ## if AllInOneOutputFile is set to yes OnlyYColumn has to be set to yes as well
                if (AllInOneOutputFile == "yes" and OnlyYColumn == "no" and print_inter_warning == "false"):
                    print " "
                    print "\t WARNING: in subroutine FittingEngine.ReadRegistrationFile!"
                    print "\t\t\t The tag <OnlyYColumn></OnlyYColumn> of output-file %i is set to no " % (OutputFileNumber + 1)
                    print "\t\t\t and the tag <AllInOneOutputFile></AllInOneOutputFile> is set to yes."
                    print " "
                    print "\t\t\t NO INTERPOLATION IS POSSIBLE for this configuration!"
                    print " "
                    print_inter_warning = "true"

            except AttributeError, err:
                LSRAdjustementFlag = "no"
            if (LSRAdjustementFlag != "yes" and LSRAdjustementFlag != "y" and LSRAdjustementFlag != "no" and LSRAdjustementFlag != "none" \
                and LSRAdjustementFlag != "n"):
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
                print "\t\t\t The flag for LSRAdjustementFlag of output-file %i has to be" % (OutputFileNumber + 1)
                print "\t\t\t set to yes, y, no, none or n."
                print "\n"
                print "\t\t\t Please correct the registration xml-file and restart MAGIX."
                print "\n\n\n\n"
                TempFile.close()
                ok = 1
                return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
                       MaxInputLines, MaxParameter
            TempFile.write("LSRAdjustementFlag = " + LSRAdjustementFlag + "|\n")



            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## interpolation method
            try:
                exec('InterpolationMethod = str(' + XmlString + '.InterpolationMethod.getValue())')
                InterpolationMethod = InterpolationMethod.lower()
                InterpolationMethod = InterpolationMethod.strip()


                ## if AllInOneOutputFile is set to yes OnlyYColumn has to be set to yes as well
                if (AllInOneOutputFile == "yes" and OnlyYColumn == "no" and print_inter_warning == "false"):
                    print " "
                    print "\t WARNING: in subroutine FittingEngine.ReadRegistrationFile!"
                    print "\t\t\t The tag <OnlyYColumn></OnlyYColumn> of output-file %i is set to no " % (OutputFileNumber + 1)
                    print "\t\t\t and the tag <AllInOneOutputFile></AllInOneOutputFile> is set to yes."
                    print " "
                    print "\t\t\t NO INTERPOLATION IS POSSIBLE for this configuration!"
                    print " "
                    print_inter_warning = "true"

            except AttributeError, err:
                InterpolationMethod = "nearest-neighbour"
            if (InterpolationMethod != "nearest-neighbour" and InterpolationMethod != "linear"):
                InterpolationMethod = "nearest-neighbour"
            TempFile.write("InterpolationMethod = " + InterpolationMethod + "|\n")


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## Normalization flag
            try:
                exec('NormalizationFlag = str(' + XmlString + '.NormalizationFlag.getValue())')
                NormalizationFlag = NormalizationFlag.lower()
                NormalizationFlag = NormalizationFlag.strip()


                ## if AllInOneOutputFile is set to yes OnlyYColumn has to be set to yes as well
                if (AllInOneOutputFile == "yes" and OnlyYColumn == "no" and print_inter_warning == "false"):
                    print " "
                    print "\t WARNING: in subroutine FittingEngine.ReadRegistrationFile!"
                    print "\t\t\t The tag <OnlyYColumn></OnlyYColumn> of output-file %i is set to no " % (OutputFileNumber + 1)
                    print "\t\t\t and the tag <AllInOneOutputFile></AllInOneOutputFile> is set to yes."
                    print " "
                    print "\t\t\t NO INTERPOLATION IS POSSIBLE for this configuration!"
                    print " "
                    print_inter_warning = "true"

            except AttributeError, err:
                NormalizationFlag = "yes"
            if (NormalizationFlag != "yes" and NormalizationFlag != "y" and NormalizationFlag != "no" and NormalizationFlag != "none" \
                and NormalizationFlag != "n"):
                print "\n\n\n\n"
                print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
                print "\t\t\t The tag <NormalizationFlag></NormalizationFlag> of output-file %i has to be" % (OutputFileNumber + 1)
                print "\t\t\t set to yes, y, no, none or n."
                print "\n"
                print "\t\t\t Please correct the registration xml-file and restart MAGIX."
                print "\n\n\n\n"
                TempFile.close()
                ok = 1
                return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
                       MaxInputLines, MaxParameter
            TempFile.write("NormalizationFlag = " + NormalizationFlag + "|\n")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get values of model function if output file is empty
    try:
        exec('ValueEmptyOutputFile = str(' + XmlString + '.ValueEmptyOutputFile.getValue())')
        if (ModelValueForEmptyOutputFile == ""):
            ModelValueForEmptyOutputFile = "0"


        ## check if content of the tag is a number
        try:
            float(ModelValueForEmptyOutputFile)
        except ValueError:
            print "\n\n\n\n"
            print "\t\t ERROR in subroutine FittingEngine.ReadRegistrationFile!"
            print "\t\t\t The tag <ValueEmptyOutputFile></ValueEmptyOutputFile> of output-file %i has to be a number." % (OutputFileNumber + 1)
            print "\n"
            print "\t\t\t Please correct the registration xml-file and restart MAGIX."
            print "\n\n\n\n"
            TempFile.close()
            ok = 1
            return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
                   MaxInputLines, MaxParameter
    except AttributeError, err:
        ModelValueForEmptyOutputFile = "0"
    TempFile.write("ModelValueForEmptyOutputFiles = " + ModelValueForEmptyOutputFile + "|\n")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## close temp file
    TempFile.write(" \n")
    TempFile.write(" \n")
    TempFile.write(" \n")
    TempFile.write(" \n")
    TempFile.close()


    ## print: we are finished
    if (print_flag != "false"):
        print "done!"
        print " "


    ## define return variables
    return ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, \
           MaxInputLines, MaxParameter
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
def NumberQ(string):
    ## is string a number
    """

    function NumberQ:       function returns true if string is a value, otherwise false

    input parameters:       string:                     string to analyze

    output parameters:      value:                      contains true or false

    working parameters:     point_counter:              count number of points in string
                            minus_counter:              count number of minus characters in string
                            e_counter:                  count number of 'e' characters in string

    """

    pointcounter = 0
    minuscounter = 0
    ecounter = 0
    pluscounter = 0
    minuscounter = 0
    value = "true"
    for element in str(string):
        if (ord(element) != 32):															## space bar is not considered
            if (ord(element) == 45):													    ## count minus signs
                minuscounter += 1
            elif (ord(element) == 46):									                    ## count points
                if (len(string.strip())>1):
                    pointcounter += 1
                else:
                    value = "false"
                    break
            elif (element.lower() == 'e'):							                        ## count exponent "E"
                if (len(string.strip())>1):
                    ecounter += 1
                    pluscounter = 0
                    minuscounter = 0
                else:
                    value = "false"
                    break
            elif (element.lower() == '+'):
                if (len(string.strip())>1):
                    pluscounter += 1
                else:
                    value = "false"
                    break
            elif (element.lower() == '-'):
                if (len(string.strip())>1):
                    minuscounter += 1
                else:
                    value = "false"
                    break
            elif (ord(element)<48) or (ord(element)>57):
                value = "false"
                break
            if (pointcounter>1) or (minuscounter>1) or (ecounter>1) or (pluscounter>1) or (minuscounter>1):
                value = "false"
                break
    return value
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
def LoadScratch(JobID):
    ## write results of the fit process to files


    ## define path and file name of scratch files
    MAGIXTempDirectory = str(os.environ.get('MAGIXTempDirectory',''))
    if (MAGIXTempDirectory.strip()[-1] == "/"):
        filename = MAGIXTempDirectory.strip() + "job_" + str(JobID) + "/"
    else:
        filename = MAGIXTempDirectory.strip() + "/job_" + str(JobID) + "/"


    ## read contents of scratch file
    scratchFuncFile = open(filename + "FitFunction.dat")
    scratchFitFunction = scratchFuncFile.readlines()
    scratchFuncFile.close()

    scratchChiFile = open(filename + "Chi2Values.dat")
    scratchChi2Values = scratchChiFile.readlines()
    scratchChiFile.close()

    #scratchFitFunction = numpy.loadtxt(filename + "FitFunction.dat")
    #scratchChi2Values = numpy.loadtxt(filename + "Chi2Values.dat")

    siteCounter = int(scratchFitFunction[0])                                                ## read number of sites
    NumFile = int(scratchFitFunction[1])                                                    ## read number exp files
    MaxL = int(scratchFitFunction[2])                                                       ## read max. number of lines
    MaxCol = int(scratchFitFunction[3])                                                     ## read max. number of y-columns

    # Debug:
    # print
    # print 'siteCounter = ',siteCounter
    # print 'NumFile = ',NumFile
    # print 'MaxL = ',MaxL
    # print 'MaxCol = ',MaxCol
    # print 'scratchFitFunction = ',scratchFitFunction
    # print 'scratchChi2Values = ',scratchChi2Values,  len(scratchChi2Values)

    ## read from scratch file
    FitFunctionAll = []
    Chi2ValuesAll = []
    LineInFileCounter = 3
    for counter in xrange(siteCounter):
        SitesFit = []
        SitesChi = []
        for numfile in xrange(NumFile):
            FileFit = []
            FileChi = []
            for lines in xrange(MaxL):
                LineInFileCounter += 1
                if (MaxCol == 1):
                    LineInFileFit = [float(scratchFitFunction[LineInFileCounter])]
                    try:
                        LineInFileChi = [float(scratchChi2Values[LineInFileCounter - 4])]
                    except IndexError:
                        LineInFileChi = [float(scratchChi2Values)]
                else:


                    ## get model function values if number of y-columns (MaxCol) is greater than 1
                    splittedline = scratchFitFunction[LineInFileCounter].split()
                    if (splittedline != []):
                        line = []
                        for columns in splittedline:
                            if (columns != ""):
                                line.append(float(columns))
                        LineInFileFit = line

                        # Debug:
                        # print 'LineInFileCounter = ', LineInFileCounter
                        # print 'scratchFitFunction = ', scratchFitFunction
                        # print 'scratchFitFunction[8] = ', scratchFitFunction[8]
                        # print 'LineInFileFit = ', LineInFileFit
                        # sys.exit(0)


                    ## get chi^2 value for each data point if number of y-columns (MaxCol) is greater than 1
                    try:
                        LineInFileChi = scratchChi2Values[LineInFileCounter - 4]
                    except IndexError:
                        LineInFileChi = float(scratchChi2Values)
                    splittedline = LineInFileChi.split()
                    if (splittedline != []):
                        line = []
                        for columns in splittedline:
                            if (columns != ""):
                                line.append(float(columns))
                        LineInFileChi = line


                ## construct list with all model function values and chi^2 values
                FileFit.append(LineInFileFit)
                FileChi.append(LineInFileChi)

            SitesFit.append(FileFit)
            SitesChi.append(FileChi)

        FitFunctionAll.append(SitesFit)
        Chi2ValuesAll.append(SitesChi)

    return FitFunctionAll, Chi2ValuesAll
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
def StoreResults(ExpInputFormat, ClassLoadExp, printflag, NumberExpFiles, LengthExpRange, NumberYColumns, ExpDataX, FitFunctionValues, Chi2SaveFlag, \
                 Chi2Values, XMLFile, NumberParameter, parameters, PreviousParameterSet, ParameterValuesOriginal, FitParameterName, \
                 ExpFilenameExtension, XMLFilenameExtension, UnmodifiedNumberParameter):
    ## write results of the fit process to files
    """

    function StoreResults:  function writes results of a fit algorithm(s) to xml-file


    input parameters:       ExpInputFormat:             format for observation files
                            ClassLoadExp:               class containing relevant subroutines for saving
                            printflag:                  flag for screen output
                            NumberExpFiles:             number of files containing the observation files
                            LengthExpRange:             number of ranges
                            NumberYColumns:             number of columns belonging to the y-column
                            ExpDataX:                   array containing the columns belonging to the x-column in the obseravation data files
                            FitFunctionValues:          array containing the values of the fit function at each point in the observation data files
                            Chi2SaveFlag:               flag for saving chi**2
                            Chi2Values:                 values of chi**2 at each point in the observation data files
                            XMLFile:                    path and file name of the paraFitFunctionAll, Chi2ValuesAllmeter file
                            NumberParameter:            number of parameters
                            parameters:                 optimized set of parameter
                            PreviousParameterSet:       optimized set of parameter
                            FitParameterName:           name of parameter file
                            ExpFilenameExtension:       extension of the observation file
                            XMLFilenameExtension:       extension of the parameter xml-file
                            UnmodifiedNumberParameter:  number of parameters without MAGIX import parameters

    output parameters:      ok:                         status of subroutines (/=0: an error occured)

    """

    # Debug:
    # print "\nNumberParameter = ", NumberParameter


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## set string values of parameters
    paramValue = []
    paramLow = []
    paramHigh = []
    for i in xrange(NumberParameter):


        ## insert original parameter values if parameter value is a string!
        if (NumberQ(str(ParameterValuesOriginal[i])) == "false" or str(ParameterValuesOriginal[i]).strip() == ""):
            paramValue.append(ParameterValuesOriginal[i])
            paramLow.append(PreviousParameterSet[2][i])
            paramHigh.append(PreviousParameterSet[3][i])
        else:
            paramValue.append(parameters[0][i])
            paramLow.append(parameters[2][i])
            paramHigh.append(parameters[3][i])
    FitParameterNameFinal = FitParameterName

    # Debug:
    # print 'NumberParameter = ', NumberParameter
    # print 'UnmodifiedNumberParameter = ', UnmodifiedNumberParameter
    # print 'ParameterValuesOriginal = ', ParameterValuesOriginal


    ## remove MAGIX import parameters from parameter list
    NumberParameter = UnmodifiedNumberParameter
    p0 = paramValue
    p1 = PreviousParameterSet[1]
    p2 = parameters[2]
    p3 = parameters[3]
    p4 = parameters[1]
    parametersFinal = []
    parametersFinal.append(p0[:NumberParameter])
    parametersFinal.append(p1[:NumberParameter])
    parametersFinal.append(p2[:NumberParameter])
    parametersFinal.append(p3[:NumberParameter])
    parametersFinal.append(p4[:NumberParameter])


    ## if error estimator algorithm is used append one line more
    if (len(parameters) == 5):

        # Debug:
        # print 'parameters = ',parameters
        # print 'parameters[4][:NumberParameter] = ',parameters[4][:NumberParameter]

        parametersFinal.append(parameters[4][:NumberParameter])


        ## remove the last line (parameters[4]) from array parameters
        parametersnew = []
        for i in xrange(4):
            parametersnew.append(parameters[i])
        parameters = parametersnew
        # parameters.remove(parameters[4])                                                    ## remove list of error values from parameters

        # Debug:
        # print 'parameters = ',parameters

    FitParameterNameFinal = FitParameterName[:NumberParameter]

    # Debug:
    # print 'FitParameterName = ',FitParameterName
    # print 'FitParameterNameFinal = ',FitParameterNameFinal
    # print 'PreviousParameterSet = ',PreviousParameterSet[1]
    # print 'FitParameterNameFinal = ',FitParameterNameFinal


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write values of the fit function to file
    if (ExpFilenameExtension.strip() != ""):
        ok = ClassLoadExp.WriteFile(ExpInputFormat, printflag, NumberExpFiles, LengthExpRange, NumberYColumns, ExpDataX, FitFunctionValues, \
                                    Chi2SaveFlag, Chi2Values, ExpFilenameExtension)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write optimized model parameter to xml-file
    XmlParameterFileOut = XMLFile.replace(".xml", "." + XMLFilenameExtension + ".xml")      ## define name of output xml-file
    shutil.copy2(XMLFile, XmlParameterFileOut)                                              ## duplicate input xml-file
    ok = GetParameterSet.WriteParameter(printflag, XmlParameterFileOut, NumberParameter, parametersFinal, FitParameterNameFinal)
    if (ok != 0):
        return ok
    if (printflag != "false"):
        print
        print
    return ok
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
def StartAlgPackage(NameOfAlgorithm, StarterExecutable, MPIFlag, MAGIXrootDirectory, JobDir, JobID, NumberExpFiles, MaxLength, MaxColX, MaxColY, \
                    MaxRangeNumber, NumberParameter, ParamSetCounter, LastAlgorithmNum, numiter, ParallelizationFlag, DeterminationChi2, PlotIteration, \
                    PlotType, RenormalizedChi2, AlgorithmCounter, chilm, GeneralAlgorithmSettings, SpecialAlgorithmSettings, parameter_set, xAxisLabel, \
                    yAxisLabel, zAxisLabel):
    ## start algorithm package from command line
    """

    function start:         starts a selected algorithm package from the command line

    input parameters:       NameOfAlgorithm:            name of algorithm
                            StarterExecutable:          name of starter executable
                            MPIFlag:                    flag for indicating MPI execution
                            MAGIXrootDirectory:         MAGIX root directory
                            JobDir:                     current job directory
                            JobID:                      job ID
                            NumberExpFiles:             number of exp. data files
                            MaxLength:                  max. number of data points per data file
                            MaxColX:                    max. number of x-columns
                            MaxColY:                    max. number of y-columns
                            MaxRangeNumber:             max. number of data ranges
                            NumberParameter:            total number of parameters (fixed and free)
                            ParamSetCounter:            number of sites
                            LastAlgorithmNum:           p
                            numiter:                    max. number of iterations
                            ParallelizationFlag:        number of cores
                            DeterminationChi2:          method for determine chi2 value
                            PlotIteration:              flag for plot for each iteration
                            PlotType:                   kind of plot (not used)
                            RenormalizedChi2:           flag for using renormalized chi2 value
                            AlgorithmCounter:           counter for algorithm call
                            chilm:                      lower limit of chi2 value
                            GeneralAlgorithmSettings:   user defined settings for algorithm
                            parameter_set:              parameter set with lower and upper bounds
                            xAxisLabel:                 x-axis label for plotting
                            yAxisLabel:                 y-axis label for plotting
                            zAxisLabel:                 z-axis label for plotting

    output paramters:       calstatus:                  status of the fit process
                            FitFunctionOut:             model function values for each data point and site
                            Chi2Values:                 chi2 values for each data point and site
                            parmCopy:                   copy of input parameter vector
                            FinalParameterSet:          optimized parameter set for all sites
    """


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write some specific algorithm variables to temp file
    VarAlgInputFileName = JobDir + "/specific-alg-variables__input.dat"
    VarAlgInputFile = open(os.path.normpath(VarAlgInputFileName), 'w')


    ## general INTEGER variables
    VarAlgInputFile.write("%30i\n" % LastAlgorithmNum)                                      ## FORTRAN name: LastAlgorithmNum
    VarAlgInputFile.write("%30i\n" % numiter)                                               ## FORTRAN name: numiter
    VarAlgInputFile.write("%30i\n" % ParallelizationFlag)                                   ## FORTRAN name: ParallelizationFlagorg
    VarAlgInputFile.write("%30i\n" % DeterminationChi2)                                     ## FORTRAN name: DeterminationChi2
    VarAlgInputFile.write("%30i\n" % PlotIteration)                                         ## FORTRAN name: DeterminationChi2
    VarAlgInputFile.write("%30i\n" % PlotType)                                              ## FORTRAN name: PlotType
    VarAlgInputFile.write("%30i\n" % RenormalizedChi2)                                      ## FORTRAN name: RenormalizedChi2Org
    VarAlgInputFile.write("%30i\n" % AlgorithmCounter)                                      ## FORTRAN name: AlgorithmCounter


    ## general REAL*8 variables
    VarAlgInputFile.write("%.15e\n" % chilm)                                                ## FORTRAN name: chilm
    for i in xrange(15):
        VarAlgInputFile.write("%.15e  " % GeneralAlgorithmSettings[i])
    VarAlgInputFile.write("\n")
    for i in xrange(NumberParameter):
        VarAlgInputFile.write("%.15e  %.15e  %.15e  %.15e\n" % (float(parameter_set[0][i]), float(parameter_set[1][i]), float(parameter_set[2][i]), \
                                                                float(parameter_set[3][i])))

        #    try:
        #        VarAlgInputFile.write("%.15e  %.15e  %.15e  %.15e\n" % (float(parameter_set[0][i]), float(parameter_set[1][i]), \
        #                                                                float(parameter_set[2][i]), float(parameter_set[3][i])))
        #    except ValueError:
        #        print "parameter_set[0][i] = ", parameter_set[0][i]
        #        print "parameter_set[1][i] = ", parameter_set[1][i]
        #        print "parameter_set[2][i] = ", parameter_set[2][i]
        #        print "parameter_set[3][i] = ", parameter_set[3][i]


    ## general CHARACTER variables
    VarAlgInputFile.write(xAxisLabel.strip() + "\n")
    VarAlgInputFile.write(yAxisLabel.strip() + "\n")
    VarAlgInputFile.write(zAxisLabel.strip() + "\n")


    ## close file
    VarAlgInputFile.close()

    # Debug:
    #    print "(PYTHON): Algorithm specific input variables:"
    #    print "LastAlgorithmNum = ", LastAlgorithmNum
    #    print "numiter = ", numiter
    #    print "ParallelizationFlag = ", ParallelizationFlag
    #    print "DeterminationChi2 = ", DeterminationChi2
    #    print "PlotIteration = ", PlotIteration
    #    print "PlotType = ", PlotType
    #    print "RenormalizedChi2 = ", RenormalizedChi2
    #    print "AlgorithmCounter = ", AlgorithmCounter
    #    print "chilm = ", chilm
    #    print "GeneralAlgorithmSettings[:] = ", GeneralAlgorithmSettings[:]
    #    print "parameter_set[:][0] = ", parameter_set[:][0]
    #    print "parameter_set[:][-1] = ", parameter_set[:][-1]
    #    print "xAxisLabel = >", xAxisLabel, "<"
    #    print "yAxisLabel = >", yAxisLabel, "<"
    #    print "zAxisLabel = >", zAxisLabel, "<"


    ## define path to algorithm executable file
    BaseDir = MAGIXrootDirectory.strip() + "/Modules/"
    BaseDir += NameOfAlgorithm
    BaseDir += "/bin/"
    BaseDir = os.path.normpath(BaseDir) + "/"

    # Debug:
    #    print "BaseDir = ", BaseDir
    #    print "MPIFlag = ", MPIFlag


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create command line call for starting algorithm package
    LocalParamSetCounter = abs(ParamSetCounter)
    if (MPIFlag == "true"):
        MPIStarterExecutable = NameOfAlgorithm + "__MPI-" + StarterExecutable
        FuncCallExt = "chmod +x " + BaseDir + MPIStarterExecutable + "; "
        FuncCallExt += "mpirun -n " + str(int(ParallelizationFlag)) + " -hostfile " + JobDir + "/hostfile_mpi.txt"
        FuncCallExt += " " + BaseDir + MPIStarterExecutable + " " + str(NumberExpFiles) + " " + str(MaxLength) + " " + str(MaxColX) + " " + str(MaxColY) \
                                                            + " " + str(MaxRangeNumber) + " " + str(NumberParameter) + " " + str(ParamSetCounter) \
                                                            + " " + JobID
    else:
        SMPStarterExecutable = NameOfAlgorithm + "__SMP-" + StarterExecutable
        FuncCallExt = "chmod +x " + BaseDir + SMPStarterExecutable + "; "
        FuncCallExt += "" + BaseDir + SMPStarterExecutable + " " + str(NumberExpFiles) + " " + str(MaxLength) + " " + str(MaxColX) + " " + str(MaxColY) \
                                                           + " " + str(MaxRangeNumber) + " " + str(NumberParameter) + " " + str(ParamSetCounter) \
                                                           + " " + JobID

    # Debug:
    # print "FuncCallExt = ", FuncCallExt


    ## start algorithm package via command line
    os.system(FuncCallExt)


    ## prepare output variables
    FitFunctionOut = numpy.zeros((LocalParamSetCounter, NumberExpFiles, MaxLength, MaxColY))
    Chi2Values = numpy.zeros((LocalParamSetCounter, NumberExpFiles, MaxLength, MaxColY))


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read output variables from temp file
    calstatus = 0
    FinalParameterSet = []
    parmCopy = []
    ResultFilesName = JobDir + "/result-variables__output.dat"
    try:
        ResultFiles = open(os.path.normpath(ResultFilesName), 'r')
    except IOError:
        print "\n\n\n\n"
        print "Error in function StartAlgPackage:" 
        print "\tFile ", ResultFilesName, " not found!"
        print "\n\n\n\n"
        for j in xrange(LocalParamSetCounter):
            FinalParameterSetPart = []
            for i in xrange(NumberParameter):
                FinalParameterSetPart.append(parameter_set[0][i])
            FinalParameterSet.append(FinalParameterSetPart)
        return (calstatus, FitFunctionOut, Chi2Values, parmCopy, FinalParameterSet)
    calstatus = int(ResultFiles.readline())
    for i in xrange(LocalParamSetCounter):


        ## get FinalParameterSet
        FinalParameterSetPart = []
        for j in xrange(NumberParameter):
            FinalParameterSetPart.append(float(ResultFiles.readline()))
        FinalParameterSet.append(FinalParameterSetPart)


        ## get FitFunctionOut and Chi2Values
        for j in xrange(NumberExpFiles):
            for k in xrange(MaxLength):
                for l in xrange(MaxColY):
                    line = ResultFiles.readline()
                    SplittedLine = line.split()
                    FitFunctionOut[i, j, k, l] = float(SplittedLine[0])
                    Chi2Values[i, j, k, l] = float(SplittedLine[1])
    parmCopyPart1 = []
    parmCopyPart2 = []
    parmCopyPart3 = []
    parmCopyPart4 = []
    for i in xrange(NumberParameter):
        line = ResultFiles.readline()
        SplittedLine = line.split()
        parmCopyPart1.append(float(SplittedLine[0]))
        parmCopyPart2.append(float(SplittedLine[1]))
        parmCopyPart3.append(float(SplittedLine[2]))
        parmCopyPart4.append(float(SplittedLine[3]))
    parmCopy.append([parmCopyPart1, parmCopyPart2, parmCopyPart3, parmCopyPart4])

    # Debug:
    #    print "\n\n"
    #    print "(PYTHON): Result output variables:"
    #    print "calstatus = ", calstatus
    #    print "FinalParameterSet = ", FinalParameterSet
    #    print "FitFunctionOut[0, 0, 0, 0] = ", FitFunctionOut[0, 0, 0, 0]
    #    print "FitFunctionOut[0, 0, -1, 0] = ", FitFunctionOut[0, 0, -1, 0]
    #    print "Chi2Values[0, 0, 0, 0] = ", Chi2Values[0, 0, 0, 0]
    #    print "Chi2Values[0, 0, -1, 0] = ", Chi2Values[0, 0, -1, 0]
    #    print "parmCopy[0][0] = ", parmCopy[0][0]
    #    print "parmCopy[0][-1] = ", parmCopy[0][-1]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define return parameters
    return (calstatus, FitFunctionOut, Chi2Values, parmCopy, FinalParameterSet)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
def start(printflag, plotflag, debugflag, modelflag, JobID, fit_control, fit_log, xml_file, XmlFileInOutput, NumberParameter, parameter_set, \
          FitParameterName, NumberExpFiles, length_exp_file, exp_data_x, exp_data_y, exp_data_error, ClassLoadExp, ExpInputFormat, NumExpRanges, \
          MinExpRange, MaxExpRange, ImportParameterSet, parameter_file_orig, MAGIXrootDirectory):
    ## call of fit algorithms
    """

    function start:         start the fit algorithm given in the fit control file
                            call subroutine ReadFitControlFile() to read fit control file

    input parameters:       printflag:                  print flag
                            plotflag:                   plot flag
                            debugflag:                  flag for debugging call of external model program
                            modelflag:                  flag indicaing the usage of MAGIX packages optimized for a certain model
                            JobID:                      the current job ID number
                            fit_control:                path and file name of the fit control file
                            fit_log:                    path and file name of the log-file
                            xml_file:                   path and file name of the xml-file containing the model paramters
                            XmlFileInOutput:            path and file name of the xml-file containing the description of the input file
                                                        for the fit function module
                            NumberParameter:            total number of model paramters
                            parameter_set:              array containing the values, the fit flags and the upper and lower limit of the
                                                        model parameters
                            FitParameterName:           names of the model parameters
                            NumberExpFiles:               contains the number of experimental data files
                            length_exp_file:            array containing the length of each experimental data file
                            exp_data_x:                 array containing the "x"-column values of the experimental data for each file
                            exp_data_y:                 array containing the "y"-column values of the experimental data for each file
                            exp_data_error:             array containing the error values of the experimental data for each file
                            NumExpRanges:               Number of exp. data ranges for each experimental data file
                            MinExpRange:                The left side of each exp. data range for each each experimental data file
                            MaxExpRange:                The right side of each exp. data range for each each experimental data file
                            ImportParameterSet:         array containing all parameters of the xml-file used by the import of the exp. data
                            parameter_file_orig:        copy of the string including the path and file name of the file containing the 
                                                        fit parameters. This copy is necessary to detect the orginal format of the model
                                                        input file.
                            MAGIXrootDirectory:         MAGIX root directory

    output paramters:       calstatus:                  status of the fit process
                            parameter_set:              optimized set of parameter

    working parameters:     algorithm:                  name of the algorithm which should be used for the fit process
                            model:                      name of model used for the determination of the fit function
                            chilm:                      limit of chi**2
                            numrange:                   number of fit ranges
                            xmn:                        starts of fit ranges
                            xmx:                        ends of fit ranges
                            numiter:                    number of iterations
                            DeterminationChi2:          method of determination of chi^2
                            SaveChi2:                   flag for saving chi^2
                            PlotIteration:              flag for plotting fit function for each iteration step
                            j:                          loop counter
                            i:                          loop counter
                            k:                          loop counter
                            l:                          loop counter
                            line:                       line of the experimental "x" column
                            LengthExpRange:             array containing the length of experimental data in each experimental file
                            ExpDataRange_x:             array containing the "x"-column values of the experimental data within the given
                                                        ranges for each file
                            ExpDataRange_y:             array containing the "y"-column values of the experimental data within the given
                                                        ranges for each file
                            ExpDataRange_Error:         array containing the "error"-column values of the experimental data within the
                                                        given ranges for each file
                            exp_dat_x:                  temp array for x-column values
                            exp_dat_y:                  temp array for y-column values
                            exp_dat_error:              temp array for error-column values
                            lengthexpdata:              temp variable for length of experimental data in the ith experimental file
                            ParameterValuesOriginal:    original values of the parameters
                            copyflag:                   temp variable for data selection
                            copyflagcolumn:             temp variable for data selection
                            currentpath:                path where the MAGIX program is executed
                            paramname:                  temp variable for construction array FitParameterName
    """


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import MAGIX packages ..


    ## .. optimized for myXCLASS model
    if (modelflag == "true" or modelflag.lower() == "myxclass"):
        StarterExecutable = "Starter__myXCLASS.exe"


    ## .. default package
    else:
        StarterExecutable = "Starter.exe"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define list of known algorithms
    KnwonAlgorithms = ["nested-sampling", "levenberg-marquardt", "simulated-annealing", "pso", "bees", "genetic", "mcmc", "ins", "interval-ns", \
                       "additionalpackages", "errorestim", "errorestim_ins"]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import Interval Nested Sampling as well as Error Estimation algorithms if plotflag is not deactivated
    if (plotflag == "true" or plotflag == "saveonly"):
        import MCMC                                                                         ## import package MCMC
        import ErrEstim_INS                                                                 ## load package ErrorEstim_INS algorithm
        import INS_paral                                                                    ## load package interval nested-sampling algorithm


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## print what you do
    if (printflag != "false"):
        print "Start fit process."


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read control file
    ok, UseHeuristic, NumberOfFitAlgorithms, AllAlgorithmSettings = ReadFitControlFile(printflag, plotflag, fit_control, KnwonAlgorithms, \
                                                                                       StarterExecutable, MAGIXrootDirectory)
    if (ok != 0):
        sys.exit(0)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get current directory
    currentpath = os.getcwd()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## set environment variable if these are not defined
    MAGIXTempDirectory = str(os.environ.get('MAGIXTempDirectory',''))
    if (MAGIXTempDirectory.strip() == ""):
        os.environ["MAGIXTempDirectory"] = 'temp/'
        MAGIXTempDirectory = str(os.environ.get('MAGIXTempDirectory','')).strip()


    ## increase size of stack
    OMP_STACKSIZE = str(os.environ.get('OMP_STACKSIZE',''))
    if (OMP_STACKSIZE.strip() == ""):
        os.environ["OMP_STACKSIZE"] = '1999M'


    ## set MAGIX working directory
    MAGIXWorkingDir = str(os.environ.get('MAGIXWorkingDir',''))
    os.environ["MAGIXWorkingDir"] = os.getcwd()
    MAGIXWorkingDir = str(os.environ.get('MAGIXWorkingDir','')).strip()


    ## set time interval for option: PlotIteration
    MAGIXTimePlotIter = str(os.environ.get('MAGIXTimePlotIter',''))
    try:
        MAGIXTimePlotIter = int(MAGIXTimePlotIter)
    except ValueError:
        MAGIXTimePlotIter = '3000'
        os.environ["MAGIXTimePlotIter"] = MAGIXTimePlotIter

    # Debug:
    # print "MAGIXTimePlotIter = ", MAGIXTimePlotIter


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create current job directory
    JobDir = MAGIXTempDirectory.strip()
    if (not JobDir.endswith("/")):
        JobDir = JobDir + "/"
    JobDir = JobDir + "job_" + str(JobID) + "/"                                             ## define current job directory
    command_string = "mkdir -p " + JobDir.strip()                                           ## create current job directory
    os.system(command_string)


    ##====================================================================================================================================================
    ## special modifications of the parameter_set for myXCLASS


    ## append names of MAGIX import parameters to list of parameter names
    for i in xrange(len(ImportParameterSet[0])):
        paramname = ImportParameterSet[0][i]


        ## determine extension
        k = 1
        if (i > 0):
            for j in xrange(i - 1):
                if (ImportParameterSet[0][j] == paramname):
                    k += 1
        FitParameterName.append(paramname + "[[" + str(k) + "]]")

        # Debug:
        # print '>>',i,'    ',ImportParameterSet[0][i], '    ',ImportParameterSet[1][i]


    ## append values of MAGIX import parameters to list of parameter_set
    UnmodifiedNumberParameter = NumberParameter
    p0 = parameter_set[0]                                                                   ## value
    p1 = parameter_set[1]                                                                   ## flag
    p2 = parameter_set[2]                                                                   ## lower limit
    p3 = parameter_set[3]                                                                   ## upper limit
    for i in xrange(len(ImportParameterSet[0])):
        NumberParameter += 1
        p0.append(ImportParameterSet[1][i])
        p1.append('0')
        p2.append(0)
        p3.append(0)


    ## construct final parameter_set array
    parameter_set = []
    parameter_set.append(p0)
    parameter_set.append(p1)
    parameter_set.append(p2)
    parameter_set.append(p3)
    ##====================================================================================================================================================


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write all values of model parameters in a array and remove non-number values from parameter value list
    FitParameterValue = []
    ParameterValuesOriginal = []
    for i in xrange(NumberParameter):
        ParameterValuesOriginal.append(parameter_set[0][i])
        if (str(parameter_set[0][i]) == " " or str(parameter_set[0][i]).strip() == ""):
            FitParameterValue.append(str(parameter_set[0][i]))
            parameter_set[0][i] = 0
        elif (NumberQ(str(parameter_set[0][i])) == "true"):
            FitParameterValue.append("nUMnUMnUMnUM")
        else:
            FitParameterValue.append(str(parameter_set[0][i]))
            parameter_set[0][i] = 0

    # Debug:
    # print 'ParameterValuesOriginal = ', ParameterValuesOriginal


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read registration file
    ok, PathStartScript, ExeCommandStartScript, CalculationMethod, ParallelizationPossible, NumberInputFiles, NumberOutputFiles, MaxInputLines, \
    MaxParameter = ReadRegistrationFile(printflag, debugflag, NumberParameter, parameter_set, FitParameterName, ParameterValuesOriginal, \
                                        XmlFileInOutput, JobDir, NumberExpFiles)
    if (ok != 0):                                                                           ## an error occurred?
        sys.exit(0)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## screen output ?
    if (printflag == "true"):
        printflagNum = 1
    else:
        printflagNum = 0


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## construct arrays containing only the experimental data within the given fit ranges
    AlgorithmSettings = AllAlgorithmSettings[0]
    numrange = AlgorithmSettings[3]                                                         ## .append(numrange)
    xmn = AlgorithmSettings[4]                                                              ## .append(xmn)
    xmx = AlgorithmSettings[5]                                                              ## .append(xmx)

    # Debug:
    #    print "numrange = ", numrange
    #    print "xmn = ", xmn
    #    print "xmx = ", xmx
    #    sys.exit(0)


    ExpDataRange_x = []
    ExpDataRange_y = []
    ExpDataRange_Error = []
    LengthExpRange = []
    NumberXColumns = []
    NumberYColumns = []
    MaxColX = 0
    MaxColY = 0
    MaxLength = 0
    for i in xrange(NumberExpFiles):
        ExpDataFile_x = []
        ExpDataFile_y = []
        ExpDataFile_Error = []
        LengthExpDataFile = 0
        NumberX = len(exp_data_x[i][0])
        NumberY = len(exp_data_y[i][0])
        NumberXColumns.append(NumberX)
        NumberYColumns.append(NumberY)
        for k in xrange(length_exp_file[i]+1):
            xcolumn = exp_data_x[i][k][0:NumberX]
            copyflag = "false"                                                              ## set copyflag to false
            if (numrange > 0):                                                              ## are there ranges defined?
                for j in xrange(numrange):
                    if (NumberX == 1):                                                      ## is there only one column for the x points?
                        if (xcolumn >= xmn[j]):
                            if (xcolumn <=xmx[j]):
                                copyflag = "true"
                    else:                                                                   ## no? Continue?
                        copyflagcolumn = 0
                        for l in xrange(len(xcolumn)):
                            if (xcolumn[l] >= xmn[j][l]):
                                if (xcolumn[l] <=xmx[j][l]):
                                    copyflagcolumn += 1
                        if (copyflagcolumn == NumberX):
                            copyflag = "true"
                    if (copyflag == "true"):                                                ## if experimental x point is within a given range ..
                        LengthExpDataFile += 1                                              ## increase length counter
                        ExpDataFile_x.append(xcolumn)                                       ## append current x point
                        ExpDataFile_y.append(exp_data_y[i][k][0:NumberY])
                        ExpDataFile_Error.append(exp_data_error[i][k][0:NumberY])
                        break
            else:
                LengthExpDataFile = length_exp_file[i] + 1                                  ## get number of x-columns
                ExpDataFile_x.append(exp_data_x[i][k][0:NumberX])                           ## get x-columns
                ExpDataFile_y.append(exp_data_y[i][k][0:NumberY])                           ## get y-columns
                ExpDataFile_Error.append(exp_data_error[i][k][0:NumberY])


        ## is data within the given range(s)?
        if (LengthExpDataFile == 0):
            ExpDataFile_x = [[0]]
            ExpDataFile_y = [[0]]
            ExpDataFile_Error = [[0]]


        ## append experimental data of the ith experimental file to the variables ExpDataRange_x, ExpDataRange_y, ExpDataRange_Error
        LengthExpRange.append(LengthExpDataFile)
        ExpDataRange_x.append(ExpDataFile_x)
        ExpDataRange_y.append(ExpDataFile_y)
        ExpDataRange_Error.append(ExpDataFile_Error)


        ## determine the max. number of x- and y-columns and the max. number of datapoints of a certain range
        if (len(ExpDataFile_x[0]) > MaxColX):
            MaxColX = len(exp_data_x[i][0])
        if (len(ExpDataFile_y[0]) > MaxColY):
            MaxColY = len(exp_data_y[i][0])
        if (LengthExpDataFile > MaxLength):
            MaxLength = LengthExpDataFile

        # Debug:
        #    print "NumberParameter = ",NumberParameter
        #    print "parameter_set = ",parameter_set
        #    print "LengthExpRange = ",LengthExpRange
        #    print "ExpDataRange_x[i][0:5] = ",ExpDataRange_x[i][0:5]
        #    print "ExpDataRange_x[i][LengthExpRange[0]-1] = ",ExpDataRange_x[i][LengthExpRange[0]-1]
        #    print "ExpDataRange_y[i][0:5] = ",ExpDataRange_y[i][0:5]
        #    print "ExpDataRange_y[i][LengthExpRange[0]-1] = ",ExpDataRange_y[i][LengthExpRange[0]-1]
        #    print "ExpDataRange_Error[i][0:5] = ",ExpDataRange_Error[i][0:5]
        #    print "ExpDataRange_Error[i][LengthExpRange[0]-1] = ",ExpDataRange_Error[i][LengthExpRange[0]-1]
        #    print "numrange = ",numrange
        #    print "fit_log = ",fit_log
        #    print "xmn = ",xmn
        #    print "xmx = ",xmx
        #    print "FitParameterName = ",FitParameterName
        #    print "MaxColX = ",MaxColX
        #    print "MaxColY = ",MaxColY
        #    print "MaxLength = ",MaxLength
        #    print "ExpDataRange_x = ", ExpDataRange_x
        #    print "####################################################"
        #    sys.exit(0)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## all experimental arrays must have the same dimensions
    if (NumberExpFiles > 1):
        ExpDataX = numpy.zeros((NumberExpFiles, MaxLength, MaxColX))                        ## construct an array of rank 3, with dimensions:
                                                                                            ##   NumberExpFiles, MaxLength, MaxColX
        ExpDataY = numpy.zeros((NumberExpFiles, MaxLength, MaxColY))                        ## construct an array of rank 3, with dimensions:
                                                                                            ##   NumberExpFiles, MaxLength, MaxColY
        ExpDataError = numpy.zeros((NumberExpFiles, MaxLength, MaxColY))                    ## construct an array of rank 3, with dimensions:
                                                                                            ##   NumberExpFiles, MaxLength, MaxColY
        for i in xrange(NumberExpFiles):                                                    ## loop over all experimental files
            length = LengthExpRange[i]
            colx = len(ExpDataRange_x[i][0])
            coly = len(ExpDataRange_y[i][0])
            ExpDataX[i, 0:length, 0:colx] = ExpDataRange_x[i]
            ExpDataY[i, 0:length, 0:coly] = ExpDataRange_y[i]
            ExpDataError[i, 0:length, 0:coly] = ExpDataRange_Error[i]
    else:
        ExpDataX = ExpDataRange_x
        ExpDataY = ExpDataRange_y
        ExpDataError = ExpDataRange_Error

    # Debug:
    #    for i in xrange(NumberExpFiles):
    #        print "File i: ",i
    #        print "ExpDataX[i][0:5] = ",ExpDataX[i][0:5]
    #        print "ExpDataX[i][LengthExpRange[i] - 1] = ",ExpDataX[i][LengthExpRange[i] - 1]
    #        print "ExpDataY[i][0:5] = ",ExpDataY[i][0:5]
    #        print "ExpDataY[i][LengthExpRange[i] - 1] = ",ExpDataY[i][LengthExpRange[i] - 1]
    #        print "ExpDataError[i][0:5] = ",ExpDataError[i][0:5]
    #        print "ExpDataError[i][length - 1] = ",ExpDataError[i][LengthExpRange[i] - 1]
    #        print "################################################"
    #    sys.exit(0)


    ##-----------------------------------------------------------------------------------------------------------------------------------------------------
    ## all (exp and fit) ranges must have the same length
    MaxRangeNumber = max(NumExpRanges) + 1
    MinRange = numpy.zeros((NumberExpFiles, MaxRangeNumber, MaxColX))
    MaxRange = numpy.zeros((NumberExpFiles, MaxRangeNumber, MaxColX))
    NumberRanges = []
    for i in xrange(NumberExpFiles):
        NumberX = NumberXColumns[i]                                                         ## get number of x-columns
        NumberRanges.append(NumExpRanges[i])
        if (NumExpRanges[i] > 0):
            for k in xrange(NumExpRanges[i]):
                MinRange[i, k, 0: NumberX] = MinExpRange[i][k]
                MaxRange[i, k, 0: NumberX] = MaxExpRange[i][k]

            # Debug:
            #    print 'i = ',i
            #    print 'NumExpRanges[i] = ',NumExpRanges[i]
            #    for k in xrange(NumExpRanges[i]):
            #        print 'k = ',k
            #        print 'MinRange[i, k, :] =  ',MinRange[i, k, :]
            #        print 'MaxRange[i, k, :] =  ',MaxRange[i, k, :]
            #        print '############################################'


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## in order to avoid using f2py write the relevant parameters to one big temp file
    VariableInputFileName = JobDir + "/general-variables__input.dat"
    VariableInputFile = open(os.path.normpath(VariableInputFileName), 'w')


    ## general INTEGER variables
    VariableInputFile.write("%30i\n" % printflagNum)                                        ## FORTRAN name: printflagNum
    VariableInputFile.write("%30i\n" % NumberOfFitAlgorithms)                               ## FORTRAN name: NumberOfFitAlgorithms
    VariableInputFile.write("%30i\n" % NumberInputFiles)                                    ## FORTRAN name: NumberInputFilesorg
    VariableInputFile.write("%30i\n" % NumberOutputFiles)                                   ## FORTRAN name: NumberOutputFilesOrg
    VariableInputFile.write("%30i\n" % MaxInputLines)                                       ## FORTRAN name: MaxInputLinesOrg
    VariableInputFile.write("%30i\n" % MaxParameter)                                        ## FORTRAN name: MaxParameterOrg
    for i in xrange(NumberExpFiles):
        VariableInputFile.write("%30i  %30i  %30i  %30i\n" % (LengthExpRange[i], NumberXColumns[i], NumberYColumns[i], NumberRanges[i]))


    ## general REAL*8 variables
    for i in xrange(NumberExpFiles):
        for j in xrange(MaxLength):
            for k in xrange(MaxColX):
                VariableInputFile.write("%.15e\n" % ExpDataX[i][j][k])
            for k in xrange(MaxColY):
                VariableInputFile.write("%.15e  %.15e\n" % (ExpDataY[i][j][k], ExpDataError[i][j][k]))
        for j in xrange(MaxRangeNumber):
            for k in xrange(MaxColX):
                VariableInputFile.write("%.15e  %.15e\n" % (MinRange[i][j][k], MaxRange[i][j][k]))


    ## general CHARACTER variables
    VariableInputFile.write(fit_log.strip() + "\n")
    VariableInputFile.write(CalculationMethod.strip() + "\n")
    VariableInputFile.write(PathStartScript.strip() + "\n")
    VariableInputFile.write(ExeCommandStartScript.strip() + "\n")
    VariableInputFile.write(currentpath.strip() + "\n")
    for k in xrange(NumberParameter):
        VariableInputFile.write(FitParameterName[k].strip() + "\n")
        VariableInputFile.write(FitParameterValue[k].strip() + "\n")
    VariableInputFile.close()

    # Debug:
    #    print "(PYTHON): General input variables:"
    #    print "printflagNum = ", printflagNum
    #    print "NumberOfFitAlgorithms = ", NumberOfFitAlgorithms
    #    print "NumberInputFiles = ", NumberInputFiles
    #    print "NumberOutputFiles = ", NumberOutputFiles
    #    print "MaxInputLines = ", MaxInputLines
    #    print "MaxParameter = ", MaxParameter
    #    print "LengthExpRange[:] = ", LengthExpRange
    #    print "NumberXColumns[:] = ", NumberXColumns
    #    print "NumberYColumns[:] = ", NumberYColumns
    #    print "NumberRanges[:] = ", NumberRanges
    #    print "ExpDataX[0][0][0] = ", ExpDataX[0][0][0]
    #    print "ExpDataX[0][-1][0] = ", ExpDataX[0][-1][0]
    #    print "ExpDataY[0][0][0] = ", ExpDataY[0][0][0]
    #    print "ExpDataY[0][-1][0] = ", ExpDataY[0][-1][0]
    #    print "ExpDataError[0][0][0] = ", ExpDataError[0][0][0]
    #    print "ExpDataError[0][-1][0] = ", ExpDataError[0][-1][0]
    #    print "MinRange[:][:][:] = ", MinRange[:][:][:]
    #    print "MaxRange[:][:][:] = ", MaxRange[:][:][:]
    #    print "fit_log = >", fit_log.strip(), "<"
    #    print "CalculationMethod = >", CalculationMethod.strip(), "<"
    #    print "PathStartScript = >", PathStartScript.strip(), "<"
    #    print "ExeCommandStartScript = >", ExeCommandStartScript.strip(), "<"
    #    print "FitParameterName[0] = >", FitParameterName[0].strip(), "<"
    #    print "FitParameterName[-1] = >", FitParameterName[-1].strip(), "<"
    #    print "FitParameterValue[0] = >", FitParameterValue[0].strip(), "<"
    #    print "FitParameterValue[-1] = >", FitParameterValue[-1].strip(), "<"
    #    print "parameter_set[0] = ",parameter_set[0]
    #    print "parameter_set[1] = ",parameter_set[1]
    #    print "parameter_set[2] = ",parameter_set[2]
    #    print "parameter_set[3] = ",parameter_set[3]
    #    print "\n\n\n"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## loop over all algorithms
    NestedSamplingCounter = 0
    LevenbergMarquardtCounter = 0
    SimulatedAnnealingCounter = 0
    PSOCounter = 0
    BeesCounter = 0
    GenCounter = 0
    ErrINSCounter = 0
    IntervalNSCounter = 0
    MCMCCounter = 0
    AdditionalPackagesCounter = 0
    NumberParameterSet = 1
    AllParameterSet = []
    AllParameterSet.append(parameter_set)                                                   ##
    LastAlgorithmNum = 0
    parameter_set_orig = parameter_set
    for algorithm in xrange(NumberOfFitAlgorithms):                                         ## loop over all algorithms in the chain


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get the different settings for the current algorithm
        AlgorithmSettings = AllAlgorithmSettings[algorithm]                                 ## get all settings for the current algorithm
        FitAlgorithm = AlgorithmSettings[0]                                                 ## .append(FitAlgorithm)


        ## get list of special settings for some algorithms (variation for LM etc.)
        SpecialAlgorithmSettings = AlgorithmSettings[1]                                     ## .append(NestedSamplingSettings)
                                                                                            ## .append(LevenbergMarquardtSettings)
                                                                                            ## .append(SimulatedAnnealingSettings)
                                                                                            ## .append(PSOSettings)
                                                                                            ## .append(BeesSettings)
        chilm = AlgorithmSettings[2]                                                        ## .append(chilm)
        numrange = AlgorithmSettings[3]                                                     ## .append(numrange)
        xmn = AlgorithmSettings[4]                                                          ## .append(xmn)
        xmx = AlgorithmSettings[5]                                                          ## .append(xmx)
        numiter = AlgorithmSettings[6]                                                      ## .append(numiter)
        NumberProcessors = AlgorithmSettings[7]                                             ## .append(NumberProcessors)
        MPIHostFileName = AlgorithmSettings[8]                                              ## .append(MPIHostFileName)
        DeterminationChi2 = AlgorithmSettings[9]                                            ## .append(DeterminationChi2)
        PlotOption = AlgorithmSettings[10]                                                  ## .append(PlotOption)
        RenormalizedChi2 = AlgorithmSettings[11]                                            ## .append(UseRenormalized Chi**2


        ## create array containing general algorithm settings
        GeneralAlgorithmSettings = numpy.zeros(15)
        GeneralAlgorithmSettings[0] = numpy.float(KnwonAlgorithms.index(FitAlgorithm.lower()))      ## get algorithm ID
        if (FitAlgorithm != "additionalpackages"):
            for i in xrange(len(SpecialAlgorithmSettings)):
                GeneralAlgorithmSettings[i + 1] = SpecialAlgorithmSettings[i]

                # Debug:
                # print "i + 1, GeneralAlgorithmSettings[i + 1] = ", i + 1, GeneralAlgorithmSettings[i + 1]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## rewrite PlotOption variable
        if (DeterminationChi2 < 0):
            Chi2SaveFlag = "true"
        else:
            Chi2SaveFlag = "false"
        PlotType = 0
        xAxisLabel = PlotOption[1]
        yAxisLabel = PlotOption[2]
        zAxisLabel = PlotOption[3]


        ## plot model function data for each iteration step?
        PlotIteration = 0                                                                   ## default: Allow plot of model function for each iter. step
        if (PlotOption[4] == "no" or PlotOption[4] == "none" or plotflag != "true"):
            PlotIteration = 1


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## if parallelization is possible, check if Number of processors is less or equal to the max. number of processors which are available
        ParallelizationFlag = 1
        if (ParallelizationPossible.strip() == "yes"):                                      ## if parallelization is possible
            ParallelizationFlag = NumberProcessors


            ## check, if MPI version of MAGIX is compiled
            MPIFlag = "false"
            currentpath = MAGIXrootDirectory + "/Modules/Levenberg-Marquardt/bin/"
            MPIStarterPath = os.path.dirname(currentpath) + "/"
            if (os.path.exists(MPIStarterPath + "Levenberg-Marquardt__MPI-" + StarterExecutable)):
                MPIFlag = "true"


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## settings for OpenMP (smp parallelization)
            if (MPIFlag == "false"):
                NumberAvailProc = multiprocessing.cpu_count()
                if (NumberProcessors > NumberAvailProc):
                    if (printflag != "false"):
                        print "\t WARNING:"
                        print "\t The defined number of processors (%i) " % NumberProcessors
                        print "\t is larger than the number of available processors (%i). " % NumberAvailProc
                        print "\t Reduce number of processors to the number of available processors."
                        print " "
                    ParallelizationFlag = NumberAvailProc
                else:
                    ParallelizationFlag = NumberProcessors


                ## set number of threads
                os.environ["OMP_NUM_THREADS"] = str(ParallelizationFlag)                        ## processors
                if (printflag != "false"):
                    print "\t Number of Processors: %i " % ParallelizationFlag
                    print " "


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## settings for MPI


            ## if no host file is defined, use localhost only
            elif (MPIHostFileName.strip() == ""):
                NumberAvailProc = multiprocessing.cpu_count()
                if (NumberProcessors > NumberAvailProc):
                    if (printflag != "false"):
                        print "\t WARNING:"
                        print "\t The defined number of processors (%i) " % NumberProcessors
                        print "\t is larger than the number of available processors (%i). " % NumberAvailProc
                        print "\t Reduce number of processors to the number of available processors."
                        print " "
                    ParallelizationFlag = NumberAvailProc
                else:
                    ParallelizationFlag = NumberProcessors
                if (printflag != "false"):
                    print "\t MPI host file not defined. Use localhost only!"
                    print "\t Number of Processors: %i " % ParallelizationFlag
                    print " "


                ## create host file
                os.environ["OMP_NUM_THREADS"] = str(ParallelizationFlag)                        ## processors
                DummyHostFileName = JobDir + "/hostfile_mpi.txt"
                DummyHostFile = open(os.path.normpath(DummyHostFileName), 'w')
                DummyHostFile.write("localhost slots=" + str(ParallelizationFlag) + "\n")
                DummyHostFile.close()

                # Debug:
                # print "host file = ", JobDir + "/hostfile_mpi.txt"


            ## settings for MPI: use global defined host file
            elif (MPIHostFileName.strip() == "MPI_HOSTS"):
                os.environ["OMP_NUM_THREADS"] = "1"
                if (printflag != "false"):
                    print "\t MPI host file defined by environment variable MPI_HOSTS!"
                    print "\t Number of Processors: %i " % ParallelizationFlag
                    print " "


            ## use defined host file
            else:


                ## read in host file
                MPIHostFile = open(MPIHostFileName)
                MPIHostFileContent = MPIHostFile.readlines()
                MPIHostFile.close()


                ## determine the number and the max. number of processors which are defined in the host file
                os.environ["OMP_NUM_THREADS"] = "1"
                ProcessorsAvailInHostFile = 0.0
                AllProcessorsAvailInHostFile = 0.0
                HostList = ""
                FirstHostFlag = "true"
                for line in MPIHostFileContent:
                    StrippedLine = line.strip()
                    if (not StrippedLine.startswith("#") and StrippedLine != ""):
                        i = StrippedLine.find(" slots=")
                        if (i > (-1)):
                            j = StrippedLine[i + 7:].find("max-slots=")
                            if (j > (-1)):
                                tempString1 = StrippedLine[i + 7: i + 7 + j].strip()
                                tempString2 = StrippedLine[i + 17 + j:].strip()
                                AllProcessorsAvailInHostFile += float(tempString2)
                            else:
                                tempString1 = StrippedLine[i + 7:].strip()

                            # Debug:
                            # print "tempString1 = ", tempString1

                            ProcessorsAvailInHostFile += float(tempString1)
                            if (FirstHostFlag == "true"):                                   ## avoid comma character at the beginning
                                HostList += " " + StrippedLine[:i] + " (" + tempString1 + " cores)"
                                FirstHostFlag = "false"
                            else:
                                HostList += ", " + StrippedLine[:i] + " (" + tempString1 + " cores)"
                        else:
                            if (FirstHostFlag == "true"):                                   ## avoid comma character at the beginning
                                HostList += " " + StrippedLine
                                FirstHostFlag = "false"
                            else:
                                HostList += ", " + StrippedLine


                ## set number of processors
                if (ProcessorsAvailInHostFile != 0.0):
                    if (NumberProcessors > ProcessorsAvailInHostFile):
                        if (printflag != "false"):
                            print " "
                            print "\t WARNING: Number of processors is larger than processors defined in host file!"
                            print "\t\t\t      Reduce number of processors to the number of processors in the host file!"
                            print "\n"
                            print "\t Number of Processors defined in host file: %i " % ProcessorsAvailInHostFile
                            print "\n"
                        ParallelizationFlag = ProcessorsAvailInHostFile


                ## print some informations to screen
                if (printflag != "false"):
                    print "\t Number of Processors: %i " % ParallelizationFlag
                    print "\t    Hosts: %s" % HostList
                    print " "


                ## copy host file to job directory
                if (MPIHostFileName.strip() != "" and MPIHostFileName.strip() != "MPI_HOSTS"):
                    command_string = "cp " + MPIHostFileName + " " + JobDir + "/hostfile_mpi.txt"
                    os.system(command_string)


        ## define default extension of the observation and the parameter xml-files
        ExpFilenameExtension = "out"
        XMLFilenameExtension = "out"


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## start the fit engine
        NewAllParameterSet = []
        NewNumberParameterSet = 0
        for ParmNum in xrange(NumberParameterSet):                                          ## loop over all parameter sets
            parameter_set = AllParameterSet[ParmNum]


            ## check, if current algorithm is last algorithm in the chain
            if ((algorithm + 1) == NumberOfFitAlgorithms and (ParmNum + 1) == NumberParameterSet):
                LastAlgorithmNum = 1


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## call Nested-Sampling algorithm
            ##
            ## input parameters:    number of objects:                                      number_objects
            ##                      ParamSetCounter:                                        number of best sites
            ##                      minmaxTrans:                                            value for min to max transition
            ##
            if (FitAlgorithm == "nested-sampling"):
                NestedSamplingCounter += 1


                ## get special settings
                ParamSetCounter = SpecialAlgorithmSettings[0]                               ## number of best sites
                number_objects = SpecialAlgorithmSettings[1]                                ## number of objects
                minmaxTrans = SpecialAlgorithmSettings[2]                                   ## value for min to max transition


                ## call fortran routine for Nested-Sampling
                parameter_set_copy = parameter_set
                parameter_set[1] = parameter_set_orig[1]
                NameOfAlgorithm = "Nested-Sampling"
                calstatus, FitFunctionAll, Chi2ValuesAll, parmCopy, FinalParameterSet = StartAlgPackage(NameOfAlgorithm, StarterExecutable, MPIFlag, \
                                                                                            MAGIXrootDirectory, JobDir, JobID, NumberExpFiles, \
                                                                                            MaxLength, MaxColX, MaxColY, MaxRangeNumber, \
                                                                                            NumberParameter, ParamSetCounter, LastAlgorithmNum, numiter, \
                                                                                            ParallelizationFlag, DeterminationChi2, PlotIteration, \
                                                                                            PlotType, RenormalizedChi2, NestedSamplingCounter, \
                                                                                            chilm, GeneralAlgorithmSettings, SpecialAlgorithmSettings, \
                                                                                            parameter_set, xAxisLabel, yAxisLabel, zAxisLabel)
                ## write best parameter sets to files
                for i in range(ParamSetCounter):
                    parameters = []
                    parameters.append(FinalParameterSet[i])
                    parameters.append(parameter_set[1])
                    parameters.append(parameter_set[2])
                    parameters.append(parameter_set[3])
                    FitFunctionValues = FitFunctionAll[i]
                    Chi2Values = Chi2ValuesAll[i]
                    parameters[1] = numpy.zeros(NumberParameter)


                    ## copy new parameter set
                    NewNumberParameterSet += 1
                    NewAllParameterSet.append(parameters)


                    ## define extension of the observation and the parameter xml-file if algorithm chain is chosed
                    if (NumberOfFitAlgorithms > 1):
                        ExpFilenameExtension = "NS__call_" + str(NestedSamplingCounter)
                    else:
                        ExpFilenameExtension = "NS"
                    PlotFilenameExtension = ExpFilenameExtension
                    ExpFilenameExtension = ExpFilenameExtension + "__site_" + str(i + 1) + ".out"
                    XMLFilenameExtension = ExpFilenameExtension


                    ## write results of fit process to files
                    parameter_set = parameter_set_copy
                    ok = StoreResults(ExpInputFormat, ClassLoadExp, printflag, NumberExpFiles, LengthExpRange, NumberYColumns, ExpDataRange_x, \
                                      FitFunctionValues, Chi2SaveFlag, Chi2Values, xml_file, NumberParameter, parameters, parameter_set, \
                                      ParameterValuesOriginal, FitParameterName, ExpFilenameExtension, XMLFilenameExtension, UnmodifiedNumberParameter)


                ## store best result
                AlgCounter = NestedSamplingCounter
                NumberSites = ParamSetCounter
                parameter_set[0] = FinalParameterSet[0]
                if (ParamSetCounter > 1):
                    FitFunctionValues = FitFunctionAll
                    Chi2Values = Chi2ValuesAll
                else:
                    FitFunctionValues = FitFunctionAll[0]
                    Chi2Values = Chi2ValuesAll[0]
                ExpFilenameExtension = PlotFilenameExtension



            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## call Levenberg-Marquardt algorithm
            ##
            ## input parameters:    max. number of iterations:                              numiter
            ##                      variation value
            ##
            elif (FitAlgorithm == "levenberg-marquardt"):
                LevenbergMarquardtCounter += 1
                if (NumberOfFitAlgorithms > 1):
                    LevenbergMarquardtCounter = -LevenbergMarquardtCounter


                ## call fortran routine for Levenberg-Marquardt
                ParamSetCounter = 1
                parameter_set_copy = parameter_set
                parameter_set[1] = parameter_set_orig[1]
                NameOfAlgorithm = "Levenberg-Marquardt"
                calstatus, FitFunctionValues, Chi2Values, parmCopy, FinalParameterSet = StartAlgPackage(NameOfAlgorithm, StarterExecutable, MPIFlag, \
                                                                                            MAGIXrootDirectory, JobDir, JobID, NumberExpFiles, \
                                                                                            MaxLength, MaxColX, MaxColY, MaxRangeNumber, \
                                                                                            NumberParameter, ParamSetCounter, LastAlgorithmNum, numiter, \
                                                                                            ParallelizationFlag, DeterminationChi2, PlotIteration, \
                                                                                            PlotType, RenormalizedChi2, LevenbergMarquardtCounter, \
                                                                                            chilm, GeneralAlgorithmSettings, SpecialAlgorithmSettings, \
                                                                                            parameter_set, xAxisLabel, yAxisLabel, zAxisLabel)
                ## we have only one "best site"
                FitFunctionValues = FitFunctionValues[0]
                Chi2Values = Chi2Values[0]
                parameters = parameter_set
                parameters[0] = FinalParameterSet[0]


                ## copy new parameter set
                NewNumberParameterSet += 1
                NewAllParameterSet.append(parameters)
                NumberSites = 1


                ## define extension of the observation and the parameter xml-file if algorithm chain is chosed
                LevenbergMarquardtCounter = abs(LevenbergMarquardtCounter)
                if (NumberOfFitAlgorithms > 1):
                    ExpFilenameExtension = "LM__call_" + str(LevenbergMarquardtCounter) + ".out"
                    XMLFilenameExtension = "LM__call_" + str(LevenbergMarquardtCounter) + ".out"
                else:
                    ExpFilenameExtension = "LM.out"
                    XMLFilenameExtension = "LM.out"
                AlgCounter = LevenbergMarquardtCounter


                ## write results of fit process to files
                parameter_set = parameter_set_copy
                ok = StoreResults(ExpInputFormat, ClassLoadExp, printflag, NumberExpFiles, LengthExpRange, NumberYColumns, ExpDataRange_x, \
                                  FitFunctionValues, Chi2SaveFlag, Chi2Values, xml_file, NumberParameter, parameters, parameter_set, \
                                  ParameterValuesOriginal, FitParameterName, ExpFilenameExtension, XMLFilenameExtension, UnmodifiedNumberParameter)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## call Simulated-Annealing algorithm
            ##
            ## input parameters:    max. number of iterations:                              numiter
            ##                      temperature:                                            used temperature for simulated annealing algorithm
            elif (FitAlgorithm == "simulated-annealing"):
                SimulatedAnnealingCounter += 1
                if (NumberOfFitAlgorithms > 1):
                    SimulatedAnnealingCounter = -SimulatedAnnealingCounter


                ## get special settings
                MethodSA = SpecialAlgorithmSettings[0]                                      ## get method of Simulated-Annealing algorithm
                ScheduleSA = SpecialAlgorithmSettings[1]                                    ## get schedule (only for scipy version)
                Temperature = SpecialAlgorithmSettings[2]                                   ## get temperature for algorithm
                TemperatureReductionKoeff = SpecialAlgorithmSettings[3]                     ## get reduction coefficient of temperature for algorithm
                NumberOfReduction = SpecialAlgorithmSettings[4]                             ## get number of reductions
                MaxNumberOfReheatingPhases = SpecialAlgorithmSettings[5]                    ## get max number of reheating phases


                ## call fortran routine for Simulated-Annealing
                ParamSetCounter = 1
                parameter_set_copy = parameter_set
                parameter_set[1] = parameter_set_orig[1]
                NameOfAlgorithm = "Simulated-Annealing"
                calstatus, FitFunctionValues, Chi2Values, parmCopy, FinalParameterSet = StartAlgPackage(NameOfAlgorithm, StarterExecutable, MPIFlag, \
                                                                                            MAGIXrootDirectory, JobDir, JobID, NumberExpFiles, \
                                                                                            MaxLength, MaxColX, MaxColY, MaxRangeNumber, \
                                                                                            NumberParameter, ParamSetCounter, LastAlgorithmNum, numiter, \
                                                                                            ParallelizationFlag, DeterminationChi2, PlotIteration, \
                                                                                            PlotType, RenormalizedChi2, SimulatedAnnealingCounter, \
                                                                                            chilm, GeneralAlgorithmSettings, SpecialAlgorithmSettings, \
                                                                                            parameter_set, xAxisLabel, yAxisLabel, zAxisLabel)
                ## we have only one "best site"
                FitFunctionValues = FitFunctionValues[0]
                Chi2Values = Chi2Values[0]
                parameters = parameter_set
                parameters[0] = FinalParameterSet[0]


                ## copy new parameter set
                NewNumberParameterSet += 1
                NewAllParameterSet.append(parameters)
                parameters[1] = numpy.zeros(NumberParameter)
                NumberSites = 1


                ## define extension of the observation and the parameter xml-file if algorithm chain is chosed
                SimulatedAnnealingCounter = abs(SimulatedAnnealingCounter)
                if (NumberOfFitAlgorithms > 1):
                    ExpFilenameExtension = "SA__call_" + str(SimulatedAnnealingCounter) + ".out"
                    XMLFilenameExtension = "SA__call_" + str(SimulatedAnnealingCounter) + ".out"
                else:
                    ExpFilenameExtension = "SA.out"
                    XMLFilenameExtension = "SA.out"
                AlgCounter = SimulatedAnnealingCounter


                ## write results of fit process to files
                parameter_set = parameter_set_copy
                ok = StoreResults(ExpInputFormat, ClassLoadExp, printflag, NumberExpFiles, LengthExpRange, NumberYColumns, ExpDataRange_x, \
                                  FitFunctionValues, Chi2SaveFlag, Chi2Values, xml_file, NumberParameter, parameters, parameter_set, \
                                  ParameterValuesOriginal, FitParameterName, ExpFilenameExtension, XMLFilenameExtension, UnmodifiedNumberParameter)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## call pso algorithm
            ##
            ## input parameters:    max. number of iterations:                              numiter
            ##                      ParamSetCounter:                                        number of best sites
            ##
            elif (FitAlgorithm == "pso"):
                PSOCounter += 1


                ## get special settings
                ParamSetCounter = SpecialAlgorithmSettings[0]                               ## get number of best sites
                NumberParticles = SpecialAlgorithmSettings[1]                               ## get number of particles


                ## call fortran routine for pso
                parameter_set_copy = parameter_set
                parameter_set[1] = parameter_set_orig[1]
                NameOfAlgorithm = "PSO"
                calstatus, FitFunctionAll, Chi2ValuesAll, parmCopy, FinalParameterSet = StartAlgPackage(NameOfAlgorithm, StarterExecutable, MPIFlag, \
                                                                                            MAGIXrootDirectory, JobDir, JobID, NumberExpFiles, \
                                                                                            MaxLength, MaxColX, MaxColY, MaxRangeNumber, \
                                                                                            NumberParameter, ParamSetCounter, LastAlgorithmNum, numiter, \
                                                                                            ParallelizationFlag, DeterminationChi2, PlotIteration, \
                                                                                            PlotType, RenormalizedChi2, PSOCounter, \
                                                                                            chilm, GeneralAlgorithmSettings, SpecialAlgorithmSettings, \
                                                                                            parameter_set, xAxisLabel, yAxisLabel, zAxisLabel)
                ## write best parameter sets to files
                for i in range(ParamSetCounter):
                    parameters = []
                    parameters.append(FinalParameterSet[i])
                    parameters.append(parameter_set[1])
                    parameters.append(parameter_set[2])
                    parameters.append(parameter_set[3])
                    FitFunctionValues = FitFunctionAll[i]
                    Chi2Values = Chi2ValuesAll[i]
                    parameters[1] = numpy.zeros(NumberParameter)


                    ## copy new parameter set
                    NewNumberParameterSet += 1
                    NewAllParameterSet.append(parameters)


                    ## define extension of the observation and the parameter xml-file if algorithm chain is chosed
                    if (NumberOfFitAlgorithms > 1):
                        ExpFilenameExtension = "PSO__call_" + str(PSOCounter)
                    else:
                        ExpFilenameExtension = "PSO"
                    PlotFilenameExtension = ExpFilenameExtension
                    ExpFilenameExtension = ExpFilenameExtension + "__site_" + str(i + 1) + ".out"
                    XMLFilenameExtension = ExpFilenameExtension


                    ## write results of fit process to files
                    parameter_set = parameter_set_copy
                    ok = StoreResults(ExpInputFormat, ClassLoadExp, printflag, NumberExpFiles, LengthExpRange, NumberYColumns, ExpDataRange_x, \
                                      FitFunctionValues, Chi2SaveFlag, Chi2Values, xml_file, NumberParameter, parameters, parameter_set, \
                                      ParameterValuesOriginal, FitParameterName, ExpFilenameExtension, XMLFilenameExtension, UnmodifiedNumberParameter)


                ## store best result
                AlgCounter = PSOCounter
                NumberSites = ParamSetCounter
                parameter_set[0] = FinalParameterSet[0]
                if (ParamSetCounter > 1):
                    FitFunctionValues = FitFunctionAll
                    Chi2Values = Chi2ValuesAll
                else:
                    FitFunctionValues = FitFunctionAll[0]
                    Chi2Values = Chi2ValuesAll[0]
                ExpFilenameExtension = PlotFilenameExtension


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## call bees algorithm
            ##
            ## input parameters:    number of bee sites:                                    bestsitescount
            ##                      max. number of iterations:                              numiter
            ##                      max. value for transformation from max to min-search:   minmaxTrans
            ##                      number of runcounts ??:                                 runcount
            elif (FitAlgorithm == "bees"):
                BeesCounter += 1


                ## get special settings
                ParamSetCounter = SpecialAlgorithmSettings[0]                               ## number of best sites
                                                                                            ## (value from the algorithm xml-file)
                NumberBees = SpecialAlgorithmSettings[1]                                    ## get number of bees

                # Debug:
                # print 'ParamSetCounter = ',ParamSetCounter
                # print 'BeesCounter = ',BeesCounter


                ## call fortran routine for bees
                parameter_set_copy = parameter_set
                parameter_set[1] = parameter_set_orig[1]
                NameOfAlgorithm = "Bees"
                calstatus, FitFunctionAll, Chi2ValuesAll, parmCopy, FinalParameterSet = StartAlgPackage(NameOfAlgorithm, StarterExecutable, MPIFlag, \
                                                                                            MAGIXrootDirectory, JobDir, JobID, NumberExpFiles, \
                                                                                            MaxLength, MaxColX, MaxColY, MaxRangeNumber, \
                                                                                            NumberParameter, ParamSetCounter, LastAlgorithmNum, numiter, \
                                                                                            ParallelizationFlag, DeterminationChi2, PlotIteration, \
                                                                                            PlotType, RenormalizedChi2, BeesCounter, \
                                                                                            chilm, GeneralAlgorithmSettings, SpecialAlgorithmSettings, \
                                                                                            parameter_set, xAxisLabel, yAxisLabel, zAxisLabel)
                ## write best parameter sets to files
                for i in range(ParamSetCounter):
                    parameters = []
                    parameters.append(FinalParameterSet[i])
                    parameters.append(parameter_set[1])
                    parameters.append(parameter_set[2])
                    parameters.append(parameter_set[3])
                    FitFunctionValues = FitFunctionAll[i]
                    Chi2Values = Chi2ValuesAll[i]
                    parameters[1] = numpy.zeros(NumberParameter)


                    ## copy new parameter set
                    NewNumberParameterSet += 1
                    NewAllParameterSet.append(parameters)


                    ## define extension of the observation and the parameter xml-file if algorithm chain is chosed
                    if (NumberOfFitAlgorithms > 1):
                        ExpFilenameExtension = "Bees__call_" + str(BeesCounter)
                    else:
                        ExpFilenameExtension = "Bees"
                    PlotFilenameExtension = ExpFilenameExtension
                    ExpFilenameExtension = ExpFilenameExtension + "__site_" + str(i + 1) + ".out"
                    XMLFilenameExtension = ExpFilenameExtension


                    ## write results of fit process to files
                    parameter_set = parameter_set_copy
                    ok = StoreResults(ExpInputFormat, ClassLoadExp, printflag, NumberExpFiles, LengthExpRange, NumberYColumns, ExpDataRange_x, \
                                      FitFunctionValues, Chi2SaveFlag, Chi2Values, xml_file, NumberParameter, parameters, parameter_set, \
                                      ParameterValuesOriginal, FitParameterName, ExpFilenameExtension, XMLFilenameExtension, UnmodifiedNumberParameter)


                ## store best result
                AlgCounter = BeesCounter
                NumberSites = ParamSetCounter
                parameter_set[0] = FinalParameterSet[0]
                if (ParamSetCounter > 1):
                    FitFunctionValues = FitFunctionAll
                    Chi2Values = Chi2ValuesAll
                else:
                    FitFunctionValues = FitFunctionAll[0]
                    Chi2Values = Chi2ValuesAll[0]
                ExpFilenameExtension = PlotFilenameExtension


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## call Genetic algorithm
            ##
            ## input parameters:    number of bee sites:                                    bestsitescount
            ##                      max. number of iterations:                              numiter
            ##                      number of runcounts ??:                                 runcount
            elif (FitAlgorithm == "genetic"):
                GenCounter += 1
                if (NumberOfFitAlgorithms > 1):
                    GenCounter = -GenCounter


                ## get special settings
                UseNewRange = SpecialAlgorithmSettings[0]                                   ## get flag for new range
                NumberChromosomes = SpecialAlgorithmSettings[1]                             ## get number of chromosomes
                ParamSetCounter = SpecialAlgorithmSettings[2]                               ## get number of best sites
                NumberSites = ParamSetCounter

                # Debug:
                # print "UseNewRange = ", UseNewRange
                # print "NumberChromosomes = ", NumberChromosomes
                # print "ParamSetCounter = ", ParamSetCounter


                ## call fortran routine for genetic
                parameter_set_copy = parameter_set
                parameter_set[1] = parameter_set_orig[1]
                paramSet = parameter_set
                NameOfAlgorithm = "Genetic"
                calstatus, FitFunctionAll, Chi2ValuesAll, parmCopy, FinalParameterSet = StartAlgPackage(NameOfAlgorithm, StarterExecutable, MPIFlag, \
                                                                                            MAGIXrootDirectory, JobDir, JobID, NumberExpFiles, \
                                                                                            MaxLength, MaxColX, MaxColY, MaxRangeNumber, \
                                                                                            NumberParameter, ParamSetCounter, LastAlgorithmNum, numiter, \
                                                                                            ParallelizationFlag, DeterminationChi2, PlotIteration, \
                                                                                            PlotType, RenormalizedChi2, GenCounter, \
                                                                                            chilm, GeneralAlgorithmSettings, SpecialAlgorithmSettings, \
                                                                                            parameter_set, xAxisLabel, yAxisLabel, zAxisLabel)
                ## check, if new ranges are useful and save new parameter ranges
                if (parmCopy != []):
                    parmCopy = parmCopy[0]
                    for iii in xrange(NumberParameter):
                        if (parameter_set[1][iii] == 1.0):                                  ## define new ranges only for free parameters
                            if (parmCopy[2][iii] < parmCopy[3][iii] and UseNewRange == 1):  ## the lower limit of the new range have to be smaller
                                parameter_set[2][iii] = parmCopy[2][iii]                    ## than the upper range, if not, use old range
                                parameter_set[3][iii] = parmCopy[3][iii]


                ## write best parameter sets to files
                ParamSetCounter = abs(ParamSetCounter)
                GenCounter = abs(GenCounter)
                for i in range(ParamSetCounter):
                    parameters = []
                    parameters.append(FinalParameterSet[i])
                    parameters.append(parameter_set[1])
                    parameters.append(parameter_set[2])
                    parameters.append(parameter_set[3])
                    FitFunctionValues = FitFunctionAll[i]
                    Chi2Values = Chi2ValuesAll[i]
                    parameters[1] = numpy.zeros(NumberParameter)


                    ## copy new parameter set
                    NewNumberParameterSet += 1
                    NewAllParameterSet.append(parameters)


                    ## define extension of the observation and the parameter xml-file if algorithm chain is chosed
                    if (NumberOfFitAlgorithms > 1):
                        ExpFilenameExtension = "Genetic__call_" + str(GenCounter)
                    else:
                        ExpFilenameExtension = "Genetic"
                    AlgCounter = GenCounter
                    PlotFilenameExtension = ExpFilenameExtension
                    ExpFilenameExtension = ExpFilenameExtension + "__site_" + str(i + 1) + ".out"
                    XMLFilenameExtension = ExpFilenameExtension


                    ## write results of fit process to files
                    parameter_set = parameter_set_copy
                    ok = StoreResults(ExpInputFormat, ClassLoadExp, printflag, NumberExpFiles, LengthExpRange, NumberYColumns, ExpDataRange_x, \
                                      FitFunctionValues, Chi2SaveFlag, Chi2Values, xml_file, NumberParameter, parameters, parameter_set, \
                                      ParameterValuesOriginal, FitParameterName, ExpFilenameExtension, XMLFilenameExtension, UnmodifiedNumberParameter)


                ## store best result
                parameter_set[0] = FinalParameterSet[0]
                if (ParamSetCounter > 1):
                    FitFunctionValues = FitFunctionAll
                    Chi2Values = Chi2ValuesAll
                else:
                    FitFunctionValues = FitFunctionAll[0]
                    Chi2Values = Chi2ValuesAll[0]
                ExpFilenameExtension = PlotFilenameExtension


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## call error estimation with interval nested sampling algorithm
            ##
            ## input parameters:    max. number of iterations: numiter
            elif (FitAlgorithm == "errorestim" or FitAlgorithm == "errorestim_ins"):
                ErrINSCounter += 1
                if (NumberOfFitAlgorithms > 1):
                    ErrINSCounter = -ErrINSCounter


                ## get special settings
                ErrorMethod = SpecialAlgorithmSettings[0]                                   ## get method for error estimation


                ## param_set_err - set of parameter properties after any fitting procedure
                ParamSetCounter = 1
                NumberSites = ParamSetCounter
                parameter_set_copy = copy.deepcopy(parameter_set)
                parameter_set[1] = parameter_set_orig[1]
                nfit = sum(parameter_set_orig[1][:UnmodifiedNumberParameter])


                ## call python routine for errors estimation with interval nested sampling algorithm
                errors_param_values, FitFunctionValues, Chi2Values, ErrorModelFunctionValues, ErrorChi2Values = ErrEstim_INS.ErrorEstimator(\
                                                                                            SpecialAlgorithmSettings, GeneralAlgorithmSettings, \
                                                                                            printflagNum, LastAlgorithmNum, \
                                                                                            ParamSetCounter, chilm, numiter, ErrINSCounter, \
                                                                                            DeterminationChi2, PlotIteration, PlotType, fit_log, \
                                                                                            NumberInputFiles, NumberOutputFiles, ParallelizationFlag, \
                                                                                            JobID, MaxInputLines, MaxParameter, RenormalizedChi2, \
                                                                                            currentpath, CalculationMethod, xAxisLabel, yAxisLabel, \
                                                                                            zAxisLabel, PathStartScript, ExeCommandStartScript, \
                                                                                            parameter_set, ExpDataX, ExpDataY, ExpDataError, \
                                                                                            NumberRanges, MinRange, MaxRange, NumberXColumns, \
                                                                                            NumberYColumns, LengthExpRange, MaxRangeNumber, \
                                                                                            NumberExpFiles, MaxLength, MaxColX, MaxColY, \
                                                                                            NumberParameter, plotflag, modelflag, MPIFlag, \
                                                                                            MAGIXrootDirectory, JobDir)
                ## construct final parameter set array
                parameters = []
                parameters.append(parameter_set[0])
                parameters.append(parameter_set[1])
                parameters.append(parameter_set[2])
                parameters.append(parameter_set[3])

                # Debug:
                # print 'param_values = ', param_values
                # print 'errors_param_values = ', errors_param_values
                # print 'nfit = ',nfit


                ## define new parameter set
                jjj = (-1)
                error_param = []
                for iii in xrange(NumberParameter):

                    # Debug:
                    # print 'iii = ', iii
                    # print 'parameters[1][iii] = ',parameters[1][iii]

                    if (parameters[1][iii] == 1.0):
                        jjj += 1

                        # Debug:
                        # print 'iii, jjj = ',iii, jjj
                        # print 'errors_param_values[jjj] = ', errors_param_values[jjj]
                        # print 'parameters[0][iii] = ', parameters[0][iii]
                        # print 'parameters[1] = ', parameters[1]

                        error_param.append(errors_param_values[jjj])
                    else:
                        error_param.append(0)


                ## store best result
                parameters.append(error_param)
                # parameter_set[0] = parameters[0]
                # parameter_set[1] = parameters[1]

                # Debug:
                # print 'parameters[4] = ', parameters[4]


                ## copy new parameter set
                NewNumberParameterSet += 1
                NewAllParameterSet.append(parameters)


                ## define extension of the observation and the parameter xml-file if algorithm chain is chosed
                ExpFilenameExtension = ""
                ExpFilenameExtensionUp = ""
                ExpFilenameExtensionLow = ""
                ErrINSCounter = abs(ErrINSCounter)
                if (NumberOfFitAlgorithms > 1):
                    ExpFilenameExtension = "ErrorEstim_INS__call_" + str(ErrINSCounter) + ".out"
                    XMLFilenameExtension = "ErrorEstim_INS__call_" + str(ErrINSCounter) + ".out"
                    ExpFilenameExtensionUp = "ErrorEstim_INS__UpperErrorValues__call_" + str(ErrINSCounter) + ".out"
                    ExpFilenameExtensionLow = "ErrorEstim_INS__LowerErrorValues__call_" + str(ErrINSCounter) + ".out"
                else:
                    ExpFilenameExtension = "ErrorEstim_INS.out"
                    XMLFilenameExtension = "ErrorEstim_INS.out"
                    ExpFilenameExtensionUp = "ErrorEstim_INS__UpperErrorValues.out"
                    ExpFilenameExtensionLow = "ErrorEstim_INS__LowerErrorValues.out"
                AlgCounter = ErrINSCounter


                ## write results of fit process to files
                parameter_set = copy.deepcopy(parameter_set_copy)
                ok = StoreResults(ExpInputFormat, ClassLoadExp, printflag, NumberExpFiles, LengthExpRange, NumberYColumns, ExpDataRange_x, \
                                  FitFunctionValues, Chi2SaveFlag, Chi2Values, xml_file, NumberParameter, parameters, parameter_set, \
                                  ParameterValuesOriginal, FitParameterName, ExpFilenameExtension, XMLFilenameExtension, UnmodifiedNumberParameter)


                ## write model function values for upper and lower error values to file
                ok = ClassLoadExp.WriteFile(ExpInputFormat, printflag, NumberExpFiles, LengthExpRange, NumberYColumns, ExpDataX, \
                                            ErrorModelFunctionValues[0][0], Chi2SaveFlag, ErrorChi2Values[0][0], ExpFilenameExtensionUp)
                ok = ClassLoadExp.WriteFile(ExpInputFormat, printflag, NumberExpFiles, LengthExpRange, NumberYColumns, ExpDataX, \
                                            ErrorModelFunctionValues[1][0], Chi2SaveFlag, ErrorChi2Values[1][0], ExpFilenameExtensionLow)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## call interval nested-sampling algorithm
            ##
            ## input parameters:    number of bee sites: bestsitescount
            ##
            elif (FitAlgorithm == "interval-ns" or FitAlgorithm == "ins"):
                IntervalNSCounter += 1
                if (NumberOfFitAlgorithms > 1):
                    IntervalNSCounter = -IntervalNSCounter


                ## call python routine for INS
                ParamSetCounter = SpecialAlgorithmSettings[3]                               ## get number of best sites
                NumberSites = ParamSetCounter
                parameter_set_copy = copy.deepcopy(parameter_set)
                parameter_set[1] = parameter_set_orig[1]
                calstatus, FinalParameterSet, FitFunctionAll, Chi2ValuesAll = INS_paral.Interval_Nested_Sampling(SpecialAlgorithmSettings, \
                                                                                            GeneralAlgorithmSettings, printflagNum, LastAlgorithmNum, \
                                                                                            ParamSetCounter, chilm, numiter, IntervalNSCounter, \
                                                                                            DeterminationChi2, PlotIteration, PlotType, fit_log, \
                                                                                            NumberInputFiles, NumberOutputFiles, ParallelizationFlag, \
                                                                                            JobID, MaxInputLines, MaxParameter, RenormalizedChi2, \
                                                                                            currentpath, CalculationMethod, xAxisLabel, yAxisLabel, \
                                                                                            zAxisLabel, PathStartScript, ExeCommandStartScript, \
                                                                                            parameter_set, ExpDataX, ExpDataY, ExpDataError, \
                                                                                            NumberRanges, MinRange, MaxRange, NumberXColumns, \
                                                                                            NumberYColumns, LengthExpRange, MaxRangeNumber, \
                                                                                            NumberExpFiles, MaxLength, MaxColX, MaxColY, \
                                                                                            NumberParameter, plotflag, modelflag, MPIFlag, \
                                                                                            MAGIXrootDirectory, JobDir)
                ## write best parameter sets to files
                IntervalNSCounter = abs(IntervalNSCounter)
                for i in range(ParamSetCounter):
                    parameters = []
                    parameters.append(FinalParameterSet[i])
                    parameters.append(parameter_set_copy[1])
                    parameters.append(parameter_set_copy[2])
                    parameters.append(parameter_set_copy[3])
                    FitFunctionValues = FitFunctionAll[i]
                    Chi2Values = Chi2ValuesAll[i]
                    parameters[1] = numpy.zeros(NumberParameter)

                    # Debug:
                    # print "NumberParameter = ", NumberParameter


                    ## copy new parameter set
                    NewNumberParameterSet += 1
                    NewAllParameterSet.append(parameters)


                    ## define extension of the observation and the parameter xml-file if algorithm chain is chosed
                    if (NumberOfFitAlgorithms > 1):
                        ExpFilenameExtension = "INS__call_" + str(INSCounter)
                    else:
                        ExpFilenameExtension = "INS"
                    PlotFilenameExtension = ExpFilenameExtension
                    if (ParamSetCounter > 1):
                        ExpFilenameExtension += "__site_" + str(i + 1)
                    ExpFilenameExtension += ".out"
                    XMLFilenameExtension = ExpFilenameExtension


                    ## write results of fit process to files
                    parameter_set = copy.deepcopy(parameter_set_copy)
                    ok = StoreResults(ExpInputFormat, ClassLoadExp, printflag, NumberExpFiles, LengthExpRange, NumberYColumns, ExpDataRange_x, \
                                      FitFunctionValues, Chi2SaveFlag, Chi2Values, xml_file, NumberParameter, parameters, parameter_set, \
                                      ParameterValuesOriginal, FitParameterName, ExpFilenameExtension, XMLFilenameExtension, UnmodifiedNumberParameter)


                ## store best result
                AlgCounter = IntervalNSCounter
                parameter_set[0] = FinalParameterSet[0]
                if (ParamSetCounter > 1):
                    FitFunctionValues = FitFunctionAll
                    Chi2Values = Chi2ValuesAll
                else:
                    FitFunctionValues = FitFunctionAll[0]
                    Chi2Values = Chi2ValuesAll[0]
                ExpFilenameExtension = PlotFilenameExtension


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## call MCMC algorithm
            ##
            ## input parameters:    number of bee sites: bestsitescount
            ##
            elif (FitAlgorithm == "mcmc"):
                MCMCCounter += 1
                if (NumberOfFitAlgorithms > 1):
                    MCMCCounter = -MCMCCounter


                ## get special settings
                ParamSetCounter = SpecialAlgorithmSettings[2]                               ## get number of best sites


                ## call python routine for MCMC
                NumberSites = ParamSetCounter
                parameter_set_copy = copy.deepcopy(parameter_set)
                parameter_set[1] = copy.deepcopy(parameter_set_orig[1])
                calstatus, FinalParameterSet, FitFunctionAll, Chi2ValuesAll = MCMC.Start(SpecialAlgorithmSettings, GeneralAlgorithmSettings, printflagNum, \
                                                                                            LastAlgorithmNum, ParamSetCounter, chilm, numiter, \
                                                                                            MCMCCounter, DeterminationChi2, PlotIteration, PlotType, \
                                                                                            fit_log, NumberInputFiles, NumberOutputFiles, \
                                                                                            ParallelizationFlag, JobID, MaxInputLines, MaxParameter, \
                                                                                            RenormalizedChi2, currentpath, CalculationMethod, \
                                                                                            xAxisLabel, yAxisLabel, zAxisLabel, PathStartScript, \
                                                                                            ExeCommandStartScript, parameter_set, ExpDataX, ExpDataY, \
                                                                                            ExpDataError, NumberRanges, MinRange, MaxRange, \
                                                                                            NumberXColumns, NumberYColumns, LengthExpRange, \
                                                                                            MaxRangeNumber, NumberExpFiles, MaxLength, MaxColX, MaxColY, \
                                                                                            NumberParameter, plotflag, modelflag, MPIFlag, \
                                                                                            MAGIXrootDirectory, JobDir)
                ## write best parameter sets to files
                MCMCCounter = abs(MCMCCounter)
                for i in range(ParamSetCounter):
                    parameters = []
                    parameters.append(FinalParameterSet[i])
                    parameters.append(parameter_set_copy[1])
                    parameters.append(parameter_set_copy[2])
                    parameters.append(parameter_set_copy[3])
                    FitFunctionValues = FitFunctionAll[i]
                    Chi2Values = Chi2ValuesAll[i]
                    parameters[1] = numpy.zeros(NumberParameter)

                    # Debug:
                    # print "NumberParameter = ", NumberParameter


                    ## copy new parameter set
                    NewNumberParameterSet += 1
                    NewAllParameterSet.append(parameters)


                    ## define extension of the observation and the parameter xml-file if algorithm chain is chosed
                    if (NumberOfFitAlgorithms > 1):
                        ExpFilenameExtension = "MCMC__call_" + str(MCMCCounter)
                    else:
                        ExpFilenameExtension = "MCMC"
                    PlotFilenameExtension = ExpFilenameExtension
                    if (ParamSetCounter > 1):
                        ExpFilenameExtension += "__site_" + str(i + 1)
                    ExpFilenameExtension += ".out"
                    XMLFilenameExtension = ExpFilenameExtension


                    ## write results of fit process to files
                    parameter_set = copy.deepcopy(parameter_set_copy)
                    ok = StoreResults(ExpInputFormat, ClassLoadExp, printflag, NumberExpFiles, LengthExpRange, NumberYColumns, ExpDataRange_x, \
                                      FitFunctionValues, Chi2SaveFlag, Chi2Values, xml_file, NumberParameter, parameters, parameter_set, \
                                      ParameterValuesOriginal, FitParameterName, ExpFilenameExtension, XMLFilenameExtension, UnmodifiedNumberParameter)


                ## store best result
                AlgCounter = MCMCCounter
                parameter_set[0] = FinalParameterSet[0]
                if (ParamSetCounter > 1):
                    FitFunctionValues = FitFunctionAll
                    Chi2Values = Chi2ValuesAll
                else:
                    FitFunctionValues = FitFunctionAll[0]
                    Chi2Values = Chi2ValuesAll[0]
                ExpFilenameExtension = PlotFilenameExtension


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## call additional packages routines
            ##
            ## input parameters:
            ##
            elif (FitAlgorithm == "additionalpackages"):
                AdditionalPackagesCounter += 1
                if (NumberOfFitAlgorithms > 1):
                    AdditionalPackagesCounter = -AdditionalPackagesCounter


                ## call fortran routine for AdditionalPackages
                ParamSetCounter = 1
                NumberSites = ParamSetCounter
                parameter_set_copy = copy.deepcopy(parameter_set)
                parameter_set[1] = parameter_set_orig[1]
                calstatus, FitFunctionValues, Chi2Values, parameters = AdditionalPackages.start(SpecialAlgorithmSettings, GeneralAlgorithmSettings, \
                                                                                            printflagNum, LastAlgorithmNum, ParamSetCounter, chilm, \
                                                                                            numiter, AdditionalPackagesCounter, DeterminationChi2, \
                                                                                            PlotIteration, PlotType, fit_log, NumberInputFiles, \
                                                                                            NumberOutputFiles, ParallelizationFlag, JobID, \
                                                                                            MaxInputLines, MaxParameter, RenormalizedChi2, currentpath, \
                                                                                            CalculationMethod, xAxisLabel, yAxisLabel, zAxisLabel, \
                                                                                            PathStartScript, ExeCommandStartScript, parameter_set, \
                                                                                            ExpDataX, ExpDataY, ExpDataError, NumberRanges, MinRange, \
                                                                                            MaxRange, NumberXColumns, NumberYColumns, LengthExpRange, \
                                                                                            MaxRangeNumber, NumberExpFiles, MaxLength, MaxColX, MaxColY, \
                                                                                            NumberParameter, modelflag, MPIFlag, MAGIXrootDirectory, \
                                                                                            JobDir)
                ## copy new parameter set
                NewNumberParameterSet += 1
                NewAllParameterSet.append(parameters)


                ## define extension of the observation and the parameter xml-file if algorithm chain is chosed
                AdditionalPackagesCounter = abs(AdditionalPackagesCounter)
                if (NumberOfFitAlgorithms > 1):
                    ExpFilenameExtension = "AP__call_" + str(AdditionalPackagesCounter) + ".out"
                    XMLFilenameExtension = "AP__call_" + str(AdditionalPackagesCounter) + ".out"
                else:
                    ExpFilenameExtension = "AP.out"
                    XMLFilenameExtension = "AP.out"
                AlgCounter = AdditionalPackagesCounter


                ## write results of fit process to files
                parameter_set = copy.deepcopy(parameter_set_copy)
                ok = StoreResults(ExpInputFormat, ClassLoadExp, printflag, NumberExpFiles, LengthExpRange, NumberYColumns, ExpDataRange_x, \
                                  FitFunctionValues, Chi2SaveFlag, Chi2Values, xml_file, NumberParameter, parameters, parameter_set, \
                                  ParameterValuesOriginal, FitParameterName, ExpFilenameExtension, XMLFilenameExtension, UnmodifiedNumberParameter)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## for myXCLASS, write optimized parameter in molfit file as well
            ii = string.rfind(fit_log, "/")
            if (ii != -1):
                BaseDir = fit_log[:ii+1]
            else:
                BaseDir = "."
            listing = os.listdir(BaseDir)
            for files in listing:
                if (files.endswith(".out.input")):
                    NewFile = files.replace(".out.input", ".out.molfit")
                    if (NewFile.startswith("myNewXCLASS__") and parameter_file_orig.endswith(".molfit")):
                        InputFileOut = parameter_file_orig.replace(".molfit", "")
                        NewFile = NewFile.replace("myNewXCLASS__", InputFileOut + "__")
                        command_string = "mv " + BaseDir + files + " " + NewFile
                        os.system(command_string)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## plot model function and experimental data in one diagram
            if ((algorithm + 1) == NumberOfFitAlgorithms and (plotflag == "true" or plotflag == "saveonly" or plotflag == "quiet")):
                if (FitAlgorithm != "errorestim" and FitAlgorithm != "errorestim_ins"):     ## special settings for error estimation
                    ErrorModelFunctionValues = []
                    ErrorChi2Values = []
                import PlotData                                                             ## import package for plotting data
                PlotOption.append("true")
                PlotData.plot(PlotOption, ExpDataRange_x, ExpDataRange_y, ExpDataRange_Error, NumberExpFiles, LengthExpRange, NumberYColumns, \
                              FitFunctionValues, Chi2Values, fit_log, plotflag, ExpFilenameExtension, ErrorModelFunctionValues, ErrorChi2Values, \
                              NumberSites, parameter_file_orig)


        ## copy new parameter set to old parameter set
        NumberParameterSet = NewNumberParameterSet
        AllParameterSet = NewAllParameterSet


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## remove temp-directory with JobID in the temp-direcotry
    TempDirectory = os.environ.get("MAGIXTempDirectory")
    command_string = "rm -r " + TempDirectory.strip() + "/job_" + str(JobID)
    os.system(command_string)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define output paramters
    return calstatus, PlotOption, parameter_set, FitFunctionValues, Chi2SaveFlag, Chi2Values, LengthExpRange, NumberYColumns, ExpDataRange_x, \
           ExpDataRange_y, ExpDataRange_Error
##--------------------------------------------------------------------------------------------------------------------------------------------------------


