/*########################################################################################################################################################
 *
 *  get entries of table PartitionFunction
 *  Copyright (C) 2012 - 2014  Thomas Moeller
 *
 *  I. Physikalisches Institut, University of Cologne
 *
 *
 *
 *
 *
 *
 *
 *  input variables:     NumberMoleculePartFunc:     number of molecules
 *                       nTkpart:                    number of temperature columns in table partition function
 *                       queryString:                query string
 *                       dbName:                     path and name database file
 *
 *  output variables:    ok:                         status variable
 *                       lgQ:                        the log of the partition function
 *                       MolNamePartFunc:            names of the molecules in database
 *
 *
 *
 *  Versions of the program:
 *
 *  Who           When        What
 *  T. Moeller    31.05.2012  Initial version
 *  T. Moeller    22.01.2013  change from MySQL to sqlite3
 *
 *
 *
 *  License:
 *
 *    GNU GENERAL PUBLIC LICENSE
 *    Version 3, 29 June 2007
 *    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
 *
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
########################################################################################################################################################*/


/* make a connection to the database */
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "sqlite3.h"
#include <time.h>
#include <math.h>
#include <sys/times.h>

#define nameLength 40


#ifdef __cplusplus
extern "C"
#endif

void readpartfunc_(int *ok,
                   int nTkpart,                                                             // number of temperature columns in table partition function
                   int NumberMoleculePartFunc,                                              // number of entries
                   int NumberOfMolecules,                                                   // number of molecules which should be considered
                   int querylength,                                                         // length of query
                   char queryString1[], char queryString2[],                                // first and second part of query string
                   char dbName[],                                                           // name of database
                   char NameOfColumnName[],                                                 // name of column containing the name
                   char MoleculeNames[NumberOfMolecules][nameLength],                       // name of molecules which should be considered
                   double lgQ[NumberMoleculePartFunc][nTkpart],                             // logarithm of partition function
                   char MolNamePartFunc[NumberMoleculePartFunc][nameLength]                 // name of molecules (taken from database)
                  )
{
    sqlite3 *db;                                                                            // sqlite3 db struct
    sqlite3_stmt *stmt;
    char *ErrorMsg = NULL;
    int rc;
    unsigned int col;                                                                       // loop variable

    char queryString[querylength];                                                          // final query string
    char querycopy[querylength];                                                            // copy of query string
    int i;                                                                                  // index variable
    int j;                                                                                  // loop variable
    int debug;                                                                              // used for debugging
    double pow();                                                                           // used for power function
    double val;                                                                             // working variable

    debug = 0;                                                                              // set equal to 1 for debugging


    //----------------------------------------------------------------------------------------------------------------------------------------------------
    // construct final query string
    // SELECT * FROM ALMADB.RadTrans...

    // Debug:
    if (debug == 1){
        fprintf (stderr,"\n\n\nNumberMoleculePartFunc = %i\n", NumberMoleculePartFunc);
        fprintf (stderr,"querylength = %d\n", querylength);
        fprintf (stderr,"queryString1 = '%s'\n", queryString1);
        fprintf (stderr,"queryString2 = '%s'\n", queryString2);
        fprintf (stderr,"NumberOfMolecules = %i\n", NumberOfMolecules);
        fprintf (stderr,"dbName = '%s'\n", dbName);
        fprintf (stderr,"NameOfColumnName = '%s'\n", NameOfColumnName);
        fprintf (stderr,"\n\n");
        for (j = 0; j < NumberOfMolecules; j++)
        {
            fprintf (stderr,"j, MoleculeNames[j] = %i, '%s' \n", j, MoleculeNames[j]);
        }
        fprintf (stderr,"\n\n");
        for (j = 0; j < NumberMoleculePartFunc; j++)
        {
            fprintf (stderr,"j = %i \n", j);
            fprintf (stderr,"MolNamePartFunc[j] = '%s' \n", MolNamePartFunc[j]);
            fprintf (stderr,"lgQ[0][j] = %f \n\n", lgQ[0][j]);
        }
    }


    //----------------------------------------------------------------------------------------------------------------------------------------------------
    // connect to sqlite3 database
    rc = sqlite3_open(dbName, &db);
    if ( rc ){
        fprintf(stderr, "\n\n");
        fprintf(stderr, "subroutine numentr_sqlite3: Failed to connect to database: Error: %s\n", sqlite3_errmsg(db));
        fprintf (stderr,"\n\nqueryString = %s\n",queryString);
        *ok = 1;
        sqlite3_close(db);

        // Debug:
        if (debug == 1){fprintf (stderr,"rc = %i", rc);}

        /* check to see that the backend connection was successfully made */
    } else{


        // Debug:
        if (debug == 1){
            fprintf(stderr,"subroutine readpartfunc: Connection to database '%s' succeeded.\n", dbName);
            fprintf (stderr,"rc = %i\n", rc);
        }
        *ok = 0;
    }


    //----------------------------------------------------------------------------------------------------------------------------------------------------
    // collect number of entries for each molecule, even if a molecule occurs more than once
    i = 0;
    if (NumberOfMolecules > 0){


        //------------------------------------------------------------------------------------------------------------------------------------------------
        // loop over all molecules
        for (j = 0; j < NumberOfMolecules; j++){


            //--------------------------------------------------------------------------------------------------------------------------------------------
            // construct query string for each molecule
            sprintf(queryString, "%s WHERE (%s = %s", queryString1, NameOfColumnName, MoleculeNames[j]);
            sprintf(queryString, "%s) %s", queryString, queryString2);


            // Debug:
            if (debug == 1){fprintf (stderr,"\nfinal queryString = '%s'\n", queryString);}


            //--------------------------------------------------------------------------------------------------------------------------------------------
            // execute query
            rc = sqlite3_prepare_v2(db, queryString, -1, &stmt, NULL);
            if ( rc ){
                fprintf(stderr, "SQL error: %s\n", ErrorMsg);
                sqlite3_free(ErrorMsg);
                ErrorMsg=NULL;
                *ok = 1;
                sqlite3_close(db);

                // Debug:
                if (debug == 1){fprintf (stderr,"rc = %i\n", rc);}
            }


            // get number of columns
            int cols = sqlite3_column_count(stmt);                                          // Read the number of rows fetched

            // Debug:
            if (debug == 1){fprintf (stderr,"cols = %i\n", cols);}


            //--------------------------------------------------------------------------------------------------------------------------------------------
            // get number of entries
            while(1)
            {
                // fetch a row’s status
                rc = sqlite3_step(stmt);
                if (rc == SQLITE_ROW)
                {
                    // sqlite3_column_text returns a const void* , typecast it to const char*
                    for (col = 0; col < cols; col++)
                    {


                        //--------------------------------------------------------------------------------------------------------------------------------
                        // get name of molecule
                        if (col == 0)
                        {
                            const char *text = (const char*)sqlite3_column_text(stmt, col);
                            sprintf(MolNamePartFunc[i], "%s", text);

                            // Debug:
                            if (debug == 1){fprintf (stderr, "name of molecule = '%s' \n", MolNamePartFunc[i]);}


                        //--------------------------------------------------------------------------------------------------------------------------------
                        // get log of partition function
                        }else if (col > 0)
                        {
                            val = sqlite3_column_double(stmt, col);

                            if (debug == 1){fprintf (stderr, "i , col, val = %i, %i, %f \n", i, col, val);}

                            // val = pow(10., valtemp);
                            lgQ[i][col - 1] = val;

                            // Debug:
                            if (debug == 1){fprintf (stderr, "col - 1, val = %i, %f \n",col - 1, val);}
                        }
                    }


                    // increase counter
                    i++;
                }
                else if(rc == SQLITE_DONE)
                {
                    break;
                }
            }


            //--------------------------------------------------------------------------------------------------------------------------------------------
            // finalize
            sqlite3_finalize(stmt);
        }
    }


    //----------------------------------------------------------------------------------------------------------------------------------------------------
    // close connection to database
    sqlite3_close(db);                                                                      // close the database connection


    if (debug == 1){fprintf (stderr, "DONE \n");}
}

