!*********************************************************************************************************************************************************
!>  Module: Model
!>
!>
!>  This module contains subroutines for the myXCLASS program for application in MAGIX  (Non-GPU version)
!>  Copyright (C) 2009 - 2014  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine SlaveWork:                     wake up MPI slave threads and do something or fall asleep
!>      - subroutine SlaveLoop:                     manage subroutine for slave threads
!>      - subroutine BroadCastMPIVariables:         broadcast MPI variables
!>      - subroutine CheckMolNames:                 checks, if a molecule name is already known
!>      - subroutine strlen:                        subroutine to determine last non blank and non char(0) character of string st
!>      - subroutine GetmyXCLASSParameter:          reads parameter for myXCLASS from paramset variable
!>      - subroutine GetPartitionFunction:          gets partition function from sqlite3 database
!>      - subroutine GetTransitionParameters:       gets informations for transitions from sqlite3 database table linecat
!>      - subroutine ModelInit:                     initialize and broadcast module variables
!>      - subroutine PrepareInputFiles:             prepare input files for MPI call of external model programs using MPI_Spawn_Multiple function
!>      - subroutine ModelCalcChiFunctionLM:        calculates the chi2 values for the Levenberg-Marquard algorithm
!>      - subroutine ModelCalcChiFunctionGeneral:   calculates the chi2 values for several given parameter vector sets
!>      - subroutine ModelCalcChiFunction:          prepare call of subroutine 'ModelCalcCore'
!>      - subroutine ModelCalcCore:                 calculates the chi2 values for a given set of parameter vectors
!>      - subroutine ModelCalcSpectrum:             calculates the myXCLASS spectrum for a given parameter vector
!>      - subroutine ModelParamFree:                free memory used by variables of the Module Model
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    12.09.2014  Initial version
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!---------------------------------------------------------------------------------------------------------------------------------------------------------
Module Model
    !> declare variables which are used by functions and subroutines of this module

    use Variables
    use FunctionCalling

    implicit none


    !< MPI variables
    include 'mpif.h'
    integer :: ierr                                                                         !< MPI error variable
    integer :: myrank                                                                       !< current thread number
    integer :: nsize                                                                        !< MPI variables
    integer :: nproc_for_mpi                                                                !< MPI variables
    integer :: icode0                                                                       !< MPI variables
    integer :: ireq0                                                                        !< MPI variables
    integer :: itag0                                                                        !< MPI variables
    integer :: TotalNumParamVectors                                                         !< used for subroutine ModelCalcCore for exchanging data
    integer, dimension(MPI_STATUS_SIZE) :: istatus0                                         !< MPI variables
    real*8, allocatable, dimension(:) :: LocalChi2ValueVector                               !< used for subroutine ModelCalcCore for exchanging data
    real*8, allocatable, dimension(:, :) :: GlobalParamVectorSet                            !< used for subroutine ModelCalcCore for exchanging data
    character(len=10) :: ParallelizationMethod                                              !< string describing the parallelization method


    !< myXCLASS variables


    !< general variables
    integer :: ErrChannelIndex                                                              !< loop variable for error handling
    integer :: ErrChannel                                                                   !< current error channel
    integer, dimension(2) :: AllErrChannels = (/0, 2222/)                                   !< list of all error channels
    real*8 :: pi                                                                            !< pi
    real*8 :: ckms                                                                          !< speed of light in km/s
    real*8, parameter :: debug_freq = 554989.699e30                                         !< frequency for debugging
    logical :: UseRegistrationXMLFile                                                       !< use registation XML file


    !< variables for molfit file
    integer :: TotalNumberDataPoints                                                        !< total number of data points
    integer :: NumberMolecules                                                              !< number of molecules in the current molfits file
    integer :: TotalNumberComponents                                                        !< counter for total number of components of all molecules
    integer :: TotalNumberOfFrequencyRanges                                                 !< total number of frequency ranges
    integer :: TotalNumberOfMolecules                                                       !< total number of molecules including isotopologues
    integer, allocatable, dimension(:) :: CompMoleculeIndex                                 !< molecule index for each component
    integer, allocatable, dimension(:) :: SpectrumIndexForFreqRange                         !< store spectrum index for each frequency range
    integer, allocatable, dimension(:) :: NumberComponentsPerMolecule                       !< number of components per molecule
    integer, allocatable, dimension(:, :) :: ConversionTableMAGIXmyXCLASSParam              !< conversion table between myXCLASS and MAGIX parameter
    integer, allocatable, dimension(:, :) :: DataPointIndexFreqRange                        !< index of data point of first and last freq. in freq. range
    real*8, allocatable, dimension(:) :: StartFrequency                                     !< first frequency for each frequency range
    real*8, allocatable, dimension(:) :: EndFrequency                                       !< last frequency for each frequency range
    real*8, allocatable, dimension(:) :: StepFrequency                                      !< stepsize for each frequency range
    real*8, allocatable, dimension(:) :: TelescopeSize                                      !< size of telescope for each frequency range
    real*8, allocatable, dimension(:) :: BackgroundTemperatureRange                         !< T_Back for each frequency range
    real*8, allocatable, dimension(:) :: TemperatureSlopeRange                              !< T_Slope for each frequency range
    real*8, allocatable, dimension(:) :: HydrogenColumnDensityRange                         !< nH for each frequency range
    real*8, allocatable, dimension(:) :: DustBetaRange                                      !< beta for each frequency range
    real*8, allocatable, dimension(:) :: KappaRange                                         !< kappa for each frequency range
    real*8, allocatable, dimension(:, :) :: myXCLASSParameter                               !< array containing all molfit parameters for each component
    real*8, allocatable, dimension(:, :) :: ObservationalDataList                           !< list containing all observational data
    character(len=40), allocatable, dimension(:) :: MoleculeNames                           !< names of all molecules (including isotopologues)
    logical :: tbFlag                                                                       !< flag for global setting T_Back and T_Slope
    logical :: nHFlag                                                                       !< flag for global setting of nH, kappa and beta
    logical :: IsoFlag                                                                      !< flag indicating use of isotopologues


    !< variables for sqlite3 database
    integer, allocatable, dimension(:, :) :: NumEntriesRanges                               !< number of transitions for each freq. range and molecule
    character(len=128) :: NameOfPartFuncTable                                               !< name of table for partition function
    character(len=128) :: NameOfRadTransTable                                               !< name of table for rad. trans.
    character(len=4096) :: dbName                                                           !< name of database


    !< variables for partition function
    integer :: NumberOfTemperatures                                                         !< number of temperatures
    integer :: NumberMoleculePartFunc                                                       !< number of entries in table PartitionFunction
    integer :: Firsti, Lasti, stepi                                                         !< working variables for interpolation
    real*8 :: TempLow, TempHigh                                                             !< working variables for extrapolation of part. func.
    real*8, allocatable, dimension(:) :: TempPartFunc                                       !< temperatures for partition function
    real*8, allocatable, dimension(:, :) :: lgQ                                             !< lgQ entries of table PartitionFunction
    character(len=40) :: ColumnNameForNamePartFunc                                          !< name of column including the names of molecules
    character(len=40), allocatable, dimension(:) :: MolNamePartFunc                         !< molecule names of table PartitionFunction
    character(len=40), allocatable, dimension(:) :: ColumnNamesPartFunc                     !< column names of table PartitionFunction


    !< variables for rad-trans parameter
    integer :: TotalNumberOfTransitions                                                     !< number of entries in table RadTrans
    integer :: Column300KPartFunc                                                           !< number of column, which contains part. func. @ 300 K
    integer, allocatable, dimension(:, :, :) :: MolecularDataIndices                        !< start and end index for each molecule and frequency range
                                                                                            !< within MolecularData variable
    real*8, allocatable, dimension(:, :) :: MolecularData                                   !< array containing the molecular data for all molecules and
                                                                                            !< all freqency ranges
    character(len=40) :: ColumnNameForNameTransitions                                       !< name of column in table transitions for molecule name
    character(len=40) :: ColumnNameForFreqTransitions                                       !< name of column in table transitions for frequency
    character(len=40) :: ColumnNameForIntTransitions                                        !< name of column in table transitions for intensity
    character(len=40) :: ColumnNameForEinsteinATransitions                                  !< name of column in table transitions for Einstein A coef.
    character(len=40) :: ColumnNameForFreqErrTransitions                                    !< name of column in table transitions for error freq.
    character(len=40) :: ColumnNameForELowTransitions                                       !< name of column in table transitions for E_low
    character(len=40) :: ColumnNameForgUpTransitions                                        !< name of column in table transitions for Upper State degen.
    character(len=40), allocatable, dimension(:) :: MolNameRadTrans                         !< molecule names of table RadTrans


    !< variables for iso table file
    integer :: NumberOfIsomasters                                                           !< number of isomasters
    integer, allocatable, dimension(:, :) :: IsoNfitConversionTable                         !< iso ratios conversion table for free paramter index
    real*8, allocatable, dimension(:, :) :: IsoRatio                                        !< iso ratios for all isomasters
    real*8, allocatable, dimension(:, :) :: IsoRatioConversionTable                         !< table with iso ratios between iso master and molecule
    character(len=4096) :: IsoTableFileName                                                 !< path and file name of the iso file
    character(len=40), allocatable, dimension(:) :: IsoMolecule                             !< names of molecules for iso table
    character(len=40), allocatable, dimension(:) :: IsoMasters                              !< names of isomasters for iso table



    contains


        !*************************************************************************************************************************************************
        !> subroutine: SlaveWork
        !>
        !> wake up MPI slave threads and do something or fall asleep
        !>
        !>
        !> input variables:     SlaveControlVar:        command which is send to the slave threads
        !>
        !>
        !> output parameters:   none
        !>
        !>
        !> \author Stefan Borowski, Thomas Moeller
        !>
        !> \date 09.09.2014
        !>
        subroutine SlaveWork(SlaveControlVar)

            implicit none
            integer :: SlaveControlVar                                                      !< command which is send to the slave threads
            integer :: i                                                                    !< loop variable

            ! Debug:
            ! print*,'MASTER: SlaveControlVar = ', SlaveControlVar


            Do i = 1, (TotalNumberOfThreads - 1)
                call MPI_Send(SlaveControlVar, 1, MPI_INTEGER, i, 0, MPI_COMM_WORLD, ierr)
            end Do
            return
        end subroutine SlaveWork


        !>************************************************************************************************************************************************
        !> subroutine: SlaveLoop
        !>
        !> manage subroutine for slave threads
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 09.09.2014
        !>
        subroutine SlaveLoop

            implicit none
            integer :: i
            integer :: SlaveControlVar                                                      !< command which is send to the slave threads
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: deallocstatus                                                        !< status of the deallocation process
            logical :: go                                                                   !< working variable
            logical :: ModelFunctionFlag                                                    !< do we need model function values?


            !< start a loop which is executed by all slave threads
            SlaveControlVar = 0
            Do
                call MPI_Irecv(SlaveControlVar, 1, MPI_INTEGER, 0, 0, MPI_COMM_WORLD, ireq0, ierr)

                ! Debug:
                ! print*,'SLAVE: myrank, SlaveControlVar = ', myrank, SlaveControlVar


                !< wait for jobs
                Do
                    Do i = 1, 10000
                    end Do
                    call MPI_Test(ireq0, go, istatus0, ierr)
                    if (go) exit
                end Do


                !< exit slave loop if (SlaveControlVar == 1)
                if (SlaveControlVar == 1) then
                    call ModelParamFree(deallocstatus)                                      !< free memory before exit
                    exit


                !< initialize variables (SlaveControlVar == 2)
                elseif (SlaveControlVar == 2) then

                    ! Debug:
                    ! print*,'********************************************************************************************>slave, myrank = ', myrank

                    call BroadCastMPIVariables


                !< determine model functions variables (SlaveControlVar == 3)
                elseif (SlaveControlVar == 3) then


                    !< determine chi2 value and model function values for different parameter vectors
                    ma = parameternumber
                    colx = MaxColX
                    NumFile = NumberExpFiles
                    MaxL = MaxExpLength
                    MaxCol = MaxColY
                    call ModelCalcCore(NumberFreeParameter, TotalNumberDataPoints, ModelFunctionFlag, ma, colx, NumFile, MaxL, MaxCol)
                endif
            end Do

            return
        end subroutine SlaveLoop


        !>************************************************************************************************************************************************
        !> subroutine: BroadCastMPIVariables
        !>
        !> broadcast MPI variables
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 09.09.2014
        !>
        subroutine BroadCastMPIVariables


            implicit none
            integer :: nelements                                                            !< working variable: size of array to be broadcast
            integer :: dealloc_status, alloc_status                                         !< working variables for allocation/deallocation
            integer :: MaxNumberOfMolecules                                                 !< working variable: max. number of molecules (incl. isotop.)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< define some global variables
            DontStoreModelFuncValuesFlag = .true.                                           !< model function values are stored for each function call
            ParallezitionMethod = 2                                                         !< set parallelization method to MPI (=2)
            myrankOfMPI = myrank                                                            !< copy thread number to global variable
            call GetEnv('MAGIXTempDirectory', TempDirectory)                                !< get temp directory for current thread


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< broadcast variables between master and slave threads


            !< broadcast scalar variables for Module 'Variables'
            call MPI_Bcast(QualityLimit, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberExpFiles, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(modelnumber, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberLinesChi2, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(MaxColX, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(MaxColY, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(parameternumber, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberInputFiles, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberOutputFiles, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(JobID, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(MaxInputLines, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(MaxParameter, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ExternalThreadNumber, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(CurrentNumberLinesCalcReduction, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(CurrentExpFile, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(CurrentYColumn, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberFreeParameter, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(MaxExpLength, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(MaxNumberRanges, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ParallezitionMethod, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(TotalNumberOfThreads, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ParallelizationFlag, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ochisq, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(NaNReplaceString, 10, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(currentpath, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(StandardWorkingDirectory, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(TempDirectory, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(RenormalizedChi2, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(initflag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(printflag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(Gradientflag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(PlotIterationFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(WriteChi2Flag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(UseCalculationReduction, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(LastAlgorithmFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(AllInOneOutputFile, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(NumOutputFilesEqualNumExpFiles, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(DontStoreModelFuncValuesFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print '(11x, "Broadcast of scalar variables of Module >>Variables<< done!")'


            !< allocate memory for variables for Module 'Variables'
            if (myrank /= 0) then                                                           !< only slave threads (de)allocate memory
                if (allocated(lengthexpdata)) then
                    deallocate(lengthexpdata, NumberXColumns, NumberYColumns, NumberHeaderLines, OutputFileFormat, NumberRanges, ConverterInfit, &
                               ValueEmptyOutputFile, LSRAdjustement, chisqValues, BestSitesParamSet, paramset, AtOnceFunction, AtOnceGradient, &
                               FirstPointExpData, LastPointExpData, expdatax, expdatay, expdatae, CalculatedParameterSets, MinRange, MaxRange, &
                               BestSitesModelValues, BestSitesChi2Values, ModelFunction, FitParameterName, FitParameterValue, CharacterForComments, &
                               CharacterSeperatingColumns, ResamplingMethod, InterpolationMethod, OnlyYColumn, LSRAdjustementFitFlag, NormalizationFlag, &
                               ExpData_reversed_flag, stat = dealloc_status)
                    if (dealloc_status /= 0) then                                           !< is all ok?
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelInit:")')
                        write(logchannel,'(2x,"Can not deallocate variables for Module Variables for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,*)
                        write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine ModelInit:")'
                        print '(2x,"Can not deallocate variables for Module Variables for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("dealloc_status = ",I4)',dealloc_status
                        print '(" ")'
                        call MPI_Abort(MPI_COMM_WORLD, ierr)
                        stop
                    endif
                endif
                allocate(lengthexpdata(NumberExpFiles), NumberXColumns(NumberExpFiles), NumberYColumns(NumberExpFiles), &
                         NumberHeaderLines(NumberOutputFiles), OutputFileFormat(NumberOutputFiles), NumberRanges(NumberExpFiles), &
                         ConverterInfit(NumberFreeParameter), ValueEmptyOutputFile(NumberOutputFiles), LSRAdjustement(NumberOutputFiles), &
                         chisqValues(0:0), BestSitesParamSet(QualityLimit, NumberFreeParameter + 1), paramset(4, parameternumber), &
                         AtOnceFunction(0:0, NumberExpFiles, MaxColY, MaxExpLength), FirstPointExpData(NumberExpFiles, MaxColX), &
                         LastPointExpData(NumberExpFiles, MaxColX), expdatax(NumberExpFiles, MaxExpLength, MaxColX), &
                         expdatay(NumberExpFiles, MaxExpLength, MaxColY), expdatae(NumberExpFiles, MaxExpLength, MaxColY), &
                         CalculatedParameterSets(0:ParallelizationFlag, CurrentNumberLinesCalcReduction, NumberFreeParameter + 1), &
                         MinRange(NumberExpFiles, MaxNumberRanges, MaxColX), MaxRange(NumberExpFiles, MaxNumberRanges, MaxColX), &
                         BestSitesModelValues(QualityLimit, NumberExpFiles, MaxExpLength, MaxColY), &
                         BestSitesChi2Values(QualityLimit, NumberExpFiles, MaxExpLength, MaxColY), &
                         ModelFunction(1, NumberExpFiles, MaxColY, MaxExpLength), FitParameterName(parameternumber), FitParameterValue(parameternumber), &
                         CharacterForComments(NumberOutputFiles), CharacterSeperatingColumns(NumberOutputFiles), ResamplingMethod(NumberOutputFiles), &
                         InterpolationMethod(NumberOutputFiles), OnlyYColumn(NumberOutputFiles), LSRAdjustementFitFlag(NumberOutputFiles), &
                         NormalizationFlag(NumberOutputFiles), ExpData_reversed_flag(NumberExpFiles), stat = alloc_status)
                if (alloc_status /= 0) then                                                 !< is all ok?
                    write(logchannel,'(" ")')
                    write(logchannel,'("Error in subroutine ModelInit:")')
                    write(logchannel,'(2x,"Can not allocate variables for Module Variables for thread = ", I5, ".")') myrank
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,'(" ")')
                    write(logchannel,'("alloc_status = ",I4)') alloc_status
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine ModelInit:")'
                    print '(2x,"Can not allocate variables for Module Variables for thread = ", I5, ".")', myrank
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("alloc_status = ",I4)',alloc_status
                    print '(" ")'
                    call MPI_Abort(MPI_COMM_WORLD, ierr)
                    stop
                endif
                lengthexpdata = 0
                NumberXColumns = 0
                NumberYColumns = 0
                NumberHeaderLines = 0
                OutputFileFormat = 0
                NumberRanges = 0
                ConverterInfit = 0
                ValueEmptyOutputFile = 0.d0
                LSRAdjustement = 0.d0
                chisqValues = 0.d0
                BestSitesParamSet = 0.d0
                paramset = 0.d0
                AtOnceFunction = 0.d0
                FirstPointExpData = 0.d0
                LastPointExpData = 0.d0
                expdatax = 0.d0
                expdatay = 0.d0
                expdatae = 0.d0
                CalculatedParameterSets = 0.d0
                MinRange = 0.d0
                MaxRange = 0.d0
                BestSitesModelValues = 0.d0
                BestSitesChi2Values = 0.d0
                ModelFunction = 0.d0
                FitParameterName = ""
                FitParameterValue = ""
                CharacterForComments = ""
                CharacterSeperatingColumns = ""
                ResamplingMethod = ""
                InterpolationMethod = ""
                OnlyYColumn = .false.
                LSRAdjustementFitFlag = .false.
                NormalizationFlag = .false.
                ExpData_reversed_flag = .false.
            endif
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print '(11x, "Allocation of variables of Module >>Variables<< done!")'


            !< broadcast scalar variables for Module 'Variables'
            !< exclude variables: AtOnceFunction, BestSitesModelValues, BestSitesChi2Values, ModelFunction
            nelements = size(lengthexpdata)
            call MPI_Bcast(lengthexpdata, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(NumberXColumns)
            call MPI_Bcast(NumberXColumns, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(NumberYColumns)
            call MPI_Bcast(NumberYColumns, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(NumberHeaderLines)
            call MPI_Bcast(NumberHeaderLines, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(OutputFileFormat)
            call MPI_Bcast(OutputFileFormat, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(NumberRanges)
            call MPI_Bcast(NumberRanges, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(ConverterInfit)
            call MPI_Bcast(ConverterInfit, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(ValueEmptyOutputFile)
            call MPI_Bcast(ValueEmptyOutputFile, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(LSRAdjustement)
            call MPI_Bcast(LSRAdjustement, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(chisqValues)
            call MPI_Bcast(chisqValues, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(BestSitesParamSet)
            call MPI_Bcast(BestSitesParamSet, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(paramset)
            call MPI_Bcast(paramset, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(FirstPointExpData)
            call MPI_Bcast(FirstPointExpData, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(LastPointExpData)
            call MPI_Bcast(LastPointExpData, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(expdatax)
            call MPI_Bcast(expdatax, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(expdatay)
            call MPI_Bcast(expdatay, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(expdatae)
            call MPI_Bcast(expdatae, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(MinRange)
            call MPI_Bcast(MinRange, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(MaxRange)
            call MPI_Bcast(MaxRange, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(FitParameterName) * 512
            call MPI_Bcast(FitParameterName, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(FitParameterValue) * 512
            call MPI_Bcast(FitParameterValue, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(CharacterForComments) * 512
            call MPI_Bcast(CharacterForComments, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(CharacterSeperatingColumns) * 512
            call MPI_Bcast(CharacterSeperatingColumns, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(ResamplingMethod) * 512
            call MPI_Bcast(ResamplingMethod, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(InterpolationMethod) * 512
            call MPI_Bcast(InterpolationMethod, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(OnlyYColumn)
            call MPI_Bcast(OnlyYColumn, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(LSRAdjustementFitFlag)
            call MPI_Bcast(LSRAdjustementFitFlag, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(NormalizationFlag)
            call MPI_Bcast(NormalizationFlag, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(ExpData_reversed_flag)
            call MPI_Bcast(ExpData_reversed_flag, nelements, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print '(11x, "Broadcast of array variables of Module >>Variables<< done!")'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< broadcast scalar variables for Module 'FunctionCalling'
            call MPI_Bcast(NumberOfUsedThreads, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberLinesOutput, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberColumnsBegin, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberColumnsEnd, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(DetChi2, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(idum, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(GradientMethod, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(GradientVariationValue, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(NaNReplaceString, 10, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(PlotPID, 100, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(CalculationMethod, 20, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(PathStartScript, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(ExeCommandStartScript, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(CommandWordOutput, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(InputDataPath, 8192, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_BCAST(debugflag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print '(11x, "Broadcast of scalar variables of Module >>FunctionCalling<< done!")'


            !< allocate memory for variables for Module 'FunctionCalling'
            if (myrank /= 0) then                                                           !< only slave threads (de)allocate memory
                if (allocated(NumberParamPerLine)) then
                    deallocate(NumberParamPerLine, ParameterName, ParameterFormat, LeadingString, TrailingString, ParamVisible, FitFktInput, &
                               FitFktOutput, stat = dealloc_status)
                    if (dealloc_status /= 0) then                                           !< is all ok?
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelInit:")')
                        write(logchannel,'(2x,"Can not deallocate variables for Module FunctionCalling for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,*)
                        write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine FunctionCalling:")'
                        print '(2x,"Can not deallocate variables for Module FunctionCalling for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("dealloc_status = ",I4)',dealloc_status
                        print '(" ")'
                        call MPI_Abort(MPI_COMM_WORLD, ierr)
                        stop
                    endif
                endif
                allocate(NumberParamPerLine(NumberInputFiles, MaxInputLines), ParameterName(NumberInputFiles, MaxInputLines, MaxParameter), &
                         ParameterFormat(0:NumberInputFiles, 0:MaxInputLines, 0:MaxParameter), &
                         LeadingString(NumberInputFiles, MaxInputLines, MaxParameter), TrailingString(NumberInputFiles, MaxInputLines, MaxParameter), &
                         ParamVisible(NumberInputFiles, MaxInputLines, MaxParameter), FitFktInput(NumberInputFiles), FitFktOutput(NumberOutputFiles), &
                         stat = alloc_status)
                if (alloc_status /= 0) then                                                 !< is all ok?
                    write(logchannel,'(" ")')
                    write(logchannel,'("Error in subroutine ModelInit:")')
                    write(logchannel,'(2x,"Can not allocate variables for Module FunctionCalling for thread = ", I5, ".")') myrank
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,'(" ")')
                    write(logchannel,'("alloc_status = ",I4)') alloc_status
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine ModelInit:")'
                    print '(2x,"Can not allocate variables for Module FunctionCalling for thread = ", I5, ".")', myrank
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("alloc_status = ",I4)',alloc_status
                    print '(" ")'
                    call MPI_Abort(MPI_COMM_WORLD, ierr)
                    stop
                endif
                NumberParamPerLine = 0
                ParameterName = ""
                ParameterFormat = ""
                LeadingString = ""
                TrailingString = ""
                ParamVisible = ""
                FitFktInput = ""
                FitFktOutput = ""
            endif
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print '(11x, "Allocation of variables of Module >>FunctionCalling<< done!")'


            !< broadcast scalar variables for Module 'FunctionCalling'
            !< exclude variables: 
            nelements = size(NumberParamPerLine)
            call MPI_Bcast(NumberParamPerLine, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(ParameterName) * 8192
            call MPI_BCAST(ParameterName, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(ParameterFormat) * 10
            call MPI_BCAST(ParameterFormat, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(LeadingString) * 8192
            call MPI_BCAST(LeadingString, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(TrailingString) * 8192
            call MPI_BCAST(TrailingString, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(ParamVisible) * 8192
            call MPI_BCAST(ParamVisible, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(FitFktInput) * 8192
            call MPI_BCAST(FitFktInput, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(FitFktOutput) * 8192
            call MPI_BCAST(FitFktOutput, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print '(11x, "Broadcast of array variables of Module >>FunctionCalling<< done!")'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< broadcast scalar variables for Module 'Model'
            call MPI_Bcast(ErrChannelIndex, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ErrChannel, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(TotalNumberDataPoints, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberMolecules, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(TotalNumberComponents, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(TotalNumberOfFrequencyRanges, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(TotalNumberOfMolecules, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberOfTemperatures, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberMoleculePartFunc, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(Firsti, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(Lasti, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(stepi, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(TotalNumberOfTransitions, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(Column300KPartFunc, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NumberOfIsomasters, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(pi, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ckms, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(TempLow, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr) 
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(TempHigh, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ColumnNameForNamePartFunc, 40, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ColumnNameForNameTransitions, 40, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ColumnNameForFreqTransitions, 40, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ColumnNameForIntTransitions, 40, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ColumnNameForEinsteinATransitions, 40, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ColumnNameForFreqErrTransitions, 40, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ColumnNameForELowTransitions, 40, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(ColumnNameForgUpTransitions, 40, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NameOfPartFuncTable, 128, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(NameOfRadTransTable, 128, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(dbName, 4096, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(IsoTableFileName, 4096, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(UseRegistrationXMLFile, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(tbFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(nHFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(IsoFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print '(11x, "Broadcast of scalar variables of Module >>Model<< done!")'


            !< allocate memory for variables for Module 'Model'
            if (myrank /= 0) then                                                           !< only slave threads (de)allocate memory
                if (allocated(CompMoleculeIndex)) then
                    deallocate(CompMoleculeIndex, SpectrumIndexForFreqRange, NumberComponentsPerMolecule, IsoNfitConversionTable, &
                               ConversionTableMAGIXmyXCLASSParam, DataPointIndexFreqRange, NumEntriesRanges, MolecularDataIndices, TempPartFunc, &
                               StartFrequency, EndFrequency, StepFrequency, TelescopeSize, BackgroundTemperatureRange, TemperatureSlopeRange, &
                               HydrogenColumnDensityRange, DustBetaRange, KappaRange, IsoRatio, IsoRatioConversionTable, MolecularData, lgQ, &
                               myXCLASSParameter, ObservationalDataList, MolNameRadTrans, MolNamePartFunc, ColumnNamesPartFunc, MoleculeNames, &
                               IsoMolecule, IsoMasters, stat = dealloc_status)
                    if (dealloc_status /= 0) then                                           !< is all ok?
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelInit:")')
                        write(logchannel,'(2x,"Can not deallocate variables for Module Model for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,*)
                        write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine FunctionCalling:")'
                        print '(2x,"Can not deallocate variables for Module Model for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("dealloc_status = ",I4)',dealloc_status
                        print '(" ")'
                        call MPI_Abort(MPI_COMM_WORLD, ierr)
                        stop
                    endif
                endif
                MaxNumberOfMolecules = NumberMolecules + 2 * NumberOfIsomasters
                allocate(CompMoleculeIndex(TotalNumberComponents), SpectrumIndexForFreqRange(TotalNumberOfFrequencyRanges), &
                         NumberComponentsPerMolecule(0:TotalNumberOfMolecules), IsoNfitConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules), &
                         ConversionTableMAGIXmyXCLASSParam(NumberFreeParameter, 2), DataPointIndexFreqRange(TotalNumberOfFrequencyRanges, 2), &
                         NumEntriesRanges(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules), &
                         MolecularDataIndices(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2), TempPartFunc(NumberOfTemperatures), &
                         StartFrequency(TotalNumberOfFrequencyRanges), EndFrequency(TotalNumberOfFrequencyRanges), &
                         StepFrequency(TotalNumberOfFrequencyRanges), TelescopeSize(TotalNumberOfFrequencyRanges), &
                         BackgroundTemperatureRange(TotalNumberOfFrequencyRanges), TemperatureSlopeRange(TotalNumberOfFrequencyRanges), &
                         HydrogenColumnDensityRange(TotalNumberOfFrequencyRanges), DustBetaRange(TotalNumberOfFrequencyRanges), &
                         KappaRange(TotalNumberOfFrequencyRanges), IsoRatioConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules), &
                         MolecularData(TotalNumberOfTransitions, 4), MolNameRadTrans(TotalNumberOfTransitions), &
                         lgQ(NumberOfTemperatures, NumberMoleculePartFunc), MolNamePartFunc(NumberMoleculePartFunc), &
                         myXCLASSParameter(11, TotalNumberComponents), ObservationalDataList(TotalNumberDataPoints, 3), &
                         ColumnNamesPartFunc(NumberOfTemperatures), MoleculeNames(MaxNumberOfMolecules), stat = alloc_status)
                if (alloc_status /= 0) then                                                 !< is all ok?
                    write(logchannel,'(" ")')
                    write(logchannel,'("Error in subroutine ModelInit:")')
                    write(logchannel,'(2x,"Can not allocate variables for Module Model for thread = ", I5, ".")') myrank
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,'(" ")')
                    write(logchannel,'("alloc_status = ",I4)') alloc_status
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine ModelInit:")'
                    print '(2x,"Can not allocate variables for Module Model for thread = ", I5, ".")', myrank
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("alloc_status = ",I4)',alloc_status
                    print '(" ")'
                    call MPI_Abort(MPI_COMM_WORLD, ierr)
                    stop
                endif
                CompMoleculeIndex = 0
                SpectrumIndexForFreqRange = 0
                NumberComponentsPerMolecule = 0
                IsoNfitConversionTable = 0
                ConversionTableMAGIXmyXCLASSParam = 0
                DataPointIndexFreqRange = 0
                NumEntriesRanges = 0
                MolecularDataIndices = 0
                TempPartFunc = 0.d0
                StartFrequency = 0.d0
                EndFrequency = 0.d0
                StepFrequency = 0.d0
                TelescopeSize = 0.d0
                BackgroundTemperatureRange = 0.d0
                TemperatureSlopeRange = 0.d0
                HydrogenColumnDensityRange = 0.d0
                DustBetaRange = 0.d0
                KappaRange = 0.d0
                IsoRatioConversionTable = 0.d0
                MolecularData = 0.d0
                lgQ = 0.d0
                myXCLASSParameter = 0.d0
                ObservationalDataList = 0.d0
                MolNameRadTrans = ""
                MolNamePartFunc = ""
                ColumnNamesPartFunc = ""
                MoleculeNames = ""
            endif
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print '(11x, "Allocation of variables of Module >>Model<< done!")'


            nelements = size(CompMoleculeIndex)
            call MPI_Bcast(CompMoleculeIndex, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            nelements = size(SpectrumIndexForFreqRange)
            call MPI_Bcast(SpectrumIndexForFreqRange, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            nelements = size(NumberComponentsPerMolecule)
            call MPI_Bcast(NumberComponentsPerMolecule, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            nelements = size(IsoNfitConversionTable)
            call MPI_Bcast(IsoNfitConversionTable, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            nelements = size(ConversionTableMAGIXmyXCLASSParam)
            call MPI_Bcast(ConversionTableMAGIXmyXCLASSParam, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            nelements = size(DataPointIndexFreqRange)
            call MPI_Bcast(DataPointIndexFreqRange, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            nelements = size(NumEntriesRanges)
            call MPI_Bcast(NumEntriesRanges, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            nelements = size(MolecularDataIndices)
            call MPI_Bcast(MolecularDataIndices, nelements, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            nelements = size(TempPartFunc)
            call MPI_Bcast(TempPartFunc, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(StartFrequency)
            call MPI_Bcast(StartFrequency, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(EndFrequency)
            call MPI_Bcast(EndFrequency, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(StepFrequency)
            call MPI_Bcast(StepFrequency, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(TelescopeSize)
            call MPI_Bcast(TelescopeSize, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(BackgroundTemperatureRange)
            call MPI_Bcast(BackgroundTemperatureRange, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(TemperatureSlopeRange)
            call MPI_Bcast(TemperatureSlopeRange, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(HydrogenColumnDensityRange)
            call MPI_Bcast(HydrogenColumnDensityRange, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(DustBetaRange)
            call MPI_Bcast(DustBetaRange, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(KappaRange)
            call MPI_Bcast(KappaRange, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(IsoRatioConversionTable)
            call MPI_Bcast(IsoRatioConversionTable, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(MolecularData)
            call MPI_Bcast(MolecularData, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(lgQ)
            call MPI_Bcast(lgQ, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(myXCLASSParameter)
            call MPI_Bcast(myXCLASSParameter, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(ObservationalDataList)
            call MPI_Bcast(ObservationalDataList, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(MolNameRadTrans) * 40
            call MPI_Bcast(MolNameRadTrans, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(MolNamePartFunc) * 40
            call MPI_Bcast(MolNamePartFunc, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(ColumnNamesPartFunc) * 40
            call MPI_Bcast(ColumnNamesPartFunc, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            nelements = size(MoleculeNames) * 40
            call MPI_Bcast(MoleculeNames, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            ! print '(11x, "Broadcast of array variables of Module >>Model<< done!")'


            if (IsoFlag) then
                if (myrank /= 0) then                                                       !< only slave threads (de)allocate memory
                    if (allocated(IsoMolecule)) then
                        deallocate(IsoMolecule, IsoMasters, IsoRatio, stat = dealloc_status)
                        if (dealloc_status /= 0) then                                           !< is all ok?
                            write(logchannel,*)
                            write(logchannel,'("Error in subroutine ModelInit:")')
                            write(logchannel,'(2x,"Can not deallocate variables IsoMolecule, IsoMasters, and IsoRatio for thread = ", I5, ".")') myrank
                            write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                            write(logchannel,*)
                            write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                            write(logchannel,'(" ")')
                            write(logchannel,'("Program aborted!")')
                            close(logchannel)

                            print '(" ")'
                            print '("Error in subroutine FunctionCalling:")'
                            print '(2x,"Can not deallocate variables IsoMolecule, IsoMasters, and IsoRatio for thread = ", I5, ".")', myrank
                            print '(2x,"Please close all other programs and restart the program!")'
                            print '(" ")'
                            print '("dealloc_status = ",I4)', dealloc_status
                            print '(" ")'
                            call MPI_Abort(MPI_COMM_WORLD, ierr)
                            stop
                        endif
                    endif
                    allocate(IsoMolecule(NumberOfIsomasters), IsoRatio(NumberOfIsomasters, 2), IsoMasters(NumberOfIsomasters), stat = alloc_status)
                    if (alloc_status /= 0) then                                                 !< is all ok?
                        write(logchannel,'(" ")')
                        write(logchannel,'("Error in subroutine ModelInit:")')
                        write(logchannel,'(2x,"Can not allocate variables IsoMolecule, IsoMasters, and IsoRatio for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,'(" ")')
                        write(logchannel,'("alloc_status = ",I4)') alloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine ModelInit:")'
                        print '(2x,"Can not allocate variables IsoMolecule, IsoMasters, and IsoRatio for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("alloc_status = ",I4)', alloc_status
                        print '(" ")'
                        call MPI_Abort(MPI_COMM_WORLD, ierr)
                        stop
                    endif
                    IsoMolecule = ""
                    IsoMasters = ""
                    IsoRatio = 0.d0
                endif
                nelements = size(IsoMolecule) * 40
                call MPI_Bcast(IsoMolecule, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
                call MPI_BARRIER(MPI_COMM_WORLD, ierr)
                nelements = size(IsoMasters) * 40
                call MPI_Bcast(IsoMasters, nelements, MPI_CHARACTER, 0, MPI_COMM_WORLD, ierr)
                call MPI_BARRIER(MPI_COMM_WORLD, ierr)
                nelements = size(IsoRatio)
                call MPI_Bcast(IsoRatio, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
                call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            endif
            return
        end subroutine BroadCastMPIVariables


        !>************************************************************************************************************************************************
        !> subroutine: CheckMolNames
        !>
        !> checks, if a molecule name is already known
        !>
        !>
        !> input variables:     TotalNumberOfMoleculesCopy: total number of all already known molecules
        !>                      CurrMol:                current molecule
        !>
        !> output variables:    AlreadyIncludedFlag:    flag indicating if current molecule is already known or not
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 27.07.2014
        !>
        subroutine CheckMolNames(AlreadyIncludedFlag, TotalNumberOfMoleculesCopy, CurrMol)

            implicit none
            integer :: i                                                                    !< loop variable
            integer :: TotalNumberOfMoleculesCopy                                           !< total number of all already known molecules
            character(len=40) :: CurrMol                                                    !< current molecule
            logical :: AlreadyIncludedFlag                                                  !< flag indicating if current molecule is already known or not


            AlreadyIncludedFlag = .false.                                                   !< initialize return parameter
            Do i = 1, TotalNumberOfMoleculesCopy
                if (trim(adjustl(MoleculeNames(i))) == trim(adjustl(CurrMol))) then
                    AlreadyIncludedFlag = .true.
                    exit
                endif
            end Do
            return
        end subroutine CheckMolNames


        !>------------------------------------------------------------------------------------------------------------------------------------------------
        !> subroutine to determine last non blank and non char(0) character of string st
        !>
        !>
        !> input variables:     st:                 string which has to be analyzed
        !>
        !> output variables:    val:                position of last non blank and non char(0) character of string st
        !>
        !> Who           When            What
        !> P. Schilke    08/12/2011      initial version
        !>
        subroutine strlen(val, st)

            implicit none
            integer :: i                                                                    !< working variable
            integer :: val                                                                  !< return value
            character :: st*(*)                                                             !< string which has to be analyzed

            ! Debug:
            ! print*,'st = ', st

            i = len(st)
            Do While ((st(i:i) == ' ' .or. st(i:i) == char(0)) .and. i > 1)
                i = i - 1
            end Do
            val = i
            return
        end subroutine strlen


        !>************************************************************************************************************************************************
        !> subroutine: GetmyXCLASSParameter
        !>
        !> read parameter for myXCLASS from paramset variable
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 27.07.2014
        !>
        subroutine GetmyXCLASSParameter

            implicit none
            integer :: i, j, k, l, m                                                        !< loop variables
            integer :: IndexComp                                                            !< index of current component
            integer :: CurrComp                                                             !< counts the current component
            integer :: IndexMolecule                                                        !< index of current molecule
            integer :: IndexFreqRange                                                       !< index of current frequency range
            integer :: IndexSpectrum                                                        !< index of current spectrum
            integer :: IndexCurrIso                                                         !< index of current iso ratio
            integer :: LastIndexFreqRange                                                   !< working variable
            integer :: FirstIndex, LastIndex                                                !< working variables for freq. range index determination
            integer :: MaxNumberOfMolecules                                                 !< working variable: max. number of molecules (incl. isotop.)
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            character(len=40) :: CurrNameMolecule                                           !< working variable: name of current molecule
            character(len=4096) :: ParamName                                                !< working variable: current name of variable without
                                                                                            !< extension
            logical :: AlreadyIncludedFlag                                                  !< working variable: indicates if current molecule name is
                                                                                            !< already known

            !< print what you do
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(9x, "Using optimized version of MAGIX for myXCLASS (MPI):")'
                print '(" ")'
                print '(" ")'
                print '(9x, "Initialize myXCLASS program:")'
                print '(" ")'
                print '(" ")'
                print '(9x, "Reading parameter for myXCLASS from paramset variable .. ", $)'
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< analyze parameter list, i.e. get all parameters wich are given in the molfit, iso-ratio, and experimental xml files
            !<      - get name of molecules and the corresponding number of components
            !<      - get parameters for each component
            !<      - get iso master, iso molecule, and iso ratios
            !<      - get frequency ranges, background temperatures and slopes, nH densities, kappa, beta, telescope sizes, and database file


            !< get general parameters from paramset variable
            dbName = "../../../Database/cdms_sqlite.db"                                     !< set default value for db file
            NumberMolecules = 0                                                             !< reset counter for total number of molecules
            TotalNumberOfFrequencyRanges = 0                                                !< total number of frequency ranges
            IsoFlag = .false.
            IndexCurrIso = 0
            Do i = 1, parameternumber                                                       !< loop over all parameter

                ! Debug:
                ! print*,"FitParameterName(i) = ", trim(adjustl(FitParameterName(i)))


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif

                ! Debug:
                ! print*,"ParamName = ", trim(adjustl(ParamName))
                ! print*,"  paramset(1, i) = ", paramset(1, i)
                ! print*,"  FitParameterValue(i) = trim(adjustl(FitParameterValue(i)))


                !< get number of molecules (without iso molecules)
                if (ParamName == "Number_Molecules") then
                    NumberMolecules = int(paramset(1, i))


                !< get tbFlag
                elseif (ParamName == "tbFlag") then
                    if (trim(adjustl(FitParameterValue(i))) == "t" .or. trim(adjustl(FitParameterValue(i))) == "T" &
                        .or. trim(adjustl(FitParameterValue(i))) == "true" .or. trim(adjustl(FitParameterValue(i))) == "True" &
                        .or. trim(adjustl(FitParameterValue(i))) == "TRUE" .or. trim(adjustl(FitParameterValue(i))) == "y" &
                        .or. trim(adjustl(FitParameterValue(i))) == "Y" .or. trim(adjustl(FitParameterValue(i))) == "yes" &
                        .or. trim(adjustl(FitParameterValue(i))) == "YES" .or. trim(adjustl(FitParameterValue(i))) == "Yes") then
                        tbFlag = .false.
                    else
                        tbFlag = .true.
                    endif


                !< get nHFlag
                elseif (ParamName == "nHFlag") then
                    if (trim(adjustl(FitParameterValue(i))) == "t" .or. trim(adjustl(FitParameterValue(i))) == "T" &
                        .or. trim(adjustl(FitParameterValue(i))) == "true" .or. trim(adjustl(FitParameterValue(i))) == "True" &
                        .or. trim(adjustl(FitParameterValue(i))) == "TRUE" .or. trim(adjustl(FitParameterValue(i))) == "y" &
                        .or. trim(adjustl(FitParameterValue(i))) == "Y" .or. trim(adjustl(FitParameterValue(i))) == "yes" &
                        .or. trim(adjustl(FitParameterValue(i))) == "YES" .or. trim(adjustl(FitParameterValue(i))) == "Yes") then
                        nHFlag = .false.
                    else
                        nHFlag = .true.
                    endif


                !< get number of frequency ranges
                elseif (ParamName == "MAGIXImportStartFrequency") then
                    TotalNumberOfFrequencyRanges = TotalNumberOfFrequencyRanges + 1


                !< get iso flag
                elseif (ParamName == "NumberOfISOLines") then
                    if (paramset(1, i) > 0) then
                        NumberOfIsomasters = int(paramset(1, i))
                        IsoFlag = .true.


                        !< allocate memory for myXCLASSParameter variable
                        if (allocated(IsoMolecule)) then
                            deallocate(IsoMolecule, IsoMasters, IsoRatio, stat = deallocstatus)
                            if (deallocstatus /= 0) then
                                Do ErrChannel = 6, 6
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                                    write(ErrChannel, '(3x,"Can not deallocate variables myXCLASSParameter, CompMoleculeIndex etc.!")')
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(3x,"Please restart the program!")')
                                end Do
                                stop 'Program aborted!'
                            endif
                        endif
                        allocate(IsoMolecule(NumberOfIsomasters), IsoMasters(NumberOfIsomasters), IsoRatio(NumberOfIsomasters, 2), stat = allocstatus)
                        if (allocstatus /= 0) then
                            Do ErrChannel = 6, 6
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                                write(ErrChannel, '(3x,"Can not allocate variables myXCLASSParameter, CompMoleculeIndex etc.!")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Please restart the program!")')
                            end Do
                            stop 'Program aborted!'
                        endif
                        IsoMolecule = ""
                        IsoMasters = ""
                        IsoRatio = 0.d0
                        IndexCurrIso = 0
                    endif


                !< get IsoMolecule
                elseif (ParamName == "IsoMolecule" .and. IsoFlag) then
                    IndexCurrIso = IndexCurrIso + 1
                    IsoMolecule(IndexCurrIso) = trim(adjustl(FitParameterValue(i)))


                !< get IsoMasters
                elseif (ParamName == "IsoMasters" .and. IsoFlag) then
                    IsoMasters(IndexCurrIso) = trim(adjustl(FitParameterValue(i)))


                !< get IsoRatio
                elseif (ParamName == "IsoRatio" .and. IsoFlag) then
                    IsoRatio(IndexCurrIso, 1) = paramset(1, i)                              !< store the value of the iso ratio
                    if (paramset(2, i) == 1) then
                        IsoRatio(IndexCurrIso, 2) = sum(paramset(2, 1:i))                   !< store the free parameter index for this iso ratio param.
                    else
                        IsoRatio(IndexCurrIso, 2) = 0
                    endif


                !< get path and name of sqlite3 database file
                elseif (ParamName == "MAGIXImportdbFilename") then
                    if (trim(adjustl(FitParameterValue(i))) /= "") then
                        dbName = trim(adjustl(FitParameterValue(i)))
                    endif
                endif
            end Do

            ! Debug:
            !    print*,"NumberMolecules = ", NumberMolecules
            !    print*,"tbFlag = ", tbFlag
            !    print*,"nHFlag = ", nHFlag
            !    print*,"TotalNumberOfFrequencyRanges = ", TotalNumberOfFrequencyRanges
            !    print*,"dbName = ", trim(adjustl(dbName))
            !    print*,"IsoFlag = ", IsoFlag
            !    if (IsoFlag) then
            !        print*,"NumberOfIsomasters = ", NumberOfIsomasters
            !        Do i = 1, NumberOfIsomasters
            !            print*,"IsoMolecule(i) = ", trim(adjustl(IsoMolecule(i)))
            !            print*,"IsoMasters(i) = ", trim(adjustl(IsoMasters(i)))
            !            print*,"IsoRatio(i, :) = ", IsoRatio(i, :)
            !        end Do
            !    endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< allocate memory for myXCLASSParameter variable
            if (allocated(StartFrequency)) then
                deallocate(StartFrequency, EndFrequency, StepFrequency, TelescopeSize, BackgroundTemperatureRange, &
                           TemperatureSlopeRange, HydrogenColumnDensityRange, DustBetaRange, KappaRange, SpectrumIndexForFreqRange, &
                           ObservationalDataList, DataPointIndexFreqRange, MoleculeNames, ConversionTableMAGIXmyXCLASSParam, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables StartFrequency, EndFrequency etc.!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            TotalNumberDataPoints = sum(lengthexpdata(1:NumberExpFiles))
            MaxNumberOfMolecules = NumberMolecules + 2 * NumberOfIsomasters
            allocate(StartFrequency(TotalNumberOfFrequencyRanges), EndFrequency(TotalNumberOfFrequencyRanges), &
                     StepFrequency(TotalNumberOfFrequencyRanges), TelescopeSize(TotalNumberOfFrequencyRanges), &
                     BackgroundTemperatureRange(TotalNumberOfFrequencyRanges), TemperatureSlopeRange(TotalNumberOfFrequencyRanges), &
                     HydrogenColumnDensityRange(TotalNumberOfFrequencyRanges), DustBetaRange(TotalNumberOfFrequencyRanges), &
                     KappaRange(TotalNumberOfFrequencyRanges), SpectrumIndexForFreqRange(TotalNumberOfFrequencyRanges), &
                     ObservationalDataList(TotalNumberDataPoints, 3), DataPointIndexFreqRange(TotalNumberOfFrequencyRanges, 2), &
                     MoleculeNames(MaxNumberOfMolecules), ConversionTableMAGIXmyXCLASSParam(NumberFreeParameter, 2), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannel = 6, 6
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x,"Can not allocate variables StartFrequency, EndFrequency etc.!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            StartFrequency = 0.d0
            EndFrequency = 0.d0
            StepFrequency = 0.d0
            TelescopeSize = 0.d0
            BackgroundTemperatureRange = 0.d0
            TemperatureSlopeRange = 0.d0
            HydrogenColumnDensityRange = 0.d0
            DustBetaRange = 0.d0
            KappaRange = 0.d0
            SpectrumIndexForFreqRange = 0
            ObservationalDataList = 0.d0
            DataPointIndexFreqRange = 0
            MoleculeNames = ""
            ConversionTableMAGIXmyXCLASSParam = 0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< get parameter for each frequency range
            IndexFreqRange = 0                                                              !< reset index for current frequency range
            IndexSpectrum = 0                                                               !< reset index for current spectrum
            LastIndexFreqRange = 0
            Do i = 1, parameternumber                                                       !< loop over all parameter


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif


                !< get size of telescope
                if (ParamName == "MAGIXImportExpFileNames") then
                    IndexSpectrum = IndexSpectrum + 1


                !< get first frequency of each frequency range
                elseif (ParamName == "MAGIXImportStartFrequency") then
                    IndexFreqRange = IndexFreqRange + 1
                    StartFrequency(IndexFreqRange) = paramset(1, i)
                    SpectrumIndexForFreqRange(IndexFreqRange) = IndexSpectrum


                !< get last frequency of each frequency range
                elseif (ParamName == "MAGIXImportEndFrequency") then
                    EndFrequency(IndexFreqRange) = paramset(1, i)


                !< get stepsize of each frequency range
                elseif (ParamName == "MAGIXImportStepFrequency") then
                    StepFrequency(IndexFreqRange) = paramset(1, i)


                !< get global definition for background temperature
                elseif (ParamName == "MAGIXImportBackgroundTemperature" .and. tbFlag) then
                    BackgroundTemperatureRange(IndexFreqRange) = paramset(1, i)


                !< get global definition for temperature slope
                elseif (ParamName == "MAGIXImportTemperatureSlope" .and. tbFlag) then
                    TemperatureSlopeRange(IndexFreqRange) = paramset(1, i)


                !< get global definition for n_H
                elseif (ParamName == "MAGIXImportHydrogenColumnDensity" .and. nHFlag) then
                    HydrogenColumnDensityRange(IndexFreqRange) = paramset(1, i)


                !< get global definition for dust beta
                elseif (ParamName == "MAGIXImportDustBeta" .and. nHFlag) then
                    DustBetaRange(IndexFreqRange) = paramset(1, i)


                !< get global definition for kappa
                elseif (ParamName == "MAGIXImportKappa" .and. nHFlag) then
                    KappaRange(IndexFreqRange) = paramset(1, i)


                !< get size of telescope
                elseif (ParamName == "MAGIXImportTelescopeSize") then
                    TelescopeSize((LastIndexFreqRange + 1):IndexFreqRange) = paramset(1, i)
                    LastIndexFreqRange = IndexFreqRange                                     !< variable 'LastIndexFreqRange' is necessary, because
                                                                                            !< size of telescope is given only for each spectrum and not
                                                                                            !< for each frequency range
                endif
            end Do

            ! Debug:
            !    print*,"SpectrumIndexForFreqRange = ", SpectrumIndexForFreqRange(:)
            !    print*,"StartFrequency = ", StartFrequency(:)
            !    print*,"EndFrequency = ", EndFrequency(:)
            !    print*,"StepFrequency = ", StepFrequency(:)
            !    print*,"BackgroundTemperatureRange = ", BackgroundTemperatureRange(:)
            !    print*,"TemperatureSlopeRange = ", TemperatureSlopeRange(:)
            !    print*,"HydrogenColumnDensityRange = ", HydrogenColumnDensityRange(:)
            !    print*,"DustBetaRange = ", DustBetaRange(:)
            !    print*,"KappaRange = ", KappaRange(:)
            !    print*,"TelescopeSize = ", TelescopeSize(:)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< prepare observational data as one long 2D list
            k = 0
            Do i = 1, IndexSpectrum                                                         !< variable 'IndexSpectrum' is NOT identical to variable
                                                                                            !< 'NumberExpFiles' !!!!!!!!!!!
                Do j = 1, lengthexpdata(i)                                                  !< loop over all points of ith spectrum
                    k = k + 1                                                               !< increase data point counter
                    ObservationalDataList(k, :) = (/ expdatax(i, j, 1), expdatay(i, j, 1), expdatae(i, j, 1) /)
                end Do
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< define indices of first and last frequency (data point) of each frequency range), respectively
            Do l = 1, TotalNumberOfFrequencyRanges                                          !< loop over all frequency ranges
                i = SpectrumIndexForFreqRange(l)                                            !< get spectrum index
                FirstIndex = 0
                LastIndex = 0 
                Do j = 1, lengthexpdata(i)                                                  !< loop over all points of ith spectrum
                    if (expdatax(i, j, 1) <= StartFrequency(l)) then
                        FirstIndex = FirstIndex + 1
                    endif
                    if (expdatax(i, j, 1) <= EndFrequency(l)) then
                        LastIndex = LastIndex + 1
                    else
                        exit
                    endif
                end Do


                !< check indices
                if (FirstIndex == 0) then
                    FirstIndex = 1
                endif
                if (LastIndex == 0) then
                    LastIndex = 1
                endif
                if (FirstIndex > LastIndex) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x,"An error occurred by calculating the indices of each frequency range!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                else


                    !< determine offset for indices for each frequency range
                    k = 0
                    if (i > 1) then
                        k = sum(lengthexpdata(1:(i - 1)))
                    endif
                    DataPointIndexFreqRange(l, :) = (/ FirstIndex + k, LastIndex + k /)
                endif
            end Do

            ! Debug:
            !    print*,"DataPointIndexFreqRange(:, 1) = ", DataPointIndexFreqRange(:, 1)
            !    print*,"DataPointIndexFreqRange(:, 2) = ", DataPointIndexFreqRange(:, 2)
            !    print*,"ObservationalDataList(DataPointIndexFreqRange(1, 1), :) = ", ObservationalDataList(DataPointIndexFreqRange(1, 1), :)
            !    print*,"ObservationalDataList(DataPointIndexFreqRange(1, 2), :) = ", ObservationalDataList(DataPointIndexFreqRange(1, 2), :)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine list of all molecules which are defined in molfit and iso ratio files
            TotalNumberOfMolecules = 0                                                      !< reset counter for total number of molecules
            Do i = 1, parameternumber                                                       !< loop over all parameter


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif


                !< determine molecule index
                if (ParamName == "Molecule_Name") then
                    CurrNameMolecule = trim(adjustl(FitParameterValue(i)))
                    call CheckMolNames(AlreadyIncludedFlag, TotalNumberOfMolecules, CurrNameMolecule)
                    if (.not. AlreadyIncludedFlag) then
                        TotalNumberOfMolecules = TotalNumberOfMolecules + 1
                        MoleculeNames(TotalNumberOfMolecules) = trim(adjustl(CurrNameMolecule))
                    endif


                    !< check, if iso ratios are defined and determine total number of molecules
                    if (IsoFlag) then
                        Do j = 1, NumberOfIsomasters
                            if (trim(adjustl(CurrNameMolecule)) == trim(adjustl(IsoMolecule(j)))) then
                                call CheckMolNames(AlreadyIncludedFlag, TotalNumberOfMolecules, IsoMasters(j))
                                if (.not. AlreadyIncludedFlag) then
                                    TotalNumberOfMolecules = TotalNumberOfMolecules + 1
                                    MoleculeNames(TotalNumberOfMolecules) = trim(adjustl(IsoMasters(j)))
                               endif
                            elseif (trim(adjustl(CurrNameMolecule)) == trim(adjustl(IsoMasters(j)))) then
                                call CheckMolNames(AlreadyIncludedFlag, TotalNumberOfMolecules, IsoMolecule(j))
                                if (.not. AlreadyIncludedFlag) then
                                    TotalNumberOfMolecules = TotalNumberOfMolecules + 1
                                    MoleculeNames(TotalNumberOfMolecules) = trim(adjustl(IsoMolecule(j)))
                               endif
                            endif
                        end Do
                    endif
                endif
            end Do

            ! Debug:
            ! print*," "
            ! Do i = 1, TotalNumberOfMolecules
            !     print*,"GetmyXCLASSParameter:    i, MoleculeNames(i) = ", i, MoleculeNames(i)
            ! end Do


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< prepare some variables for isotopologues only


            !< allocate memory for iso conversion tables
            if (allocated(IsoRatioConversionTable)) then
                deallocate(IsoRatioConversionTable, IsoNfitConversionTable, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables IsoNfitConversionTable and IsoRatioConversionTable!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            allocate(IsoNfitConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules), &
                     IsoRatioConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannel = 6, 6
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x,"Can not allocate variables IsoNfitConversionTable and IsoRatioConversionTable!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            IsoNfitConversionTable = 0
            IsoRatioConversionTable = 0.d0



            !< define conversion tables IsoNfitConversionTable and IsoRatioConversionTable
            if (IsoFlag) then
                Do i = 1, parameternumber                                                   !< loop over all parameter


                    !< remove [[..]] expansion of parameter name
                    ParamName = trim(adjustl(FitParameterName(i)))
                    k = index(ParamName, "[[")
                    if (k > 0) then
                        ParamName = ParamName(:(k - 1))
                    endif


                    !< determine molecule index for current molecule
                    if (ParamName == "Molecule_Name") then
                        CurrNameMolecule = trim(adjustl(FitParameterValue(i)))
                        IndexMolecule = 1
                        Do k = 1, TotalNumberOfMolecules
                            if (trim(adjustl(CurrNameMolecule)) == trim(adjustl(MoleculeNames(k)))) then
                                IndexMolecule = k
                                exit
                            endif
                        end Do


                        !< create conversion table between molecules and isotopologues
                        Do j = 1, NumberOfIsomasters                                        !< loop over all defined isotopologues

                            ! Debug:
                            ! print*,"j, CurrNameMolecule = ", j, trim(adjustl(CurrNameMolecule))
                            ! print*,"IsoMolecule(j), IsoMasters(j), IsoRatio(j, 1) = ", trim(adjustl(IsoMolecule(j))), "    ", &
                            !                                                            trim(adjustl(IsoMasters(j))), "   ", IsoRatio(j, 1)


                            if (trim(adjustl(CurrNameMolecule)) == trim(adjustl(IsoMolecule(j)))) then
                                l = IndexMolecule
                                m = 0
                                Do k = 1, TotalNumberOfMolecules
                                    if (trim(adjustl(IsoMasters(j))) == trim(adjustl(MoleculeNames(k)))) then
                                        m = k
                                        exit
                                    endif
                                end Do
                                IsoRatioConversionTable(l, m) = IsoRatio(j, 1)
                                IsoNfitConversionTable(l, m) = int(IsoRatio(j, 2))

                                ! Debug:
                                ! print*,"IsoMolecule(j), IsoRatio(j, 1) = ", trim(adjustl(IsoMolecule(j))), "    ", IsoRatio(j, 1)


                            elseif (trim(adjustl(CurrNameMolecule)) == trim(adjustl(IsoMasters(j)))) then
                                l = IndexMolecule
                                m = 0
                                Do k = 1, TotalNumberOfMolecules
                                    if (trim(adjustl(IsoMolecule(j))) == trim(adjustl(MoleculeNames(k)))) then
                                        m = k
                                        exit
                                    endif
                                end Do
                                if (dabs(IsoRatio(j, 1)) > 1.e-30) then
                                    IsoRatioConversionTable(l, m) = 1.d0 / IsoRatio(j, 1)
                                    IsoNfitConversionTable(l, m) = -int(IsoRatio(j, 2))
                                else
                                    IsoRatioConversionTable(l, m) = 1.d0
                                    IsoNfitConversionTable(l, m) = -int(IsoRatio(j, 2))
                                endif

                                ! Debug:
                                ! print*,"IsoMasters(j), 1.d0 / IsoRatio(j, 1) = ", trim(adjustl(IsoMasters(j))), "    ", 1.d0 / IsoRatio(j, 1)

                            endif
                        end Do
                    endif
                end Do
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< determine the number of components for each molecule (including isotopologues if selected)
            if (allocated(NumberComponentsPerMolecule)) then
                deallocate(NumberComponentsPerMolecule, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x,"Can not deallocate variable NumberComponentsPerMolecule!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            allocate(NumberComponentsPerMolecule(0:TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannel = 6, 6
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x,"Can not allocate variable NumberComponentsPerMolecule!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            NumberComponentsPerMolecule = 0
            IndexMolecule = 1
            Do i = 1, parameternumber                                                       !< loop over all parameter


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif


                !< determine molecule index
                if (ParamName == "Molecule_Name") then
                    CurrNameMolecule = trim(adjustl(FitParameterValue(i)))
                    IndexMolecule = 1
                    Do k = 1, TotalNumberOfMolecules
                        if (trim(adjustl(CurrNameMolecule)) == trim(adjustl(MoleculeNames(k)))) then
                            IndexMolecule = k
                            exit
                        endif
                    end Do


                !< get number of components for current molecule
                elseif (ParamName == "Number_MolLine") then
                    NumberComponentsPerMolecule(IndexMolecule) = int(paramset(1, i))
                endif
            end Do


            !< special handling for isotopologues
            if (IsoFlag) then
                Do k = 1, TotalNumberOfMolecules

                    ! Debug:
                    ! print*,'>>>k, NumberComponentsPerMolecule(k) = ', k, NumberComponentsPerMolecule(k)
                    ! print*,'IsoRatioConversionTable(k, :) = ', IsoRatioConversionTable(k, :)


                    Do j = 1, TotalNumberOfMolecules

                        ! Debug:
                        ! print*,'j, IsoRatioConversionTable(k, j) = ', j, IsoRatioConversionTable(k, j)


                        if (NumberComponentsPerMolecule(j) == 0 .and. IsoRatioConversionTable(k, j) /= 0.d0 .and. k /= j) then
                            NumberComponentsPerMolecule(j) = NumberComponentsPerMolecule(k)

                            ! Debug:
                            ! print*,'-->j, NumberComponentsPerMolecule(j) = ', j, NumberComponentsPerMolecule(j)
                        endif
                    end Do
                end Do
            endif


            !< determine total number of components and allocate memory
            TotalNumberComponents = sum(NumberComponentsPerMolecule(:))
            if (allocated(myXCLASSParameter)) then
                deallocate(myXCLASSParameter, CompMoleculeIndex, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables myXCLASSParameter, CompMoleculeIndex etc.!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            allocate(myXCLASSParameter(11, TotalNumberComponents), CompMoleculeIndex(TotalNumberComponents), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannel = 6, 6
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x,"Can not allocate variables myXCLASSParameter, CompMoleculeIndex etc.!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            myXCLASSParameter = 0.d0
            CompMoleculeIndex = 1


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< get calculation parameter: source_size
            IndexMolecule = 1                                                               !< reset index for current molecule
            IndexComp = 0                                                                   !< reset index for current component
            CurrComp = 0                                                                    !< reset index for current component of current molecule
            Do i = 1, parameternumber                                                       !< loop over all parameter


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif


                !< determine molecule index
                if (ParamName == "Molecule_Name") then
                    CurrNameMolecule = trim(adjustl(FitParameterValue(i)))
                    IndexMolecule = 0
                    Do k = 1, TotalNumberOfMolecules
                        if (trim(adjustl(CurrNameMolecule)) == trim(adjustl(MoleculeNames(k)))) then
                            IndexMolecule = k
                            exit
                        endif
                    end Do
                    if (IndexMolecule == 0) then
                        print '(" ")'
                        print '(2x, "Error in subroutine ReadInMolfitsParameter:")'
                        print '(4x, "Can not determine molecule index for molecule: ", A, " ")', trim(adjustl(CurrNameMolecule))
                        print '(4x, "Set index to 1!")'
                        print '(" ")'
                        print '(4x,"Known molecules:")'
                        Do k = 1, TotalNumberOfMolecules
                            print '(4x, I5, 3x, A)', k, trim(adjustl(MoleculeNames(k)))
                        end Do
                        IndexMolecule = 1
                    endif


                !< determine molecule index
                elseif (ParamName == "Number_MolLine") then
                    CurrComp = 0
                    IndexComp = sum(NumberComponentsPerMolecule(:(IndexMolecule - 1)))


                !< get source size
                elseif (ParamName == "source_size") then
                    IndexComp = IndexComp + 1                                               !< increase indec for current component
                    CurrComp = CurrComp + 1
                    CompMoleculeIndex(IndexComp) = IndexMolecule
                    myXCLASSParameter(1, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 1/)
                    endif


                !< get T_rot
                elseif (ParamName == "T_rot") then
                    myXCLASSParameter(2, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 2/)
                    endif


                    !< check for zero temperature
                    if (myXCLASSParameter(2, IndexComp) == 0.d0) then
                        Do ErrChannelIndex = 1, 1
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(2x, "Error in subroutine ReadInMolfitsParameter:")')
                            write(ErrChannel, '(4x, "The rotation temperature for component ", I4, " of molecule ", A, " ")') CurrComp, &
                                                                                                          trim(adjustl(CurrNameMolecule))
                            write(ErrChannel, '(4x, "is identical to zero!")')
                            write(ErrChannel, '(4x, "Set rotation temperature for this component to 1.0 to avoid division by zero!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                        end Do
                        myXCLASSParameter(2, IndexComp) = 1.d0
                    endif


                !< get N_tot
                elseif (ParamName == "N_tot") then
                    myXCLASSParameter(3, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 3/)
                    endif


                    !< check for overflow
                    if (myXCLASSParameter(3, IndexComp) > 300.d0) then
                        Do ErrChannelIndex = 1, 1
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(2x, "Error in subroutine GetmyXCLASSParameter:")')
                            write(ErrChannel, '(4x, "The log10 value of column density for component ", I4, " of molecule ", A, " ")') CurrComp, &
                                                                                                                   trim(adjustl(CurrNameMolecule))
                            write(ErrChannel, '(4x, "causes an overflow (to large number)!")')
                            write(ErrChannel, '(4x, "Set log10 value for this column density to 300!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                        end Do
                        myXCLASSParameter(3, IndexComp) = 300.d0
                    endif


                    !< convert log10 value back to linear value
                    myXCLASSParameter(3, IndexComp) = 10.d0**myXCLASSParameter(3, IndexComp)

                    ! Debug:
                    ! print*,'paramset(1, i) = ', paramset(1, i)
                    ! print*,'IndexComp, myXCLASSParameter(3, IndexComp) = ', IndexComp, myXCLASSParameter(3, IndexComp)


                !< get V_width
                elseif (ParamName == "V_width") then
                    myXCLASSParameter(4, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 4/)
                    endif


                !< get V_off
                elseif (ParamName == "V_off") then
                    myXCLASSParameter(5, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 5/)
                    endif


                !< get AbsorptionFlag
                elseif (ParamName == "AbsorptionFlag") then
                    if (trim(adjustl(FitParameterValue(i))) == "a") then
                        myXCLASSParameter(11, IndexComp) = 2.d0
                    else
                        myXCLASSParameter(11, IndexComp) = 1.d0
                    endif
                endif


                !< get T_Back and T_Slope for each component
                if (.not. tbFlag) then


                    !< get T_Back
                    if (ParamName == "T_Back") then
                        myXCLASSParameter(6, IndexComp) = paramset(1, i)
                        if (paramset(2, i) == 1) then
                            k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                            ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 6/)
                        endif


                    !< get T_Slope
                    elseif (ParamName == "T_Slope") then
                        myXCLASSParameter(7, IndexComp) = paramset(1, i)
                        if (paramset(2, i) == 1) then
                            k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                            ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 7/)
                        endif
                    endif
                endif


                !< get nH, kappa, and beta for each component
                if (.not. nHFlag) then


                    !< get nHcolumn
                    if (ParamName == "nHcolumn") then
                        myXCLASSParameter(8, IndexComp) = paramset(1, i)
                        if (paramset(2, i) == 1) then
                            k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                            ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 8/)
                        endif


                        !< check for overflow
                        if (myXCLASSParameter(8, IndexComp) > 300.d0) then
                            Do ErrChannelIndex = 1, 1
                                ErrChannel = AllErrChannels(ErrChannelIndex)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(2x, "Error in subroutine GetmyXCLASSParameter:")')
                                write(ErrChannel, '(4x, "The log10 value of hydrogen column density for component ", I4, " of molecule ", A, " ")') &
                                                                                                            CurrComp, trim(adjustl(CurrNameMolecule))

                                write(ErrChannel, '(4x, "causes an overflow (to large number)!")')
                                write(ErrChannel, '(4x, "Set log10 value for this column density to 300!")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                            end Do
                            myXCLASSParameter(8, IndexComp) = 300.d0
                        endif


                        !< convert log10 value back to linear value
                        myXCLASSParameter(8, IndexComp) = 10.d0**myXCLASSParameter(8, IndexComp)


                    !< get beta
                    elseif (ParamName == "beta") then
                        myXCLASSParameter(9, IndexComp) = paramset(1, i)
                        if (paramset(2, i) == 1) then
                            k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                            ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 10/)
                        endif


                    !< get kappa
                    elseif (ParamName == "kappa") then
                        myXCLASSParameter(10, IndexComp) = paramset(1, i) * 2.d0 * 1.66d-24 / 100.d0     !< kappa_1300 * M(H2)/M_d_M_g
                        if (paramset(2, i) == 1) then
                            k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                            ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 9/)
                        endif
                    endif
                endif
            end Do

            ! Debug:
            !    print*,"myXCLASSParameter(1, :) = ", myXCLASSParameter(1, :)
            !    print*,"myXCLASSParameter(2, :) = ", myXCLASSParameter(2, :)
            !    print*,"myXCLASSParameter(3, :) = ", myXCLASSParameter(3, :)
            !    print*,"myXCLASSParameter(4, :) = ", myXCLASSParameter(4, :)
            !    print*,"myXCLASSParameter(5, :) = ", myXCLASSParameter(5, :)
            !    print*,"myXCLASSParameter(6, :) = ", myXCLASSParameter(6, :)
            !    print*,"myXCLASSParameter(7, :) = ", myXCLASSParameter(7, :)
            !    print*,"myXCLASSParameter(8, :) = ", myXCLASSParameter(8, :)
            !    print*,"myXCLASSParameter(9, :) = ", myXCLASSParameter(9, :)
            !    print*,"myXCLASSParameter(10, :) = ", myXCLASSParameter(10, :)
            !    print*,"myXCLASSParameter(11, :) = ", myXCLASSParameter(11, :)
            !    if (IsoFlag) then
            !        Do i = 1, TotalNumberOfMolecules
            !            print*,"i, IsoRatioConversionTable(i, :) = ", i, IsoRatioConversionTable(i, :)
            !        end Do
            !    endif


            !< print what you do
            if (printflag) then
                print '("done!")'
            endif


            return
        end subroutine GetmyXCLASSParameter


        !>************************************************************************************************************************************************
        !> subroutine: GetPartitionFunction
        !>
        !> get partition function from sqlite3 database
        !>
        !> input variables:     none
        !>
        !> output variables:    ok:             status variable
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 29.07.2014
        !>
        subroutine GetPartitionFunction(ok)

            implicit none
            integer :: ok                                                                       !< status variable
            integer :: number_entries                                                           !< number of entries for the PartitionFunction table
            integer :: countlen                                                                 !< total length of query
            integer :: i, j, k, l                                                               !< loop/working variables
            integer :: allocstatus, deallocstatus                                               !< variables for (de)allocation
            real*8, allocatable, dimension(:, :) :: CopylgQ                                     !< copy of lgQ variable
            character(len=16384) :: queryString1                                                !< query string (first part)
            character(len=16384) :: queryString2                                                !< query string (second part)
            character(len=4096) :: dbNameCopy                                                   !< name of database
            character(len=40) :: ColumnNameForNamePartFuncCopy                                  !< copy of name of column including the names of molecules
            character(len=40), dimension(TotalNumberOfMolecules) :: CopyMoleculeNames           !< copy of names of molecules


            !< print what you do
            if (printflag) then
                print '(9x, "Reading partition functions for all molecules from sqlite3 database .. ", $)'
            endif



            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define query string


            !< first part
            queryString1 = "SELECT count(*) FROM " // trim(adjustl(NameOfPartFuncTable)) // char(0)
            queryString2 = " " // char(0)


            !< create query string
            ok = 0                                                                          !< reset status variable
            number_entries = 0                                                              !< reset result variable
            j = TotalNumberOfMolecules
            ColumnNameForNamePartFuncCopy = trim(adjustl(ColumnNameForNamePartFunc)) // char(0) !< get name of column for molecule names
            countlen = (len_trim(queryString1) - 1)                                         !< length of string queryString1
            countlen = countlen + 8                                                         !< length plus ' WHERE ('
            CopyMoleculeNames = ""
            Do k = 1, TotalNumberOfMolecules                                                !< loop over all molecules
                CopyMoleculeNames(k) = char(34) // trim(adjustl(MoleculeNames(k))) // char(34) // char(0)
                countlen = countlen + len_trim(CopyMoleculeNames(k)) - 1                    !< length plus length of name (incl. '""') 
                countlen = countlen + len_trim(ColumnNameForNamePartFuncCopy)               !< length plus 'PF_Name'
                countlen = countlen + 7                                                     !< length plus ' or ' and ' = '
            end Do
            countlen = countlen + 2                                                         !< length plus ') '
            countlen = countlen + (len_trim(queryString2) - 1)                              !< length plus length of string queryString2
            countlen = countlen + len_trim(ColumnNameForNamePartFuncCopy) + 2               !< length plus 'PF_Name'
            dbNameCopy = trim(adjustl(dbName)) // char(0)

            ! Debug:
            ! print*,' '
            ! print*,'queryString1 = >', trim(adjustl(queryString1)), '<'
            ! print*,'queryString2 = >', trim(adjustl(queryString2)), '<'
            ! print*,'countlen = ', countlen


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get number of entries for the partition function table
            call numentr_sqlite3(ok, number_entries, %val(countlen), trim(adjustl(queryString1)), trim(adjustl(queryString2)), %val(j), &
                                 trim(adjustl(dbNameCopy)), trim(adjustl(ColumnNameForNamePartFuncCopy)), CopyMoleculeNames)
            NumberMoleculePartFunc = number_entries

            ! Debug:
            ! print*,'NumberMoleculePartFunc = ', NumberMoleculePartFunc
            ! print*,'ok = ', ok


            !< allocate variables for partition function
            if (NumberMoleculePartFunc > 0 .and. ok == 0) then
                if (allocated(MolNamePartFunc)) then
                    deallocate(MolNamePartFunc, lgQ, CopylgQ, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        Do ErrChannelIndex = 1, 1
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(1x,"Error in subroutine GetPartitionFunction!")')
                            write(ErrChannel, '(3x,"Can not deallocate variables MolNamePartFunc and lgQ!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(3x,"Please restart the program!")')
                        end Do
                        stop 'Program aborted!'
                    endif
                endif
                allocate(MolNamePartFunc(NumberMoleculePartFunc), lgQ(NumberOfTemperatures, NumberMoleculePartFunc), &
                         CopylgQ(NumberOfTemperatures, NumberMoleculePartFunc), stat = allocstatus)
                if (allocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetPartitionFunction!")')
                        write(ErrChannel, '(3x,"Can not allocate variables MolNamePartFunc and lgQ!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
                MolNamePartFunc = ""
                lgQ = 0.d0
                CopylgQ = 0.d0


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< construct query string
                ok = 0                                                                      !< reset status variable
                i = NumberMoleculePartFunc
                queryString1 = "SELECT " // trim(adjustl(ColumnNameForNamePartFunc))
                Do j = 1, NumberOfTemperatures
                    queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNamesPartFunc(j)))
                end Do
                queryString1 = trim(adjustl(queryString1)) // " FROM " // trim(adjustl(NameOfPartFuncTable)) // char(0)

                ! Debug:
                ! print*,"queryString1 = ", queryString1


                !< get information from sqlite3 database
                ok = 0                                                                      !< reset status variable
                l = NumberOfTemperatures
                i = NumberMoleculePartFunc
                j = TotalNumberOfMolecules
                countlen = (len_trim(queryString1) - 1)
                CopyMoleculeNames = ""
                Do k = 1, TotalNumberOfMolecules
                    CopyMoleculeNames(k) = char(34) // trim(adjustl(MoleculeNames(k))) // char(34) // char(0)
                    countlen = countlen + len_trim(CopyMoleculeNames(k)) + 14
                end Do
                countlen = countlen + (len_trim(queryString2) - 1) + 17


                !< bug fix: prevent segmentation fault, if database contains double entries
                if (NumberMoleculePartFunc > TotalNumberOfMolecules) then
                    Do k = TotalNumberOfMolecules, NumberMoleculePartFunc
                        countlen = countlen + 40 + 14
                    end Do
                endif


                !< get information from sqlite3 database
                ! ok,   status
                ! i,    number of entries in the database
                ! l,    number of temperatures
                ! j,    number of molecules in the query string
                dbNameCopy = trim(adjustl(dbName)) // char(0)
                ColumnNameForNamePartFuncCopy = trim(adjustl(ColumnNameForNamePartFunc)) // char(0)
                call ReadPartFunc(ok, %val(l), %val(i), %val(j), %val(countlen), trim(adjustl(queryString1)), trim(adjustl(queryString2)), &
                                  trim(adjustl(dbNameCopy)), trim(adjustl(ColumnNameForNamePartFuncCopy)), CopyMoleculeNames, lgQ, MolNamePartFunc)

                ! Debug:
                ! print*,'NumberMoleculePartFunc = ', NumberMoleculePartFunc
                ! print*,'lgQ(1:NumberOfTemperatures, 1) = ', lgQ(1:NumberOfTemperatures, 1)
                ! print*,'lgQ(1:NumberOfTemperatures, NumberMoleculePartFunc) = ', lgQ(1:NumberOfTemperatures, NumberMoleculePartFunc)
                ! print*,'MolNamePartFunc(1) = ', MolNamePartFunc(1)
                ! print*,'MolNamePartFunc(NumberMoleculePartFunc) = ', MolNamePartFunc(NumberMoleculePartFunc)
                ! stop
            else
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetPartitionFunction!")')
                    write(ErrChannel, '(3x,"There are no entries in the sqlite3 database ", A, "!")') trim(adjustl(NameOfPartFuncTable))
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please correct the name of the database and restart the program!")')
                end Do
                ok = 1
                return
            endif


            !< bring partition function in variable lgQ in the same order like MoleculeNames
            CopylgQ(:, :) = lgQ(:, :)
            lgQ = 0.d0
            Do i = 1, TotalNumberOfMolecules
                Do j = 1, TotalNumberOfMolecules

                    ! Debug:
                    ! print*,'----->i, trim(adjustl(MoleculeNames(i))) = ', i, trim(adjustl(MoleculeNames(i)))
                    ! print*,'>>>>>>j, trim(adjustl(MolNamePartFunc(j))) = ', j, trim(adjustl(MolNamePartFunc(j)))

                    if (trim(adjustl(MoleculeNames(i))) // char(0) == trim(adjustl(MolNamePartFunc(j)))) then
                        lgQ(:, i) = CopylgQ(:, j)


                        !< calculate log10 of partition function of the current molecule and check for entries which are equal to zero
                        if (count(lgQ(:, i) == 0) > 0) then
                            Do ErrChannelIndex = 1, 1
                                ErrChannel = AllErrChannels(ErrChannelIndex)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(2x, "Error in subroutine qinterp!")')
                                write(ErrChannel, '(4x, "The are some entries in the parittion function table for molecule ", A, " which are not defined or zero!")') &
                                                                                                                           trim(adjustl(MoleculeNames(i)))
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(4x, "Number of entries which are identical to zero = ", I10)') count(lgQ(:, i) == 0)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(4x,"Set values at these positions to 1!")')
                                write(ErrChannel, '(" ")')
                                where (lgQ(:, i) == 0.d0)
                                    lgQ(:, i) = 1.d0
                                end where
                            end Do
                        endif


                        !< determine log10 values
                        Do l = 1, NumberOfTemperatures
                            lgQ(l, i) = dlog10(lgQ(l, i))
                        end Do
                        exit
                    endif
                end Do

                ! Debug:
                ! print*,'i, trim(adjustl(MoleculeNames(i))), lgQ(:, i) = ', i, trim(adjustl(MoleculeNames(i))), lgQ(:, i)
            end Do


            !< print some status information to screen
            if (printflag) then
                print '("done!")'
                print '(" ")'
                print '(11x, "Number of entries = ",I10)', NumberMoleculePartFunc
                call strlen(i, MolNamePartFunc(1))
                if (NumberMoleculePartFunc > 1) then
                    print '(11x, "First Molecule in database: ",A)', trim(adjustl(MolNamePartFunc(1)(:i)))
                    call strlen(i, MolNamePartFunc(NumberMoleculePartFunc))
                    print '(11x, "Last Molecule in database:  ",A)', trim(adjustl(MolNamePartFunc(NumberMoleculePartFunc)(:i)))
                else
                    print '(11x, "Molecule in database: ",A)', trim(adjustl(MolNamePartFunc(1)(:i)))
                endif
                print '(" ")'
                print '(" ")'
            endif


            ! Debug:
            ! Do i = 1, TotalNumberOfMolecules
            !     print*,"i, MoleculeNames(i), MolNamePartFunc(i) = ", i, trim(adjustl(MoleculeNames(i))), "         ", trim(adjustl(MolNamePartFunc(i)))
            ! end Do


            !< free memory of CopylgQ variable
            if (allocated(CopylgQ)) then
                deallocate(CopylgQ, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetPartitionFunction!")')
                        write(ErrChannel, '(3x,"Can not deallocate variable CopylgQ!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif

            return
        end subroutine GetPartitionFunction


        !>************************************************************************************************************************************************
        !> subroutine: GetTransitionParameters
        !>
        !> get informations for transitions from sqlite3 database
        !>
        !> input variables:     none
        !>
        !> output variables:    ok:             status variable
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 30.07.2014
        !>
        subroutine GetTransitionParameters(ok)

            implicit none
            integer :: ok                                                                   !< status variable
            integer :: number_entries                                                       !< number of entries for the PartitionFunction table
            integer :: countlen                                                             !< total length of query
            integer :: i, j, k                                                              !< loop variables
            integer :: RangeIndex                                                           !< index for frequency ranges
            integer :: MolecularDataCounter                                                 !< overall counter for MolecularData variable
            integer :: NumEntriesLocal                                                      !< working variable: number of data within current frequency
                                                                                            !< range and current molecule
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            integer, allocatable, dimension(:) :: UpperStateDegeneracy                      !< upper state degeneracy
            real*8 :: freqmin                                                               !< working variable lowest frequency of current freq. range
            real*8 :: freqmax                                                               !< working variable highest frequency of current freq. range
            real*8, allocatable, dimension(:) :: lFreq                                      !< frequencies of transition
            real*8, allocatable, dimension(:) :: lFreqErr                                   !< uncertainty of frequencies of transition
            real*8, allocatable, dimension(:) :: lElt                                       !< energies of lower states
            real*8, allocatable, dimension(:) :: icatMol                                    !< intensity of the transitions
            real*8, allocatable, dimension(:) :: EinsteinA                                  !< Einstein A coefficients
            character(len=40), allocatable, dimension(:) :: MolNameLocal                    !< names of molecules in current freq. range
            character(len=25) :: numberString1, numberString2                               !< working variables for number to string conversion
            character(len=8192) :: queryString1                                             !< query string (first part)
            character(len=8192) :: queryString2                                             !< query string (second part)
            character(len=4096) :: dbNameCopy                                               !< name of database
            character(len=40) :: ColumnNameForNameTransitionsCopy                           !< copy of name of column including the names of molecules
            character(len=40), dimension(1) :: CopyMoleculeNames                            !< copy of names of molecules


            !< initialize output variable
            ok = 0


            !< print what you do
            if (printflag) then
                print '(9x, "Reading parameters for radiative transitions for all molecules from sqlite database .. ", $)'
            endif


            !< allocate memory for variable NumEntriesRanges
            if (allocated(NumEntriesRanges)) then
                deallocate(NumEntriesRanges, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                        write(ErrChannel, '(3x,"Can not deallocate variable NumEntriesRanges!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    ok = 1
                    return
                endif
            endif
            allocate(NumEntriesRanges(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                    write(ErrChannel, '(3x,"Can not allocate variable NumEntriesRanges!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                ok = 1
                return
            endif
            NumEntriesRanges = 0


            !< determine total number of all transitions in all frequency ranges
            TotalNumberOfTransitions = 0
            Do RangeIndex = 1, TotalNumberOfFrequencyRanges                                 !< loop over all frequency ranges
                freqmin = StartFrequency(RangeIndex)
                freqmax = EndFrequency(RangeIndex)
                Do k = 1, TotalNumberOfMolecules                                            !< loop over all molecules


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< define query
                    write(numberString1, '(ES25.15)') freqmin
                    write(numberString2, '(ES25.15)') freqmax
                    queryString1 = "SELECT count(*) FROM " // trim(adjustl(NameOfRadTransTable)) // char(0)
                    queryString2 = "AND (" // trim(adjustl(ColumnNameForFreqTransitions)) // " >= " // trim(adjustl(numberString1)) &
                                   // " AND " // trim(adjustl(ColumnNameForFreqTransitions)) // " <= " // trim(adjustl(numberString2)) &
                                   // ") ORDER by " // trim(adjustl(ColumnNameForFreqTransitions)) // char(0)

                    ! Debug:
                    ! print*,' '
                    ! print*,'queryString1 = >', trim(adjustl(queryString1)), '<'
                    ! print*,'queryString2 = >', trim(adjustl(queryString2)), '<'


                    !< get number of entries for the transition table
                    ok = 0                                                                  !< reset status variable
                    number_entries = 0                                                      !< reset result variable
                    j = 1
                    countlen = (len_trim(queryString1) - 1)
                    CopyMoleculeNames = ""
                    CopyMoleculeNames(1) = char(34) // trim(adjustl(MoleculeNames(k))) // char(34) // char(0)
                    countlen = countlen + len_trim(CopyMoleculeNames(1)) + 12
                    countlen = countlen + (len_trim(queryString2) - 1) + 17
                    dbNameCopy = trim(adjustl(dbName)) // char(0)
                    ColumnNameForNameTransitionsCopy = trim(adjustl(ColumnNameForNameTransitions)) // char(0)
                    call numentr_sqlite3(ok, number_entries, %val(countlen), trim(adjustl(queryString1)), trim(adjustl(queryString2)), %val(j), &
                                         trim(adjustl(dbNameCopy)), trim(adjustl(ColumnNameForNameTransitionsCopy)), CopyMoleculeNames)
                    NumEntriesRanges(RangeIndex, k) = number_entries
                    TotalNumberOfTransitions = TotalNumberOfTransitions + number_entries

                    ! Debug:
                    ! print*,"Frequency range index = ", RangeIndex
                    ! print*,"Molecule index = ", k
                    ! print*,"Number of database entries within the current freq. range (number_entries) = ", NumEntriesRanges(RangeIndex, k)
                    ! print*,'TotalNumberOfTransitions = ', TotalNumberOfTransitions
                    ! print*,'ok = ', ok
                end Do
            end Do


            !< allocate variables for parameter
            if (TotalNumberOfTransitions > 0 .and. ok == 0) then


                !< allocate memory for variable NumEntriesRanges
                if (allocated(MolecularData)) then
                    deallocate(MolecularData, MolecularDataIndices, MolNameRadTrans, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        Do ErrChannelIndex = 1, 1
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                            write(ErrChannel, '(3x,"Can not deallocate variables MolecularData, MolecularDataIndices, and MolNameRadTrans!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(3x,"Please restart the program!")')
                        end Do
                        ok = 1
                        return
                    endif
                endif
                allocate(MolecularData(TotalNumberOfTransitions, 4), MolecularDataIndices(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2), &
                         MolNameRadTrans(TotalNumberOfTransitions), stat = allocstatus)
                if (allocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                        write(ErrChannel, '(3x,"Can not allocate variables MolecularData, MolecularDataIndices, and MolNameRadTrans!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    ok = 1
                    return
                endif
                MolecularData = 0.d0
                MolecularDataIndices = 0
                MolNameRadTrans = ""


                !< read in molecular data from database for each molecule
                MolecularDataCounter = 0
                Do RangeIndex = 1, TotalNumberOfFrequencyRanges                             !< loop over all frequency ranges
                    freqmin = StartFrequency(RangeIndex)                                    !< get first frequency of current frequency range
                    freqmax = EndFrequency(RangeIndex)                                      !< get last frequency of current frequency range
                    Do k = 1, TotalNumberOfMolecules                                        !< loop over all molecules


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< define query
                        write(numberString1, '(ES25.15)') freqmin
                        write(numberString2, '(ES25.15)') freqmax
                        queryString2 = "AND (" // trim(adjustl(ColumnNameForFreqTransitions)) // " >= " // trim(adjustl(numberString1)) &
                                       // " AND " // trim(adjustl(ColumnNameForFreqTransitions)) // " <= " // trim(adjustl(numberString2)) &
                                       // ") ORDER by " // trim(adjustl(ColumnNameForFreqTransitions)) // char(0)

                        ! Debug:
                        ! print*,' '
                        ! print*,"RangeIndex, k, NumEntriesRanges(RangeIndex, k) = ", RangeIndex, k, NumEntriesRanges(RangeIndex, k)


                        !< allocate memory for worker
                        if (allocated(lFreq)) then
                            deallocate(MolNameLocal, lFreq, lFreqErr, icatMol, EinsteinA, lElt, UpperStateDegeneracy, stat = deallocstatus)
                            if (deallocstatus /= 0) then
                                Do ErrChannelIndex = 1, 1
                                    ErrChannel = AllErrChannels(ErrChannelIndex)
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                                    write(ErrChannel, '(3x,"Can not deallocate variables lFreq and ...!")')
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(3x,"Please restart the program!")')
                                end Do
                                ok = 1
                                return
                            endif
                        endif
                        NumEntriesLocal = NumEntriesRanges(RangeIndex, k)
                        allocate(MolNameLocal(NumEntriesLocal), lFreq(NumEntriesLocal), lFreqErr(NumEntriesLocal), icatMol(NumEntriesLocal), &
                                 EinsteinA(NumEntriesLocal), lElt(NumEntriesLocal), UpperStateDegeneracy(NumEntriesLocal), stat = allocstatus)
                        if (allocstatus /= 0) then
                            Do ErrChannelIndex = 1, 1
                                ErrChannel = AllErrChannels(ErrChannelIndex)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                                write(ErrChannel, '(3x,"Can not allocate variables lFreq and ...!")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Please restart the program!")')
                            end Do
                            ok = 1
                            return
                        endif
                        MolNameLocal = ""
                        lFreq = 0.d0
                        lFreqErr = 0.d0
                        icatMol = 0.d0
                        EinsteinA = 0.d0
                        lElt = 0.d0
                        UpperStateDegeneracy = 0
                        Do j = 1, NumEntriesLocal
                            MolNameLocal(j) = char(0)
                        end Do


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< get molecular data
                        if (NumEntriesRanges(RangeIndex, k) > 0) then                       !< check, if molecular data are contained in the database
                                                                                            !< for the current frequency range and molecule

                            !< construct query string
                            queryString1 = "SELECT " // trim(adjustl(ColumnNameForNameTransitions))                                 !< pos. 1: molecule
                                                                                                                                    !< name
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForFreqTransitions))       !< pos. 2: frequency
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForFreqErrTransitions))    !< pos. 3: error of
                                                                                                                                    !< frequency
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForIntTransitions))        !< pos. 4: intensity
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForEinsteinATransitions))  !< pos. 5: Einstein A
                                                                                                                                    !<         coefficient
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForELowTransitions))       !< pos. 6: E_lower
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForgUpTransitions))        !< pos. 7: upper state
                                                                                                                                    !<         degeneracy
                            queryString1 = trim(adjustl(queryString1)) // " FROM " // trim(adjustl(NameOfRadTransTable)) // char(0)

                            ! Debug:
                            ! print*," "
                            ! print*,"queryString1 = -->", trim(adjustl(queryString1)), "<--"


                            !< get information from sqlite3 database
                            ok = 0                                                          !< reset status variable
                            i = TotalNumberOfTransitions
                            j = 1
                            countlen = (len_trim(queryString1) - 1)
                            CopyMoleculeNames = ""
                            CopyMoleculeNames(1) = char(34) // trim(adjustl(MoleculeNames(k))) // char(34) // char(0)
                            countlen = countlen + len_trim(CopyMoleculeNames(1)) + 12
                            countlen = countlen + (len_trim(queryString2) - 1) + 17
                            dbNameCopy = trim(adjustl(dbName)) // char(0)
                            ColumnNameForNameTransitionsCopy = trim(adjustl(ColumnNameForNameTransitions)) // char(0)


                            !< read informations from sqlite3 database
                            call radtrans(ok, %val(i), %val(countlen), trim(adjustl(queryString1)), trim(adjustl(queryString2)), %val(j), &
                                          trim(adjustl(dbNameCopy)), trim(adjustl(ColumnNameForNameTransitionsCopy)), CopyMoleculeNames, &
                                          MolNameLocal, lFreq, lFreqErr, icatMol, EinsteinA, lElt, UpperStateDegeneracy)
                            if (ok /= 0) then
                                ErrChannel = 6
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                                write(ErrChannel, '(3x,"An error occurs while reading entries from sqlite3 database ", A, " within the given frequency range!")') &
                                                                                                                     trim(adjustl(NameOfRadTransTable))
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Molecule:")')
                                write(ErrChannel, '(3x,A)') trim(adjustl(MoleculeNames(k)))
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Frequency range:")')
                                write(ErrChannel, '(3x,"Min. frequency = ", ES25.15)') freqmin
                                write(ErrChannel, '(3x,"Max. frequency = ", ES25.15)') freqmax
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Please check your inputs and restart the program!")')
                                ok = 1
                                return

                                ! Debug:
                                ! print*,'MolNameLocal(1) = ', MolNameLocal(1)
                                ! print*,'MolNameLocal(TotalNumberOfTransitions) = ', MolNameLocal(TotalNumberOfTransitions)
                                ! print*,'lFreq(1) = ', lFreq(1)
                                ! print*,'lFreq(TotalNumberOfTransitions) = ', lFreq(TotalNumberOfTransitions)
                                ! print*,'lFreqErr(1) = ', lFreqErr(1)
                                ! print*,'lFreqErr(TotalNumberOfTransitions) = ', lFreqErr(TotalNumberOfTransitions)
                                ! print*,'icatMol(1) = ', icatMol(1)
                                ! print*,'icatMol(TotalNumberOfTransitions) = ', icatMol(TotalNumberOfTransitions)
                                ! print*,'lElt(1) = ', lElt(1)
                                ! print*,'lElt(TotalNumberOfTransitions) = ', lElt(TotalNumberOfTransitions)
                                ! stop
                            endif
                        endif


                        !< copy molecular data for current molecule and frequency range to MolecularData variable
                        MolecularDataIndices(RangeIndex, k, 1) = MolecularDataCounter + 1   !< save first index for current freq. range and molecule
                                                                                            !< within MolecularData variable
                        Do j = 1, NumEntriesLocal                                           !< loop over all transitions within the current frequency
                                                                                            !< range and molecule
                            MolecularDataCounter = MolecularDataCounter + 1
                            MolecularData(MolecularDataCounter, 1) = lFreq(j)
                            MolecularData(MolecularDataCounter, 2) = EinsteinA(j)
                            MolecularData(MolecularDataCounter, 3) = lElt(j) / 1.438769     !< only for debugging
                            MolecularData(MolecularDataCounter, 4) = UpperStateDegeneracy(j)

                            ! Debug:
                            ! print*,"trim(adjustl(MoleculeNames(k))), j, MolecularData(j, :) = ", trim(adjustl(MoleculeNames(k))), j, MolecularData(j, :)


                            !< correct entry for lower energy
                            MolecularData(MolecularDataCounter, 3) = lElt(j)
                            MolNameRadTrans(MolecularDataCounter) = MolNameLocal(j)
                        end Do
                        MolecularDataIndices(RangeIndex, k, 2) = MolecularDataCounter       !< save last index for current freq. range and molecule
                                                                                            !< within MolecularData variable
                    end Do
                end Do
            endif


            !< print what you do
            if (printflag) then
                print '("done!")'
            endif


            !< print some informations to screen about transitions
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(11x, "Number of transitions for each frequency range and molecule:")'
                Do RangeIndex = 1, TotalNumberOfFrequencyRanges                             !< loop over all frequency ranges
                    freqmin = StartFrequency(RangeIndex)                                    !< get first frequency of current frequency range
                    freqmax = EndFrequency(RangeIndex)                                      !< get last frequency of current frequency range
                    write(numberString1, '(ES25.15)') freqmin
                    write(numberString2, '(ES25.15)') freqmax
                    print '(" ")'
                    print '(11x, "Frequency range: ", A, " MHz - ", A, " MHz:")', trim(adjustl(numberString1)), trim(adjustl(numberString2))
                    Do k = 1, TotalNumberOfMolecules                                        !< loop over all molecules
                        print '(13x, "Number of transitions for molecule ", A, A, A, ": ", T92, I10)', char(34), trim(adjustl(MoleculeNames(k))), &
                                                                                                       char(34), NumEntriesRanges(RangeIndex, k)
                    end Do
                end Do
                print '(" ")'
                print '(" ")'
            endif


            !< free memory of local variables
            if (allocated(MolNameLocal)) deallocate(MolNameLocal, stat = deallocstatus)
            if (allocated(lFreq) .and. deallocstatus == 0) deallocate(lFreq, stat = deallocstatus)
            if (allocated(lFreqErr) .and. deallocstatus == 0) deallocate(lFreqErr, stat = deallocstatus)
            if (allocated(icatMol) .and. deallocstatus == 0) deallocate(icatMol, stat = deallocstatus)
            if (allocated(EinsteinA) .and. deallocstatus == 0) deallocate(EinsteinA, stat = deallocstatus)
            if (allocated(lElt) .and. deallocstatus == 0) deallocate(lElt, stat = deallocstatus)
            if (allocated(UpperStateDegeneracy) .and. deallocstatus == 0) deallocate(UpperStateDegeneracy, stat = deallocstatus)
            if (deallocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                    write(ErrChannel, '(3x,"Can not deallocate variables MolNameLocal etc.!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                ok = 1
            endif
            return
        end subroutine GetTransitionParameters


        !>************************************************************************************************************************************************
        !> subroutine: ModelInit
        !>
        !> initialize and broadcast module variables
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 26.08.2014
        !>
        subroutine ModelInit

            implicit none
            integer :: ok                                                                   !< status variablee: name of parameter
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation


            !< initialize variables
            pi = 4.d0 * datan(1.d0)                                                         !< determine pi
            ckms = 299792.458d0                                                             !< speed of light in km?


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define variables for sqlite3 database
            NameOfRadTransTable = "transitions"
            NameOfPartFuncTable = "partitionfunctions"
            NumberOfTemperatures = 110


            !< allocate variables for partition function
            if (allocated(ColumnNamesPartFunc)) then
                deallocate(ColumnNamesPartFunc, TempPartFunc, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables ColumnNamesPartFunc and TempPartFunc!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            allocate(ColumnNamesPartFunc(NumberOfTemperatures), TempPartFunc(NumberOfTemperatures), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variables ColumnNamesPartFunc and TempPartFunc!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            ColumnNamesPartFunc = ""
            TempPartFunc = 0.d0


            !< define column names of table partitionfunctions
            !< Due to a bug in the current gfortran compiler a statement like ColumnNamesPartFunc = (/.../) is not possible
            ColumnNameForNamePartFunc = 'PF_Name'
            ColumnNamesPartFunc(1) = 'PF_1_072'
            ColumnNamesPartFunc(2) = 'PF_1_148'
            ColumnNamesPartFunc(3) = 'PF_1_230'
            ColumnNamesPartFunc(4) = 'PF_1_318'
            ColumnNamesPartFunc(5) = 'PF_1_413'
            ColumnNamesPartFunc(6) = 'PF_1_514'
            ColumnNamesPartFunc(7) = 'PF_1_622'
            ColumnNamesPartFunc(8) = 'PF_1_738'
            ColumnNamesPartFunc(9) = 'PF_1_862'
            ColumnNamesPartFunc(10) = 'PF_1_995'
            ColumnNamesPartFunc(11) = 'PF_2_138'
            ColumnNamesPartFunc(12) = 'PF_2_291'
            ColumnNamesPartFunc(13) = 'PF_2_455'
            ColumnNamesPartFunc(14) = 'PF_2_630'
            ColumnNamesPartFunc(15) = 'PF_2_725'
            ColumnNamesPartFunc(16) = 'PF_2_818'
            ColumnNamesPartFunc(17) = 'PF_3_020'
            ColumnNamesPartFunc(18) = 'PF_3_236'
            ColumnNamesPartFunc(19) = 'PF_3_467'
            ColumnNamesPartFunc(20) = 'PF_3_715'
            ColumnNamesPartFunc(21) = 'PF_3_981'
            ColumnNamesPartFunc(22) = 'PF_4_266'
            ColumnNamesPartFunc(23) = 'PF_4_571'
            ColumnNamesPartFunc(24) = 'PF_4_898'
            ColumnNamesPartFunc(25) = 'PF_5_000'
            ColumnNamesPartFunc(26) = 'PF_5_248'
            ColumnNamesPartFunc(27) = 'PF_5_623'
            ColumnNamesPartFunc(28) = 'PF_6_026'
            ColumnNamesPartFunc(29) = 'PF_6_457'
            ColumnNamesPartFunc(30) = 'PF_6_918'
            ColumnNamesPartFunc(31) = 'PF_7_413'
            ColumnNamesPartFunc(32) = 'PF_7_943'
            ColumnNamesPartFunc(33) = 'PF_8_511'
            ColumnNamesPartFunc(34) = 'PF_9_120'
            ColumnNamesPartFunc(35) = 'PF_9_375'
            ColumnNamesPartFunc(36) = 'PF_9_772'
            ColumnNamesPartFunc(37) = 'PF_10_471'
            ColumnNamesPartFunc(38) = 'PF_11_220'
            ColumnNamesPartFunc(39) = 'PF_12_023'
            ColumnNamesPartFunc(40) = 'PF_12_882'
            ColumnNamesPartFunc(41) = 'PF_13_804'
            ColumnNamesPartFunc(42) = 'PF_14_791'
            ColumnNamesPartFunc(43) = 'PF_15_849'
            ColumnNamesPartFunc(44) = 'PF_16_982'
            ColumnNamesPartFunc(45) = 'PF_18_197'
            ColumnNamesPartFunc(46) = 'PF_18_750'
            ColumnNamesPartFunc(47) = 'PF_19_498'
            ColumnNamesPartFunc(48) = 'PF_20_893'
            ColumnNamesPartFunc(49) = 'PF_22_387'
            ColumnNamesPartFunc(50) = 'PF_23_988'
            ColumnNamesPartFunc(51) = 'PF_25_704'
            ColumnNamesPartFunc(52) = 'PF_27_542'
            ColumnNamesPartFunc(53) = 'PF_29_512'
            ColumnNamesPartFunc(54) = 'PF_31_623'
            ColumnNamesPartFunc(55) = 'PF_33_884'
            ColumnNamesPartFunc(56) = 'PF_36_308'
            ColumnNamesPartFunc(57) = 'PF_37_500'
            ColumnNamesPartFunc(58) = 'PF_38_905'
            ColumnNamesPartFunc(59) = 'PF_41_687'
            ColumnNamesPartFunc(60) = 'PF_44_668'
            ColumnNamesPartFunc(61) = 'PF_47_863'
            ColumnNamesPartFunc(62) = 'PF_51_286'
            ColumnNamesPartFunc(63) = 'PF_54_954'
            ColumnNamesPartFunc(64) = 'PF_58_884'
            ColumnNamesPartFunc(65) = 'PF_63_096'
            ColumnNamesPartFunc(66) = 'PF_67_608'
            ColumnNamesPartFunc(67) = 'PF_72_444'
            ColumnNamesPartFunc(68) = 'PF_75_000'
            ColumnNamesPartFunc(69) = 'PF_77_625'
            ColumnNamesPartFunc(70) = 'PF_83_176'
            ColumnNamesPartFunc(71) = 'PF_89_125'
            ColumnNamesPartFunc(72) = 'PF_95_499'
            ColumnNamesPartFunc(73) = 'PF_102_329'
            ColumnNamesPartFunc(74) = 'PF_109_648'
            ColumnNamesPartFunc(75) = 'PF_117_490'
            ColumnNamesPartFunc(76) = 'PF_125_893'
            ColumnNamesPartFunc(77) = 'PF_134_896'
            ColumnNamesPartFunc(78) = 'PF_144_544'
            ColumnNamesPartFunc(79) = 'PF_150_000'
            ColumnNamesPartFunc(80) = 'PF_154_882'
            ColumnNamesPartFunc(81) = 'PF_165_959'
            ColumnNamesPartFunc(82) = 'PF_177_828'
            ColumnNamesPartFunc(83) = 'PF_190_546'
            ColumnNamesPartFunc(84) = 'PF_204_174'
            ColumnNamesPartFunc(85) = 'PF_218_776'
            ColumnNamesPartFunc(86) = 'PF_225_000'
            ColumnNamesPartFunc(87) = 'PF_234_423'
            ColumnNamesPartFunc(88) = 'PF_251_189'
            ColumnNamesPartFunc(89) = 'PF_269_153'
            ColumnNamesPartFunc(90) = 'PF_288_403'
            ColumnNamesPartFunc(91) = 'PF_300_000'
            ColumnNamesPartFunc(92) = 'PF_309_030'
            ColumnNamesPartFunc(93) = 'PF_331_131'
            ColumnNamesPartFunc(94) = 'PF_354_813'
            ColumnNamesPartFunc(95) = 'PF_380_189'
            ColumnNamesPartFunc(96) = 'PF_407_380'
            ColumnNamesPartFunc(97) = 'PF_436_516'
            ColumnNamesPartFunc(98) = 'PF_467_735'
            ColumnNamesPartFunc(99) = 'PF_500_000'
            ColumnNamesPartFunc(100) = 'PF_501_187'
            ColumnNamesPartFunc(101) = 'PF_537_032'
            ColumnNamesPartFunc(102) = 'PF_575_440'
            ColumnNamesPartFunc(103) = 'PF_616_595'
            ColumnNamesPartFunc(104) = 'PF_660_693'
            ColumnNamesPartFunc(105) = 'PF_707_946'
            ColumnNamesPartFunc(106) = 'PF_758_578'
            ColumnNamesPartFunc(107) = 'PF_812_831'
            ColumnNamesPartFunc(108) = 'PF_870_964'
            ColumnNamesPartFunc(109) = 'PF_933_254'
            ColumnNamesPartFunc(110) = 'PF_1000_000'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define log10 of temperatures for partition funciton for
            !< T = 1.072, 1.148, 1.23, 1.318, 1.413, 1.514, 1.622, 1.738, 1.862, 1.995, 2.138, 2.291, 2.455, 2.63, 2.725, 2.818, 3.02, 3.236, 3.467, 
            !<     3.715, 3.981, 4.266, 4.571, 4.898, 5, 5.248, 5.623, 6.026, 6.457, 6.918, 7.413, 7.943, 8.511, 9.12, 9.375, 9.772, 10.471, 11.22, 
            !<     12.023, 12.882, 13.804, 14.791, 15.849, 16.982, 18.197, 18.75, 19.498, 20.893, 22.387, 23.988, 25.704, 27.542, 29.512, 31.623, 
            !<     33.884, 36.308, 37.5, 38.905, 41.687, 44.668, 47.863, 51.286, 54.954, 58.884, 63.096, 67.608, 72.444, 75, 77.625, 83.176, 89.125,
            !<     95.499, 102.329, 109.648, 117.49, 125.893, 134.896, 144.544, 150, 154.882, 165.959, 177.828, 190.546, 204.174, 218.776, 225, 
            !<     234.423, 251.189, 269.153, 288.403, 300, 309.03, 331.131, 354.813, 380.189, 407.38, 436.516, 467.735, 500, 501.187, 537.032, 
            !<     575.44, 616.595, 660.693, 707.946, 758.578, 812.831, 870.964, 933.254, 1000
            TempPartFunc = (/0.030194785356751d0, 0.059941888061955d0, 0.089905111439398d0, 0.119915410257991d0, 0.150142161848559d0, &
                             0.180125875164054d0, 0.210050849875137d0, 0.240049772112648d0, 0.269979676645324d0, 0.299942900022767d0, &
                             0.330007700872759d0, 0.360025089189397d0, 0.390051496458987d0, 0.419955748489758d0, 0.435366506612661d0, &
                             0.449940988773338d0, 0.480006942957151d0, 0.510008512940235d0, 0.539953841656397d0, 0.569958818096594d0, &
                             0.599992177584098d0, 0.630020851113410d0, 0.660011221289331d0, 0.690018780788695d0, 0.698970004336019d0, &
                             0.719993826367604d0, 0.749968083509403d0, 0.780029127337338d0, 0.810030786405839d0, 0.839980557678343d0, &
                             0.869994000121742d0, 0.899984562549391d0, 0.929980590515515d0, 0.959994838328416d0, 0.971971276399756d0, &
                             0.989983458301399d0, 1.019988159591290d0, 1.049992856920140d0, 1.080012847107930d0, 1.109983294819890d0, &
                             1.140004950619450d0, 1.169997537066570d0, 1.200001865406600d0, 1.229988836544810d0, 1.259999795081890d0, &
                             1.273001272063740d0, 1.289990066054320d0, 1.320000804264730d0, 1.349995899262860d0, 1.379994040165740d0, &
                             1.410000712543460d0, 1.439995473953810d0, 1.469998642218750d0, 1.500003068051690d0, 1.529994673069140d0, &
                             1.560002326773370d0, 1.574031267727720d0, 1.590005419651330d0, 1.620000642300070d0, 1.649996507466040d0, &
                             1.679999916229220d0, 1.709998828025190d0, 1.739999309401970d0, 1.769997304028540d0, 1.800001827817970d0, &
                             1.829998088697380d0, 1.860002421901100d0, 1.875061263391700d0, 1.890001613184640d0, 1.919998030964020d0, &
                             1.949999542859920d0, 1.979998823973740d0, 2.009998730048060d0, 2.040000714473320d0, 2.070000903802340d0, &
                             2.100001582801350d0, 2.129999071957430d0, 2.160000068881300d0, 2.176091259055680d0, 2.190000948069660d0, &
                             2.220000809286660d0, 2.250000144081290d0, 2.279999836361080d0, 2.310000437185500d0, 2.339999677628230d0, &
                             2.352182518111360d0, 2.370000219475120d0, 2.400000616976380d0, 2.429999224858320d0, 2.459999773650270d0, &
                             2.477121254719660d0, 2.490000641890950d0, 2.519999840669680d0, 2.549999523574670d0, 2.579999547278480d0, &
                             2.609999703842230d0, 2.640000166906100d0, 2.669999868814090d0, 2.698970004336020d0, 2.699999797554580d0, &
                             2.730000164674160d0, 2.760000047292730d0, 2.789999998688880d0, 2.819999705510490d0, 2.850000132271100d0, &
                             2.880000243300670d0, 2.910000258512930d0, 2.940000204462930d0, 2.969999860022620d0, 3.000000000000000d0/)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< determine order of partition function in database: ascending or descending order
            TempLow = min(TempPartFunc(1), TempPartFunc(NumberOfTemperatures))
            TempHigh = max(TempPartFunc(1), TempPartFunc(NumberOfTemperatures))
            if (TempPartFunc(1) < TempPartFunc(NumberOfTemperatures)) then
                Firsti = NumberOfTemperatures
                Lasti = 1
                stepi = (-1)
            else
                Firsti = 1
                Lasti = NumberOfTemperatures
                stepi = 1
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define column names of table transitions
            ColumnNameForNameTransitions = "T_Name"
            ColumnNameForFreqTransitions = "T_Frequency"
            ColumnNameForIntTransitions = "T_Intensity"
            ColumnNameForEinsteinATransitions = "T_EinsteinA"
            ColumnNameForFreqErrTransitions = "T_Uncertainty"
            ColumnNameForELowTransitions = "T_EnergyLower"
            ColumnNameForgUpTransitions = "T_UpperStateDegeneracy"
            Column300KPartFunc = 91                                                         !< (only used, if use_intensity_flag == .true): define the
                                                                                            !< column number, containing the partition function for 300 K

            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get myXCLASS parameter
            call GetmyXCLASSParameter


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get partition functions for selected molecules and describe partition functions with polynominal
            call GetPartitionFunction(ok)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get informations for each transition in the given frequency ranges
            call GetTransitionParameters(ok)

            ! Debug:
            ! stop 'Subroutine ModelInit finished!'


            !< initialize MPI environment
            if (printflag) then
                print '(9x, "Initialize MPI environment .. ", $)'
            endif


            !< resize AtOnceFunction variable (only for MASTER thread)
            !< this section is neccessary to avoid changes in the Algorithm core routine
            if (allocated(AtOnceFunction)) then
                deallocate(AtOnceFunction, stat = deallocstatus)
                if (deallocstatus /= 0) then                                                !< is all ok?
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine ModelInit:")')
                    write(logchannel,'(2x,"Can not deallocate variable AtOnceFunction for thread = ", I5, ".")') myrank
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("deallocstatus = ", I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine ModelInit:")'
                    print '(2x,"Can not deallocate variable AtOnceFunction for thread = ", I5, ".")', myrank
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("deallocstatus = ", I4)', deallocstatus
                    print '(" ")'
                    call MPI_Abort(MPI_COMM_WORLD, ierr)
                    stop
                endif
            endif
            allocate(AtOnceFunction(0:0, NumberExpFiles, MaxColY, MaxExpLength), stat = allocstatus)
            if (allocstatus /= 0) then                                                 !< is all ok?
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine ModelInit:")')
                write(logchannel,'(2x,"Can not allocate variable AtOnceFunction for thread = ", I5, ".")') myrank
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')
                close(logchannel)

                print '(" ")'
                print '("Error in subroutine ModelInit:")'
                print '(2x,"Can not allocate variable AtOnceFunction for thread = ", I5, ".")', myrank
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("allocstatus = ",I4)',allocstatus
                print '(" ")'
                call MPI_Abort(MPI_COMM_WORLD, ierr)
                stop
            endif
            AtOnceFunction = 0.d0


            !< redefine variable 'ParallelizationFlag'
            TotalNumberOfThreads = ParallelizationFlag
            NumberOfUsedThreads = 1                                                         !< in Module 'FunctionCalling' we need only one thread
            ParallelizationMethod = "MPI"


            !< wake up slave threads
            call SlaveWork(2)


            !< broadcast variables
            call BroadCastMPIVariables


            !< we've finished
            if (printflag) then
                print '("done!")'
            endif


            !< sleep little slave threads
            call SlaveWork(4)
            return
        end subroutine ModelInit


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunctionLM
        !>
        !> calculates the chi2 values for the Levenberg-Marquard algorithm
        !>
        !>
        !> input variables:     ma:                 total number of parameters
        !>                      a:                  array containing the parameter set
        !>                      ia:                 flags for including/excluding parameter in the fit
        !>                      NumberFreeParameterCopy:    number of fitted parameters
        !>                      fitparam:           parameter which have to be optimized
        !>                      colx:               number of columns in experimental x data
        !>                      NumFile:            number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>                      FitFunctionOut:     values of the fit function at all calculated points
        !>                      Chi2Values:         values of the fit function at all calculated points
        !>                      alpha:              matrix alpha (only used for Levenberg-Marquardt algorithm)
        !>                      beta2:              beta2 array (only used for Levenberg-Marquardt algorithm)
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 31.07.2014
        !>
        subroutine ModelCalcChiFunctionLM(ma, a, ia, NumberFreeParameterCopy, fitparam, colx, NumFile, MaxL, MaxCol, FitFunctionOut, Chi2Values, &
                                          alpha, beta2)

            implicit none
            integer :: i, j, k, l, m, n, fitnum, NumberFile                                 !< loop variables
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: counter                                                              !< counter for ModelFunction
            integer :: NumberFreeParameterCopy                                              !< number of fitted parameters
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            integer :: NumParameterVectors                                                  !< number of parameter vectors in ParameterVectorSet
            real*8 :: dy, sig2i, wt, ymod                                                   !< working variables
            real*8 :: value                                                                 !< calculated value of the fit function
            real*8 :: variation, d1, d2                                                     !< working variables
            real*8, dimension(ma) :: steph                                                  !< working variable
            real*8, dimension(ma) :: modelparam                                             !< array containing the parameter set
            real*8, dimension(ma) :: modelparamcopy                                         !< copy of modelparam
            real*8, dimension(ma) :: a                                                      !< array containing the parameter set
            real*8, dimension(ma, ma) :: alpha                                              !< matrix alpha
            real*8, dimension(NumberFreeParameterCopy) :: beta2                             !< beta2 array
            real*8, dimension(NumberFreeParameterCopy) :: fitparam                          !< parameter which have to be optimized
            real*8, dimension(NumFile, MaxL, MaxCol) :: FitFunctionOut                      !< values of the fit function at all calculated points
            real*8, dimension(NumFile, MaxL, MaxCol) :: Chi2Values                          !< values of the fit function at all calculated points
            real*8, dimension(NumberFreeParameterCopy, NumFile, MaxCol, MaxL) :: GradientHelp
            real*8, dimension(ma) :: dyda                                                   !< gradient of the fit function
            real*8, dimension(NumberFreeParameterCopy + 1) :: chi2ValuesVector              !< vector containing chi2 values for each parameter vector
                                                                                            !< in ParameterVectorSet
            real*8, dimension(NumberFreeParameterCopy + 1, NumberFreeParameterCopy) :: ParameterVectorSet
                                                                                            !< set of parameter vectors calculated by model module
            logical :: IntegerTrue                                                          !< flag for identification of integer numbers
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit
            character(len=100) :: HelpString                                                !< working variables
            character(len=100), dimension(NumberFreeParameterCopy) :: FormattedParmValues   !< formatted parameter values for chi2 log file


            character(len=50) :: valueh, valuel                                             !< working variable for determine gradient
            logical :: equal_flag                                                           !< required for string comparison
            logical, dimension(ma) :: ChangeSign_Flag                                       !< change sign flag
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function is stored or not


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize values for fit function calculation
            Do j = 1, NumberFreeParameterCopy                                               !< Initialize (symmetric) alpha, beta2.
                Do k = 1, j
                    alpha(j,k) = 0.d0                                                       !< clear alpha partially
                end Do
                beta2(j) = 0.d0                                                             !< clear beta2
            end Do


            !< update the parameter set with the current values of the fit parameters some algorithms optimize only the parameters in the array fitparam
            !< all other parameter in the parameter set a are not included in these algorithms but are required for the calculation of the fit function
            k = 0
            Do i = 1, ma
                if (ia(i)) then
                    k = k + 1
                    a(i) = fitparam(k)
                endif
            end Do
            ParameterVectorSet(1, :) = fitparam(:)                                          !< first entry of ParameterVectorSet contains the current
                                                                                            !< parameter vector
            ! Debug:
            ! print*,'>>>>>>>>>>>>>fitparam = ', fitparam(:)


            !< check, if gradient is necessary (for Levenberg-Marquardt it is always necessary!)
            NumParameterVectors = 1
            if (Gradientflag) then
                NumParameterVectors = NumberFreeParameterCopy + 1


                !< initialize some working parameter
                steph = 0.d0
                ChangeSign_Flag = .false.
                modelparamcopy = a                                                          !< dublicate array containing the parameter set


                !< calculate parameter vector for gradient calculation

                !$omp parallel default(shared) &
                !$omp shared(OutputFileFormat, NumberRanges, ValueEmptyOutputFile, LSRAdjustement, FirstPointExpData, LastPointExpData, idum) &
                !$omp shared(MinRange, MaxRange, NaNReplaceString, StandardWorkingDirectory, CharacterForComments, CharacterSeperatingColumns) &
                !$omp shared(ResamplingMethod, InterpolationMethod, RenormalizedChi2, OnlyYColumn, LSRAdjustementFitFlag, NormalizationFlag) &
                !$omp shared(AtOnceGradient, ia, paramset, modelparamcopy, ConverterInfit) &
                !$omp shared(ParameterName, ParameterFormat, LeadingString, TrailingString, ParamVisible, NumberLinesOutput) &
                !$omp shared(printflag, chisqValues, NumberExpFiles, modelnumber, lengthexpdata, currentpath, NumberHeaderLines, QualityLimit) &
                !$omp shared(FitParameterName, FitParameterValue, NumberColumnsBegin, NumberColumnsEnd, NumberParamPerLine, CalculationMethod) &
                !$omp shared(CommandWordOutput, DetChi2, MaxParameter, ExternalThreadNumber, expdatax, expdatay, expdatae, TempDirectory) &
                !$omp shared(NumberXColumns, NumberYColumns, MaxColX, MaxColY, parameternumber, NumberInputFiles, ParallelizationFlag, JobID) &
                !$omp shared(MaxInputLines, UseCalculationReduction, WriteChi2Flag, Gradientflag,  initflag, CalculatedParameterSets) &
                !$omp shared(LastAlgorithmFlag, ChangeSign_Flag, steph, NumberFreeParameterCopy, ParameterVectorSet, fitparam) &
                !$omp shared(CurrentNumberLinesCalcReduction, ochisq, NumberLinesChi2, InputDataPath, FitFktInput, ModelFunction, FitFktOutput) &
                !$omp shared(ExeCommandStartScript, NumberOutputFiles, CurrentExpFile, GradientMethod, PathStartScript, CurrentYColumn) &
                !$omp shared(ExpData_reversed_flag, BestSitesModelValues, BestSitesChi2Values, BestSitesParamSet, GradientVariationValue) &
                !$omp private(fitnum, i, j, k, modelparam, d1, d2, sig2i, NumberFile, variation, value, valueh, valuel) &
                !$omp private(IntegerTrue, NumInputFile_index, i_index, j_index, equal_flag)
                !$omp do

                Do fitnum = 1, NumberFreeParameterCopy                                      !< loop over all free parameter
                    !$omp critical
                    ParameterVectorSet(fitnum + 1, :) = fitparam(:)
                    !$omp end critical
                    modelparam = modelparamcopy                                             !< load unmodified parameter values
                    i = ConverterInfit(fitnum)                                              !< get appropriate parameter index within parameter set
                    call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)

                    ! Debug:
                    ! print*,'fitnum, NumInputFile_index, i_index, j_index = ', fitnum, NumInputFile_index, i_index, j_index


                    !< check if parameter is within parameter limits
                    if (modelparamcopy(i) < paramset(3, i) .or. modelparamcopy(i) > paramset(4, i)) then
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                        write(logchannel,'(2x,"The parameter ",A," is out of limits.")') &
                                          trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        write(logchannel,'(2x,"Upper limit = ", ES25.15)') paramset(4, i)
                        write(logchannel,'(2x,"Lower limit = ", ES25.15)') paramset(3, i)
                        write(logchannel,'(2x,"Value of parameter = ", ES25.15)') modelparamcopy(i)
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')

                        print '(" ")'
                        print '(" ")'
                        print '(" ")'
                        print '(11x,"Error in subroutine ModelCalcChiFunctionLM:")'
                        print '(13x,"The parameter ",A," is out of limits.")', trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        print '(13x,"Upper limit = ", ES25.15)', paramset(4, i)
                        print '(13x,"Lower limit = ", ES25.15)', paramset(3, i)
                        print '(13x,"Value of parameter = ", ES25.15)',modelparamcopy(i)
                        print '(" ")'
                        print '(13x,"Program aborted!")'
                        stop
                    endif


                    !< determine strength of variation
                    variation = GradientVariationValue                                      !< variation of the parameter in percent/100
                    !$omp critical
                    steph(i) = dabs(modelparamcopy(i) * variation)                          !< define stepsize for foating point numbers
                    if (modelparamcopy(i) == 0.d0) then
                        if (IntegerTrue) then                                               !< is parameter an integer ??
                            steph(i) = 1.d0                                                 !< variation of the parameter in percent/100
                        else
                            steph(i) = variation                                            !< variation of the parameter in percent/100
                        endif
                    elseif (IntegerTrue .and. steph(i) < 1.d0) then
                        steph(i) = 1.d0
                    endif
                    !$omp end critical

                    ! Debug:
                    ! print*,'i = ', i
                    ! print*,'modelparamcopy(i) = ', modelparamcopy(i)
                    ! print*,'steph(i) = ',steph(i)
                    ! print*,'modelparamcopy(i) + steph(i) = ', modelparamcopy(i) + steph(i)
                    ! print*,'paramset(3, i) = ', paramset(3, i)
                    ! print*,'paramset(4, i) = ', paramset(4, i)
                    ! print*,'modelparamcopy(i) - steph(i) = ', modelparamcopy(i) - steph(i)


                    !< test, if we can accelerate the calculation
                    value = modelparamcopy(i) + steph(i)
                    if (value < paramset(3, i) .or. paramset(4, i) < value) then
                        if (value > paramset(4, i)) then                                    !< is f(x_i + h) > upper limit ?
                            value = modelparamcopy(i) - steph(i)                            !< use f(x_i - h)
                            if (value < paramset(3, i)) then
                                if (dabs(modelparamcopy(i) - paramset(4, i)) < 1.d-12) then
                                    write(logchannel,*)
                                    write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                                    write(logchannel,'(2x,"The gradient for parameter ",A," cannot be calculated.")') &
                                                      trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                                    write(logchannel,'(2x,"The variation runs out of limits.")')
                                    write(logchannel,*)
                                    write(logchannel,'(2x,"Please increase upper and lower limits for this parameter or reduce value of variation.")')
                                    write(logchannel,*)
                                    write(logchannel,*)
                                    write(logchannel,'(2x,"Upper limit = ",ES25.15)') paramset(4, i)
                                    write(logchannel,'(2x,"Lower limit = ",ES25.15)') paramset(3, i)
                                    write(logchannel,'(2x,"value of variation = ", ES25.15)') variation
                                    write(logchannel,'(2x,"Value of parameter = ", ES25.15)') modelparamcopy(i)
                                    write(logchannel,'(" ")')
                                    write(logchannel,'("Program aborted!")')

                                    print '(" ")'
                                    print '(" ")'
                                    print '(" ")'
                                    print '("Error in subroutine ModelCalcChiFunctionLM:")'
                                    print '(2x,"The gradient for parameter ",A," cannot be calculated.")', &
                                                      trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                                    print '(2x,"The variation runs out of limits.")'
                                    print '(" ")'
                                    print '(2x,"Please increase upper and lower limits for this parameter or reduce value of variation.")'
                                    print '(" ")'
                                    print '(" ")'
                                    print '(2x,"Upper limit = ", ES25.15)', paramset(4, i)
                                    print '(2x,"Lower limit = ", ES25.15)', paramset(3, i)
                                    print '(2x,"value of variation = ", ES25.15)', variation
                                    print '(2x,"Value of parameter = ", ES25.15)', modelparamcopy(i)
                                    print '(" ")'
                                    print '("Program aborted!")'
                                    stop
                                endif
                                value = paramset(4, i)
                            else
                                !$omp critical
                                ChangeSign_Flag(i) = .true.
                                !$omp end critical
                            endif
                        endif
                    endif

                    ! Debug:
                    ! print*,'> value, modelparamcopy(i), steph(i) = ',value, modelparamcopy(i), steph(i)


                    !< check, if the variation leads in combination with the Format to a variation in the current parameter
                    if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                        .or. index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                        write(valueh, ParameterFormat(NumInputFile_index, i_index, j_index)) int(value)
                        write(valuel, ParameterFormat(NumInputFile_index, i_index, j_index)) int(modelparamcopy(i))
                    else
                        write(valueh, ParameterFormat(NumInputFile_index, i_index, j_index)) value
                        write(valuel, ParameterFormat(NumInputFile_index, i_index, j_index)) modelparamcopy(i)
                    endif
                    equal_flag = .true.
                    valueh = adjustl(valueh)
                    valuel = adjustl(valuel)
                    if (len_trim(valueh) == len_trim(valuel)) then
                        Do j = 1, len_trim(valueh)
                            if (valueh(j:j) /= valuel(j:j)) then
                                equal_flag = .false.
                                exit
                            endif
                        end Do
                    else
                        equal_flag = .false.
                    endif
                    if (equal_flag) then                                                    !< both expressions are equal
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                        write(logchannel,'(2x,"The format specification of the parameter ",A)') &
                                trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        write(logchannel,'(2x,"prevents the variation of the current parameter.")')
                        write(logchannel,'(2x," ")')
                        write(logchannel,'(2x,"The gradient entry for this parameter is set to zero. Therefore")')
                        write(logchannel,'(2x,"no variation of this parameter in the current iteration is done")')
                        write(logchannel,'(" ")')

                        print '(" ")'
                        print '(" ")'
                        print '(" ")'
                        print '(11x,"Error in subroutine ModelCalcChiFunctionLM:")'
                        print '(13x,"The format specification of the parameter ",A)', &
                               trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        print '(13x,"prevents the variation of the current parameter.")'
                        print '(" ")'
                        print '(13x,"The gradient entry for this parameter is set to zero. Therefore")'
                        print '(13x,"no variation of this parameter in the current iteration is done")'
                        print '(" ")'
                    endif


                    !< modify the ith parameter
                    !$omp critical
                    modelparam(i) = value                                                   !< modify value of the ith parameter
                    ParameterVectorSet(fitnum + 1, fitnum) = value
                    !$omp end critical

                end Do

                !$omp end do
                !$omp end parallel                                                          !< end of parallel environment

                modelparam = modelparamcopy                                                 !< restore old paramter values
            endif

            ! Debug:
            !    Do l = 2, NumParameterVectors
            !        Do k = 1, NumberFreeParameterCopy
            !            if (dabs(ParameterVectorSet(l, k) - ParameterVectorSet(1, k)) > 1.d-6) then
            !                print*,'l, k, ParameterVectorSet(l, k) = ', l, k, ParameterVectorSet(l, k), ParameterVectorSet(1, k), &
            !                                                            dabs(ParameterVectorSet(l, k) - ParameterVectorSet(1, k))
            !            endif
            !        end Do
            !    end Do
            !    stop


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            ModelFunctionFlag = .true.
            call ModelCalcChiFunction(NumberFreeParameterCopy, NumParameterVectors, ParameterVectorSet(1:NumParameterVectors,:), ModelFunctionFlag, &
                                      ma, a, ia, colx, NumFile, MaxL, MaxCol, chi2ValuesVector)
            ! Debug:
            ! print*,' '
            ! print*,'chi2ValuesVector(:) = ', chi2ValuesVector(:)
            ! stop 'Test LM-algorithm up to here!'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< start loop for determine chi**2
            GradientHelp = 0.d0
            FitFunctionOut = 0.d0                                                           !< reset array containing the values of the fit function
            Chi2Values = 0.d0
            chisqValues = 0.d0
            counter = 0
            Do NumberFile = 1, NumberExpFiles                                               !< loop over exp. data files
                CurrentExpFile = NumberFile


                !< start loop over all lines(expdata)
                Do n = 1, NumberYColumns(NumberFile)                                        !< loop over y-columns
                    Do i = 1, lengthexpdata(NumberFile)                                     !< loop over all line of current exp. data file
                        CurrentYColumn = n
                        counter = counter + 1


                        !< get fit function
                        ymod = ModelFunction(1, NumberFile, n, i)

                        ! Debug:
                        !    print*,'>> NumberFile, n, i, counter, ymod = ', NumberFile, n, i, counter, ymod
                        !    print*,'>> ModelFunction(1, NumberFile, n, i) = ', ModelFunction(1, NumberFile, n, i)


                        !< get gradient
                        if (Gradientflag) then
                            dyda = 0.d0
                            Do l = 1, NumberFreeParameterCopy
                                j = ConverterInfit(l)                                       !< get appropriate parameter index within parameter set

                                ! order new: NumberFile, n, i, l,      j
                                ! order old: NumberFile, n, i, l,      j
                                ! order org: NumberFile, k, j, fitnum, i

                                ! Debug:
                                !    print*,'l, j = ', l, j
                                !    print*,'ParameterVectorSet(l + 1, l) = ', ParameterVectorSet(l + 1, l)
                                !    print*,'steph(j) = ', steph(j)
                                !    print*,'ChangeSign_Flag(j) = ', ChangeSign_Flag(j)
                                !    print*,'ModelFunction(l + 1, NumberFile, n, i) = ', ModelFunction(l + 1, NumberFile, n, i)
                                !    print*,'GradientMethod = ', GradientMethod


                                !< determine gradient for numerical recepies version
                                if (GradientMethod == 1) then
                                    if (ParameterVectorSet(l + 1, l) > paramset(4, j) .or. ChangeSign_Flag(j)) then
                                        dyda(j) = (ModelFunction(1, NumberFile, n, i) - ModelFunction(l + 1, NumberFile, n, i)) / steph(j)
                                    else
                                        dyda(j) = (ModelFunction(l + 1, NumberFile, n, i) - ModelFunction(1, NumberFile, n, i)) / steph(j)
                                    endif


                                !< determine gradient for minpack version
                                elseif (GradientMethod == 2) then
                                    sig2i = 1.d0
                                    if (expdatae(NumberFile, i, n) /= 0.d0) then            !< do the experimental datas include errors
                                        sig2i = 1.d0 / (expdatae(NumberFile, i, n) * expdatae(NumberFile, i, n))  !< define sig2i factor
                                    endif
                                    d1 = (expdatay(NumberFile, i, n) - ModelFunction(1, NumberFile, n, i))**2 * sig2i
                                    d2 = (expdatay(NumberFile, i, n) - ModelFunction(l + 1, NumberFile, n, i))**2 * sig2i
                                    if (ParameterVectorSet(l + 1, l) > paramset(4, j) .or. ChangeSign_Flag(j)) then
                                        ! dyda(j) = (d1 - d2) / steph(j)
                                        GradientHelp(l, NumberFile, n, i) = (d1 - d2) / steph(j)
                                    else
                                        ! dyda(j) = (d2 - d1) / steph(j)
                                        GradientHelp(l, NumberFile, n, i) = (d2 - d1) / steph(j)
                                    endif
                                    dyda(j) = GradientHelp(l, NumberFile, n, i)
                                endif
                            end Do

                            ! Debug:
                            ! print*,'ymod = ', ymod
                            ! print*,'dyda(:) = ', dyda(:)
                            ! print*,'a = ', a
                            ! print*,'############################################################################'
                            ! stop
                        endif
                        FitFunctionOut(NumberFile, i, n) = ymod                             !< save value of fit function

                        ! Debug:
                        !    print*,' '
                        !    print*,' '
                        !    print*,'NumberExpFiles = ',NumberExpFiles
                        !    print*,'lengthexpdata(NumberFile) = ',lengthexpdata(NumberFile)
                        !    print*,'NumberYColumns(NumberFile) = ',NumberYColumns(NumberFile)
                        !    print*,'NumberFile = ',NumberFile
                        !    print*,'i = ',i
                        !    print*,'n = ',n
                        !    print*,'lenposdatexp = ',lenposdatexp
                        !    print*,'posdatexp = ',posdatexp
                        !    print*,'expdatay(NumberFile,i,n) = ',expdatay(NumberFile,i,n)
                        !    print*,'ymod = ',ymod
                        !    print*,'dyda = ',dyda(1:NumberFreeParameterCopy)
                        !    if (i==3) stop


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< determine chi**2 by calculating the difference (y_i^{obs) - y_i(fit))**2
                        if (abs(DetChi2) == 1) then

                            sig2i = 1.d0
                            if (expdatae(NumberFile,i,n) /= 0.d0) then                      !< do the experimental datas include errors
                                sig2i = 1.d0/(expdatae(NumberFile,i,n) * expdatae(NumberFile,i,n))      !< define sig2i factor
                            endif
                            dy = (expdatay(NumberFile, i, n) - ymod)                        !< define distance between fit and data
                            Chi2Values(NumberFile, i, n) = dy * dy * sig2i                  !< save chi^2
                            chisqValues(0) = chisqValues(0) + dy * dy * sig2i               !< And find chi^2.


                            if (Gradientflag) then
                                j = 0
                                Do l = 1, ma                                                !< loop over all parameters
                                    if (ia(l)) then                                         !< is the lth parameters optimized?
                                        j = j + 1
                                        wt = dyda(l) * sig2i                                !< define weighting factor
                                        k = 0
                                        Do m = 1, l                                         !< determine alpha matrix
                                            if (ia(m)) then
                                                k = k + 1
                                                alpha(j,k) = alpha(j,k) + wt * dyda(m)
                                            endif
                                        end Do
                                        beta2(j) = beta2(j) + dy * wt                       !< calculate beta2 array
                                    endif
                                end Do
                            endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< determine chi**2 by calculating the difference (y_i^{obs)**2 - y_i(fit)**2)
                        elseif (abs(DetChi2) == 2) then

                            sig2i = 1.d0
                            if (expdatae(NumberFile,i,n) /= 0.d0) then                      !< do the experimental datas include errors
                                sig2i = 1.d0/(expdatae(NumberFile,i,n) * expdatae(NumberFile,i,n))      !< define sig2i factor
                            endif
                            dy = (expdatay(NumberFile, i, n)**2 - ymod**2)
                            Chi2Values(NumberFile, i, n) = dy * dy * sig2i                  !< save chi^2
                            chisqValues(0) = chisqValues(0) + dy * dy * sig2i               !< And find chi^2.


                            if (Gradientflag) then
                                j = 0
                                Do l = 1, ma                                                !< loop over all parameters
                                    if (ia(l)) then                                         !< is the lth parameters optimized?
                                        j = j + 1
                                        wt = dyda(l) * sig2i                                !< define weighting factor
                                        k = 0
                                        Do m = 1, l                                         !< determine alpha matrix
                                            if (ia(m)) then
                                                k = k + 1
                                                alpha(j, k) = alpha(j, k) + wt * dyda(m)
                                            endif
                                        end Do
                                        beta2(j) = beta2(j) + dy * wt                       !< calculate beta2 array
                                    endif
                                end Do
                            endif

                        endif
                    end Do                                                                  !< loop over all line of current exp. data file
                end Do                                                                      !< loop over y-columns
            end Do                                                                          !< loop over exp. data files

            ! Debug:
            ! print*,'chisqValues(0) = ', chisqValues(0)


            !< only used for MINPACK version of Levenberg-Marquardt algorithm
            if (GradientMethod == 2) then
                Do l = 1, NumberFreeParameterCopy
                    ModelFunction(l + 1, :, :, :) = GradientHelp(l, :, :, :)
                end Do
            endif


            !< Fill in the symmetric side.
            Do j = 2, NumberFreeParameterCopy
                Do k = 1, (j - 1)
                   alpha(k, j) = alpha(j, k)
                end Do

                ! Debug:
                ! print*,'j = ', j
                ! print*,'alpha(j,:) = ', alpha(j,:NumberFreeParameterCopy)
                ! print*,'beta2(j) = ', beta2(j)
            end Do


            !< writing current value of chi**2 and corresponding values of parameters to file
            if (WriteChi2Flag) then
                NumberLinesChi2 = NumberLinesChi2 + 1
                k = 0
                FormattedParmValues(:)(:) = ""
                Do i = 1, ma
                    if (ia(i)) then
                        k = k + 1
                        a(i) = fitparam(k)

                        ! Debug:
                        !   print*,'fitparam(k) = ',k,fitparam(k)


                        !< build list with fit parameters
                        HelpString = ""
                        call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)
                        if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                            .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(i))
                        else
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(i)
                        endif
                        FormattedParmValues(k) = trim(adjustl(HelpString))
                    endif
                end Do


                !< write line of formatted parameter values to log file
                write(Chi2Channel,'(ES25.15,$)') chisqValues(0)
                Do i = 1, NumberFreeParameterCopy
                    write(Chi2Channel,'(1x,A,$)') " " // trim(adjustl(FormattedParmValues(i)))
                end Do
                write(Chi2Channel,*)
            endif


            return
        end subroutine ModelCalcChiFunctionLM


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunctionGeneral
        !>
        !> calculates the chi2 values for several given parameter vector sets
        !>
        !>
        !> input variables:     ma:                 total number of parameters
        !>                      ia:                 flags for including/excluding parameter in the fit
        !>                      a:                  array containing the parameter set
        !>                      NumberParamVectors:         number of parameter vectors
        !>                      NumberFreeParameterCopy:    number of fitted parameters
        !>                      ParameterVectorSet:     
        !>                      NumFile:            number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>                      FitFunctionOut:     values of the fit function at all calculated points
        !>                      Chi2Values:         values of the fit function at all calculated points
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 19.08.2014
        !>
        subroutine ModelCalcChiFunctionGeneral(ma, ia, a, NumberParamVectors, NumberFreeParameterCopy, NumFile, MaxL, MaxCol, ParameterVectorSet, &
                                               chi2ValuesVector)

            implicit none
            integer :: ma                                                                   !< total number of parameters
            integer :: NumberParamVectors                                                   !< number of parameter vectors
            integer :: NumberFreeParameterCopy                                              !< number of fitted parameters
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: colx                                                                 !< max. number of x columns
            real*8, dimension(ma) :: a                                                      !< array containing the parameter set
            real*8, dimension(NumberParamVectors, NumberFreeParameterCopy) :: ParameterVectorSet   !< set of parameter vectors calculated by model module
            real*8, dimension(NumberParamVectors) :: chi2ValuesVector                       !< vector containing chi2 values for each parameter vector
                                                                                            !< in ParameterVectorSet
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit


            !< working variables
            integer :: i, k, l                                                              !< loop variables
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            character(len=100) :: HelpString                                                !< working variables
            character(len=100), dimension(NumberFreeParameterCopy) :: FormattedParmValues   !< formatted parameter values for chi2 log file
            logical :: IntegerTrue                                                          !< flag for identification of integer numbers
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function is stored or not


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            ModelFunctionFlag = .false.
            chi2ValuesVector = 0.d0
            colx = MaxColX
            call ModelCalcChiFunction(NumberFreeParameterCopy, NumberParamVectors, ParameterVectorSet, ModelFunctionFlag, ma, a, ia, colx, NumFile, &
                                      MaxL, MaxCol, chi2ValuesVector)

            ! Debug:
            ! print*,' '
            ! print*,'chi2ValuesVector(:) = ', chi2ValuesVector(:)
            ! stop 'Test LM-algorithm up to here!'


            !< writing current value of chi**2 and corresponding values of parameters to file
            Do l = 1, NumberParamVectors
                if (WriteChi2Flag) then
                    NumberLinesChi2 = NumberLinesChi2 + 1
                    k = 0
                    FormattedParmValues(:)(:) = ""

                    Do i = 1, ma
                        if (ia(i)) then
                            k = k + 1
                            a(i) = ParameterVectorSet(l, k)


                            !< build list with fit parameters
                            HelpString = ""
                            call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)
                            if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                                .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(i))
                            else
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(i)
                            endif
                            FormattedParmValues(k) = trim(adjustl(HelpString))
                        endif
                    end Do


                    !< write line of formatted parameter values to log file
                    write(Chi2Channel,'(ES25.15,$)') chi2ValuesVector(l)
                    Do i = 1, NumberFreeParameterCopy
                        write(Chi2Channel,'(1x,A,$)') " " // trim(adjustl(FormattedParmValues(i)))
                    end Do
                    write(Chi2Channel,*)
                endif
            end Do
            return
        end subroutine ModelCalcChiFunctionGeneral


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunction
        !>
        !> prepare call of subroutine 'ModelCalcCore'
        !>
        !>
        !> input variables:     NumberFreeParam:        number of free parameters
        !>                      NumParamVectors:        number of parameter vectors
        !>                      ParameterVectorSet:     parameter vector
        !>                      ModelFunctionFlag:      flag for indicating if model function values are stored or not
        !>                      ma:                     total number of parameters
        !>                      a:                      array containing the parameter set
        !>                      ia:                     flags for including/excluding parameter in the fit
        !>                      colx:                   number of columns in experimental x data
        !>                      NumFile:                number of experimental files
        !>                      MaxL:                   max number of lines of all experimental files
        !>                      MaxCol:                 max number of columns of all experimental files
        !>
        !> output variables:    chi2ValuesVector:       chi2 value for parameter vector
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 15.08.2014
        !>
        subroutine ModelCalcChiFunction(NumberFreeParam, NumParamVectors, ParameterVectorSet, ModelFunctionFlag, ma, a, ia, colx, NumFile, &
                                        MaxL, MaxCol, chi2ValuesVector)

            use omp_lib

            implicit none
            integer :: NumberFile                                                           !< loop variables
            integer :: NumberFreeParam                                                      !< number of free parameters
            integer :: NumParamVectors                                                      !< number of parameter vectors
            integer :: length                                                               !< total length of AtOnceFunction
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: dealloc_status, alloc_status                                         !< working variables for allocation/deallocation
            real*8, dimension(ma) :: a, acopy                                               !< array containing the parameter set
            real*8, dimension(NumParamVectors) :: chi2ValuesVector                          !< chi2 value for parameter vector
            real*8, dimension(NumParamVectors, NumberFreeParam) :: ParameterVectorSet       !< parameter vector set
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function values are stored or
                                                                                            !< not
            logical :: AllocateFlag                                                         !< used to avoid multiple allocation of memory
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit


            ! Debug:
            ! print*,'NumFile, MaxL, MaxCol = ', NumFile, MaxL, MaxCol
            ! print*,'ParallezitionMethod = ', ParallezitionMethod


            !< reset output variables
            chi2ValuesVector = 0.d0


            !< determine total length of AtOnceFunction
            length = 0
            Do NumberFile = 1, NumberExpFiles
                length = length + (NumberYColumns(NumberFile) * lengthexpdata(NumberFile))
            end Do


            !< make a copy of a
            acopy = a


            !< make a copy of NumParamVectors
            TotalNumParamVectors = NumParamVectors


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< (de)allocate memory for array containing all parameter vectors and chi2 vector
            !< the following lines are included to aviod reallocation for each call of this subroutine
            AllocateFlag = .false.
            if (.not. allocated(GlobalParamVectorSet)) then
                AllocateFlag = .true.
            else
                if (size(GlobalParamVectorSet) /= (NumParamVectors * NumberFreeParameter)) then
                    AllocateFlag = .true.
                endif
            endif

            ! Debug:
            ! print*,'>>>>>>>>>>>>>>>>>>myrank, AllocateFlag, TotalNumParamVectors = ', myrank ,AllocateFlag, TotalNumParamVectors


            if (AllocateFlag) then
                if (allocated(GlobalParamVectorSet)) then
                    deallocate(GlobalParamVectorSet, LocalChi2ValueVector, stat = dealloc_status)
                    if (dealloc_status /= 0) then                                           !< is all ok?
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelCalcChiFunction:")')
                        write(logchannel,'(2x,"Can not deallocate variables GlobalParamVectorSet and LocalChi2ValueVector for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,*)
                        write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine ModelCalcChiFunction:")'
                        print '(2x,"Can not deallocate variables GlobalParamVectorSet and LocalChi2ValueVector for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("dealloc_status = ",I4)', dealloc_status
                        print '(" ")'
                        call MPI_Abort(MPI_COMM_WORLD, ierr)
                        stop
                    endif
                endif
                allocate(GlobalParamVectorSet(NumParamVectors, NumberFreeParameter), LocalChi2ValueVector(NumParamVectors), stat = alloc_status)
                if (alloc_status /= 0) then                                                 !< is all ok?
                    write(logchannel,'(" ")')
                    write(logchannel,'("Error in subroutine ModelCalcChiFunction:")')
                    write(logchannel,'(2x,"Can not allocate variables GlobalParamVectorSet and LocalChi2ValueVector for thread = ", I5, ".")') myrank
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,'(" ")')
                    write(logchannel,'("alloc_status = ",I4)') alloc_status
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine ModelCalcChiFunction:")'
                    print '(2x,"Can not allocate variables GlobalParamVectorSet and LocalChi2ValueVector for thread = ", I5, ".")', myrank
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("alloc_status = ",I4)', alloc_status
                    print '(" ")'
                    call MPI_Abort(MPI_COMM_WORLD, ierr)
                    stop
                endif
            endif
            GlobalParamVectorSet(:, :) = ParameterVectorSet(:, :)
            LocalChi2ValueVector = 0.d0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< weak up slave threads and compute model function
            call SlaveWork(3)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine chi2 value and model function values for different parameter vectors
            call ModelCalcCore(NumberFreeParam, TotalNumberDataPoints, ModelFunctionFlag, ma, colx, NumFile, MaxL, MaxCol)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< sleep little slave threads
            call SlaveWork(4)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< get result of model function calculation
            chi2ValuesVector(:) = LocalChi2ValueVector(:)

            ! Debug:
            !    print*,'chi2ValuesVector = ', chi2ValuesVector
            !    call MPI_Abort(MPI_COMM_WORLD, ierr)
            !    stop
            return
        end subroutine ModelCalcChiFunction


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcCore
        !>
        !> calculates the chi2 values for a given set of parameter vectors
        !>
        !>
        !> input variables:     NumberFreeParam:        number of free parameters
        !>                      TotalNumDataPts:        total number of data points
        !>                      ModelFunctionFlag:      flag for indicating if model function values are stored or not
        !>                      ma:                     total number of parameters
        !>                      a:                      array containing the parameter set
        !>                      ia:                     flags for including/excluding parameter in the fit
        !>                      colx:                   number of columns in experimental x data
        !>                      NumFile:                number of experimental files
        !>                      MaxL:                   max number of lines of all experimental files
        !>                      MaxCol:                 max number of columns of all experimental files
        !>
        !> output variables:    chi2Value:              chi2 value
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 09.09.2014
        !>
        subroutine ModelCalcCore(NumberFreeParam, TotalNumDataPts, ModelFunctionFlag, ma, colx, NumFile, MaxL, MaxCol)

            implicit none
            integer :: m, l, j, n, i, NumberFile                                            !< loop variables
            integer :: k                                                                    !< counter variable
            integer :: ParamIndex                                                           !< working variable: index of parameter vector
            integer :: NumberFreeParam                                                      !< number of free parameters
            integer :: nelements                                                            !< working variable: size of an array to send
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: CurrentNumberWorkers                                                 !< the number of workers
            integer :: LastWorker                                                           !< number of last worker
            integer :: NumWorkers                                                           !< total number of workers
            integer :: TotalNumDataPts                                                      !< total number of data points
            integer :: dealloc_status, alloc_status                                         !< working variables for allocation/deallocation
            real*8 :: chi2Value                                                             !< working variable for chi2 value
            real*8 :: sig2i, ymod, val                                                      !< working variables
            real*8, dimension(NumberFreeParam) :: LocalParameterVector                      !< local copy of part of ParameterVectorSet
            real*8, dimension(NumberFreeParam) :: LocalParameterVector2                     !< further local copy of part of ParameterVectorSet
            real*8, dimension(TotalNumDataPts) :: ModelFunctionListLocal                    !< model function values
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function values are stored or
                                                                                            !< not
            logical :: NoCalFlag                                                            !< flag for not calculating model function for a specific
                                                                                            !< parameter vector
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit
            logical :: OutOfRangeFlag                                                       !< indicates, if a parameter is out of allowed range
            logical :: StoredBefore_Flag                                                    !< flag indicating if parameter vector is stored before or not
            logical :: AllocateFlag                                                         !< used to avoid multiple allocation of memory
            logical :: LocalModelFunctionFlag                                               !< in contrast to the GPU version we can store the model
                                                                                            !< function for all parameter vectors


            !< broadcast some variables between master and slave threads
            call MPI_Bcast(ModelFunctionFlag, 1, MPI_LOGICAL, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
            call MPI_Bcast(TotalNumParamVectors, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)


            !< (de)allocate memory for array containing all parameter vectors and chi2 vector
            !< the following lines are included to aviod reallocation for each call of this subroutine
            if (myrank /= 0) then

                ! Debug:
                ! print*,'-------------------myrank = ', myrank


                AllocateFlag = .false.
                if (.not. allocated(GlobalParamVectorSet)) then
                    AllocateFlag = .true.
                else
                    if (size(GlobalParamVectorSet) /= (TotalNumParamVectors * NumberFreeParameter)) then
                        AllocateFlag = .true.
                    endif
                endif

                ! Debug:
                ! print*,'-------------------myrank, AllocateFlag, TotalNumParamVectors = ', myrank ,AllocateFlag, TotalNumParamVectors


                if (AllocateFlag) then
                    if (allocated(GlobalParamVectorSet)) then
                        deallocate(GlobalParamVectorSet, stat = dealloc_status)
                        if (dealloc_status /= 0) then                                           !< is all ok?
                            write(logchannel,*)
                            write(logchannel,'("Error in subroutine ModelCalcChiFunction:")')
                            write(logchannel,'(2x,"Can not deallocate variable GlobalParamVectorSet for thread = ", I5, ".")') &
                                                                                                                                                    myrank
                            write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                            write(logchannel,*)
                            write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                            write(logchannel,'(" ")')
                            write(logchannel,'("Program aborted!")')
                            close(logchannel)

                            print '(" ")'
                            print '("Error in subroutine ModelCalcChiFunction:")'
                            print '(2x,"Can not deallocate variable GlobalParamVectorSet for thread = ", I5, ".")', myrank
                            print '(2x,"Please close all other programs and restart the program!")'
                            print '(" ")'
                            print '("dealloc_status = ",I4)',dealloc_status
                            print '(" ")'
                            call MPI_Abort(MPI_COMM_WORLD, ierr)
                            stop
                        endif
                    endif
                    allocate(GlobalParamVectorSet(TotalNumParamVectors, NumberFreeParameter), stat = alloc_status)
                    if (alloc_status /= 0) then                                                 !< is all ok?
                        write(logchannel,'(" ")')
                        write(logchannel,'("Error in subroutine ModelCalcChiFunction:")')
                        write(logchannel,'(2x,"Can not allocate variable GlobalParamVectorSet for thread = ", I5, ".")') myrank
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,'(" ")')
                        write(logchannel,'("alloc_status = ",I4)') alloc_status
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine ModelCalcChiFunction:")'
                        print '(2x,"Can not allocate variable GlobalParamVectorSet for thread = ", I5, ".")', myrank
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("alloc_status = ",I4)',alloc_status
                        print '(" ")'
                        call MPI_Abort(MPI_COMM_WORLD, ierr)
                        stop
                    endif
                endif
                GlobalParamVectorSet = 0.d0
            endif


            !< broadcast some variables between master and slave threads
            nelements = size(GlobalParamVectorSet)
            call MPI_Bcast(GlobalParamVectorSet, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
            call MPI_BARRIER(MPI_COMM_WORLD, ierr)
!            nelements = size(paramset)
!            call MPI_Bcast(paramset, nelements, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
!            call MPI_BARRIER(MPI_COMM_WORLD, ierr)

            ! Debug:
            !    print*,'myrank, ModelFunctionFlag = ', myrank, ModelFunctionFlag
            !    print*,'myrank, colx = ', myrank, colx
            !    print*,'myrank, NumFile = ', myrank, NumFile
            !    print*,'myrank, MaxL = ', myrank, MaxL
            !    print*,'myrank, MaxCol = ', myrank, MaxCol
            !    print*,'myrank, TotalNumParamVectors = ', myrank, TotalNumParamVectors
            !    print*,'myrank, GlobalParamVectorSet(myrank + 1, :) = ', myrank, GlobalParamVectorSet(myrank + 1, :)
            !    print*,'>>>myrank, paramset(1, :) = ', myrank, paramset(1, :)


            !< reset output variables
            if (myrank == 0) then
                LocalChi2ValueVector = 0.d0
            endif
            ModelFunction = 0.d0


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            Do l = 1, TotalNumParamVectors, TotalNumberOfThreads                                 !< loop over all parameter vectors

                ! Debug:
                ! print*,'l + myrank, TotalNumParamVectors, TotalNumberOfThreads = ', l + myrank, TotalNumParamVectors, TotalNumberOfThreads


                !< determine max. number of workers
                LastWorker = min(TotalNumParamVectors, (l + TotalNumberOfThreads - 1))
                NumWorkers = LastWorker - l + 1
                CurrentNumberWorkers = NumWorkers

                ! Debug:
                ! print*,'myrank, CurrentNumberWorkers = ', myrank, CurrentNumberWorkers


                !< do nothing if no more parameter vector is left
                if (l + myrank > TotalNumParamVectors) then
                    exit
                endif


                !< select a parameter vector for each thread (master and slave)
                LocalParameterVector(:) = GlobalParamVectorSet(l + myrank, :)


                !< initialize result variables
                chi2Value = 1.d99
                ModelFunctionListLocal = 0.d0


                !< is calculation reduction wanted (to be done soon ........... !)
                NoCalFlag = .false.


                !< is a calculation still neccessary
                if (.not. NoCalFlag) then


                    !< make copy of current parameter vector
                    OutOfRangeFlag = .false.
                    k = 0
                    ia = .false.
                    Do i = 1, ma
                        if (paramset(2, i) == 1.d0) then
                            ia(i) = .true.
                            k = k + 1
                            if (LocalParameterVector(k) < paramset(3, i) .or. paramset(4, i) < LocalParameterVector(k)) then
                                OutOfRangeFlag = .true.
                                exit
                            endif
                        endif
                    end Do

                    ! Debug:
                    ! print*,'myrank, LocalParameterVector = ', myrank, LocalParameterVector
                    ! print*,'myrank, paramset(1, :) = ', myrank, paramset(1, :)
                    ! print*,'myrank, NumberFreeParam = ', myrank, NumberFreeParam
                    ! print*,'myrank, OutOfRangeFlag = ', myrank, OutOfRangeFlag



                    !-------------------------------------------------------------------------------------------------------------------------------------
                    !< calculate model function for current parameter vector
                    if (.not. OutOfRangeFlag) then


                        !< call subroutine to calculate model function
                        LocalModelFunctionFlag = .true.
                        call ModelCalcSpectrum(NumberFreeParam, LocalParameterVector, LocalModelFunctionFlag, TotalNumberComponents, chi2Value, &
                                               CompMoleculeIndex, myXCLASSParameter, TotalNumberOfMolecules, IsoRatioConversionTable, &
                                               TotalNumberDataPoints, ModelFunctionListLocal)
                        ! Debug:
                        ! print*,'myrank, ModelFunctionListLocal(100) = ', myrank, ModelFunctionListLocal(100)
                    endif
                endif


                !< send results back to master thread
                if (myrank /= 0) then

                    ! Debug:
                    ! print*,'SLAVE: myrank, chi2Value = ', myrank, chi2Value


                    !< send some other parameters back to master thread
                    call MPI_Send(chi2Value, 1, MPI_DOUBLE_PRECISION, 0, 0, MPI_COMM_WORLD, ierr)
                    nelements = size(ModelFunctionListLocal)
                    call MPI_Send(ModelFunctionListLocal, nelements, MPI_DOUBLE_PRECISION, 0, 0, MPI_COMM_WORLD, ierr)


                !< MASTER: recieve results from slave threads
                else
                    Do j = 0, (CurrentNumberWorkers - 1)
                        ParamIndex = l + j


                        !< get chi2 value
                        if (j > 0) then
                            call MPI_Recv(chi2Value, 1, MPI_DOUBLE_PRECISION, j, 0, MPI_COMM_WORLD, istatus0, ierr)

                            ! Debug:
                            ! print*,'MPI_Recv(chi2Value): myrank, ierr = ', myrank, ierr
                        endif
                        LocalChi2ValueVector(ParamIndex) = chi2Value

                        ! Debug:
                        ! print*,'myrank, j, chi2Value = ', myrank, j, chi2Value


                        !< get model function values
                        if (j > 0) then
                            nelements = size(ModelFunctionListLocal)
                            call MPI_Recv(ModelFunctionListLocal, nelements, MPI_DOUBLE_PRECISION, j, 0, MPI_COMM_WORLD, istatus0, ierr)

                            ! Debug:
                            ! print*,'MPI_Recv(ModelFunctionListLocal): myrank, ierr = ', myrank, ierr
                        endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< do we need to determine the best fit parameter vector
                        if (ModelFunctionFlag) then
                            k = 0
                            Do NumberFile = 1, NumberExpFiles                               !< loop over exp. data files
                                Do n = 1, NumberYColumns(NumberFile)                        !< loop over y-columns
                                    Do i = 1, lengthexpdata(NumberFile)                     !< loop over all line of current exp. data file
                                        k = k + 1                                           !< increase data point counter
                                        ModelFunction(ParamIndex, NumberFile, n, i) = ModelFunctionListLocal(k)
                                    end Do
                                end Do
                            end Do
                        else
                            LocalParameterVector(:) = GlobalParamVectorSet(ParamIndex, :)
                            Do k = 1, QualityLimit
                                if (chi2Value < BestSitesParamSet(k, 1)) then
                                    StoredBefore_Flag = .false.
                                    if (QualityLimit > 1) then
                                        Do m = 1, QualityLimit
                                            if (chi2Value == BestSitesParamSet(m, 1)) then
                                                LocalParameterVector2(:) = BestSitesParamSet(m, 2:)
                                                call CompareTwoParameterSets(StoredBefore_Flag, ma, NumberFreeParameter, ia, LocalParameterVector2(:), &
                                                                             LocalParameterVector(:))
                                                if (StoredBefore_Flag) then
                                                    exit
                                                endif
                                            endif
                                        end Do
                                    endif
                                    if (.not. StoredBefore_Flag) then
                                        if (k < QualityLimit) then
                                            Do m = QualityLimit, (k + 1), (-1)
                                                BestSitesParamSet(m, :) = BestSitesParamSet((m - 1), :)
                                                BestSitesModelValues(m, :, :, :) = BestSitesModelValues((m - 1), :, :, :)
                                                BestSitesChi2Values(m, :, :, :) = BestSitesChi2Values((m - 1), :, :, :)
                                            end Do
                                        endif
                                        BestSitesParamSet(k, 1) = chi2Value
                                        BestSitesParamSet(k, 2:) = LocalParameterVector(:)
                                        m = 0
                                        Do NumberFile = 1, NumberExpFiles                   !< loop over exp. data files
                                            Do n = 1, NumberYColumns(NumberFile)            !< loop over y-columns
                                                Do i = 1, lengthexpdata(NumberFile)         !< loop over all line of current exp. data file
                                                    m = m + 1                               !< increase data point counter
                                                    ymod = ModelFunctionListLocal(m)
                                                    sig2i = 1.d0
                                                    if (expdatae(NumberFile, i, n) /= 0.d0) then    !< do the experimental datas include errors
                                                        sig2i = 1.d0/(expdatae(NumberFile, i, n) * expdatae(NumberFile, i, n))
                                                    endif


                                                    !< determine chi**2 by calculating the difference (y_i^{obs) - y_i(fit))**2
                                                    val = 0.d0
                                                    if (abs(DetChi2) == 1) then
                                                        val = (expdatay(NumberFile, i, n) - ymod)**2


                                                    !< determine chi**2 by calculating the difference (y_i^{obs)**2 - y_i(fit)**2)
                                                    elseif (abs(DetChi2) == 2) then
                                                        val = (expdatay(NumberFile, i, n)**2 - ymod**2)
                                                    endif
                                                    BestSitesChi2Values(k, NumberFile, i, n) = val * sig2i
                                                    BestSitesModelValues(k, NumberFile, i, n) = ymod
                                                    ModelFunction(1, NumberFile, n, i) = ymod
                                                end Do
                                            end Do
                                        end Do
                                    endif
                                endif
                            end Do
                        endif
                    end Do
                endif

                ! Debug:
                ! print*,'myrank = ', myrank
                ! if (myrank == 0) print*,'#######################################################################################'
                ! print '(".", $)'
                ! call MPI_BARRIER(MPI_COMM_WORLD, ierr)
                ! call MPI_Abort(MPI_COMM_WORLD, ierr)
                ! stop 'lllklklkklkl'

            end Do
            return
        end subroutine ModelCalcCore


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcSpectrum
        !>
        !> calculates the myXCLASS spectrum for a given parameter vector
        !>
        !>
        !> input variables:     NumberFreeParameterCopy:    number of free parameters
        !>                      ParameterVector:        parameter vector
        !>                      ModelFunctionFlag:      flag for indicating if model function values are stored or not
        !>                      NumComp:                total number of components
        !>                      CompMoleculeIndex:      molecule index for each component
        !>                      myXCLASSParameterOrig:  array containing all molfit parameters for each component
        !>                      TotalNumMol:            total number of molecules
        !>                      IsoRatioConversionTable: table with iso ratios between iso master and molecule
        !>                      NumDataPoints:          total number of data points
        !>
        !> output variables:    chi2Value:              chi2 value for parameter vector
        !>                      ModelFuncList:          model function values
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 31.07.2014
        !>
        subroutine ModelCalcSpectrum(NumberFreeParameterCopy, ParameterVector, ModelFunctionFlag, NumComp, chi2Value, CompMoleculeIndexOrig, &
                                     myXCLASSParameterOrig, TotalNumMol, IsoRatioConversionTableOrig, NumDataPoints, ModelFuncList)
            use omp_lib

            implicit none
            integer :: NumberFreeParameterCopy                                              !< number of free parameters
            integer :: NumComp                                                              !< total number of components
            integer :: TotalNumMol                                                          !< total number of molecules
            real*8 :: chi2Value                                                             !< chi2 value for parameter vector
            real*8, dimension(NumberFreeParameterCopy) :: ParameterVector                   !< parameter vector
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function values are stored or
                                                                                            !< not
            !< myXCLASS variables
            integer :: fitnum, i, j, k, l, ae, c, t                                         !< loop variables
            integer :: i1, i2, ff1, ff2                                                     !< working variables for copying
            integer :: IndexComp                                                            !< overall component index
            integer :: varIndex                                                             !< myXCLASS variable index
            integer :: ErrChannelIndexLocal                                                 !< working varibale for error handling
            integer :: ErrChannelLocal                                                      !< working varibale for error handling
            integer :: MoleculeIndex                                                        !< working variable: index for molecule
            integer :: FirstMolecularDataIndex                                              !< working variable: first index for molecular data table
            integer :: LastMolecularDataIndex                                               !< working variable: last index for molecular data table
            integer :: FirstIndex                                                           !< working variable: index for first freq. data point
            integer :: LastIndex                                                            !< working variable: index for last freq. data point
            integer :: FreqIndex                                                            !< working variable: index for frequency point
            integer :: ThreadNumber                                                         !< thread number
            integer :: NumDataPoints                                                        !< total number of data points
            integer, dimension(NumComp) :: CompMoleculeIndexOrig                            !< molecule index for each component
            integer, dimension(NumComp) :: CopyCompMoleculeIndex                            !< working variable: local copy of CompMoleculeIndexOrig
            real*8 :: eta_source                                                            !< working variable: beam filling factor
            real*8 :: logt, x0, x1, f0, f1                                                  !< working variables for interpolation
            real*8 :: freq                                                                  !< working variable: frequency for index 'FreqIndex'
            real*8 :: freq_t                                                                !< working variable: frequency as temperature
            real*8 :: beam_size                                                             !< working variable: beam size (eqn. 2 - 3)
            real*8 :: j_tk, j_cb, j_back                                                    !< working variables: Rayleigh temperatures
            real*8 :: tau_d                                                                 !< working variable: tau dust
            real*8 :: tau_t                                                                 !< working variable: tau for each transition
            real*8 :: tau_l                                                                 !< working variable: sum over all taus
            real*8 :: vLSR                                                                  !< working variable: v_LSR
            real*8 :: sigma                                                                 !< working variable: sigma
            real*8 :: Temperature                                                           !< working variable: temperature
            real*8 :: LocalIntensity                                                        !< working variable: calc. intensity at current data point
            real*8, dimension(NumComp) :: Q                                                 !< partition function at the rotation temperatures
            real*8, dimension(NumDataPoints) :: ModelFuncList                               !< output variable: model function values
            real*8, dimension(11, NumComp) :: myXCLASSParameterOrig                         !< array containing all molfit parameters for each component
            real*8, dimension(11, NumComp) :: CopymyXCLASSParameter                         !< working variable: local copy of myXCLASSParameterOrig
            real*8, dimension(TotalNumMol, TotalNumMol) :: IsoRatioConversionTableOrig      !< table with iso ratios between iso master and molecule
            real*8, dimension(TotalNumMol, TotalNumMol) :: CopyIsoRatioConversionTable      !< working variable: local copy of IsoRatioConversionTableOrig


            !< PRIVATE variables for this subroutine (in addition to the local defined variables):
            !< CompMoleculeIndex:
            !< myXCLASSParameter:
            !< chi2Value:
            !< ParameterVector:
            !< ModelFunctionFlag???


            !< get threadnumber
            ThreadNumber = myrank
            ModelFuncList = 0.d0


            !<============================================================================================================================================
            !< variables from the global module definition
            !<      Firsti                                  shared      integer             working variable for interpolation
            !<      Lasti                                   shared      integer             working variable for interpolation
            !<      stepi                                   shared      integer             working variable for interpolation
            !<      TotalNumberComponents                   shared      integer             counter for total number of components of all molecules
            !<      TotalNumberOfFrequencyRanges            shared      integer             total number of frequency ranges
            !<      TotalNumberOfMolecules                  shared      integer             total number of molecules including isotopologues
            !<      NumberOfTemperatures                    shared      integer             number of temperatures
            !<      NumberComponentsPerMolecule             shared      integer             number of components per molecule
            !<                                                          dimension: (0:TotalNumberOfMolecules)
            !<      IsoNfitConversionTable                  shared      integer             iso ratios conversion table for free paramter index
            !<                                                          dimension: (TotalNumberOfMolecules, TotalNumberOfMolecules)
            !<      SpectrumIndexForFreqRange               shared      integer             store spectrum index for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      DataPointIndexFreqRange                 shared      integer                 index of data point of first and last freq. in freq. range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges, 2)
            !<      MolecularDataIndices                    shared      integer             start and end index for each molecule and frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2)
            !<      ConversionTableMAGIXmyXCLASSParam       shared      integer             conversion table between myXCLASS and MAGIX parameter
            !<                                                          dimension: (NumberFreeParameterCopy, 2)
            !<      CompMoleculeIndex                       private     integer             molecule index for each component
            !<                                                          dimension: (TotalNumberComponents)
            !<      debug_freq                              shared      real*8              frequency for debugging
            !<      ckms                                    shared      real*8              speed of light in km/s
            !<      pi                                      shared      real*8              pi
            !<      TempLow                                 shared      real*8              working variable for extrapolation of part. func.
            !<      TempHigh                                shared      real*8              working variable for extrapolation of part. func.
            !<      TempPartFunc                            shared      real*8              temperatures for partition function
            !<                                                          dimension: (NumberOfTemperatures)
            !<      BackgroundTemperatureRange              shared      real*8              T_Back for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      TemperatureSlopeRange                   shared      real*8              T_Slope for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      HydrogenColumnDensityRange              shared      real*8              nH for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      DustBetaRange                           shared      real*8              beta for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      KappaRange                              shared      real*8              kappa for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      TelescopeSize                           shared      real*8              size of telescope for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      StartFrequency                          shared      real*8              first frequency for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      ObservationalDataList                   shared      real*8              list containing all observational data
            !<                                                          dimension: (TotalNumberDataPoints, 3)
            !<      MolecularData                           shared      real*8              array containing the molecular data for all molecules and
            !<                                                          dimension: (TotalNumberOfTransitions, 4)
            !<      MoleculeNames                           shared      character(len=40)   names of all molecules (including isotopologues)
            !<                                                          dimension: (MaxNumberOfMolecules)
            !<      IsoRatioConversionTable                 private     real*8              table with iso ratios between iso master and molecule
            !<                                                          dimension: (TotalNumberOfMolecules, TotalNumberOfMolecules)
            !<      lgQ                                     shared      real*8              lgQ entries of table PartitionFunction
            !<                                                          dimension: (NumberOfTemperatures, NumberMoleculePartFunc)
            !<      myXCLASSParameter                       private     real*8              array containing all molfit parameters for each component
            !<                                                          dimension: (11, TotalNumberComponents)
            !<      nHFlag                                  shared      logical             flag for global setting of nH, kappa and beta
            !<      tbFlag                                  shared      logical             flag for global setting T_Back and T_Slope
            !<      IsoFlag                                 shared      logical             flag indicating use of isotopologues
            !<============================================================================================================================================


            !< make private copy of both variables
            CopyCompMoleculeIndex = CompMoleculeIndexOrig
            CopymyXCLASSParameter = myXCLASSParameterOrig
            if (IsoFlag) then
                CopyIsoRatioConversionTable = IsoRatioConversionTableOrig
            endif


            !<============================================================================================================================================
            !< update myXCLASS parameter vector with new parameter values
            !< CopymyXCLASSParameter(1, :)      source size
            !< CopymyXCLASSParameter(2, :)      T_rot
            !< CopymyXCLASSParameter(3, :)      N_tot
            !< CopymyXCLASSParameter(4, :)      V_width
            !< CopymyXCLASSParameter(5, :)      V_off
            !< CopymyXCLASSParameter(6, :)      T_Back
            !< CopymyXCLASSParameter(7, :)      T_Slope
            !< CopymyXCLASSParameter(8, :)      nH_column
            !< CopymyXCLASSParameter(9, :)      beta
            !< CopymyXCLASSParameter(10, :)     kappa
            !< CopymyXCLASSParameter(11, :)     Emission/Absorption flag
            Do fitnum = 1, NumberFreeParameterCopy                                          !< loop over all free parameters


                !< update molfit file parameters
                IndexComp = ConversionTableMAGIXmyXCLASSParam(fitnum, 1)                    !< get current overall component index
                varIndex = ConversionTableMAGIXmyXCLASSParam(fitnum, 2)                     !< get current myXCLASS variable index
                if (IndexComp > 0 .and. varIndex > 0 .and. varIndex < 11) then              !< check, if current free param. corresponds to molfit param.

                ! Debug:
                    ! print*,'varIndex, IndexComp = ', varIndex, IndexComp
                    ! print*,'CopymyXCLASSParameter(:, IndexComp) = ', CopymyXCLASSParameter(varIndex, IndexComp)
                    ! print*,'ParameterVector(fitnum) = ', ParameterVector(fitnum)


                    CopymyXCLASSParameter(varIndex, IndexComp) = ParameterVector(fitnum)


                    !< check for overflow
                    if (varIndex == 3) then
                        if (CopymyXCLASSParameter(3, IndexComp) > 300.d0) then
                            j = CopyCompMoleculeIndex(IndexComp)
                            k = IndexComp - sum(NumberComponentsPerMolecule(:(j - 1)))
                            Do ErrChannelIndexLocal = 1, 1
                                ErrChannelLocal = AllErrChannels(ErrChannelIndexLocal)
                                write(ErrChannelLocal, '(" ")')
                                write(ErrChannelLocal, '(" ")')
                                write(ErrChannelLocal, '(2x, "Error in subroutine ModelCalcSpectrum:")')
                                if (j /= 0) then
                                    write(ErrChannelLocal, '(4x, "The log10 value of column density for component ", I4, " of molecule ", A, " ")') k, &
                                                                                                                    trim(adjustl(MoleculeNames(j)))
                                else
                                    write(ErrChannelLocal, '(4x, "The log10 value of column density for component ", I4, " ")') k
                                endif
                                write(ErrChannelLocal, '(4x, "causes an overflow (to large number)!")')
                                write(ErrChannelLocal, '(4x, "Set log10 value for this column density to 300!")')
                                write(ErrChannelLocal, '(" ")')
                                write(ErrChannelLocal, '(" ")')
                            end Do
                            CopymyXCLASSParameter(3, IndexComp) = 300.d0
                        endif


                        !< convert log10 value back to linear value
                        CopymyXCLASSParameter(3, IndexComp) = 10.d0**CopymyXCLASSParameter(3, IndexComp)
                    endif


                !< update iso ratio table
                elseif (IsoFlag) then
                    Do j = 1, TotalNumberOfMolecules
                        Do k = 1, TotalNumberOfMolecules
                            if (int(IsoNfitConversionTable(j, k)) == fitnum) then
                                CopyIsoRatioConversionTable(j, k) = ParameterVector(fitnum)
                            elseif (int(IsoNfitConversionTable(j, k)) == -fitnum) then
                                CopyIsoRatioConversionTable(j, k) = 1.d0 / ParameterVector(fitnum)
                            endif
                        end Do
                    end Do
                endif
            end Do


            !<============================================================================================================================================
            !< update parameters for isotopologues
            if (IsoFlag) then
                Do j = 1, TotalNumberOfMolecules
                    i1 = sum(NumberComponentsPerMolecule(:(j - 1))) + 1
                    i2 = i1 + NumberComponentsPerMolecule(j) - 1

                    ! Debug:
                    !    print*,"#####################################################################"
                    !    print*,"j, MoleculeNames(j) = ", j, MoleculeNames(j)
                    !    print*,"NumberComponentsPerMolecule(j) = ", j, NumberComponentsPerMolecule(j)
                    !    print*,"CopyIsoRatioConversionTable(j, :) = ", CopyIsoRatioConversionTable(j, :)
                    !    Do k = i1, i2
                    !        print*,"CopymyXCLASSParameter(:, k) = ", k, CopymyXCLASSParameter(:, k)
                    !    end Do
                    !    print*," "
                    !    print*," "


                    Do k = 1, TotalNumberOfMolecules

                        ! Debug:
                        !    print*,"k, MoleculeNames(j) = ", k, MoleculeNames(k)
                        !    print*,"CopyIsoRatioConversionTable(j, k) = ", CopyIsoRatioConversionTable(j, k)
                        !    print*,"NumberComponentsPerMolecule(k) = ", NumberComponentsPerMolecule(k)
                        !    print*," "


                        if (CopyIsoRatioConversionTable(j, k) /= 0.d0 .and. NumberComponentsPerMolecule(k) /= 0) then

                            ff1 = sum(NumberComponentsPerMolecule(:(k - 1))) + 1
                            ff2 = ff1 + NumberComponentsPerMolecule(k) - 1

                            ! Debug:
                            ! print*,"i1, i2 = ",i1, i2
                            ! print*,"ff1, ff2 = ",ff1, ff2


                            !< copy parameters for current isotopologues from isomaster and scale column densities with iso ratio factor
                            CopymyXCLASSParameter(:, ff1:ff2) = CopymyXCLASSParameter(:, i1:i2)
                            CopyCompMoleculeIndex(ff1:ff2) = k
                            Do i = ff1, ff2
                                CopymyXCLASSParameter(3, i) = CopyIsoRatioConversionTable(j, k) * CopymyXCLASSParameter(3, i)
                            end Do
                        endif
                    end Do
                end Do
            endif

            ! Debug:
            !    Do i = 1, TotalNumberComponents
            !        print*,"i = ", i
            !        j = CopyCompMoleculeIndex(i)
            !        print*,"MoleculeNames = ", trim(adjustl(MoleculeNames(j)))
            !        print*,"CopymyXCLASSParameter(:, i) = ", CopymyXCLASSParameter(:, i)
            !    end Do


            !<============================================================================================================================================
            !< determine partiton functions for all rotation temperatures
            Q = 0.d0
            Do l = 1, TotalNumberComponents                                                 !< loop over all components
                Temperature = CopymyXCLASSParameter(2, l)                                   !< get excitation temperatures
                if (Temperature <= 0.d0) then
                    Temperature = 10.d0**TempLow
                    j = CopyCompMoleculeIndex(l)
                    k = l - sum(NumberComponentsPerMolecule(:(j - 1)))
                    Do ErrChannelIndexLocal = 1, 1
                        ErrChannelLocal = AllErrChannels(ErrChannelIndexLocal)
                        write(ErrChannelLocal, '(" ")')
                        write(ErrChannelLocal, '(" ")')
                        write(ErrChannelLocal, '(2x, "Error in subroutine ModelCalcSpectrum!")')
                        if (j /= 0) then
                            write(ErrChannelLocal, '(4x, "The excitation temperature for component ", I4, " of molecule ", A, " ")') k, &
                                                                                                     trim(adjustl(MoleculeNames(j)))
                        else
                            write(ErrChannelLocal, '(4x, "The excitation temperature for component ", I4, " ")') k
                        endif
                        write(ErrChannelLocal, '(4x, "is less than or equal to zero!")')
                        write(ErrChannelLocal, '(" ")')
                        write(ErrChannelLocal, '(" ")')
                        write(ErrChannelLocal, '(4x, "Set temperature to ", F25.15)') Temperature
                        write(ErrChannelLocal, '(" ")')
                    end Do
                endif

                logt = dlog10(Temperature)                                              !< calculate the logarithm of the excitation temperature tk
                MoleculeIndex = CopyCompMoleculeIndex(l)                                !< get molecule index


                !< is logt out of range?
                if (logt < TempLow .or. logt > TempHigh) then


                    !< extrapolate if neccessary ---------------------------------------- is not valid for all cases !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                    x0 = 0.d0
                    x1 = 0.d0
                    f0 = 0.d0
                    f1 = 0.d0
                    if (logt < TempPartFunc(NumberOfTemperatures)) then
                        x0 = TempPartFunc((NumberOfTemperatures - 1))
                        x1 = TempPartFunc(NumberOfTemperatures)
                        f0 = lgQ((NumberOfTemperatures - 1), MoleculeIndex)
                        f1 = lgQ(NumberOfTemperatures, MoleculeIndex)

                    elseif (logt > TempPartFunc(1)) then
                        x0 = TempPartFunc(2)
                        x1 = TempPartFunc(1)
                        f0 = lgQ(2, MoleculeIndex)
                        f1 = lgQ(1, MoleculeIndex)

                    endif
                    Q(l) = f0 + (f1 - f0)/(x1 - x0) * (logt - x0)
                    Q(l) = 10.d0**Q(l)

                    ! Debug:
                    ! print*,'logt > TempPartFunc(1) .or. logt < TempPartFunc(NumberOfTemperatures):'
                    ! print*,'x0, f0 = ', x0, f0
                    ! print*,'x1, f1 = ', x1, f1
                    ! print*,'logt = ',logt
                    ! print*,'Q(l) = ', Q(l)
                    ! print*,'log10(Q(l)) = ', dlog10(Q(l))
                else


                    !< start linear interpolation
                    Do i = Firsti, Lasti, stepi
                        if (TempPartFunc(i) == logt) then
                            Q(l) = lgQ(i, MoleculeIndex)
                            exit


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< linear interpolation: f(x) = f_0 + \frac{f_1 - f_0}{x_1 - x_0} (x - x_0)
                        elseif (TempPartFunc(i) < logt) then
                            if (i == NumberOfTemperatures) then
                                x0 = TempPartFunc(i - 1)
                                x1 = TempPartFunc(i)
                                f0 = lgQ(i - 1, MoleculeIndex)
                                f1 = lgQ(i, MoleculeIndex)
                            else
                                x0 = TempPartFunc(i)
                                x1 = TempPartFunc(i + 1)
                                f0 = lgQ(i, MoleculeIndex)
                                f1 = lgQ(i + 1, MoleculeIndex)
                            endif
                            Q(l) = f0 + (f1 - f0)/(x1 - x0) * (logt - x0)

                            ! Debug:
                            ! print*,'x0, f0 = ', x0, f0
                            ! print*,'x1, f1 = ', x1, f1
                            ! print*,'logt = ',logt
                            ! print*,'Q(l) = ', Q(l)
                            ! print*,'10.d0**Q(l) = ', 10.d0**Q(l)
                            exit
                        endif
                    end Do
                    Q(l) = 10.d0**Q(l)

                    ! Debug:
                    !    print*,' '
                    !    print*,' '
                    !    print*,'l = ', l
                    !    print*,"MoleculeNames = ", trim(adjustl(MoleculeNames(MoleculeIndex)))
                    !    print*,'Temperature = ', Temperature
                    !    print*,'Q(l) = ', Q(l)
                    !    print*,'logt = ', logt
                    !    Do i = 1, NumberOfTemperatures
                    !        print*,'i = ', i
                    !        print*,'10.d0**lgQ(i, MoleculeIndex) = ', 10.d0**lgQ(i, MoleculeIndex), "           ", Q(l)
                    !        print*,'10.d0**TempPartFunc(i) = ', 10.d0**TempPartFunc(i), "           ",Temperature
                    !    end Do
                    !    print*,'log10(Q(l)) = ', dlog10(Q(l))
                endif
            end Do


            !<============================================================================================================================================
            !< start loop over molecules, frequency ranges, components, transitions
            chi2Value = 0.d0
            Do l = 1, TotalNumberOfFrequencyRanges                                          !< loop over all frequency ranges
                i = SpectrumIndexForFreqRange(l)                                            !< get spectrum index
                FirstIndex = DataPointIndexFreqRange(l, 1)                                  !< get index of first freq. point in 'ObservationalDataList'
                LastIndex = DataPointIndexFreqRange(l, 2)                                   !< get index of last freq. point in 'ObservationalDataList'

                ! Debug:
                ! print*,"l = ", l
                ! print*,"StartFrequency(l) = ", StartFrequency(l)
                ! print*,"EndFrequency(l)   = ", EndFrequency(l)
                ! print*,"StepFrequency(l)  = ", StepFrequency(l)
                ! print*,"FirstIndex = ", FirstIndex
                ! print*,"LastIndex  = ", LastIndex


                !< update 
                if (tbFlag) then
                    CopymyXCLASSParameter(6, :) = BackgroundTemperatureRange(l)
                    CopymyXCLASSParameter(7, :) = TemperatureSlopeRange(l)
                endif
                if (nHFlag) then
                    CopymyXCLASSParameter(8, :) = HydrogenColumnDensityRange(l)
                    CopymyXCLASSParameter(9, :) = DustBetaRange(l)
                    CopymyXCLASSParameter(10, :) = KappaRange(l) * 2.d0 * 1.66d-24 / 100.d0 !< kappa_1300 * M(H2)/M_d_M_g
                endif
                Do FreqIndex = FirstIndex, LastIndex                                        !< loop over all frequency points in the current freq. range
                    freq = ObservationalDataList(FreqIndex, 1)                              !< get frequency

                    ! Debug:
                    ! print*,"myrank, l, freq = ", myrank, l, freq


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< determine beam size (eq. 3):
                    !< telescope beam FWHM size is related to the diameter of the telescope by the diffraction limit:
                    !<
                    !<     \theta_t = 1.22 \frac{c}{\lambda D}
                    !<
                    !< where D describes the diameter of the telescope and c the speed of light.
                    !<
                    beam_size = 1.22d-3 * ckms * 180.d0 * 3600.d0 / pi / freq / TelescopeSize(l)


                    LocalIntensity = 0.d0                                                   !< reset local intensity
                    Do ae = 1, 2                                                            !< loop over emission (ae=1) and absorption (ae=2) components
                        Do c = 1, TotalNumberComponents                                     !< loop over all components

                            !< CopymyXCLASSParameter(1, c)      source size
                            !< CopymyXCLASSParameter(2, c)      T_rot
                            !< CopymyXCLASSParameter(3, c)      N_tot
                            !< CopymyXCLASSParameter(4, c)      V_width
                            !< CopymyXCLASSParameter(5, c)      V_off
                            !< CopymyXCLASSParameter(6, c)      T_Back
                            !< CopymyXCLASSParameter(7, c)      T_Slope
                            !< CopymyXCLASSParameter(8, c)      nH_column
                            !< CopymyXCLASSParameter(9, c)      beta
                            !< CopymyXCLASSParameter(10, c)     kappa
                            !< CopymyXCLASSParameter(11, c)     Emission/Absorption flag

                            ! Debug:
                            if (dabs(freq - debug_freq) < 1.d-1 .and. ThreadNumber == (-1)) then
                                print*,"####################################################################################"
                                print*,"ae = ", ae
                                print*,"c, TotalNumberComponents = ", c, TotalNumberComponents
                                print*,"CopymyXCLASSParameter(:, c) = ", CopymyXCLASSParameter(:, c)
                            endif


                            if (int(CopymyXCLASSParameter(11, c)) == ae) then               !< check, if current component correspond to ae index
                                MoleculeIndex = CopyCompMoleculeIndex(c)                    !< get molecule index


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< determine beam filling (dilution) factor (eq. 2):
                                !< \eta(\theta_{m,c}) of molecule m and component c for a source with a Gaussian brightness profile, and a Gaussian beam
                                !< is given by
                                !< 
                                !<     \eta(\theta_{m,c}) = \frac{\theta_{m,c}^2}{\theta_{m,c}^2 + \theta_t^2}
                                !< 
                                !< where \theta_{m,c} and \theta_t represents the source and telescope beam FWHM sizes, respectively.
                                !< 
                                eta_source = CopymyXCLASSParameter(1, c)**2 / (beam_size**2 + CopymyXCLASSParameter(1, c)**2)

                                ! Debug:
                                if (dabs(freq - debug_freq) < 1.d-1 .and. ThreadNumber == (-1)) then
                                    print*,"MoleculeIndex = ", MoleculeIndex
                                    print*,"eta_source = ", eta_source
                                endif


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< determine Rayleight-Jeans temperatures
                                freq_t = freq * 4.8d-5                                      !< convert frequency in temperature (Kelvin)
                                j_tk = freq_t / (dexp(freq_t / CopymyXCLASSParameter(2, c)) - 1.d0)
                                j_cb = freq_t / (dexp(freq_t / 2.7d0) - 1.d0)

                                ! Debug:
                                if (dabs(freq - debug_freq) < 1.d-1 .and. ThreadNumber == (-1)) then
                                    print*,"freq_t = ", freq_t
                                    print*,"j_tk = ", j_tk
                                    print*,"j_cb = ", j_cb
                                endif


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< background temperature and slope are given for each component and molecule
                                !< the background temperature J(T_bg; \nu) is described by
                                !< 
                                !<     J (T_{\rm bg}, \nu ) = | T_{\rm bg} | * (\frac{\nu}{\nu_{\rm min}} )^{T_{\rm slope}},
                                !< 
                                !< and defines the continuum contribution for each frequency range, individually. Here, \nu_{\rm min} indicates the
                                !< lowest frequency of a given frequency range. T_{\rm bg} and T_{\rm slope}, describe the background continuum
                                !< temperature and the temperature slope, respectively.
                                !< 
                                !< CopymyXCLASSParameter(6, c)      T_Back
                                !< CopymyXCLASSParameter(7, c)      T_Slope
                                !< 
                                j_back = dabs(CopymyXCLASSParameter(6, c)) * (freq / StartFrequency(l))**CopymyXCLASSParameter(7, c)

                                ! Debug:
                                if (dabs(freq - debug_freq) < 1.d-1 .and. ThreadNumber == (-1)) print*,"j_back = ", j_back


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< calculate the dust contribution (eq. 4):
                                !< 
                                !<     \tau_d = N_H * \kappa_{\rm 1.3 \, mm} * ( \frac{\nu}{\nu_{\rm 1.3 \, mm}} )^\beta * m_{H_2}
                                !<                  * \frac{1}{\zeta_{\rm gas-dust}},
                                !< 
                                !< where the hydrogen column density $N_H$, the dust mass opacity $\kappa_{\rm 1.3 \, mm}$, and the exponent $\beta$.
                                !< In addition, $\nu_{\rm 1.3 \, mm}$ = 230~GHz indicates the frequency for a wavelength of 1.3~mm, $m_{H_2}$ describes
                                !< the mass of a hydrogen molecule, and $\zeta_{\rm gas-dust}$ represents the gas to dust mass ratio,
                                !< i.e.\ $1 / \zeta_{\rm gas-dust}$ describes the dust to gas ratio and is set to (1/100).
                                !< 
                                !< CopymyXCLASSParameter(8, c)      nH_column
                                !< CopymyXCLASSParameter(9, c)      beta
                                !< CopymyXCLASSParameter(10, c)     kappa   (contains factor (m_{H_2} * \frac{1}{\zeta_{\rm gas-dust}}) as well)
                                !< 
                                tau_d = CopymyXCLASSParameter(10, c) * (freq / 2.3d5)**CopymyXCLASSParameter(9, c) * CopymyXCLASSParameter(8, c)

                                ! Debug:
                                if (dabs(freq - debug_freq) < 1.d-1 .and. ThreadNumber == (-1)) then
                                    print*,"tau_d = ", tau_d
                                    print*,"l = ", l
                                    print*,"MoleculeIndex = ", MoleculeIndex
                                    print*,"MolecularDataIndices(l, :, 1) = ", MolecularDataIndices(l, :, 1)
                                    print*,"MolecularDataIndices(l, :, 2) = ", MolecularDataIndices(l, :, 2)
                                endif


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< determine optical depth $\tau(\nu)^{m,c}$ for each molecule m and component c
                                !<
                                !<     \tau(\nu)^{m,c} = \sum_t \frac{c^3}{8 \pi \nu^3} \, A_{ul} \, N_{\rm tot}^{m,c}
                                !<                              \frac{g_u  e^{-E_l/k_B T_{\rm ex}^{m,c}}}{Q \left(m, T_{\rm ex}^{m,c} \right)}
                                !<                              \left(1 - e^{-h \, \nu /k_B \, T_{\rm ex}^{m,c}} \right)
                                !<                              \times \phi(\nu)^{m,c},
                                !<
                                !< where the sum with index $t$ runs over all spectral line transitions of molecule $m$ within the given frequency
                                !< range. Additionally, the Einstein $A_{ul}$ coefficient\footnote{The indices $u$ and $l$ represent upper and lower
                                !< state, respectively.}, the energy of the lower state $E_l$, the upper state degeneracy $g_u$, and the
                                !< partition function $Q \left(m, T_{\rm ex}^{m,c} \right)$ of molecule $m$ are taken from the embedded
                                !< SQLite3 database, described in  Sect.~\ref{sec:db}. In addition, the values of the excitation temperatures
                                !< $T_{\rm ex}^{m,c}$ and the column densities $N_{\rm tot}^{m,c}$ for the different components and molecules are taken
                                !< from the user defined molfit file. Furthermore, the profile of a spectral line $t$ is given by a Gaussian line profile
                                !<
                                !<     \phi(\nu)^{m,c} = \frac{1}{\sqrt{2 \pi} \, \sigma} 
                                !<                       \cdot e^{-\frac{\left(\nu - \left( \nu_t + \nu_{\rm LSR}^{m,c} \right) \right)^2} {2 \sigma^2}},
                                !<
                                !< which is normalized, i.e.\ $\int_0^{\infty} \phi(\nu) \, d\nu$ = 1. The source velocity $\nu_{\rm LSR}^{m,c}$ for
                                !< each component $c$ of a molecule $m$ is related to the user defined velocity offset
                                !< $\left(v_{\rm offset}^{m,c}\right)$ taken from the aforementioned molfit file, by the following expression
                                !<
                                !<     \nu_{\rm LSR}^{m,c} = -v_{\rm offset}^{m,c} \cdot \nu_t,
                                !<
                                !< where $\nu_t$ indicates the frequency of transition $t$ taken from the SQLite3 database mentioned above. Additionally,
                                !< the variance $\sigma$ of the profile is defined by the velocity width $\left(v_{\rm width}^{m,c}\right)$ described in
                                !< the molfit file for each component $c$ of a molecule $m$:
                                !<
                                !<     \sigma = \frac{v_{\rm width}^{m,c} \cdot \left(\nu_t + \nu_{\rm LSR}^{m,c} \right)}{2 \, \sqrt{2 \, \ln 2}}.
                                !<
                                tau_l = 0.d0


                                !    t = CopyCompMoleculeIndex(c)
                                !    write(11112,*) trim(adjustl(MoleculeNames(t)))
                                !    write(11112,'("TotalNumberOfFrequencyRanges = ", I5)') TotalNumberOfFrequencyRanges
                                !    write(11112,'("TotalNumberOfMolecules = ", I5)') TotalNumberOfMolecules
                                !    write(11112,'("l = ", I5)') l
                                !    write(11112,'("c = ", I5)') c
                                !    write(11112,'("t = ", I5)') t
                                !    write(11112,'("MoleculeIndex = ", I5)') MoleculeIndex




                                FirstMolecularDataIndex = MolecularDataIndices(l, MoleculeIndex, 1) !< get first mol. data index of current freq. range
                                LastMolecularDataIndex = MolecularDataIndices(l, MoleculeIndex, 2)  !< get last mol. data index of current freq. range
                                Do t = FirstMolecularDataIndex, LastMolecularDataIndex      !< loop over all transitions of current molecule and 
                                    !< MolecularData(t, 1) = lFreq
                                    !< MolecularData(t, 2) = EinsteinA
                                    !< MolecularData(t, 3) = lElt
                                    !< MolecularData(t, 4) = UpperStateDegeneracy

                                    ! Debug:
                                    if (dabs(freq - debug_freq) < 1.d-1 .and. ThreadNumber == (-1)) then
                                        print*,"t = ", t
                                        print*,"MolecularData(t, :) = ", MolecularData(t, :)
                                        print*,"FirstMolecularDataIndex, LastMolecularDataIndex = ", FirstMolecularDataIndex, LastMolecularDataIndex
                                    endif


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< determine part of eq. (5):
                                    !<
                                    !<      \frac{c^3}{8 \pi \nu^3} \, A_{ul} \, N_{\rm tot}^{m,c}
                                    !<                              \frac{g_u  e^{-E_l/k_B T_{\rm ex}^{m,c}}}{Q \left(m, T_{\rm ex}^{m,c} \right)}
                                    !<                              \left(1 - e^{-h \, \nu /k_B \, T_{\rm ex}^{m,c}} \right)
                                    !<
                                    !< MolecularData(t, 2) = EinsteinA
                                    !< MolecularData(t, 4) = UpperStateDegeneracy
                                    !< CopymyXCLASSParameter(2, c) = T_rot
                                    !< CopymyXCLASSParameter(3, c) = N_tot
                                    !<
                                    if (Q(c) == 0.d0) then
                                        tau_t = 0.d0
                                    else
                                        tau_t = (((ckms * 1.d3)**2 / (8.d0 * pi * freq**3)) * MolecularData(t, 2) * CopymyXCLASSParameter(3, c) &
                                                 * MolecularData(t, 4) * 1.d-14 * ckms * (dexp(-MolecularData(t, 3)/CopymyXCLASSParameter(2, c)) &
                                                 - dexp(-(MolecularData(t, 3) + (freq/20836.74d0))/ CopymyXCLASSParameter(2, c)))) / Q(c)
                                    endif

                                    ! Debug:
                                    if (dabs(freq - debug_freq) < 1.d-1 .and. ThreadNumber == (-1)) then
                                        print*,'e_low   = ', MolecularData(t, 3)
                                        print*,'e_up    = ', (MolecularData(t, 3) + (freq/20836.74d0))
                                        print*,'>>tau_t = ', tau_t
                                        print*,'--> MolecularData(t, 2) = ', MolecularData(t, 2)
                                        print*,'--> CopymyXCLASSParameter(3, c) = ', CopymyXCLASSParameter(3, c)
                                        print*,'--> MolecularData(t, 4) = ', MolecularData(t, 4)
                                        print*,'--> MolecularData(t, 3) = ', MolecularData(t, 3)
                                        print*,'--> CopymyXCLASSParameter(2, c) = ', CopymyXCLASSParameter(2, c)
                                    endif


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< determine \nu_{\rm LSR}^{m,c}:
                                    !<
                                    !<     \nu_{\rm LSR}^{m,c} = -v_{\rm offset}^{m,c} \cdot \nu_t,
                                    !<
                                    !< CopymyXCLASSParameter(5, c) = V_off
                                    !< MolecularData(t, 1): lFreq
                                    !<
                                    vLSR = -CopymyXCLASSParameter(5, c) / ckms * MolecularData(t, 1)

                                    ! Debug:
                                    if (dabs(freq - debug_freq) < 1.d-1 .and. ThreadNumber == (-1)) print*,'vLSR = ', vLSR


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< determine \sigma:
                                    !<
                                    !<     \sigma = \frac{v_{\rm width}^{m,c} \cdot \left(\nu_t + \nu_{\rm LSR}^{m,c} \right)}{2 \, \sqrt{2 \, \ln 2}}.
                                    !<
                                    !< CopymyXCLASSParameter(4, c) = V_width
                                    !< MolecularData(t, 1): lFreq
                                    !<
                                    sigma = (CopymyXCLASSParameter(4, c) / ckms * (MolecularData(t, 1) + vLSR)) / (2.d0 * dsqrt(2.d0 * dlog(2.d0)))

                                    ! Debug:
                                    if (dabs(freq - debug_freq) < 1.d-1 .and. ThreadNumber == (-1)) print*,'sigma = ', sigma


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< determine \phi(\nu)^{m,c}:
                                    !<
                                    !<     \phi(\nu)^{m,c} = \frac{1}{\sqrt{2 \pi} \, \sigma} 
                                    !<                       * e^{-\frac{\left(\nu - \left( \nu_t + \nu_{\rm LSR}^{m,c} \right) \right)^2} {2 \sigma^2}},
                                    !<
                                    !< MolecularData(t, 1): lFreq
                                    !<
                                    tau_t = tau_t * 1.d0 / (dsqrt(2.d0 * pi) * sigma) &
                                                  * dexp(-1.d0 * ((MolecularData(t, 1) - freq + vLSR)**2) / (2.d0 * (sigma)**2)) * freq / ckms


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< add tau for current transition to sum of taus
                                    tau_l = tau_l + tau_t
                                end Do                                                      !< t: loop over all transitions

                                ! Debug:
                                if (dabs(freq - debug_freq) < 1.d-1 .and. ThreadNumber == (-1)) print*,'tau_l = ', tau_l


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< calculate intensity using eqn. (1) and (10):
                                !< The model function considers finite source size and dust attenuation and is given by:
                                !< 
                                !<     T_{\rm mb}(\nu) = \sum_m \sum_c \eta \left(\theta_{m,c} \right) \left[ J \left(T_{\rm ex}^{m,c}, \nu \right) 
                                !<                                                                          - J \left(T_{\rm bg}, \nu \right) 
                                !<                                                                          - J \left(T_{\rm cbg}, \nu \right) \right]
                                !<                                      \times \left(1 - e^{-\left( \tau(\nu)^{m,c} + \tau_d\right)} \right)
                                !< 
                                !< The sums go over the indices $m$ for molecule, and $c$ for component, respectively. Here, components for each
                                !< molecule can be identified as spatially distinct sources such as clumps, hot dense cores, colder envelopes or
                                !< outflows. They can usually be distinguished by different radial velocities, and do not interact with each other
                                !< radiatively but superimpose in the model. So, the myXCLASS program is able to model a spectrum with an arbitrary
                                !< number of molecules and components, simultaneously.
                                !<
                                !<     T_{\rm mb}(\nu) = T_{\rm mb}(\nu) - \left[ \eta \left(\theta_{m,c} \right) 
                                !<                                         \cdot J \left(T_{\rm ex}^{m,c}, \nu \right)
                                !<                                         \cdot \left(1 - e^{-\tau_d} \right) \right]
                                !< 
                                !< allows the user to define the continuum contribution for each frequency range, individually. Here,
                                !< $\nu_{\rm min}$ indicates the lowest frequency of a given frequency range. $T_{\rm bg}$ and $T_{\rm slope}$, 
                                !< describe the background continuum temperature and the temperature slope, respectively. Please note, if the user 
                                !< defines a background temperature $T_{\rm bg} \leq 0$, the dust continuum offset for each component $c$ of a
                                !< molecule $m$ is subtracted from the model function.
                                !< 
                                if (CopymyXCLASSParameter(11, c) == 1.d0) then              !< determine intensity for emission
                                    LocalIntensity = LocalIntensity + eta_source * (j_tk - j_back - j_cb) * (1.d0 - dexp(-(tau_l + tau_d)))

                                    ! Debug:
                                    if (dabs(freq - debug_freq) < 1.d-1 .and. ThreadNumber == (-1)) then
                                        print*,'>>>>>>>>>>>>>>> LocalIntensity = ', LocalIntensity
                                    endif


                                    if (CopymyXCLASSParameter(6, c) <= 0.d0) then           !< substract dust continuum offset if T_Back <= 0
                                        LocalIntensity = LocalIntensity - eta_source * (j_tk - j_back - j_cb) * (1.d0 - dexp(-tau_d))
                                    endif
                                 else                                                       !< absorption loop here
                                    LocalIntensity = LocalIntensity * dexp(-tau_l) + eta_source * (j_tk - j_back - j_cb) * (1.d0 - dexp(-tau_l))
                                endif
                            endif                                                           !< continue here, if current comp. is not equal to ae


                            !< add background continuum
                            if (c == TotalNumberComponents .and. ae == 2) then
                                LocalIntensity = LocalIntensity + j_back

                                ! Debug:
                                if (dabs(freq - debug_freq) < 1.d-1 .and. ThreadNumber == (-1)) then
                                    print*,'--------------> freq, LocalIntensity = ', freq, LocalIntensity
                                endif
                            endif

                        end Do                                                              !< c: loop over all components
                    end Do                                                                  !< ae: loop over emission/absorption

                    ! Debug:
                    ! if (dabs(freq - debug_freq) < 1.d-1 .and. ThreadNumber == (-1)) stop


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< determine chi^2 value using local calculated intensity
                    chi2Value = chi2Value + (LocalIntensity - ObservationalDataList(FreqIndex, 2))**2


                    !< save model function
                    if (ModelFunctionFlag) then
                        ModelFuncList(FreqIndex) = LocalIntensity

                        ! Debug:
                        !    if (isnan(LocalIntensity)) then
                        !        print*,'ThreadNumber = ', ThreadNumber
                        !        print*,'FreqIndex = ', FreqIndex
                        !    endif
                    endif
                end Do                                                                      !< FreqIndex: loop over frequencies
            end Do                                                                          !< l: loop over frequencies

            ! Debug:
            !    print*,"DataPointIndexFreqRange(:, 1) = ", DataPointIndexFreqRange(:, 1)
            !    print*,"DataPointIndexFreqRange(:, 2) = ", DataPointIndexFreqRange(:, 2)
            !    print*,"ObservationalDataList(DataPointIndexFreqRange(1, 1), :) = ", ObservationalDataList(DataPointIndexFreqRange(1, 1), :)
            !    print*,"ObservationalDataList(DataPointIndexFreqRange(1, 2), :) = ", ObservationalDataList(DataPointIndexFreqRange(1, 2), :)
            !    print*,'>>>>>>>>>>count(isnan(ModelFuncList(:))) = ', count(isnan(ModelFuncList(:)))


            ! Debug:
            ! stop '###### stop subroutine ModelCalcSpectrum ######'
            return
        end subroutine ModelCalcSpectrum


        !>************************************************************************************************************************************************
        !> subroutine: ModelParamFree
        !>
        !> free memory used by variables of the Module Model
        !>
        !>
        !> input variables:     deallocstatus           status of the previous deallocation process
        !>
        !> output variables:    deallocstatus           status of the deallocation process
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 26.08.2014
        !>
        subroutine ModelParamFree(deallocstatus)

            implicit none
            integer :: deallocstatus                                                        !< status of the deallocation process


            !< free memory of slave threads and kill slave threads
            if (myrank == 0) then
                call SlaveWork(1)
            endif


            !< deallocate memory of variables of Modules 'Variables' and 'FunctionCalling'
            if (allocated(lengthexpdata)) deallocate(lengthexpdata, stat = deallocstatus)
            if (allocated(NumberXColumns)) deallocate(NumberXColumns, stat = deallocstatus)
            if (allocated(NumberYColumns)) deallocate(NumberYColumns, stat = deallocstatus)
            if (allocated(NumberHeaderLines)) deallocate(NumberHeaderLines, stat = deallocstatus)
            if (allocated(OutputFileFormat)) deallocate(OutputFileFormat, stat = deallocstatus)
            if (allocated(NumberRanges)) deallocate(NumberRanges, stat = deallocstatus)
            if (allocated(ConverterInfit)) deallocate(ConverterInfit, stat = deallocstatus)
            if (allocated(ValueEmptyOutputFile)) deallocate(ValueEmptyOutputFile, stat = deallocstatus)
            if (allocated(LSRAdjustement)) deallocate(LSRAdjustement, stat = deallocstatus)
            if (allocated(chisqValues)) deallocate(chisqValues, stat = deallocstatus)
            if (allocated(BestSitesParamSet)) deallocate(BestSitesParamSet, stat = deallocstatus)
            if (allocated(paramset)) deallocate(paramset, stat = deallocstatus)
            if (allocated(AtOnceFunction)) deallocate(AtOnceFunction, stat = deallocstatus)
            if (allocated(AtOnceGradient)) deallocate(AtOnceGradient, stat = deallocstatus)
            if (allocated(FirstPointExpData)) deallocate(FirstPointExpData, stat = deallocstatus)
            if (allocated(LastPointExpData)) deallocate(LastPointExpData, stat = deallocstatus)
            if (allocated(expdatax)) deallocate(expdatax, stat = deallocstatus)
            if (allocated(expdatay)) deallocate(expdatay, stat = deallocstatus)
            if (allocated(expdatae)) deallocate(expdatae, stat = deallocstatus)
            if (allocated(CalculatedParameterSets)) deallocate(CalculatedParameterSets, stat = deallocstatus)
            if (allocated(MinRange)) deallocate(MinRange, stat = deallocstatus)
            if (allocated(MaxRange)) deallocate(MaxRange, stat = deallocstatus)
            if (allocated(BestSitesModelValues)) deallocate(BestSitesModelValues, stat = deallocstatus)
            if (allocated(BestSitesChi2Values)) deallocate(BestSitesChi2Values, stat = deallocstatus)
            if (allocated(ModelFunction)) deallocate(ModelFunction, stat = deallocstatus)
            if (allocated(FitParameterName)) deallocate(FitParameterName, stat = deallocstatus)
            if (allocated(FitParameterValue)) deallocate(FitParameterValue, stat = deallocstatus)
            if (allocated(CharacterForComments)) deallocate(CharacterForComments, stat = deallocstatus)
            if (allocated(CharacterSeperatingColumns)) deallocate(CharacterSeperatingColumns, stat = deallocstatus)
            if (allocated(ResamplingMethod)) deallocate(ResamplingMethod, stat = deallocstatus)
            if (allocated(InterpolationMethod)) deallocate(InterpolationMethod, stat = deallocstatus)
            if (allocated(OnlyYColumn)) deallocate(OnlyYColumn, stat = deallocstatus)
            if (allocated(LSRAdjustementFitFlag)) deallocate(LSRAdjustementFitFlag, stat = deallocstatus)
            if (allocated(NormalizationFlag)) deallocate(NormalizationFlag, stat = deallocstatus)
            if (allocated(ExpData_reversed_flag)) deallocate(ExpData_reversed_flag, stat = deallocstatus)
            if (allocated(NumberParamPerLine)) deallocate(NumberParamPerLine, stat = deallocstatus)
            if (allocated(ParameterName)) deallocate(ParameterName, stat = deallocstatus)
            if (allocated(ParameterFormat)) deallocate(ParameterFormat, stat = deallocstatus)
            if (allocated(LeadingString)) deallocate(LeadingString, stat = deallocstatus)
            if (allocated(TrailingString)) deallocate(TrailingString, stat = deallocstatus)
            if (allocated(ParamVisible)) deallocate(ParamVisible, stat = deallocstatus)
            if (allocated(FitFktInput)) deallocate(FitFktInput, stat = deallocstatus)
            if (allocated(FitFktOutput)) deallocate(FitFktOutput, stat = deallocstatus)
            if (allocated(valuesModel_output)) deallocate(valuesModel_output, stat = deallocstatus)
            if (allocated(xPointsModel_output)) deallocate(xPointsModel_output, stat = deallocstatus)


            !< deallocate memory of variables of Module 'Model'
            if (allocated(IsoMolecule)) deallocate(IsoMolecule, stat = deallocstatus)
            if (allocated(IsoMasters)) deallocate(IsoMasters, stat = deallocstatus)
            if (allocated(IsoRatio)) deallocate(IsoRatio, stat = deallocstatus)
            if (allocated(StartFrequency)) deallocate(StartFrequency, stat = deallocstatus)
            if (allocated(EndFrequency)) deallocate(EndFrequency, stat = deallocstatus)
            if (allocated(StepFrequency)) deallocate(StepFrequency, stat = deallocstatus)
            if (allocated(TelescopeSize)) deallocate(TelescopeSize, stat = deallocstatus)
            if (allocated(BackgroundTemperatureRange)) deallocate(BackgroundTemperatureRange, stat = deallocstatus)
            if (allocated(TemperatureSlopeRange)) deallocate(TemperatureSlopeRange, stat = deallocstatus)
            if (allocated(HydrogenColumnDensityRange)) deallocate(HydrogenColumnDensityRange, stat = deallocstatus)
            if (allocated(DustBetaRange)) deallocate(DustBetaRange, stat = deallocstatus)
            if (allocated(KappaRange)) deallocate(KappaRange, stat = deallocstatus)
            if (allocated(SpectrumIndexForFreqRange)) deallocate(SpectrumIndexForFreqRange, stat = deallocstatus)
            if (allocated(ObservationalDataList)) deallocate(ObservationalDataList, stat = deallocstatus)
            if (allocated(DataPointIndexFreqRange)) deallocate(DataPointIndexFreqRange, stat = deallocstatus)
            if (allocated(MoleculeNames)) deallocate(MoleculeNames, stat = deallocstatus)
            if (allocated(ConversionTableMAGIXmyXCLASSParam)) deallocate(ConversionTableMAGIXmyXCLASSParam, stat = deallocstatus)
            if (allocated(IsoRatioConversionTable)) deallocate(IsoRatioConversionTable, stat = deallocstatus)
            if (allocated(IsoNfitConversionTable)) deallocate(IsoNfitConversionTable, stat = deallocstatus)
            if (allocated(NumberComponentsPerMolecule)) deallocate(NumberComponentsPerMolecule, stat = deallocstatus)
            if (allocated(myXCLASSParameter)) deallocate(myXCLASSParameter, stat = deallocstatus)
            if (allocated(CompMoleculeIndex)) deallocate(CompMoleculeIndex, stat = deallocstatus)
            if (allocated(MolNamePartFunc)) deallocate(MolNamePartFunc, stat = deallocstatus)
            if (allocated(lgQ)) deallocate(lgQ, stat = deallocstatus)
            if (allocated(NumEntriesRanges)) deallocate(NumEntriesRanges, stat = deallocstatus)
            if (allocated(MolecularData)) deallocate(MolecularData, stat = deallocstatus)
            if (allocated(MolecularDataIndices)) deallocate(MolecularDataIndices, stat = deallocstatus)
            if (allocated(MolNameRadTrans)) deallocate(MolNameRadTrans, stat = deallocstatus)
            if (allocated(ColumnNamesPartFunc)) deallocate(ColumnNamesPartFunc, stat = deallocstatus)
            if (allocated(TempPartFunc)) deallocate(TempPartFunc, stat = deallocstatus)
            return
        end subroutine ModelParamFree
end Module Model
!*********************************************************************************************************************************************************

