!*********************************************************************************************************************************************************
!>  Module: Model
!>
!>
!>  This module contains subroutines for the myXCLASS program for application in GPU version of MAGIX
!>  Copyright (C) 2009 - 2015  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine ModelInit:                     initializes myXCLASS program
!>      - subroutine ModelCalcChiFunctionLM:        calculates the chi2 values for the Levenberg-Marquard algorithm
!>      - subroutine ModelCalcChiFunctionGeneral:   calculates the chi2 values for several given parameter vector sets
!>      - subroutine ModelCalcChiFunction:          calculates the chi2 values for a given set of parameter vectors
!>      - subroutine ModelCalcSpectrumGPU:          calculates the myXCLASS spectrum for a given parameter vector
!>      - subroutine ModelParamFree:                free memory used by variables of the Module Model
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    29.07.2014  Initial version
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!---------------------------------------------------------------------------------------------------------------------------------------------------------
Module Model
    !> declare variables which are used by functions and subroutines of this module

    use myXCLASSCore
    use cudafor

    implicit none


    !< GPU variables
    integer, device :: Firsti_d                                                             !< working variable for interpolation
    integer, device :: Lasti_d                                                              !< working variable for interpolation
    integer, device :: stepi_d                                                              !< working variable for interpolation
    integer, device :: TotalNumberComponents_d                                              !< counter for total number of components of all mol.
    integer, device :: TotalNumberOfFrequencyRanges_d                                       !< total number of frequency ranges
    integer, device :: TotalNumberOfMolecules_d                                             !< total number of molecules including isotopologues
    integer, device :: NumberOfTemperatures_d                                               !< number of temperatures
    integer, device :: NumberFreeParameter_d                                                !< number of free parameter
    integer, dimension(:), allocatable, device :: NumberComponentsPerMolecule_d             !< number of components per molecule
    !        dimension: (0:TotalNumberOfMolecules)
    integer, dimension(:, :), allocatable, device :: IsoNfitConversionTable_d               !< iso ratios conversion table for free param. index
    !        dimension: (TotalNumberOfMolecules, TotalNumberOfMolecules)
    integer, dimension(:), allocatable, device :: SpectrumIndexForFreqRange_d               !< store spectrum index for each frequency range
    !        dimension: (TotalNumberOfFrequencyRanges)
    integer, dimension(:, :), allocatable, device :: DataPointIndexFreqRange_d              !< index of data pt. of first and last freq. in freq. range
    !        dimension: (TotalNumberOfFrequencyRanges, 2)
    integer, dimension(:, :, :), allocatable, device :: MolecularDataIndices_d              !< start and end index for each molecule and frequency range
    !        dimension: (TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2)
    integer, dimension(:, :), allocatable, device :: ConversionTableMAGIXmyXCLASSParam_d    !< conversion table between myXCLASS and MAGIX param.
    !        dimension: (NumberFreeParameter, 2)
    integer, dimension(:), allocatable, device :: CompMoleculeIndex_d                       !< molecule index for each component
    !        dimension: (TotalNumberComponents)
    real*8, device :: debug_freq_d                                                          !< frequency for debugging
    real*8, device :: ckms_d                                                                !< speed of light in km/s
    real*8, device :: pi_d                                                                  !< pi
    real*8, device :: TempLow_d                                                             !< working variable for extrapolation of part. func.
    real*8, device :: TempHigh_d                                                            !< working variable for extrapolation of part. func.
    real*8, dimension(:), allocatable, device :: TempPartFunc_d                             !< temperatures for partition function
    !<      dimension: (NumberOfTemperatures)
    real*8, dimension(:), allocatable, device :: BackgroundTemperatureRange_d               !< T_Back for each frequency range
    !<      dimension: (TotalNumberOfFrequencyRanges)
    real*8, dimension(:), allocatable, device :: TemperatureSlopeRange_d                    !< T_Slope for each frequency range
    !<      dimension: (TotalNumberOfFrequencyRanges)
    real*8, dimension(:), allocatable, device :: HydrogenColumnDensityRange_d               !< nH for each frequency range
    !<      dimension: (TotalNumberOfFrequencyRanges)
    real*8, dimension(:), allocatable, device :: DustBetaRange_d                            !< beta for each frequency range
    !<      dimension: (TotalNumberOfFrequencyRanges)
    real*8, dimension(:), allocatable, device :: KappaRange_d                               !< kappa for each frequency range
    !<      dimension: (TotalNumberOfFrequencyRanges)
    real*8, dimension(:), allocatable, device :: TelescopeSize_d                            !< size of telescope for each frequency range
    !<      dimension: (TotalNumberOfFrequencyRanges)
    real*8, dimension(:), allocatable, device :: StartFrequency_d                           !< first frequency for each frequency range
    !<      dimension: (TotalNumberOfFrequencyRanges)
    real*8, dimension(:, :), allocatable, device :: ObservationalDataList_d                 !< list containing all observational data
    !<      dimension: (TotalNumberDataPoints, 3)
    real*8, dimension(:, :), allocatable, device :: MolecularData_d                         !< array containing the molecular data for all molecules and
    !<      dimension: (TotalNumberOfTransitions, 4)
    character(len=40), dimension(:), allocatable, device :: MoleculeNames_d                 !< names of all molecules (including isotopologues)
    !<      dimension: (MaxNumberOfMolecules)
    real*8, dimension(:, :), allocatable, device :: IsoRatioConversionTable_d               !< table with iso ratios between iso master and molecule
    !<      dimension: (TotalNumberOfMolecules, TotalNumberOfMolecules)
    real*8, dimension(:, :), allocatable, device :: lgQ_d                                   !< lgQ entries of table PartitionFunction
    !<      dimension: (NumberOfTemperatures, NumberMoleculePartFunc)
    real*8, dimension(:, :), allocatable, device :: myXCLASSParameter_d                     !< array containing all molfit parameters for each component
    !<      dimension: (11, TotalNumberComponents)
    logical, device :: nHFlag_d                                                             !< flag for global setting of nH, kappa and beta
    logical, device :: tbFlag_d                                                             !< flag for global setting T_Back and T_Slope
    logical, device :: IsoFlag_d                                                            !< flag indicating use of isotopologues
    logical, device :: ModelFunctionFlag_d                                                  !< flag for saving model function values


    !< constant arrays for device subprogram
    integer, allocatable, dimension(:, :), device :: CopyCompMoleculeIndex                  !< working variable: local copy of CompMoleculeIndexOrig
    real*8, allocatable, dimension(:, :), device :: PartFunc                                !< working variable: local array for partition function
    real*8, allocatable, dimension(:, :, :), device :: CopymyXCLASSParameter                !< working variable: local copy of myXCLASSParameterOrig
    real*8, allocatable, dimension(:, :, :), device :: CopyIsoRatioConversionTable          !< working variable: local copy of IsoRatioConversionTableOrig


    contains


        !>************************************************************************************************************************************************
        !> subroutine: ModelInit
        !>
        !> initialize myXCLASS program
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 27.07.2014
        !>
        subroutine ModelInit

            implicit none
            integer :: ok                                                                   !< status variablee: name of parameter
            integer :: i, j, k, m, n                                                        !< working variables
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation


            !< initalize myXCLASS variables
            ParallelizationMethod = "GPU"
            IntegrationFlag = .true.
            LogFlag = .false.
            call myXCLASSInitVar


            !<============================================================================================================================================
            !< define variables to GPU variables


            !< print what you do
            if (printflag) then
                print '(11x, "Copy data to GPU RAM .. ", $)'
            endif


            !< allocate variables for partition function
            if (allocated(NumberComponentsPerMolecule_d)) then
                deallocate(NumberComponentsPerMolecule_d, IsoNfitConversionTable_d, SpectrumIndexForFreqRange_d, DataPointIndexFreqRange_d, &
                           MolecularDataIndices_d, ConversionTableMAGIXmyXCLASSParam_d, CompMoleculeIndex_d, TempPartFunc_d, &
                           BackgroundTemperatureRange_d, TemperatureSlopeRange_d, HydrogenColumnDensityRange_d, DustBetaRange_d, KappaRange_d, &
                           TelescopeSize_d, StartFrequency_d, ObservationalDataList_d, MolecularData_d, MoleculeNames_d, IsoRatioConversionTable_d, &
                           lgQ_d, myXCLASSParameter_d, CopyCompMoleculeIndex, PartFunc, CopymyXCLASSParameter, CopyIsoRatioConversionTable, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables NumberComponentsPerMolecule_d, etc.!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            if (printflag) print '(".", $)'


            allocate(NumberComponentsPerMolecule_d(0:TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable NumberComponentsPerMolecule_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(IsoNfitConversionTable_d(TotalNumberOfMolecules, TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable IsoNfitConversionTable_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(SpectrumIndexForFreqRange_d(TotalNumberOfFrequencyRanges), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable SpectrumIndexForFreqRange_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(DataPointIndexFreqRange_d(TotalNumberOfFrequencyRanges, 2), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable DataPointIndexFreqRange_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(MolecularDataIndices_d(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable MolecularDataIndices_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(ConversionTableMAGIXmyXCLASSParam_d(NumberFreeParameter, 2), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable ConversionTableMAGIXmyXCLASSParam_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(CompMoleculeIndex_d(TotalNumberComponents), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable CompMoleculeIndex_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(TempPartFunc_d(NumberOfTemperatures), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable TempPartFunc_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(BackgroundTemperatureRange_d(TotalNumberOfFrequencyRanges), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable BackgroundTemperatureRange_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(TemperatureSlopeRange_d(TotalNumberOfFrequencyRanges), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable TemperatureSlopeRange_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(HydrogenColumnDensityRange_d(TotalNumberOfFrequencyRanges), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable HydrogenColumnDensityRange_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(DustBetaRange_d(TotalNumberOfFrequencyRanges), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable DustBetaRange_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(KappaRange_d(TotalNumberOfFrequencyRanges), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable KappaRange_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(TelescopeSize_d(TotalNumberOfFrequencyRanges), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable TelescopeSize_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(StartFrequency_d(TotalNumberOfFrequencyRanges), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable StartFrequency_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(ObservationalDataList_d(TotalNumberDataPoints, 3), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable ObservationalDataList_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(MolecularData_d(TotalNumberOfTransitions, 4), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable MolecularData_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(MoleculeNames_d(TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable MoleculeNames_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(IsoRatioConversionTable_d(TotalNumberOfMolecules, TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable IsoRatioConversionTable_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(lgQ_d(NumberOfTemperatures, NumberMoleculePartFunc), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable lgQ_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(myXCLASSParameter_d(11, TotalNumberComponents), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable myXCLASSParameter_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            if (printflag) print '(".", $)'


            allocate(CopyCompMoleculeIndex(2000, TotalNumberComponents), PartFunc(2000, TotalNumberComponents), &
                     CopymyXCLASSParameter(2000, 11, TotalNumberComponents), &
                     CopyIsoRatioConversionTable(2000, TotalNumberOfMolecules, TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable myXCLASSParameter_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            NumberFreeParameter_d = NumberFreeParameter
            NumberComponentsPerMolecule_d = NumberComponentsPerMolecule
            IsoNfitConversionTable_d = IsoNfitConversionTable
            SpectrumIndexForFreqRange_d = SpectrumIndexForFreqRange
            DataPointIndexFreqRange_d = DataPointIndexFreqRange
            MolecularDataIndices_d = MolecularDataIndices
            ConversionTableMAGIXmyXCLASSParam_d = ConversionTableMAGIXmyXCLASSParam
            CompMoleculeIndex_d = CompMoleculeIndex
            TempPartFunc_d = TempPartFunc
            BackgroundTemperatureRange_d = BackgroundTemperatureRange
            TemperatureSlopeRange_d = TemperatureSlopeRange
            HydrogenColumnDensityRange_d = HydrogenColumnDensityRange
            DustBetaRange_d = DustBetaRange
            KappaRange_d = KappaRange
            TelescopeSize_d = TelescopeSize
            StartFrequency_d = StartFrequency
            ObservationalDataList_d = ObservationalDataList
            MolecularData_d = MolecularData
            MoleculeNames_d = MoleculeNames(1:TotalNumberOfMolecules)
            IsoRatioConversionTable_d = IsoRatioConversionTable
            lgQ_d = lgQ
            myXCLASSParameter_d = myXCLASSParameter
            Firsti_d = Firsti
            Lasti_d = Lasti
            stepi_d = stepi
            TotalNumberComponents_d = TotalNumberComponents
            TotalNumberOfFrequencyRanges_d = TotalNumberOfFrequencyRanges
            TotalNumberOfMolecules_d = TotalNumberOfMolecules
            NumberOfTemperatures_d = NumberOfTemperatures
            debug_freq_d = debug_freq
            ckms_d = ckms
            pi_d = pi
            TempLow_d = TempLow
            TempHigh_d = TempHigh
            nHFlag_d = nHFlag
            tbFlag_d = tbFlag
            IsoFlag_d = IsoFlag
            if (printflag) print '(".", $)'


            CopyIsoRatioConversionTable = 0.d0
            CopyCompMoleculeIndex = 0
            CopymyXCLASSParameter = 0.d0
            k = min(1000, TotalNumberOfMolecules)
            i = min(1000, TotalNumberComponents)
            Do n = 1, 1000


                !< copy iso ratio conversion table
                Do m = 1, k
                    Do j = 1, k
                        CopyIsoRatioConversionTable(n, m, j) = IsoRatioConversionTable(m, j)
                    end Do
                end Do


                !< copy CompMoleculeIndex
                CopyCompMoleculeIndex(n, 1:i) = CompMoleculeIndex(1:i)


                !< copy myXCLASS parameter array
                Do j = 1, 11
                    CopymyXCLASSParameter(n, j, 1:i) = myXCLASSParameter(j, 1:i)
                end Do
            end Do
            PartFunc = 0.d0


            !< we've done
            if (printflag) then
                print '("done!")'
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< set storage flag for model function values
            DontStoreModelFuncValuesFlag = .true.                                           !< model function values are stored for each function call
            ParallezitionMethod = 1                                                         !< set parallelization method to OpenMP (=1)
            myrankOfMPI = 0                                                                 !< copy thread number to global variable
            NumberOfUsedThreads = ParallelizationFlag                                       !< in Module 'FunctionCalling' we need only all thread


            return
        end subroutine ModelInit


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunctionLM
        !>
        !> calculates the chi2 values for the Levenberg-Marquard algorithm
        !>
        !>
        !> input variables:     ma:                 total number of parameters
        !>                      a:                  array containing the parameter set
        !>                      ia:                 flags for including/excluding parameter in the fit
        !>                      NumberFreeParameterCopy:             number of fitted parameters
        !>                      fitparam:           parameter which have to be optimized
        !>                      colx:               number of columns in experimental x data
        !>                      NumFile:            number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>                      FitFunctionOut:     values of the fit function at all calculated points
        !>                      Chi2Values:         values of the fit function at all calculated points
        !>                      alpha:              matrix alpha (only used for Levenberg-Marquardt algorithm)
        !>                      beta2:              beta2 array (only used for Levenberg-Marquardt algorithm)
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 31.07.2014
        !>
        subroutine ModelCalcChiFunctionLM(ma, a, ia, NumberFreeParameterCopy, fitparam, colx, NumFile, MaxL, MaxCol, FitFunctionOut, Chi2Values, &
                                          alpha, beta2)

            use omp_lib

            implicit none
            integer :: i, j, k, l, m, n, fitnum, NumberFile                                 !< loop variables
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: counter                                                              !< counter for ModelFunction
            integer :: NumberFreeParameterCopy                                              !< number of fitted parameters
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            integer :: NumParameterVectors                                                  !< number of parameter vectors in ParameterVectorSet
            real*8 :: dy, sig2i, wt, ymod                                                   !< working variables
            real*8 :: value                                                                 !< calculated value of the fit function
            real*8 :: variation, d1, d2                                                     !< working variables
            real*8, dimension(ma) :: steph                                                  !< working variable
            real*8, dimension(ma) :: modelparam                                             !< array containing the parameter set
            real*8, dimension(ma) :: modelparamcopy                                         !< copy of modelparam
            real*8, dimension(ma) :: a                                                      !< array containing the parameter set
            real*8, dimension(ma, ma) :: alpha                                              !< matrix alpha
            real*8, dimension(NumberFreeParameterCopy) :: beta2                             !< beta2 array
            real*8, dimension(NumberFreeParameterCopy) :: fitparam                          !< parameter which have to be optimized
            real*8, dimension(NumFile, MaxL, MaxCol) :: FitFunctionOut                      !< values of the fit function at all calculated points
            real*8, dimension(NumFile, MaxL, MaxCol) :: Chi2Values                          !< values of the fit function at all calculated points
            real*8, dimension(NumberFreeParameterCopy, NumFile, MaxCol, MaxL) :: GradientHelp
            real*8, dimension(ma) :: dyda                                                   !< gradient of the fit function
            real*8, dimension(NumberFreeParameterCopy + 1) :: chi2ValuesVector              !< vector containing chi2 values for each parameter vector
                                                                                            !< in ParameterVectorSet
            real*8, dimension(NumberFreeParameterCopy + 1, NumberFreeParameterCopy) :: ParameterVectorSet
                                                                                            !< set of parameter vectors calculated by model module
            logical :: IntegerTrue                                                          !< flag for identification of integer numbers
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit
            character(len=100) :: HelpString                                                !< working variables
            character(len=100), dimension(NumberFreeParameterCopy) :: FormattedParmValues   !< formatted parameter values for chi2 log file


            character(len=50) :: valueh, valuel                                             !< working variable for determine gradient
            logical :: equal_flag                                                           !< required for string comparison
            logical, dimension(ma) :: ChangeSign_Flag                                       !< change sign flag
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function is stored or not


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize values for fit function calculation
            Do j = 1, NumberFreeParameterCopy                                               !< Initialize (symmetric) alpha, beta2.
                Do k = 1, j
                    alpha(j,k) = 0.d0                                                       !< clear alpha partially
                end Do
                beta2(j) = 0.d0                                                             !< clear beta2
            end Do


            !< update the parameter set with the current values of the fit parameters some algorithms optimize only the parameters in the array fitparam
            !< all other parameter in the parameter set a are not included in these algorithms but are required for the calculation of the fit function
            k = 0
            Do i = 1, ma
                if (ia(i)) then
                    k = k + 1
                    a(i) = fitparam(k)
                endif
            end Do
            ParameterVectorSet(1, :) = fitparam(:)                                          !< first entry of ParameterVectorSet contains the current
                                                                                            !< parameter vector
            ! Debug:
            ! print*,'>>>>>>>>>>>>>fitparam = ', fitparam(:)


            !< check, if gradient is necessary (for Levenberg-Marquardt it is always necessary!)
            NumParameterVectors = 1
            if (Gradientflag) then
                NumParameterVectors = NumberFreeParameterCopy + 1


                !< initialize some working parameter
                steph = 0.d0
                ChangeSign_Flag = .false.
                modelparamcopy = a                                                 !< dublicate array containing the parameter set


                !< calculate parameter vector for gradient calculation

                !$omp parallel default(shared) &
                !$omp shared(OutputFileFormat, NumberRanges, ValueEmptyOutputFile, LSRAdjustement, FirstPointExpData, LastPointExpData, idum) &
                !$omp shared(MinRange, MaxRange, NaNReplaceString, StandardWorkingDirectory, CharacterForComments, CharacterSeperatingColumns) &
                !$omp shared(ResamplingMethod, InterpolationMethod, RenormalizedChi2, OnlyYColumn, LSRAdjustementFitFlag, NormalizationFlag) &
                !$omp shared(AtOnceGradient, ia, paramset, modelparamcopy, ConverterInfit) &
                !$omp shared(ParameterName, ParameterFormat, LeadingString, TrailingString, ParamVisible, NumberLinesOutput) &
                !$omp shared(printflag, chisqValues, NumberExpFiles, modelnumber, lengthexpdata, currentpath, NumberHeaderLines, QualityLimit) &
                !$omp shared(FitParameterName, FitParameterValue, NumberColumnsBegin, NumberColumnsEnd, NumberParamPerLine, CalculationMethod) &
                !$omp shared(CommandWordOutput, DetChi2, MaxParameter, ExternalThreadNumber, expdatax, expdatay, expdatae, TempDirectory) &
                !$omp shared(NumberXColumns, NumberYColumns, MaxColX, MaxColY, parameternumber, NumberInputFiles, ParallelizationFlag, JobID) &
                !$omp shared(MaxInputLines, UseCalculationReduction, WriteChi2Flag, Gradientflag,  initflag, CalculatedParameterSets) &
                !$omp shared(LastAlgorithmFlag, ChangeSign_Flag, steph, NumberFreeParameterCopy, ParameterVectorSet, fitparam) &
                !$omp shared(CurrentNumberLinesCalcReduction, ochisq, NumberLinesChi2, InputDataPath, FitFktInput, ModelFunction, FitFktOutput) &
                !$omp shared(ExeCommandStartScript, NumberOutputFiles, CurrentExpFile, GradientMethod, PathStartScript, CurrentYColumn) &
                !$omp shared(ExpData_reversed_flag, BestSitesModelValues, BestSitesChi2Values, BestSitesParamSet, GradientVariationValue) &
                !$omp private(fitnum, i, j, k, modelparam, d1, d2, sig2i, NumberFile, variation, value, valueh, valuel) &
                !$omp private(IntegerTrue, NumInputFile_index, i_index, j_index, equal_flag)
                !$omp do

                Do fitnum = 1, NumberFreeParameterCopy                                      !< loop over all free parameter
                    !$omp critical
                    ParameterVectorSet(fitnum + 1, :) = fitparam(:)
                    !$omp end critical
                    modelparam = modelparamcopy                                             !< load unmodified parameter values
                    i = ConverterInfit(fitnum)                                              !< get appropriate parameter index within parameter set
                    call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)

                    ! Debug:
                    ! print*,'fitnum, NumInputFile_index, i_index, j_index = ', fitnum, NumInputFile_index, i_index, j_index


                    !< check if parameter is within parameter limits
                    if (modelparamcopy(i) < paramset(3, i) .or. modelparamcopy(i) > paramset(4, i)) then
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                        write(logchannel,'(2x,"The parameter ",A," is out of limits.")') &
                                              trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        write(logchannel,'(2x,"Upper limit = ", ES25.15)') paramset(4, i)
                        write(logchannel,'(2x,"Lower limit = ", ES25.15)') paramset(3, i)
                        write(logchannel,'(2x,"Value of parameter = ", ES25.15)') modelparamcopy(i)
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')

                        print '(" ")'
                        print '(" ")'
                        print '(" ")'
                        print '(11x,"Error in subroutine ModelCalcChiFunctionLM:")'
                        print '(13x,"The parameter ",A," is out of limits.")', trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        print '(13x,"Upper limit = ", ES25.15)', paramset(4, i)
                        print '(13x,"Lower limit = ", ES25.15)', paramset(3, i)
                        print '(13x,"Value of parameter = ", ES25.15)',modelparamcopy(i)
                        print '(" ")'
                        print '(13x,"Program aborted!")'
                        stop
                    endif


                    !< determine strength of variation
                    variation = GradientVariationValue                                      !< variation of the parameter in percent/100
                    !$omp critical
                    steph(i) = dabs(modelparamcopy(i) * variation)                          !< define stepsize for foating point numbers
                    if (modelparamcopy(i) == 0.d0) then
                        if (IntegerTrue) then                                               !< is parameter an integer ??
                            steph(i) = 1.d0                                                 !< variation of the parameter in percent/100
                        else
                            steph(i) = variation                                            !< variation of the parameter in percent/100
                        endif
                    elseif (IntegerTrue .and. steph(i) < 1.d0) then
                        steph(i) = 1.d0
                    endif
                    !$omp end critical

                    ! Debug:
                    ! print*,'i = ', i
                    ! print*,'modelparamcopy(i) = ', modelparamcopy(i)
                    ! print*,'steph(i) = ',steph(i)
                    ! print*,'modelparamcopy(i) + steph(i) = ', modelparamcopy(i) + steph(i)
                    ! print*,'paramset(3, i) = ', paramset(3, i)
                    ! print*,'paramset(4, i) = ', paramset(4, i)
                    ! print*,'modelparamcopy(i) - steph(i) = ', modelparamcopy(i) - steph(i)


                    !< test, if we can accelerate the calculation
                    value = modelparamcopy(i) + steph(i)
                    if (value < paramset(3, i) .or. paramset(4, i) < value) then
                        if (value > paramset(4, i)) then                                    !< is f(x_i + h) > upper limit ?
                            value = modelparamcopy(i) - steph(i)                            !< use f(x_i - h)
                            if (value < paramset(3, i)) then
                                if (dabs(modelparamcopy(i) - paramset(4, i)) < 1.d-12) then
                                    write(logchannel,*)
                                    write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                                    write(logchannel,'(2x,"The gradient for parameter ",A," cannot be calculated.")') &
                                                      trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                                    write(logchannel,'(2x,"The variation runs out of limits.")')
                                    write(logchannel,*)
                                    write(logchannel,'(2x,"Please increase upper and lower limits for this parameter or reduce value of variation.")')
                                    write(logchannel,*)
                                    write(logchannel,*)
                                    write(logchannel,'(2x,"Upper limit = ",ES25.15)') paramset(4, i)
                                    write(logchannel,'(2x,"Lower limit = ",ES25.15)') paramset(3, i)
                                    write(logchannel,'(2x,"value of variation = ", ES25.15)') variation
                                    write(logchannel,'(2x,"Value of parameter = ", ES25.15)') modelparamcopy(i)
                                    write(logchannel,'(" ")')
                                    write(logchannel,'("Program aborted!")')

                                    print '(" ")'
                                    print '(" ")'
                                    print '(" ")'
                                    print '("Error in subroutine ModelCalcChiFunctionLM:")'
                                    print '(2x,"The gradient for parameter ",A," cannot be calculated.")', &
                                                      trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                                    print '(2x,"The variation runs out of limits.")'
                                    print '(" ")'
                                    print '(2x,"Please increase upper and lower limits for this parameter or reduce value of variation.")'
                                    print '(" ")'
                                    print '(" ")'
                                    print '(2x,"Upper limit = ", ES25.15)', paramset(4, i)
                                    print '(2x,"Lower limit = ", ES25.15)', paramset(3, i)
                                    print '(2x,"value of variation = ", ES25.15)', variation
                                    print '(2x,"Value of parameter = ", ES25.15)', modelparamcopy(i)
                                    print '(" ")'
                                    print '("Program aborted!")'
                                    stop
                                endif
                                value = paramset(4, i)
                            else
                                !$omp critical
                                ChangeSign_Flag(i) = .true.
                                !$omp end critical
                            endif
                        endif
                    endif

                    ! Debug:
                    ! print*,'> value, modelparamcopy(i), steph(i) = ',value, modelparamcopy(i), steph(i)


                    !< check, if the variation leads in combination with the Format to a variation in the current parameter
                    if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                        .or. index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                        write(valueh, ParameterFormat(NumInputFile_index, i_index, j_index)) int(value)
                        write(valuel, ParameterFormat(NumInputFile_index, i_index, j_index)) int(modelparamcopy(i))
                    else
                        write(valueh, ParameterFormat(NumInputFile_index, i_index, j_index)) value
                        write(valuel, ParameterFormat(NumInputFile_index, i_index, j_index)) modelparamcopy(i)
                    endif
                    equal_flag = .true.
                    valueh = adjustl(valueh)
                    valuel = adjustl(valuel)
                    if (len_trim(valueh) == len_trim(valuel)) then
                        Do j = 1, len_trim(valueh)
                            if (valueh(j:j) /= valuel(j:j)) then
                                equal_flag = .false.
                                exit
                            endif
                        end Do
                    else
                        equal_flag = .false.
                    endif
                    if (equal_flag) then                                                    !< both expressions are equal
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                        write(logchannel,'(2x,"The format specification of the parameter ",A)') &
                                trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        write(logchannel,'(2x,"prevents the variation of the current parameter.")')
                        write(logchannel,'(2x," ")')
                        write(logchannel,'(2x,"The gradient entry for this parameter is set to zero. Therefore")')
                        write(logchannel,'(2x,"no variation of this parameter in the current iteration is done")')
                        write(logchannel,'(" ")')

                        print '(" ")'
                        print '(" ")'
                        print '(" ")'
                        print '(11x,"Error in subroutine ModelCalcChiFunctionLM:")'
                        print '(13x,"The format specification of the parameter ",A)', &
                               trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        print '(13x,"prevents the variation of the current parameter.")'
                        print '(" ")'
                        print '(13x,"The gradient entry for this parameter is set to zero. Therefore")'
                        print '(13x,"no variation of this parameter in the current iteration is done")'
                        print '(" ")'
                    endif


                    !< modify the ith parameter
                    !$omp critical
                    modelparam(i) = value                                                   !< modify value of the ith parameter
                    ParameterVectorSet(fitnum + 1, fitnum) = value
                    !$omp end critical

                end Do

                !$omp end do
                !$omp end parallel                                                          !< end of parallel environment

                modelparam = modelparamcopy                                                 !< restore old paramter values
            endif

            ! Debug:
            !    Do l = 2, NumParameterVectors
            !        Do k = 1, NumberFreeParameterCopy
            !            if (dabs(ParameterVectorSet(l, k) - ParameterVectorSet(1, k)) > 1.d-6) then
            !                print*,'l, k, ParameterVectorSet(l, k) = ', l, k, ParameterVectorSet(l, k), ParameterVectorSet(1, k), &
            !                                                            dabs(ParameterVectorSet(l, k) - ParameterVectorSet(1, k))
            !            endif
            !        end Do
            !    end Do
            !    stop


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            ModelFunctionFlag = .true.
            call ModelCalcChiFunction(NumberFreeParameterCopy, NumParameterVectors, ParameterVectorSet(1:NumParameterVectors,:), ModelFunctionFlag, &
                                      TotalNumberComponents, TotalNumberOfMolecules, chi2ValuesVector)

            ! Debug:
            ! print*,' '
            ! print*,'chi2ValuesVector(:) = ', chi2ValuesVector(:)
            ! stop 'Test LM-algorithm up to here!'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< start loop for determine chi**2
            GradientHelp = 0.d0
            FitFunctionOut = 0.d0                                                           !< reset array containing the values of the fit function
            Chi2Values = 0.d0
            chisqValues = 0.d0
            counter = 0
            Do NumberFile = 1, NumberExpFiles                                               !< loop over exp. data files
                CurrentExpFile = NumberFile


                !< start loop over all lines(expdata)
                Do n = 1, NumberYColumns(NumberFile)                                        !< loop over y-columns
                    Do i = 1, lengthexpdata(NumberFile)                                     !< loop over all line of current exp. data file
                        CurrentYColumn = n
                        counter = counter + 1


                        !< get fit function
                        ymod = ModelFunction(1, NumberFile, n, i)

                        ! Debug:
                        !    print*,'>> NumberFile, n, i, counter, ymod = ', NumberFile, n, i, counter, ymod
                        !    print*,'>> ModelFunction(1, NumberFile, n, i) = ', ModelFunction(1, NumberFile, n, i)


                        !< get gradient
                        if (Gradientflag) then
                            dyda = 0.d0
                            Do l = 1, NumberFreeParameterCopy
                                j = ConverterInfit(l)                                       !< get appropriate parameter index within parameter set

                                ! order new: NumberFile, n, i, l,      j
                                ! order old: NumberFile, n, i, l,      j
                                ! order org: NumberFile, k, j, fitnum, i

                                ! Debug:
                                !    print*,'l, j = ', l, j
                                !    print*,'ParameterVectorSet(l + 1, l) = ', ParameterVectorSet(l + 1, l)
                                !    print*,'steph(j) = ', steph(j)
                                !    print*,'ChangeSign_Flag(j) = ', ChangeSign_Flag(j)
                                !    print*,'ModelFunction(l + 1, NumberFile, n, i) = ', ModelFunction(l + 1, NumberFile, n, i)
                                !    print*,'GradientMethod = ', GradientMethod


                                !< determine gradient for numerical recepies version
                                if (GradientMethod == 1) then
                                    if (ParameterVectorSet(l + 1, l) > paramset(4, j) .or. ChangeSign_Flag(j)) then
                                        dyda(j) = (ModelFunction(1, NumberFile, n, i) - ModelFunction(l + 1, NumberFile, n, i)) / steph(j)
                                    else
                                        dyda(j) = (ModelFunction(l + 1, NumberFile, n, i) - ModelFunction(1, NumberFile, n, i)) / steph(j)
                                    endif


                                !< determine gradient for minpack version
                                elseif (GradientMethod == 2) then
                                    sig2i = 1.d0
                                    if (expdatae(NumberFile, i, n) /= 0.d0) then            !< do the experimental datas include errors
                                        sig2i = 1.d0 / (expdatae(NumberFile, i, n) * expdatae(NumberFile, i, n))  !< define sig2i factor
                                    endif
                                    d1 = (expdatay(NumberFile, i, n) - ModelFunction(1, NumberFile, n, i))**2 * sig2i
                                    d2 = (expdatay(NumberFile, i, n) - ModelFunction(l + 1, NumberFile, n, i))**2 * sig2i
                                    if (ParameterVectorSet(l + 1, l) > paramset(4, j) .or. ChangeSign_Flag(j)) then
                                        ! ModelFunction(l + 1, NumberFile, n, i) = (d1 - d2) / steph(j)
                                        GradientHelp(l, NumberFile, n, i) = (d1 - d2) / steph(j)
                                    else
                                        ! ModelFunction(l + 1, NumberFile, n, i) = (d2 - d1) / steph(j)
                                        GradientHelp(l, NumberFile, n, i) = (d2 - d1) / steph(j)
                                    endif
                                    dyda(j) = GradientHelp(l, NumberFile, n, i)
                                endif
                            end Do

                            ! Debug:
                            ! print*,'ymod = ', ymod
                            ! print*,'dyda(:) = ', dyda(:)
                            ! print*,'a = ', a
                            ! print*,'############################################################################'
                            ! stop
                        endif
                        FitFunctionOut(NumberFile, i, n) = ymod                             !< save value of fit function

                        ! Debug:
                        !    print*,' '
                        !    print*,' '
                        !    print*,'NumberExpFiles = ',NumberExpFiles
                        !    print*,'lengthexpdata(NumberFile) = ',lengthexpdata(NumberFile)
                        !    print*,'NumberYColumns(NumberFile) = ',NumberYColumns(NumberFile)
                        !    print*,'NumberFile = ',NumberFile
                        !    print*,'i = ',i
                        !    print*,'n = ',n
                        !    print*,'lenposdatexp = ',lenposdatexp
                        !    print*,'posdatexp = ',posdatexp
                        !    print*,'expdatay(NumberFile,i,n) = ',expdatay(NumberFile,i,n)
                        !    print*,'ymod = ',ymod
                        !    print*,'dyda = ',dyda(1:NumberFreeParameterCopy)
                        !    if (i==3) stop


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< determine chi**2 by calculating the difference (y_i^{obs) - y_i(fit))**2
                        if (abs(DetChi2) == 1) then

                            sig2i = 1.d0
                            if (expdatae(NumberFile,i,n) /= 0.d0) then                      !< do the experimental datas include errors
                                sig2i = 1.d0/(expdatae(NumberFile,i,n) * expdatae(NumberFile,i,n))      !< define sig2i factor
                            endif
                            dy = (expdatay(NumberFile, i, n) - ymod)                        !< define distance between fit and data
                            Chi2Values(NumberFile, i, n) = dy * dy * sig2i                  !< save chi^2
                            chisqValues(0) = chisqValues(0) + dy * dy * sig2i               !< And find chi^2.


                            if (Gradientflag) then
                                j = 0
                                Do l = 1, ma                                                !< loop over all parameters
                                    if (ia(l)) then                                         !< is the lth parameters optimized?
                                        j = j + 1
                                        wt = dyda(l) * sig2i                                !< define weighting factor
                                        k = 0
                                        Do m = 1, l                                         !< determine alpha matrix
                                            if (ia(m)) then
                                                k = k + 1
                                                alpha(j,k) = alpha(j,k) + wt * dyda(m)
                                            endif
                                        end Do
                                        beta2(j) = beta2(j) + dy * wt                       !< calculate beta2 array
                                    endif
                                end Do
                            endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< determine chi**2 by calculating the difference (y_i^{obs)**2 - y_i(fit)**2)
                        elseif (abs(DetChi2) == 2) then

                            sig2i = 1.d0
                            if (expdatae(NumberFile,i,n) /= 0.d0) then                      !< do the experimental datas include errors
                                sig2i = 1.d0/(expdatae(NumberFile,i,n) * expdatae(NumberFile,i,n))      !< define sig2i factor
                            endif
                            dy = (expdatay(NumberFile, i, n)**2 - ymod**2)
                            Chi2Values(NumberFile, i, n) = dy * dy * sig2i                  !< save chi^2
                            chisqValues(0) = chisqValues(0) + dy * dy * sig2i               !< And find chi^2.


                            if (Gradientflag) then
                                j = 0
                                Do l = 1, ma                                                !< loop over all parameters
                                    if (ia(l)) then                                         !< is the lth parameters optimized?
                                        j = j + 1
                                        wt = dyda(l) * sig2i                                !< define weighting factor
                                        k = 0
                                        Do m = 1, l                                         !< determine alpha matrix
                                            if (ia(m)) then
                                                k = k + 1
                                                alpha(j, k) = alpha(j, k) + wt * dyda(m)
                                            endif
                                        end Do
                                        beta2(j) = beta2(j) + dy * wt                       !< calculate beta2 array
                                    endif
                                end Do
                            endif

                        endif
                    end Do                                                                  !< loop over all line of current exp. data file
                end Do                                                                      !< loop over y-columns
            end Do                                                                          !< loop over exp. data files

            ! Debug:
            ! print*,'chisqValues(0) = ', chisqValues(0)


            !< only used for MINPACK version of Levenberg-Marquardt algorithm
            if (GradientMethod == 2) then
                Do l = 1, NumberFreeParameterCopy
                    ModelFunction(l + 1, :, :, :) = GradientHelp(l, :, :, :)
                end Do
            endif


            !< Fill in the symmetric side.
            Do j = 2, NumberFreeParameterCopy
                Do k = 1, (j - 1)
                   alpha(k, j) = alpha(j, k)
                end Do

                ! Debug:
                ! print*,'j = ', j
                ! print*,'alpha(j,:) = ', alpha(j,:NumberFreeParameterCopy)
                ! print*,'beta2(j) = ', beta2(j)
            end Do


            !< writing current value of chi**2 and corresponding values of parameters to file
            if (WriteChi2Flag) then
                NumberLinesChi2 = NumberLinesChi2 + 1
                k = 0
                FormattedParmValues(:)(:) = ""
                Do i = 1, ma
                    if (ia(i)) then
                        k = k + 1
                        a(i) = fitparam(k)

                        ! Debug:
                        !   print*,'fitparam(k) = ',k,fitparam(k)


                        !< build list with fit parameters
                        HelpString = ""
                        call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)
                        if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                            .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(i))
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) int(a(i))
                            endif
                        else
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(i)
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) a(i)
                            endif
                        endif
                        FormattedParmValues(k) = trim(adjustl(HelpString))
                    endif
                end Do


                !< write line of formatted parameter values to log file
                write(Chi2Channel,'(ES25.15,$)') chisqValues(0)
                Do i = 1, NumberFreeParameterCopy
                    write(Chi2Channel,'(1x,A,$)') " " // trim(adjustl(FormattedParmValues(i)))
                end Do
                write(Chi2Channel,*)
            endif


            return
        end subroutine ModelCalcChiFunctionLM


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunctionGeneral
        !>
        !> calculates the chi2 values for several given parameter vector sets
        !>
        !>
        !> input variables:     ma:                 total number of parameters
        !>                      ia:                 flags for including/excluding parameter in the fit
        !>                      a:                  array containing the parameter set
        !>                      NumberParamVectors:         number of parameter vectors
        !>                      NumberFreeParameterCopy:    number of fitted parameters
        !>                      ParameterVectorSet:
        !>                      NumFile:            number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>                      FitFunctionOut:     values of the fit function at all calculated points
        !>                      Chi2Values:         values of the fit function at all calculated points
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 19.08.2014
        !>
        subroutine ModelCalcChiFunctionGeneral(ma, ia, a, NumberParamVectors, NumberFreeParameterCopy, NumFile, MaxL, MaxCol, ParameterVectorSet, &
                                               chi2ValuesVector)

            use omp_lib

            implicit none
            integer :: ma                                                                   !< total number of parameters
            integer :: NumberParamVectors                                                   !< number of parameter vectors
            integer :: NumberFreeParameterCopy                                              !< number of fitted parameters
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            real*8, dimension(ma) :: a                                                      !< array containing the parameter set
            real*8, dimension(NumberParamVectors, NumberFreeParameterCopy) :: ParameterVectorSet   !< set of parameter vectors calculated by model module
            real*8, dimension(NumberParamVectors) :: chi2ValuesVector                       !< vector containing chi2 values for each parameter vector
                                                                                            !< in ParameterVectorSet
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit


            !< working variables
            integer :: i, k, l, m, n, NumberFile                                            !< loop variables
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            real*8 :: sig2i                                                                 !< working variable for error (sigma)
            real*8 :: val                                                                   !< working variable
            real*8, dimension(1) :: chi2valDummy                                            !< dummy argument
            character(len=100) :: HelpString                                                !< working variables
            character(len=100), dimension(NumberFreeParameterCopy) :: FormattedParmValues   !< formatted parameter values for chi2 log file
            logical :: IntegerTrue                                                          !< flag for identification of integer numbers
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function is stored or not
            logical :: StoredBefore_Flag                                                    !< used to avoid double entries in BestSitesParamSet


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            ModelFunctionFlag = .false.
            chi2ValuesVector = 0.d0
            call ModelCalcChiFunction(NumberFreeParameterCopy, NumberParamVectors, ParameterVectorSet, ModelFunctionFlag, TotalNumberComponents, &
                                      TotalNumberOfMolecules, chi2ValuesVector)
            ! Debug:
            ! print*,' '
            ! print*,'chi2ValuesVector(:) = ', chi2ValuesVector(:)
            ! stop 'Test LM-algorithm up to here!'


            !< writing current value of chi**2 and corresponding values of parameters to file
            Do l = 1, NumberParamVectors
                if (WriteChi2Flag) then
                    NumberLinesChi2 = NumberLinesChi2 + 1
                    k = 0
                    FormattedParmValues(:)(:) = ""
                    Do i = 1, ma
                        if (ia(i)) then
                            k = k + 1
                            a(i) = ParameterVectorSet(l, k)


                            !< build list with fit parameters
                            HelpString = ""
                            call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)
                            if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                                .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(i))
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) int(a(i))
                                endif
                            else
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(i)
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) a(i)
                                endif
                            endif
                            FormattedParmValues(k) = trim(adjustl(HelpString))
                        endif
                    end Do


                    !< write line of formatted parameter values to log file
                    write(Chi2Channel,'(ES25.15,$)') chi2ValuesVector(l)
                    Do i = 1, NumberFreeParameterCopy
                        write(Chi2Channel,'(1x,A,$)') " " // trim(adjustl(FormattedParmValues(i)))
                    end Do
                    write(Chi2Channel,*)
                endif


                !< store model function values and correpsonding chi2 values for the best fitness
                Do k = 1, QualityLimit
                    if (chi2ValuesVector(l) < BestSitesParamSet(k, 1) .or. PlotIterationFlag) then
                        StoredBefore_Flag = .false.
                        if (QualityLimit > 1) then
                            Do m = 1, QualityLimit
                                if (chi2ValuesVector(l) == BestSitesParamSet(m, 1)) then
                                    call CompareTwoParameterSets(StoredBefore_Flag, ma, NumberFreeParameterCopy, ia, BestSitesParamSet(m, 2:), &
                                                                 ParameterVectorSet(l, :))
                                    if (StoredBefore_Flag) then
                                        exit
                                    endif
                                endif
                            end Do
                        endif
                        if (.not. StoredBefore_Flag .or. PlotIterationFlag) then
                            if (k < QualityLimit) then
                                Do m = QualityLimit, (k + 1), (-1)
                                    BestSitesParamSet(m, :) = BestSitesParamSet((m - 1), :)
                                    BestSitesModelValues(m, :, :, :) = BestSitesModelValues((m - 1), :, :, :)
                                    BestSitesChi2Values(m, :, :, :) = BestSitesChi2Values((m - 1), :, :, :)
                                end Do
                            endif
                            BestSitesParamSet(k, 1) = chi2ValuesVector(l)
                            BestSitesParamSet(k, 2:) = ParameterVectorSet(l, :)


                            !< determine model function values for lth chi2 values
                            if (PlotIterationFlag) then                                     !< we need the model function values and the chi2 values only
                                                                                            !< if we want to fit the model function for each iteration
                                ModelFunctionFlag = .true.
                                chi2valDummy = 0.d0
                                m = 1
                                call ModelCalcChiFunction(NumberFreeParameterCopy, m, ParameterVectorSet(l, :), ModelFunctionFlag, &
                                                          TotalNumberComponents, TotalNumberOfMolecules, chi2valDummy)


                                !< determine chi2 values for each frequency point
                                Do NumberFile = 1, NumberExpFiles                           !< loop over exp. data files
                                    Do n = 1, NumberYColumns(NumberFile)                    !< loop over y-columns
                                        Do i = 1, lengthexpdata(NumberFile)                 !< loop over all line of current exp. data file
                                            sig2i = 1.d0
                                            if (expdatae(NumberFile, i, n) /= 0.d0) then    !< do the experimental datas include errors
                                                sig2i = 1.d0/(expdatae(NumberFile, i, n) * expdatae(NumberFile, i, n))
                                            endif


                                            !< determine chi**2 by calculating the difference (y_i^{obs) - y_i(fit))**2
                                            val = 0.d0
                                            if (abs(DetChi2) == 1) then
                                                val = (expdatay(NumberFile, i, n) - ModelFunction(1, NumberFile, n, i))**2


                                            !< determine chi**2 by calculating the difference (y_i^{obs)**2 - y_i(fit)**2)
                                            elseif (abs(DetChi2) == 2) then
                                                val = (expdatay(NumberFile, i, n)**2 - ModelFunction(1, NumberFile, n, i)**2)
                                            endif
                                            BestSitesChi2Values(k, NumberFile, i, n) = val * sig2i
                                            BestSitesModelValues(k, NumberFile, i, n) = ModelFunction(1, NumberFile, n, i)
                                        end Do
                                    end Do
                                end Do
                            endif
                            exit
                        endif
                    endif
                end Do
            end Do
            return
        end subroutine ModelCalcChiFunctionGeneral


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunction
        !>
        !> calculates the chi2 values for a given set of parameter vectors
        !>
        !>
        !> input variables:     NumberFreeParam:        number of free parameters
        !>                      NumParamVectors:        number of parameter vectors
        !>                      ParameterVectorSet:     parameter vector
        !>                      ModelFunctionFlag:      flag for indicating if model function values are stored or not
        !>                      NumComp:                total number of components
        !>
        !> output variables:    chi2ValuesVector:       chi2 value for parameter vector
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 15.08.2014
        !>
        subroutine ModelCalcChiFunction(NumberFreeParam, NumParamVectors, ParameterVectorSet, ModelFunctionFlag, NumComp, TotalNumMol, chi2ValuesVector)

            implicit none
            integer :: l, n, i, NumberFile                                                  !< loop variables
            integer :: k                                                                    !< counter variable
            integer :: num                                                                  !< working variable for model function value array
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            integer :: NumberFreeParam                                                      !< number of free parameters
            integer :: NumParamVectors                                                      !< number of parameter vectors
            integer :: NumComp                                                              !< total number of components
            integer :: errcode, LastErrcode                                                 !< error code for GPU
            integer :: TotalNumMol                                                          !< total number of molecules
            real :: t1, t2                                                                  !< used for debugging: time variables
            real*8, dimension(NumParamVectors) :: chi2ValuesVector                          !< chi2 value for parameter vector
            real*8, dimension(NumParamVectors, NumberFreeParam) :: ParameterVectorSet       !< parameter vector set
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function values are stored


            !< CUDA variables
            integer :: istat
            integer :: MaxNumThreads                                                        !< max. number of threads
            integer :: MaxGridThreads                                                       !< max. number of all threads
            integer, dimension(3) :: MaxGrid                                                !< maximum grid dimensions


            !< kernel variables
            real*8, dimension(NumParamVectors, NumberFreeParam), device :: ParameterVectorSet_d
            real*8, allocatable, dimension(:, :), device :: ModelFunctionListLocal_d
            real*8, dimension(NumParamVectors), device :: chi2ValuesVector_d


            type (cudaDeviceProp) :: prop                                                   !< CUDA variable:
            type(dim3) :: grid, tBlock                                                      !< CUDA variables:


            !< make settings for current GPU
            MaxNumThreads = prop%maxThreadsPerBlock                                         !< get max. number of threads allowed on the current GPU
            MaxNumThreads = 1024
            MaxGrid = prop%maxGridSize                                                      !< get max. grid dimensions allowed on the current GPU


            !< check, if number of parameter vectors is less than max. grid dimension
            MaxGridThreads = min(NumParamVectors, MaxGrid(0))


            !< define
            tBlock = dim3(MaxNumThreads, 1, 1)
            grid = dim3(ceiling(real(NumParamVectors)/tBlock%x), 1, 1)


            !< do we need the model function values
            if (ModelFunctionFlag) then
                num = TotalNumberDataPoints
            else
                num = 1
            endif
            if (allocated(ModelFunctionListLocal_d)) then
                deallocate(ModelFunctionListLocal_d, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine ModelCalcChiFunction!")')
                        write(ErrChannel, '(3x,"Can not deallocate variable ModelFunctionListLocal_d!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    return
                endif
            endif
            allocate(ModelFunctionListLocal_d(NumParamVectors, num), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelCalcChiFunction!")')
                    write(ErrChannel, '(3x,"Can not allocate variable ModelFunctionListLocal_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                return
            endif
            ModelFunctionListLocal_d = 0.d0
            chi2ValuesVector = 1.d99


            !< copy to GPU variables
            ModelFunctionFlag_d = ModelFunctionFlag
            ParameterVectorSet_d = ParameterVectorSet
            chi2ValuesVector_d = chi2ValuesVector


            !< check, if parameter are out of allowed range
            Do l = 1, NumParamVectors
                Do n = 1, NumberFreeParam
                    k = ConverterInfit(n)
                    if (ParameterVectorSet(l, n) < paramset(3, k) .or. paramset(4, k) < ParameterVectorSet(l, n)) then
                        ParameterVectorSet_d(l, 1) = 1.d99
                        exit
                    endif
                end Do
            end Do

            ! Debug:
            print '(" ")'
            print '("NumParamVectors = ", I5)', NumParamVectors
            print '("tBlock = ", 3(I5))', tBlock
            print '("grid   = ", 3(I5))', grid
            print '("ModelFunctionFlag = ", L1)', ModelFunctionFlag
            CALL CPU_TIME(t1)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            call ModelCalcSpectrumGPU<<< grid, tBlock >>>(ParameterVectorSet_d, chi2ValuesVector_d, ModelFunctionListLocal_d, NumParamVectors)
            CALL CPU_TIME(t2)
            istat = cudaDeviceSynchronize()
            chi2ValuesVector = chi2ValuesVector_d
            ParameterVectorSet = ParameterVectorSet_d

            ! Debug:
            print '("chi2ValuesVector(1:10) = ", 10(ES25.15))', chi2ValuesVector(1:min(10,NumParamVectors))
            print '("Time: t2 - t1 = ", F25.10, " sec")', t2 - t1


            !< check if an error occurred on the GPU
            errcode = cudaGetLastError()
            LastErrcode = cudaPeekAtLastError()
            if (errcode /= 0 .or. LastErrcode /= 0) then
                print '(" ")'
                print '(" ")'
                print '(1x, "Error on GPU:")'
                print '(3x, "An error occurred on the GPU!")'
                print '(" ")'
                print '(3x, "Error code = ", I5)', errcode
                print '(3x, "Last error code = ", I5)', LastErrcode
                print '(" ")'
                print '(3x, "Error message: = ", A)', cudaGetErrorString(errcode)
                print '(3x, "Last error message: = ", A)', cudaGetErrorString(LastErrcode)
                print '(" ")'
                print '(" ")'
            endif
stop

            !< copy model function values to ModelFunction array
            if (ModelFunctionFlag) then                                                     !< store model function values
                Do l = 1, NumParamVectors
                    k = 0
                    Do NumberFile = 1, NumberExpFiles                                       !< loop over exp. data files
                        Do n = 1, NumberYColumns(NumberFile)                                !< loop over y-columns
                            Do i = 1, lengthexpdata(NumberFile)                             !< loop over all line of current exp. data file
                                k = k + 1                                                   !< increase data point counter
                                ModelFunction(l, NumberFile, n, i) = ModelFunctionListLocal_d(l, k)
                            end Do

                            ! Debug:
                            ! print*,'minval(ModelFunction(l, NumberFile, n, :)) = ', minval(ModelFunction(l, NumberFile, n, :))
                            ! print*,'maxval(ModelFunction(l, NumberFile, n, :)) = ', maxval(ModelFunction(l, NumberFile, n, :))
                        end Do
                    end Do
                end Do
            endif

            ! Debug:
            !   print*,' '
            !   print*,'chi2ValuesVector(:) = ', chi2ValuesVector(:)
            !   stop 'Test LM-algorithm up to here!'

            return
        end subroutine ModelCalcChiFunction


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcSpectrumGPU
        !>
        !> calculates the myXCLASS spectrum for a given parameter vector (GPU specialized version)
        !>
        !>
        !> input variables:     ParameterVector:        parameter vector set
        !>                      NumParamVecs:           number of parameter vectors
        !>
        !> output variables:    chi2ValuesVector:       chi2 values for each parameter vector
        !>                      ModelFunctionListLocal: model function values for each parameter vector
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 31.07.2014
        !>
        attributes(global) subroutine ModelCalcSpectrumGPU(ParameterVector, chi2ValuesVector, ModelFunctionListLocal, NumParamVecs)

            implicit none
            real*8 :: ParameterVector(:, :)                                                 !< (input) parameter vector set
            real*8 :: chi2ValuesVector(:)                                                   !< (output) chi2 values for each parameter vector
            real*8 :: ModelFunctionListLocal(:, :)                                          !< (output) model function values for each parameter vector
            integer, value :: NumParamVecs                                                  !< (input) total number of parameter vectors


            !< myXCLASS variables
            integer :: fitnum, i, j, k, l, ae, c, t                                         !< loop variables
            integer :: i1, i2, ff1, ff2                                                     !< working variables for copying
            integer :: IndexComp                                                            !< overall component index
            integer :: ThreadNumber                                                         !< current thread number
            integer :: varIndex                                                             !< myXCLASS variable index
            integer :: ErrChannelIndexLocal                                                 !< working varibale for error handling
            integer :: ErrChannelLocal                                                      !< working varibale for error handling
            integer :: MoleculeIndex                                                        !< working variable: index for molecule
            integer :: FirstMolecularDataIndex                                              !< working variable: first index for molecular data table
            integer :: LastMolecularDataIndex                                               !< working variable: last index for molecular data table
            integer :: FirstIndex                                                           !< working variable: index for first freq. data point
            integer :: LastIndex                                                            !< working variable: index for last freq. data point
            integer :: FreqIndex                                                            !< working variable: index for frequency point
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation

            real*8 :: chi2Value                                                             !< wokring variable: resulting chi2 value
            real*8 :: eta_source                                                            !< working variable: beam filling factor
            real*8 :: logt, x0, x1, f0, f1                                                  !< working variables for interpolation
            real*8 :: freq                                                                  !< working variable: frequency for index 'FreqIndex'
            real*8 :: freq_t                                                                !< working variable: frequency as temperature
            real*8 :: beam_size                                                             !< working variable: beam size (eqn. 2 - 3)
            real*8 :: j_tk, j_cb, j_back                                                    !< working variables: Rayleigh temperatures
            real*8 :: tau_d                                                                 !< working variable: tau dust
            real*8 :: tau_t                                                                 !< working variable: tau for each transition
            real*8 :: tau_l                                                                 !< working variable: sum over all taus
            real*8 :: vLSR                                                                  !< working variable: v_LSR
            real*8 :: sigma                                                                 !< working variable: sigma
            real*8 :: Temperature                                                           !< working variable: temperature
            real*8 :: LocalIntensity                                                        !< working variable: calc. intensity at current data point


            !< reset output variable
            ModelFunctionListLocal = 0.d0


            !<============================================================================================================================================
            !< variables being used from the global module definition
            !<      Firsti                                  shared      integer             working variable for interpolation
            !<      Lasti                                   shared      integer             working variable for interpolation
            !<      stepi                                   shared      integer             working variable for interpolation
            !<      TotalNumberComponents                   shared      integer             counter for total number of components of all molecules
            !<      TotalNumberOfFrequencyRanges            shared      integer             total number of frequency ranges
            !<      TotalNumberOfMolecules                  shared      integer             total number of molecules including isotopologues
            !<      NumberOfTemperatures                    shared      integer             number of temperatures
            !<      NumberComponentsPerMolecule             shared      integer             number of components per molecule
            !<                                                          dimension: (0:TotalNumberOfMolecules)
            !<      IsoNfitConversionTable                  shared      integer             iso ratios conversion table for free paramter index
            !<                                                          dimension: (TotalNumberOfMolecules, TotalNumberOfMolecules)
            !<      SpectrumIndexForFreqRange               shared      integer             store spectrum index for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      DataPointIndexFreqRange                 shared      integer                 index of data point of first and last freq. in freq. range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges, 2)
            !<      MolecularDataIndices                    shared      integer             start and end index for each molecule and frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2)
            !<      ConversionTableMAGIXmyXCLASSParam       shared      integer             conversion table between myXCLASS and MAGIX parameter
            !<                                                          dimension: (NumberFreeParameterCopy, 2)
            !<      CompMoleculeIndex                       private     integer             molecule index for each component
            !<                                                          dimension: (TotalNumberComponents)
            !<      debug_freq                              shared      real*8              frequency for debugging
            !<      ckms                                    shared      real*8              speed of light in km/s
            !<      pi                                      shared      real*8              pi
            !<      TempLow                                 shared      real*8              working variable for extrapolation of part. func.
            !<      TempHigh                                shared      real*8              working variable for extrapolation of part. func.
            !<      TempPartFunc                            shared      real*8              temperatures for partition function
            !<                                                          dimension: (NumberOfTemperatures)
            !<      BackgroundTemperatureRange              shared      real*8              T_Back for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      TemperatureSlopeRange                   shared      real*8              T_Slope for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      HydrogenColumnDensityRange              shared      real*8              nH for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      DustBetaRange                           shared      real*8              beta for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      KappaRange                              shared      real*8              kappa for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      TelescopeSize                           shared      real*8              size of telescope for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      StartFrequency                          shared      real*8              first frequency for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      ObservationalDataList                   shared      real*8              list containing all observational data
            !<                                                          dimension: (TotalNumberDataPoints, 3)
            !<      MolecularData                           shared      real*8              array containing the molecular data for all molecules and
            !<                                                          dimension: (TotalNumberOfTransitions, 4)
            !<      MoleculeNames                           shared      character(len=40)   names of all molecules (including isotopologues)
            !<                                                          dimension: (MaxNumberOfMolecules)
            !<      IsoRatioConversionTable                 shared      real*8              table with iso ratios between iso master and molecule
            !<                                                          dimension: (TotalNumberOfMolecules, TotalNumberOfMolecules)
            !<      lgQ                                     shared      real*8              lgQ entries of table PartitionFunction
            !<                                                          dimension: (NumberOfTemperatures, NumberMoleculePartFunc)
            !<      myXCLASSParameter                       shared      real*8              array containing all molfit parameters for each component
            !<                                                          dimension: (11, TotalNumberComponents)
            !<      nHFlag                                  shared      logical             flag for global setting of nH, kappa and beta
            !<      tbFlag                                  shared      logical             flag for global setting T_Back and T_Slope
            !<      IsoFlag                                 shared      logical             flag indicating use of isotopologues
            !<============================================================================================================================================


            !< get current thread number and check, if thread ID is within allowed range
            ThreadNumber = blockDim%x * (blockIdx%x - 1) + threadIdx%x
            if (ThreadNumber > NumParamVecs) return


            !< check, if parameters are within allowed range
            if (ParameterVector(ThreadNumber, 1) == 1.d99) then
                chi2ValuesVector(ThreadNumber) = 1.d99
                return
            endif


            !<============================================================================================================================================
            !< update myXCLASS parameter vector with new parameter values
            !< CopymyXCLASSParameter(1, :)      source size
            !< CopymyXCLASSParameter(2, :)      T_rot
            !< CopymyXCLASSParameter(3, :)      N_tot
            !< CopymyXCLASSParameter(4, :)      V_width
            !< CopymyXCLASSParameter(5, :)      V_off
            !< CopymyXCLASSParameter(6, :)      T_Back
            !< CopymyXCLASSParameter(7, :)      T_Slope
            !< CopymyXCLASSParameter(8, :)      nH_column
            !< CopymyXCLASSParameter(9, :)      beta
            !< CopymyXCLASSParameter(10, :)     kappa
            !< CopymyXCLASSParameter(11, :)     core / foreground flag
            Do fitnum = 1, NumberFreeParameter_d                                            !< loop over all free parameters


                !< update molfit file parameters
                IndexComp = ConversionTableMAGIXmyXCLASSParam_d(fitnum, 1)                  !< get current overall component index
                varIndex = ConversionTableMAGIXmyXCLASSParam_d(fitnum, 2)                   !< get current myXCLASS variable index
                if (IndexComp > 0 .and. varIndex > 0 .and. varIndex < 11) then              !< check, if current free param. corresponds to molfit param.

                    ! Debug:
                    ! print*,'varIndex, IndexComp = ', varIndex, IndexComp
                    ! print*,'CopymyXCLASSParameter(ThreadNumber, :, IndexComp) = ', CopymyXCLASSParameter(ThreadNumber, varIndex, IndexComp)
                    ! print*,'ParameterVector(ThreadNumber, fitnum) = ', ParameterVector(ThreadNumber fitnum)


                    CopymyXCLASSParameter(ThreadNumber, varIndex, IndexComp) = ParameterVector(ThreadNumber, fitnum)


                    !< convert log10 value back to linear value check for overflow
                    if (varIndex == 3) then
                        CopymyXCLASSParameter(ThreadNumber, 3, IndexComp) = 10.d0**min(300.d0, CopymyXCLASSParameter(ThreadNumber, 3, IndexComp))
                    endif


                !< update iso ratio table
                elseif (IsoFlag_d) then
                    Do j = 1, TotalNumberOfMolecules_d
                        Do k = 1, TotalNumberOfMolecules_d
                            if (int(IsoNfitConversionTable_d(j, k)) == fitnum) then
                                CopyIsoRatioConversionTable(ThreadNumber, j, k) = ParameterVector(ThreadNumber, fitnum)
                            elseif (int(IsoNfitConversionTable_d(j, k)) == -fitnum) then
                                CopyIsoRatioConversionTable(ThreadNumber, j, k) = 1.d0 / ParameterVector(ThreadNumber, fitnum)
                            endif
                        end Do
                    end Do
                endif
            end Do


            !<============================================================================================================================================
            !< update parameters for isotopologues
            if (IsoFlag_d) then
                Do j = 1, TotalNumberOfMolecules_d
                    i1 = sum(NumberComponentsPerMolecule_d(:(j - 1))) + 1
                    i2 = i1 + NumberComponentsPerMolecule_d(j) - 1

                    ! Debug:
                    !    print*,"#####################################################################"
                    !    print*,"j, MoleculeNames_d(j) = ", j, MoleculeNames_d(j)
                    !    print*,"NumberComponentsPerMolecule_d(j) = ", j, NumberComponentsPerMolecule_d(j)
                    !    print*,"CopyIsoRatioConversionTable(ThreadNumber, j, :) = ", CopyIsoRatioConversionTable(ThreadNumber, j, :)
                    !    Do k = i1, i2
                    !        print*,"CopymyXCLASSParameter(ThreadNumber, :, k) = ", k, CopymyXCLASSParameter(ThreadNumber, :, k)
                    !    end Do
                    !    print*," "
                    !    print*," "


                    Do k = 1, TotalNumberOfMolecules_d

                        ! Debug:
                        !    print*,"k, MoleculeNames_d(j) = ", k, MoleculeNames_d(k)
                        !    print*,"CopyIsoRatioConversionTable(ThreadNumber, j, k) = ", CopyIsoRatioConversionTable(ThreadNumber, j, k)
                        !    print*,"NumberComponentsPerMolecule_d(k) = ", NumberComponentsPerMolecule_d(k)
                        !    print*," "


                        if (CopyIsoRatioConversionTable(ThreadNumber, j, k) /= 0.d0 .and. NumberComponentsPerMolecule_d(k) /= 0) then

                            ff1 = sum(NumberComponentsPerMolecule_d(:(k - 1))) + 1
                            ff2 = ff1 + NumberComponentsPerMolecule_d(k) - 1

                            ! Debug:
                            ! print*,"i1, i2 = ",i1, i2
                            ! print*,"ff1, ff2 = ",ff1, ff2


                            !< copy parameters for current isotopologues from isomaster and scale column densities with iso ratio factor
                            !CopymyXCLASSParameter(ThreadNumber, :, ff1:ff2) = CopymyXCLASSParameter(ThreadNumber, :, i1:i2)
                            CopyCompMoleculeIndex(ThreadNumber, ff1:ff2) = k
                            Do i = ff1, ff2
                                CopymyXCLASSParameter(ThreadNumber, :, i) = CopymyXCLASSParameter(ThreadNumber, :, i - ff1 + i1)
                                CopymyXCLASSParameter(ThreadNumber, 3, i) = CopyIsoRatioConversionTable(ThreadNumber, j, k) &
                                                                            * CopymyXCLASSParameter(ThreadNumber, 3, i)
                            end Do
                        endif
                    end Do
                end Do
            endif

            ! Debug:
            !    Do i = 1, TotalNumberComponents_d
            !        print*,"i = ", i
            !        j = CopyCompMoleculeIndex(ThreadNumber, i)
            !        print*,"MoleculeNames_d = ", trim(adjustl(MoleculeNames_d(j)))
            !        print*,"CopymyXCLASSParameter(ThreadNumber, :, i) = ", CopymyXCLASSParameter(ThreadNumber, :, i)
            !    end Do


            !<============================================================================================================================================
            !< determine partiton functions for all rotation temperatures
            PartFunc = 0.d0
            Do l = 1, TotalNumberComponents_d                                               !< loop over all components
                Temperature = CopymyXCLASSParameter(ThreadNumber, 2, l)                     !< get excitation temperatures
                if (Temperature <= 0.d0) then
                    Temperature = 10.d0**TempLow_d
                endif
                logt = dlog10(Temperature)                                              !< calculate the logarithm of the excitation temperature tk
                MoleculeIndex = CopyCompMoleculeIndex(ThreadNumber, l)                                !< get molecule index


                !< is logt out of range?
                if (logt < TempLow_d .or. logt > TempHigh_d) then


                    !< extrapolate if neccessary ---------------------------------------- is not valid for all cases !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                    x0 = 0.d0
                    x1 = 0.d0
                    f0 = 0.d0
                    f1 = 0.d0
                    if (logt < TempPartFunc_d(NumberOfTemperatures_d)) then
                        x0 = TempPartFunc_d((NumberOfTemperatures_d - 1))
                        x1 = TempPartFunc_d(NumberOfTemperatures_d)
                        f0 = lgQ_d((NumberOfTemperatures_d - 1), MoleculeIndex)
                        f1 = lgQ_d(NumberOfTemperatures_d, MoleculeIndex)

                    elseif (logt > TempPartFunc_d(1)) then
                        x0 = TempPartFunc_d(2)
                        x1 = TempPartFunc_d(1)
                        f0 = lgQ_d(2, MoleculeIndex)
                        f1 = lgQ_d(1, MoleculeIndex)

                    endif
                    PartFunc(ThreadNumber, l) = f0 + (f1 - f0)/(x1 - x0) * (logt - x0)
                    PartFunc(ThreadNumber, l) = 10.d0**PartFunc(ThreadNumber, l)

                    ! Debug:
                    ! print*,'logt > TempPartFunc_d(1) .or. logt < TempPartFunc_d(NumberOfTemperatures_d):'
                    ! print*,'x0, f0 = ', x0, f0
                    ! print*,'x1, f1 = ', x1, f1
                    ! print*,'logt = ',logt
                    ! print*,'PartFunc(ThreadNumber, l) = ', PartFunc(ThreadNumber, l)
                    ! print*,'log10(PartFunc(ThreadNumber, l)) = ', dlog10(PartFunc(ThreadNumber, l))
                else


                    !< start linear interpolation
                    Do i = Firsti_d, Lasti_d, stepi_d
                        if (TempPartFunc_d(i) == logt) then
                            PartFunc(ThreadNumber, l) = lgQ_d(i, MoleculeIndex)
                            exit


                        !<----------------------------------------------------------------------------------------------------------------------------
                        !< linear interpolation: f(x) = f_0 + \frac{f_1 - f_0}{x_1 - x_0} (x - x_0)
                        elseif (TempPartFunc_d(i) < logt) then
                            if (i == NumberOfTemperatures_d) then
                                x0 = TempPartFunc_d(i - 1)
                                x1 = TempPartFunc_d(i)
                                f0 = lgQ_d(i - 1, MoleculeIndex)
                                f1 = lgQ_d(i, MoleculeIndex)
                            else
                                x0 = TempPartFunc_d(i)
                                x1 = TempPartFunc_d(i + 1)
                                f0 = lgQ_d(i, MoleculeIndex)
                                f1 = lgQ_d(i + 1, MoleculeIndex)
                            endif
                            PartFunc(ThreadNumber, l) = f0 + (f1 - f0)/(x1 - x0) * (logt - x0)

                            ! Debug:
                            ! print*,'x0, f0 = ', x0, f0
                            ! print*,'x1, f1 = ', x1, f1
                            ! print*,'logt = ',logt
                            ! print*,'PartFunc(ThreadNumber, l) = ', PartFunc(ThreadNumber, l)
                            ! print*,'10.d0**PartFunc(ThreadNumber, l) = ', 10.d0**PartFunc(ThreadNumber, l)
                            exit
                        endif
                    end Do
                    PartFunc(ThreadNumber, l) = 10.d0**PartFunc(ThreadNumber, l)

                    ! Debug:
                    !    print*,' '
                    !    print*,' '
                    !    print*,'l = ', l
                    !    print*,"MoleculeNames_d = ", trim(adjustl(MoleculeNames_d(MoleculeIndex)))
                    !    print*,'Temperature = ', Temperature
                    !    print*,'PartFunc(ThreadNumber, l) = ', PartFunc(ThreadNumber, l)
                    !    print*,'logt = ', logt
                    !    Do i = 1, NumberOfTemperatures_d
                    !        print*,'i = ', i
                    !        print*,'10.d0**lgQ_d(i, MoleculeIndex) = ', 10.d0**lgQ_d(i, MoleculeIndex), "           ", PartFunc(ThreadNumber, l)
                    !        print*,'10.d0**TempPartFunc_d(i) = ', 10.d0**TempPartFunc_d(i), "           ",Temperature
                    !    end Do
                    !    print*,'log10(PartFunc(ThreadNumber, l)) = ', dlog10(PartFunc(ThreadNumber, l))
                endif
            end Do


            !<============================================================================================================================================
            !< start loop over molecules, frequency ranges, components, transitions
            chi2Value = 0.d0
            Do l = 1, TotalNumberOfFrequencyRanges_d                                        !< loop over all frequency ranges
                i = SpectrumIndexForFreqRange_d(l)                                          !< get spectrum index
                FirstIndex = DataPointIndexFreqRange_d(l, 1)                                !< get index of first freq. point in 'ObservationalDataList'
                LastIndex = DataPointIndexFreqRange_d(l, 2)                                 !< get index of last freq. point in 'ObservationalDataList'

                ! Debug:
                ! print*,"l = ", l
                ! print*,"StartFrequency_d(l) = ", StartFrequency_d(l)
                ! print*,"EndFrequency(l)   = ", EndFrequency(l)
                ! print*,"StepFrequency(l)  = ", StepFrequency(l)
                ! print*,"FirstIndex = ", FirstIndex
                ! print*,"LastIndex  = ", LastIndex


                !< update
                if (tbFlag_d) then
                    CopymyXCLASSParameter(ThreadNumber, 6, :) = BackgroundTemperatureRange_d(l)
                    CopymyXCLASSParameter(ThreadNumber, 7, :) = TemperatureSlopeRange_d(l)
                endif
                if (nHFlag_d) then
                    CopymyXCLASSParameter(ThreadNumber, 8, :) = HydrogenColumnDensityRange_d(l)
                    CopymyXCLASSParameter(ThreadNumber, 9, :) = DustBetaRange_d(l)
                    CopymyXCLASSParameter(ThreadNumber, 10, :) = KappaRange_d(l) * 2.d0 * 1.66d-24 / 100.d0 !< kappa_1300 * M(H2)/M_d_M_g
                endif
                Do FreqIndex = FirstIndex, LastIndex                                        !< loop over all frequency points in the current freq. range
                    freq = ObservationalDataList_d(FreqIndex, 1)                            !< get frequency

                    ! Debug:
                    ! print*,"freq = ", freq


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< determine beam size (eq. 3):
                    !< telescope beam FWHM size is related to the diameter of the telescope by the diffraction limit:
                    !<
                    !<     \theta_t = 1.22 \frac{c}{\lambda D}
                    !<
                    !< where D describes the diameter of the telescope and c the speed of light.
                    !<
                    beam_size = 1.22d-3 * ckms_d * 180.d0 * 3600.d0 / pi_d / freq / TelescopeSize_d(l)


                    LocalIntensity = 0.d0                                                   !< reset local intensity
                    Do ae = 1, 2                                                            !< loop over core (ae=1) and foreground (ae=2) components
                        Do c = 1, TotalNumberComponents_d                                   !< loop over all components

                            !< CopymyXCLASSParameter(1, c)      source size
                            !< CopymyXCLASSParameter(2, c)      T_rot
                            !< CopymyXCLASSParameter(3, c)      N_tot
                            !< CopymyXCLASSParameter(4, c)      V_width
                            !< CopymyXCLASSParameter(5, c)      V_off
                            !< CopymyXCLASSParameter(6, c)      T_Back
                            !< CopymyXCLASSParameter(7, c)      T_Slope
                            !< CopymyXCLASSParameter(8, c)      nH_column
                            !< CopymyXCLASSParameter(9, c)      beta
                            !< CopymyXCLASSParameter(10, c)     kappa
                            !< CopymyXCLASSParameter(11, c)     core / foreground flag

                            ! Debug:
                            !    if (dabs(freq - debug_freq_d) < 1.d-6) then
                            !        print*,"####################################################################################"
                            !        print*,"ae = ", ae
                            !        print*,"c, TotalNumberComponents_d = ", c, TotalNumberComponents_d
                            !        print*,"CopymyXCLASSParameter(ThreadNumber, :, c) = ", CopymyXCLASSParameter(ThreadNumber, :, c)
                            !    endif


                            if (int(CopymyXCLASSParameter(ThreadNumber, 11, c)) == ae) then !< check, if current component correspond to ae index
                                MoleculeIndex = CopyCompMoleculeIndex(ThreadNumber, c)      !< get molecule index


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< determine beam filling (dilution) factor (eq. 2):
                                !< \eta(\theta_{m,c}) of molecule m and component c for a source with a Gaussian brightness profile, and a Gaussian beam
                                !< is given by
                                !<
                                !<     \eta(\theta_{m,c}) = \frac{\theta_{m,c}^2}{\theta_{m,c}^2 + \theta_t^2}
                                !<
                                !< where \theta_{m,c} and \theta_t represents the source and telescope beam FWHM sizes, respectively.
                                !<
                                eta_source = CopymyXCLASSParameter(ThreadNumber, 1, c)**2 / (beam_size**2 + CopymyXCLASSParameter(ThreadNumber, 1, c)**2)

                                ! Debug:
                                !    if (dabs(freq - debug_freq_d) < 1.d-6) then
                                !        print*,"MoleculeIndex = ", MoleculeIndex
                                !        print*,"eta_source = ", eta_source
                                !    endif


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< determine Rayleight-Jeans temperatures
                                freq_t = freq * 4.8d-5                                      !< convert frequency in temperature (Kelvin)
                                j_tk = freq_t / (dexp(freq_t / CopymyXCLASSParameter(ThreadNumber, 2, c)) - 1.d0)
                                j_cb = freq_t / (dexp(freq_t / 2.7d0) - 1.d0)

                                ! Debug:
                                !    if (dabs(freq - debug_freq_d) < 1.d-6) then
                                !        print*,"freq_t = ", freq_t
                                !        print*,"j_tk = ", j_tk
                                !        print*,"j_cb = ", j_cb
                                !    endif


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< background temperature and slope are given for each component and molecule
                                !< the background temperature J(T_bg; \nu) is described by
                                !<
                                !<     J (T_{\rm bg}, \nu ) = | T_{\rm bg} | * (\frac{\nu}{\nu_{\rm min}} )^{T_{\rm slope}},
                                !<
                                !< and defines the continuum contribution for each frequency range, individually. Here, \nu_{\rm min} indicates the
                                !< lowest frequency of a given frequency range. T_{\rm bg} and T_{\rm slope}, describe the background continuum
                                !< temperature and the temperature slope, respectively.
                                !<
                                !< CopymyXCLASSParameter(6, c)      T_Back
                                !< CopymyXCLASSParameter(7, c)      T_Slope
                                !<
                                j_back = dabs(CopymyXCLASSParameter(ThreadNumber, 6, c)) &
                                         * (freq / StartFrequency_d(l))**CopymyXCLASSParameter(ThreadNumber, 7, c)

                                ! Debug:
                                !    if (dabs(freq - debug_freq_d) < 1.d-6) print*,"j_back = ", j_back


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< calculate the dust contribution (eq. 4):
                                !<
                                !<     \tau_d = N_H * \kappa_{\rm 1.3 \, mm} * ( \frac{\nu}{\nu_{\rm 1.3 \, mm}} )^\beta * m_{H_2}
                                !<                  * \frac{1}{\zeta_{\rm gas-dust}},
                                !<
                                !< where the hydrogen column density $N_H$, the dust mass opacity $\kappa_{\rm 1.3 \, mm}$, and the exponent $\beta$.
                                !< In addition, $\nu_{\rm 1.3 \, mm}$ = 230~GHz indicates the frequency for a wavelength of 1.3~mm, $m_{H_2}$ describes
                                !< the mass of a hydrogen molecule, and $\zeta_{\rm gas-dust}$ represents the gas to dust mass ratio,
                                !< i.e.\ $1 / \zeta_{\rm gas-dust}$ describes the dust to gas ratio and is set to (1/100).
                                !<
                                !< CopymyXCLASSParameter( 8, c)      nH_column
                                !< CopymyXCLASSParameter( 9, c)      beta
                                !< CopymyXCLASSParameter(10, c)     kappa   (contains factor (m_{H_2} * \frac{1}{\zeta_{\rm gas-dust}}) as well)
                                !<
                                tau_d = CopymyXCLASSParameter(ThreadNumber, 10, c) * (freq / 2.3d5)**CopymyXCLASSParameter(ThreadNumber, 9, c) &
                                        * CopymyXCLASSParameter(ThreadNumber, 8, c)

                                ! Debug:
                                !    if (dabs(freq - debug_freq_d) < 1.d-6) then
                                !        print*,"tau_d = ", tau_d
                                !        print*,"l = ", l
                                !        print*,"MoleculeIndex = ", MoleculeIndex
                                !        print*,"MolecularDataIndices_d(l, :, 1) = ", MolecularDataIndices_d(l, :, 1)
                                !        print*,"MolecularDataIndices_d(l, :, 2) = ", MolecularDataIndices_d(l, :, 2)
                                !    endif


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< determine optical depth $\tau(\nu)^{m,c}$ for each molecule m and component c
                                !<
                                !<     \tau(\nu)^{m,c} = \sum_t \frac{c^3}{8 \pi \nu^3} \, A_{ul} \, N_{\rm tot}^{m,c}
                                !<                              \frac{g_u  e^{-E_l/k_B T_{\rm ex}^{m,c}}}{Q \left(m, T_{\rm ex}^{m,c} \right)}
                                !<                              \left(1 - e^{-h \, \nu /k_B \, T_{\rm ex}^{m,c}} \right)
                                !<                              \times \phi(\nu)^{m,c},
                                !<
                                !< where the sum with index $t$ runs over all spectral line transitions of molecule $m$ within the given frequency
                                !< range. Additionally, the Einstein $A_{ul}$ coefficient\footnote{The indices $u$ and $l$ represent upper and lower
                                !< state, respectively.}, the energy of the lower state $E_l$, the upper state degeneracy $g_u$, and the
                                !< partition function $Q \left(m, T_{\rm ex}^{m,c} \right)$ of molecule $m$ are taken from the embedded
                                !< SQLite3 database, described in  Sect.~\ref{sec:db}. In addition, the values of the excitation temperatures
                                !< $T_{\rm ex}^{m,c}$ and the column densities $N_{\rm tot}^{m,c}$ for the different components and molecules are taken
                                !< from the user defined molfit file. Furthermore, the profile of a spectral line $t$ is given by a Gaussian line profile
                                !<
                                !<     \phi(\nu)^{m,c} = \frac{1}{\sqrt{2 \pi} \, \sigma}
                                !<                       \cdot e^{-\frac{\left(\nu - \left( \nu_t + \nu_{\rm LSR}^{m,c} \right) \right)^2} {2 \sigma^2}},
                                !<
                                !< which is normalized, i.e.\ $\int_0^{\infty} \phi(\nu) \, d\nu$ = 1. The source velocity $\nu_{\rm LSR}^{m,c}$ for
                                !< each component $c$ of a molecule $m$ is related to the user defined velocity offset
                                !< $\left(v_{\rm offset}^{m,c}\right)$ taken from the aforementioned molfit file, by the following expression
                                !<
                                !<     \nu_{\rm LSR}^{m,c} = -v_{\rm offset}^{m,c} \cdot \nu_t,
                                !<
                                !< where $\nu_t$ indicates the frequency of transition $t$ taken from the SQLite3 database mentioned above. Additionally,
                                !< the variance $\sigma$ of the profile is defined by the velocity width $\left(v_{\rm width}^{m,c}\right)$ described in
                                !< the molfit file for each component $c$ of a molecule $m$:
                                !<
                                !<     \sigma = \frac{v_{\rm width}^{m,c} \cdot \left(\nu_t + \nu_{\rm LSR}^{m,c} \right)}{2 \, \sqrt{2 \, \ln 2}}.
                                !<
                                tau_l = 0.d0
                                FirstMolecularDataIndex = MolecularDataIndices_d(l, MoleculeIndex, 1) !< get first mol. data index of current freq. range
                                LastMolecularDataIndex = MolecularDataIndices_d(l, MoleculeIndex, 2)  !< get last mol. data index of current freq. range
                                Do t = FirstMolecularDataIndex, LastMolecularDataIndex - 0  !< loop over all transitions of current molecule and
                                    !< MolecularData_d(t, 1) = lFreq
                                    !< MolecularData_d(t, 2) = EinsteinA
                                    !< MolecularData_d(t, 3) = lElt
                                    !< MolecularData_d(t, 4) = UpperStateDegeneracy

                                    ! Debug:
                                    !    if (dabs(freq - debug_freq_d) < 1.d-6) then
                                    !        print*,"t = ", t
                                    !        print*,"MolecularData_d(t, :) = ", MolecularData_d(t, :)
                                    !        print*,"FirstMolecularDataIndex, LastMolecularDataIndex = ", FirstMolecularDataIndex, LastMolecularDataIndex
                                    !    endif


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< determine part of eq. (5):
                                    !<
                                    !<      \frac{c^3}{8 \pi \nu^3} \, A_{ul} \, N_{\rm tot}^{m,c}
                                    !<                              \frac{g_u  e^{-E_l/k_B T_{\rm ex}^{m,c}}}{Q \left(m, T_{\rm ex}^{m,c} \right)}
                                    !<                              \left(1 - e^{-h \, \nu /k_B \, T_{\rm ex}^{m,c}} \right)
                                    !<
                                    !< MolecularData_d(t, 2) = EinsteinA
                                    !< MolecularData_d(t, 4) = UpperStateDegeneracy
                                    !< CopymyXCLASSParameter(ThreadNumber, 2, c) = T_rot
                                    !< CopymyXCLASSParameter(ThreadNumber, 3, c) = N_tot
                                    !<
                                    if (PartFunc(ThreadNumber, c) == 0.d0) then
                                        tau_t = 0.d0
                                    else
                                        tau_t = (((ckms_d * 1.d3)**2 / (8.d0 * pi_d * freq**3)) &
                                                 * MolecularData_d(t, 2) * CopymyXCLASSParameter(ThreadNumber, 3, c) &
                                                 * MolecularData_d(t, 4) * 1.d-14 * ckms_d &
                                                 * (dexp(-MolecularData_d(t, 3)/CopymyXCLASSParameter(ThreadNumber, 2, c)) &
                                                 - dexp(-(MolecularData_d(t, 3) &
                                                 + (freq/20836.74d0))/ CopymyXCLASSParameter(ThreadNumber, 2, c)))) / PartFunc(ThreadNumber, c)
                                    endif

                                    ! Debug:
                                    !    if (dabs(freq - debug_freq_d) < 1.d-6) then
                                    !        print*,'e_low   = ', MolecularData_d(t, 3)
                                    !        print*,'e_up    = ', (MolecularData_d(t, 3) + (freq/20836.74d0))
                                    !        print*,'>>tau_t = ', tau_t
                                    !        print*,'--> MolecularData_d(t, 2) = ', MolecularData_d(t, 2)
                                    !        print*,'--> CopymyXCLASSParameter(ThreadNumber, 3, c) = ', CopymyXCLASSParameter(ThreadNumber, 3, c)
                                    !        print*,'--> MolecularData_d(t, 4) = ', MolecularData_d(t, 4)
                                    !        print*,'--> MolecularData_d(t, 3) = ', MolecularData_d(t, 3)
                                    !        print*,'--> CopymyXCLASSParameter(ThreadNumber, 2, c) = ', CopymyXCLASSParameter(ThreadNumber, 2, c)
                                    !    endif


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< determine \nu_{\rm LSR}^{m,c}:
                                    !<
                                    !<     \nu_{\rm LSR}^{m,c} = -v_{\rm offset}^{m,c} \cdot \nu_t,
                                    !<
                                    !< CopymyXCLASSParameter(ThreadNumber, 5, c) = V_off
                                    !< MolecularData_d(t, 1): lFreq
                                    !<
                                    vLSR = -CopymyXCLASSParameter(ThreadNumber, 5, c) / ckms_d * MolecularData_d(t, 1)

                                    ! Debug:
                                    !    if (dabs(freq - debug_freq_d) < 1.d-6) print*,'vLSR = ', vLSR


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< determine \sigma:
                                    !<
                                    !<     \sigma = \frac{v_{\rm width}^{m,c} \cdot \left(\nu_t + \nu_{\rm LSR}^{m,c} \right)}{2 \, \sqrt{2 \, \ln 2}}.
                                    !<
                                    !< CopymyXCLASSParameter(ThreadNumber, 4, c) = V_width
                                    !< MolecularData_d(t, 1): lFreq
                                    !<
                                    sigma = (CopymyXCLASSParameter(ThreadNumber, 4, c) / ckms_d &
                                            * (MolecularData_d(t, 1) + vLSR)) / (2.d0 * dsqrt(2.d0 * dlog(2.d0)))

                                    ! Debug:
                                    !    if (dabs(freq - debug_freq_d) < 1.d-6) print*,'sigma = ', sigma


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< determine \phi(\nu)^{m,c}:
                                    !<
                                    !<     \phi(\nu)^{m,c} = \frac{1}{\sqrt{2 \pi} \, \sigma}
                                    !<                       * e^{-\frac{\left(\nu - \left( \nu_t + \nu_{\rm LSR}^{m,c} \right) \right)^2} {2 \sigma^2}},
                                    !<
                                    !< MolecularData_d(t, 1): lFreq
                                    !<
                                    tau_t = tau_t * 1.d0 / (dsqrt(2.d0 * pi_d) * sigma) &
                                                  * dexp(-1.d0 * ((MolecularData_d(t, 1) - freq + vLSR)**2) / (2.d0 * (sigma)**2)) * freq / ckms_d


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< add tau for current transition to sum of taus
                                    tau_l = tau_l + tau_t
                                end Do                                                      !< t: loop over all transitions

                                ! Debug:
                                !    if (dabs(freq - debug_freq_d) < 1.d-6) print*,'tau_l = ', tau_l


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< calculate intensity using eqn. (1) and (10):
                                !< The model function considers finite source size and dust attenuation and is given by:
                                !<
                                !<     T_{\rm mb}(\nu) = \sum_m \sum_c \eta \left(\theta_{m,c} \right) \left[ J \left(T_{\rm ex}^{m,c}, \nu \right)
                                !<                                                                          - J \left(T_{\rm bg}, \nu \right)
                                !<                                                                          - J \left(T_{\rm cbg}, \nu \right) \right]
                                !<                                      \times \left(1 - e^{-\left( \tau(\nu)^{m,c} + \tau_d\right)} \right)
                                !<
                                !< The sums go over the indices $m$ for molecule, and $c$ for component, respectively. Here, components for each
                                !< molecule can be identified as spatially distinct sources such as clumps, hot dense cores, colder envelopes or
                                !< outflows. They can usually be distinguished by different radial velocities, and do not interact with each other
                                !< radiatively but superimpose in the model. So, the myXCLASS program is able to model a spectrum with an arbitrary
                                !< number of molecules and components, simultaneously.
                                !<
                                !<     T_{\rm mb}(\nu) = T_{\rm mb}(\nu) - \left[ \eta \left(\theta_{m,c} \right)
                                !<                                         \cdot J \left(T_{\rm ex}^{m,c}, \nu \right)
                                !<                                         \cdot \left(1 - e^{-\tau_d} \right) \right]
                                !<
                                !< allows the user to define the continuum contribution for each frequency range, individually. Here,
                                !< $\nu_{\rm min}$ indicates the lowest frequency of a given frequency range. $T_{\rm bg}$ and $T_{\rm slope}$,
                                !< describe the background continuum temperature and the temperature slope, respectively. Please note, if the user
                                !< defines a background temperature $T_{\rm bg} \leq 0$, the dust continuum offset for each component $c$ of a
                                !< molecule $m$ is subtracted from the model function.
                                !<
                                if (CopymyXCLASSParameter(ThreadNumber, 11, c) == 1.d0) then    !< determine intensity for emission
                                    LocalIntensity = LocalIntensity + eta_source * (j_tk - j_back - j_cb) * (1.d0 - dexp(-(tau_l + tau_d)))

                                    ! Debug:
                                    !    if (dabs(freq - debug_freq_d) < 1.d-6) then
                                    !        print*,'>>>>>>>>>>>>>>> LocalIntensity = ', LocalIntensity
                                    !    endif


                                    if (CopymyXCLASSParameter(ThreadNumber, 6, c) <= 0.d0) then !< substract dust continuum offset if T_Back <= 0
                                        LocalIntensity = LocalIntensity - eta_source * (j_tk - j_back - j_cb) * (1.d0 - dexp(-tau_d))
                                    endif
                                else                                                        !< foreground loop here
                                    LocalIntensity = LocalIntensity * dexp(-tau_l) + eta_source * (j_tk - j_back - j_cb) * (1.d0 - dexp(-tau_l))
                                endif
                            endif                                                           !< continue here, if current comp. is not equal to ae


                            !< add background continuum
                            if (c == TotalNumberComponents_d .and. ae == 2) then
                                LocalIntensity = LocalIntensity + j_back

                                ! Debug:
                                !    if (dabs(freq - debug_freq_d) < 1.d-6) then
                                !        print*,'--------------> LocalIntensity = ', LocalIntensity
                                !    endif
                            endif

                        end Do                                                              !< c: loop over all components
                    end Do                                                                  !< ae: loop over core / foreground

                    ! Debug:
                    !    if (dabs(freq - debug_freq_d) < 1.d-6) stop


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< determine chi^2 value using local calculated intensity
                    chi2Value = chi2Value + (LocalIntensity - ObservationalDataList_d(FreqIndex, 2))**2


                    !< save model function
                    if (ModelFunctionFlag_d) then
                        ModelFunctionListLocal(ThreadNumber, FreqIndex) = LocalIntensity
                    endif
                end Do                                                                      !< FreqIndex: loop over frequencies
            end Do                                                                          !< l: loop over frequencies
            chi2ValuesVector(ThreadNumber) = chi2Value

            ! Debug:
            !    print*,"DataPointIndexFreqRange_d(:, 1) = ", DataPointIndexFreqRange_d(:, 1)
            !    print*,"DataPointIndexFreqRange_d(:, 2) = ", DataPointIndexFreqRange_d(:, 2)
            !    print*,"ObservationalDataList_d(DataPointIndexFreqRange_d(1, 1), :) = ", ObservationalDataList_d(DataPointIndexFreqRange_d(1, 1), :)
            !    print*,"ObservationalDataList_d(DataPointIndexFreqRange_d(1, 2), :) = ", ObservationalDataList_d(DataPointIndexFreqRange_d(1, 2), :)

            return
        end subroutine ModelCalcSpectrumGPU


        !>************************************************************************************************************************************************
        !> subroutine: ModelParamFree
        !>
        !> free memory used by variables of the Module Model
        !>
        !>
        !> input variables:     deallocstatus           status of the previous deallocation process
        !>
        !> output variables:    deallocstatus           status of the deallocation process
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 26.08.2014
        !>
        subroutine ModelParamFree(deallocstatus)

            implicit none
            !    integer :: i
            integer :: deallocstatus                                                        !< status of the deallocation process


            !< deallocate memory of myXCLASS variables
            call myXCLASSParamFree(deallocstatus)


            !< free device array variables
            !    i = cudaFreeArray(NumberComponentsPerMolecule_d)
            !    i = cudaFreeArray(IsoNfitConversionTable_d)
            !    i = cudaFreeArray(SpectrumIndexForFreqRange_d)
            !    i = cudaFreeArray(DataPointIndexFreqRange_d)
            !    i = cudaFreeArray(MolecularDataIndices_d)
            !    i = cudaFreeArray(ConversionTableMAGIXmyXCLASSParam_d)
            !    i = cudaFreeArray(CompMoleculeIndex_d)
            !    i = cudaFreeArray(TempPartFunc_d)
            !    i = cudaFreeArray(BackgroundTemperatureRange_d)
            !    i = cudaFreeArray(TemperatureSlopeRange_d)
            !    i = cudaFreeArray(HydrogenColumnDensityRange_d)
            !    i = cudaFreeArray(DustBetaRange_d)
            !    i = cudaFreeArray(KappaRange_d)
            !    i = cudaFreeArray(TelescopeSize_d)
            !    i = cudaFreeArray(StartFrequency_d)
            !    i = cudaFreeArray(ObservationalDataList_d)
            !    i = cudaFreeArray(MolecularData_d)
            !    i = cudaFreeArray(MoleculeNames_d)
            !    i = cudaFreeArray(IsoRatioConversionTable_d)
            !    i = cudaFreeArray(lgQ_d)
            !    i = cudaFreeArray(myXCLASSParameter_d)
            !    i = cudaFreeArray(CopyCompMoleculeIndex)
            !    i = cudaFreeArray(PartFunc)
            !    i = cudaFreeArray(CopymyXCLASSParameter)
            !    i = cudaFreeArray(CopyIsoRatioConversionTable)

            return
        end subroutine ModelParamFree
end Module Model
!*********************************************************************************************************************************************************

