!*********************************************************************************************************************************************************
!>  Module: myXCLASSCore
!>
!>
!>  This module contains the core subroutines for the myXCLASS program
!>  Copyright (C) 2009 - 2016  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine CheckMolNames:                 checks, if a molecule name is already known
!>      - subroutine strlen:                        subroutine to determine last non blank and non char(0) character of string st
!>      - subroutine GetmyXCLASSParameter:          reads parameter for myXCLASS from paramset variable
!>      - subroutine GetPartitionFunction:          gets partition function from sqlite3 database
!>      - subroutine GetTransitionParameters:       gets informations for transitions from sqlite3 database table linecat
!>      - subroutine myXCLASSInitVar:               initialize myXCLASS variables
!>      - subroutine InterpolateQ:                  interpolates partition function for a given temperature
!>      - subroutine DetectionEquation:             calculates the detection equation for a given frequency
!>      - subroutine ModelCalcSpectrum:             calculates the myXCLASS spectrum for a given parameter vector
!>      - subroutine myXCLASSParamFree:             free memory used by maXCLASS variables
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    04.02.2015  Initial version
!>  T. Moeller    11.06.2015  improved dust handling, introduction of dust temperature
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!---------------------------------------------------------------------------------------------------------------------------------------------------------
Module myXCLASSCore
    !> declare variables which are used by functions and subroutines of this module

    use Variables
    use FunctionCalling

    implicit none


    !< general variables
    integer :: ErrChannelIndex                                                              !< loop variable for error handling
    integer :: ErrChannel                                                                   !< current error channel
    integer, parameter :: DebugChannel = 98765                                              !< channel for debugging
    integer, dimension(2) :: AllErrChannels = (/0, 2222/)                                   !< list of all error channels
    real*8 :: pi                                                                            !< pi
    real*8 :: cms, ckms                                                                     !< speed of light in (m/s) and (km/s)
    real*8 :: MHz2Kelvin                                                                    !< conversion factor for MHz in Kelvin
    real*8 :: Wavenumber2Kelvin                                                             !< conversion factor for cm^(-1) in Kelvin
    real*8 :: Tcbg                                                                          !< temperature of the cosmic background
    real*8 :: tt1, tt2                                                                      !< working variables for debugging: used for timing
    real*8, parameter :: debug_freq = 569937.147d0                                          !< frequency for debugging
    logical :: UseRegistrationXMLFile                                                       !< use registation XML file
    logical :: LogFlag                                                                      !< write log information to files (used only for subroutine
                                                                                            !< ModelCalcSpectrum)
    logical :: AllOutputFilesFlag                                                           !< flag for writing intensities and optical depths to file
    logical :: IntegrationFlag                                                              !< flag indicating integration over channel width
    character(len=10) :: ParallelizationMethod                                              !< string describing the parallelization method


    !< variables for molfit file
    integer :: TotalNumberDataPoints                                                        !< total number of data points
    integer :: NumberMolecules                                                              !< number of molecules in the current molfits file
    integer :: TotalNumberComponents                                                        !< counter for total number of components of all molecules
    integer :: TotalNumberOfFrequencyRanges                                                 !< total number of frequency ranges
    integer :: TotalNumberOfMolecules                                                       !< total number of molecules including isotopologues
    integer :: NumberMolfitMolecules                                                        !< number of molecules in molfit file only
    integer, allocatable, dimension(:) :: CompMoleculeIndex                                 !< molecule index for each component
    integer, allocatable, dimension(:) :: SpectrumIndexForFreqRange                         !< store spectrum index for each frequency range
    integer, allocatable, dimension(:) :: NumberComponentsPerMolecule                       !< number of components per molecule
    integer, allocatable, dimension(:, :) :: ConversionTableMAGIXmyXCLASSParam              !< conversion table between myXCLASS and MAGIX parameter
    integer, allocatable, dimension(:, :) :: DataPointIndexFreqRange                        !< index of data point of first and last freq. in freq. range
    real*8, allocatable, dimension(:) :: StartFrequency                                     !< first frequency for each frequency range
    real*8, allocatable, dimension(:) :: EndFrequency                                       !< last frequency for each frequency range
    real*8, allocatable, dimension(:) :: StepFrequency                                      !< stepsize for each frequency range
    real*8, allocatable, dimension(:) :: TelescopeSize                                      !< size of telescope for each frequency range
    real*8, allocatable, dimension(:) :: GlobalvLSR                                         !< global velocity for each freq. range
    real*8, allocatable, dimension(:) :: BackgroundTemperatureRange                         !< T_Back for each frequency range
    real*8, allocatable, dimension(:) :: TemperatureSlopeRange                              !< T_Slope for each frequency range
    real*8, allocatable, dimension(:) :: HydrogenColumnDensityRange                         !< nH for each frequency range
    real*8, allocatable, dimension(:) :: DustBetaRange                                      !< beta for each frequency range
    real*8, allocatable, dimension(:) :: KappaRange                                         !< kappa for each frequency range
    real*8, allocatable, dimension(:, :) :: myXCLASSParameter                               !< array containing all molfit parameters for each component
    real*8, allocatable, dimension(:, :) :: vWidthLimits                                    !< lower and upper limits of velocity widths parameters
    real*8, allocatable, dimension(:, :) :: vOffLimits                                      !< lower and upper limits of velocity offsets parameters
    real*8, allocatable, dimension(:, :) :: ObservationalDataList                           !< list containing all observational data
    character(len=40), allocatable, dimension(:) :: MoleculeNames                           !< names of all molecules (including isotopologues)
    logical, allocatable, dimension(:) :: tbFlag                                            !< flag indicating that T_Back and T_Slope describes
                                                                                            !< continuum completely
    logical, allocatable, dimension(:) :: InterFlag                                         !< flag indicating that observation is interferrometric data
    logical, allocatable, dimension(:) :: EmisFlag                                          !< flag indicating components with a source size definition
    logical, allocatable, dimension(:) :: tdFlag                                            !< flag for reading T_dOff and T_dSlope from molfit file
    logical, allocatable, dimension(:) :: nHFlag                                            !< flag for global setting of nH, kappa and beta
    logical :: IsoFlag                                                                      !< flag indicating use of isotopologues


    !< variables for sqlite3 database
    integer :: MaxNumTransitionsSQL                                                         !< number of transitions which are considered in SQL query
    integer, allocatable, dimension(:, :) :: NumEntriesRanges                               !< number of transitions for each freq. range and molecule
    character(len=128) :: NameOfPartFuncTable                                               !< name of table for partition function
    character(len=128) :: NameOfRadTransTable                                               !< name of table for rad. trans.
    character(len=4096) :: dbName                                                           !< name of database


    !< variables for partition function
    integer :: NumberOfTemperatures                                                         !< number of temperatures
    integer :: NumberMoleculePartFunc                                                       !< number of entries in table PartitionFunction
    integer :: Firsti, Lasti, stepi                                                         !< working variables for interpolation
    real*8 :: TempLow, TempHigh                                                             !< working variables for extrapolation of part. func.
    real*8, allocatable, dimension(:) :: TempPartFunc                                       !< temperatures for partition function
    real*8, allocatable, dimension(:, :) :: lgQ                                             !< lgQ entries of table PartitionFunction
    character(len=40) :: ColumnNameForNamePartFunc                                          !< name of column including the names of molecules
    character(len=40), allocatable, dimension(:) :: MolNamePartFunc                         !< molecule names of table PartitionFunction
    character(len=40), allocatable, dimension(:) :: ColumnNamesPartFunc                     !< column names of table PartitionFunction


    !< variables for rad-trans parameter
    integer :: TotalNumberOfTransitions                                                     !< number of entries in table RadTrans
    integer :: Column300KPartFunc                                                           !< number of column, which contains part. func. @ 300 K
    integer, allocatable, dimension(:, :, :) :: MolecularDataIndices                        !< start and end index for each molecule and frequency range
                                                                                            !< within MolecularData variable
    real*8, allocatable, dimension(:, :) :: MolecularData                                   !< array containing the molecular data for all molecules and
                                                                                            !< all freqency ranges
    character(len=40) :: ColumnNameForNameTransitions                                       !< name of column in table transitions for molecule name
    character(len=40) :: ColumnNameForFreqTransitions                                       !< name of column in table transitions for frequency
    character(len=40) :: ColumnNameForIntTransitions                                        !< name of column in table transitions for intensity
    character(len=40) :: ColumnNameForEinsteinATransitions                                  !< name of column in table transitions for Einstein A coef.
    character(len=40) :: ColumnNameForFreqErrTransitions                                    !< name of column in table transitions for error freq.
    character(len=40) :: ColumnNameForELowTransitions                                       !< name of column in table transitions for E_low
    character(len=40) :: ColumnNameForgUpTransitions                                        !< name of column in table transitions for Upper State degen.
    character(len=40), allocatable, dimension(:) :: MolNameRadTrans                         !< molecule names of table RadTrans


    !< variables for iso table file
    integer :: NumberOfIsomasters                                                           !< number of isomasters
    integer :: NumberOfGlobalISORatios                                                      !< number of global iso ratio definitions
    integer, allocatable, dimension(:, :, :) :: IsoNfitConversionTable                      !< iso ratios conversion table for free paramter index
    integer, allocatable, dimension(:, :, :) :: GlobalIsoRatioParameter                     !< parameters for globally defined iso ratios
    real*8, allocatable, dimension(:, :) :: IsoRatio                                        !< iso ratios for all isomasters
    real*8, allocatable, dimension(:, :) :: IsoRatioConversionTable                         !< table with iso ratios between iso master and molecule
    character(len=4096) :: IsoTableFileName                                                 !< path and file name of the iso file
    character(len=40), allocatable, dimension(:) :: IsoMolecule                             !< names of molecules for iso table
    character(len=40), allocatable, dimension(:) :: IsoMasters                              !< names of isomasters for iso table


    !< for myXCLASS function only
    real*8, allocatable, dimension(:) :: TauHelperArray1                                    !< working array for intensity per component calculation
    real*8, allocatable, dimension(:) :: TauHelperArray2                                    !< working array for intensity per component calculation
    real*8, allocatable, dimension(:) :: LocalIntArray                                      !< working array for intensity per component calculation
    real*8, allocatable, dimension(:, :) :: IntegHelperArray1                               !< working array for integrated line intensity
    real*8, allocatable, dimension(:, :) :: IntegHelperArray2                               !< working array for integrated line intensity
    real*8, allocatable, dimension(:, :) :: PrevIntPerPeak                                  !< working array for integrated line intensity
    real*8, allocatable, dimension(:, :, :, :) :: IntensityPerPeak                          !< working array for integrated line intensity
    character(len=4096), allocatable, dimension(:) :: TauCompFileName                       !< store name of file including tau for each component
    character(len=4096), allocatable, dimension(:) :: IntCompFileName                       !< store name of file including intensity for each component


    contains


        !>************************************************************************************************************************************************
        !> subroutine: CheckMolNames
        !>
        !> checks, if a molecule name is already known
        !>
        !>
        !> input variables:     TotalNumberOfMoleculesCopy: total number of all already known molecules
        !>                      CurrMol:                current molecule
        !>
        !> output variables:    AlreadyIncludedFlag:    flag indicating if current molecule is already known or not
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 27.07.2014
        !>
        subroutine CheckMolNames(AlreadyIncludedFlag, TotalNumberOfMoleculesCopy, CurrMol)

            implicit none
            integer :: i                                                                    !< loop variable
            integer :: TotalNumberOfMoleculesCopy                                           !< total number of all already known molecules
            character(len=40) :: CurrMol                                                    !< current molecule
            logical :: AlreadyIncludedFlag                                                  !< flag indicating if current molecule is already known or not


            AlreadyIncludedFlag = .false.                                                   !< initialize return parameter
            Do i = 1, TotalNumberOfMoleculesCopy
                if (trim(adjustl(MoleculeNames(i))) == trim(adjustl(CurrMol))) then
                    !AlreadyIncludedFlag = .true.
                    exit
                endif
            end Do
            return
        end subroutine CheckMolNames


        !>------------------------------------------------------------------------------------------------------------------------------------------------
        !> subroutine to determine last non blank and non char(0) character of string st
        !>
        !>
        !> input variables:     st:                 string which has to be analyzed
        !>
        !> output variables:    val:                position of last non blank and non char(0) character of string st
        !>
        !> Who           When            What
        !> P. Schilke    08/12/2011      initial version
        !>
        subroutine strlen(val, st)

            implicit none
            integer :: i                                                                    !< working variable
            integer :: val                                                                  !< return value
            character :: st*(*)                                                             !< string which has to be analyzed

            ! Debug:
            ! print*,'st = ', st

            i = len(st)
            Do While ((st(i:i) == ' ' .or. st(i:i) == char(0)) .and. i > 1)
                i = i - 1
            end Do
            val = i
            return
        end subroutine strlen


        !>------------------------------------------------------------------------------------------------------------------------------------------------
        !> subroutine to count occurrences of a substring
        !>
        !>
        !> input variables:     s1:                 string which is analyzed
        !>
        !>                      s2:                 string which is searched
        !>
        !> output variables:    c:                  number of occurrences
        !>
        !>
        !> partly taken from:
        !>      http://rosettacode.org/wiki/Count_occurrences_of_a_substring#Fortran
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.01.2016
        !>
        subroutine countsubstring(c, s1, s2)

            implicit none
            character(*), intent(in) :: s1, s2
            integer :: c, p, posn

            c = 0
            if (len(s2) == 0) return
            p = 1
            Do
                posn = index(s1(p:), s2)
                if (posn == 0) return
                c = c + 1
                p = p + posn + len(s2)
            end Do
        end subroutine countsubstring


        !>************************************************************************************************************************************************
        !> subroutine: GetmyXCLASSParameter
        !>
        !> read parameter for myXCLASS from paramset variable
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 27.07.2014
        !>
        subroutine GetmyXCLASSParameter

            implicit none
            integer :: i, j, k, l, m                                                        !< loop variables
            integer :: IndexComp                                                            !< index of current component
            integer :: CurrComp                                                             !< counts the current component
            integer :: IndexMolecule                                                        !< index of current molecule
            integer :: IndexFreqRange                                                       !< index of current frequency range
            integer :: IndexSpectrum                                                        !< index of current spectrum
            integer :: IndexCurrIso                                                         !< index of current iso ratio
            integer :: LastIndexFreqRange                                                   !< working variable
            integer :: FirstIndex, LastIndex                                                !< working variables for freq. range index determination
            integer :: MaxNumberOfMolecules                                                 !< working variable: max. number of molecules (incl. isotop.)
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            integer :: c                                                                    !< working variable: number of occurrences of string s2 in s1
            integer, dimension(1000) :: HelpIntArray                                        !< working variable: used for globally defined iso ratios
            character(len=1) :: s2                                                          !< working variable: used for subroutine countsubstring
            character(len=40) :: CurrNameMolecule                                           !< working variable: name of current molecule
            character(len=4096) :: ParamName                                                !< working variable: current name of variable without
                                                                                            !< extension
            character(len=4096) :: s1                                                       !< working variable: used for subroutine countsubstring
            logical :: AlreadyIncludedFlag                                                  !< working variable: indicates if current molecule name is
                                                                                            !< already known

            !< print what you do
            if (printflag .and. trim(adjustl(ParallelizationMethod)) /= "NoMAGIX") then
                print '(" ")'
                print '(" ")'
                print '(9x, "Using optimized version of MAGIX for myXCLASS (", A, "):")', trim(adjustl(ParallelizationMethod))
                print '(" ")'
                print '(" ")'
                print '(9x, "Initialize myXCLASS program:")'
                print '(" ")'
                print '(" ")'
                print '(9x, "Reading parameter for myXCLASS from paramset variable .. ", $)'
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< analyze parameter list, i.e. get all parameters wich are given in the molfit, iso-ratio, and experimental xml files
            !<      - get name of molecules and the corresponding number of components
            !<      - get parameters for each component
            !<      - get iso master, iso molecule, and iso ratios
            !<      - get frequency ranges, background temperatures and slopes, nH densities, kappa, beta, telescope sizes, and database file


            !< get general parameters from paramset variable
            dbName = "../../../Database/cdms_sqlite.db"                                     !< set default value for db file
            NumberMolecules = 0                                                             !< reset counter for total number of molecules
            TotalNumberOfFrequencyRanges = 0                                                !< total number of frequency ranges
            NumberOfGlobalISORatios = 0
            IsoFlag = .false.
            MaxNumTransitionsSQL = 0
            IndexCurrIso = 0
            Do i = 1, parameternumber                                                       !< loop over all parameter

                ! Debug:
                ! print*,"FitParameterName(i) = ", trim(adjustl(FitParameterName(i)))


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif

                ! Debug:
                ! print*,"ParamName = ", trim(adjustl(ParamName))
                ! print*,"  paramset(1, i) = ", paramset(1, i)
                ! print*,"  FitParameterValue(i) = trim(adjustl(FitParameterValue(i)))


                !< get number of molecules (without iso molecules)
                if (ParamName == "Number_Molecules") then
                    NumberMolecules = int(paramset(1, i))


                !< get number of frequency ranges
                elseif (ParamName == "MAGIXImportStartFrequency") then
                    TotalNumberOfFrequencyRanges = TotalNumberOfFrequencyRanges + 1


                !< get iso flag
                elseif (ParamName == "NumberOfISOLines") then
                    if (paramset(1, i) > 0) then
                        NumberOfIsomasters = int(paramset(1, i))
                        IsoFlag = .true.


                        !< allocate memory for myXCLASSParameter variable
                        if (allocated(IsoMolecule)) then
                            deallocate(IsoMolecule, IsoMasters, IsoRatio, stat = deallocstatus)
                            if (deallocstatus /= 0) then
                                Do ErrChannel = 6, 6
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                                    write(ErrChannel, '(3x,"Can not deallocate variables IsoMolecule etc.!")')
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(3x,"Please restart the program!")')
                                end Do
                                stop 'Program aborted!'
                            endif
                        endif
                        allocate(IsoMolecule(NumberOfIsomasters), IsoMasters(NumberOfIsomasters), IsoRatio(NumberOfIsomasters, 2), stat = allocstatus)
                        if (allocstatus /= 0) then
                            Do ErrChannel = 6, 6
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                                write(ErrChannel, '(3x,"Can not allocate variables IsoMolecule etc.!")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Please restart the program!")')
                            end Do
                            stop 'Program aborted!'
                        endif
                        IsoMolecule = ""
                        IsoMasters = ""
                        IsoRatio = 0.d0
                        IndexCurrIso = 0
                    endif


                !< get number of global iso ratio definitions
                elseif (ParamName == "NumberOfGlobalISORatios") then
                    if (paramset(1, i) > 0) then
                        NumberOfGlobalISORatios = int(paramset(1, i))

                        ! Debug:
                        ! print*,'NumberOfGlobalISORatios = ', NumberOfGlobalISORatios


                        !< allocate memory for myXCLASSParameter variable
                        if (allocated(GlobalIsoRatioParameter)) then
                            deallocate(GlobalIsoRatioParameter, stat = deallocstatus)
                            if (deallocstatus /= 0) then
                                Do ErrChannel = 6, 6
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                                    write(ErrChannel, '(3x,"Can not deallocate variable GlobalIsoRatioParameter etc.!")')
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(3x,"Please restart the program!")')
                                end Do
                                stop 'Program aborted!'
                            endif
                        endif
                        allocate(GlobalIsoRatioParameter(NumberOfIsomasters, NumberOfGlobalISORatios, 2), stat = allocstatus)
                        if (allocstatus /= 0) then
                            Do ErrChannel = 6, 6
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                                write(ErrChannel, '(3x,"Can not allocate variable GlobalIsoRatioParameter etc.!")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Please restart the program!")')
                            end Do
                            stop 'Program aborted!'
                        endif
                        GlobalIsoRatioParameter = 0
                    endif


                !< get IsoMolecule
                elseif (ParamName == "IsoMolecule" .and. IsoFlag) then
                    IndexCurrIso = IndexCurrIso + 1
                    IsoMolecule(IndexCurrIso) = trim(adjustl(FitParameterValue(i)))


                !< get IsoMasters
                elseif (ParamName == "IsoMasters" .and. IsoFlag) then
                    IsoMasters(IndexCurrIso) = trim(adjustl(FitParameterValue(i)))


                !< get IsoRatio
                elseif (ParamName == "IsoRatio" .and. IsoFlag) then
                    IsoRatio(IndexCurrIso, 1) = paramset(1, i)                              !< store the value of the iso ratio
                    if (paramset(2, i) == 1) then
                        IsoRatio(IndexCurrIso, 2) = sum(paramset(2, 1:i))                   !< store the free parameter index for this iso ratio param.
                    else
                        IsoRatio(IndexCurrIso, 2) = 0
                    endif


                !< get parameters for global iso ratio definitions
                elseif (ParamName == "IsoGlobalParameters" .and. IsoFlag .and. NumberOfGlobalISORatios > 0) then
                    s1 = trim(adjustl(FitParameterValue(i)))
                    s2 = ","
                    c = 0
                    call countsubstring(c, s1, s2)                                          !< determine number of "," character in parameter string
                    if (c > 0) then
                        c = c + 1
                        read(s1, *) HelpIntArray(1:c)
                        c = c / 2
                        k = 0
                        Do j = 1, c
                            k = k + 1
                            GlobalIsoRatioParameter(IndexCurrIso, j, 1) = HelpIntArray(k)
                            k = k + 1
                            GlobalIsoRatioParameter(IndexCurrIso, j, 2) = HelpIntArray(k)

                            ! Debug:
                            ! print *,"GlobalIsoRatioParameter(IndexCurrIso, j, :) = ", GlobalIsoRatioParameter(IndexCurrIso, j, :)
                        end Do
                    endif


                !< get path and name of sqlite3 database file
                elseif (ParamName == "MAGIXImportdbFilename") then
                    if (trim(adjustl(FitParameterValue(i))) /= "") then
                        dbName = trim(adjustl(FitParameterValue(i)))
                    endif
                endif
            end Do

            ! Debug:
            !    print*,"NumberMolecules = ", NumberMolecules
            !    print*,"TotalNumberOfFrequencyRanges = ", TotalNumberOfFrequencyRanges
            !    print*,"dbName = ", trim(adjustl(dbName))
            !    print*,"IsoFlag = ", IsoFlag
            !    if (IsoFlag) then
            !        print*,"NumberOfIsomasters = ", NumberOfIsomasters
            !        Do i = 1, NumberOfIsomasters
            !            print*,"IsoMolecule(i) = ", trim(adjustl(IsoMolecule(i)))
            !            print*,"IsoMasters(i) = ", trim(adjustl(IsoMasters(i)))
            !            print*,"IsoRatio(i, :) = ", IsoRatio(i, :)
            !        end Do
            !    endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< allocate memory for myXCLASSParameter variable
            if (allocated(StartFrequency)) then
                deallocate(StartFrequency, EndFrequency, StepFrequency, tbFlag, BackgroundTemperatureRange, TemperatureSlopeRange, &
                           HydrogenColumnDensityRange, DustBetaRange, KappaRange, TelescopeSize, InterFlag, GlobalvLSR, SpectrumIndexForFreqRange, &
                           ObservationalDataList, DataPointIndexFreqRange, MoleculeNames, ConversionTableMAGIXmyXCLASSParam, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables StartFrequency, EndFrequency etc.!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            TotalNumberDataPoints = sum(lengthexpdata(1:NumberExpFiles))
            MaxNumberOfMolecules = NumberMolecules + 2 * NumberOfIsomasters
            allocate(StartFrequency(TotalNumberOfFrequencyRanges), EndFrequency(TotalNumberOfFrequencyRanges), &
                     StepFrequency(TotalNumberOfFrequencyRanges), tbFlag(TotalNumberOfFrequencyRanges), &
                     BackgroundTemperatureRange(TotalNumberOfFrequencyRanges), TemperatureSlopeRange(TotalNumberOfFrequencyRanges), &
                     HydrogenColumnDensityRange(TotalNumberOfFrequencyRanges), DustBetaRange(TotalNumberOfFrequencyRanges), &
                     KappaRange(TotalNumberOfFrequencyRanges), TelescopeSize(TotalNumberOfFrequencyRanges), InterFlag(TotalNumberOfFrequencyRanges), &
                     GlobalvLSR(TotalNumberOfFrequencyRanges), SpectrumIndexForFreqRange(TotalNumberOfFrequencyRanges), &
                     ObservationalDataList(TotalNumberDataPoints, 3), DataPointIndexFreqRange(TotalNumberOfFrequencyRanges, 2), &
                     MoleculeNames(MaxNumberOfMolecules), ConversionTableMAGIXmyXCLASSParam(NumberFreeParameter, 2), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannel = 6, 6
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x,"Can not allocate variables StartFrequency, EndFrequency etc.!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            StartFrequency = 0.d0
            EndFrequency = 0.d0
            StepFrequency = 0.d0
            tbFlag = .true.
            BackgroundTemperatureRange = 0.d0
            TemperatureSlopeRange = 0.d0
            HydrogenColumnDensityRange = 0.d0
            DustBetaRange = 0.d0
            KappaRange = 0.d0
            TelescopeSize = 0.d0
            GlobalvLSR = 0.d0
            InterFlag = .false.
            SpectrumIndexForFreqRange = 0
            ObservationalDataList = 0.d0
            DataPointIndexFreqRange = 0
            MoleculeNames = ""
            ConversionTableMAGIXmyXCLASSParam = 0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< get parameter for each frequency range
            IndexFreqRange = 0                                                              !< reset index for current frequency range
            IndexSpectrum = 0                                                               !< reset index for current spectrum
            LastIndexFreqRange = 0
            Do i = 1, parameternumber                                                       !< loop over all parameter


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif


                !< increase counter for number of obs. data files
                if (ParamName == "MAGIXImportExpFileNames") then
                    IndexSpectrum = IndexSpectrum + 1


                !< get first frequency of each frequency range
                elseif (ParamName == "MAGIXImportStartFrequency") then
                    IndexFreqRange = IndexFreqRange + 1
                    StartFrequency(IndexFreqRange) = paramset(1, i)
                    SpectrumIndexForFreqRange(IndexFreqRange) = IndexSpectrum


                !< get last frequency of each frequency range
                elseif (ParamName == "MAGIXImportEndFrequency") then
                    EndFrequency(IndexFreqRange) = paramset(1, i)


                !< get stepsize of each frequency range
                elseif (ParamName == "MAGIXImportStepFrequency") then
                    StepFrequency(IndexFreqRange) = paramset(1, i)


                !< get definition for t_back_flag
                elseif (ParamName == "MAGIXImportTBackFlag") then
                    if (paramset(1, i) == 0.d0) then
                        tbFlag(IndexFreqRange) = .false.
                    else
                        tbFlag(IndexFreqRange) = .true.
                    endif


                !< get global definition for background temperature
                elseif (ParamName == "MAGIXImportBackgroundTemperature") then
                    BackgroundTemperatureRange(IndexFreqRange) = paramset(1, i)


                !< get global definition for temperature slope
                elseif (ParamName == "MAGIXImportTemperatureSlope") then
                    TemperatureSlopeRange(IndexFreqRange) = paramset(1, i)


                !< get global definition for n_H
                elseif (ParamName == "MAGIXImportHydrogenColumnDensity") then
                    HydrogenColumnDensityRange(IndexFreqRange) = paramset(1, i)


                !< get global definition for dust beta
                elseif (ParamName == "MAGIXImportDustBeta") then
                    DustBetaRange(IndexFreqRange) = paramset(1, i)


                !< get global definition for kappa
                elseif (ParamName == "MAGIXImportKappa") then
                    KappaRange(IndexFreqRange) = paramset(1, i) * 2.d0 * 1.66d-24 / 100.d0  !< kappa_1300 * M(H2)/M_d_M_g


                !< get global v_LSR
                elseif (ParamName == "MAGIXImportGlobalvLSR") then
                    GlobalvLSR((LastIndexFreqRange + 1):IndexFreqRange) = paramset(1, i)


                !< get size of telescope
                elseif (ParamName == "MAGIXImportTelescopeSize") then
                    TelescopeSize((LastIndexFreqRange + 1):IndexFreqRange) = paramset(1, i)


                !< get interferrometer flag
                elseif (ParamName == "MAGIXImportInterferrometerFlag") then
                    if (paramset(1, i) == 0.d0) then
                        InterFlag((LastIndexFreqRange + 1):IndexFreqRange) = .false.
                    else
                        InterFlag((LastIndexFreqRange + 1):IndexFreqRange) = .true.
                    endif
                    LastIndexFreqRange = IndexFreqRange                                     !< variable 'LastIndexFreqRange' is necessary, because
                                                                                            !< size of telescope is given only for each spectrum and not
                                                                                            !< for each frequency range
                endif

                ! Debug:
                ! print*,'i, ParamName, paramset(1, i) = ', i, trim(adjustl(ParamName)), paramset(1, i)
            end Do

            ! Debug:
            !    print*,"SpectrumIndexForFreqRange = ", SpectrumIndexForFreqRange(:)
            !    print*,"StartFrequency = ", StartFrequency(:)
            !    print*,"EndFrequency = ", EndFrequency(:)
            !    print*,"StepFrequency = ", StepFrequency(:)
            !    print*,"BackgroundTemperatureRange = ", BackgroundTemperatureRange(:)
            !    print*,"TemperatureSlopeRange = ", TemperatureSlopeRange(:)
            !    print*,"HydrogenColumnDensityRange = ", HydrogenColumnDensityRange(:)
            !    print*,"DustBetaRange = ", DustBetaRange(:)
            !    print*,"KappaRange = ", KappaRange(:)
            !    print*,"TelescopeSize = ", TelescopeSize(:)
            !    print*,"GlobalvLSR = ", GlobalvLSR(:)
            !    print*,'tbFlag(:) = ', tbFlag(:)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< prepare observational data as one long 2D list
            k = 0
            Do i = 1, IndexSpectrum                                                         !< variable 'IndexSpectrum' is NOT identical to variable
                                                                                            !< 'NumberExpFiles' !!!!!!!!!!!
                Do j = 1, lengthexpdata(i)                                                  !< loop over all points of ith spectrum
                    k = k + 1                                                               !< increase data point counter
                    ObservationalDataList(k, :) = (/ expdatax(i, j, 1), expdatay(i, j, 1), expdatae(i, j, 1) /)
                end Do
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< define indices of first and last frequency (data point) of each frequency range), respectively
            Do l = 1, TotalNumberOfFrequencyRanges                                          !< loop over all frequency ranges
                i = SpectrumIndexForFreqRange(l)                                            !< get spectrum index
                FirstIndex = 0
                LastIndex = 0
                Do j = 1, lengthexpdata(i)                                                  !< loop over all points of ith spectrum
                    if (expdatax(i, j, 1) <= StartFrequency(l)) then
                        FirstIndex = FirstIndex + 1
                    endif
                    if (expdatax(i, j, 1) <= EndFrequency(l)) then
                        LastIndex = LastIndex + 1
                    else
                        exit
                    endif
                end Do


                !< check indices
                if (FirstIndex == 0) then
                    FirstIndex = 1
                endif
                if (LastIndex == 0) then
                    LastIndex = 1
                endif
                if (FirstIndex > LastIndex) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x,"An error occurred by calculating the indices of each frequency range!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                else


                    !< determine offset for indices for each frequency range
                    k = 0
                    if (i > 1) then
                        k = sum(lengthexpdata(1:(i - 1)))
                    endif
                    DataPointIndexFreqRange(l, :) = (/ FirstIndex + k, LastIndex + k /)
                endif
            end Do

            ! Debug:
            !    print*,"DataPointIndexFreqRange(:, 1) = ", DataPointIndexFreqRange(:, 1)
            !    print*,"DataPointIndexFreqRange(:, 2) = ", DataPointIndexFreqRange(:, 2)
            !    print*,"ObservationalDataList(DataPointIndexFreqRange(1, 1), :) = ", ObservationalDataList(DataPointIndexFreqRange(1, 1), :)
            !    print*,"ObservationalDataList(DataPointIndexFreqRange(1, 2), :) = ", ObservationalDataList(DataPointIndexFreqRange(1, 2), :)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine list of all molecules which are defined in molfit and iso ratio files
            TotalNumberOfMolecules = 0                                                      !< reset counter for total number of molecules
            NumberMolfitMolecules = 0                                                       !< reset counter for number of molecules in molfit file
            Do i = 1, parameternumber                                                       !< loop over all parameter


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif

                ! Debug:
                !    print*,"i, trim(adjustl(FitParameterName(i))) = ", i, trim(adjustl(FitParameterName(i)))


                !< determine names of molecules
                if (ParamName == "Molecule_Name") then
                    CurrNameMolecule = trim(adjustl(FitParameterValue(i)))
                    TotalNumberOfMolecules = TotalNumberOfMolecules + 1
                    NumberMolfitMolecules = NumberMolfitMolecules + 1
                    MoleculeNames(TotalNumberOfMolecules) = trim(adjustl(CurrNameMolecule))

                    ! Debug:
                    ! print*,"CurrNameMolecule = ", trim(adjustl(CurrNameMolecule))
                    ! print*,"TotalNumberOfMolecules = ", TotalNumberOfMolecules
                endif
            end Do


            !< check, if iso ratios are defined and determine total number of molecules
            if (IsoFlag) then
                k = TotalNumberOfMolecules
                Do i = 1, k
                    CurrNameMolecule = trim(adjustl(MoleculeNames(i)))
                    Do j = 1, NumberOfIsomasters
                        if (trim(adjustl(CurrNameMolecule)) == trim(adjustl(IsoMolecule(j)))) then
                            call CheckMolNames(AlreadyIncludedFlag, TotalNumberOfMolecules, IsoMasters(j))
                            if (.not. AlreadyIncludedFlag) then
                                TotalNumberOfMolecules = TotalNumberOfMolecules + 1
                                MoleculeNames(TotalNumberOfMolecules) = trim(adjustl(IsoMasters(j)))
                           endif
                        elseif (trim(adjustl(CurrNameMolecule)) == trim(adjustl(IsoMasters(j)))) then
                            call CheckMolNames(AlreadyIncludedFlag, TotalNumberOfMolecules, IsoMolecule(j))
                            if (.not. AlreadyIncludedFlag) then
                                TotalNumberOfMolecules = TotalNumberOfMolecules + 1
                                MoleculeNames(TotalNumberOfMolecules) = trim(adjustl(IsoMolecule(j)))
                           endif
                        endif
                    end Do
                end Do
            endif

            ! Debug:
            ! print*," "
            ! Do i = 1, TotalNumberOfMolecules
            !     print*,"GetmyXCLASSParameter:    i, MoleculeNames(i) = ", i, MoleculeNames(i)
            ! end Do


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< prepare some variables for isotopologues only


            !< allocate memory for iso conversion tables
            if (allocated(IsoRatioConversionTable)) then
                deallocate(IsoRatioConversionTable, IsoNfitConversionTable, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables IsoNfitConversionTable and IsoRatioConversionTable!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            allocate(IsoNfitConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules, 2), &
                     IsoRatioConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannel = 6, 6
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x,"Can not allocate variables IsoNfitConversionTable and IsoRatioConversionTable!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            IsoNfitConversionTable = 0
            IsoRatioConversionTable = 0.d0


            !< define conversion tables IsoNfitConversionTable and IsoRatioConversionTable
            if (IsoFlag) then
                Do i = 1, parameternumber                                                   !< loop over all parameter


                    !< remove [[..]] expansion of parameter name
                    ParamName = trim(adjustl(FitParameterName(i)))
                    k = index(ParamName, "[[")
                    if (k > 0) then
                        ParamName = ParamName(:(k - 1))
                    endif


                    !< determine name of current molecule
                    if (ParamName == "Molecule_Name") then
                        CurrNameMolecule = trim(adjustl(FitParameterValue(i)))

                        ! Debug:
                        ! print*,"CurrNameMolecule = ", trim(adjustl(CurrNameMolecule))


                    !< determine index for current molecule
                    elseif (ParamName == "Molecule_Inex") then
                        IndexMolecule = int(paramset(1, i))
                        if (IndexMolecule == 0) then
                            print '(" ")'
                            print '(2x, "Error in subroutine GetmyXCLASSParameter:")'
                            print '(4x, "Can not determine molecule index for molecule: ", A, " ")', trim(adjustl(CurrNameMolecule))
                            print '(4x, "Set index to 1!")'
                            print '(" ")'
                            print '(4x,"Known molecules:")'
                            Do k = 1, TotalNumberOfMolecules
                                print '(4x, I5, 3x, A)', k, trim(adjustl(MoleculeNames(k)))
                            end Do
                            IndexMolecule = 1
                        endif

                        ! Debug:
                        ! print*,"IndexMolecule = ", IndexMolecule


                        !< create conversion table between molecules and isotopologues
                        IsoRatioConversionTable(IndexMolecule, IndexMolecule) = 1.d0
                        IsoNfitConversionTable(IndexMolecule, IndexMolecule, :) = 0
                        Do j = 1, NumberOfIsomasters                                        !< loop over all defined isotopologues

                            ! Debug:
                            !    print*," "
                            !    print*," "
                            !    print*,"j = ", j
                            !    print*,"CurrNameMolecule = ", trim(adjustl(CurrNameMolecule))
                            !    print*,"IsoMolecule(j)   = ", trim(adjustl(IsoMolecule(j)))
                            !    print*,"IsoMasters(j)    = ", trim(adjustl(IsoMasters(j)))
                            !    print*,"IsoRatio(j, 1)   = ", IsoRatio(j, 1)


                            if (trim(adjustl(CurrNameMolecule)) == trim(adjustl(IsoMolecule(j)))) then
                                l = IndexMolecule
                                m = 0
                                Do k = 1, TotalNumberOfMolecules
                                    if (l == k) then
                                        IsoRatioConversionTable(l, k) = 1.d0
                                        IsoNfitConversionTable(l, k, 1) = 1
                                    elseif (trim(adjustl(IsoMasters(j))) == trim(adjustl(MoleculeNames(k)))) then
                                        m = k
                                        IsoNfitConversionTable(l, m, 1) = int(IsoRatio(j, 2))
                                        IsoNfitConversionTable(l, m, 2) = j
                                        IsoRatioConversionTable(l, m) = IsoRatio(j, 1)
                                        if (dabs(IsoRatio(j, 1)) > 1.e-30) then
                                            IsoRatioConversionTable(m, l) = 1.d0 / IsoRatio(j, 1)
                                        else
                                            IsoRatioConversionTable(m, l) = 1.d0
                                        endif
                                    endif
                                end Do

                                ! Debug:
                                ! print*,"IsoMolecule(j), IsoRatio(j, 1) = ", trim(adjustl(IsoMolecule(j))), "    ", IsoRatio(j, 1)


                            elseif (trim(adjustl(CurrNameMolecule)) == trim(adjustl(IsoMasters(j)))) then
                                l = IndexMolecule
                                m = 0
                                Do k = 1, TotalNumberOfMolecules
                                    if (l == k) then
                                        IsoRatioConversionTable(l, k) = 1.d0
                                        IsoNfitConversionTable(l, k, 1) = 1
                                    elseif (trim(adjustl(IsoMolecule(j))) == trim(adjustl(MoleculeNames(k)))) then
                                        m = k
                                        IsoNfitConversionTable(l, m, 1) = int(IsoRatio(j, 2))
                                        IsoNfitConversionTable(l, m, 2) = -j
                                        if (dabs(IsoRatio(j, 1)) > 1.e-30) then
                                            IsoRatioConversionTable(l, m) = 1.d0 / IsoRatio(j, 1)
                                        else
                                            IsoRatioConversionTable(l, m) = 1.d0
                                        endif
                                        IsoRatioConversionTable(m, l) = IsoRatio(j, 1)
                                    endif
                                end Do

                                ! Debug:
                                ! print*,"IsoMasters(j), 1.d0 / IsoRatio(j, 1) = ", trim(adjustl(IsoMasters(j))), "    ", 1.d0 / IsoRatio(j, 1)
                            endif
                        end Do
                    endif
                end Do
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< determine the number of components for each molecule
            if (allocated(NumberComponentsPerMolecule)) then
                deallocate(NumberComponentsPerMolecule, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x,"Can not deallocate variable NumberComponentsPerMolecule!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            allocate(NumberComponentsPerMolecule(0:NumberMolfitMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannel = 6, 6
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x,"Can not allocate variable NumberComponentsPerMolecule!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            NumberComponentsPerMolecule = 0
            IndexMolecule = 1
            Do i = 1, parameternumber                                                       !< loop over all parameter


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif


                !< determine name of current molecule
                if (ParamName == "Molecule_Name") then
                    CurrNameMolecule = trim(adjustl(FitParameterValue(i)))

                    ! Debug:
                    ! print*,"CurrNameMolecule = ", trim(adjustl(CurrNameMolecule))


                !< determine index for current molecule
                elseif (ParamName == "Molecule_Inex") then
                    IndexMolecule = int(paramset(1, i))
                    if (IndexMolecule == 0) then
                        print '(" ")'
                        print '(2x, "Error in subroutine GetmyXCLASSParameter:")'
                        print '(4x, "Can not determine molecule index for molecule: ", A, " ")', trim(adjustl(CurrNameMolecule))
                        print '(4x, "Set index to 1!")'
                        print '(" ")'
                        print '(4x,"Known molecules:")'
                        Do k = 1, TotalNumberOfMolecules
                            print '(4x, I5, 3x, A)', k, trim(adjustl(MoleculeNames(k)))
                        end Do
                        IndexMolecule = 1
                    endif

                    ! Debug:
                    ! print*,"IndexMolecule = ", IndexMolecule


                !< get number of components for current molecule
                elseif (ParamName == "Number_MolLine") then
                    NumberComponentsPerMolecule(IndexMolecule) = int(paramset(1, i))

                    ! Debug:
                    ! print*,"NumberComponentsPerMolecule(IndexMolecule) = ", NumberComponentsPerMolecule(IndexMolecule)
                endif
            end Do


            !< determine total number of components and allocate memory
            TotalNumberComponents = sum(NumberComponentsPerMolecule(:))
            if (allocated(myXCLASSParameter)) then
                deallocate(myXCLASSParameter, CompMoleculeIndex, EmisFlag, tdFlag, nHFlag, vWidthLimits, vOffLimits, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables myXCLASSParameter, CompMoleculeIndex etc.!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            allocate(myXCLASSParameter(11, TotalNumberComponents), CompMoleculeIndex(TotalNumberComponents), EmisFlag(TotalNumberComponents), &
                     tdFlag(TotalNumberComponents), nHFlag(TotalNumberComponents), vWidthLimits(TotalNumberComponents, 2), &
                     vOffLimits(TotalNumberComponents, 3), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannel = 6, 6
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x,"Can not allocate variables myXCLASSParameter, CompMoleculeIndex etc.!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            myXCLASSParameter = 0.d0
            vWidthLimits = 0.d0
            vOffLimits = 0.d0
            CompMoleculeIndex = 1
            EmisFlag = .true.
            tdFlag = .false.
            nHFlag = .false.


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< get calculation parameters
            IndexMolecule = 1                                                               !< reset index for current molecule
            IndexComp = 0                                                                   !< reset index for current component
            CurrComp = 0                                                                    !< reset index for current component of current molecule
            Do i = 1, parameternumber                                                       !< loop over all parameter


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif


                !< get name of current molecule
                if (ParamName == "Molecule_Name") then
                    CurrNameMolecule = trim(adjustl(FitParameterValue(i)))

                    ! Debug:
                    ! print*,"CurrNameMolecule = ", trim(adjustl(CurrNameMolecule))


                !< get max. number of transitions
                elseif (ParamName == "MaxNumTransitionsSQL") then
                    MaxNumTransitionsSQL = int(paramset(1, i))

                    ! Debug:
                    ! print*,"MaxNumTransitionsSQL = ", MaxNumTransitionsSQL


                !< determine index for current molecule
                elseif (ParamName == "Molecule_Inex") then
                    IndexMolecule = int(paramset(1, i))
                    if (IndexMolecule == 0) then
                        print '(" ")'
                        print '(2x, "Error in subroutine GetmyXCLASSParameter:")'
                        print '(4x, "Can not determine molecule index for molecule: ", A, " ")', trim(adjustl(CurrNameMolecule))
                        print '(4x, "Set index to 1!")'
                        print '(" ")'
                        print '(4x,"Known molecules:")'
                        Do k = 1, TotalNumberOfMolecules
                            print '(4x, I5, 3x, A)', k, trim(adjustl(MoleculeNames(k)))
                        end Do
                        IndexMolecule = 1
                    endif

                    ! Debug:
                    ! print*,"IndexMolecule = ", IndexMolecule


                !< determine molecule index
                elseif (ParamName == "Number_MolLine") then
                    CurrComp = 0
                    IndexComp = sum(NumberComponentsPerMolecule(:(IndexMolecule - 1)))


                !< get EmissionFlag
                elseif (ParamName == "EmissionFlag") then
                    IndexComp = IndexComp + 1                                               !< increase indec for current component
                    CurrComp = CurrComp + 1
                    CompMoleculeIndex(IndexComp) = IndexMolecule
                    if (trim(adjustl(FitParameterValue(i))) == "E") then
                        EmisFlag(IndexComp) = .true.
                    else
                        EmisFlag(IndexComp) = .false.
                    endif


                !< get tdFlag
                elseif (ParamName == "tdFlag") then
                    if (trim(adjustl(FitParameterValue(i))) == "T") then
                        tdFlag(IndexComp) = .true.
                    else
                        tdFlag(IndexComp) = .false.
                    endif


                !< get nHFlag
                elseif (ParamName == "nHFlag") then
                    if (trim(adjustl(FitParameterValue(i))) == "T") then
                        nHFlag(IndexComp) = .true.
                    else
                        nHFlag(IndexComp) = .false.
                    endif


                !< get source size
                elseif (ParamName == "source_size") then
                    myXCLASSParameter(1, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 1/)
                    endif


                !< get T_rot
                elseif (ParamName == "T_rot") then
                    myXCLASSParameter(2, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 2/)
                    endif


                    !< check for zero temperature
                    if (myXCLASSParameter(2, IndexComp) == 0.d0) then
                        Do ErrChannelIndex = 1, 1
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(2x, "Error in subroutine ReadInMolfitsParameter:")')
                            write(ErrChannel, '(4x, "The rotation temperature for component ", I4, " of molecule ", A, " ")') CurrComp, &
                                                                                                          trim(adjustl(CurrNameMolecule))
                            write(ErrChannel, '(4x, "is identical to zero!")')
                            write(ErrChannel, '(4x, "Set rotation temperature for this component to 1.0 to avoid division by zero!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                        end Do
                        myXCLASSParameter(2, IndexComp) = 1.d0
                    endif


                !< get N_tot
                elseif (ParamName == "N_tot") then
                    myXCLASSParameter(3, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 3/)
                    endif


                    !< check for overflow
                    if (myXCLASSParameter(3, IndexComp) > 300.d0) then
                        Do ErrChannelIndex = 1, 1
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(2x, "Error in subroutine GetmyXCLASSParameter:")')
                            write(ErrChannel, '(4x, "The log10 value of column density for component ", I4, " of molecule ", A, " ")') CurrComp, &
                                                                                                                   trim(adjustl(CurrNameMolecule))
                            write(ErrChannel, '(4x, "causes an overflow (to large number)!")')
                            write(ErrChannel, '(4x, "Set log10 value for this column density to 300!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                        end Do
                        myXCLASSParameter(3, IndexComp) = 300.d0
                    endif


                    !< convert log10 value back to linear value
                    myXCLASSParameter(3, IndexComp) = 10.d0**myXCLASSParameter(3, IndexComp)

                    ! Debug:
                    ! print*,'i, paramset(1, i) = ', i, paramset(1, i)
                    ! print*,'IndexComp, myXCLASSParameter(3, IndexComp) = ', IndexComp, myXCLASSParameter(3, IndexComp)


                !< get V_width
                elseif (ParamName == "V_width") then
                    myXCLASSParameter(4, IndexComp) = paramset(1, i)
                    if (myXCLASSParameter(4, IndexComp) < 0.0001) then
                        myXCLASSParameter(4, IndexComp) = 0.0001
                    endif
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 4/)
                    endif
                    vWidthLimits(IndexComp, :) = (/dmin1(paramset(3, i), paramset(4, i)), dmax1(paramset(3, i), paramset(4, i))/)


                !< get V_off
                elseif (ParamName == "V_off") then
                    myXCLASSParameter(5, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 5/)
                    endif
                    vOffLimits(IndexComp, :) = (/paramset(2, i), dmin1(paramset(3, i), paramset(4, i)), dmax1(paramset(3, i), paramset(4, i))/)


                !< get CFFlag
                elseif (ParamName == "CFFlag") then
                    if (trim(adjustl(FitParameterValue(i))) == "a" .or. trim(adjustl(FitParameterValue(i))) == "f") then
                        myXCLASSParameter(11, IndexComp) = 2.d0
                    else
                        myXCLASSParameter(11, IndexComp) = 1.d0
                    endif
                endif


                !< get T_dOff and T_dSlope for each component
                if (IndexComp > 0) then
                    if (tdFlag(IndexComp)) then


                        !< get T_dOff
                        if (ParamName == "T_dOff") then
                            myXCLASSParameter(6, IndexComp) = paramset(1, i)
                            if (paramset(2, i) == 1) then
                                k = int(sum(paramset(2, 1:i)))                              !< index of free parameter
                                ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 6/)
                            endif


                        !< get T_dSlope
                        elseif (ParamName == "T_dSlope") then
                            myXCLASSParameter(7, IndexComp) = paramset(1, i)
                            if (paramset(2, i) == 1) then
                                k = int(sum(paramset(2, 1:i)))                              !< index of free parameter
                                ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 7/)
                            endif
                        endif
                    endif


                    !< get nH, kappa, and beta for each component
                    if (nHFlag(IndexComp)) then


                        !< get nHcolumn
                        if (ParamName == "nHcolumn") then
                            myXCLASSParameter(8, IndexComp) = paramset(1, i)
                            if (paramset(2, i) == 1) then
                                k = int(sum(paramset(2, 1:i)))                              !< index of free parameter
                                ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 8/)
                            endif


                            !< check for overflow
                            if (myXCLASSParameter(8, IndexComp) > 300.d0) then
                                Do ErrChannelIndex = 1, 1
                                    ErrChannel = AllErrChannels(ErrChannelIndex)
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(2x, "Error in subroutine GetmyXCLASSParameter:")')
                                    write(ErrChannel, '(4x, "The log10 value of hydrogen column density for component ", I4, " of molecule ", A, " ")') &
                                                                                                                CurrComp, trim(adjustl(CurrNameMolecule))
                                    write(ErrChannel, '(4x, "causes an overflow (to large number)!")')
                                    write(ErrChannel, '(4x, "Set log10 value for this column density to 300!")')
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(" ")')
                                end Do
                                myXCLASSParameter(8, IndexComp) = 300.d0
                            endif


                            !< convert log10 value back to linear value
                            myXCLASSParameter(8, IndexComp) = 10.d0**myXCLASSParameter(8, IndexComp)


                        !< get beta
                        elseif (ParamName == "beta") then
                            myXCLASSParameter(9, IndexComp) = paramset(1, i)
                            if (paramset(2, i) == 1) then
                                k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                                ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 10/)
                            endif


                        !< get kappa
                        elseif (ParamName == "kappa") then
                            myXCLASSParameter(10, IndexComp) = paramset(1, i) * 2.d0 * 1.66d-24 / 100.d0     !< kappa_1300 * M(H2)/M_d_M_g
                            if (paramset(2, i) == 1) then
                                k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                                ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 9/)
                            endif
                        endif
                    endif
                endif
            end Do

            ! Debug:
            !    print*,"myXCLASSParameter(1, :) = ", myXCLASSParameter(1, :)
            !    print*,"myXCLASSParameter(2, :) = ", myXCLASSParameter(2, :)
            !    print*,"myXCLASSParameter(3, :) = ", myXCLASSParameter(3, :)
            !    print*,"myXCLASSParameter(4, :) = ", myXCLASSParameter(4, :)
            !    print*,"myXCLASSParameter(5, :) = ", myXCLASSParameter(5, :)
            !    print*,"myXCLASSParameter(6, :) = ", myXCLASSParameter(6, :)
            !    print*,"myXCLASSParameter(7, :) = ", myXCLASSParameter(7, :)
            !    print*,"myXCLASSParameter(8, :) = ", myXCLASSParameter(8, :)
            !    print*,"myXCLASSParameter(9, :) = ", myXCLASSParameter(9, :)
            !    print*,"myXCLASSParameter(10, :) = ", myXCLASSParameter(10, :)
            !    print*,"myXCLASSParameter(11, :) = ", myXCLASSParameter(11, :)
            !    if (IsoFlag) then
            !        Do i = 1, TotalNumberOfMolecules
            !            print*,"i, IsoRatioConversionTable(i, :) = ", i, IsoRatioConversionTable(i, :)
            !        end Do
            !    endif


            !< print what you do
            if (printflag .and. trim(adjustl(ParallelizationMethod)) /= "NoMAGIX") then
                print '("done!")'
            endif


            return
        end subroutine GetmyXCLASSParameter


        !>************************************************************************************************************************************************
        !> subroutine: GetPartitionFunction
        !>
        !> get partition function from sqlite3 database
        !>
        !> input variables:     none
        !>
        !> output variables:    ok:             status variable
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 29.07.2014
        !>
        subroutine GetPartitionFunction(ok)

            implicit none
            integer :: ok                                                                   !< status variable
            integer :: number_entries                                                       !< number of entries for the PartitionFunction table
            integer :: countlen                                                             !< total length of query
            integer :: i, j, k, l                                                           !< loop/working variables
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            real*8, allocatable, dimension(:, :) :: CopylgQ                                 !< copy of lgQ variable
            character(len=16384) :: queryString1                                            !< query string (first part)
            character(len=16384) :: queryString2                                            !< query string (second part)
            character(len=4096) :: dbNameCopy                                               !< name of database
            character(len=40) :: ColumnNameForNamePartFuncCopy                              !< copy of name of column including the names of molecules
            character(len=40), dimension(TotalNumberOfMolecules) :: CopyMoleculeNames       !< copy of names of molecules


            !< print what you do
            if (printflag) then
                print '(9x, "Reading partition functions for all molecules from sqlite3 database .. ", $)'
            endif



            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define query string


            !< first part
            queryString1 = "SELECT count(*) FROM " // trim(adjustl(NameOfPartFuncTable)) // char(0)
            queryString2 = " " // char(0)


            !< create query string
            ok = 0                                                                          !< reset status variable
            number_entries = 0                                                              !< reset result variable
            j = TotalNumberOfMolecules
            ColumnNameForNamePartFuncCopy = trim(adjustl(ColumnNameForNamePartFunc)) // char(0) !< get name of column for molecule names
            countlen = (len_trim(queryString1) - 1)                                         !< length of string queryString1
            countlen = countlen + 8                                                         !< length plus ' WHERE ('
            CopyMoleculeNames = ""
            Do k = 1, TotalNumberOfMolecules                                                !< loop over all molecules
                CopyMoleculeNames(k) = char(34) // trim(adjustl(MoleculeNames(k))) // char(34) // char(0)
                countlen = countlen + len_trim(CopyMoleculeNames(k)) - 1                    !< length plus length of name (incl. '""')
                countlen = countlen + len_trim(ColumnNameForNamePartFuncCopy)               !< length plus 'PF_Name'
                countlen = countlen + 7                                                     !< length plus ' or ' and ' = '
            end Do
            countlen = countlen + 2                                                         !< length plus ') '
            countlen = countlen + (len_trim(queryString2) - 1)                              !< length plus length of string queryString2
            countlen = countlen + len_trim(ColumnNameForNamePartFuncCopy) + 2               !< length plus 'PF_Name'
            dbNameCopy = trim(adjustl(dbName)) // char(0)

            ! Debug:
            ! print*,' '
            ! print*,'queryString1 = >', trim(adjustl(queryString1)), '<'
            ! print*,'queryString2 = >', trim(adjustl(queryString2)), '<'
            ! print*,'countlen = ', countlen


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get number of entries for the partition function table
            call numentr_sqlite3(ok, number_entries, %val(countlen), trim(adjustl(queryString1)), trim(adjustl(queryString2)), %val(j), &
                                 trim(adjustl(dbNameCopy)), trim(adjustl(ColumnNameForNamePartFuncCopy)), CopyMoleculeNames)
            NumberMoleculePartFunc = max(number_entries, TotalNumberOfMolecules)

            ! Debug:
            ! print*,'NumberMoleculePartFunc = ', NumberMoleculePartFunc
            ! print*,'ok = ', ok


            !< allocate variables for partition function
            if (NumberMoleculePartFunc > 0 .and. ok == 0) then
                if (allocated(MolNamePartFunc)) then
                    deallocate(MolNamePartFunc, lgQ, CopylgQ, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        Do ErrChannelIndex = 1, 1
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(1x,"Error in subroutine GetPartitionFunction!")')
                            write(ErrChannel, '(3x,"Can not deallocate variables MolNamePartFunc and lgQ!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(3x,"Please restart the program!")')
                        end Do
                        stop 'Program aborted!'
                    endif
                endif
                allocate(MolNamePartFunc(NumberMoleculePartFunc), lgQ(NumberOfTemperatures, NumberMoleculePartFunc), &
                         CopylgQ(NumberOfTemperatures, NumberMoleculePartFunc), stat = allocstatus)
                if (allocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetPartitionFunction!")')
                        write(ErrChannel, '(3x,"Can not allocate variables MolNamePartFunc and lgQ!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
                MolNamePartFunc = ""
                lgQ = 0.d0
                CopylgQ = 0.d0


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< construct query string
                ok = 0                                                                      !< reset status variable
                i = NumberMoleculePartFunc
                queryString1 = "SELECT " // trim(adjustl(ColumnNameForNamePartFunc))
                Do j = 1, NumberOfTemperatures
                    queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNamesPartFunc(j)))
                end Do
                queryString1 = trim(adjustl(queryString1)) // " FROM " // trim(adjustl(NameOfPartFuncTable)) // char(0)

                ! Debug:
                ! print*,"queryString1 = ", queryString1


                !< get information from sqlite3 database
                ok = 0                                                                      !< reset status variable
                l = NumberOfTemperatures
                i = NumberMoleculePartFunc
                j = TotalNumberOfMolecules
                countlen = (len_trim(queryString1) - 1)
                CopyMoleculeNames = ""
                Do k = 1, TotalNumberOfMolecules
                    CopyMoleculeNames(k) = char(34) // trim(adjustl(MoleculeNames(k))) // char(34) // char(0)
                    countlen = countlen + len_trim(CopyMoleculeNames(k)) + 14

                    ! Debug:
                    ! print*,'k, CopyMoleculeNames(k) = ', k, trim(adjustl(CopyMoleculeNames(k)))
                end Do
                countlen = countlen + (len_trim(queryString2) - 1) + 17


                !< bug fix: prevent segmentation fault, if database contains double entries
                if (NumberMoleculePartFunc > TotalNumberOfMolecules) then
                    Do k = TotalNumberOfMolecules, NumberMoleculePartFunc
                        countlen = countlen + 40 + 14
                    end Do
                endif


                !< get information from sqlite3 database
                ! ok,   status
                ! i,    number of entries in the database
                ! l,    number of temperatures
                ! j,    number of molecules in the query string
                dbNameCopy = trim(adjustl(dbName)) // char(0)
                ColumnNameForNamePartFuncCopy = trim(adjustl(ColumnNameForNamePartFunc)) // char(0)
                call ReadPartFunc(ok, %val(l), %val(i), %val(j), %val(countlen), trim(adjustl(queryString1)), trim(adjustl(queryString2)), &
                                  trim(adjustl(dbNameCopy)), trim(adjustl(ColumnNameForNamePartFuncCopy)), CopyMoleculeNames, lgQ, MolNamePartFunc)

                ! Debug:
                ! print*,'NumberMoleculePartFunc = ', NumberMoleculePartFunc
                ! print*,'lgQ(1:NumberOfTemperatures, 1) = ', lgQ(1:NumberOfTemperatures, 1)
                ! print*,'lgQ(1:NumberOfTemperatures, NumberMoleculePartFunc) = ', lgQ(1:NumberOfTemperatures, NumberMoleculePartFunc)
                ! print*,'MolNamePartFunc(1) = ', MolNamePartFunc(1)
                ! print*,'MolNamePartFunc(NumberMoleculePartFunc) = ', MolNamePartFunc(NumberMoleculePartFunc)
                ! stop
            else
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetPartitionFunction!")')
                    write(ErrChannel, '(3x,"There are no entries in the sqlite3 database ", A, "!")') trim(adjustl(NameOfPartFuncTable))
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please correct the name of the database and restart the program!")')
                end Do
                ok = 1
                return
            endif


            !< bring partition function in variable lgQ in the same order like MoleculeNames
            CopylgQ(:, :) = lgQ(:, :)
            lgQ = 0.d0
            Do i = 1, TotalNumberOfMolecules
                Do j = 1, NumberMoleculePartFunc

                    ! Debug:
                    ! print*,'----->i, trim(adjustl(MoleculeNames(i))) = ', i, trim(adjustl(MoleculeNames(i)))
                    ! print*,'>>>>>>j, trim(adjustl(MolNamePartFunc(j))) = ', j, trim(adjustl(MolNamePartFunc(j)))

                    if (trim(adjustl(MoleculeNames(i))) // char(0) == trim(adjustl(MolNamePartFunc(j)))) then
                        lgQ(:, i) = CopylgQ(:, j)


                        !< calculate log10 of partition function of the current molecule and check for entries which are equal to zero
                        if (count(lgQ(:, i) == 0) > 0) then
                            Do ErrChannelIndex = 1, 1
                                ErrChannel = AllErrChannels(ErrChannelIndex)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(2x, "Error in subroutine qinterp!")')
                                write(ErrChannel, '(4x, "The are some entries in the parittion function table for molecule ", A, " which are not defined or zero!")') &
                                                                                                                           trim(adjustl(MoleculeNames(i)))
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(4x, "Number of entries which are identical to zero = ", I10)') count(lgQ(:, i) == 0)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(4x,"Set values at these positions to 1!")')
                                write(ErrChannel, '(" ")')
                                where (lgQ(:, i) == 0.d0)
                                    lgQ(:, i) = 1.d0
                                end where
                            end Do
                        endif


                        !< determine log10 values
                        Do l = 1, NumberOfTemperatures
                            lgQ(l, i) = dlog10(lgQ(l, i))
                        end Do
                        exit
                    endif
                end Do

                ! Debug:
                ! print*,'i, trim(adjustl(MoleculeNames(i))), lgQ(:, i) = ', i, trim(adjustl(MoleculeNames(i))), lgQ(:, i)
            end Do


            !< print some status information to screen
            if (printflag) then
                print '("done!")'
                print '(" ")'
                print '(11x, "Number of entries = ",I10)', NumberMoleculePartFunc
                call strlen(i, MolNamePartFunc(1))
                if (NumberMoleculePartFunc > 1) then
                    print '(11x, "First Molecule in database: ",A)', trim(adjustl(MolNamePartFunc(1)(:i)))
                    call strlen(i, MolNamePartFunc(NumberMoleculePartFunc))
                    print '(11x, "Last Molecule in database:  ",A)', trim(adjustl(MolNamePartFunc(NumberMoleculePartFunc)(:i)))
                else
                    print '(11x, "Molecule in database: ",A)', trim(adjustl(MolNamePartFunc(1)(:i)))
                endif
                print '(" ")'
                print '(" ")'
            endif


            ! Debug:
            ! Do i = 1, TotalNumberOfMolecules
            !     print*,"i, MoleculeNames(i), MolNamePartFunc(i) = ", i, trim(adjustl(MoleculeNames(i))), "         ", trim(adjustl(MolNamePartFunc(i)))
            ! end Do


            !< free memory of CopylgQ variable
            if (allocated(CopylgQ)) then
                deallocate(CopylgQ, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetPartitionFunction!")')
                        write(ErrChannel, '(3x,"Can not deallocate variable CopylgQ!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif

            return
        end subroutine GetPartitionFunction


        !>************************************************************************************************************************************************
        !> subroutine: GetTransitionParameters
        !>
        !> get informations for transitions from sqlite3 database
        !>
        !> input variables:     none
        !>
        !> output variables:    ok:             status variable
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 30.07.2014
        !>
        subroutine GetTransitionParameters(ok)

            implicit none
            integer :: ok                                                                   !< status variable
            integer :: number_entries                                                       !< number of entries for the PartitionFunction table
            integer :: countlen                                                             !< total length of query
            integer :: i, j, k                                                              !< loop variables
            integer :: RangeIndex                                                           !< index for frequency ranges
            integer :: MolecularDataCounter                                                 !< overall counter for MolecularData variable
            integer :: NumEntriesLocal                                                      !< working variable: number of data within current frequency
                                                                                            !< range and current molecule
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            integer, allocatable, dimension(:) :: UpperStateDegeneracy                      !< upper state degeneracy
            real*8 :: freqmin                                                               !< working variable lowest frequency of current freq. range
            real*8 :: freqmax                                                               !< working variable highest frequency of current freq. range
            real*8 :: NewFreqMin, NewFreqMax                                                !< working variables for min. and max. freq. of current range
            real*8 :: NewFreqMin1, NewFreqMax1                                              !< working variables for min. and max. freq. of current range
            real*8 :: vLocal                                                                !< working variable local velocity offset frequency
            real*8 :: vwidthLocal                                                           !< working variable local velocity width
            real*8 :: vMin, vMax                                                            !< min. and max. offset velocity of each molecule
            real*8 :: vwidthMax                                                             !< max. velocity width
            real*8, allocatable, dimension(:) :: lFreq                                      !< frequencies of transition
            real*8, allocatable, dimension(:) :: lFreqErr                                   !< uncertainty of frequencies of transition
            real*8, allocatable, dimension(:) :: lElt                                       !< energies of lower states
            real*8, allocatable, dimension(:) :: icatMol                                    !< intensity of the transitions
            real*8, allocatable, dimension(:) :: EinsteinA                                  !< Einstein A coefficients
            character(len=40), allocatable, dimension(:) :: MolNameLocal                    !< names of molecules in current freq. range
            character(len=25) :: numberString1, numberString2, numberString3                !< working variables for number to string conversion
            character(len=8192) :: queryString1                                             !< query string (first part)
            character(len=31384) :: queryString2                                            !< query string (second part)
            character(len=4096) :: dbNameCopy                                               !< name of database
            character(len=40) :: ColumnNameForNameTransitionsCopy                           !< copy of name of column including the names of molecules
            character(len=40), dimension(1) :: CopyMoleculeNames                            !< copy of names of molecules


            !< initialize output variable
            ok = 0


            !< print what you do
            if (printflag) then
                print '(9x, "Reading parameters for radiative transitions for all molecules from sqlite database .. ", $)'
            endif


            !< allocate memory for variable NumEntriesRanges
            if (allocated(NumEntriesRanges)) then
                deallocate(NumEntriesRanges, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                        write(ErrChannel, '(3x,"Can not deallocate variable NumEntriesRanges!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    ok = 1
                    return
                endif
            endif
            allocate(NumEntriesRanges(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                    write(ErrChannel, '(3x,"Can not allocate variable NumEntriesRanges!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                ok = 1
                return
            endif
            NumEntriesRanges = 0


            !< determine total number of all transitions in all frequency ranges
            TotalNumberOfTransitions = 0
            Do RangeIndex = 1, TotalNumberOfFrequencyRanges                                 !< loop over all frequency ranges
                freqmin = StartFrequency(RangeIndex)
                freqmax = EndFrequency(RangeIndex)
                Do k = 1, TotalNumberOfMolecules                                            !< loop over all molecules


                    !< determine min. and max. velocity offset
                    NewFreqMin = 0.d0
                    NewFreqMax = 0.d0
                    vMin = 1.d99
                    vMax = -1.d99
                    vwidthMax = -1.d99
                    Do j = 1, TotalNumberComponents
                        i = CompMoleculeIndex(j)
                        if (i == k) then


                            !< for myXCLASS function or not-fitted parameters consider the velocity offset and width parameters directly
                            !< (vOffLimits(j, 1) describes if current parameter is fitted (=1) or not (=0))
                            if (LogFlag .or. vOffLimits(j, 1) == 0) then


                                !< determine max. velocity width for current molecule
                                vwidthLocal = myXCLASSParameter(4, j)
                                if (vwidthLocal > vwidthMax) then
                                    vwidthMax = vwidthLocal
                                endif


                                !< determine min. and max. velocity offsets for current molecule
                                vLocal = myXCLASSParameter(5, j)
                                if (vLocal < vMin) then
                                    vMin = vLocal
                                endif
                                if (vLocal > vMax) then
                                    vMax = vLocal
                                endif


                            !< for myXCLASSFit function consider the lower and upper limits of velocity offset and width parameters
                            else


                                !< determine max. velocity width for current molecule
                                vwidthLocal = vWidthLimits(j, 2)                            !< get upper limit of velocity width
                                if (vwidthLocal > vwidthMax) then
                                    vwidthMax = vwidthLocal
                                endif


                                !< determine min. and max. velocity offsets for current molecule
                                vLocal = vOffLimits(j, 2)                                   !< get lower limit of velocity offset
                                if (vLocal < vMin) then
                                    vMin = vLocal
                                endif
                                vLocal = vOffLimits(j, 3)                                   !< get upper limit of velocity offset
                                if (vLocal > vMax) then
                                    vMax = vLocal
                                endif
                            endif
                        endif
                    end Do
                    if (vMin == 1.d99) then
                        vMin = 0.d0
                    endif
                    if (vMax == -1.d99) then
                        vMax = 0.d0
                    endif
                    if (vwidthMax == -1.d99) then
                        vwidthMax = 0.d0
                    endif


                    !< define new min. and max. frequencies for current range if user-defined vLSR is unequal zero
                    if (GlobalvLSR(RangeIndex) /= 0.d0) then
                        NewFreqMin = (freqmin * (1.d0 - GlobalvLSR(RangeIndex) / ckms))
                        NewFreqMax = (freqmax * (1.d0 - GlobalvLSR(RangeIndex) / ckms))
                    else
                        NewFreqMin = freqmin
                        NewFreqMax = freqmax
                    endif


                    !< shift the frequency range
                    NewFreqMin1 = (NewFreqMin * (1.d0 - dmax1(vMin, vMax) / ckms))
                    NewFreqMin = dmin1(NewFreqMin, (NewFreqMin1 * 0.9d0))
                    NewFreqMax1 = (NewFreqMax * (1.d0 + dmin1(vMin, vMax) / ckms))
                    NewFreqMax = dmax1(NewFreqMax, (NewFreqMax1 * 1.1d0))


                    !< convert new limits to string
                    NewFreqMin1 = dmin1(NewFreqMin, NewFreqMax)
                    NewFreqMax1 = dmax1(NewFreqMin, NewFreqMax)
                    write(numberString1, '(ES25.15)') NewFreqMin1
                    write(numberString2, '(ES25.15)') NewFreqMax1


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< define query
                    if (MaxNumTransitionsSQL <= 0) then
                        queryString1 = "SELECT count(*) FROM " // trim(adjustl(NameOfRadTransTable)) // char(0)
                        queryString2 = "AND (" // trim(adjustl(ColumnNameForFreqTransitions)) // " >= " // trim(adjustl(numberString1)) &
                                       // " AND " // trim(adjustl(ColumnNameForFreqTransitions)) // " <= " // trim(adjustl(numberString2)) &
                                       // ") ORDER by " // trim(adjustl(ColumnNameForFreqTransitions)) // char(0)
                    else
                        write(numberString3, '(I5)') MaxNumTransitionsSQL                   !< convert float to string for freqmax
                        queryString1 = "SELECT count(*) FROM (select * from " // trim(adjustl(NameOfRadTransTable)) // char(0)
                        queryString2 = "AND (" // trim(adjustl(ColumnNameForFreqTransitions)) // " >= " // trim(adjustl(numberString1)) &
                                       // " AND " // trim(adjustl(ColumnNameForFreqTransitions)) // " <= " // trim(adjustl(numberString2)) &
                                       // ") ORDER BY " // trim(adjustl(ColumnNameForEinsteinATransitions)) // " * " &
                                       // trim(adjustl(ColumnNameForgUpTransitions)) // " DESC limit " // trim(adjustl(numberString3)) &
                                       // ") ORDER BY " // trim(adjustl(ColumnNameForFreqTransitions)) // char(0)
                    endif

                    ! Debug:
                    ! print*,' '
                    ! print*,'queryString1 = >', trim(adjustl(queryString1)), '<'
                    ! print*,'queryString2 = >', trim(adjustl(queryString2)), '<'


                    !< get number of entries for the transition table
                    ok = 0                                                                  !< reset status variable
                    number_entries = 0                                                      !< reset result variable
                    j = 1
                    countlen = (len_trim(queryString1) - 1)
                    CopyMoleculeNames = ""
                    CopyMoleculeNames(1) = char(34) // trim(adjustl(MoleculeNames(k))) // char(34) // char(0)
                    countlen = countlen + len_trim(CopyMoleculeNames(1)) + 12
                    countlen = countlen + (len_trim(queryString2) - 1) + 17
                    dbNameCopy = trim(adjustl(dbName)) // char(0)
                    ColumnNameForNameTransitionsCopy = trim(adjustl(ColumnNameForNameTransitions)) // char(0)
                    call numentr_sqlite3(ok, number_entries, %val(countlen), trim(adjustl(queryString1)), trim(adjustl(queryString2)), %val(j), &
                                         trim(adjustl(dbNameCopy)), trim(adjustl(ColumnNameForNameTransitionsCopy)), CopyMoleculeNames)
                    NumEntriesRanges(RangeIndex, k) = number_entries
                    TotalNumberOfTransitions = TotalNumberOfTransitions + number_entries

                    ! Debug:
                    ! print*,"Frequency range index = ", RangeIndex
                    ! print*,"Molecule index = ", k
                    ! print*,"Number of database entries within the current freq. range (number_entries) = ", NumEntriesRanges(RangeIndex, k)
                    ! print*,'TotalNumberOfTransitions = ', TotalNumberOfTransitions
                    ! print*,'ok = ', ok
                end Do
            end Do


            !< allocate memory for variable NumEntriesRanges
            if (allocated(MolecularData)) then
                deallocate(MolecularData, MolecularDataIndices, MolNameRadTrans, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables MolecularData, MolecularDataIndices, and MolNameRadTrans!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    ok = 1
                    return
                endif
            endif
            j = max(1, TotalNumberOfTransitions)
            allocate(MolecularData(j, 4), MolecularDataIndices(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2), &
                     MolNameRadTrans(j), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                    write(ErrChannel, '(3x,"Can not allocate variables MolecularData, MolecularDataIndices, and MolNameRadTrans!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                ok = 1
                return
            endif
            MolecularData = 0.d0
            MolecularDataIndices = 0
            MolNameRadTrans = ""


            !< allocate variables for parameter
            if (TotalNumberOfTransitions > 0 .and. ok == 0) then


                !< read in molecular data from database for each molecule
                MolecularDataCounter = 0
                Do RangeIndex = 1, TotalNumberOfFrequencyRanges                             !< loop over all frequency ranges
                    freqmin = StartFrequency(RangeIndex)                                    !< get first frequency of current frequency range
                    freqmax = EndFrequency(RangeIndex)                                      !< get last frequency of current frequency range
                    Do k = 1, TotalNumberOfMolecules                                        !< loop over all molecules


                        !< determine min. and max. velocity offset
                        NewFreqMin = 0.d0
                        NewFreqMax = 0.d0
                        vMin = 1.d99
                        vMax = -1.d99
                        vwidthMax = -1.d99
                        Do j = 1, TotalNumberComponents
                            i = CompMoleculeIndex(j)
                            if (i == k) then


                                !< for myXCLASS function or not-fitted parameters consider the velocity offset and width parameters directly
                                !< (vOffLimits(j, 1) describes if current parameter is fitted (=1) or not (=0))
                                if (LogFlag .or. vOffLimits(j, 1) == 0) then


                                    !< determine max. velocity width for current molecule
                                    vwidthLocal = myXCLASSParameter(4, j)
                                    if (vwidthLocal > vwidthMax) then
                                        vwidthMax = vwidthLocal
                                    endif


                                    !< determine min. and max. velocity offsets for current molecule
                                    vLocal = myXCLASSParameter(5, j)
                                    if (vLocal < vMin) then
                                        vMin = vLocal
                                    endif
                                    if (vLocal > vMax) then
                                        vMax = vLocal
                                    endif


                                !< for myXCLASSFit function consider the lower and upper limits of velocity offset and width parameters
                                else


                                    !< determine max. velocity width for current molecule
                                    vwidthLocal = vWidthLimits(j, 2)                        !< get upper limit of velocity width
                                    if (vwidthLocal > vwidthMax) then
                                        vwidthMax = vwidthLocal
                                    endif


                                    !< determine min. and max. velocity offsets for current molecule
                                    vLocal = vOffLimits(j, 2)                               !< get lower limit of velocity offset
                                    if (vLocal < vMin) then
                                        vMin = vLocal
                                    endif
                                    vLocal = vOffLimits(j, 3)                               !< get upper limit of velocity offset
                                    if (vLocal > vMax) then
                                        vMax = vLocal
                                    endif
                                endif
                            endif
                        end Do
                        if (vMin == 1.d99) then
                            vMin = 0.d0
                        endif
                        if (vMax == -1.d99) then
                            vMax = 0.d0
                        endif
                        if (vwidthMax == -1.d99) then
                            vwidthMax = 0.d0
                        endif


                        !< define new min. and max. frequencies for current range if user-defined vLSR is unequal zero
                        if (GlobalvLSR(RangeIndex) /= 0.d0) then
                            NewFreqMin = (freqmin * (1.d0 - GlobalvLSR(RangeIndex) / ckms))
                            NewFreqMax = (freqmax * (1.d0 - GlobalvLSR(RangeIndex) / ckms))
                        else
                            NewFreqMin = freqmin
                            NewFreqMax = freqmax
                        endif


                        !< shift the frequency range
                        NewFreqMin1 = (NewFreqMin * (1.d0 - dmax1(vMin, vMax) / ckms))
                        NewFreqMin = dmin1(NewFreqMin, (NewFreqMin1 * 0.9d0))
                        NewFreqMax1 = (NewFreqMax * (1.d0 + dmin1(vMin, vMax) / ckms))
                        NewFreqMax = dmax1(NewFreqMax, (NewFreqMax1 * 1.1d0))


                        !< convert new limits to string
                        NewFreqMin1 = dmin1(NewFreqMin, NewFreqMax)
                        NewFreqMax1 = dmax1(NewFreqMin, NewFreqMax)
                        write(numberString1, '(ES25.15)') NewFreqMin1
                        write(numberString2, '(ES25.15)') NewFreqMax1


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< define query
                        if (MaxNumTransitionsSQL <= 0) then
                            queryString2 = "AND (" // trim(adjustl(ColumnNameForFreqTransitions)) // " >= " // trim(adjustl(numberString1)) &
                                           // " AND " // trim(adjustl(ColumnNameForFreqTransitions)) // " <= " // trim(adjustl(numberString2)) &
                                           // ") ORDER by " // trim(adjustl(ColumnNameForFreqTransitions)) // char(0)
                        else
                            queryString2 = "AND (" // trim(adjustl(ColumnNameForFreqTransitions)) // " >= " // trim(adjustl(numberString1)) &
                                           // " AND " // trim(adjustl(ColumnNameForFreqTransitions)) // " <= " // trim(adjustl(numberString2)) &
                                           // ") ORDER BY " // trim(adjustl(ColumnNameForEinsteinATransitions)) // " DESC limit " &
                                           // trim(adjustl(numberString3)) // ") ORDER BY " // trim(adjustl(ColumnNameForFreqTransitions)) // char(0)
                        endif

                        ! Debug:
                        ! print*,' '
                        ! print*,"RangeIndex, k, NumEntriesRanges(RangeIndex, k) = ", RangeIndex, k, NumEntriesRanges(RangeIndex, k)


                        ! select * from (select * from transitions where (T_Name = "CH3OH;v=0;" and T_Frequency >= 300000 and T_Frequency <= 305000)
                        ! order by T_EinsteinA DESC limit 5) order by T_Frequency


                        !< allocate memory for worker
                        if (allocated(lFreq)) then
                            deallocate(MolNameLocal, lFreq, lFreqErr, icatMol, EinsteinA, lElt, UpperStateDegeneracy, stat = deallocstatus)
                            if (deallocstatus /= 0) then
                                Do ErrChannelIndex = 1, 1
                                    ErrChannel = AllErrChannels(ErrChannelIndex)
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                                    write(ErrChannel, '(3x,"Can not deallocate variables lFreq and ...!")')
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(3x,"Please restart the program!")')
                                end Do
                                ok = 1
                                return
                            endif
                        endif
                        NumEntriesLocal = NumEntriesRanges(RangeIndex, k)
                        allocate(MolNameLocal(NumEntriesLocal), lFreq(NumEntriesLocal), lFreqErr(NumEntriesLocal), icatMol(NumEntriesLocal), &
                                 EinsteinA(NumEntriesLocal), lElt(NumEntriesLocal), UpperStateDegeneracy(NumEntriesLocal), stat = allocstatus)
                        if (allocstatus /= 0) then
                            Do ErrChannelIndex = 1, 1
                                ErrChannel = AllErrChannels(ErrChannelIndex)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                                write(ErrChannel, '(3x,"Can not allocate variables lFreq and ...!")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Please restart the program!")')
                            end Do
                            ok = 1
                            return
                        endif
                        MolNameLocal = ""
                        lFreq = 0.d0
                        lFreqErr = 0.d0
                        icatMol = 0.d0
                        EinsteinA = 0.d0
                        lElt = 0.d0
                        UpperStateDegeneracy = 0
                        Do j = 1, NumEntriesLocal
                            MolNameLocal(j) = char(0)
                        end Do


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< get molecular data
                        if (NumEntriesRanges(RangeIndex, k) > 0) then                       !< check, if molecular data are contained in the database
                                                                                            !< for the current frequency range and molecule

                            !< construct query string
                            if (MaxNumTransitionsSQL <= 0) then
                                queryString1 = "SELECT " // trim(adjustl(ColumnNameForNameTransitions))                             !< pos. 1: molecule
                                                                                                                                    !< name
                            else
                                queryString1 = "SELECT * FROM (SELECT " // trim(adjustl(ColumnNameForNameTransitions))              !< pos. 1: molecule
                                                                                                                                    !< name
                            endif
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForFreqTransitions))       !< pos. 2: frequency
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForFreqErrTransitions))    !< pos. 3: error of
                                                                                                                                    !< frequency
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForIntTransitions))        !< pos. 4: intensity
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForEinsteinATransitions))  !< pos. 5: Einstein A
                                                                                                                                    !<         coefficient
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForELowTransitions))       !< pos. 6: E_lower
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForgUpTransitions))        !< pos. 7: upper state
                                                                                                                                    !<         degeneracy
                            queryString1 = trim(adjustl(queryString1)) // " FROM " // trim(adjustl(NameOfRadTransTable)) // char(0)

                            ! Debug:
                            ! print*," "
                            ! print*,"queryString1 = -->", trim(adjustl(queryString1)), "<--"


                            !< get information from sqlite3 database
                            ok = 0                                                          !< reset status variable
                            i = TotalNumberOfTransitions
                            j = 1
                            countlen = (len_trim(queryString1) - 1)
                            CopyMoleculeNames = ""
                            CopyMoleculeNames(1) = char(34) // trim(adjustl(MoleculeNames(k))) // char(34) // char(0)
                            countlen = countlen + len_trim(CopyMoleculeNames(1)) + 12
                            countlen = countlen + (len_trim(queryString2) - 1) + 17
                            dbNameCopy = trim(adjustl(dbName)) // char(0)
                            ColumnNameForNameTransitionsCopy = trim(adjustl(ColumnNameForNameTransitions)) // char(0)


                            !< read informations from sqlite3 database
                            call radtrans(ok, %val(i), %val(countlen), trim(adjustl(queryString1)), trim(adjustl(queryString2)), %val(j), &
                                          trim(adjustl(dbNameCopy)), trim(adjustl(ColumnNameForNameTransitionsCopy)), CopyMoleculeNames, &
                                          MolNameLocal, lFreq, lFreqErr, icatMol, EinsteinA, lElt, UpperStateDegeneracy)
                            if (ok /= 0) then
                                ErrChannel = 6
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                                write(ErrChannel, '(3x,"An error occurs while reading entries from sqlite3 database ", A, " within the given frequency range!")') &
                                                                                                                     trim(adjustl(NameOfRadTransTable))
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Molecule:")')
                                write(ErrChannel, '(3x,A)') trim(adjustl(MoleculeNames(k)))
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Frequency range:")')
                                write(ErrChannel, '(3x,"Min. frequency = ", A)') trim(adjustl(numberString1))
                                write(ErrChannel, '(3x,"Max. frequency = ", A)') trim(adjustl(numberString2))
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Please check your inputs and restart the program!")')
                                ok = 1
                                return

                                ! Debug:
                                ! print*,'MolNameLocal(1) = ', MolNameLocal(1)
                                ! print*,'MolNameLocal(TotalNumberOfTransitions) = ', MolNameLocal(TotalNumberOfTransitions)
                                ! print*,'lFreq(1) = ', lFreq(1)
                                ! print*,'lFreq(TotalNumberOfTransitions) = ', lFreq(TotalNumberOfTransitions)
                                ! print*,'lFreqErr(1) = ', lFreqErr(1)
                                ! print*,'lFreqErr(TotalNumberOfTransitions) = ', lFreqErr(TotalNumberOfTransitions)
                                ! print*,'icatMol(1) = ', icatMol(1)
                                ! print*,'icatMol(TotalNumberOfTransitions) = ', icatMol(TotalNumberOfTransitions)
                                ! print*,'lElt(1) = ', lElt(1)
                                ! print*,'lElt(TotalNumberOfTransitions) = ', lElt(TotalNumberOfTransitions)
                                ! stop
                            endif
                        endif


                        !< copy molecular data for current molecule and frequency range to MolecularData variable
                        MolecularDataIndices(RangeIndex, k, 1) = MolecularDataCounter + 1   !< save first index for current freq. range and molecule
                                                                                            !< within MolecularData variable
                        Do j = 1, NumEntriesLocal                                           !< loop over all transitions within the current frequency
                                                                                            !< range and molecule
                            MolecularDataCounter = MolecularDataCounter + 1
                            MolecularData(MolecularDataCounter, 1) = lFreq(j)
                            MolecularData(MolecularDataCounter, 2) = EinsteinA(j)
                            MolecularData(MolecularDataCounter, 3) = lElt(j) / 1.438769     !< only for debugging, convert K to cm-1?
                            MolecularData(MolecularDataCounter, 4) = UpperStateDegeneracy(j)

                            ! Debug:
                            ! print*,"trim(adjustl(MoleculeNames(k))), j, MolecularData(j, :) = ", trim(adjustl(MoleculeNames(k))), j, MolecularData(j, :)


                            !< correct entry for lower energy
                            MolecularData(MolecularDataCounter, 3) = lElt(j)
                            MolNameRadTrans(MolecularDataCounter) = MolNameLocal(j)
                        end Do
                        MolecularDataIndices(RangeIndex, k, 2) = MolecularDataCounter       !< save last index for current freq. range and molecule
                                                                                            !< within MolecularData variable
                    end Do
                end Do
            endif


            !< print what you do
            if (printflag) then
                print '("done!")'
            endif


            !< print some informations to screen about transitions
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(11x, "Number of transitions for each frequency range and molecule:")'
                Do RangeIndex = 1, TotalNumberOfFrequencyRanges                             !< loop over all frequency ranges
                    freqmin = StartFrequency(RangeIndex)                                    !< get first frequency of current frequency range
                    freqmax = EndFrequency(RangeIndex)                                      !< get last frequency of current frequency range
                    write(numberString1, '(ES25.15)') freqmin
                    write(numberString2, '(ES25.15)') freqmax
                    print '(" ")'
                    print '(11x, "Frequency range: ", A, " MHz - ", A, " MHz:")', trim(adjustl(numberString1)), trim(adjustl(numberString2))
                    Do k = 1, TotalNumberOfMolecules                                        !< loop over all molecules
                        print '(13x, "Number of transitions for molecule ", A, A, A, ": ", T92, I10)', char(34), trim(adjustl(MoleculeNames(k))), &
                                                                                                       char(34), NumEntriesRanges(RangeIndex, k)
                    end Do
                end Do
                print '(" ")'
                print '(" ")'
            endif


            !< free memory of local variables
            if (allocated(MolNameLocal)) deallocate(MolNameLocal, stat = deallocstatus)
            if (allocated(lFreq) .and. deallocstatus == 0) deallocate(lFreq, stat = deallocstatus)
            if (allocated(lFreqErr) .and. deallocstatus == 0) deallocate(lFreqErr, stat = deallocstatus)
            if (allocated(icatMol) .and. deallocstatus == 0) deallocate(icatMol, stat = deallocstatus)
            if (allocated(EinsteinA) .and. deallocstatus == 0) deallocate(EinsteinA, stat = deallocstatus)
            if (allocated(lElt) .and. deallocstatus == 0) deallocate(lElt, stat = deallocstatus)
            if (allocated(UpperStateDegeneracy) .and. deallocstatus == 0) deallocate(UpperStateDegeneracy, stat = deallocstatus)
            if (deallocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                    write(ErrChannel, '(3x,"Can not deallocate variables MolNameLocal etc.!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                ok = 1
            endif
            return
        end subroutine GetTransitionParameters


        !>************************************************************************************************************************************************
        !> subroutine: myXCLASSInitVar
        !>
        !> initialize myXCLASS variables
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 27.07.2014
        !>
        subroutine myXCLASSInitVar

            implicit none
            integer :: ok                                                                   !< status variablee: name of parameter
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation


            !< initialize variables
            pi = 4.d0 * datan(1.d0)                                                         !< determine pi
            Tcbg = 2.725d0                                                                  !< temperature of the cosmic background
            ckms = 299792.458d0                                                             !< speed of light in km/s
            cms = ckms * 1.d3                                                               !< speed of light in m/s
            MHz2Kelvin = 4.7992433484894915d-5                                              !< MHz to Kelvin conversion factor
                                                                                            !< = (h / k_B)
                                                                                            !< = (6.62606957d-34 / (1.d-6 * 1.3806488d-23) (J s) / (J/K)
                                                                                            !< = 4.7992433484894915e-05
            Wavenumber2Kelvin = 1.4387769599838154d0                                        !< cm-1 to Kelvin conversion factor ()
                                                                                            !< = (h * c / k_B * 100)
                                                                                            !< = 1.4387769599838154
            tt1 = 0.d0                                                                      !< initialize debug parameter tt1
            tt2 = 0.d0                                                                      !< initialize debug parameter tt2


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define variables for sqlite3 database
            NameOfRadTransTable = "transitions"
            NameOfPartFuncTable = "partitionfunctions"
            NumberOfTemperatures = 110


            !< allocate variables for partition function
            if (allocated(ColumnNamesPartFunc)) then
                deallocate(ColumnNamesPartFunc, TempPartFunc, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine myXCLASSInitVar!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables ColumnNamesPartFunc and TempPartFunc!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            allocate(ColumnNamesPartFunc(NumberOfTemperatures), TempPartFunc(NumberOfTemperatures), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine myXCLASSInitVar!")')
                    write(ErrChannel, '(3x,"Can not allocate variables ColumnNamesPartFunc and TempPartFunc!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            ColumnNamesPartFunc = ""
            TempPartFunc = 0.d0


            !< define column names of table partitionfunctions
            !< Due to a bug in the current gfortran compiler a statement like ColumnNamesPartFunc = (/.../) is not possible
            ColumnNameForNamePartFunc = 'PF_Name'
            ColumnNamesPartFunc(1) = 'PF_1_072'
            ColumnNamesPartFunc(2) = 'PF_1_148'
            ColumnNamesPartFunc(3) = 'PF_1_230'
            ColumnNamesPartFunc(4) = 'PF_1_318'
            ColumnNamesPartFunc(5) = 'PF_1_413'
            ColumnNamesPartFunc(6) = 'PF_1_514'
            ColumnNamesPartFunc(7) = 'PF_1_622'
            ColumnNamesPartFunc(8) = 'PF_1_738'
            ColumnNamesPartFunc(9) = 'PF_1_862'
            ColumnNamesPartFunc(10) = 'PF_1_995'
            ColumnNamesPartFunc(11) = 'PF_2_138'
            ColumnNamesPartFunc(12) = 'PF_2_291'
            ColumnNamesPartFunc(13) = 'PF_2_455'
            ColumnNamesPartFunc(14) = 'PF_2_630'
            ColumnNamesPartFunc(15) = 'PF_2_725'
            ColumnNamesPartFunc(16) = 'PF_2_818'
            ColumnNamesPartFunc(17) = 'PF_3_020'
            ColumnNamesPartFunc(18) = 'PF_3_236'
            ColumnNamesPartFunc(19) = 'PF_3_467'
            ColumnNamesPartFunc(20) = 'PF_3_715'
            ColumnNamesPartFunc(21) = 'PF_3_981'
            ColumnNamesPartFunc(22) = 'PF_4_266'
            ColumnNamesPartFunc(23) = 'PF_4_571'
            ColumnNamesPartFunc(24) = 'PF_4_898'
            ColumnNamesPartFunc(25) = 'PF_5_000'
            ColumnNamesPartFunc(26) = 'PF_5_248'
            ColumnNamesPartFunc(27) = 'PF_5_623'
            ColumnNamesPartFunc(28) = 'PF_6_026'
            ColumnNamesPartFunc(29) = 'PF_6_457'
            ColumnNamesPartFunc(30) = 'PF_6_918'
            ColumnNamesPartFunc(31) = 'PF_7_413'
            ColumnNamesPartFunc(32) = 'PF_7_943'
            ColumnNamesPartFunc(33) = 'PF_8_511'
            ColumnNamesPartFunc(34) = 'PF_9_120'
            ColumnNamesPartFunc(35) = 'PF_9_375'
            ColumnNamesPartFunc(36) = 'PF_9_772'
            ColumnNamesPartFunc(37) = 'PF_10_471'
            ColumnNamesPartFunc(38) = 'PF_11_220'
            ColumnNamesPartFunc(39) = 'PF_12_023'
            ColumnNamesPartFunc(40) = 'PF_12_882'
            ColumnNamesPartFunc(41) = 'PF_13_804'
            ColumnNamesPartFunc(42) = 'PF_14_791'
            ColumnNamesPartFunc(43) = 'PF_15_849'
            ColumnNamesPartFunc(44) = 'PF_16_982'
            ColumnNamesPartFunc(45) = 'PF_18_197'
            ColumnNamesPartFunc(46) = 'PF_18_750'
            ColumnNamesPartFunc(47) = 'PF_19_498'
            ColumnNamesPartFunc(48) = 'PF_20_893'
            ColumnNamesPartFunc(49) = 'PF_22_387'
            ColumnNamesPartFunc(50) = 'PF_23_988'
            ColumnNamesPartFunc(51) = 'PF_25_704'
            ColumnNamesPartFunc(52) = 'PF_27_542'
            ColumnNamesPartFunc(53) = 'PF_29_512'
            ColumnNamesPartFunc(54) = 'PF_31_623'
            ColumnNamesPartFunc(55) = 'PF_33_884'
            ColumnNamesPartFunc(56) = 'PF_36_308'
            ColumnNamesPartFunc(57) = 'PF_37_500'
            ColumnNamesPartFunc(58) = 'PF_38_905'
            ColumnNamesPartFunc(59) = 'PF_41_687'
            ColumnNamesPartFunc(60) = 'PF_44_668'
            ColumnNamesPartFunc(61) = 'PF_47_863'
            ColumnNamesPartFunc(62) = 'PF_51_286'
            ColumnNamesPartFunc(63) = 'PF_54_954'
            ColumnNamesPartFunc(64) = 'PF_58_884'
            ColumnNamesPartFunc(65) = 'PF_63_096'
            ColumnNamesPartFunc(66) = 'PF_67_608'
            ColumnNamesPartFunc(67) = 'PF_72_444'
            ColumnNamesPartFunc(68) = 'PF_75_000'
            ColumnNamesPartFunc(69) = 'PF_77_625'
            ColumnNamesPartFunc(70) = 'PF_83_176'
            ColumnNamesPartFunc(71) = 'PF_89_125'
            ColumnNamesPartFunc(72) = 'PF_95_499'
            ColumnNamesPartFunc(73) = 'PF_102_329'
            ColumnNamesPartFunc(74) = 'PF_109_648'
            ColumnNamesPartFunc(75) = 'PF_117_490'
            ColumnNamesPartFunc(76) = 'PF_125_893'
            ColumnNamesPartFunc(77) = 'PF_134_896'
            ColumnNamesPartFunc(78) = 'PF_144_544'
            ColumnNamesPartFunc(79) = 'PF_150_000'
            ColumnNamesPartFunc(80) = 'PF_154_882'
            ColumnNamesPartFunc(81) = 'PF_165_959'
            ColumnNamesPartFunc(82) = 'PF_177_828'
            ColumnNamesPartFunc(83) = 'PF_190_546'
            ColumnNamesPartFunc(84) = 'PF_204_174'
            ColumnNamesPartFunc(85) = 'PF_218_776'
            ColumnNamesPartFunc(86) = 'PF_225_000'
            ColumnNamesPartFunc(87) = 'PF_234_423'
            ColumnNamesPartFunc(88) = 'PF_251_189'
            ColumnNamesPartFunc(89) = 'PF_269_153'
            ColumnNamesPartFunc(90) = 'PF_288_403'
            ColumnNamesPartFunc(91) = 'PF_300_000'
            ColumnNamesPartFunc(92) = 'PF_309_030'
            ColumnNamesPartFunc(93) = 'PF_331_131'
            ColumnNamesPartFunc(94) = 'PF_354_813'
            ColumnNamesPartFunc(95) = 'PF_380_189'
            ColumnNamesPartFunc(96) = 'PF_407_380'
            ColumnNamesPartFunc(97) = 'PF_436_516'
            ColumnNamesPartFunc(98) = 'PF_467_735'
            ColumnNamesPartFunc(99) = 'PF_500_000'
            ColumnNamesPartFunc(100) = 'PF_501_187'
            ColumnNamesPartFunc(101) = 'PF_537_032'
            ColumnNamesPartFunc(102) = 'PF_575_440'
            ColumnNamesPartFunc(103) = 'PF_616_595'
            ColumnNamesPartFunc(104) = 'PF_660_693'
            ColumnNamesPartFunc(105) = 'PF_707_946'
            ColumnNamesPartFunc(106) = 'PF_758_578'
            ColumnNamesPartFunc(107) = 'PF_812_831'
            ColumnNamesPartFunc(108) = 'PF_870_964'
            ColumnNamesPartFunc(109) = 'PF_933_254'
            ColumnNamesPartFunc(110) = 'PF_1000_000'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define log10 of temperatures for partition funciton for
            !< T = 1.072, 1.148, 1.23, 1.318, 1.413, 1.514, 1.622, 1.738, 1.862, 1.995, 2.138, 2.291, 2.455, 2.63, 2.725, 2.818, 3.02, 3.236, 3.467,
            !<     3.715, 3.981, 4.266, 4.571, 4.898, 5, 5.248, 5.623, 6.026, 6.457, 6.918, 7.413, 7.943, 8.511, 9.12, 9.375, 9.772, 10.471, 11.22,
            !<     12.023, 12.882, 13.804, 14.791, 15.849, 16.982, 18.197, 18.75, 19.498, 20.893, 22.387, 23.988, 25.704, 27.542, 29.512, 31.623,
            !<     33.884, 36.308, 37.5, 38.905, 41.687, 44.668, 47.863, 51.286, 54.954, 58.884, 63.096, 67.608, 72.444, 75, 77.625, 83.176, 89.125,
            !<     95.499, 102.329, 109.648, 117.49, 125.893, 134.896, 144.544, 150, 154.882, 165.959, 177.828, 190.546, 204.174, 218.776, 225,
            !<     234.423, 251.189, 269.153, 288.403, 300, 309.03, 331.131, 354.813, 380.189, 407.38, 436.516, 467.735, 500, 501.187, 537.032,
            !<     575.44, 616.595, 660.693, 707.946, 758.578, 812.831, 870.964, 933.254, 1000
            TempPartFunc = (/0.030194785356751d0, 0.059941888061955d0, 0.089905111439398d0, 0.119915410257991d0, 0.150142161848559d0, &
                             0.180125875164054d0, 0.210050849875137d0, 0.240049772112648d0, 0.269979676645324d0, 0.299942900022767d0, &
                             0.330007700872759d0, 0.360025089189397d0, 0.390051496458987d0, 0.419955748489758d0, 0.435366506612661d0, &
                             0.449940988773338d0, 0.480006942957151d0, 0.510008512940235d0, 0.539953841656397d0, 0.569958818096594d0, &
                             0.599992177584098d0, 0.630020851113410d0, 0.660011221289331d0, 0.690018780788695d0, 0.698970004336019d0, &
                             0.719993826367604d0, 0.749968083509403d0, 0.780029127337338d0, 0.810030786405839d0, 0.839980557678343d0, &
                             0.869994000121742d0, 0.899984562549391d0, 0.929980590515515d0, 0.959994838328416d0, 0.971971276399756d0, &
                             0.989983458301399d0, 1.019988159591290d0, 1.049992856920140d0, 1.080012847107930d0, 1.109983294819890d0, &
                             1.140004950619450d0, 1.169997537066570d0, 1.200001865406600d0, 1.229988836544810d0, 1.259999795081890d0, &
                             1.273001272063740d0, 1.289990066054320d0, 1.320000804264730d0, 1.349995899262860d0, 1.379994040165740d0, &
                             1.410000712543460d0, 1.439995473953810d0, 1.469998642218750d0, 1.500003068051690d0, 1.529994673069140d0, &
                             1.560002326773370d0, 1.574031267727720d0, 1.590005419651330d0, 1.620000642300070d0, 1.649996507466040d0, &
                             1.679999916229220d0, 1.709998828025190d0, 1.739999309401970d0, 1.769997304028540d0, 1.800001827817970d0, &
                             1.829998088697380d0, 1.860002421901100d0, 1.875061263391700d0, 1.890001613184640d0, 1.919998030964020d0, &
                             1.949999542859920d0, 1.979998823973740d0, 2.009998730048060d0, 2.040000714473320d0, 2.070000903802340d0, &
                             2.100001582801350d0, 2.129999071957430d0, 2.160000068881300d0, 2.176091259055680d0, 2.190000948069660d0, &
                             2.220000809286660d0, 2.250000144081290d0, 2.279999836361080d0, 2.310000437185500d0, 2.339999677628230d0, &
                             2.352182518111360d0, 2.370000219475120d0, 2.400000616976380d0, 2.429999224858320d0, 2.459999773650270d0, &
                             2.477121254719660d0, 2.490000641890950d0, 2.519999840669680d0, 2.549999523574670d0, 2.579999547278480d0, &
                             2.609999703842230d0, 2.640000166906100d0, 2.669999868814090d0, 2.698970004336020d0, 2.699999797554580d0, &
                             2.730000164674160d0, 2.760000047292730d0, 2.789999998688880d0, 2.819999705510490d0, 2.850000132271100d0, &
                             2.880000243300670d0, 2.910000258512930d0, 2.940000204462930d0, 2.969999860022620d0, 3.000000000000000d0/)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< determine order of partition function in database: ascending or descending order
            TempLow = min(TempPartFunc(1), TempPartFunc(NumberOfTemperatures))
            TempHigh = max(TempPartFunc(1), TempPartFunc(NumberOfTemperatures))
            if (TempPartFunc(1) < TempPartFunc(NumberOfTemperatures)) then
                Firsti = NumberOfTemperatures
                Lasti = 1
                stepi = (-1)
            else
                Firsti = 1
                Lasti = NumberOfTemperatures
                stepi = 1
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define column names of table transitions
            ColumnNameForNameTransitions = "T_Name"
            ColumnNameForFreqTransitions = "T_Frequency"
            ColumnNameForIntTransitions = "T_Intensity"
            ColumnNameForEinsteinATransitions = "T_EinsteinA"
            ColumnNameForFreqErrTransitions = "T_Uncertainty"
            ColumnNameForELowTransitions = "T_EnergyLower"
            ColumnNameForgUpTransitions = "T_UpperStateDegeneracy"
            Column300KPartFunc = 91                                                         !< (only used, if use_intensity_flag == .true): define the
                                                                                            !< column number, containing the partition function for 300 K

            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get myXCLASS parameter
            call GetmyXCLASSParameter


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get partition functions for selected molecules and describe partition functions with polynominal
            call GetPartitionFunction(ok)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get informations for each transition in the given frequency ranges
            call GetTransitionParameters(ok)

            ! Debug:
            ! stop 'Subroutine myXCLASSInitVar finished!'

            return
        end subroutine myXCLASSInitVar


        !>************************************************************************************************************************************************
        !> subroutine: InterpolateQ
        !>
        !> interpolates partition function for a given temperature
        !>
        !>
        !> input variables:     Temperature:            temperature
        !>                      MoleculeIndex:          molecule index
        !>
        !> output variable:     QLocal:                 interpolated partition function
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 17.02.2014
        !>
        subroutine InterpolateQ(QLocal, Temperature, MoleculeIndex)

            implicit none
            integer :: i                                                                    !< loop variable
            integer :: MoleculeIndex                                                        !< molecule index
            real*8 :: Temperature                                                           !< temperature
            real*8 :: logt, x0, x1, f0, f1                                                  !< working variables for interpolation
            real*8 :: QLocal                                                                !< interpolated partition function


            !< is logt out of range?
            logt = dlog10(Temperature)                                                      !< calculate the logarithm of the excitation temperature tk
            if (logt < TempLow .or. logt > TempHigh) then


                !< extrapolate if neccessary ---------------------------------------- is not valid for all cases !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                x0 = 0.d0
                x1 = 0.d0
                f0 = 0.d0
                f1 = 0.d0
                if (logt < TempLow) then
                    if (Firsti > Lasti) then                                                !< TempPartFunc(1) describes TempLow
                        x0 = TempPartFunc(1)
                        x1 = TempPartFunc(2)
                        f0 = lgQ(1, MoleculeIndex)
                        f1 = lgQ(2, MoleculeIndex)
                    else                                                                    !< TempPartFunc(1) describes TempHigh
                        x0 = TempPartFunc(NumberOfTemperatures)
                        x1 = TempPartFunc((NumberOfTemperatures - 1))
                        f0 = lgQ(NumberOfTemperatures, MoleculeIndex)
                        f1 = lgQ((NumberOfTemperatures - 1), MoleculeIndex)
                    endif


                elseif (logt > TempHigh) then

                    ! Debug:
                    ! print*," "
                    ! print*," "
                    ! print*,"Firsti = ", Firsti
                    ! print*,"Lasti = ", Lasti
                    ! print*,"Temperature, logt = ", Temperature, logt
                    ! print*,"TempHigh = ", TempHigh
                    ! print*,"NumberOfTemperatures = ", NumberOfTemperatures


                    if (Firsti > Lasti) then                                                !< TempPartFunc(1) describes TempLow
                        x0 = TempPartFunc((NumberOfTemperatures - 1))
                        x1 = TempPartFunc(NumberOfTemperatures)
                        f0 = lgQ((NumberOfTemperatures - 1), MoleculeIndex)
                        f1 = lgQ(NumberOfTemperatures, MoleculeIndex)

                        ! Debug:
                        ! print*,"x0 = ", x0
                        ! print*,"f0 = ", f0
                        ! print*,"x1 = ", x1
                        ! print*,"f1 = ", f1
                    else                                                                    !< TempPartFunc(1) describes TempHigh
                        x0 = TempPartFunc(2)
                        x1 = TempPartFunc(1)
                        f0 = lgQ(2, MoleculeIndex)
                        f1 = lgQ(1, MoleculeIndex)
                    endif
                endif
                QLocal = f0 + (f1 - f0)/(x1 - x0) * (logt - x0)
                QLocal = 10.d0**QLocal

                ! Debug:
                ! print*,'logt > TempPartFunc(1) .or. logt < TempPartFunc(NumberOfTemperatures):'
                ! print*,'x0, f0 = ', x0, f0
                ! print*,'x1, f1 = ', x1, f1
                ! print*,'logt = ',logt
                ! print*,'>QLocal = ', QLocal
                ! print*,'>log10(QLocal) = ', dlog10(QLocal)
                ! stop
            else


                !< start linear interpolation
                Do i = Firsti, Lasti, stepi
                    if (TempPartFunc(i) == logt) then
                        QLocal = lgQ(i, MoleculeIndex)
                        exit


                    !<--------------------------------------------------------------------------------------------------------------------------------
                    !< linear interpolation: f(x) = f_0 + \frac{f_1 - f_0}{x_1 - x_0} (x - x_0)
                    elseif (TempPartFunc(i) < logt) then
                        if (i == NumberOfTemperatures) then
                            x0 = TempPartFunc(i - 1)
                            x1 = TempPartFunc(i)
                            f0 = lgQ(i - 1, MoleculeIndex)
                            f1 = lgQ(i, MoleculeIndex)
                        else
                            x0 = TempPartFunc(i)
                            x1 = TempPartFunc(i + 1)
                            f0 = lgQ(i, MoleculeIndex)
                            f1 = lgQ(i + 1, MoleculeIndex)
                        endif
                        QLocal = f0 + (f1 - f0)/(x1 - x0) * (logt - x0)

                        ! Debug:
                        !    print*," "
                        !    print*," "
                        !    print*,"Firsti = ", Firsti
                        !    print*,"Lasti = ", Lasti
                        !    print*,'x0, f0 = ', x0, f0
                        !    print*,'x1, f1 = ', x1, f1
                        !    print*,'Temperature, logt = ', Temperature, logt
                        !    print*,'QLocal = ', QLocal
                        !    print*,'10.d0**QLocal = ', 10.d0**QLocal
                        !    stop
                        exit
                    endif
                end Do
                QLocal = 10.d0**QLocal

                ! Debug:
                !    print*,' '
                !    print*,' '
                !    print*,"MoleculeNames = ", trim(adjustl(MoleculeNames(MoleculeIndex)))
                !    print*,'Temperature = ', Temperature
                !    print*,'QLocal = ', QLocal
                !    print*,'logt = ', logt
                !    Do i = 1, NumberOfTemperatures
                !        print*,'i = ', i
                !        print*,'10.d0**lgQ(i, MoleculeIndex) = ', 10.d0**lgQ(i, MoleculeIndex), "           ", QLocal
                !        print*,'10.d0**TempPartFunc(i) = ', 10.d0**TempPartFunc(i), "           ",Temperature
                !    end Do
                !    print*,'log10(QLocal) = ', dlog10(QLocal)
            endif

            return
        end subroutine InterpolateQ


        !>************************************************************************************************************************************************
        !> subroutine: DetectionEquation
        !>
        !> calculates the detection equation for a given frequency
        !>
        !>
        !> input variables:     NumComp:                total number of components
        !>                      freq:                   current frequency
        !>                      l:                      frequency range index
        !>                      MoleculeIndex:          current molecule index
        !>                      CopymyXCLASSParameter:  local copy of myXCLASSParameterOrig
        !>                      Q:                      partition function
        !>                      beam_size:              beam size
        !>                      ae:                     core / foreground index
        !>                      c:                      current component index
        !>                      TotalLocalIntensity:    total intensity (used for foreground)
        !>                      EmissionCompCounter:    counter for core components
        !>                      AbsorptionCompCounter:  counter for foreground components
        !>                      NumEmissionComp:        total number of core components
        !>                      TotalNumMol:            total number of molecules
        !>
        !> output variable:     LocalIntensity:         calc. intensity at current frequency
        !>                      tau_l:                  optical depth
        !>                      T_d:                    dust temperature
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 17.02.2014
        !>
        subroutine DetectionEquation(LocalIntensity, NumComp, freq, l, MoleculeIndex, CopymyXCLASSParameter, CopyIsoRatioConversionTable, Q, T_d, &
                                     beam_size, ae, c, tau_l, TotalLocalIntensity, EmissionCompCounter, AbsorptionCompCounter, NumEmissionComp, &
                                     TotalNumMol)

            implicit none
            integer :: ae, c, t, MoleculeIndexLoop, wm, ic                                  !< loop variables
            integer :: l                                                                    !< frequency range index
            integer :: NumComp                                                              !< total number of components
            integer :: MoleculeIndex                                                        !< working variable: index for molecule
            integer :: FirstMolecularDataIndex                                              !< working variable: first index for molecular data table
            integer :: LastMolecularDataIndex                                               !< working variable: last index for molecular data table
            integer :: EmissionCompCounter                                                  !< counter for core components
            integer :: AbsorptionCompCounter                                                !< counter for foreground components
            integer :: NumEmissionComp                                                      !< total number of emission components
            integer :: TotalNumMol                                                          !< total number of molecules
            integer :: ll, kk                                                               !< used for handling of isotoplogues
            integer :: FirstIndex                                                           !< index for first freq. data point
            integer :: LastIndex                                                            !< index for last freq. data point
            integer :: FreqIndex                                                            !< index for frequency point
            integer :: IsoCounter                                                           !< working variable: counter for isotopologue
            real*8 :: eta_source                                                            !< working variable: beam filling factor
            real*8 :: freq                                                                  !< working variable: current frequency
            real*8 :: freq_t                                                                !< working variable: frequency as temperature
            real*8 :: beam_size                                                             !< working variable: beam size (eqn. 2 - 3)
            real*8 :: j_tk                                                                  !< working variables: Rayleigh temperatures
            real*8 :: SourceFunc                                                            !< working variable: source function
            real*8 :: j_td                                                                  !< working variable: Planck function for T_d
            real*8 :: T_d                                                                   !< dust temperature
            real*8 :: j_back                                                                !< brightness for background temperature
            real*8 :: j_cb                                                                  !< brightness for cosmic microwave background temperature
            real*8 :: Ibg                                                                   !< working variable: background intensity (from emission)
            real*8 :: Ntot                                                                  !< working variable: total column density
            real*8 :: tau_d                                                                 !< working variable: tau dust
            real*8 :: tau_t                                                                 !< working variable: tau for each transition
            real*8 :: tau_l                                                                 !< working variable: sum over all taus
            real*8 :: tau_total                                                             !< working variable: total optical depth
            real*8 :: vLSR                                                                  !< working variable: v_LSR
            real*8 :: sigma                                                                 !< working variable: sigma
            real*8 :: val                                                                   !< working variable: val (for peak integration)
            real*8 :: LocalIntensity                                                        !< working variable: calc. intensity at current data point
            real*8 :: TotalLocalIntensity                                                   !< total intensity (used for foreground)
            ! real*8 :: debug_freqLocal                                                       !< for debugging: define local debug frequency
            real*8, dimension(NumComp, TotalNumMol) :: Q                                    !< partition function at the rotation temperatures
            real*8, dimension(11, NumComp) :: CopymyXCLASSParameter                         !< working variable: local copy of myXCLASSParameterOrig
            real*8, dimension(TotalNumMol, TotalNumMol) :: CopyIsoRatioConversionTable      !< local copy of IsoRatioConversionTableOrig
            logical :: CalcFalg                                                             !< used for calculation of isotopoluges


            !< initialize some variables
            LocalIntensity = 0.d0                                                           !< reset local intensity
            j_tk = 0.d0                                                                     !< initialize j_tk variable
            eta_source = 0.d0                                                               !< initialize eta_source variable
            tau_d = 0.d0                                                                    !< initialize tau_d variable
            SourceFunc = 0.d0                                                               !< initialize S variable
            j_td = 0.d0                                                                     !< initialize j_td variable
            Ibg = 0.d0                                                                      !< initialize Ibg variable
            T_d = 0.d0                                                                      !< initialize T_d variable
            wm = 0                                                                          !< initialize wm variable


            !< if no transition is located within the given frequency range
            if (TotalNumberOfTransitions == 0) then
                return
            endif


            !< CopymyXCLASSParameter(1, c)      source size
            !< CopymyXCLASSParameter(2, c)      T_rot
            !< CopymyXCLASSParameter(3, c)      N_tot
            !< CopymyXCLASSParameter(4, c)      V_width
            !< CopymyXCLASSParameter(5, c)      V_off
            !< CopymyXCLASSParameter(6, c)      T_Back
            !< CopymyXCLASSParameter(7, c)      T_Slope
            !< CopymyXCLASSParameter(8, c)      nH_column
            !< CopymyXCLASSParameter(9, c)      beta
            !< CopymyXCLASSParameter(10, c)     kappa
            !< CopymyXCLASSParameter(11, c)     core/foreground flag

            ! Debug:
            !    if (dabs(freq - debug_freq) < 1.d-1) then
            !        write(DebugChannel, *) "####################################################################################"
            !        write(DebugChannel, *) "ae = ", ae
            !        write(DebugChannel, *) "c, TotalNumberComponents = ", c, TotalNumberComponents
            !        write(DebugChannel, *) "CopymyXCLASSParameter(:, c) = ", CopymyXCLASSParameter(:, c)
            !    endif


            !<############################################################################################################################################
            !< Only used by myXCLASS function!
            !<############################################################################################################################################
            !< initialize working variables
            if (LogFlag) then
                if (AllOutputFilesFlag) then
                    IntegHelperArray1 = 0.d0
                    IntegHelperArray2 = 0.d0
                    TauHelperArray1 = 0.d0
                    TauHelperArray2 = 0.d0
                    LocalIntArray = 0.d0
                endif
            endif
            !<############################################################################################################################################
            !<############################################################################################################################################


            !< check, if current component correspond to ae index
            if (int(CopymyXCLASSParameter(11, c)) == ae) then


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< determine beam filling (dilution) factor (eq. 2):
                !< \eta(\theta_{m,c}) of molecule m and component c for a source with a Gaussian brightness profile, and a Gaussian
                !< beam is given by
                !<
                !<     \eta(\theta_{m,c}) = \frac{\theta_{m,c}^2}{\theta_{m,c}^2 + \theta_t^2}
                !<
                !< where \theta_{m,c} and \theta_t represents the source and telescope beam FWHM sizes, respectively.
                !<
                eta_source = CopymyXCLASSParameter(1, c)**2 / (beam_size**2 + CopymyXCLASSParameter(1, c)**2)

                ! Debug:
                !    if (dabs(freq - debug_freq) < 1.d-1) then
                !        write(DebugChannel, *) "MoleculeIndex = ", MoleculeIndex
                !        write(DebugChannel, *) "eta_source = ", eta_source
                !    endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< determine Rayleight-Jeans temperatures
                freq_t = MHz2Kelvin * freq                                                  !< convert frequency in temperature (Kelvin)
                                                                                            !< = \frac{h \nu}{k_B}
                j_tk = freq_t / (dexp(freq_t / CopymyXCLASSParameter(2, c)) - 1.d0)

                ! Debug:
                !    if (dabs(freq - debug_freq) < 1.d-1) then
                !        write(DebugChannel, *) "freq_t = ", freq_t
                !        write(DebugChannel, *) "j_tk = ", j_tk
                !    endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< calculate the dust contribution (eq. 4):
                !<
                !<     \tau_d^{m,c} = N_H^{m,c} * \kappa_{\rm 1.3 \, mm}^{m,c} * (\frac{\nu}{\nu_{\rm 1.3 \, mm}} )^\beta^{m,c}
                !<                              * m_{H_2} * \frac{1}{\zeta_{\rm gas-dust}},
                !<
                !< where the hydrogen column density $N_H$, the dust mass opacity $\kappa_{\rm 1.3 \, mm}$, and the exponent $\beta$.
                !< In addition, $\nu_{\rm 1.3 \, mm}$ = 230~GHz indicates the frequency for a wavelength of 1.3~mm, $m_{H_2}$
                !< describes the mass of a hydrogen molecule, and $\zeta_{\rm gas-dust}$ represents the gas to dust mass ratio,
                !< i.e.\ $1 / \zeta_{\rm gas-dust}$ describes the dust to gas ratio and is set to (1/100).
                !<
                !< CopymyXCLASSParameter(8, c) =    N_H^{m,c}
                !< CopymyXCLASSParameter(9, c) =    \beta^{m,c}
                !< CopymyXCLASSParameter(10, c) =   \kappa_{\rm 1.3 \, mm}^{m,c}   (contains factor (m_{H_2} * \frac{1}{\zeta_{\rm gas-dust}}) as well)
                !<
                tau_d = CopymyXCLASSParameter(8, c) * (freq / 2.3d5)**CopymyXCLASSParameter(9, c) * CopymyXCLASSParameter(10, c)

                ! Debug:
                !    if (dabs(freq - debug_freq) < 1.d-1) then
                !        write(DebugChannel, *) "tau_d = ", tau_d
                !        write(DebugChannel, *) "l = ", l
                !        write(DebugChannel, *) "MoleculeIndex = ", MoleculeIndex
                !        write(DebugChannel, *) "MolecularDataIndices(l, :, 1) = ", MolecularDataIndices(l, :, 1)
                !        write(DebugChannel, *) "MolecularDataIndices(l, :, 2) = ", MolecularDataIndices(l, :, 2)
                !    endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< determine optical depth $\tau(\nu)^{m,c}$ for each molecule m and component c
                !<
                !<     \tau(\nu)^{m,c} = \sum_t \frac{c^3}{8 \pi \nu^3} \, A_{ul} \, N_{\rm tot}^{m,c}
                !<                              \frac{g_u  e^{-E_l/k_B T_{\rm ex}^{m,c}}}{Q \left(m, T_{\rm ex}^{m,c} \right)}
                !<                              \left(1 - e^{-h \, \nu /k_B \, T_{\rm ex}^{m,c}} \right)
                !<                              \times \phi(\nu)^{m,c},
                !<
                !< where the sum with index $t$ runs over all spectral line transitions of molecule $m$ within the given frequency
                !< range. Additionally, the Einstein $A_{ul}$ coefficient\footnote{The indices $u$ and $l$ represent upper and lower
                !< state, respectively.}, the energy of the lower state $E_l$, the upper state degeneracy $g_u$, and the
                !< partition function $Q \left(m, T_{\rm ex}^{m,c} \right)$ of molecule $m$ are taken from the embedded
                !< SQLite3 database, described in  Sect.~\ref{sec:db}. In addition, the values of the excitation temperatures
                !< $T_{\rm ex}^{m,c}$ and the column densities $N_{\rm tot}^{m,c}$ for the different components and molecules are
                !< taken from the user defined molfit file. Furthermore, the profile of a spectral line $t$ is given by a Gaussian
                !< line profile
                !<
                !<     \phi(\nu)^{m,c} = \frac{1}{\sqrt{2 \pi} \, \sigma}
                !<                       \cdot e^{-\frac{\left(\nu-\left(\nu_t + \nu_{\rm LSR}^{m,c} \right) \right)^2}{2 \sigma^2}},
                !<
                !< which is normalized, i.e.\ $\int_0^{\infty} \phi(\nu) \, d\nu$ = 1. The source velocity $\nu_{\rm LSR}^{m,c}$ for
                !< each component $c$ of a molecule $m$ is related to the user defined velocity offset
                !< $\left(v_{\rm offset}^{m,c}\right)$ taken from the aforementioned molfit file, by the following expression
                !<
                !<     \nu_{\rm LSR}^{m,c} = -v_{\rm offset}^{m,c} \cdot \nu_t,
                !<
                !< where $\nu_t$ indicates the frequency of transition $t$ taken from the SQLite3 database mentioned above.
                !< Additionally, the variance $\sigma$ of the profile is defined by the velocity width
                !< $\left(v_{\rm width}^{m,c}\right)$ described in the molfit file for each component $c$ of a molecule $m$:
                !<
                !<     \sigma = \frac{v_{\rm width}^{m,c} \cdot \left(\nu_t + \nu_{\rm LSR}^{m,c} \right)}{2 \, \sqrt{2 \, \ln 2}}.
                !<
                tau_l = 0.d0


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< if isotopologues are used, each component has to be calculated for all isotopoluges of the current molecule as well
                !< so a further loop is necessary which runs over all molecule indices of the isotopologues related to the current
                !< molecule
                ll = MoleculeIndex                                                          !< get molecule index
                if (IsoFlag) then                                                           !< use isotopologues?
                    kk = TotalNumMol
                else
                    kk = ll
                endif
                IsoCounter = 0
                Do MoleculeIndexLoop = ll, kk                                               !< loop over all molecule indices (including isotopologues!)
                    CalcFalg = .true.
                    if (IsoFlag) then                                                       !< use isotopologues?
                        if (CopyIsoRatioConversionTable(ll, MoleculeIndexLoop) == 0.d0) then
                            CalcFalg = .false.
                        endif
                    endif
                    if (CalcFalg) then                                                      !< consider current molecule
                        IsoCounter = IsoCounter + 1


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< perform sum over all transition frequencies
                        wm = 0                                                              !< initialize wm variable
                        FirstMolecularDataIndex = MolecularDataIndices(l, MoleculeIndexLoop, 1) !< get first mol. data index of current frequency range
                        LastMolecularDataIndex = MolecularDataIndices(l, MoleculeIndexLoop, 2)  !< get last mol. data index of current freq. range
                        Do t = FirstMolecularDataIndex, LastMolecularDataIndex              !< loop over all transitions of current molecule and
                            wm = wm + 1
                            !< MolecularData(t, 1) = lFreq
                            !< MolecularData(t, 2) = EinsteinA
                            !< MolecularData(t, 3) = lElt
                            !< MolecularData(t, 4) = UpperStateDegeneracy

                            ! Debug:
                            !    if (dabs(freq - debug_freq) < 1.d-1) then
                            !        write(DebugChannel, *) "t = ", t
                            !        write(DebugChannel, *) "MolecularData(t, :) = ", MolecularData(t, :)
                            !        write(DebugChannel, *) "FirstMolecularDataIndex = ", FirstMolecularDataIndex
                            !        write(DebugChannel, *) "LastMolecularDataIndex = ", LastMolecularDataIndex
                            !    endif


                            !< determine total column density (scale with iso ratio if defined)
                            Ntot = CopymyXCLASSParameter(3, c)
                            if (IsoFlag) then                                               !< use isotopologues?
                                Ntot = Ntot * CopyIsoRatioConversionTable(ll, MoleculeIndexLoop)
                            endif


                            !<----------------------------------------------------------------------------------------------------------------------------
                            !< determine part of eq. (5):
                            !<
                            !<      \frac{c^3}{8 \pi \nu^3} \, A_{u,l} \, N_{\rm tot}^{m,c}
                            !<                              \frac{g_u  e^{-E_l/k_B T_{\rm ex}^{m,c}}}{Q \left(m, T_{\rm ex}^{m,c} \right)}
                            !<                              \left(1 - e^{-h \, \nu_{u,l} /k_B \, T_{\rm ex}^{m,c}} \right)
                            !<
                            !< MolecularData(t, 1) =         \nu_{u,l}          (transition frequency)
                            !< MolecularData(t, 2) =         A_{u,l}            (EinsteinA)
                            !< MolecularData(t, 3) =         E_low              (energy of lower level in Kelvin)
                            !< MolecularData(t, 4) =         g_u                (degeneracy of upper level)
                            !< CopymyXCLASSParameter(2, c) = T_{\rm ex}^{m,c}   (excitation temperature)
                            !< CopymyXCLASSParameter(3, c) = N_{\rm tot}^{m,c}  (total column density)
                            !<
                            if (Q(c, MoleculeIndexLoop) == 0.d0) then
                                tau_t = 0.d0
                            else
                                tau_t = (cms**2 / (8.d0 * pi * freq**2)) * MolecularData(t, 2) * Ntot * MolecularData(t, 4) &
                                                                 * (dexp(-MolecularData(t, 3) / CopymyXCLASSParameter(2, c)) / Q(c, MoleculeIndexLoop)) &
                                                                 * (1.d0 - dexp(-(MHz2Kelvin * MolecularData(t, 1) / CopymyXCLASSParameter(2, c)))) &
                                                                 * 1.d-8                    !< conversion factor: [\nu] = MHz (10^{-6}),
                                                                                            !<                    [N_{\rm tot}^{m,c}] = cm-2 (10^{+4}),
                                                                                            !<                    [A_{u,l}] = s^(-1)
                                                                                            !<  tau = tau * 10^{-6} * 10^{-6} * 10^{-4}
                                                                                            !<      = 10^{-8} s^(-1) (phi is missing here)
                                                                                            !<      = 10^{-14} s^(-1) (with phi (in MHz^(-1)))
                            endif

                            ! Debug:
                            !    debug_freqLocal = MolecularData(t, 1) * (1.d0 - (CopymyXCLASSParameter(5, c) + GlobalvLSR(l)) / ckms)
                            !    if (dabs(freq - debug_freqLocal) < 1.d0) then
                            !        write(DebugChannel, *) "t = ", t
                            !        write(DebugChannel, *) "freq = ", freq
                            !        write(DebugChannel, *) 'transition frequency    = ', MolecularData(t, 1)
                            !        write(DebugChannel, *) 'EinsteinA               = ', MolecularData(t, 2)
                            !        write(DebugChannel, *) 'E_low                   = ', MolecularData(t, 3)
                            !        write(DebugChannel, *) 'e_up                    = ', (MolecularData(t, 3) + (freq / 20836.74d0))
                            !        write(DebugChannel, *) 'g_u                     = ', MolecularData(t, 4)
                            !        write(DebugChannel, *) 'Tex                  = ', CopymyXCLASSParameter(2, c)
                            !        write(DebugChannel, *) 'Ntot                 = ', Ntot
                            !        write(DebugChannel, *) "j_tk = ", j_tk
                            !        write(DebugChannel, *) "j_cb = ", j_cb
                            !        write(DebugChannel, *) "Q    = ", Q(c, MoleculeIndexLoop)
                            !        write(DebugChannel, *) "eta_source = ", eta_source
                            !        write(DebugChannel, *) '------> tau_t (wihtout phi) = ', tau_t
                            !    endif


                            !<----------------------------------------------------------------------------------------------------------------------------
                            !< determine \nu_{\rm LSR}^{m,c}:
                            !<
                            !<     \nu_{\rm LSR}^{m,c} = -v_{\rm offset}^{m,c} \cdot \nu_t,
                            !<
                            !< CopymyXCLASSParameter(5, c) = V_off
                            !< MolecularData(t, 1) =         \nu_{u,l}          (transition frequency)
                            !<
                            vLSR = -(CopymyXCLASSParameter(5, c) + GlobalvLSR(l)) / ckms * MolecularData(t, 1)

                            ! Debug:
                            !    if (dabs(freq - debug_freq) < 1.d-1) write(DebugChannel, *) 'vLSR = ', vLSR


                            !<----------------------------------------------------------------------------------------------------------------------------
                            !< determine \sigma:
                            !<
                            !<     \sigma = \frac{v_{\rm width}^{m,c} \cdot \left(\nu_t + \nu_{\rm LSR}^{m,c}\right)}{2 \, \sqrt{2 \, \ln 2}}.
                            !<
                            !< CopymyXCLASSParameter(4, c) = V_width
                            !< MolecularData(t, 1) =         \nu_{u,l}          (transition frequency)
                            !<
                            sigma = (CopymyXCLASSParameter(4, c) / ckms * (MolecularData(t, 1) + vLSR)) / (2.d0 * dsqrt(2.d0 * dlog(2.d0)))

                            ! Debug:
                            !    if (dabs(freq - debug_freqLocal) < 1.d0) write(DebugChannel, *) 'sigma = ', sigma


                            !<----------------------------------------------------------------------------------------------------------------------------
                            !< determine \phi(\nu)^{m,c}:
                            !<
                            !<     \phi(\nu)^{m,c} = \frac{1}{\sqrt{2 \pi} \, \sigma}
                            !<                       * e^{-\frac{\left(\nu - \left(\nu_t + \nu_{\rm LSR}^{m,c}\right)\right)^2} {2 \sigma^2}},
                            !<
                            !< MolecularData(t, 1): lFreq
                            !<
                            tau_t = tau_t * 1.d0 / (dsqrt(2.d0 * pi) * sigma * 1.d6) &
                                          * dexp(-((freq - (MolecularData(t, 1) + vLSR))**2) / (2.d0 * sigma**2))

                            ! Debug:
                            !    if (dabs(freq - debug_freqLocal) < 1.d0) then
                            !        write(DebugChannel, *) 'phi = ', 1.d0 / (dsqrt(2.d0 * pi) * sigma * 1.d6) &
                            !                                         * dexp(-((freq - (MolecularData(t, 1) + vLSR))**2) / (2.d0 * sigma**2))
                            !        write(DebugChannel, *) 'tau_t (with phi) = ', tau_t
                            !        write(DebugChannel, *) ' '
                            !        write(DebugChannel, *) ' '
                            !        write(DebugChannel, *) ' '
                            !    endif


                            !<----------------------------------------------------------------------------------------------------------------------------
                            !< add tau for current transition to sum of taus
                            tau_l = tau_l + tau_t


                            !<############################################################################################################################
                            !< Only used by myXCLASS function!
                            !<############################################################################################################################
                            !< allocate variables for output variables
                            if (LogFlag) then
                                if (AllOutputFilesFlag) then
                                    !< IntegHelperArray1(IsoCounter, wm) = tau_t         (for a single transition)
                                    IntegHelperArray1(IsoCounter, wm) = tau_t               !< store tau_t
                                    TauHelperArray1(IsoCounter) = TauHelperArray1(IsoCounter) + tau_t
                                endif
                            endif
                            !<############################################################################################################################
                            !<############################################################################################################################
                        end Do                                                              !< MoleculeIndexLoop: loop over all indices of molecules
                    endif
                end Do

                ! Debug:
                !    if (dabs(freq - debug_freq) < 1.d-1) write(DebugChannel, *) 'tau_l = ', tau_l


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< define total optical depth \tau_{\rm total}, Eq. (7)
                tau_total = (tau_l + tau_d)


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< Calculate effective dust temperature using eq. (15)
                !<
                !<
                !< Tthe continuum contribution is described through the source function $S^{m,c}(\nu)$, Eq. (7), by an
                !< effective dust temperature $T_d^{m,c}$ (through $J(T_d^{m,c}, \nu)$) for each component which is given by
                !<
                !<      T_d^{m,c} (\nu) &= T_{\rm ex}^{m,c} (\nu) + \Delta T_d^{m,c}(\nu) \\
                !<                      &= T_{\rm ex}^{m,c} (\nu) + T_{d, {\rm off}}^{m,c} \\
                !<                                                \cdot \left(\frac{\nu}{\nu_{\rm min}} \right)^{T_{d, {\rm slope}}^{m,c}},
                !<
                !< where $T_{d, {\rm off}}^{m,c}$ and $T_{\rm d,  slope}^{m,c}$ can be defined by the user for each component in the molfit. If
                !< $T_{d, {\rm off}}^{m,c}$ and $T_{\rm d,  slope}^{m,c}$ are not defined for a certain component, we assume
                !< $T_d^{m,c} (\nu) \equiv T_{\rm ex}^{m,c} (\nu)$ for all components.
                !<
                !<
                !< CopymyXCLASSParameter(2, c)      T_ex
                !< CopymyXCLASSParameter(6, c)      T_d
                !< CopymyXCLASSParameter(7, c)      T_Slope
                !<
                T_d = 0.d0
                j_td = 0.d0
                if (tdFlag(c)) then
                    T_d = CopymyXCLASSParameter(2, c) + CopymyXCLASSParameter(6, c) * (freq / StartFrequency(l))**CopymyXCLASSParameter(7, c)


                    !< determine brightness for dust temperature
                    if (dabs((dexp(freq_t / T_d) - 1.d0)) > tiny(1.d0)) then
                        j_td = freq_t / (dexp(freq_t / T_d) - 1.d0)
                    endif
                else
                    j_td = j_tk
                    T_d = CopymyXCLASSParameter(2, c)
                endif

                ! Debug:
                ! write(9233, *) freq, c, T_d, tau_d, (1.d0 / huge(1.d0)), tiny(1.d0)


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< Calculate source function using eq. (7)
                !<
                !<
                !< The expression $S^{m,c}(\nu)$ describes the source function and is according to Kirchhoff's law of thermal radiation given by
                !<
                !<      S^{m,c}(\nu) &= \frac{\epsilon_l^{m,c}(\nu) + \epsilon_d^{m,c}(\nu)}{\kappa_l^{m,c}(\nu) + \kappa_d^{m,c}(\nu)}\\
                !<                   &= \frac{\kappa_l^{m,c}(\nu) \, J(T_\mathrm{ex}^{m,c}, \nu) + \kappa_d^{m,c}(\nu) \, J(T_d^{m,c}, \nu)}
                !<                           {\kappa_l^{m,c}(\nu) + \kappa_d^{m,c}(\nu)}\\
                !<                   &= \frac{\tau_l^{m,c}(\nu) \, J(T_\mathrm{ex}^{m,c}, \nu) + \gamma \, \tau_d^{m,c}(\nu) \, J(T_d^{m,c}, \nu)}
                !<                           {\tau_l^{m,c}(\nu) + \tau_d^{m,c}(\nu)},
                !<
                !< where $\epsilon_{l,d}(\nu)$ and $\kappa_{l,d}(\nu)$ are the emission and absorption coefficients for line and dust, respectively.
                !< Additionally, the optical depth is given by $\tau(\nu) = \int \kappa(\nu) \, ds = \kappa(\nu) \, s$. This assumes that molecules
                !< and dust are well mixed, i.e.\ it would not be correct if the molecule exists only in part of the cloud, but the dust everywhere.
                !< Since the line form function $\phi$ is part of $\tau_l$, outside the line this reduces to $J(T_d^{m,c}, \nu)$, as it should.
                !< $\tau_l^{m,c}(\nu)$ is actually the sum over all opacities of all lines of molecule $m$ and component $c$ at the given frequency
                !< point $\nu$.
                !<
                if (tbFlag(l)) then
                    SourceFunc = j_tk
                else
                    if (dabs(tau_total) > tiny(1.d0)) then                                  !< prevent division by zero
                        SourceFunc = (tau_l * j_tk + tau_d * j_td) / tau_total
                    else
                        SourceFunc = 0.d0
                    endif
                endif


                !<########################################################################################################################################
                !< Only used by myXCLASS function!
                !<########################################################################################################################################
                !< allocate variables for output variables
                if (LogFlag) then
                    if (AllOutputFilesFlag) then
                        Do ic = 1, IsoCounter                                               !< loop over isotopologues


                            !< define arrays for intensity array calculation
                            if (dabs(TauHelperArray1(ic) + tau_d) > tiny(1.d0)) then        !< prevent division by zero
                                TauHelperArray2(ic) = (TauHelperArray1(ic) * j_tk + tau_d * j_td) / (TauHelperArray1(ic) + tau_d)
                            else
                                TauHelperArray2(ic) = 0.d0
                            endif


                            !< define arrays for integrated line calculation
                            Do t = 1, wm
                                !< IntegHelperArray1(ic, t) = tau_t         (for a single transition)
                                !< IntegHelperArray2(ic, t) = SourceFunc    (for a single transition)
                                if (dabs(IntegHelperArray1(ic, t) + tau_d) > tiny(1.d0)) then   !< prevent division by zero
                                    IntegHelperArray2(ic, t) = (IntegHelperArray1(ic, t) * j_tk + tau_d * j_td) / (IntegHelperArray1(ic, t) + tau_d)
                                else
                                    IntegHelperArray2(ic, t) = 0.d0
                                endif
                            end Do
                        end Do
                    endif
                endif
                !<########################################################################################################################################
                !<########################################################################################################################################


                !<========================================================================================================================================
                !< determine background temperature
                !<
                !< The background temperature $J(T_{\rm bg}, \nu)$ is parametrized as
                !<
                !<      I_{\rm bg} (\nu) = T_{\rm bg} * \left(\frac{\nu}{\nu_{\rm min}} \right)^{T_{\rm slope}} + J_\mathrm{CMB}
                !<
                !< to allow the user to define the continuum contribution for each frequency range, individually. Here, $\delta_{T_{\rm bg}, 0}$
                !< represents the Kronecker delta, i.e.\ $I_{\rm bg} (\nu) \equiv J_\mathrm{CMB}$ for $T_{\rm bg} \equiv 0$. Additionally,
                !< $\nu_{\rm min}$ indicates the lowest frequency of a given frequency range. $T_{\rm bg}$ and $T_{\rm slope}$, defined by the
                !< user, describe the background continuum temperature and the temperature slope, respectively. The treatment of the dust is
                !< not entirely self-consistent. To amend that, we would need to define a more precise source model, which we consider to be
                !< outside the scope of this effort.
                !<
                j_cb = freq_t / (dexp(freq_t / Tcbg) - 1.d0)
                if (BackgroundTemperatureRange(l) == 0.d0) then
                    j_back = j_cb
                else
                    j_back = BackgroundTemperatureRange(l) * (freq / StartFrequency(l))**TemperatureSlopeRange(l) + j_cb
                endif

                ! Debug:
                !    if (dabs(Obsfreq - debug_freq) < 1.d-1) then
                !        write(DebugChannel, *) "j_back = ", j_back
                !        write(DebugChannel, *) "j_cb   = ", j_cb
                !    endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !<
                !< Emission:
                !<
                !<----------------------------------------------------------------------------------------------------------------------------------------
                if (CopymyXCLASSParameter(11, c) == 1.d0) then                              !< determine intensity for emission


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< Calculate intensity using eq. (1)
                    !<
                    !<
                    !< We assume that emission components do not interact with each other radiatively, i.e.\ one emission layer is not influenced by
                    !< the others. But the emission layers may overlap to offer the possibility to model sources consisting of several molecules and
                    !< compounds. The solution of the radiative transfer equation for emission layers is
                    !<
                    !<      T_{\rm mb}^{\rm em}(\nu) &= \left[\sum_m \sum_c \left[\eta \left(\theta^{m,c}\right) S^{m,c}(\nu) 
                    !<                                  \left(1 - e^{-\tau_{\rm total}^{m,c}(\nu)}\right)
                    !<                                  \left( e^{-\tau_{\rm total}^{m,c}(\nu)} - 1\right) I_{\rm bg} (\nu) \right] \right] \\
                    !<                                  + I_{\rm bg} (\nu) - J_\mathrm{CMB},
                    !<
                    !< where the sums go over the indices $m$ for molecule, and $c$ for (emission) component, respectively. The last term
                    !< $J_\mathrm{CMB}$ describes the OFF position for single dish observations, where we have an intensity caused
                    !< by the cosmic background $J_\mathrm{CMB}$. For interferometric observationss, the contribution of the cosmic background
                    !< is filtered out and has to be subtracted as well.
                    !<
                    LocalIntensity = (eta_source * (SourceFunc * (1.d0 - dexp(-tau_total)) + j_back * dexp(-tau_total) - j_back)) &
                                     + (1.d0 / NumEmissionComp) * (j_back - j_cb)

                    ! Debug:
                    !    if (dabs(freq - debug_freq) < 1.d-1) then
                    !        print*,' '
                    !        print*,'l, ae, c, MoleculeIndexLoop = ', l, ae, c, MoleculeIndexLoop
                    !        print*,'freq = ', freq
                    !        print*,'CopymyXCLASSParameter(1, c) = ', CopymyXCLASSParameter(1, c)
                    !        print*,'CopymyXCLASSParameter(2, c) = ', CopymyXCLASSParameter(2, c)
                    !        print*,'CopymyXCLASSParameter(3, c) = ', CopymyXCLASSParameter(3, c)
                    !        print*,'CopymyXCLASSParameter(4, c) = ', CopymyXCLASSParameter(4, c)
                    !        print*,'CopymyXCLASSParameter(5, c) = ', CopymyXCLASSParameter(5, c)
                    !        print*,'CopymyXCLASSParameter(6, c) = ', CopymyXCLASSParameter(6, c)
                    !        print*,'CopymyXCLASSParameter(7, c) = ', CopymyXCLASSParameter(7, c)
                    !        print*,'CopymyXCLASSParameter(8, c) = ', CopymyXCLASSParameter(8, c)
                    !        print*,'CopymyXCLASSParameter(9, c) = ', CopymyXCLASSParameter(9, c)
                    !        print*,'CopymyXCLASSParameter(10, c) = ', CopymyXCLASSParameter(10, c)
                    !        print*,'CopymyXCLASSParameter(11, c) = ', CopymyXCLASSParameter(11, c)
                    !        print*,'eta_source = ', eta_source
                    !        print*,'SourceFunc = ', SourceFunc
                    !        print*,'j_back     = ', j_back
                    !        print*,'InterFlag(l) = ', InterFlag(l)
                    !        print*,'tbFlag(l) = ', tbFlag(l)
                    !        print*,'tdFlag(c) = ', tdFlag(c)
                    !        print*,'tau_l, j_tk, tau_l * j_tk = ', tau_l, j_tk, tau_l * j_tk
                    !        print*,'tau_d, j_td, tau_d * j_td = ', tau_d, j_td, tau_d * j_td
                    !        print*,'tau_total, dexp(-tau_total) = ', tau_total, dexp(-tau_total)
                    !        print*,' '
                    !        print*,'eta_source * SourceFunc * (1.d0 - dexp(-tau_total)) = ', eta_source * SourceFunc * (1.d0 - dexp(-tau_total))
                    !        print*,'eta_source * j_back * dexp(-tau_total) = ', eta_source * j_back * dexp(-tau_total)
                    !        print*,'LocalIntensity = ', LocalIntensity
                    !    endif


                    !<####################################################################################################################################
                    !< Only used by myXCLASS function!
                    !<####################################################################################################################################
                    !< calculate intensity
                    if (LogFlag) then
                        if (AllOutputFilesFlag) then
                            FirstIndex = DataPointIndexFreqRange(l, 1)                      !< get index of first freq. point in 'ObservationalDataList'
                            LastIndex = DataPointIndexFreqRange(l, 2)                       !< get index of last freq. point in 'ObservationalDataList'
                            FreqIndex = minloc(dabs(freq - ObservationalDataList(FirstIndex:LastIndex, 1)), dim=1)
                            Do ic = 1, IsoCounter                                           !< loop over isotopologues


                                !< define arrays for intensity array calculation
                                LocalIntArray(ic) = (eta_source * (TauHelperArray2(ic) * (1.d0 - dexp(-(TauHelperArray1(ic) + tau_d))) &
                                                     + j_back * dexp(-(TauHelperArray1(ic) + tau_d)) - j_back)) &
                                                     + (1.d0 / NumEmissionComp) * (j_back - j_cb)


                                !< define arrays for integrated line calculation
                                Do t = 1, wm
                                    !< IntegHelperArray1(ic, t) = tau_total = tau_l             (for a single transition)
                                    !< IntegHelperArray2(ic, t) = SourceFunc                    (for a single transition)


                                    !< calculate intensity
                                    IntegHelperArray1(ic, t) = IntegHelperArray1(ic, t) + tau_d
                                    val = (eta_source * (IntegHelperArray2(ic, t) * (1.d0 - dexp(-IntegHelperArray1(ic, t))) &
                                                        + j_back * dexp(-IntegHelperArray1(ic, t)) - j_back)) + (1.d0 / NumEmissionComp) * (j_back - j_cb)


                                    !< integrate peak
                                    if (FreqIndex /= FirstIndex) then
                                        IntensityPerPeak(l, c, ic, t) = IntensityPerPeak(l, c, ic, t) + (freq - ObservationalDataList(FreqIndex - 1, 1)) &
                                                                                                        * (PrevIntPerPeak(ic, t) + val) / 2.d0
                                    endif
                                    PrevIntPerPeak(ic, t) = val
                                end Do
                            end Do
                        endif
                    endif
                    !<####################################################################################################################################
                    !<####################################################################################################################################


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !<
                !< foreground:
                !<
                !<----------------------------------------------------------------------------------------------------------------------------------------
                else                                                                        !< foreground loop here
                    Ibg = TotalLocalIntensity
                    if (EmissionCompCounter == 0 .and. AbsorptionCompCounter == 1) then
                        Ibg = j_back
                        if (.not. InterFlag(l)) then
                            Ibg = Ibg - j_cb
                        endif
                    endif


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< Calculate intensity using eq. (16)
                    !<
                    !<
                    !< In contrast to emission layers, absorbing components may interact with each other, where absorption takes places only, if the
                    !< excitation temperature for the absorbing layer is lower than the temperature of the background. 
                    !<
                    !< foreground components have to be considered in an iterative manner. The solution of the radiative transfer equation for
                    !< foreground layers can be expressed as
                    !<
                    !<      T_{\rm mb}^{\rm abs}(\nu)_{m,c=1} &= \eta \left(\theta^{m,c=1} \right) \\
                    !<                                        &  \Bigg[S^{m,c=1}(\nu) \left(1 - e^{-\tau_{\rm total}^{m,c=1}(\nu)}\right) \\
                    !<                                        & + I_{\rm bg} (\nu) e^{-\tau_{\rm total}^{m,c=1}(\nu)}\Bigg] \\
                    !<                                        & + \left(1 - \eta \left(\theta^{m,c=1} \right)\right) \, I_{\rm bg} (\nu) \\
                    !<      T_{\rm mb}^{\rm abs}(\nu)_{m,c=i} &= \eta \left(\theta^{m,c=i} \right) 
                    !<                                        &  \Big[S^{m,c=i}(\nu) \left(1 - e^{-\tau_{\rm total}^{m,c=i}(\nu)}\right) \\
                    !<                                        & + T_{\rm mb}^{\rm abs}(\nu)_{m,c=(i-1)} e^{-\tau_{\rm total}^{m,c=i}(\nu)}\Big]\\
                    !<                                        & + \left(1-\eta \left(\theta^{m,c=i} \right)\right) T_{\rm mb}^{\rm abs}(\nu)_{m,c=(i-1)},
                    !<
                    !< where $m$ indicates the index of the current molecule and $i$ represents an index running over all foreground components of
                    !< all molecules. Additionally, we assume that each absorbing component covers the whole beam, i.e.
                    !< $\eta \left(\theta^{m,c=1} \right) \equiv 1$ for all absorbing layer. So, we achieve
                    !<
                    !<      T_{\rm mb}^{\rm abs}(\nu)_{m,c=1} &= \Big[ S^{m,c=1}(\nu) \left(1 - e^{-\tau_{\rm total}^{m,c=1}(\nu)}\right) \\
                    !<                                        & + I_{\rm bg} (\nu) e^{-\tau_{\rm total}^{m,c=1}(\nu)}\Big] \\
                    !<      T_{\rm mb}^{\rm abs}(\nu)_{m,c=i} &= \Big[S^{m,c=i}(\nu) \left(1 - e^{-\tau_{\rm total}^{m,c=i}(\nu)}\right) \\
                    !<                                        & + T_{\rm mb}^{\rm abs}(\nu)_{m,c=(i-1)} e^{-\tau_{\rm total}^{m,c=i}(\nu)}\Big],
                    !<
                    !< where $I_{\rm bg} (\nu)$ describes the emission spectrum plus continuum. For absorption lines the emission by other components
                    !< is considered by first calculating the emission lines and then use this emission as new continuum for absorption lines
                    !< reflecting the fact that cold absorption layers are often found in front of hotter emission sources. The order of components
                    !< along the line of sight is defined by the occurrence of a certain foreground component in the molfit file.
                    !<
                    !<
                    LocalIntensity = SourceFunc * (1.d0 - dexp(-tau_total)) + Ibg * dexp(-tau_total)


                    !<####################################################################################################################################
                    !< Only used by myXCLASS function!
                    !<####################################################################################################################################
                    !< allocate variables for output variables
                    if (LogFlag) then
                        if (AllOutputFilesFlag) then
                            FirstIndex = DataPointIndexFreqRange(l, 1)                      !< get index of first freq. point in 'ObservationalDataList'
                            LastIndex = DataPointIndexFreqRange(l, 2)                       !< get index of last freq. point in 'ObservationalDataList'
                            FreqIndex = minloc(dabs(freq - ObservationalDataList(FirstIndex:LastIndex, 1)), dim=1)
                            Do ic = 1, IsoCounter                                           !< loop over isotopologues


                                !< define arrays for intensity array calculation
                                LocalIntArray(ic) = TauHelperArray2(ic) * (1.d0 - dexp(-(TauHelperArray1(ic) + tau_d))) &
                                                    + Ibg * dexp(-(TauHelperArray1(ic) + tau_d))


                                !< define arrays for integrated line calculation
                                Do t = 1, wm
                                    !< IntegHelperArray1(ic, t) = tau_total = tau_t  + tau_d    (for a single transition)
                                    !< IntegHelperArray2(ic, t) = SourceFunc                    (for a single transition)


                                    !< calculate intensity
                                    IntegHelperArray1(ic, t) = IntegHelperArray1(ic, t) + tau_d
                                    val = IntegHelperArray2(ic, t) * (1.d0 - dexp(-IntegHelperArray1(ic, t))) + Ibg * dexp(-IntegHelperArray1(ic, t))


                                    !< integrate peak
                                    if (FreqIndex /= FirstIndex) then
                                        IntensityPerPeak(l, c, ic, t) = IntensityPerPeak(l, c, ic, t) &
                                                                        + (freq - ObservationalDataList(FreqIndex - 1, 1)) &
                                                                          * (PrevIntPerPeak(ic, t) + val) / 2.d0
                                    endif
                                    PrevIntPerPeak(ic, t) = val
                                end Do
                            end Do
                        endif
                    endif
                    !<####################################################################################################################################
                    !<####################################################################################################################################
                endif

                ! Debug:
                !    if (dabs(freq - debug_freq) < 1.d-1) then
                !        print*,' '
                !        print*,'l, ae, c, MoleculeIndex = ', l, ae, c, MoleculeIndex
                !        print*,'freq = ', freq
                !        print*,'TotalNumMol = ', TotalNumMol
                !        print*,'CopymyXCLASSParameter(1, c) = ', CopymyXCLASSParameter(1, c)
                !        print*,'CopymyXCLASSParameter(2, c) = ', CopymyXCLASSParameter(2, c)
                !        print*,'CopymyXCLASSParameter(3, c) = ', CopymyXCLASSParameter(3, c)
                !        print*,'CopymyXCLASSParameter(4, c) = ', CopymyXCLASSParameter(4, c)
                !        print*,'CopymyXCLASSParameter(5, c) = ', CopymyXCLASSParameter(5, c)
                !        print*,'CopymyXCLASSParameter(6, c) = ', CopymyXCLASSParameter(6, c)
                !        print*,'CopymyXCLASSParameter(7, c) = ', CopymyXCLASSParameter(7, c)
                !        print*,'CopymyXCLASSParameter(8, c) = ', CopymyXCLASSParameter(8, c)
                !        print*,'CopymyXCLASSParameter(9, c) = ', CopymyXCLASSParameter(9, c)
                !        print*,'CopymyXCLASSParameter(10, c) = ', CopymyXCLASSParameter(10, c)
                !        print*,'CopymyXCLASSParameter(11, c) = ', CopymyXCLASSParameter(11, c)
                !        print*,'eta_source = ', eta_source
                !        print*,'SourceFunc = ', SourceFunc
                !        print*,'NumEmissionComp = ', NumEmissionComp
                !        print*,'j_back     = ', j_back
                !        print*,'InterFlag(l) = ', InterFlag(l)
                !        print*,'tbFlag(l) = ', tbFlag(l)
                !        print*,'tdFlag(c) = ', tdFlag(c)
                !        print*,'IsoFlag = ', IsoFlag
                !        print*,'CalcFalg = ', CalcFalg
                !        print*,'Q(c, :) = ', Q(c, :)
                !        print*,'tau_l, j_tk, tau_l * j_tk = ', tau_l, j_tk, tau_l * j_tk
                !        print*,'tau_d, j_td, tau_d * j_td = ', tau_d, j_td, tau_d * j_td
                !        print*,'tau_total, dexp(-tau_total) = ', tau_total, dexp(-tau_total)
                !        print*,' '
                !        print*,'eta_source * SourceFunc * (1.d0 - dexp(-tau_total)) = ', eta_source * SourceFunc * (1.d0 - dexp(-tau_total))
                !        print*,'(1.d0 / NumEmissionComp) * (j_back - j_cb) = ', (1.d0 / NumEmissionComp) * (j_back - j_cb)
                !        print*,'LocalIntensity = ', LocalIntensity
                !        print*,'CopyIsoRatioConversionTable(MoleculeIndex, :) = ', CopyIsoRatioConversionTable(MoleculeIndex, :)
                !        print*," "
                !        print*," "
                !        print*," "
                !    endif
            endif                                                                           !< continue here, if current comp. is not equal to ae

            return
        end subroutine DetectionEquation


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcSpectrum
        !>
        !> calculates the myXCLASS spectrum for a given parameter vector
        !>
        !>
        !> input variables:     NumberFreeParameterCopy:    number of free parameters
        !>                      ParameterVector:        parameter vector
        !>                      ModelFunctionFlag:      flag for indicating if model function values are stored or not
        !>                      NumComp:                total number of components
        !>                      TotalNumFreqRan:        total number of frequency ranges
        !>                      TotalNumTransFreq:      TotalNumTransFreq
        !>                      CompMoleculeIndex:      molecule index for each component
        !>                      myXCLASSParameterOrig:  array containing all molfit parameters for each component
        !>                      TotalNumMol:            total number of molecules
        !>                      IsoRatioConversionTable: table with iso ratios between iso master and molecule
        !>                      NumDataPoints:          total number of data points
        !>                      ThreadNumber:           current thread number
        !>
        !> output variables:    chi2Value:              chi2 value for parameter vector
        !>                      ModelFuncList:          model function values
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 31.07.2014
        !>
        subroutine ModelCalcSpectrum(NumberFreeParameterCopy, ParameterVector, ModelFunctionFlag, NumComp, TotalNumFreqRan, TotalNumTransFreq, &
                                     chi2Value, CompMoleculeIndexOrig, myXCLASSParameterOrig, TotalNumMol, IsoRatioConversionTableOrig, &
                                     NumDataPoints, ModelFuncList, ThreadNumber)

            implicit none

            !< input parameters
            integer :: NumberFreeParameterCopy                                              !< number of free parameters
            integer :: NumComp                                                              !< total number of components
            integer :: TotalNumMol                                                          !< total number of molecules
            integer :: TotalNumFreqRan                                                      !< total number of frequency ranges
            integer :: TotalNumTransFreq                                                    !< total number of transition frequencies
            integer :: NumDataPoints                                                        !< total number of data points
            integer :: ThreadNumber                                                         !< current thread number
            integer, dimension(NumComp) :: CompMoleculeIndexOrig                            !< molecule index for each component
            real*8, dimension(NumberFreeParameterCopy) :: ParameterVector                   !< parameter vector
            real*8, dimension(11, NumComp) :: myXCLASSParameterOrig                         !< array containing all molfit parameters for each component
            real*8, dimension(TotalNumMol, TotalNumMol) :: IsoRatioConversionTableOrig      !< table with iso ratios between iso master and molecule
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model func. values are stored or not

            !< output parameters
            real*8 :: chi2Value                                                             !< chi2 value for parameter vector
            real*8, dimension(NumDataPoints) :: ModelFuncList                               !< output variable: model function values

            !< working variables
            integer, parameter :: MaxFreqResampling = 100                                   !< max. number of frequency points for resampling
            integer :: fitnum, i, j, k, l, m, ae, c, t, fi, ll, kk, MoleculeIndexLoop       !< loop variables
            integer :: IndexComp                                                            !< overall component index
            integer :: varIndex                                                             !< myXCLASS variable index
            integer :: ErrChannelIndexLocal                                                 !< used for error handling
            integer :: ErrChannelLocal                                                      !< used for error handling
            integer :: MoleculeIndex                                                        !< index for molecule
            integer :: FirstMolecularDataIndex                                              !< first index for molecular data table
            integer :: LastMolecularDataIndex                                               !< last index for molecular data table
            integer :: FirstIndex                                                           !< index for first freq. data point
            integer :: LastIndex                                                            !< index for last freq. data point
            integer :: FreqIndex                                                            !< index for frequency point
            integer :: IsoCounter, fff                                                      !< working variable: counter for isotopologue
            integer :: n                                                                    !< current comp. index
            integer :: ncomp                                                                !< total number of comp. of curr. mol.
            integer :: etaMaxIndex                                                          !< index of emission comp. with the largest beam fill. factor
            integer :: NumEmissionComp                                                      !< total number of emission components
            integer :: EmissionCompCounter                                                  !< counter for core components
            integer :: AbsorptionCompCounter                                                !< counter for foreground components
            integer :: fiFirst                                                              !< first index of frequency list
            integer :: LastDSTransFreqIndex                                                 !< index of last doppler-shifted freq.
            integer :: NumFreqList                                                          !< length of frequency list
            integer :: NumDopplerTransFreq                                                  !< number of doppler-shifted frequencies
            integer :: NumInsertedPoints                                                    !< used for resampling: number of inserted points
            integer :: MaxNumIso                                                            !< working variable max. number of isotopologues per component
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            integer, dimension(NumComp) :: CopyCompMoleculeIndex                            !< local copy of CompMoleculeIndexOrig
            integer, dimension(TotalNumFreqRan) :: NumTransFreqPerFreqRange                 !< number of doppler-shifted trans. freq. per freq. range
            real*8 :: x0, x1, f0, f1                                                        !< working variables for integration
            real*8 :: Obsfreq                                                               !< observed frequency
            real*8 :: sig2i                                                                 !< sigma factor (for weighting of obs. data)
            real*8 :: freq                                                                  !< frequency for index 'FreqIndex'
            real*8 :: val                                                                   !< value for integration
            real*8 :: j_back                                                                !< brightness for background temperatures
            real*8 :: j_cb                                                                  !< brightness for cosmic microwave background temperature
            real*8 :: MinDistance, DeltaFreq                                                !< used for resampling
            real*8 :: sigma, MinSigma, MinSigmaHalf                                         !< used for resampling check
            real*8 :: Temperature                                                           !< temperature
            real*8 :: FreqChannelFirst                                                      !< first frequency of current channel
            real*8 :: FreqChannelLast                                                       !< last frequency of current channel
            real*8 :: ChannelWidth                                                          !< channel width
            real*8 :: beam_size                                                             !< beam size (eqn. 2 - 3)
            real*8 :: tau_l                                                                 !< optical depth (for myXCLASS only)
            real*8 :: Td                                                                    !< dust temperature (for myXCLASS only)
            real*8 :: LocalIntensity                                                        !< intensities for curret comp.
            real*8 :: ChannelIntensity, ChannelInt                                          !< calc. intensity for current channel
            real*8 :: TotalLocalIntensity                                                   !< total calc. intensity for current channel
            real*8 :: etaMax                                                                !< max. beam filling factor of all emission components
            real*8 :: SumEtaEmission                                                        !< sum of beam filling factor of all emission components
            real*8, dimension(NumComp, TotalNumMol) :: Q                                    !< partition function at the rotation temperatures
            real*8, dimension(NumComp) :: LastChannelInt                                    !< right intensity of last channel for each comp.
            real*8, dimension(MaxFreqResampling) :: FreqList, CopyFreqList                  !< list (and copy) of frequencies for integration
            real*8, dimension(MaxFreqResampling) :: IntFreqList                             !< list of intensities for frequency list
            real*8, dimension(11, NumComp) :: CopymyXCLASSParameter                         !< local copy of myXCLASSParameterOrig
            real*8, dimension(TotalNumMol, TotalNumMol) :: CopyIsoRatioConversionTable      !< local copy of IsoRatioConversionTableOrig
            real*8, dimension(TotalNumTransFreq) :: DopplerShiftedTransFreq                 !< list of all doppler-shifted transition frequencies
            real*8, dimension(TotalNumTransFreq) :: CopyDopplerShiftedTransFreq             !< copy of DopplerShiftedTransFreq array
            real*8, allocatable, dimension(:, :, :) :: IntPerCompArray                      !< array containing intensities for each comp. (myXCLASS only)
            real*8, allocatable, dimension(:, :, :) :: TauPerCompArray                      !< array containing tau for each comp. (myXCLASS only)
            real*8, allocatable, dimension(:, :) :: TdPerCompArray                          !< array containing T_d for each comp. (myXCLASS only)
            character(len=30) :: numberstring1, numberstring2                               !< used for integer number to string conversion
            character(len=40) :: CurrMoleculeName                                           !< name of molecule
            character(len=40) :: ModifiedMoleculeName                                       !< name of optical depth file
            logical :: IncludeTransFreqFlag                                                 !< flag indicating that current trans. freq. is considered
            logical :: CalcFalg                                                             !< used for calculation of isotopoluges
            logical, dimension(NumComp) :: PrevChannelTransFlag                             !< flag indicating trans. freq. in previous channel
            logical, dimension(NumDataPoints) :: FirstDataFlag                              !< indicates if first freq. of channel has to be calculated
                                                                                            !< or not
            logical, dimension(NumComp) :: GeneralSamplingFlag                              !< indicates if a finer sampling is necessary for each comp.


            ! Debug:
            !    CALL CPU_TIME(tt1)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize some variables
            chi2Value = 0.d0
            ModelFuncList = 0.d0
            FirstDataFlag = .false.
            LastChannelInt = 0.d0
            FreqChannelFirst = 0.d0
            FreqChannelLast = 0.d0
            ChannelWidth = 1.d0
            NumDopplerTransFreq = 0
            fiFirst = 1
            MaxNumIso = 1
            MinSigma = 0.d0
            MinSigmaHalf = 0.d0
            IncludeTransFreqFlag = .true.
            GeneralSamplingFlag = .false.
            DopplerShiftedTransFreq = 0.d0
            CopyDopplerShiftedTransFreq = 0.d0
            PrevChannelTransFlag = .false.
            j_back = 0.d0                                                                   !< initialize j_back variable
            j_cb = 0.d0                                                                     !< initialize j_cb variable
            if (ThreadNumber < 0) then
                print '(2x, "Error in subroutine ModelCalcSpectrum:")'
                print '(4x, "Thread number negativ!!")'
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< make private copies of variables CompMoleculeIndexOrig, myXCLASSParameterOrig, and IsoRatioConversionTableOrig
            CopyCompMoleculeIndex = CompMoleculeIndexOrig
            CopymyXCLASSParameter = myXCLASSParameterOrig
            if (IsoFlag) then
                CopyIsoRatioConversionTable = IsoRatioConversionTableOrig
            endif


            !<============================================================================================================================================
            !< update myXCLASS parameter vector with new parameter values
            !< CopymyXCLASSParameter(1, :)      source size
            !< CopymyXCLASSParameter(2, :)      T_rot
            !< CopymyXCLASSParameter(3, :)      N_tot
            !< CopymyXCLASSParameter(4, :)      V_width
            !< CopymyXCLASSParameter(5, :)      V_off
            !< CopymyXCLASSParameter(6, :)      T_Back
            !< CopymyXCLASSParameter(7, :)      T_Slope
            !< CopymyXCLASSParameter(8, :)      nH_column
            !< CopymyXCLASSParameter(9, :)      beta
            !< CopymyXCLASSParameter(10, :)     kappa
            !< CopymyXCLASSParameter(11, :)     core / foreground flag
            Do fitnum = 1, NumberFreeParameterCopy                                          !< loop over all free parameters


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< update molfit file parameters
                IndexComp = ConversionTableMAGIXmyXCLASSParam(fitnum, 1)                    !< get current overall component index
                varIndex = ConversionTableMAGIXmyXCLASSParam(fitnum, 2)                     !< get current myXCLASS variable index
                if (IndexComp > 0 .and. varIndex > 0 .and. varIndex < 11) then              !< check, if current free param. corresponds to molfit param.

                    ! Debug:
                    ! print*,'varIndex, IndexComp = ', varIndex, IndexComp
                    ! print*,'CopymyXCLASSParameter(:, IndexComp) = ', CopymyXCLASSParameter(varIndex, IndexComp)
                    ! print*,'ParameterVector(fitnum) = ', ParameterVector(fitnum)


                    CopymyXCLASSParameter(varIndex, IndexComp) = ParameterVector(fitnum)


                    !< check for overflow of N_tot
                    if (varIndex == 3) then
                        if (CopymyXCLASSParameter(3, IndexComp) > 300.d0) then
                            j = CopyCompMoleculeIndex(IndexComp)
                            k = IndexComp - sum(NumberComponentsPerMolecule(:(j - 1)))
                            Do ErrChannelIndexLocal = 1, 1
                                ErrChannelLocal = AllErrChannels(ErrChannelIndexLocal)
                                write(ErrChannelLocal, '(" ")')
                                write(ErrChannelLocal, '(" ")')
                                write(ErrChannelLocal, '(2x, "Error in subroutine ModelCalcSpectrum:")')
                                if (j /= 0) then
                                    write(ErrChannelLocal, '(4x, "The log10 value of column density for component ", I4, " of molecule ", A, " ")') k, &
                                                                                                                    trim(adjustl(MoleculeNames(j)))
                                else
                                    write(ErrChannelLocal, '(4x, "The log10 value of column density for component ", I4, " ")') k
                                endif
                                write(ErrChannelLocal, '(4x, "causes an overflow (to large number)!")')
                                write(ErrChannelLocal, '(4x, "Set log10 value for this column density to 300!")')
                                write(ErrChannelLocal, '(" ")')
                                write(ErrChannelLocal, '(" ")')
                            end Do
                            CopymyXCLASSParameter(3, IndexComp) = 300.d0
                        endif


                        !< convert log10 value back to linear value
                        CopymyXCLASSParameter(3, IndexComp) = 10.d0**CopymyXCLASSParameter(3, IndexComp)


                    !< check for V_width < 0.01
                    elseif (varIndex == 4) then
                        if (CopymyXCLASSParameter(4, IndexComp) < 0.0001) then
                            CopymyXCLASSParameter(4, IndexComp) = 0.0001
                        endif
                    endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< update iso ratio table with new iso ratios
                elseif (IsoFlag) then
                    Do m = 1, TotalNumberOfMolecules
                        Do k = (m + 1), TotalNumberOfMolecules
                            if (IsoNfitConversionTable(m, k, 1) == fitnum) then


                                !< continue here, if iso master and isotopologues are exchanged, i.e. ratio = 1.0 / ratio
                                if (IsoNfitConversionTable(m, k, 2) < 0) then
                                    CopyIsoRatioConversionTable(m, k) = 1.d0 / ParameterVector(fitnum)
                                else
                                    CopyIsoRatioConversionTable(m, k) = ParameterVector(fitnum)
                                endif
                                if (dabs(CopyIsoRatioConversionTable(m, k)) < 1.e-30) then
                                    CopyIsoRatioConversionTable(k, m) = 1.0
                                else
                                    CopyIsoRatioConversionTable(k, m) = 1.d0 / CopyIsoRatioConversionTable(m, k)
                                endif

                                ! Debug:
                                !    print*,' '
                                !    print*,' '
                                !    print*, m, adjustl(MoleculeNames(m))
                                !    print*, k, adjustl(MoleculeNames(k))
                                !    print*, CopyIsoRatioConversionTable(k, m)
                                !    print*, CopyIsoRatioConversionTable(m, k)
                            endif
                        end Do                                                              !< loop over k
                    end Do                                                                  !< loop over m
                endif
            end Do                                                                          !< loop over fitnum


            !<============================================================================================================================================
            !< check, if some iso ratios were globally defined and update corresponding iso ratios
            if (IsoFlag .and. NumberOfGlobalISORatios > 0) then
                Do m = 1, TotalNumberOfMolecules
                    Do k = (m + 1), TotalNumberOfMolecules
                        j = abs(IsoNfitConversionTable(m, k, 2))                            !< get j index for iso ratio

                        ! Debug:
                        ! print*,'m, k, j = ', m, k, j

                        if (j > 0) then


                            !< determine if globally defined iso ratios are involved in the current iso ratio
                            if (GlobalIsoRatioParameter(j, 1, 1) > 0) then
                                val = 1.d0
                                !val = CopyIsoRatioConversionTable(m, k)
                                Do fi = 1, NumberOfGlobalISORatios
                                    l = GlobalIsoRatioParameter(j, fi, 1)                   !< get j index for globally defined iso ratio
                                    if (l == 0) then
                                        exit
                                    endif
                                    t = GlobalIsoRatioParameter(j, fi, 2)                   !< get number of replications
                                    fitnum = int(IsoRatio(l, 2))                            !< get index for fitted variable

                                    ! Debug:
                                    !    print*," "
                                    !    print*," "
                                    !    print*,"fi, l, t, fitnum = ", fi, l, t, fitnum
                                    !    print*,"IsoMolecule(l) = ", trim(adjustl(IsoMolecule(l)))
                                    !    print*,"IsoMasters(l)  = ", trim(adjustl(IsoMasters(l)))
                                    !    print*,"IsoRatio(l, 1) = ", IsoRatio(l, 1)

                                    if (fitnum == 0) then                                   !< is globally defined ratio fitted
                                        val = val * (IsoRatio(l, 1)**float(t))              !< calculated new ratio with non-fitted iso ratio
                                    else
                                        val = val * (float(t) * ParameterVector(fitnum))    !< calculated new ratio with fitted iso ratio

                                        ! Debug:
                                        ! print*,"m, k, fitnum, ParameterVector(fitnum) = ", m, k, fitnum, ParameterVector(fitnum)
                                    endif
                                end Do


                                !< is current iso ratio inverted
                                if (IsoNfitConversionTable(m, k, 2) < 0) then
                                    if (dabs(val) < 1.e-30) then
                                        val = 1.d0
                                    else
                                        val = 1.d0 / val
                                    endif
                                endif
                                val = val * CopyIsoRatioConversionTable(m, k)


                                !< update entry in current working copy of IsoRatioConversionTable
                                CopyIsoRatioConversionTable(m, k) = val
                                if (dabs(val) < 1.e-30) then
                                    CopyIsoRatioConversionTable(k, m) = 1.0
                                else
                                    CopyIsoRatioConversionTable(k, m) = 1.d0 / val
                                endif

                                ! Debug:
                                !    print*,"IsoRatioConversionTableOrig(m, k) = ", IsoRatioConversionTableOrig(m, k)
                                !    print*,"m,k,val = ", val
                            endif
                        endif

                        ! Debug:
                        !    if (CopyIsoRatioConversionTable(m, k) == 0.d0) then
                        !        write(23424, '("     .     ", $)')
                        !    else
                        !        write(23424, '(1x, ES10.2, $)') CopyIsoRatioConversionTable(m, k)
                        !    endif
                        !    if (IsoNfitConversionTable(m, k, 2) == 0.d0) then
                        !        write(23425, '("     .     ", $)')
                        !    else
                        !        write(23425, '(1x, I10, $)') IsoNfitConversionTable(m, k, 2)
                        !    endif
                    end Do                                                                  !< loop over k

                    ! Debug:
                    !    write(23424, '(" ")')
                    !    write(23425, '(" ")')
                end Do                                                                      !< loop over m

                ! Debug:
                !    write(23424, '(" ")')
                !    write(23424, '(" ")')
                !    write(23425, '(" ")')
                !    write(23425, '(" ")')
            endif

            ! Debug:
            !    print*," "
            !    print*," "
            !    Do i = 1, TotalNumberComponents
            !        j = CopyCompMoleculeIndex(i)
            !        print*,"MoleculeNames = ", i, trim(adjustl(MoleculeNames(j)))
            !        print*,"CopymyXCLASSParameter(:, i) = ", CopymyXCLASSParameter(:, i)
            !    end Do
            !    Do j = 1, TotalNumberOfMolecules
            !        print*,j
            !        print*,">>", trim(adjustl(MoleculeNames(j))), "<<"
            !    end Do


            !<============================================================================================================================================
            !< determine partiton functions for all rotation temperatures and all doppler-shifted transition frequencies
            Q = 0.d0
            NumTransFreqPerFreqRange = 0
            Do c = 1, TotalNumberComponents                                                 !< loop over all components
                Temperature = CopymyXCLASSParameter(2, c)                                   !< get excitation temperatures


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< check excitation temperature
                if (Temperature <= 0.d0) then
                    Temperature = 10.d0**TempLow
                    j = CopyCompMoleculeIndex(c)
                    k = c - sum(NumberComponentsPerMolecule(:(j - 1)))
                    Do ErrChannelIndexLocal = 1, 1
                        ErrChannelLocal = AllErrChannels(ErrChannelIndexLocal)
                        write(ErrChannelLocal, '(" ")')
                        write(ErrChannelLocal, '(" ")')
                        write(ErrChannelLocal, '(2x, "Error in subroutine ModelCalcSpectrum!")')
                        if (j /= 0) then
                            write(ErrChannelLocal, '(4x, "The excitation temperature for component ", I4, " of molecule ", A, " ")') k, &
                                                                                                     trim(adjustl(MoleculeNames(j)))
                        else
                            write(ErrChannelLocal, '(4x, "The excitation temperature for component ", I4, " ")') k
                        endif
                        write(ErrChannelLocal, '(4x, "is less than or equal to zero!")')
                        write(ErrChannelLocal, '(" ")')
                        write(ErrChannelLocal, '(" ")')
                        write(ErrChannelLocal, '(4x, "Set temperature to ", F25.15)') Temperature
                        write(ErrChannelLocal, '(" ")')
                    end Do
                endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< get partition function at temperature, for molecule and if used for isotopologues as well
                MoleculeIndex = CopyCompMoleculeIndex(c)                                    !< get molecule index
                l = MoleculeIndex
                if (IsoFlag) then
                    Do m = l, TotalNumberOfMolecules
                        if (CopyIsoRatioConversionTable(l, m) /= 0.d0) then                 !< is a iso ratio defined for mth molecule?
                            call InterpolateQ(Q(c, m), Temperature, m)                      !< get partition function at temperature
                        endif
                    end Do
                else
                    call InterpolateQ(Q(c, l), Temperature, MoleculeIndex)                  !< get partition function at temperature
                endif
            end Do


            !<############################################################################################################################################
            !< Only used by myXCLASS function!
            !<############################################################################################################################################
            !< allocate variables for output variables
            if (LogFlag) then
                if (AllOutputFilesFlag) then


                    !< determine max. number of isotopologues per component
                    if (IsoFlag) then
                        MaxNumIso = 0
                        Do l = 1, TotalNumberOfMolecules
                            k = 0
                            Do m = 1, TotalNumberOfMolecules
                                if (CopyIsoRatioConversionTable(l, m) /= 0.d0) then         !< is a iso ratio defined for mth molecule?
                                    k = k + 1
                                endif
                            end Do
                            if (MaxNumIso < k) then
                                MaxNumIso = k
                            endif
                        end Do
                    else
                        MaxNumIso = 1
                    endif


                    !< allocate variables for output variables
                    i = NumDataPoints
                    j = TotalNumberComponents
                    if (allocated(IntPerCompArray)) then
                        deallocate(IntPerCompArray, TauPerCompArray, TdPerCompArray, TauHelperArray1, TauHelperArray2, LocalIntArray, &
                                   stat = deallocstatus)
                        if (deallocstatus /= 0) then
                            Do ErrChannelIndex = 1, 1
                                ErrChannel = AllErrChannels(ErrChannelIndex)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(1x,"Error in subroutine ModelCalcSpectrum!")')
                                write(ErrChannel, '(3x,"Cannot deallocate variables IntPerCompArray, TempPartFunc, TdPerCompArray, and TauHelperArray!")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Please restart the program!")')
                            end Do
                            stop 'Program aborted!'
                        endif
                    endif
                    allocate(IntPerCompArray(i, j, MaxNumIso), TauPerCompArray(i, j, MaxNumIso), TdPerCompArray(i, j), TauHelperArray1(MaxNumIso), &
                             TauHelperArray2(MaxNumIso), LocalIntArray(MaxNumIso), stat = allocstatus)
                    if (allocstatus /= 0) then
                        Do ErrChannelIndex = 1, 1
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(1x,"Error in subroutine ModelCalcSpectrum!")')
                            write(ErrChannel, '(3x,"Can not allocate variables IntPerCompArray, TempPartFunc, TdPerCompArray, and TauHelperArray1!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(3x,"Please restart the program!")')
                        end Do
                        stop 'Program aborted!'
                    endif
                    IntPerCompArray = 0.d0
                    TauPerCompArray = 0.d0
                    TdPerCompArray = 0.d0
                    TauHelperArray1 = 0.d0
                    TauHelperArray2 = 0.d0
                    LocalIntArray = 0.d0
                endif
            endif
            !<############################################################################################################################################
            !<############################################################################################################################################


            !< determine number of emission components
            NumEmissionComp = count(CopymyXCLASSParameter(11, :) == 1)                      !< determine number of emission components

            ! Debug:
            ! print*,'NumEmissionComp = ', NumEmissionComp


            !<============================================================================================================================================
            !< start loop over molecules, frequency ranges, components, transitions
            Do l = 1, TotalNumberOfFrequencyRanges                                          !< loop over all frequency ranges
                i = SpectrumIndexForFreqRange(l)                                            !< get spectrum index
                FirstIndex = DataPointIndexFreqRange(l, 1)                                  !< get index of first freq. point in 'ObservationalDataList'
                LastIndex = DataPointIndexFreqRange(l, 2)                                   !< get index of last freq. point in 'ObservationalDataList'
                LastDSTransFreqIndex = 1                                                    !< reset index for doppler-shifted trans. freq.

                ! Debug:
                ! print*,"l = ", l
                ! print*,"StartFrequency(l) = ", StartFrequency(l)
                ! print*,"EndFrequency(l)   = ", EndFrequency(l)
                ! print*,"StepFrequency(l)  = ", StepFrequency(l)
                ! print*,"FirstIndex = ", FirstIndex
                ! print*,"LastIndex  = ", LastIndex


                !<########################################################################################################################################
                !< Only used by myXCLASS function!
                !<########################################################################################################################################
                !< allocate variables for output variables
                if (LogFlag) then
                    if (AllOutputFilesFlag) then
                        i = NumDataPoints
                        j = TotalNumberComponents
                        k = maxval(NumEntriesRanges(l, :))
                        if (allocated(IntensityPerPeak)) then
                            deallocate(IntensityPerPeak, PrevIntPerPeak, IntegHelperArray1, IntegHelperArray2, stat = deallocstatus)
                            if (deallocstatus /= 0) then
                                Do ErrChannelIndex = 1, 1
                                    ErrChannel = AllErrChannels(ErrChannelIndex)
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(1x,"Error in subroutine ModelCalcSpectrum!")')
                                    write(ErrChannel, '(3x,"Can not deallocate variables IntensityPerPeak, PrevIntPerPeak, IntegHelperArray1, and IntegHelperArray2!")')
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(3x,"Please restart the program!")')
                                end Do
                                stop 'Program aborted!'
                            endif
                        endif
                        allocate(IntensityPerPeak(TotalNumberOfFrequencyRanges, j, MaxNumIso, k), PrevIntPerPeak(MaxNumIso, k), &
                                 IntegHelperArray1(MaxNumIso, k), IntegHelperArray2(MaxNumIso, k), stat = allocstatus)
                        if (allocstatus /= 0) then
                            Do ErrChannelIndex = 1, 1
                                ErrChannel = AllErrChannels(ErrChannelIndex)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(1x,"Error in subroutine ModelCalcSpectrum!")')
                                write(ErrChannel, '(3x,"Can not allocate variables IntensityPerPeak, PrevIntPerPeak, IntegHelperArray1, and IntegHelperArray2!")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Please restart the program!")')
                            end Do
                            stop 'Program aborted!'
                        endif
                        IntensityPerPeak = 0.d0
                        PrevIntPerPeak = 0.d0
                        IntegHelperArray1 = 0.d0
                        IntegHelperArray2 = 0.d0
                    endif
                endif
                !<############################################################################################################################################
                !<############################################################################################################################################


                !<========================================================================================================================================
                !< start loop over frequencies
                Do FreqIndex = FirstIndex, LastIndex                                        !< loop over all frequency points in the current freq. range
                    Obsfreq = ObservationalDataList(FreqIndex, 1)                           !< get frequency
                    FreqList = 0.d0                                                         !< reset frequency list

                    ! Debug:
                    ! print*,"l, Obsfreq = ", l, Obsfreq


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< determine list of frequencies
                    if (IntegrationFlag) then                                               !< prepare integration over channel width


                        !< determine first and last frequency of current channel
                        FreqChannelFirst = 0.d0
                        FreqChannelLast = 0.d0
                        if (FreqIndex == FirstIndex) then
                            FreqChannelFirst = Obsfreq - ((ObservationalDataList((FreqIndex + 1), 1) - Obsfreq) / 2.d0)
                            FreqChannelLast = Obsfreq + ((ObservationalDataList((FreqIndex + 1), 1) - Obsfreq) / 2.d0)
                        elseif (FreqIndex == LastIndex) then
                            FreqChannelFirst = Obsfreq - ((Obsfreq - ObservationalDataList((FreqIndex - 1), 1)) / 2.d0)
                            FreqChannelLast = Obsfreq + ((Obsfreq - ObservationalDataList((FreqIndex - 1), 1)) / 2.d0)
                        else
                            FreqChannelFirst = Obsfreq - ((Obsfreq - ObservationalDataList((FreqIndex - 1), 1)) / 2.d0)
                            FreqChannelLast = Obsfreq + ((ObservationalDataList((FreqIndex + 1), 1) - Obsfreq) / 2.d0)
                        endif
                        ChannelWidth = (FreqChannelLast - FreqChannelFirst)                 !< determine channel width

                        ! Debug:
                        !    print*,'LastDSTransFreqIndex = ', LastDSTransFreqIndex
                        !    print*,'l, NumTransFreqPerFreqRange(l) = ', l, NumTransFreqPerFreqRange(l)
                    endif


                    !<====================================================================================================================================
                    !< determine beam size (eq. 3):
                    !<
                    !< for single dish data (InterFlag(l) = .false.):
                    !<
                    !<              telescope beam FWHM size is related to the diameter of the telescope by the diffraction limit:
                    !<
                    !<                  \theta_t = 1.22 \frac{c}{\nu D} * \zeta = 1.22 \frac{c}{\nu D} * (180.d0 * 3600.d0 / pi)
                    !<
                    !<              where D describes the diameter of the telescope and c the speed of light.
                    !<
                    !<
                    !< for interferometric data (InterFlag(l) = .true.):
                    !<
                    !<              interferometric beam FWHM size is directly given by the user (and is constant over the whole frequency range!)
                    !<
                    !<                  \theta_t = D
                    !<
                    !<              where D describes the interferometric beam FWHM size given by the user.
                    !<
                    if (InterFlag(l)) then
                        beam_size = TelescopeSize(l)
                    else
                        beam_size = 1.22d-3 * ckms / (Obsfreq * TelescopeSize(l)) * (180.d0 * 3600.d0 / pi)
                    endif


                    !< determine max. beam filling factor and the corresponding component index of all emission components
                    etaMax = 0.d0
                    etaMaxIndex = 0
                    SumEtaEmission = 0.d0
                    Do c = 1, TotalNumberComponents
                        if (int(CopymyXCLASSParameter(11, c)) == 1) then
                            val = CopymyXCLASSParameter(1, c)**2 / (beam_size**2 + CopymyXCLASSParameter(1, c)**2)
                            SumEtaEmission = SumEtaEmission + val
                            if (val > etaMax) then
                                etaMax = val
                                etaMaxIndex = c
                            endif
                        endif
                    end Do


                    !<====================================================================================================================================
                    !< determine intensity
                    TotalLocalIntensity = 0.d0
                    EmissionCompCounter = 0                                                 !< reset counter for emission components
                    AbsorptionCompCounter = 0                                               !< reset counter for foreground components
                    Do ae = 1, 2                                                            !< loop over core (ae=1) and foreground (ae=2) components
                        Do c = 1, TotalNumberComponents                                     !< loop over all components
                            if (int(CopymyXCLASSParameter(11, c)) == ae) then               !< check, if current component correspond to ae index
                                MoleculeIndex = CopyCompMoleculeIndex(c)                    !< get molecule index
                                ChannelIntensity = 0.d0                                     !< reset channel intensity
                                if (ae == 1) then
                                    EmissionCompCounter = EmissionCompCounter + 1           !< set flag if at least one component describe emission
                                else
                                    AbsorptionCompCounter = AbsorptionCompCounter + 1       !< increase counter for foreground components
                                endif


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< if nH flag is not set for the current component, use global definition for nH, dust beta, and kappa
                                if (FreqIndex == FirstIndex) then                           !< this has to be done only for the first frequency point
                                    if (.not. nHFlag(c)) then
                                        if (ae == 2 .or. c == etaMaxIndex) then
                                            CopymyXCLASSParameter(8, c) = HydrogenColumnDensityRange(l)
                                            CopymyXCLASSParameter(9, c) = DustBetaRange(l)
                                            CopymyXCLASSParameter(10, c) = KappaRange(l)
                                        else
                                            CopymyXCLASSParameter(8, c) = 0.d0
                                            CopymyXCLASSParameter(9, c) = 0.d0
                                            CopymyXCLASSParameter(10, c) = 0.d0
                                        endif
                                    endif
                                endif


                                !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                                !< prepare integration of spectrum
                                if (IntegrationFlag) then                                   !< prepare integration over channel width
                                    FirstMolecularDataIndex = MolecularDataIndices(l, MoleculeIndex, 1) !< get first mol. index of current freq. range
                                    LastMolecularDataIndex = MolecularDataIndices(l, MoleculeIndex, 2)  !< get last mol. index of current freq. range


                                    !< include first frequency of current channel only for the first channel of the current freq. range
                                    NumFreqList = 1
                                    FreqList(NumFreqList) = FreqChannelFirst


                                    !< check, if previous channel was calculated
                                    fiFirst = 1
                                    if (FreqIndex == FirstIndex) then
                                        FirstDataFlag(FreqIndex) = .true.
                                    elseif (FirstDataFlag(FreqIndex - 1)) then
                                        fiFirst = 2
                                    endif


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< check if width of lines of current component is smaller than two channel widths
                                    if (l == 1 .and. FreqIndex == FirstIndex) then
                                        MinSigma = 1.d99
                                        Do m = LastDSTransFreqIndex, LastMolecularDataIndex !< loop over all doppler-shifted transition freq.
                                            val = MolecularData(m, 1) * (1.d0 - (CopymyXCLASSParameter(5, c) + GlobalvLSR(l)) / ckms)
                                            DopplerShiftedTransFreq(m) = val
                                            sigma = (CopymyXCLASSParameter(4, c) / ckms * val) / (2.d0 * dsqrt(2.d0 * dlog(2.d0)))
                                            if (sigma < MinSigma) then
                                                MinSigma = sigma
                                            endif
                                        end Do
                                        MinSigmaHalf = MinSigma / 2.d0
                                        if (MinSigma < (2.d0 * ChannelWidth)) then          !< check if min. width is lower than two channel widths
                                            GeneralSamplingFlag(c) = .true.
                                        endif

                                        ! Debug:
                                        !    if (GeneralSamplingFlag(c)) then
                                        !        Do m = FirstMolecularDataIndex, LastMolecularDataIndex
                                        !            val = MolecularData(m, 1) * (1.d0 - (CopymyXCLASSParameter(5, c) + GlobalvLSR(l)) / ckms)
                                        !            write(54321, *) m, val
                                        !        end Do
                                        !    endif
                                    endif


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< if necessary: define freq. list of current channel for resampling
                                    NumDopplerTransFreq = 0
                                    if (GeneralSamplingFlag(c)) then
                                        t = 1
                                        Do m = LastDSTransFreqIndex, LastMolecularDataIndex !< loop over all doppler-shifted transition freq.
                                            val = DopplerShiftedTransFreq(m)
                                            if (FreqChannelFirst <= val .and. val <= FreqChannelLast) then


                                                !< check if doppler-shifted transition frequencies are not to close together, i.e. the current
                                                !< doppler-shifted transition frequency is closer than 10^(-2) of the channel width away from the
                                                !< last doppler-shifted transition frequency
                                                IncludeTransFreqFlag = .true.
                                                if (NumDopplerTransFreq > 0) then
                                                    if (dabs(FreqList(NumFreqList) - val) < 1.d-2 * ChannelWidth) then
                                                        IncludeTransFreqFlag = .false.
                                                    endif
                                                endif


                                                !< add current doppler-shifted transition frequency to list
                                                if (IncludeTransFreqFlag) then
                                                    NumDopplerTransFreq = NumDopplerTransFreq + 1
                                                    NumFreqList = NumFreqList + 1
                                                    FreqList(NumFreqList) = val
                                                    t = m
                                                endif
                                            elseif (val > FreqChannelLast) then
                                                exit
                                            endif
                                        end Do
                                        LastDSTransFreqIndex = t


                                        !< include always last frequency of current channel in the frequency list
                                        NumFreqList = NumFreqList + 1
                                        FreqList(NumFreqList) = FreqChannelLast

                                        ! Debug:
                                        !    print*,'FreqChannelFirst = ', FreqChannelFirst
                                        !    print*,'FreqChannelLast = ', FreqChannelLast
                                        !    print*,'t = ', t
                                        !    print*,'FreqList(1:t) = ', FreqList(1:t)
                                        !    print*,'NumDopplerTransFreq = ', NumDopplerTransFreq
                                        !    print*,'LastDSTransFreqIndex = ', LastDSTransFreqIndex


                                        !<----------------------------------------------------------------------------------------------------------------
                                        !< insert frequencies between doppler-shifted transition frequencies
                                        if (NumDopplerTransFreq > 0) then                   !< contains current channel at least one trans. freq.
                                            CopyFreqList(:) = FreqList(:)                   !< make a copy of list with trans. freq.
                                            FreqList = 0.d0                                 !< reset list
                                            t = 0                                           !< initialize counter
                                            Do m = 1, (NumFreqList - 1)                     !< loop over all doppler-shifted trans. frequencies
                                                val = CopyFreqList(m + 1) - CopyFreqList(m) !< get distance between doppler-shifted trans. freq.


                                                !< define number of points which should be inserted between doppler-shifted transition frequencies
                                                if (val < (MinSigmaHalf / 3.d0)) then       !< case:    val < MinSigmaHalf / 3.d0
                                                    NumInsertedPoints = 30
                                                elseif (val < (MinSigmaHalf / 2.d0)) then   !< case:    val < MinSigmaHalf / 2.d0
                                                    NumInsertedPoints = 20
                                                elseif (val < MinSigmaHalf) then            !< case:    val < MinSigmaHalf
                                                    NumInsertedPoints = 10
                                                else
                                                    NumInsertedPoints = 40
                                                endif


                                                !< insert points between doppler-shifted transition frequencies
                                                Do j = 0, NumInsertedPoints                 !< loop over new inserted points
                                                    if (t < (MaxFreqResampling - 1)) then   !< prevent out of bounds
                                                        t = t + 1
                                                        FreqList(t) = CopyFreqList(m) + j * (val / (NumInsertedPoints + 1.d0))
                                                    elseif (t == (MaxFreqResampling - 1)) then
                                                        t = t + 1
                                                        FreqList(t) = FreqChannelLast
                                                    endif
                                                end Do                                      !< j: loop over inserted points
                                            end Do                                          !< m: loop over doppler-shifted trans. frequencies
                                            if (t < (MaxFreqResampling - 1)) then           !< prevent out of bounds
                                                t = t + 1
                                                FreqList(t) = CopyFreqList(NumFreqList)     !< store highest frequency of current channel
                                            elseif (t == (MaxFreqResampling - 1)) then
                                                t = t + 1
                                                FreqList(t) = FreqChannelLast
                                            endif
                                            NumFreqList = t                                 !< define new number of resampling points
                                        endif
                                    else


                                        !< include always last frequency of current channel in the frequency list
                                        NumFreqList = NumFreqList + 1
                                        FreqList(NumFreqList) = FreqChannelLast
                                    endif

                                    ! Debug:
                                    !    print*,'NumFreqList = ', NumFreqList
                                    !    print*,'FreqList(1:NumFreqList) = ', FreqList(1:NumFreqList)


                                    !<------------------------------------------------------------------------------------------------------------
                                    !< only for resampling: Check, if current channel includes a transition frequency and define frequency list
                                    if (GeneralSamplingFlag(c)) then


                                        !<----------------------------------------------------------------------------------------------------------------
                                        !< does the current channel contain a transition?
                                        if (NumDopplerTransFreq == 0) then


                                            !<------------------------------------------------------------------------------------------------------------
                                            !< Check, if current channel with no doppler-shifted transition frequency lies close to a channel with
                                            !< at least one doppler-shifted transition frequency. Therefore, determine smallest distance to a
                                            !< doppler shifted transition frequency and check, if this distance is smaller than the channel width
                                            CopyDopplerShiftedTransFreq(1:LastMolecularDataIndex) = &
                                                                            DopplerShiftedTransFreq(1:LastMolecularDataIndex)
                                            CopyDopplerShiftedTransFreq(1:LastMolecularDataIndex) = &
                                                                            dabs(CopyDopplerShiftedTransFreq(1:LastMolecularDataIndex) - Obsfreq)
                                            MinDistance = minval(CopyDopplerShiftedTransFreq(1:LastMolecularDataIndex))
                                            if (MinDistance < (2.d0 * ChannelWidth) .or. PrevChannelTransFlag(c)) then   !< resampling necessary!

                                                ! Debug:
                                                !   write(5678, *) Obsfreq, MinDistance, (2.d0 * ChannelWidth), FreqChannelFirst, FreqChannelLast


                                                !< define freq. list for resampling for an empty but neighboring channel
                                                NumFreqList = 20
                                                DeltaFreq = ChannelWidth / (NumFreqList - 1.d0)
                                                Do m = 2, NumFreqList
                                                    FreqList(m) = FreqList(m - 1) + DeltaFreq
                                                end Do
                                            endif


                                            !< used for next channel
                                            PrevChannelTransFlag(c) = .false.


                                        !<----------------------------------------------------------------------------------------------------------------
                                        !< current channel contains at least one doppler-shifted transition frequency
                                        else
                                            PrevChannelTransFlag(c) = .true.
                                        endif
                                    endif


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< determine intensities for frequencies defined in the list FreqList
                                    IntFreqList = 0.d0                                      !< reset list of intensities
                                    Do fi = fiFirst, NumFreqList                            !< loop over all frequencies in the list
                                        freq = FreqList(fi)                                 !< get frequency where intensity is required


                                        !< calculate detection equation
                                        LocalIntensity = 0.d0
                                        call DetectionEquation(LocalIntensity, NumComp, freq, l, MoleculeIndex, CopymyXCLASSParameter, &
                                                               CopyIsoRatioConversionTable, Q, Td, beam_size, ae, c, tau_l, TotalLocalIntensity, &
                                                               EmissionCompCounter, AbsorptionCompCounter, NumEmissionComp, TotalNumMol)
                                        ! Debug:
                                        !    if (GeneralSamplingFlag(c)) write(12341 + c,*) freq, LocalIntensity


                                        !< store intensities in a list
                                        if (IntegrationFlag) then                           !< prepare integration over channel width
                                            IntFreqList(fi) = LocalIntensity
                                        endif
                                    end Do                                                  !< fi: loop over all frequencies in freq. list


                                    !<####################################################################################################################
                                    !< Only used by myXCLASS function!
                                    !<####################################################################################################################
                                    !< for myxclass program: determine intensity at channel center
                                    if (LogFlag) then
                                        if (AllOutputFilesFlag) then
                                            LocalIntensity = 0.d0
                                            tau_l = 0.d0
                                            Td = 0.d0
                                            call DetectionEquation(LocalIntensity, NumComp, Obsfreq, l, MoleculeIndex, CopymyXCLASSParameter, &
                                                                   CopyIsoRatioConversionTable, Q, Td, beam_size, ae, c, tau_l, TotalLocalIntensity, &
                                                                   EmissionCompCounter, AbsorptionCompCounter, NumEmissionComp, TotalNumMol)
                                            TauPerCompArray(FreqIndex, c, :) = TauHelperArray1(:)
                                            TdPerCompArray(FreqIndex, c) = Td
                                        endif
                                    endif
                                    !<####################################################################################################################
                                    !<####################################################################################################################

                                    ! Debug:
                                    !    if (dabs(Obsfreq - debug_freq) < 1.d-4) then
                                    !        print*,"Obsfreq = ", Obsfreq
                                    !        print*,'IntFreqList(1:NumFreqList) = ', IntFreqList(1:NumFreqList)
                                    !        print*,'IntLimit = ', IntLimit
                                    !        print*,'NumDopplerTransFreq = ',NumDopplerTransFreq
                                    !        print*,'####################################################################################################'
                                    !    endif


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< integrate spectrum over channel width
                                    FirstDataFlag(FreqIndex) = .true.


                                    !< add calculated intensity of last channel to frequency list
                                    if (fiFirst == 2) then
                                        if (ae == 1) then
                                            IntFreqList(1) = LastChannelInt(c)
                                        else
                                            IntFreqList(1) = IntFreqList(2)                 !< bug-fix: avoid shift of modeled spectrum by adding f-comp.
                                        endif
                                    endif
                                    LastChannelInt(c) = IntFreqList(NumFreqList)

                                    ! Debug:
                                    !    print*,'fiFirst = ', fiFirst
                                    !    print*,'LastChannelInt(c) = ',LastChannelInt(c)
                                    !    print*,'NumDopplerTransFreq = ',NumDopplerTransFreq
                                    !    print*,"NumFreqList = ", NumFreqList
                                    !    print*,"FreqList(1:NumFreqList) = ", FreqList(1:NumFreqList)
                                    !    print*,'IntFreqList(1:NumFreqList) = ', IntFreqList(1:NumFreqList)
                                    !    print*,'FreqChannelFirst = ', FreqChannelFirst
                                    !    print*,'FreqChannelLast = ', FreqChannelLast
                                    !    print*,'########################################################################################################'
                                    !    print*,'  '


                                    !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                                    !< do integration for each component for each channel
                                    ChannelInt = 0.d0
                                    Do fi = 1, (NumFreqList - 1)
                                        x0 = FreqList(fi)
                                        x1 = FreqList(fi + 1)
                                        f0 = IntFreqList(fi)
                                        f1 = IntFreqList(fi + 1)
                                        val = (x1 - x0) * (f0 + f1) / 2.d0
                                        ChannelInt = ChannelInt + val
                                    end Do
                                    ChannelIntensity = (1.d0 / ChannelWidth) * ChannelInt
                                    !<++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

                                    ! Debug:
                                    !    if (GeneralSamplingFlag(c)) write(8888 + c,*) Obsfreq, ChannelInt
                                    !    print*,'Obsfreq, FreqIndex, TotalLocalIntensity = ', Obsfreq, FreqIndex, TotalLocalIntensity
                                    !    print*,'#############################################################'
                                else
                                    freq = Obsfreq
                                    call DetectionEquation(LocalIntensity, NumComp, freq, l, MoleculeIndex, CopymyXCLASSParameter, &
                                                           CopyIsoRatioConversionTable, Q, Td, beam_size, ae, c, tau_l, TotalLocalIntensity, &
                                                           EmissionCompCounter, AbsorptionCompCounter, NumEmissionComp, TotalNumMol)
                                    ChannelIntensity = LocalIntensity

                                    ! Debug:
                                    !    if (dabs(Obsfreq - debug_freq) < 1.d-4) then
                                    !        print*,"Obsfreq, LocalIntensity = ", Obsfreq, LocalIntensity
                                    !        print*,'****************************************************************************************************'
                                    !    endif
                                endif


                                !< determine total intensity of all components
                                if (ae == 1) then
                                    TotalLocalIntensity = TotalLocalIntensity + ChannelIntensity
                                else
                                    TotalLocalIntensity = ChannelIntensity
                                endif


                                !<########################################################################################################################
                                !< Only used by myXCLASS function!
                                !<########################################################################################################################
                                !< write modelled function to output file
                                if (LogFlag) then
                                    if (AllOutputFilesFlag) then
                                        if (FreqIndex == FirstIndex) then
                                            IntPerCompArray(FreqIndex, c, :) = LocalIntArray(:)
                                        else
                                            Do fi = 1, MaxNumIso
                                                x0 = ObservationalDataList(FreqIndex - 1, 1)
                                                x1 = ObservationalDataList(FreqIndex, 1)
                                                f0 = IntPerCompArray(FreqIndex - 1, c, fi)
                                                f1 = LocalIntArray(fi)
                                                val = (x1 - x0) * (f0 + f1) / 2.d0
                                                ! IntPerCompArray(FreqIndex, c, fi) = (1.d0 / ChannelWidth) * val
                                                IntPerCompArray(FreqIndex, c, fi) = f1
                                            end Do
                                        endif
                                    endif
                                endif
                                !<########################################################################################################################
                                !<########################################################################################################################
                            endif                                                           !< continue here if current comp. does not correspond to ae
                        end Do                                                              !< c: loop over all components
                    end Do                                                                  !< ae: loop over core / foreground

                    ! Debug:
                    ! print*,"Obsfreq, TotalLocalIntensity = ", Obsfreq, TotalLocalIntensity


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< determine sigma (sig2i) factor
                    sig2i = 1.d0
                    if (ObservationalDataList(FreqIndex, 3) /= 0.d0) then                   !< do the experimental datas include errors
                        sig2i = 1.d0 / (ObservationalDataList(FreqIndex, 3)**2)
                    endif


                    !< determine chi^2 value using local calculated intensity
                    chi2Value = chi2Value + ((TotalLocalIntensity - ObservationalDataList(FreqIndex, 2))**2 * sig2i)
                    if (ModelFunctionFlag) then                                             !< save model function
                        ModelFuncList(FreqIndex) = TotalLocalIntensity
                    endif
                end Do                                                                      !< FreqIndex: loop over frequencies
            end Do                                                                          !< l: loop over frequency ranges

            ! Debug:
            !    print*,"DataPointIndexFreqRange(:, 1) = ", DataPointIndexFreqRange(:, 1)
            !    print*,"DataPointIndexFreqRange(:, 2) = ", DataPointIndexFreqRange(:, 2)
            !    print*,"ObservationalDataList(DataPointIndexFreqRange(1, 1), :) = ", ObservationalDataList(DataPointIndexFreqRange(1, 1), :)
            !    print*,"ObservationalDataList(DataPointIndexFreqRange(1, 2), :) = ", ObservationalDataList(DataPointIndexFreqRange(1, 2), :)
            !    print*,'>>>>>>>>>>ThreadNumber, count(isnan(ModelFuncList(:))) = ', ThreadNumber, count(isnan(ModelFuncList(:)))


            !<############################################################################################################################################
            !< Only used by myXCLASS function!
            !<############################################################################################################################################
            !< write output variables to file
            if (LogFlag) then


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< open output file for transition energies
                if (AllOutputFilesFlag) then
                    open(2223, file = 'transition_energies.dat', status = 'unknown')
                    write(2223, '(4x, "% Trans. freq. (MHz):", 4x, "Doppler-shift freq. (MHz):", 16x, "Intensity (K):", $)')
                    write(2223, '(14x, "Integrated Int.:", 20x, "E_low (K):", 25x, "g_up:", 11x, "Einstein A (s^(-1):", $)')
                    write(2223, '(5x, "Molecule name:")')
                endif


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< write contents of output arrays to file
                Do l = 1, TotalNumberOfFrequencyRanges                                      !< loop over all frequency ranges
                    i = SpectrumIndexForFreqRange(l)                                        !< get spectrum index
                    FirstIndex = DataPointIndexFreqRange(l, 1)                              !< get index of first freq. point in 'ObservationalDataList'
                    LastIndex = DataPointIndexFreqRange(l, 2)                               !< get index of last freq. point in 'ObservationalDataList'


                    !< write modeled spectrum to output file
                    Do FreqIndex = FirstIndex, LastIndex                                    !< loop over all frequency points in the current freq. range
                        Obsfreq = ObservationalDataList(FreqIndex, 1)                       !< get frequency
                        write(2222, '(1x, ES25.10, 1x, ES25.16E3)') Obsfreq, ModelFuncList(FreqIndex)
                    end Do
                    if (AllOutputFilesFlag) then


                        !< write telescope beam full width half maximum sizes for \nu_min and \nu_max to log file
                        write(2224,'(" ")')
                        Do t = 1, 2
                            write(2224,'(" ")')
                            if (t == 1) then
                                x0 = ObservationalDataList(FirstIndex, 1)
                                write(2224,'("Beginning of frequency range", $)')
                            else
                                x0 = ObservationalDataList(LastIndex, 1)
                                write(2224,'("End of frequency range", $)')
                            endif
                            write(numberstring1, '(F25.3)') x0
                            write(2224,'(" (\nu = ", A, " MHz):")') trim(adjustl(numberstring1))
                            write(2224,'(" ")')


                            !< write telescope beam full width half maximum sizes for \nu_min and \nu_max to log file
                            if (InterFlag(l)) then
                                beam_size = TelescopeSize(l)
                            else
                                beam_size = 1.22d-3 * ckms / (x0 * TelescopeSize(l)) * (180.d0 * 3600.d0 / pi)
                            endif
                            write(numberstring2, '(F18.5)') beam_size
                            write(2224, '(2x, "Telescope beam FWHM size:                       \theta_T (\nu) = ", A, " arcsec")') &
                                        trim(adjustl(numberstring2))
                            write(2224,'(" ")')
                        end Do
                        write(2224,'(" ")')


                        !< write intensities and opacities for each component to files
                        fff = 0
                        Do c = 1, TotalNumberComponents                                     !< loop over all components
                            MoleculeIndex = CopyCompMoleculeIndex(c)                        !< get molecule index
                            ll = MoleculeIndex                                              !< get molecule index
                            if (NumberOfGlobalISORatios > 0) then
                                kk = TotalNumMol
                            else
                                kk = ll
                            endif
                            IsoCounter = 0
                            Do MoleculeIndexLoop = ll, kk                                   !< loop over all molecule indices (including isotopologues!)
                                CalcFalg = .true.
                                if (NumberOfGlobalISORatios > 0) then
                                    if (CopyIsoRatioConversionTable(ll, MoleculeIndexLoop) == 0.d0) then
                                        CalcFalg = .false.
                                    endif
                                endif
                                if (CalcFalg) then                                          !< consider current molecule
                                    IsoCounter = IsoCounter + 1
                                    fff = fff + 1
                                    CurrMoleculeName = trim(adjustl(MoleculeNames(MoleculeIndexLoop)))  !< get name of current molecule


                                    !< correct molecule name
                                    ModifiedMoleculeName = CurrMoleculeName
                                    Do n = 1, len_trim(CurrMoleculeName)
                                        if (CurrMoleculeName(n:n) == ";" .or. CurrMoleculeName(n:n) == "(" .or. CurrMoleculeName(n:n) == ")" &
                                            .or. CurrMoleculeName(n:n) == "'" .or. CurrMoleculeName(n:n) == ";") then
                                            ModifiedMoleculeName(n:n) = "_"
                                        endif
                                    end Do


                                    !< determine current comp. index and total number of component of current molecule
                                    ncomp = 0
                                    n = 0
                                    Do i = 1, TotalNumberComponents
                                        if (CopyCompMoleculeIndex(i) == CopyCompMoleculeIndex(c)) then
                                            ncomp = ncomp + 1
                                            if (i <= c) then
                                                n = n + 1
                                            endif
                                        endif
                                    end Do
                                    write(numberstring2, '(I30)') n                         !< convert integer to string


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< write dust temperature for each component to log-file
                                    if (.not. tbFlag(l)) then
                                        write(2224,'(" ")')
                                        write(numberstring2, '(I30)') n
                                        write(2224, '("Molecule: ", A, ", component: ", A)') trim(adjustl(CurrMoleculeName)), trim(adjustl(numberstring2))
                                        x0 = TdPerCompArray(FirstIndex, c)
                                        write(2224, '(2x, "Dust temperature (beginning of frequency range):       T_d (\nu_{min}) = ", 5x, F15.5, " K")') x0
                                        x0 = TdPerCompArray(LastIndex, c)
                                        write(2224, '(2x, "Dust temperature (end of frequency range):             T_d (\nu_{max}) = ", 5x, F15.5, " K")') x0
                                    endif


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< open output files for intensities of each component of each molecule
                                    IntCompFileName(c) = "intensity__" // trim(adjustl(ModifiedMoleculeName)) // "__comp__" &
                                                         // trim(adjustl(numberstring2)) // ".dat"
                                    open(19143 + fff, file = trim(adjustl(IntCompFileName(c))))
                                    write(19143 + fff, '("name of current molecule = ", A)') trim(adjustl(CurrMoleculeName))
                                    write(19143 + fff, '("current component = ", I10)') n
                                    write(19143 + fff, '("total number of components = ", I10)') ncomp
                                    write(19143 + fff, '(10x, "Frequency [MHz]:", 21x, "T_mb [K]:")')


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< open output files for optical depths of each component of each molecule
                                    TauCompFileName(c) = "optical_depth__" // trim(adjustl(ModifiedMoleculeName)) // "__comp__" &
                                                         // trim(adjustl(numberstring2)) // ".dat"
                                    open(142 + fff, file = trim(adjustl(TauCompFileName(c))))
                                    write(142 + fff, '("name of current molecule = ", A)') trim(adjustl(CurrMoleculeName))
                                    write(142 + fff, '("current component = ", I10)') n
                                    write(142 + fff, '("total number of components = ", I10)') ncomp
                                    write(142 + fff, '(10x, "Frequency [MHz]:", 26x, "tau:")')


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< open output files for optical depths of each component of each molecule
                                    Do FreqIndex = FirstIndex, LastIndex                    !< loop over all frequency points in the current freq. range
                                        Obsfreq = ObservationalDataList(FreqIndex, 1)       !< get frequency


                                        !< write intensities for each component and molecule to file
                                        write(19143 + fff, '(1x, ES25.10, 5x, ES25.15E3)') Obsfreq, IntPerCompArray(FreqIndex, c, IsoCounter)


                                        !< write optical depth tau for each component and molecule to file
                                        write(142 + fff, '(1x, ES25.10, 5x, ES25.15E3)') Obsfreq, TauPerCompArray(FreqIndex, c, IsoCounter)
                                    end Do                                                  !< FreqIndex: loop over frequencies


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< write transition energies to output file
                                    FirstMolecularDataIndex = MolecularDataIndices(l, MoleculeIndexLoop, 1) !< get first mol. index of current freq. range
                                    LastMolecularDataIndex = MolecularDataIndices(l, MoleculeIndexLoop, 2)  !< get last mol. index of current freq. range
                                    Do t = FirstMolecularDataIndex, LastMolecularDataIndex  !< loop over all transitions of current molecule
                                        !    'transition frequency = ', MolecularData(t, 1)
                                        !    'EinsteinA            = ', MolecularData(t, 2)
                                        !    'E_low                = ', MolecularData(t, 3)
                                        !    'g_u                  = ', MolecularData(t, 4)
                                        if (t > 0) then


                                            !< determine Doppler-shifted transition frequencies
                                            x0 = MolecularData(t, 1) * ( 1.d0 - (CopymyXCLASSParameter(5, c) + GlobalvLSR(l)) / ckms)


                                            !< determine modeled spectrum at Doppler-shifted transition frequencies
                                            f0 = 0.d0
                                            FreqIndex = minloc(dabs(x0 - ObservationalDataList(FirstIndex:LastIndex, 1)), dim=1)
                                            f0 = ModelFuncList(FreqIndex)


                                            !< write integrated peak intensity of each line to output file
                                            f1 = IntensityPerPeak(l, c, IsoCounter, t - FirstMolecularDataIndex + 1)


                                            !< write frequencies to output file
                                            if (n == 1) then
                                                write(2223,'(ES25.15, $)') MolecularData(t, 1)
                                            else
                                                write(2223,'(24x, " ", $)') 
                                            endif
                                            write(2223,'(3(5x, ES25.15), $)') x0, f0, f1
                                            if (n == 1) then                   !< add other transition parameter to log file
                                                write(2223,'(5x, ES25.15, 5x, ES25.15, 5x, ES25.15, $)') MolecularData(t, 3), MolecularData(t, 4), &
                                                                                                         MolecularData(t, 2)
                                            else
                                                write(2223,'(29x, " ", 29x, " ", 29x, " ", $)') 
                                            endif
                                            write(2223,'(5x, A)') trim(adjustl(CurrMoleculeName))
                                        else
                                            exit
                                        endif
                                    end Do                                                  !< t: loop over transitions
                                    close(142 + fff)                                        !< close output file for optical depths for each component
                                    close(19143 + fff)                                      !< close output file for intensities for each component
                                endif                                                       !< CalcFlag
                            end Do                                                          !< MoleculeIndexLoop: loop over molecules
                        end Do                                                              !< c: loop over all components
                    endif
                end Do                                                                      !< l: loop over frequency ranges
                close(2223)                                                                 !< close output file for transition energies


                !< if global iso ratio files are defined, write used iso ratios for all isotopologue to log file
                if (NumberOfGlobalISORatios > 0) then
                    if (AllOutputFilesFlag) then
                        write(2224,'(" ")')
                        write(2224,'(" ")')
                        write(2224,'("Used iso ratios:")')
                        write(2224,'(" ")')
                        write(2224,'(4x, "Isotopologue:", 27x, "Iso-master:", 44x, "Iso-ratio:")') 
                        write(2224,'(" ")')
                        Do n = 1, TotalNumberOfMolecules
                            Do l = (n + 1), TotalNumberOfMolecules
                                if (CopyIsoRatioConversionTable(l, n) /= 0.d0) then
                                    write(2224,'(4x, A40, 10x, $)') adjustl(MoleculeNames(l))
                                    write(2224,'(A40, 10x, $)') adjustl(MoleculeNames(n))
                                    write(2224,'(ES25.5)') CopyIsoRatioConversionTable(l, n)
                                endif
                            end Do
                        end Do
                    endif
                endif
            endif
            !<############################################################################################################################################
            !<############################################################################################################################################

            ! Debug:
            !    CALL CPU_TIME(tt2)
            !    print '("Total time for subroutine ModelCalcSpectrum: ", F20.3, " sec. = ", F20.3, " min.")', (tt2 - tt1), (tt2 - tt1) / 60.d0
            !    stop '###### stop subroutine ModelCalcSpectrum ######'
            return
        end subroutine ModelCalcSpectrum


        !>************************************************************************************************************************************************
        !> subroutine: myXCLASSParamFree
        !>
        !> free memory used by myXCLASS variables
        !>
        !>
        !> input variables:     deallocstatus           status of the previous deallocation process
        !>
        !> output variables:    deallocstatus           status of the deallocation process
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 26.08.2014
        !>
        subroutine myXCLASSParamFree(deallocstatus)

            implicit none
            integer :: deallocstatus                                                        !< status of the deallocation process


            !< deallocate memory of variables of Modules 'Variables' and 'FunctionCalling'
            if (allocated(lengthexpdata)) deallocate(lengthexpdata, stat = deallocstatus)
            if (allocated(NumberXColumns)) deallocate(NumberXColumns, stat = deallocstatus)
            if (allocated(NumberYColumns)) deallocate(NumberYColumns, stat = deallocstatus)
            if (allocated(NumberHeaderLines)) deallocate(NumberHeaderLines, stat = deallocstatus)
            if (allocated(OutputFileFormat)) deallocate(OutputFileFormat, stat = deallocstatus)
            if (allocated(NumberRanges)) deallocate(NumberRanges, stat = deallocstatus)
            if (allocated(ConverterInfit)) deallocate(ConverterInfit, stat = deallocstatus)
            if (allocated(ValueEmptyOutputFile)) deallocate(ValueEmptyOutputFile, stat = deallocstatus)
            if (allocated(LSRAdjustement)) deallocate(LSRAdjustement, stat = deallocstatus)
            if (allocated(chisqValues)) deallocate(chisqValues, stat = deallocstatus)
            if (allocated(BestSitesParamSet)) deallocate(BestSitesParamSet, stat = deallocstatus)
            if (allocated(paramset)) deallocate(paramset, stat = deallocstatus)
            if (allocated(AtOnceFunction)) deallocate(AtOnceFunction, stat = deallocstatus)
            if (allocated(AtOnceGradient)) deallocate(AtOnceGradient, stat = deallocstatus)
            if (allocated(FirstPointExpData)) deallocate(FirstPointExpData, stat = deallocstatus)
            if (allocated(LastPointExpData)) deallocate(LastPointExpData, stat = deallocstatus)
            if (allocated(expdatax)) deallocate(expdatax, stat = deallocstatus)
            if (allocated(expdatay)) deallocate(expdatay, stat = deallocstatus)
            if (allocated(expdatae)) deallocate(expdatae, stat = deallocstatus)
            if (allocated(CalculatedParameterSets)) deallocate(CalculatedParameterSets, stat = deallocstatus)
            if (allocated(MinRange)) deallocate(MinRange, stat = deallocstatus)
            if (allocated(MaxRange)) deallocate(MaxRange, stat = deallocstatus)
            if (allocated(BestSitesModelValues)) deallocate(BestSitesModelValues, stat = deallocstatus)
            if (allocated(BestSitesChi2Values)) deallocate(BestSitesChi2Values, stat = deallocstatus)
            if (allocated(ModelFunction)) deallocate(ModelFunction, stat = deallocstatus)
            if (allocated(FitParameterName)) deallocate(FitParameterName, stat = deallocstatus)
            if (allocated(FitParameterValue)) deallocate(FitParameterValue, stat = deallocstatus)
            if (allocated(CharacterForComments)) deallocate(CharacterForComments, stat = deallocstatus)
            if (allocated(CharacterSeperatingColumns)) deallocate(CharacterSeperatingColumns, stat = deallocstatus)
            if (allocated(ResamplingMethod)) deallocate(ResamplingMethod, stat = deallocstatus)
            if (allocated(InterpolationMethod)) deallocate(InterpolationMethod, stat = deallocstatus)
            if (allocated(OnlyYColumn)) deallocate(OnlyYColumn, stat = deallocstatus)
            if (allocated(LSRAdjustementFitFlag)) deallocate(LSRAdjustementFitFlag, stat = deallocstatus)
            if (allocated(NormalizationFlag)) deallocate(NormalizationFlag, stat = deallocstatus)
            if (allocated(ExpData_reversed_flag)) deallocate(ExpData_reversed_flag, stat = deallocstatus)
            if (allocated(NumberParamPerLine)) deallocate(NumberParamPerLine, stat = deallocstatus)
            if (allocated(ParameterName)) deallocate(ParameterName, stat = deallocstatus)
            if (allocated(ParameterFormat)) deallocate(ParameterFormat, stat = deallocstatus)
            if (allocated(LeadingString)) deallocate(LeadingString, stat = deallocstatus)
            if (allocated(TrailingString)) deallocate(TrailingString, stat = deallocstatus)
            if (allocated(ParamVisible)) deallocate(ParamVisible, stat = deallocstatus)
            if (allocated(FitFktInput)) deallocate(FitFktInput, stat = deallocstatus)
            if (allocated(FitFktOutput)) deallocate(FitFktOutput, stat = deallocstatus)
            if (allocated(valuesModel_output)) deallocate(valuesModel_output, stat = deallocstatus)
            if (allocated(xPointsModel_output)) deallocate(xPointsModel_output, stat = deallocstatus)


            !< deallocate memory of variables of Module 'Model'
            if (allocated(IsoMolecule)) deallocate(IsoMolecule, stat = deallocstatus)
            if (allocated(IsoMasters)) deallocate(IsoMasters, stat = deallocstatus)
            if (allocated(IsoRatio)) deallocate(IsoRatio, stat = deallocstatus)
            if (allocated(StartFrequency)) deallocate(StartFrequency, stat = deallocstatus)
            if (allocated(EndFrequency)) deallocate(EndFrequency, stat = deallocstatus)
            if (allocated(StepFrequency)) deallocate(StepFrequency, stat = deallocstatus)
            if (allocated(tbFlag)) deallocate(tbFlag, stat = deallocstatus)
            if (allocated(BackgroundTemperatureRange)) deallocate(BackgroundTemperatureRange, stat = deallocstatus)
            if (allocated(TemperatureSlopeRange)) deallocate(TemperatureSlopeRange, stat = deallocstatus)
            if (allocated(HydrogenColumnDensityRange)) deallocate(HydrogenColumnDensityRange, stat = deallocstatus)
            if (allocated(DustBetaRange)) deallocate(DustBetaRange, stat = deallocstatus)
            if (allocated(KappaRange)) deallocate(KappaRange, stat = deallocstatus)
            if (allocated(TelescopeSize)) deallocate(TelescopeSize, stat = deallocstatus)
            if (allocated(InterFlag)) deallocate(InterFlag, stat = deallocstatus)
            if (allocated(GlobalvLSR)) deallocate(GlobalvLSR, stat = deallocstatus)
            if (allocated(SpectrumIndexForFreqRange)) deallocate(SpectrumIndexForFreqRange, stat = deallocstatus)
            if (allocated(ObservationalDataList)) deallocate(ObservationalDataList, stat = deallocstatus)
            if (allocated(DataPointIndexFreqRange)) deallocate(DataPointIndexFreqRange, stat = deallocstatus)
            if (allocated(MoleculeNames)) deallocate(MoleculeNames, stat = deallocstatus)
            if (allocated(ConversionTableMAGIXmyXCLASSParam)) deallocate(ConversionTableMAGIXmyXCLASSParam, stat = deallocstatus)
            if (allocated(IsoRatioConversionTable)) deallocate(IsoRatioConversionTable, stat = deallocstatus)
            if (allocated(IsoNfitConversionTable)) deallocate(IsoNfitConversionTable, stat = deallocstatus)
            if (allocated(NumberComponentsPerMolecule)) deallocate(NumberComponentsPerMolecule, stat = deallocstatus)
            if (allocated(myXCLASSParameter)) deallocate(myXCLASSParameter, stat = deallocstatus)
            if (allocated(CompMoleculeIndex)) deallocate(CompMoleculeIndex, stat = deallocstatus)
            if (allocated(MolNamePartFunc)) deallocate(MolNamePartFunc, stat = deallocstatus)
            if (allocated(lgQ)) deallocate(lgQ, stat = deallocstatus)
            if (allocated(NumEntriesRanges)) deallocate(NumEntriesRanges, stat = deallocstatus)
            if (allocated(MolecularData)) deallocate(MolecularData, stat = deallocstatus)
            if (allocated(MolecularDataIndices)) deallocate(MolecularDataIndices, stat = deallocstatus)
            if (allocated(MolNameRadTrans)) deallocate(MolNameRadTrans, stat = deallocstatus)
            if (allocated(ColumnNamesPartFunc)) deallocate(ColumnNamesPartFunc, stat = deallocstatus)
            if (allocated(TempPartFunc)) deallocate(TempPartFunc, stat = deallocstatus)
            return
        end subroutine myXCLASSParamFree
end Module myXCLASSCore
!*********************************************************************************************************************************************************

