!*********************************************************************************************************************************************************
!>  Module: Model
!>
!>
!>  This module contains subroutines for the myXCLASS program for application in MAGIX  (Non-GPU version)
!>  Copyright (C) 2009 - 2014  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine CheckMolNames:                 checks, if a molecule name is already known
!>      - subroutine strlen:                        subroutine to determine last non blank and non char(0) character of string st
!>      - subroutine GetmyXCLASSParameter:          reads parameter for myXCLASS from paramset variable
!>      - subroutine GetPartitionFunction:          gets partition function from sqlite3 database
!>      - subroutine GetTransitionParameters:       gets informations for transitions from sqlite3 database table linecat
!>      - subroutine ModelInit:                     initializes myXCLASS program
!>      - subroutine ModelCalcChiFunctionLM:        calculates the chi2 values for the Levenberg-Marquard algorithm
!>      - subroutine ModelCalcChiFunctionGeneral:   calculates the chi2 values for several given parameter vector sets
!>      - subroutine ModelCalcChiFunction:          calculates the chi2 values for a given set of parameter vectors
!>      - subroutine ModelCalcSpectrum:             calculates the myXCLASS spectrum for a given parameter vector
!>      - subroutine ModelParamFree:                free memory used by variables of the Module Model
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    29.07.2014  Initial version
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!---------------------------------------------------------------------------------------------------------------------------------------------------------
Module Model
    !> declare variables which are used by functions and subroutines of this module

    use Variables
    use FunctionCalling
    use cudafor

    implicit none


    !< general variables
    integer :: ErrChannelIndex                                                              !< loop variable for error handling
    integer :: ErrChannel                                                                   !< current error channel
    integer, dimension(2) :: AllErrChannels = (/0, 2222/)                                   !< list of all error channels
    real*8 :: pi                                                                            !< pi
    real*8 :: ckms                                                                          !< speed of light in km/s
    real*8, parameter :: debug_freq = 554989.699e30                                         !< frequency for debugging
    logical :: UseRegistrationXMLFile                                                       !< use registation XML file
    character(len=3) :: ParallelizationMethod                                               !< string describing the parallelization method


    !< variables for molfit file
    integer :: TotalNumberDataPoints                                                        !< total number of data points
    integer :: NumberMolecules                                                              !< number of molecules in the current molfits file
    integer :: TotalNumberComponents                                                        !< counter for total number of components of all molecules
    integer :: TotalNumberOfFrequencyRanges                                                 !< total number of frequency ranges
    integer :: TotalNumberOfMolecules                                                       !< total number of molecules including isotopologues
    integer, allocatable, dimension(:) :: CompMoleculeIndex                                 !< molecule index for each component
    integer, allocatable, dimension(:) :: SpectrumIndexForFreqRange                         !< store spectrum index for each frequency range
    integer, allocatable, dimension(:) :: NumberComponentsPerMolecule                       !< number of components per molecule
    integer, allocatable, dimension(:, :) :: ConversionTableMAGIXmyXCLASSParam              !< conversion table between myXCLASS and MAGIX parameter
    integer, allocatable, dimension(:, :) :: DataPointIndexFreqRange                        !< index of data point of first and last freq. in freq. range
    real*8, allocatable, dimension(:) :: StartFrequency                                     !< first frequency for each frequency range
    real*8, allocatable, dimension(:) :: EndFrequency                                       !< last frequency for each frequency range
    real*8, allocatable, dimension(:) :: StepFrequency                                      !< stepsize for each frequency range
    real*8, allocatable, dimension(:) :: TelescopeSize                                      !< size of telescope for each frequency range
    real*8, allocatable, dimension(:) :: BackgroundTemperatureRange                         !< T_Back for each frequency range
    real*8, allocatable, dimension(:) :: TemperatureSlopeRange                              !< T_Slope for each frequency range
    real*8, allocatable, dimension(:) :: HydrogenColumnDensityRange                         !< nH for each frequency range
    real*8, allocatable, dimension(:) :: DustBetaRange                                      !< beta for each frequency range
    real*8, allocatable, dimension(:) :: KappaRange                                         !< kappa for each frequency range
    real*8, allocatable, dimension(:, :) :: myXCLASSParameter                               !< array containing all molfit parameters for each component
    real*8, allocatable, dimension(:, :) :: ObservationalDataList                           !< list containing all observational data
    character(len=40), allocatable, dimension(:) :: MoleculeNames                           !< names of all molecules (including isotopologues)
    logical :: tbFlag                                                                       !< flag for global setting T_Back and T_Slope
    logical :: nHFlag                                                                       !< flag for global setting of nH, kappa and beta
    logical :: IsoFlag                                                                      !< flag indicating use of isotopologues


    !< variables for sqlite3 database
    integer, allocatable, dimension(:, :) :: NumEntriesRanges                               !< number of transitions for each freq. range and molecule
    character(len=128) :: NameOfPartFuncTable                                               !< name of table for partition function
    character(len=128) :: NameOfRadTransTable                                               !< name of table for rad. trans.
    character(len=4096) :: dbName                                                           !< name of database


    !< variables for partition function
    integer :: NumberOfTemperatures                                                         !< number of temperatures
    integer :: NumberMoleculePartFunc                                                       !< number of entries in table PartitionFunction
    integer :: Firsti, Lasti, stepi                                                         !< working variables for interpolation
    real*8 :: TempLow, TempHigh                                                             !< working variables for extrapolation of part. func.
    real*8, allocatable, dimension(:) :: TempPartFunc                                       !< temperatures for partition function
    real*8, allocatable, dimension(:, :) :: lgQ                                             !< lgQ entries of table PartitionFunction
    character(len=40) :: ColumnNameForNamePartFunc                                          !< name of column including the names of molecules
    character(len=40), allocatable, dimension(:) :: MolNamePartFunc                         !< molecule names of table PartitionFunction
    character(len=40), allocatable, dimension(:) :: ColumnNamesPartFunc                     !< column names of table PartitionFunction


    !< variables for rad-trans parameter
    integer :: TotalNumberOfTransitions                                                     !< number of entries in table RadTrans
    integer :: Column300KPartFunc                                                           !< number of column, which contains part. func. @ 300 K
    integer, allocatable, dimension(:, :, :) :: MolecularDataIndices                        !< start and end index for each molecule and frequency range
                                                                                            !< within MolecularData variable
    real*8, allocatable, dimension(:, :) :: MolecularData                                   !< array containing the molecular data for all molecules and
                                                                                            !< all freqency ranges
    character(len=40) :: ColumnNameForNameTransitions                                       !< name of column in table transitions for molecule name
    character(len=40) :: ColumnNameForFreqTransitions                                       !< name of column in table transitions for frequency
    character(len=40) :: ColumnNameForIntTransitions                                        !< name of column in table transitions for intensity
    character(len=40) :: ColumnNameForEinsteinATransitions                                  !< name of column in table transitions for Einstein A coef.
    character(len=40) :: ColumnNameForFreqErrTransitions                                    !< name of column in table transitions for error freq.
    character(len=40) :: ColumnNameForELowTransitions                                       !< name of column in table transitions for E_low
    character(len=40) :: ColumnNameForgUpTransitions                                        !< name of column in table transitions for Upper State degen.
    character(len=40), allocatable, dimension(:) :: MolNameRadTrans                         !< molecule names of table RadTrans


    !< variables for iso table file
    integer :: NumberOfIsomasters                                                           !< number of isomasters
    integer, allocatable, dimension(:, :) :: IsoNfitConversionTable                         !< iso ratios conversion table for free paramter index
    real*8, allocatable, dimension(:, :) :: IsoRatio                                        !< iso ratios for all isomasters
    real*8, allocatable, dimension(:, :) :: IsoRatioConversionTable                         !< table with iso ratios between iso master and molecule
    character(len=4096) :: IsoTableFileName                                                 !< path and file name of the iso file
    character(len=40), allocatable, dimension(:) :: IsoMolecule                             !< names of molecules for iso table
    character(len=40), allocatable, dimension(:) :: IsoMasters                              !< names of isomasters for iso table


    !< GPU variables
    integer, device :: Firsti_d                                                             !< working variable for interpolation
    integer, device :: Lasti_d                                                              !< working variable for interpolation
    integer, device :: stepi_d                                                              !< working variable for interpolation
    integer, device :: TotalNumberComponents_d                                              !< counter for total number of components of all mol.
    integer, device :: TotalNumberOfFrequencyRanges_d                                       !< total number of frequency ranges
    integer, device :: TotalNumberOfMolecules_d                                             !< total number of molecules including isotopologues
    integer, device :: NumberOfTemperatures_d                                               !< number of temperatures
    integer, device :: NumberFreeParameter_d                                                !< number of free parameter
    integer, dimension(:), allocatable, device :: NumberComponentsPerMolecule_d             !< number of components per molecule
    !        dimension: (0:TotalNumberOfMolecules)
    integer, dimension(:, :), allocatable, device :: IsoNfitConversionTable_d               !< iso ratios conversion table for free param. index
    !        dimension: (TotalNumberOfMolecules, TotalNumberOfMolecules)
    integer, dimension(:), allocatable, device :: SpectrumIndexForFreqRange_d               !< store spectrum index for each frequency range
    !        dimension: (TotalNumberOfFrequencyRanges)
    integer, dimension(:, :), allocatable, device :: DataPointIndexFreqRange_d              !< index of data pt. of first and last freq. in freq. range
    !        dimension: (TotalNumberOfFrequencyRanges, 2)
    integer, dimension(:, :, :), allocatable, device :: MolecularDataIndices_d              !< start and end index for each molecule and frequency range
    !        dimension: (TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2)
    integer, dimension(:, :), allocatable, device :: ConversionTableMAGIXmyXCLASSParam_d    !< conversion table between myXCLASS and MAGIX param.
    !        dimension: (NumberFreeParameter, 2)
    integer, dimension(:), allocatable, device :: CompMoleculeIndex_d                       !< molecule index for each component
    !        dimension: (TotalNumberComponents)
    real*8, device :: debug_freq_d                                                          !< frequency for debugging
    real*8, device :: ckms_d                                                                !< speed of light in km/s
    real*8, device :: pi_d                                                                  !< pi
    real*8, device :: TempLow_d                                                             !< working variable for extrapolation of part. func.
    real*8, device :: TempHigh_d                                                            !< working variable for extrapolation of part. func.
    real*8, dimension(:), allocatable, device :: TempPartFunc_d                             !< temperatures for partition function
    !<      dimension: (NumberOfTemperatures)
    real*8, dimension(:), allocatable, device :: BackgroundTemperatureRange_d               !< T_Back for each frequency range
    !<      dimension: (TotalNumberOfFrequencyRanges)
    real*8, dimension(:), allocatable, device :: TemperatureSlopeRange_d                    !< T_Slope for each frequency range
    !<      dimension: (TotalNumberOfFrequencyRanges)
    real*8, dimension(:), allocatable, device :: HydrogenColumnDensityRange_d               !< nH for each frequency range
    !<      dimension: (TotalNumberOfFrequencyRanges)
    real*8, dimension(:), allocatable, device :: DustBetaRange_d                            !< beta for each frequency range
    !<      dimension: (TotalNumberOfFrequencyRanges)
    real*8, dimension(:), allocatable, device :: KappaRange_d                               !< kappa for each frequency range
    !<      dimension: (TotalNumberOfFrequencyRanges)
    real*8, dimension(:), allocatable, device :: TelescopeSize_d                            !< size of telescope for each frequency range
    !<      dimension: (TotalNumberOfFrequencyRanges)
    real*8, dimension(:), allocatable, device :: StartFrequency_d                           !< first frequency for each frequency range
    !<      dimension: (TotalNumberOfFrequencyRanges)
    real*8, dimension(:, :), allocatable, device :: ObservationalDataList_d                 !< list containing all observational data
    !<      dimension: (TotalNumberDataPoints, 3)
    real*8, dimension(:, :), allocatable, device :: MolecularData_d                         !< array containing the molecular data for all molecules and
    !<      dimension: (TotalNumberOfTransitions, 4)
    character(len=40), dimension(:), allocatable, device :: MoleculeNames_d                 !< names of all molecules (including isotopologues)
    !<      dimension: (MaxNumberOfMolecules)
    real*8, dimension(:, :), allocatable, device :: IsoRatioConversionTable_d               !< table with iso ratios between iso master and molecule
    !<      dimension: (TotalNumberOfMolecules, TotalNumberOfMolecules)
    real*8, dimension(:, :), allocatable, device :: lgQ_d                                   !< lgQ entries of table PartitionFunction
    !<      dimension: (NumberOfTemperatures, NumberMoleculePartFunc)
    real*8, dimension(:, :), allocatable, device :: myXCLASSParameter_d                     !< array containing all molfit parameters for each component
    !<      dimension: (11, TotalNumberComponents)
    logical, device :: nHFlag_d                                                             !< flag for global setting of nH, kappa and beta
    logical, device :: tbFlag_d                                                             !< flag for global setting T_Back and T_Slope
    logical, device :: IsoFlag_d                                                            !< flag indicating use of isotopologues
    logical, device :: ModelFunctionFlag_d                                                  !< flag for saving model function values


    !< constant arrays for device subprogram
    integer, allocatable, dimension(:, :), device :: CopyCompMoleculeIndex                  !< working variable: local copy of CompMoleculeIndexOrig
    real*8, allocatable, dimension(:, :), device :: PartFunc                                !< working variable: local array for partition function
!    real*8, allocatable, dimension(:, :, :), device :: CopymyXCLASSParameter                !< working variable: local copy of myXCLASSParameterOrig
    real*8, allocatable, dimension(:, :, :), device :: CopyIsoRatioConversionTable          !< working variable: local copy of IsoRatioConversionTableOrig


    contains


        !>************************************************************************************************************************************************
        !> subroutine: CheckMolNames
        !>
        !> checks, if a molecule name is already known
        !>
        !>
        !> input variables:     TotalNumberOfMoleculesCopy: total number of all already known molecules
        !>                      CurrMol:                current molecule
        !>
        !> output variables:    AlreadyIncludedFlag:    flag indicating if current molecule is already known or not
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 27.07.2014
        !>
        subroutine CheckMolNames(AlreadyIncludedFlag, TotalNumberOfMoleculesCopy, CurrMol)

            implicit none
            integer :: i                                                                    !< loop variable
            integer :: TotalNumberOfMoleculesCopy                                           !< total number of all already known molecules
            character(len=40) :: CurrMol                                                    !< current molecule
            logical :: AlreadyIncludedFlag                                                  !< flag indicating if current molecule is already known or not


            AlreadyIncludedFlag = .false.                                                   !< initialize return parameter
            Do i = 1, TotalNumberOfMoleculesCopy
                if (trim(adjustl(MoleculeNames(i))) == trim(adjustl(CurrMol))) then
                    AlreadyIncludedFlag = .true.
                    exit
                endif
            end Do
            return
        end subroutine CheckMolNames


        !>------------------------------------------------------------------------------------------------------------------------------------------------
        !> subroutine to determine last non blank and non char(0) character of string st
        !>
        !>
        !> input variables:     st:                 string which has to be analyzed
        !>
        !> output variables:    val:                position of last non blank and non char(0) character of string st
        !>
        !> Who           When            What
        !> P. Schilke    08/12/2011      initial version
        !>
        subroutine strlen(val, st)

            implicit none
            integer :: i                                                                    !< working variable
            integer :: val                                                                  !< return value
            character :: st*(*)                                                             !< string which has to be analyzed

            ! Debug:
            ! print*,'st = ', st

            i = len(st)
            Do While ((st(i:i) == ' ' .or. st(i:i) == char(0)) .and. i > 1)
                i = i - 1
            end Do
            val = i
            return
        end subroutine strlen


        !>************************************************************************************************************************************************
        !> subroutine: GetmyXCLASSParameter
        !>
        !> read parameter for myXCLASS from paramset variable
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 27.07.2014
        !>
        subroutine GetmyXCLASSParameter

            implicit none
            integer :: i, j, k, l, m                                                        !< loop variables
            integer :: IndexComp                                                            !< index of current component
            integer :: CurrComp                                                             !< counts the current component
            integer :: IndexMolecule                                                        !< index of current molecule
            integer :: IndexFreqRange                                                       !< index of current frequency range
            integer :: IndexSpectrum                                                        !< index of current spectrum
            integer :: IndexCurrIso                                                         !< index of current iso ratio
            integer :: LastIndexFreqRange                                                   !< working variable
            integer :: FirstIndex, LastIndex                                                !< working variables for freq. range index determination
            integer :: MaxNumberOfMolecules                                                 !< working variable: max. number of molecules (incl. isotop.)
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            character(len=40) :: CurrNameMolecule                                           !< working variable: name of current molecule
            character(len=4096) :: ParamName                                                !< working variable: current name of variable without
                                                                                            !< extension
            logical :: AlreadyIncludedFlag                                                  !< working variable: indicates if current molecule name is
                                                                                            !< already known

            !< print what you do
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(9x, "Using GPU optimized versions of myXCLASS and MAGIX:")'
                print '(" ")'
                print '(" ")'
                print '(9x, "Initialize myXCLASS program:")'
                print '(" ")'
                print '(" ")'
                print '(9x, "Reading parameter for myXCLASS from paramset variable .. ", $)'
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< analyze parameter list, i.e. get all parameters wich are given in the molfit, iso-ratio, and experimental xml files
            !<      - get name of molecules and the corresponding number of components
            !<      - get parameters for each component
            !<      - get iso master, iso molecule, and iso ratios
            !<      - get frequency ranges, background temperatures and slopes, nH densities, kappa, beta, telescope sizes, and database file


            !< get general parameters from paramset variable
            dbName = "../../../Database/cdms_sqlite.db"                                     !< set default value for db file
            NumberMolecules = 0                                                             !< reset counter for total number of molecules
            TotalNumberOfFrequencyRanges = 0                                                !< total number of frequency ranges
            IsoFlag = .false.
            Do i = 1, parameternumber                                                       !< loop over all parameter

                ! Debug:
                ! print*,"FitParameterName(i) = ", trim(adjustl(FitParameterName(i)))


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif

                ! Debug:
                ! print*,"ParamName = ", trim(adjustl(ParamName))
                ! print*,"  paramset(1, i) = ", paramset(1, i)
                ! print*,"  FitParameterValue(i) = trim(adjustl(FitParameterValue(i)))


                !< get number of molecules (without iso molecules)
                if (ParamName == "Number_Molecules") then
                    NumberMolecules = int(paramset(1, i))


                !< get tbFlag
                elseif (ParamName == "tbFlag") then
                    if (trim(adjustl(FitParameterValue(i))) == "t" .or. trim(adjustl(FitParameterValue(i))) == "T" &
                        .or. trim(adjustl(FitParameterValue(i))) == "true" .or. trim(adjustl(FitParameterValue(i))) == "True" &
                        .or. trim(adjustl(FitParameterValue(i))) == "TRUE" .or. trim(adjustl(FitParameterValue(i))) == "y" &
                        .or. trim(adjustl(FitParameterValue(i))) == "Y" .or. trim(adjustl(FitParameterValue(i))) == "yes" &
                        .or. trim(adjustl(FitParameterValue(i))) == "YES" .or. trim(adjustl(FitParameterValue(i))) == "Yes") then
                        tbFlag = .false.
                    else
                        tbFlag = .true.
                    endif


                !< get nHFlag
                elseif (ParamName == "nHFlag") then
                    if (trim(adjustl(FitParameterValue(i))) == "t" .or. trim(adjustl(FitParameterValue(i))) == "T" &
                        .or. trim(adjustl(FitParameterValue(i))) == "true" .or. trim(adjustl(FitParameterValue(i))) == "True" &
                        .or. trim(adjustl(FitParameterValue(i))) == "TRUE" .or. trim(adjustl(FitParameterValue(i))) == "y" &
                        .or. trim(adjustl(FitParameterValue(i))) == "Y" .or. trim(adjustl(FitParameterValue(i))) == "yes" &
                        .or. trim(adjustl(FitParameterValue(i))) == "YES" .or. trim(adjustl(FitParameterValue(i))) == "Yes") then
                        nHFlag = .false.
                    else
                        nHFlag = .true.
                    endif


                !< get number of frequency ranges
                elseif (ParamName == "MAGIXImportStartFrequency") then
                    TotalNumberOfFrequencyRanges = TotalNumberOfFrequencyRanges + 1


                !< get iso flag
                elseif (ParamName == "NumberOfISOLines") then
                    if (paramset(1, i) > 0) then
                        NumberOfIsomasters = int(paramset(1, i))
                        IsoFlag = .true.


                        !< allocate memory for myXCLASSParameter variable
                        if (allocated(IsoMolecule)) then
                            deallocate(IsoMolecule, IsoMasters, IsoRatio, stat = deallocstatus)
                            if (deallocstatus /= 0) then
                                Do ErrChannel = 6, 6
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                                    write(ErrChannel, '(3x,"Can not deallocate variables myXCLASSParameter, CompMoleculeIndex etc.!")')
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(3x,"Please restart the program!")')
                                end Do
                                stop 'Program aborted!'
                            endif
                        endif
                        allocate(IsoMolecule(NumberOfIsomasters), IsoMasters(NumberOfIsomasters), IsoRatio(NumberOfIsomasters, 2), stat = allocstatus)
                        if (allocstatus /= 0) then
                            Do ErrChannel = 6, 6
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                                write(ErrChannel, '(3x,"Can not allocate variables myXCLASSParameter, CompMoleculeIndex etc.!")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Please restart the program!")')
                            end Do
                            stop 'Program aborted!'
                        endif
                        IsoMolecule = ""
                        IsoMasters = ""
                        IsoRatio = 0.d0
                        IndexCurrIso = 0
                    endif


                !< get IsoMolecule
                elseif (ParamName == "IsoMolecule" .and. IsoFlag) then
                    IndexCurrIso = IndexCurrIso + 1
                    IsoMolecule(IndexCurrIso) = trim(adjustl(FitParameterValue(i)))


                !< get IsoMasters
                elseif (ParamName == "IsoMasters" .and. IsoFlag) then
                    IsoMasters(IndexCurrIso) = trim(adjustl(FitParameterValue(i)))


                !< get IsoRatio
                elseif (ParamName == "IsoRatio" .and. IsoFlag) then
                    IsoRatio(IndexCurrIso, 1) = paramset(1, i)                              !< store the value of the iso ratio
                    if (paramset(2, i) == 1) then
                        IsoRatio(IndexCurrIso, 2) = sum(paramset(2, 1:i))                   !< store the free parameter index for this iso ratio param.
                    else
                        IsoRatio(IndexCurrIso, 2) = 0
                    endif


                !< get path and name of sqlite3 database file
                elseif (ParamName == "MAGIXImportdbFilename") then
                    if (trim(adjustl(FitParameterValue(i))) /= "") then
                        dbName = trim(adjustl(FitParameterValue(i)))
                    endif
                endif
            end Do

            ! Debug:
            !    print*,"NumberMolecules = ", NumberMolecules
            !    print*,"tbFlag = ", tbFlag
            !    print*,"nHFlag = ", nHFlag
            !    print*,"TotalNumberOfFrequencyRanges = ", TotalNumberOfFrequencyRanges
            !    print*,"dbName = ", trim(adjustl(dbName))
            !    print*,"IsoFlag = ", IsoFlag
            !    if (IsoFlag) then
            !        print*,"NumberOfIsomasters = ", NumberOfIsomasters
            !        Do i = 1, NumberOfIsomasters
            !            print*,"IsoMolecule(i) = ", trim(adjustl(IsoMolecule(i)))
            !            print*,"IsoMasters(i) = ", trim(adjustl(IsoMasters(i)))
            !            print*,"IsoRatio(i, :) = ", IsoRatio(i, :)
            !        end Do
            !    endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< allocate memory for myXCLASSParameter variable
            if (allocated(StartFrequency)) then
                deallocate(StartFrequency, EndFrequency, StepFrequency, TelescopeSize, BackgroundTemperatureRange, &
                           TemperatureSlopeRange, HydrogenColumnDensityRange, DustBetaRange, KappaRange, SpectrumIndexForFreqRange, &
                           ObservationalDataList, DataPointIndexFreqRange, MoleculeNames, ConversionTableMAGIXmyXCLASSParam, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables StartFrequency, EndFrequency etc.!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            TotalNumberDataPoints = sum(lengthexpdata(1:NumberExpFiles))
            MaxNumberOfMolecules = NumberMolecules + 2 * NumberOfIsomasters
            allocate(StartFrequency(TotalNumberOfFrequencyRanges), EndFrequency(TotalNumberOfFrequencyRanges), &
                     StepFrequency(TotalNumberOfFrequencyRanges), TelescopeSize(TotalNumberOfFrequencyRanges), &
                     BackgroundTemperatureRange(TotalNumberOfFrequencyRanges), TemperatureSlopeRange(TotalNumberOfFrequencyRanges), &
                     HydrogenColumnDensityRange(TotalNumberOfFrequencyRanges), DustBetaRange(TotalNumberOfFrequencyRanges), &
                     KappaRange(TotalNumberOfFrequencyRanges), SpectrumIndexForFreqRange(TotalNumberOfFrequencyRanges), &
                     ObservationalDataList(TotalNumberDataPoints, 3), DataPointIndexFreqRange(TotalNumberOfFrequencyRanges, 2), &
                     MoleculeNames(MaxNumberOfMolecules), ConversionTableMAGIXmyXCLASSParam(NumberFreeParameter, 2), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannel = 6, 6
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x,"Can not allocate variables StartFrequency, EndFrequency etc.!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            StartFrequency = 0.d0
            EndFrequency = 0.d0
            StepFrequency = 0.d0
            TelescopeSize = 0.d0
            BackgroundTemperatureRange = 0.d0
            TemperatureSlopeRange = 0.d0
            HydrogenColumnDensityRange = 0.d0
            DustBetaRange = 0.d0
            KappaRange = 0.d0
            SpectrumIndexForFreqRange = 0
            ObservationalDataList = 0.d0
            DataPointIndexFreqRange = 0
            MoleculeNames = ""
            ConversionTableMAGIXmyXCLASSParam = 0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< get parameter for each frequency range
            IndexFreqRange = 0                                                              !< reset index for current frequency range
            IndexSpectrum = 0                                                               !< reset index for current spectrum
            LastIndexFreqRange = 0
            Do i = 1, parameternumber                                                       !< loop over all parameter


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif


                !< get size of telescope
                if (ParamName == "MAGIXImportExpFileNames") then
                    IndexSpectrum = IndexSpectrum + 1


                !< get first frequency of each frequency range
                elseif (ParamName == "MAGIXImportStartFrequency") then
                    IndexFreqRange = IndexFreqRange + 1
                    StartFrequency(IndexFreqRange) = paramset(1, i)
                    SpectrumIndexForFreqRange(IndexFreqRange) = IndexSpectrum


                !< get last frequency of each frequency range
                elseif (ParamName == "MAGIXImportEndFrequency") then
                    EndFrequency(IndexFreqRange) = paramset(1, i)


                !< get stepsize of each frequency range
                elseif (ParamName == "MAGIXImportStepFrequency") then
                    StepFrequency(IndexFreqRange) = paramset(1, i)


                !< get global definition for background temperature
                elseif (ParamName == "MAGIXImportBackgroundTemperature" .and. tbFlag) then
                    BackgroundTemperatureRange(IndexFreqRange) = paramset(1, i)


                !< get global definition for temperature slope
                elseif (ParamName == "MAGIXImportTemperatureSlope" .and. tbFlag) then
                    TemperatureSlopeRange(IndexFreqRange) = paramset(1, i)


                !< get global definition for n_H
                elseif (ParamName == "MAGIXImportHydrogenColumnDensity" .and. nHFlag) then
                    HydrogenColumnDensityRange(IndexFreqRange) = paramset(1, i)


                !< get global definition for dust beta
                elseif (ParamName == "MAGIXImportDustBeta" .and. nHFlag) then
                    DustBetaRange(IndexFreqRange) = paramset(1, i)


                !< get global definition for kappa
                elseif (ParamName == "MAGIXImportKappa" .and. nHFlag) then
                    KappaRange(IndexFreqRange) = paramset(1, i)


                !< get size of telescope
                elseif (ParamName == "MAGIXImportTelescopeSize") then
                    TelescopeSize((LastIndexFreqRange + 1):IndexFreqRange) = paramset(1, i)
                    LastIndexFreqRange = IndexFreqRange                                     !< variable 'LastIndexFreqRange' is necessary, because
                                                                                            !< size of telescope is given only for each spectrum and not
                                                                                            !< for each frequency range
                endif
            end Do

            ! Debug:
            !    print*,"SpectrumIndexForFreqRange = ", SpectrumIndexForFreqRange(:)
            !    print*,"StartFrequency = ", StartFrequency(:)
            !    print*,"EndFrequency = ", EndFrequency(:)
            !    print*,"StepFrequency = ", StepFrequency(:)
            !    print*,"BackgroundTemperatureRange = ", BackgroundTemperatureRange(:)
            !    print*,"TemperatureSlopeRange = ", TemperatureSlopeRange(:)
            !    print*,"HydrogenColumnDensityRange = ", HydrogenColumnDensityRange(:)
            !    print*,"DustBetaRange = ", DustBetaRange(:)
            !    print*,"KappaRange = ", KappaRange(:)
            !    print*,"TelescopeSize = ", TelescopeSize(:)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< prepare observational data as one long 2D list
            k = 0
            Do i = 1, IndexSpectrum                                                         !< variable 'IndexSpectrum' is NOT identical to variable
                                                                                            !< 'NumberExpFiles' !!!!!!!!!!!
                Do j = 1, lengthexpdata(i)                                                  !< loop over all points of ith spectrum
                    k = k + 1                                                               !< increase data point counter
                    ObservationalDataList(k, :) = (/ expdatax(i, j, 1), expdatay(i, j, 1), expdatae(i, j, 1) /)
                end Do
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< define indices of first and last frequency (data point) of each frequency range), respectively
            Do l = 1, TotalNumberOfFrequencyRanges                                          !< loop over all frequency ranges
                i = SpectrumIndexForFreqRange(l)                                            !< get spectrum index
                FirstIndex = 0
                LastIndex = 0
                Do j = 1, lengthexpdata(i)                                                  !< loop over all points of ith spectrum
                    if (expdatax(i, j, 1) <= StartFrequency(l)) then
                        FirstIndex = FirstIndex + 1
                    endif
                    if (expdatax(i, j, 1) <= EndFrequency(l)) then
                        LastIndex = LastIndex + 1
                    else
                        exit
                    endif
                end Do


                !< check indices
                if (FirstIndex == 0) then
                    FirstIndex = 1
                endif
                if (LastIndex == 0) then
                    LastIndex = 1
                endif
                if (FirstIndex > LastIndex) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x,"An error occurred by calculating the indices of each frequency range!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                else


                    !< determine offset for indices for each frequency range
                    k = 0
                    if (i > 1) then
                        k = sum(lengthexpdata(1:(i - 1)))
                    endif
                    DataPointIndexFreqRange(l, :) = (/ FirstIndex + k, LastIndex + k /)
                endif
            end Do

            ! Debug:
            !    print*,"DataPointIndexFreqRange(:, 1) = ", DataPointIndexFreqRange(:, 1)
            !    print*,"DataPointIndexFreqRange(:, 2) = ", DataPointIndexFreqRange(:, 2)
            !    print*,"ObservationalDataList(DataPointIndexFreqRange(1, 1), :) = ", ObservationalDataList(DataPointIndexFreqRange(1, 1), :)
            !    print*,"ObservationalDataList(DataPointIndexFreqRange(1, 2), :) = ", ObservationalDataList(DataPointIndexFreqRange(1, 2), :)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine list of all molecules which are defined in molfit and iso ratio files
            TotalNumberOfMolecules = 0                                                      !< reset counter for total number of molecules
            Do i = 1, parameternumber                                                       !< loop over all parameter


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif


                !< determine molecule index
                if (ParamName == "Molecule_Name") then
                    CurrNameMolecule = trim(adjustl(FitParameterValue(i)))
                    call CheckMolNames(AlreadyIncludedFlag, TotalNumberOfMolecules, CurrNameMolecule)
                    if (.not. AlreadyIncludedFlag) then
                        TotalNumberOfMolecules = TotalNumberOfMolecules + 1
                        MoleculeNames(TotalNumberOfMolecules) = trim(adjustl(CurrNameMolecule))
                    endif


                    !< check, if iso ratios are defined and determine total number of molecules
                    if (IsoFlag) then
                        Do j = 1, NumberOfIsomasters
                            if (trim(adjustl(CurrNameMolecule)) == trim(adjustl(IsoMolecule(j)))) then
                                call CheckMolNames(AlreadyIncludedFlag, TotalNumberOfMolecules, IsoMasters(j))
                                if (.not. AlreadyIncludedFlag) then
                                    TotalNumberOfMolecules = TotalNumberOfMolecules + 1
                                    MoleculeNames(TotalNumberOfMolecules) = trim(adjustl(IsoMasters(j)))
                               endif
                            elseif (trim(adjustl(CurrNameMolecule)) == trim(adjustl(IsoMasters(j)))) then
                                call CheckMolNames(AlreadyIncludedFlag, TotalNumberOfMolecules, IsoMolecule(j))
                                if (.not. AlreadyIncludedFlag) then
                                    TotalNumberOfMolecules = TotalNumberOfMolecules + 1
                                    MoleculeNames(TotalNumberOfMolecules) = trim(adjustl(IsoMolecule(j)))
                               endif
                            endif
                        end Do
                    endif
                endif
            end Do

            ! Debug:
            ! print*," "
            ! Do i = 1, TotalNumberOfMolecules
            !     print*,"GetmyXCLASSParameter:    i, MoleculeNames(i) = ", i, MoleculeNames(i)
            ! end Do


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< prepare some variables for isotopologues only


            !< allocate memory for iso conversion tables
            if (allocated(IsoRatioConversionTable)) then
                deallocate(IsoRatioConversionTable, IsoNfitConversionTable, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables IsoNfitConversionTable and IsoRatioConversionTable!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            allocate(IsoNfitConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules), &
                     IsoRatioConversionTable(TotalNumberOfMolecules, TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannel = 6, 6
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x,"Can not allocate variables IsoNfitConversionTable and IsoRatioConversionTable!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            IsoNfitConversionTable = 0
            IsoRatioConversionTable = 0.d0



            !< define conversion tables IsoNfitConversionTable and IsoRatioConversionTable
            if (IsoFlag) then
                Do i = 1, parameternumber                                                   !< loop over all parameter


                    !< remove [[..]] expansion of parameter name
                    ParamName = trim(adjustl(FitParameterName(i)))
                    k = index(ParamName, "[[")
                    if (k > 0) then
                        ParamName = ParamName(:(k - 1))
                    endif


                    !< determine molecule index for current molecule
                    if (ParamName == "Molecule_Name") then
                        CurrNameMolecule = trim(adjustl(FitParameterValue(i)))
                        IndexMolecule = 1
                        Do k = 1, TotalNumberOfMolecules
                            if (trim(adjustl(CurrNameMolecule)) == trim(adjustl(MoleculeNames(k)))) then
                                IndexMolecule = k
                                exit
                            endif
                        end Do


                        !< create conversion table between molecules and isotopologues
                        Do j = 1, NumberOfIsomasters                                        !< loop over all defined isotopologues

                            ! Debug:
                            ! print*,"j, CurrNameMolecule = ", j, trim(adjustl(CurrNameMolecule))
                            ! print*,"IsoMolecule(j), IsoMasters(j), IsoRatio(j, 1) = ", trim(adjustl(IsoMolecule(j))), "    ", &
                            !                                                            trim(adjustl(IsoMasters(j))), "   ", IsoRatio(j, 1)


                            if (trim(adjustl(CurrNameMolecule)) == trim(adjustl(IsoMolecule(j)))) then
                                l = IndexMolecule
                                m = 0
                                Do k = 1, TotalNumberOfMolecules
                                    if (trim(adjustl(IsoMasters(j))) == trim(adjustl(MoleculeNames(k)))) then
                                        m = k
                                        exit
                                    endif
                                end Do
                                IsoRatioConversionTable(l, m) = IsoRatio(j, 1)
                                IsoNfitConversionTable(l, m) = int(IsoRatio(j, 2))

                                ! Debug:
                                ! print*,"IsoMolecule(j), IsoRatio(j, 1) = ", trim(adjustl(IsoMolecule(j))), "    ", IsoRatio(j, 1)


                            elseif (trim(adjustl(CurrNameMolecule)) == trim(adjustl(IsoMasters(j)))) then
                                l = IndexMolecule
                                m = 0
                                Do k = 1, TotalNumberOfMolecules
                                    if (trim(adjustl(IsoMolecule(j))) == trim(adjustl(MoleculeNames(k)))) then
                                        m = k
                                        exit
                                    endif
                                end Do
                                if (dabs(IsoRatio(j, 1)) > 1.e-30) then
                                    IsoRatioConversionTable(l, m) = 1.d0 / IsoRatio(j, 1)
                                    IsoNfitConversionTable(l, m) = -int(IsoRatio(j, 2))
                                else
                                    IsoRatioConversionTable(l, m) = 1.d0
                                    IsoNfitConversionTable(l, m) = -int(IsoRatio(j, 2))
                                endif

                                ! Debug:
                                ! print*,"IsoMasters(j), 1.d0 / IsoRatio(j, 1) = ", trim(adjustl(IsoMasters(j))), "    ", 1.d0 / IsoRatio(j, 1)

                            endif
                        end Do
                    endif
                end Do
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< determine the number of components for each molecule (including isotopologues if selected)
            if (allocated(NumberComponentsPerMolecule)) then
                deallocate(NumberComponentsPerMolecule, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x,"Can not deallocate variable NumberComponentsPerMolecule!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            allocate(NumberComponentsPerMolecule(0:TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannel = 6, 6
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x,"Can not allocate variable NumberComponentsPerMolecule!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            NumberComponentsPerMolecule = 0
            IndexMolecule = 1
            Do i = 1, parameternumber                                                       !< loop over all parameter


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif


                !< determine molecule index
                if (ParamName == "Molecule_Name") then
                    CurrNameMolecule = trim(adjustl(FitParameterValue(i)))
                    IndexMolecule = 1
                    Do k = 1, TotalNumberOfMolecules
                        if (trim(adjustl(CurrNameMolecule)) == trim(adjustl(MoleculeNames(k)))) then
                            IndexMolecule = k
                            exit
                        endif
                    end Do


                !< get number of components for current molecule
                elseif (ParamName == "Number_MolLine") then
                    NumberComponentsPerMolecule(IndexMolecule) = int(paramset(1, i))
                endif
            end Do


            !< special handling for isotopologues
            if (IsoFlag) then
                Do k = 1, TotalNumberOfMolecules

                    ! Debug:
                    ! print*,'>>>k, NumberComponentsPerMolecule(k) = ', k, NumberComponentsPerMolecule(k)
                    ! print*,'IsoRatioConversionTable(k, :) = ', IsoRatioConversionTable(k, :)


                    Do j = 1, TotalNumberOfMolecules

                        ! Debug:
                        ! print*,'j, IsoRatioConversionTable(k, j) = ', j, IsoRatioConversionTable(k, j)


                        if (NumberComponentsPerMolecule(j) == 0 .and. IsoRatioConversionTable(k, j) /= 0.d0 .and. k /= j) then
                            NumberComponentsPerMolecule(j) = NumberComponentsPerMolecule(k)

                            ! Debug:
                            ! print*,'-->j, NumberComponentsPerMolecule(j) = ', j, NumberComponentsPerMolecule(j)
                        endif
                    end Do
                end Do
            endif


            !< determine total number of components and allocate memory
            TotalNumberComponents = sum(NumberComponentsPerMolecule(:))
            if (allocated(myXCLASSParameter)) then
                deallocate(myXCLASSParameter, CompMoleculeIndex, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannel = 6, 6
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables myXCLASSParameter, CompMoleculeIndex etc.!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            allocate(myXCLASSParameter(11, TotalNumberComponents), CompMoleculeIndex(TotalNumberComponents), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannel = 6, 6
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetmyXCLASSParameter!")')
                    write(ErrChannel, '(3x,"Can not allocate variables myXCLASSParameter, CompMoleculeIndex etc.!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            myXCLASSParameter = 0.d0
            CompMoleculeIndex = 1


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< get calculation parameter: source_size
            IndexMolecule = 1                                                               !< reset index for current molecule
            IndexComp = 0                                                                   !< reset index for current component
            CurrComp = 0                                                                    !< reset index for current component of current molecule
            Do i = 1, parameternumber                                                       !< loop over all parameter


                !< remove [[..]] expansion of parameter name
                ParamName = trim(adjustl(FitParameterName(i)))
                k = index(ParamName, "[[")
                if (k > 0) then
                    ParamName = ParamName(:(k - 1))
                endif


                !< determine molecule index
                if (ParamName == "Molecule_Name") then
                    CurrNameMolecule = trim(adjustl(FitParameterValue(i)))
                    IndexMolecule = 0
                    Do k = 1, TotalNumberOfMolecules
                        if (trim(adjustl(CurrNameMolecule)) == trim(adjustl(MoleculeNames(k)))) then
                            IndexMolecule = k
                            exit
                        endif
                    end Do
                    if (IndexMolecule == 0) then
                        print '(" ")'
                        print '(2x, "Error in subroutine ReadInMolfitsParameter:")'
                        print '(4x, "Can not determine molecule index for molecule: ", A, " ")', trim(adjustl(CurrNameMolecule))
                        print '(4x, "Set index to 1!")'
                        print '(" ")'
                        print '(4x,"Known molecules:")'
                        Do k = 1, TotalNumberOfMolecules
                            print '(4x, I5, 3x, A)', k, trim(adjustl(MoleculeNames(k)))
                        end Do
                        IndexMolecule = 1
                    endif


                !< determine molecule index
                elseif (ParamName == "Number_MolLine") then
                    CurrComp = 0
                    IndexComp = sum(NumberComponentsPerMolecule(:(IndexMolecule - 1)))


                !< get source size
                elseif (ParamName == "source_size") then
                    IndexComp = IndexComp + 1                                               !< increase indec for current component
                    CurrComp = CurrComp + 1
                    CompMoleculeIndex(IndexComp) = IndexMolecule
                    myXCLASSParameter(1, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 1/)
                    endif


                !< get T_rot
                elseif (ParamName == "T_rot") then
                    myXCLASSParameter(2, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 2/)
                    endif


                    !< check for zero temperature
                    if (myXCLASSParameter(2, IndexComp) == 0.d0) then
                        Do ErrChannelIndex = 1, 1
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(2x, "Error in subroutine ReadInMolfitsParameter:")')
                            write(ErrChannel, '(4x, "The rotation temperature for component ", I4, " of molecule ", A, " ")') CurrComp, &
                                                                                                          trim(adjustl(CurrNameMolecule))
                            write(ErrChannel, '(4x, "is identical to zero!")')
                            write(ErrChannel, '(4x, "Set rotation temperature for this component to 1.0 to avoid division by zero!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                        end Do
                        myXCLASSParameter(2, IndexComp) = 1.d0
                    endif


                !< get N_tot
                elseif (ParamName == "N_tot") then
                    myXCLASSParameter(3, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 3/)
                    endif


                    !< check for overflow
                    if (myXCLASSParameter(3, IndexComp) > 300.d0) then
                        Do ErrChannelIndex = 1, 1
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(2x, "Error in subroutine GetmyXCLASSParameter:")')
                            write(ErrChannel, '(4x, "The log10 value of column density for component ", I4, " of molecule ", A, " ")') CurrComp, &
                                                                                                                   trim(adjustl(CurrNameMolecule))
                            write(ErrChannel, '(4x, "causes an overflow (to large number)!")')
                            write(ErrChannel, '(4x, "Set log10 value for this column density to 300!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(" ")')
                        end Do
                        myXCLASSParameter(3, IndexComp) = 300.d0
                    endif


                    !< convert log10 value back to linear value
                    myXCLASSParameter(3, IndexComp) = 10.d0**myXCLASSParameter(3, IndexComp)

                    ! Debug:
                    ! print*,'paramset(1, i) = ', paramset(1, i)
                    ! print*,'IndexComp, myXCLASSParameter(3, IndexComp) = ', IndexComp, myXCLASSParameter(3, IndexComp)


                !< get V_width
                elseif (ParamName == "V_width") then
                    myXCLASSParameter(4, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 4/)
                    endif


                !< get V_off
                elseif (ParamName == "V_off") then
                    myXCLASSParameter(5, IndexComp) = paramset(1, i)
                    if (paramset(2, i) == 1) then
                        k = int(sum(paramset(2, 1:i)))                                      !< index of free parameter
                        ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 5/)
                    endif


                !< get AbsorptionFlag
                elseif (ParamName == "AbsorptionFlag") then
                    if (trim(adjustl(FitParameterValue(i))) == "a") then
                        myXCLASSParameter(11, IndexComp) = 2.d0
                    else
                        myXCLASSParameter(11, IndexComp) = 1.d0
                    endif
                endif


                !< get T_Back and T_Slope for each component
                if (.not. tbFlag) then


                    !< get T_Back
                    if (ParamName == "T_Back") then
                        myXCLASSParameter(6, IndexComp) = paramset(1, i)
                        if (paramset(2, i) == 1) then
                            k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                            ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 6/)
                        endif


                    !< get T_Slope
                    elseif (ParamName == "T_Slope") then
                        myXCLASSParameter(7, IndexComp) = paramset(1, i)
                        if (paramset(2, i) == 1) then
                            k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                            ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 7/)
                        endif
                    endif
                endif


                !< get nH, kappa, and beta for each component
                if (.not. nHFlag) then


                    !< get nHcolumn
                    if (ParamName == "nHcolumn") then
                        myXCLASSParameter(8, IndexComp) = paramset(1, i)
                        if (paramset(2, i) == 1) then
                            k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                            ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 8/)
                        endif


                        !< check for overflow
                        if (myXCLASSParameter(8, IndexComp) > 300.d0) then
                            Do ErrChannelIndex = 1, 1
                                ErrChannel = AllErrChannels(ErrChannelIndex)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(2x, "Error in subroutine GetmyXCLASSParameter:")')
                                write(ErrChannel, '(4x, "The log10 value of hydrogen column density for component ", I4, " of molecule ", A, " ")') &
                                                                                                            CurrComp, trim(adjustl(CurrNameMolecule))

                                write(ErrChannel, '(4x, "causes an overflow (to large number)!")')
                                write(ErrChannel, '(4x, "Set log10 value for this column density to 300!")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                            end Do
                            myXCLASSParameter(8, IndexComp) = 300.d0
                        endif


                        !< convert log10 value back to linear value
                        myXCLASSParameter(8, IndexComp) = 10.d0**myXCLASSParameter(8, IndexComp)


                    !< get beta
                    elseif (ParamName == "beta") then
                        myXCLASSParameter(9, IndexComp) = paramset(1, i)
                        if (paramset(2, i) == 1) then
                            k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                            ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 10/)
                        endif


                    !< get kappa
                    elseif (ParamName == "kappa") then
                        myXCLASSParameter(10, IndexComp) = paramset(1, i) * 2.d0 * 1.66d-24 / 100.d0     !< kappa_1300 * M(H2)/M_d_M_g
                        if (paramset(2, i) == 1) then
                            k = int(sum(paramset(2, 1:i)))                                  !< index of free parameter
                            ConversionTableMAGIXmyXCLASSParam(k, :) = (/IndexComp, 9/)
                        endif
                    endif
                endif
            end Do

            ! Debug:
            !    print*,"myXCLASSParameter(1, :) = ", myXCLASSParameter(1, :)
            !    print*,"myXCLASSParameter(2, :) = ", myXCLASSParameter(2, :)
            !    print*,"myXCLASSParameter(3, :) = ", myXCLASSParameter(3, :)
            !    print*,"myXCLASSParameter(4, :) = ", myXCLASSParameter(4, :)
            !    print*,"myXCLASSParameter(5, :) = ", myXCLASSParameter(5, :)
            !    print*,"myXCLASSParameter(6, :) = ", myXCLASSParameter(6, :)
            !    print*,"myXCLASSParameter(7, :) = ", myXCLASSParameter(7, :)
            !    print*,"myXCLASSParameter(8, :) = ", myXCLASSParameter(8, :)
            !    print*,"myXCLASSParameter(9, :) = ", myXCLASSParameter(9, :)
            !    print*,"myXCLASSParameter(10, :) = ", myXCLASSParameter(10, :)
            !    print*,"myXCLASSParameter(11, :) = ", myXCLASSParameter(11, :)
            !    if (IsoFlag) then
            !        Do i = 1, TotalNumberOfMolecules
            !            print*,"i, IsoRatioConversionTable(i, :) = ", i, IsoRatioConversionTable(i, :)
            !        end Do
            !    endif


            !< print what you do
            if (printflag) then
                print '("done!")'
            endif


            return
        end subroutine GetmyXCLASSParameter


        !>************************************************************************************************************************************************
        !> subroutine: GetPartitionFunction
        !>
        !> get partition function from sqlite3 database
        !>
        !> input variables:     none
        !>
        !> output variables:    ok:             status variable
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 29.07.2014
        !>
        subroutine GetPartitionFunction(ok)

            implicit none
            integer :: ok                                                                       !< status variable
            integer :: number_entries                                                           !< number of entries for the PartitionFunction table
            integer :: countlen                                                                 !< total length of query
            integer :: i, j, k, l                                                               !< loop/working variables
            integer :: allocstatus, deallocstatus                                               !< variables for (de)allocation
            real*8, allocatable, dimension(:, :) :: CopylgQ                                     !< copy of lgQ variable
            character(len=16384) :: queryString1                                                !< query string (first part)
            character(len=16384) :: queryString2                                                !< query string (second part)
            character(len=4096) :: dbNameCopy                                                   !< name of database
            character(len=40) :: ColumnNameForNamePartFuncCopy                                  !< copy of name of column including the names of molecules
            character(len=40), dimension(TotalNumberOfMolecules) :: CopyMoleculeNames           !< copy of names of molecules


            !< print what you do
            if (printflag) then
                print '(9x, "Reading partition functions for all molecules from sqlite3 database .. ", $)'
            endif



            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define query string


            !< first part
            queryString1 = "SELECT count(*) FROM " // trim(adjustl(NameOfPartFuncTable)) // char(0)
            queryString2 = " " // char(0)


            !< create query string
            ok = 0                                                                          !< reset status variable
            number_entries = 0                                                              !< reset result variable
            j = TotalNumberOfMolecules
            ColumnNameForNamePartFuncCopy = trim(adjustl(ColumnNameForNamePartFunc)) // char(0) !< get name of column for molecule names
            countlen = (len_trim(queryString1) - 1)                                         !< length of string queryString1
            countlen = countlen + 8                                                         !< length plus ' WHERE ('
            CopyMoleculeNames = ""
            Do k = 1, TotalNumberOfMolecules                                                !< loop over all molecules
                CopyMoleculeNames(k) = char(34) // trim(adjustl(MoleculeNames(k))) // char(34) // char(0)
                countlen = countlen + len_trim(CopyMoleculeNames(k)) - 1                    !< length plus length of name (incl. '""')
                countlen = countlen + len_trim(ColumnNameForNamePartFuncCopy)               !< length plus 'PF_Name'
                countlen = countlen + 7                                                     !< length plus ' or ' and ' = '
            end Do
            countlen = countlen + 2                                                         !< length plus ') '
            countlen = countlen + (len_trim(queryString2) - 1)                              !< length plus length of string queryString2
            countlen = countlen + len_trim(ColumnNameForNamePartFuncCopy) + 2               !< length plus 'PF_Name'
            dbNameCopy = trim(adjustl(dbName)) // char(0)

            ! Debug:
            ! print*,' '
            ! print*,'queryString1 = >', trim(adjustl(queryString1)), '<'
            ! print*,'queryString2 = >', trim(adjustl(queryString2)), '<'
            ! print*,'countlen = ', countlen


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get number of entries for the partition function table
            call numentr_sqlite3(ok, number_entries, %val(countlen), trim(adjustl(queryString1)), trim(adjustl(queryString2)), %val(j), &
                                 trim(adjustl(dbNameCopy)), trim(adjustl(ColumnNameForNamePartFuncCopy)), CopyMoleculeNames)
            NumberMoleculePartFunc = number_entries

            ! Debug:
            ! print*,'NumberMoleculePartFunc = ', NumberMoleculePartFunc
            ! print*,'ok = ', ok


            !< allocate variables for partition function
            if (NumberMoleculePartFunc > 0 .and. ok == 0) then
                if (allocated(MolNamePartFunc)) then
                    deallocate(MolNamePartFunc, lgQ, CopylgQ, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        Do ErrChannelIndex = 1, 1
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(1x,"Error in subroutine GetPartitionFunction!")')
                            write(ErrChannel, '(3x,"Can not deallocate variables MolNamePartFunc and lgQ!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(3x,"Please restart the program!")')
                        end Do
                        stop 'Program aborted!'
                    endif
                endif
                allocate(MolNamePartFunc(NumberMoleculePartFunc), lgQ(NumberOfTemperatures, NumberMoleculePartFunc), &
                         CopylgQ(NumberOfTemperatures, NumberMoleculePartFunc), stat = allocstatus)
                if (allocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetPartitionFunction!")')
                        write(ErrChannel, '(3x,"Can not allocate variables MolNamePartFunc and lgQ!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
                MolNamePartFunc = ""
                lgQ = 0.d0
                CopylgQ = 0.d0


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< construct query string
                ok = 0                                                                      !< reset status variable
                i = NumberMoleculePartFunc
                queryString1 = "SELECT " // trim(adjustl(ColumnNameForNamePartFunc))
                Do j = 1, NumberOfTemperatures
                    queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNamesPartFunc(j)))
                end Do
                queryString1 = trim(adjustl(queryString1)) // " FROM " // trim(adjustl(NameOfPartFuncTable)) // char(0)

                ! Debug:
                ! print*,"queryString1 = ", queryString1


                !< get information from sqlite3 database
                ok = 0                                                                      !< reset status variable
                l = NumberOfTemperatures
                i = NumberMoleculePartFunc
                j = TotalNumberOfMolecules
                countlen = (len_trim(queryString1) - 1)
                CopyMoleculeNames = ""
                Do k = 1, TotalNumberOfMolecules
                    CopyMoleculeNames(k) = char(34) // trim(adjustl(MoleculeNames(k))) // char(34) // char(0)
                    countlen = countlen + len_trim(CopyMoleculeNames(k)) + 14
                end Do
                countlen = countlen + (len_trim(queryString2) - 1) + 17


                !< bug fix: prevent segmentation fault, if database contains double entries
                if (NumberMoleculePartFunc > TotalNumberOfMolecules) then
                    Do k = TotalNumberOfMolecules, NumberMoleculePartFunc
                        countlen = countlen + 40 + 14
                    end Do
                endif


                !< get information from sqlite3 database
                ! ok,   status
                ! i,    number of entries in the database
                ! l,    number of temperatures
                ! j,    number of molecules in the query string
                dbNameCopy = trim(adjustl(dbName)) // char(0)
                ColumnNameForNamePartFuncCopy = trim(adjustl(ColumnNameForNamePartFunc)) // char(0)
                call ReadPartFunc(ok, %val(l), %val(i), %val(j), %val(countlen), trim(adjustl(queryString1)), trim(adjustl(queryString2)), &
                                  trim(adjustl(dbNameCopy)), trim(adjustl(ColumnNameForNamePartFuncCopy)), CopyMoleculeNames, lgQ, MolNamePartFunc)

                ! Debug:
                ! print*,'NumberMoleculePartFunc = ', NumberMoleculePartFunc
                ! print*,'lgQ(1:NumberOfTemperatures, 1) = ', lgQ(1:NumberOfTemperatures, 1)
                ! print*,'lgQ(1:NumberOfTemperatures, NumberMoleculePartFunc) = ', lgQ(1:NumberOfTemperatures, NumberMoleculePartFunc)
                ! print*,'MolNamePartFunc(1) = ', MolNamePartFunc(1)
                ! print*,'MolNamePartFunc(NumberMoleculePartFunc) = ', MolNamePartFunc(NumberMoleculePartFunc)
                ! stop
            else
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetPartitionFunction!")')
                    write(ErrChannel, '(3x,"There are no entries in the sqlite3 database ", A, "!")') trim(adjustl(NameOfPartFuncTable))
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please correct the name of the database and restart the program!")')
                end Do
                ok = 1
                return
            endif


            !< bring partition function in variable lgQ in the same order like MoleculeNames
            CopylgQ(:, :) = lgQ(:, :)
            lgQ = 0.d0
            Do i = 1, TotalNumberOfMolecules
                Do j = 1, TotalNumberOfMolecules

                    ! Debug:
                    ! print*,'----->i, trim(adjustl(MoleculeNames(i))) = ', i, trim(adjustl(MoleculeNames(i)))
                    ! print*,'>>>>>>j, trim(adjustl(MolNamePartFunc(j))) = ', j, trim(adjustl(MolNamePartFunc(j)))

                    if (trim(adjustl(MoleculeNames(i))) // char(0) == trim(adjustl(MolNamePartFunc(j)))) then
                        lgQ(:, i) = CopylgQ(:, j)


                        !< calculate log10 of partition function of the current molecule and check for entries which are equal to zero
                        if (count(lgQ(:, i) == 0) > 0) then
                            Do ErrChannelIndex = 1, 1
                                ErrChannel = AllErrChannels(ErrChannelIndex)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(2x, "Error in subroutine qinterp!")')
                                write(ErrChannel, '(4x, "The are some entries in the parittion function table for molecule ", A, " which are not defined or zero!")') &
                                                                                                                           trim(adjustl(MoleculeNames(i)))
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(4x, "Number of entries which are identical to zero = ", I10)') count(lgQ(:, i) == 0)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(4x,"Set values at these positions to 1!")')
                                write(ErrChannel, '(" ")')
                                where (lgQ(:, i) == 0.d0)
                                    lgQ(:, i) = 1.d0
                                end where
                            end Do
                        endif


                        !< determine log10 values
                        Do l = 1, NumberOfTemperatures
                            lgQ(l, i) = dlog10(lgQ(l, i))
                        end Do
                        exit
                    endif
                end Do

                ! Debug:
                ! print*,'i, trim(adjustl(MoleculeNames(i))), lgQ(:, i) = ', i, trim(adjustl(MoleculeNames(i))), lgQ(:, i)
            end Do


            !< print some status information to screen
            if (printflag) then
                print '("done!")'
                print '(" ")'
                print '(11x, "Number of entries = ",I10)', NumberMoleculePartFunc
                call strlen(i, MolNamePartFunc(1))
                if (NumberMoleculePartFunc > 1) then
                    print '(11x, "First Molecule in database: ",A)', trim(adjustl(MolNamePartFunc(1)(:i)))
                    call strlen(i, MolNamePartFunc(NumberMoleculePartFunc))
                    print '(11x, "Last Molecule in database:  ",A)', trim(adjustl(MolNamePartFunc(NumberMoleculePartFunc)(:i)))
                else
                    print '(11x, "Molecule in database: ",A)', trim(adjustl(MolNamePartFunc(1)(:i)))
                endif
                print '(" ")'
                print '(" ")'
            endif


            ! Debug:
            ! Do i = 1, TotalNumberOfMolecules
            !     print*,"i, MoleculeNames(i), MolNamePartFunc(i) = ", i, trim(adjustl(MoleculeNames(i))), "         ", trim(adjustl(MolNamePartFunc(i)))
            ! end Do


            !< free memory of CopylgQ variable
            if (allocated(CopylgQ)) then
                deallocate(CopylgQ, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetPartitionFunction!")')
                        write(ErrChannel, '(3x,"Can not deallocate variable CopylgQ!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif

            return
        end subroutine GetPartitionFunction


        !>************************************************************************************************************************************************
        !> subroutine: GetTransitionParameters
        !>
        !> get informations for transitions from sqlite3 database
        !>
        !> input variables:     none
        !>
        !> output variables:    ok:             status variable
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 30.07.2014
        !>
        subroutine GetTransitionParameters(ok)

            implicit none
            integer :: ok                                                                   !< status variable
            integer :: number_entries                                                       !< number of entries for the PartitionFunction table
            integer :: countlen                                                             !< total length of query
            integer :: i, j, k                                                              !< loop variables
            integer :: RangeIndex                                                           !< index for frequency ranges
            integer :: MolecularDataCounter                                                 !< overall counter for MolecularData variable
            integer :: NumEntriesLocal                                                      !< working variable: number of data within current frequency
                                                                                            !< range and current molecule
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            integer, allocatable, dimension(:) :: UpperStateDegeneracy                      !< upper state degeneracy
            real*8 :: freqmin                                                               !< working variable lowest frequency of current freq. range
            real*8 :: freqmax                                                               !< working variable highest frequency of current freq. range
            real*8, allocatable, dimension(:) :: lFreq                                      !< frequencies of transition
            real*8, allocatable, dimension(:) :: lFreqErr                                   !< uncertainty of frequencies of transition
            real*8, allocatable, dimension(:) :: lElt                                       !< energies of lower states
            real*8, allocatable, dimension(:) :: icatMol                                    !< intensity of the transitions
            real*8, allocatable, dimension(:) :: EinsteinA                                  !< Einstein A coefficients
            character(len=40), allocatable, dimension(:) :: MolNameLocal                    !< names of molecules in current freq. range
            character(len=25) :: numberString1, numberString2                               !< working variables for number to string conversion
            character(len=8192) :: queryString1                                             !< query string (first part)
            character(len=8192) :: queryString2                                             !< query string (second part)
            character(len=4096) :: dbNameCopy                                               !< name of database
            character(len=40) :: ColumnNameForNameTransitionsCopy                           !< copy of name of column including the names of molecules
            character(len=40), dimension(1) :: CopyMoleculeNames                            !< copy of names of molecules


            !< initialize output variable
            ok = 0


            !< print what you do
            if (printflag) then
                print '(9x, "Reading parameters for radiative transitions for all molecules from sqlite database .. ", $)'
            endif


            !< allocate memory for variable NumEntriesRanges
            if (allocated(NumEntriesRanges)) then
                deallocate(NumEntriesRanges, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                        write(ErrChannel, '(3x,"Can not deallocate variable NumEntriesRanges!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    ok = 1
                    return
                endif
            endif
            allocate(NumEntriesRanges(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                    write(ErrChannel, '(3x,"Can not allocate variable NumEntriesRanges!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                ok = 1
                return
            endif
            NumEntriesRanges = 0


            !< determine total number of all transitions in all frequency ranges
            TotalNumberOfTransitions = 0
            Do RangeIndex = 1, TotalNumberOfFrequencyRanges                                 !< loop over all frequency ranges
                freqmin = StartFrequency(RangeIndex)
                freqmax = EndFrequency(RangeIndex)
                Do k = 1, TotalNumberOfMolecules                                            !< loop over all molecules


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< define query
                    write(numberString1, '(ES25.15)') freqmin
                    write(numberString2, '(ES25.15)') freqmax
                    queryString1 = "SELECT count(*) FROM " // trim(adjustl(NameOfRadTransTable)) // char(0)
                    queryString2 = "AND (" // trim(adjustl(ColumnNameForFreqTransitions)) // " >= " // trim(adjustl(numberString1)) &
                                   // " AND " // trim(adjustl(ColumnNameForFreqTransitions)) // " <= " // trim(adjustl(numberString2)) &
                                   // ") ORDER by " // trim(adjustl(ColumnNameForFreqTransitions)) // char(0)

                    ! Debug:
                    ! print*,' '
                    ! print*,'queryString1 = >', trim(adjustl(queryString1)), '<'
                    ! print*,'queryString2 = >', trim(adjustl(queryString2)), '<'


                    !< get number of entries for the transition table
                    ok = 0                                                                  !< reset status variable
                    number_entries = 0                                                      !< reset result variable
                    j = 1
                    countlen = (len_trim(queryString1) - 1)
                    CopyMoleculeNames = ""
                    CopyMoleculeNames(1) = char(34) // trim(adjustl(MoleculeNames(k))) // char(34) // char(0)
                    countlen = countlen + len_trim(CopyMoleculeNames(1)) + 12
                    countlen = countlen + (len_trim(queryString2) - 1) + 17
                    dbNameCopy = trim(adjustl(dbName)) // char(0)
                    ColumnNameForNameTransitionsCopy = trim(adjustl(ColumnNameForNameTransitions)) // char(0)
                    call numentr_sqlite3(ok, number_entries, %val(countlen), trim(adjustl(queryString1)), trim(adjustl(queryString2)), %val(j), &
                                         trim(adjustl(dbNameCopy)), trim(adjustl(ColumnNameForNameTransitionsCopy)), CopyMoleculeNames)
                    NumEntriesRanges(RangeIndex, k) = number_entries
                    TotalNumberOfTransitions = TotalNumberOfTransitions + number_entries

                    ! Debug:
                    ! print*,"Frequency range index = ", RangeIndex
                    ! print*,"Molecule index = ", k
                    ! print*,"Number of database entries within the current freq. range (number_entries) = ", NumEntriesRanges(RangeIndex, k)
                    ! print*,'TotalNumberOfTransitions = ', TotalNumberOfTransitions
                    ! print*,'ok = ', ok
                end Do
            end Do


            !< allocate variables for parameter
            if (TotalNumberOfTransitions > 0 .and. ok == 0) then


                !< allocate memory for variable NumEntriesRanges
                if (allocated(MolecularData)) then
                    deallocate(MolecularData, MolecularDataIndices, MolNameRadTrans, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        Do ErrChannelIndex = 1, 1
                            ErrChannel = AllErrChannels(ErrChannelIndex)
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                            write(ErrChannel, '(3x,"Can not deallocate variables MolecularData, MolecularDataIndices, and MolNameRadTrans!")')
                            write(ErrChannel, '(" ")')
                            write(ErrChannel, '(3x,"Please restart the program!")')
                        end Do
                        ok = 1
                        return
                    endif
                endif
                allocate(MolecularData(TotalNumberOfTransitions, 4), MolecularDataIndices(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2), &
                         MolNameRadTrans(TotalNumberOfTransitions), stat = allocstatus)
                if (allocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                        write(ErrChannel, '(3x,"Can not allocate variables MolecularData, MolecularDataIndices, and MolNameRadTrans!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    ok = 1
                    return
                endif
                MolecularData = 0.d0
                MolecularDataIndices = 0
                MolNameRadTrans = ""


                !< read in molecular data from database for each molecule
                MolecularDataCounter = 0
                Do RangeIndex = 1, TotalNumberOfFrequencyRanges                             !< loop over all frequency ranges
                    freqmin = StartFrequency(RangeIndex)                                    !< get first frequency of current frequency range
                    freqmax = EndFrequency(RangeIndex)                                      !< get last frequency of current frequency range
                    Do k = 1, TotalNumberOfMolecules                                        !< loop over all molecules


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< define query
                        write(numberString1, '(ES25.15)') freqmin
                        write(numberString2, '(ES25.15)') freqmax
                        queryString2 = "AND (" // trim(adjustl(ColumnNameForFreqTransitions)) // " >= " // trim(adjustl(numberString1)) &
                                       // " AND " // trim(adjustl(ColumnNameForFreqTransitions)) // " <= " // trim(adjustl(numberString2)) &
                                       // ") ORDER by " // trim(adjustl(ColumnNameForFreqTransitions)) // char(0)

                        ! Debug:
                        ! print*,' '
                        ! print*,"RangeIndex, k, NumEntriesRanges(RangeIndex, k) = ", RangeIndex, k, NumEntriesRanges(RangeIndex, k)


                        !< allocate memory for worker
                        if (allocated(lFreq)) then
                            deallocate(MolNameLocal, lFreq, lFreqErr, icatMol, EinsteinA, lElt, UpperStateDegeneracy, stat = deallocstatus)
                            if (deallocstatus /= 0) then
                                Do ErrChannelIndex = 1, 1
                                    ErrChannel = AllErrChannels(ErrChannelIndex)
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                                    write(ErrChannel, '(3x,"Can not deallocate variables lFreq and ...!")')
                                    write(ErrChannel, '(" ")')
                                    write(ErrChannel, '(3x,"Please restart the program!")')
                                end Do
                                ok = 1
                                return
                            endif
                        endif
                        NumEntriesLocal = NumEntriesRanges(RangeIndex, k)
                        allocate(MolNameLocal(NumEntriesLocal), lFreq(NumEntriesLocal), lFreqErr(NumEntriesLocal), icatMol(NumEntriesLocal), &
                                 EinsteinA(NumEntriesLocal), lElt(NumEntriesLocal), UpperStateDegeneracy(NumEntriesLocal), stat = allocstatus)
                        if (allocstatus /= 0) then
                            Do ErrChannelIndex = 1, 1
                                ErrChannel = AllErrChannels(ErrChannelIndex)
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                                write(ErrChannel, '(3x,"Can not allocate variables lFreq and ...!")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Please restart the program!")')
                            end Do
                            ok = 1
                            return
                        endif
                        MolNameLocal = ""
                        lFreq = 0.d0
                        lFreqErr = 0.d0
                        icatMol = 0.d0
                        EinsteinA = 0.d0
                        lElt = 0.d0
                        UpperStateDegeneracy = 0
                        Do j = 1, NumEntriesLocal
                            MolNameLocal(j) = char(0)
                        end Do


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< get molecular data
                        if (NumEntriesRanges(RangeIndex, k) > 0) then                       !< check, if molecular data are contained in the database
                                                                                            !< for the current frequency range and molecule

                            !< construct query string
                            queryString1 = "SELECT " // trim(adjustl(ColumnNameForNameTransitions))                                 !< pos. 1: molecule
                                                                                                                                    !< name
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForFreqTransitions))       !< pos. 2: frequency
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForFreqErrTransitions))    !< pos. 3: error of
                                                                                                                                    !< frequency
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForIntTransitions))        !< pos. 4: intensity
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForEinsteinATransitions))  !< pos. 5: Einstein A
                                                                                                                                    !<         coefficient
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForELowTransitions))       !< pos. 6: E_lower
                            queryString1 = trim(adjustl(queryString1)) // ", " // trim(adjustl(ColumnNameForgUpTransitions))        !< pos. 7: upper state
                                                                                                                                    !<         degeneracy
                            queryString1 = trim(adjustl(queryString1)) // " FROM " // trim(adjustl(NameOfRadTransTable)) // char(0)

                            ! Debug:
                            ! print*," "
                            ! print*,"queryString1 = -->", trim(adjustl(queryString1)), "<--"


                            !< get information from sqlite3 database
                            ok = 0                                                          !< reset status variable
                            i = TotalNumberOfTransitions
                            j = 1
                            countlen = (len_trim(queryString1) - 1)
                            CopyMoleculeNames = ""
                            CopyMoleculeNames(1) = char(34) // trim(adjustl(MoleculeNames(k))) // char(34) // char(0)
                            countlen = countlen + len_trim(CopyMoleculeNames(1)) + 12
                            countlen = countlen + (len_trim(queryString2) - 1) + 17
                            dbNameCopy = trim(adjustl(dbName)) // char(0)
                            ColumnNameForNameTransitionsCopy = trim(adjustl(ColumnNameForNameTransitions)) // char(0)


                            !< read informations from sqlite3 database
                            call radtrans(ok, %val(i), %val(countlen), trim(adjustl(queryString1)), trim(adjustl(queryString2)), %val(j), &
                                          trim(adjustl(dbNameCopy)), trim(adjustl(ColumnNameForNameTransitionsCopy)), CopyMoleculeNames, &
                                          MolNameLocal, lFreq, lFreqErr, icatMol, EinsteinA, lElt, UpperStateDegeneracy)
                            if (ok /= 0) then
                                ErrChannel = 6
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                                write(ErrChannel, '(3x,"An error occurs while reading entries from sqlite3 database ", A, " within the given frequency range!")') &
                                                                                                                     trim(adjustl(NameOfRadTransTable))
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Molecule:")')
                                write(ErrChannel, '(3x,A)') trim(adjustl(MoleculeNames(k)))
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Frequency range:")')
                                write(ErrChannel, '(3x,"Min. frequency = ", ES25.15)') freqmin
                                write(ErrChannel, '(3x,"Max. frequency = ", ES25.15)') freqmax
                                write(ErrChannel, '(" ")')
                                write(ErrChannel, '(3x,"Please check your inputs and restart the program!")')
                                ok = 1
                                return

                                ! Debug:
                                ! print*,'MolNameLocal(1) = ', MolNameLocal(1)
                                ! print*,'MolNameLocal(TotalNumberOfTransitions) = ', MolNameLocal(TotalNumberOfTransitions)
                                ! print*,'lFreq(1) = ', lFreq(1)
                                ! print*,'lFreq(TotalNumberOfTransitions) = ', lFreq(TotalNumberOfTransitions)
                                ! print*,'lFreqErr(1) = ', lFreqErr(1)
                                ! print*,'lFreqErr(TotalNumberOfTransitions) = ', lFreqErr(TotalNumberOfTransitions)
                                ! print*,'icatMol(1) = ', icatMol(1)
                                ! print*,'icatMol(TotalNumberOfTransitions) = ', icatMol(TotalNumberOfTransitions)
                                ! print*,'lElt(1) = ', lElt(1)
                                ! print*,'lElt(TotalNumberOfTransitions) = ', lElt(TotalNumberOfTransitions)
                                ! stop
                            endif
                        endif


                        !< copy molecular data for current molecule and frequency range to MolecularData variable
                        MolecularDataIndices(RangeIndex, k, 1) = MolecularDataCounter + 1   !< save first index for current freq. range and molecule
                                                                                            !< within MolecularData variable
                        Do j = 1, NumEntriesLocal                                           !< loop over all transitions within the current frequency
                                                                                            !< range and molecule
                            MolecularDataCounter = MolecularDataCounter + 1
                            MolecularData(MolecularDataCounter, 1) = lFreq(j)
                            MolecularData(MolecularDataCounter, 2) = EinsteinA(j)
                            MolecularData(MolecularDataCounter, 3) = lElt(j) / 1.438769     !< only for debugging
                            MolecularData(MolecularDataCounter, 4) = UpperStateDegeneracy(j)

                            ! Debug:
                            ! print*,"trim(adjustl(MoleculeNames(k))), j, MolecularData(j, :) = ", trim(adjustl(MoleculeNames(k))), j, MolecularData(j, :)


                            !< correct entry for lower energy
                            MolecularData(MolecularDataCounter, 3) = lElt(j)
                            MolNameRadTrans(MolecularDataCounter) = MolNameLocal(j)
                        end Do
                        MolecularDataIndices(RangeIndex, k, 2) = MolecularDataCounter       !< save last index for current freq. range and molecule
                                                                                            !< within MolecularData variable
                    end Do
                end Do
            endif


            !< print what you do
            if (printflag) then
                print '("done!")'
            endif


            !< print some informations to screen about transitions
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(11x, "Number of transitions for each frequency range and molecule:")'
                Do RangeIndex = 1, TotalNumberOfFrequencyRanges                             !< loop over all frequency ranges
                    freqmin = StartFrequency(RangeIndex)                                    !< get first frequency of current frequency range
                    freqmax = EndFrequency(RangeIndex)                                      !< get last frequency of current frequency range
                    write(numberString1, '(ES25.15)') freqmin
                    write(numberString2, '(ES25.15)') freqmax
                    print '(" ")'
                    print '(11x, "Frequency range: ", A, " MHz - ", A, " MHz:")', trim(adjustl(numberString1)), trim(adjustl(numberString2))
                    Do k = 1, TotalNumberOfMolecules                                        !< loop over all molecules
                        print '(13x, "Number of transitions for molecule ", A, A, A, ": ", T92, I10)', char(34), trim(adjustl(MoleculeNames(k))), &
                                                                                                       char(34), NumEntriesRanges(RangeIndex, k)
                    end Do
                end Do
                print '(" ")'
                print '(" ")'
            endif


            !< free memory of local variables
            if (allocated(MolNameLocal)) deallocate(MolNameLocal, stat = deallocstatus)
            if (allocated(lFreq) .and. deallocstatus == 0) deallocate(lFreq, stat = deallocstatus)
            if (allocated(lFreqErr) .and. deallocstatus == 0) deallocate(lFreqErr, stat = deallocstatus)
            if (allocated(icatMol) .and. deallocstatus == 0) deallocate(icatMol, stat = deallocstatus)
            if (allocated(EinsteinA) .and. deallocstatus == 0) deallocate(EinsteinA, stat = deallocstatus)
            if (allocated(lElt) .and. deallocstatus == 0) deallocate(lElt, stat = deallocstatus)
            if (allocated(UpperStateDegeneracy) .and. deallocstatus == 0) deallocate(UpperStateDegeneracy, stat = deallocstatus)
            if (deallocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine GetTransitionParameters!")')
                    write(ErrChannel, '(3x,"Can not deallocate variables MolNameLocal etc.!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                ok = 1
            endif
            return
        end subroutine GetTransitionParameters


        !>************************************************************************************************************************************************
        !> subroutine: ModelInit
        !>
        !> initialize myXCLASS program
        !>
        !>
        !> input variables:     none
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 27.07.2014
        !>
        subroutine ModelInit

            implicit none
            integer :: ok                                                                   !< status variablee: name of parameter
            integer :: i, j, k, m, n                                                        !< working variables
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation


            !< initialize variables
            pi = 4.d0 * datan(1.d0)                                                         !< determine pi
            ckms = 299792.458d0                                                             !< speed of light in km?


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define variables for sqlite3 database
            NameOfRadTransTable = "transitions"
            NameOfPartFuncTable = "partitionfunctions"
            NumberOfTemperatures = 110


            !< allocate variables for partition function
            if (allocated(ColumnNamesPartFunc)) then
                deallocate(ColumnNamesPartFunc, TempPartFunc, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables ColumnNamesPartFunc and TempPartFunc!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            allocate(ColumnNamesPartFunc(NumberOfTemperatures), TempPartFunc(NumberOfTemperatures), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variables ColumnNamesPartFunc and TempPartFunc!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            ColumnNamesPartFunc = ""
            TempPartFunc = 0.d0


            !< define column names of table partitionfunctions
            !< Due to a bug in the current gfortran compiler a statement like ColumnNamesPartFunc = (/.../) is not possible
            ColumnNameForNamePartFunc = 'PF_Name'
            ColumnNamesPartFunc(1) = 'PF_1_072'
            ColumnNamesPartFunc(2) = 'PF_1_148'
            ColumnNamesPartFunc(3) = 'PF_1_230'
            ColumnNamesPartFunc(4) = 'PF_1_318'
            ColumnNamesPartFunc(5) = 'PF_1_413'
            ColumnNamesPartFunc(6) = 'PF_1_514'
            ColumnNamesPartFunc(7) = 'PF_1_622'
            ColumnNamesPartFunc(8) = 'PF_1_738'
            ColumnNamesPartFunc(9) = 'PF_1_862'
            ColumnNamesPartFunc(10) = 'PF_1_995'
            ColumnNamesPartFunc(11) = 'PF_2_138'
            ColumnNamesPartFunc(12) = 'PF_2_291'
            ColumnNamesPartFunc(13) = 'PF_2_455'
            ColumnNamesPartFunc(14) = 'PF_2_630'
            ColumnNamesPartFunc(15) = 'PF_2_725'
            ColumnNamesPartFunc(16) = 'PF_2_818'
            ColumnNamesPartFunc(17) = 'PF_3_020'
            ColumnNamesPartFunc(18) = 'PF_3_236'
            ColumnNamesPartFunc(19) = 'PF_3_467'
            ColumnNamesPartFunc(20) = 'PF_3_715'
            ColumnNamesPartFunc(21) = 'PF_3_981'
            ColumnNamesPartFunc(22) = 'PF_4_266'
            ColumnNamesPartFunc(23) = 'PF_4_571'
            ColumnNamesPartFunc(24) = 'PF_4_898'
            ColumnNamesPartFunc(25) = 'PF_5_000'
            ColumnNamesPartFunc(26) = 'PF_5_248'
            ColumnNamesPartFunc(27) = 'PF_5_623'
            ColumnNamesPartFunc(28) = 'PF_6_026'
            ColumnNamesPartFunc(29) = 'PF_6_457'
            ColumnNamesPartFunc(30) = 'PF_6_918'
            ColumnNamesPartFunc(31) = 'PF_7_413'
            ColumnNamesPartFunc(32) = 'PF_7_943'
            ColumnNamesPartFunc(33) = 'PF_8_511'
            ColumnNamesPartFunc(34) = 'PF_9_120'
            ColumnNamesPartFunc(35) = 'PF_9_375'
            ColumnNamesPartFunc(36) = 'PF_9_772'
            ColumnNamesPartFunc(37) = 'PF_10_471'
            ColumnNamesPartFunc(38) = 'PF_11_220'
            ColumnNamesPartFunc(39) = 'PF_12_023'
            ColumnNamesPartFunc(40) = 'PF_12_882'
            ColumnNamesPartFunc(41) = 'PF_13_804'
            ColumnNamesPartFunc(42) = 'PF_14_791'
            ColumnNamesPartFunc(43) = 'PF_15_849'
            ColumnNamesPartFunc(44) = 'PF_16_982'
            ColumnNamesPartFunc(45) = 'PF_18_197'
            ColumnNamesPartFunc(46) = 'PF_18_750'
            ColumnNamesPartFunc(47) = 'PF_19_498'
            ColumnNamesPartFunc(48) = 'PF_20_893'
            ColumnNamesPartFunc(49) = 'PF_22_387'
            ColumnNamesPartFunc(50) = 'PF_23_988'
            ColumnNamesPartFunc(51) = 'PF_25_704'
            ColumnNamesPartFunc(52) = 'PF_27_542'
            ColumnNamesPartFunc(53) = 'PF_29_512'
            ColumnNamesPartFunc(54) = 'PF_31_623'
            ColumnNamesPartFunc(55) = 'PF_33_884'
            ColumnNamesPartFunc(56) = 'PF_36_308'
            ColumnNamesPartFunc(57) = 'PF_37_500'
            ColumnNamesPartFunc(58) = 'PF_38_905'
            ColumnNamesPartFunc(59) = 'PF_41_687'
            ColumnNamesPartFunc(60) = 'PF_44_668'
            ColumnNamesPartFunc(61) = 'PF_47_863'
            ColumnNamesPartFunc(62) = 'PF_51_286'
            ColumnNamesPartFunc(63) = 'PF_54_954'
            ColumnNamesPartFunc(64) = 'PF_58_884'
            ColumnNamesPartFunc(65) = 'PF_63_096'
            ColumnNamesPartFunc(66) = 'PF_67_608'
            ColumnNamesPartFunc(67) = 'PF_72_444'
            ColumnNamesPartFunc(68) = 'PF_75_000'
            ColumnNamesPartFunc(69) = 'PF_77_625'
            ColumnNamesPartFunc(70) = 'PF_83_176'
            ColumnNamesPartFunc(71) = 'PF_89_125'
            ColumnNamesPartFunc(72) = 'PF_95_499'
            ColumnNamesPartFunc(73) = 'PF_102_329'
            ColumnNamesPartFunc(74) = 'PF_109_648'
            ColumnNamesPartFunc(75) = 'PF_117_490'
            ColumnNamesPartFunc(76) = 'PF_125_893'
            ColumnNamesPartFunc(77) = 'PF_134_896'
            ColumnNamesPartFunc(78) = 'PF_144_544'
            ColumnNamesPartFunc(79) = 'PF_150_000'
            ColumnNamesPartFunc(80) = 'PF_154_882'
            ColumnNamesPartFunc(81) = 'PF_165_959'
            ColumnNamesPartFunc(82) = 'PF_177_828'
            ColumnNamesPartFunc(83) = 'PF_190_546'
            ColumnNamesPartFunc(84) = 'PF_204_174'
            ColumnNamesPartFunc(85) = 'PF_218_776'
            ColumnNamesPartFunc(86) = 'PF_225_000'
            ColumnNamesPartFunc(87) = 'PF_234_423'
            ColumnNamesPartFunc(88) = 'PF_251_189'
            ColumnNamesPartFunc(89) = 'PF_269_153'
            ColumnNamesPartFunc(90) = 'PF_288_403'
            ColumnNamesPartFunc(91) = 'PF_300_000'
            ColumnNamesPartFunc(92) = 'PF_309_030'
            ColumnNamesPartFunc(93) = 'PF_331_131'
            ColumnNamesPartFunc(94) = 'PF_354_813'
            ColumnNamesPartFunc(95) = 'PF_380_189'
            ColumnNamesPartFunc(96) = 'PF_407_380'
            ColumnNamesPartFunc(97) = 'PF_436_516'
            ColumnNamesPartFunc(98) = 'PF_467_735'
            ColumnNamesPartFunc(99) = 'PF_500_000'
            ColumnNamesPartFunc(100) = 'PF_501_187'
            ColumnNamesPartFunc(101) = 'PF_537_032'
            ColumnNamesPartFunc(102) = 'PF_575_440'
            ColumnNamesPartFunc(103) = 'PF_616_595'
            ColumnNamesPartFunc(104) = 'PF_660_693'
            ColumnNamesPartFunc(105) = 'PF_707_946'
            ColumnNamesPartFunc(106) = 'PF_758_578'
            ColumnNamesPartFunc(107) = 'PF_812_831'
            ColumnNamesPartFunc(108) = 'PF_870_964'
            ColumnNamesPartFunc(109) = 'PF_933_254'
            ColumnNamesPartFunc(110) = 'PF_1000_000'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define log10 of temperatures for partition funciton for
            !< T = 1.072, 1.148, 1.23, 1.318, 1.413, 1.514, 1.622, 1.738, 1.862, 1.995, 2.138, 2.291, 2.455, 2.63, 2.725, 2.818, 3.02, 3.236, 3.467,
            !<     3.715, 3.981, 4.266, 4.571, 4.898, 5, 5.248, 5.623, 6.026, 6.457, 6.918, 7.413, 7.943, 8.511, 9.12, 9.375, 9.772, 10.471, 11.22,
            !<     12.023, 12.882, 13.804, 14.791, 15.849, 16.982, 18.197, 18.75, 19.498, 20.893, 22.387, 23.988, 25.704, 27.542, 29.512, 31.623,
            !<     33.884, 36.308, 37.5, 38.905, 41.687, 44.668, 47.863, 51.286, 54.954, 58.884, 63.096, 67.608, 72.444, 75, 77.625, 83.176, 89.125,
            !<     95.499, 102.329, 109.648, 117.49, 125.893, 134.896, 144.544, 150, 154.882, 165.959, 177.828, 190.546, 204.174, 218.776, 225,
            !<     234.423, 251.189, 269.153, 288.403, 300, 309.03, 331.131, 354.813, 380.189, 407.38, 436.516, 467.735, 500, 501.187, 537.032,
            !<     575.44, 616.595, 660.693, 707.946, 758.578, 812.831, 870.964, 933.254, 1000
            TempPartFunc = (/0.030194785356751d0, 0.059941888061955d0, 0.089905111439398d0, 0.119915410257991d0, 0.150142161848559d0, &
                             0.180125875164054d0, 0.210050849875137d0, 0.240049772112648d0, 0.269979676645324d0, 0.299942900022767d0, &
                             0.330007700872759d0, 0.360025089189397d0, 0.390051496458987d0, 0.419955748489758d0, 0.435366506612661d0, &
                             0.449940988773338d0, 0.480006942957151d0, 0.510008512940235d0, 0.539953841656397d0, 0.569958818096594d0, &
                             0.599992177584098d0, 0.630020851113410d0, 0.660011221289331d0, 0.690018780788695d0, 0.698970004336019d0, &
                             0.719993826367604d0, 0.749968083509403d0, 0.780029127337338d0, 0.810030786405839d0, 0.839980557678343d0, &
                             0.869994000121742d0, 0.899984562549391d0, 0.929980590515515d0, 0.959994838328416d0, 0.971971276399756d0, &
                             0.989983458301399d0, 1.019988159591290d0, 1.049992856920140d0, 1.080012847107930d0, 1.109983294819890d0, &
                             1.140004950619450d0, 1.169997537066570d0, 1.200001865406600d0, 1.229988836544810d0, 1.259999795081890d0, &
                             1.273001272063740d0, 1.289990066054320d0, 1.320000804264730d0, 1.349995899262860d0, 1.379994040165740d0, &
                             1.410000712543460d0, 1.439995473953810d0, 1.469998642218750d0, 1.500003068051690d0, 1.529994673069140d0, &
                             1.560002326773370d0, 1.574031267727720d0, 1.590005419651330d0, 1.620000642300070d0, 1.649996507466040d0, &
                             1.679999916229220d0, 1.709998828025190d0, 1.739999309401970d0, 1.769997304028540d0, 1.800001827817970d0, &
                             1.829998088697380d0, 1.860002421901100d0, 1.875061263391700d0, 1.890001613184640d0, 1.919998030964020d0, &
                             1.949999542859920d0, 1.979998823973740d0, 2.009998730048060d0, 2.040000714473320d0, 2.070000903802340d0, &
                             2.100001582801350d0, 2.129999071957430d0, 2.160000068881300d0, 2.176091259055680d0, 2.190000948069660d0, &
                             2.220000809286660d0, 2.250000144081290d0, 2.279999836361080d0, 2.310000437185500d0, 2.339999677628230d0, &
                             2.352182518111360d0, 2.370000219475120d0, 2.400000616976380d0, 2.429999224858320d0, 2.459999773650270d0, &
                             2.477121254719660d0, 2.490000641890950d0, 2.519999840669680d0, 2.549999523574670d0, 2.579999547278480d0, &
                             2.609999703842230d0, 2.640000166906100d0, 2.669999868814090d0, 2.698970004336020d0, 2.699999797554580d0, &
                             2.730000164674160d0, 2.760000047292730d0, 2.789999998688880d0, 2.819999705510490d0, 2.850000132271100d0, &
                             2.880000243300670d0, 2.910000258512930d0, 2.940000204462930d0, 2.969999860022620d0, 3.000000000000000d0/)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< determine order of partition function in database: ascending or descending order
            TempLow = min(TempPartFunc(1), TempPartFunc(NumberOfTemperatures))
            TempHigh = max(TempPartFunc(1), TempPartFunc(NumberOfTemperatures))
            if (TempPartFunc(1) < TempPartFunc(NumberOfTemperatures)) then
                Firsti = NumberOfTemperatures
                Lasti = 1
                stepi = (-1)
            else
                Firsti = 1
                Lasti = NumberOfTemperatures
                stepi = 1
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define column names of table transitions
            ColumnNameForNameTransitions = "T_Name"
            ColumnNameForFreqTransitions = "T_Frequency"
            ColumnNameForIntTransitions = "T_Intensity"
            ColumnNameForEinsteinATransitions = "T_EinsteinA"
            ColumnNameForFreqErrTransitions = "T_Uncertainty"
            ColumnNameForELowTransitions = "T_EnergyLower"
            ColumnNameForgUpTransitions = "T_UpperStateDegeneracy"
            Column300KPartFunc = 91                                                         !< (only used, if use_intensity_flag == .true): define the
                                                                                            !< column number, containing the partition function for 300 K

            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get myXCLASS parameter
            call GetmyXCLASSParameter


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get partition functions for selected molecules and describe partition functions with polynominal
            call GetPartitionFunction(ok)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< get informations for each transition in the given frequency ranges
            call GetTransitionParameters(ok)

            ! Debug:
            ! stop 'Subroutine ModelInit finished!'


            !<============================================================================================================================================
            !< define variables to GPU variables


            !< print what you do
            if (printflag) then
                print '(11x, "Copy data to GPU RAM .. ", $)'
            endif


            !< allocate variables for partition function
            if (allocated(NumberComponentsPerMolecule_d)) then
                deallocate(NumberComponentsPerMolecule_d, IsoNfitConversionTable_d, SpectrumIndexForFreqRange_d, DataPointIndexFreqRange_d, &
                           MolecularDataIndices_d, ConversionTableMAGIXmyXCLASSParam_d, CompMoleculeIndex_d, TempPartFunc_d, &
                           BackgroundTemperatureRange_d, TemperatureSlopeRange_d, HydrogenColumnDensityRange_d, DustBetaRange_d, KappaRange_d, &
                           TelescopeSize_d, StartFrequency_d, ObservationalDataList_d, MolecularData_d, MoleculeNames_d, IsoRatioConversionTable_d, &
                           lgQ_d, myXCLASSParameter_d, CopyCompMoleculeIndex, PartFunc, CopyIsoRatioConversionTable, &
!                           CopymyXCLASSParameter, &
                           stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                        write(ErrChannel, '(3x,"Can not deallocate variables NumberComponentsPerMolecule_d, etc.!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    stop 'Program aborted!'
                endif
            endif
            if (printflag) print '(".", $)'


            allocate(NumberComponentsPerMolecule_d(0:TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable NumberComponentsPerMolecule_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(IsoNfitConversionTable_d(TotalNumberOfMolecules, TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable IsoNfitConversionTable_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(SpectrumIndexForFreqRange_d(TotalNumberOfFrequencyRanges), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable SpectrumIndexForFreqRange_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(DataPointIndexFreqRange_d(TotalNumberOfFrequencyRanges, 2), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable DataPointIndexFreqRange_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(MolecularDataIndices_d(TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable MolecularDataIndices_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(ConversionTableMAGIXmyXCLASSParam_d(NumberFreeParameter, 2), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable ConversionTableMAGIXmyXCLASSParam_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(CompMoleculeIndex_d(TotalNumberComponents), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable CompMoleculeIndex_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(TempPartFunc_d(NumberOfTemperatures), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable TempPartFunc_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(BackgroundTemperatureRange_d(TotalNumberOfFrequencyRanges), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable BackgroundTemperatureRange_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(TemperatureSlopeRange_d(TotalNumberOfFrequencyRanges), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable TemperatureSlopeRange_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(HydrogenColumnDensityRange_d(TotalNumberOfFrequencyRanges), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable HydrogenColumnDensityRange_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(DustBetaRange_d(TotalNumberOfFrequencyRanges), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable DustBetaRange_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(KappaRange_d(TotalNumberOfFrequencyRanges), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable KappaRange_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(TelescopeSize_d(TotalNumberOfFrequencyRanges), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable TelescopeSize_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(StartFrequency_d(TotalNumberOfFrequencyRanges), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable StartFrequency_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(ObservationalDataList_d(TotalNumberDataPoints, 3), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable ObservationalDataList_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(MolecularData_d(TotalNumberOfTransitions, 4), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable MolecularData_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(MoleculeNames_d(TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable MoleculeNames_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(IsoRatioConversionTable_d(TotalNumberOfMolecules, TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable IsoRatioConversionTable_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(lgQ_d(NumberOfTemperatures, NumberMoleculePartFunc), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable lgQ_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            allocate(myXCLASSParameter_d(11, TotalNumberComponents), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable myXCLASSParameter_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            if (printflag) print '(".", $)'


            allocate(CopyCompMoleculeIndex(2000, TotalNumberComponents), PartFunc(2000, TotalNumberComponents), &
!                     CopymyXCLASSParameter(2000, 11, TotalNumberComponents), &
                     CopyIsoRatioConversionTable(2000, TotalNumberOfMolecules, TotalNumberOfMolecules), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelInit!")')
                    write(ErrChannel, '(3x,"Can not allocate variable myXCLASSParameter_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"allocstatus = ", I5)') allocstatus
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                stop 'Program aborted!'
            endif
            NumberFreeParameter_d = NumberFreeParameter
            NumberComponentsPerMolecule_d = NumberComponentsPerMolecule
            IsoNfitConversionTable_d = IsoNfitConversionTable
            SpectrumIndexForFreqRange_d = SpectrumIndexForFreqRange
            DataPointIndexFreqRange_d = DataPointIndexFreqRange
            MolecularDataIndices_d = MolecularDataIndices
            ConversionTableMAGIXmyXCLASSParam_d = ConversionTableMAGIXmyXCLASSParam
            CompMoleculeIndex_d = CompMoleculeIndex
            TempPartFunc_d = TempPartFunc
            BackgroundTemperatureRange_d = BackgroundTemperatureRange
            TemperatureSlopeRange_d = TemperatureSlopeRange
            HydrogenColumnDensityRange_d = HydrogenColumnDensityRange
            DustBetaRange_d = DustBetaRange
            KappaRange_d = KappaRange
            TelescopeSize_d = TelescopeSize
            StartFrequency_d = StartFrequency
            ObservationalDataList_d = ObservationalDataList
            MolecularData_d = MolecularData
            MoleculeNames_d = MoleculeNames(1:TotalNumberOfMolecules)
            IsoRatioConversionTable_d = IsoRatioConversionTable
            lgQ_d = lgQ
            myXCLASSParameter_d = myXCLASSParameter
            Firsti_d = Firsti
            Lasti_d = Lasti
            stepi_d = stepi
            TotalNumberComponents_d = TotalNumberComponents
            TotalNumberOfFrequencyRanges_d = TotalNumberOfFrequencyRanges
            TotalNumberOfMolecules_d = TotalNumberOfMolecules
            NumberOfTemperatures_d = NumberOfTemperatures
            debug_freq_d = debug_freq
            ckms_d = ckms
            pi_d = pi
            TempLow_d = TempLow
            TempHigh_d = TempHigh
            nHFlag_d = nHFlag
            tbFlag_d = tbFlag
            IsoFlag_d = IsoFlag
            if (printflag) print '(".", $)'


            CopyIsoRatioConversionTable = 0.d0
            CopyCompMoleculeIndex = 0
!            CopymyXCLASSParameter = 0.d0
            k = min(1000, TotalNumberOfMolecules)
            i = min(1000, TotalNumberComponents)
            Do n = 1, 2000


                !< copy iso ratio conversion table
                Do m = 1, k
                    Do j = 1, k
                        CopyIsoRatioConversionTable(n, m, j) = IsoRatioConversionTable(m, j)
                    end Do
                end Do


                !< copy CompMoleculeIndex
                CopyCompMoleculeIndex(n, 1:i) = CompMoleculeIndex(1:i)


                !< copy myXCLASS parameter array
                Do j = 1, 11
!                    CopymyXCLASSParameter(n, j, 1:i) = myXCLASSParameter(j, 1:i)
                end Do
            end Do
            PartFunc = 0.d0


            !< we've done
            if (printflag) then
                print '("done!")'
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< set storage flag for model function values
            DontStoreModelFuncValuesFlag = .true.                                           !< model function values are stored for each function call
            ParallezitionMethod = 1                                                         !< set parallelization method to OpenMP (=1)
            myrankOfMPI = 0                                                                 !< copy thread number to global variable
            NumberOfUsedThreads = ParallelizationFlag                                       !< in Module 'FunctionCalling' we need only all thread
            ParallelizationMethod = "GPU"


            return
        end subroutine ModelInit


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunctionLM
        !>
        !> calculates the chi2 values for the Levenberg-Marquard algorithm
        !>
        !>
        !> input variables:     ma:                 total number of parameters
        !>                      a:                  array containing the parameter set
        !>                      ia:                 flags for including/excluding parameter in the fit
        !>                      NumberFreeParameterCopy:             number of fitted parameters
        !>                      fitparam:           parameter which have to be optimized
        !>                      colx:               number of columns in experimental x data
        !>                      NumFile:            number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>                      FitFunctionOut:     values of the fit function at all calculated points
        !>                      Chi2Values:         values of the fit function at all calculated points
        !>                      alpha:              matrix alpha (only used for Levenberg-Marquardt algorithm)
        !>                      beta2:              beta2 array (only used for Levenberg-Marquardt algorithm)
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 31.07.2014
        !>
        subroutine ModelCalcChiFunctionLM(ma, a, ia, NumberFreeParameterCopy, fitparam, colx, NumFile, MaxL, MaxCol, FitFunctionOut, Chi2Values, &
                                          alpha, beta2)

            use omp_lib

            implicit none
            integer :: i, j, k, l, m, n, fitnum, NumberFile                                 !< loop variables
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: ma                                                                   !< total number of parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: counter                                                              !< counter for ModelFunction
            integer :: NumberFreeParameterCopy                                              !< number of fitted parameters
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            integer :: NumParameterVectors                                                  !< number of parameter vectors in ParameterVectorSet
            real*8 :: dy, sig2i, wt, ymod                                                   !< working variables
            real*8 :: value                                                                 !< calculated value of the fit function
            real*8 :: variation, d1, d2                                                     !< working variables
            real*8, dimension(ma) :: steph                                                  !< working variable
            real*8, dimension(ma) :: modelparam                                             !< array containing the parameter set
            real*8, dimension(ma) :: modelparamcopy                                         !< copy of modelparam
            real*8, dimension(ma) :: a                                                      !< array containing the parameter set
            real*8, dimension(ma, ma) :: alpha                                              !< matrix alpha
            real*8, dimension(NumberFreeParameterCopy) :: beta2                             !< beta2 array
            real*8, dimension(NumberFreeParameterCopy) :: fitparam                          !< parameter which have to be optimized
            real*8, dimension(NumFile, MaxL, MaxCol) :: FitFunctionOut                      !< values of the fit function at all calculated points
            real*8, dimension(NumFile, MaxL, MaxCol) :: Chi2Values                          !< values of the fit function at all calculated points
            real*8, dimension(ma) :: dyda                                                   !< gradient of the fit function
            real*8, dimension(NumberFreeParameterCopy + 1) :: chi2ValuesVector              !< vector containing chi2 values for each parameter vector
                                                                                            !< in ParameterVectorSet
            real*8, dimension(NumberFreeParameterCopy + 1, NumberFreeParameterCopy) :: ParameterVectorSet
                                                                                            !< set of parameter vectors calculated by model module
            logical :: IntegerTrue                                                          !< flag for identification of integer numbers
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit
            character(len=100) :: HelpString                                                !< working variables
            character(len=100), dimension(NumberFreeParameterCopy) :: FormattedParmValues   !< formatted parameter values for chi2 log file


            character(len=50) :: valueh, valuel                                             !< working variable for determine gradient
            logical :: equal_flag                                                           !< required for string comparison
            logical, dimension(ma) :: ChangeSign_Flag                                       !< change sign flag
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function is stored or not


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize values for fit function calculation
            Do j = 1, NumberFreeParameterCopy                                               !< Initialize (symmetric) alpha, beta2.
                Do k = 1, j
                    alpha(j,k) = 0.d0                                                       !< clear alpha partially
                end Do
                beta2(j) = 0.d0                                                             !< clear beta2
            end Do


            !< update the parameter set with the current values of the fit parameters some algorithms optimize only the parameters in the array fitparam
            !< all other parameter in the parameter set a are not included in these algorithms but are required for the calculation of the fit function
            k = 0
            Do i = 1, ma
                if (ia(i)) then
                    k = k + 1
                    a(i) = fitparam(k)
                endif
            end Do
            ParameterVectorSet(1, :) = fitparam(:)                                          !< first entry of ParameterVectorSet contains the current
                                                                                            !< parameter vector
            ! Debug:
            ! print*,'>>>>>>>>>>>>>fitparam = ', fitparam(:)


            !< check, if gradient is necessary (for Levenberg-Marquardt it is always necessary!)
            NumParameterVectors = 1
            if (Gradientflag) then
                NumParameterVectors = NumberFreeParameterCopy + 1


                !< initialize some working parameter
                steph = 0.d0
                ChangeSign_Flag = .false.
                modelparamcopy = a                                                 !< dublicate array containing the parameter set


                !< calculate parameter vector for gradient calculation

                !$omp parallel default(shared) &
                !$omp shared(OutputFileFormat, NumberRanges, ValueEmptyOutputFile, LSRAdjustement, FirstPointExpData, LastPointExpData, idum) &
                !$omp shared(MinRange, MaxRange, NaNReplaceString, StandardWorkingDirectory, CharacterForComments, CharacterSeperatingColumns) &
                !$omp shared(ResamplingMethod, InterpolationMethod, RenormalizedChi2, OnlyYColumn, LSRAdjustementFitFlag, NormalizationFlag) &
                !$omp shared(AtOnceGradient, ia, paramset, modelparamcopy, ConverterInfit, UseRegistrationXMLFile) &
                !$omp shared(ParameterName, ParameterFormat, LeadingString, TrailingString, ParamVisible, NumberLinesOutput) &
                !$omp shared(printflag, chisqValues, NumberExpFiles, modelnumber, lengthexpdata, currentpath, NumberHeaderLines, QualityLimit) &
                !$omp shared(FitParameterName, FitParameterValue, NumberColumnsBegin, NumberColumnsEnd, NumberParamPerLine, CalculationMethod) &
                !$omp shared(CommandWordOutput, DetChi2, MaxParameter, ExternalThreadNumber, expdatax, expdatay, expdatae, TempDirectory) &
                !$omp shared(NumberXColumns, NumberYColumns, MaxColX, MaxColY, parameternumber, NumberInputFiles, ParallelizationFlag, JobID) &
                !$omp shared(MaxInputLines, UseCalculationReduction, WriteChi2Flag, Gradientflag,  initflag, CalculatedParameterSets) &
                !$omp shared(LastAlgorithmFlag, ChangeSign_Flag, steph, NumberFreeParameterCopy, ParameterVectorSet, fitparam) &
                !$omp shared(CurrentNumberLinesCalcReduction, ochisq, NumberLinesChi2, InputDataPath, FitFktInput, ModelFunction, FitFktOutput) &
                !$omp shared(ExeCommandStartScript, NumberOutputFiles, CurrentExpFile, GradientMethod, PathStartScript, CurrentYColumn) &
                !$omp shared(ExpData_reversed_flag, BestSitesModelValues, BestSitesChi2Values, BestSitesParamSet, GradientVariationValue) &
                !$omp private(fitnum, i, j, k, modelparam, d1, d2, sig2i, NumberFile, variation, value, valueh, valuel) &
                !$omp private(IntegerTrue, NumInputFile_index, i_index, j_index, equal_flag)
                !$omp do

                Do fitnum = 1, NumberFreeParameterCopy                                      !< loop over all free parameter
                    !$omp critical
                    ParameterVectorSet(fitnum + 1, :) = fitparam(:)
                    !$omp end critical
                    modelparam = modelparamcopy                                             !< load unmodified parameter values
                    i = ConverterInfit(fitnum)                                              !< get appropriate parameter index within parameter set
                    if (UseRegistrationXMLFile) then
                        call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)
                    endif

                    ! Debug:
                    ! print*,'fitnum, NumInputFile_index, i_index, j_index = ', fitnum, NumInputFile_index, i_index, j_index


                    !< check if parameter is within parameter limits
                    if (modelparamcopy(i) < paramset(3, i) .or. modelparamcopy(i) > paramset(4, i)) then
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                        if (UseRegistrationXMLFile) then
                            write(logchannel,'(2x,"The parameter ",A," is out of limits.")') &
                                              trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        else
                            write(logchannel,'(2x,"The free parameter ", I10, " is out of limits.")') fitnum
                        endif
                        write(logchannel,'(2x,"Upper limit = ", ES25.15)') paramset(4, i)
                        write(logchannel,'(2x,"Lower limit = ", ES25.15)') paramset(3, i)
                        write(logchannel,'(2x,"Value of parameter = ", ES25.15)') modelparamcopy(i)
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')

                        print '(" ")'
                        print '(" ")'
                        print '(" ")'
                        print '(11x,"Error in subroutine ModelCalcChiFunctionLM:")'
                        if (UseRegistrationXMLFile) then
                            print '(13x,"The parameter ",A," is out of limits.")', trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                        else
                            print '(13x,"The free parameter ", I10, " is out of limits.")', fitnum
                        endif
                        print '(13x,"Upper limit = ", ES25.15)', paramset(4, i)
                        print '(13x,"Lower limit = ", ES25.15)', paramset(3, i)
                        print '(13x,"Value of parameter = ", ES25.15)',modelparamcopy(i)
                        print '(" ")'
                        print '(13x,"Program aborted!")'
                        stop
                    endif


                    !< determine strength of variation
                    variation = GradientVariationValue                                      !< variation of the parameter in percent/100
                    !$omp critical
                    steph(i) = dabs(modelparamcopy(i) * variation)                          !< define stepsize for foating point numbers
                    if (UseRegistrationXMLFile) then
                        if (modelparamcopy(i) == 0.d0) then
                            if (IntegerTrue) then                                           !< is parameter an integer ??
                                steph(i) = 1.d0                                             !< variation of the parameter in percent/100
                            else
                                steph(i) = variation                                        !< variation of the parameter in percent/100
                            endif
                        elseif (IntegerTrue .and. steph(i) < 1.d0) then
                            steph(i) = 1.d0
                        endif
                    else
                        if (modelparamcopy(i) == 0.d0) then
                            steph(i) = variation                                            !< variation of the parameter in percent/100
                        endif
                    endif
                    !$omp end critical

                    ! Debug:
                    ! print*,'i = ', i
                    ! print*,'modelparamcopy(i) = ', modelparamcopy(i)
                    ! print*,'steph(i) = ',steph(i)
                    ! print*,'modelparamcopy(i) + steph(i) = ', modelparamcopy(i) + steph(i)
                    ! print*,'paramset(3, i) = ', paramset(3, i)
                    ! print*,'paramset(4, i) = ', paramset(4, i)
                    ! print*,'modelparamcopy(i) - steph(i) = ', modelparamcopy(i) - steph(i)


                    !< test, if we can accelerate the calculation
                    value = modelparamcopy(i) + steph(i)
                    if (value < paramset(3, i) .or. paramset(4, i) < value) then
                        if (value > paramset(4, i)) then                                    !< is f(x_i + h) > upper limit ?
                            value = modelparamcopy(i) - steph(i)                            !< use f(x_i - h)
                            if (value < paramset(3, i)) then
                                if (dabs(modelparamcopy(i) - paramset(4, i)) < 1.d-12) then
                                    write(logchannel,*)
                                    write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                                    if (UseRegistrationXMLFile) then
                                        write(logchannel,'(2x,"The gradient for parameter ",A," cannot be calculated.")') &
                                                          trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                                    else
                                        write(logchannel,'(2x,"The gradient for free parameter ", I10, " cannot be calculated.")') fitnum
                                    endif
                                    write(logchannel,'(2x,"The variation runs out of limits.")')
                                    write(logchannel,*)
                                    write(logchannel,'(2x,"Please increase upper and lower limits for this parameter or reduce value of variation.")')
                                    write(logchannel,*)
                                    write(logchannel,*)
                                    write(logchannel,'(2x,"Upper limit = ",ES25.15)') paramset(4, i)
                                    write(logchannel,'(2x,"Lower limit = ",ES25.15)') paramset(3, i)
                                    write(logchannel,'(2x,"value of variation = ", ES25.15)') variation
                                    write(logchannel,'(2x,"Value of parameter = ", ES25.15)') modelparamcopy(i)
                                    write(logchannel,'(" ")')
                                    write(logchannel,'("Program aborted!")')

                                    print '(" ")'
                                    print '(" ")'
                                    print '(" ")'
                                    print '("Error in subroutine ModelCalcChiFunctionLM:")'
                                    if (UseRegistrationXMLFile) then
                                        print '(2x,"The gradient for parameter ",A," cannot be calculated.")', &
                                                          trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                                    else
                                        print '(2x,"The gradient for free parameter ", I10, " cannot be calculated.")', fitnum
                                    endif
                                    print '(2x,"The variation runs out of limits.")'
                                    print '(" ")'
                                    print '(2x,"Please increase upper and lower limits for this parameter or reduce value of variation.")'
                                    print '(" ")'
                                    print '(" ")'
                                    print '(2x,"Upper limit = ", ES25.15)', paramset(4, i)
                                    print '(2x,"Lower limit = ", ES25.15)', paramset(3, i)
                                    print '(2x,"value of variation = ", ES25.15)', variation
                                    print '(2x,"Value of parameter = ", ES25.15)', modelparamcopy(i)
                                    print '(" ")'
                                    print '("Program aborted!")'
                                    stop
                                endif
                                value = paramset(4, i)
                            else
                                !$omp critical
                                ChangeSign_Flag(i) = .true.
                                !$omp end critical
                            endif
                        endif
                    endif

                    ! Debug:
                    ! print*,'> value, modelparamcopy(i), steph(i) = ',value, modelparamcopy(i), steph(i)


                    !< check, if the variation leads in combination with the Format to a variation in the current parameter
                    if (UseRegistrationXMLFile) then
                        if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                            .or. index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                            write(valueh, ParameterFormat(NumInputFile_index, i_index, j_index)) int(value)
                            write(valuel, ParameterFormat(NumInputFile_index, i_index, j_index)) int(modelparamcopy(i))
                        else
                            write(valueh, ParameterFormat(NumInputFile_index, i_index, j_index)) value
                            write(valuel, ParameterFormat(NumInputFile_index, i_index, j_index)) modelparamcopy(i)
                        endif
                        equal_flag = .true.
                        valueh = adjustl(valueh)
                        valuel = adjustl(valuel)
                        if (len_trim(valueh) == len_trim(valuel)) then
                            Do j = 1, len_trim(valueh)
                                if (valueh(j:j) /= valuel(j:j)) then
                                    equal_flag = .false.
                                    exit
                                endif
                            end Do
                        else
                            equal_flag = .false.
                        endif
                    else
                        if (dabs(value - modelparamcopy(i)) < 1.d-12) then
                            equal_flag = .true.
                        else
                            equal_flag = .false.
                        endif
                    endif
                    if (equal_flag) then                                                    !< both expressions are equal
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine ModelCalcChiFunctionLM:")')
                        if (UseRegistrationXMLFile) then
                            write(logchannel,'(2x,"The format specification of the parameter ",A)') &
                                    trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                            write(logchannel,'(2x,"prevents the variation of the current parameter.")')
                            write(logchannel,'(2x," ")')
                            write(logchannel,'(2x,"The gradient entry for this parameter is set to zero. Therefore")')
                        else
                            write(logchannel,'(2x," ")')
                            write(logchannel,'(2x,"The gradient entry for this parameter is zero. Therefore")')
                        endif
                        write(logchannel,'(2x,"no variation of this parameter in the current iteration is done")')
                        write(logchannel,'(" ")')

                        print '(" ")'
                        print '(" ")'
                        print '(" ")'
                        print '(11x,"Error in subroutine ModelCalcChiFunctionLM:")'
                        if (UseRegistrationXMLFile) then
                            print '(13x,"The format specification of the parameter ",A)', &
                                   trim(adjustl(ParameterName(NumInputFile_index, i_index, j_index)))
                            print '(13x,"prevents the variation of the current parameter.")'
                            print '(" ")'
                            print '(13x,"The gradient entry for this parameter is set to zero. Therefore")'
                        else
                            print '(" ")'
                            print '(13x,"The gradient entry for this parameter is zero. Therefore")'
                        endif
                        print '(13x,"no variation of this parameter in the current iteration is done")'
                        print '(" ")'
                    endif


                    !< modify the ith parameter
                    !$omp critical
                    modelparam(i) = value                                                   !< modify value of the ith parameter
                    ParameterVectorSet(fitnum + 1, fitnum) = value
                    !$omp end critical

                end Do

                !$omp end do
                !$omp end parallel                                                          !< end of parallel environment

                modelparam = modelparamcopy                                                 !< restore old paramter values
            endif

            ! Debug:
            !    Do l = 2, NumParameterVectors
            !        Do k = 1, NumberFreeParameterCopy
            !            if (dabs(ParameterVectorSet(l, k) - ParameterVectorSet(1, k)) > 1.d-6) then
            !                print*,'l, k, ParameterVectorSet(l, k) = ', l, k, ParameterVectorSet(l, k), ParameterVectorSet(1, k), &
            !                                                            dabs(ParameterVectorSet(l, k) - ParameterVectorSet(1, k))
            !            endif
            !        end Do
            !    end Do
            !    stop


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            ModelFunctionFlag = .true.
            call ModelCalcChiFunction(NumberFreeParameterCopy, NumParameterVectors, ParameterVectorSet, ModelFunctionFlag, TotalNumberComponents, &
                                      TotalNumberOfMolecules, chi2ValuesVector)

            ! Debug:
            ! print*,' '
            ! print*,'chi2ValuesVector(:) = ', chi2ValuesVector(:)
            ! stop 'Test LM-algorithm up to here!'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< start loop for determine chi**2
            FitFunctionOut = 0.d0                                                           !< reset array containing the values of the fit function
            Chi2Values = 0.d0
            chisqValues = 0.d0
            counter = 0
            Do NumberFile = 1, NumberExpFiles                                               !< loop over exp. data files
                CurrentExpFile = NumberFile


                !< start loop over all lines(expdata)
                Do n = 1, NumberYColumns(NumberFile)                                        !< loop over y-columns
                    Do i = 1, lengthexpdata(NumberFile)                                     !< loop over all line of current exp. data file
                        CurrentYColumn = n
                        counter = counter + 1


                        !< get fit function
                        ymod = ModelFunction(1, NumberFile, n, i)

                        ! Debug:
                        !    print*,'>> NumberFile, n, i, counter, ymod = ', NumberFile, n, i, counter, ymod
                        !    print*,'>> ModelFunction(1, NumberFile, n, i) = ', ModelFunction(1, NumberFile, n, i)


                        !< get gradient
                        if (Gradientflag) then
                            dyda = 0.d0
                            Do l = 1, NumberFreeParameterCopy
                                j = ConverterInfit(l)                                       !< get appropriate parameter index within parameter set

                                ! order new: NumberFile, n, i, l,      j
                                ! order old: NumberFile, n, i, l,      j
                                ! order org: NumberFile, k, j, fitnum, i

                                ! Debug:
                                !    print*,'l, j = ', l, j
                                !    print*,'ParameterVectorSet(l + 1, l) = ', ParameterVectorSet(l + 1, l)
                                !    print*,'steph(j) = ', steph(j)
                                !    print*,'ChangeSign_Flag(j) = ', ChangeSign_Flag(j)
                                !    print*,'ModelFunction(l + 1, NumberFile, n, i) = ', ModelFunction(l + 1, NumberFile, n, i)
                                !    print*,'GradientMethod = ', GradientMethod


                                !< determine gradient for numerical recepies version
                                if (GradientMethod == 1) then
                                    if (ParameterVectorSet(l + 1, l) > paramset(4, j) .or. ChangeSign_Flag(j)) then
                                        dyda(j) = (ModelFunction(1, NumberFile, n, i) - ModelFunction(l + 1, NumberFile, n, i)) / steph(j)
                                    else
                                        dyda(j) = (ModelFunction(l + 1, NumberFile, n, i) - ModelFunction(1, NumberFile, n, i)) / steph(j)
                                    endif


                                !< determine gradient for minpack version
                                elseif (GradientMethod == 2) then
                                    sig2i = 1.d0
                                    if (expdatae(NumberFile, i, n) /= 0.d0) then            !< do the experimental datas include errors
                                        sig2i = 1.d0 / (expdatae(NumberFile, i, n) * expdatae(NumberFile, i, n))  !< define sig2i factor
                                    endif
                                    d1 = (expdatay(NumberFile, i, n) - ModelFunction(1, NumberFile, n, i))**2 * sig2i
                                    d2 = (expdatay(NumberFile, i, n) - ModelFunction(l + 1, NumberFile, n, i))**2 * sig2i
                                    if (ParameterVectorSet(l + 1, l) > paramset(4, j) .or. ChangeSign_Flag(j)) then
                                        dyda(j) = (d1 - d2) / steph(j)
                                    else
                                        dyda(j) = (d2 - d1) / steph(j)
                                    endif
                                endif
                                ! dyda(j) = ModelFunction(l + 1, NumberFile, n, i)
                            end Do

                            ! Debug:
                            ! print*,'ymod = ', ymod
                            ! print*,'dyda(:) = ', dyda(:)
                            ! print*,'a = ', a
                            ! print*,'############################################################################'
                            ! stop
                        endif
                        FitFunctionOut(NumberFile, i, n) = ymod                             !< save value of fit function

                        ! Debug:
                        !    print*,' '
                        !    print*,' '
                        !    print*,'NumberExpFiles = ',NumberExpFiles
                        !    print*,'lengthexpdata(NumberFile) = ',lengthexpdata(NumberFile)
                        !    print*,'NumberYColumns(NumberFile) = ',NumberYColumns(NumberFile)
                        !    print*,'NumberFile = ',NumberFile
                        !    print*,'i = ',i
                        !    print*,'n = ',n
                        !    print*,'lenposdatexp = ',lenposdatexp
                        !    print*,'posdatexp = ',posdatexp
                        !    print*,'expdatay(NumberFile,i,n) = ',expdatay(NumberFile,i,n)
                        !    print*,'ymod = ',ymod
                        !    print*,'dyda = ',dyda(1:NumberFreeParameterCopy)
                        !    if (i==3) stop


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< determine chi**2 by calculating the difference (y_i^{obs) - y_i(fit))**2
                        if (abs(DetChi2) == 1) then

                            sig2i = 1.d0
                            if (expdatae(NumberFile,i,n) /= 0.d0) then                      !< do the experimental datas include errors
                                sig2i = 1.d0/(expdatae(NumberFile,i,n) * expdatae(NumberFile,i,n))      !< define sig2i factor
                            endif
                            dy = (expdatay(NumberFile, i, n) - ymod)                        !< define distance between fit and data
                            Chi2Values(NumberFile, i, n) = dy * dy * sig2i                  !< save chi^2
                            chisqValues(0) = chisqValues(0) + dy * dy * sig2i               !< And find chi^2.


                            if (Gradientflag) then
                                j = 0
                                Do l = 1, ma                                                !< loop over all parameters
                                    if (ia(l)) then                                         !< is the lth parameters optimized?
                                        j = j + 1
                                        wt = dyda(l) * sig2i                                !< define weighting factor
                                        k = 0
                                        Do m = 1, l                                         !< determine alpha matrix
                                            if (ia(m)) then
                                                k = k + 1
                                                alpha(j,k) = alpha(j,k) + wt * dyda(m)
                                            endif
                                        end Do
                                        beta2(j) = beta2(j) + dy * wt                       !< calculate beta2 array
                                    endif
                                end Do
                            endif


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< determine chi**2 by calculating the difference (y_i^{obs)**2 - y_i(fit)**2)
                        elseif (abs(DetChi2) == 2) then

                            sig2i = 1.d0
                            if (expdatae(NumberFile,i,n) /= 0.d0) then                      !< do the experimental datas include errors
                                sig2i = 1.d0/(expdatae(NumberFile,i,n) * expdatae(NumberFile,i,n))      !< define sig2i factor
                            endif
                            dy = (expdatay(NumberFile, i, n)**2 - ymod**2)
                            Chi2Values(NumberFile, i, n) = dy * dy * sig2i                  !< save chi^2
                            chisqValues(0) = chisqValues(0) + dy * dy * sig2i               !< And find chi^2.


                            if (Gradientflag) then
                                j = 0
                                Do l = 1, ma                                                !< loop over all parameters
                                    if (ia(l)) then                                         !< is the lth parameters optimized?
                                        j = j + 1
                                        wt = dyda(l) * sig2i                                !< define weighting factor
                                        k = 0
                                        Do m = 1, l                                         !< determine alpha matrix
                                            if (ia(m)) then
                                                k = k + 1
                                                alpha(j, k) = alpha(j, k) + wt * dyda(m)
                                            endif
                                        end Do
                                        beta2(j) = beta2(j) + dy * wt                       !< calculate beta2 array
                                    endif
                                end Do
                            endif

                        endif
                    end Do                                                                  !< loop over all line of current exp. data file
                end Do                                                                      !< loop over y-columns
            end Do                                                                          !< loop over exp. data files

            ! Debug:
            ! print*,'chisqValues(0) = ', chisqValues(0)


            !< Fill in the symmetric side.
            Do j = 2, NumberFreeParameterCopy
                Do k = 1, (j - 1)
                   alpha(k, j) = alpha(j, k)
                end Do

                ! Debug:
                ! print*,'j = ', j
                ! print*,'alpha(j,:) = ', alpha(j,:NumberFreeParameterCopy)
                ! print*,'beta2(j) = ', beta2(j)
            end Do


            !< writing current value of chi**2 and corresponding values of parameters to file
            if (WriteChi2Flag) then
                NumberLinesChi2 = NumberLinesChi2 + 1
                k = 0
                FormattedParmValues(:)(:) = ""
                Do i = 1, ma
                    if (ia(i)) then
                        k = k + 1
                        a(i) = fitparam(k)

                        ! Debug:
                        !   print*,'fitparam(k) = ',k,fitparam(k)


                        !< build list with fit parameters
                        HelpString = ""
                        if (UseRegistrationXMLFile) then
                            call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)
                            if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                                .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(i))
                            else
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(i)
                            endif
                        else
                            write(HelpString, '(ES25.15)') a(i)
                        endif
                        FormattedParmValues(k) = trim(adjustl(HelpString))
                    endif
                end Do


                !< write line of formatted parameter values to log file
                write(Chi2Channel,'(ES25.15,$)') chisqValues(0)
                Do i = 1, NumberFreeParameterCopy
                    write(Chi2Channel,'(1x,A,$)') " " // trim(adjustl(FormattedParmValues(i)))
                end Do
                write(Chi2Channel,*)
            endif


            return
        end subroutine ModelCalcChiFunctionLM


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunctionGeneral
        !>
        !> calculates the chi2 values for several given parameter vector sets
        !>
        !>
        !> input variables:     ma:                 total number of parameters
        !>                      ia:                 flags for including/excluding parameter in the fit
        !>                      a:                  array containing the parameter set
        !>                      NumberParamVectors:         number of parameter vectors
        !>                      NumberFreeParameterCopy:    number of fitted parameters
        !>                      ParameterVectorSet:
        !>                      NumFile:            number of experimental files
        !>                      MaxL:               max number of lines of all experimental files
        !>                      MaxCol:             max number of columns of all experimental files
        !>                      FitFunctionOut:     values of the fit function at all calculated points
        !>                      Chi2Values:         values of the fit function at all calculated points
        !>
        !> output variables:    none
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 19.08.2014
        !>
        subroutine ModelCalcChiFunctionGeneral(ma, ia, a, NumberParamVectors, NumberFreeParameterCopy, NumFile, MaxL, MaxCol, ParameterVectorSet, &
                                               chi2ValuesVector)

            use omp_lib

            implicit none
            integer :: ma                                                                   !< total number of parameters
            integer :: NumberParamVectors                                                   !< number of parameter vectors
            integer :: NumberFreeParameterCopy                                              !< number of fitted parameters
            integer :: NumFile                                                              !< number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            real*8, dimension(ma) :: a                                                      !< array containing the parameter set
            real*8, dimension(NumberParamVectors, NumberFreeParameterCopy) :: ParameterVectorSet   !< set of parameter vectors calculated by model module
            real*8, dimension(NumberParamVectors) :: chi2ValuesVector                       !< vector containing chi2 values for each parameter vector
                                                                                            !< in ParameterVectorSet
            logical, dimension(ma) :: ia                                                    !< flags for including/excluding parameter in the fit


            !< working variables
            integer :: i, k, l, m, n, NumberFile                                            !< loop variables
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            real*8 :: sig2i                                                                 !< working variable for error (sigma)
            real*8 :: val                                                                   !< working variable
            real*8, dimension(1) :: chi2valDummy                                            !< dummy argument
            character(len=100) :: HelpString                                                !< working variables
            character(len=100), dimension(NumberFreeParameterCopy) :: FormattedParmValues   !< formatted parameter values for chi2 log file
            logical :: IntegerTrue                                                          !< flag for identification of integer numbers
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function is stored or not
            logical :: StoredBefore_Flag                                                    !< used to avoid double entries in BestSitesParamSet


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            ModelFunctionFlag = .false.
            chi2ValuesVector = 1.d99
            call ModelCalcChiFunction(NumberFreeParameterCopy, NumberParamVectors, ParameterVectorSet, ModelFunctionFlag, TotalNumberComponents, &
                                      TotalNumberOfMolecules, chi2ValuesVector)
            ! Debug:
            ! print*,' '
            ! print*,'chi2ValuesVector(:) = ', chi2ValuesVector(:)
            ! stop 'Test LM-algorithm up to here!'


            !< writing current value of chi**2 and corresponding values of parameters to file
            Do l = 1, NumberParamVectors
                if (WriteChi2Flag) then
                    NumberLinesChi2 = NumberLinesChi2 + 1
                    k = 0
                    FormattedParmValues(:)(:) = ""
                    Do i = 1, ma
                        if (ia(i)) then
                            k = k + 1
                            a(i) = ParameterVectorSet(l, k)


                            !< build list with fit parameters
                            HelpString = ""
                            if (UseRegistrationXMLFile) then
                                call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)
                                if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                                    .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                                    write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(i))
                                else
                                    write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(i)
                                endif
                            else
                                write(HelpString, '(ES25.15)') a(i)
                            endif
                            FormattedParmValues(k) = trim(adjustl(HelpString))
                        endif
                    end Do


                    !< write line of formatted parameter values to log file
                    write(Chi2Channel,'(ES25.15,$)') chi2ValuesVector(l)
                    Do i = 1, NumberFreeParameterCopy
                        write(Chi2Channel,'(1x,A,$)') " " // trim(adjustl(FormattedParmValues(i)))
                    end Do
                    write(Chi2Channel,*)
                endif


                !< store model function values and correpsonding chi2 values for the best fitness
                Do k = 1, QualityLimit
                    if (chi2ValuesVector(l) < BestSitesParamSet(k, 1) .or. PlotIterationFlag) then
                        StoredBefore_Flag = .false.
                        if (QualityLimit > 1) then
                            Do m = 1, QualityLimit
                                if (chi2ValuesVector(l) == BestSitesParamSet(m, 1)) then
                                    call CompareTwoParameterSets(StoredBefore_Flag, ma, NumberFreeParameterCopy, ia, BestSitesParamSet(m, 2:), &
                                                                 ParameterVectorSet(l, :))
                                    if (StoredBefore_Flag) then
                                        exit
                                    endif
                                endif
                            end Do
                        endif
                        if (.not. StoredBefore_Flag .or. PlotIterationFlag) then
                            if (k < QualityLimit) then
                                Do m = QualityLimit, (k + 1), (-1)
                                    BestSitesParamSet(m, :) = BestSitesParamSet((m - 1), :)
                                    BestSitesModelValues(m, :, :, :) = BestSitesModelValues((m - 1), :, :, :)
                                    BestSitesChi2Values(m, :, :, :) = BestSitesChi2Values((m - 1), :, :, :)
                                end Do
                            endif
                            BestSitesParamSet(k, 1) = chi2ValuesVector(l)
                            BestSitesParamSet(k, 2:) = ParameterVectorSet(l, :)


                            !< determine model function values for lth chi2 values
                            if (PlotIterationFlag) then                                     !< we need the model function values and the chi2 values only
                                                                                            !< if we want to fit the model function for each iteration
                                ModelFunctionFlag = .true.
                                chi2valDummy = 1.d99
                                m = 1
                                call ModelCalcChiFunction(NumberFreeParameterCopy, m, ParameterVectorSet(l, :), ModelFunctionFlag, &
                                                          TotalNumberComponents, TotalNumberOfMolecules, chi2valDummy)


                                !< determine chi2 values for each frequency point
                                Do NumberFile = 1, NumberExpFiles                           !< loop over exp. data files
                                    Do n = 1, NumberYColumns(NumberFile)                    !< loop over y-columns
                                        Do i = 1, lengthexpdata(NumberFile)                 !< loop over all line of current exp. data file
                                            sig2i = 1.d0
                                            if (expdatae(NumberFile, i, n) /= 0.d0) then    !< do the experimental datas include errors
                                                sig2i = 1.d0/(expdatae(NumberFile, i, n) * expdatae(NumberFile, i, n))
                                            endif


                                            !< determine chi**2 by calculating the difference (y_i^{obs) - y_i(fit))**2
                                            if (abs(DetChi2) == 1) then
                                                val = (expdatay(NumberFile, i, n) - ModelFunction(1, NumberFile, n, i))**2


                                            !< determine chi**2 by calculating the difference (y_i^{obs)**2 - y_i(fit)**2)
                                            elseif (abs(DetChi2) == 2) then
                                                val = (expdatay(NumberFile, i, n)**2 - ModelFunction(1, NumberFile, n, i)**2)
                                            endif
                                            BestSitesChi2Values(k, NumberFile, i, n) = val * sig2i
                                            BestSitesModelValues(k, NumberFile, i, n) = ModelFunction(1, NumberFile, n, i)
                                        end Do
                                    end Do
                                end Do
                            endif
                            exit
                        endif
                    endif
                end Do
            end Do
            return
        end subroutine ModelCalcChiFunctionGeneral


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcChiFunction
        !>
        !> calculates the chi2 values for a given set of parameter vectors
        !>
        !>
        !> input variables:     NumberFreeParam:        number of free parameters
        !>                      NumParamVectors:        number of parameter vectors
        !>                      ParameterVectorSet:     parameter vector
        !>                      ModelFunctionFlag:      flag for indicating if model function values are stored or not
        !>                      NumComp:                total number of components
        !>
        !> output variables:    chi2ValuesVector:       chi2 value for parameter vector
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 15.08.2014
        !>
        subroutine ModelCalcChiFunction(NumberFreeParam, NumParamVectors, ParameterVectorSet, ModelFunctionFlag, NumComp, TotalNumMol, chi2ValuesVector)

            implicit none
            integer :: l, n, i, NumberFile                                                  !< loop variables
            integer :: k                                                                    !< counter variable
            integer :: num                                                                  !< working variable for model function value array
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            integer :: NumberFreeParam                                                      !< number of free parameters
            integer :: NumParamVectors                                                      !< number of parameter vectors
            integer :: NumComp                                                              !< total number of components
            integer :: errcode, LastErrcode                                                 !< error code for GPU
            integer :: TotalNumMol                                                          !< total number of molecules
            real*8, dimension(NumParamVectors) :: chi2ValuesVector                          !< chi2 value for parameter vector
            real*8, dimension(NumParamVectors, NumberFreeParam) :: ParameterVectorSet       !< parameter vector set
            logical :: ModelFunctionFlag                                                    !< flag for indicating if model function values are stored


            !< CUDA variables
            integer :: istat
            integer :: MaxNumThreadsX                                                       !< max. number of threads (X)
            integer :: MaxNumThreadsY                                                       !< max. number of threads (Y)
            integer :: MaxNumThreadsZ                                                       !< max. number of threads (Z)
            integer :: MaxGridThreads                                                       !< max. number of all threads
            integer, dimension(3) :: MaxGrid                                                !< maximum grid dimensions


            !< kernel variables
            real*8, dimension(NumParamVectors, NumberFreeParam), device :: ParameterVectorSet_d
            real*8, allocatable, dimension(:, :), device :: ModelFunctionListLocal_d
            real*8, dimension(NumParamVectors), device :: chi2ValuesVector_d


            type (cudaDeviceProp) :: prop                                                   !< CUDA variable:
            type(dim3) :: grid, tBlock                                                      !< CUDA variables:


            !< make settings for current GPU
            MaxNumThreadsX = prop%maxThreadsPerBlock                                        !< get max. number of threads allowed on the current GPU
            MaxNumThreadsX = min(1024, NumParamVectors)

!            MaxNumThreadsY = prop%maxThreadsPerBlock                                        !< get max. number of threads allowed on the current GPU
!            MaxNumThreadsY = min(1024, NumParamVectors)
!            MaxNumThreadsZ = prop%maxThreadsPerBlock                                        !< get max. number of threads allowed on the current GPU
!            MaxNumThreadsZ = min(1024, NumParamVectors)


            MaxGrid = prop%maxGridSize                                                      !< get max. grid dimensions allowed on the current GPU


            !< check, if number of parameter vectors is less than max. grid dimension
            MaxGridThreads = min(NumParamVectors, MaxGrid(0))


            !< define
            tBlock = dim3(MaxNumThreadsX, 1, 1)
            grid = dim3(ceiling(real(NumParamVectors)/tBlock%x), 1, 1)


            !< do we need the model function values
            if (ModelFunctionFlag) then
                num = TotalNumberDataPoints
            else
                num = 1
            endif
            if (allocated(ModelFunctionListLocal_d)) then
                deallocate(ModelFunctionListLocal_d, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    Do ErrChannelIndex = 1, 1
                        ErrChannel = AllErrChannels(ErrChannelIndex)
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(1x,"Error in subroutine ModelCalcChiFunction!")')
                        write(ErrChannel, '(3x,"Can not deallocate variable ModelFunctionListLocal_d!")')
                        write(ErrChannel, '(" ")')
                        write(ErrChannel, '(3x,"Please restart the program!")')
                    end Do
                    return
                endif
            endif
            allocate(ModelFunctionListLocal_d(NumParamVectors, num), stat = allocstatus)
            if (allocstatus /= 0) then
                Do ErrChannelIndex = 1, 1
                    ErrChannel = AllErrChannels(ErrChannelIndex)
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(1x,"Error in subroutine ModelCalcChiFunction!")')
                    write(ErrChannel, '(3x,"Can not allocate variable ModelFunctionListLocal_d!")')
                    write(ErrChannel, '(" ")')
                    write(ErrChannel, '(3x,"Please restart the program!")')
                end Do
                return
            endif
            ModelFunctionListLocal_d = 0.d0
            chi2ValuesVector = 1.d99


            !< copy variables to GPU RAM
            ModelFunctionFlag_d = ModelFunctionFlag
            chi2ValuesVector_d = chi2ValuesVector
            ParameterVectorSet_d = ParameterVectorSet


            !< check, if parameter are out of allowed range
            Do l = 1, NumParamVectors
                Do n = 1, NumberFreeParam
                    k = ConverterInfit(n)
                    if (ParameterVectorSet(l, n) < paramset(3, k) .or. paramset(4, k) < ParameterVectorSet(l, n)) then
                        ParameterVectorSet_d(l, 1) = 1.d99
                        exit
                    endif
                end Do
            end Do

            ! Debug:
                print*,' '
                print*,'NumParamVectors = ', NumParamVectors
                print*,'tBlock = ', tBlock
                print*,'grid = ', grid


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate model function for all parameter vectors in variable 'ParameterVectorSet'
            call ModelCalcSpectrum<<< grid, tBlock >>>(ParameterVectorSet_d, chi2ValuesVector_d, ModelFunctionListLocal_d, NumParamVectors)
            istat = cudaDeviceSynchronize()
            chi2ValuesVector = chi2ValuesVector_d
            ParameterVectorSet = ParameterVectorSet_d

            ! Debug:
            print '("chi2ValuesVector(1:5) = ", 5(ES25.15))', chi2ValuesVector(1:min(5,NumParamVectors))


            !< check if an error occurred on the GPU
            errcode = cudaGetLastError()
            LastErrcode = cudaPeekAtLastError()
            if (errcode /= 0 .or. LastErrcode /= 0) then
                print '(" ")'
                print '(" ")'
                print '(1x, "Error on GPU:")'
                print '(3x, "An error occurred on the GPU!")'
                print '(" ")'
                print '(3x, "Error code = ", I5)', errcode
                print '(3x, "Last error code = ", I5)', LastErrcode
                print '(" ")'
                print '(3x, "Error message: = ", A)', cudaGetErrorString(errcode)
                print '(3x, "Last error message: = ", A)', cudaGetErrorString(LastErrcode)
                print '(" ")'
                print '(" ")'
                stop
            endif
stop

            !< copy model function values to ModelFunction array
            if (ModelFunctionFlag) then                                                     !< store model function values
                Do l = 1, NumParamVectors
                    k = 0
                    Do NumberFile = 1, NumberExpFiles                                       !< loop over exp. data files
                        Do n = 1, NumberYColumns(NumberFile)                                !< loop over y-columns
                            Do i = 1, lengthexpdata(NumberFile)                             !< loop over all line of current exp. data file
                                k = k + 1                                                   !< increase data point counter
                                ModelFunction(l, NumberFile, n, i) = ModelFunctionListLocal_d(l, k)
                            end Do

                            ! Debug:
                            ! print*,'minval(ModelFunction(l, NumberFile, n, :)) = ', minval(ModelFunction(l, NumberFile, n, :))
                            ! print*,'maxval(ModelFunction(l, NumberFile, n, :)) = ', maxval(ModelFunction(l, NumberFile, n, :))
                        end Do
                    end Do
                end Do
            endif

            ! Debug:
            !   print*,' '
            !   print*,'chi2ValuesVector(:) = ', chi2ValuesVector(:)
            !   stop 'Test LM-algorithm up to here!'

            return
        end subroutine ModelCalcChiFunction


        !>************************************************************************************************************************************************
        !> subroutine: ModelCalcSpectrum
        !>
        !> calculates the myXCLASS spectrum for a given parameter vector
        !>
        !>
        !> input variables:     ParameterVector:        parameter vector set
        !>                      NumParamVecs:           number of parameter vectors
        !>
        !> output variables:    chi2ValuesVector:       chi2 values for each parameter vector
        !>                      ModelFunctionListLocal: model function values for each parameter vector
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 31.07.2014
        !>
        attributes(global) subroutine ModelCalcSpectrum(ParameterVector, chi2ValuesVector, ModelFunctionListLocal, NumParamVecs)

            implicit none
            real*8 :: ParameterVector(:, :)                                                 !< (input) parameter vector set
            real*8 :: chi2ValuesVector(:)                                                   !< (output) chi2 values for each parameter vector
            real*8 :: ModelFunctionListLocal(:, :)                                          !< (output) model function values for each parameter vector
            integer, value :: NumParamVecs                                                  !< (input) total number of parameter vectors


            !< myXCLASS variables
            integer :: fitnum, i, j, k, l, ae, c, t                                         !< loop variables
            integer :: i1, i2, ff1, ff2                                                     !< working variables for copying
            integer :: IndexComp                                                            !< overall component index
            integer :: ThreadNumberX                                                        !< current thread number (X)
            integer :: ThreadNumberY                                                        !< current thread number (Y)
            integer :: ThreadNumberZ                                                        !< current thread number (Z)
            integer :: varIndex                                                             !< myXCLASS variable index
            integer :: ErrChannelIndexLocal                                                 !< working varibale for error handling
            integer :: ErrChannelLocal                                                      !< working varibale for error handling
            integer :: MoleculeIndex                                                        !< working variable: index for molecule
            integer :: FirstMolecularDataIndex                                              !< working variable: first index for molecular data table
            integer :: LastMolecularDataIndex                                               !< working variable: last index for molecular data table
            integer :: FirstIndex                                                           !< working variable: index for first freq. data point
            integer :: LastIndex                                                            !< working variable: index for last freq. data point
            integer :: FreqIndex                                                            !< working variable: index for frequency point
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            real*8 :: chi2Value                                                             !< wokring variable: resulting chi2 value
            real*8 :: eta_source                                                            !< working variable: beam filling factor
            real*8 :: logt, x0, x1, f0, f1                                                  !< working variables for interpolation
            real*8 :: freq                                                                  !< working variable: frequency for index 'FreqIndex'
            real*8 :: freq_t                                                                !< working variable: frequency as temperature
            real*8 :: beam_size                                                             !< working variable: beam size (eqn. 2 - 3)
            real*8 :: j_tk, j_cb, j_back                                                    !< working variables: Rayleigh temperatures
            real*8 :: tau_d                                                                 !< working variable: tau dust
            real*8 :: tau_t                                                                 !< working variable: tau for each transition
            real*8 :: tau_l                                                                 !< working variable: sum over all taus
            real*8 :: vLSR                                                                  !< working variable: v_LSR
            real*8 :: sigma                                                                 !< working variable: sigma
            real*8 :: Temperature                                                           !< working variable: temperature
            real*8 :: LocalIntensity                                                        !< working variable: calc. intensity at current data point
            real*8 :: val
            real*8, dimension(11, 10000) :: CopymyXCLASSParameter


            !< reset output variable
            ModelFunctionListLocal = 0.d0


            !<============================================================================================================================================
            !< variables being used from the global module definition
            !<      Firsti                                  shared      integer             working variable for interpolation
            !<      Lasti                                   shared      integer             working variable for interpolation
            !<      stepi                                   shared      integer             working variable for interpolation
            !<      TotalNumberComponents                   shared      integer             counter for total number of components of all molecules
            !<      TotalNumberOfFrequencyRanges            shared      integer             total number of frequency ranges
            !<      TotalNumberOfMolecules                  shared      integer             total number of molecules including isotopologues
            !<      NumberOfTemperatures                    shared      integer             number of temperatures
            !<      NumberComponentsPerMolecule             shared      integer             number of components per molecule
            !<                                                          dimension: (0:TotalNumberOfMolecules)
            !<      IsoNfitConversionTable                  shared      integer             iso ratios conversion table for free paramter index
            !<                                                          dimension: (TotalNumberOfMolecules, TotalNumberOfMolecules)
            !<      SpectrumIndexForFreqRange               shared      integer             store spectrum index for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      DataPointIndexFreqRange                 shared      integer             index of data point of first and last freq. in freq. range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges, 2)
            !<      MolecularDataIndices                    shared      integer             start and end index for each molecule and frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges, TotalNumberOfMolecules, 2)
            !<      ConversionTableMAGIXmyXCLASSParam       shared      integer             conversion table between myXCLASS and MAGIX parameter
            !<                                                          dimension: (NumberFreeParameterCopy, 2)
            !<      CompMoleculeIndex                       private     integer             molecule index for each component
            !<                                                          dimension: (TotalNumberComponents)
            !<      debug_freq                              shared      real*8              frequency for debugging
            !<      ckms                                    shared      real*8              speed of light in km/s
            !<      pi                                      shared      real*8              pi
            !<      TempLow                                 shared      real*8              working variable for extrapolation of part. func.
            !<      TempHigh                                shared      real*8              working variable for extrapolation of part. func.
            !<      TempPartFunc                            shared      real*8              temperatures for partition function
            !<                                                          dimension: (NumberOfTemperatures)
            !<      BackgroundTemperatureRange              shared      real*8              T_Back for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      TemperatureSlopeRange                   shared      real*8              T_Slope for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      HydrogenColumnDensityRange              shared      real*8              nH for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      DustBetaRange                           shared      real*8              beta for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      KappaRange                              shared      real*8              kappa for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      TelescopeSize                           shared      real*8              size of telescope for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      StartFrequency                          shared      real*8              first frequency for each frequency range
            !<                                                          dimension: (TotalNumberOfFrequencyRanges)
            !<      ObservationalDataList                   shared      real*8              list containing all observational data
            !<                                                          dimension: (TotalNumberDataPoints, 3)
            !<      MolecularData                           shared      real*8              array containing the molecular data for all molecules and
            !<                                                          dimension: (TotalNumberOfTransitions, 4)
            !<      MoleculeNames                           shared      character(len=40)   names of all molecules (including isotopologues)
            !<                                                          dimension: (MaxNumberOfMolecules)
            !<      IsoRatioConversionTable                 shared      real*8              table with iso ratios between iso master and molecule
            !<                                                          dimension: (TotalNumberOfMolecules, TotalNumberOfMolecules)
            !<      lgQ                                     shared      real*8              lgQ entries of table PartitionFunction
            !<                                                          dimension: (NumberOfTemperatures, NumberMoleculePartFunc)
            !<      myXCLASSParameter                       shared      real*8              array containing all molfit parameters for each component
            !<                                                          dimension: (11, TotalNumberComponents)
            !<      nHFlag                                  shared      logical             flag for global setting of nH, kappa and beta
            !<      tbFlag                                  shared      logical             flag for global setting T_Back and T_Slope
            !<      IsoFlag                                 shared      logical             flag indicating use of isotopologues
            !<============================================================================================================================================


            !< get current thread number (for x direction) and check, if thread ID is within allowed range
            ThreadNumberX = blockDim%x * (blockIdx%x - 1) + threadIdx%x
            if (ThreadNumberX > NumParamVecs) return


            !< check, if parameters are within allowed range
            if (ParameterVector(ThreadNumberX, 1) == 1.d99) then
                chi2ValuesVector(ThreadNumberX) = 1.d99
                return
            endif


            !< get current thread number (for ydirection) and check, if thread ID is within allowed range
            ThreadNumberY = blockDim%y * (blockIdx%y - 1) + threadIdx%y
            ThreadNumberZ = blockDim%z * (blockIdx%z - 1) + threadIdx%z
            if (ThreadNumberY > NumParamVecs .or. ThreadNumberZ > NumParamVecs) return


            i = min(10000, TotalNumberComponents_d)
            CopymyXCLASSParameter(:, 1:i) = myXCLASSParameter_d(:, 1:i)


            !<============================================================================================================================================
            !< update myXCLASS parameter vector with new parameter values
            !< CopymyXCLASSParameter(1, :)      source size
            !< CopymyXCLASSParameter(2, :)      T_rot
            !< CopymyXCLASSParameter(3, :)      N_tot
            !< CopymyXCLASSParameter(4, :)      V_width
            !< CopymyXCLASSParameter(5, :)      V_off
            !< CopymyXCLASSParameter(6, :)      T_Back
            !< CopymyXCLASSParameter(7, :)      T_Slope
            !< CopymyXCLASSParameter(8, :)      nH_column
            !< CopymyXCLASSParameter(9, :)      beta
            !< CopymyXCLASSParameter(10, :)     kappa
            !< CopymyXCLASSParameter(11, :)     Emission/Absorption flag
            Do fitnum = 1, NumberFreeParameter_d                                            !< loop over all free parameters


                !< update molfit file parameters
                IndexComp = ConversionTableMAGIXmyXCLASSParam_d(fitnum, 1)                  !< get current overall component index
                varIndex = ConversionTableMAGIXmyXCLASSParam_d(fitnum, 2)                   !< get current myXCLASS variable index
                if (IndexComp > 0 .and. varIndex > 0 .and. varIndex < 11) then              !< check, if current free param. corresponds to molfit param.

                    ! Debug:
                    ! print*,'varIndex, IndexComp = ', varIndex, IndexComp
                    ! print*,'CopymyXCLASSParameter(:, IndexComp) = ', CopymyXCLASSParameter(varIndex, IndexComp)
                    ! print*,'ParameterVector(ThreadNumberX, fitnum) = ', ParameterVector(ThreadNumberX fitnum)


                    val = ParameterVector(ThreadNumberX, fitnum)


                    !< convert log10 value back to linear value check for overflow
                    if (varIndex == 3) then
                        if (val > 300.d0) val = 300.d0
                        val = 10.d0**val
                    endif
                    CopymyXCLASSParameter(varIndex, IndexComp) = val


                !< update iso ratio table
                elseif (IsoFlag_d) then
                    Do j = 1, TotalNumberOfMolecules_d
                        Do k = 1, TotalNumberOfMolecules_d
                            if (int(IsoNfitConversionTable_d(j, k)) == fitnum) then
                                CopyIsoRatioConversionTable(ThreadNumberX, j, k) = ParameterVector(ThreadNumberX, fitnum)
                            elseif (int(IsoNfitConversionTable_d(j, k)) == -fitnum) then
                                CopyIsoRatioConversionTable(ThreadNumberX, j, k) = 1.d0 / ParameterVector(ThreadNumberX, fitnum)
                            endif
                        end Do
                    end Do
                endif
            end Do
return


            !<============================================================================================================================================
            !< update parameters for isotopologues
            if (IsoFlag_d) then
                Do j = 1, TotalNumberOfMolecules_d
                    i1 = sum(NumberComponentsPerMolecule_d(:(j - 1))) + 1
                    i2 = i1 + NumberComponentsPerMolecule_d(j) - 1

                    ! Debug:
                    !    print*,"#####################################################################"
                    !    print*,"j, MoleculeNames_d(j) = ", j, MoleculeNames_d(j)
                    !    print*,"NumberComponentsPerMolecule_d(j) = ", j, NumberComponentsPerMolecule_d(j)
                    !    print*,"CopyIsoRatioConversionTable(ThreadNumberX, j, :) = ", CopyIsoRatioConversionTable(ThreadNumberX, j, :)
                    !    Do k = i1, i2
                    !        print*,"CopymyXCLASSParameter(:, k) = ", k, CopymyXCLASSParameter(:, k)
                    !    end Do
                    !    print*," "
                    !    print*," "


                    Do k = 1, TotalNumberOfMolecules_d

                        ! Debug:
                        !    print*,"k, MoleculeNames_d(j) = ", k, MoleculeNames_d(k)
                        !    print*,"CopyIsoRatioConversionTable(ThreadNumberX, j, k) = ", CopyIsoRatioConversionTable(ThreadNumberX, j, k)
                        !    print*,"NumberComponentsPerMolecule_d(k) = ", NumberComponentsPerMolecule_d(k)
                        !    print*," "


                        if (CopyIsoRatioConversionTable(ThreadNumberX, j, k) /= 0.d0 .and. NumberComponentsPerMolecule_d(k) /= 0) then

                            ff1 = sum(NumberComponentsPerMolecule_d(:(k - 1))) + 1
                            ff2 = ff1 + NumberComponentsPerMolecule_d(k) - 1

                            ! Debug:
                            ! print*,"i1, i2 = ",i1, i2
                            ! print*,"ff1, ff2 = ",ff1, ff2


                            !< copy parameters for current isotopologues from isomaster and scale column densities with iso ratio factor
                            !CopymyXCLASSParameter(:, ff1:ff2) = CopymyXCLASSParameter(:, i1:i2)
                            CopyCompMoleculeIndex(ThreadNumberX, ff1:ff2) = k
                            Do i = ff1, ff2
                                CopymyXCLASSParameter(:, i) = CopymyXCLASSParameter(:, i - ff1 + i1)
                                CopymyXCLASSParameter(3, i) = CopyIsoRatioConversionTable(ThreadNumberX, j, k) * CopymyXCLASSParameter(3, i)
                            end Do
                        endif
                    end Do
                end Do
            endif

            ! Debug:
            !    Do i = 1, TotalNumberComponents_d
            !        print*,"i = ", i
            !        j = CopyCompMoleculeIndex(ThreadNumberX, i)
            !        print*,"MoleculeNames_d = ", trim(adjustl(MoleculeNames_d(j)))
            !        print*,"CopymyXCLASSParameter(:, i) = ", CopymyXCLASSParameter(:, i)
            !    end Do


            !<============================================================================================================================================
            !< determine partiton functions for all rotation temperatures
            PartFunc = 0.d0
            Do l = 1, TotalNumberComponents_d                                               !< loop over all components
                Temperature = CopymyXCLASSParameter(2, l)                                   !< get excitation temperatures
                if (Temperature <= 0.d0) then
                    Temperature = 10.d0**TempLow_d
                endif
                logt = dlog10(Temperature)                                                  !< calculate the logarithm of the excitation temperature tk
                MoleculeIndex = CopyCompMoleculeIndex(ThreadNumberX, l)                     !< get molecule index


                !< is logt out of range?
                if (logt < TempLow_d .or. logt > TempHigh_d) then


                    !< extrapolate if neccessary ---------------------------------------- is not valid for all cases !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
                    x0 = 0.d0
                    x1 = 0.d0
                    f0 = 0.d0
                    f1 = 0.d0
                    if (logt < TempPartFunc_d(NumberOfTemperatures_d)) then
                        x0 = TempPartFunc_d((NumberOfTemperatures_d - 1))
                        x1 = TempPartFunc_d(NumberOfTemperatures_d)
                        f0 = lgQ_d((NumberOfTemperatures_d - 1), MoleculeIndex)
                        f1 = lgQ_d(NumberOfTemperatures_d, MoleculeIndex)

                    elseif (logt > TempPartFunc_d(1)) then
                        x0 = TempPartFunc_d(2)
                        x1 = TempPartFunc_d(1)
                        f0 = lgQ_d(2, MoleculeIndex)
                        f1 = lgQ_d(1, MoleculeIndex)

                    endif
                    PartFunc(ThreadNumberX, l) = f0 + (f1 - f0)/(x1 - x0) * (logt - x0)
                    PartFunc(ThreadNumberX, l) = 10.d0**PartFunc(ThreadNumberX, l)

                    ! Debug:
                    ! print*,'logt > TempPartFunc_d(1) .or. logt < TempPartFunc_d(NumberOfTemperatures_d):'
                    ! print*,'x0, f0 = ', x0, f0
                    ! print*,'x1, f1 = ', x1, f1
                    ! print*,'logt = ',logt
                    ! print*,'PartFunc(ThreadNumberX, l) = ', PartFunc(ThreadNumberX, l)
                    ! print*,'log10(PartFunc(ThreadNumberX, l)) = ', dlog10(PartFunc(ThreadNumberX, l))
                else


                    !< start linear interpolation
                    Do i = Firsti_d, Lasti_d, stepi_d
                        if (TempPartFunc_d(i) == logt) then
                            PartFunc(ThreadNumberX, l) = lgQ_d(i, MoleculeIndex)
                            exit


                        !<----------------------------------------------------------------------------------------------------------------------------
                        !< linear interpolation: f(x) = f_0 + \frac{f_1 - f_0}{x_1 - x_0} (x - x_0)
                        elseif (TempPartFunc_d(i) < logt) then
                            if (i == NumberOfTemperatures_d) then
                                x0 = TempPartFunc_d(i - 1)
                                x1 = TempPartFunc_d(i)
                                f0 = lgQ_d(i - 1, MoleculeIndex)
                                f1 = lgQ_d(i, MoleculeIndex)
                            else
                                x0 = TempPartFunc_d(i)
                                x1 = TempPartFunc_d(i + 1)
                                f0 = lgQ_d(i, MoleculeIndex)
                                f1 = lgQ_d(i + 1, MoleculeIndex)
                            endif
                            PartFunc(ThreadNumberX, l) = f0 + (f1 - f0)/(x1 - x0) * (logt - x0)

                            ! Debug:
                            ! print*,'x0, f0 = ', x0, f0
                            ! print*,'x1, f1 = ', x1, f1
                            ! print*,'logt = ',logt
                            ! print*,'PartFunc(ThreadNumberX, l) = ', PartFunc(ThreadNumberX, l)
                            ! print*,'10.d0**PartFunc(ThreadNumberX, l) = ', 10.d0**PartFunc(ThreadNumberX, l)
                            exit
                        endif
                    end Do
                    PartFunc(ThreadNumberX, l) = 10.d0**PartFunc(ThreadNumberX, l)

                    ! Debug:
                    !    print*,' '
                    !    print*,' '
                    !    print*,'l = ', l
                    !    print*,"MoleculeNames_d = ", trim(adjustl(MoleculeNames_d(MoleculeIndex)))
                    !    print*,'Temperature = ', Temperature
                    !    print*,'PartFunc(ThreadNumberX, l) = ', PartFunc(ThreadNumberX, l)
                    !    print*,'logt = ', logt
                    !    Do i = 1, NumberOfTemperatures_d
                    !        print*,'i = ', i
                    !        print*,'10.d0**lgQ_d(i, MoleculeIndex) = ', 10.d0**lgQ_d(i, MoleculeIndex), "           ", PartFunc(ThreadNumberX, l)
                    !        print*,'10.d0**TempPartFunc_d(i) = ', 10.d0**TempPartFunc_d(i), "           ",Temperature
                    !    end Do
                    !    print*,'log10(PartFunc(ThreadNumberX, l)) = ', dlog10(PartFunc(ThreadNumberX, l))
                endif
            end Do


            !<============================================================================================================================================
            !< start loop over molecules, frequency ranges, components, transitions
            chi2Value = 0.d0
            Do l = 1, TotalNumberOfFrequencyRanges_d                                        !< loop over all frequency ranges
                i = SpectrumIndexForFreqRange_d(l)                                          !< get spectrum index
                FirstIndex = DataPointIndexFreqRange_d(l, 1)                                !< get index of first freq. point in 'ObservationalDataList'
                LastIndex = DataPointIndexFreqRange_d(l, 2)                                 !< get index of last freq. point in 'ObservationalDataList'

                ! Debug:
                ! print*,"l = ", l
                ! print*,"StartFrequency_d(l) = ", StartFrequency_d(l)
                ! print*,"EndFrequency(l)   = ", EndFrequency(l)
                ! print*,"StepFrequency(l)  = ", StepFrequency(l)
                ! print*,"FirstIndex = ", FirstIndex
                ! print*,"LastIndex  = ", LastIndex


                !< update
                if (tbFlag_d) then
                    CopymyXCLASSParameter(6, :) = BackgroundTemperatureRange_d(l)
                    CopymyXCLASSParameter(7, :) = TemperatureSlopeRange_d(l)
                endif
                if (nHFlag_d) then
                    CopymyXCLASSParameter(8, :) = HydrogenColumnDensityRange_d(l)
                    CopymyXCLASSParameter(9, :) = DustBetaRange_d(l)
                    CopymyXCLASSParameter(10, :) = KappaRange_d(l) * 2.d0 * 1.66d-24 / 100.d0 !< kappa_1300 * M(H2)/M_d_M_g
                endif
                Do FreqIndex = FirstIndex, LastIndex                                        !< loop over all frequency points in the current freq. range
                    freq = ObservationalDataList_d(FreqIndex, 1)                            !< get frequency

                    ! Debug:
                    ! print*,"freq = ", freq


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< determine beam size (eq. 3):
                    !< telescope beam FWHM size is related to the diameter of the telescope by the diffraction limit:
                    !<
                    !<     \theta_t = 1.22 \frac{c}{\lambda D}
                    !<
                    !< where D describes the diameter of the telescope and c the speed of light.
                    !<
                    beam_size = 1.22d-3 * ckms_d * 180.d0 * 3600.d0 / pi_d / freq / TelescopeSize_d(l)


                    LocalIntensity = 0.d0                                                   !< reset local intensity
                    Do ae = 1, 2                                                            !< loop over emission (ae=1) and absorption (ae=2) components
                        Do c = 1, TotalNumberComponents_d                                   !< loop over all components

                            !< CopymyXCLASSParameter(1, c)      source size
                            !< CopymyXCLASSParameter(2, c)      T_rot
                            !< CopymyXCLASSParameter(3, c)      N_tot
                            !< CopymyXCLASSParameter(4, c)      V_width
                            !< CopymyXCLASSParameter(5, c)      V_off
                            !< CopymyXCLASSParameter(6, c)      T_Back
                            !< CopymyXCLASSParameter(7, c)      T_Slope
                            !< CopymyXCLASSParameter(8, c)      nH_column
                            !< CopymyXCLASSParameter(9, c)      beta
                            !< CopymyXCLASSParameter(10, c)     kappa
                            !< CopymyXCLASSParameter(11, c)     Emission/Absorption flag

                            ! Debug:
                            !    if (dabs(freq - debug_freq_d) < 1.d-6) then
                            !        print*,"####################################################################################"
                            !        print*,"ae = ", ae
                            !        print*,"c, TotalNumberComponents_d = ", c, TotalNumberComponents_d
                            !        print*,"CopymyXCLASSParameter(:, c) = ", CopymyXCLASSParameter(:, c)
                            !    endif


                            if (int(CopymyXCLASSParameter(11, c)) == ae) then    !< check, if current component correspond to ae index
                                MoleculeIndex = CopyCompMoleculeIndex(ThreadNumberX, c)     !< get molecule index


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< determine beam filling (dilution) factor (eq. 2):
                                !< \eta(\theta_{m,c}) of molecule m and component c for a source with a Gaussian brightness profile, and a Gaussian beam
                                !< is given by
                                !<
                                !<     \eta(\theta_{m,c}) = \frac{\theta_{m,c}^2}{\theta_{m,c}^2 + \theta_t^2}
                                !<
                                !< where \theta_{m,c} and \theta_t represents the source and telescope beam FWHM sizes, respectively.
                                !<
                                eta_source = CopymyXCLASSParameter(1, c)**2 &
                                             / (beam_size**2 + CopymyXCLASSParameter(1, c)**2)

                                ! Debug:
                                !    if (dabs(freq - debug_freq_d) < 1.d-6) then
                                !        print*,"MoleculeIndex = ", MoleculeIndex
                                !        print*,"eta_source = ", eta_source
                                !    endif


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< determine Rayleight-Jeans temperatures
                                freq_t = freq * 4.8d-5                                      !< convert frequency in temperature (Kelvin)
                                j_tk = freq_t / (dexp(freq_t / CopymyXCLASSParameter(2, c)) - 1.d0)
                                j_cb = freq_t / (dexp(freq_t / 2.7d0) - 1.d0)

                                ! Debug:
                                !    if (dabs(freq - debug_freq_d) < 1.d-6) then
                                !        print*,"freq_t = ", freq_t
                                !        print*,"j_tk = ", j_tk
                                !        print*,"j_cb = ", j_cb
                                !    endif


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< background temperature and slope are given for each component and molecule
                                !< the background temperature J(T_bg; \nu) is described by
                                !<
                                !<     J (T_{\rm bg}, \nu ) = | T_{\rm bg} | * (\frac{\nu}{\nu_{\rm min}} )^{T_{\rm slope}},
                                !<
                                !< and defines the continuum contribution for each frequency range, individually. Here, \nu_{\rm min} indicates the
                                !< lowest frequency of a given frequency range. T_{\rm bg} and T_{\rm slope}, describe the background continuum
                                !< temperature and the temperature slope, respectively.
                                !<
                                !< CopymyXCLASSParameter(6, c)      T_Back
                                !< CopymyXCLASSParameter(7, c)      T_Slope
                                !<
                                j_back = dabs(CopymyXCLASSParameter(6, c)) &
                                         * (freq / StartFrequency_d(l))**CopymyXCLASSParameter(7, c)

                                ! Debug:
                                !    if (dabs(freq - debug_freq_d) < 1.d-6) print*,"j_back = ", j_back


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< calculate the dust contribution (eq. 4):
                                !<
                                !<     \tau_d = N_H * \kappa_{\rm 1.3 \, mm} * ( \frac{\nu}{\nu_{\rm 1.3 \, mm}} )^\beta * m_{H_2}
                                !<                  * \frac{1}{\zeta_{\rm gas-dust}},
                                !<
                                !< where the hydrogen column density $N_H$, the dust mass opacity $\kappa_{\rm 1.3 \, mm}$, and the exponent $\beta$.
                                !< In addition, $\nu_{\rm 1.3 \, mm}$ = 230~GHz indicates the frequency for a wavelength of 1.3~mm, $m_{H_2}$ describes
                                !< the mass of a hydrogen molecule, and $\zeta_{\rm gas-dust}$ represents the gas to dust mass ratio,
                                !< i.e.\ $1 / \zeta_{\rm gas-dust}$ describes the dust to gas ratio and is set to (1/100).
                                !<
                                !< CopymyXCLASSParameter( 8, c)      nH_column
                                !< CopymyXCLASSParameter( 9, c)      beta
                                !< CopymyXCLASSParameter(10, c)     kappa   (contains factor (m_{H_2} * \frac{1}{\zeta_{\rm gas-dust}}) as well)
                                !<
                                tau_d = CopymyXCLASSParameter(10, c) * (freq / 2.3d5)**CopymyXCLASSParameter(9, c) &
                                        * CopymyXCLASSParameter(8, c)

                                ! Debug:
                                !    if (dabs(freq - debug_freq_d) < 1.d-6) then
                                !        print*,"tau_d = ", tau_d
                                !        print*,"l = ", l
                                !        print*,"MoleculeIndex = ", MoleculeIndex
                                !        print*,"MolecularDataIndices_d(l, :, 1) = ", MolecularDataIndices_d(l, :, 1)
                                !        print*,"MolecularDataIndices_d(l, :, 2) = ", MolecularDataIndices_d(l, :, 2)
                                !    endif


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< determine optical depth $\tau(\nu)^{m,c}$ for each molecule m and component c
                                !<
                                !<     \tau(\nu)^{m,c} = \sum_t \frac{c^3}{8 \pi \nu^3} \, A_{ul} \, N_{\rm tot}^{m,c}
                                !<                              \frac{g_u  e^{-E_l/k_B T_{\rm ex}^{m,c}}}{Q \left(m, T_{\rm ex}^{m,c} \right)}
                                !<                              \left(1 - e^{-h \, \nu /k_B \, T_{\rm ex}^{m,c}} \right)
                                !<                              \times \phi(\nu)^{m,c},
                                !<
                                !< where the sum with index $t$ runs over all spectral line transitions of molecule $m$ within the given frequency
                                !< range. Additionally, the Einstein $A_{ul}$ coefficient\footnote{The indices $u$ and $l$ represent upper and lower
                                !< state, respectively.}, the energy of the lower state $E_l$, the upper state degeneracy $g_u$, and the
                                !< partition function $Q \left(m, T_{\rm ex}^{m,c} \right)$ of molecule $m$ are taken from the embedded
                                !< SQLite3 database, described in  Sect.~\ref{sec:db}. In addition, the values of the excitation temperatures
                                !< $T_{\rm ex}^{m,c}$ and the column densities $N_{\rm tot}^{m,c}$ for the different components and molecules are taken
                                !< from the user defined molfit file. Furthermore, the profile of a spectral line $t$ is given by a Gaussian line profile
                                !<
                                !<     \phi(\nu)^{m,c} = \frac{1}{\sqrt{2 \pi} \, \sigma}
                                !<                       \cdot e^{-\frac{\left(\nu - \left( \nu_t + \nu_{\rm LSR}^{m,c} \right) \right)^2} {2 \sigma^2}},
                                !<
                                !< which is normalized, i.e.\ $\int_0^{\infty} \phi(\nu) \, d\nu$ = 1. The source velocity $\nu_{\rm LSR}^{m,c}$ for
                                !< each component $c$ of a molecule $m$ is related to the user defined velocity offset
                                !< $\left(v_{\rm offset}^{m,c}\right)$ taken from the aforementioned molfit file, by the following expression
                                !<
                                !<     \nu_{\rm LSR}^{m,c} = -v_{\rm offset}^{m,c} \cdot \nu_t,
                                !<
                                !< where $\nu_t$ indicates the frequency of transition $t$ taken from the SQLite3 database mentioned above. Additionally,
                                !< the variance $\sigma$ of the profile is defined by the velocity width $\left(v_{\rm width}^{m,c}\right)$ described in
                                !< the molfit file for each component $c$ of a molecule $m$:
                                !<
                                !<     \sigma = \frac{v_{\rm width}^{m,c} \cdot \left(\nu_t + \nu_{\rm LSR}^{m,c} \right)}{2 \, \sqrt{2 \, \ln 2}}.
                                !<
                                tau_l = 0.d0
                                FirstMolecularDataIndex = MolecularDataIndices_d(l, MoleculeIndex, 1) !< get first mol. data index of current freq. range
                                LastMolecularDataIndex = MolecularDataIndices_d(l, MoleculeIndex, 2)  !< get last mol. data index of current freq. range
                                Do t = FirstMolecularDataIndex, LastMolecularDataIndex - 0  !< loop over all transitions of current molecule and
                                    !< MolecularData_d(t, 1) = lFreq
                                    !< MolecularData_d(t, 2) = EinsteinA
                                    !< MolecularData_d(t, 3) = lElt
                                    !< MolecularData_d(t, 4) = UpperStateDegeneracy

                                    ! Debug:
                                    !    if (dabs(freq - debug_freq_d) < 1.d-6) then
                                    !        print*,"t = ", t
                                    !        print*,"MolecularData_d(t, :) = ", MolecularData_d(t, :)
                                    !        print*,"FirstMolecularDataIndex, LastMolecularDataIndex = ", FirstMolecularDataIndex, LastMolecularDataIndex
                                    !    endif


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< determine part of eq. (5):
                                    !<
                                    !<      \frac{c^3}{8 \pi \nu^3} \, A_{ul} \, N_{\rm tot}^{m,c}
                                    !<                              \frac{g_u  e^{-E_l/k_B T_{\rm ex}^{m,c}}}{Q \left(m, T_{\rm ex}^{m,c} \right)}
                                    !<                              \left(1 - e^{-h \, \nu /k_B \, T_{\rm ex}^{m,c}} \right)
                                    !<
                                    !< MolecularData_d(t, 2) = EinsteinA
                                    !< MolecularData_d(t, 4) = UpperStateDegeneracy
                                    !< CopymyXCLASSParameter(2, c) = T_rot
                                    !< CopymyXCLASSParameter(3, c) = N_tot
                                    !<
                                    if (PartFunc(ThreadNumberX, c) == 0.d0) then
                                        tau_t = 0.d0
                                    else
                                        tau_t = (((ckms_d * 1.d3)**2 / (8.d0 * pi_d * freq**3)) &
                                                 * MolecularData_d(t, 2) * CopymyXCLASSParameter(3, c) &
                                                 * MolecularData_d(t, 4) * 1.d-14 * ckms_d &
                                                 * (dexp(-MolecularData_d(t, 3)/CopymyXCLASSParameter(2, c)) &
                                                 - dexp(-(MolecularData_d(t, 3) &
                                                 + (freq/20836.74d0))/ CopymyXCLASSParameter(2, c)))) / PartFunc(ThreadNumberX, c)
                                    endif

                                    ! Debug:
                                    !    if (dabs(freq - debug_freq_d) < 1.d-6) then
                                    !        print*,'e_low   = ', MolecularData_d(t, 3)
                                    !        print*,'e_up    = ', (MolecularData_d(t, 3) + (freq/20836.74d0))
                                    !        print*,'>>tau_t = ', tau_t
                                    !        print*,'--> MolecularData_d(t, 2) = ', MolecularData_d(t, 2)
                                    !        print*,'--> CopymyXCLASSParameter(3, c) = ', CopymyXCLASSParameter(3, c)
                                    !        print*,'--> MolecularData_d(t, 4) = ', MolecularData_d(t, 4)
                                    !        print*,'--> MolecularData_d(t, 3) = ', MolecularData_d(t, 3)
                                    !        print*,'--> CopymyXCLASSParameter(2, c) = ', CopymyXCLASSParameter(2, c)
                                    !    endif


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< determine \nu_{\rm LSR}^{m,c}:
                                    !<
                                    !<     \nu_{\rm LSR}^{m,c} = -v_{\rm offset}^{m,c} \cdot \nu_t,
                                    !<
                                    !< CopymyXCLASSParameter(5, c) = V_off
                                    !< MolecularData_d(t, 1): lFreq
                                    !<
                                    vLSR = -CopymyXCLASSParameter(5, c) / ckms_d * MolecularData_d(t, 1)

                                    ! Debug:
                                    !    if (dabs(freq - debug_freq_d) < 1.d-6) print*,'vLSR = ', vLSR


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< determine \sigma:
                                    !<
                                    !<     \sigma = \frac{v_{\rm width}^{m,c} \cdot \left(\nu_t + \nu_{\rm LSR}^{m,c} \right)}{2 \, \sqrt{2 \, \ln 2}}.
                                    !<
                                    !< CopymyXCLASSParameter(4, c) = V_width
                                    !< MolecularData_d(t, 1): lFreq
                                    !<
                                    sigma = (CopymyXCLASSParameter(4, c) / ckms_d &
                                            * (MolecularData_d(t, 1) + vLSR)) / (2.d0 * dsqrt(2.d0 * dlog(2.d0)))

                                    ! Debug:
                                    !    if (dabs(freq - debug_freq_d) < 1.d-6) print*,'sigma = ', sigma


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< determine \phi(\nu)^{m,c}:
                                    !<
                                    !<     \phi(\nu)^{m,c} = \frac{1}{\sqrt{2 \pi} \, \sigma}
                                    !<                       * e^{-\frac{\left(\nu - \left( \nu_t + \nu_{\rm LSR}^{m,c} \right) \right)^2} {2 \sigma^2}},
                                    !<
                                    !< MolecularData_d(t, 1): lFreq
                                    !<
                                    tau_t = tau_t * 1.d0 / (dsqrt(2.d0 * pi_d) * sigma) &
                                                  * dexp(-1.d0 * ((MolecularData_d(t, 1) - freq + vLSR)**2) / (2.d0 * (sigma)**2)) * freq / ckms_d


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< add tau for current transition to sum of taus
                                    tau_l = tau_l + tau_t
                                end Do                                                      !< t: loop over all transitions

                                ! Debug:
                                !    if (dabs(freq - debug_freq_d) < 1.d-6) print*,'tau_l = ', tau_l


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< calculate intensity using eqn. (1) and (10):
                                !< The model function considers finite source size and dust attenuation and is given by:
                                !<
                                !<     T_{\rm mb}(\nu) = \sum_m \sum_c \eta \left(\theta_{m,c} \right) \left[ J \left(T_{\rm ex}^{m,c}, \nu \right)
                                !<                                                                          - J \left(T_{\rm bg}, \nu \right)
                                !<                                                                          - J \left(T_{\rm cbg}, \nu \right) \right]
                                !<                                      \times \left(1 - e^{-\left( \tau(\nu)^{m,c} + \tau_d\right)} \right)
                                !<
                                !< The sums go over the indices $m$ for molecule, and $c$ for component, respectively. Here, components for each
                                !< molecule can be identified as spatially distinct sources such as clumps, hot dense cores, colder envelopes or
                                !< outflows. They can usually be distinguished by different radial velocities, and do not interact with each other
                                !< radiatively but superimpose in the model. So, the myXCLASS program is able to model a spectrum with an arbitrary
                                !< number of molecules and components, simultaneously.
                                !<
                                !<     T_{\rm mb}(\nu) = T_{\rm mb}(\nu) - \left[ \eta \left(\theta_{m,c} \right)
                                !<                                         \cdot J \left(T_{\rm ex}^{m,c}, \nu \right)
                                !<                                         \cdot \left(1 - e^{-\tau_d} \right) \right]
                                !<
                                !< allows the user to define the continuum contribution for each frequency range, individually. Here,
                                !< $\nu_{\rm min}$ indicates the lowest frequency of a given frequency range. $T_{\rm bg}$ and $T_{\rm slope}$,
                                !< describe the background continuum temperature and the temperature slope, respectively. Please note, if the user
                                !< defines a background temperature $T_{\rm bg} \leq 0$, the dust continuum offset for each component $c$ of a
                                !< molecule $m$ is subtracted from the model function.
                                !<
                                if (CopymyXCLASSParameter(11, c) == 1.d0) then       !< determine intensity for emission
                                    LocalIntensity = LocalIntensity + eta_source * (j_tk - j_back - j_cb) * (1.d0 - dexp(-(tau_l + tau_d)))

                                    ! Debug:
                                    !    if (dabs(freq - debug_freq_d) < 1.d-6) then
                                    !        print*,'>>>>>>>>>>>>>>> LocalIntensity = ', LocalIntensity
                                    !    endif


                                    if (CopymyXCLASSParameter(6, c) <= 0.d0) then    !< substract dust continuum offset if T_Back <= 0
                                        LocalIntensity = LocalIntensity - eta_source * (j_tk - j_back - j_cb) * (1.d0 - dexp(-tau_d))
                                    endif
                                else                                                        !< absorption loop here
                                    LocalIntensity = LocalIntensity * dexp(-tau_l) + eta_source * (j_tk - j_back - j_cb) * (1.d0 - dexp(-tau_l))
                                endif
                            endif                                                           !< continue here, if current comp. is not equal to ae


                            !< add background continuum
                            if (c == TotalNumberComponents_d .and. ae == 2) then
                                LocalIntensity = LocalIntensity + j_back

                                ! Debug:
                                !    if (dabs(freq - debug_freq_d) < 1.d-6) then
                                !        print*,'--------------> LocalIntensity = ', LocalIntensity
                                !    endif
                            endif

                        end Do                                                              !< c: loop over all components
                    end Do                                                                  !< ae: loop over emission/absorption

                    ! Debug:
                    !    if (dabs(freq - debug_freq_d) < 1.d-6) stop


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< determine chi^2 value using local calculated intensity
                    chi2Value = chi2Value + (LocalIntensity - ObservationalDataList_d(FreqIndex, 2))**2


                    !< save model function
                    if (ModelFunctionFlag_d) then
                        ModelFunctionListLocal(ThreadNumberX, FreqIndex) = LocalIntensity
                    endif
                end Do                                                                      !< FreqIndex: loop over frequencies
            end Do                                                                          !< l: loop over frequencies
            chi2ValuesVector(ThreadNumberX) = chi2Value

            ! Debug:
            !    print*,"DataPointIndexFreqRange_d(:, 1) = ", DataPointIndexFreqRange_d(:, 1)
            !    print*,"DataPointIndexFreqRange_d(:, 2) = ", DataPointIndexFreqRange_d(:, 2)
            !    print*,"ObservationalDataList_d(DataPointIndexFreqRange_d(1, 1), :) = ", ObservationalDataList_d(DataPointIndexFreqRange_d(1, 1), :)
            !    print*,"ObservationalDataList_d(DataPointIndexFreqRange_d(1, 2), :) = ", ObservationalDataList_d(DataPointIndexFreqRange_d(1, 2), :)

            return
        end subroutine ModelCalcSpectrum


        !>************************************************************************************************************************************************
        !> subroutine: ModelParamFree
        !>
        !> free memory used by variables of the Module Model
        !>
        !>
        !> input variables:     deallocstatus           status of the previous deallocation process
        !>
        !> output variables:    deallocstatus           status of the deallocation process
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 26.08.2014
        !>
        subroutine ModelParamFree(deallocstatus)

            implicit none
            integer :: i
            integer :: deallocstatus                                                        !< status of the deallocation process


            if (allocated(IsoMolecule)) deallocate(IsoMolecule, stat = deallocstatus)
            if (allocated(IsoMasters)) deallocate(IsoMasters, stat = deallocstatus)
            if (allocated(IsoRatio)) deallocate(IsoRatio, stat = deallocstatus)
            if (allocated(StartFrequency)) deallocate(StartFrequency, stat = deallocstatus)
            if (allocated(EndFrequency)) deallocate(EndFrequency, stat = deallocstatus)
            if (allocated(StepFrequency)) deallocate(StepFrequency, stat = deallocstatus)
            if (allocated(TelescopeSize)) deallocate(TelescopeSize, stat = deallocstatus)
            if (allocated(BackgroundTemperatureRange)) deallocate(BackgroundTemperatureRange, stat = deallocstatus)
            if (allocated(TemperatureSlopeRange)) deallocate(TemperatureSlopeRange, stat = deallocstatus)
            if (allocated(HydrogenColumnDensityRange)) deallocate(HydrogenColumnDensityRange, stat = deallocstatus)
            if (allocated(DustBetaRange)) deallocate(DustBetaRange, stat = deallocstatus)
            if (allocated(KappaRange)) deallocate(KappaRange, stat = deallocstatus)
            if (allocated(SpectrumIndexForFreqRange)) deallocate(SpectrumIndexForFreqRange, stat = deallocstatus)
            if (allocated(ObservationalDataList)) deallocate(ObservationalDataList, stat = deallocstatus)
            if (allocated(DataPointIndexFreqRange)) deallocate(DataPointIndexFreqRange, stat = deallocstatus)
            if (allocated(MoleculeNames)) deallocate(MoleculeNames, stat = deallocstatus)
            if (allocated(ConversionTableMAGIXmyXCLASSParam)) deallocate(ConversionTableMAGIXmyXCLASSParam, stat = deallocstatus)
            if (allocated(IsoRatioConversionTable)) deallocate(IsoRatioConversionTable, stat = deallocstatus)
            if (allocated(IsoNfitConversionTable)) deallocate(IsoNfitConversionTable, stat = deallocstatus)
            if (allocated(NumberComponentsPerMolecule)) deallocate(NumberComponentsPerMolecule, stat = deallocstatus)
            if (allocated(myXCLASSParameter)) deallocate(myXCLASSParameter, stat = deallocstatus)
            if (allocated(CompMoleculeIndex)) deallocate(CompMoleculeIndex, stat = deallocstatus)
            if (allocated(MolNamePartFunc)) deallocate(MolNamePartFunc, stat = deallocstatus)
            if (allocated(lgQ)) deallocate(lgQ, stat = deallocstatus)
            if (allocated(NumEntriesRanges)) deallocate(NumEntriesRanges, stat = deallocstatus)
            if (allocated(MolecularData)) deallocate(MolecularData, stat = deallocstatus)
            if (allocated(MolecularDataIndices)) deallocate(MolecularDataIndices, stat = deallocstatus)
            if (allocated(MolNameRadTrans)) deallocate(MolNameRadTrans, stat = deallocstatus)
            if (allocated(ColumnNamesPartFunc)) deallocate(ColumnNamesPartFunc, stat = deallocstatus)
            if (allocated(TempPartFunc)) deallocate(TempPartFunc, stat = deallocstatus)
            if (allocated(ConverterInfit)) deallocate(ConverterInfit, stat = deallocstatus)


            !< free device array variables
!            i = cudaFreeArray(NumberComponentsPerMolecule_d)
!            i = cudaFreeArray(IsoNfitConversionTable_d)
!            i = cudaFreeArray(SpectrumIndexForFreqRange_d)
!            i = cudaFreeArray(DataPointIndexFreqRange_d)
!            i = cudaFreeArray(MolecularDataIndices_d)
!            i = cudaFreeArray(ConversionTableMAGIXmyXCLASSParam_d)
!            i = cudaFreeArray(CompMoleculeIndex_d)
!            i = cudaFreeArray(TempPartFunc_d)
!            i = cudaFreeArray(BackgroundTemperatureRange_d)
!            i = cudaFreeArray(TemperatureSlopeRange_d)
!            i = cudaFreeArray(HydrogenColumnDensityRange_d)
!            i = cudaFreeArray(DustBetaRange_d)
!            i = cudaFreeArray(KappaRange_d)
!            i = cudaFreeArray(TelescopeSize_d)
!            i = cudaFreeArray(StartFrequency_d)
!            i = cudaFreeArray(ObservationalDataList_d)
!            i = cudaFreeArray(MolecularData_d)
!            i = cudaFreeArray(MoleculeNames_d)
!            i = cudaFreeArray(IsoRatioConversionTable_d)
!            i = cudaFreeArray(lgQ_d)
!            i = cudaFreeArray(myXCLASSParameter_d)
!            i = cudaFreeArray(CopyCompMoleculeIndex)
!            i = cudaFreeArray(PartFunc)
!            i = cudaFreeArray(CopymyXCLASSParameter)
!            i = cudaFreeArray(CopyIsoRatioConversionTable)

            return
        end subroutine ModelParamFree
end Module Model
!*********************************************************************************************************************************************************

