!*********************************************************************************************************************************************************
!>  Module:  FunctionCalling
!>
!>
!>  This module contains some subroutines and functions which are used by all other optimization modules
!>  Copyright (C) 2009 - 2016  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine GetThreadNumberForTempDir:     returns the current thread number depending on the polarization method
!>      - subroutine GetThreadNumber:               returns the current thread number. Here, for MPI the return value is always 0
!>      - subroutine ran1:                          random number generator for numbers between 0 and 1
!>      - function RandomWithLimits:                random number generator for numbers between given limits
!>      - subroutine sort:                          starts the differnt sort algorithms
!>      - subroutine sort2:                         Starts the differnt sort2 algorithms
!>      - subroutine sort_noNR:                     sort an array using heapsort algorithm (non-numerical recepies sort routine)
!>      - subroutine sort2_noNR:                    sort an 2D array using heapsort algorithm (non-numerical recepies sort routine)
!>      - subroutine vertausche:                    used for non-numerical recepies sort routine
!>      - subroutine versickere:                    used for non-numerical recepies sort routine
!>      - subroutine ueberfuehreInHeap:             used for non-numerical recepies sort routine
!>      - subroutine versickere2:                   used for non-numerical recepies sort routine
!>      - subroutine ueberfuehreInHeap2:            used for non-numerical recepies sort routine
!>      - subroutine MultidimensionalIntegration:   determines the integral of the given function using Simpson's rule
!>      - subroutine ReadFitsFile:                  reads in a fits file (with and) without using cfitsio
!>      - subroutine CheckResamplingFlag:           determines parameters for resampling
!>      - subroutine MultidimensionalIntegration:   determines the integral of the given function using Simpson's rule
!>      - subroutine ReadFitsFile:                  reads in a fits file (with and) without using cfitsio
!>      - subroutine CheckResamplingFlag:           determines parameters for resampling
!>      - subroutine RegistrationMask:              open registration file and construct input and output mask
!>      - subroutine DetermineParameterValue:       determines the value of the parameter indicated by the name of the parameter
!>      - subroutine IndexFormat:                   determines the format(string) of a parameter of number listindex in the parameter list
!>      - subroutine WriteParameter:                write input file for given model
!>      - subroutine WriteDataForFitFunction:       write all experimental data points to file (only x-column values)
!>      - subroutine ReadDataOfFitFunction:         reads values of the fit function
!>      - subroutine CallFitFunctionAtOnce:         call the external model function defined in the CommandLine variable
!>      - subroutine FitFunctionAtOnce:             determines the value of the external model function and its gradient (method: at once)
!>      - subroutine parameterlimitcheck:           check if parameters are within defined range
!>      - subroutine CheckCalculatedParameterSets:  reads previously calculated parameter sets
!>      - subroutine ChiFunction:                   calculates chi^2 using different methods saved in DetChi2
!>      - subroutine CompareTwoParameterSets:       compares to different paramter sets using the format specifications defined in registration file
!>      - subroutine SortChi2File:                  reads in all calculated parameter vectors with the corresponding chi^2 values and sort theses
!>                                                  parameter vectors in ascending numerical order of chi^2
!>      - subroutine PlotFitFunction:               plots the current model function for each iteration step using gplot.sh stored in "Modules/gnuplot/"
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    30/11/2009  Initial version
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!---------------------------------------------------------------------------------------------------------------------------------------------------------
Module FunctionCalling
    !> declare variables which are used by functions and subroutines of this module
    use Variables

    implicit none
    integer :: NumberOfUsedThreads                                                          !< number of used threads with this module
    integer :: NumberLinesOutput                                                            !< number of lines of the position of the fit value
                                                                                            !<      within the output file
    integer :: NumberColumnsBegin                                                           !< number of (begin) columns of the position of the fit value
                                                                                            !<      within the output file
    integer :: NumberColumnsEnd                                                             !< number of (end) columns of the position of the fit value
                                                                                            !<      within the output file
    integer :: DetChi2                                                                      !< method being used for the determination of chi^2
    integer :: idum                                                                         !< used for random number generator
    integer :: GradientMethod                                                               !< selects the method how the gradient is determined
    integer, allocatable, dimension(:, :) :: NumberParamPerLine                             !< number of parameter for every line in the input file

    real*8 :: GradientVariationValue                                                        !< value of the variation (in procent) used for gradient
    real*8, allocatable, dimension(:, :, :) :: xPointsModel_output                          !< x-column values of the output file(s)
                                                                                            !< (used only for OnlyYColumn == .false.)
    real*8, allocatable, dimension(:, :) :: valuesModel_output                              !< model function values of output file(s)
                                                                                            !< (used only for OnlyYColumn == .false.)
    character(len=100) :: PlotPID                                                           !< pid for plot program
    character(len=20) :: CalculationMethod                                                  !< method of computation
    character(len=8192) :: PathStartScript                                                  !< path of start script
    character(len=8192) :: ExeCommandStartScript                                            !< execution command for start script
    character(len=8192) :: CommandWordOutput                                                !< command word indicating the position of the fit value
                                                                                            !<      within the output file
    character(len=8192) :: InputDataPath                                                    !< path and file name of the data file for input
    character(len=8192), allocatable, dimension(:, :, :) :: ParameterName                   !< name of model parameters
    character(len=10), allocatable, dimension(:, :, :) :: ParameterFormat                   !< format of the model parameter values
    character(len=8192), allocatable, dimension(:, :, :) :: LeadingString                   !< leading string of each model parameter
    character(len=8192), allocatable, dimension(:, :, :) :: TrailingString                  !< trailing string of each model parameter
    character(len=8192), allocatable, dimension(:, :, :) :: ParamVisible                    !< visible flag of each model parameter
    character(len=8192), allocatable, dimension(:) :: FitFktInput                           !< string containing the path and the file name of the input
                                                                                            !<   file for the fit function module
    character(len=8192), allocatable, dimension(:) :: FitFktOutput                          !< string containing the path and the file name of the ouput
                                                                                            !<   file for the fit function module
    logical :: debugflag                                                                    !< flag for debugging call of external model program

    save :: xPointsModel_output                                                             !< used within an OpenMP environment (thread private)
    save :: valuesModel_output                                                              !< used within an OpenMP environment (thread private)
    !$omp THREADPRIVATE(xPointsModel_output, valuesModel_output)

    contains


            !*********************************************************************************************************************************************
            !> subroutine: GetThreadNumberForTempDir
            !>
            !> returns the current thread number depending on the polarization method (two different subroutines are neccessary to avoid to Modules)
            !>
            !>
            !> input variables:     None
            !>
            !> output variables:    ThreadNumber:   current thread number
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 08.09.2014
            !>
            subroutine GetThreadNumberForTempDir(ThreadNumber)

                use omp_lib

                implicit none
                integer :: ThreadNumber


                ThreadNumber = 0                                                            !< initialize return variable
                if (ParallezitionMethod == 1) then                                          !< OpenMP
                    if (NumberOfUsedThreads > 1) then
                        ThreadNumber = omp_get_thread_num()                                 !< call OpenMP function to determine thread number
                    endif
                elseif (ParallezitionMethod == 2) then                                      !< MPI
                    ThreadNumber = myrankOfMPI
                endif


                return
            end subroutine GetThreadNumberForTempDir


            !*********************************************************************************************************************************************
            !> subroutine: GetThreadNumber
            !>
            !> returns the current thread number. Here, for MPI the return value is always 0
            !>
            !>
            !> input variables:     None
            !>
            !> output variables:    ThreadNumber:   current thread number
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 08.09.2014
            !>
            subroutine GetThreadNumber(ThreadNumber)

                use omp_lib

                implicit none
                integer :: ThreadNumber


                ThreadNumber = 0                                                            !< initialize return variable
                if (ParallezitionMethod == 1) then                                          !< OpenMP
                    if (NumberOfUsedThreads > 1) then
                        ThreadNumber = omp_get_thread_num()                                 !< call OpenMP function to determine thread number
                    endif
                elseif (ParallezitionMethod == 2) then                                      !< MPI
                    ThreadNumber = 0
                endif


                return
            end subroutine GetThreadNumber


            !*********************************************************************************************************************************************
            !> subroutine: ran1
            !>
            !> random number generator: generates random number between 0 and 1
            !>
            !>
            !> input variables:     idum:           number indicating (< 0) the initialization process or normal mode
            !>
            !> output variables:    value:          contains the random number
            !>
            !>
            !> \author Numercial Recipies Fortran 90 (chapter B7, p. 1142) and Thomas Moeller
            !>
            !> \date 26.05.2010
            !>
            subroutine ran1(value, dummy)
                !< "Minimal" random number generator of Park and Miller with Bays-Durham shuffle and
                !< added safeguards. Returns a uniform random deviate between 0.0 and 1.0 (exclusive of
                !< the endpoint values). Call with idum a negative integer to initialize; thereafter, do not
                !< alter idum between successive deviates in a sequence. RNMX should approximate the largest
                !< floating value that is less than 1.

                implicit none
                integer :: dummy
                integer, parameter :: IA1 = 16807                                           !< parameter for numerical recepies version
                integer, parameter :: IM = 2147483647                                       !< parameter for numerical recepies version
                integer, parameter :: IQ = 127773                                           !< parameter for numerical recepies version
                integer, parameter :: IR = 2836                                             !< parameter for numerical recepies version
                integer, parameter :: NTAB = 32                                             !< parameter for numerical recepies version
                integer, parameter :: NDIV = 1 + (IM - 1)/NTAB                              !< parameter for numerical recepies version
                integer, parameter :: version_switch = 3                                    !< =1: numerical recepies, =2: use FORTRAN intrinsic function

                real*8 :: value                                                             !< output variable
                real*8, parameter :: AM = 1.d0 / IM                                         !< parameter for numerical recepies version
                real*8, parameter :: EPS = 1.2d-7                                           !< parameter for numerical recepies version
                real*8, parameter :: RNMX = 1.d0 - EPS                                      !< parameter for numerical recepies version

                !< working variables
                integer :: j, k, iy
                integer, dimension(NTAB) :: iv
                SAVE iv, iy
                DATA iv /NTAB*0/, iy /0/


                !-----------------------------------------------------------------------------------------------------------------------------------------
                !< numerical recepies version
                if (version_switch == 1) then
                    if (idum <= 0 .or. iy == 0) then                                        !< Initialize.
                        idum = max(-idum, 1)                                                !< Be sure to prevent idum = 0.
                        Do j = (NTAB + 8), 1, (-1)                                          !< Load the shuffle table (after 8 warm-ups).
                            k = idum / IQ
                            idum = IA1 * (idum - k * IQ) - IR * k
                            if (idum < 0) idum = idum + IM
                            if (j <= NTAB) iv(j) = idum
                        end Do
                        iy = iv(1)
                    endif
                    k = idum / IQ                                                           !< Start here when not initializing.
                    idum = IA1 * (idum - k * IQ) - IR * k                                   !< Compute idum = mod(IA1*idum,IM) without over ows by
                    if (idum < 0) idum = idum + IM                                          !< Schrage's method.
                    j = 1 + iy / NDIV                                                       !< Will be in the range 1:NTAB.
                    iy = iv(j)                                                              !< Output previously stored value and re ll the shu e table.
                    iv(j) = idum
                    value = min(AM * iy, RNMX)                                              !< Because users don't expect endpoint values.


                !-----------------------------------------------------------------------------------------------------------------------------------------
                !< use the RANDOM_NUMBER command included in FORTRAN 95
                elseif (version_switch == 2) then
                    call RANDOM_NUMBER(value)


                !-----------------------------------------------------------------------------------------------------------------------------------------
                !< use a corrected version of numerical recepies algorithm
                elseif (version_switch == 3) then
                    call ran1_corrected(value, dummy)
                endif
                return
            end subroutine ran1


            !*********************************************************************************************************************************************
            !> subroutine: ran1_corrected
            !>
            !> random number generator: generates random number between 0 and 1
            !>
            !>
            !> input variables:     idum:           number indicating (< 0) the initialization process or normal mode
            !>
            !> output variables:    value:          contains the random number
            !>
            !>
            !> \author Timothy H. Kaiser, http://www.sdsc.edu/~tkaiser/f90.html, call 14.11.2011
            !>
            !> \date 14.11.2011
            !>
            subroutine ran1_corrected(value, idum)

                implicit none
                real*8 :: value
                integer :: idum
                real*8 :: rm1,rm2
                real*8, dimension(97) :: r
                integer, parameter :: m1 = 259200, ia1 = 7141, ic1 = 54773
                integer, parameter :: m2 = 134456, ia2 = 8121, ic2 = 28411
                integer, parameter :: m3 = 243000, ia3 = 4561, ic3 = 51349
                integer :: j
                integer :: iff, ix1, ix2, ix3
                data iff /0/
                save                                                                        !< corrects a bug in the original routine
                if (idum < 0 .or. iff == 0)then
                    rm1 = 1.d0/m1
                    rm2 = 1.d0/m2
                    iff = 1
                    ix1 = mod(ic1 - idum, m1)
                    ix1 = mod(ia1 * ix1 + ic1, m1)
                    ix2 = mod(ix1, m2)
                    ix1 = mod(ia1 * ix1 + ic1, m1)
                    ix3 = mod(ix1, m3)
                    Do j = 1,97
                        ix1 = mod(ia1 * ix1 + ic1, m1)
                        ix2 = mod(ia2 * ix2 + ic2, m2)
                        r(j) = (real(ix1) + real(ix2) * rm2) * rm1
                    end Do
                    idum = 1
                endif
                ix1 = mod(ia1 * ix1 + ic1, m1)
                ix2 = mod(ia2 * ix2 + ic2, m2)
                ix3 = mod(ia3 * ix3 + ic3, m3)
                j = 1 + (97 * ix3)/m3
                if(j > 97 .or. j < 1)then
                    write(logchannel,*)
                    write(logchannel,'(11x,"Error in subroutine ran1_corrected:")')
                    write(logchannel,'(13x,"An error occurred!")')
                    write(logchannel,*)
                    write(logchannel,'(11x,"j = ", I5)') j
                    write(logchannel,*)
                    write(logchannel,'(11x,"Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '(11x,"Error in subroutine ran1_corrected:")'
                    print '(13x,"An error occurred!")'
                    print '(" ")'
                    print '(13x,"j = ", I5)', j
                    print '(" ")'
                    stop '           Program aborted!'
                endif
                value = r(j)
                r(j) = (real(ix1) + real(ix2) * rm2) * rm1
                return
            end subroutine ran1_corrected


            !*********************************************************************************************************************************************
            !> function: RandomWithLimits (determine a random number between lowLimit and UpperLimit
            !>
            !> generates a random number between lower limit and upper limit using subroutine ran1
            !>
            !>
            !> input variables:     LowLimit:           lower limit
            !>                      UpperLimit:         upper limit
            !>
            !> output variables:    RandomWithLimits:   random number
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 09.06.2010
            !>
            function RandomWithLimits(LowLimit, UpperLimit)

                implicit none
                real*8 :: LowLimit, UpperLimit                                              !< lower and upper limit of the random number interval
                real*8 :: RandomWithLimits                                                  !< output value
                real*8 :: value                                                             !< random number generated by ran1

                !if (abs(LowLimit) > HUGE(0.d0)) then
                !    LowLimit = sign(HUGE(0.d0), LowLimit)
                !endif
                !if (abs(LowLimit) < TINY(0.d0)) then
                !    LowLimit = 0.d0     !sign(TINY(0.d0), LowLimit)
                !endif
                !if (abs(UpperLimit) > HUGE(0.d0)) then
                !    UpperLimit = sign(HUGE(0.d0), UpperLimit)
                !endif
                !if (abs(UpperLimit) < TINY(0.d0)) then
                !    UpperLimit = 0.d0   !sign(TINY(0.d0), UpperLimit)
                !endif

                if (abs(LowLimit - UpperLimit) < TINY(0.d0)) then
                    RandomWithLimits = LowLimit
                else
                    call ran1(value, 0)                                                     !< call ran1 to generate random number between 0 and 1
                    if (LowLimit > UpperLimit) then
                        RandomWithLimits = UpperLimit + (LowLimit - UpperLimit) * value
                    else
                        RandomWithLimits = LowLimit + (UpperLimit - LowLimit) * value
                    endif
                endif
                return
            end function RandomWithLimits


            !*********************************************************************************************************************************************
            !> subroutine: sort
            !>
            !> input variables:     n:                  length of array ra
            !>                      ra:                 array, which should be sort
            !>
            !> output variables:    ra:                 sorted array
            !>
            !> \author Thomas Moeller
            !>
            !> \date 05.08.2014
            !>
            subroutine sort(n, ra)
                !< Starts the differnt sort algorithms

                implicit none
                integer :: n                                                                !< length of list ra
                real*8, dimension(n) :: ra                                                  !< input/output variable



                ! ############################################################ Begin: Non-NR version only ################################################
                call sort_noNR(n, ra)
                return
                ! ############################################################ End: Non-NR version only   ################################################
            end subroutine sort




            !*********************************************************************************************************************************************
            !> subroutine: sort_noNR (No Numerical Recepies version)
            !>
            !> sort an array using heapsort algorithm
            !>
            !>
            !> input variables:     a:                  array, which should be sort
            !>                      size_bn:            length of array a
            !>
            !> output variables:    a:                  modified array a
            !>
            !>
            !> \author http://de.wikibooks.org/wiki/Algorithmensammlung:_Sortierverfahren:_Heapsort
            !>
            !> \date 09.03.2012
            !>
            subroutine sort_noNR(size_bn, a)

                implicit none
                integer :: size_bn
                integer :: i                                                                !< loop variable
                real*8, dimension(size_bn) :: a

                call ueberfuehreInHeap(a, size_bn)
                Do i = size_bn, 1, (-1)
                    call vertausche(a, size_bn, 1, i)
                    call versickere(a, i - 1, 1)
                end Do
                return
            end subroutine sort_noNR


            !*********************************************************************************************************************************************
            !> subroutine: sort2
            !>
            !> input variables:     n:                  length of array ra
            !>                      arr:                1st array, which should be sort
            !>                      brr:                2nd array, which should be sort
            !>
            !> output variables:    arr:                1st array, which should be sort
            !>                      brr:                2nd array, which should be sort
            !>
            !> \author Thomas Moellerstarter fors
            !>
            !> \date 05.08.2014
            !>
            subroutine sort2(n, arr, brr)
                !< Starts the differnt sort algorithms

                implicit none
                integer :: n                                                                !< length of list ra
                real*8, dimension(n) :: arr, brr                                            !< input/output variables



                ! ############################################################ Begin: Non-NR version only ################################################
                call sort2_noNR(n, arr, brr)
                return
                ! ############################################################ End: Non-NR version only   ################################################
            end subroutine sort2




            !*********************************************************************************************************************************************
            !> subroutine: sort2_noNR (No Numerical Recepies version)
            !>
            !> sort an array using heapsort algorithm
            !>
            !>
            !> input variables:     a:                  1st array, which should be sort
            !>                      b:                  2nd array, which should be sort
            !>                      size_bn:            length of array a
            !>
            !> output variables:    a:                  1st modified array a
            !>                      b:                  2nd modified array b
            !>
            !>
            !> \author http://de.wikibooks.org/wiki/Algorithmensammlung:_Sortierverfahren:_Heapsort and Thomas Moeller
            !>
            !> \date 09.03.2012
            !>
            subroutine sort2_noNR(size_bn, a, b)

                implicit none
                integer :: size_bn
                integer :: i                                                                !< loop variable
                real*8, dimension(size_bn) :: a, b

                call ueberfuehreInHeap2(a, b, size_bn)
                Do i = size_bn, 1, (-1)
                    call vertausche(a, size_bn, 1, i)
                    call vertausche(b, size_bn, 1, i)
                    call versickere2(a, b, i - 1, 1)
                end Do
                return
            end subroutine sort2_noNR


            !*********************************************************************************************************************************************
            !> subroutine: vertausche (No Numerical Recepies version)
            !>
            !> swap array entries
            !>
            !>
            !> input variables:     a:                  array, which should be sort
            !>                      size_bn:            length of array a
            !>                      x:                  index x
            !>                      y:                  index y
            !>
            !> output variables:    a:                  sorted array
            !>
            !>
            !> \author http://de.wikibooks.org/wiki/Algorithmensammlung:_Sortierverfahren:_Heapsort
            !>
            !> \date 09.03.2012
            !>
            subroutine vertausche(a, size_bn, x, y)

                implicit none
                integer :: x, y, size_bn
                real*8 :: z
                real*8, dimension(size_bn) :: a

                z = a(x)
                a(x) = a(y)
                a(y) = z
                return
            end subroutine vertausche


            !*********************************************************************************************************************************************
            !> subroutine: versickere (No Numerical Recepies version)
            !>
            !> sink an item
            !>
            !>
            !> input variables:     a:                  array, which should be sort
            !>                      size_bn:            length of array a
            !>                      i:                  index of element which is sunk
            !>
            !> output variables:    a:                  modified array a
            !>
            !>
            !> \author http://de.wikibooks.org/wiki/Algorithmensammlung:_Sortierverfahren:_Heapsort
            !>
            !> \date 09.03.2012
            !>
            subroutine versickere(a, size_bn, i)

                implicit none
                integer :: i, size_bn, ln, j
                real*8, dimension(size_bn) :: a

                j = i
                Do While (j <= (size_bn/2))
                    ln = 2 * j
                    if (ln < size_bn) then
                        if (a(ln + 1) > a(ln)) ln = ln + 1
                    end if
                    if (a(ln) > a(j)) then
                        call vertausche(a, size_bn, j, ln)
                        j = ln
                    else
                        j = size_bn
                    endif
                end Do
                return
            end subroutine versickere


            !*********************************************************************************************************************************************
            !> subroutine: ueberfuehreInHeap (No Numerical Recepies version)
            !>
            !> generate a maxheap tree
            !>
            !>
            !> input variables:     a:                  array, which should be sort
            !>                      size_bn:            length of array a
            !>
            !> output variables:    a:                  modified array a
            !>
            !>
            !> \author http://de.wikibooks.org/wiki/Algorithmensammlung:_Sortierverfahren:_Heapsort
            !>
            !> \date 09.03.2012
            !>
            subroutine ueberfuehreInHeap(a, size_bn)

                implicit none
                integer :: size_bn, i
                real*8, dimension(size_bn) :: a

                Do i = size_bn / 2 , 1 , (-1)
                    call versickere(a, size_bn, i)
                end Do
                return
            end subroutine ueberfuehreInHeap


            !*********************************************************************************************************************************************
            !> subroutine: versickere2 (No Numerical Recepies version)
            !>
            !> sink an item
            !>
            !>
            !> input variables:     a:                  1st array, which should be sort
            !>                      b:                  2nd array, which should be sort
            !>                      size_bn:            length of array a
            !>                      i:                  index of element which is sunk
            !>
            !> output variables:    a:                  1st modified array a
            !>                      b:                  2nd modified array b
            !>
            !>
            !> \author http://de.wikibooks.org/wiki/Algorithmensammlung:_Sortierverfahren:_Heapsort and Thomas Moeller
            !>
            !> \date 09.03.2012
            !>
            subroutine versickere2(a, b, size_bn, i)

                implicit none
                integer :: i, size_bn, ln, j
                real*8, dimension(size_bn) :: a, b

                j = i
                Do While (j <= (size_bn/2))
                    ln = 2 * j
                    if (ln < size_bn) then
                        if (a(ln + 1) > a(ln)) ln = ln + 1
                    end if
                    if (a(ln) > a(j)) then
                        call vertausche(a, size_bn, j, ln)
                        call vertausche(b, size_bn, j, ln)
                        j = ln
                    else
                        j = size_bn
                    endif
                end Do
                return
            end subroutine versickere2


            !*********************************************************************************************************************************************
            !> subroutine: ueberfuehreInHeap2 (No Numerical Recepies version)
            !>
            !> generate a maxheap tree
            !>
            !>
            !> input variables:     a:                  1st array, which should be sort
            !>                      b:                  2nd array, which should be sort
            !>                      size_bn:            length of array a
            !>
            !> output variables:    a:                  1st modified array a
            !>                      b:                  2nd modified array b
            !>
            !>
            !> \author http://de.wikibooks.org/wiki/Algorithmensammlung:_Sortierverfahren:_Heapsort and Thomas Moeller
            !>
            !> \date 09.03.2012
            !>
            subroutine ueberfuehreInHeap2(a, b, size_bn)

                implicit none
                integer :: size_bn, i
                real*8, dimension(size_bn) :: a, b

                Do i = size_bn / 2 , 1 , (-1)
                    call versickere2(a, b, size_bn, i)
                end Do
                return
            end subroutine ueberfuehreInHeap2


            !*********************************************************************************************************************************************
            !> subroutine: MultidimensionalIntegration
            !>
            !> input variables:     NumberPoints:       number of points
            !>                      x_array:            array containing the x-positions
            !>                      y_array:            array containing the values of the functions which has to be integrated
            !>
            !> output variables:    value:              value of the integral
            !>
            !> \author Numercial Recipies, modifications by Thomas Moeller
            !>
            !> \date 11.01.2011
            !>
            subroutine MultidimensionalIntegration(value, MaxColumnX, NumberPoints, x_array, y_array)
                !< determines the integral of the given function using Simpson's rule

                implicit none
                integer :: i, j, k                                                          !< loop variables
                integer :: MaxColumnX                                                       !< max number of x-columns
                integer :: NumberPoints                                                     !< number of points
                real*8 :: h, ymax, ymin                                                     !< working variables
                real*8, dimension(NumberPoints, MaxColumnX) :: x_array                      !< array containing the x-positions
                real*8, dimension(NumberPoints) :: y_array                                  !< array containing the values of the functions which has
                                                                                            !< to be integrated
                real*8 :: value                                                             !< value of the integral

                value = 0.d0
                Do j = 1, MaxColumnX
                    k = 1
                    Do i = 2, NumberPoints
                        k = k + 1
                        h = (x_array(i, j) - x_array((i - 1), j))
                        ymax = max(y_array(k), y_array(k - 1))
                        ymin = min(y_array(k), y_array(k - 1))
                        value = value + (h * ymin + 0.5d0 * (ymax - ymin) * h)
                    end Do
                end Do
                value = dabs(value)
                return
            end subroutine MultidimensionalIntegration


            !*********************************************************************************************************************************************
            !> subroutine: readfitsFile
            !>
            !> input variables:     NumOut:             number of current output file
            !>                      filename:           name of the current fits file
            !>
            !> output variables:    FinalCounter:       number of all lines
            !>
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 08.02.2011
            !>
            subroutine ReadFitsFile(MaxX, NumOut, filename, FinalCounter, resampling_flag, FirstPointOutputFile, LastPointOutputFile, &
                                                                                    FirstPointValueOutputFile, LastPointValueOutputFile)
                !< this subroutine reads in a fits file with or without using cfitsio

                implicit none
                integer :: MaxX                                                             !< max. number of x-columns
                integer :: NumOut                                                           !< number of current output file
                integer :: unitNum                                                          !< unit number
                integer :: FinalCounter                                                     !< number of all lines
                integer :: ThreadNumber                                                     !< thread number
                integer :: i, k, x, y, z, t                                                 !< loop variables
                integer :: io_stat                                                          !< status number
                ! integer :: blocksize                                                        !<
                integer :: readwriteMode                                                    !< read write mode (used for ftopen)
                integer :: counter, group, firstpix, nbuffer, npixels                       !< working variables
                integer :: ndim                                                             !< dimension of the current fits file
                integer :: allocstatus, deallocstatus                                       !< variables for (de)allocation
                integer, dimension(4) :: crpix, crpixCopy                                   !< values of crpix for each axis
                integer, dimension(4) :: length, lengthCopy                                 !< length of the image in each dimension
                real*8 :: zero                                                              !< working variable
                ! real*8 :: bzero                                                             !< value of bzero
                ! real*8 :: bscale                                                            !< value of bscale
                real*8 :: FirstPointValueOutputFile, LastPointValueOutputFile               !< value of first point and last point
                real*8, dimension(4) :: ValueOfFirstPoint                                   !< value of the first point of each axis
                real*8, dimension(0:4) :: crval, crvalCopy                                  !< values of crval for each axis
                real*8, dimension(0:4) :: cdelt, cdeltCopy                                  !< values of cdelt for each axis
                real*8, dimension(100) :: buffer                                            !< buffer array for reading
                real*8, dimension(MaxX) :: xArray                                           !< help array
                real*8, dimension(MaxX) :: FirstPointOutputFile                             !< first point (lowest value of x-column vector)
                real*8, dimension(MaxX) :: LastPointOutputFile                              !< last point (highest value of x-column vector)
                character(len=1) :: NumString                                               !< working variables
                character(len=30) :: NumberString                                           !< working variables
                ! character(len=80) :: dummy                                                  !< working variable
                character(len=512) :: filename                                              !< name of the fits file
                character(len=512) :: LongHelpString1                                       !< help string
                character(len=8), dimension(0:4) :: units(0:4)                              !< string for units
                character(len=40), dimension(0:4) :: comments                               !< string for comments
                logical :: anyf                                                             !<
                logical :: resampling_flag                                                  !< resampling flag


                !< get threadnumber
                ThreadNumber = 0                                                            !< default value for thread number
                call GetThreadNumber(ThreadNumber)


                !< Get an unused Logical Unit Number to use to open the FITS file.
                io_stat = 0
                !call ftgiou(unitNum, io_stat)

                ! Debug:
                ! print*,' '
                print*,'UnitNumber = ',unitNum
                print*,'Filename = ',trim(adjustl(filename))


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< open fits file
                readwriteMode = 0
                io_stat = 0
                !call ftopen(unitNum, trim(adjustl(filename)), readwriteMode, blocksize, io_stat)
                if (io_stat == 0) then


                    !< try moving to the next extension in the FITS file, if it exists.
                    !call ftmrhd(unitNum, 1, hdutype, io_stat)


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< read required header entries
                    ndim = 0
                    !call ftgkyj(unitNum, 'NAXIS ', ndim, comments(0), io_stat)              !< get number of axis
                    if (io_stat /= 0) then                                                  !< exception handling
                        ndim = 1
                        io_stat = 0
                    endif
                    !call ftgkys(unitNum, 'BUNIT ', units(0), comments(0), io_stat)
                    if (io_stat /= 0) then                                                  !< exception handling
                        units(0) = "        "                                               !< Heidelberg cubes often have no data scaling
                        comments(0) = "                                       "
                        io_stat = 0
                    endif
                    !call ftgkyd(unitNum, 'BZERO ', bzero, dummy, io_stat)
                    if (io_stat /= 0) then                                                  !< exception handling
                        crval(0) = 0.d0
                        io_stat = 0
                    endif
                    !call ftgkyd(unitNum, 'BSCALE', bscale, dummy, io_stat)
                    if (io_stat /= 0) then                                                  !< exception handling
                        cdelt(0) = 1.d0
                        io_stat = 0
                    endif

                    ! Debug:
                    ! print*,'NAXIS  = ', ndim
                    ! print*,'BUNIT  = ', units(0)
                    ! print*,'BZERO  = ', bzero
                    ! print*,'BSCALE = ', bscale


                    !< read further header parts
                    crpix = 0
                    crval = 0.d0
                    cdelt = 0.d0
                    length = 0
                    Do i = 1, ndim                                                          !< loop over all axis
                        write(NumString,'(I1)') i
                        !call ftgkyj(unitNum, 'NAXIS' // NumString, length(i), comments(i), io_stat)
                        !call ftgkys(unitNum, 'CTYPE' // NumString, units(i), dummy, io_stat)
                        !call ftgkyj(unitNum, 'CRPIX' // NumString, crpix(i), comments(i), io_stat)
                        !call ftgkyd(unitNum, 'CRVAL' // NumString, crval(i), dummy, io_stat)
                        !call ftgkyd(unitNum, 'CDELT' // NumString, cdelt(i), dummy, io_stat)
                        if (io_stat /= 0) then                                              !< exception handling
                            units(i) = "        "
                            comments(i) = "                                       "
                            crval(i) = 0.d0
                            cdelt(i) = 1.d0
                            io_stat = 0
                        endif

                        ! Debug:
                        ! print*,'>',i
                        ! print*,'NAXIS' // NumString,' = ', length(i)
                        ! print*,'CTYPE' // NumString,' = ', units(i)
                        ! print*,'CRPIX' // NumString,' = ', crpix(i)
                        ! print*,'CRVAL' // NumString,' = ', crval(i)
                        ! print*,'CDELT' // NumString,' = ', cdelt(i)
                    end Do


                    !< remove the axes with length == 1
                    lengthCopy(:) = length(:)
                    crpixCopy(:) = crpix(:)
                    crvalCopy(:) = crval(:)
                    cdeltCopy(:) = cdelt(:)
                    length = 0
                    crpix = 0
                    crval = 0.d0
                    cdelt = 0.d0
                    x = 0
                    npixels = 1                                                             !< total number of pixel of current image
                    Do i = 1, ndim
                        if (lengthCopy(i) > 1) then                                         !< only those dimensions are relevant with length > 1
                            x = x + 1
                            length(x) = lengthCopy(i)
                            crpix(x) = crpixCopy(i)
                            crval(x) = crvalCopy(i)
                            cdelt(x) = cdeltCopy(i)
                            npixels = npixels * length(x)                                   !< calculate total number of pixels
                        endif
                    end Do
                    ndim = x                                                                !< set ndim to new value

                    ! Debug:
                    ! print*,'ndim = ',ndim
                    ! print*,'npixels = ',npixels


                    !< determine lsr adjustement value if desired
                    if (LSRAdjustementFitFlag(NumOut)) then                                 !< if LSRAdjustementFitFlag is true get lsr adjustement
                        write(NumberString,'(I30)') NumOut                                  !< from parameter list
                        LongHelpString1 = "MAGIXOutputLSRAdjustementName" // trim(adjustl(NumberString))
                        Do k = 1, parameternumber
                            if (trim(adjustl(FitParameterName(k))) == trim(adjustl(LongHelpString1))) then
                                LSRAdjustement(NumOut) = paramset(0, k)
                                exit
                            endif
                        end Do

                        ! Debug:
                        ! print*,'NumOut = ', NumOut
                        ! print*,'LSRAdjustement(NumOut) = ', LSRAdjustement(NumOut)
                        ! stop
                    endif


                    !< (deallocate and) allocate memory for output variables
                    if (allocated(xPointsModel_output)) then
                        deallocate(xPointsModel_output, valuesModel_output, stat = deallocstatus)
                        if (deallocstatus /= 0) then
                            write(logchannel,*)
                            write(logchannel,'(11x,"Error in subroutine ReadFitsFile:")')
                            write(logchannel,'(13x,"Can not deallocate variables xPointsModel_output etc.")')
                            write(logchannel,'(13x,"Please close all other programs and restart the program!")')
                            write(logchannel,*)
                            write(logchannel,'(13x,"deallocstatus = ",I4)') deallocstatus
                            write(logchannel,'(" ")')
                            write(logchannel,'(11x,"Program aborted!")')
                            close(logchannel)

                            print '(" ")'
                            print '(11x,"Error in subroutine ReadFitsFile:")'
                            print '(13x,"Can not deallocate variables xPointsModel_output etc.")'
                            print '(13x,"Please close all other programs and restart the program!")'
                            print '(" ")'
                            print '(13x,"deallocstatus = ",I4)',deallocstatus
                            print '(" ")'
                            stop '           Program aborted!'
                        endif
                    endif
                    allocate(xPointsModel_output(0:NumberOfUsedThreads - 1, npixels, MaxX), valuesModel_output(0:NumberOfUsedThreads - 1, npixels) &
                                                                                                                                , stat = allocstatus)
                    if (allocstatus /= 0) then
                        write(logchannel,'(" ")')
                        write(logchannel,'(11x,"Error in subroutine ReadFitsFile:")')
                        write(logchannel,'(13x,"Can not allocate variables xPointsModel_output etc.")')
                        write(logchannel,'(13x,"Please close all other programs and restart the program!")')
                        write(logchannel,'(" ")')
                        write(logchannel,'(13x,"allocstatus = ",I4)') allocstatus
                        write(logchannel,'(" ")')
                        write(logchannel,'(11x,"Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '(11x,"Error in subroutine ReadFitsFile:")'
                        print '(13x,"Can not allocate variables xPointsModel_output etc.")'
                        print '(13x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '(13x,"allocstatus = ",I4)',allocstatus
                        print '(" ")'
                        stop '           Program aborted!'
                    endif
                    xPointsModel_output(ThreadNumber, : ,:) = 0.d0
                    valuesModel_output(ThreadNumber, :) = 0.d0


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< read final image
                    !< Initialize variables
                    group = 1
                    firstpix = 1
                    zero = -999.d0
                    anyf = .false.
                    counter = 0
                    buffer = 0.d0
                    valuesModel_output = 0.d0
                    Do while (npixels > 0)


                        !< read up to 100 pixels at a time
                        nbuffer = min(100, npixels)
                        !call ftgpvd(unitNum, group, firstpix, nbuffer, zero, buffer, anyf, io_stat)


                        !< add to valuesModel_output array
                        Do i = 1, nbuffer
                            counter = counter + 1
                            valuesModel_output(ThreadNumber, counter) = buffer(i)
                        end Do


                        !< increment pointers and loop back to read the next group of pixels
                        npixels = npixels - nbuffer
                        firstpix = firstpix + nbuffer
                    end Do


                    !< read image depending of the dimension of the image
                    if (.not.OnlyYColumn(NumOut)) then
                        counter = 0
                        ValueOfFirstPoint(1) = crval(1) - ((crpix(1) - 1) * cdelt(1))       !< calculate value at first pixel
                        if (ndim == 1) then
                            Do x = 1, length(1)                                             !< loop over all points of the ith axis
                                counter = counter + 1                                       !< increase counter
                                xArray(1) = ValueOfFirstPoint(1) + ((x - 1) * cdelt(1)) - LSRAdjustement(NumOut)
                                xPointsModel_output(ThreadNumber, counter, 1) = xArray(1)
                                call CheckResamplingFlag(MaxX, NumOut, counter, xArray, valuesModel_output(ThreadNumber, counter), resampling_flag, &
                                                         FirstPointOutputFile, LastPointOutputFile, FirstPointValueOutputFile, LastPointValueOutputFile)
                            end Do

                        elseif (ndim == 2) then
                            Do x = 1, length(1)                                             !< loop over all points of the ith axis
                                ValueOfFirstPoint(2) = crval(2) - ((crpix(2) - 1) * cdelt(2))           !< calculate value at first pixel
                                Do y = 1, length(2)                                         !< loop over all points of the ith axis
                                    counter = counter + 1                                   !< increase counter
                                    xArray(1) = ValueOfFirstPoint(1) + ((x - 1) * cdelt(1)) - LSRAdjustement(NumOut)
                                    xArray(2) = ValueOfFirstPoint(2) + ((y - 1) * cdelt(2)) - LSRAdjustement(NumOut)
                                    xPointsModel_output(ThreadNumber, counter, 1:2) = xArray(1:2)
                                    call CheckResamplingFlag(MaxX, NumOut, counter, xArray, valuesModel_output(ThreadNumber, counter), resampling_flag, &
                                                        FirstPointOutputFile, LastPointOutputFile, FirstPointValueOutputFile, LastPointValueOutputFile)
                                end Do
                            end Do

                        elseif (ndim == 3) then
                            Do x = 1, length(1)                                             !< loop over all points of the ith axis
                                ValueOfFirstPoint(2) = crval(2) - ((crpix(2) - 1) * cdelt(2))           !< calculate value at first pixel
                                Do y = 1, length(2)                                         !< loop over all points of the ith axis
                                    ValueOfFirstPoint(3) = crval(3) - ((crpix(3) - 1) * cdelt(3))       !< calculate value at first pixel
                                    Do z = 1, length(3)                                     !< loop over all points of the ith axis
                                        counter = counter + 1                               !< increase counter
                                        xArray(1) = ValueOfFirstPoint(1) + ((x - 1) * cdelt(1)) - LSRAdjustement(NumOut)
                                        xArray(2) = ValueOfFirstPoint(2) + ((y - 1) * cdelt(2)) - LSRAdjustement(NumOut)
                                        xArray(3) = ValueOfFirstPoint(3) + ((z - 1) * cdelt(3)) - LSRAdjustement(NumOut)
                                        xPointsModel_output(ThreadNumber, counter, 1:3) = xArray(1:3)
                                        call CheckResamplingFlag(MaxX, NumOut, counter, xArray, valuesModel_output(ThreadNumber, counter), &
                                                                 resampling_flag, FirstPointOutputFile, LastPointOutputFile, &
                                                                 FirstPointValueOutputFile, LastPointValueOutputFile)
                                    end Do
                                end Do
                            end Do

                        elseif (ndim == 3) then
                            Do x = 1, length(1)                                             !< loop over all points of the ith axis
                                ValueOfFirstPoint(2) = crval(2) - ((crpix(2) - 1) * cdelt(2))           !< calculate value at first pixel
                                Do y = 1, length(2)                                         !< loop over all points of the ith axis
                                    ValueOfFirstPoint(3) = crval(3) - ((crpix(3) - 1) * cdelt(3))       !< calculate value at first pixel
                                    Do z = 1, length(3)                                     !< loop over all points of the ith axis
                                        ValueOfFirstPoint(4) = crval(4) - ((crpix(4) - 1) * cdelt(4))   !< calculate value at first pixel
                                        Do t = 1, length(4)                                 !< loop over all points of the ith axis
                                            counter = counter + 1                           !< increase counter
                                            xArray(1) = ValueOfFirstPoint(1) + ((x - 1) * cdelt(1)) - LSRAdjustement(NumOut)
                                            xArray(2) = ValueOfFirstPoint(2) + ((y - 1) * cdelt(2)) - LSRAdjustement(NumOut)
                                            xArray(3) = ValueOfFirstPoint(3) + ((z - 1) * cdelt(3)) - LSRAdjustement(NumOut)
                                            xArray(4) = ValueOfFirstPoint(4) + ((t - 1) * cdelt(4)) - LSRAdjustement(NumOut)
                                            xPointsModel_output(ThreadNumber, counter, 1:4) = xArray(1:4)
                                            call CheckResamplingFlag(MaxX, NumOut, counter, xArray, valuesModel_output(ThreadNumber, counter), &
                                                                     resampling_flag, FirstPointOutputFile, LastPointOutputFile, &
                                                                     FirstPointValueOutputFile, LastPointValueOutputFile)
                                        end Do
                                    end Do
                                end Do
                            end Do
                        endif
                        FinalCounter = counter
                    endif


                    !< close file
                    !call ftclos(unitNum, io_stat)                                          !< FITS file must always be closed before exiting the program.
                    !call ftfiou(unitNum, io_stat)                                          !< Any unit numbers allocated with FTGIOU must be freed with
                endif                                                                       !< FTFIOU.

                ! Debug:
                ! print*,'-->',xPointsModel_output(ThreadNumber, :counter, 1)
                ! print*,'==>',valuesModel_output(ThreadNumber, :counter)


                return
            end subroutine ReadFitsFile


            !*********************************************************************************************************************************************
            !> subroutine: CheckResamplingFlag
            !>
            !> input variables:     NumOut:             number of current output file
            !>                      filename:           name of the current fits file
            !>
            !> output variables:    FinalCounter:       number of all lines
            !>
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 08.02.2011
            !>
            subroutine CheckResamplingFlag(MaxX, NumOut, counter, xArray, yValue, resampling_flag, FirstPointOutputFile, LastPointOutputFile, &
                                                                                            FirstPointValueOutputFile, LastPointValueOutputFile)
                !< determine parameters for resampling

                implicit none
                integer :: MaxX                                                             !< max. number of x-columns
                integer :: k                                                                !< loop variable
                integer :: ii, jj                                                           !< working variables
                integer :: NumOut                                                           !< current number of output file
                integer :: counter                                                          !< current length of fits file
                real*8 :: yValue                                                            !< current y-column value
                real*8 :: FirstPointValueOutputFile, LastPointValueOutputFile               !< value of first point and last point
                real*8, dimension(MaxX) :: xArray                                           !< the current x-column values
                real*8, dimension(MaxX) :: FirstPointOutputFile                             !< first point (lowest value of x-column vector)
                real*8, dimension(MaxX) :: LastPointOutputFile                              !< last point (highest value of x-column vector)
                logical :: resampling_flag


                !< do a x-column check if OnlyYColumn == .false.
                if (.not.resampling_flag) then
                    ii = 0
                    if (counter <= lengthexpdata(NumOut)) then
                        Do k = 1, NumberXColumns(NumOut)
                            jj = counter
                            if (xArray(k) == expdatax(NumOut, jj, k)) then
                                ii = ii + 1
                            else
                                exit
                            endif
                        end Do
                    endif
                    if (ii /= NumberXColumns(NumOut)) then
                        resampling_flag = .true.                                            !< set general resampling flag
                    endif
                endif


                !< determine first and last point of the output file array
                ii = 0
                jj = 0
                Do k = 1, NumberXColumns(NumOut)
                    if (xArray(k) <= FirstPointOutputFile(k)) then
                        ii = ii + 1
                    endif
                    if (xArray(k) >= LastPointOutputFile(k)) then
                        jj = jj + 1
                    endif
                end Do
                if (ii == NumberXColumns(NumOut)) then
                    FirstPointOutputFile(1:NumberXColumns(NumOut)) = xArray(1:NumberXColumns(NumOut))
                    FirstPointValueOutputFile = yValue
                endif
                if (jj == NumberXColumns(NumOut)) then
                    LastPointOutputFile(1:NumberXColumns(NumOut)) = xArray(1:NumberXColumns(NumOut))
                    LastPointValueOutputFile = yValue
                endif
                return
            end subroutine CheckResamplingFlag


            !*********************************************************************************************************************************************
            !> subroutine: RegistrationMask
            !>
            !> open registration file and construct input and output mask
            !>
            !>
            !> input variables:     ok:                 number of current output file
            !>
            !> output variables:    None
            !>
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 08.02.2011
            !>
            subroutine RegistrationMask(ok)

                implicit none
                integer, parameter :: MaxStringLength = 16384                               !< define max. length of line in registration file
                integer :: i, j, k, l, n, m, ll, kk, lcPV                                   !< working variables
                integer :: ok                                                               !< subroutine status
                integer :: eol                                                              !< position of the character defining the end of line (eol)
                integer :: io_err                                                           !< error value
                integer :: OutCounter                                                       !< counter for output file names
                integer :: NumInputFile                                                     !< index of the current input-file
                integer :: NumOutputFile                                                    !< index of the current output-file
                integer :: LengthInOutputFile                                               !< number of lines of the InOutputFile
                integer :: NumberLines                                                      !< number of lines
                integer :: LineIndex                                                        !< indicate the current line in the input file
                integer :: ColumnIndex                                                      !< indicate the current column in the current line
                integer :: NumberReplicationLine                                            !< number of replications of the current line
                integer :: NumberParameterLine                                              !< number of parameters within the current line
                integer :: NumberReplicationParameter                                       !< number of replications of the current parameter
                integer :: indexbracket                                                     !< index in string where '[[' occur
                integer :: counter                                                          !< count number of apperance of a name of a model parameter
                integer :: allocstatus, deallocstatus                                       !< status of the allocate/deallocate process
                integer :: localCounter                                                     !< working variable: local counter for LocalParamNames array
                integer :: LocalNameCounter                                                 !< working variable: counter for LocalParamNames array
                integer :: OccurrenceIndex                                                  !< occurrence index
                real*8 :: floatvalue                                                        !< working variable
                character(len=5) :: GroupLineFlag                                           !< string containing the group line flag
                character(len=5) :: GroupParameterFlag                                      !< string containing the group parameter flag
                character(len=5) :: hilfString                                              !< help string for number conversion
                character(len=MaxStringLength) :: FilenameRegistration, LongHelpString1     !< working variables
                character(len=MaxStringLength) :: string, helpstring                        !< working variables
                character(len=MaxStringLength) :: line                                      !< working variable: contains the current line from reg. file
                character(len=MaxStringLength) :: space                                     !< working variable: only blank characters
                character(len=MaxStringLength) :: ParamVisibleLocal, TempString, helpString2    !< local copy of ParamVisible
                character(len=512), allocatable, dimension(:) :: LocalParamNames            !< working variable: contains all parameter names
                character(len=512), allocatable, dimension(:) :: VisibleLocalParamNames     !< working variable: contains all visible parameter names
                logical :: EqualFlag                                                        !< working variable: used for comparison of two strings
                logical :: UseAcurateMethod                                                 !< working variable: use accurate method or not
                logical :: LocalVisibleFlag                                                 !< working variable
                logical :: CheckFlag                                                        !< working variable


                !< define space
                Do i = 1, MaxStringLength
                    space(i:i) = " "
                end Do


                !-----------------------------------------------------------------------------------------------------------------------------------------
                !< deallocate if necessary and allocate memory
                ok = 0
                if (allocated(ParameterName)) then
                    deallocate(NumberParamPerLine, ParameterName, ParameterFormat, LeadingString, TrailingString, ParamVisible, FitFktInput, &
                                FitFktOutput, OutputFileFormat, OnlyYColumn, NumberHeaderLines, CharacterForComments, CharacterSeperatingColumns, &
                                ResamplingMethod, LSRAdjustement, LSRAdjustementFitFlag, InterpolationMethod, NormalizationFlag, ValueEmptyOutputFile, &
                                stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine RegistrationMask:")')
                        write(logchannel,'(2x,"Can not deallocate variables NumberParamPerLine etc.")')
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,*)
                        write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')

                        print '(" ")'
                        print '("Error in subroutine RegistrationMask:")'
                        print '(2x,"Can not deallocate variables NumberParamPerLine etc.")'
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("deallocstatus = ",I4)',deallocstatus
                        print '(" ")'
                        ok = 1
                        return
                    endif
                endif
                allocate(NumberParamPerLine(NumberInputFiles, MaxInputLines), ParameterName(NumberInputFiles, MaxInputLines, MaxParameter), &
                         ParameterFormat(0:NumberInputFiles, 0:MaxInputLines, 0:MaxParameter), &
                         LeadingString(NumberInputFiles, MaxInputLines, MaxParameter), TrailingString(NumberInputFiles, MaxInputLines, MaxParameter), &
                         FitFktInput(NumberInputFiles), FitFktOutput(NumberOutputFiles), ParamVisible(NumberInputFiles, MaxInputLines, MaxParameter), &
                         OutputFileFormat(NumberOutputFiles), OnlyYColumn(NumberOutputFiles), NumberHeaderLines(NumberOutputFiles), &
                         CharacterForComments(NumberOutputFiles), CharacterSeperatingColumns(NumberOutputFiles), ResamplingMethod(NumberOutputFiles), &
                         LSRAdjustement(NumberOutputFiles), LSRAdjustementFitFlag(NumberOutputFiles), InterpolationMethod(NumberOutputFiles), &
                         NormalizationFlag(NumberOutputFiles), ValueEmptyOutputFile(NumberOutputFiles), LocalParamNames(MaxInputLines * MaxParameter), &
                         VisibleLocalParamNames(MaxInputLines * MaxParameter), stat = allocstatus)
                if (allocstatus /= 0) then
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine RegistrationMask:")')
                    write(logchannel,'(2x,"Can not allocate variables NumberParamPerLine etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("allocstatus = ",I4)') allocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine RegistrationMask:")'
                    print '(2x,"Can not allocate variables NumberParamPerLine etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("allocstatus = ",I4)',allocstatus
                    print '(" ")'
                    ok = 1
                    return
                endif
                NumberParamPerLine = 0
                ParameterName = ""
                ParameterFormat = ""
                LeadingString = ""
                TrailingString = ""
                FitFktInput = ""
                FitFktOutput = ""
                ParamVisible = ""
                OutputFileFormat = 0
                OnlyYColumn = .false.
                NumberHeaderLines = 0
                CharacterForComments = ""
                CharacterSeperatingColumns = ""
                ResamplingMethod = "simple"
                LSRAdjustement = 0.d0
                LSRAdjustementFitFlag = .false.
                InterpolationMethod = "nearest-neighbour"
                NormalizationFlag = .false.
                ValueEmptyOutputFile = 0.d0


                !< initialize some variables
                GroupLineFlag = ""
                NumberLines = 0
                NumberReplicationLine = 0
                NumberParameterLine = 0
                NumberReplicationParameter = 0
                GroupParameterFlag = ""
                CommandWordOutput = ""
                CalculationMethod = "atonce"
                InputDataPath = ""
                AllInOneOutputFile = .false.
                NumOutputFilesEqualNumExpFiles = .false.
                debugflag = .false.


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< loop over all input-files
                OutCounter = 0
                Do NumInputFile = 1, NumberInputFiles


                    !-------------------------------------------------------------------------------------------------------------------------------------
                    !< open temp-file including description of the registration mask
                    write(LongHelpString1,'(I30)') JobID                                    !< write JobID to string
                    write(hilfString,'(I5)') NumInputFile                                   !< write number of input file to string
                    FilenameRegistration = trim(adjustl(TempDirectory))//"job_"//trim(adjustl(LongHelpString1)) // "/"
                    FilenameRegistration = trim(adjustl(FilenameRegistration)) // "Registration_Input_" // trim(adjustl(hilfString)) // ".txt"
                    open(tempChannel,file = trim(adjustl(FilenameRegistration)))            !< open temp file for registration


                    !< determine number of lines of the temp file
                    LengthInOutputFile = 0
                    io_err = 0
                    Do While (io_err == 0)
                        read(tempChannel,'(A)', iostat = io_err) line
                        LengthInOutputFile = LengthInOutputFile + 1
                    end Do
                    LengthInOutputFile = LengthInOutputFile - 2
                    ! Debug:
                    !    print*,"LengthInOutputFile = ",LengthInOutputFile


                    !-------------------------------------------------------------------------------------------------------------------------------------
                    !< determine number of groups and number of lines per group
                    rewind tempChannel


                    !-------------------------------------------------------------------------------------------------------------------------------------
                    !< read registration file
                    LineIndex = 0
                    ColumnIndex = 0
                    LocalNameCounter = 0
                    LocalParamNames = ""
                    VisibleLocalParamNames = ""
                    Do i = 1, LengthInOutputFile


                        !< read next line of the temp file
                        line = space(:)
                        read(tempChannel,'(A)') line
                        eol = index(line,'|',back = .true.) - 1                             !< determine position of the last character


                        !< get value of debug flag
                        if (NumInputFile == 1) then
                            j = index(line,'debugflag = ')                                  !< get number of replications of the line
                            if (j /= 0 .and. line(j+12:eol) /= "") then
                                if (trim(adjustl(line(j+12:eol))) == "true") then
                                    debugflag = .true.
                                    ! Debug:
                                    !  print*,"debugflag = ", debugflag
                                endif
                            endif
                        endif


                        !< get name and path of input file
                        j = index(line,'FitFunctionIn = ')                                  !< get number of replications of the line
                        if (j /= 0 .and. line(j+16:eol) /= "") then
                            FitFktInput(NumInputFile) = trim(adjustl(line(j+16:eol)))
                            ! Debug:
                            !  print*,"FitFktInput(NumInputFile) = ",FitFktInput(NumInputFile)
                        endif


                        !< get line index
                        j = index(line,'LineIndex = ')                                      !< get number of replications of the line
                        if (j /= 0 .and. line(j+12:eol) /= "") then
                            read(line(j+12:eol),*) floatvalue                               !< convert to number
                            LineIndex = int(floatvalue) + 1
                            ! Debug:
                            !   print*,"LineIndex = ",LineIndex
                        endif


                        !< get number of parameter of each line
                        j = index(line,'NumberParameterLine = ')
                        if (j /= 0 .and. line(j+22:eol) /= "" .and. LineIndex > 0) then
                            read(line(j+22:eol),*) floatvalue
                            NumberParamPerLine(NumInputFile,LineIndex) = int(floatvalue)
                        endif


                        !< get column index
                        j = index(line,'ColumnIndex = ')                                    !< get number of replications of the line
                        if (j /= 0 .and. line(j+14:eol) /= "") then
                            read(line(j+14:eol),*) floatvalue                               !< convert to number
                            ColumnIndex = int(floatvalue) + 1
                            ! Debug:
                            !   print*,"ColumnIndex = ",ColumnIndex
                        endif


                        !< continue only, if LineIndex and ColumnIndex are known
                        if (LineIndex > 0 .and. ColumnIndex > 0) then


                            !< get leading string
                            j = index(line,'LeadingString = ')
                            if (j /= 0 .and. (j + 15) < eol) then
                                LeadingString(NumInputFile,LineIndex,ColumnIndex) = line(j+16:eol) // "|"
                                ! Debug:
                                ! print*,"LeadingString(NumInputFile,LineIndex,ColumnIndex) = ", line(j+16:eol) // "|"
                            endif


                            !< get name of parameter
                            j = index(line, 'ParameterName = ')
                            if (j /= 0 .and. line(j + 16:eol) /= "") then
                                ParameterName(NumInputFile, LineIndex, ColumnIndex) = trim(adjustl(line(j + 16:eol)))
                                if (index(ParameterName(NumInputFile, LineIndex, ColumnIndex), "[[]]", back = .true.) == 0) then
                                    LocalNameCounter = LocalNameCounter + 1
                                    LocalParamNames(LocalNameCounter) = trim(adjustl(ParameterName(NumInputFile, LineIndex, ColumnIndex)))
                                endif
                                ! Debug:
                                !  print*,"ParameterName(NumInputFile, LineIndex, ColumnIndex) = ",trim(ParameterName(NumInputFile,LineIndex,ColumnIndex))
                            endif


                            !< get format of parameter
                            j = index(line,'ParameterFormat = ')
                            if (j /= 0 .and. line(j + 18:eol) /= "") then
                                ParameterFormat(NumInputFile,LineIndex,ColumnIndex) = "(" // trim(adjustl(line(j+18:eol))) // ")"
                                ! Debug:
                                !  print*,"ParameterFormat(NumInputFile,LineIndex,ColumnIndex) = ",trim(ParameterFormat(NumInputFile,LineIndex,ColumnIndex))
                            endif


                            !< get trailing string of parameter
                            j = index(line,'TrailingString = ')
                            if (j /= 0 .and. (j + 16) < eol) then
                                TrailingString(NumInputFile, LineIndex,ColumnIndex) = line(j+17:eol) // "|"
                                ! Debug:
                                ! print*,"TrailingString(NumInputFile, LineIndex,ColumnIndex) = ",TrailingString(NumInputFile, LineIndex,ColumnIndex)
                            endif


                            !< get ParamVisible flag of parameter
                            j = index(line,'ParamVisible = ')
                            if (j /= 0 .and. line(j+15:eol) /= "") then
                                ParamVisible(NumInputFile, LineIndex,ColumnIndex) = line(j+15:eol) // "|"
                                ! Debug:
                                !  print*,"ParamVisible(NumInputFile, LineIndex,ColumnIndex) = ",ParamVisible(NumInputFile, LineIndex,ColumnIndex)
                            endif
                        endif


                        !< read specialized output file description
                        if (CalculationMethod == "pointtopoint") then                       !< if calculation method is "pointtopoint"


                            !< get number of lines of output file
                            j = index(line,'NumberLinesOutput = ')
                            if (j /= 0 .and. line(j+19:eol) /= "") then
                                read(line(j+19:eol),*) floatvalue
                                NumberLinesOutput = int(floatvalue)
                                ! Debug:
                                !   print*,"NumberLinesOutput = ",NumberLinesOutput
                            endif


                            !< get number of (begin) columns of output file
                            j = index(line,'NumberColumnsBegin = ')                         !< read number of columns in the output file
                            if (j /= 0 .and. line(j+21:eol) /= "") then
                                read(line(j+21:eol),*) floatvalue
                                NumberColumnsBegin = int(floatvalue)
                                ! Debug:
                                !   print*,"NumberColumnsBegin = ",NumberColumnsBegin
                            endif


                            !< get number of (end) columns of output file
                            j = index(line,'NumberColumnsEnd = ')                           !< read number of columns in the output file
                            if (j /= 0 .and. line(j+21:eol) /= "") then
                                read(line(j+21:eol),*) floatvalue
                                NumberColumnsEnd = int(floatvalue)
                                ! Debug:
                                !   print*,"NumberColumnsEnd = ",NumberColumnsEnd
                            endif


                            !< get command word
                            j = index(line,'CommandWordOutput = ')                          !< read number of columns in the output file
                            if (j /= 0) then
                                CommandWordOutput = trim(adjustl(line(j+19:eol)))
                                ! Debug:
                                !   print*,"CommandWordOutput = ",CommandWordOutput
                            endif


                        elseif (CalculationMethod == "atonce") then                         !< if calculation method is "atonce"


                            !< get path and file name of the data file for input
                            j = index(line,'InputDataPath = ')
                            if (j /= 0 .and. line(j + 15:eol) /= "") then
                                InputDataPath = trim(adjustl(line(j + 15:eol)))
                                ! Debug:
                                !    print*,"InputDataPath = ",InputDataPath,'<<'
                            endif
                        endif
                    end Do


                    !< close temp file and delete it (if LastAlgorithmFlag == .true.)
                    if (LastAlgorithmFlag) then
                        close(tempChannel, status = 'delete')
                    else
                        close(tempChannel)
                    endif


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< extend the name of model parameter with an unique index to make each parameter name unique
                    UseAcurateMethod = .false.                                              !< use fast method
                    localCounter = 0                                                        !< reset local counter
                    Do i = 1, MaxInputLines                                                 !< loop over all lines of the NumInputFile-th input file
                        Do j = 1, NumberParamPerLine(NumInputFile, i)                       !< loop over all parameters of the ith line
                            if (index(ParameterName(NumInputFile, i, j), "[[]]", back = .true.) == 0) then  !< ignore parameter names ending with '[[]]'
                                counter = 1                                                 !< initialize counter variable
                                localCounter = localCounter + 1                             !< increase counter


                                !< analyze jth parameter name of the ith line of the current input file
                                indexbracket = index(ParameterName(NumInputFile, i, j),"[[",back = .true.)
                                if (indexbracket == 0) then
                                    string = trim(ParameterName(NumInputFile, i, j))
                                else
                                    string = trim(ParameterName(NumInputFile, i, j)(:indexbracket - 1))
                                endif
                                string = trim(adjustl(string))


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< determine the number of occurences of the current parameter up to now


                                !< use fast method
                                if (.not. UseAcurateMethod) then
                                    if (localCounter > 1 .and. localCounter <= LocalNameCounter) then
                                        counter = count(LocalParamNames(:(localCounter - 1)) == string) + 1

                                        ! Debug:
                                        !    print*,"string = ", trim(adjustl(string))
                                        !    print*,"counter = ", counter
                                        !    print*,"localCounter = ", localCounter
                                    else
                                        counter = localCounter
                                    endif


                                !< use an accurate method to determine the index of the current parameter
                                else


                                    !< the following two loops (over k and l) run over all the param. used in the previous lines of the current input file
                                    Do k = 1, i                                                 !< loop over all previous lines of the current input file
                                        if (k < i) then
                                            m = NumberParamPerLine(NumInputFile, k)
                                        else
                                            m = (j - 1)
                                        endif
                                        Do l = 1, m                                             !< loop over all param. of up to the current kth parameter


                                            !< analyze the lth parameter name of the kth line in the current input file
                                            if (index(ParameterName(NumInputFile, k, l), "[[]]", back = .true.) /= 0) then
                                                helpString = trim(ParameterName(NumInputFile, k, l))
                                            else
                                                indexbracket = index(ParameterName(NumInputFile, k, l),"[[", back = .true.)
                                                if (indexbracket == 0) then
                                                    helpString = trim(ParameterName(NumInputFile, k, l))
                                                else
                                                    helpString = trim(ParameterName(NumInputFile, k, l)(:indexbracket - 1))
                                                endif
                                            endif
                                            helpString = trim(adjustl(helpString))


                                            !< compare both parameter names
                                            EqualFlag = .false.                                 !< reset equal flag
                                            if (len_trim(helpString) == len_trim(string)) then
                                                EqualFlag = .true.
                                                Do n = 1, len_trim(helpString)
                                                    if (helpString(n:n) /= string(n:n)) then
                                                        EqualFlag = .false.
                                                        exit
                                                    endif
                                                end Do
                                            endif


                                            !< if both parameter names are identical, increase counter variable
                                            if (EqualFlag) then
                                                counter = counter + 1
                                            endif
                                        end Do
                                    end Do
                                endif


                                !< expand name
                                write(hilfString, '(I5)') counter
                                ParameterName(NumInputFile, i, j) = trim(adjustl(ParameterName(NumInputFile, i, j))) // "[[" &
                                                                    // trim(adjustl(hilfString)) // "]]"
                            endif

                            ! Debug:
                            !    print*,'NumInputFile, i, j = ', NumInputFile, i, j
                            !    print*,'ParameterName(NumInputFile, i, j) = ',trim(ParameterName(NumInputFile, i, j))


                            !<----------------------------------------------------------------------------------------------------------------------------
                            !< determine visibility flag for each parameter


                            !< determine posiotion of last character of leading string
                            lcPV = index(ParamVisible(NumInputFile, i, j), "|", back = .true.)
                            if (lcPV == 0) then
                                lcPV = len_trim(ParamVisible(NumInputFile, i, j))
                            else
                                lcPV = lcPV - 1
                            endif
                            ParamVisibleLocal = ParamVisible(NumInputFile, i, j)(1:lcPV)
                            ParamVisibleLocal = trim(adjustl(ParamVisibleLocal))
                            if (trim(adjustl(ParamVisibleLocal)) == "f") then
                                LocalVisibleFlag = .false.
                            else

                                ! Debug:
                                !    call GetThreadNumberForTempDir(ThreadNumber)
                                !    print*,'ThreadNumber, NumInputFile, i, j = ', ThreadNumber, NumInputFile, i, j
                                !    print*,'ParameterName(NumInputFile, i, j) = ', '>>',trim(adjustl(ParameterName(NumInputFile,i, j))), '<<'
                                !    print*,'lcPV = ', lcPV
                                !    print*,'ParamVisible(NumInputFile, i, j) = ', trim(adjustl(ParamVisible(NumInputFile, i, j)))
                                !    print*,'ParamVisibleLocal = ', trim(adjustl(ParamVisibleLocal))


                                !< determine the occurrence index of the current parameter
                                TempString = trim(adjustl(ParameterName(NumInputFile, i, j)))
                                OccurrenceIndex = 1
                                ll = index(TempString, '[[', back = .true.)
                                if (ll /= 0) then
                                    helpString2 = trim(adjustl(TempString(ll + 2:)))
                                    ll = index(helpString2, ']]', back = .true.)
                                    if (ll /= 0) then
                                        helpString2 = trim(adjustl(helpString2(:ll - 1)))
                                        if (trim(adjustl(helpString2)) /= "") then
                                            read(helpString2, '(I10)') OccurrenceIndex
                                        endif
                                    endif
                                endif


                                !< determine if parameter is visible or not
                                LocalVisibleFlag = .true.
                                ll = 0                                                      !< reset counter
                                Do kk = 1, len_trim(ParamVisibleLocal)                      !< loop over all entries of the visibility flag
                                    if (ParamVisibleLocal(kk:kk) == ";") then               !< are there more than one criterium for visibility
                                        ll = 0                                              !< reset counter
                                    else
                                        ll = ll + 1                                         !< increase counter
                                        CheckFlag = .true.                                  !< initialize check flag
                                        if (len_trim(ParamVisibleLocal) > 1) then           !< parameter describing visibility is multiple defined
                                            if (ll /= OccurrenceIndex) then
                                                CheckFlag = .false.
                                            endif
                                        endif
                                        if (CheckFlag) then
                                            if (ParamVisibleLocal(kk:kk) == "f") then
                                                LocalVisibleFlag = .false.
                                                exit
                                            endif
                                        endif
                                    endif
                                end Do
                            endif


                            !< define final visible flag for current parameter
                            if (LocalVisibleFlag) then
                                ParamVisible(NumInputFile, i, j) = "true|"


                                !< add name of the current visible parameter
                                VisibleLocalParamNames(localCounter) = trim(adjustl(LocalParamNames(localCounter)))


                                !< determine the number of ocurrances of the current parameter which are really visible
                                if (localCounter > 1 .and. localCounter <= LocalNameCounter) then
                                    counter = count(VisibleLocalParamNames(:(localCounter - 1)) == VisibleLocalParamNames(localCounter)) + 1

                                    ! Debug:
                                    !    print*,"string = ", trim(adjustl(string))
                                    !    print*,"counter = ", counter
                                    !    print*,"localCounter = ", localCounter
                                else
                                    counter = localCounter
                                endif


                                !< expand parameter name with the ocurrance number of this parameter
                                if (index(ParameterName(NumInputFile, i, j), "[[]]", back = .true.) == 0) then
                                    write(hilfString, '(I5)') counter
                                    ParameterName(NumInputFile, i, j) = trim(adjustl(VisibleLocalParamNames(localCounter))) // "[[" &
                                                                        // trim(adjustl(hilfString)) // "]]"
                                endif
                            else
                                ParamVisible(NumInputFile, i, j) = "false|"
                            endif
                        end Do
                    end Do

                    ! Debug:
                    !    Do i = 1, MaxInputLines
                    !        print*,'i = ', i
                    !        print*,'NumberParamPerLine(NumInputFile,i) = ',NumberParamPerLine(NumInputFile, i)
                    !        Do j = 1, NumberParamPerLine(NumInputFile, i)
                    !            print*,'j = ', j
                    !            print*,'ParameterName(NumInputFile, i, j) = ', trim(ParameterName(NumInputFile, i, j)), '-'
                    !            print*,'ParameterFormat(NumInputFile, i, j) = ', trim(ParameterFormat(NumInputFile, i, j)), '-'
                    !            print*,'LeadingString(NumInputFile, i, j) = ', trim(LeadingString(NumInputFile, i, j)), '-'
                    !            print*,'TrailingString(NumInputFile, i, j) = ', trim(TrailingString(NumInputFile, i, j)), '-'
                    !            print*,'ParamVisible(NumInputFile, i, j) = ', trim(ParamVisible(NumInputFile, i, j)), '-'
                    !        end Do
                    !    end Do
                    !    stop
                end Do


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< read in settings for output file
                Do NumOutputFile = 1, NumberOutputFiles


                    !-------------------------------------------------------------------------------------------------------------------------------------
                    !< open temp-file including description of the registration mask
                    write(LongHelpString1,'(I30)') JobID                                    !< write JobID to string
                    write(hilfString,'(I5)') NumOutputFile                                  !< write number of input file to string
                    FilenameRegistration = trim(adjustl(TempDirectory))//"job_"//trim(adjustl(LongHelpString1)) // "/"
                    FilenameRegistration = trim(adjustl(FilenameRegistration)) // "Registration_Output_" // trim(adjustl(hilfString)) // ".txt"
                    open(tempChannel,file = trim(adjustl(FilenameRegistration)))            !< open temp file for registration


                    !< determine number of lines of the temp file
                    LengthInOutputFile = 0
                    io_err = 0
                    Do While (io_err == 0)
                        read(tempChannel,'(A)', iostat = io_err) line
                        LengthInOutputFile = LengthInOutputFile + 1
                    end Do
                    LengthInOutputFile = LengthInOutputFile - 2
                    ! Debug:
                    !    print*,"LengthInOutputFile = ",LengthInOutputFile


                    !-------------------------------------------------------------------------------------------------------------------------------------
                    !< determine number of groups and number of lines per group
                    rewind tempChannel


                    !-------------------------------------------------------------------------------------------------------------------------------------
                    !< read registration file
                    LineIndex = 0
                    ColumnIndex = 0
                    Do i = 1,LengthInOutputFile


                        !< read next line of the temp file
                        line = space(:)
                        read(tempChannel,'(A)') line
                        eol = index(line,'|',back = .true.) - 1                             !< determine position of the last character


                        !< read output file names
                        j = index(line,'OutputFileName = ')                                 !< get filename of output files
                        if (j /= 0 .and. line(j + 16:eol) /= "") then
                            FitFktOutput(NumOutputFile) = trim(adjustl(line(j + 16:eol) ))
                            ! Debug:
                            !   print*,"FitFktOutput(NumOutputFile) = ", FitFktOutput(NumOutputFile)
                        endif


                        !< read format of output file
                        j = index(line,'OutputFileFormat = ')                               !< get format of output files
                        if (j /= 0 .and. line(j + 18:eol) /= "") then
                            read(line(j+18:eol),*) floatvalue
                            OutputFileFormat(NumOutputFile) = int(floatvalue)
                            ! Debug:
                            !   print*,"OutputFileFormat(NumOutputFile) = ",OutputFileFormat(NumOutputFile)
                        endif


                        !< get OnlyYColumn flag
                        j = index(line,'OnlyYColumn = ')                                    !< read OnlyYColumn flag
                        if (j /= 0) then
                            if (trim(adjustl(line(j + 13:eol))) == "yes".or.trim(adjustl(line(j + 13:eol))) == "y" ) then
                                OnlyYColumn(NumOutputFile) = .true.
                            else
                                OnlyYColumn(NumOutputFile) = .false.
                            endif
                            ! Debug:
                            !   print*,"OnlyYColumn(NumOutputFile) = ",OnlyYColumn(NumOutputFile)
                        endif


                        !< get AllInOneOutputFile flag
                        j = index(line,'AllInOneOutputFile = ')                             !< read AllInOneOutputFile flag
                        if (j /= 0) then
                            if (trim(adjustl(line(j + 20:eol))) == "yes") then
                                AllInOneOutputFile = .true.
                            else
                                AllInOneOutputFile = .false.
                            endif
                            ! Debug:
                            !   print*,"AllInOneOutputFile = ", AllInOneOutputFile
                        endif


                        !< get NumOutputFilesEqualNumExpFiles flag
                        j = index(line,'NumOutputFilesEqualNumExpFiles = ')                 !< read NumOutputFilesEqualNumExpFiles flag
                        if (j /= 0) then
                            if (trim(adjustl(line(j + 32:eol))) == "yes" .or. trim(adjustl(line(j + 32:eol))) == "true") then
                                NumOutputFilesEqualNumExpFiles = .true.
                            else
                                NumOutputFilesEqualNumExpFiles = .false.
                            endif
                            ! Debug:
                            ! print*,"NumOutputFilesEqualNumExpFiles = ", NumOutputFilesEqualNumExpFiles
                        endif


                        !< get Number of Header Lines flag
                        j = index(line,'NumberHeaderLines = ')                              !< read Number of Header Lines flag
                        if (j /= 0 .and. line(j+19:eol) /= "") then
                            read(line(j+19:eol),*) floatvalue
                            NumberHeaderLines(NumOutputFile) = int(floatvalue)
                            ! Debug:
                            !   print*,"NumberHeaderLines(NumOutputFile) = ", NumberHeaderLines(NumOutputFile)
                        endif


                        !< get character indicating comments
                        j = index(line,'CharacterForComments = ')                           !< read character indicating comments
                        if (j /= 0 .and. line(j+22:eol) /= "") then
                            CharacterForComments(NumOutputFile) = trim(adjustl(line(j + 22:eol) ))
                            ! Debug:
                            !   print*,"CharacterForComments(NumOutputFile) = ", CharacterForComments(NumOutputFile)
                        endif


                        !< get character separating columns in the output file
                        !j = index(line,'CharacterForComments = ')                          !< read character separating columns in the output file
                        !if (j /= 0 .and. line(j+22:eol) /= "") then
                        !    CharacterSeperatingColumns(NumOutputFile) = line(j + 22:eol)
                        !    ! Debug:
                        !    !   print*,"CharacterSeperatingColumns(NumOutputFile) = ", CharacterSeperatingColumns(NumOutputFile)
                        !endif


                        !< get resampling method
                        j = index(line,'ResamplingMethod = ')                               !< read resampling method
                        if (j /= 0 .and. line(j+18:eol) /= "") then
                            ResamplingMethod(NumOutputFile) = trim(adjustl(line(j + 18:eol) ))
                            ! Debug:
                            !   print*,"ResamplingMethod(NumOutputFile) = ", trim(adjustl(ResamplingMethod(NumOutputFile)))
                        endif


                        !< get LSRAdjustement parameter
                        j = index(line,'LSRAdjustement = ')                                 !< read LSRAdjustement parameter
                        if (j /= 0 .and. line(j + 16:eol) /= "") then
                            read(line(j + 16:eol),*) floatvalue
                            LSRAdjustement(NumOutputFile) = floatvalue

                            ParameterFormat(0, 0, 0) = "(E20.10)"

                            ! Debug:
                            !   print*,"LSRAdjustement(NumOutputFile) = ", LSRAdjustement(NumOutputFile)
                        endif


                        !< get fit flag for LSRAdjustement parameter
                        j = index(line,'LSRAdjustementFlag = ')                             !< read fit flag for LSRAdjustement parameter
                        if (j /= 0 .and. line(j + 20:eol) /= "") then
                            if (trim(adjustl(line(j + 20:eol))) == "yes".or.trim(adjustl(line(j + 20:eol))) == "y" ) then
                                LSRAdjustementFitFlag(NumOutputFile) = .true.
                            else
                                LSRAdjustementFitFlag(NumOutputFile) = .false.
                            endif

                            ! Debug:
                            !   print*,"LSRAdjustementFitFlag(NumOutputFile) = ", LSRAdjustementFitFlag(NumOutputFile)
                        endif


                        !< get interpolation method
                        j = index(line,'InterpolationMethod = ')                            !< read interpolation method
                        if (j /= 0 .and. line(j+21:eol) /= "") then
                            InterpolationMethod(NumOutputFile) = trim(adjustl(line(j + 21:eol) ))
                            ! Debug:
                            !   print*,"InterpolationMethod(NumOutputFile) = ", trim(adjustl(InterpolationMethod(NumOutputFile)))
                        endif


                        !< get interpolation method
                        j = index(line,'NormalizationFlag = ')                              !< read normalization flag
                        if (j /= 0 .and. line(j+19:eol) /= "") then
                            if (trim(adjustl(line(j + 19:eol))) == "yes".or.trim(adjustl(line(j + 19:eol))) == "y" ) then
                                NormalizationFlag(NumOutputFile) = .true.
                            else
                                NormalizationFlag(NumOutputFile) = .false.
                            endif
                            ! Debug:
                            !   print*,"NormalizationFlag(NumOutputFile) = ", trim(adjustl(InterpolationMethod(NumOutputFile)))
                        endif


                        !< define string which is used instead of NaN
                        NaNReplaceString = "0.00"


                        !< define values of model function if output file is empty
                        j = index(line,'ValueEmptyOutputFiles = ')                          !< read model function value for empty output files
                        if (j /= 0 .and. line(j+22:eol) /= "") then
                            read(line(j + 22:eol),*) floatvalue
                            ValueEmptyOutputFile(NumOutputFile) = floatvalue
                            ! Debug:
                            !   print*,"ValueEmptyOutputFile(NumOutputFile) = ", ValueEmptyOutputFile(NumOutputFile)
                        endif
                    end Do


                    !< close temp file and delete it (if LastAlgorithmFlag == .true.)
                    if (LastAlgorithmFlag) then
                        close(tempChannel, status = 'delete')
                    else
                        close(tempChannel)
                    endif
                end Do
                return
            end subroutine RegistrationMask


            !*********************************************************************************************************************************************
            !> subroutine: DetermineParameterValue
            !>
            !> determine the value of the parameter indicated by the name of the parameter
            !>
            !>
            !> input variables:     string:             string containing name of parameter
            !>                      dataexpflag:        flag for checking exsistence of parameter
            !>
            !> output variables:    listindex:          index of parameter in list modelparam
            !>
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 14.10.2011
            !>
            subroutine DetermineParameterValue(listindex, string, dataexpflag)
                !< determine the value of the parameter indicated by the name of the parameter given by "string"

                implicit none
                integer :: i, j                                                             !< loop variables
                integer :: listindex                                                        !< index of parameter in list modelparam
                character(len=512) :: string                                                !< string containing name of parameter
                character(len=512) :: helpString                                            !< working variable
                logical :: EqualFlag                                                        !< working variable for comparing strings
                logical :: dataexpflag                                                      !< flag for checking exsistence of parameter


                string = trim(adjustl(string))
                listindex = (-1)
                Do i = 1, parameternumber                                                   !< loop over all parameter


                    !< check, if ith fit parameter name is identical to the name of the incoming parameter
                    EqualFlag = .false.
                    helpString = trim(adjustl(FitParameterName(i)))
                    if (len_trim(helpString) == len_trim(string)) then
                        EqualFlag = .true.
                        Do j = 1, len_trim(helpString)
                            if (helpString(j:j) /= string(j:j)) then
                                EqualFlag = .false.
                                exit
                            endif
                        end Do
                    endif


                    !< if both parameter names are identical, store parameter index
                    if (EqualFlag) then
                        listindex = i
                    endif
                end Do


                !< check, if an error occurred, i.e. it was not possible to determine the parameter index
                if (listindex == (-1) .and. dataexpflag) then                               !< cannot find parameter name
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine DetermineParameterValue:")')
                    write(logchannel,'(2x,"Can not find the parameter ",A)') trim(adjustl(string))
                    write(logchannel,'(2x,"in the list of parameters given by the parameter file!")')
                    write(logchannel,'(2x,"Please correct parameter or registration file")')
                    write(logchannel,'(2x,"and restart MAGIX!")')
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print*,' '
                    print*,' Error in subroutine DetermineParameterValue:'
                    print*,'   Can not find the parameter ',trim(adjustl(string))
                    print*,'   in the list of parameters given by the parameter file!'
                    print*,'   Please correct parameter or registration file'
                    print*,'   and restart MAGIX!'
                    print*,' '
                    stop 'Program aborted!'
                endif
                return
            end subroutine DetermineParameterValue


            !*********************************************************************************************************************************************
            !> subroutine: IndexFormat
            !>
            !> determine the format(string) of a parameter of number listindex in the parameter list
            !>
            !>
            !> input variables:     listindex:          index of parameter which format string is searched for
            !>
            !> output variables:    IntegerTrueFlag:    flag indicating if parameter is integer or not
            !>                      NumInputFile_index: index of input file
            !>                      i_index:            index of line in the (NumInputFile_index)th input file
            !>                      j_index:            index of a parameter in the (i_index)th line
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 14.10.2011
            !>
            subroutine IndexFormat(IntegerTrueFlag, NumInputFile_index, i_index, j_index, listindex)
                !< determine the format(string) of a parameter of number listindex in the parameter list

                implicit none
                integer :: i, j, k                                                          !< loop variables
                integer :: NumInputFile_index                                               !< index of input file
                integer :: i_index                                                          !< index of line in the (NumInputFile_index)th input file
                integer :: j_index                                                          !< index of a parameter in the (i_index)th line
                integer :: NumInputFile                                                     !< index of input-file
                integer :: listindex                                                        !< index of parameter which format string is searched for
                character(len=512) :: helpString                                            !< working variable
                character(len=512) :: string                                                !< working variable
                logical :: IntegerTrueFlag                                                  !< flag indicating if parameter is integer or not
                logical :: EqualFlag, finish_flag                                           !< working variables


                !< reset logical parameter
                IntegerTrueFlag = .false.
                finish_flag = .false.
                EqualFlag = .false.
                helpString = trim(adjustl(FitParameterName(listindex)))


                !< do not look for the format string of parameter 'MAGIXOutputLSRAdjustementName'
                if (index(helpString,'MAGIXOutputLSRAdjustementName') > 0) then
                    NumInputFile_index = 0
                    i_index = 0
                    j_index = 0
                    return
                endif


                !< determine indices NumInputFile_index, i_index, j_index
                NumInputFile_index = 0
                i_index = 0
                j_index = 0
                Do NumInputFile = 1, NumberInputFiles
                    Do i = 1, MaxInputLines
                        Do j = 1, NumberParamPerLine(NumInputFile,i)
                            string = trim(adjustl(ParameterName(NumInputFile, i, j)))       !< get current parameter name


                            !< compare strings 'helpString' and 'string'
                            EqualFlag = .false.
                            if (len_trim(helpString) == len_trim(string)) then
                                EqualFlag = .true.
                                Do k = 1, len_trim(helpString)
                                    if (helpString(k:k) /= string(k:k)) then
                                        EqualFlag = .false.
                                        exit
                                    endif
                                end Do
                            endif


                            !< both strings are identical
                            if (EqualFlag) then


                                !< check, if parameter is an integer number
                                if (index(ParameterFormat(NumInputFile, i, j), 'I') /= 0 .or. index(ParameterFormat(NumInputFile, i, j), 'i') /= 0) then
                                    IntegerTrueFlag = .true.
                                endif
                                NumInputFile_index = NumInputFile
                                i_index = i
                                j_index = j
                                finish_flag = .true.
                                exit
                            endif
                        end Do
                        if (finish_flag) exit
                    end Do
                    if (finish_flag) exit
                end Do


                !< error handling: It was not possible, to determine the string describing the format of the current parameter
                if (NumInputFile_index == 0 .or. i_index == 0 .or. j_index == 0) then
                    write(logchannel,*)
                    write(logchannel,'(11x,"Error in subroutine IndexFormat:")')
                    write(logchannel,'(13x,"Can not determine indices for parameter ", A)') trim(adjustl(FitParameterName(listindex)))
                    write(logchannel,'(13x,"with index listindex = ", I5)') listindex
                    write(logchannel,*)
                    write(logchannel,'(13x,"NumInputFile_index = ", I5)') NumInputFile_index
                    write(logchannel,'(13x,"i_index = ", I5)') i_index
                    write(logchannel,'(13x,"j_index = ", I5)') j_index
                    write(logchannel,*)
                    write(logchannel,'(13x,"List of parameter names:")')
                    Do NumInputFile = 1, NumberInputFiles
                        Do i = 1, MaxInputLines
                            Do j = 1, NumberParamPerLine(NumInputFile,i)
                                write(logchannel,'(13x,"NumInputFile, i, j, parameter: ",I5, ", ", I5, ", ", I5, ", ", A)') NumInputFile, i, j, &
                                                                                                   trim(adjustl(ParameterName(NumInputFile,i,j)))
                            end Do
                        end Do
                    end Do
                    write(logchannel,'(11x,"Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '(11x,"Error in subroutine IndexFormat:")'
                    print '(13x,"Can not determine indices for parameter ", A)', trim(adjustl(FitParameterName(listindex)))
                    print '(13x,"with index listindex = ", I5)', listindex
                    print '(" ")'
                    print '(13x,"NumInputFile_index = ", I5)', NumInputFile_index
                    print '(13x,"i_index = ", I5)', i_index
                    print '(13x,"j_index = ", I5)', j_index
                    print '(" ")'
                    print '(13x,"List of parameter names:")'
                    Do NumInputFile = 1, NumberInputFiles
                        Do i = 1, MaxInputLines
                            Do j = 1, NumberParamPerLine(NumInputFile,i)
                                print '(13x,"NumInputFile, i, j, parameter: ",I5, ", ", I5, ", ", I5, ", ", A)', NumInputFile, i, j, &
                                                                                        trim(adjustl(ParameterName(NumInputFile,i,j)))
                            end Do
                        end Do
                    end Do
                    stop '           Program aborted!'
                endif
                return
            end subroutine IndexFormat


            !*********************************************************************************************************************************************
            !> subroutine: WriteParameter
            !>
            !> write input file for given model
            !>
            !>
            !> input variables:     channel:            number of output channel, where the input file is written to
            !>                      dataexpflag:
            !>                      lenposdatexp:       number of lines in the experimental data file
            !>                      posdataexp:         array defining the experimental x point
            !>                      modelparamnumber:   number of parameters
            !>                      modelparam:         array containing the parameter set
            !>                      NumInputFile:       number of the input file
            !>
            !> output variables:    None
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 14.10.2011
            !>
            subroutine WriteParameter(channel, dataexpflag, lenposdatexp, posdataexp, modelparamnumber, modelparam, NumInputFile)
                !<
                !<
                !< ATTENTION !
                !< -----------
                !<
                !<      The following names MUST NOT be part of a name of a parameter:
                !<
                !<
                !<      XValue, NumberXValue, CurrentExpFile, CurrentYColumn, MAGIXImport
                !<
                !<
                !<----------------------------------------------------------------------------------------------------------------------------------------

                implicit none
                integer :: channel                                                          !< contains the channel number where the parameter are written
                integer :: lenposdatexp                                                     !< length of posdataexp
                integer :: modelparamnumber                                                 !< number of parameters
                integer :: NumInputFile                                                     !< number of the input file
                integer :: i, j, k, l, lcLS, lcTS, lcPV                                     !< working variables
                integer :: ParameterIndex                                                   !< index of parameter in list modelparam
                real*8 :: value                                                             !< working variable
                real*8 :: ValueDouble                                                       !< working variable
                real*8,dimension(lenposdatexp) :: posdataexp                                !< array defining the experimental x point
                real*8,dimension(modelparamnumber) :: modelparam                            !< array containing the parameter set
                character(len=5) :: FormatString                                            !< carries the number of x-column index as string
                character(len=50) :: helpString, helpString2                                !< working variables
                character(len=512) :: TempString                                            !< working variable
                logical :: EqualFlag                                                        !< working variable
                logical :: NonNumberFlag                                                    !< working variable
                logical :: dummyFlag                                                        !< flag for lines without a parameter
                logical :: dataexpflag                                                      !< flag is true, if the additional parameters "XValue" and
                                                                                            !< "NumberXValue" are included in the input file
                logical :: MAGIXCommandFlag                                                 !< working variables


                Do i = 1, MaxInputLines
                    Do j = 1, NumberParamPerLine(NumInputFile, i)
                        dummyFlag = .false.                                                 !< set dummy flag to false (needed for lines without parameters)
                        MAGIXCommandFlag = .false.


                        !< model parameter name to parameter value converter
                        NonNumberFlag = .false.
                        if (index(ParameterName(NumInputFile,i,j),"XValue") /= 0 .and. CalculationMethod == "pointtopoint") then
                            MAGIXCommandFlag = .true.
                            if (dataexpflag) then
                                Do k = 1, lenposdatexp                                      !< special case: value of the "x"-column values
                                    write(FormatString,'(I5)') k                            !< write value of k to string FormatString
                                    helpString = "XValue" // trim(adjustl(FormatString))    !< construct XValue1, XValue2, depending on k


                                    !< compare both strings: are they identical ????
                                    EqualFlag = .false.
                                    if (len_trim(helpString) == len_trim(ParameterName(NumInputFile,i,j))) then
                                        EqualFlag = .true.
                                        Do l = 1, len_trim(helpString)
                                            if (helpString(l:l) /= ParameterName(NumInputFile,i,j)(l:l)) then
                                                EqualFlag = .false.
                                                exit
                                            endif
                                        end Do
                                    endif
                                    if (EqualFlag) then
                                        value = posdataexp(k)
                                    endif
                                end Do
                            endif


                        elseif (index(ParameterName(NumInputFile,i,j),"NumberXValueAll") /= 0 .and. CalculationMethod == "atonce") then
                            MAGIXCommandFlag = .true.
                            NonNumberFlag = .true.                                          !<     ParameterIndex a number or a string?
                            TempString = ""
                            Do k = 1, NumberExpFiles
                                write(helpString,*) lengthexpdata(k)
                                if (k > 1) then
                                    TempString = trim(adjustl(TempString)) // ", " // trim(adjustl(helpString))
                                else
                                    TempString = trim(adjustl(helpString))
                                endif
                            end Do

                            ! Debug:
                            ! print*,' '
                            ! print*,'TempString = ', trim(adjustl(TempString)), "<<"


                        !< write number of x-points to input file (if calculation method "atonce" is chosen)
                        elseif (index(ParameterName(NumInputFile,i,j),"NumberXValue") /= 0 &
                            .and. index(ParameterName(NumInputFile,i,j),"NumberXValueAll") == 0 .and. CalculationMethod == "atonce") then
                            MAGIXCommandFlag = .true.
                            if (dataexpflag) then
                                Do k = 1, lenposdatexp                                      !< special case: value of the "x"-column values
                                    write(FormatString,'(I5)') k
                                    helpString = "NumberXValue" // trim(adjustl(FormatString))  !< construct NumberXValue1, NumberXValue2, depending on k


                                    !< remove '[[' from file name
                                    l = index(ParameterName(NumInputFile,i,j),'[[',back = .true.)
                                    if (l /= 0) then
                                        helpString2 = trim(adjustl(ParameterName(NumInputFile,i,j)(:l-1)))
                                    else
                                        helpString2 = trim(adjustl(ParameterName(NumInputFile,i,j)))
                                    endif


                                    !< compare both strings: are they identical ????
                                    EqualFlag = .false.
                                    if (len_trim(helpString) == len_trim(helpString2)) then
                                        EqualFlag = .true.
                                        Do l = 1, len_trim(helpString)
                                            if (helpString(l:l) /= helpString2(l:l)) then
                                                EqualFlag = .false.
                                                exit
                                            endif
                                        end Do
                                    endif
                                    if (EqualFlag) then
                                        value = posdataexp(k)
                                    endif
                                end Do
                                TempString = ""
                            endif

                        else                                                                !< parameter is not a "x"-column value


                            !< determine posiotion of last character of leading string
                            lcPV = index(ParamVisible(NumInputFile,i,j),"|", back = .true.)
                            if (lcPV == 0) then
                                lcPV = len_trim(ParamVisible(NumInputFile,i,j))
                            else
                                lcPV = lcPV - 1
                            endif

                            ! Debug:
                            ! call GetThreadNumberForTempDir(ThreadNumber)
                            ! print*,'ThreadNumber, NumInputFile, i, j = ', ThreadNumber, NumInputFile, i, j
                            ! print*,'ParameterName(NumInputFile, i, j) = ', '>>',trim(adjustl(ParameterName(NumInputFile,i, j))), '<<'
                            ! print*,'ParamVisible(NumInputFile, i, j)(1:lcPV) = >', ParamVisible(NumInputFile, i, j)(1:lcPV),'<'


                            !< get value of current parameter
                            MAGIXCommandFlag = .false.
                            TempString = trim(adjustl(ParameterName(NumInputFile,i,j)))
                            l = len_trim(TempString)
                            if (ParamVisible(NumInputFile,i,j)(1:lcPV) == "false" .or. (TempString(1:2) == "[[" .and. TempString(l-1:l) == "]]")) then
                                dummyFlag = .true.

                                ! Debug:
                                ! print*,'trim(adjustl(TempString)) = >',trim(adjustl(TempString)),'<'
                            else
                                call DetermineParameterValue(ParameterIndex, ParameterName(NumInputFile, i, j), dataexpflag)
                                if (FitParameterValue(ParameterIndex) /= "nUMnUMnUMnUM") then   !< is value of the parameter with index
                                    NonNumberFlag = .true.                                  !< ParameterIndex a number or a string?
                                else
                                    value = modelparam(ParameterIndex)
                                endif
                            endif
                        endif


                        !< write value with format to string
                        if (dataexpflag .or. ((.not. dataexpflag) .and. (.not. MAGIXCommandFlag))) then


                            !< determine posiotion of last character of leading string
                            lcPV = index(ParamVisible(NumInputFile, i, j),"|", back = .true.)
                            if (lcPV == 0) then
                                lcPV = len_trim(ParamVisible(NumInputFile, i, j))
                            else
                                lcPV = lcPV - 1
                            endif
                            if (ParamVisible(NumInputFile, i, j)(1:lcPV) == "true") then


                                !< get paramter value
                                if (dummyFlag) then
                                    TempString = ""
                                else
                                    if (NonNumberFlag) then
                                        ! TempString = trim(adjustl(FitParameterValue(ParameterIndex)))
                                        if (.not. MAGIXCommandFlag) TempString = FitParameterValue(ParameterIndex)
                                    else
                                        if (index(ParameterFormat(NumInputFile,i,j),'I') /= 0 .or. index(ParameterFormat(NumInputFile,i,j),'i') /= 0) then
                                            write(TempString, ParameterFormat(NumInputFile, i, j)) int(value)
                                        else
                                            ValueDouble = 0.d0
                                            ValueDouble = value
                                            write(TempString, ParameterFormat(NumInputFile, i, j)) value
                                        endif
                                    endif
                                endif


                                !< determine posiotion of last character of leading string
                                lcLS = index(LeadingString(NumInputFile,i,j),"|", back = .true.)
                                if (lcLS == 0) then
                                    lcLS = len_trim(LeadingString(NumInputFile,i,j))
                                else
                                    lcLS = lcLS - 1
                                endif

                                ! Debug:
                                ! print*,'>',LeadingString(NumInputFile, i, j),'<'


                                !< determine posiotion of last character of trailing string
                                lcTS = index(TrailingString(NumInputFile, i, j), "|", back = .true.)
                                if (lcTS == 0) then
                                    lcTS = len_trim(TrailingString(NumInputFile, i, j))
                                else
                                    lcTS = lcTS - 1
                                endif


                                !< write parameter
                                write(channel,'(A,$)') LeadingString(NumInputFile, i, j)(1:lcLS) // trim(TempString) &
                                                        // TrailingString(NumInputFile, i, j)(1:lcTS)
                                ! Debug:
                                !    print*,'>',ThreadNumber,LeadingString(NumInputFile,i,j)(1:lcLS) // trim(TempString) &
                                !                          // TrailingString(NumInputFile,i,j)(1:lcTS),'<'
                                !    print '(A,$)',LeadingString(NumInputFile,i,j)(1:lcLS) // trim(TempString) &
                                !                           // TrailingString(NumInputFile,i,j)(1:lcTS)
                            endif
                        endif
                    end Do
                    if (NumberParamPerLine(NumInputFile,i) > 0) write(channel,'(" ")')
                end Do
                return
            end subroutine WriteParameter



            !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
            !
            !   subroutines for calculation method: 'at once'
            !
            !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++



            !*********************************************************************************************************************************************
            !> subroutine: WriteDataForFitFunction
            !>
            !> write all experimental data points to file (only x-column values)
            !>
            !>
            !> input variables:     None
            !>
            !> output variables:    None
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 14.10.2011
            !>
            subroutine WriteDataForFitFunction
                !< write all experimental data points to file (only x-column values)

                implicit none
                integer :: i, NumFile                                                       !< working variables
                integer :: ThreadNumber                                                     !< current thread number
                integer :: tempChannelLocal                                                 !< working variable
                integer :: FirstLeftBlank, LastSlash                                        !< working variables
                character(len=4) :: filenameending                                          !< format of the file name
                character(len=512) :: LongHelpString1, LongHelpString2                      !< working variables
                character(len=512) :: InputDataPathLocal                                    !< path for data files
                character(len=512) :: WorkingDirectory                                      !< string containing current working directory


                !< determine format of the input file for data
                i = index(InputDataPath,'.',back = .true.)
                if (i /= 0) then
                    filenameending = trim(adjustl(InputDataPath(i+1:)))
                endif
                tempChannelLocal = tempChannel


                !< if parallelization is enabled, extend InputDataPath variable
                InputDataPathLocal = trim(adjustl(InputDataPath))
                tempChannelLocal = DataForFitFunctionChannel                            !< get number of channel for data
                ThreadNumber = 0                                                        !< default value for thread number
                call GetThreadNumberForTempDir(ThreadNumber)                            !< get thread number for temp. directory
                tempChannelLocal = tempChannelLocal + ThreadNumber                      !< write thread number to string
                write(LongHelpString1,'(I30)') JobID                                    !< write JobID to string
                write(LongHelpString2,'(I30)') ThreadNumber                             !< write thread number to string
                WorkingDirectory = trim(adjustl(trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(LongHelpString1)) // "/" &
                                   // trim(adjustl(LongHelpString2)) // "/"))


                !< change InputDataPath
                FirstLeftBlank = index(trim(adjustl(InputDataPathLocal)),' ')           !< determine the first blank in the string CommandLine
                                                                                        !< counted from the left side without leading blanks
                if (FirstLeftBlank == 0) then                                           !< if no blank found, set variable to total length of string
                    FirstLeftBlank = len(trim(adjustl(InputDataPathLocal))) + 1
                endif
                LastSlash = index(trim(adjustl(InputDataPathLocal(:FirstLeftBlank-1))),'/', back = .true.)
                if (LastSlash == 0) then                                                !< if no slash occurs on the left side of the model-exe
                    LastSlash = 0                                                       !< command
                endif
                InputDataPathLocal = trim(adjustl(WorkingDirectory)) // trim(adjustl(InputDataPathLocal(LastSlash+1:FirstLeftBlank-1)))

                ! Debug:
                ! print*,'-> InputDataPathLocal = ',trim(adjustl(InputDataPathLocal)),'<-'


                !-----------------------------------------------------------------------------------------------------------------------------------------
                !< write all exp. data x-column values in one big file
                open(tempChannelLocal, file = trim(adjustl(InputDataPathLocal)))            !< open output file
                Do NumFile = 1, NumberExpFiles                                              !< loop over all exp. data files
                    Do i = 1, lengthexpdata(NumFile)                                        !< loop over all lines
                        write(tempChannelLocal,*) expdatax(NumFile,i,:NumberXColumns(NumFile))
                    end Do
                end Do
                close(tempChannelLocal)                                                     !< close file
                return
            end subroutine WriteDataForFitFunction


            !*********************************************************************************************************************************************
            !> subroutine: ReadDataOfFitFunction
            !>
            !> reads values of the fit function
            !>
            !>
            !> input variables:     NumOutputFiles:     number of output files
            !>                      NumThreads:         max number of threads
            !>                      NumFile:            number of experimental files
            !>                      MaxL:               max number of lines of all experimental files
            !>                      MaxCol:             max number of columns of all experimental files
            !>                      colx:               number of columns in experimental x data
            !>
            !> output variables:    AtOnceFunctionLocal: array containing the fit model function
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 14.10.2011
            !>
            subroutine ReadDataOfFitFunction(AtOnceFunctionLocal, NumOutputFiles, NumThreads, NumFileOrig, MaxL, MaxCol, colx)
                !< reads values of the fit function

                implicit none
                integer :: NumFileOrig                                                      !< number of experimental files
                integer :: MaxL                                                             !< max number of lines of all experimental files
                integer :: MaxCol                                                           !< max number of columns of all experimental files
                integer :: colx                                                             !< number of columns in experimental x data
                integer :: open_stat                                                        !< status for open output file
                integer :: deallocstatus, allocstatus                                       !< status variables for allocation and deallocation
                integer :: NumberLinesOutputFiles                                           !< number of lines in the current output file
                integer :: NumberLinesOutputFilesTotal                                      !< total number of lines in current output file (incl. header)
                integer :: NumOutputFiles                                                   !< copy of NumberOutputFiles
                integer :: NumThreads                                                       !< max number of threads (processors)
                integer :: i, j, k, ii, kk, jj, xc, NumOut, yColumn                         !< working variables
                integer :: ieff, NumOutEff, lengthsum                                       !< working variables
                integer :: io_err                                                           !< error number of i/o traffic
                integer :: ChannelLocal                                                     !< channel number for output file
                integer :: ThreadNumber                                                     !< current thread number
                integer :: OutputLineCounter                                                !< counter for lines in output file(s)
                integer :: NumberPoints                                                     !< working variable
                integer :: first_element_index                                              !< index of the first index
                integer :: startLoop, endLoop, stepLoop                                     !< working variables
                integer :: shortest_distance_index                                          !< used for interpolation method: nearest neighbour
                integer :: second_shortest_distance_index                                   !< used for interpolation method: linear
                integer, dimension(0:NumThreads - 1, NumOutputFiles) :: counter_lines_output    !< counter of lines for each exp. file
                real*8 :: d1, d2, d3, y1, y2                                                !< working variable for interpolation routine
                real*8 :: interpol_value                                                    !< working variable for interpolation routine
                real*8 :: distance                                                          !< used for interpolation method: nearest neighbour
                real*8 :: shortest_distance                                                 !< used for interpolation method: nearest neighbour
                real*8 :: second_shortest_distance                                          !< used for interpolation method: nearest neighbour
                real*8 :: shortest_distance_value, second_shortest_distance_value, grad     !< used for interpolation method: linear
                real*8 :: IntegralValue1, IntegralValue2                                    !< values of integration
                real*8 :: FirstPointValueOutputFile, LastPointValueOutputFile               !< value of first point and last point
                real*8, dimension(colx) :: FirstPointOutputFile                             !< first point (lowest value of x-column vector)
                real*8, dimension(colx) :: LastPointOutputFile                              !< last point (highest value of x-column vector)
                real*8, dimension(MaxCol) :: yValuesArray                                   !< array for reading all model function values of one line
                real*8, allocatable, dimension(:, :) :: xPointsModel_expanded               !< x-column values of the output file(s)
                                                                                            !< (used only for OnlyYColumn == .false.)
                real*8, allocatable, dimension(:) :: valuesModel_expanded                   !< model function values expanded
                                                                                            !< (used only for OnlyYColumn == .false.)
                real*8, dimension(0:NumThreads-1, NumFileOrig, MaxCol, MaxL) :: AtOnceFunctionLocal    !< array containing the fit model function
                !character(len=3) :: DebugString                                            !< used for debug
                character(len=30) :: NumberString                                           !< used for number to string conversion
                character(len=8192) :: OutputFile                                           !< output file name
                character(len=8192) :: WorkingDirectory, CommandString                      !< working string variables
                character(len=8192) :: LongHelpString1, LongHelpString2                     !< working string variables
                character(len=8192) :: OutputLineString, OutputLineStringPart               !< strings for reading line in output file
                logical :: cover_flag                                                       !< used for resampling: flag for model function
                logical :: resampling_flag                                                  !< flag for resampling
                logical :: EmptyFile_flag                                                   !< flag for empty output files
                logical :: Output_reversed_order_flag                                       !< flag for reversed order


                !< expand channel number depending on the number of the thread
                ChannelLocal = ReadDataForFitFunctionChannel
                Do i = 1, 8192
                    WorkingDirectory(i:i) = " "
                end Do
                ThreadNumber = 0                                                            !< default value for thread number
                call GetThreadNumberForTempDir(ThreadNumber)                                !< get thread number for temp. directory
                ChannelLocal = ChannelLocal + ThreadNumber                                  !< write thread number to string
                write(LongHelpString1,'(I30)') JobID                                        !< write JobID to string
                write(LongHelpString2,'(I30)') ThreadNumber                                 !< write thread number to string
                WorkingDirectory = trim(adjustl(trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(LongHelpString1)) // "/" &
                                                                                       // trim(adjustl(LongHelpString2)) // "/"))

                !< get current thread number. The following lines are neccessary, because for MPI we need only one dimension for arrays
                !< 'counter_lines_output' and 'AtOnceFunctionLocal'
                call GetThreadNumber(ThreadNumber)


                !< read fit data from output file(s)
                FirstPointValueOutputFile = 0.d0                                            !< reset FirstPointValueOutputFile
                LastPointValueOutputFile = 0.d0                                             !< reset LastPointValueOutputFile
                counter_lines_output(ThreadNumber, :) = 0                                   !< reset counter for lines in output file(s)
                first_element_index = 0                                                     !< reset index for first element
                AtOnceFunctionLocal(ThreadNumber, :, :, :) = 0.d0                           !< reset final output array
                Do NumOut = 1, NumOutputFiles                                               !< loop over all output files
                    NumOutEff = NumOut                                                      !< if all output is stored in only one output file,
                                                                                            !< the effective output file index NumOutEff has to be
                                                                                            !< determined for all arrays/variables, which are indexed
                                                                                            !< in conjunction with the experimental data files, i.e. for
                                                                                            !< the variables expdatax etc. But, variables which belong to
                                                                                            !< the real given output file, like CharacterForComments has
                                                                                            !< to be indexed with the current output file index (NumOut).

                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< determine format of the input file
                    if (NumOutputFilesEqualNumExpFiles) then
                        i = index(FitFktOutput(NumOut), '.', back = .true.)
                        OutputFile = FitFktOutput(NumOut)
                        write(NumberString,'(I30)') NumOut
                        OutputFile = OutputFile(:i-1) // "_" // trim(adjustl(NumberString)) // OutputFile(i:)

                        ! Debug:
                        ! print*,' '
                        ! print*,'i,NumOut  = ', i, NumOut
                        ! print*,'NumberString = ', trim(adjustl(NumberString))
                    else
                        OutputFile = FitFktOutput(NumOut)
                    endif


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< write data to destinated file, if output file has fits-format, use fits2dat conversion module to create ascii-file format
                    if (OutputFileFormat(NumOut) == 2) then


                        !< call fits2dat converter
                        if (OnlyYColumn(NumOut)) then
                            CommandString = "(cd " // trim(adjustl(WorkingDirectory)) // "; ./fits2dat.py y " // char(34) &
                                                                    // trim(adjustl(OutputFile))// char(34) // ")"
                        else
                            CommandString = "(cd " // trim(adjustl(WorkingDirectory)) // "; ./fits2dat.py n " // char(34) &
                                                                    // trim(adjustl(OutputFile)) // char(34) // ")"
                        endif
                        call system(trim(adjustl(CommandString)))


                        !< rename OutputFile variable
                        i = index(OutputFile, '.', back = .true.)
                        OutputFile = trim(adjustl(OutputFile(:i))) // "dat"                 !< define output file name
                    endif

                    ! Debug:
                    ! print*,'WorkingDirectory = ',trim(adjustl(WorkingDirectory))
                    ! print*,'OutputFile = ',trim(adjustl(OutputFile))
                    ! print*,'FitFktOutput(NumOut) = ',trim(adjustl(FitFktOutput(NumOut)))


                    !< make path of output file absolute
                    OutputFile = trim(adjustl(WorkingDirectory)) // trim(adjustl(OutputFile))


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< open (NumOut)th output file
                    EmptyFile_flag = .false.                                                !< flag for an empty file
                    open(ChannelLocal, file = trim(adjustl(OutputFile)), iostat = open_stat)    !< open output file
                    if (open_stat > 0) then
                        EmptyFile_flag = .true.
                        write(logchannel,*)
                        write(logchannel,*)
                        write(logchannel,'(11x,"Error in subroutine ReadDataOfFitFunction:")')
                        write(logchannel,'(13x,"Can not open output file ",A)') trim(adjustl(OutputFile))
                        write(logchannel,'(13x,"Set all model function values for this output file to ",ES25.15,"!")') ValueEmptyOutputFile(NumOut)
                        write(logchannel,*)

                        print '(" ")'
                        print '(" ")'
                        print '(11x,"Error in subroutine ReadDataOfFitFunction:")'
                        print '(13x,"Can not open output file ",A)', trim(adjustl(OutputFile))
                        print '(13x,"Set all model function values for this output file to ",ES25.15,"!")', ValueEmptyOutputFile(NumOut)
                        print '(" ")'
                    endif
                    if (.not.EmptyFile_flag) then                                           !< continue here, if output file is not empty


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        !< determine number of lines in the current (NumOut)th output file
                        NumberLinesOutputFilesTotal = 0
                        Do
                            read(ChannelLocal,'(A)',iostat = io_err) OutputLineString
                            if (io_err /= 0) then
                                exit
                            else


                                !< ignore empty lines and all lines starting with a comment character
                                if (trim(adjustl(OutputLineString)) /= "") then
                                    if (trim(adjustl(CharacterForComments(NumOut))) == "") then
                                        NumberLinesOutputFilesTotal = NumberLinesOutputFilesTotal + 1
                                    else
                                        ii = index(OutputLineString, trim(adjustl(CharacterForComments(NumOut))))
                                        if (ii /= 1) then
                                            NumberLinesOutputFilesTotal = NumberLinesOutputFilesTotal + 1
                                        endif
                                    endif
                                endif
                            endif
                        end Do


                        !< correct the number of lines in the output file(s)
                        if (AllInOneOutputFile) then
                            NumberLinesOutputFiles = (NumberLinesOutputFilesTotal - NumberHeaderLines(NumOut))
                        else


                            !< check if length of ouput file is longer then the experimental data file
                            i = (NumberLinesOutputFilesTotal - NumberHeaderLines(NumOut))

                            ! Debug:
                            ! print*,'i, lengthexpdata(NumOut), NumberHeaderLines(NumOut) = ', i, lengthexpdata(NumOut), NumberHeaderLines(NumOut)

                            if (lengthexpdata(NumOut) < i .and. OnlyYColumn(NumOut)) then
                                NumberLinesOutputFilesTotal = min(lengthexpdata(NumOut), i) + NumberHeaderLines(NumOut)
                            endif
                            NumberLinesOutputFiles = NumberLinesOutputFilesTotal - NumberHeaderLines(NumOut)
                        endif

                        ! Debug:
                        ! print*,'NumberLinesOutputFiles = ', NumberLinesOutputFiles


                        !< substract the number of header lines
                        if (NumberLinesOutputFiles == 0) then
                            EmptyFile_flag = .true.
                        endif

                        ! Debug:
                        ! print*,'NumberLinesOutputFilesTotal = ', NumberLinesOutputFilesTotal
                        ! print*,'NumberHeaderLines(NumOut) = ', NumberHeaderLines(NumOut)
                        ! print*,'NumberLinesOutputFiles = ', NumberLinesOutputFiles
                        ! print*,'NumOut = ', NumOut
                        ! print*,'lengthexpdata(NumOut) = ', lengthexpdata(NumOut)
                        ! stop
                    endif


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< if output file is empty fill part of AtOnceFunctionLocal with value for empty output files
                    if (EmptyFile_flag) then                                                !< the output file is empty
                        if (AllInOneOutputFile) then
                            AtOnceFunctionLocal(ThreadNumber, :, :, :) = ValueEmptyOutputFile(NumOut)
                        else
                            AtOnceFunctionLocal(ThreadNumber, NumOut, :, :) = ValueEmptyOutputFile(NumOut)
                        endif

                        ! Debug:
                        ! print*,'ValueEmptyOutputFile(NumOut) = ', ValueEmptyOutputFile(NumOut)
                    else


                        !<--------------------------------------------------------------------------------------------------------------------------------
                        OutputLineCounter = 0                                               !< reset line counter for AtOnceFunctionLocal
                        Do yColumn = 1, NumberYColumns(NumOut)                              !< loop over all y-columns in the current output file
                                                                                            !< Note, here it is assumed, that the number of y-columns
                                                                                            !< in the (NumOut)th output file is identical to the number
                                                                                            !< of y-columns in the (NumOut)th experimental data file
                            !< get current thread number. Herem we do not use subroutine GetThreadNumber because for MPI we need only one dimension
                            !< for arrays 'xPointsModel_output' and 'valuesModel_output'
                            call GetThreadNumber(ThreadNumber)


                            !< read current (jth) output file
                            resampling_flag = .false.                                       !< reset general resampling flag


                            !<----------------------------------------------------------------------------------------------------------------------------
                            ! if (OutputFileFormat(NumOut) == -2) then                        !< option deactivated at the moment

                                !< read fits file
                                ! call ReadFitsFile(colx, NumOut, OutputFile, counter_lines_output(ThreadNumber, NumOut), resampling_flag, &
                                !                      FirstPointOutputFile, LastPointOutputFile, FirstPointValueOutputFile, LastPointValueOutputFile)
                                ! NumberLinesOutputFiles = counter_lines_output(ThreadNumber, NumOut)

                                ! Debug:
                                ! print*,'resampling_flag = ',resampling_flag
                                ! print*,'FirstPointOutputFile = ', FirstPointOutputFile
                                ! print*,'LastPointOutputFile = ', LastPointOutputFile
                                ! print*,'FirstPointValueOutputFile = ', FirstPointValueOutputFile
                                ! print*,'LastPointValueOutputFile = ', LastPointValueOutputFile
                                ! stop
                            ! else


                                !< (deallocate and) allocate memory for output variables
                                if (.not.OnlyYColumn(NumOut)) then
                                    if (allocated(xPointsModel_output)) then
                                        deallocate(xPointsModel_output, valuesModel_output, stat = deallocstatus)
                                        if (deallocstatus /= 0) then
                                            write(logchannel,*)
                                            write(logchannel,'(11x,"Error in subroutine ReadDataOfFitFunction:")')
                                            write(logchannel,'(13x,"Can not deallocate variables xPointsModel_output etc.")')
                                            write(logchannel,'(13x,"Please close all other programs and restart the program!")')
                                            write(logchannel,*)
                                            write(logchannel,'(13x,"deallocstatus = ",I4)') deallocstatus
                                            write(logchannel,'(" ")')
                                            write(logchannel,'(11x,"Program aborted!")')
                                            close(logchannel)

                                            print '(" ")'
                                            print '(11x,"Error in subroutine ReadDataOfFitFunction:")'
                                            print '(13x,"Can not deallocate variables xPointsModel_output etc.")'
                                            print '(13x,"Please close all other programs and restart the program!")'
                                            print '(" ")'
                                            print '(13x,"deallocstatus = ",I4)',deallocstatus
                                            print '(" ")'
                                            stop '           Program aborted!'
                                        endif
                                    endif
                                    allocate(xPointsModel_output(0:NumberOfUsedThreads - 1, NumberLinesOutputFiles, colx), &
                                             valuesModel_output(0:NumberOfUsedThreads - 1, NumberLinesOutputFiles), &
                                             stat = allocstatus)
                                    if (allocstatus /= 0) then
                                        write(logchannel,'(" ")')
                                        write(logchannel,'(11x,"Error in subroutine ReadDataOfFitFunction:")')
                                        write(logchannel,'(13x,"Can not allocate variables xPointsModel_output etc.")')
                                        write(logchannel,'(13x,"Please close all other programs and restart the program!")')
                                        write(logchannel,'(" ")')
                                        write(logchannel,'(13x,"allocstatus = ",I4)') allocstatus
                                        write(logchannel,'(" ")')
                                        write(logchannel,'(11x,"Program aborted!")')
                                        close(logchannel)

                                        print '(" ")'
                                        print '(11x,"Error in subroutine ReadDataOfFitFunction:")'
                                        print '(13x,"Can not allocate variables xPointsModel_output etc.")'
                                        print '(13x,"Please close all other programs and restart the program!")'
                                        print '(" ")'
                                        print '(13x,"allocstatus = ",I4)',allocstatus
                                        print '(" ")'
                                        stop '           Program aborted!'
                                    endif
                                    xPointsModel_output(ThreadNumber, :, :) = 0.d0
                                    valuesModel_output(ThreadNumber, :) = 0.d0
                                    FirstPointOutputFile(:) = 1.d99
                                    FirstPointValueOutputFile = 0.d0
                                    LastPointOutputFile(:) = -1.d99
                                    LastPointValueOutputFile = 0.d0
                                endif


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< read each line of the current (NumOut)th output file
                                rewind ChannelLocal                                         !< go back to the beginning of the current output file
                                ieff = 0                                                    !< reset counter of meaningful lines
                                Do i = 1, NumberLinesOutputFilesTotal                       !< loop over all lines in the current input file


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< In order to save memory the output file is read for each y-column!
                                    read(ChannelLocal,'(A)', iostat = io_err) OutputLineString
                                    if (io_err > 0) then
                                        write(logchannel,*)
                                        write(logchannel,'(11x,"Error in subroutine ReadDataOfFitFunction:")')
                                        write(logchannel,'(13x,"Can not read line ", I5, " of the ", I4, "th model output file!")') i, NumOut
                                        write(logchannel,'(13x,"Error Code: io_err = ", I15)') io_err
                                        write(logchannel,*)
                                        write(logchannel,'(13x,"Assume empty line!")')

                                        print '(" ")'
                                        print '(" ")'
                                        print '(" ")'
                                        print '(11x,"Error in subroutine ReadDataOfFitFunction:")'
                                        print '(13x,"Can not read line ", I5 ," of the ", I4, "th model output file!")', i, NumOut
                                        print '(13x,"Error Code: io_err = ", I15)', io_err
                                        print '(" ")'
                                        print '(13x,"Assume empty line!")'
                                        OutputLineString = ""
                                    endif


                                    !< check, if line starts with comment character
                                    ii = 0
                                    if (trim(adjustl(CharacterForComments(NumOut))) /= "") then
                                        ii = index(OutputLineString, trim(adjustl(CharacterForComments(NumOut))))
                                        if (ii == (-1) .or. ii > 1) then
                                            ii = 2
                                        endif
                                    endif

                                    ! Debug:
                                    ! print*,'ii = ', ii


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< ignore header lines, empty lines, and lines starting with a comment character
                                    if (i > NumberHeaderLines(NumOut) .and. trim(adjustl(OutputLineString)) /= "" .and. ii /= 1) then
                                        ieff = ieff + 1                                     !< increase counter for meaningful lines

                                        ! Debug:
                                        ! print*,'ieff = ', ieff


                                        !<----------------------------------------------------------------------------------------------------------------
                                        !< increase line counter for AtOnceFunctionLocal
                                        if (AllInOneOutputFile) then                        !< if all output is stored in one big output file then the
                                                                                            !< whole thing gets pretty complicate !
                                            lengthsum = 0
                                            NumOutEff = 1
                                            if (ieff > lengthexpdata(1)) then
                                                Do While (lengthsum <= ieff .and. NumOutEff < NumFileOrig)
                                                    if (ieff > lengthsum + lengthexpdata(NumOutEff)) then
                                                        lengthsum = lengthsum + lengthexpdata(NumOutEff)
                                                        if (ieff > lengthsum) then
                                                            NumOutEff = NumOutEff + 1
                                                        endif
                                                    else
                                                        exit
                                                    endif
                                                end Do
                                            endif
                                            if (NumOutEff > NumFileOrig) then
                                                exit
                                            else
                                                OutputLineCounter = ieff - lengthsum
                                            endif

                                            ! Debug:
                                            ! print*,'ieff = ', ieff
                                            ! print*,'lengthsum = ', lengthsum
                                            ! print*,'NumOutEff = ', NumOutEff
                                            ! print*,'lengthexpdata(NumOutEff) = ', lengthexpdata(NumOutEff)
                                            ! print*,"OutputLineCounter = ", OutputLineCounter
                                        else
                                            OutputLineCounter = OutputLineCounter + 1       !< increase counter for all lines in all output file
                                        endif


                                        !< remove comments
                                        OutputLineStringPart = OutputLineString
                                        if (trim(adjustl(CharacterForComments(NumOut))) /= "") then
                                            ii = index(OutputLineString, trim(adjustl(CharacterForComments(NumOut))))
                                            if (ii > 0) then
                                                OutputLineStringPart = OutputLineString(:(ii - 1))
                                            endif
                                        endif

                                        ! Debug:
                                        ! if ((i/100.) == int(i/100.)*1.) print*,'>>',i, trim(adjustl(OutputLineString))


                                        !< replace NaN in the current line of the current outfile
                                        ii = index(OutputLineStringPart,'NaN')
                                        if (ii > 0) then

                                            ! Debug:
                                            ! print*,'>>',i,'>',OutputLineStringPart,'<'

                                            k = ii
                                            Do While (k > 0)
                                                if (k > 0) then
                                                    OutputLineStringPart = OutputLineString(:(ii - 1)) // trim(adjustl(NaNReplaceString)) &
                                                                                                            // OutputLineString((ii + 3):)
                                                    k = index(OutputLineStringPart(ii:),'NaN')
                                                    if (k > 0) ii = ii + k
                                                endif
                                            end Do

                                            ! Debug:
                                            ! print*,'->',i,'>',OutputLineStringPart,'<'
                                            ! print*,'########################################################################################'
                                        endif


                                        !<----------------------------------------------------------------------------------------------------------------
                                        !< read only Y-columns
                                        if (OnlyYColumn(NumOut)) then
                                            if (OutputLineCounter <= MaxL) then             !< take care, that no error occurs, if output file
                                                                                            !< conatains more lines than expected
                                                yValuesArray = 0.d0
                                                read(OutputLineStringPart,*) yValuesArray(1:NumberYColumns(NumOutEff))
                                                AtOnceFunctionLocal(ThreadNumber, NumOutEff, yColumn, OutputLineCounter) = yValuesArray(yColumn)

                                                ! Debug:
                                                ! print*,'NumOutEff, yColumn, OutputLineCounter, ThreadNumber = ', NumOutEff, yColumn, &
                                                !         OutputLineCounter, ThreadNumber
                                                ! print*,'yValuesArray(1:NumberYColumns(NumOutEff)) = ', yValuesArray(1:NumberYColumns(NumOutEff))
                                                ! print*,'yColumn, yValuesArray(yColumn) = ', yColumn, yValuesArray(yColumn)
                                                ! print*,'AtOnceFunctionLocal(ThreadNumber, NumOutEff, yColumn, OutputLineCounter) = ', &
                                                !         AtOnceFunctionLocal(ThreadNumber, NumOutEff, yColumn, OutputLineCounter)
                                                ! print*,'#######################################################################################'
                                            endif


                                        !<----------------------------------------------------------------------------------------------------------------
                                        !< read x and y-columns
                                        else
                                            k = NumberXColumns(NumOutEff)
                                            read(OutputLineStringPart,*) xPointsModel_output(ThreadNumber, ieff, 1:k), &
                                                                         yValuesArray(1:NumberYColumns(NumOutEff))
                                            valuesModel_output(ThreadNumber, ieff) = yValuesArray(yColumn)

                                            ! Debug:
                                            ! print*,'>',trim(adjustl(OutputLineStringPart)),'<',k
                                            ! print*,NumOutEff, yColumn, yValuesArray(1:NumberYColumns(NumOutEff))
                                            ! print*,'-->', ieff, xPointsModel_output(ThreadNumber, ieff, 1:k), valuesModel_output(ThreadNumber, ieff)
                                            ! stop


                                            !<------------------------------------------------------------------------------------------------------------
                                            !< do a lsr adjustment if desired
                                            if (LSRAdjustementFitFlag(NumOut)) then         !< if LSRAdjustementFitFlag is true get lsr adjustement
                                                write(NumberString,'(I30)') NumOut          !< from parameter list
                                                LongHelpString1 = "MAGIXOutputLSRAdjustementName" // trim(adjustl(NumberString))
                                                Do k = 1, parameternumber
                                                    if (trim(adjustl(FitParameterName(k))) == trim(adjustl(LongHelpString1))) then
                                                        LSRAdjustement(NumOut) = paramset(0, k)
                                                        exit
                                                    endif
                                                end Do

                                                ! Debug:
                                                ! print*,'NumOut = ', NumOut
                                                ! print*,'LSRAdjustement(NumOut) = ', LSRAdjustement(NumOut)
                                                ! stop
                                            endif
                                            if (LSRAdjustement(NumOut) /= 0.d0) then
                                                Do k = 1, NumberXColumns(NumOutEff)
                                                    xPointsModel_output(ThreadNumber, ieff, k) = xPointsModel_output(ThreadNumber, ieff, k) &
                                                                                                                 - LSRAdjustement(NumOut)
                                                end Do
                                            endif


                                            !<------------------------------------------------------------------------------------------------------------
                                            !< determine first and last point of the output file array
                                            ii = 0
                                            jj = 0
                                            Do k = 1, NumberXColumns(NumOutEff)
                                                if (xPointsModel_output(ThreadNumber, ieff, k) <= FirstPointOutputFile(k)) then
                                                    ii = ii + 1
                                                endif
                                                if (xPointsModel_output(ThreadNumber, ieff, k) >= LastPointOutputFile(k)) then
                                                    jj = jj + 1
                                                endif
                                            end Do
                                            if (ii == NumberXColumns(NumOutEff)) then
                                                FirstPointOutputFile(1:NumberXColumns(NumOutEff)) = xPointsModel_output(ThreadNumber, ieff, &
                                                                                                                 1:NumberXColumns(NumOutEff))
                                                FirstPointValueOutputFile = valuesModel_output(ThreadNumber, ieff)
                                            endif
                                            if (jj == NumberXColumns(NumOutEff)) then
                                                LastPointOutputFile(1:NumberXColumns(NumOutEff)) = xPointsModel_output(ThreadNumber, ieff, &
                                                                                                                1:NumberXColumns(NumOutEff))
                                                LastPointValueOutputFile = valuesModel_output(ThreadNumber, ieff)
                                            endif


                                            !<------------------------------------------------------------------------------------------------------------
                                            !< do a x-column check if OnlyYColumn == .false.
                                            if (.not.resampling_flag) then
                                                ii = 0
                                                if (OutputLineCounter <= MaxL) then
                                                    Do k = 1, NumberXColumns(NumOutEff)
                                                        if (dabs(xPointsModel_output(ThreadNumber, ieff, k) &
                                                            - expdatax(NumOutEff, OutputLineCounter, k)) < 1.d-4) then
                                                            ii = ii + 1
                                                        else
                                                            exit
                                                        endif
                                                    end Do
                                                    if (ii /= NumberXColumns(NumOutEff)) then
                                                        resampling_flag = .true.                !< set general resampling flag

                                                        ! Debug:
                                                        ! print*,'NumOut, yColumn, ii, NumberXColumns(NumOut), ieff = ', NumOut, yColumn, ii, &
                                                        !                                                          NumberXColumns(NumOut), ieff
                                                        ! stop
                                                    endif
                                                endif
                                            endif                                           !< continue here, if resampling flag is already set
                                        endif                                               !< continue here, if OnlyYColumn flag is set or not
                                    endif                                                   !< continue here, if lines are ignored (i.e. empty etc.)
                                end Do                                                      !< loop over all lines in the current input file
                            ! endif                                                         !< continue here, if deactivated fits import routine is used
                            !                                                               !< again
                            counter_lines_output(ThreadNumber, NumOut) = ieff               !< save number of meaningful lines in the current output file

                            ! Debug:
                            ! print*,'NumOut, NumOutEff, yColumn, resampling_flag = ', NumOut, NumOutEff, yColumn, resampling_flag, &
                            !                                                                     valuesModel_output(ThreadNumber, 1)
                            ! print*,'OutputLineCounter, MaxL = ', OutputLineCounter, MaxL
                            ! print*,xPointsModel_output(ThreadNumber, 1:5, :)
                            ! print*,expdatax(NumOutEff, 1:5, :)
                            ! print*,'#############################################################'
                            ! Do k = 1, NumOutputFiles
                            !     Do i = 1,lengthexpdata(k)
                            !         print*,'>',expdatax(k, i, :NumberXColumns(k))
                            !     end Do
                            !     print*,'###############'
                            ! end Do
                            ! print*,' '
                            ! write(DebugString,'(I3)') NumOut
                            ! open(368,file='/home/moeller/CATS/magix/test'//trim(adjustl(DebugString))//'_raw.txt')
                            ! Do j = 1, counter_lines_output(ThreadNumber, NumOut)
                            !   write(368,'(9999(F20.10))') xPointsModel_output(ThreadNumber, j,:NumberXColumns(NumOut)), &
                            !   valuesModel_output(ThreadNumber, j)
                            ! end Do
                            ! close(368)


                            !<----------------------------------------------------------------------------------------------------------------------------
                            !< check, if resampling is really necessary
                            if (resampling_flag .and. AllInOneOutputFile) then              !< all output is stored in one big output file but the
                                                                                            !< x positions are not the same
                                write(logchannel,*)
                                write(logchannel,'(11x,"Error in subroutine ReadDataOfFitFunction:")')
                                write(logchannel,'(13x,"The model function values are not given at the same x-column points")')
                                write(logchannel,'(13x,"which are defined in the experimental data file(s)!")')
                                write(logchannel,*)
                                write(logchannel,'(13x,"Due to the setting in the registration xml-file, store all data")')
                                write(logchannel,'(13x,"in one big output file, interpolation is not possible!")')
                                write(logchannel,'(13x," ")')
                                write(logchannel,'(13x,"Set the tag <AllInOneOutputFile></AllInOneOutputFile>")')
                                write(logchannel,'(13x,"to no and modify your model program!")')
                                write(logchannel,'(13x," ")')
                                write(logchannel,'(13x,"For the current job, I assume an empty output file")')
                                write(logchannel,'(13x,"and set all model function values to ", ES25.15, " !")') ValueEmptyOutputFile(NumOut)

                                print '(" ")'
                                print '(" ")'
                                print '(" ")'
                                print '(11x,"Error in subroutine ReadDataOfFitFunction:")'
                                print '(13x,"The model function values are not given at the same x-column points")'
                                print '(13x,"which are defined in the experimental data file(s)!")'
                                print '(" ")'
                                print '(13x,"Due to the setting in the registration xml-file, store all data")'
                                print '(" ")'
                                print '(13x,"in one big output file, interpolation is not possible!")'
                                print '(13x,"Set the tag <AllInOneOutputFile></AllInOneOutputFile>")'
                                print '(13x,"to no and modify your model program!")'
                                print '(" ")'
                                print '(13x,"For the current job, I assume an empty output file")'
                                print '(13x,"and set all model function values to ", ES25.15, " !")', ValueEmptyOutputFile(NumOut)
                                AtOnceFunctionLocal(ThreadNumber, :, yColumn, :) = ValueEmptyOutputFile(NumOut)

                            elseif ((.not. resampling_flag) .and. (.not. OnlyYColumn(NumOut))) then   !< output file is not empty and x-column values
                                                                                            !< are at the correct positions, i.e. no interpolation /
                                if (AllInOneOutputFile) then                                !< resampling is necessary
                                    ii = 0
                                    Do kk = 1, ieff                                         !< loop over all lines of the current output file

                                            lengthsum = 0
                                            NumOutEff = 1
                                            if (kk > lengthexpdata(1)) then
                                                Do While (lengthsum <= kk .and. NumOutEff < NumFileOrig)
                                                    if (kk > lengthsum + lengthexpdata(NumOutEff)) then
                                                        lengthsum = lengthsum + lengthexpdata(NumOutEff)
                                                        if (kk > lengthsum) then
                                                            NumOutEff = NumOutEff + 1
                                                        endif
                                                    else
                                                        exit
                                                    endif
                                                end Do
                                            endif
                                            if (NumOutEff > NumFileOrig) then
                                                exit
                                            else
                                                ii = kk - lengthsum
                                            endif

                                        ! Debug:
                                        ! print*,'kk = ', kk
                                        ! print*,'lengthsum = ', lengthsum
                                        ! print*,'NumOutEff = ', NumOutEff
                                        ! print*,'lengthexpdata(NumOutEff) = ', lengthexpdata(NumOutEff)
                                        ! print*,"ii = ", ii

                                        AtOnceFunctionLocal(ThreadNumber, NumOutEff, yColumn, ii) = valuesModel_output(ThreadNumber, kk)

                                        ! Debug:
                                        ! print*,'NumOutEff, yColumn, ii, AtOnceFunctionLocal(ThreadNumber, NumOutEff, yColumn, ii) = ', &
                                        !                NumOutEff, yColumn, ii, AtOnceFunctionLocal(ThreadNumber, NumOutEff, yColumn, ii)
                                    end Do
                                else
                                    Do kk = 1, min(ieff, MaxL)                              !< loop over all lines of the current output file
                                        AtOnceFunctionLocal(ThreadNumber, NumOutEff, yColumn, kk) = valuesModel_output(ThreadNumber, kk)

                                        ! Debug:
                                        ! print*,'NumOutEff, yColumn, kk, AtOnceFunctionLocal(ThreadNumber, NumOutEff, yColumn, kk) = ', &
                                        !                NumOutEff, yColumn, kk, AtOnceFunctionLocal(ThreadNumber, NumOutEff, yColumn, kk)
                                    end Do
                                endif


                            !<----------------------------------------------------------------------------------------------------------------------------
                            !< do resampling if model function values are not at the same positions than the x-points in the experimental data and output
                            !< file is not empty
                            elseif (resampling_flag .and. (.not.AllInOneOutputFile)) then   !< resampling is necessar, but only, if not all output data
                                                                                            !< is stored in one output file. So, NumOut == NumOutEff
!---------------------------------------------------------------------------------------------------------------------------------------------------------
! muss für interpolation bei AllInOneOutputFile angepasst werden
                                !< check output file starts with the highest x-column value interchange FirstPointOutputFile and LastPointOutputFile
                                ii = 0
                                Do k = 1, NumberXColumns(NumOut)
                                    if (xPointsModel_output(ThreadNumber, 1, k) &
                                        >= xPointsModel_output(ThreadNumber, counter_lines_output(ThreadNumber, NumOut), k)) then
                                        ii = ii + 1
                                    endif
                                end Do
                                if (ii == NumberXColumns(NumOut)) then
                                    Output_reversed_order_flag = .true.
                                else
                                    Output_reversed_order_flag = .false.
                                endif
!---------------------------------------------------------------------------------------------------------------------------------------------------------


                                !< (deallocate and) allocate memory for output variables
                                if (allocated(xPointsModel_expanded)) then
                                    deallocate(xPointsModel_expanded, valuesModel_expanded, stat = deallocstatus)
                                    if (deallocstatus /= 0) then
                                        write(logchannel,*)
                                        write(logchannel,'(11x,"Error in subroutine ReadDataOfFitFunction:")')
                                        write(logchannel,'(13x,"Can not deallocate variables xPointsModel_expanded etc.")')
                                        write(logchannel,'(13x,"Please close all other programs and restart the program!")')
                                        write(logchannel,*)
                                        write(logchannel,'(13x,"deallocstatus = ", I4)') deallocstatus
                                        write(logchannel,'(" ")')
                                        write(logchannel,'(11x,"Program aborted!")')
                                        close(logchannel)

                                        print '(" ")'
                                        print '(11x,"Error in subroutine ReadDataOfFitFunction:")'
                                        print '(13x,"Can not deallocate variables xPointsModel_expanded etc.")'
                                        print '(13x,"Please close all other programs and restart the program!")'
                                        print '(" ")'
                                        print '(13x,"deallocstatus = ",I4)',deallocstatus
                                        print '(" ")'
                                        stop '           Program aborted!'
                                    endif
                                endif
                                allocate(xPointsModel_expanded(ieff + lengthexpdata(NumOut), colx), &
                                        valuesModel_expanded(ieff + lengthexpdata(NumOut)), stat = allocstatus)
                                if (allocstatus /= 0) then
                                    write(logchannel,'(" ")')
                                    write(logchannel,'(11x,"Error in subroutine ReadDataOfFitFunction:")')
                                    write(logchannel,'(13x,"Can not allocate variables xPointsModel_expanded etc.")')
                                    write(logchannel,'(13x,"Please close all other programs and restart the program!")')
                                    write(logchannel,'(" ")')
                                    write(logchannel,'(13x,"allocstatus = ", I4)') allocstatus
                                    write(logchannel,'(" ")')
                                    write(logchannel,'(11x,"Program aborted!")')
                                    close(logchannel)

                                    print '(" ")'
                                    print '(11x,"Error in subroutine ReadDataOfFitFunction:")'
                                    print '(13x,"Can not allocate variables xPointsModel_expanded etc.")'
                                    print '(13x,"Please close all other programs and restart the program!")'
                                    print '(" ")'
                                    print '(13x,"allocstatus = ", I4)',allocstatus
                                    print '(" ")'
                                    stop '           Program aborted!'
                                endif
                                xPointsModel_expanded = 0.d0
                                valuesModel_expanded = 0.d0


                                !< get current thread number. Herem we do not use subroutine GetThreadNumber because for MPI we need only one dimension
                                !< for arrays 'xPointsModel_output' and 'valuesModel_output'
                                call GetThreadNumber(ThreadNumber)


                                !<------------------------------------------------------------------------------------------------------------------------
                                !< simple resampling method
                                !<
                                !<
                                !< First of all, we have to distinguish between six different cases:
                                !< -----------------------------------------------------------------
                                !<
                                !< 1. case: The model function starts at a later x-column position than exp. data:
                                !<
                                !<                  [*****************************************************************]     = model function range
                                !<              |----|----|----|----|----|----|----|----|----|----|----|----|----|----|     = exp. data range
                                !<
                                !<          Here, the model function is extrapolated to the start of the exp data range, with the value of the model
                                !<          function at the first point of the model function range.
                                !<
                                !<
                                !< 2. case: The model function starts at a previous x-column4.654163130 position than exp. data:
                                !<
                                !<          [*************************************************************************]     = model function range
                                !<              |----|----|----|----|----|----|----|----|----|----|----|----|----|----|     = exp. data range
                                !<
                                !<          Here, the values of the model function between the start point of the model function range and the start point
                                !<          of the exp. data range will be removed.
                                !<
                                !<
                                !< 3. case: The model function ends at a previous x-column position than exp. data:
                                !<
                                !<              [**********************************************************]                = model function range
                                !<              |----|----|----|----|----|----|----|----|----|----|----|----|----|----|     = exp. data range
                                !<
                                !<          Here, the model function is extrapolated to the end of the exp data range, with the value of the model
                                !<          function at the last point of the model function range.
                                !<
                                !<
                                !< 4. case: The model function ends at a later x-column pexpdatax(NumOut, kk, k)osition than exp. data:
                                !<
                                !<              [*************************************************************************] = model function range
                                !<              |----|----|----|----|----|----|----|----|----|----|----|----|----|----|     = exp. data range
                                !<
                                !<          Here, the values of the model function between the last point of the model function range and the last point
                                !<          of the exp. data range will be removed.
                                !<
                                !<
                                !< 5. case: The model function range does not cover the exp. data range.
                                !<
                                !<          [*]                                                                             = model function range
                                !<              |----|----|----|----|----|----|----|----|----|----|----|----|----|----|     = exp. data range
                                !<
                                !<          Here, all values of the model function are set to zero.
                                !<
                                !<
                                !< 6. case: The model function range does not cover the exp. data range.
                                !<
                                !<                                                                                      [*] = model function range
                                !<              |----|----|----|----|----|----|----|----|----|----|----|----|----|----|     = exp. data range
                                !<
                                !<          Here, all values of the model function are set to zero.
                                !<
                                !<
                                !<  Remark:   - At the end of the subroutine, the variable AtOnceFunctionLocal contains the model function values for all
                                !<              x-points of the exp. data.
                                !<            - In general, the model function values are not calculated at the x-points given in the exp. data.
                                !<              Therefore,the model function have to be interpolated. The user have to define the interpolation method:
                                !<              nearest-neighbour, linear interpolation, or bicubic interpolation.
                                !<
                                !<------------------------------------------------------------------------------------------------------------------------
                                if (trim(adjustl(ResamplingMethod(NumOut))) == "simple") then
                                    xc = 0                                                  !< counter for expanded range
                                    cover_flag = .true.                                     !< set cover_flag to true, i. e. model function covers
                                                                                            !< (partly) exp. data range


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< 1. case: The model function starts at a later x-column position than exp. data:
                                    !<
                                    !<                  [*****************************************************************]     = model function range
                                    !<              |----|----|----|----|----|----|----|----|----|----|----|----|----|----|     = exp. data range
                                    !<
                                    !<          Here, the model function is extrapolated to the start of the exp data range, with the value of the model
                                    !<          function at the first point of the model function range.
                                    !<
                                    if (.not.ExpData_reversed_flag(NumOut)) then
                                        Do kk = 1, lengthexpdata(NumOut)
                                            j = 0                                           !< counter for x-columns which are identical to the last
                                                                                            !< expdatax position
                                            ii = 0                                          !< counter for x-columns which are smaller then the last
                                                                                            !< expdatax position
                                            Do k = 1, NumberXColumns(NumOut)                !< loop over all x-columns
                                                if (expdatax(NumOut, kk, k) == FirstPointOutputFile(k)) then
                                                    j = j + 1
                                                elseif (expdatax(NumOut, kk, k) < FirstPointOutputFile(k)) then
                                                    ii = ii + 1
                                                endif
                                            end Do
                                            if ((ii + j) == NumberXColumns(NumOut) .and. (j /= NumberXColumns(NumOut))) then
                                                xc = xc + 1
                                                xPointsModel_expanded(xc, :) = expdatax(NumOut, kk, :)
                                            endif
                                        end Do
                                        if (xc > 0) then                                    !< extrapolate model function to the start of the exp data
                                            valuesModel_expanded(:xc) = FirstPointValueOutputFile   !< range, with the value of the model function at the
                                        endif                                               !< first point of the model function range.
                                    else !<---------------------------------------------------------------------------------------------------------------
                                        Do kk = 1, lengthexpdata(NumOut)
                                            j = 0                                           !< counter for x-columns which are identical to the last
                                                                                            !< expdatax position
                                            ii = 0                                          !< counter for x-columns which are smaller then the last
                                                                                            !< expdatax position
                                            Do k = 1, NumberXColumns(NumOut)                !< loop over all x-columns
                                                if (expdatax(NumOut, kk, k) == LastPointOutputFile(k)) then
                                                    j = j + 1
                                                elseif (expdatax(NumOut, kk, k) > LastPointOutputFile(k)) then
                                                    ii = ii + 1
                                                endif
                                            end Do
                                            if ((ii + j) == NumberXColumns(NumOut) .and. (j /= NumberXColumns(NumOut))) then
                                                xc = xc + 1
                                                xPointsModel_expanded(xc, :) = expdatax(NumOut, kk, :)
                                            endif
                                        end Do
                                        if (xc > 0) then                                    !< extrapolate model function to the start of the exp data
                                            valuesModel_expanded(:xc) = LastPointValueOutputFile   !< range, with the value of the model function at the
                                        endif                                               !< first point of the model function range.
                                    endif


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< save all model function values which are in the same range than the current observation data
                                    if ((.not. ExpData_reversed_flag(NumOut)) .and. (.not. (Output_reversed_order_flag)) &
                                        .or. (ExpData_reversed_flag(NumOut)) .and. (Output_reversed_order_flag)) then
                                        startLoop = 1
                                        endLoop = counter_lines_output(ThreadNumber, NumOut)
                                        stepLoop = 1
                                    else
                                        startLoop = counter_lines_output(ThreadNumber, NumOut)
                                        endLoop = 1
                                        stepLoop = (-1)
                                    endif
!---------------------------------------------------------------------------------------------------------------------------------------------------------
! muss für interpolation bei AllInOneOutputFile angepasst werden
                                    Do j = startLoop, endLoop, stepLoop                     !< loop over all "x"-points in the current output file
                                        ii = 0
                                        Do k = 1, NumberXColumns(NumOut)                    !< loop over all x-columns
                                            if (xPointsModel_output(ThreadNumber, j, k) >= FirstPointExpData(NumOut, k) &
                                                .and. xPointsModel_output(ThreadNumber, j, k) <= LastPointExpData(NumOut, k)) then
                                                ii = ii + 1
                                            endif
                                        end Do
                                        if (ii == NumberXColumns(NumOut)) then              !< if all x-columns are within exp. data range


                                            !< is the jth point of the output model function within a range
                                            kk = 0
                                            if (NumberRanges(NumOut) > 0) then
                                                kk = 1
                                                Do jj = 1, NumberRanges(NumOut)

                                                    ii = 0
                                                    Do k = 1, NumberXColumns(NumOut)        !< loop over all x-columns
                                                        if (xPointsModel_output(ThreadNumber, j, k) >= MinRange(NumOut, jj, k) &
                                                            .and. xPointsModel_output(ThreadNumber, j, k) <= MaxRange(NumOut, jj, k)) then
                                                            ii = ii + 1
                                                        endif
                                                    end Do
                                                    if (ii == NumberXColumns(NumOut)) then
                                                        kk = 0
                                                        exit
                                                    endif
                                                end Do
                                            endif
                                            if (kk == 0) then
                                                xc = xc + 1
                                                xPointsModel_expanded(xc , :) = xPointsModel_output(ThreadNumber, j, :)
                                                valuesModel_expanded(xc) = valuesModel_output(ThreadNumber, j)
                                            endif
                                        endif
                                    end Do
!---------------------------------------------------------------------------------------------------------------------------------------------------------


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< 3. case: The model function ends at a previous x-column position than exp. data:
                                    !<
                                    !<              [**********************************************************]                = model function range
                                    !<              |----|----|----|----|----|----|----|----|----|----|----|----|----|----|     = exp. data range
                                    !<
                                    !<          Here, the model function is extrapolated to the end of the exp data range, with the value of the model
                                    !<          function at the last point of the model function range.
                                    !<
                                    !<
                                    !< first of all, find index in exp. data range which is close to the last index of the model function range
                                    if (xc > 0) then
                                        if (((.not. ExpData_reversed_flag(NumOut)) .and. (.not. (Output_reversed_order_flag))) &
                                            .or. (ExpData_reversed_flag(NumOut) .and. Output_reversed_order_flag)) then
                                            startLoop = lengthexpdata(NumOut)
                                            endLoop = 1
                                            stepLoop = (-1)
                                        else
                                            startLoop = 1
                                            endLoop = lengthexpdata(NumOut)
                                            stepLoop = 1
                                        endif
                                        kk = 0
                                        Do j = startLoop, endLoop, stepLoop                 !< find index in exp. data range which is close to the
                                                                                            !< last index of the model function range
                                            jj = 0                                          !< counter for x-columns which are identical to the last
                                                                                            !< expdatax position
                                            ii = 0                                          !< counter for x-columns which are smaller then the last
                                                                                            !< expdatax position
                                            Do k = 1, NumberXColumns(NumOut)                !< loop over all x-columns0
                                                if (.not.ExpData_reversed_flag(NumOut)) then
                                                    if (expdatax(NumOut, j, k) == LastPointOutputFile(k)) then
                                                        jj = jj + 1
                                                    elseif (expdatax(NumOut, j, k) > LastPointOutputFile(k)) then
                                                        ii = ii + 1
                                                    endif
                                                else
                                                    if (expdatax(NumOut, j, k) == FirstPointOutputFile(k)) then
                                                        jj = jj + 1
                                                    elseif (expdatax(NumOut, j, k) < FirstPointOutputFile(k)) then
                                                        ii = ii + 1
                                                    endif
                                                endif
                                            end Do
                                            if ((ii + jj) == NumberXColumns(NumOut) .and. (jj /= NumberXColumns(NumOut))) then
                                                if (kk < lengthexpdata(NumOut)) then
                                                    kk = j + 1
                                                else
                                                    kk = j
                                                endif
                                                exit
                                            endif
                                        end Do


                                        !< extrapolate the model function up to the last point of the exp. data range with the last model function
                                        !< value at the last point of the model function range
                                        if (kk /= 0) then                                   !< continue, if 3. case is matched
                                            Do j = kk, lengthexpdata(NumOut)
                                                xc = xc + 1
                                                xPointsModel_expanded(xc , :) = expdatax(NumOut, j, :)
                                                valuesModel_expanded(xc) = LastPointValueOutputFile
                                            end Do
                                        endif
                                    endif


                                    !<--------------------------------------------------------------------------------------------------------------------
                                    !< does the fit function cover the exp. data range (case 5 and case 6)?
                                    if (xc == 0) then
                                        valuesModel_expanded(:) = ValueEmptyOutputFile(NumOut)  !< set all model function values to zero
                                        cover_flag = .false.                                !< set cover flag
                                        write(logchannel,*)
                                        write(logchannel,'(11x,"WARNING:")')
                                        write(logchannel,'(11x,"The model function does not cover the experimental data.")')
                                        write(logchannel,'(11x,"Set model function values to ", ES25.15, ".")') ValueEmptyOutputFile(NumOut)
                                        write(logchannel,*)
                                        write(logchannel,'(11x,"Number of current Output File  = ",I4)') NumOut
                                        write(logchannel,'(11x,"First point in Output File (FirstPointOutputFile(:))  = ",99999(1x, ES25.15))') &
                                                                                                                          FirstPointOutputFile(:)
                                        write(logchannel,'(11x,"Last point in Output File (LastPointOutputFile(:))  = ",99999(1x, ES25.15))') &
                                                                                                                         LastPointOutputFile(:)

                                        write(logchannel,'(11x,"First point in observation File (FirstPointExpData(NumOut,:))  = ",99999(1x, ES25.15))') &
                                                         FirstPointExpData(NumOut,:)
                                        write(logchannel,'(11x,"Last point in observation File (LastPointExpData(NumOut, :))  = ",99999(1x, ES25.15))') &
                                                         LastPointExpData(NumOut, :)
                                        write(logchannel,'(" ")')
                                        write(logchannel,'(11x,"Program aborted!")')

                                        print '(" ")'
                                        print '(" ")'
                                        print '(" ")'
                                        print '(11x,"WARNING:")'
                                        print '(11x,"The model function does not cover the experimental data.")'
                                        print '(11x,"Set model function values to ",ES25.15,".")', ValueEmptyOutputFile(NumOut)
                                        print '(" ")'
                                        print '(11x,"Number of current Output File  = ",I4)', NumOut
                                        print '(11x,"First point in Output File FirstPointOutputFile(:)  = ",99999(1x, ES25.15))', FirstPointOutputFile(:)
                                        print '(11x,"Last point in Output File LastPointOutputFile(:)  = ",99999(1x, ES25.15))', LastPointOutputFile(:)

                                        print '(11x,"First point in observation File FirstPointExpData(NumOut,:)  = ",99999(1x, ES25.15))', &
                                                                                                                 FirstPointExpData(NumOut, :)
                                        print '(11x,"Last point in observation File LastPointExpData(NumOut,:)  = ",99999(1x, ES25.15))', &
                                                                                                                LastPointExpData(NumOut, :)
                                        print '(" ")'

                                        ! Debug:
                                        ! Do i = 1,lengthexpdata(NumOut)
                                        !     print*,'>>',i,expdatax(NumOut, i, :)
                                        ! end Do
                                        ! print*,'##############################################################'
                                        ! Do i = 1,counter_lines_output(ThreadNumber, NumOut)
                                        !     print*,'->',i,xPointsModel_output(ThreadNumber, i, :)
                                        ! end Do
                                        ! stop

                                    else    !< continue only, if model function covers the exp. data range at least partly

                                        ! Debug:
                                        ! print*,' '
                                        ! print*,'xc = ',xc
                                        ! Do j = 1, xc
                                        !     print*,'>',j,xPointsModel_expanded(j , :), valuesModel_expanded(j)
                                        ! end Do
                                        ! stop '####'


                                        !<----------------------------------------------------------------------------------------------------------------
                                        !< do interpolation, to determine model function at the same positions than in the exp. data range
                                        !< determine the integral over the expanded model function
                                        if (NormalizationFlag(NumOut)) then
                                            IntegralValue1 = 0.d0
                                            NumberPoints = xc
                                            call MultidimensionalIntegration(IntegralValue1, colx, NumberPoints, xPointsModel_expanded(:xc, :), &
                                                                                                                       valuesModel_expanded(:xc))

                                            ! Debug:
                                            ! print*,'>', IntegralValue1
                                            ! open(368, file='/home/moeller/CATS/magix/test'//trim(adjustl(DebugString))//'_int1.txt')
                                            ! Do j = 1, xc
                                            !   write(368,'(9999(F20.10))') xPointsModel_expanded(j, 1), valuesModel_expanded(j)
                                            ! end Do
                                            ! close(368)
                                            ! stop
                                        endif


                                        !< do interpolation
                                        Do j = 1, lengthexpdata(NumOut)                     !< loop over all "x"-points in the current exp. file
                                            interpol_value = 0.d0


                                            !< determine the index of the point in the model function range which is next to the current jth point
                                            !< in the exp. data range
                                            shortest_distance = 1.d99                       !< reset shortest distance
                                            shortest_distance_index = 0
                                            second_shortest_distance = 1.d99                !< reset shortest distance
                                            second_shortest_distance_index = 0
                                            shortest_distance_value = 0.d0
                                            Do k = 1, xc                                    !< loop over all points in the expanded model function array


                                                !< determine distance between the jth point in the exp. data range and the kth point in the model
                                                !< function range
                                                distance = 0.d0
                                                Do kk = 1, NumberXColumns(NumOut)           !< loop over each "x"-point in the current line
                                                    distance = distance + (expdatax(NumOut, j, kk) - xPointsModel_expanded(k, kk))**2
                                                end Do
                                                distance = dsqrt(distance)


                                                !< store shortest distance and the corresponding k index
                                                if (distance == 0.d0) then
                                                    interpol_value = valuesModel_expanded(k)
                                                    exit
                                                endif


                                                !< store shortest distance and the corresponding k index
                                                if (distance < shortest_distance) then
                                                    shortest_distance = distance
                                                    shortest_distance_index = k
                                                    shortest_distance_value = valuesModel_expanded(k)
                                                endif
                                            end Do
                                            if (distance /= 0.d0) then                      !< interpolation only for distances /= 0


                                                !< interpolation method: nearest-neighbour
                                                if (trim(adjustl(InterpolationMethod(NumOut))) == "nearest-neighbour") then
                                                    interpol_value = shortest_distance_value


                                                !< interpolation method: linear
                                                elseif (trim(adjustl(InterpolationMethod(NumOut))) == "linear") then
                                                    second_shortest_distance = 1.d99        !< reset shortest distance
                                                    second_shortest_distance_index = 0
                                                    second_shortest_distance_value = 0.d0
                                                    Do k = 1, xc                            !< loop over all points in the expanded model function array


                                                        !< determine distance between the jth point in the exp. data range and the kth point in the model
                                                        !< function range
                                                        distance = 0.d0
                                                        Do kk = 1, NumberXColumns(NumOut)    !< loop over each "x"-point in the current line
                                                            distance = distance + (expdatax(NumOut, j, kk) - xPointsModel_expanded(k, kk))**2
                                                        end Do
                                                        distance = dsqrt(distance)


                                                        if (distance < second_shortest_distance .and. distance > shortest_distance) then  !< store second
                                                            second_shortest_distance = distance !< shortest distance and the corresponding k index
                                                            second_shortest_distance_index = k
                                                            second_shortest_distance_value = valuesModel_expanded(k)
                                                        endif
                                                    end Do
                                                    if (second_shortest_distance_index == 0) then
                                                        second_shortest_distance_index = shortest_distance_index
                                                        second_shortest_distance_value = shortest_distance_value
                                                    endif


                                                    !< determine distance between nearest and second nearest point
                                                    d1 = 0.d0
                                                    d2 = 0.d0
                                                    d3 = 0.d0
                                                    distance = 0.d0
                                                    Do kk = 1, NumberXColumns(NumOut)
                                                        d1 = d1 + xPointsModel_expanded(shortest_distance_index, kk)**2
                                                        d2 = d2 + xPointsModel_expanded(second_shortest_distance_index, kk)**2
                                                        d3 = d3 + expdatax(NumOut, j, kk)**2
                                                        distance = distance + (xPointsModel_expanded(shortest_distance_index, kk) - &
                                                                               xPointsModel_expanded(second_shortest_distance_index, kk))**2
                                                    end Do
                                                    d1 = dsqrt(d1)
                                                    d2 = dsqrt(d2)
                                                    d3 = dsqrt(d3)
                                                    distance = dsqrt(distance)


                                                    !< determine x1, x2, y1, and y2
                                                    y1 = shortest_distance_value
                                                    y2 = second_shortest_distance_value


                                                    !< determine gradient
                                                    if (distance /= 0.d0) then
                                                        grad = (y2 - y1) / distance


                                                        !< calculate final interpolation value
                                                        interpol_value = (d3 - d1) * grad + y1
                                                    else
                                                        interpol_value = shortest_distance_value
                                                    endif
                                                endif
                                            endif
                                            AtOnceFunctionLocal(ThreadNumber, NumOut, yColumn, j) = interpol_value

                                            ! Debug:
                                            ! print*,'>',OutputLineCounter, expdatax(NumOut, j, :NumberXColumns(NumOut)), interpol_value
                                            ! if (distance /= 0.d0) print*,'>', NumOut, OutputLineCounter, j, shortest_distance_index, distance,
                                            !                                   interpol_value
                                        end Do                                              !< loop j: loop over all "x"-points in the current exp. file



                                        !<----------------------------------------------------------------------------------------------------------------
                                        !< do interpolation, to determine model function at the same positions than in the exp. data range
                                        !< determine the integral over the expanded model function
                                        if (NormalizationFlag(NumOut)) then
                                            IntegralValue2 = 0.d0
                                            NumberPoints = lengthexpdata(NumOut)
                                            call MultidimensionalIntegration(IntegralValue2, NumberXColumns(NumOut), NumberPoints, &
                                                                             expdatax(NumOut, :NumberPoints, :NumberXColumns(NumOut)), &
                                                                             AtOnceFunctionLocal(ThreadNumber, NumOut, yColumn, :) )
                                            ! Debug:
                                            ! print*,'>',IntegralValue2
                                            ! open(368,file='/home/moeller/CATS/magix/test'//trim(adjustl(DebugString))//'_int2.txt')
                                            ! Do j = 1, NumberPoints
                                            !   write(368,'(9999(F20.10))') expdatax(NumOut, j, :NumberXColumns(NumOut)), &
                                            !                                AtOnceFunctionLocal(ThreadNumber, NumOut, yColumn, j)
                                            ! end Do
                                            ! close(368)
                                            ! open(368,file='/home/moeller/CATS/magix/test'//trim(adjustl(DebugString))//'_exp.txt')
                                            ! Do j = 1, lengthexpdata(NumOut)
                                            !   write(368,'(9999(F20.10))') expdatax(NumOut, j, :NumberXColumns(NumOut)), &
                                            !                               expdatay(NumOut, j, :NumberYColumns(NumOut))
                                            ! end Do
                                            ! close(368)
                                            ! print*,'####################'
                                        endif


                                        !< make that the interpolated output model function have the same integrated area than original output
                                        !< model function
                                        if (NormalizationFlag(NumOut) .and. IntegralValue1 /= 0.d0 .and. IntegralValue2 /= 0.d0) then
                                            AtOnceFunctionLocal(ThreadNumber, NumOut, yColumn, :) = &
                                            IntegralValue1/IntegralValue2 * AtOnceFunctionLocal(ThreadNumber, NumOut, yColumn, :)
                                        endif
                                    endif
                                endif                                                       !< continue if resampling is finished


                                !< (deallocate and) allocate memory for output variables
                                if (allocated(xPointsModel_expanded)) then
                                    deallocate(xPointsModel_expanded, valuesModel_expanded, stat = deallocstatus)
                                    if (deallocstatus /= 0) then
                                        write(logchannel,*)
                                        write(logchannel,'(11x,"Error in subroutine ReadDataOfFitFunction:")')
                                        write(logchannel,'(13x,"Can not deallocate variables xPointsModel_expanded etc.")')
                                        write(logchannel,'(13x,"Please close all other programs and restart the program!")')
                                        write(logchannel,*)
                                        write(logchannel,'(13x,"deallocstatus = ",I4)') deallocstatus
                                        write(logchannel,'(" ")')
                                        write(logchannel,'(11x,"Program aborted!")')
                                        close(logchannel)

                                        print '(" ")'
                                        print '(11x,"Error in subroutine ReadDataOfFitFunction:")'
                                        print '(13x,"Can not deallocate variables xPointsModel_expanded etc.")'
                                        print '(13x,"Please close all other programs and restart the program!")'
                                        print '(" ")'
                                        print '(13x,"deallocstatus = ",I4)',deallocstatus
                                        print '(" ")'
                                        stop '           Program aborted!'
                                    endif
                                endif
                            endif                                                           !< continue here, if output file is empty


                            !<----------------------------------------------------------------------------------------------------------------------------
                            !< deallocate memory for output variables
                            if (allocated(xPointsModel_output)) then
                                deallocate(xPointsModel_output, valuesModel_output, stat = deallocstatus)
                                if (deallocstatus /= 0) then
                                    write(logchannel,*)
                                    write(logchannel,'(11x,"Error in subroutine ReadDataOfFitFunction:")')
                                    write(logchannel,'(13x,"Can not deallocate variables xPointsModel_output etc.")')
                                    write(logchannel,'(13x,"Please close all other programs and restart the program!")')
                                    write(logchannel,*)
                                    write(logchannel,'(13x,"deallocstatus = ",I4)') deallocstatus
                                    write(logchannel,'(" ")')
                                    write(logchannel,'(11x,"Program aborted!")')
                                    close(logchannel)

                                    print '(" ")'
                                    print '(11x,"Error in subroutine ReadDataOfFitFunction:")'
                                    print '(13x,"Can not deallocate variables xPointsModel_output etc.")'
                                    print '(13x,"Please close all other programs and restart the program!")'
                                    print '(" ")'
                                    print '(13x,"deallocstatus = ",I4)',deallocstatus
                                    print '(" ")'
                                    stop '           Program aborted!'
                                endif
                            endif
                            first_element_index = OutputLineCounter
                        end Do                                                              !< loop yColumn: loop over all y-columns of current output
                                                                                            !< files
                    endif                                                                   !< continue here, if output file is empty or not


                    !< close output file
                    close(ChannelLocal) !, status = 'delete')                               !< close and delete current output file after reading the
                end Do                                                                      !< loop over all output files
                return
            end subroutine ReadDataOfFitFunction


            !*********************************************************************************************************************************************
            !> subroutine: CallFitFunctionAtOnce
            !>
            !> call the external model function defined in the CommandLine variable
            !>
            !>
            !> input variables:     modelparamnumber:   number of parameters
            !>                      modelparam:         array containing the parameter set
            !>                      NumThreads:         max number of threads
            !>                      NumFileOrig:        number of experimental files
            !>                      MaxL:               max number of lines of all experimental files
            !>                      MaxCol:             max number of columns of all experimental files
            !>                      colx:               number of columns in experimental x data
            !>
            !> output variables:    AtOnceFunctionLocal: array containing the fit model function
            !>                      ok:                 status variable (needed for MPI version)
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 14.10.2011
            !>
            subroutine CallFitFunctionAtOnce(ok, modelparamnumber, modelparam, AtOnceFunctionLocal, NumThreads, NumFileOrig, MaxL, MaxCol, colx)
                !< call the function defined in the CommandLine variable

                implicit none
                integer :: ok                                                               !< status variable (needed for MPI version)
                integer :: NumFileOrig                                                      !< number of experimental files
                integer :: MaxL                                                             !< max number of lines of all experimental files
                integer :: MaxCol                                                           !< max number of columns of all experimental files
                integer :: colx                                                             !< number of columns in experimental x data
                integer :: NumFile, NumberXExpPoints                                        !< working variables
                integer :: NumInputFiles                                                    !< index for input file
                integer :: modelparamnumber                                                 !< number of parameters
                integer :: ThreadNumber                                                     !< ID number of the current thread
                integer :: NumThreads                                                       !< max number of threads
                integer :: ChannelLocal1                                                    !< copy of InputFileChannel
                real*8, dimension(1) :: dummy                                               !< dummy argument required by subroutine WriteParameter
                real*8, dimension(modelparamnumber) :: modelparam                           !< array containing the parameter set
                real*8, dimension(0:NumThreads-1, NumFileOrig, MaxCol, MaxL) :: AtOnceFunctionLocal    !< array containing the fit model function
                character(len=30) :: LongHelpString1, LongHelpString2                       !< help strings for converting numbers to strings
                character(len=8192) :: CommandString                                        !< command string for creating subdirectory
                character(len=8192) :: filename                                             !< string containing the filename with path
                character(len=8192) :: WorkingDir                                           !< containing current working directory


                !< initialize status variable
                ok = 0


                !< if parallelization is required, create new subdirectory in temp-directory
                WorkingDir = " "                                                            !< working directory is current directory
                ChannelLocal1 = InputFileChannel
                write(LongHelpString1, '(I30)') JobID                                       !< write JobID to string


                !< get thread number(ThreadNumber)
                ThreadNumber = 0                                                            !< default value for thread number
                call GetThreadNumberForTempDir(ThreadNumber)                                !< get thread number for temp. directory
                write(LongHelpString2, '(I30)') ThreadNumber                                !< write thread number to string
                ChannelLocal1 = ChannelLocal1 + ThreadNumber                                !< modify channel number of input file


                !< create name of current working directory
                WorkingDir = trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(LongHelpString1)) // "/" // trim(adjustl(LongHelpString2)) // "/"


                !< create working directory
                CommandString = "mkdir -p "// trim(adjustl(WorkingDir))
                call system(CommandString)

                ! Debug:
                ! print*,'-> CommandString = ',trim(adjustl(CommandString)),'<-'


                !< copy start script to current temp directory
                CommandString = "cp " // trim(adjustl(PathStartScript)) // " " // trim(adjustl(WorkingDir))
                call system(CommandString)

                ! Debug:
                ! print*,'-> CommandString = ',trim(adjustl(CommandString)),'<-'


                !< copy fits2dat converter to working directory if one or more output files of the external model program are in fits format
                if (maxval(OutputFileFormat(:)) == 2) then
                    CommandString = "cp Modules/magix-parts_python/fits2dat.py " // trim(adjustl(WorkingDir))
                    call system(CommandString)
                endif


                !< write input file for several models
                Do NumInputFiles = 1, NumberInputFiles
                    filename = trim(adjustl(WorkingDir)) // trim(adjustl(FitFktInput(NumInputFiles)))    !< define path and filename of input-file


                    !< open input-file
                    open(ChannelLocal1,file = trim(filename))
                    NumberXExpPoints = 0
                    Do NumFile = 1, NumberExpFiles                                          !< loop over all exp. data files
                        NumberXExpPoints = NumberXExpPoints + lengthexpdata(NumFile)        !< loop over all lines

                        ! Debug:
                        ! print*,'NumFile,NumberXExpPoints, lengthexpdata(NumFile) = ',NumFile,NumberXExpPoints, lengthexpdata(NumFile)
                    end Do
                    dummy = dfloat(NumberXExpPoints)
                    call WriteParameter(ChannelLocal1, .true., 1, dummy, modelparamnumber, modelparam, NumInputFiles)    !< write parameter to file
                    close(ChannelLocal1)                                                    !< close input-file
                end Do


                !< write X column values of exp data file to one bit ASCII file
                if (InputDataPath /= "") then                                               !< if path of data input file is not empty
                    call WriteDataForFitFunction                                            !< write experimental data to file
                endif


                !-----------------------------------------------------------------------------------------------------------------------------------------
                !< change to defined directory, call external program to calculate model and go back to MAGIX standard working directory


                !< change working directory (use intrinsic function ChDir)
                CommandString = "(cd"
                CommandString = trim(adjustl(CommandString)) // " " // trim(adjustl(WorkingDir)) // "; " // trim(adjustl(ExeCommandStartScript)) // ")"


                !< call external program
                call system(CommandString)


                !< in order to debug the call of the external model program stop here
                if (debugflag) then
                    ok = 1
                    return
                endif


                !< read output file
                call ReadDataOfFitFunction(AtOnceFunctionLocal, NumberOutputFiles, NumThreads, NumFileOrig, MaxL, MaxCol, colx)


                !< remove temp-directory
                CommandString = "(rm -rf "//trim(adjustl(WorkingDir)) // " 2>/dev/null)"
                call system(CommandString)
                return
            end subroutine CallFitFunctionAtOnce


            !*********************************************************************************************************************************************
            !> subroutine: FitFunctionAtOnce
            !>
            !> determines the value of the external model function and its gradient (method: at once)
            !>
            !>
            !> input variables:     modelparamnumber:   number of parameters
            !>                      modelparam:         array containing the parameter set
            !>                      ia:                 flags indicating parameter which are optimized
            !>                      NumFile:            number of experimental files
            !>                      MaxL:               max number of lines of all experimental files
            !>                      MaxCol:             max number of columns of all experimental files
            !>                      colx:               number of columns in experimental x data
            !>
            !> output variables:    ok:                 status variable (needed for MPI version)
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 14.10.2011
            !>
            subroutine FitFunctionAtOnce(ok, modelparamnumber, modelparam, NumFile, MaxL, MaxCol, colx)
                !< determines the value of the external model function and its gradient (method: at once)

                implicit none
                integer :: ok                                                               !< status variable (needed for MPI version)
                integer :: NumFile                                                          !< number of experimental files
                integer :: MaxL                                                             !< max number of lines of all experimental files
                integer :: MaxCol                                                           !< max number of columns of all experimental files
                integer :: colx                                                             !< number of columns in experimental x data
                integer :: modelparamnumber                                                 !< number of parameters
                integer :: NumThreads                                                       !< max number of threads (processor)
                integer :: ThreadNumber                                                     !< current thread id-number
                real*8, dimension(modelparamnumber) :: modelparam                           !< array containing the parameter set


                !< determine the values of the fit functions for the unmodified parameter set
                NumThreads = NumberOfUsedThreads


                !< get current thread number. Herem we do not use subroutine GetThreadNumber because for MPI we need only one dimension
                !< for arrays 'xPointsModel_output' and 'valuesModel_output'
                call GetThreadNumber(ThreadNumber)


                !< calculate model functions
                AtOnceFunction(ThreadNumber, :, :, :) = 0.d0                                !< the values are stored in the array AtOnceFunction
                call CallFitFunctionAtOnce(ok, modelparamnumber, modelparam, AtOnceFunction, NumThreads, NumFile, MaxL, MaxCol, colx)

                return
            end subroutine FitFunctionAtOnce


            !*********************************************************************************************************************************************
            !> subroutine: parameterlimitcheck
            !>
            !> check if parameters are within defined range
            !>
            !>
            !> input variables:     npar:               number of parameter
            !>                      parmval:            array containing the values of the parameters
            !>
            !> output variables:    OutOfRangeFlag:     flag is true if one parameter is out of range
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 14.10.2011
            !>
            subroutine parameterlimitcheck(OutOfRangeFlag, npar, parmval)
                !< check if all parameters are within defined range

                implicit none
                integer :: i                                                        !< working variable
                integer :: npar                                                     !< number of parameter
                real*8, dimension(npar) :: parmval                                  !< array containing the values of the parameters
                logical :: OutOfRangeFlag                                           !< flag is true if one parameter is out of range


                OutOfRangeFlag = .false.
                Do i = 1, npar                                                      !< loop over all parameters


                    !< is the value of the ith parameter out of range ? Yes? Print warning message!
                    if (paramset(2,i) == 1 .and. (parmval(i) < paramset(3,i) .or. parmval(i) > paramset(4,i))) then
                        OutOfRangeFlag = .true.
                        if (printflag) print '(11x,"WARNING:  parameter ",A,"( = ",F15.8,") out of range [",F10.5,",",F10.5,"].")', &
                                trim(adjustl(FitParameterName(i))), parmval(i), paramset(3,i), paramset(4,i)
                        write(paramchannel,'(11x,"WARNING:  parameter ",A,"( = ",F15.8,") out of range [",F10.5,",",F10.5,"].")') &
                                trim(adjustl(FitParameterName(i))), parmval(i), paramset(3,i), paramset(4,i)
                        write(logchannel,'(11x,"WARNING:  parameter ",A,"( = ",F15.8,") out of range [",F10.5,",",F10.5,"].")') &
                                trim(adjustl(FitParameterName(i))), parmval(i), paramset(3,i), paramset(4,i)

                        if (parmval(i) < paramset(3,i)) then
                            if (paramset(3,i) == 0.d0) then
                                parmval(i) = -parmval(i)
                            else
                                parmval(i) = paramset(3,i)
                            endif
                            if (printflag) print '(21x,"parameter ",A," corrected to ",F20.10)', trim(adjustl(FitParameterName(i))), parmval(i)
                        elseif (parmval(i) > paramset(4,i)) then
                            parmval(i) = paramset(4,i)
                            if (printflag) print '(21x,"parameter ",A," corrected to ",F20.10)', trim(adjustl(FitParameterName(i))), parmval(i)
                        endif
                    endif
                end Do
                return
            end subroutine parameterlimitcheck


            !*********************************************************************************************************************************************
            !> subroutine: CheckCalculatedParameterSets
            !>
            !> reads previously calculated parameter sets
            !>
            !>
            !> input variables:     nfit:               number of free parameters
            !>                      NumberThreads:      number of threads
            !>
            !> output variables:    None
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 14.10.2011
            !>
            subroutine CheckCalculatedParameterSets(nfit, NumberThreads)
                !< reads previously calculated parameter sets

                implicit none
                integer :: i, j                                                             !< loop variables
                integer :: nfit                                                             !< number of free parameters
                integer :: NumberThreads                                                    !< number of threads
                integer :: allocstatus, deallocstatus                                       !< status variables for allocation and deallocation of memory
                real*8, dimension(nfit + 1) :: Entries                                      !< temp variable


                !-----------------------------------------------------------------------------------------------------------------------------------------
                !< deallocate/allocate memory for some working variables, clear contents of the variables and print error message if necessary
                if (allocated(CalculatedParameterSets)) then
                    deallocate(CalculatedParameterSets, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine CheckCalculatedParameterSets:")')
                        write(logchannel,'(2x,"Can not deallocate variables CalculatedParameterSets.")')
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,*)
                        write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine CheckCalculatedParameterSets:")'
                        print '(2x,"Can not deallocate variables CalculatedParameterSets.")'
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("deallocstatus = ",I4)',deallocstatus
                        print '(" ")'
                        stop ' Program aborted!'
                    endif
                endif
                allocate(CalculatedParameterSets(NumberThreads, CurrentNumberLinesCalcReduction, nfit + 1), stat = allocstatus)
                if (allocstatus /= 0) then
                    write(logchannel,'(" ")')
                    write(logchannel,'("Error in subroutine CheckCalculatedParameterSets:")')
                    write(logchannel,'(2x,"Can not allocate variables CalculatedParameterSets.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,'(" ")')
                    write(logchannel,'("allocstatus = ",I4)') allocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine CheckCalculatedParameterSets:")'
                    print '(2x,"Can not allocate variables CalculatedParameterSets.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("allocstatus = ",I4)',allocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
                CalculatedParameterSets = 0.d0


                !-----------------------------------------------------------------------------------------------------------------------------------------
                !< read contents of log-file
                rewind Chi2Channel
                Do i = 1, CurrentNumberLinesCalcReduction
                    Entries = 0.d0
                    CalculatedParameterSets(:, i, 1) = 1.d99
                    read(Chi2Channel,*) Entries(:)
                    if (minval(Entries(:)) /= 0.d0 .or. maxval(Entries(:)) /= 0.d0) then    !< check line
                        Do j = 1, NumberThreads
                            CalculatedParameterSets(j, i, :) = Entries(:)
                        end Do
                    endif
                end Do
                return
            end subroutine CheckCalculatedParameterSets


            !*********************************************************************************************************************************************
            !> subroutine: ChiFunction
            !>
            !> calculates chi^2 using different methods saved in DetChi2
            !>
            !>
            !> input variables:     ma:                 total number of parameters
            !>                      a:                  array containing the parameter set
            !>                      ia:                 flags for including/excluding parameter in the fit
            !>                      numfit:             number of fitted parameters
            !>                      fitparam:           parameter which have to be optimized
            !>                      colx:               number of columns in experimental x data
            !>                      NumFile:            number of experimental files
            !>                      MaxL:               max number of lines of all experimental files
            !>                      MaxCol:             max number of columns of all experimental files
            !>                      FitFunctionOut:     values of the fit function at all calculated points
            !>                      Chi2Values:         values of the fit function at all calculated points
            !>                      alpha:              matrix alpha (only used for Levenberg-Marquardt algorithm)
            !>                      beta2:              beta2 array (only used for Levenberg-Marquardt algorithm)
            !>
            !> output variables:    None
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 14.10.2011
            !>
            subroutine ChiFunction(ma, a, ia, numfit, fitparam, colx, NumFile, MaxL, MaxCol, FitFunctionOut, Chi2Values, alpha, beta2)

                implicit none
                integer :: ok                                                               !< status variable (needed for MPI version)
                integer :: i, j, k, l, m, n                                                 !< working variables
                integer :: NumFile                                                          !< number of experimental files
                integer :: MaxL                                                             !< max number of lines of all experimental files
                integer :: MaxCol                                                           !< max number of columns of all experimental files
                integer :: ma                                                               !< total number of parameters
                integer :: colx                                                             !< number of columns in experimental x data
                integer :: counter                                                          !< counter for AtOnceFunction
                integer :: numfit                                                           !< number of fitted parameters
                integer :: NumInputFile_index                                               !< contains index for input file
                integer :: i_index                                                          !< contains index for i
                integer :: j_index                                                          !< contains index for j
                integer :: length                                                           !< total length of AtOnceFunction
                integer :: NumberFile                                                       !< loop variable for counting file
                integer :: ThreadNumber                                                     !< current thread number
                real*8 :: dy, sig2i, wt, ymod                                               !< working variables
                real*8, dimension(ma) :: a                                                  !< array containing the parameter set
                real*8, dimension(ma, ma) :: alpha                                          !< matrix alpha
                real*8, dimension(numfit) :: beta2                                          !< beta2 array
                real*8, dimension(numfit) :: fitparam                                       !< parameter which have to be optimized
                real*8, dimension(numfit) :: LocalParamSet                                  !< local copy of parameter set
                real*8, dimension(NumFile, MaxL, MaxCol) :: FitFunctionOut                  !< values of the fit function at all calculated points
                real*8, dimension(NumFile, MaxL, MaxCol) :: Chi2Values                      !< values of the fit function at all calculated points
                real*8, dimension(ma) :: dyda                                               !< gradient of the fit function
                logical :: EqualParam_Flag                                                  !< flag for comparing two parameter sets
                logical :: IntegerTrue                                                      !< flag for identification of integer numbers
                logical, dimension(ma) :: ia                                                !< flags for including/excluding parameter in the fit
                character(len=100) :: HelpString                                            !< working variables
                character(len=100), dimension(numfit) :: FormattedParmValues                !< formatted parameter values for chi2 log file


                !<----------------------------------------------------------------------------------------------------------------------------------------
                !< initialize values for fit function calculation
                Do j = 1, numfit                                                            !< Initialize (symmetric) alpha, beta2.
                    Do k = 1, j
                        alpha(j, k) = 0.d0                                                  !< clear alpha partially
                    end Do
                    beta2(j) = 0.d0                                                         !< clear beta2
                end Do


                !< determine current thread number and reset array chisqValues
                ThreadNumber = 0
                call GetThreadNumber(ThreadNumber)
                chisqValues(ThreadNumber) = 0.d0                                            !< set value of chi**2 to 0


                !< is calculation reduction wanted
                if (UseCalculationReduction) then
                    Do i = 1, CurrentNumberLinesCalcReduction
                        EqualParam_Flag = .false.
                        LocalParamSet(:) = CalculatedParameterSets(ThreadNumber + 1, i, 2:)
                        call CompareTwoParameterSets(EqualParam_Flag, ma, numfit, ia, LocalParamSet(:), fitparam)


                        !< if all parameter are identical set flag QualityFlag
                        if (EqualParam_Flag) then
                            chisqValues(ThreadNumber) = CalculatedParameterSets(ThreadNumber + 1, i, 1)
                            return
                        endif
                    end Do
                endif

                ! Debug:
                !    print*,'ChiFunction:'
                !    print*,'ma = ',ma
                !    print*,'a = ',a
                !    print*,'ia = ',ia
                !    print*,' '

                !< update the parameter set with the current values of the fit parameters
                !< some algorithms optimize only the parameters in the array fitparam
                !< all other parameter in the parameter set a are not included in these algorithms
                !< but are required for the calculation of the fit function

                k = 0
                Do i = 1, ma
                    if (ia(i)) then
                        k = k + 1
                        a(i) = fitparam(k)
                    endif
                end Do


                !< if calculation method == 'atonce' then determine model function
                if (CalculationMethod == "atonce") then
                    length = 0
                    Do NumberFile = 1, NumberExpFiles
                        length = length + (NumberYColumns(NumberFile) * lengthexpdata(NumberFile))
                    end Do
                    call FitFunctionAtOnce(ok, ma, a, NumFile, MaxL, MaxCol, colx)
                    if (ok == 1) stop
                endif


                !< start loop for determine chi**2
                FitFunctionOut = 0.d0                                                       !< reset array containing the values of the fit function
                Chi2Values = 0.d0
                counter = 0
                Do NumberFile = 1, NumberExpFiles                                           !< loop over exp. data files
                    CurrentExpFile = NumberFile


                    !< start loop over all lines(expdata)
                    Do n = 1, NumberYColumns(NumberFile)                                    !< loop over y-columns
                        Do i = 1, lengthexpdata(NumberFile)                                 !< loop over all line of current exp. data file
                            CurrentYColumn = n


                            !< get thread number
                            ThreadNumber = 0
                            call GetThreadNumber(ThreadNumber)


                            !< calculate fit function
                            ymod = 0.d0
                            if (CalculationMethod == "atonce") then                         !< if calculation method == 'atonce' read values from memory
                                counter = counter + 1
                                ymod = AtOnceFunction(ThreadNumber, NumberFile, n, i)

                                ! Debug:
                                ! print*,'>>',NumberFile, n, i, counter, ymod


                                if (Gradientflag) then
                                    dyda = 0.d0
                                    l = 0
                                    Do j = 1, ma
                                        if (ia(j)) then
                                            l = l + 1
                                            dyda(j) = AtOnceGradient(NumberFile, n, i, l)
                                        endif
                                    end Do

                                    ! Debug:
                                    ! print*,'ymod = ', ymod
                                    ! print*,'dyda(1:nfit) = ', dyda(1:l)
                                    ! print*,'a = ', a
                                    ! print*,'############################################################################'
                                endif
                            endif
                            FitFunctionOut(NumberFile, i, n) = ymod                         !< save value of fit function

                            ! Debug:
                            !    print*,' '
                            !    print*,' '
                            !    print*,'NumberExpFiles = ',NumberExpFiles
                            !    print*,'lengthexpdata(NumberFile) = ',lengthexpdata(NumberFile)
                            !    print*,'NumberYColumns(NumberFile) = ',NumberYColumns(NumberFile)
                            !    print*,'NumberFile = ',NumberFile
                            !    print*,'i = ',i
                            !    print*,'n = ',n
                            !    print*,'lenposdatexp = ',lenposdatexp
                            !    print*,'posdatexp = ',posdatexp
                            !    print*,'expdatay(NumberFile,i,n) = ',expdatay(NumberFile,i,n)
                            !    print*,'ymod = ',ymod
                            !    print*,'dyda = ',dyda(1:numfit)
                            !    if (i==3) stop


                            !<----------------------------------------------------------------------------------------------------------------------------
                            !< determine chi**2 by calculating the difference (y_i^{obs) - y_i(fit))**2
                            if (abs(DetChi2) == 1) then

                                sig2i = 1.d0
                                if (expdatae(NumberFile,i,n) /= 0.d0) then                  !< do the experimental datas include errors
                                    sig2i = 1.d0/(expdatae(NumberFile,i,n) * expdatae(NumberFile,i,n))      !< define sig2i factor
                                endif
                                ThreadNumber = 0
                                call GetThreadNumber(ThreadNumber)


                                dy = (expdatay(NumberFile, i, n) - ymod)                    !< define distance between fit and data
                                Chi2Values(NumberFile, i, n) = dy * dy * sig2i              !< save chi^2
                                chisqValues(ThreadNumber) = chisqValues(ThreadNumber) + dy * dy * sig2i     !< And find chi^2.


                                if (Gradientflag) then
                                    j = 0
                                    Do l = 1, ma                                            !< loop over all parameters
                                        if (ia(l)) then                                     !< is the lth parameters optimized?
                                            j = j + 1
                                            wt = dyda(l) * sig2i                            !< define weighting factor
                                            k = 0
                                            Do m = 1, l                                     !< determine alpha matrix
                                                if (ia(m)) then
                                                    k = k + 1
                                                    alpha(j,k) = alpha(j,k) + wt * dyda(m)
                                                endif
                                            end Do
                                            beta2(j) = beta2(j) + dy * wt                   !< calculate beta2 array
                                        endif
                                    end Do
                                endif


                            !<----------------------------------------------------------------------------------------------------------------------------
                            !< determine chi**2 by calculating the difference (y_i^{obs)**2 - y_i(fit)**2)
                            elseif (abs(DetChi2) == 2) then

                                sig2i = 1.d0
                                if (expdatae(NumberFile,i,n) /= 0.d0) then                  !< do the experimental datas include errors
                                    sig2i = 1.d0/(expdatae(NumberFile,i,n) * expdatae(NumberFile,i,n))      !< define sig2i factor
                                endif
                                ThreadNumber = 0
                                call GetThreadNumber(ThreadNumber)
                                dy = (expdatay(NumberFile, i, n)**2 - ymod**2)
                                Chi2Values(NumberFile, i, n) = dy * dy * sig2i              !< save chi^2
                                chisqValues(ThreadNumber) = chisqValues(ThreadNumber) + dy * dy * sig2i     !< And find chi^2.


                                if (Gradientflag) then
                                    j = 0
                                    Do l = 1, ma                                            !< loop over all parameters
                                        if (ia(l)) then                                     !< is the lth parameters optimized?
                                            j = j + 1
                                            wt = dyda(l) * sig2i                            !< define weighting factor
                                            k = 0
                                            Do m = 1, l                                     !< determine alpha matrix
                                                if (ia(m)) then
                                                    k = k + 1
                                                    alpha(j,k) = alpha(j,k) + wt * dyda(m)
                                                endif
                                            end Do
                                            beta2(j) = beta2(j) + dy * wt                   !< calculate beta2 array
                                        endif
                                    end Do
                                endif

                            endif
                        end Do                                                              !< loop over all line of current exp. data file
                    end Do                                                                  !< loop over y-columns
                end Do                                                                      !< loop over exp. data files


                !< Fill in the symmetric side.
                Do j = 2, numfit
                    Do k = 1, (j - 1)
                       alpha(k,j) = alpha(j,k)
                    end Do
                end Do


                !< writing current value of chi**2 and corresponding values of parameters to file
                if (WriteChi2Flag) then
                    NumberLinesChi2 = NumberLinesChi2 + 1
                    k = 0
                    FormattedParmValues(:)(:) = ""
                    Do i = 1, ma
                        if (ia(i)) then
                            k = k + 1
                            a(i) = fitparam(k)

                            ! Debug:
                            !   print*,'fitparam(k) = ',k,fitparam(k)


                            !< build list with fit parameters
                            HelpString = ""
                            call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, i)
                            if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                                .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(i))
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) int(a(i))
                                endif
                            else
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(i)
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) a(i)
                                endif
                            endif
                            FormattedParmValues(k) = trim(adjustl(HelpString))
                        endif
                    end Do


                    !< write line of formatted parameter values to log file
                    !$omp critical
                    write(Chi2Channel,'(ES25.15,$)') chisqValues(ThreadNumber)
                    Do i = 1, numfit
                        write(Chi2Channel,'(1x,A,$)') " " // trim(adjustl(FormattedParmValues(i)))
                    end Do
                    write(Chi2Channel,*)
                    !$omp end critical
                endif
                return
            end subroutine ChiFunction


            !*********************************************************************************************************************************************
            !> subroutine: CompareTwoParameterSets
            !>
            !> compares to different paramter sets using the format specifications defined in registration file
            !>
            !>
            !> input variables:     parameternum:       total number of parameters (fixed + non-fixed)
            !>                      nfit:               number of non-fixed parameters (length of param_set_1 and _2)
            !>                      ia:                 logical list marking the optimized (non-fixed) parameters
            !>                      param_set_1:        first parameter set
            !>                      param_set_2:        second parameter set
            !>
            !> Output variables:    Equal_Flag:         Flag is true, if param_set_1 and param_set_2 are identical
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 14.10.2011
            !>
            subroutine CompareTwoParameterSets(Equal_Flag, parameternum, nfit, ia, param_set_1, param_set_2)
                !< compares to different paramter sets using the format specifications defined in registration file

                implicit none
                integer :: parameternum                                                     !< total number of parameters
                integer :: nfit                                                             !< number of free parameters
                integer :: n                                                                !< loop variable
                integer :: k, nfitCounter                                                   !< working variables
                integer :: NumInputFile_index, i_index, j_index                             !< working variables
                real*8, dimension(nfit) :: param_set_1                                      !< first parameter set
                real*8, dimension(nfit) :: param_set_2                                      !< second parameter set
                character(len=100) :: HelpString, HelpString2                               !< working variables
                logical :: Equal_Flag                                                       !< Flag is true, if param_set_1 and param_set_2 are identical
                logical :: IntegerTrue                                                      !< flag for identification of integer numbers
                logical, dimension(parameternum) :: ia                                      !< logical list marking the optimized (non-fixed) parameters


                ! Debug:
                !    print*,'>> param_set_1 = ', param_set_1
                !    print*,'>> param_set_2 = ', param_set_2


                Equal_Flag = .false.
                nfitCounter = 0
                k = 0
                Do n = 1, parameternum                                                      !< loop over all parameters
                    if (ia(n)) then
                        k = k + 1


                        !< determine parameter value within specified format for param_set_1(k)
                        HelpString = ""
                        call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, n)
                        if (IntegerTrue) then
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(param_set_1(k))
                            if (index(HelpString2, "*") > 0) then                           !< search for bad real number
                                write(HelpString2, *) int(param_set_1(k))
                            endif
                        else
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) param_set_1(k)
                            if (index(HelpString2, "*") > 0) then                           !< search for bad real number
                                write(HelpString2, *) param_set_1(k)
                            endif
                        endif


                        !< determine parameter value within specified format for param_set_2(k)
                        HelpString2 = ""
                        call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, n)
                        if (IntegerTrue) then
                            write(HelpString2, ParameterFormat(NumInputFile_index, i_index, j_index)) int(param_set_2(k))
                            if (index(HelpString2, "*") > 0) then                           !< search for bad real number
                                write(HelpString2, *) int(param_set_2(k))
                            endif
                        else
                            write(HelpString2, ParameterFormat(NumInputFile_index, i_index, j_index)) param_set_2(k)
                            if (index(HelpString2, "*") > 0) then                           !< search for bad real number
                                write(HelpString2, *) param_set_2(k)
                            endif
                        endif


                        !< compare value of both parameters
                        if (trim(adjustl(HelpString)) == trim(adjustl(HelpString2))) then
                            nfitCounter = nfitCounter + 1

                            ! Debug:
                            !    print*,'HelpString = ', trim(adjustl(HelpString))
                            !    print*,'HelpString2 = ', trim(adjustl(HelpString2))
                        endif
                    endif
                end Do


                !< if all parameter are identical set flag Equal_Flag
                if (nfitCounter == nfit) then
                    Equal_Flag = .true.
                endif
                return
            end subroutine CompareTwoParameterSets


            !*********************************************************************************************************************************************
            !> subroutine: SortChi2File
            !>
            !> reads in all calculated parameter vectors with the corresponding chi^2 values and sort theses parameter sets in ascending numerical order
            !> of chi^2
            !>
            !>
            !> input variables:     nfit:               number of non-fixed parameters (length of param_set_1 and _2)
            !>                      parameternum:       total number of parameters (fixed + non-fixed)
            !>                      ia:                 logical list marking the optimized (non-fixed) parameters
            !>                      a:                  all parameters
            !>
            !> Output variables:    None
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 14.10.2011
            !>
            subroutine SortChi2File(nfit, parameternum, ia, a)
                !< reads in all calculated parameter vectors with the corresponding chi^2 values and sort theses parameter vectors in ascending
                !< numerical order of chi^2

                implicit none
                integer :: i, j, k, m                                                       !< loop variables
                integer :: counter_SortChi2FileContents                                     !< working variables
                integer :: io_err, deallocstatus, allocstatus                               !< working variables
                integer :: nfit                                                             !< number of free parameters
                integer :: parameternum                                                     !< total number of parameters
                integer :: NumLines                                                         !< total number of lines in log-file
                integer :: NumInputFile_index                                               !< contains index for input file
                integer :: i_index                                                          !< contains index for i
                integer :: j_index                                                          !< contains index for j
                real*8, dimension(parameternum) :: a, a_copy                                !< all parameters
                real*8, allocatable, dimension(:) :: LogValuesChi2, IndexArray              !< array containing all values of chi**2 of log-file
                real*8, allocatable, dimension(:, :) :: Entries                             !< contents of log file
                real*8, allocatable, dimension(:, :) :: SortChi2FileContents                !< contents of chi^2 file
                real*8, dimension(nfit) :: LocalCopyParamVector1, LocalCopyParamVector2     !< needed for MPI version to avoid warning messages
                character(len=30) :: Number1, Number2                                       !< working variables
                character(len=100) :: HelpString                                            !< working variables
                character(len=512) :: line                                                  !< working variable
                logical :: IntegerTrue                                                      !< flag for identification of integer numbers
                logical :: AlreadyStored                                                    !< flag for SortChi2FileContents
                logical, dimension(parameternum) :: ia                                      !< flags for including/excluding parameter in the fit


                !< print what you do
                if (printflag) then
                    print '(" ")'
                    print '(" ")'
                    print '(9x,"Sorting chi2-log file .. ",$)'
                endif


                !< copy parameter set
                a_copy = a


                !< determine number of lines in the chi2 log-file
                rewind Chi2Channel
                NumLines = 0
                io_err = 0
                Do While (io_err == 0)
                    read(Chi2Channel,'(A)', iostat = io_err) line
                    NumLines = NumLines + 1
                end Do
                NumLines = NumLines - 1


                !< deallocate and allocate variables
                if (allocated(LogValuesChi2)) then
                    deallocate(LogValuesChi2, Entries, SortChi2FileContents, IndexArray, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        write(logchannel,*)
                        write(logchannel,'("Error in subroutine SortChi2File:")')
                        write(logchannel,'(2x,"Can not deallocate variables LogValuesChi2 etc.")')
                        write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                        write(logchannel,*)
                        write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                        write(logchannel,'(" ")')
                        write(logchannel,'("Program aborted!")')
                        close(logchannel)

                        print '(" ")'
                        print '("Error in subroutine SortChi2File:")'
                        print '(2x,"Can not deallocate variables LogValuesChi2 etc.")'
                        print '(2x,"Please close all other programs and restart the program!")'
                        print '(" ")'
                        print '("deallocstatus = ",I4)',deallocstatus
                        print '(" ")'
                        stop ' Program aborted!'
                    endif
                endif
                allocate(LogValuesChi2(NumLines), IndexArray(NumLines), Entries(NumLines, nfit + 1), SortChi2FileContents(NumLines, nfit + 1), &
                         stat = allocstatus)
                if (allocstatus /= 0) then
                    write(logchannel,'(" ")')
                    write(logchannel,'("Error in subroutine SortChi2File:")')
                    write(logchannel,'(2x,"Can not allocate variables LogValuesChi2 etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,'(" ")')
                    write(logchannel,'("allocstatus = ",I4)') allocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')
                    close(logchannel)

                    print '(" ")'
                    print '("Error in subroutine SortChi2File:")'
                    print '(2x,"Can not allocate variables LogValuesChi2 etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("allocstatus = ",I4)',allocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
                LogValuesChi2 = 0.d0
                IndexArray = 0.d0
                Entries = 0.d0
                SortChi2FileContents = 0.d0


                !< read contents of log-file
                rewind Chi2Channel
                Do i = 1, NumLines
                    read(Chi2Channel,*) Entries(i,:)
                    LogValuesChi2(i) = Entries(i, 1)
                    IndexArray(i) = i
                end Do


                !< sort chi**2 values in increasing order
                call sort2(NumLines, LogValuesChi2, IndexArray)


                !< write new order to file
                write(Number2,'(I30)') NumLines
                counter_SortChi2FileContents = 0
                SortChi2FileContents = 0.d0                                                 !< reset contents of chi2-file
                Do i = 1, NumLines


                    !< print what you do
                    if (printflag) then
                        write(Number1,'(I30)') i
                        print '(A,11x,"Sorting chi2-log file (",A,"/",A,") .. ",A,$)',char(13), trim(adjustl(Number1)), trim(adjustl(Number2)), char(13)
                    endif


                    !< compare current parameter set with previous stored parameter set
                    if (i == 1 .or. dabs(SortChi2FileContents(i, 1) - 1.d99) < 1.0) then
                        counter_SortChi2FileContents = counter_SortChi2FileContents + 1
                        m = int(IndexArray(i))
                        SortChi2FileContents(counter_SortChi2FileContents, :) = Entries(m, :)
                    else
                        m = int(IndexArray(i))
                        AlreadyStored = .false.
                        Do j = counter_SortChi2FileContents, 1, (-1)
                            if (SortChi2FileContents(j, 1) < LogValuesChi2(i)) then
                                exit
                            endif
                            LocalCopyParamVector1(:) = SortChi2FileContents(j, 2:)
                            LocalCopyParamVector2(:) = Entries(m, 2:)
                            call CompareTwoParameterSets(AlreadyStored, parameternum, nfit, ia, LocalCopyParamVector1(:), LocalCopyParamVector2(:))
                            if (AlreadyStored) then
                                exit
                            endif
                        end Do
                        if (.not. AlreadyStored) then
                            counter_SortChi2FileContents = counter_SortChi2FileContents + 1
                            m = int(IndexArray(i))
                            SortChi2FileContents(counter_SortChi2FileContents, :) = Entries(m, :)
                        endif
                    endif
                end Do


                !< write chi2 file
                rewind Chi2Channel
                write(Chi2Channel,'(4x,"Nr.:",15x,"chi**2:",5x,"parameters:")')
                Do i = 1, counter_SortChi2FileContents
                    write(Chi2Channel,'(I8,2x,ES20.10,$)') i, SortChi2FileContents(i, 1)


                    !< build list with fit parameters
                    k = 0
                    Do j = 1, parameternumber
                        if (ia(j)) then
                            k = k + 1
                            a_copy(j) = SortChi2FileContents(i, k + 1)
                            HelpString = ""
                            call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, j)
                            if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                                .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a_copy(j))
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) int(a_copy(j))
                                endif
                            else
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a_copy(j)
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) a_copy(j)
                                endif
                            endif
                            write(Chi2Channel, '(5x,A,$)') trim(HelpString)
                        endif
                    end Do
                    write(Chi2Channel, *)
                end Do


                !< print what you do
                if (printflag) then
                    print '(A,11x,"Sorting chi2-log file .. done!",120(" "),A,$)', char(13), char(13)
                    print '(" ")'
                endif
                return
            end subroutine SortChi2File


            !*********************************************************************************************************************************************
            !> subroutine: PlotFitFunction
            !>
            !> plots the current model function for each iteration step using gplot.sh stored in "Modules/gnuplot/"
            !>
            !>
            !> input variables:     nfit:               number of non-fixed parameters (length of param_set_1 and _2)
            !>                      parameternum:       total number of parameters (fixed + non-fixed)
            !>                      ia:                 logical list marking the optimized (non-fixed) parameters
            !>                      a:                  all parameters
            !>
            !> Output variables:    None
            !>
            !>
            !> \author Thomas Moeller
            !>
            !> \date 14.10.2011
            !>
            subroutine PlotFitFunction(InitPlotFlag, xAxisLabel, yAxisLabel, zAxisLabel)
                !< plots the current model function for each iteration step using gplot.sh stored in "Modules/gnuplot/"

                implicit none
                integer :: NumberFile, i, j                                                 !< loop variables
                integer :: io_err, open_stat                                                !< i/o variables
                character(len=30) :: Number1, Number2                                       !< working variables
                character(len=256) :: xAxisLabel, yAxisLabel, zAxisLabel                    !< axis labels
                character(len=512) :: WorkingDirectory                                      !< current job directory
                character(len=8192) :: CommandString                                        !< command string for call of gnuplot script
                character(len=8192) :: filename                                             !< complete path and file name for plot_data_NUM.dat file(s)
                logical :: InitPlotFlag                                                     !< save plot to file?


                !< get path of current working directory
                write(Number1,'(I30)') JobID                                                !< write JobID to string
                WorkingDirectory = trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(Number1)) // "/"


                !< stop plot program if it is already started to avoid i/o problems
                if (.not.InitPlotFlag) then
                    if (len_trim(adjustl(PlotPID)) /= 0) then
                        CommandString = "kill -19 " // trim(adjustl(PlotPID)) // " 1>/dev/null 2>&1"
                        call system(trim(adjustl(CommandString)))

                        ! Debug:
                        ! print*,'len_trim(adjustl(PlotPID)) = ', len_trim(adjustl(PlotPID))
                        ! print "CommandString = ", trim(adjustl(CommandString))
                    endif
                endif


                !< construct plot_data_NUM.dat files
                Do NumberFile = 1, NumberExpFiles                                           !< loop over exp. data files
                    write(Number2,'(I30)') NumberFile                                       !< write exp. file number to string


                    !< construct path and file name for plot_data_NUM.dat file and open channel
                    filename = trim(adjustl(WorkingDirectory)) // "plot_data_" // trim(adjustl(Number2)) // ".dat"
                    open(111, file = trim(adjustl(filename)))                               !< open channel for plot_data_NUM.dat file
                    Do i = 1, lengthexpdata(NumberFile)                                     !< loop over all line of current exp. data file


                        !< construct content of plot_data_NUM.dat file
                        write(111,*) (expdatax(NumberFile, i, j), j = 1, 1 + 0*NumberXColumns(NumberFile)), expdatay(NumberFile, i, 1), &
                                     BestSitesModelValues(1, NumberFile, i, 1), BestSitesChi2Values(1, NumberFile, i, 1)
                    end Do
                    close(111)
                end Do


                !< execute plot script
                if (InitPlotFlag) then
                    PlotPID = ""
                    write(Number2,'(I30)') NumberExpFiles
                    CommandString = "python Modules/magix-parts_python/PlotPerIteration.py -dGTKAgg " // trim(adjustl(WorkingDirectory)) &
                                    // " " // trim(adjustl(Number2)) // " " &
                                    // char(34) // trim(adjustl(xAxisLabel)) // char(34) // " " &
                                    // char(34) // trim(adjustl(yAxisLabel)) // char(34) // " " &
                                    // char(34) // trim(adjustl(zAxisLabel)) // char(34) // " 1>/dev/null 2>&1 &"
                    call system(trim(adjustl(CommandString)))

                    ! Debug:
                    ! print*,"CommandString = ", trim(adjustl(CommandString))
                    ! stop
                else


                    !< read process id of plot.py program from file
                    io_err = 0
                    open_stat = 0
                    filename = trim(adjustl(WorkingDirectory)) // "pid.txt"
                    open(111, file = trim(adjustl(filename)), iostat = open_stat)           !< open output file
                    if (open_stat == 0) then
                        read(111,'(A)', iostat = io_err) PlotPID
                        close(111)


                        !< send wake up signal to plot.py program
                        if (trim(adjustl(PlotPID)) /= "") then
                            CommandString = "kill -18 " // trim(adjustl(PlotPID)) // " 1>/dev/null 2>&1"

                            ! Debug:
                            ! print*,"CommandString = ", trim(adjustl(CommandString))

                            call system(trim(adjustl(CommandString)))
                        endif
                    endif
                endif


                ! Debug:
                ! print*,'>', trim(adjustl(CommandString)), '<'
                ! stop


                ! call system(trim(adjustl(CommandString)))
                return
            end subroutine PlotFitFunction
end Module FunctionCalling
!*********************************************************************************************************************************************************

