!*********************************************************************************************************************************************************
!> Module: FunctionCallPython
!>
!>
!>  This module contains the subroutines for the module function interface
!>  Copyright (C) 2009 - 2016  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - Module FunctionCallPython:                contains some global variables and subroutines
!>      - subroutine DetermineModelFunction:        determines the model function for a given parameter set
!>      - Module Algorithm:                         module contains the main subroutine used to start the different versions of the Bees algorithm
!>      - subroutine MainAlg:                       main subroutine which starts the Bees algorithm
!>      - Module FunctionCall:                      module contains the start subroutine
!>      - subroutine MainCaller:                    main subroutine which starts DetermineModelFunction subroutine
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    09.06.2009  Initial version
!>  T. Moeller    13.01.2012  Updated version
!>  T. Moeller    20.08.2014  myXCLASS (model) optimized version
!>  T. Moeller    01.09.2014  modified and restructured for GPU
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!> Module: FunctionCallPython
!>
!>         Module contains some global variables and subroutines
!>
!>
!> \author Thomas Moeller
!>
!> \date 09.06.2010
!>
Module FunctionCallPython

    use FunctionCalling
    use Model

    logical, allocatable, dimension(:) :: ia                                                !< array indicating the parameter which should be optimized


    contains

        !*************************************************************************************************************************************************
        !> subroutine: DetermineModelFunction
        !>
        !> determines the model function for a given parameter set
        !>
        !> input variables:     ma:                     total number of all parameter
        !>                      nfit:                   number of parameters which should be optimized
        !>                      number_param_sets:      number of parameter sets
        !>                      NumFile:                number of input files
        !>                      MaxL:                   max. total length
        !>                      MaxCol:                 max. number of columns
        !>                      parameter_vector_set:   the required parameter set(s)
        !>
        !> output variables:    funcValues:             output values
        !>
        !>
        !> \author Irina Bernst, Thomas Moeller
        !>
        !> \date 09.06.2010
        !>
        subroutine DetermineModelFunction(ma, nfit, number_param_sets, NumFile, MaxL, MaxCol, FitFunctionOut, Chi2Values, &
                                          parameter_vector_set, funcValues, alpha)

            use Variables

            implicit none
            integer :: i, k, m                                                              !< loop variables
            integer :: ma                                                                   !< total number of all parameters
            integer :: nfit                                                                 !< (= effective parameter number) number of model
            integer :: number_param_sets                                                    !< number of parameter sets
            integer :: NumFile, MaxL, MaxCol                                                !< working variables
            real*8 :: sqrtarg                                                               !< working variable for the determination of the error
            real*8, dimension(1) :: chi2ValuesVector                                        !< one chi2 value
            real*8, dimension(1, nfit) :: PureParameterVector                               !< array containing only the optimized parameter
            real*8, dimension(ma) :: a, acopy                                               !< current parameter vector and copy of parameter vector
            real*8, dimension(ma, ma) :: alpha                                              !< matrix alpha
            real*8, dimension(nfit) :: beta2                                                !< beta2 array
            real*8, dimension(number_param_sets) :: funcValues                              !< chi2 value vector
            real*8, dimension(number_param_sets, ma) :: parameter_vector_set                !< parameter sets
            real*8, dimension(NumFile, MaxL, MaxCol) :: FitFunctionOutDummy                 !< values of the model function at the calculated points
            real*8, dimension(NumFile, MaxL, MaxCol) :: Chi2ValuesDummy                     !< values of the model function at the calculated points
            real*8, dimension(number_param_sets, NumFile, MaxL, MaxCol) :: FitFunctionOut   !< values of the model function at the calculated points
            real*8, dimension(number_param_sets, NumFile, MaxL, MaxCol) :: Chi2Values       !< values of the model function at the calculated points
            logical :: PlotFlagCopy                                                         !< copy of plotiteration flag
            logical :: OutOfRangeFlag                                                       !< flag for parameter is out of range


            !< reset output variables
            FitFunctionOut = 0.d0
            Chi2Values = 0.d0
            alpha = 0.d0


            !< remember plotiteration flag
            PlotFlagCopy = .false.
            if (PlotIterationFlag) PlotFlagCopy = .true.


            !< calculate chi2 values
            Do i = 1, number_param_sets
                a = parameter_vector_set(i, 1:parameternumber)                                  !< get current parameter vector


                !< test if all parameter values are within the given ranges and define PureParameterVector array
                OutOfRangeFlag = .false.
                PureParameterVector = 0.d0
                m = 0
                Do k = 1, parameternumber
                    if (ia(k)) then
                        if (a(k) < paramset(3, k) .or. paramset(4, k) < a(k)) then
                            OutOfRangeFlag = .true.
                            exit
                        else
                            m = m + 1
                            PureParameterVector(1, m) = a(k)
                        endif
                    endif
                end Do
                if (.not. OutOfRangeFlag) then
                    chi2ValuesVector = 0.d0


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< determine gradient and covariance matrix using subroutine for Levenberg-Marquardt algorithm
                    if (Gradientflag) then
                        PlotIterationFlag = .true.
                        FitFunctionOutDummy = 0.d0
                        Chi2ValuesDummy = 0.d0
                        alpha = 0.d0
                        beta2 = 0.d0
                        call ModelCalcChiFunctionLM(ma, paramset(1, :), ia, nfit, PureParameterVector, MaxColX, NumFile, MaxL, MaxCol, &
                                                    FitFunctionOutDummy, Chi2ValuesDummy, alpha, beta2)
                        funcValues(i) = chisqValues(0)
                        !< alpha contains the non-inverted covariance matrix


                    !<------------------------------------------------------------------------------------------------------------------------------------
                    !< determine gradient and covariance matrix using subroutine for Levenberg-Marquardt algorithm
                    else
                        PlotIterationFlag = .true.
                        call ModelCalcChiFunctionGeneral(ma, ia, paramset(1, :), 1, nfit, NumFile, MaxL, MaxCol, PureParameterVector, chi2ValuesVector)
                        funcValues(i) = chi2ValuesVector(1)
                    endif
                    FitFunctionOut(i ,:, :, :) = BestSitesModelValues(1, :, :, :)
                    Chi2Values(i, :, :, :) = BestSitesChi2Values(1, :, :, :)
                else
                    funcValues(i) = 1.d+99
                endif
            end Do


            !< restore plot iteration flag
            PlotIterationFlag = .false.
            if (PlotFlagCopy) PlotIterationFlag = .true.


            return
        end subroutine DetermineModelFunction
end Module FunctionCallPython
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!> Module: Algorithm
!>
!>         Module contains the main subroutine used to start the different versions of the Bees algorithm
!>
!>
!> \author Thomas Moeller
!>
!> \date 26.08.2014
!>
Module Algorithm

    use Variables
    use FunctionCallPython

    implicit none

    contains


        !*************************************************************************************************************************************************
        !> subroutine: MainAlg
        !>
        !> NOTE, do not use capital letters in the name of the subroutine which is called by python
        !>
        !>
        !> main subroutine which starts DetermineModelFunction subroutine
        !>
        !>
        !> input variables:         printflagNum:           flag for screen output 1 (=yes) or 0 (=no)
        !>                          DeterminationChi2:      method being used for the determination of chi^2
        !>                          PlotIteration:          plot model function for each iteration set 1(=yes) or 0(=no)
        !>                          PlotTypeOrg:            get type of plot
        !>                          NumberInputFilesorg:    number of input files for the external model program
        !>                          NumberOutputFilesOrg:   number of output files for the external model program
        !>                          ParallelizationFlagorg: contains the number of processors used for parallelization
        !>                          JobIDorg:               job identification number
        !>                          MaxInputLinesOrg:       max number of lines in an input file
        !>                          MaxParameterOrg:        max number of parameters in a line of an input file
        !>                          RenormalizedChi2Org:    flag for using renormalized chi**2
        !>                          fitlog:                 path for log-file containing the current values of chi**2
        !>                          currentpathorg:         path of the working directory
        !>                          FitParameterNameOrg:    array containing the names of the model parameters
        !>                          FitParameterValueLineOrg:   array containing the values of the model parameters as string
        !>                          CalculationMethodOrg:   method of computation (at once or point-to-point)
        !>                          xAxisLabel:             label of the x-axis (for plot)
        !>                          yAxisLabel:             label of the y-axis (for plot)
        !>                          zAxisLabel:             label of the z-axis (for plot)
        !>                          PathStartScriptOrg:     path and name of the start script for calling model function
        !>                          ExeCommandStartScriptOrg:   command for calling model function
        !>                          parametersetorg:        the complete set of paramters (incl. flags and limits)
        !>                          parametersetorg:        array containing ParamSetCounter sets of values
        !>                          expdataxorg:            array containing the experimental x side
        !>                          expdatayorg:            array containing the experimental y side
        !>                          expdataerrororg:        array containing the experimental error of the y side
        !>                          NumberRangesOrg:        number of y-columns for each experimental file
        !>                          MinRangeOrg:            array containing the minimal exp. ranges
        !>                          MaxRangeOrg:            array containing the maximal exp. ranges
        !>                          NumberXColumnsOrg:      number of x-columns for each experimental file
        !>                          NumberYColumnsOrg:      number of y-columns for each experimental file
        !>                          lengthexpdataorg:       number of lines in experimental data
        !>                          MaxRangeNumber:         max. number of ranges
        !>                          NumFileOrg:             number of experimental files
        !>                          MaxLengthOrg:           max length of experimental data
        !>                          MaxColXOrg:             number of columns concerning to the experimental x side
        !>                          MaxColYOrg:             number of columns concerning to the experimental y side
        !>                          parameternum:           number of model parameter
        !>                          ParamSetCounter:        which positions should be used
        !>
        !> output variables:        ValueOfChi2:            value of chi^2
        !>                          FitFunctionOut:         values of the model function at the calculated points
        !>                          Chi2Values:             values of the chi^2 function at the calculated points
        !>
        subroutine MainAlg(printflagNum, LastAlgorithmNum, calstatus, FitFunctionOut, Chi2Values, chilm, NumberOfFitAlgorithms, numiter, AlgCounter, &
                           ParamSetCounter, GeneralAlgorithmSettings, DeterminationChi2, PlotIteration, PlotType, fitlog, NumberInputFilesorg, &
                           NumberOutputFilesOrg, ParallelizationFlagorg, JobIDorg, MaxInputLinesOrg, MaxParameterOrg, RenormalizedChi2Org, &
                           currentpathorg, FitParameterNameLocal, FitParameterValueLocal, CalculationMethodOrg, xAxisLabel, yAxisLabel, zAxisLabel, &
                           PathStartScriptOrg, ExeCommandStartScriptOrg, parametersetorg, FinalParameterSet, expdataxorg, expdatayorg, expdataerrororg, &
                           NumberRangesOrg, MinRangeOrg, MaxRangeOrg, NumberXColumnsOrg, NumberYColumnsOrg, lengthexpdataorg, MaxRangeNumber, &
                           NumFileOrg, MaxLengthOrg, MaxColXOrg, MaxColYOrg, parameternum)


            implicit none
            ! ********** input variables **********
            integer :: parameternum                                                         !< number of model parameter
            integer :: NumberOfFitAlgorithms                                                !< total number of all algorithms in the chain
            integer :: numiter                                                              !< max. number of iterations
            integer :: AlgCounter                                                           !< (not used) counter for calls
            integer :: NumFileOrg                                                           !< number of experimental files
            integer, dimension(NumFileOrg) :: lengthexpdataorg                              !< number of lines in experimental data
            integer, dimension(NumFileOrg) :: NumberXColumnsOrg                             !< number of x-columns for each experimental file
            integer, dimension(NumFileOrg) :: NumberYColumnsOrg                             !< number of y-columns for each experimental file
            integer, dimension(NumFileOrg) :: NumberRangesOrg                               !< number of y-columns for each experimental file
            integer :: MaxColXOrg                                                           !< number of columns concerning to the experimental x side
            integer :: MaxColYOrg                                                           !< number of columns concerning to the experimental y side
            integer :: MaxLengthOrg                                                         !< max length of experimental data
            integer :: printflagNum                                                         !< flag for screen output 1 (=yes) or 0 (=no)
            integer :: LastAlgorithmNum                                                     !< flag for screen output 1 (=yes) or 0 (=no)
            integer :: DeterminationChi2                                                    !< method being used for the determination of chi^2
            integer :: PlotIteration                                                        !< plot model func. for each iteration set 1(=yes) or 0(=no)
            integer :: PlotTypeOrg                                                          !< get type of plot
            integer :: NumberInputFilesorg                                                  !< number of input files for the external model program
            integer :: NumberOutputFilesOrg                                                 !< number of output files for the external model program
            integer :: ParallelizationFlagorg                                               !< contains the number of processors used for parallelization
            integer :: JobIDorg                                                             !< job identification number
            integer :: MaxInputLinesOrg                                                     !< max number of lines in an input file
            integer :: MaxParameterOrg                                                      !< max number of parameters in a line of an input file
            integer :: RenormalizedChi2Org                                                  !< flag for using renormalized chi**2
            integer :: MaxRangeNumber                                                       !< max. number of ranges
            integer :: ParamSetCounter                                                      !< which positions should be used
            real*8 :: chilm                                                                 !< user defined abort criteria for chi**2
            real*8, dimension(ParamSetCounter) :: ValueOfChi2                               !< chi2 value vector
            real*8, dimension(15) :: GeneralAlgorithmSettings                               !< special algorithm settings
            real*8, dimension(ParamSetCounter, parameternum) :: parameter_vector            !<
            real*8, dimension(NumFileOrg, MaxLengthOrg, MaxColXOrg) :: expdataxorg          !< array containing the experimental x side
            real*8, dimension(NumFileOrg, MaxLengthOrg, MaxColYOrg) :: expdatayorg          !< array containing the experimental y side
            real*8, dimension(NumFileOrg, MaxLengthOrg, MaxColYOrg) :: expdataerrororg      !< array containing the experimental error of the y side
            real*8, dimension(NumFileOrg, MaxRangeNumber, MaxColXOrg) :: MinRangeOrg        !< array containing the minimal exp. ranges
            real*8, dimension(NumFileOrg, MaxRangeNumber, MaxColXOrg) :: MaxRangeOrg        !< array containing the maximal exp. ranges
            character(len=8192) :: fitlog                                                   !< path for log-file containing the current values of chi**2
            character(len=256) :: xAxisLabel                                                !< label of the x-axis (for plot)
            character(len=256) :: yAxisLabel                                                !< label of the y-axis (for plot)
            character(len=256) :: zAxisLabel                                                !< label of the z-axis (for plot)
            character(len=20) :: CalculationMethodOrg                                       !< method of computation
            character(len=8192) :: PathStartScriptOrg                                       !< command for calling model function
            character(len=8192) :: ExeCommandStartScriptOrg                                 !< command for calling model function
            character(len=8192) :: currentpathorg                                           !< path of the working directory
            character(len=512), dimension(parameternum) :: FitParameterNameLocal            !< array containing the names of the model parameters
            character(len=512), dimension(parameternum) :: FitParameterValueLocal           !< array containing the values of the model parameters as


            ! ********** in/output variables **********
            real*8, dimension(4, parameternum) :: parametersetorg                           !< the non-optimized (initial parameter set)


            ! ********** output variables **********
            integer :: calstatus                                                            !< the following line is necessary for f2py
            real*8, dimension(ParamSetCounter, parameternum) :: FinalParameterSet           !< array containing the optimized parameter set
            real*8, dimension(ParamSetCounter, NumFileOrg, MaxLengthOrg, MaxColYOrg) :: FitFunctionOut !< values of the model func. at the calculated pts.
            real*8, dimension(ParamSetCounter, NumFileOrg, MaxLengthOrg, MaxColYOrg) :: Chi2Values     !< values of the model func. at the calculated pts.


            !<********** working variabels **********
            integer, parameter :: WorkChannel = 2151                                        !< define channel for writing chi2 values to tmp file
            integer :: i, j, k, ii, jj, NumInputFile_index, i_index, j_index                !< working variables
            integer :: nfit, ma, MaxLength, PlotType, CurrIter                              !< working variables
            integer :: stat1, stat2                                                         !< status variables for i/o
            integer :: ok                                                                   !< status of calculation
            integer :: actualiteration                                                      !< contains the current iteration within the iteration loop
            integer :: flag                                                                 !< working variable used within the iteration loop
            integer :: allocstatus, deallocstatus                                           !< working variables for allocation/deallocation
            real*8, dimension(MaxColXOrg) :: posdatexp                                      !< array defining the experimental x point
            real*8, dimension(parameternum, parameternum) :: alpha                          !< covariance matrix
            character(len=30) :: Number1, Number2                                           !< working variable
            character(len=100) :: HelpString                                                !< help string for writing parameter in correct format
            character(len=512) :: WorkingDirectory                                          !< current job directory
            character(len=5196) :: ListParamFormated                                        !< list of all formated free parameters
            character(len=8192) :: filename, filename2                                      !< complete path and file name for plot_data_NUM.dat file(s)
            logical :: IntegerTrue                                                          !< flag for identification of integer numbers
            logical :: InitPlotFlag                                                         !< used for final plot
            logical :: PlotFunctionFlag                                                     !< flag for plotting model function


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< set print flag and last algorithm flag
            if (printflagNum == 1) then                                                     !< set printflag
                printflag = .true.
            else
                printflag = .false.
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy contents of some input variables to module variables
            NumberExpFiles = NumFileOrg                                                     !< copy number of experimental files to global variable
            currentpath = trim(adjustl(currentpathorg))                                     !< copy path of working directory to module variable without
                                                                                            !< trailing and leading blanks
            MaxColX = MaxColXOrg                                                            !< copy number of columns of the experimental x data to
                                                                                            !< module variable
            MaxColY = MaxColYOrg                                                            !< copy number of columns of the experimental y data to
                                                                                            !< module variable
            MaxLength = MaxLengthOrg                                                        !< copy max. number of lines
            MaxExpLength = MaxLength                                                        !< copy max. number of exp. data points
            MaxNumberRanges = MaxRangeNumber                                                !< copy of max. number of data ranges in a exp. data file
            parameternumber = parameternum                                                  !< copy input variable containing the number of parameters
                                                                                            !< to module variable
            DetChi2 = DeterminationChi2                                                     !< copy method of chi**2 determination
            NumberInputFiles = NumberInputFilesorg                                          !< copy number of input files for the external program to
                                                                                            !< global variable
            NumberOutputFiles = NumberOutputFilesOrg                                        !< copy number of output files for the external program to
                                                                                            !< global variable
            ParallelizationFlag = ParallelizationFlagorg                                    !< copy number of used processors to global variable
            JobID = JobIDorg                                                                !< copy job-ID number to global variable
            PlotType = PlotTypeOrg                                                          !< copy flag for plotting
            CurrIter = numiter                                                              !< copy number of current iteration
            MaxInputLines = MaxInputLinesOrg                                                !< copy max number of input lines in an input file
            QualityLimit = ParamSetCounter                                                  !< which positions should be used
            MaxParameter = MaxParameterOrg                                                  !< copy max number of parameters in a line of an input file
            RenormalizedChi2 = .true.                                                       !< define flag for using renormalized chi**2
            if (RenormalizedChi2Org /= 1) then
                RenormalizedChi2 = .false.
            endif
            PlotIterationFlag = .false.
            if (PlotIteration == 0) PlotIterationFlag = .true.

            ! Debug:
            !    print*,'PathStartScriptOrg = ',trim(PathStartScriptOrg)
            !    print*,'ExeCommandStartScriptOrg = ',trim(ExeCommandStartScriptOrg)
            !    print*,'FitFktInputOrg = ',trim(FitFktInputOrg)
            !    print*,'MaxNumberParameter = ',MaxNumberParameter
            !    print*,'NumFileOrg = ',NumFileOrg
            !    print*,'MaxColXOrg = ',MaxColXOrg
            !    print*,'MaxColYOrg = ',MaxColYOrg
            !    Do i=1,NumFileOrg
            !        print*,'    Experimental file: i = ',i
            !        print*,'    lengthexpdataorg(i) = ',lengthexpdataorg(i)
            !        print*,'    NumberYColumnsOrg(i) = ',NumberYColumnsOrg(i)
            !        print*,'    expdataxorg(i,1:5,1) = ',expdataxorg(i,1:5,1)
            !        print*,'    expdatayorg(i,1:5,1) = ',expdatayorg(i,1:5,1)
            !        print*,'    expdataerrororg(i,1:5,1) = ',expdataerrororg(i,1:5,1)
            !    end Do
            !    print*,'fitlog = ',trim(fitlog)
            !    print*,'currentpathorg = ',trim(adjustl(currentpathorg))
            !    print*,'len(FitParameterNameOrg) = ',len(FitParameterNameOrg
            !    print*,'FitParameterNameOrg = ',FitParameterNameOrg
            !    print*,'FitParameterValueLineOrg = ',FitParameterValueLineOrg
            !    print*,"parametersetorg(1,:) = ",parametersetorg(1,:)
            !    print*,"parametersetorg(2,:) = ",parametersetorg(2,:)
            !    print*,"parametersetorg(3,:) = ",parametersetorg(3,:)
            !    print*,"parametersetorg(4,:) = ",parametersetorg(4,:)
            !    print*,"RenormalizedChi2 = ",RenormalizedChi2
            !    return


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< set temp-directory
            TempDirectory = " "
            CALL GetEnv('MAGIXTempDirectory', TempDirectory)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< open file containing the values of chi**2 and the corresponding values of the parameters
            WriteChi2Flag = .true.
            NumberLinesChi2 = 0


            !< get path of current working directory
            write(Number1,'(I30)') JobID                                                    !< write JobID to string
            WorkingDirectory = trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(Number1)) // "/"
            filename = trim(adjustl(WorkingDirectory)) // "log_chi2_single_call.dat"
            open(Chi2Channel, file = trim(adjustl(filename)), status='replace')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< deallocate if necessary and print error message if necessay
            if (allocated(expdatax)) then
                deallocate(expdatax, expdatay, expdatae, lengthexpdata, NumberXColumns, NumberYColumns, FirstPointExpData, LastPointExpData, &
                           NumberRanges, MinRange, MaxRange, ExpData_reversed_flag, stat = deallocstatus)
                if (deallocstatus /= 0) then                                                !< is all ok?
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine MainAlg:")')
                    write(logchannel,'(2x,"Can not deallocate variables expdatax etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine MainAlg:")'
                    print '(2x,"Can not deallocate variables expdatax etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< allocate memory for variables, clear content of the variables and print error message if necessay
            allocate(expdatax(NumberExpFiles, MaxLength, MaxColX), expdatay(NumberExpFiles, MaxLength, MaxColY), &
                     expdatae(NumberExpFiles, MaxLength, MaxColY), lengthexpdata(NumberExpFiles), NumberXColumns(NumberExpFiles), &
                     NumberYColumns(NumberExpFiles), FirstPointExpData(NumberExpFiles, MaxColX), LastPointExpData(NumberExpFiles, MaxColX), &
                     NumberRanges(NumberExpFiles), MinRange(NumberExpFiles, MaxRangeNumber, MaxColX), MaxRange(NumberExpFiles, MaxRangeNumber, MaxColX), &
                     ExpData_reversed_flag(NumberExpFiles), stat = allocstatus)
            if (allocstatus /= 0) then                                                      !< is all ok?
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not allocate variables expdatax etc.")')
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not allocate variables expdatax,expdatay etc.")'
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("allocstatus = ",I4)',allocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            MaxRangeNumber = MaxRangeNumber - 1                                             !< get real value
            expdatax = 0.d0
            expdatay = 0.d0
            expdatae = 0.d0
            lengthexpdata = 0
            NumberXColumns = 0
            NumberYColumns = 0
            FirstPointExpData = 1.d99
            LastPointExpData = -1.d99
            NumberRanges = 0
            MinRange = 0.d0
            MaxRange = 0.d0
            ExpData_reversed_flag = .false.


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy input variables to module variabels
            expdatax(:,:,:) = expdataxorg(:,:,:)
            expdatay(:,:,:) = expdatayorg(:,:,:)
            expdatae(:,:,:) = expdataerrororg(:,:,:)
            lengthexpdata = lengthexpdataorg
            NumberXColumns = NumberXColumnsOrg
            NumberYColumns = NumberYColumnsOrg
            CalculationMethod = CalculationMethodOrg
            PathStartScript = PathStartScriptOrg
            ExeCommandStartScript = ExeCommandStartScriptOrg
            NumberRanges = NumberRangesOrg
            MinRange = MinRangeOrg
            MaxRange = MaxRangeOrg


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine first and last point of each exp. data file
            Do i = 1, NumberExpFiles
                Do j = 1, lengthexpdata(i)
                    ii = 0
                    jj = 0
                    Do k = 1,NumberXColumns(i)
                        if (expdatax(i, j, k) <= FirstPointExpData(i, k)) then
                            ii = ii + 1
                        endif
                        if (expdatax(i, j, k) >= LastPointExpData(i, k)) then
                            jj = jj + 1
                        endif
                    end Do
                    if (ii == NumberXColumns(i)) then
                        FirstPointExpData(i, 1:NumberXColumns(i)) = expdatax(i, j, 1:NumberXColumns(i))
                    endif
                    if (jj == NumberXColumns(i)) then
                        LastPointExpData(i, 1:NumberXColumns(i)) = expdatax(i, j, 1:NumberXColumns(i))
                    endif
                end Do


                !< check output file starts with the highest x-column value interchange FirstPointOutputFile and LastPointOutputFile
                ii = 0
                Do k = 1, NumberXColumns(i)
                    if (expdatax(i, 1, k) >= expdatax(i, lengthexpdata(i), k)) then
                        ii = ii + 1
                    endif
                end Do
                if (ii == NumberXColumns(i)) then
                    ExpData_reversed_flag(i) = .true.
                endif

                ! Debug:
                ! print*,' '
                ! print*,'File = ',i
                ! print*,'FirstPointExpData(i, 1:NumberXColumns(i)) = ', FirstPointExpData(i, 1:NumberXColumns(i))
                ! print*,'LastPointExpData(i, 1:NumberXColumns(i)) = ', LastPointExpData(i, 1:NumberXColumns(i))
                ! print*,'##########################################'
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy number of parameters required for the model function to working variable
            nfit = sum(parametersetorg(2, :))                                               !< number of parameters which should be optimized
            ma = parameternumber
            UseCalculationReduction = .true.                                                !< activate calculation reduction
            CurrentNumberLinesCalcReduction = 0                                             !< reset number of lines


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< deallocate/allocate memory for some working variables, clear contents of the variables and print error message if necessary
            if (allocated(paramset)) then
                deallocate(paramset, FitParameterName, FitParameterValue, ia, ModelFunction, chisqValues, BestSitesParamSet, BestSitesModelValues, &
                           BestSitesChi2Values, ConverterInfit, AtOnceFunction, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    write(logchannel,*)
                    write(logchannel,'(11x,"Error in subroutine MainAlg:")')
                    write(logchannel,'(13x,"Can not deallocate variables paramset etc.")')
                    write(logchannel,'(13x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'(13x,"deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'(13x,"Program aborted!")')

                    print '(" ")'
                    print '(11x,"Error in subroutine MainAlg:")'
                    print '(13x,"Can not deallocate variables paramset etc.")'
                    print '(13x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '(13x,"deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif
            allocate(paramset(4, parameternumber), ia(parameternumber), FitParameterName(parameternumber), FitParameterValue(parameternumber), &
                     ModelFunction(nfit + 1, NumberExpFiles, MaxColY, MaxLength), chisqValues(0:0), ConverterInfit(nfit), &
                     AtOnceFunction(0:ParallelizationFlag - 1, NumberExpFiles, MaxColY, MaxLength), &
                     BestSitesParamSet(ParamSetCounter, nfit + 1), BestSitesModelValues(ParamSetCounter, NumberExpFiles, MaxLength, MaxColY), &
                     BestSitesChi2Values(ParamSetCounter, NumberExpFiles, MaxLength, MaxColY), stat = allocstatus)
            if (allocstatus /= 0) then
                write(logchannel,'(" ")')
                write(logchannel,'(11x,"Error in subroutine MainAlg:")')
                write(logchannel,'(13x,"Can not allocate variables paramset etc.")')
                write(logchannel,'(13x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'(13x,"allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'(13x,"Program aborted!")')

                print '(" ")'
                print '(11x,"Error in subroutine MainAlg:")'
                print '(13x,"Can not allocate variables paramset etc.")'
                print '(13x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '(13x,"allocstatus = ",I4)',allocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            ConverterInfit = 0
            ia = .false.
            FitParameterName = FitParameterNameLocal
            FitParameterValue = FitParameterValueLocal
            ModelFunction = 0.d0
            chisqValues = 0.d0
            AtOnceFunction = 0.d0
            BestSitesParamSet = 0.d0
            BestSitesParamSet(:, 1) = 1.d99
            BestSitesModelValues = 0.d0
            BestSitesChi2Values = 0.d0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< save last Plot to png and eps file
            if (PlotIteration == 0) then


                !< read in best function values from file
                PlotFunctionFlag = .true.
                Do i = 1, ParamSetCounter
                    write(Number1,'(I30)') i
                    Do j = 1, NumberExpFiles
                        write(Number2,'(I30)') j
                        filename = trim(adjustl(WorkingDirectory)) // "best_model_function_values_call__" // trim(adjustl(Number1)) // "__" &
                                   // trim(adjustl(Number2)) // ".dat"
                        filename2 = trim(adjustl(WorkingDirectory)) // "best_chi2_values_call__" // trim(adjustl(Number1)) // "__" &
                                   // trim(adjustl(Number2)) // ".dat"


                        !< check if file exists
                        inquire(file = trim(adjustl(filename)), exist = PlotFunctionFlag, iostat = stat1)
                        if (PlotFunctionFlag) then
                            inquire(file = trim(adjustl(filename2)), exist = PlotFunctionFlag, iostat = stat2)
                        endif


                        !< read in data
                        if (PlotFunctionFlag) then
                            open(67878, file = trim(adjustl(filename)), status = 'old')
                            open(67979, file = trim(adjustl(filename2)), status = 'old')
                            Do k = 1, MaxLength
                                read(67878, *) BestSitesModelValues(i, j, k, 1)
                                read(67979, *) BestSitesChi2Values(i, j, k, 1)
                            end Do
                            close(67979)
                            close(67878)
                        else
                            exit
                        endif
                    end Do
                    if (.not. PlotFunctionFlag) exit
                end Do
                if (CurrIter == (-1) .or. PlotFunctionFlag) then
                    if (CurrIter == (-1)) then
                        InitPlotFlag = .true.
                    else
                        InitPlotFlag = .false.
                    endif
                    call PlotFitFunction(InitPlotFlag, xAxisLabel, yAxisLabel, zAxisLabel)
                endif
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy parameter set to module variable and define ia variable
            paramset = parametersetorg
            ia = .false.
            Do i = 1, parameternumber
                if (paramset(2, i) == 1) then
                    ia(i) = .true.
                endif
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< write registration mask for the fit model
            call RegistrationMask(ok)
            if (ok /= 0) then
                return
            endif


            !< define ia variable
            ConverterInfit = 0
            ia = .false.
            k = 0
            Do i = 1, parameternumber
                if (paramset(2, i) == 1) then
                    k = k + 1
                    ia(i) = .true.
                    ConverterInfit(k) = i
                endif
            end Do
            NumberFreeParameter = int(sum(paramset(2, :)))                                  !< determine number of free parameter


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize model program
            printflag = .false.
            call ModelInit
            if (printflagNum == 1) then                                                     !< set printflag
                printflag = .true.
            else
                printflag = .false.
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< start iteration ..
            flag = 0                                                                        !< set flag variable to 0
            actualiteration = 0                                                             !< set working variable containing the current iteration
                                                                                            !< number to 0

            !< write input files to log-files
            write(Number1,'(I30)') JobID                                                    !< write JobID to string
            WorkingDirectory = trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(Number1)) // "/"


            !< read in new parameter values from file
            open(WorkChannel, file = trim(adjustl(WorkingDirectory)) // "new-parameters.dat")
            Do k = 1, ParamSetCounter
                read(WorkChannel, *) parameter_vector(k, :)
            end Do
            close(WorkChannel, status = 'delete')


            !< get gradient flag
            Gradientflag = .false.                                                          !< we do not need the gradient of the function here
            inquire(file = trim(adjustl(WorkingDirectory)) // "gradientflag.dat", exist = Gradientflag, iostat = stat1)
            if (Gradientflag) then
                open(WorkChannel, file = trim(adjustl(WorkingDirectory)) // "gradientflag.dat")
                read(WorkChannel, *) GradientVariationValue
                close(WorkChannel, status = 'delete')
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< call subroutine to call pso algorithm
            alpha = 0.d0
            call DetermineModelFunction(parameternumber, nfit, ParamSetCounter, NumberExpFiles, MaxLength, MaxColY, FitFunctionOut, Chi2Values, &
                                        parameter_vector, ValueOfChi2, alpha)

            !< write errors to file
            if (Gradientflag) then
                open(WorkChannel, file = trim(adjustl(WorkingDirectory)) // "covariance-matrix.dat")
                Do k = 1, parameternumber
                    Do i = 1, parameternumber
                        write(WorkChannel, *) alpha(k, i)
                    end Do
                end Do
                close(WorkChannel)
            endif


            !< open file to chi2 channel
            open(WorkChannel, file = trim(adjustl(WorkingDirectory)) // "chi2Values.dat")
            Do k = 1, ParamSetCounter
                write(Number1,'(I30)') k                                                    !< write JobID to string


                !< write chi2 value to temp file
                write(WorkChannel, *) ValueOfChi2(k)


                !< update parametersetorg array
                parametersetorg(1, :) = parameter_vector(k, :)


                !< write corresponding input file(s) to temp file
                filename = trim(adjustl(WorkingDirectory)) // "log_input-files__single_call__" // trim(adjustl(Number1)) // ".dat"
                open(paramchannel,file = trim(adjustl(filename)))
                write(paramchannel,'("-",61(" -"))')
                Do j = 1, NumberInputFiles
                    write(paramchannel,'("Input-File ",I5,":  , file: ",A)') j, trim(adjustl(FitFktInput(j)))
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("-start_input-file",106("-"))')
                    posdatexp = 0.d0
                    call WriteParameter(paramchannel, .true., MaxColX, posdatexp, parameternumber, parametersetorg(1, :), j)
                    write(paramchannel,'("-end_input-file",108("-"))')
                end Do
                close(paramchannel)


                !< write corresponding line for log-file to temp file
                filename = trim(adjustl(WorkingDirectory)) // "log_lines__single_call__" // trim(adjustl(Number1)) // ".dat"
                open(678, file = trim(adjustl(filename)))


                !< build list with fit parameters
                i = 0
                ListParamFormated = ""
                Do j = 1, parameternumber
                    if (ia(j)) then
                        i = i + 1
                        HelpString = ""
                        call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, j)
                        if (IntegerTrue) then
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(parametersetorg(1, j))
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) int(parametersetorg(1, j))
                            endif
                        else
                            write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) parametersetorg(1, j)
                            if (index(HelpString, "*") > 0) then                            !< search for bad real number
                                write(HelpString, *) parametersetorg(1, j)
                            endif
                        endif
                        if (i == 1) then
                            ListParamFormated = trim(adjustl(ListParamFormated)) // trim(adjustl(HelpString))
                        else
                            ListParamFormated = trim(adjustl(ListParamFormated)) // ',  ' // trim(adjustl(HelpString))
                        endif
                    endif
                end Do
                write(678,'(ES26.15,5x,A)') ValueOfChi2(k), trim(adjustl(ListParamFormated))
                close(678)
            end Do
            close(WorkChannel)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< close log-files
            close(Chi2Channel)

            ! Debug:
            !    print*,'lengthexpdata = ',lengthexpdata
            !    print*,'MaxColX = ',MaxColX
            !    print*,'MaxColY = ',MaxColY
            !    print*,'expdataxorg(1:5,1) = ',expdataxorg(1:5,1)
            !    print*,'expdatayorg(1:5,1) = ',expdatayorg(1:5,1)
            !    print*,'expdataerrororg(1:5,1) = ',expdataerrororg(1:5,1)
            !    print*,'fitlog = ',trim(fitlog)
            !    print*,'model = ',trim(model)
            !    print*,'currentpathorg = ',currentpathorg
            !    print*,'parametersetorg(1,:) = ',parametersetorg(1,:)
            !    print*,'parametersetorg(2,:) = ',parametersetorg(2,:)
            !    print*,'parametersetorg(3,:) = ',parametersetorg(3,:)
            !    print*,'parametersetorg(4,:) = ',parametersetorg(4,:)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< free memory of general model function call variables
            if (allocated(ia)) deallocate(ia, stat = deallocstatus)


            !< free memory of model variables
            call ModelParamFree(deallocstatus)
            if (deallocstatus /= 0) then
                write(logchannel,*)
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not deallocate expdatax etc.")')
                write(logchannel,*)
                write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not deallocate variables expdatax etc.")'
                print '(" ")'
                print '("deallocstatus = ",I4)',deallocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            return
    end subroutine MainAlg
end Module Algorithm

