#!/usr/bin/env python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  emcee package
##
##  Copyright (C) 2012 - 2013, Dan Foreman-Mackey & contributors. 
##
##  http://dan.iel.fm/emcee/current/
##
##
##
##  The following subroutines and functions are included in this module:
##
##      - subroutine Sampler.__init__:                  initialize class Sampler
##      - subroutine Sampler.random_state:              The state of the internal random number generator. In practice, it's the result of calling
##                                                      ``get_state()`` on a ``numpy.random.mtrand.RandomState`` object. You can try to set this
##                                                      property but be warned that if you do this and it fails, it will do so silently.
##      - subroutine Sampler.random_state:              Try to set the state of the random number generator but fail silently if it doesn't work.
##                                                      Don't say I didn't warn you...
##      - subroutine Sampler.acceptance_fraction:       The fraction of proposed steps that were accepted.
##      - subroutine Sampler.chain:                     A pointer to the Markov chain.
##      - subroutine Sampler.flatchain:                 Alias of ``chain`` provided for compatibility.
##      - subroutine Sampler.lnprobability:             A list of the log-probability values associated with each step in the chain.
##      - subroutine Sampler.acor:                      ???
##      - subroutine Sampler.get_autocorr_time:         check NotImplementedError
##      - subroutine Sampler.get_lnprob:                Return the log-probability at the given position.
##      - subroutine Sampler.reset:                     Clear ``chain``, ``lnprobability`` and the bookkeeping parameters.
##      - subroutine Sampler.clear_chain:               An alias for :func:`reset` kept for backwards compatibility.
##      - subroutine Sampler.sample:                    check NotImplementedError
##      - subroutine Sampler.run_mcmc:                  Iterate :func:`sample` for ``N`` iterations and return the result.
##      - subroutine EnsembleSampler.__init__:          initialize class ModelFunctionCallClass
##      - subroutine EnsembleSampler.clear_blobs:       Clear the ``blobs`` list.
##      - subroutine EnsembleSampler.reset:             Clear the ``chain`` and ``lnprobability`` array. Also reset the bookkeeping parameters.
##      - subroutine EnsembleSampler.sample:            Advance the chain ``iterations`` steps as a generator.
##      - subroutine EnsembleSampler._propose_stretch:  Propose a new position for one sub-ensemble given the positions of another.
##      - subroutine EnsembleSampler._get_lnprob:       Calculate the vector of log-probability for the walkers.
##      - subroutine EnsembleSampler.blobs:             Get the list of "blobs" produced by sampling. The result is a list (of length ``iterations``)
##                                                      of ``list`` s (of length ``nwalkers``) of arbitrary objects. **Note**: this will actually be
##                                                      an empty list if your ``lnpostfn`` doesn't return any metadata.
##      - subroutine EnsembleSampler.chain:             A pointer to the Markov chain itself. The shape of this array is ``(k, iterations, dim)``.
##      - subroutine EnsembleSampler.flatchain:         A shortcut for accessing chain flattened along the zeroth (walker) axis.
##      - subroutine EnsembleSampler.lnprobability:     A pointer to the matrix of the value of ``lnprobfn`` produced at each step for each walker.
##                                                      The shape is ``(k, iterations)``.
##      - subroutine EnsembleSampler.flatlnprobability: A shortcut to return the equivalent of ``lnprobability`` but aligned to ``flatchain`` rather
##                                                      than ``chain``.
##      - subroutine EnsembleSampler.acceptance_fraction:   An array (length: ``k``) of the fraction of steps accepted for each walker.
##      - subroutine EnsembleSampler.acor:              An estimate of the autocorrelation time for each parameter (length: ``dim``).
##      - subroutine EnsembleSampler.get_autocorr_time: Compute an estimate of the autocorrelation time for each parameter (length: ``dim``).
##      - subroutine EnsembleSampler.function:          Estimate the autocorrelation function of a time series using the FFT.
##      - subroutine EnsembleSampler.integrated_time:   Estimate the integrated autocorrelation time of a time series.
##      - subroutine _function_wrapper.__init__:        initialize class _function_wrapper
##      - subroutine _function_wrapper.__call__:        call likelihood function
##
##      Corner package:
##
##      - subroutine corner:                            Make a *sick* corner plot showing the projections of a data set in a multi-dimensional space.
##                                                      kwargs are passed to hist2d() or used for `matplotlib` styling.
##      - subroutine quantile:                          like numpy.percentile
##      - subroutine hist2d:                            plot a 2-D histogram of samples.
##
##      MCMC credible regions (taken from http://bebi103.caltech.edu/2015/tutorials/l06_credible_regions.html):
##
##      - subroutine hpd:                               Returns highest probability density region given by a set of samples.
##      - subroutine hpdPyMC:                           Calculate highest posterior density (HPD) of array for given alpha. The HPD is the minimum
##                                                      width Bayesian credible interval (BCI).
##      - subroutine calc_min_interval:                 Internal method to determine the minimum interval of a given width.
##
##
##
##  Versions of the program:
##
##  Who                                 When            What
##
##  Dan Foreman-Mackey & contributors   2012 - 2014     initial version
##  T. Moeller                          2016-05-19      improved documentation, modifications for XCLASS
##
##
##
##  License:
##
##    The MIT License (MIT)
##
##    Copyright (c) 2010 - 2013 Daniel Foreman-Mackey & contributors.
##
##    Permission is hereby granted, free of charge, to any person obtaining a copy
##    of this software and associated documentation files (the "Software"), to deal
##    in the Software without restriction, including without limitation the rights
##    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
##    copies of the Software, and to permit persons to whom the Software is
##    furnished to do so, subject to the following conditions:
##
##    The above copyright notice and this permission notice shall be included in all
##    copies or substantial portions of the Software.
##
##    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
##    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
##    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
##    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
##    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
##    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
##    SOFTWARE.
##
##********************************************************************************************************************************************************


##******************************************************************** load packages *********************************************************************
from math import *                                                                          ## load math package
import numpy as np                                                                          ## load numpy package
import random                                                                               ## load random package
import sys                                                                                  ## load sys package
import time                                                                                 ## load time package
import os                                                                                   ## load os package
import pylab                                                                                ## load python package for plotting pylab
import logging                                                                              ## load python package logging
import matplotlib.pyplot as pl                                                              ## load python package matplotlib.pyplot
from matplotlib.ticker import MaxNLocator                                                   ## load python package matplotlib.MaxNLocator
from matplotlib.colors import LinearSegmentedColormap, colorConverter                       ## load python package matplotlib.LinearSegmentedColormap
from matplotlib.ticker import ScalarFormatter                                               ## load python package ScalarFormatter
try:
    from scipy.ndimage import gaussian_filter                                               ## load python package gaussian_filter
except ImportError:
    gaussian_filter = None
##********************************************************************************************************************************************************


##********************************************************************************************************************************************************
## class Sampler
class Sampler(object):
    """
    An abstract sampler object that implements various helper functions

    :param dim:
        The number of dimensions in the parameter space.

    :param lnpostfn:
        A function that takes a vector in the parameter space as input and
        returns the natural logarithm of the posterior probability for that
        position.

    :param args: (optional)
        A list of extra positional arguments for ``lnpostfn``. ``lnpostfn``
        will be called with the sequence ``lnpostfn(p, *args, **kwargs)``.

    :param kwargs: (optional)
        A list of extra keyword arguments for ``lnpostfn``. ``lnpostfn``
        will be called with the sequence ``lnpostfn(p, *args, **kwargs)``.

    """
    ##****************************************************************************************************************************************************
    ## initialize class Sampler
    def __init__(self, dim, lnprobfn, args=[], kwargs={}):
        self.dim = dim
        self.lnprobfn = lnprobfn
        self.args = args
        self.kwargs = kwargs


        ## This is a random number generator that we can easily set the state of without affecting the numpy-wide generator
        self._random = np.random.mtrand.RandomState()

        self.reset()


        ## we've done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## The state of the internal random number generator. In practice, it's the result of calling ``get_state()`` on a
    ## ``numpy.random.mtrand.RandomState`` object. You can try to set this property but be warned that if you do this and it fails, it will do
    ## so silently.
    @property
    def random_state(self):
        """
        

        """


        ## we've done
        return self._random.get_state()
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## Try to set the state of the random number generator but fail silently if it doesn't work. Don't say I didn't warn you...
    @random_state.setter  # NOQA
    def random_state(self, state):
        """


        """
        try:
            self._random.set_state(state)
        except:
            pass


        ## we've done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## The fraction of proposed steps that were accepted.
    @property
    def acceptance_fraction(self):
        """


        """


        ## define return parameter
        return self.naccepted / self.iterations
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## A pointer to the Markov chain.
    @property
    def chain(self):
        """


        """


        ## define return parameter
        return self._chain
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## Alias of ``chain`` provided for compatibility.
    @property
    def flatchain(self):
        """


        """


        ## define return parameter
        return self._chain
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## A list of the log-probability values associated with each step in the chain.
    @property
    def lnprobability(self):
        """


        """


        ## define return parameter
        return self._lnprob
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## ???
    @property
    def acor(self):


        ## define return parameter
        return self.get_autocorr_time()
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## check NotImplementedError
    def get_autocorr_time(self, window=50):
        raise NotImplementedError("The acor method must be implemented by subclasses")


        ## we've done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## Return the log-probability at the given position.
    def get_lnprob(self, p):


        ## define return parameter
        return self.lnprobfn(p, *self.args, **self.kwargs)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## Clear ``chain``, ``lnprobability`` and the bookkeeping parameters.
    def reset(self):
        """


        """
        self.iterations = 0
        self.naccepted = 0
        self._last_run_mcmc_result = None


        ## we've done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## An alias for :func:`reset` kept for backwards compatibility.
    def clear_chain(self):


        ## define return parameter
        return self.reset()
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## check NotImplementedError
    def sample(self, *args, **kwargs):
        raise NotImplementedError("The sampling routine must be implemented by subclasses")


        ## we've done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## Iterate :func:`sample` for ``N`` iterations and return the result.
    def run_mcmc(self, pos0, N, LocalModelClass, rstate0=None, lnprob0=None, **kwargs):
        """
        :param pos0:
            The initial position vector.  Can also be None to resume from
            where :func:``run_mcmc`` left off the last time it executed.

        :param N:
            The number of steps to run.

        :param lnprob0: (optional)
            The log posterior probability at position ``p0``. If ``lnprob``
            is not provided, the initial value is calculated.

        :param rstate0: (optional)
            The state of the random number generator. See the
            :func:`random_state` property for details.

        :param kwargs: (optional)
            Other parameters that are directly passed to :func:`sample`.

        This returns the results of the final sample in whatever form
        :func:`sample` yields.  Usually, that's:
        ``pos``, ``lnprob``, ``rstate``, ``blobs`` (blobs optional)
        """
        if pos0 is None:
            if self._last_run_mcmc_result is None:
                raise ValueError("Cannot have pos0 = None if run_mcmc has never been called.")
            pos0 = self._last_run_mcmc_result[0]
            if lnprob0 is None:
                rstate0 = self._last_run_mcmc_result[1]
            if rstate0 is None:
                rstate0 = self._last_run_mcmc_result[2]

        CounterIter = 0
        for results in self.sample(pos0, lnprob0, rstate0, iterations = N, **kwargs):
            CounterIter += 1


            ## write informations to log files
            LocalModelClass.WriteLogLines(CounterIter)
            pass


        ## store so that the ``pos0=None`` case will work.  We throw out the blob if it's there because we don't need it
        self._last_run_mcmc_result = results[:3]


        ## define return parameter
        return results
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## class containing main subroutines
class EnsembleSampler(Sampler):
    """
    A generalized Ensemble sampler that uses 2 ensembles for parallelization.
    The ``__init__`` function will raise an ``AssertionError`` if
    ``k < 2 * dim`` (and you haven't set the ``live_dangerously`` parameter)
    or if ``k`` is odd.

    **Warning**: The :attr:`chain` member of this object has the shape:
    ``(nwalkers, nlinks, dim)`` where ``nlinks`` is the number of steps
    taken by the chain and ``k`` is the number of walkers.  Use the
    :attr:`flatchain` property to get the chain flattened to
    ``(nlinks, dim)``. For users of pre-1.0 versions, this shape is
    different so be careful!

    :param nwalkers:
        The number of Goodman & Weare "walkers".

    :param dim:
        Number of dimensions in the parameter space.

    :param lnpostfn:
        A function that takes a vector in the parameter space as input and
        returns the natural logarithm of the posterior probability for that
        position.

    :param a: (optional)
        The proposal scale parameter. (default: ``2.0``)

    :param args: (optional)
        A list of extra positional arguments for ``lnpostfn``. ``lnpostfn``
        will be called with the sequence ``lnpostfn(p, *args, **kwargs)``.

    :param kwargs: (optional)
        A list of extra keyword arguments for ``lnpostfn``. ``lnpostfn``
        will be called with the sequence ``lnpostfn(p, *args, **kwargs)``.

    :param postargs: (optional)
        Alias of ``args`` for backwards compatibility.

    :param threads: (optional)
        The number of threads to use for parallelization. If ``threads == 1``,
        then the ``multiprocessing`` module is not used but if
        ``threads > 1``, then a ``Pool`` object is created and calls to
        ``lnpostfn`` are run in parallel.

    :param pool: (optional)
        An alternative method of using the parallelized algorithm. If
        provided, the value of ``threads`` is ignored and the
        object provided by ``pool`` is used for all parallelization. It
        can be any object with a ``map`` method that follows the same
        calling sequence as the built-in ``map`` function.

    :param runtime_sortingfn: (optional)
        A function implementing custom runtime load-balancing. See
        :ref:`loadbalance` for more information.

    """

    ##****************************************************************************************************************************************************
    ## initialize class
    def __init__(self, nwalkers, dim, lnpostfn, a = 2.0, args=[], kwargs = {}, postargs = None, threads = 1, pool = None, live_dangerously = False, \
                 runtime_sortingfn = None):
        """

        """
        self.k = nwalkers
        self.a = a
        self.threads = threads
        self.pool = pool
        self.runtime_sortingfn = runtime_sortingfn

        if postargs is not None:
            args = postargs
        super(EnsembleSampler, self).__init__(dim, lnpostfn, args=args, kwargs=kwargs)


        ## Do a little bit of _magic_ to make the likelihood call with ``args`` and ``kwargs`` pickleable.
        # self.lnprobfn = _function_wrapper(self.lnprobfn, self.args, self.kwargs)          ## for XCLASS, we don't need it

        assert self.k % 2 == 0, "The number of walkers must be even."
        if not live_dangerously:
            assert self.k >= 2 * self.dim, (
                "The number of walkers needs to be more than twice the "
                "dimension of your parameter space... unless you're "
                "crazy!")

        if self.threads > 1 and self.pool is None:
            self.pool = InterruptiblePool(self.threads)


        ## we've done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## Clear the ``blobs`` list.
    def clear_blobs(self):
        """

        """
        self._blobs = []


        ## we've done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## Clear the ``chain`` and ``lnprobability`` array. Also reset the bookkeeping parameters.
    def reset(self):
        """

        """
        super(EnsembleSampler, self).reset()
        self.naccepted = np.zeros(self.k)
        self._chain = np.empty((self.k, 0, self.dim))
        self._lnprob = np.empty((self.k, 0))


        ## Initialize list for storing optional metadata blobs.
        self.clear_blobs()


        ## we've done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## Advance the chain ``iterations`` steps as a generator.
    def sample(self, p0, lnprob0 = None, rstate0 = None, blobs0 = None, iterations = 1, thin = 1, storechain = True, mh_proposal = None):
        """
        :param p0:
            A list of the initial positions of the walkers in the
            parameter space. It should have the shape ``(nwalkers, dim)``.

        :param lnprob0: (optional)
            The list of log posterior probabilities for the walkers at
            positions given by ``p0``. If ``lnprob is None``, the initial
            values are calculated. It should have the shape ``(k, dim)``.

        :param rstate0: (optional)
            The state of the random number generator.
            See the :attr:`Sampler.random_state` property for details.

        :param iterations: (optional)
            The number of steps to run.

        :param thin: (optional)
            If you only want to store and yield every ``thin`` samples in the
            chain, set thin to an integer greater than 1.

        :param storechain: (optional)
            By default, the sampler stores (in memory) the positions and
            log-probabilities of the samples in the chain. If you are
            using another method to store the samples to a file or if you
            don't need to analyse the samples after the fact (for burn-in
            for example) set ``storechain`` to ``False``.

        :param mh_proposal: (optional)
            A function that returns a list of positions for ``nwalkers``
            walkers given a current list of positions of the same size. See
            :class:`utils.MH_proposal_axisaligned` for an example.

        At each iteration, this generator yields:

        * ``pos`` - A list of the current positions of the walkers in the
          parameter space. The shape of this object will be
          ``(nwalkers, dim)``.

        * ``lnprob`` - The list of log posterior probabilities for the
          walkers at positions given by ``pos`` . The shape of this object
          is ``(nwalkers, dim)``.

        * ``rstate`` - The current state of the random number generator.

        * ``blobs`` - (optional) The metadata "blobs" associated with the
          current position. The value is only returned if ``lnpostfn``
          returns blobs too.

        """
        ## Try to set the initial value of the random number generator. This fails silently if it doesn't work but that's what we want because
        ## we'll just interpret any garbage as letting the generator stay in it's current state.
        self.random_state = rstate0


        ## convert p0 to numpy array
        p = np.array(p0)


        ## define half number of walkers
        halfk = int(self.k / 2)

        # Debug:
        # print "\nself.k = ", self.k
        # print "halfk = ", halfk


        ## If the initial log-probabilities were not provided, calculate them now.
        lnprob = lnprob0
        blobs = blobs0
        if lnprob is None:
            lnprob, blobs = self._get_lnprob(p)


        ## Check to make sure that the probability function didn't return ``np.nan``.
        if np.any(np.isnan(lnprob)):
            raise ValueError("The initial lnprob was NaN.")


        ## Store the initial size of the stored chain.
        i0 = self._chain.shape[1]


        ## Here, we resize chain in advance for performance. This actually makes a pretty big difference.
        if storechain:
            N = int(iterations / thin)
            self._chain = np.concatenate((self._chain, np.zeros((self.k, N, self.dim))), axis=1)
            self._lnprob = np.concatenate((self._lnprob, np.zeros((self.k, N))), axis=1)


        for i in range(int(iterations)):
            self.iterations += 1


            ## If we were passed a Metropolis-Hastings proposal function, use it.
            if mh_proposal is not None:


                ## Draw proposed positions & evaluate lnprob there
                q = mh_proposal(p)
                newlnp, blob = self._get_lnprob(q)


                ## Accept if newlnp is better; and ...
                acc = (newlnp > lnprob)


                ## ... sometimes accept for steps that got worse
                worse = np.flatnonzero(~acc)
                acc[worse] = ((newlnp[worse] - lnprob[worse]) > np.log(self._random.rand(len(worse))))
                del worse


                ## Update the accepted walkers.
                lnprob[acc] = newlnp[acc]
                p[acc] = q[acc]
                self.naccepted[acc] += 1

                if blob is not None:
                    assert blobs is not None, (
                        "If you start sampling with a given lnprob, you also "
                        "need to provide the current list of blobs at that "
                        "position.")
                    ind = np.arange(self.k)[acc]
                    for j in ind:
                        blobs[j] = blob[j]

            else:
                ## Loop over the two ensembles, calculating the proposed positions.


                ## Slices for the first and second halves
                first, second = slice(halfk), slice(halfk, self.k)
                for S0, S1 in [(first, second), (second, first)]:
                    q, newlnp, acc, blob = self._propose_stretch(p[S0], p[S1], lnprob[S0])
                    if np.any(acc):


                        ## Update the positions, log probabilities and acceptance counts.
                        lnprob[S0][acc] = newlnp[acc]
                        p[S0][acc] = q[acc]
                        self.naccepted[S0][acc] += 1

                        if blob is not None:
                            assert blobs is not None, (
                                "If you start sampling with a given lnprob, "
                                "you also need to provide the current list of "
                                "blobs at that position.")
                            ind = np.arange(len(acc))[acc]
                            indfull = np.arange(self.k)[S0][acc]
                            for j in range(len(ind)):
                                blobs[indfull[j]] = blob[ind[j]]

            if storechain and i % thin == 0:
                ind = i0 + int(i / thin)
                self._chain[:, ind, :] = p
                self._lnprob[:, ind] = lnprob
                if blobs is not None:
                    self._blobs.append(list(blobs))


            ## Yield the result as an iterator so that the user can do all sorts of fun stuff with the results so far.
            if blobs is not None:


                ## This is a bit of a hack to keep things backwards compatible.
                yield p, lnprob, self.random_state, blobs
            else:
                yield p, lnprob, self.random_state


        ## we've done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## Propose a new position for one sub-ensemble given the positions of another.
    def _propose_stretch(self, p0, p1, lnprob0):
        """
        :param p0:
            The positions from which to jump.

        :param p1:
            The positions of the other ensemble.

        :param lnprob0:
            The log-probabilities at ``p0``.

        This method returns:

        * ``q`` - The new proposed positions for the walkers in ``ensemble``.

        * ``newlnprob`` - The vector of log-probabilities at the positions
          given by ``q``.

        * ``accept`` - A vector of type ``bool`` indicating whether or not
          the proposed position for each walker should be accepted.

        * ``blob`` - The new meta data blobs or ``None`` if nothing was
          returned by ``lnprobfn``.

        """
        s = np.atleast_2d(p0)
        Ns = len(s)
        c = np.atleast_2d(p1)
        Nc = len(c)


        ## Generate the vectors of random numbers that will produce the proposal.
        zz = ((self.a - 1.) * self._random.rand(Ns) + 1) ** 2. / self.a
        rint = self._random.randint(Nc, size=(Ns,))


        ## Calculate the proposed positions and the log-probability there.
        q = c[rint] - zz[:, np.newaxis] * (c[rint] - s)
        newlnprob, blob = self._get_lnprob(q)


        ## Decide whether or not the proposals should be accepted.
        lnpdiff = (self.dim - 1.) * np.log(zz) + newlnprob - lnprob0
        accept = (lnpdiff > np.log(self._random.rand(len(lnpdiff))))


        ## define return parameters
        return q, newlnprob, accept, blob
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## Calculate the vector of log-probability for the walkers.
    def _get_lnprob(self, pos=None):
        """
        :param pos: (optional)
            The position vector in parameter space where the probability
            should be calculated. This defaults to the current position
            unless a different one is provided.

        This method returns:

        * ``lnprob`` - A vector of log-probabilities with one entry for each
          walker in this sub-ensemble.

        * ``blob`` - The list of meta data returned by the ``lnpostfn`` at
          this position or ``None`` if nothing was returned.

        """
        if pos is None:
            p = self.pos
        else:
            p = pos


        ## Check that the parameters are in physical ranges.
        if np.any(np.isinf(p)):
            raise ValueError("At least one parameter value was infinite.")
        if np.any(np.isnan(p)):
            raise ValueError("At least one parameter value was NaN.")


        ## If the `pool` property of the sampler has been set (i.e. we want to use `multiprocessing`), use the `pool`'s map method. Otherwise,
        ## just use the built-in `map` function.
        if self.pool is not None:
            M = self.pool.map
        else:
            M = map


        ## sort the tasks according to (user-defined) some runtime guess
        if self.runtime_sortingfn is not None:
            p, idx = self.runtime_sortingfn(p)

        # Debug:
        # print ("len(p) = ", len(p))
        # print ("p = ", p)


        ## Run the log-probability calculations (optionally in parallel).
        # results = list(M(self.lnprobfn, [p[i] for i in range(len(p))]))                   ## original emcee statement
        NumParamVectors = len(p)                                                            ## get number of parameter vectors
        Chi2List = self.lnprobfn(p, NumParamVectors, 0.0)                                   ## call logLhood, here self.lnprobfn = LocalModelClass
        results = list([chi2value for chi2value in Chi2List])                               ## define result array, valid for other functions

        # Debug:
        # print "\n\n"
        # print "len(results) = ", len(results)
        # print "results = ", results
        # print "results[0] = ", results[0]
        # sys.exit(0)


        try:
            lnprob = np.array([float(l[0]) for l in results])
            blob = [l[1] for l in results]
        except (IndexError, TypeError):
            lnprob = np.array([float(l) for l in results])
            blob = None


        ## sort it back according to the original order - get the same chain irrespective of the runtime sorting fn
        if self.runtime_sortingfn is not None:
            orig_idx = np.argsort(idx)
            lnprob = lnprob[orig_idx]
            p = [p[i] for i in orig_idx]
            if blob is not None:
                blob = [blob[i] for i in orig_idx]


        ## Check for lnprob returning NaN.
        if np.any(np.isnan(lnprob)):


            ## Print some debugging stuff.
            print "Error in subroutine emcee__modified.EnsembleSampler._get_lnprob:"
            print "\tNaN value of lnprob for parameters:"
            for pars in p[np.isnan(lnprob)]:
                print "\t", pars


            ## Finally raise exception.
            raise ValueError("lnprob returned NaN.")

        # Debug:
        # print "\n\nlnprob = ", lnprob
        # print "blob =   ", blob


        ## define return parameters
        return lnprob, blob
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## Get the list of "blobs" produced by sampling. The result is a list (of length ``iterations``) of ``list`` s (of length ``nwalkers``) of
    ## arbitrary objects. **Note**: this will actually be an empty list if your ``lnpostfn`` doesn't return any metadata.
    @property
    def blobs(self):
        """

        """


        ## define return parameter
        return self._blobs
    ##----------------------------------------------------------------------------------------------------------------------------------------------------

     
    ##****************************************************************************************************************************************************
    ## A pointer to the Markov chain itself. The shape of this array is ``(k, iterations, dim)``.
    @property
    def chain(self):
        """

        """


        ## define return parameter
        return super(EnsembleSampler, self).chain
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## A shortcut for accessing chain flattened along the zeroth (walker) axis.
    @property
    def flatchain(self):
        """

        """
        s = self.chain.shape


        ## define return parameter
        return self.chain.reshape(s[0] * s[1], s[2])
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## A pointer to the matrix of the value of ``lnprobfn`` produced at each step for each walker. The shape is ``(k, iterations)``.
    @property
    def lnprobability(self):
        """

        """


        ## define return parameter
        return super(EnsembleSampler, self).lnprobability
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ##  A shortcut to return the equivalent of ``lnprobability`` but aligned to ``flatchain`` rather than ``chain``.
    @property
    def flatlnprobability(self):
        """

        """


        ## define return parameter
        return super(EnsembleSampler, self).lnprobability.flatten()
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## An array (length: ``k``) of the fraction of steps accepted for each walker.
    @property
    def acceptance_fraction(self):
        """

        """


        ## define return parameter
        return super(EnsembleSampler, self).acceptance_fraction
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## An estimate of the autocorrelation time for each parameter (length: ``dim``).
    @property
    def acor(self):
        """

        """


        ## define return parameter
        return self.get_autocorr_time()
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## Estimate the autocorrelation function of a time series using the FFT.
    def function(self, x, axis = 0, fast = False):
        """
        :param x:
            The time series. If multidimensional, set the time axis using the
            ``axis`` keyword argument and the function will be computed for every
            other axis.

        :param axis: (optional)
            The time axis of ``x``. Assumed to be the first axis if not specified.

        :param fast: (optional)
            If ``True``, only use the largest ``2^n`` entries for efficiency.
            (default: False)

        """
        x = np.atleast_1d(x)
        m = [slice(None), ] * len(x.shape)


        ## For computational efficiency, crop the chain to the largest power of two if requested.
        if fast:
            n = int(2**np.floor(np.log2(x.shape[axis])))
            m[axis] = slice(0, n)
            x = x
        else:
            n = x.shape[axis]


        ## Compute the FFT and then (from that) the auto-correlation function.
        f = np.fft.fft(x-np.mean(x, axis=axis), n=2*n, axis=axis)
        m[axis] = slice(0, n)
        acf = np.fft.ifft(f * np.conjugate(f), axis=axis)[m].real
        m[axis] = 0


        ## define return parameter
        return acf / acf[m]
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## Estimate the integrated autocorrelation time of a time series.
    def integrated_time(self, x, axis = 0, window = 50, fast = False):
        """

        See `Sokal's notes <http://www.stat.unc.edu/faculty/cji/Sokal.pdf>`_ on
        MCMC and sample estimators for autocorrelation times.

        :param x:
            The time series. If multidimensional, set the time axis using the
            ``axis`` keyword argument and the function will be computed for every
            other axis.

        :param axis: (optional)
            The time axis of ``x``. Assumed to be the first axis if not specified.

        :param window: (optional)
            The size of the window to use. (default: 50)

        :param fast: (optional)
            If ``True``, only use the largest ``2^n`` entries for efficiency.
            (default: False)

        """
        ## Compute the autocorrelation function.
        f = self.function(x, axis = axis, fast = fast)


        ## Special case 1D for simplicity.
        if len(f.shape) == 1:
            return 1 + 2 * np.sum(f[1:window])


        ## N-dimensional case.
        m = [slice(None), ] * len(f.shape)
        m[axis] = slice(1, window)
        tau = 1 + 2 * np.sum(f[m], axis = axis)


        ## define return parameter
        return tau
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## Compute an estimate of the autocorrelation time for each parameter (length: ``dim``).
    def get_autocorr_time(self, window = 50, fast = False):
        """
        :param window: (optional)
            The size of the windowing function. This is equivalent to the
            maximum number of lags to use. (default: 50)
        """


        ## define return parameter
        return self.integrated_time(np.mean(self.chain, axis = 0), window = window, fast = fast)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## class _function_wrapper
class _function_wrapper(object):
    """
    This is a hack to make the likelihood function pickleable when ``args``
    or ``kwargs`` are also included.

    """
    ##****************************************************************************************************************************************************
    ## initialize class _function_wrapper
    def __init__(self, f, args, kwargs):
        self.f = f
        self.args = args
        self.kwargs = kwargs


        ## we've done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## call likelihood function
    def __call__(self, x):
        try:
            return self.f(x, *self.args, **self.kwargs)
        except:
            import traceback
            print("emcee: Exception while calling your likelihood function:")
            print("  params:", x)
            print("  args:", self.args)
            print("  kwargs:", self.kwargs)
            print("  exception:")
            traceback.print_exc()
            raise


        ## we've done
        return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## corner package
def corner(xs, bins = 20, range = None, weights = None, color = "k", smooth = None, smooth1d = None, labels = None, label_kwargs = None, \
           show_titles = False, title_fmt = ".2f", title_kwargs = None, truths = None, truth_color = "#4682b4", scale_hist = False, quantiles = None, \
           ErrorTyp = "hpd", verbose = False, fig = None, max_n_ticks = 5, top_ticks = False, use_math_text = False, hist_kwargs = None, \
           **hist2d_kwargs):
    """
    Make a *sick* corner plot showing the projections of a data set in a
    multi-dimensional space. kwargs are passed to hist2d() or used for
    `matplotlib` styling.

    taken from
    https://pypi.python.org/pypi/corner, call May 22th 2016

    Parameters
    ----------
    xs : array_like (nsamples, ndim)
        The samples. This should be a 1- or 2-dimensional array. For a 1-D
        array this results in a simple histogram. For a 2-D array, the zeroth
        axis is the list of samples and the next axis are the dimensions of
        the space.

    bins : int or array_like (ndim,) (optional)
        The number of bins to use in histograms, either as a fixed value for
        all dimensions, or as a list of integers for each dimension.

    weights : array_like (nsamples,)
        The weight of each sample. If `None` (default), samples are given
        equal weight.

    color : str (optional)
        A ``matplotlib`` style color for all histograms.

    smooth, smooth1d : float (optional)
       The standard deviation for Gaussian kernel passed to
       `scipy.ndimage.gaussian_filter` to smooth the 2-D and 1-D histograms
       respectively. If `None` (default), no smoothing is applied.

    labels : iterable (ndim,) (optional)
        A list of names for the dimensions. If a ``xs`` is a
        ``pandas.DataFrame``, labels will default to column names.

    label_kwargs : dict (optional)
        Any extra keyword arguments to send to the `set_xlabel` and
        `set_ylabel` methods.

    show_titles : bool (optional)
        Displays a title above each 1-D histogram showing the 0.5 quantile
        with the upper and lower errors supplied by the quantiles argument.

    title_fmt : string (optional)
        The format string for the quantiles given in titles. If you explicitly
        set ``show_titles=True`` and ``title_fmt=None``, the labels will be
        shown as the titles. (default: ``.2f``)

    title_kwargs : dict (optional)
        Any extra keyword arguments to send to the `set_title` command.

    range : iterable (ndim,) (optional)
        A list where each element is either a length 2 tuple containing
        lower and upper bounds or a float in range (0., 1.)
        giving the fraction of samples to include in bounds, e.g.,
        [(0.,10.), (1.,5), 0.999, etc.].
        If a fraction, the bounds are chosen to be equal-tailed.

    truths : iterable (ndim,) (optional)
        A list of reference values to indicate on the plots.  Individual
        values can be omitted by using ``None``.

    truth_color : str (optional)
        A ``matplotlib`` style color for the ``truths`` makers.

    scale_hist : bool (optional)
        Should the 1-D histograms be scaled in such a way that the zero line
        is visible?

    quantiles : iterable (optional)
        A list of fractional quantiles to show on the 1-D histograms as
        vertical dashed lines.

    ErrorTyp: str (optional)
        defines the type of error (``gauss``, ``quantile``, ``hpd``)

    verbose : bool (optional)
        If true, print the values of the computed quantiles.

    plot_contours : bool (optional)
        Draw contours for dense regions of the plot.

    use_math_text : bool (optional)
        If true, then axis tick labels for very large or small exponents will
        be displayed as powers of 10 rather than using `e`.

    max_n_ticks: int (optional)
        Maximum number of ticks to try to use

    top_ticks : bool (optional)
        If true, label the top ticks of each axis

    fig : matplotlib.Figure (optional)
        Overplot onto the provided figure object.

    hist_kwargs : dict (optional)
        Any extra keyword arguments to send to the 1-D histogram plots.

    **hist2d_kwargs : (optional)
        Any remaining keyword arguments are sent to `corner.hist2d` to generate
        the 2-D histogram plots.
    """
    if quantiles is None:
        quantiles = []
    if title_kwargs is None:
        title_kwargs = dict()
    if label_kwargs is None:
        label_kwargs = dict()


    ## Try filling in labels from pandas.DataFrame columns.
    if labels is None:
        try:
            labels = xs.columns
        except AttributeError:
            pass


    ## Deal with 1D sample lists.
    xs = np.atleast_1d(xs)
    if len(xs.shape) == 1:
        xs = np.atleast_2d(xs)
    else:
        assert len(xs.shape) == 2, "The input sample array must be 1- or 2-D."
        xs = xs.T
    assert xs.shape[0] <= xs.shape[1], "I don't believe that you want more dimensions than samples!"


    ## Parse the weight array.
    if weights is not None:
        weights = np.asarray(weights)
        if weights.ndim != 1:
            raise ValueError("Weights must be 1-D")
        if xs.shape[1] != weights.shape[0]:
            raise ValueError("Lengths of weights must match number of samples")


    ## Parse the parameter ranges.
    if range is None:
        if "extents" in hist2d_kwargs:
            logging.warn("Deprecated keyword argument 'extents'. Use 'range' instead.")
            range = hist2d_kwargs.pop("extents")
        else:
            range = [[x.min(), x.max()] for x in xs]


            ## Check for parameters that never change.
            m = np.array([e[0] == e[1] for e in range], dtype=bool)
            if np.any(m):
                raise ValueError(("It looks like the parameter(s) in column(s) {0} have no dynamic range. "
                                  "Please provide a `range` argument.")
                                 .format(", ".join(map("{0}".format, np.arange(len(m))[m]))))

    else:
        ## If any of the extents are percentiles, convert them to ranges. Also make sure it's a normal list.
        range = list(range)
        for i, _ in enumerate(range):
            try:
                emin, emax = range[i]
            except TypeError:
                q = [0.5 - 0.5*range[i], 0.5 + 0.5*range[i]]
                range[i] = quantile(xs[i], q, weights=weights)


    if len(range) != xs.shape[0]:
        raise ValueError("Dimension mismatch between samples and range")


    ## Parse the bin specifications.
    try:
        bins = [int(bins) for _ in range]
    except TypeError:
        if len(bins) != len(range):
            raise ValueError("Dimension mismatch between bins and range")


    ## Some magic numbers for pretty axis layout.
    K = len(xs)
    factor = 2.0                                                                            ## size of one side of one panel
    lbdim = 0.5 * factor                                                                    ## size of left/bottom margin
    trdim = 0.2 * factor                                                                    ## size of top/right margin
    whspace = 0.05                                                                          ## w/hspace size
    plotdim = factor * K + factor * (K - 1.) * whspace
    dim = lbdim + plotdim + trdim


    ## Create a new figure if one wasn't provided.
    if fig is None:
        fig, axes = pl.subplots(K, K, figsize=(dim, dim))
    else:
        try:
            axes = np.array(fig.axes).reshape((K, K))
        except:
            raise ValueError("Provided figure has {0} axes, but data has dimensions K={1}".format(len(fig.axes), K))


    ## Format the figure.
    lb = lbdim / dim
    tr = (lbdim + plotdim) / dim
    fig.subplots_adjust(left=lb, bottom=lb, right=tr, top=tr, wspace=whspace, hspace=whspace)


    ## Set up the default histogram keywords.
    if hist_kwargs is None:
        hist_kwargs = dict()
    hist_kwargs["color"] = hist_kwargs.get("color", color)
    if smooth1d is None:
        hist_kwargs["histtype"] = hist_kwargs.get("histtype", "step")
    for i, x in enumerate(xs):


        ## Deal with masked arrays.
        if hasattr(x, "compressed"):
            x = x.compressed()


        if np.shape(xs)[0] == 1:
            ax = axes
        else:
            ax = axes[i, i]


        ## Plot the histograms.
        if smooth1d is None:
            n, _, _ = ax.hist(x, bins=bins[i], weights=weights, range=np.sort(range[i]), **hist_kwargs)
        else:
            if gaussian_filter is None:
                raise ImportError("Please install scipy for smoothing")
            n, b = np.histogram(x, bins=bins[i], weights=weights, range=np.sort(range[i]))
            n = gaussian_filter(n, smooth1d)
            x0 = np.array(list(zip(b[:-1], b[1:]))).flatten()
            y0 = np.array(list(zip(n, n))).flatten()
            ax.plot(x0, y0, **hist_kwargs)
        if truths is not None and truths[i] is not None:
            ax.axvline(truths[i], color=truth_color)


        ## Plot quantiles if wanted.
        if len(quantiles[i]) > 0:
            if (ErrorTyp != "quantile"):
                for q in quantiles[i]:
                    ax.axvline(q, ls="dashed", color=color)
            else:
                qvalues = quantile(x, quantiles[i], weights=weights)
                for q in qvalues:
                    ax.axvline(q, ls="dashed", color=color)
                if verbose:
                    print("Quantiles:")
                    print([item for item in zip(quantiles[i], qvalues)])


        if show_titles:
            title = None
            if title_fmt is not None:


                ## Compute the quantiles for the title. This might redo unneeded computation but who cares.
                if (ErrorTyp == "quantile"):
                    qLeft, qMiddle, qRight = quantile(x, quantiles[i], weights=weights)
                else:
                    qLeft = quantiles[i][0]
                    qMiddle = quantiles[i][1]
                    qRight = quantiles[i][2]
                q_m, q_p = qMiddle - qLeft, qRight - qMiddle


                ## Format the quantile display.
                fmt = "{{0:{0}}}".format(title_fmt).format
                title = r"${{{0}}}_{{-{1}}}^{{+{2}}}$"
                title = title.format(fmt(qMiddle), fmt(q_m), fmt(q_p))


                ## Add in the column name if it's given.
                if labels is not None:
                    title = "{0} = {1}".format(labels[i], title)


            elif labels is not None:
                title = "{0}".format(labels[i])


            if title is not None:
                ax.set_title(title, **title_kwargs)


        ## Set up the axes.
        ax.set_xlim(range[i])
        if scale_hist:
            maxn = np.max(n)
            ax.set_ylim(-0.1 * maxn, 1.1 * maxn)
        else:
            ax.set_ylim(0, 1.1 * np.max(n))
        ax.set_yticklabels([])
        ax.xaxis.set_major_locator(MaxNLocator(max_n_ticks, prune="lower"))
        if i < K - 1:
            if top_ticks:
                ax.xaxis.set_ticks_position("top")
                [l.set_rotation(45) for l in ax.get_xticklabels()]
            else:
                ax.set_xticklabels([])
        else:
            [l.set_rotation(45) for l in ax.get_xticklabels()]
            if labels is not None:
                ax.set_xlabel(labels[i], **label_kwargs)
                ax.xaxis.set_label_coords(0.5, -0.3)


            ## use MathText for axes ticks
            ax.xaxis.set_major_formatter(
                ScalarFormatter(useMathText=use_math_text))


        for j, y in enumerate(xs):
            if np.shape(xs)[0] == 1:
                ax = axes
            else:
                ax = axes[i, j]
            if j > i:
                ax.set_frame_on(False)
                ax.set_xticks([])
                ax.set_yticks([])
                continue
            elif j == i:
                continue


            ## Deal with masked arrays.
            if hasattr(y, "compressed"):
                y = y.compressed()
            hist2d(y, x, ax=ax, range=[range[j], range[i]], weights=weights, color=color, smooth=smooth, bins=[bins[j], bins[i]], **hist2d_kwargs)

            if truths is not None:
                if truths[i] is not None and truths[j] is not None:
                    ax.plot(truths[j], truths[i], "s", color=truth_color)
                if truths[j] is not None:
                    ax.axvline(truths[j], color=truth_color)
                if truths[i] is not None:
                    ax.axhline(truths[i], color=truth_color)

            ax.xaxis.set_major_locator(MaxNLocator(max_n_ticks, prune="lower"))
            ax.yaxis.set_major_locator(MaxNLocator(max_n_ticks, prune="lower"))

            if i < K - 1:
                ax.set_xticklabels([])
            else:
                [l.set_rotation(45) for l in ax.get_xticklabels()]
                if labels is not None:
                    ax.set_xlabel(labels[j], **label_kwargs)
                    ax.xaxis.set_label_coords(0.5, -0.3)


                ## use MathText for axes ticks
                ax.xaxis.set_major_formatter(ScalarFormatter(useMathText=use_math_text))


            if j > 0:
                ax.set_yticklabels([])
            else:
                [l.set_rotation(45) for l in ax.get_yticklabels()]
                if labels is not None:
                    ax.set_ylabel(labels[i], **label_kwargs)
                    ax.yaxis.set_label_coords(-0.3, 0.5)


                ## use MathText for axes ticks
                ax.yaxis.set_major_formatter(ScalarFormatter(useMathText=use_math_text))


    ## define return parameter
    return fig
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Like numpy.percentile
def quantile(x, q, weights = None):
    """
    Compute sample quantiles with support for weighted samples.

    Note
    ----
    When ``weights`` is ``None``, this method simply calls numpy's percentile
    function with the values of ``q`` multiplied by 100.

    Parameters
    ----------
    x : array_like[nsamples,]
       The samples.

    q : array_like[nquantiles,]
       The list of quantiles to compute. These should all be in the range
       ``[0, 1]``.

    weights : Optional[array_like[nsamples,]]
        An optional weight corresponding to each sample. These

    Returns
    -------
    quantiles : array_like[nquantiles,]
        The sample quantiles computed at ``q``.

    Raises
    ------
    ValueError
        For invalid quantiles; ``q`` not in ``[0, 1]`` or dimension mismatch
        between ``x`` and ``weights``.

    """
    x = np.atleast_1d(x)
    q = np.atleast_1d(q)

    if np.any(q < 0.0) or np.any(q > 1.0):
        raise ValueError("Quantiles must be between 0 and 1")

    if weights is None:
        return np.percentile(x, 100.0 * q)
    else:
        weights = np.atleast_1d(weights)
        if len(x) != len(weights):
            raise ValueError("Dimension mismatch: len(weights) != len(x)")
        idx = np.argsort(x)
        sw = weights[idx]
        cdf = np.cumsum(sw)[:-1]
        cdf /= cdf[-1]
        cdf = np.append(0, cdf)
        return np.interp(q, cdf, x[idx]).tolist()


    ## we've done
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Plot a 2-D histogram of samples.
def hist2d(x, y, bins = 20, range = None, weights = None, levels = None, smooth = None, ax = None, color = None, plot_datapoints = True, \
           plot_density = True, plot_contours = True, no_fill_contours = False, fill_contours = False, contour_kwargs = None, contourf_kwargs = None, \
           data_kwargs = None, **kwargs):
    """

    Parameters
    ----------
    x : array_like[nsamples,]
       The samples.

    y : array_like[nsamples,]
       The samples.

    levels : array_like
        The contour levels to draw.

    ax : matplotlib.Axes (optional)
        A axes instance on which to add the 2-D histogram.

    plot_datapoints : bool (optional)
        Draw the individual data points.

    plot_density : bool (optional)
        Draw the density colormap.

    plot_contours : bool (optional)
        Draw the contours.

    no_fill_contours : bool (optional)
        Add no filling at all to the contours (unlike setting
        ``fill_contours=False``, which still adds a white fill at the densest
        points).

    fill_contours : bool (optional)
        Fill the contours.

    contour_kwargs : dict (optional)
        Any additional keyword arguments to pass to the `contour` method.

    contourf_kwargs : dict (optional)
        Any additional keyword arguments to pass to the `contourf` method.

    data_kwargs : dict (optional)
        Any additional keyword arguments to pass to the `plot` method when
        adding the individual data points.
    """
    if ax is None:
        ax = pl.gca()


    ## Set the default range based on the data range if not provided.
    if range is None:
        if "extent" in kwargs:
            logging.warn("Deprecated keyword argument 'extent'. "
                         "Use 'range' instead.")
            range = kwargs["extent"]
        else:
            range = [[x.min(), x.max()], [y.min(), y.max()]]


    ## Set up the default plotting arguments.
    if color is None:
        color = "k"


    ## Choose the default "sigma" contour levels.
    if levels is None:
        levels = 1.0 - np.exp(-0.5 * np.arange(0.5, 2.1, 0.5) ** 2)


    ## This is the color map for the density plot, over-plotted to indicate the density of the points near the center.
    density_cmap = LinearSegmentedColormap.from_list("density_cmap", [color, (1, 1, 1, 0)])


    ## This color map is used to hide the points at the high density areas.
    white_cmap = LinearSegmentedColormap.from_list("white_cmap", [(1, 1, 1), (1, 1, 1)], N=2)


    ## This "color map" is the list of colors for the contour levels if the contours are filled.
    rgba_color = colorConverter.to_rgba(color)
    contour_cmap = [list(rgba_color) for l in levels] + [rgba_color]
    for i, l in enumerate(levels):
        contour_cmap[i][-1] *= float(i) / (len(levels)+1)


    ## We'll make the 2D histogram to directly estimate the density.
    try:
        H, X, Y = np.histogram2d(x.flatten(), y.flatten(), bins=bins, range=list(map(np.sort, range)), weights=weights)
    except ValueError:
        raise ValueError("It looks like at least one of your sample columns "
                         "have no dynamic range. You could try using the "
                         "'range' argument.")

    if smooth is not None:
        if gaussian_filter is None:
            raise ImportError("Please install scipy for smoothing")
        H = gaussian_filter(H, smooth)


    ## Compute the density levels.
    Hflat = H.flatten()
    inds = np.argsort(Hflat)[::-1]
    Hflat = Hflat[inds]
    sm = np.cumsum(Hflat)
    sm /= sm[-1]
    V = np.empty(len(levels))
    for i, v0 in enumerate(levels):
        try:
            V[i] = Hflat[sm <= v0][-1]
        except:
            V[i] = Hflat[0]
    V.sort()
    m = np.diff(V) == 0
    if np.any(m):
        logging.warning("Too few points to create valid contours")
    while np.any(m):
        V[np.where(m)[0][0]] *= 1.0 - 1e-4
        m = np.diff(V) == 0
    V.sort()


    ## Compute the bin centers.
    X1, Y1 = 0.5 * (X[1:] + X[:-1]), 0.5 * (Y[1:] + Y[:-1])


    ## Extend the array for the sake of the contours at the plot edges.
    H2 = H.min() + np.zeros((H.shape[0] + 4, H.shape[1] + 4))
    H2[2:-2, 2:-2] = H
    H2[2:-2, 1] = H[:, 0]
    H2[2:-2, -2] = H[:, -1]
    H2[1, 2:-2] = H[0]
    H2[-2, 2:-2] = H[-1]
    H2[1, 1] = H[0, 0]
    H2[1, -2] = H[0, -1]
    H2[-2, 1] = H[-1, 0]
    H2[-2, -2] = H[-1, -1]
    X2 = np.concatenate([X1[0] + np.array([-2, -1]) * np.diff(X1[:2]), X1, X1[-1] + np.array([1, 2]) * np.diff(X1[-2:]), ])
    Y2 = np.concatenate([Y1[0] + np.array([-2, -1]) * np.diff(Y1[:2]), Y1, Y1[-1] + np.array([1, 2]) * np.diff(Y1[-2:]), ])

    if plot_datapoints:
        if data_kwargs is None:
            data_kwargs = dict()
        data_kwargs["color"] = data_kwargs.get("color", color)
        data_kwargs["ms"] = data_kwargs.get("ms", 2.0)
        data_kwargs["mec"] = data_kwargs.get("mec", "none")
        data_kwargs["alpha"] = data_kwargs.get("alpha", 0.1)
        ax.plot(x, y, "o", zorder=-1, rasterized=True, **data_kwargs)


    ## Plot the base fill to hide the densest data points.
    if (plot_contours or plot_density) and not no_fill_contours:
        ax.contourf(X2, Y2, H2.T, [V.min(), H.max()], cmap=white_cmap, antialiased=False)

    if plot_contours and fill_contours:
        if contourf_kwargs is None:
            contourf_kwargs = dict()
        contourf_kwargs["colors"] = contourf_kwargs.get("colors", contour_cmap)
        contourf_kwargs["antialiased"] = contourf_kwargs.get("antialiased", False)
        ax.contourf(X2, Y2, H2.T, np.concatenate([[0], V, [H.max()*(1+1e-4)]]), **contourf_kwargs)


    ## Plot the density map. This can't be plotted at the same time as the contour fills.
    elif plot_density:
        ax.pcolor(X, Y, H.max() - H.T, cmap=density_cmap)


    ## Plot the contour edge colors.
    if plot_contours:
        if contour_kwargs is None:
            contour_kwargs = dict()
        contour_kwargs["colors"] = contour_kwargs.get("colors", color)
        ax.contour(X2, Y2, H2.T, V, **contour_kwargs)

    ax.set_xlim(range[0])
    ax.set_ylim(range[1])


    ## we've done
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Returns highest probability density region given by a set of samples.
def hpd(trace, mass_frac) :
    """

    taken from
    http://bebi103.caltech.edu/2015/tutorials/l06_credible_regions.html

    Computation of the HPD is a little trickier. The function below will compute the HPD interval. The idea is that we rank-order the MCMC trace.
    We know that the number of samples that are included in the HPD is 0.95 times the total number of MCMC sample. We then consider all intervals
    that contain that many samples and find the shortest one.


    Parameters
    ----------
    trace : array
        1D array of MCMC samples for a single variable
    mass_frac : float with 0 < mass_frac <= 1
        The fraction of the probability to be included in
        the HPD.  For example, `massfrac` = 0.95 gives a
        95% HPD.
        
    Returns
    -------
    output : array, shape (2,)
        The bounds of the HPD
    """

    # Debug:
    #    print "trace = ", trace
    #    print "trace[0] = ", trace[0]


    ## Get sorted list
    d = np.sort(np.copy(trace))

    # Debug:
    #    print "d = ", d
    #    print "d[0] = ", d[0]


    ## Number of total samples taken
    n = len(trace)

    # Debug:
    #    print "n = ", n


    ## Get number of samples that should be included in HPD
    n_samples = np.floor(mass_frac * n).astype(int)

    # Debug:
    #    print "n_samples = ", n_samples


    ## Get width (in units of data) of all intervals with n_samples samples
    int_width = d[n_samples:] - d[:n-n_samples]

    # Debug:
    #    print "int_width = ", int_width


    ## Pick out minimal interval
    min_int = np.argmin(int_width)
    dMin = max(0, min(min_int, (n - 1)))
    dMax = max(0, min(min_int + n_samples, (n - 1)))

    # Debug:
    #    print "min_int = ", min_int
    #    print "dMin = ", dMin
    #    print "dMax = ", dMax


    ## Return interval
    return np.array([d[dMin], d[dMax]])
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Calculate highest posterior density (HPD) of array for given alpha. The HPD is the minimum width Bayesian credible interval (BCI).
def hpdPyMC(y, alpha):
    """

    taken from:
    https://github.com/pymc-devs/pymc/blob/master/pymc/utils.py


    Arguments:
      x : Numpy array
          An array containing MCMC samples
      alpha : float
          Desired probability of type I error
    """


    ## Make a copy of trace
    x = y.copy()

    # Debug:
    #    print "x = ", x
    #    print "x.ndim = ", x.ndim


    ## For multivariate node
    if (x.ndim > 1):
        from numpy import (sqrt, ndarray, asmatrix, array, prod,
                           asarray, atleast_1d, iterable, linspace, diff,
                           around, log10, zeros, arange, digitize, apply_along_axis,
                           concatenate, bincount, sort, hsplit, argsort, inf, shape,
                           ndim, swapaxes, ravel, diag, cov, transpose as tr)


        ## Transpose first, then sort
        tx = tr(x, list(range(x.ndim)[1:]) + [0])
        dims = shape(tx)


        ## Container list for intervals
        intervals = np.resize(0.0, (2,) + dims[:-1])
        for index in make_indices(dims[:-1]):
            try:
                index = tuple(index)
            except TypeError:
                pass


            ## Sort trace
            sx = sort(tx[index])


            ## Append to list
            intervals[0][index], intervals[1][index] = calc_min_interval(sx, alpha)


        ## Transpose back before returning
        return array(intervals)

    else:
        ## Sort univariate node
        sx = sort(x)


        ## Return interval
        return array(calc_min_interval(sx, alpha))
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Internal method to determine the minimum interval of a given width. Assumes that x is sorted numpy array.
def calc_min_interval(x, alpha):
    """

    taken from:
    https://github.com/pymc-devs/pymc/blob/master/pymc/utils.py


    """


    n = len(x)
    cred_mass = 1.0 - alpha


    interval_idx_inc = int(np.floor(cred_mass * n))
    n_intervals = n - interval_idx_inc
    interval_width = x[interval_idx_inc:] - x[:n_intervals]

    if len(interval_width) == 0:
        print_('Too few elements for interval calculation')
        return [None, None]

    min_idx = np.argmin(interval_width)
    hdi_min = x[min_idx]
    hdi_max = x[min_idx + interval_idx_inc]


    ## define return parameter
    return [hdi_min, hdi_max]
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Generates complete set of indices for given dimensions
def make_indices(dimensions):
    """

    taken from:
    https://github.com/pymc-devs/pymc/blob/master/pymc/utils.py


    """

    level = len(dimensions)

    if level == 1:
        return range(dimensions[0])

    indices = [[]]

    while level:

        _indices = []

        for j in range(dimensions[level - 1]):

            _indices += [[j] + i for i in indices]

        indices = _indices

        level -= 1

    try:
        return [tuple(i) for i in indices]
    except TypeError:


        ## define return parameter
        return indices
##--------------------------------------------------------------------------------------------------------------------------------------------------------

