#!/usr/bin/env python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
## Package: MCMC algorithm
##
##
##  This module contains the subroutines for MCMC algorithm
##  Copyright (C) 2009 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following subroutines and functions are included in this module:
##
##      - subroutine ModelFunctionCallClass.__init__:   initialize class ModelFunctionCallClass
##      - subroutine ModelFunctionCallClass.logLhood:   call of model-function
##      - subroutine ModelFunctionCallClass.GetBestResult:                  get best result
##      - subroutine ModelFunctionCallClass.GetBestFunctionValues:          get function values of best result
##      - subroutine ModelFunctionCallClass.GetPoints:                      get points, i.e. list of all chi2 values
##      - subroutine ModelFunctionCallClass.WriteLogLines:                  write informatons of current iteration to log files
##      - subroutine ModelFunctionCallClass.SetCurrentParameterIndex:       set parameter CurrentParameterIndex
##      - subroutine ModelFunctionCallClass.SaveChi2Distribution:           get parameter values and the corresponding chi2 values for current parameter
##      - subroutine MCMC:                              main subroutine for MCMC
##      - subroutine start:                             subroutine to start MCMC algorithm
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    24.05.2016   initial version
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##******************************************************************** load packages *********************************************************************
from math import *                                                                          ## load math package
from scipy.special import erfinv                                                            ## load scipy.special package
from copy import *                                                                          ## import random copy
import numpy                                                                                ## load numpy package
import random                                                                               ## load random package
import sys                                                                                  ## load sys package
import time                                                                                 ## load time package
import os                                                                                   ## load os package
import matplotlib                                                                           ## load python package for plotting matplotlib
# matplotlib.rcParams['backend'] = 'Agg'                                                      ## Do not remove the "#" sign! Important line for matplotlib
from matplotlib.ticker import MaxNLocator
import pylab                                                                                ## load python package for plotting pylab
import FittingEngine                                                                        ## import package containing the fitting engine
##********************************************************************************************************************************************************


##********************************************************************************************************************************************************
## class containing main subroutines
class ModelFunctionCallClass:


    ##****************************************************************************************************************************************************
    ## initialize class variables
    def __init__(self, printflagNumIn, GeneralAlgorithmSettingsIn, LastAlgorithmNumIn, DeterminationChi2In, PlotIterationIn, \
                 PlotTypeIn, fit_logIn, NumberInputFilesIn, NumberOutputFilesIn, ParallelizationFlagIn, JobIDIn, MaxInputLinesIn, MaxParameterIn, \
                 RenormalizedChi2In, currentpathIn, CalculationMethodIn, xAxisLabelIn, yAxisLabelIn, zAxisLabelIn, PathStartScriptIn, \
                 ExeCommandStartScriptIn, parameter_setIn, ExpDataXIn, ExpDataYIn, ExpDataErrorIn, NumberRangesIn, MinRangeIn, MaxRangeIn, \
                 NumberXColumnsIn, NumberYColumnsIn, LengthExpRangeIn, MaxRangeNumberIn, NumberExpFilesIn, MaxLengthIn, MaxColXIn, MaxColYIn, \
                 NumberParameterIn, MPIFlagIn, MAGIXrootDirectoryIn, JobDirIn, SpecialAlgorithmSettingsIn, StarterExecutable, Chi2Channel, logchannel, \
                 paramchannel, MCMCBestSiteCounter):
            self.printflagNum = printflagNumIn
            self.GeneralAlgorithmSettings = GeneralAlgorithmSettingsIn
            self.LastAlgorithmNum = LastAlgorithmNumIn
            self.DeterminationChi2 = DeterminationChi2In
            self.PlotIteration = PlotIterationIn
            self.PlotType = PlotTypeIn
            self.fitlog = fit_logIn
            self.NumberInputFiles = NumberInputFilesIn
            self.NumberOutputFiles = NumberOutputFilesIn
            self.ParallelizationFlag = ParallelizationFlagIn
            self.JobID = JobIDIn
            self.MaxInputLines = MaxInputLinesIn
            self.MaxParameter = MaxParameterIn
            self.RenormalizedChi2 = RenormalizedChi2In
            self.currentpath = currentpathIn
            self.CalculationMethod = CalculationMethodIn
            self.xAxisLabel = xAxisLabelIn
            self.yAxisLabel = yAxisLabelIn
            self.zAxisLabel = zAxisLabelIn
            self.PathStartScript = PathStartScriptIn
            self.ExeCommandStartScript = ExeCommandStartScriptIn
            self.parameter_set = parameter_setIn
            self.ExpDataX = ExpDataXIn
            self.ExpDataY = ExpDataYIn
            self.ExpDataError = ExpDataErrorIn
            self.NumberRanges = NumberRangesIn
            self.MinRange = MinRangeIn
            self.MaxRange = MaxRangeIn
            self.NumberXColumns = NumberXColumnsIn
            self.NumberYColumns = NumberYColumnsIn
            self.LengthExpRange = LengthExpRangeIn
            self.MaxRangeNumber = MaxRangeNumberIn
            self.NumberExpFiles = NumberExpFilesIn
            self.MaxLength = MaxLengthIn
            self.MaxColX = MaxColXIn
            self.MaxColY = MaxColYIn
            self.NumberParameter = NumberParameterIn
            self.MPIFlag = MPIFlagIn
            self.MAGIXrootDirectory = MAGIXrootDirectoryIn
            self.JobDir = JobDirIn
            self.SpecialAlgorithmSettings = SpecialAlgorithmSettingsIn
            self.StarterExecutable = StarterExecutable
            self.Chi2Channel = Chi2Channel
            self.logchannel = logchannel
            self.paramchannel = paramchannel
            self.MCMCBestSiteCounter = int(MCMCBestSiteCounter)
            self.BestChi2 = numpy.ones(self.MCMCBestSiteCounter) * 1.e99
            self.BestFunctionValues = []
            self.BestChi2Values = []
            self.BestParameter = []
            self.BestInputFiles = []
            for i in xrange(self.MCMCBestSiteCounter):
                self.BestParameter.append(0.0)
                self.BestFunctionValues.append(0.0)
                self.BestChi2Values.append(0.0)
                self.BestInputFiles.append(0.0)
            self.BestLogLine = ""
            self.FuncCallCounter = 0
            self.TempDir = str(os.environ.get('MAGIXTempDirectory',''))
            self.CurrentParameterIndex = (-1)
            self.Chi2DistributionParam = []
            self.Chi2DistributionChi2 = []
            self.CurrIter = (-1)                                                            ## turn initialization flag on
            self.Points = []
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## call of logLhood
    def logLhood(self, parameter_vector, number_param_set, dummy):
        """
        input variables:    number_param_set:       total number of parameter
                            parameter_vector:       parameter vector

        output variables:   chi2value:              value of chi^2
        """

        # Debug:
        # print 'number_param_set = ',number_param_set
        # print 'parameter_vector = ',parameter_vector


        ## reset output value
        chi2value = 0.0


        ## make parameter_vector always has the same length
        if (number_param_set == 1):
            parameter_vector = [parameter_vector]


        ## update parameter_set array and write new parameter vectors to file
        WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"
        NewParamFile = open(WorkingDirectory.strip() + "new-parameters.dat", 'w')
        for k in xrange(number_param_set):
            j = (-1)
            line = ""
            for i in xrange(self.NumberParameter):
                if (self.parameter_set[1][i] == 1):
                    j += 1
                    line += "   " + str(parameter_vector[k][j])
                else:
                    line += "   " + str(self.parameter_set[0][i])
            NewParamFile.write(line + "\n")
        NewParamFile.close()

        # Debug:
        # print 'ModelFunctionCall.startcall.__doc__ = ',ModelFunctionCall.startcall.__doc__
        # print 'self.printflagNum = ',self.printflagNum
        # print 'number_param_set = ',number_param_set
        # print 'self.DeterminationChi2 = ',self.DeterminationChi2
        # print 'self.PlotIteration = ', self.PlotIteration
        # print 'self.PlotType = ',self.PlotType
        # print 'self.NumberInputFiles = ',self.NumberInputFiles
        # print 'self.NumberOutputFiles = ',self.NumberOutputFiles
        # print 'self.ParallelizationFlag = ',self.ParallelizationFlag
        # print 'self.JobID = ',self.JobID
        # print 'self.MaxInputLines = ',self.MaxInputLines
        # print 'self.MaxParameter = ',self.MaxParameter
        # print 'self.RenormalizedChi2 = ',self.RenormalizedChi2


        ## define dummy arguments
        chilm = 1.0
        NumberOfFitAlgorithms = 1
        AlgCounter = 1


        ## call model function package
        ParamSetCounter = number_param_set
        numiter = self.CurrIter
        NameOfAlgorithm = "Model-Function_Call"
        calstatus, FitFunctionOut, Chi2Values, parmCopy, FinalParameterSet = FittingEngine.StartAlgPackage(NameOfAlgorithm, self.StarterExecutable, \
                                                                                            self.MPIFlag, self.MAGIXrootDirectory, self.JobDir, \
                                                                                            self.JobID, self.NumberExpFiles, self.MaxLength, \
                                                                                            self.MaxColX, self.MaxColY, self.MaxRangeNumber, \
                                                                                            self.NumberParameter, ParamSetCounter, \
                                                                                            self.LastAlgorithmNum, numiter, self.ParallelizationFlag, \
                                                                                            self.DeterminationChi2, self.PlotIteration, self.PlotType, \
                                                                                            self.RenormalizedChi2, AlgCounter, chilm, \
                                                                                            self.GeneralAlgorithmSettings, \
                                                                                            self.SpecialAlgorithmSettings, self.parameter_set, \
                                                                                            self.xAxisLabel, self.yAxisLabel, self.zAxisLabel)
        self.CurrIter = 0                                                                   ## turn initialization flag off


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get calculated chi2 values from temp file
        WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"
        ResultFile = open(WorkingDirectory.strip() + "chi2Values.dat", 'r')
        ResultFileContents = ResultFile.readlines()
        ResultFile.close()
        ValuesOfChi2 = []
        for val in ResultFileContents:
            ValuesOfChi2.append(float(val))

        # Debug:
        # print ">>>>ValuesOfChi2 = ", ValuesOfChi2


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check if parameter vector is within given range
        for i in xrange(number_param_set):                                                  ## loop over parameter sets
            k = (-1)
            for j in xrange(self.NumberParameter):                                          ## loop over all parameter
                if (self.parameter_set[1][j] == 1):
                    k += 1
                    value = float(parameter_vector[i][k])
                    if (value < float(self.parameter_set[2][j]) or value > float(self.parameter_set[3][j])):
                        ValuesOfChi2[i] = numpy.inf
                        FitFunctionOut[i] = 0.e0
                        Chi2Values[i] = 0.e0

        # Debug:
        # print 'parameter_vector = ',parameter_vector
        # sys.exit(0)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## save chi2 value and corresponding error parameter to arrays
        if (self.CurrentParameterIndex > (-1)):
            for k, chi2value in enumerate(ValuesOfChi2):
                self.Chi2DistributionParam.append(parameter_vector[k][self.CurrentParameterIndex])
                self.Chi2DistributionChi2.append(chi2value)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define complete list of chi2 values
        AllChi2Values = []
        RealNumParamVec = len(ValuesOfChi2)
        ParamVector = []
        for k, chi2value in enumerate(ValuesOfChi2):
            if (chi2value != numpy.inf):
                lll = [x for x in parameter_vector[k]]
                if (not (lll in ParamVector)):
                    AllChi2Values.append([chi2value, k])
                    ParamVector.append(lll)
        for k, chi2value in enumerate(self.BestChi2):
            AllChi2Values.append([chi2value, RealNumParamVec + k])

        # Debug:
        # print "\nAllChi2Values = ", AllChi2Values


        ## sort list of chi2 values
        AllChi2Values = sorted(AllChi2Values, key=lambda l:l[0])

        # Debug:
        # print "Sorted: AllChi2Values = ", AllChi2Values


        ## make a copy of the current best site variables
        OldBestParameter = deepcopy(self.BestParameter)
        OldBestFunctionValues = deepcopy(self.BestFunctionValues)
        OldBestChi2Values = deepcopy(self.BestChi2Values)
        OldBestInputFiles = deepcopy(self.BestInputFiles)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## store parameter vectors and function values for best sites
        for i in xrange(self.MCMCBestSiteCounter):
            LocalChi2Value = AllChi2Values[i][0]
            LocalChi2Index = AllChi2Values[i][1]

            # Debug:
            #    print "\nLocalChi2Value = ", LocalChi2Value
            #    print "LocalChi2Index = ", LocalChi2Index
            #    print "number_param_set = ", RealNumParamVec
            #    print "LocalChi2Index - RealNumParamVec = ", (LocalChi2Index - RealNumParamVec)


            ## identify best chi2 values
            if (LocalChi2Index < RealNumParamVec):
                self.BestChi2[i] = LocalChi2Value
                self.BestParameter[i] = deepcopy(parameter_vector[LocalChi2Index])
                self.BestFunctionValues[i] = deepcopy(FitFunctionOut[LocalChi2Index])
                self.BestChi2Values[i] = deepcopy(Chi2Values[LocalChi2Index])


                ## get corresponding line with formatted parameters
                if (i == 0):
                    f = open(WorkingDirectory.strip() + "log_lines__single_call__" + str(LocalChi2Index + 1) + ".dat", 'r')
                    self.BestLogLine = f.readline()
                    f.close()
                    self.BestLogLine = self.BestLogLine[0:len(self.BestLogLine)-1]


                ## write best model function values to temp file
                for j in xrange(self.NumberExpFiles):
                    f = open(WorkingDirectory.strip() + "best_model_function_values_call__" + str(i + 1) + "__" + str(j + 1) + ".dat", 'w')
                    for lines in self.BestFunctionValues[i][j]:
                        f.write(str(lines[0]) + "\n")
                    f.close()


                ## write chi2 values to temp file
                for j in xrange(self.NumberExpFiles):
                    f = open(WorkingDirectory.strip() + "best_chi2_values_call__" + str(i + 1) + "__" + str(j + 1) + ".dat", 'w')
                    for lines in self.BestChi2Values[i][j]:
                        f.write(str(lines[0]) + "\n")
                    f.close()


                ## get corresponding contents of the input file(s)
                f = open(WorkingDirectory.strip() + "log_input-files__single_call__" + str(LocalChi2Index + 1) + ".dat", 'r')
                fcontentLocal = f.readlines()
                f.close()
                self.BestInputFiles[i] = fcontentLocal
            else:
                k = LocalChi2Index - RealNumParamVec
                self.BestChi2[i] = LocalChi2Value
                self.BestParameter[i] = deepcopy(OldBestParameter[k])
                self.BestFunctionValues[i] = deepcopy(OldBestFunctionValues[k])
                self.BestChi2Values[i] = deepcopy(OldBestChi2Values[k])
                self.BestInputFiles[i] = deepcopy(OldBestInputFiles[k])


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## append content of chi2-log file to total list of all calculated chi^2 values
        WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"
        for i in xrange(number_param_set):
            if (ValuesOfChi2[i] != numpy.inf):
                f = open(WorkingDirectory.strip() + "log_chi2_single_call.dat", 'r')
                contentChi2LogFile = f.readlines()
                for line in contentChi2LogFile:
                    self.Chi2Channel.write(line)
                    self.Chi2Channel.flush()
                f.close()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## print what you do
        self.FuncCallCounter += 1                                                           ## increase counter for model function calls


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## store chi2 values
        for k in xrange(number_param_set):
            if (ValuesOfChi2[i] != numpy.inf):
                NewArray = []
                for param in parameter_vector[k]:
                    NewArray.append(param)
                NewArray.append(ValuesOfChi2[k])
                self.Points.append(NewArray)

        # Debug:
        # print ">>self.BestChi2 = ", self.BestChi2
        # print "self.BestLogLine = ", self.BestLogLine
        # print "self.BestInputFiles = ", self.BestInputFiles
        # print "self.BestParameter = ", self.BestParameter


        ## define return variable
        ValuesOfChi2[:] = [x * (-1) for x in ValuesOfChi2] 
        return ValuesOfChi2
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## get best result
    def GetBestResult(self):
        """
        input variables:    None

        output variables:   self.BestChi2:              chi2 value for best result
                            self.BestParameter:         parameter vector for best result
                            self.BestLogLine:           line in log file for best result
                            self.BestInputFiles:        content of input file for best result
        """


        ## define return parameters
        return self.BestChi2, self.BestParameter, self.BestLogLine, self.BestInputFiles
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## get function values of best result
    def GetBestFunctionValues(self):
        """
        input variables:    None

        output variables:   self.BestParameter:         parameter vector of best result
                            self.BestFunctionValues:    model function values of best result
                            self.BestChi2Values:        chi2 function values of best result
        """


        ## define return parameters
        return self.BestParameter, self.BestFunctionValues, self.BestChi2Values
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## set parameter SetCurrentParameterIndex
    def SetCurrentParameterIndex(self, paramvalue):
        """
        input variables:    paramvalue:             the new value for CurrentParameterIndex

        output variables:   None

        """
        self.CurrentParameterIndex = paramvalue
        self.Chi2DistributionParam = []
        self.Chi2DistributionChi2 = []


        ## we've done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## get parameter values and the corresponding chi2 values for current parameter
    def SaveChi2Distribution(self):
        """
        input variables:    None

        output variables:   self.Chi2DistributionParam: parameter values for current parameter
                            self.Chi2DistributionChi2:  corresponding chi2 values for current parameter
        """


        ## define return parameters
        return self.Chi2DistributionParam, self.Chi2DistributionChi2
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## get points, i.e. list of all chi2 values
    def GetPoints(self):
        """
        input variables:    None

        output variables:   self.Points:            list of all chi2 values
        """


        ## define return parameters
        return self.Points
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## write informatons of current iteration to log files
    def WriteLogLines(self, CurrentIteration):
        """
        input variables:    CurrentIteration:       number of current iteration

        output variables:   None
        """


        ## print what you do
        bestchi, bestparam, bestline, bestinput = self.GetBestResult()
        chisq = bestchi[0]
        if (self.printflagNum == 1):
            print "\r                %5d%s" % (CurrentIteration, bestline)


        ## wirte information to param-log file
        self.paramchannel.write("\n")
        self.paramchannel.write("\n")
        outputstring = "*" * 122
        self.paramchannel.write(outputstring + "\n")
        outputstring = "Iteration: %5d,  chi^2 = %25.15e" % (CurrentIteration, chisq)
        self.paramchannel.write(outputstring + "\n")
        self.paramchannel.write("\n")
        self.paramchannel.write("\n")
        self.paramchannel.write("\n")
        self.paramchannel.write("Parameters: " + str(bestparam[0]) + "\n")
        outputstring = "-" * 122
        self.paramchannel.write(outputstring + "\n")
        self.paramchannel.write("\n")
        self.paramchannel.flush()


        ## write contents of input file(s) to param-log file
        for lineinput in bestinput[0]:
            self.paramchannel.write(lineinput)
            self.paramchannel.flush()
        self.paramchannel.flush()


        ## wirte information to log file
        outputstring = "                %5d%s" % (CurrentIteration, bestline)
        self.logchannel.write(outputstring + "\n")
        self.logchannel.flush()


        ## define return parameters
        return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Main subroutine to call MCMC
def MCMC(printflag, ModelCallClass, fit_log, paramchannel, logchannel, fitlogChi2, Chi2Channel, RenormalizedChi2, NumberExpFiles, NumberYColumns, \
         LengthExpRange, num_par, chilm, max_iter, param_up_down, min_point, plotflag, MCMCCounter, NumberMCMCSampler, NumberBurnInIter, JobDir, \
         UsePreviousResultsFlag, SampleMethod, UsedFor):
    """
    input variables:            printflag:              flag indicating if messages are printed to screen or not
                                ModelCallClass:         class for model function call
                                fit_log:                path and name of log file
                                paramchannel:           channel for parameter log file
                                logchannel:             channel for main log file
                                fitlogChi2:             path and name of chi2 log file
                                Chi2Channel:            channel for chi2 log file
                                RenormalizedChi2:       renormalization flag
                                NumberExpFiles:         number of obs. data files
                                NumberYColumns:         number of y columns for each data file
                                LengthExpRange:         number of data points for each data file
                                num_par:                number of parameter
                                chilm:                  lower limit for chi^2
                                max_iter:               max. number of iterations
                                param_up_down:          parameters with upper and lower limits
                                min_point:              parameter describing point of minimum
                                plotflag:               flag for plotting
                                MCMCCounter:            counter for MCMC algorithm call
                                NumberMCMCSampler:      number of samplers
                                NumberBurnInIter:       number of iterations for burn-in phase
                                JobDir:                 path and name of current job directory
                                UsePreviousResultsFlag: flag indicating usage of results of previous algorithms
                                SampleMethod:           defines how new walkers are distributed (global or local)
                                UsedFor:                defines from which algorithm this subroutine is called

    output variables:           calstatus:              status flag of calculation (= 0: all ok)
                                AddInfo:                best chi2 value and corresponding parameter vector etc.
    """


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import modified emcee package
    import emcee__modified


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return parameter
    calstatus = 0
    AddInfo = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check user input


    ## check user-defined number of samples (walkers)
    if (NumberMCMCSampler <= 0):                                                           ## if number is <= 0, re-define number
        NumberMCMCSampler = (num_par * 10)
    elif (NumberMCMCSampler < (2 * num_par)):
        NumberMCMCSampler = (num_par * 2)
    NumberMCMCSampler = int(NumberMCMCSampler)
    if (NumberMCMCSampler % 2 != 0):
        NumberMCMCSampler = NumberMCMCSampler - 1
        if (NumberMCMCSampler == 0):
            NumberMCMCSampler = 2


    ## check user-defined number of iterations for burn-in phase
    if (NumberBurnInIter <= 0):
        NumberBurnInIter = NumberMCMCSampler
    NumberBurnInIter = int(NumberBurnInIter)

    # Debug:
    # print "\n\n\n"
    # print "num_par = ", num_par
    # print "NumberMCMCSampler = ", NumberMCMCSampler
    # print "NumberBurnInIter = ", NumberBurnInIter


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    # Set up the sampler.
    ndim = int(num_par)                                                                     ## dimension of parameter space
    nwalkers = NumberMCMCSampler                                                            ## user defined number of samples
    NumVectors = 1
    NumberOfPreCalculatedParamVectors = 0
    p0 = []
    lnprop = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read in the .chi2.log files from other algorithms
    if (printflag == "true"):
        print "\n\n\t Initialize MCMC run .."
        sys.stdout.flush()
    WorkingDir = os.path.dirname(fit_log) + "/"                                             ## determine working directory
    listing = os.listdir(WorkingDir)                                                        ## read in all files in the current job directory
    ListChi2LogFiles = []                                                                   ## determine number of output files
    if (UsePreviousResultsFlag):
        for files in listing:
            LocalFileName = WorkingDir + files                                              ## define absolute path of file
            if (files.endswith(".log.chi2") and LocalFileName != fitlogChi2):
                try:
                    LocalChi2Content = numpy.loadtxt(LocalFileName, skiprows = 1)           ## read in parameter vectors and chi2 values from file
                except (ValueError, IOError):
                    LocalChi2Content = []
                if (len(LocalChi2Content) > 0):
                    LocalChi2Content = numpy.delete(LocalChi2Content, 0, 1)                 ## delete column with numbering
                    if (ListChi2LogFiles != []):
                        ListChi2LogFiles = numpy.concatenate((ListChi2LogFiles, LocalChi2Content), axis = 0)
                    else:
                        ListChi2LogFiles = LocalChi2Content

                    # Debug:
                    # print "len(ListChi2LogFiles) = ", len(ListChi2LogFiles)


    ## analyze previous calculated parameter vectors
    if (ListChi2LogFiles != []):


        ## sort multidimensional array along 1st column, taken from http://stackoverflow.com/questions/2828059/sorting-arrays-in-numpy-by-column
        ListChi2LogFiles = ListChi2LogFiles[numpy.lexsort(numpy.fliplr(ListChi2LogFiles).T)]
        lnpropTemp = ListChi2LogFiles[:, 0]                                                 ## get (sorted) chi2 values
        p0Temp = ListChi2LogFiles[:, 1:]                                                    ## get corresponding parameter vectors

        # Debug:
        # print "ListChi2LogFiles = ", ListChi2LogFiles
        # print "ListChi2LogFiles[:, 0] = ", ListChi2LogFiles[:, 0]
        # print "ListChi2LogFiles[0, :] = ", ListChi2LogFiles[0, :]
        # print "lnpropTemp = ", lnpropTemp
        # print "p0Temp = ", p0Temp


        ## exclude chi2 values with chi^2 >= 1.e99 or chi^2 = +/-inf or chi^2 = NaN
        for i, Localchi2Value in enumerate(lnpropTemp):
            if ((i + 1) > nwalkers):                                                        ## we need only "nwalkers" samples
                break
            else:
                if (not (numpy.isnan(Localchi2Value)) and abs(Localchi2Value) < 1.e90 and abs(Localchi2Value) != numpy.inf):
                    lnprop.append(Localchi2Value)                                           ## save current chi2 value
                    p0.append(p0Temp[i])                                                    ## save corresponding parameter vector


    # determine additional walkers
    NumberOfPreCalculatedParamVectors = len(lnprop)                                         ## get number of already calculated parameter vectors
    if (NumberOfPreCalculatedParamVectors < nwalkers):
        p0Add = []
        for i in xrange(NumberOfPreCalculatedParamVectors, nwalkers):
            NewVector = []
            for j in xrange(ndim):
                a = param_up_down[j, 0]
                b = param_up_down[j, 1]
                if (SampleMethod == "global"):                                              ## distribute walkers in the entire parameter space
                    NewValue = (b - a) * numpy.random.random_sample() + a
                elif (SampleMethod == "local"):                                             ## distribute walkers in the neighbourhood of min. point
                    MinVal = min_point[j]
                    aMin = MinVal - min(a, b)
                    bMin = max(a, b) - MinVal
                    NewValue = (-1.0)**numpy.random.random_integers(2) * min(aMin, bMin) * numpy.random.random_sample() + MinVal
                NewVector.append(NewValue)
            p0Add.append(NewVector)
            p0.append(NewVector)


        ## calculate corresponding chi^2 value
        if (printflag == "true"):
            print "\n\n\t Determine additional walkers .."
            sys.stdout.flush()
        NumParamVectors = len(p0Add)                                                        ## get number of parameter vectors
        lnpropAdd = ModelCallClass.logLhood(p0Add, NumParamVectors, 0.0)                    ## call ModelCallClass.logLhood
        lnprop = lnprop + lnpropAdd
    if (lnprop != []):                                                                      ## convert to numpy array
        lnprop = numpy.asarray(lnprop)

    # Debug:
    # print "len(p0) = ", p0
    # print "p0[0] = ", p0[0]
    # print "nwalkers = ", nwalkers
    # print "lnprop = ", lnprop
    # print "len(lnprop) = ", len(lnprop)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start with the minimum from the previous algorithm(s)
    sampler = emcee__modified.EnsembleSampler(nwalkers, ndim, ModelCallClass.logLhood, args=(NumVectors,))


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## print what you do to log file (and screen)
    OutputString1 = "\n\n           Calculate burn-in steps:"
    logchannel.write(OutputString1 + "\n")
    OutputString2 = "\n\n           Iteration:                    chi^2:     Parameter:"
    logchannel.write(OutputString2 + "\n")
    logchannel.flush()
    if (printflag == "true"):
        print OutputString1
        sys.stdout.flush()
        print OutputString2
        sys.stdout.flush()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Burn-In: Run NumberBurnInIter steps as a burn-in.
    pos, prob, state = sampler.run_mcmc(p0, NumberBurnInIter, ModelCallClass, lnprob0 = lnprop)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Reset the chain to remove the burn-in samples.
    sampler.reset()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## print what you do to log file (and screen)
    OutputString1 = "\n\n           Calculate final MCMC run:"
    logchannel.write(OutputString1 + "\n")
    OutputString2 = "\n\n           Iteration:                    chi^2:     Parameter:"
    logchannel.write(OutputString2 + "\n")
    logchannel.flush()
    if (printflag == "true"):
        print OutputString1
        sys.stdout.flush()
        print OutputString2
        sys.stdout.flush()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Starting from the final position in the burn-in chain, sample for max_iter steps.
    pos, prob, state  = sampler.run_mcmc(pos, max_iter, ModelCallClass, rstate0 = state)

    # Debug:
    #    print "\npos = ", pos
    #    print "\nprob = ", prob
    #    print "\nstate = ", state
    #    print "\n\n"
    #    for i in xrange(len(pos)):
    #        print prob[i], pos[i]


    ## get results
    bestchi, bestparam, bestline, bestinput = ModelCallClass.GetBestResult()
    AddInfo = [pos, prob, bestinput, sampler, bestchi, bestparam]

    # Debug:
    #    print "bestparam = ", bestparam


    ## Print out the mean acceptance fraction. In general, acceptance_fraction has an entry for each walker so, in this case, it is a ndim-dimensional
    ## vector.
    print "\n\n\t Mean acceptance fraction:  %.6f \n\n" % (numpy.mean(sampler.acceptance_fraction))

    # Debug:
    # sys.exit(0)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## post processing


    ## define working directory
    LocalFitLog = fitlogChi2.strip()
    mm = LocalFitLog.rfind("/")
    if (mm > (-1)):
        path_to_save_png = LocalFitLog[:mm + 1]
    else:
        path_to_save_png = ""
    if (UsedFor == "error"):
        path_to_save_png += "ErrorEstim_"


    ## create plots for each parameter describing the parameter values for each iteration
    LabelList = []
    ErrorList = []
    for ParamIndex in xrange(ndim):
        LabelList.append("free param. %i" % (ParamIndex + 1))
        LocalErrorList = []


        ##----------------------------------------------------------------------------------------------------------------------------------------------------
        ## create plots for each parameter describing the parameter values for each iteration
        fig = pylab.figure(figsize=(15, 10))
        fig.clear()
        layer = pylab.subplot(1, 1, 1)
        pylab.grid(True)
        pylab.xlabel("step number for free parameter %i" % (ParamIndex + 1))
        pylab.ylabel("parameter value")
        layer.plot(sampler.chain[:, :, ParamIndex].T, color="k", alpha=0.4)
        # layer.yaxis.set_major_locator(MaxNLocator(5))
        if (plotflag == "true" or plotflag == "saveonly"):
            pylab.savefig(path_to_save_png + "MCMC__parameter-evolution_of_free-parameter_parm_" + str(ParamIndex + 1) + "__call_" \
                          + str(abs(MCMCCounter)) + ".png", format='png')
        pylab.draw()
        matplotlib.pyplot.close(fig)


        ## save results of mcmc call
        numpy.savetxt(path_to_save_png + "MCMC__walkers_of_free-parameter_parm_" + str(ParamIndex + 1) + "__call_" + str(abs(MCMCCounter)) + ".dat", \
                      sampler.chain[:, :, ParamIndex])


    ## define return parameters
    return (calstatus, AddInfo)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## main routinte for MCMC algorithm
def Start(SpecialAlgorithmSettings, GeneralAlgorithmSettings, printflagNum, LastAlgorithmNum, ParamSetCounter, chilm, numiter, MCMCCounter, \
          DeterminationChi2, PlotIteration, PlotType, fit_log, NumberInputFiles, NumberOutputFiles, ParallelizationFlag, JobID, MaxInputLines, \
          MaxParameter, RenormalizedChi2, currentpath, CalculationMethod, xAxisLabel, yAxisLabel, zAxisLabel, PathStartScript, \
          ExeCommandStartScript, parameter_set, ExpDataX, ExpDataY, ExpDataError, NumberRanges, MinRange, MaxRange, NumberXColumns, \
          NumberYColumns, LengthExpRange, MaxRangeNumber, NumberExpFiles, MaxLength, MaxColX, MaxColY, NumberParameter, plotflag, modelflag, \
          MPIFlag, MAGIXrootDirectory, JobDir):
    """
    input variables:            SpecialAlgorithmSettings:   algorithm user setttings
                                GeneralAlgorithmSettings:   special algorithm settings
                                printflagNum:           flag for screen output 1 (=yes) or 0 (=no)
                                LastAlgorithmNum:       number of last algorithm
                                ParamSetCounter:        number of best sites
                                chilm:                  user defined abort criteria for chi**2
                                numiter:                max. number of iterations
                                MCMCCounter:            counts number of calls
                                DeterminationChi2:      method being used for the determination of chi^2
                                PlotIteration:          plot model function for each iteration set 1(=yes) or 0(=no)
                                PlotType:               get type of plot
                                fit_log:                path for log-file containing the current values of chi**2
                                NumberInputFiles:       number of input files for the external model program
                                NumberOutputFiles:      number of output files for the external model program
                                ParallelizationFlag:    contains the number of processors used for parallelization
                                JobID:                  job identification number
                                MaxInputLines:          max number of lines in an input file
                                MaxParameter:           max number of parameters in a line of an input file
                                RenormalizedChi2:       flag for using renormalized chi**2
                                currentpath:            path of the working directory
                                CalculationMethod:      method of computation (at once or point-to-point)
                                xAxisLabel:             label of the x-axis (for plot)
                                yAxisLabel:             label of the y-axis (for plot)
                                zAxisLabel:             label of the z-axis (for plot)
                                PathStartScript:        path and name of the start script for calling model function
                                ExeCommandStartScript:  command for calling model function
                                parameter_set:          the complete set of paramters (incl. flags and limits)
                                ExpDataX:               array containing the experimental x side
                                ExpDataY:               array containing the experimental y side
                                ExpDataError:           array containing the experimental error of the y side
                                NumberRanges:           number of ranges
                                MinRange:               array containing lower limit of each range for each file
                                MaxRange:               array containing upper limit of each range for each file
                                NumberXColumns:         number x-columns
                                NumberYColumns:         number y-columns
                                LengthExpRange:         total number of data points
                                MaxRangeNumber:         max number of ranges over all files
                                NumberExpFiles:         number of exp. data files
                                MaxLength:              max. length of experimental data
                                MaxColX:                number of columns concerning to the experimental x side
                                MaxColY:                number of columns concerning to the experimental y side
                                NumberParameter:        number of model parameter
                                plotflag:               flag for plotting histograms
                                modelflag:              flag to use optimized MAGIX packages for a certain model or not
                                MPIFlag:                mpi flag
                                MAGIXrootDirectory:     MAGIX root directory
                                JobDir:                 job directory

    output variables:           FitFunctionValues:      values of the model function of the last function call
                                Chi2Values:             corresponding chi^2 values
    """


    ## initialize return parameters
    FitFunctionValues = []
    Chi2Values = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get special settings
    NumberMCMCSampler = SpecialAlgorithmSettings[0]                                         ## get number of samplers (only used for MCMC)
    NumberBurnInIter = SpecialAlgorithmSettings[1]                                          ## get number of iterations for burn-in (only used for MCMC)
    MCMCBestSiteCounter = SpecialAlgorithmSettings[2]                                       ## get number of best sites
    max_iter = numiter                                                                      ## get max. number of iterations

    # Debug:
    #    print "NumberMCMCSampler = ", NumberMCMCSampler
    #    print "NumberBurnInIter = ", NumberBurnInIter
    #    print "MCMCBestSiteCounter = ", MCMCBestSiteCounter
    #    print "max_iter = ", max_iter
    #    print ">>>NumberParameter = ", NumberParameter


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define num_par, min_point and param_up_down from parameter_set
    for i in range(NumberParameter):
        parameter_set[1][i] = float(parameter_set[1][i])
    num_par = int(sum(parameter_set[1]))                                                    ## determine num_par
    min_point = numpy.zeros(num_par, dtype = 'float')                                       ## determine minimum point
    param_up_down = numpy.zeros((num_par, 2), dtype = 'float')                              ## define param_up_down
    j = (-1)
    for i in range(NumberParameter):
        parameter_set[1][i] = float(parameter_set[1][i])
        if (parameter_set[1][i] == 1):
            j += 1
            min_point[j] = parameter_set[0][i]
            param_up_down[j, 0] = parameter_set[2][i]
            param_up_down[j, 1] = parameter_set[3][i]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## print what you do
    printflag = "true"
    if (printflagNum == 1):
        printflag = "true"
    else:
        printflag = "false"
    if (printflag == "true"):
        print "\n         temporary files are stored in: " + str(os.environ.get('MAGIXTempDirectory','')).strip() + "job_" + str(JobID).strip() + "/"
        print "\n\n         Start MCMC algorithm .."


    ## modify file names for log-files
    fit_log = fit_log.strip()
    i = fit_log.rfind("/")
    j = fit_log.rfind(".")
    if (j > i):
        fit_log = fit_log[:j] + "__MCMC__call_" + str(abs(MCMCCounter)).strip() + ".log"
    else:
        fit_log = fit_log + "__MCMC__call_" + str(abs(MCMCCounter)).strip() + ".log"

    # Debug:
    # print 'fit_log = ',fit_log


    ## open param-log file
    fitlogparam = fit_log.strip() + ".param"                                                ## define file name for param-log file
    paramchannel = open(fitlogparam.strip(), 'w')
    paramchannel.write("\n")
    LogString = "log-file containing the actual values of the parameters used in the MCMC algorithm:"
    paramchannel.write(LogString + "\n")
    LogString = "-" * len(LogString)
    paramchannel.write(LogString + "\n")
    paramchannel.write("\n")
    paramchannel.flush()


    ## open log file and get current time and date
    fitlogparam = fit_log.strip()                                                           ## define file name for param-log file
    logchannel = open(fitlogparam.strip(), 'w')
    logchannel.write("\n")
    LogString = "log-file for MCMC algorithm:"
    logchannel.write(LogString + "\n")
    LogString = "-" * len(LogString)
    logchannel.write(LogString + "\n")
    logchannel.write("\n")
    logchannel.write("\n")


    ## get local time
    lt = time.localtime()
    datestring = time.strftime("algorithm starts at Date: %d.%m.%Y", lt) + time.strftime(",     Time: %H:%M:%S", lt)
    logchannel.write(str(datestring) + "\n")
    logchannel.write("\n")
    logchannel.write("\n")
    logchannel.flush()


    ## open log file and get current time and date
    fitlogChi2 = fit_log.strip() + ".chi2"                                                  ## define file name for param-log file
    Chi2Channel = open(fitlogChi2.strip(), 'w')


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import MAGIX package ModelFunctionCall ..


    ## define name of MAGIX executable
    if (modelflag == "true" or modelflag.lower() == "myxclass"):                            ## .. optimized for myXCLASS model
        StarterExecutable = "Starter__myXCLASS.exe"
    else:                                                                                   ## .. default package
        StarterExecutable = "Starter.exe"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define parameters for class
    ModelCallClass = ModelFunctionCallClass(printflagNum, GeneralAlgorithmSettings, LastAlgorithmNum, DeterminationChi2, PlotIteration, PlotType, \
                                            fit_log, NumberInputFiles, NumberOutputFiles, ParallelizationFlag, JobID, MaxInputLines, MaxParameter, \
                                            RenormalizedChi2, currentpath, CalculationMethod, xAxisLabel, yAxisLabel, zAxisLabel, PathStartScript, \
                                            ExeCommandStartScript, parameter_set, ExpDataX, ExpDataY, ExpDataError, NumberRanges, MinRange, MaxRange, \
                                            NumberXColumns, NumberYColumns, LengthExpRange, MaxRangeNumber, NumberExpFiles, MaxLength, MaxColX, \
                                            MaxColY, NumberParameter, MPIFlag, MAGIXrootDirectory, JobDir, SpecialAlgorithmSettings, StarterExecutable, \
                                            Chi2Channel, logchannel, paramchannel, MCMCBestSiteCounter)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start MCMC algorithm
    UsePreviousResultsFlag = False
    SampleMethod = "global"
    UsedFor = "mcmc"
    calstatus, AddInfo = MCMC(printflag, ModelCallClass, fit_log, paramchannel, logchannel, fitlogChi2, Chi2Channel, RenormalizedChi2, NumberExpFiles, \
                              NumberYColumns, LengthExpRange, num_par, chilm, max_iter, param_up_down, min_point, plotflag, MCMCCounter, \
                              NumberMCMCSampler, NumberBurnInIter, JobDir, UsePreviousResultsFlag, SampleMethod, UsedFor)
    AllChi2DistributionParamValues = AddInfo[0]
    AllChi2DistributionChi2Values = AddInfo[1]
    Bestinput = AddInfo[2]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## close param-log file
    Chi2Channel.close()
    paramchannel.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read in contents of log file
    Chi2Channel = open(fitlogChi2.strip(), 'r')
    contens = Chi2Channel.readlines()                                                       ## read in contents of input file
    Chi2Channel.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## sort log.chi^2 file
    sortflag = "true"                                                                       ## define if log file is sorted or not
    if (sortflag == "true"):                                                                ## !!! sorting does not work properly !!!!
        Chi2Contents = []
        for inputline in contens:                                                           ## loop over all lines of the input file
            line = inputline.split()
            Currentline = []
            count = 0
            for elements in line:
                count += 1
                if (count == 1):
                    elements = float(elements)
                Currentline.append(elements)
            Chi2Contents.append(Currentline)


        ## sort belonging to chi^2 values
        sortchi2 = sorted(Chi2Contents)

        # Debug:
        # print Chi2Contents[0]
        # print sortchi2[0]


        ## write sorted chi^2 parameter vectors
        Chi2Channel = open(fitlogChi2.strip(), 'w')
        Chi2Channel.write("    Nr.:                   chi**2:  Parameters:\n")
        lastline = ""
        countLine = 0
        for inputline in sortchi2:
            count = 0
            formatline = ""
            for elements in inputline:
                count += 1
                if (count == 1):
                    formatline = '%27.15e        ' % float(elements)
                else:
                    k = len(elements)
                    if (k < 20):
                        formatelements = ' %20s ' % elements
                    else:
                        formatelements = " " + elements + " "
                    formatline += formatelements
            if (lastline != formatline):
                lastline = formatline
                countLine += 1
                countLineString = '%8d' % countLine
                Chi2Channel.write(countLineString + " " + str(formatline) + "\n")
        Chi2Channel.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## if log file is not sorted just write a header line at the beginning of the log file
    else:
        Chi2Channel = open(fitlogChi2.strip(), 'w')
        Chi2Channel.write("    Nr.:                   chi**2:  Parameters:\n")
        countLine = 0
        for line in contens:
            countLine += 1
            countLineString = '%8d' % countLine
            Chi2Channel.write(countLineString + " " + line)
        Chi2Channel.close()


    ## close log file
    logchannel.write("\n")
    lt = time.localtime()
    datestring = time.strftime("algorithm ends at Date: %d.%m.%Y", lt) + time.strftime(",     Time: %H:%M:%S", lt)
    logchannel.write(str(datestring) + "\n")
    logchannel.write("\n")
    helpString = "-" * 150
    logchannel.write(helpString + "\n")
    logchannel.flush()
    logchannel.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create final input files


    ## define base directory, i.e. the path where the final input file(s) are written to
    BaseDir = fit_log.strip()
    i = BaseDir.rfind("/")
    if (i > (-1)):
        BaseDir = BaseDir[:i + 1]
    else:
        BaseDir = ""


    ## extract informations from Bestinput variable
    NameInputFile = ""
    WriteFlag = False
    for LocalBestSite in xrange(MCMCBestSiteCounter):
        for line in Bestinput[LocalBestSite]:
            StrippedLine = line.strip()


            ## get name of input file
            if (StrippedLine.startswith("Input-File ")):
                k = StrippedLine.find(":  , file:")
                if (k > (-1)):
                    NameInputFile = StrippedLine[k + 10:].strip()


                    ## modify input file name
                    j = NameInputFile.rfind(".")
                    if (j > (-1)):
                        NameInputFile = NameInputFile[:j] + "__MCMC__call_" + str(abs(MCMCCounter))
                    else:
                        NameInputFile = NameInputFile + "__MCMC__call_" + str(abs(MCMCCounter))
                    if (MCMCBestSiteCounter > 1):
                        NameInputFile += "__site_" + str(LocalBestSite + 1)
                    NameInputFile += ".out.input"


                    ## open channel for final input file
                    NameInputFile = BaseDir + "/" + NameInputFile
                    InputFile = open(NameInputFile, 'w')


            ## identify begin of contents of final input file(s)
            elif (StrippedLine.startswith("-start_input-file-")):
                WriteFlag = True


            ## identify end of contents of final input file(s)
            elif (StrippedLine.startswith("-end_input-file-")):
                try:
                    InputFile.close()
                except AttributeError:
                    NameInputFile = ""
                WriteFlag = False


            ## write contents of final input file(s) to current input file
            elif (WriteFlag == True and NameInputFile != ""):
                InputFile.write(line)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get parameters and function values for best sites
    FinalParameter, FitFunctionAll, Chi2ValuesAll = ModelCallClass.GetBestFunctionValues()


    ## define new parameter set
    FinalParameterSet = []
    for LocalBestSite in xrange(MCMCBestSiteCounter):
        j = (-1)
        LocalParameterSet = []
        for i in range(NumberParameter):
            if (parameter_set[1][i] == 1):
                j += 1
                LocalParameterSet.append(FinalParameter[LocalBestSite][j])
            else:
                LocalParameterSet.append(parameter_set[0][i])
        FinalParameterSet.append(LocalParameterSet)

    # Debug:
    # print "NumberParameter = ", NumberParameter
    # print "FinalParameter = ", FinalParameter


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define return variables
    calstatus = 0
    return (calstatus, FinalParameterSet, FitFunctionAll, Chi2ValuesAll)
##--------------------------------------------------------------------------------------------------------------------------------------------------------
