!*********************************************************************************************************************************************************
!>  Package: Levenberg-Marquardt algorithm
!>
!>
!>  These modules contain the subroutines for the different versions of the Levenberg-Marquardt algorithm
!>  Copyright (C) 2009 - 2016  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - Module LevenbergMarquardt_Variables:      module containing global variables for all Levenberg-Marquardt subroutines
!>      - Module LevenbergMarquardt_MINPACK:        module containing global variables for the Levenberg-Marquardt algorithm (MINPACK version)
!>      - subroutine enorm:                         (MINPACK): computes the Euclidean norm of a vector.
!>      - subroutine lmder:                         (MINPACK): minimizes M functions in N variables by the Levenberg-Marquardt method.
!>      - subroutine lmder1:                        (MINPACK): minimizes M functions in N variables by the Levenberg-Marquardt method.
!>      - subroutine lmpar:                         (MINPACK): computes a parameter for the Levenberg-Marquardt method.
!>      - subroutine qrfac:                         (MINPACK): computes a QR factorization using Householder transformations.
!>      - subroutine qrsolv:                        (MINPACK): solves a rectangular linear system A*x=b in the least squares sense.
!>      - subroutine covar_minpack:                 (MINPACK): covar_minpack
!>      - subroutine fcn:                           (MINPACK): fcn
!>      - subroutine call_LevenbergMarquardt_MINPACK: (MINPACK): calls subroutines for MINPACK verison of Levenberg-Marquardt algorithm
!>      - Module Algorithm:                         module contains the main subroutine used to start the different versions of the Levenberg-Marquardt
!>                                                  algorithm
!>      - subroutine MainAlg:                       main subroutine which starts the Levenberg-Marquardt algorithm
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    06.22.2009  Initial version
!>  T. Moeller    13.01.2012  Updated version
!>  T. Moeller    26.08.2014  modified and restructured for GPU
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!> Module: LevenbergMarquardt_Variables
!>
!>         Module contains global variables and subroutines for all Levenberg-Marquardt subroutines
!>
!>
!> \author Thomas Moeller
!>
!> \date 09.06.2010
!>
Module LevenbergMarquardt_Variables

    use Model

    integer :: nfit                                                                         !< (= effective parameter number) number of model
                                                                                            !< parameters which are optimized
    integer :: NumIterGlobal                                                                !< max. number of iterations
    integer :: MaxExpLines                                                                  !< max. number of lines in exp. file
    integer :: PlotIterationGlobal                                                          !< plot model function for each iter. set 1 (=yes) or 0 (=no)
    integer :: PlotTypeGlobal                                                               !< get type of plot
    real*8, allocatable, dimension(:) :: currentparmval                                     !< array containing the current parameter set within the
                                                                                            !< iteration loop
    real*8,allocatable,dimension(:) :: posdatexp                                            !< array defining the experimental x point
    logical, allocatable, dimension(:) :: ia                                                !< array indicating if parameter should be optimized (true)
                                                                                            !< or not (false)
    character(len=256) :: xAxisLabelGlobal                                                  !< label of the x-axis (for plot)
    character(len=256) :: yAxisLabelGlobal                                                  !< label of the y-axis (for plot)
    character(len=256) :: zAxisLabelGlobal                                                  !< label of the z-axis (for plot)
    character(len=8192) :: fitlogGlobal                                                     !< path for log-file containing the current values of chi**2
end Module LevenbergMarquardt_Variables
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!> Module: LevenbergMarquardt_MINPACK
!>
!>         Module contains global variables and subroutines for the Levenberg-Marquardt (MINPACK version) subroutines
!>
!>
!> \author Thomas Moeller
!>
!> \date 09.06.2010
!>
Module LevenbergMarquardt_MINPACK

    use Variables
    use FunctionCalling
    use LevenbergMarquardt_Variables

    implicit none

    contains


        !>************************************************************************************************************************************************
        !> subroutine: enorm
        !>
        !>  (MINPACK): ENORM computes the Euclidean norm of a vector.
        !>
        !> input variables:     n:              is the length of the vector.
        !>                      x:              the vector whose norm is desired.
        !>
        !> output variables:    val:            the Euclidean norm of the vector
        !>
        !>
        !> \author MINPACK and Thomas Moeller
        !>
        !> \date 06.04.2010
        !>
        subroutine enorm(val, n, x)
            !<
            !<  Discussion:
            !<
            !<    This is an extremely simplified version of the original ENORM
            !<    routine, which has been renamed to "ENORM2".
            !<
            !<  Licensing:
            !<
            !<    This code is distributed under the GNU LGPL license.
            !<
            !<  Modified:
            !<
            !<    06 April 2010
            !<
            !<  Author:
            !<
            !<    Original FORTRAN77 version by Jorge More, Burton Garbow, Kenneth Hillstrom.
            !<    FORTRAN90 version by John Burkardt.
            !<
            !<  Reference:
            !<
            !<    Jorge More, Burton Garbow, Kenneth Hillstrom,
            !<    User Guide for MINPACK-1,
            !<    Technical Report ANL-80-74,
            !<    Argonne National Laboratory, 1980.
            !<
            !<  Parameters:
            !<
            !<    Input, integer :: N, is the length of the vector.
            !<
            !<    Input, real ( kind = 8 ) X(N), the vector whose norm is desired.
            !<
            !<    Output, real ( kind = 8 ) val, the Euclidean norm of the vector.
            !<
            !<--------------------------------------------------------------------------------------------------------------------------------------------

            implicit none
            integer :: n                                                                    !< INPUT:   is the length of the vector.
            real*8 :: val                                                                   !< OUTPUT:  the Euclidean norm of the vector
            real*8, dimension(n) :: x                                                       !< INPUT:   the vector whose norm is desired.

            val = dsqrt(sum(x(1:n)**2))
            return
        end subroutine enorm


        !>************************************************************************************************************************************************
        !> subroutine: lmder
        !>
        !> (MINPACK): LMDER minimizes M functions in N variables by the Levenberg-Marquardt method.
        !>
        !> input variables:     m:                  is the number of functions
        !>                      n:                  is the number of variables.  N must not exceed M.
        !>                      x:                  On input, X must contain an initial estimate of the solution vector. On output X contains the
        !>                                          final estimate of the solution vector.
        !>                      ldfjac:             LDFJAC, the leading dimension of the array FJAC. LDFJAC must be at least M.
        !>                      ftol:               Termination occurs when both the actual and predicted relative reductions in the sum of squares
        !>                                          are at most FTOL. Therefore, FTOL measures the relative error desired in the sum of squares.
        !>                                          FTOL should be nonnegative.
        !>                      xtol:               Termination occurs when the relative error between two consecutive iterates is at most XTOL.
        !>                                          XTOL should be nonnegative.
        !>                      gtol:               Termination occurs when the cosine of the angle between FVEC and any column of the jacobian is
        !>                                          at most GTOL in absolute value.  Therefore, GTOL measures the orthogonality desired between
        !>                                          the function vector and the columns of the jacobian. GTOL should  be nonnegative.
        !>                      maxfev:             Termination occurs when the number of calls to FCN with IFLAG = 1 is at least MAXFEV by the end
        !>                                          of an iteration.
        !>                      diag:               If MODE = 1, then DIAG is set internally.  If MODE = 2, then DIAG must contain positive entries
        !>                                          that serve as multiplicative scale factors for the variables.
        !>                      mode:               MODE, scaling option.
        !>                                          1, variables will be scaled internally.
        !>                                          2, scaling is specified by the input DIAG vector.
        !>                      factor:             FACTOR, determines the initial step bound. This bound is set to the product of FACTOR and the
        !>                                          euclidean norm of DIAG*X if nonzero, or else to FACTOR itself.  In most cases, FACTOR should lie
        !>                                          in the interval (0.1, 100) with 100 the recommended value.
        !>                      nprint:             NPRINT, enables controlled printing of iterates if it is positive.  In this case, FCN is called
        !>                                          with IFLAG = 0 at the beginning of the first iteration and every NPRINT iterations thereafter
        !>                                          and immediately prior to return, with X and FVEC available for printing.  If NPRINT is not
        !>                                          positive, no special calls of FCN with IFLAG = 0 are made.
        !>
        !> output variables:    x:                  On input, X must contain an initial estimate of the solution vector. On output X contains the
        !>                                          final estimate of the solution vector.
        !>                      fvec:               the functions evaluated at the output X
        !>                      fjac:               n M by N array. The upper N by N submatrix of FJAC contains an upper triangular matrix R
        !>                                          with diagonal elements of nonincreasing magnitude such that
        !>                                          P' * ( JAC' * JAC ) * P = R' * R,
        !>                                          where P is a permutation matrix and JAC is the final calculated jacobian. Column J of P is column
        !>                                          IPVT(J) of the identity matrix. The lower trapezoidal part of FJAC contains information
        !>                                          generated during the computation of R.
        !>                      qtf:                contains the first N elements of Q'*FVEC.
        !>                      info:               error flag
        !>                      nfev:               the number of calls to FCN with IFLAG = 1.
        !>                      njev:               the number of calls to FCN with IFLAG = 2.
        !>                      ipvt:               defines a permutation matrix P such that  JAC*P = Q*R, where JAC is the final calculated
        !>                                          jacobian, Q is orthogonal (not stored), and R is upper triangular with diagonal elements of
        !>                                          nonincreasing magnitude.  Column J of P is column IPVT(J) of the identity matrix.
        !>
        !>
        !> \author MINPACK and Thomas Moeller
        !>
        !> \date 06.04.2010
        !>
        subroutine lmder(m, n, x, fvec, fjac, ldfjac, ftol, xtol, gtol, maxfev, diag, mode, factor, nprint, info, nfev, njev, ipvt, qtf)
            !<
            !<  Discussion:
            !<
            !<    LMDER minimizes the sum of the squares of M nonlinear functions in
            !<    N variables by a modification of the Levenberg-Marquardt algorithm.
            !<    The user must provide a subroutine which calculates the functions
            !<    and the jacobian.
            !<
            !<  Licensing:
            !<
            !<    This code is distributed under the GNU LGPL license.
            !<
            !<  Modified:
            !<
            !<    06 April 2010
            !<
            !<  Author:
            !<
            !<    Original FORTRAN77 version by Jorge More, Burton Garbow, Kenneth Hillstrom.
            !<    FORTRAN90 version by John Burkardt.
            !<
            !<  Reference:
            !<
            !<    Jorge More, Burton Garbow, Kenneth Hillstrom,
            !<    User Guide for MINPACK-1,
            !<    Technical Report ANL-80-74,
            !<    Argonne National Laboratory, 1980.
            !<
            !<  Parameters:
            !<
            !<    Input, external FCN, the name of the user-supplied subroutine which
            !<    calculates the functions and the jacobian.  FCN should have the form:
            !<
            !<      subroutine fcn ( m, n, x, fvec, fjac, ldfjac, iflag )
            !<      integer :: ldfjac
            !<      integer :: n
            !<      real fjac(ldfjac,n)
            !<      real fvec(m)
            !<      integer :: iflag
            !<      real x(n)
            !<
            !<    If IFLAG = 1 on intput, FCN should calculate the functions at X and
            !<    return this vector in FVEC.
            !<    If IFLAG = 2 on input, FCN should calculate the jacobian at X and
            !<    return this matrix in FJAC.
            !<    To terminate the algorithm, the user may set IFLAG negative.
            !<
            !<    Input, integer :: M, is the number of functions.
            !<
            !<    Input, integer :: N, is the number of variables.  N must not exceed M.
            !<
            !<    Input/output, real ( kind = 8 ) X(N).  On input, X must contain an initial
            !<    estimate of the solution vector.  On output X contains the final
            !<    estimate of the solution vector.
            !<
            !<    Output, real ( kind = 8 ) FVEC(M), the functions evaluated at the output X.
            !<
            !<    Output, real ( kind = 8 ) FJAC(LDFJAC,N), an M by N array.  The upper
            !<    N by N submatrix of FJAC contains an upper triangular matrix R with
            !<    diagonal elements of nonincreasing magnitude such that
            !<      P' * ( JAC' * JAC ) * P = R' * R,
            !<    where P is a permutation matrix and JAC is the final calculated jacobian.
            !<    Column J of P is column IPVT(J) of the identity matrix.  The lower defines a permutation matrix P such that
            !<    JAC*P = Q*R, where JAC is the final calculated jacobian, Q is
            !<    orthogonal (not stored), and R is upper triangular with diagonal
            !<    elements of nonincreasing magnitude.  Column J of P is column
            !<    IPVT(J) of the identity matrix.
            !<
            !<    Output, real ( kind = 8 ) QTF(N), contains the first N elements of Q'*FVEC.
            !<    trapezoidal part of FJAC contains information generated during
            !<    the computation of R.
            !<
            !<    Input, integer :: LDFJAC, the leading dimension of the array FJAC.
            !<    LDFJAC must be at least M.
            !<
            !<    Input, real ( kind = 8 ) FTOL.  Termination occurs when both the actual
            !<    and predicted relative reductions in the sum of squares are at most FTOL.
            !<    Therefore, FTOL measures the relative error desired in the sum of
            !<    squares.  FTOL should be nonnegative.
            !<
            !<    Input, real ( kind = 8 ) XTOL.  Termination occurs when the relative error
            !<    between two consecutive iterates is at most XTOL.  XTOL should be
            !<    nonnegative.
            !<
            !<    Input, real ( kind = 8 ) GTOL.  Termination occurs when the cosine of the
            !<    angle between FVEC and any column of the jacobian is at most GTOL in
            !<    absolute value.  Therefore, GTOL measures the orthogonality desired
            !<    between the function vector and the columns of the jacobian.  GTOL should
            !<    be nonnegative.
            !<
            !<    Input, integer :: MAXFEV.  Termination occurs when the number of calls
            !<    to FCN with IFLAG = 1 is at least MAXFEV by the end of an iteration.
            !<
            !<    Input/output, real ( kind = 8 ) DIAG(N).  If MODE = 1, then DIAG is set
            !<    internally.  If MODE = 2, then DIAG must contain positive entries that
            !<    serve as multiplicative scale factors for the variables.
            !<
            !<    Input, integer :: MODE, scaling option.
            !<    1, variables will be scaled internally.
            !<    2, scaling is specified by the input DIAG vector.
            !<
            !<    Input, real ( kind = 8 ) FACTOR, determines the initial step bound.  This
            !<    bound is set to the product of FACTOR and the euclidean norm of DIAG*X if
            !<    nonzero, or else to FACTOR itself.  In most cases, FACTOR should lie
            !<    in the interval (0.1, 100) with 100 the recommended value.
            !<
            !<    Input, integer :: NPRINT, enables controlled printing of iterates if it
            !<    is positive.  In this case, FCN is called with IFLAG = 0 at the
            !<    beginning of the first iteration and every NPRINT iterations thereafter
            !<    and immediately prior to return, with X and FVEC available
            !<    for printing.  If NPRINT is not positive, no special calls
            !<    of FCN with IFLAG = 0 are made.
            !<
            !<    Output, integer :: INFO, error flag.  If the user has terminated
            !<    execution, INFO is set to the (negative) value of IFLAG. See the description
            !<    of FCN.  Otherwise, INFO is set as follows:
            !<    0, improper input parameters.
            !<    1, both actual and predicted relative reductions in the sum of
            !<       squares are at most FTOL.
            !<    2, relative error between two consecutive iterates is at most XTOL.
            !<    3, conditions for INFO = 1 and INFO = 2 both hold.
            !<    4, the cosine of the angle between FVEC and any column of the jacobian
            !<       is at most GTOL in absolute value.
            !<    5, number of calls to FCN with IFLAG = 1 has reached MAXFEV.
            !<    6, FTOL is too small.  No further reduction in the sum of squares
            !<       is possible.
            !<    7, XTOL is too small.  No further improvement in the approximate
            !<       solution X is possible.
            !<    8, GTOL is too small.  FVEC is orthogonal to the columns of the
            !<       jacobian to machine precision.
            !<
            !<    Output, integer :: NFEV, the number of calls to FCN with IFLAG = 1.
            !<
            !<    Output, integer :: NJEV, the number of calls to FCN with IFLAG = 2.
            !<
            !<    Output, integer :: IPVT(N), defines a permutation matrix P such that
            !<    JAC*P = Q*R, where JAC is the final calculated jacobian, Q is
            !<    orthogonal (not stored), and R is upper triangular with diagonal
            !<    elements of nonincreasing magnitude.  Column J of P is column
            !<    IPVT(J) of the identity matrix.
            !<
            !<    Output, real ( kind = 8 ) QTF(N), contains the first N elements of Q'*FVEC.
            !<
            !<--------------------------------------------------------------------------------------------------------------------------------------------

            implicit none
            integer :: ldfjac                                                               !< INPUT:   the leading dimension of the array FJAC.
                                                                                            !<          LDFJAC must be at least M.
            integer :: m                                                                    !< INPUT:   is the number of functions.
            integer :: n                                                                    !< INPUT:   is the number of variables. N must not exceed M.
            integer :: info                                                                 !< OUTPUT:  error flag
            integer :: iflag                                                                !<
            integer :: iter                                                                 !<
            integer :: i, j                                                                 !< loop variables
            integer :: l                                                                    !<
            integer :: k, NumInputFile_index, i_index, j_index, NumInputFiles               !< working variables
            integer :: maxfev                                                               !< INPUT:   Termination occurs when the number of calls to
                                                                                            !<          FCN with IFLAG = 1 is at least MAXFEV by the
                                                                                            !<          end of an iteration.
            integer :: mode                                                                 !< INPUT:   MODE, scaling option.
                                                                                            !<          1, variables will be scaled internally.
                                                                                            !<          2, scaling is specified by the input DIAG vector.
            integer :: nfev                                                                 !< OUTPUT:  the number of calls to FCN with IFLAG = 1.
            integer :: njev                                                                 !< OUTPUT:  the number of calls to FCN with IFLAG = 2.
            integer :: nprint                                                               !< INPUT:   enables controlled printing of iterates if it
                                                                                            !<          is positive
            integer, dimension(n) :: ipvt                                                   !< OUTPUT:  defines a permutation matrix P such that
                                                                                            !<          JAC*P = Q*R, where JAC is the final calculated
                                                                                            !<          jacobian, Q is orthogonal (not stored), and R
                                                                                            !<          is upper triangular with diagonal elements of
                                                                                            !<          nonincreasing magnitude. Column J of P is column
                                                                                            !<          IPVT(J) of the identity matrix.
            real*8 :: factor                                                                !< INPUT:   determines the initial step bound. This bound
                                                                                            !<          is set to the product of FACTOR and the
                                                                                            !<          euclidean norm of DIAG*X if nonzero, or else
                                                                                            !<          to FACTOR itself. In most cases, FACTOR should
                                                                                            !<          lie in the interval (0.1, 100) with 100 the
                                                                                            !<          recommended value.
            real*8 :: ftol                                                                  !< INPUT:   Termination occurs when both the actual and
                                                                                            !<          predicted relative reductions in the sum of
                                                                                            !<          squares are at most FTOL. Therefore, FTOL
                                                                                            !<          measures the relative error desired in the sum of
                                                                                            !<          squares. FTOL should be nonnegative.
            real*8 :: xtol                                                                  !< INPUT:   Termination occurs when the relative error between
                                                                                            !<          two consecutive iterates is at most XTOL. XTOL
                                                                                            !<          should be nonnegative.
            real*8 :: gtol                                                                  !< INPUT:   Termination occurs when the cosine of the angle
                                                                                            !<          between FVEC and any column of the jacobian is
                                                                                            !<          at most GTOL in absolute value. Therefore, GTOL
                                                                                            !<          measures the orthogonality desired between the
                                                                                            !<          function vector and the columns of the jacobian.
                                                                                            !<          GTOL should be nonnegative.
            real*8 :: fnorm                                                                 !<
            real*8 :: fnorm1                                                                !<
            real*8 :: gnorm                                                                 !<
            real*8 :: actred                                                                !<
            real*8 :: delta                                                                 !<
            real*8 :: dirder                                                                !<
            real*8 :: epsmch                                                                !<
            real*8 :: par                                                                   !<
            real*8 :: pnorm                                                                 !<
            real*8 :: prered                                                                !<
            real*8 :: ratio                                                                 !<
            real*8 :: sum2                                                                  !<
            real*8 :: temp                                                                  !<
            real*8 :: temp1                                                                 !<
            real*8 :: temp2                                                                 !<
            real*8 :: xnorm                                                                 !<
            real*8 :: chisq                                                                 !< current value of chi^2
            real*8, dimension(n) :: x                                                       !< INPUT/OUTPUT: On input, X must contain an initial estimate
                                                                                            !<          of the solution vector. On output X contains the
                                                                                            !<          final estimate of the solution vector.
            real*8, dimension(n) :: qtf                                                     !< OUTPUT:  contains the first N elements of Q'*FVEC.
            real*8, dimension(n) :: diag                                                    !< INPUT/OUTPUT:  If MODE = 1, then DIAG is set internally.
                                                                                            !<          If MODE = 2, then DIAG must contain positive
                                                                                            !<          entries that serve as multiplicative scale
                                                                                            !<          factors for the variables.
            real*8, dimension(n) :: wa1                                                     !<
            real*8, dimension(n) :: wa2                                                     !<
            real*8, dimension(n) :: wa3                                                     !<
            real*8, dimension(m) :: wa4                                                     !<
            real*8, dimension(m) :: fvec                                                    !< OUTPUT:  the functions evaluated at the output X
            real*8, dimension(ldfjac,n) :: fjac                                             !< OUTPUT:  n M by N array. The upper N by N submatrix
                                                                                            !<          of FJAC contains an upper triangular matrix R
                                                                                            !<          with diagonal elements of nonincreasing magnitude
                                                                                            !<          such that
                                                                                            !<            P' * ( JAC' * JAC ) * P = R' * R,
                                                                                            !<          where P is a permutation matrix and JAC is the
                                                                                            !<          final calculated jacobian. Column J of P is column
                                                                                            !<          IPVT(J) of the identity matrix. The lower
                                                                                            !<          trapezoidal part of FJAC contains information
                                                                                            !<          generated during the computation of R.
            character(len=25) :: LongNumber1, LongNumber2                                   !< working variables
            character(len=100) :: HelpString                                                !< working variable
            character(len=8192) :: ListParamFormated                                        !< needed for formated printout of parameters
            logical :: IntegerTrue                                                          !< is parameter an integer ?
            logical :: InitPlotFlag                                                         !< flag for saving plot to file


            epsmch = epsilon(epsmch)


            !< reset output variables
            info = 0
            ipvt = 0
            iflag = 0
            nfev = 0
            njev = 0
            qtf = 0.d0
            fvec = 0.d0
            fjac = 0.d0


            !<  Check the input parameters for errors.
            if (n < 0) then
                print '(" ")'
                print '(" ")'
                print '(13x,"Error in subroutine lmder:")'
                print '(15x, "Number of free parameters n < 0!")'
                print '(" ")'
                print '(15x, "n = ", I9)', n
                print '(" ")'
                print '(" ")'
                print '(13x, "Stop algorithm!")'
                go to 10300
            endif
            if (m < n) then
                print '(" ")'
                print '(" ")'
                print '(13x,"Error in subroutine lmder:")'
                print '(15x, "Number of data points m is smaller than number of free parameters n!")'
                print '(" ")'
                print '(15x, "m, n = ", I6, ", ", I6)', m, n
                print '(" ")'
                print '(" ")'
                print '(13x, "Stop algorithm!")'
                go to 10300
            endif
            if (ldfjac < m .or. ftol < 0.d0 .or. xtol < 0.d0 .or. gtol < 0.d0 .or. maxfev <= 0 .or. factor <= 0.d0) then
                print '(" ")'
                print '(" ")'
                print '(13x,"Error in subroutine lmder:")'
                print '(15x, "One of the following problems occured:")'
                print '(" ")'
                print '(15x, "ldfjac < m: ldfjac, m = ", I6, ", ", ES25.15)', ldfjac, m
                print '(15x, "ftol < 0:   ftol = ", ES25.15)', ftol
                print '(15x, "xtol < 0:   xtol = ", ES25.15)', xtol
                print '(15x, "gtol < 0:   gtol = ", ES25.15)', gtol
                print '(15x, "maxfev < 0: maxfev = ", I6)', maxfev
                print '(15x, "factor < 0: factor = ", ES25.15)', factor
                print '(" ")'
                print '(" ")'
                print '(13x, "Stop algorithm!")'
                go to 10300
            endif
            if (mode == 2) then
                Do j = 1, n
                    if (diag(j) <= 0.d0) then
                        print '(" ")'
                        print '(" ")'
                        print '(13x,"Error in subroutine lmder:")'
                        print '(15x, "The following problem occured:")'
                        print '(" ")'
                        print '(15x, "diag(j) < 0: j, diag(j) = ", I6, ", ", ES25.15)', j, diag(j)
                        print '(" ")'
                        print '(" ")'
                        print '(13x, "Stop algorithm!")'
                        go to 10300
                    endif
                end Do
            endif


            !< Evaluate the function at the starting point and calculate its norm.
            iflag = 1
            call fcn(chisq, n, x, fvec, fjac, ldfjac, iflag, parameternumber, nfit, MaxColX, NumberExpFiles, MaxExpLines, MaxColY)
            if (n == 0) then
                print '(A, "                                                                             ")', char(13)
                print '(" ")'
                print '(15x, "No free parameter to optimize!")'
                print '(" ")'
                go to 10300
                return
            endif
            nfev = 1
            if (iflag < 0) then
                print '(" ")'
                print '(" ")'
                print '(13x,"Error in subroutine lmder:")'
                print '(15x, "The following problem occured:")'
                print '(" ")'
                print '(15x, "iflag < 0: iflag = ", I6)', iflag
                print '(" ")'
                print '(" ")'
                print '(13x, "Stop algorithm!")'
                go to 10300
            endif
            call enorm(fnorm, m, fvec)



            !< Initialize Levenberg-Marquardt parameter and iteration counter.
            par = 0.d0
            iter = 1


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< Beginning of the outer loop.
      1030  continue


            !< Calculate the jacobian matrix.
            iflag = 2
            call fcn(chisq, n, x, fvec, fjac, ldfjac, iflag, parameternumber, nfit, MaxColX, NumberExpFiles, MaxExpLines, MaxColY)
            njev = njev + 1
            if (iflag < 0) then
                print '(" ")'
                print '(" ")'
                print '(13x,"Error in subroutine lmder:")'
                print '(15x, "The following problem occured:")'
                print '(" ")'
                print '(15x, "Within outer loop: iflag < 0: iflag = ", I9)', iflag
                print '(" ")'
                print '(" ")'
                print '(13x, "Stop algorithm!")'
                go to 10300
            endif

            ! Debug:
            ! write(HelpString,'(I5)') iter
            ! open(1234,file='/home/moeller/CATS/magix/fjac_' // trim(adjustl(HelpString)) // '.dat')
            ! Do j = 1, m
            !     write(1234,*) fjac(j,:)
            ! end Do
            ! close(1234)


            !< Compute the QR factorization of the jacobian.
            call qrfac(m, n, fjac, ldfjac, .true., ipvt, n, wa1, wa2)

            ! Debug:
            ! print*,'--> wa1 = ',wa1
            ! print*,'--> wa2 = ',wa2


            !< On the first iteration and if mode is 1, scale according to the norms of the columns of the initial jacobian.
            if (iter == 1) then
                if (mode /= 2) then
                    diag(1:n) = wa2(1:n)
                    Do j = 1, n
                        if (wa2(j) == 0.d0) then
                            diag(j) = 1.d0
                        endif
                    end Do
                endif


                !< On the first iteration, calculate the norm of the scaled X and initialize the step bound DELTA.
                wa3(1:n) = diag(1:n) * x(1:n)
                call enorm(xnorm, n, wa3)
                delta = factor * xnorm
                if (delta == 0.d0) then
                    delta = factor
                endif
            endif


            !< Form Q'*FVEC and store the first N components in QTF.
            wa4(1:m) = fvec(1:m)
            Do j = 1, n
                if (fjac(j,j) /= 0.d0) then
                    sum2 = dot_product(wa4(j:m), fjac(j:m,j))
                    temp = -sum2 / fjac(j,j)
                    wa4(j:m) = wa4(j:m) + fjac(j:m,j) * temp
                endif
                fjac(j,j) = wa1(j)
                qtf(j) = wa4(j)
            end Do


            !< Compute the norm of the scaled gradient.
            gnorm = 0.d0
            if (fnorm /= 0.d0) then
                Do j = 1, n
                    l = ipvt(j)

                    ! Debug:
                    ! print*,'n, j, l, wa2(l) = ', n, j, l, wa2(l)


                    if (wa2(l) /= 0.d0) then
                        sum2 = dot_product(qtf(1:j), fjac(1:j,j)) / fnorm
                        gnorm = dmax1(gnorm, dabs(sum2 / wa2(l)))

                        ! Debug:
                        ! print*,'n, j, l, wa2(l) = ', n, j, l, wa2(l)
                        ! print*,'sum2 = ', sum2
                        ! print*,'gnorm = ', gnorm
                    endif
                end Do
            endif


            !< Test for convergence of the gradient norm.
            !if (chisq <= gtol) then
            !    info = 4
            !    print '(" ")'
            !    print '(" ")'
            !    print '(13x,"Error in subroutine lmder:")'
            !    print '(15x, "Norm lower than tolerance:")'
            !    print '(" ")'
            !    print '(15x, "gnorm <= gtol: gnorm, gtol = ", ES25.15, ", ", ES25.15)', gnorm, gtol
            !    print '(" ")'
            !    print '(" ")'
            !    print '(13x, "Stop algorithm!")'
            !    go to 10300
            !endif


            !< Rescale if necessary.
            if (mode /= 2) then
                Do j = 1, n
                    diag(j) = dmax1(diag(j), wa2(j))
                end Do
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< Beginning of the inner loop.
      10200 continue

            ! Debug:
            ! print*,'>>>wa1 = ',wa1
            ! print*,'>>>wa2 = ',wa2


            !< Determine the Levenberg-Marquardt parameter.
            call lmpar(n, fjac, ldfjac, ipvt, diag, qtf, delta, par, wa1, wa2)

            ! Debug:
            ! print*,'diag = ',diag
            ! print*,'wa1 = ',wa1
            ! print*,'wa2 = ',wa2
            ! print*,'fjac = ',fjac


            !< Store the direction p and x + p. calculate the norm of p.
            wa1(1:n) = -wa1(1:n)


            !< modify parameter vector and check if parameters are within limits
            k = 0
            Do i = 1, parameternumber                                                       !< Did the trial succeed?
                if (ia(i)) then                                                             !< modify only the parameter which are optimized
                    k = k + 1
                    wa2(k) = x(k) + wa1(k)
                    if (wa2(k) < paramset(3, i) .or. wa2(k) > paramset(4, i)) then          !< check if parameters are within limits
                        wa1(k) = 0.d0
                        !if (wa2(k) < paramset(3,i)) then
                        !    wa1(k) = (x(k) - paramset(3,i)) !* RandomWithLimits(0.d0, 1.d0)
                        !else
                        !    wa1(k) = (x(k) - paramset(4,i)) !* RandomWithLimits(0.d0, 1.d0)
                        !endif
                        wa2(k) = x(k) + wa1(k)
                    endif
                endif
            end Do


            !< modify wa3 array
            wa3(1:n) = diag(1:n) * wa1(1:n)
            call enorm(pnorm, n, wa3)


            !< On the first iteration, adjust the initial step bound.
            if (iter == 1) then
                delta = dmin1(delta, pnorm)
            endif


            !< Evaluate the function at x + p and calculate its norm.
            iflag = 1
            call fcn(chisq, n, wa2, wa4, fjac, ldfjac, iflag, parameternumber, nfit, MaxColX, NumberExpFiles, MaxExpLines, MaxColY)
            nfev = nfev + 1
            if (iflag < 0) then
                print '(" ")'
                print '(" ")'
                print '(13x,"Error in subroutine lmder:")'
                print '(15x,"The following problem occured:")'
                print '(" ")'
                print '(15x, "last call: iflag < 0: iflag = ", I9)', iflag
                print '(" ")'
                print '(" ")'
                print '(13x, "Stop algorithm!")'
                go to 10300
            endif
            call enorm(fnorm1, m, wa4)


            !< Compute the scaled actual reduction.
            actred = -1.d0
            if (0.1D+00 * fnorm1 < fnorm) then
                actred = 1.d0 - (fnorm1 / fnorm)**2
            endif

            ! Debug:
            ! print*,'actred = ',actred
            ! print*,'fnorm1 = ',fnorm1
            ! print*,'fnorm  = ',fnorm


            !< Compute the scaled predicted reduction and the scaled directional derivative.
            Do j = 1, n
                wa3(j) = 0.d0
                l = ipvt(j)
                temp = wa1(l)
                wa3(1:j) = wa3(1:j) + fjac(1:j,j) * temp
            end Do
            call enorm(temp1, n, wa3)
            temp1 = temp1 / fnorm
            temp2 = (dsqrt(par) * pnorm) / fnorm
            prered = temp1**2 + temp2**2 / 0.5D+00
            dirder = -(temp1**2 + temp2**2)


            !< Compute the ratio of the actual to the predicted reduction.
            if (prered /= 0.d0) then
                ratio = actred / prered
            else
                ratio = 0.d0
            endif

            ! Debug:
            ! print*,'ratio  = ',ratio
            ! print*,'actred = ',actred
            ! print*,'prered = ',prered


            !< Update the step bound.
            if (ratio <= 0.25D+00) then
                if (actred >= 0.d0) then
                    temp = 0.5D+00
                endif
                if (actred < 0.d0) then
                    temp = 0.5D+00 * dirder / (dirder + 0.5D+00 * actred)
                endif
                if (0.1D+00 * fnorm1 >= fnorm .or. temp < 0.1D+00) then
                    temp = 0.1D+00
                endif
                delta = temp * dmin1(delta, pnorm / 0.1D+00)
                par = par / temp

            else

                if (par == 0.d0 .or. ratio >= 0.75D+00) then
                    delta = 2.d0 * pnorm
                    par = 0.5D+00 * par
                endif
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< Successful iteration:  Update X, FVEC, and their norms.
            if (ratio >= 0.0001D+00) then
                x(1:n) = wa2(1:n)
                wa2(1:n) = diag(1:n) * x(1:n)
                fvec(1:m) = wa4(1:m)
                call enorm(xnorm, n, wa2)
                fnorm = fnorm1
            endif


            !< build list with fit parameters
            k = 0
            ListParamFormated = ""
            Do j = 1, parameternumber
                if (ia(j)) then
                    k = k + 1
                    HelpString = ""
                    call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, j)
                    if (IntegerTrue) then
                        write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(BestSitesParamSet(1, k + 1))
                        if (index(HelpString, "*") > 0) then                                !< search for bad real number
                            write(HelpString, *) int(BestSitesParamSet(1, k + 1))
                        endif
                    else
                        write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) BestSitesParamSet(1, k + 1)
                        if (index(HelpString, "*") > 0) then                                !< search for bad real number
                            write(HelpString, *) BestSitesParamSet(1, k + 1)
                        endif
                    endif
                    if (k == 1) then
                        if (len_trim(adjustl(ListParamFormated)) + len_trim(adjustl(HelpString)) > 8192) then
                            exit
                        else
                            ListParamFormated = trim(adjustl(ListParamFormated)) // trim(adjustl(HelpString))
                        endif
                    else
                        if ((len_trim(adjustl(ListParamFormated)) + len_trim(adjustl(HelpString)) + 3) > 8192) then
                            exit
                        else
                            ListParamFormated = trim(adjustl(ListParamFormated)) // ',  ' // trim(adjustl(HelpString))
                        endif
                    endif
                endif
            end Do


            !< print status of iteration process ..
            if (printflag) then
                print '(11x, I10, ES26.15, 5x, A)', iter, BestSitesParamSet(1, 1), trim(adjustl(ListParamFormated))
            endif
            write(paramchannel,'("  ")')
            write(paramchannel,'("  ")')
            write(paramchannel,'(123("*"))')
            write(paramchannel,'("Iteration: ",I5,",  chi^2 = ",ES25.15)') iter, BestSitesParamSet(1, 1)
            write(logchannel,'(11x, I10, ES26.15, 5x, A)') iter, BestSitesParamSet(1, 1), trim(adjustl(ListParamFormated))


            !< write actual parameters to files
            write(paramchannel,'("  ")')
            write(paramchannel,'("  ")')
            write(paramchannel,'("Parameters: ",A)') trim(adjustl(ListParamFormated))
            write(paramchannel,'(123("-"))')
            write(paramchannel,'("  ")')


            !< save current experimental x point of the first experimental file to variable posdatexp
            posdatexp(1:MaxColX) = 0.d0


            !< call subroutine to write current values of the parameter to file
            write(paramchannel,'("-",61(" -"))')
            Do NumInputFiles = 1, NumberInputFiles
                write(paramchannel,'("Input-File ",I5,":  , file: ",A)') NumInputFiles, trim(adjustl(FitFktInput(NumInputFiles)))
                write(paramchannel,'("  ")')
                write(paramchannel,'("-start_input-file",106("-"))')
                k = 0
                Do i = 1, parameternumber
                    if (ia(i)) then
                        k = k + 1
                        currentparmval(i) = BestSitesParamSet(1, k + 1)
                    endif
                end Do
                call WriteParameter(paramchannel, .true., MaxColX, posdatexp, parameternumber, currentparmval, NumInputFiles)
                write(paramchannel,'("-end_input-file",108("-"))')
            end Do


            !< plot experimental data, model function, and chi**2
            if (PlotIterationGlobal == 0) then
                if (iter == 1) then
                    InitPlotFlag = .true.
                else
                    InitPlotFlag = .false.
                endif
                call PlotFitFunction(InitPlotFlag, xAxisLabelGlobal, yAxisLabelGlobal, zAxisLabelGlobal)
            endif


            !< print seperator
            write(paramchannel,'("  ")')
            write(paramchannel,'(123("="))')
            write(paramchannel,'("  ")')


            !< increase iteration counter
            iter = iter + 1


            !< test, if max. number of iteration is reached
            if (iter > NumIterGlobal) then
                if (printflag) then
                    print '(" ")'
                    print '(11x,"Iteration stopped. Number of iterations is equal to max. number of iterations = ",I6)',NumIterGlobal
                endif
                write(logchannel,'("  ")')
                write(logchannel,'(11x,"Iteration stopped. Number of iterations is equal to max. number of iterations = ",I6)') NumIterGlobal
                go to 10300
            endif


            !< test if chi^2 is lower than limit of chi^2
            if (BestSitesParamSet(1, 1) < ftol) then
                write(LongNumber1,'(ES25.15)') BestSitesParamSet(1, 1)
                write(LongNumber2,'(ES25.15)') ftol
                if (printflag) then
                    print '(" ")'
                    print '(11x,"Iteration stopped. chi^2 (=",A,") dropped below limit = ",A)', trim(adjustl(LongNumber1)), trim(adjustl(LongNumber2))
                endif
                write(logchannel,'("  ")')
                write(logchannel,'(11x,"Iteration stopped. chi^2 (=",A,") dropped below limit = ",A)') trim(adjustl(LongNumber1)), &
                                                                                                            trim(adjustl(LongNumber2))
                go to 10300
            endif


            !< Tests for convergence.
            !if (dabs(actred) <= ftol .and. prered <= ftol .and. 0.5D+00 * ratio <= 1.d0) then
            !    info = 1

            !    ! Debug:
            !    print*,'info   = ',info
            !    print*,'actred = ',actred
            !    print*,'ftol   = ',ftol
            !    print*,'prered = ',prered
            !    print*,'ratio  = ',ratio
            !endif
            !if (delta <= xtol * xnorm) then
            !    info = 2

            !    ! Debug:
            !    print*,'info  = ',info
            !    print*,'delta = ',delta
            !    print*,'xtol  = ',xtol
            !    print*,'xnorm = ',xnorm
            !endif
            !if (dabs(actred) <= ftol .and. prered <= ftol .and. 0.5D+00 * ratio <= 1.d0 .and. info == 2) then
            !    info = 3

            !    ! Debug:
            !    print*,'actred = ',actred
            !    print*,'ftol   = ',ftol
            !    print*,'prered = ',prered
            !    print*,'ratio  = ',ratio
            !endif
            !if (info /= 0) go to 10300


            !< Tests for termination and stringent tolerances.
            !if (nfev > maxfev) then                                                         !< ------------------ changed
            !    if (printflag) then
            !        print '(" ")'
            !        print '(11x,"Iteration stopped. Number of iterations is equal to max. number of iterations = ",I6)',NumIterGlobal
            !    endif
            !    write(logchannel,'("  ")')
            !    write(logchannel,'(11x,"Iteration stopped. Number of iterations is equal to max. number of iterations = ",I6)') NumIterGlobal
            !    info = 5
            !endif
            !if (dabs(actred) <= epsmch .and. prered <= epsmch .and. 0.5D+00 * ratio <= 1.d0) then
            !    info = 6
            !endif
            !if (delta <= epsmch * xnorm) then
            !    info = 7
            !endif
            !if (gnorm <= epsmch) then
            !    info = 8
            !endif
            !if (info /= 0) then
            !    go to 10300
            !endif


            !< End of the inner loop. repeat if iteration unsuccessful.
            if (ratio < 0.0001D+00) then
                go to 10200
            endif


            !< End of the outer loop.
            go to 1030
      10300 continue


            !< Termination, either normal or user imposed.
            if (iflag < 0) then
                info = iflag
            endif

            iflag = 0
            !if (0 < nprint) then
            !    call fcn(chisq, n, x, fvec, fjac, ldfjac, iflag, parameternumber, nfit, MaxColX, NumberExpFiles, MaxExpLines, MaxColY)
            !endif
            return
        end subroutine lmder


        !>************************************************************************************************************************************************
        !> subroutine: lmder1
        !>
        !> (MINPACK): LMDER1 minimizes M functions in N variables by the Levenberg-Marquardt method.
        !>
        !> input variables:     m:                      the number of functions
        !>                      n:                      is the number of variables.  N must not exceed M.
        !>                      x:                      On input, X must contain an initial estimate of the solution vector. On output X contains the
        !>                                              final estimate of the solution vector.
        !>                      ldfjac:                 is the leading dimension of FJAC, which must be no less than M.
        !>                      tol:                    Termination occurs when the algorithm estimates either that the relative error in the sum of
        !>                                              squares is at most TOL or that the relative error between X and the solution is at most TOL.
        !>
        !> output variables:    fvec:                   the functions evaluated at the output X.
        !>                      x:                      On input, X must contain an initial estimate of the solution vector. On output X contains the
        !>                                              final estimate of the solution vector.
        !>                      fjac:                   an M by N array.  The upper N by N submatrix contains an upper triangular matrix R with
        !>                                              diagonal elements of nonincreasing magnitude such that
        !>                                              P' * ( JAC' * JAC ) * P = R' * R,
        !>                                              where P is a permutation matrix and JAC is the final calculated jacobian.  Column J of P is column
        !>                                              IPVT(J) of the identity matrix. The lower trapezoidal part of FJAC contains information generated
        !>                                              during the computation of R.
        !>                      info:                   error flag
        !>
        !>
        !> \author MINPACK and Thomas Moeller
        !>
        !> \date 06.04.2010
        !>
        subroutine lmder1(m, n, x, fvec, fjac, ldfjac, tol, info)
            !<
            !<  Discussion:
            !<
            !<    LMDER1 minimizes the sum of the squares of M nonlinear functions in
            !<    N variables by a modification of the Levenberg-Marquardt algorithm.
            !<    This is done by using the more general least-squares solver LMDER.
            !<    The user must provide a subroutine which calculates the functions
            !<    and the jacobian.
            !<
            !<  Licensing:
            !<
            !<    This code is distributed under the GNU LGPL license.
            !<
            !<  Modified:
            !<
            !<    06 April 2010
            !<
            !<  Author:
            !<
            !<    Original FORTRAN77 version by Jorge More, Burton Garbow, Kenneth Hillstrom.
            !<    FORTRAN90 version by John Burkardt.
            !<
            !<  Reference:
            !<
            !<    Jorge More, Burton Garbow, Kenneth Hillstrom,
            !<    User Guide for MINPACK-1,
            !<    Technical Report ANL-80-74,
            !<    Argonne National Laboratory, 1980.
            !<
            !<  Parameters:
            !<
            !<    Input, external FCN, the name of the user-supplied subroutine which
            !<    calculates the functions and the jacobian.  FCN should have the form:
            !<
            !<      subroutine fcn ( m, n, x, fvec, fjac, ldfjac, iflag )
            !<      integer :: ldfjac
            !<      integer :: n
            !<      real fjac(ldfjac,n)
            !<      real fvec(m)
            !<      integer :: iflag
            !<      real x(n)
            !<
            !<    If IFLAG = 1 on intput, FCN should calculate the functions at X and
            !<    return this vector in FVEC.
            !<    If IFLAG = 2 on input, FCN should calculate the jacobian at X and
            !<    return this matrix in FJAC.
            !<    To terminate the algorithm, the user may set IFLAG negative.
            !<
            !<    Input, integer :: M, the number of functions.
            !<
            !<    Input, integer :: N, is the number of variables.  N must not exceed M.
            !<
            !<    Input/output, real ( kind = 8 ) X(N).  On input, X must contain an initial
            !<    estimate of the solution vector.  On output X contains the final
            !<    estimate of the solution vector.
            !<
            !<    Output, real ( kind = 8 ) FVEC(M), the functions evaluated at the output X.
            !<
            !<    Output, real ( kind = 8 ) FJAC(LDFJAC,N), an M by N array.  The upper
            !<    N by N submatrix contains an upper triangular matrix R with
            !<    diagonal elements of nonincreasing magnitude such that
            !<      P' * ( JAC' * JAC ) * P = R' * R,
            !<    where P is a permutation matrix and JAC is the final calculated
            !<    jacobian.  Column J of P is column IPVT(J) of the identity matrix.
            !<    The lower trapezoidal part of FJAC contains information generated during
            !<    the computation of R.
            !<
            !<    Input, integer :: LDFJAC, is the leading dimension of FJAC,
            !<    which must be no less than M.
            !<
            !<    Input, real ( kind = 8 ) TOL.  Termination occurs when the algorithm
            !<    estimates either that the relative error in the sum of squares is at
            !<    most TOL or that the relative error between X and the solution is at
            !<    most TOL.
            !<
            !<    Output, integer :: INFO, error flag.  If the user has terminated
            !<    execution, INFO is set to the (negative) value of IFLAG. See the description
            !<    of FCN.  Otherwise, INFO is set as follows:
            !<    0, improper input parameters.
            !<    1, algorithm estimates that the relative error in the sum of squares
            !<       is at most TOL.
            !<    2, algorithm estimates that the relative error between X and the
            !<       solution is at most TOL.
            !<    3, conditions for INFO = 1 and INFO = 2 both hold.
            !<    4, FVEC is orthogonal to the columns of the jacobian to machine precision.
            !<    5, number of calls to FCN with IFLAG = 1 has reached 100*(N+1).
            !<    6, TOL is too small.  No further reduction in the sum of squares is
            !<       possible.
            !<    7, TOL is too small.  No further improvement in the approximate
            !<       solution X is possible.
            !<
            !<--------------------------------------------------------------------------------------------------------------------------------------------


            implicit none
            integer :: ldfjac                                                               !< INPUT:   is the leading dimension of FJAC, which must be
                                                                                            !<          no less than M.
            integer :: m                                                                    !< INPUT:   M, the number of functions.
            integer :: n                                                                    !< INPUT:   N, is the number of variables. N must not exceed M
            integer :: info                                                                 !< OUTPUT:  INFO, error flag.
            integer :: maxfev                                                               !<
            integer :: mode                                                                 !<
            integer :: nfev                                                                 !<
            integer :: njev                                                                 !<
            integer :: nprint                                                               !<
            integer, dimension(n) :: ipvt                                                   !<
            real*8 :: factor                                                                !<
            real*8 :: ftol                                                                  !<
            real*8 :: gtol                                                                  !<
            real*8 :: tol                                                                   !< INPUT:   Termination occurs when the algorithm estimates
                                                                                            !<          either that the relative error in the sum of
                                                                                            !<          squares is at most TOL or that the relative error
                                                                                            !<          between X and the solution is at most TOL.
            real*8 :: xtol                                                                  !<
            real*8, dimension(n) :: x                                                       !< INPUT/OUTPUT: On input, X must contain an initial estimate
                                                                                            !<          of the solution vector. On output X contains the
                                                                                            !<          final estimate of the solution vector.
            real*8, dimension(n) :: qtf                                                     !<
            real*8, dimension(n) :: diag                                                    !<
            real*8, dimension(m) :: fvec                                                    !< OUTPUT:  FVEC(M), the functions evaluated at the output X.
            real*8, dimension(ldfjac,n) :: fjac                                             !< OUTPUT:  FJAC(LDFJAC,N), an M by N array.  The upper
                                                                                            !<          N by N submatrix contains an upper triangular
                                                                                            !<          matrix R with diagonal elements of nonincreasing
                                                                                            !<          magnitude such that
                                                                                            !<            P' * ( JAC' * JAC ) * P = R' * R,
                                                                                            !<          where P is a permutation matrix and JAC is the
                                                                                            !<          final calculated  jacobian. Column J of P is
                                                                                            !<          column IPVT(J) of the identity matrix. The lower
                                                                                            !<          trapezoidal part of FJAC contains information
                                                                                            !<          generated during the computation of R.


            !< reset output variables
            info = 0
            fvec = 0.d0
            fjac = 0.d0


            !< check input parameters
            if (m < n) then
                return
            elseif (ldfjac < m) then
                return
            elseif (tol < 0.d0) then
                return
            endif


            !< set algorithm parameter
            maxfev = 100 * (n + 1)
            factor = 100.d0
            ftol = tol
            xtol = tol
            gtol = tol
            mode = 1
            nprint = 0
            call lmder(m, n, x, fvec, fjac, ldfjac, ftol, xtol, gtol, maxfev, diag, mode, factor, nprint, info, nfev, njev, ipvt, qtf)

            ! print*,'==============>',info


            if (info == 8) then
                info = 4
            endif

            return
        end subroutine lmder1


        !>************************************************************************************************************************************************
        !> subroutine: lmpar
        !>
        !> (MINPACK): LMPAR computes a parameter for the Levenberg-Marquardt method.
        !>
        !> input variables:     n:                      N, the order of R
        !>                      r:                      R(LDR,N),the N by N matrix.
        !>                      ldr:                    the leading dimension of R. LDR must be no less than N.
        !>                      ipvt:                   IPVT(N), defines the permutation matrix P such that A*P = Q*R.  Column J of P is column IPVT(J)
        !>                                              of the identity matrix.
        !>                      diag:                   the diagonal elements of the matrix D.
        !>                      qtb:                    the first N elements of the vector Q'*B.
        !>                      delta:                  an upper bound on the euclidean norm of D*X. DELTA should be positive.
        !>                      par:                    On input an initial estimate of the Levenberg-Marquardt parameter. On output the final estimate.
        !>                                              PAR should be nonnegative.
        !>
        !> output variables:    par:                    On input an initial estimate of the Levenberg-Marquardt parameter. On output the final estimate.
        !>                                              PAR should be nonnegative.
        !>                      x:                      the least squares solution of the system A*X = B, sqrt(PAR)*D*X = 0, for the output value of PAR.
        !>                      sdiag:                  the diagonal elements of the upper triangular matrix S.
        !>
        !>
        !> \author MINPACK and Thomas Moeller
        !>
        !> \date 06.04.2010
        !>
        subroutine lmpar(n, r, ldr, ipvt, diag, qtb, delta, par, x, sdiag)
            !<
            !<  Discussion:
            !<
            !<    Given an M by N matrix A, an N by N nonsingular diagonal
            !<    matrix D, an M-vector B, and a positive number DELTA,
            !<    the problem is to determine a value for the parameter
            !<    PAR such that if X solves the system
            !<
            !<      A*X = B,
            !<      sqrt ( PAR ) * D * X = 0,
            !<
            !<    in the least squares sense, and DXNORM is the euclidean
            !<    norm of D*X, then either PAR is zero and
            !<
            !<      ( DXNORM - DELTA ) <= 0.1 * DELTA,
            !<
            !<    or PAR is positive and
            !<
            !<      dabs(DXNORM - DELTA) <= 0.1 * DELTA.
            !<
            !<    This subroutine completes the solution of the problem
            !<    if it is provided with the necessary information from the
            !<    QR factorization, with column pivoting, of A.  That is, if
            !<    A*P = Q*R, where P is a permutation matrix, Q has orthogonal
            !<    columns, and R is an upper triangular matrix with diagonal
            !<    elements of nonincreasing magnitude, then LMPAR expects
            !<    the full upper triangle of R, the permutation matrix P,
            !<    and the first N components of Q'*B.  On output
            !<    LMPAR also provides an upper triangular matrix S such that
            !<
            !<      P' * ( A' * A + PAR * D * D ) * P = S'* S.
            !<
            !<    S is employed within LMPAR and may be of separate interest.
            !<
            !<    Only a few iterations are generally needed for convergence
            !<    of the algorithm.  If, however, the limit of 10 iterations
            !<    is reached, then the output PAR will contain the best
            !<    value obtained so far.
            !<
            !<  Licensing:
            !<
            !<    This code is distributed under the GNU LGPL license.
            !<
            !<  Modified:
            !<
            !<    06 April 2010
            !<
            !<  Author:
            !<
            !<    Original FORTRAN77 version by Jorge More, Burton Garbow, Kenneth Hillstrom.
            !<    FORTRAN90 version by John Burkardt.
            !<
            !<  Reference:
            !<
            !<    Jorge More, Burton Garbow, Kenneth Hillstrom,
            !<    User Guide for MINPACK-1,
            !<    Technical Report ANL-80-74,
            !<    Argonne National Laboratory, 1980.
            !<
            !<  Parameters:
            !<
            !<    Input, integer :: N, the order of R.
            !<
            !<    Input/output, real ( kind = 8 ) R(LDR,N),the N by N matrix.  The full
            !<    upper triangle must contain the full upper triangle of the matrix R.
            !<    On output the full upper triangle is unaltered, and the strict lower
            !<    triangle contains the strict upper triangle (transposed) of the upper
            !<    triangular matrix S.
            !<
            !<    Input, integer :: LDR, the leading dimension of R.  LDR must be
            !<    no less than N.
            !<
            !<    Input, integer :: IPVT(N), defines the permutation matrix P such that
            !<    A*P = Q*R.  Column J of P is column IPVT(J) of the identity matrix.
            !<
            !<    Input, real ( kind = 8 ) DIAG(N), the diagonal elements of the matrix D.
            !<
            !<    Input, real ( kind = 8 ) QTB(N), the first N elements of the vector Q'*B.
            !<
            !<    Input, real ( kind = 8 ) DELTA, an upper bound on the euclidean norm of D*X.
            !<    DELTA should be positive.
            !<
            !<    Input/output, real ( kind = 8 ) PAR.  On input an initial estimate of the
            !<    Levenberg-Marquardt parameter.  On output the final estimate.
            !<    PAR should be nonnegative.
            !<
            !<    Output, real ( kind = 8 ) X(N), the least squares solution of the system
            !<    A*X = B, sqrt(PAR)*D*X = 0, for the output value of PAR.
            !<
            !<    Output, real ( kind = 8 ) SDIAG(N), the diagonal elements of the upper
            !<    triangular matrix S.
            !<
            !<--------------------------------------------------------------------------------------------------------------------------------------------


            implicit none
            integer :: ldr                                                                  !< INPUT:   the leading dimension of R.  LDR must be
                                                                                            !<          no less than N.
            integer :: n                                                                    !< INPUT:   N, the order of R.
            integer :: iter                                                                 !<
            integer :: i, j, k, l                                                           !< loop variables
            integer :: jm1                                                                  !<
            integer :: nsing                                                                !<
            integer, dimension(n) :: ipvt                                                   !< INPUT:   defines the permutation matrix P such that
                                                                                            !<          A*P = Q*R. Column J of P is column IPVT(J) of
                                                                                            !<          the identity matrix.
            real*8 :: delta                                                                 !< INPUT:   an upper bound on the euclidean norm of D*X.
                                                                                            !<          DELTA should be positive.
            real*8 :: dwarf                                                                 !<
            real*8 :: dxnorm                                                                !<
            real*8 :: gnorm                                                                 !<
            real*8 :: fp                                                                    !<
            real*8 :: par                                                                   !< INPUT/OUTPUT: On input an initial estimate of the
                                                                                            !<          Levenberg-Marquardt parameter. On output the
                                                                                            !<          final estimate. PAR should be nonnegative.
            real*8 :: parc                                                                  !<
            real*8 :: parl                                                                  !<
            real*8 :: paru                                                                  !<
            real*8 :: sum2                                                                  !<
            real*8 :: temp                                                                  !<
            real*8, dimension(n) :: diag                                                    !< INPUT:   the diagonal elements of the matrix D.
            real*8, dimension(n) :: qtb                                                     !< INPUT:   the first N elements of the vector Q'*B.
            real*8, dimension(n) :: wa1                                                     !<
            real*8, dimension(n) :: wa2                                                     !<
            real*8, dimension(n) :: sdiag                                                   !< OUTPUT:  the diagonal elements of the upper
                                                                                            !<          triangular matrix S.
            real*8, dimension(n) :: x                                                       !< OUTPUT:  the least squares solution of the system
                                                                                            !<          A*X = B, sqrt(PAR)*D*X = 0,
                                                                                            !<          for the output value of PAR.
            real*8, dimension(ldr,n) :: r                                                   !< INPUT/OUTPUT: R(LDR,N),the N by N matrix. The full
                                                                                            !<          upper triangle must contain the full upper
                                                                                            !<          triangle of the matrix R. On output the full
                                                                                            !<          upper triangle is unaltered, and the strict lower
                                                                                            !<          triangle contains the strict upper triangle
                                                                                            !<          (transposed) of the upper triangular matrix S.

            !< reset output variables
            x = 0.d0
            sdiag = 0.d0



            !< DWARF is the smallest positive magnitude.
            dwarf = tiny(dwarf)


            !< Compute and store in X the Gauss-Newton direction. If the jacobian is rank-deficient, obtain a least squares solution.
            wa1 = 0.d0
            wa2 = 0.d0
            sdiag = 0.d0
            x = 0.d0
            nsing = n
            Do j = 1, n
                wa1(j) = qtb(j)
                if (r(j,j) == 0.d0 .and. nsing == n) then
                    nsing = j - 1
                endif
                if (nsing < n) then
                    wa1(j) = 0.d0
                endif
            end Do
            Do k = 1, nsing
                j = nsing - k + 1
                wa1(j) = wa1(j) / r(j,j)
                temp = wa1(j)
                jm1 = j - 1
                if (jm1 >= 1) then
                    Do i = 1, jm1
                        wa1(i) = wa1(i) - r(i,j) * temp
                    end Do
                endif
                !wa1(1:j-1) = wa1(1:j-1) - r(1:j-1,j) * temp
            end Do
            Do j = 1, n
                l = ipvt(j)
                x(l) = wa1(j)
            end Do


            !< Initialize the iteration counter. Evaluate the function at the origin, and test for acceptance of the Gauss-Newton direction.
            iter = 0
            wa2(1:n) = diag(1:n) * x(1:n)
            call enorm(dxnorm, n, wa2)
            fp = dxnorm - delta
            if (fp <= 0.1D+00 * delta) then
                go to 30220
            endif


            !< If the jacobian is not rank deficient, the Newton step provides a lower bound, PARL, for the zero of the function.
            !< Otherwise set this bound to zero.
            parl = 0.d0
            if (nsing >= n) then
                Do j = 1, n
                    l = ipvt(j)
                    wa1(j) = diag(l) * (wa2(l) / dxnorm)
                end Do
                Do j = 1, n
                    sum2 = dot_product(wa1(1:j-1), r(1:j-1,j))
                    wa1(j) = (wa1(j) - sum2) / r(j,j)
                end Do
                call enorm(temp, n, wa1)
                parl = ((fp / delta) / temp) / temp
            endif


            !< Calculate an upper bound, PARU, for the zero of the function.
            Do j = 1, n
                sum2 = dot_product(qtb(1:j), r(1:j,j))
                l = ipvt(j)
                wa1(j) = sum2 / diag(l)
            end Do
            call enorm(gnorm, n, wa1)
            paru = gnorm / delta
            if (paru == 0.d0) then
                paru = dwarf / dmin1(delta, 0.1D+00)
            endif


            !< If the input PAR lies outside of the interval (PARL, PARU), set PAR to the closer endpoint.
            par = dmax1(par, parl)
            par = dmin1(par, paru)
            if (par == 0.d0) then
                par = gnorm / dxnorm
            endif


            !< Beginning of an iteration.
      30150 continue
            iter = iter + 1                                                                 !< increase counter for iteration


            !< Evaluate the function at the current value of PAR.
            if (par == 0.d0) then
                par = dmax1(dwarf, 0.001D+00 * paru)
            endif
            wa1(1:n) = dsqrt(par) * diag(1:n)
            call qrsolv(n, r, ldr, ipvt, wa1, qtb, x, sdiag)
            wa2(1:n) = diag(1:n) * x(1:n)
            call enorm(dxnorm, n, wa2)
            temp = fp
            fp = dxnorm - delta


            !< If the function is small enough, accept the current value of PAR.
            if (dabs(fp) <= 0.1D+00 * delta) then
                go to 30220
            endif


            !< Test for the exceptional cases where PARL is zero or the number of iterations has reached 10.
            if (parl == 0.d0 .and. fp <= temp .and. temp < 0.d0) then
                go to 30220
            !elseif (iter == 10) then
            !    go to 30220
            endif

            !< Compute the Newton correction.
            Do j = 1, n
                l = ipvt(j)
                wa1(j) = diag(l) * (wa2(l) / dxnorm)
            end Do

            Do j = 1, n
                wa1(j) = wa1(j) / sdiag(j)
                temp = wa1(j)
                wa1(j+1:n) = wa1(j+1:n) - r(j+1:n,j) * temp
            end Do
            call enorm(temp, n, wa1)
            parc = ((fp / delta) / temp) / temp


            !< Depending on the sign of the function, update PARL or PARU.
            if (0.d0 < fp) then
                parl = dmax1(parl, par)
            elseif (fp < 0.d0) then
                paru = dmin1(paru, par)
            endif


            !< Compute an improved estimate for PAR.
            par = dmax1(parl, par + parc)

            !< End of an iteration.
            go to 30150
      30220 continue


            !< Termination.
            if (iter == 0) then
                par = 0.d0
            endif
            return
        end subroutine lmpar


        !>************************************************************************************************************************************************
        !> subroutine: qrfac
        !>
        !> (MINPACK): QRFAC computes a QR factorization using Householder transformations.
        !>
        !> input variables:     m:                      the number of rows of A.
        !>                      n:                      the number of columns of A.
        !>                      a:                      A(LDA,N), the M by N array. On input, A contains the matrix for which the QR factorization is to
        !>                                              be computed. On output, the strict upper trapezoidal part of A contains the strict upper
        !>                                              trapezoidal part of R, and the lower trapezoidal part of A contains a factored form of Q (the
        !>                                              non-trivial elements of the U vectors described above).
        !>                      lda:                    the leading dimension of A, which must be no less than M.
        !>                      pivot:                  is TRUE if column pivoting is to be carried out.
        !>                      lipvt:                  the dimension of IPVT, which should be N if pivoting is used.
        !>
        !> output variables:    ipvt:                   defines the permutation matrix P such that A*P = Q*R. Column J of P is column IPVT(J) of the
        !>                                              identity matrix. If PIVOT is false, IPVT is not referenced.
        !>                      rdiag:                  contains the diagonal elements of R.
        !>                      acnorm:                 the norms of the corresponding columns of the input matrix A.  If this information is not needed,
        !>                                              then ACNORM can coincide with RDIAG.
        !>
        !>
        !> \author MINPACK and Thomas Moeller
        !>
        !> \date 06.04.2010
        !>
        subroutine qrfac(m, n, a, lda, pivot, ipvt, lipvt, rdiag, acnorm)
            !<
            !<  Discussion:
            !<
            !<    This subroutine uses Householder transformations with column
            !<    pivoting (optional) to compute a QR factorization of the
            !<    M by N matrix A.  That is, QRFAC determines an orthogonal
            !<    matrix Q, a permutation matrix P, and an upper trapezoidal
            !<    matrix R with diagonal elements of nonincreasing magnitude,
            !<    such that A*P = Q*R.  The Householder transformation for
            !<    column K, K = 1,2,...,min(M,N), is of the form
            !<
            !<      I - ( 1 / U(K) ) * U * U'
            !<
            !<    where U has zeros in the first K-1 positions.  The form of
            !<    this transformation and the method of pivoting first
            !<    appeared in the corresponding LINPACK subroutine.
            !<
            !<  Licensing:
            !<
            !<    This code is distributed under the GNU LGPL license.
            !<
            !<  Modified:
            !<
            !<    06 April 2010
            !<
            !<  Author:
            !<
            !<    Original FORTRAN77 version by Jorge More, Burton Garbow, Kenneth Hillstrom.
            !<    FORTRAN90 version by John Burkardt.
            !<
            !<  Reference:
            !<
            !<    Jorge More, Burton Garbow, Kenneth Hillstrom,
            !<    User Guide for MINPACK-1,
            !<    Technical Report ANL-80-74,
            !<    Argonne National Laboratory, 1980.
            !<
            !<  Parameters:
            !<
            !<    Input, integer :: M, the number of rows of A.
            !<
            !<    Input, integer :: N, the number of columns of A.
            !<
            !<    Input/output, real ( kind = 8 ) A(LDA,N), the M by N array.
            !<    On input, A contains the matrix for which the QR factorization is to
            !<    be computed.  On output, the strict upper trapezoidal part of A contains
            !<    the strict upper trapezoidal part of R, and the lower trapezoidal
            !<    part of A contains a factored form of Q (the non-trivial elements of
            !<    the U vectors described above).
            !<
            !<    Input, integer :: LDA, the leading dimension of A, which must
            !<    be no less than M.
            !<
            !<    Input, logical PIVOT, is TRUE if column pivoting is to be carried out.
            !<
            !<    Output, integer :: IPVT(LIPVT), defines the permutation matrix P such
            !<    that A*P = Q*R.  Column J of P is column IPVT(J) of the identity matrix.
            !<    If PIVOT is false, IPVT is not referenced.
            !<
            !<    Input, integer :: LIPVT, the dimension of IPVT, which should be N if
            !<    pivoting is used.
            !<
            !<    Output, real ( kind = 8 ) RDIAG(N), contains the diagonal elements of R.
            !<
            !<    Output, real ( kind = 8 ) ACNORM(N), the norms of the corresponding
            !<    columns of the input matrix A.  If this information is not needed,
            !<    then ACNORM can coincide with RDIAG.
            !<
            !<--------------------------------------------------------------------------------------------------------------------------------------------


            implicit none
            integer :: lda                                                                  !< INPUT:   LDA, the leading dimension of A, which must
                                                                                            !<          be no less than M.
            integer :: lipvt                                                                !< INPUT:   the dimension of IPVT, which should be N if
                                                                                            !<          pivoting is used.
            integer :: m                                                                    !< INPUT:   M, the number of rows of A.
            integer :: n                                                                    !< INPUT:   N, the number of columns of A.
            integer :: i4_temp                                                              !<
            integer :: j                                                                    !<
            integer :: k                                                                    !<
            integer :: kmax                                                                 !<
            integer :: minmn                                                                !<
            integer, dimension(lipvt) :: ipvt                                               !< OUTPUT:  defines the permutation matrix P such that
                                                                                            !<          A*P = Q*R.  Column J of P is column IPVT(J)
                                                                                            !<          of the identity matrix. If PIVOT is false,
                                                                                            !<          IPVT is not referenced.
            real*8 :: ajnorm                                                                !<
            real*8 :: epsmch                                                                !<
            real*8 :: temp                                                                  !<
            real*8, dimension(m) :: r8_temp                                                 !< working variable
            real*8, dimension(n) :: rdiag                                                   !< OUTPUT:  contains the diagonal elements of R.
            real*8, dimension(n) :: wa                                                      !<
            real*8, dimension(n) :: acnorm                                                  !< OUTPUT:  the norms of the corresponding columns of the
                                                                                            !<          input matrix A.  If this information is not
                                                                                            !<          needed, then ACNORM can coincide with RDIAG.
            real*8, dimension(lda,n) :: a                                                   !< INPUT/OUTPUT: the M by N array. On input, A contains
                                                                                            !<          the matrix for which the QR factorization is to
                                                                                            !<          be computed.  On output, the strict upper
                                                                                            !<          trapezoidal part of A contains the strict upper
                                                                                            !<          trapezoidal part of R, and the lower trapezoidal
                                                                                            !<          part of A contains a factored form of Q (the
                                                                                            !<          non-trivial elements of the U vectors described
                                                                                            !<          above).
            logical :: pivot                                                                !< INPUT:   is TRUE if column pivoting is to be carried out


            epsmch = epsilon(epsmch)


            !< reset output variables
            ipvt = 0
            rdiag = 0.d0
            acnorm = 0.d0


            !< Compute the initial column norms and initialize several arrays.
            Do j = 1, n
                call enorm(acnorm(j), m, a(1:m,j))
            end Do
            rdiag(1:n) = acnorm(1:n)
            wa(1:n) = acnorm(1:n)
            if (pivot) then
                Do j = 1, n
                    ipvt(j) = j
                end Do
            endif


            !< Reduce A to R with Householder transformations.
            minmn = min(m, n)
            Do j = 1, minmn


                !< Bring the column of largest norm into the pivot position.
                if (pivot) then
                    kmax = j
                    Do k = j, n
                        if (rdiag(k) > rdiag(kmax)) then
                            kmax = k
                        endif
                    end Do
                    if (kmax /= j) then
                        r8_temp(1:m) = a(1:m,j)
                        a(1:m,j) = a(1:m,kmax)
                        a(1:m,kmax) = r8_temp(1:m)
                        rdiag(kmax) = rdiag(j)
                        wa(kmax) = wa(j)
                        i4_temp = ipvt(j)
                        ipvt(j) = ipvt(kmax)
                        ipvt(kmax) = i4_temp
                    endif
                endif


                !< Compute the Householder transformation to reduce the J-th column of A to a multiple of the J-th unit vector.
                call enorm(ajnorm, (m - j + 1), a(j, j))
                if (ajnorm /= 0.d0) then
                    if (a(j,j) < 0.d0) then
                        ajnorm = -ajnorm
                    endif
                    a(j:m,j) = a(j:m,j) / ajnorm
                    a(j,j) = a(j,j) + 1.d0


                    !< Apply the transformation to the remaining columns and update the norms.
                    Do k = (j + 1), n
                        temp = dot_product(a(j:m,j), a(j:m,k)) / a(j,j)
                        a(j:m,k) = a(j:m,k) - temp * a(j:m,j)
                        if (pivot .and. rdiag(k) /= 0.d0) then
                            temp = a(j,k) / rdiag(k)
                            rdiag(k) = rdiag(k) * dsqrt(dmax1(0.d0, 1.d0 - temp**2))
                            if (0.05D+00 * (rdiag(k) / wa(k))**2 <= epsmch) then
                                call enorm(rdiag(k), (m - j), a(j + 1, k))
                                wa(k) = rdiag(k)
                            endif
                        endif
                    end Do
                endif
                rdiag(j) = -ajnorm
            end Do
            return
        end subroutine qrfac


        !>************************************************************************************************************************************************
        !> subroutine: qrsolv
        !>
        !> (MINPACK): QRSOLV solves a rectangular linear system A*x=b in the least squares sense.
        !>
        !> input variables:     n:                      the order of R.
        !>                      r:                      the N by N matrix. On input the full upper triangle must contain the full upper triangle of the
        !>                                              matrix R.  On output the full upper triangle is unaltered, and the strict lower triangle contains
        !>                                              the strict upper triangle (transposed) of the upper triangular matrix S.
        !>                      ldr:                    the leading dimension of R, which must be at least N.
        !>                      ipvt:                   defines the permutation matrix P such that A*P = Q*R.  Column J of P is column IPVT(J) of the
        !>                                              identity matrix.
        !>                      diag:                   the diagonal elements of the matrix D.
        !>                      qtb:                    the first N elements of the vector Q'*B.
        !>
        !> output variables:    r:                      the N by N matrix. On input the full upper triangle must contain the full upper triangle of the
        !>                                              matrix R.  On output the full upper triangle is unaltered, and the strict lower triangle contains
        !>                                              the strict upper triangle (transposed) of the upper triangular matrix S.
        !>                      x:                      the least squares solution.
        !>                      sdiag:                  the diagonal elements of the upper triangular matrix S.
        !>
        !>
        !> \author MINPACK and Thomas Moeller
        !>
        !> \date 06.04.2010
        !>
        subroutine qrsolv(n, r, ldr, ipvt, diag, qtb, x, sdiag)
            !<
            !<  Discussion:
            !<
            !<    Given an M by N matrix A, an N by N diagonal matrix D,
            !<    and an M-vector B, the problem is to determine an X which
            !<    solves the system
            !<
            !<      A*X = B
            !<      D*X = 0
            !<
            !<    in the least squares sense.
            !<
            !<    This subroutine completes the solution of the problem
            !<    if it is provided with the necessary information from the
            !<    QR factorization, with column pivoting, of A.  That is, if
            !<    Q*P = Q*R, where P is a permutation matrix, Q has orthogonal
            !<    columns, and R is an upper triangular matrix with diagonal
            !<    elements of nonincreasing magnitude, then QRSOLV expects
            !<    the full upper triangle of R, the permutation matrix p,
            !<    and the first N components of Q'*B.
            !<
            !<    The system is then equivalent to
            !<
            !<      R*Z = Q'*B
            !<      P'*D*P*Z = 0
            !<
            !<    where X = P*Z.  If this system does not have full rank,
            !<    then a least squares solution is obtained.  On output QRSOLV
            !<    also provides an upper triangular matrix S such that
            !<
            !<      P'*(A'*A + D*D)*P = S'*S.
            !<
            !<    S is computed within QRSOLV and may be of separate interest.
            !<
            !<  Licensing:
            !<
            !<    This code is distributed under the GNU LGPL license.
            !<
            !<  Modified:
            !<
            !<    06 April 2010
            !<
            !<  Author:
            !<
            !<    Original FORTRAN77 version by Jorge More, Burton Garbow, Kenneth Hillstrom.
            !<    FORTRAN90 version by John Burkardt.
            !<
            !<  Reference:
            !<
            !<    Jorge More, Burton Garbow, Kenneth Hillstrom,
            !<    User Guide for MINPACK-1,
            !<    Technical Report ANL-80-74,
            !<    Argonne National Laboratory, 1980.
            !<
            !<  Parameters:
            !<
            !<    Input, integer :: N, the order of R.
            !<
            !<    Input/output, real ( kind = 8 ) R(LDR,N), the N by N matrix.
            !<    On input the full upper triangle must contain the full upper triangle
            !<    of the matrix R.  On output the full upper triangle is unaltered, and
            !<    the strict lower triangle contains the strict upper triangle
            !<    (transposed) of the upper triangular matrix S.
            !<
            !<    Input, integer :: LDR, the leading dimension of R, which must be
            !<    at least N.
            !<
            !<    Input, integer :: IPVT(N), defines the permutation matrix P such that
            !<    A*P = Q*R.  Column J of P is column IPVT(J) of the identity matrix.
            !<
            !<    Input, real ( kind = 8 ) DIAG(N), the diagonal elements of the matrix D.
            !<
            !<    Input, real ( kind = 8 ) QTB(N), the first N elements of the vector Q'*B.
            !<
            !<    Output, real ( kind = 8 ) X(N), the least squares solution.
            !<
            !<    Output, real ( kind = 8 ) SDIAG(N), the diagonal elements of the upper
            !<    triangular matrix S.
            !<
            !<--------------------------------------------------------------------------------------------------------------------------------------------


            implicit none
            integer :: ldr                                                                  !< INPUT:   LDR, the leading dimension of R, which must be
                                                                                            !<          at least N.
            integer :: n                                                                    !< INPUT:   N, the order of R.
            integer :: i, j, k                                                              !< loop variables
            integer :: l, nsing                                                             !< working variables
            integer, dimension(n) :: ipvt                                                   !< INPUT:   defines the permutation matrix P such that
                                                                                            !<          A*P = Q*R. Column J of P is column IPVT(J) of
                                                                                            !<          the identity matrix.
            real*8 :: c                                                                     !<
            real*8 :: cotan                                                                 !<
            real*8 :: qtbpj                                                                 !<
            real*8 :: s                                                                     !<
            real*8 :: sum2                                                                  !<
            real*8 :: t                                                                     !<
            real*8 :: temp                                                                  !<
            real*8, dimension(n) :: wa                                                      !<
            real*8, dimension(n) :: x                                                       !< OUTPUT:  X(N), the least squares solution.
            real*8, dimension(n) :: diag                                                    !< INPUT:   the diagonal elements of the matrix D.
            real*8, dimension(n) :: qtb                                                     !< INPUT:   the first N elements of the vector Q'*B.
            real*8, dimension(n) :: sdiag                                                   !< OUTPUT:  SDIAG(N), the diagonal elements of the upper
                                                                                            !<          triangular matrix S.
            real*8, dimension(ldr,n) :: r                                                   !< INPUT/OUTPUT: R(LDR,N), the N by N matrix. On input the
                                                                                            !<          triangle of the matrix R. On output the full
                                                                                            !<          upper triangle is unaltered, and the strict
                                                                                            !<          lower triangle contains the strict upper triangle
                                                                                            !<          (transposed) of the upper triangular matrix S.


            !< reset output variabels
            x = 0.d0
            sdiag = 0.d0


            !< Copy R and Q'*B to preserve input and initialize S. In particular, save the diagonal elements of R in X.
            Do j = 1, n
                r(j:n,j) = r(j,j:n)
                x(j) = r(j,j)
            end Do
            wa(1:n) = qtb(1:n)


            !< Eliminate the diagonal matrix D using a Givens rotation.
            Do j = 1, n


                !< Prepare the row of D to be eliminated, locating the diagonal element using P from the QR factorization.
                l = ipvt(j)
                if (diag(l) /= 0.d0) then
                    sdiag(j:n) = 0.d0
                    sdiag(j) = diag(l)


                    !< The transformations to eliminate the row of D modify only a single element of Q'*B beyond the first N, which is initially zero.
                    qtbpj = 0.d0
                    Do k = j, n


                        !< Determine a Givens rotation which eliminates the appropriate element in the current row of D.
                        if (sdiag(k) /= 0.d0) then
                            if (dabs(r(k,k)) < dabs(sdiag(k))) then
                                cotan = r(k,k) / sdiag(k)
                                s = 0.5D+00 / dsqrt(0.25D+00 + 0.25D+00 * cotan**2)
                                c = s * cotan
                            else
                                t = sdiag(k) / r(k,k)
                                c = 0.5D+00 / dsqrt(0.25D+00 + 0.25D+00 * t**2)
                                s = c * t
                            endif


                            !< Compute the modified diagonal element of R and the modified element of (Q'*B,0).
                            r(k,k) = c * r(k,k) + s * sdiag(k)
                            temp = c * wa(k) + s * qtbpj
                            qtbpj = -s * wa(k) + c * qtbpj
                            wa(k) = temp


                            !< Accumulate the tranformation in the row of S.
                            Do i = (k + 1), n
                                temp = c * r(i,k) + s * sdiag(i)
                                sdiag(i) = -s * r(i,k) + c * sdiag(i)
                                r(i,k) = temp
                            end Do
                        endif
                    end Do
                endif


                !< Store the diagonal element of S and restore the corresponding diagonal element of R.
                sdiag(j) = r(j,j)
                r(j,j) = x(j)
            end Do


            !< Solve the triangular system for Z.  If the system is singular, then obtain a least squares solution.
            nsing = n
            Do j = 1, n
                if (sdiag(j) == 0.d0 .and. nsing == n) then
                    nsing = j - 1
                endif
                if (nsing < n) then
                    wa(j) = 0.d0
                endif
            end Do
            Do j = nsing, 1, (-1)
                sum2 = dot_product(wa(j+1:nsing), r(j+1:nsing,j))
                wa(j) = (wa(j) - sum2) / sdiag(j)
            end Do


            !< Permute the components of Z back to components of X.
            Do j = 1, n
                l = ipvt(j)
                x(l) = wa(j)
            end Do

            return
        end subroutine qrsolv


        !>************************************************************************************************************************************************
        !> subroutine: covar_minpack
        !>
        !> (MINPACK): subroutine covar_minpack
        !>
        !> input variables:     n:                      is a positive integer input variable set to the order of r.
        !>                      r:                      is an n by n array. on input the full upper triangle must contain the full upper triangle of the
        !>                                              matrix r. on output r contains the square symmetric covariance matrix.
        !>                      ldr:                    is a positive integer input variable not less than n which specifies the leading dimension of
        !>                                              the array r.
        !>                      ipvt:                   is an integer input array of length n which defines the permutation matrix p such that a*p = q*r.
        !>                                              column j of p is column ipvt(j) of the identity matrix.
        !>                      tol:                    is a nonnegative input variable used to define the numerical rank of a in the manner described
        !>                                              above.
        !>                      wa:                     is a work array of length n.
        !>
        !> output variables:    r:                      is an n by n array. on input the full upper triangle must contain the full upper triangle of the
        !>                                              matrix r. on output r contains the square symmetric covariance matrix.
        !>
        !>
        !> \author MINPACK and Thomas Moeller
        !>
        !> \date 06.04.2010
        !>
        subroutine covar_minpack(n, r, ldr, ipvt, tol, wa)
            !<
            !<    given an m by n matrix a, the problem is to determine
            !<    the covariance matrix corresponding to a, defined as
            !<
            !<                   t
            !<          inverse(a *a) .
            !<
            !<    this subroutine completes the solution of the problem
            !<    if it is provided with the necessary information from the
            !<    qr factorization, with column pivoting, of a. that is, if
            !<    a*p = q*r, where p is a permutation matrix, q has orthogonal
            !<    columns, and r is an upper triangular matrix with diagonal
            !<    elements of nonincreasing magnitude, then covar expects
            !<    the full upper triangle of r and the permutation matrix p.
            !<    the covariance matrix is then computed as
            !<
            !<                     t     t
            !<          p*inverse(r *r)*p  .
            !<
            !<    if a is nearly rank deficient, it may be desirable to compute
            !<    the covariance matrix corresponding to the linearly independent
            !<    columns of a. to define the numerical rank of a, covar uses
            !<    the tolerance tol. if l is the largest integer such that
            !<
            !<          abs(r(l,l)) .gt. tol*abs(r(1,1)) ,
            !<
            !<    then covar computes the covariance matrix corresponding to
            !<    the first l columns of r. for k greater than l, column
            !<    and row ipvt(k) of the covariance matrix are set to zero.
            !<
            !<    the subroutine statement is
            !<
            !<      subroutine covar(n,r,ldr,ipvt,tol,wa)
            !<
            !<    where
            !<
            !<      n is a positive integer input variable set to the order of r.
            !<
            !<      r is an n by n array. on input the full upper triangle must
            !<        contain the full upper triangle of the matrix r. on output
            !<        r contains the square symmetric covariance matrix.
            !<
            !<      ldr is a positive integer input variable not less than n
            !<        which specifies the leading dimension of the array r.
            !<
            !<      ipvt is an integer input array of length n which defines the
            !<        permutation matrix p such that a*p = q*r. column j of p
            !<        is column ipvt(j) of the identity matrix.
            !<
            !<      tol is a nonnegative input variable used to define the
            !<        numerical rank of a in the manner described above.
            !<
            !<      wa is a work array of length n.
            !<
            !<    subprograms called
            !<
            !<      fortran-supplied ... dabs
            !<
            !<    argonne national laboratory. minpack project. august 1980.
            !<    burton s. garbow, kenneth e. hillstrom, jorge j. more
            !<
            !<--------------------------------------------------------------------------------------------------------------------------------------------


            implicit none
            integer :: n,ldr
            integer :: i, ii, j, jj, k, km1, l
            integer, dimension(n) :: ipvt
            real*8, parameter :: zero = 0.d0
            real*8, parameter :: one = 1.d0
            real*8 :: temp, tolr
            real*8 :: tol
            real*8, dimension(n) :: wa
            real*8, dimension(ldr,n) :: r
            logical :: sing


            !< form the inverse of r in the full upper triangle of r.
            tolr = tol * dabs(r(1,1))
            l = 0
            Do k = 1, n
                if (dabs(r(k,k)) .le. tolr) go to 50
                r(k,k) = one/r(k,k)
                km1 = k - 1
                if (km1 .lt. 1) go to 30
                Do j = 1, km1
                    temp = r(k,k)*r(j,k)
                    r(j,k) = zero
                    Do i = 1, j
                            r(i,k) = r(i,k) - temp*r(i,j)
                    end Do
                end Do
        30      continue
                l = k
            end Do
        50  continue


            !< form the full upper triangle of the inverse of (r transpose)*r in the full upper triangle of r.
            if (l .lt. 1) go to 110
            Do k = 1, l
                km1 = k - 1
                if (km1 .lt. 1) go to 80
                    Do j = 1, km1
                        temp = r(j,k)
                        Do i = 1, j
                            r(i,j) = r(i,j) + temp*r(i,k)
                        end Do
                    end Do
        80      continue
                temp = r(k,k)
                Do i = 1, k
                    r(i,k) = temp*r(i,k)
                end Do
            end Do
        110 continue


            !< form the full lower triangle of the covariance matrix in the strict lower triangle of r and in wa.
            Do j = 1, n
                jj = ipvt(j)
                sing = j .gt. l
                Do i = 1, j
                    if (sing) r(i,j) = zero
                    ii = ipvt(i)
                    if (ii > jj) r(ii,jj) = r(i,j)
                    if (ii < jj) r(jj,ii) = r(i,j)
                end Do
                wa(jj) = r(j,j)
            end Do


            !< symmetrize the covariance matrix in r.
            Do j = 1, n
                Do i = 1, j
                    r(i,j) = r(j,i)
                end Do
                r(j,j) = wa(j)
            end Do
            return
        end subroutine covar_minpack


        !>************************************************************************************************************************************************
        !> subroutine: fcn
        !>
        !> (MINPACK): Input, external FCN, the name of the user-supplied subroutine which calculates the functions and the jacobian.
        !>
        !> input variables:     chisq:                  value of chi^2
        !>                      n:                      the number of functions and variables.
        !>                      x:                      On input, X must contain an initial estimate of the solution vector. On output X contains the
        !>                                              final estimate of the solution vector.
        !>                      ldfjac:                 the leading dimension of the array FJAC.  LDFJAC must be at least N.
        !>                      iflag:                  ?
        !>                      TotalNumPar:            total number of all parameters
        !>                      numfit:                 number of free parameters
        !>                      colx:                   number of columns belonging to the x-column
        !>                      NumFile:                number of input files
        !>                      MaxL:                   max. total length
        !>                      MaxCol:                 max. number of columns
        !>
        !> output variables:    x:                      On input, X must contain an initial estimate of the solution vector. On output X contains the
        !>                                              final estimate of the solution vector.
        !>                      fvec:                   the functions evaluated at the output X.
        !>                      fjac:                   an N by N matrix, containing the orthogonal matrix Q produced by the QR factorization
        !>                                              of the final approximate jacobian.
        !>
        !>
        !> \author MINPACK and Thomas Moeller
        !>
        !> \date 06.04.2010
        !>
        subroutine fcn(chisq, n, x, fvec, fjac, ldfjac, iflag, TotalNumPar, numfit, colx, NumFile, MaxL, MaxCol)
            !<
            !<      FCN should have the form:
            !<
            !<      subroutine fcn(n, x, fvec, fjac, ldfjac, iflag)
            !<      integer :: ldfjac
            !<      integer :: n
            !<      real fjac(ldfjac,n)
            !<      real fvec(n)
            !<      integer :: iflag
            !<      real x(n)
            !<
            !<  If IFLAG = 1 on intput, FCN should calculate the functions at X and
            !<  return this vector in FVEC.
            !<  If IFLAG = 2 on input, FCN should calculate the jacobian at X and
            !<  return this matrix in FJAC.
            !<  To terminate the algorithm, the user may set IFLAG negative.
            !<
            !<  Input, integer :: N, the number of functions and variables.
            !<
            !<  Input/output, real ( kind = 8 ) X(N).  On input, X must contain an initial
            !<  estimate of the solution vector.  On output X contains the final
            !<  estimate of the solution vector.
            !<
            !<  Output, real ( kind = 8 ) FVEC(N), the functions evaluated at the output X.
            !<
            !<  Output, real ( kind = 8 ) FJAC(LDFJAC,N), an N by N matrix, containing
            !<  the orthogonal matrix Q produced by the QR factorization
            !<  of the final approximate jacobian.
            !<
            !<  Input, integer :: LDFJAC, the leading dimension of the
            !<  array FJAC.  LDFJAC must be at least N.
            !<

            implicit none
            integer :: n                                                                    !< the number of functions and variables
            integer :: ldfjac                                                               !< the leading dimension of the array FJAC. LDFJAC must be
                                                                                            !< at least N
            integer :: iflag                                                                !< ?
            real*8, dimension(n) :: x                                                       !< On input, X must contain an initial estimate of the
                                                                                            !< solution vector. On output X contains the final
                                                                                            !< estimate of the solution vector.
            real*8, dimension(ldfjac) :: fvec                                               !< the functions evaluated at the output X
            real*8, dimension(ldfjac, n) :: fjac                                            !< an N by N matrix, containing the orthogonal matrix Q
                                                                                            !< produced by the QR factorization of the final
                                                                                            !< approximate jacobian
            integer :: TotalNumPar                                                          !< total number of parameter
            integer :: numfit                                                               !< number of fitted parameters
            integer :: colx                                                                 !< number of columns in experimental x data
            integer :: NumFile                                                              !< Number of experimental files
            integer :: MaxL                                                                 !< max number of lines of all experimental files
            integer :: MaxCol                                                               !< max number of columns of all experimental files
            integer :: i, j, k, NumberFile                                                  !< working variables
            real*8 :: chisq                                                                 !< chi^2
            real*8, dimension(numfit) :: beta2                                              !< beta2 array
            real*8, dimension(TotalNumPar, TotalNumPar) :: alpha                            !< matrix alpha
            real*8, dimension(NumFile, MaxL, MaxCol) :: FitFunctionOut                      !< values of the model function at all calculated points
            real*8, dimension(NumFile, MaxL, MaxCol) :: Chi2Values                          !< values of chi**2 for each experimental data point


            !< print what you Do ..
            if (printflag) then
                print '(A1, 11x, "Calculate model function .. ", A1, $)', char(13), char(13)
            endif
            fvec = 0.d0
            if (iflag == 1) then
                Gradientflag = .false.
            elseif (iflag == 2) then
                Gradientflag = .true.
                fjac = 0.d0
            endif


            !< call subroutine for the determination of chi**2
            k = 0
            Do i = 1, parameternumber
                if (ia(i)) then
                    k = k + 1
                    currentparmval(i) = x(k)
                endif
            end Do
            alpha = 0.d0
            beta2 = 0.d0
            FitFunctionOut = 0.d0
            Chi2Values = 0.d0
            call ModelCalcChiFunctionLM(parameternumber, currentparmval, ia, nfit, x, colx, NumFile, MaxL, MaxCol, FitFunctionOut, Chi2Values, alpha, beta2)
            chisq = chisqValues(0)


            !< store model function values and correpsonding chi2 values for the best fitness
            if (chisq < BestSitesParamSet(1, 1)) then
                BestSitesParamSet(1, 1) = chisq
                BestSitesParamSet(1, 2:) = x(:)
                BestSitesModelValues(1, :, :, :) = FitFunctionOut
                BestSitesChi2Values(1, :, :, :) = Chi2Values
            endif


            !< print what you Do ..
            ! if (printflag) then
            !    print '("done!",A,$)', char(13)
            ! endif


            !< determine output arrays
            j = 0
            Do NumberFile = 1, NumberExpFiles                                               !< loop over exp. data files
                Do i = 1, lengthexpdata(NumberFile)                                         !< loop over all line of current exp. data file
                    Do k = 1, NumberYColumns(NumberFile)                                    !< loop over y-columns
                        j = j + 1
                        fvec(j) = Chi2Values(NumberFile, i, k)
                        if (iflag == 2) then
                            fjac(j, :) = ModelFunction(2:, NumberFile, k, i)
                        endif
                    end Do
                end Do
            end Do


            !< determine chi^2
            chisq = sum(fvec(:))

            ! Debug:
            ! print*,'>',x,chisq

            return
        end subroutine fcn


        !>************************************************************************************************************************************************
        !> subroutine: call_LevenbergMarquardt_MINPACK
        !>
        !> (MINPACK): call_LevenbergMarquardt_MINPACK calls subroutines for MINPACK verison of Levenberg-Marquardt algorithm
        !>
        !> input variables:     m:                      total number of exp. data points
        !>                      n:                      number of fit parameters
        !>                      tol:                    limit of chi^2 (?)
        !>
        !> output variables:    none:
        !>
        !>
        !> \author Irina Bernst, Thomas Moeller
        !>
        !> \date 09.06.2010
        !>
        subroutine call_LevenbergMarquardt_MINPACK(m, n, tol)
            !< subroutine for calling minpack version

            implicit none
            integer :: i, k                                                                 !< working variables
            integer :: m                                                                    !< total number of exp. data points
            integer :: n                                                                    !< number of fit parameters
            integer :: ldfjac                                                               !< working variable
            integer :: iflag                                                                !< working variable
            integer :: info                                                                 !< status variable
            real*8 :: tol                                                                   !< abort criteria (limit of chi^2)
            real*8, dimension(n) :: x                                                       !< array containing fit parameters
            real*8, dimension(m) :: fvec                                                    !< minimization function value
            real*8, dimension(m, n) :: fjac                                                 !< corresponding gradient


            !< define array x
            k = 0
            x = 0.d0
            Do i = 1, parameternumber
                if (ia(i)) then
                    k = k + 1
                    x(k) = currentparmval(i)
                endif
            end Do


            !< call minpack version
            ldfjac = m
            GradientMethod = 2                                                              !< select method for gradient
            iflag = 1
            call lmder1(m, n, x, fvec, fjac, ldfjac, tol, info)
            return
        end subroutine call_LevenbergMarquardt_MINPACK
end Module LevenbergMarquardt_MINPACK
!*********************************************************************************************************************************************************




!*********************************************************************************************************************************************************
!> Module: Algorithm
!>
!>         Module contains the main subroutine used to start the different versions of the Levenberg-Marquardt algorithm
!>
!>
!> \author Thomas Moeller
!>
!> \date 26.08.2014
!>
Module Algorithm

    use Variables
    use FunctionCalling
    use LevenbergMarquardt_MINPACK

    implicit none

    contains


        !*************************************************************************************************************************************************
        !> subroutine: MainAlg
        !>
        !> main subroutine which starts the Levenberg-Marquardt algorithm
        !>
        !>
        !> input variables:         printflagNum:           flag for screen output 1 (=yes) or 0 (=no)
        !>                          LastAlgorithmNum:       number of last algorithm
        !>                          FinalParameterSet:      array containing the parameter set with SampleslogL(i) >= logZ
        !>                          chilm:                  user defined abort criteria for chi**2
        !>                          NumberOfFitAlgorithms:  total number of all algorithms in the chain
        !>                          numiter:                max. number of iterations
        !>                          LevenbergMarquardtCounter:  counts number of calls
        !>                          DeterminationChi2:      method being used for the determination of chi^2
        !>                          PlotIterationOrg:       plot model function for each iteration set 1(=yes) or 0(=no)
        !>                          PlotTypeOrg:            get type of plot
        !>                          fitlog:                 path for log-file containing the current values of chi**2
        !>                          NumberInputFilesorg:    number of input files for the external model program
        !>                          NumberOutputFilesOrg:   number of output files for the external model program
        !>                          ParallelizationFlagorg: contains the number of processors used for parallelization
        !>                          JobIDorg:               job identification number
        !>                          MaxInputLinesOrg:       max number of lines in an input file
        !>                          MaxParameterOrg:        max number of parameters in a line of an input file
        !>                          RenormalizedChi2Org:    flag for using renormalized chi**2
        !>                          currentpathorg:         path of the working directory
        !>                          FitParameterNameLocal:  array containing the names of the model parameters
        !>                          FitParameterValueLocal: array containing the values of the model parameters as string
        !>                          CalculationMethodOrg:   method of computation (at once or point-to-point)
        !>                          xAxisLabel:             label of the x-axis (for plot)
        !>                          yAxisLabel:             label of the y-axis (for plot)
        !>                          zAxisLabel:             label of the z-axis (for plot)
        !>                          PathStartScriptOrg:     path and name of the start script for calling model function
        !>                          ExeCommandStartScriptOrg:   command for calling model function
        !>                          parametersetorg:        the complete set of paramters (incl. flags and limits)
        !>                          expdataxorg:            array containing the experimental x side
        !>                          expdatayorg:            array containing the experimental y side
        !>                          expdataerrororg:        array containing the experimental error of the y side
        !>                          NumberRangesOrg:        number of y-columns for each experimental file
        !>                          MinRangeOrg:            array containing the minimal exp. ranges
        !>                          MaxRangeOrg:            array containing the maximal exp. ranges
        !>                          NumberXColumnsOrg:      number of x-columns for each experimental file
        !>                          NumberYColumnsOrg:      number of y-columns for each experimental file
        !>                          lengthexpdataorg:       number of lines in experimental data
        !>                          MaxRangeNumber:         max. number of ranges
        !>                          NumFileOrg:             number of experimental files
        !>                          MaxLengthOrg:           max length of experimental data
        !>                          MaxColXOrg:             number of columns concerning to the experimental x side
        !>                          MaxColYOrg:             number of columns concerning to the experimental y side
        !>                          parameternum:           number of model parameter
        !>
        !> output variables:        calstatus:              status flag of calculation (= 0: all ok)
        !>                          FitFunctionOut:         values of the model function at the calculated points
        !>                          Chi2Values:             values of the chi^2 function at the calculated points
        !>                          FinalParameterSet:      the complete set of paramters (incl. flags and limits)
        !>
        subroutine MainAlg(printflagNum, LastAlgorithmNum, calstatus, FitFunctionOut, Chi2Values, chilm, NumberOfFitAlgorithms, numiter, &
                           LevenbergMarquardtCounter, NumberSites, GeneralAlgorithmSettings, DeterminationChi2, PlotIteration, PlotType, fitlog, &
                           NumberInputFilesorg, NumberOutputFilesOrg, ParallelizationFlagorg, JobIDorg, MaxInputLinesOrg, MaxParameterOrg, &
                           RenormalizedChi2Org, currentpathorg, FitParameterNameLocal, FitParameterValueLocal, CalculationMethodOrg, xAxisLabel, &
                           yAxisLabel, zAxisLabel, PathStartScriptOrg, ExeCommandStartScriptOrg, parametersetorg, FinalParameterSet, expdataxorg, &
                           expdatayorg, expdataerrororg, NumberRangesOrg, MinRangeOrg, MaxRangeOrg, NumberXColumnsOrg, NumberYColumnsOrg, &
                           lengthexpdataorg, MaxRangeNumber, NumFileOrg, MaxLength, MaxColXOrg, MaxColYOrg, parameternum)


            implicit none
            ! ********** input variables **********
            integer :: parameternum                                                         !< number of model parameter
            integer :: NumberOfFitAlgorithms                                                !< total number of all algorithms in the chain
            integer :: numiter                                                              !< max. number of iterations
            integer :: NumFileOrg                                                           !< number of experimental files
            integer, dimension(NumFileOrg) :: lengthexpdataorg                              !< number of lines in experimental data
            integer, dimension(NumFileOrg) :: NumberXColumnsOrg                             !< number of x-columns for each experimental file
            integer, dimension(NumFileOrg) :: NumberYColumnsOrg                             !< number of y-columns for each experimental file
            integer, dimension(NumFileOrg) :: NumberRangesOrg                               !< number of y-columns for each experimental file
            integer :: MaxColXOrg                                                           !< number of columns concerning to the experimental x side
            integer :: MaxColYOrg                                                           !< number of columns concerning to the experimental y side
            integer :: MaxLength                                                            !< max length of experimental data
            integer :: LastAlgorithmNum                                                     !< flag for screen output 1 (=yes) or 0 (=no)
            integer :: printflagNum                                                         !< flag for screen output 1 (=yes) or 0 (=no)
            integer :: DeterminationChi2                                                    !< method being used for the determination of chi^2
            integer :: PlotIteration                                                        !< plot model func. for each iteration set 1 (=yes) or 0 (=no)
            integer :: PlotType                                                             !< get type of plot
            integer :: NumberInputFilesorg                                                  !< number of input files for the external model program
            integer :: NumberOutputFilesOrg                                                 !< number of output files for the external model program
            integer :: ParallelizationFlagorg                                               !< contains the number of processors used for parallelization
            integer :: JobIDorg                                                             !< job identification number
            integer :: MaxInputLinesOrg                                                     !< max number of lines in an input file
            integer :: MaxParameterOrg                                                      !< max number of parameters in a line of an input file
            integer :: RenormalizedChi2Org                                                  !< flag for using renormalized chi**2
            integer :: LevenbergMarquardtMethod                                             !< flag for using renormalized chi**2
            integer :: LevenbergMarquardtCounter                                            !< counts number of calls
            integer :: NumberSites                                                          !< number of best sites
            integer :: MaxRangeNumber                                                       !< max. number of ranges
            real*8 :: chilm                                                                 !< user defined abort criteria for chi**2
            real*8 :: VariationValue                                                        !< user defined value of variation (for gradient)
            real*8, dimension(15) :: GeneralAlgorithmSettings                               !< special algorithm settings
            real*8, dimension(NumFileOrg, MaxLength, MaxColXOrg) :: expdataxorg             !< array containing the experimental x side
            real*8, dimension(NumFileOrg, MaxLength, MaxColYOrg) :: expdatayorg             !< array containing the experimental y side
            real*8, dimension(NumFileOrg, MaxLength, MaxColYOrg) :: expdataerrororg         !< array containing the experimental error of the y side
            real*8, dimension(NumFileOrg, MaxRangeNumber, MaxColXOrg) :: MinRangeOrg        !< array containing the minimal exp. ranges
            real*8, dimension(NumFileOrg, MaxRangeNumber, MaxColXOrg) :: MaxRangeOrg        !< array containing the maximal exp. ranges
            character(len=8192) :: fitlog                                                   !< path for log-file containing the current values of chi**2
            character(len=256) :: xAxisLabel                                                !< label of the x-axis (for plot)
            character(len=256) :: yAxisLabel                                                !< label of the y-axis (for plot)
            character(len=256) :: zAxisLabel                                                !< label of the z-axis (for plot)
            character(len=20) :: CalculationMethodOrg                                       !< method of computation
            character(len=8192) :: PathStartScriptOrg                                       !< command for calling model function
            character(len=8192) :: ExeCommandStartScriptOrg                                 !< command for calling model function
            character(len=8192) :: currentpathorg                                           !< path of the working directory
            character(len=512), dimension(parameternum) :: FitParameterNameLocal            !< array containing the names of the model parameters
            character(len=512), dimension(parameternum) :: FitParameterValueLocal           !< array containing the values of the model parameters as
                                                                                            !< string

            ! ********** in/output variables **********
            real*8, dimension(4, parameternum) :: parametersetorg                           !< the non-optimized (initial parameter set)


            ! ********** output variables **********
            integer :: calstatus                                                            !< the following line is necessary for f2py
            real*8, dimension(NumberSites, parameternum) :: FinalParameterSet               !< array containing the optimized parameter set
            real*8, dimension(NumberSites, NumFileOrg, MaxLength, MaxColYOrg) :: FitFunctionOut !< values of the model function at the calculated points
            real*8, dimension(NumberSites, NumFileOrg, MaxLength, MaxColYOrg) :: Chi2Values     !< values of the model function at the calculated points


            ! ********** working variabels **********
            integer :: i, j, k, ii, jj                                                      !< working variables
            integer :: length                                                               !< total number of exp. data points
            integer :: LM_method                                                            !< method of Levenberg-Marquardt algorithm
            integer :: ok                                                                   !< status of calculation
            integer :: NumInputFiles                                                        !< needed for loop over input files
            integer :: allocstatus, deallocstatus                                           !< working variables for allocation/deallocation
            integer, dimension(8) :: VALUES                                                 !< value for the date_and_time subroutine
            real*8 :: chilim                                                                !< lower limit of chi**2 normalized to the total number of
                                                                                            !< all data points
            real*8, dimension(parameternum) :: errors                                       !< array containing the errors of the optimized model param.
            character(len=512) :: fitlogparam                                               !< path for log-file containing the current parameter values
            character(len=512) :: fitlogChi2                                                !< path for log-file containing chi**2 and the corresponding
                                                                                            !< parameter values
            character(len=8) :: DATE                                                        !< variable for the date_and_time subroutine
            character(len=10) :: TIME                                                       !< variable for the date_and_time subroutine
            character(len=5) :: ZONE                                                        !< variable for the date_and_time subroutine
            character(len=10) :: Number1                                                    !< variable for number to string converting
            character(len=8192) :: SiteExt, NumExt, BaseDir, FuncCallExt                    !< working variables for final input file name


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< set print flag and last algorithm flag
            calstatus = 0                                                                   !< set calculation status to 0 = everything is ok
            if (printflagNum == 1) then                                                     !< set printflag
                printflag = .true.
            else
                printflag = .false.
            endif
            if (LastAlgorithmNum == 1) then
                LastAlgorithmFlag = .true.
            else
                LastAlgorithmFlag = .false.
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy contents of some input variables to module variables
            NumberExpFiles = NumFileOrg                                                     !< copy number of experimental files to global variable
            MaxExpLines = MaxLength                                                         !< copy max. number of lines in exp file(s) to global param.
            MaxExpLength = MaxLength                                                        !< copy max. number of exp. data points
            MaxNumberRanges = MaxRangeNumber                                                !< copy of max. number of data ranges in a exp. data file
            fitlogGlobal = fitlog                                                           !< copy path of log-file to global param.
            xAxisLabelGlobal = xAxisLabel                                                   !< copy label of x-axis to global param.
            yAxisLabelGlobal = yAxisLabel                                                   !< copy label of y-axis to global param.
            zAxisLabelGlobal = zAxisLabel                                                   !< copy label of z-axis to global param.
            PlotIterationGlobal = PlotIteration                                             !< copy flag for plot for each iteration to global parameter
            PlotTypeGlobal = PlotType                                                       !< copy type of plot to global parameter
            NumIterGlobal = NumIter                                                         !< copy max. number of iterations to global parameter
            currentpath = trim(adjustl(currentpathorg))                                     !< copy path of working directory to module variable without
                                                                                            !< trailing and leading blanks
            MaxColX = MaxColXOrg                                                            !< copy number of columns of the experimental x data to
                                                                                            !< module variable
            MaxColY = MaxColYOrg                                                            !< copy number of columns of the experimental y data to
                                                                                            !< module variable
            parameternumber = parameternum                                                  !< copy input variable containing the number of parameters
                                                                                            !< to module variable
            DetChi2 = DeterminationChi2                                                     !< copy method of chi^2 determination
            NumberInputFiles = NumberInputFilesorg                                          !< copy number of input files for the external program to
                                                                                            !< global variable
            NumberOutputFiles = NumberOutputFilesOrg                                        !< copy number of output files for the external program to
                                                                                            !< global variable
            ParallelizationFlag = ParallelizationFlagorg                                    !< copy number of used processors to global variable
            JobID = JobIDorg                                                                !< copy job-ID number to global variable
            MaxInputLines = MaxInputLinesOrg                                                !< copy max number of input lines in an input file
            MaxParameter = MaxParameterOrg                                                  !< copy max number of parameters in a line of an input file
            RenormalizedChi2 = .true.                                                       !< define flag for using renormalized chi**2
            QualityLimit = 1                                                                !< set number of best sites to 1
            if (RenormalizedChi2Org /= 1) then                                              !< set renomalization flag
                RenormalizedChi2 = .false.
            endif
            PlotIterationFlag = .false.
            if (PlotIterationGlobal == 0) PlotIterationFlag = .true.


            !< get special algorithm settings
            LevenbergMarquardtMethod = int(GeneralAlgorithmSettings(2))
            VariationValue = GeneralAlgorithmSettings(3)
            GradientVariationValue = VariationValue                                         !< copy value of the variation (in percent) to global value

            ! Debug:
            !    print*,'LevenbergMarquardtMethod = ', LevenbergMarquardtMethod
            !    print*,'VariationValue = ', VariationValue
            !    print*,'PathStartScriptOrg = ',trim(PathStartScriptOrg)
            !    print*,'ExeCommandStartScriptOrg = ',trim(ExeCommandStartScriptOrg)
            !    print*,'FitFunctionOutOrg = ',trim(FitFunctionOutOrg)
            !    print*,'MaxLength = ',MaxLength
            !    print*,'NumFileOrg = ',NumFileOrg
            !    print*,'MaxColXOrg = ',MaxColXOrg
            !    print*,'MaxColYOrg = ',MaxColYOrg
            !    print*,'MaxInputLines = ',MaxInputLines
            !    print*,'MaxParameter = ',MaxParameter
            !    Do i=1,NumFileOrg
            !        print*,'    Experimental file: i = ',i
            !        print*,'    lengthexpdataorg(i) = ',lengthexpdataorg(i)
            !        print*,'    NumberYColumnsOrg(i) = ',NumberYColumnsOrg(i)
            !        print*,'    expdataxorg(i,1:5,1) = ',expdataxorg(i,1:5,1)
            !        print*,'    expdatayorg(i,1:5,1) = ',expdatayorg(i,1:5,1)
            !        print*,'    expdataerrororg(i,1:5,1) = ',expdataerrororg(i,1:5,1)
            !    end Do
            !    print*,'chilm = ',chilm
            !    print*,'numiter = ',numiter
            !    print*,'fitlog = ',trim(fitlog)
            !    print*,'currentpathorg = ',trim(adjustl(currentpathorg))
            !    print*,"parametersetorg(1,:) = ",parametersetorg(1,:)
            !    print*,"parametersetorg(2,:) = ",parametersetorg(2,:)
            !    print*,"parametersetorg(3,:) = ",parametersetorg(3,:)
            !    print*,"parametersetorg(4,:) = ",parametersetorg(4,:)
            !    print*,"RenormalizedChi2 = ",RenormalizedChi2
            !    return


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< set temp-directory
            TempDirectory = " "
            CALL GetEnv('MAGIXTempDirectory', TempDirectory)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< modify name of log file
            i = index(fitlog,"/",back = .true.)
            j = index(fitlog,".",back = .true.)
            Number1 = "          "
            write(Number1,'(I10)') abs(LevenbergMarquardtCounter)
            if (j > i) then
                if (NumberOfFitAlgorithms > 1) then
                    fitlog = trim(adjustl(fitlog(:j-1))) // "__LM__call_" // trim(adjustl(Number1)) // trim(adjustl(fitlog(j:)))
                else
                    fitlog = trim(adjustl(fitlog(:j-1))) // "__LM" // trim(adjustl(fitlog(j:)))
                endif
            else
                if (NumberOfFitAlgorithms > 1) then
                    fitlog = trim(adjustl(fitlog)) // "__LM__call_" // trim(adjustl(Number1)) // ".log"
                else
                    fitlog = trim(adjustl(fitlog)) // "__LM.log"
                endif
            endif

            ! Debug:
            !   print*,'>',trim(adjustl(fitlog)),'<'


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< open log file and write header
            open(logchannel,file = trim(fitlog), status='replace')
            write(logchannel,'(" ")')
            write(logchannel,'("log-file for Levenberg-Marquardt algorithm:")')
            write(logchannel,'(43("-"))')
            write(logchannel,'(" ")')


            !< get current local time and date and write to log-file
            call date_and_time(DATE, TIME, ZONE, VALUES)
            write(logchannel,'(" ")')
            write(logchannel,'("algorithm starts at Date: ",A2,".",A2,".",A4,",     Time: ",A2,":",A2,":",A2)') DATE(7:8),DATE(5:6),DATE(1:4), &
                                                                                                                TIME(1:2),TIME(3:4),TIME(5:6)
            write(logchannel,'(" ")')
            write(logchannel,'(" ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< open log-file for the parameter and write header
            fitlogparam = trim(fitlog)//".param"
            open(paramchannel,file = trim(fitlogparam), status='replace')
            write(paramchannel,'(" ")')
            write(paramchannel,'("log-file containing the actual values of the parameters used in the Levenberg-Marquardt algorithm:")')
            write(paramchannel,'(98("-"))')
            write(paramchannel,'(" ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< open file containing the values of chi**2 and the corresponding values of the parameters
            WriteChi2Flag = .true.
            NumberLinesChi2 = 0
            fitlogChi2 = trim(fitlog) // ".chi2"
            open(Chi2Channel,file = trim(fitlogChi2), status='replace')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< deallocate if necessary and print error message if necessay
            if (allocated(expdatax)) then
                deallocate(expdatax, expdatay, expdatae, posdatexp, lengthexpdata, NumberXColumns, NumberYColumns, FirstPointExpData, LastPointExpData, &
                           NumberRanges, MinRange, MaxRange, ExpData_reversed_flag, stat = deallocstatus)
                if (deallocstatus /= 0) then                                                !< is all ok?
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine MainAlg:")')
                    write(logchannel,'(2x,"Can not deallocate variables expdatax etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine MainAlg:")'
                    print '(2x,"Can not deallocate variables expdatax etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< allocate memory for variables, clear content of the variables and print error message if necessay
            allocate(expdatax(NumberExpFiles, MaxLength, MaxColX), expdatay(NumberExpFiles, MaxLength, MaxColY), &
                     expdatae(NumberExpFiles, MaxLength, MaxColY), posdatexp(MaxColX), lengthexpdata(NumberExpFiles), &
                     NumberXColumns(NumberExpFiles), NumberYColumns(NumberExpFiles), FirstPointExpData(NumberExpFiles, MaxColX), &
                     LastPointExpData(NumberExpFiles, MaxColX), NumberRanges(NumberExpFiles), MinRange(NumberExpFiles, MaxRangeNumber, MaxColX), &
                     MaxRange(NumberExpFiles, MaxRangeNumber, MaxColX), ExpData_reversed_flag(NumberExpFiles), stat = allocstatus)
            if (allocstatus /= 0) then                                                      !< is all ok?
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not allocate variables expdatax etc.")')
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not allocate variables expdatax,expdatay etc.")'
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("allocstatus = ",I4)',allocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            MaxRangeNumber = MaxRangeNumber - 1                                             !< get real value
            expdatax = 0.d0
            expdatay = 0.d0
            expdatae = 0.d0
            posdatexp = 0.d0
            lengthexpdata = 0
            NumberXColumns = 0
            NumberYColumns = 0
            FirstPointExpData = 1.d99
            LastPointExpData = -1.d99
            NumberRanges = 0
            MinRange = 0.d0
            MaxRange = 0.d0
            ExpData_reversed_flag = .false.


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy input variables to module variabels
            expdatax(:,:,:) = expdataxorg(:,:,:)
            expdatay(:,:,:) = expdatayorg(:,:,:)
            expdatae(:,:,:) = expdataerrororg(:,:,:)
            lengthexpdata = lengthexpdataorg
            NumberXColumns = NumberXColumnsOrg
            NumberYColumns = NumberYColumnsOrg
            CalculationMethod = CalculationMethodOrg
            PathStartScript = PathStartScriptOrg
            ExeCommandStartScript = ExeCommandStartScriptOrg
            NumberRanges = NumberRangesOrg
            MinRange = MinRangeOrg
            MaxRange = MaxRangeOrg


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine first and last point of each exp. data file
            Do i = 1, NumberExpFiles
                Do j = 1, lengthexpdata(i)
                    ii = 0
                    jj = 0
                    Do k = 1,NumberXColumns(i)
                        if (expdatax(i, j, k) <= FirstPointExpData(i, k)) then
                            ii = ii + 1
                        endif
                        if (expdatax(i, j, k) >= LastPointExpData(i, k)) then
                            jj = jj + 1
                        endif
                    end Do
                    if (ii == NumberXColumns(i)) then
                        FirstPointExpData(i, 1:NumberXColumns(i)) = expdatax(i, j, 1:NumberXColumns(i))
                    endif
                    if (jj == NumberXColumns(i)) then
                        LastPointExpData(i, 1:NumberXColumns(i)) = expdatax(i, j, 1:NumberXColumns(i))
                    endif
                end Do


                !< check output file starts with the highest x-column value interchange FirstPointOutputFile and LastPointOutputFile
                ii = 0
                Do k = 1, NumberXColumns(i)
                    if (expdatax(i, 1, k) >= expdatax(i, lengthexpdata(i), k)) then
                        ii = ii + 1
                    endif
                end Do
                if (ii == NumberXColumns(i)) then
                    ExpData_reversed_flag(i) = .true.
                endif

                ! Debug:
                ! print*,' '
                ! print*,'File = ',i
                ! print*,'FirstPointExpData(i, 1:NumberXColumns(i)) = ', FirstPointExpData(i, 1:NumberXColumns(i))
                ! print*,'LastPointExpData(i, 1:NumberXColumns(i)) = ', LastPointExpData(i, 1:NumberXColumns(i))
                ! print*,'##########################################'
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy number of parameters required for the model function to working variable
            nfit = int(sum(parametersetorg(2, :)))                                          !< number of parameters which should be optimized


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< dimension of the arrays 'ModelFunction'
            i = 0
            Do k = 1, NumberExpFiles
                i = i + (NumberYColumns(k) * lengthexpdata(k))
            end Do
            j = nfit


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< deallocate/allocate memory for some working variables, clear contents of the variables and print error message if necessary
            if (allocated(paramset)) then
                deallocate(paramset, currentparmval, ia, ConverterInfit, FitParameterName, FitParameterValue, ModelFunction, chisqValues, &
                           AtOnceFunction, BestSitesParamSet, BestSitesModelValues, BestSitesChi2Values, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine MainAlg:")')
                    write(logchannel,'(2x,"Can not deallocate variables paramset etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine MainAlg:")'
                    print '(2x,"Can not deallocate variables paramset etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif
            allocate(paramset(4, parameternumber), currentparmval(parameternumber), ia(parameternumber), ConverterInfit(nfit), &
                     FitParameterName(parameternumber), FitParameterValue(parameternumber), &
                     ModelFunction(nfit + 1, NumberExpFiles, MaxColY, MaxLength), chisqValues(0:0), &
                     AtOnceFunction(0:ParallelizationFlag - 1, NumberExpFiles, MaxColY, MaxLength), &
                     BestSitesParamSet(QualityLimit, nfit + 1), BestSitesModelValues(QualityLimit, NumberExpFiles, MaxLength, MaxColY), &
                     BestSitesChi2Values(QualityLimit, NumberExpFiles, MaxLength, MaxColY), stat = allocstatus)
            if (allocstatus /= 0) then
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not allocate variables paramset etc.")')
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not allocate variables paramset etc.")'
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("allocstatus = ",I4)',allocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            paramset = 0.d0
            currentparmval = 0.d0
            ia = .false.
            ConverterInfit = 0
            FitParameterName = FitParameterNameLocal
            FitParameterValue = FitParameterValueLocal
            ModelFunction = 0.d0
            chisqValues = 0.d0
            BestSitesParamSet = 0.d0
            BestSitesParamSet(:,1) = 1.d99
            BestSitesModelValues = 0.d0
            BestSitesChi2Values = 0.d0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy parameter set to module variable
            paramset = parametersetorg


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< write registration mask for the fit model
            if (printflag) print '(9x, "Writing registration mask for the fit model .. ", $)'
            call RegistrationMask(ok)
            if (ok /= 0) then
                return
            endif
            if (printflag) print '("done!")'


            !< define ia variable
            ConverterInfit = 0
            ia = .false.
            k = 0
            Do i = 1, parameternumber
                if (paramset(2, i) == 1) then
                    k = k + 1
                    ia(i) = .true.
                    ConverterInfit(k) = i
                endif
            end Do
            NumberFreeParameter = int(sum(paramset(2, :)))                                  !< determine number of free parameter


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize model program
            call ModelInit


            !< write on the screen what you Do ..
            if (printflag) then
                print '(" ")'
                print '(" ")'
                write(Number1,'(I10)') JobID                                                !< write JobID to string
                print '(9x,"Temporary files are stored in: ",A)', trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(Number1)) // "/"
                print '(" ")'
                print '(" ")'
                print '(9x,"Start Levenberg-Marquardt algorithm (", A, " version) ..")', trim(adjustl(ParallelizationMethod))
                print '(" ")'
            endif
            write(logchannel,'(11x,"Start Levenberg-Marquardt algorithm (", A, " version) ..")') trim(adjustl(ParallelizationMethod))
            write(logchannel,'(" ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine total number of exp. data points
            length = 0
            Do i = 1, NumberExpFiles
                length = length + (NumberYColumns(i) * lengthexpdata(i))
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< lower limit of chi^2 for stopping condition normalized to the number of calculation points
            chilim = 0.E0
            if (RenormalizedChi2) then
                Do i = 1, NumberExpFiles
                    chilim = chilim + (NumberYColumns(i) * lengthexpdata(i) - parameternumber) * dabs(chilm)
                end Do
                chilim = dabs(chilim)
                write(logchannel,'(11x,"Renormalized limit for chi^2 = ", ES25.15)') chilim
                write(logchannel,'(" ")')
                write(logchannel,'(" ")')
                if (printflag) then
                    print '(" ")'
                    print '(11x,"Renormalized limit for chi^2 = ",ES20.10)', chilim
                    print '(" ")'
                    print '(" ")'
                endif
            else
                chilim = dabs(chilm)
                write(logchannel,'(11x,"Limit for chi^2 = ", ES25.15)') chilim
                write(logchannel,'(" ")')
                write(logchannel,'(" ")')
                if (printflag) then
                    print '(" ")'
                    print '(11x,"Limit for chi^2 = ",ES20.10)', chilim
                    print '(" ")'
                    print '(" ")'
                endif
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize variables for fit process
            Gradientflag = .true.                                                           !< we need the gradient of the function here
            currentparmval = paramset(1, 1:parameternumber)                                 !< copy the values of the parameters to another array
            FitFunctionOut = 0.d0                                                           !< reset FitFunctionOut array
            Chi2Values = 0.d0                                                               !< reset Chi2Values array
            errors = 0.d0                                                                   !< reset errors array
            UseCalculationReduction = .false.                                               !< no calculation reduction
            CurrentNumberLinesCalcReduction = 0                                             !< reset CurrentNumberLinesCalcReduction variable


            !< call subroutine to write current values of the parameter to file
            write(paramchannel,'("  ")')
            write(paramchannel,'("  ")')
            write(paramchannel,'("Input file with start values of all parameters:")')
            write(paramchannel,'(46("-"))')
            write(paramchannel,'("  ")')


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< call Levenberg-Marquardt algorithm
            LM_method = LevenbergMarquardtMethod


                !< print what you Do
                if (printflag) then
                    print '(11x,"Using MINACK version!")'
                    print '(" ")'
                    print '(" ")'
                    print '(11x,"Iteration:",20x,"chi^2:",5x,"Parameter:")'
                    print '(11x,"Initialize model function ..",20(" "),A1,$ )',char(13)
                endif
                write(logchannel,'(11x,"Using MINACK version!")')
                write(logchannel,'(" ")')
                write(logchannel,'(" ")')
                write(logchannel,'(11x,"Iteration:",20x,"chi^2:",5x,"Parameter:")')
                call call_LevenbergMarquardt_MINPACK(length, nfit, chilim)




            !< save best fit results
            k = 0
            FinalParameterSet = 0.d0
            Do i = 1, parameternumber
                if (ia(i)) then
                    k = k + 1
                    parametersetorg(1, i) = BestSitesParamSet(1, k + 1)
                endif
            end Do
            FitFunctionOut(1, :, :, :) = BestSitesModelValues(1, :, :, :)
            Chi2Values(1, :, :, :) = BestSitesChi2Values(1, :, :, :)
            FinalParameterSet(1, :) = parametersetorg(1, :)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< close file
            call SortChi2File(nfit, parameternumber, ia, currentparmval)
            close(Chi2Channel)
            close(paramchannel)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< copy current values of the model parameter to variable paramset
            paramset(1, 1:parameternumber) = currentparmval
            paramset(2, 1:parameternumber) = errors
            parametersetorg = paramset

            ! Debug:
            !    print*,'paramset = ', paramset
            !    print*,'FinalParameterSet(1, :) = ', FinalParameterSet(1, :)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< write final parameter sets for each site to final input files


            !< define base directory, i.e. path where final input file is written to
            k = index(fitlog, '/', back = .true.)
            if (k == 0) then
                BaseDir = ""
            else
                BaseDir = trim(adjustl(fitlog(:k)))
            endif


            !< define file name extension for number of algorithm call
            write(Number1,'(I10)') abs(LevenbergMarquardtCounter)
            FuncCallExt = "__LM__call_" // trim(adjustl(Number1))


            !< write files
            SiteExt = ".out"


            !< write parameter sets to file
            Do NumInputFiles = 1, NumberInputFiles                                          !< loop over all input files
                NumExt = trim(adjustl(FitFktInput(NumInputFiles)))
                k = index(NumExt, '/', back = .true.)
                if (k > 0) then                                                             !< we only need the file name
                    NumExt = trim(adjustl(NumExt(k:)))
                endif
                j = index(NumExt, '.', back = .true.)
                if (j > 1) then                                                             !< remove file name extension
                    NumExt = trim(adjustl(NumExt(:j - 1)))
                endif

                ! Debug:
                ! print*,"Site = ", i
                ! print*,"Nr. Final input file = ", NumInputFiles
                ! print*,"Final input file = ", trim(adjustl(BaseDir)) // trim(adjustl(NumExt)) // trim(adjustl(FuncCallExt)) // trim(adjustl(SiteExt)) &
                !                               // ".input"


                !< write parameter sets to file
                open(235,file = trim(adjustl(BaseDir)) // trim(adjustl(NumExt)) // trim(adjustl(FuncCallExt)) // trim(adjustl(SiteExt)) // ".input")
                call WriteParameter(235, .true., MaxColX, posdatexp, parameternumber, currentparmval, NumInputFiles)
                close(235)
            end Do


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< write end of log-file and print message to screen
            call date_and_time(DATE, TIME, ZONE, VALUES)
            write(logchannel,*)
            write(logchannel,'("algorithm ends at Date: ",A2,".",A2,".",A4,",     Time: ",A2,":",A2,":",A2)') DATE(7:8),DATE(5:6),DATE(1:4), &
                                                                                                              TIME(1:2),TIME(3:4),TIME(5:6)
            write(logchannel,'(" ")')
            write(logchannel,'(150("-"))')
            close(logchannel)
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(9x,"Finished Levenberg-Marquardt algorithm!")'
                print '(" ")'
                print '(" ")'
            endif


            !< clear temp files
            open(113, file = "experimental_data.dat")
            open(112, file = "Fit-Function_Values.dat")
            open(111, file = "chi2_Values.dat")
            close(111, status = 'delete')
            close(112, status = 'delete')
            close(113, status = 'delete')

            ! Debug:
            !    print*,'lengthexpdata = ',lengthexpdata
            !    print*,'MaxColX = ',MaxColX
            !    print*,'MaxColY = ',MaxColY
            !    print*,'expdataxorg(1:5,1) = ',expdataxorg(1:5,1)
            !    print*,'expdatayorg(1:5,1) = ',expdatayorg(1:5,1)
            !    print*,'expdataerrororg(1:5,1) = ',expdataerrororg(1:5,1)
            !    print*,'chilm = ',chilm
            !    print*,'numiter = ',numiter
            !    print*,'fitlog = ',trim(fitlog)
            !    print*,'model = ',trim(model)
            !    print*,'currentpathorg = ',currentpathorg
            !    print*,'parametersetorg(1,:) = ',parametersetorg(1,:)
            !    print*,'parametersetorg(2,:) = ',parametersetorg(2,:)
            !    print*,'parametersetorg(3,:) = ',parametersetorg(3,:)
            !    print*,'parametersetorg(4,:) = ',parametersetorg(4,:)


            !< send plot program the kill signal
            if (PlotIteration == 0) then
                call system("kill -9 " // trim(adjustl(PlotPID)) // " 1>/dev/null 2>&1")
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< free memory of general Levenberg-Marquardt variables
            if (allocated(posdatexp)) deallocate(posdatexp, stat = deallocstatus)
            if (allocated(currentparmval)) deallocate(currentparmval, stat = deallocstatus)
            if (allocated(ia)) deallocate(ia, stat = deallocstatus)




            !< free memory of model variables
            call ModelParamFree(deallocstatus)
            if (deallocstatus /= 0) then
                write(logchannel,*)
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not deallocate expdatax etc.")')
                write(logchannel,*)
                write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not deallocate variables expdatax etc.")'
                print '(" ")'
                print '("deallocstatus = ",I4)',deallocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            return
        end subroutine MainAlg
end Module Algorithm
!*********************************************************************************************************************************************************


