#!/usr/bin/env python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  Error estimator of parameter values based on Interval Nested Sampling algorithm
##  Copyright (C) 2009 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following subroutines and functions are included in this module:
##
##------- ModelFunctionCallClass: ------------------------------------------------------------------------------------------------------------------------
##      - subroutine ModelFunctionCallClass.__init__:   initialize class ModelFunctionCallClass
##      - subroutine ModelFunctionCallClass.logLhood:   call of model-function
##      - subroutine ModelFunctionCallClass.GetBestResult:                  get best result
##      - subroutine ModelFunctionCallClass.GetBestFunctionValues:          get function values of best result
##      - subroutine ModelFunctionCallClass.GetCovarianceMatrix:            get covariance matrix
##      - subroutine ModelFunctionCallClass.GetPoints:                      get points, i.e. list of all chi2 values
##      - subroutine ModelFunctionCallClass.WriteLogLines:                  write informatons of current iteration to log files
##      - subroutine ModelFunctionCallClass.SetCurrentParameterIndex:       set parameter CurrentParameterIndex
##      - subroutine ModelFunctionCallClass.SaveChi2Distribution:           get parameter values and the corresponding chi2 values for current parameter
##      - subroutine ModelFunctionCallClass.CalcErrorModelFunctionValues:   call of model-function for a given parameter set
##      - subroutine ModelFunctionCallClass.extract_distribution:           subroutine extracts the chi^2 distribution for each parameter from the
##                                                                          Error Estimation algorithm log file .log.chi2
##------- error estimation using INS: --------------------------------------------------------------------------------------------------------------------
##      - subroutine check_point_paral:                 Checking of point from the list Vectors (evaluated or not yet): if -yes- that don't evaluate
##                                                      (function value from the list -Points-)
##      - subroutine my_hist1:                          Create histigramms for each parameter after Error estimation using matplotlib
##      - subroutine del_chi2:                          Calculation of quantiles of chi2-distribution on dependence of the number of free parameter
##                                                      (np) for 1-sigma, 2-sigma and 3-sigma contours
##      - subroutine error_info:                        Get information about error for variable parameter (nvar)
##      - subroutine bisection:                         Bisection of parametric space (par_space) for k-coordinate direction
##      - subroutine interval_division:                 Division of num_par-dimension parameter space to subdomains with bisection
##      - subroutine calc_mid_point:                    Calculation of center point of the interval <par_space>
##      - subroutine inclusion_function:                Calculation of inclusion function: return inclusion - interval F1 = (minimal value, maximal value)
##      - subroutine volume_box:                        Calculation of volume of interval box and return ratio (volume_box/whole_volume)
##      - subroutine calc_cen_point:                    Calculation of center point of the interval par_space
##      - subroutine interval_method:                   Interval analysis for part of space - par_space
##      - subroutine explore:                           Evolve object within function constraint contentsEvolve object within function constraint contents
##                                                      interval method for analysing of optimization function
##      - subroutine plus(x,y):                         logarithmic addition log(exp(x)+exp(y))
##      - subroutine ndim_nest:                         This is an implementation of John Skilling's Nested Sampling algorithm for computing the
##                                                      normalizing constant of a probability distribution (usually the posterior in Bayesian inference).
##      - subroutine loglike_nest:                      Initial List of working objects
##      - subroutine Interval_NS:                       Return for ONE parameter --- mean value, standard deviation and Paramset for histogramm
##      - subroutine ErrorEstimINS:                     Error estimator using INS
##------- error estimation using MCMC: -------------------------------------------------------------------------------------------------------------------
##      - subroutine modeList:                          calculate mode(s) of list
##      - subroutine ErrorEstimMCMC:                    Error estimator using MCMC
##------- error estimation using Fisher-matrix: ----------------------------------------------------------------------------------------------------------
##      - subroutine ErrorEstimFisher:                  Error estimator using Fihser-matrix
##------- central subroutine: ----------------------------------------------------------------------------------------------------------------------------
##      - subroutine ErrorEstimator:                    main subroutine for Error estimator
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  I. Bernst     04.11.2011   Working version
##  T. Moeller    04.11.2011   combination with ModelFunctionCall package
##  I. Bernst     29.11.2011   Update
##  T. Moeller    16.01.2012   improve documentation of source code
##  T. Moeller    25.08.2013   bug fix, calculate model function and chi2 function for upper and lower parameter error values, chi2 distribution
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##******************************************************************** load packages *********************************************************************
from math import *                                                                          ## load math package
from scipy import stats                                                                     ## load scipy.stats package
from scipy.special import (erfinv, ndtr)                                                    ## load scipy.special package
from copy import deepcopy                                                                   ## load deepcopy package
import numpy                                                                                ## load numpy package
import random                                                                               ## load random package
import sys                                                                                  ## load sys package
import time                                                                                 ## load time package
import os                                                                                   ## load os package
import matplotlib                                                                           ## load python package for plotting matplotlib
# matplotlib.rcParams['backend'] = 'Agg'                                                      ## Do not remove the "#" sign! Important line for matplotlib
from matplotlib.ticker import MaxNLocator
import pylab                                                                                ## load python package for plotting pylab
import FittingEngine                                                                        ## import package containing the fitting engine
##********************************************************************************************************************************************************


##********************************************************************************************************************************************************
## class containing main subroutines
class ModelFunctionCallClass:


    ##****************************************************************************************************************************************************
    ## initialize class variables
    def __init__(self, printflagNumIn, GeneralAlgorithmSettingsIn, LastAlgorithmNumIn, DeterminationChi2In, PlotIterationIn, \
                 PlotTypeIn, fit_logIn, NumberInputFilesIn, NumberOutputFilesIn, ParallelizationFlagIn, JobIDIn, MaxInputLinesIn, MaxParameterIn, \
                 RenormalizedChi2In, currentpathIn, CalculationMethodIn, xAxisLabelIn, yAxisLabelIn, zAxisLabelIn, PathStartScriptIn, \
                 ExeCommandStartScriptIn, parameter_setIn, ExpDataXIn, ExpDataYIn, ExpDataErrorIn, NumberRangesIn, MinRangeIn, MaxRangeIn, \
                 NumberXColumnsIn, NumberYColumnsIn, LengthExpRangeIn, MaxRangeNumberIn, NumberExpFilesIn, MaxLengthIn, MaxColXIn, MaxColYIn, \
                 NumberParameterIn, MPIFlagIn, MAGIXrootDirectoryIn, JobDirIn, SpecialAlgorithmSettingsIn, StarterExecutable, Chi2Channel, logchannel, \
                 paramchannel, MCMCBestSiteCounter):
            self.printflagNum = printflagNumIn
            self.GeneralAlgorithmSettings = deepcopy(GeneralAlgorithmSettingsIn)
            self.LastAlgorithmNum = LastAlgorithmNumIn
            self.DeterminationChi2 = DeterminationChi2In
            self.PlotIteration = PlotIterationIn
            self.PlotType = PlotTypeIn
            self.fitlog = fit_logIn
            self.NumberInputFiles = NumberInputFilesIn
            self.NumberOutputFiles = NumberOutputFilesIn
            self.ParallelizationFlag = ParallelizationFlagIn
            self.JobID = JobIDIn
            self.MaxInputLines = MaxInputLinesIn
            self.MaxParameter = MaxParameterIn
            self.RenormalizedChi2 = RenormalizedChi2In
            self.currentpath = currentpathIn
            self.CalculationMethod = CalculationMethodIn
            self.xAxisLabel = xAxisLabelIn
            self.yAxisLabel = yAxisLabelIn
            self.zAxisLabel = zAxisLabelIn
            self.PathStartScript = PathStartScriptIn
            self.ExeCommandStartScript = ExeCommandStartScriptIn
            self.parameter_set = deepcopy(parameter_setIn)
            self.ExpDataX = deepcopy(ExpDataXIn)
            self.ExpDataY = deepcopy(ExpDataYIn)
            self.ExpDataError = deepcopy(ExpDataErrorIn)
            self.NumberRanges = NumberRangesIn
            self.MinRange = deepcopy(MinRangeIn)
            self.MaxRange = deepcopy(MaxRangeIn)
            self.NumberXColumns = NumberXColumnsIn
            self.NumberYColumns = NumberYColumnsIn
            self.LengthExpRange = LengthExpRangeIn
            self.MaxRangeNumber = MaxRangeNumberIn
            self.NumberExpFiles = NumberExpFilesIn
            self.MaxLength = MaxLengthIn
            self.MaxColX = MaxColXIn
            self.MaxColY = MaxColYIn
            self.NumberParameter = NumberParameterIn
            self.MPIFlag = MPIFlagIn
            self.MAGIXrootDirectory = MAGIXrootDirectoryIn
            self.JobDir = JobDirIn
            self.SpecialAlgorithmSettings = deepcopy(SpecialAlgorithmSettingsIn)
            self.StarterExecutable = StarterExecutable
            self.Chi2Channel = Chi2Channel
            self.logchannel = logchannel
            self.paramchannel = paramchannel
            self.MCMCBestSiteCounter = MCMCBestSiteCounter
            self.BestChi2 = 1.e99
            self.BestFunctionValues = 0.0
            self.BestChi2Values = 0.0
            self.BestParameter = 0.0
            self.BestLogLine = ""
            self.BestInputFiles = ""
            self.FuncCallCounter = 0
            self.TempDir = str(os.environ.get('MAGIXTempDirectory',''))
            self.CovarianceMatrix = 0.0
            self.CurrentParameterIndex = (-1)
            self.Chi2DistributionParam = []
            self.Chi2DistributionChi2 = []
            self.CurrIter = (-1)                                                            ## turn initialization flag on
            self.Points = []
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## call of logLhood
    def logLhood(self, parameter_vector, number_param_set, GradientVariation):
        """
        input variables:    number_param_set:       total number of parameter
                            parameter_vector:       parameter vector
                            GradientVariation:      variation for gradient calculation

        output variables:   chi2value:              value of chi^2
        """

        # Debug:
        # print 'number_param_set = ', number_param_set
        # print 'parameter_vector = ', parameter_vector
        # print 'GradientFlag = ', GradientFlag


        ## reset output value
        chi2value = 0.0


        ## make parameter_vector always has the same length
        if (number_param_set == 1):
            parameter_vector = [parameter_vector]


        ## update parameter_set array and write new parameter vectors to file
        WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"
        NewParamFile = open(WorkingDirectory.strip() + "new-parameters.dat", 'w')
        for k in xrange(number_param_set):
            j = (-1)
            line = ""
            for i in xrange(self.NumberParameter):
                if (self.parameter_set[1][i] == 1):
                    j += 1
                    line += "   " + str(parameter_vector[k][j])
                else:
                    line += "   " + str(self.parameter_set[0][i])
            NewParamFile.write(line + "\n")
        NewParamFile.close()

        # Debug:
        # print 'ModelFunctionCall.startcall.__doc__ = ',ModelFunctionCall.startcall.__doc__
        # print 'self.printflagNum = ',self.printflagNum
        # print 'number_param_set = ',number_param_set
        # print 'self.DeterminationChi2 = ',self.DeterminationChi2
        # print 'self.PlotIteration = ', self.PlotIteration
        # print 'self.PlotType = ',self.PlotType
        # print 'self.NumberInputFiles = ',self.NumberInputFiles
        # print 'self.NumberOutputFiles = ',self.NumberOutputFiles
        # print 'self.ParallelizationFlag = ',self.ParallelizationFlag
        # print 'self.JobID = ',self.JobID
        # print 'self.MaxInputLines = ',self.MaxInputLines
        # print 'self.MaxParameter = ',self.MaxParameter
        # print 'self.RenormalizedChi2 = ',self.RenormalizedChi2


        ## write gradientflag
        GradientFlagFileName = WorkingDirectory.strip() + "gradientflag.dat"
        if (GradientVariation != 0.0):
            GradientFlag = True
            GradientFlagFile = open(GradientFlagFileName, 'w')
            GradientFlagFile.write(str(GradientVariation) + "\n")
            GradientFlagFile.close()
        else:
            GradientFlag = False
            cmdString = "rm -rf " + GradientFlagFileName
            os.system(cmdString)


        ## define dummy arguments
        chilm = 1.0
        NumberOfFitAlgorithms = 1
        AlgCounter = 1


        ## call model function package
        ParamSetCounter = number_param_set
        numiter = self.CurrIter
        NameOfAlgorithm = "Model-Function_Call"
        calstatus, FitFunctionOut, Chi2Values, parmCopy, FinalParameterSet = FittingEngine.StartAlgPackage(NameOfAlgorithm, self.StarterExecutable, \
                                                                                            self.MPIFlag, self.MAGIXrootDirectory, self.JobDir, \
                                                                                            self.JobID, self.NumberExpFiles, self.MaxLength, \
                                                                                            self.MaxColX, self.MaxColY, self.MaxRangeNumber, \
                                                                                            self.NumberParameter, ParamSetCounter, \
                                                                                            self.LastAlgorithmNum, numiter, self.ParallelizationFlag, \
                                                                                            self.DeterminationChi2, self.PlotIteration, self.PlotType, \
                                                                                            self.RenormalizedChi2, AlgCounter, chilm, \
                                                                                            self.GeneralAlgorithmSettings, \
                                                                                            self.SpecialAlgorithmSettings, self.parameter_set, \
                                                                                            self.xAxisLabel, self.yAxisLabel, self.zAxisLabel)
        self.CurrIter = 0                                                                   ## turn initialization flag off


        ## get calculated chi2 values from temp file
        WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"
        ResultFile = open(WorkingDirectory.strip() + "chi2Values.dat", 'r')
        ResultFileContents = ResultFile.readlines()
        ResultFile.close()
        ValuesOfChi2 = []
        for val in ResultFileContents:
            ValuesOfChi2.append(float(val))

        # Debug:
        # print ">>>>ValuesOfChi2 = ", ValuesOfChi2


        ## check if parameter vector is within given range
        for i in xrange(number_param_set):                                                  ## loop over parameter sets
            k = (-1)
            for j in xrange(self.NumberParameter):                                          ## loop over all parameter
                if (self.parameter_set[1][j] == 1):
                    k += 1
                    value = float(parameter_vector[i][k])
                    if (value < float(self.parameter_set[2][j]) or value > float(self.parameter_set[3][j])):
                        ValuesOfChi2[i] = numpy.inf
                        FitFunctionOut[i] = 0.e0
                        Chi2Values[i] = 0.e0

        # Debug:
        # print '\nparameter_vector = ',parameter_vector
        # print ">>>>ValuesOfChi2 = ", ValuesOfChi2
        # print "self.CurrentParameterIndex = ", self.CurrentParameterIndex
        # sys.exit(0)


        ## get covariance matrix
        if (GradientFlag):
            NumberFitParameter = 0
            for k in xrange(self.NumberParameter):
                if (self.parameter_set[1][k] == 1):
                    NumberFitParameter += 1
            self.CovarianceMatrix = numpy.zeros((NumberFitParameter, NumberFitParameter), dtype = 'float')
            ResultFile = open(WorkingDirectory.strip() + "covariance-matrix.dat", 'r')
            ResultFileContents = ResultFile.readlines()
            ResultFile.close()
            Counterlines = (-1)
            kk = (-1)
            for k in xrange(self.NumberParameter):
                if (self.parameter_set[1][k] == 1):
                    kk += 1
                ii = (-1)
                for i in xrange(self.NumberParameter):
                    if (self.parameter_set[1][i] == 1):
                        ii += 1
                    Counterlines += 1
                    if (self.parameter_set[1][k] == 1 and self.parameter_set[1][i] == 1):
                        self.CovarianceMatrix[kk, ii] = ResultFileContents[Counterlines]


        ## save chi2 value and corresponding error parameter to arrays
        if (self.CurrentParameterIndex > (-1)):
            for k, chi2value in enumerate(ValuesOfChi2):
                self.Chi2DistributionParam.append(parameter_vector[k][self.CurrentParameterIndex])
                self.Chi2DistributionChi2.append(chi2value)


        ## save best chi2
        WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"
        for i in xrange(number_param_set):


            ## append content of chi2-log file to total list of all calculated chi^2 values
            f = open(WorkingDirectory.strip() + "log_chi2_single_call.dat", 'r')
            contentChi2LogFile = f.readlines()
            for line in contentChi2LogFile:
                self.Chi2Channel.write(line)
                self.Chi2Channel.flush()
            f.close()


            ## current value of chi^2 better than saved result ?
            if (ValuesOfChi2[i] != numpy.inf and ValuesOfChi2[i] < self.BestChi2):
                self.BestChi2 = ValuesOfChi2[i]
                self.BestParameter = parameter_vector[i]
                self.BestFunctionValues = FitFunctionOut[i]
                self.BestChi2Values = Chi2Values[i]


                ## write best model function values to temp file
                for j in xrange(self.NumberExpFiles):
                    f = open(WorkingDirectory.strip() + "best_model_function_values_call__" + str(i + 1) + "__" + str(j + 1) + ".dat", 'w')
                    for lines in self.BestFunctionValues[j]:
                        f.write(str(lines[0]) + "\n")
                    f.close()


                ## write chi2 values to temp file
                for j in xrange(self.NumberExpFiles):
                    f = open(WorkingDirectory.strip() + "best_chi2_values_call__" + str(i + 1) + "__" + str(j + 1) + ".dat", 'w')
                    for lines in self.BestFunctionValues[j]:
                        f.write(str(lines[0]) + "\n")
                    f.close()


                ## get corresponding line with formatted parameters
                f = open(WorkingDirectory.strip() + "log_lines__single_call__" + str(i + 1) + ".dat", 'r')
                self.BestLogLine = f.readline()
                self.BestLogLine = self.BestLogLine[0:len(self.BestLogLine)-1]
                f.close()


                ## get corresponding contents of the input file(s)
                f = open(WorkingDirectory.strip() + "log_input-files__single_call__" + str(i + 1) + ".dat", 'r')
                self.BestInputFiles = f.readlines()
                f.close()


        ## print what you do
        self.FuncCallCounter += 1                                                           ## increase counter for model function calls


        ## store chi2 values
        for k in xrange(number_param_set):
            NewArray = []
            for param in parameter_vector[k]:
                NewArray.append(param)
            NewArray.append(ValuesOfChi2[k])
            self.Points.append(NewArray)


        ## define return variable
        ValuesOfChi2[:] = [x * (-1) for x in ValuesOfChi2]
        return ValuesOfChi2
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## get best result
    def GetBestResult(self):
        """
        input variables:    None

        output variables:   self.BestChi2:              chi2 value for best result
                            self.BestParameter:         parameter vector for best result
                            self.BestLogLine:           line in log file for best result
                            self.BestInputFiles:        content of input file for best result
        """


        ## define return parameters
        return self.BestChi2, self.BestParameter, self.BestLogLine, self.BestInputFiles
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## get function values of best result
    def GetBestFunctionValues(self):
        """
        input variables:    None

        output variables:   self.BestParameter:         parameter vector of best result
                            self.BestFunctionValues:    model function values of best result
                            self.BestChi2Values:        chi2 function values of best result
        """


        ## define return parameters
        return self.BestParameter, self.BestFunctionValues, self.BestChi2Values
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## get covariance matrix
    def GetCovarianceMatrix(self):
        """
        input variables:    None

        output variables:   self.CovarianceMatrix:  covariance matrix
        """


        ## define return parameters
        return self.CovarianceMatrix
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## set parameter SetCurrentParameterIndex
    def SetCurrentParameterIndex(self, paramvalue):
        """
        input variables:    paramvalue:             the new value for CurrentParameterIndex

        output variables:   None

        """
        self.CurrentParameterIndex = paramvalue
        self.Chi2DistributionParam = []
        self.Chi2DistributionChi2 = []


        ## we've done
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## get parameter values and the corresponding chi2 values for current parameter
    def SaveChi2Distribution(self):
        """
        input variables:    None

        output variables:   self.Chi2DistributionParam: parameter values for current parameter
                            self.Chi2DistributionChi2:  corresponding chi2 values for current parameter
        """


        ## define return parameters
        return self.Chi2DistributionParam, self.Chi2DistributionChi2
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## get points, i.e. list of all chi2 values
    def GetPoints(self):
        """
        input variables:    None

        output variables:   self.Points:            list of all chi2 values
        """


        ## define return parameters
        return self.Points
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## write informatons of current iteration to log files
    def WriteLogLines(self, CurrentIteration):
        """
        input variables:    CurrentIteration:       number of current iteration

        output variables:   None
        """


        ## print what you do
        bestchi, bestparam, bestline, bestinput = self.GetBestResult()
        chisq = bestchi
        if (self.printflagNum == 1):
            print "\r                %5d%s" % (CurrentIteration, bestline)


        ## wirte information to param-log file
        self.paramchannel.write("\n")
        self.paramchannel.write("\n")
        outputstring = "*" * 122
        self.paramchannel.write(outputstring + "\n")
        outputstring = "Iteration: %5d,  chi^2 = %25.15e" % (CurrentIteration, bestchi)
        self.paramchannel.write(outputstring + "\n")
        self.paramchannel.write("\n")
        self.paramchannel.write("\n")
        self.paramchannel.write("\n")
        self.paramchannel.write("Parameters: " + str(bestparam) + "\n")
        outputstring = "-" * 122
        self.paramchannel.write(outputstring + "\n")
        self.paramchannel.write("\n")
        self.paramchannel.flush()


        ## write contents of input file(s) to param-log file
        for lineinput in bestinput:
            self.paramchannel.write(lineinput)
            self.paramchannel.flush()
        self.paramchannel.flush()


        ## wirte information to log file
        outputstring = "                %5d%s" % (CurrentIteration, bestline)
        self.logchannel.write(outputstring + "\n")
        self.logchannel.flush()


        ## define return parameters
        return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## call of model-function for a given parameter set
    def CalcErrorModelFunctionValues(self, number_param_set, parameter_vector):
        """
        input variables:    number_param_set:       total number of parameter
                            parameter_vector:       parameter vector

        output variables:   FitFunctionOut:         model function values
                            Chi2Values:             the difference between experimental data and model function values at each data point
        """


        ## reset output value
        chi2value = 0.0


        ## update parameter_set array
        if (number_param_set == 1):
            dummyVector = deepcopy(parameter_vector)
            parameter_vector = []
            parameter_vector.append(dummyVector)
        parameter_vector_set = deepcopy(self.parameter_set)
        for k in xrange(number_param_set):
            j = (-1)
            for i in xrange(self.NumberParameter):
                if (self.parameter_set[1][i] == 1):
                    j += 1
                    parameter_vector_set[k][i] = parameter_vector[k][j]


        ## write new parameter vectors to file
        WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"
        NewParamFile = open(WorkingDirectory.strip() + "new-parameters.dat", 'w')
        for k in xrange(number_param_set):
            j = (-1)
            line = ""
            for i in xrange(self.NumberParameter):
                if (self.parameter_set[1][i] == 1):
                    j += 1
                    line += "   " + str(parameter_vector[k][j])
                else:
                    line += "   " + str(self.parameter_set[0][i])
            NewParamFile.write(line + "\n")
        NewParamFile.close()

        # Debug:
        # print 'ModelFunctionCall.startcall.__doc__ = ',ModelFunctionCall.startcall.__doc__
        # print 'self.printflagNum = ',self.printflagNum
        # print 'number_param_set = ',number_param_set
        # print 'self.DeterminationChi2 = ',self.DeterminationChi2
        # print 'self.PlotIteration = ', self.PlotIteration
        # print 'self.PlotType = ',self.PlotType
        # print 'self.NumberInputFiles = ',self.NumberInputFiles
        # print 'self.NumberOutputFiles = ',self.NumberOutputFiles
        # print 'self.ParallelizationFlag = ',self.ParallelizationFlag
        # print 'self.JobID = ',self.JobID
        # print 'self.MaxInputLines = ',self.MaxInputLines
        # print 'self.MaxParameter = ',self.MaxParameter
        # print 'self.RenormalizedChi2 = ',self.RenormalizedChi2


        ## write gradientflag
        GradientFlag = False
        GradientFlagFileName = WorkingDirectory.strip() + "gradientflag.dat"
        cmdString = "rm -rf " + GradientFlagFileName
        os.system(cmdString)


        ## define dummy arguments
        chilm = 1.0
        NumberOfFitAlgorithms = 1
        AlgCounter = 1
        numiter = 1


        ## call model function package
        ParamSetCounter = number_param_set
        NameOfAlgorithm = "Model-Function_Call"
        calstatus, FitFunctionOut, Chi2Values, parmCopy, FinalParameterSet = FittingEngine.StartAlgPackage(NameOfAlgorithm, self.StarterExecutable, \
                                                                                            self.MPIFlag, self.MAGIXrootDirectory, self.JobDir, \
                                                                                            self.JobID, self.NumberExpFiles, self.MaxLength, \
                                                                                            self.MaxColX, self.MaxColY, self.MaxRangeNumber, \
                                                                                            self.NumberParameter, ParamSetCounter, \
                                                                                            self.LastAlgorithmNum, numiter, self.ParallelizationFlag, \
                                                                                            self.DeterminationChi2, self.PlotIteration, self.PlotType, \
                                                                                            self.RenormalizedChi2, AlgCounter, chilm, \
                                                                                            self.GeneralAlgorithmSettings, \
                                                                                            self.SpecialAlgorithmSettings, parameter_vector_set, \
                                                                                            self.xAxisLabel, self.yAxisLabel, self.zAxisLabel)
        ## get calculated chi2 values from temp file
        WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"
        ResultFile = open(WorkingDirectory.strip() + "chi2Values.dat", 'r')
        ResultFileContents = ResultFile.readlines()
        ResultFile.close()
        ValueOfChi2 = []
        for val in ResultFileContents:
            ValueOfChi2.append(float(val))

        # Debug:
        # print "ValueOfChi2 = ", ValueOfChi2


        ## define return parameters
        return (FitFunctionOut, Chi2Values)
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##****************************************************************************************************************************************************
    ## subroutine extracts the chi^2 distribution for each parameter from the Error Estimation algorithm log file .log.chi2
    def extract_distribution(self, num_par, NameOfLogFile, plotflag, ErrINSCounter, param_up_down, min_point, Error_left, Error_right, \
                             AllChi2DistributionParamValues, AllChi2DistributionChi2Values):
        """
        input variables:    num_par:                number of free parameters
                            NameOfLogFile:          path and name of chi2 log file
                            plotflag:               flag for plotting
                            ErrINSCounter:          counter for algorithm calling
                            param_up_down:          upper and lower limits for each free parameters
                            min_point:              point of minimum
                            Error_left:             left errors
                            Error_right:            right errors
                            AllChi2DistributionParamValues:
                            AllChi2DistributionChi2Values:

        output variables:   None
        """


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## plot distributions for each parameter


        ## create plot
        for i in xrange(num_par):                                                           ## loop over each distribution

            # Debug:
            # print "i = ", i
            # print "AllChi2DistributionParamValues[i] = ", AllChi2DistributionParamValues[i]
            # print "AllChi2DistributionChi2Values[i] =  ", AllChi2DistributionChi2Values[i]


            ## initialize figure window
            # ax1 = pylab.figure()
            ax1 = pylab.subplot(1, 1, 1)
            pylab.clf()


            ## sort list with parameter values and apply changes to chi2 list as well
            distr = zip(AllChi2DistributionParamValues[i], AllChi2DistributionChi2Values[i])
            sorted_distr = sorted(distr)
            paramvalues = [point[0] for point in sorted_distr]
            chi2values = [point[1] for point in sorted_distr]

            # Debug:
            # print "paramvalues = ", paramvalues
            # print "chi2values = ", chi2values


            ## make plot
            pylab.plot(paramvalues, chi2values, 'b.-')
            pylab.grid(True)
            pylab.xlabel("free parameter %i" % (i + 1))
            pylab.ylabel(r'$\chi^2$')
            pylab.xlim(param_up_down[i, 0], param_up_down[i, 1])


            ## add lines to plot indicating the point of minimum, the left and the right errors respectively
            pylab.axvline(x=min_point[i], color='k', linewidth=2, linestyle='-')
            pylab.axvline(x=(min_point[i] - Error_left[i]), color='r', linewidth=1, linestyle='--')
            pylab.axvline(x=(min_point[i] + Error_right[i]), color='r', linewidth=1, linestyle='--')


            ## save distribution plot to file
            fit_log = NameOfLogFile.strip()
            mm = fit_log.rfind("/")
            if (mm > (-1)):
                path_to_save_png = fit_log[:mm + 1]
            else:
                path_to_save_png = ""
            if (plotflag == "true"):
                pylab.savefig(path_to_save_png + "ErrorEstim_INS__chi2-distribution_of_free-parameter_parm_" + str(i + 1) + "__call_" \
                              + str(abs(ErrINSCounter)) + ".png", format='png')
                # pylab.show()
            elif (plotflag == "saveonly"):
                pylab.savefig(path_to_save_png + "ErrorEstim_INS__chi2-distribution_of_free-parameter_parm_" + str(i + 1) + "__call_" \
                              + str(abs(ErrINSCounter)) )
            pylab.draw()


            ## save data to ASCII file
            ASCIIOutputFile = open(path_to_save_png + "ErrorEstim_INS__chi2-distribution_of_free-parameter_parm_" + str(i + 1) + "__call_" \
                              + str(abs(ErrINSCounter)) + ".dat", 'w')
            OutputContents = []
            for i in xrange(len(paramvalues)):
                x = paramvalues[i]
                y = chi2values[i]
                XVal = "%.15e" % x
                YVal = "%.15e" % y
                NewLine = XVal + "\t\t" + YVal


                ## check, if current line is not already written to file
                NotAlreadyIncluded = "true"
                if (OutputContents != []):
                    for PrevLine in OutputContents:
                        if (PrevLine.strip() == NewLine.strip()):
                            NotAlreadyIncluded = "false"
                            break
                if (NotAlreadyIncluded == "true"):
                    OutputContents.append(NewLine)
                    ASCIIOutputFile.write(NewLine + "\n")
            ASCIIOutputFile.close()

            # Debug:
            # print "path_to_save_png = ", path_to_save_png


        ## we've done
        return
    ##--------------------------------------------------------------------------------------------------------------------------------------------------------


    ##********************************************************************************************************************************************************
    def extract_distribution_old(self, NameOfLogFile, plotflag, ErrINSCounter, param_up_down, min_point, Error_left, Error_right):
        """
        subroutine extracts the chi^2 distribution for each parameter from the Error Estimation algorithm log file .log.chi2

        input variables:    NameOfLogFile:          path and name of chi2 log file
                            plotflag:               flag for plotting
                            ErrINSCounter:          counter for algorithm calling
                            param_up_down:          upper and lower limits for each free parameters
                            min_point:              point of minimum
                            Error_left:             left errors
                            Error_right:            right errors

        output variables:   None
        """


        ## print what you do
        print "\tDetermine the chi^2 distribution for each parameter.."


        ## read in contents of log file
        print "\n\tRead in contents of log.chi2 file ", NameOfLogFile, "..",
        LogFile = open(NameOfLogFile, 'r')                                                  ## open log file (read only)
        LogFileContentsOrig = LogFile.readlines()                                           ## read in contents of log file
        LogFile.close()                                                                     ## close log file
        print "done!"


        ## remove first line and first column and build up new array
        print "\tConstruct reduced contents array ..",
        LogFileContents = []                                                                ## initialize array for new log file content
        CounterLines = 0                                                                    ## reset counter for lines
        for line in LogFileContentsOrig:                                                    ## loop over all lines of log file
            CounterLines += 1                                                               ## increase counter for lines
            if (CounterLines > 1):                                                          ## ignore first line
                StrippedLine = line.strip()                                                 ## remove leading and tailing blanks
                SplittedLine = StrippedLine.split()                                         ## split line into columns
                CounterColumns = 0                                                          ## reset counter for columns
                NewLine = []                                                                ## initialize working variable
                for column in SplittedLine:                                                 ## loop over all columns of current line
                    CounterColumns += 1                                                     ## increase counter for columns
                    if (CounterColumns > 1):                                                ## ignore first column
                        NewColumn = column.strip()                                          ## remove leading and tailing blanks
                        NewLine.append(NewColumn)                                           ## build up new line
                LogFileContents.append(NewLine)                                             ## add to new log file
        NumberLines = len(LogFileContents[:])                                               ## get total number of lines
        NumberParameters = len(LogFileContents[0]) - 1                                      ## get number of parameters
        print "done!"

        # Debug:
        # print "\tLogFileContents[0] = ", LogFileContents[0]
        # print "\tNumberLines = ", NumberLines
        # print "\tNumberParameters = ", NumberParameters


        ## analyze contents of log file
        print "\tExtract distribution for each parameter .."
        ListOfAllDistributions = []
        for NumParam in xrange(NumberParameters):                                           ## loop over each parameter
            CurrentDistributions = []                                                       ## reset array for the distribution of the current parameter


            ## print what you do
            print "\t\t Extract distribution for parameter %i " % (NumParam + 1)


            ## find all lines where the parameters are identical except the NumParam-th parameter
            FoundLines = []                                                                 ## initalize array for already found lines
            for LineNumOut in xrange(NumberLines):                                          ## outer loop over all lines in the log file

                # Debug:
                # print "LineNumOut = ", LineNumOut


                ## check if current line is used before
                try:
                    FoundLines.index(LineNumOut)
                except ValueError:                                                          ## continue here, if LineNumOut-th line is not used before
                    CounterLineDistr = 0
                    for LineNumIn in xrange(LineNumOut + 1, NumberLines):                   ## inner loop over all lines in the log file


                        ## check if current line is used before
                        try:
                            FoundLines.index(LineNumIn)
                        except ValueError:                                                  ## continue here, if LineNumIn-th line is not used before

                            # Debug:
                            # print "LineNumOut, LineNumIn = ", LineNumOut, LineNumIn


                            ## compare both lines
                            CounterIdenticalParam = 0
                            for NumParamCompare in xrange(NumberParameters):                ## loop over all parameters in the line
                                if (NumParamCompare != NumParam):                           ## ignore the NumParam-th parameter
                                    paramOut = LogFileContents[LineNumOut][NumParamCompare + 1].strip() ## get parameter in outer loop line
                                    paramIn = LogFileContents[LineNumIn][NumParamCompare + 1].strip()   ## get parameter in inner loop line
                                    if (paramOut == paramIn):                               ## do both parameters coincide?
                                        CounterIdenticalParam += 1                          ## if so, increase counter
                                    else:                                                   ## otherwise stop loop
                                        break


                            ## if both lines coincide add current line to distribution array
                            if (CounterIdenticalParam == (NumberParameters - 1)):           ## correspond the line of the inner loop to the line of the
                                                                                            ## outer loop?
                                CounterLineDistr += 1                                       ## increase counter for added lines
                                NewLine = [LogFileContents[LineNumIn][NumParam + 1], LogFileContents[LineNumIn][0]]
                                NewLine = [float(LogFileContents[LineNumIn][NumParam + 1]), float(LogFileContents[LineNumIn][0])]
                                CurrentDistributions.append(NewLine)
                                FoundLines.append(LineNumIn)

                                # Debug:
                                # print "chi^2 = ", LogFileContents[LineNumIn][0]
                                # print "parameter = ", LogFileContents[LineNumIn][NumParam + 1]
                                # print "LogFileContents[LineNumIn] = ", LogFileContents[LineNumIn]

                    if (CounterLineDistr > 0):
                        NewLine = [LogFileContents[LineNumOut][NumParam + 1], LogFileContents[LineNumOut][0]]
                        CurrentDistributions.insert(0, NewLine)
                        FoundLines.insert(0, LineNumOut)

            # Debug:
            # print "len(FoundLines) = ", len(FoundLines)
            # print "FoundLines = ", FoundLines
            # print "CurrentDistributions = ", CurrentDistributions
            # print "sort.CurrentDistributions = ", sorted(CurrentDistributions, key=lambda x: float(x[0]))
            # print "len(CurrentDistributions) = ", len(CurrentDistributions)


            ## add list of current distribition to general list if list is not empty
            if (CurrentDistributions != []):
                CurrentDistributions = sorted(CurrentDistributions, key=lambda x: float(x[0]))
                ListOfAllDistributions.append(CurrentDistributions)


        ## print what you do
        print "\tExtraction finished!\n\n"

        # Debug:
        # print "\tNumber of distributions found = ", len(ListOfAllDistributions)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## plot distributions for each parameter


        ## create plot
        for i in xrange(len(ListOfAllDistributions)):                                       ## loop over each distribution
            CurrentDistr = ListOfAllDistributions[i]

            # Debug
            # print "i, ListOfAllDistributions[i] = ", i, ListOfAllDistributions[i]
            # print "------------------------------------------------------------------------------------------------------------------------------\n"


            ## initialize figure window
            # ax1 = pylab.figure()
            ax1 = pylab.subplot(1, 1, 1)
            pylab.clf()


            ## make plot
            # subplots_adjust(hspace=0.45, wspace=0.2)
            pylab.plot([line[0] for line in CurrentDistr], [line[1] for line in CurrentDistr], 'b.-')   ## add exp. data to left panel
            pylab.grid(True)
            pylab.xlabel("free parameter %i" % (i + 1))
            pylab.ylabel(r'$\chi^2$')
            pylab.xlim(param_up_down[i, 0], param_up_down[i, 1])


            ## add lines to plot indicating the point of minimum, the left and the right errors respectively
            pylab.axvline(x=min_point[i], color='k', linewidth=2, linestyle='-')
            pylab.axvline(x=(min_point[i] - Error_left[i]), color='r', linewidth=1, linestyle='--')
            pylab.axvline(x=(min_point[i] + Error_right[i]), color='r', linewidth=1, linestyle='--')


            ## save distribution plot to file
            fit_log = NameOfLogFile.strip()
            mm = fit_log.rfind("/")
            if (mm > (-1)):
                path_to_save_png = fit_log[:mm + 1]
            else:
                path_to_save_png = ""
            if (plotflag == "true"):
                pylab.savefig(path_to_save_png + "ErrorEstim_INS__chi2-distribution_of_free-parameter_parm_" + str(i + 1) + "__call_" \
                              + str(abs(ErrINSCounter)) + ".png", format='png')
                # pylab.show()
            elif (plotflag == "saveonly"):
                pylab.savefig(path_to_save_png + "ErrorEstim_INS__chi2-distribution_of_free-parameter_parm_" + str(i + 1) + "__call_" \
                              + str(abs(ErrINSCounter)) )
            pylab.draw()


            ## save data to ASCII file
            ASCIIOutputFile = open(path_to_save_png + "ErrorEstim_INS__chi2-distribution_of_free-parameter_parm_" + str(i + 1) + "__call_" \
                                   + str(abs(ErrINSCounter)) + ".dat", 'w')
            counterLine = 0
            OutputContents = []
            for line in CurrentDistr:
                counterLine += 1
                if (counterLine > 1):
                    XVal = "%.15e" % float(line[0])
                    YVal = "%.15e" % float(line[1])
                    NewLine = XVal + "\t\t" + YVal


                    ## check, if current line is not already written to file
                    NotAlreadyIncluded = "true"
                    if (OutputContents != []):
                        for PrevLine in OutputContents:
                            if (PrevLine.strip() == NewLine.strip()):
                                NotAlreadyIncluded = "false"
                                break
                    if (NotAlreadyIncluded == "true"):
                        OutputContents.append(NewLine)
                        ASCIIOutputFile.write(NewLine + "\n")
            ASCIIOutputFile.close()

            # Debug:
            # print "path_to_save_png = ", path_to_save_png


        ## we've done
        return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##========================================================================================================================================================
##
##  subroutine for interval nested-sampling algorithm
##
##========================================================================================================================================================


##********************************************************************************************************************************************************
## Checking of point from the list Vectors (evaluated or not yet): if -yes- that don't evaluate (function value from the list -Points-)
def check_point_paral(Vectors, printflag, modelClass, num_par):
    """
    input variables:        Vectors:                list of parameter vectors of dimension (num_par+1, nvec)
                                                        nvec --> number of parameter vectors;
                                                        without parallelization nvec=1 --> only 1 parameter vector in the list <Vectors>

                                                        for each vectors:
                                                            0:num_par) --> parameter values
                                                            (num_par+1)--> function values (in input is 0.0)

                            printflag:              flag for screen output
                            modelClass:             class for model function call
                            num_par:                number of parameters

    output variables:       Vectors:                list of parameter vectors of dimension (num_par+1,nvec)
                                                        nvec --> number of parameter vectors;
                                                        (0:num_par) --> parameter values
                                                        (num_par+1)--> calculated function values
    """

    ## Points - list of already evaluated points
    nvec = len(Vectors)                                                                     ## number of points in the list <Vectors>
                                                                                            ## without parallelization nvec=1 --> only 1 parameter
                                                                                            ## vector in the list <Vectors>
    Points_paral = []                                                                       ## form a list for parallel calculation of optimization func.


    ## print what you do
    if (printflag == "true"):
        print "Calculate model function ..",


    ## check if in the list <Points> the function values for list <Vectors> already exist
    for i7 in xrange(nvec):
        punkt1 = Vectors[i7]
        Points_paral.append(punkt1)

        # Debug:
        # print 'punkt1[0:num_par] = ',punkt1[0:num_par]


        ### without parallelization to use:
        Chi2List = modelClass.logLhood(punkt1[0:num_par], 1, 0.0)
        punkt1[num_par] = -Chi2List[0]
        Vectors[i7][num_par] = punkt1[num_par]


        ## for further parallelization to use list of parameter vectors <Points_paral> and than update list <Vectors> (see above) !!!
        ## ..

    ## print what you do
    if (printflag == "true"):
        print "\r                                                             \r",


    ## define return parameter
    return (Vectors)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Create histigramms for each parameter after Error estimation using matplotlib
def my_hist1(para, pud, mp_val, CurrParamIndex, TotalNumFreeParam, fit_log, ErrINSCounter, plotflag):
    """
    input variables:        para:                   set of parameter values after ErrorEstimator
                            pud:                    parameter space for creating of histogram
                            mp_val:                 parameter value at the point of minimum after first optimization procedure (LM, SA ets.)
                            CurrParamIndex:         current index of free parameter
                            TotalNumFreeParam:      total number of free parameter
                            fit_log:                path and name of log-files
                            ErrINSCounter:          counter for algorithm calling
                            plotflag:               flag for plotting

    output variables:       none
    """
    if (plotflag == "false"):
        return


    ## chose type of histograms
    PlotType = "alone"


    ## define variables
    # num_interval = 30                                                                     ## set number of intervals to 30 (why 30?)
    num_interval = 1000                                                                     ## set number of intervals to 5000
    npoint = len(para)                                                                      ## get number of parameters


    ## sort array para containing parameter values
    para.sort()


    ## draw all histograms in one plot
    if (PlotType == "alltogether"):


        ## create plot for all histogramms
        if (CurrParamIndex == 0):
            pylab.subplots_adjust(hspace = 0.3, wspace = 0.2, left = 0.14, bottom = 0.10, right = 0.95, top = 0.93)
            ax1 = pylab.subplot((TotalNumFreeParam + 1), 1, 1)
        else:
            ax1 = pylab.subplot(TotalNumFreeParam, 1, (CurrParamIndex + 1))
        pylab.clf()


        ## create the histogram for the current free parameter
        n, bins, patches = ax1.hist(para, num_interval, normed = 1, facecolor = 'green', alpha = 0.75, label = str(CurrParamIndex + 1))

        # Debug:
        # print 'n = ', n
        # print 'bins = ',bins
        # print 'patches = ', patches


        ## define label for y-axis
        y1lab = 'weight of \n param ' + str(CurrParamIndex + 1)
        ax1.set_ylabel(y1lab)


        ## set range of x-axis and add grid
        ax1.set_xlim(pud[0], pud[1])
        ax1.grid(True)


        ## if last free parameter is reached add label to x-axis and save histograms to file
        if (CurrParamIndex == (TotalNumFreeParam - 1)):
            ax1.set_xlabel('Parameter range')                                               ## add label to x-axis


            ## save histograms to file
            fit_log = fit_log.strip()
            i = fit_log.rfind("/")
            if (i > (-1)):
                path_to_save_png = fit_log[:i + 1]
            else:
                path_to_save_png = ""
            if (plotflag == "true"):
                pylab.savefig(path_to_save_png + "ErrorEstim_INS__Histograms_of_all-free-parameter__call_" + str(abs(ErrINSCounter)) \
                                                                                                           + ".png", format='png')
                # pylab.show()
            elif (plotflag == "saveonly"):
                pylab.savefig(path_to_save_png + "ErrorEstim_INS__Histograms_of_all-free-parameter__call_" + str(abs(ErrINSCounter)))
            pylab.draw()


        ## construct ascii file for histogram data
        histascii_file = open(path_to_save_png + "ErrorEstim_INS__Histograms_of_free-parameter_parm__" + str(CurrParamIndex + 1) \
                                               + "__call_" + str(abs(ErrINSCounter)) + ".dat", 'w')
        count = -1
        for i in n:
            count += 1
            histascii_file.write(str(bins[count]) + "   " + str(i) + "\n")
        histascii_file.close()


    ## draw each histogram in a separate file
    elif (PlotType == "alone"):


        ## create plot for all histogramms
        ax1 = pylab.subplot(1, 1, 1)
        pylab.clf()
        # ax1 = pylab.figure()


        ## create the histogram for the current free parameter
        n, bins, patches = pylab.hist(para, num_interval, normed = 1, facecolor = 'green', alpha = 0.75, \
                                      label = 'parameter ' + str(CurrParamIndex + 1))

        # Debug:
        # print 'n = ', n
        # print 'bins = ',bins
        # print 'patches = ', patches


        ## define label for y-axis
        y1lab = 'propotional weight'
        pylab.ylabel(y1lab)


        ## set range of x-axis and add grid
        pylab.xlim(pud[0], pud[1])
        pylab.grid(True)


        ## add label to x-axis and add legend
        pylab.xlabel('Parameter range')
        pylab.legend(fancybox = False, shadow = False, loc = 1)


        ## save histograms to file
        fit_log = fit_log.strip()
        i = fit_log.rfind("/")
        if (i > (-1)):
            path_to_save_png = fit_log[:i + 1]
        else:
            path_to_save_png = ""
        if (plotflag == "true"):
            pylab.savefig(path_to_save_png + "ErrorEstim_INS__Histograms_of_free-parameter_parm_" + str(CurrParamIndex + 1) + "__call_" \
                          + str(abs(ErrINSCounter)) + ".png", format='png')
        elif (plotflag == "saveonly"):
            pylab.savefig(path_to_save_png + "ErrorEstim_INS__Histograms_of_free-parameter_parm_" + str(CurrParamIndex + 1) + "__call_" \
                          + str(abs(ErrINSCounter)))
        pylab.draw()


        ## construct ascii file for histogram data
        histascii_file = open(path_to_save_png + "ErrorEstim_INS__Histograms_of_free-parameter_parm_" + str(CurrParamIndex + 1) + "__call_" \
                              + str(abs(ErrINSCounter)) + ".dat", 'w')
        count = (-1)
        for i in n:
            count += 1
            histascii_file.write(str(bins[count]) + "   " + str(i) + "\n")
        histascii_file.close()


    ## the histogram need not be shown to the user, but saved in different files
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Calculation of quantiles of chi2-distribution on dependence of the number of free parameter (np) for 1-sigma, 2-sigma and 3-sigma contours
def del_chi2(np):
    """
    input variables:        np:                     number of free parameter

    output variables:       delta_chi2:             set of quantiles of chi2-distribution for 1-sigma (68.3%), 2-sigma(95.5%) and 3-sigma(99.7%)
                                                    contours

    Source: "http://en.wikipedia.org/wiki/68-95-99.7_rule", call 18.11.2011
    """

    alfa = [0.683, 0.955, 0.997]                                                            ## definition of the confidence levels
    delta_chi2 = numpy.array(range(3), dtype = 'float')
    for i in xrange(3):
        a1 = log(1.0/(1.0 - alfa[i]))
        a2 = (a1 - 0.16)**0.4274
        d = 2.0637 * a2 - 1.5774
        d0 = 2**0.5
        A = d * d0
        d1 = d * d - 1.0
        B = (2.0 * d1)/3.0
        d2 = d * d - 7
        C = (d * d2)/(9.0 * d0)
        d3 = 6.0 * d**4 + 14 * d * d - 32
        D = d3/405.0
        d4 = 9 * d**4 + 256 * d * d - 433
        E = d * d4/(4860 * d0)
        np0 = np**0.5
        delta_chi2[i] = np + A * np0 + B + (C/np0) + (D/np) + E/(np * np0)


    ## define return parameter
    return (delta_chi2)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Get information about error for variable parameter (nvar)
def error_info(param, nvar, Paramset):
    """
    input variables:        param:                  point of minimum (after other optimization procedure)
                                                    contains only the free (optimized) parameter, no chi^2
                            nvar:                   number of parameter in parameter vector "param" for which we would like to estimate errors
                            Paramset:               set of parameter values with function values and proportional weights
                                                    Paramset[:, 0] contains all parameter values of parameter "param[nvar]"
                                                    Paramset[:, 1] contains the corresponding chi^2 values
                                                    Paramset[:, 2] contains the corresponding proportional weights ?????

    output variables:       mean_val:               mean value for parameter <nvar>
                            dev:                    standard deviation
    """

    ## define variables num_par and numpo
    num_par = len(param) - 1
    numpo = len(Paramset)

    # Debug:
    # print "\n\n\nnum_par = ", num_par
    # print "numpo = ", numpo
    # print "param = ", param
    # print "nvar = ", nvar
    # print "Paramset[:, 0] = ", Paramset[:, 0]
    # print "Paramset[:, 1] = ", Paramset[:, 1]
    # print "Paramset[:, 2] = ", Paramset[:, 2]


    ## calculation of chi2-contours (1sigma, 2sigma, 3sigma) for ONE variable
    delta_chi2 = del_chi2(1)

    # Debug:
    # print "delta_chi2 = ", delta_chi2


    ## initialize variables
    param_err = numpy.array(range(numpo*4), dtype = 'float')
    param_err.shape = (numpo, 4)
    param_prom = numpy.array(range(4), dtype = 'float')


    ## calculation of distances for all points from point of minimum
    for j in xrange(numpo):
        dist = (Paramset[j, 0] - param[nvar])**2
        param_err[j, 0] = Paramset[j, 0]
        param_err[j, 1] = Paramset[j, 1]
        param_err[j, 2] = dist
        param_err[j, 3] = Paramset[j, 2]

        # Debug:
        # print "j = ", j
        # print "dist = ", dist
        # print "param_err[j, :] = ", param_err[j, :]


    ## sort on distances (minimal above)
    for j in xrange(numpo):
        for k in xrange(j,(numpo)):
            if (param_err[k, 2] < param_err[j, 2]):
                param_prom[:] = param_err[j, :]
                param_err[j, :] = param_err[k, :]
                param_err[k, :] = param_prom[:]

    # Debug: (print sorted param_err array)
    # for j in xrange(numpo):
    #     print "j = ", j
    #     print "param_err[j, :] = ", param_err[j, :]


    ## form list <Min_points> for error estimation get all points with delta_chi2 < 3sigma contour
    cont = 0
    ncont = 2                                                                               ## delta_chi2[ncont=2] = 3sigma contour
    nconira = 0
    while(cont == 0):


        ## define array "Min_points" containing all points with delta_chi2 < 3sigma contour
        ## chi2-contours (1sigma, 2sigma, 3sigma):  delta_chi2[ncont=2] = 3sigma contour
        ii = 0
        mett_contur = 0
        Min_points = []                                                                     ## reset Min_points array
        while (mett_contur == 0 and ii < numpo):
            if (param_err[ii, 1] <= delta_chi2[ncont]):                                     ## check, if current distance is within 3sigma contour
                Min_points.append(param_err[ii,:])
                ii += 1
            else:
                mett_contur = 1

        # Debug:
        # if (ii != 0):
        #     print "ii, ncont = ", ii, ncont
        #     print "param_err[ii,:] = ", param_err[ii,:]
        #     print "delta_chi2[ncont] = ", delta_chi2[ncont]


        ## check, if statistic is reliable
        if (len(Min_points) > 3):                                                           ## if more than 3 points are within 3sigma contour stop loop
            cont = 1
        else:
            if (ncont >= 2 and nconira == 0):                                               ## continue here, if less than 3 pts are within 3sigma contour
                delta_chi2[ncont] = 2.0 * delta_chi2[ncont]                                 ## increase 3sigma contour by a factor 2
                nconira = 1
            else:
                ## if we does not have enough points for error estimation in 3sigma contour, we use all points from INS-sequence
                Min_points = []
                for iira in xrange(numpo):
                    Min_points.append(param_err[iira,:])

                # Debug:
                # print "\n\nToo small statistical sample for error estimation of parameter ", (nvar + 1)
                # print 'Maximum number of iteration has to be 10 or more'

                mett_contur = 1
                ii = numpo
                cont = 1


    ## define how many points we have for error estimation
    nminpo = len(Min_points)


    ## determine eqn. (10) in MAGIX paper: Calculate mean value and standard deviation for parameter <nvar>
    ## for eqn. (10): nominator is stored in mean_val
    ## for eqn. (10): denominator is stored in sum_wes
    mean_val = 0.0
    sum_wes = 0.0
    for i in xrange(nminpo):                                                                ## sum form i = 1 to m (here nminpo)
           if (Min_points[i][3] != 0.0):                                                    ## is weight equal to zero?
               mean_val += Min_points[i][0] * Min_points[i][3]
               sum_wes += Min_points[i][3]
           else:
               mean_val += Min_points[i][0]
               sum_wes += 0.5

    # Debug:
    # print ">sum_wes = ", sum_wes


    ## define mean_val --> mean value for parameter <nvar> for the sample points from the list <Min_points>
    if (sum_wes <> 0.0):
        mean_val = mean_val/sum_wes                                                         ## final part of eqn. (10) in MAGIX paper

    # Debug:
    # print "sum_wes = ", sum_wes
    # print "mean_val = ", mean_val


    ## determine eqn. (9) in MAGIX paper: calculate standard deviation for the sample points from the list <Min_points>
    dev = 0.0
    for i in xrange(nminpo):
         dev += (Min_points[i][0] - mean_val)**2.0                                          ## first part of eqn. (9) in MAGIX paper (non-weighted version)
         # dev += (Min_points[i][0] - mean_val)**2.0 * Min_points[i][3]                     ## first part of eqn. (9) in MAGIX paper (correct version)


    ## check if we have enough points for error estimation
    if ((nminpo - 1) != 0):
         dev = (dev/(nminpo - 1))**0.5                                                      ## seond part of eqn. (9) in MAGIX paper
    else:
         dev = (dev)**0.5

    # Debug:
    # print "dev = ", dev
    # sys.exit(0)


    ## define return (output) variables
    return (mean_val, dev)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Process of bisection of parametric space (par_space) for k-coordinate
def bisection(k, par_space):
    """
    input variables:        k:                      direction of bisection
                            par_space:              parental interval box in form (lower_limit, upper_limit)

    output variables:       par_space1:             interval box1 after bisection in form (lower_limit, upper_limit)
                            par_space2:             interval box2 after bisection in form (lower_limit, upper_limit)
    """


    ## Dimension of space = 1
    par_space1 = numpy.array(range(2), dtype = 'float')
    par_space2 = numpy.array(range(2), dtype = 'float')


    ## calculate center point of the interval par_space
    bi_point = (par_space[1] + par_space[0])/2.0


    ## give bounds for new interval boxes
    par_space1[0] = par_space[0]
    par_space1[1] = bi_point
    par_space2[0] = bi_point
    par_space2[1] = par_space[1]


    ## define return (output) variables
    return (par_space1, par_space2)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Calculation of center point of the interval <par_space>
def calc_mid_point(par_space):
    """
    input variables:        par_space:              interval in form (lower_limit, upper_limit)

    output variables:       mid_point:              center point of the interval <par_space>
    """


    ## create vector for center point --> (0:num_par) - parameter values; (num_par) - function value (!only for Python!)
    mid_point = numpy.array(range(num_par + 1), dtype = 'float')
    for i in xrange(num_par):
        mid_point[i] = (par_space[i,1] + par_space[i,0])/2.0


    ## define function value in cenetr point --> input: 0.0
    mid_point[num_par] = 0.0


    ## calculate function value in cenetr point --> call subroutine <check_point>
    mid_point = check_point(mid_point)


    ## define return (output) variables
    return (mid_point)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Calculation of inclusion function: return inclusion - interval F1 = (F1_min, F1_max)
def inclusion_function(V1, mid, mid1, direc):
    """
    input variables:        V1:                     initial interval box
                            mid:                    center point of interval box
                            mid1:                   center point of parental interval box
                            direc:                  direction of bisection

    output variables:       F1:                     inclusion function in form of the interval F1 = (F1_min, F1_max)
    """


    ## define number of parameters
    np = len(mid) - 1


    ## calculate inclusion function with help interval arithmetic rules by calculation it is used the inclusion function with slopes
    ## (for more information see Documentation)
    aa = V1[0]
    bb = V1[1]
    aa = aa - mid[direc]
    bb = bb - mid[direc]


    ## check for division by zero:
    if ((mid1[direc] - mid[direc]) == 0.0):
        s1 = 0.0
    else:
        s1 = (mid1[np] - mid[np])/(mid1[direc] - mid[direc])
    s1_aa = s1 * aa
    s1_bb = s1 * bb
    F1 = [0, 0]
    F1[0] = mid1[np] + s1_aa
    F1[1] = mid1[np] + s1_bb
    if (F1[0] > F1[1]):
       cc = F1[0]
       F1[0] = F1[1]
       F1[1] = cc


    ## define return (output) variables
    return (F1)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Calculation of volume of interval box and return ratio (volume_box/whole_volume)
def volume_box(box, param_up_down):
    """
    input variables:        box:                    current interval box in form (lower_limit, upper_limit)
                            param_up_down:          the whole parameter space in form (lower_limit, upper_limit)

    output variables:       volume_ratio:           ration of the volume of the current interval box to the whole volume of parameter space
    """


    ## define the volume of current interval box and the whole volume of parameter space
    whole_volume = 1.0
    vol_box = 1.0


    ## calculate the volume of current interval box and the whole volume of parameter space
    for j in xrange(len(param_up_down[:, 0])):
        pud0 = param_up_down[j, 0]
        pud1 = param_up_down[j, 1]
        try:
            vol_box = vol_box * abs(box[1] - box[0])
        except ValueError:
            print "box[0] = ", box[0]
            print "box[1] = ", box[1]
            print "pud11[0] = ", pud11[0]
            print "pud11[1] = ", pud11[1]
        whole_volume = whole_volume * abs(pud1 - pud0)


    ## define ration of volumes
    volume_ratio = vol_box / whole_volume


    ## define return (output) variables
    return (volume_ratio)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Calculation of center point of the interval par_space
def calc_cen_point(par_space, nvar, min_point):
    """
    input variables:        par_space:              current interval box in form (lower_limit, upper_limit)
                            nvar:                   number of the variable parameter (from 0 to num_par)
                            min_point:              point of minimum after first optimization algorithm (LM, SA ets.)

    output variables:       cen_point:              center point of the current interval box <par_space>
    """


    ## define npar
    npar = len(min_point)


    ## define center point: only one parameter value is variable, other parameter held as constants
    cen_point = numpy.array(range(npar+1), dtype = 'float')
    if (npar == 1):
       cen_point[0] = (par_space[1] + par_space[0])/2.0
       cen_point[1] = 0.0
    else:
        for jj in xrange(npar):


            ## only one parameter value is variable, other parameter held as constants
            if (jj == nvar):
               cen_point[jj] = (par_space[1] + par_space[0])/2.0
            else:
               cen_point[jj] = min_point[jj]


        ## define function value in the center point
        cen_point[npar] = 0.0


    ## define return (output) variables
    return (cen_point)
##----------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Interval analysis for part of space - par_space
def interval_method(par_space, direc, min_point, printflag, modelClass, num_par):
    """
    1. Division of par_space on this direction - def bisection
    2. Finding of inclusion function in accordance with interval arithmetic rules

    input variables:        par_space:              current interval box
                            direc:                  direction for bisection
                            min_point:              point of the minimum after first optimization procedure (LM, SA ets.)
                            printflag:              flag for screen output
                            modelClass:             class for model function call
                            num_par:                number of parameters

    output variables:       cen_point:              center point od the current interval box <par_space>
                            V1:                     first interval box after bisection of <par_space>
                            cen_point1:             center point of first interval box
                            V2:                     second interval box after bisection of <par_space>
                            cen_point2:             center point of second interval box
                            F1:                     inclusion function of V1 in form F1 = (F1_min, F1_max)
                            F2:                     inclusion function of V2 in form F2 = (F2_min, F2_max)
    """


    ## calculate the center point of current interval box
    Centers = []
    cen_point = calc_cen_point(par_space, direc, min_point)
    Centers.append(cen_point)


    ## bisection of current interval box in <direc> direction
    V1, V2 = bisection (direc, par_space)


    ## center point for V1
    cen_point1 = calc_cen_point(V1, direc, min_point)
    Centers.append(cen_point1)


    ## center point for V2
    cen_point2 = calc_cen_point(V2, direc, min_point)
    Centers.append(cen_point2)


    ## calculate function values for points from the list <Centers>
    Centers = check_point_paral(Centers, printflag, modelClass, num_par)


    ## define output variable cen_point, cen_point1, cen_point2
    cen_point = Centers[0]
    cen_point1 = Centers[1]
    cen_point2 = Centers[2]


    ## calculate inclusion function for V1 and V2
    F1 = inclusion_function(V1, cen_point, cen_point1, direc)
    F2 = inclusion_function(V2, cen_point, cen_point2, direc)


    ## define return (output) variables
    return (cen_point, V1, cen_point1, V2, cen_point2, F1, F2)
##----------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Evolve object within function constraint contents interval method for analysing of optimization function
def explore(Obj, logLstar, min_point, nvar, L00, Cen_L00, Best, stop_crit, printflag, modelClass, num_par, param_up_down):
    """
    input variables:        Obj:                    Object being evolved
                            logLstar:               best function value (for the moment)
                            min_point:              point of minimum
                            nvar:                   number of variable parameter
                            L00:                    list of working intervals
                            Cen_L00:                mid_points for intervals from L00
                            Best:                   list of best intervals
                            stop_crit:              terminating criterion
                            printflag:              flag for screen output
                            modelClass:             class for model function call
                            num_par:                number of parameters
                            param_up_down:          parameter limits

    output variables:       ret:                    dictionary with best Object
                            stop_crit:              termination criteria of inclusion function (not used in ErrorEstimator!)
    """

    ## define <ret>
    ret = Object()
    ret.__dict__ = Obj.__dict__.copy()


    ## reset variables
    step = 0.1;                                                                             ## Initial guess suitable step-size in (0,1)
    accept = 0;                                                                             ## MCMC acceptances
    reject = 0;                                                                             ## MCMC rejections
    Try = Object();                                                                         ## Trial object
    LL = []
    uniform = random.random


    ## do two attempts for search of new best Object
    for m in range(2):                                                                      ## pre-judged number of steps
        L1 = []


        ## Trial object
        Try.prior = ret.prior + step * (2.0 * uniform() - 1.0)
        Try.prior = Try.prior - floor(Try.prior)
        if (len(L00) <> 0):                                                                 ## check working list of interval boxes
            LL = L00[0]
            del L00[0]
            del Cen_L00[0]
        else:
            if (len(Best) <> 0):                                                            ## if the main working list <LL> is empty, use the list <Best>
                LL = Best[0]
                del Best[0]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## Interval method for investigation of parameter space


        ## internal parameters of INS-algorithm are defined by default (for ErrorEstimator!)
        vol_bound = 0.1
        delta_incl = 0.0001


        ## calculate ration of volumes
        volume_ratio = volume_box(LL, param_up_down)


        ## define transitional vector
        prom_vec = numpy.array(range(num_par), dtype='float')


        ## check if the volume of working interval box is small enough
        if (volume_ratio < vol_bound):                                                      ## if working interval is small enough
            cen, V1, cen1, V2, cen2, F1, F2 = interval_method(LL, nvar, min_point, printflag, modelClass, num_par)


            ## if function value in center points are equal th add to the working list <L00> both of interval boxes V1 and V2
            if (cen1[num_par] <= cen2[num_par] and F1[0] <= F2[0]):
                L00.append(V1)
                Cen_L00.append(cen1)
                Best.append(V1)
                L00.append(V2)
                Cen_L00.append(cen2)


                ## define point for choosing of best Object
                if (m == 0):
                    Try.position = [cen1[i] for i in xrange (num_par) ]
                    Try.logL = cen1[num_par]


                ## define point for choosing of best Object
                if (m > 0 and Try.logL > cen1[num_par]):
                    Try.position = [cen1[i] for i in xrange (num_par) ]
                    Try.logL = cen1[num_par]


            ## mid_point test
            if (cen1[num_par] > cen2[num_par] and F1[0] >= F2[0]):


                ## both of interval boxes are added to the working list <L00>
                L1.append(V2)
                L1.append(cen2)
                Best.append(L1)
                L00.append(V2)
                Cen_L00.append(cen2)
                L00.append(V1)
                Cen_L00.append(cen1)


                ## define point for choosing of best Object
                if (m == 0):
                    Try.position = [cen2[i] for i in xrange (num_par) ]
                    Try.logL = cen2[num_par]
                if (m > 0 and Try.logL > cen2[num_par]):
                    Try.position = [cen2[i] for i in xrange (num_par) ]
                    Try.logL = cen2[num_par]
                if (abs(F2[1] - F2[0]) < delta_incl):
                   stop_crit = 1
        else:                                                                               ## if working interval is big
            cen, V1, cen1, V2, cen2, F1, F2 = interval_method(LL, nvar, min_point, printflag, modelClass, num_par)


            ##  if function value in center points are equal th add to the working list <L00> both of interval boxes V1 and V2
            if (cen1[num_par] == cen2[num_par]):
                L00.append(V1)
                Cen_L00.append(cen1)
                if (m == 0):
                    Try.position = [cen1[i] for i in xrange (num_par) ]
                    Try.logL = cen1[num_par]
                if (m > 0 and Try.logL > cen1[num_par]):
                    Try.position = [cen1[i] for i in xrange (num_par) ]
                    Try.logL = cen1[num_par]
                L00.append(V2)
                Cen_L00.append(cen2)


            ## mid_point test
            ## only best interval box is added to the working list <L00>
            if (cen1[num_par] < cen2[num_par]):
                L00.append(V1)
                Cen_L00.append(cen1)
                if (m == 0):
                    Try.position = [cen1[i] for i in xrange (num_par) ]
                    Try.logL = cen1[num_par]
                if (m > 0 and Try.logL > cen1[num_par]):
                    Try.position = [cen1[i] for i in xrange (num_par) ]
                    Try.logL = cen1[num_par]


            ## only best interval box is added to the working list <L00>
            if (cen1[num_par] > cen2[num_par]):
                L00.append(V2)
                Cen_L00.append(cen2)
                if (m == 0):
                    Try.position = [cen2[i] for i in xrange (num_par) ]
                    Try.logL = cen2[num_par]
                if (m > 0 and Try.logL > cen2[num_par]):
                    Try.position = [cen2[i] for i in xrange (num_par) ]
                    Try.logL = cen2[num_par]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## Accept if and only if within hard likelihood constraint
        if (Try.logL < logLstar):
             ret.__dict__ = Try.__dict__.copy()
             accept += 1
        else:
             reject += 1


        ## Refine step-size to let acceptance ratio converge around 50%
        if (accept > reject):
            step *= exp(1.0 / accept);
        if (accept < reject):
            step /= exp(1.0 / reject);

    # Debug:
    # print 'Stop_crit = ', stop_crit


    ## define return parameters
    return (ret, stop_crit)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## logarithmic addition log(exp(x)+exp(y))
def plus(x,y):
    """
    input variables:        x:                      floating point
                            y:                      floating point

    output variables:       log(exp(x)+exp(y))
    """

    if (x > y):
       return x + log(1 + exp(y - x))
    else:
       return y + log(1 + exp(x - y))
##----------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Initial list of working objects
def loglike_nest(i, num_par, LL):
    """
    input variables:        i:                      number of the object
                            num_par:                number of parameters
                            LL:                     ?

    output variables:       Obj:                    interval box from the working list <LL>
    """

    Obj = Object()
    Obj.prior = random.uniform(0, 1)
    Obj.position = [ LL[i][1][j] for j in xrange (num_par) ]
    Obj.logL = LL[i][1][j + 1]


    ## define return parameters
    return(Obj)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## This is an implementation of John Skilling's Nested Sampling algorithm for computing the normalizing constant of a probability distribution
## (usually the posterior in Bayesian inference).
def ndim_nest(n, explore, pud, min_point, nvar, LL, printflag, paramchannel, logchannel, modelClass, max_iter, param_up_down):
    """
    input variables:        n:                      number of objects
                            explore:                subroutine for searching of the next best point in parameter space (uses the interval method)
                            pud:                    lower and upper limit for single parameter
                            min_point:              point of minimum after first optimization procedure (LM, SA, ets.)
                            nvar:                   number of variable parameter (from 0 to num_par)
                            LL:                     working list of interval boxes
                            printflag:              flag for screen output
                            paramchannel:           number of channel for parameter log-file
                            logchannel:             number of channel for log-file
                            modelClass:             class for model function call
                            max_iter:               max. number of iterations
                            param_up_down:          parameter space

    output variables:       samples:                a set of the rejected points from nested sampling process;
                                                        for these points we have proportional weights
                            num_iterations:         number of iterations of nested sampling process
                            logZ:                   logarithm of the Bayesian evidence
                            logZ_sdev:              standers deviation of log(Z)
                            Best:                   list with interval boxes
                            InputFile:              content of final input files
    """


    ##================================================================================================================================================
    Obj = []                                                                                ## Collection of n objects
    Samples = []                                                                            ## Objects stored for posterior results
    logwidth = None                                                                         ## ln(width in prior mass)
    logLstar = None                                                                         ## ln(Likelihood constraint)
    H = 0.0                                                                                 ## Information, initially 0
    DBL_MAX = 1e300
    logZ =-DBL_MAX                                                                          ## ln(Evidence Z, initially 0)
    logZnew = None                                                                          ## Updated logZ
    copy = None                                                                             ## Duplicated object
    worst = None                                                                            ## Worst object
    nest = None                                                                             ## Nested sampling iteration count
    num_par = len(min_point)                                                                ## number of free parameters
    uniform = random.random


    ## reset L00 and Cen_L00
    L00 = []
    Cen_L00 = []


    ## Set prior objects
    for i in xrange(n):
        Obj.append(loglike_nest(i, num_par, LL))
        L00.append(LL[i][0])
        Cen_L00.append(LL[i][1])


    ## Outermost interval of prior mass
    logwidth = log(1.0 - exp(-1.0 / n));


    ## NESTED SAMPLING LOOP ______________________________________________________________________________________________________________________________
    nest = 0
    Best = []
    stop_crit = 0
    chisq = 1.e99
    glavmet = 0                                                                             ## flag for stopping criterion <stop_crit>
    while (nest < max_iter and glavmet == 0):


        ## determine worst index
        worst = 0
        for i in range(1, n):


            ## bug fix:
            if (Obj[i].logL == None):
                Obj[i].logL = 0
                Obj[i].position = [0.0 for j in xrange(num_par)]


            ## determine worst index
            if (Obj[i].logL > Obj[worst].logL):
               worst = i


        ## bug fix
        if (n == 0 and Obj[0].logL == None):
            Obj[0].logL = 0
            Obj[0].position = [0.0 for j in xrange(num_par)]


        ## calculate new contour for value Obj[worst].logL
        try:
            Obj[worst].logWt = logwidth - Obj[worst].logL
        except TypeError, err:
            print " "
            print " "
            print "Error in subroutine ErrorEstimator:"
            print " "
            print "\t logwidth = ", logwidth
            print "\t Obj[worst].logL = ", Obj[worst].logL
            print "\t n = ", n
            for i in range(1, n):
               print "\t i,Obj[i].logL = ",i,Obj[i].logL
            sys.exit(0)


        ## Update Evidence Z and Information H
        logZnew = plus(logZ, Obj[worst].logWt)
        H = exp(Obj[worst].logWt - logZnew) * Obj[worst].logL + exp(logZ - logZnew) * (H + logZ) - logZnew;
        logZ = logZnew


        ## Posterior Samples (optional)
        Samples.append(Obj[worst])


        ## Kill worst object in favour of copy of different survivor
        if (n > 1):                                                                         ## don't kill if n is only 1
            while True:
                copy = int(n * uniform()) % n                                               ## force 0 <= copy < n
                if (copy != worst):
                    break


        ## choose new best object
        logLstar = Obj[worst].logL                                                          ## new likelihood constraint
        Obj[worst] = Obj[copy]                                                              ## overwrite worst object


        ## Evolve copied object within constraint
        updated, stop_crit = explore(Obj[worst], logLstar, min_point, nvar, L00, Cen_L00, Best, stop_crit, printflag, modelClass, num_par, param_up_down)


        ## if stop_crit is reached and number of iteration <nest> is enough --> the loop is ended
        if (stop_crit == 1 and nest >= 5):
            # glavmet = 1
            glavmet = 0


        ## Clean working list <L00> (each 5 iterations)
        iira = (nest/3.0) - int(nest/3.0)
        if (nest <> 0 and iira == 0):
            new_len = int(len(L00)/2.0)
            nl = len(L00)
            for i in xrange(nl):
                for j in xrange(i,nl):
                    L11 = []
                    C11 = []
                    if (Cen_L00[i][num_par] > Cen_L00[j][num_par]):
                        L11 = L00[i]
                        L00[i] = L00[j]
                        L00[j] = L11

                        C11 = Cen_L00[i]
                        Cen_L00[i] = Cen_L00[j]
                        Cen_L00[j] = C11

            i = new_len + 1
            ir = len(L00) - i
            for i5 in xrange(ir):
                del L00[i]
                del Cen_L00[i]


        ## write informations to log files
        modelClass.WriteLogLines(nest + 1)


        ## update the worst object
        assert(updated != None)                                                             ## Make sure explore didn't update in-place
        Obj[worst] = updated


        ## Shrink interval
        logwidth -= 1.0 / n
        nest += 1


    ## Exit with evidence Z, information H, and optional posterior Samples
    sdev_H = H/log(2.)
    if (H > 0):
        sdev_logZ = sqrt(H/n)
    else:
        sdev_logZ = 0.0
    nest0 = nest - 1


    ## write to log file
    logchannel.write("  \n")
    logchannel.flush()


    ## get results from chi^2 calculation
    bestchi, bestparam, bestline, bestinput = modelClass.GetBestResult()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## CAN NOT COLLECT A GOOD STATISTIC FOR ERROR ESTIMATION IF NEXT LINES 931-953 ARE OPENED   !!!!!!!!!!!!!!!!!!
    ## print reason for stop of iteration to screen and to log-file
    # paramchannel.write("\n")
    # paramchannel.write("=======================================================================================================================\n")
    # paramchannel.write("\n")
    # paramchannel.flush()
    # if (bestchi < chilim):
    #     outputstring = "           Iteration stopped. chi^2 (=%25.15e) dropped below limit = %25.15e" % (bestchi, chilim)
    #     if (printflag == "true"):
    #         print "\n" + outputstring
    #     logchannel.write("\n")
    #     logchannel.write(outputstring + "\n")
    # elif (len(L00) == 0):
    #     if (printflag == "true"):
    #         print "\n           Iteration stopped. len(L00) == 0"
    #     logchannel.write("\n          Iteration stopped. len(L00) == 0\n")
    # else:
    #     if (printflag == "true"):
    #         print "\n           Iteration stopped. Number of iterations is equal to max. number of iterations = %6d" % max_iter
    #     logchannel.write("\n          Iteration stopped. Number of iterations is equal to max. number of iterations = " + str(max_iter) + "\n")


    ## define return parameters
    return {"samples":Samples, "num_iterations":(nest0), "logZ":logZ, "logZ_sdev":sdev_logZ, "Best":Best, "InputFile":bestinput}
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## class Object
class Object:


    ##****************************************************************************************************************************************************
    ## initialize class Object
    def __init__(self):
        self.prior = None                                                                   ## a prior volume for self.position (multidimensional)
        self.position = None                                                                ## a set of parameters values (multidimensional)
        self.logL = None                                                                    ## logLikelihood = ln Prob(data | position)
        self.logWt = None                                                                   ## log(Weight), adding to SUM(Wt) = Evidence Z


##********************************************************************************************************************************************************
## Return for ONE parameter --- mean value, standard deviation and Paramset for histogramm
## Other parameters are held as CONSTANTS in min_point
def Interval_NS(min_point, nvar, pud, max_iter, paramchannel, logchannel, printflag, modelClass, num_par, param_up_down):
    """
    input variables:        min_point:              the point of minimum after first optimization algorithm (LM, SA, ets.)
                            nvar:                   number of variable parameter (from 0 to num_par)
                            pud:                    parameter space
                            max_iter:               maximum number of iteration of INS algorithm
                            paramchannel:           channel for parameter log file
                            logchannel:             channel for main log file
                            printflag:              flag indicating if messages are printed to screen or not
                            modelClass:             class for model function call
                            num_par:                number of parameters
                            param_up_down:          lower and upper limits for all parameters

    output variables:       Points:                 a list with all points which have been calculated while INS algorithm
                            Paramset2[:,nvar]:      a set of parameter values for parameter <nvar> with proportional weights
                            num_set2:               number of the vectors in <Paramset2>
                            minima:                 mean value for parameter <nvar>
                            stdev:                  standard deviation for parameter <nvar>
                            logZ:                   logarithm of the evidence
    """

    ## define working lists
    L0 = []
    Points11 = []


    ## calculation of initial interval for error estimation
    bound_00 = numpy.array(range(4*2), dtype = 'float')
    bound_00.shape = (4,2)


    ## number of objects for INS process
    Num_int0 = 4                                                                            ## Number if initial intervals
    n = 4                                                                                   ## Number of working objects for NS-process


    ## initial interval division of parameter space <pud>
    ddel1 = abs(pud[0] - min_point[nvar])
    ddel2 = abs(pud[1] - min_point[nvar])
    if (ddel1 < ddel2):
        ddel = ddel1/3.0
    else:
        ddel = ddel2/3.0


    ## create initial working list
    dmet = 0


    ## if <min_point> is very close to left bound of parameter space
    if (ddel1 <= 0.01 * (pud[1] - pud[0])):
        ddel = ddel2/10.0
        dmet = 1

        ## initial boxes in form (lower_limit, upper_limit)
        bound_00[0,0] = pud[0]
        bound_00[0,1] = min_point[nvar]

        bound_00[1,0] = min_point[nvar]
        bound_00[1,1] = min_point[nvar] + ddel/5.0

        bound_00[2,0] = min_point[nvar] + ddel/5.0
        bound_00[2,1] = min_point[nvar] + 2.0 * ddel

        bound_00[3,0] = min_point[nvar] + 2.0 * ddel
        bound_00[3,1] = pud[1]


    ## if <min_point> is very close to right bound of parameter space
    if (ddel2 <= 0.01 * (pud[1] - pud[0])):
        ddel = ddel1/10.0
        dmet = 1


        ## initial boxes in form (lower_limit, upper_limit)
        bound_00[0,0] = pud[0]
        bound_00[0,1] = pud[0] + ddel1/2.0

        bound_00[1,0] = pud[0] + ddel1/2.0
        bound_00[1,1] = pud[0] + 3.0 * ddel1/4.0

        bound_00[2,0] = pud[0] + 3.0 * ddel1/4.0
        bound_00[2,1] = min_point[nvar] - ddel

        bound_00[3,0] = min_point[nvar] - ddel
        bound_00[3,1] = pud[1]


    ## if <min_point> is not close to left and right bounds of parameter space
    if (dmet == 0):
        bound_00[0,0] = pud[0]
        bound_00[0,1] = min_point[nvar] - ddel

        bound_00[1,0] = min_point[nvar] - ddel
        bound_00[1,1] = min_point[nvar]

        bound_00[2,0] = min_point[nvar]
        bound_00[2,1] = min_point[nvar] + ddel

        bound_00[3,0] = min_point[nvar] + ddel
        bound_00[3,1] = pud[1]


    ## define center points of the initial interval boxes
    npar = len(min_point)
    mid_point00 = numpy.array(range(Num_int0 * (npar + 1)), dtype = 'float')
    mid_point00.shape = (Num_int0, (npar + 1))


    ## define initial center points for initial intervals
    Init_centers = []
    for jj in xrange(Num_int0):
        L0.append(bound_00[jj,:])
        if (npar == 1):
            mid_point00[jj,:] = calc_cen_point(bound_00[jj,:], nvar, min_point)
        else:
           for jj1 in xrange(npar):
                if (jj1 == nvar):
                   mpo = calc_cen_point(bound_00[jj,:], nvar, min_point)
                   mid_point00[jj, jj1] = mpo[nvar]
                else:
                   mid_point00[jj, jj1] = min_point[jj1]
           mid_point00[jj, npar] = 0.0


        ## add to the list <Init_centers>
        Init_centers.append(mid_point00[jj, :])


    ## length of initial lists L0
    n_list = len(L0)


    ## print what you do to (screen), log and param-log file
    paramchannel.write("==========================================================================================================================\n")
    paramchannel.write("\n Nested sampling loop for free parameter " + str((nvar + 1)) + "\n\n")
    paramchannel.write("==========================================================================================================================\n")
    paramchannel.flush()
    outputstring = "\n           Nested sampling loop for free parameter " + str((nvar + 1)) + "\n\n"
    logchannel.write(outputstring + "\n")
    if (printflag == "true"):
        print outputstring
        print "           Iteration:                    chi^2:     Parameter:"
        logchannel.write("           Iteration:                    chi^2:     Parameter:\n")
        command_string = "echo -n \"           Initialize model function ..\""
        os.system(command_string)
    logchannel.flush()


    ## Checking of point from the list Vectors (evaluated or not yet): if -yes- that don't evaluate
    Init_centers = check_point_paral(Init_centers, printflag, modelClass, num_par)


    ## Form list <L00> for using in <def explore()>
    L00 = []
    for i in xrange(n_list):
        L01 = []
        L01.append(L0[i])
        L01.append(Init_centers[i])
        L00.append(L01)


    ## call ndim_nest subroutine which performs the nested-sampling steps
    Best = []
    results = ndim_nest(n, explore, pud, min_point, nvar, L00, printflag, paramchannel, logchannel, modelClass, max_iter, param_up_down)
    ni = results['num_iterations']                                                          ## get number of iterations
    samples = results['samples']                                                            ## get parameter vectors and function values
    logZ = results['logZ']                                                                  ## get logZ
    Best = results['Best']                                                                  ## get best of interval boxes
    Bestinput = results['InputFile']


    ## Definition of first set of best parameter vectors for weight's calculating
    Paramset1 = numpy.array(range((ni + 1) * (num_par + 2)), dtype = 'float')
    Paramset1.shape = ((ni + 1), (num_par + 2))
    min_prob = 1e+03
    max_prob = 0.0
    ffmax = 1.0
    for i in range(ni):                                                                     ## loop over all calculated paremeter vectors done in INS
        w = exp(samples[i].logWt - logZ);                                                   ## poportional weight
        Paramset1[i,0:num_par] = samples[i].position                                        ## store the ith parameter vector
        Paramset1[i, num_par] = samples[i].logL                                             ## and the corresponding chi^2 (logL)
        Paramset1[i, num_par + 1] = w                                                       ## at the num_par+1 position the poportional weight is added


        ## save the position with the highest poportional weight
        if (max_prob < w):
            max_prob = w                                                                    ## store the current poportional weight to variable max_prob
            ffmax = Paramset1[i, num_par]                                                   ## save the corresponding chi^2 (logL)


    ## add to array "Paramset1" the point of minimum
    Paramset1[ni, 0:num_par] = min_point[0:num_par]                                         ## add the parameter vector for the point of minimum
    Paramset1[ni, num_par] = ffmax / 2.0                                                    ## reduce the found minimal chi^2 (logL) by a factor of 2
                                                                                            ## to make the chi^2 (logL) value for the minimum really the
                                                                                            ## lowest one
    Paramset1[ni, num_par + 1] = 0.988                                                      ## proportional weight for the point of minimum is equal to
                                                                                            ## the 3 sigma weight?
    Points = modelClass.GetPoints()
    npp = len(Points)


    ## define minimal proportional weight
    HelpList = Paramset1[0:(ni - 1), num_par + 1]
    min_prob = min(HelpList)                                                                ## get the smallest proportional weight
    i = numpy.where(HelpList == min_prob)[0][0]                                             ## get index where the smallest weight is located
    ffmin = Paramset1[i, num_par]                                                           ## get the corresponding chi^2 (logL) value


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Calculation of weights for all best points from the list <Points> (linear interpolation)
    aa5 = max_prob - min_prob                                                               ## determine difference between lowest and highest
                                                                                            ## proportional weight
    bb5 = ffmax - ffmin                                                                     ## define the difference between lowest and highest chi^2
                                                                                            ## (logL) value

    ## determine gradient m of f(x) = m*x + b
    if (bb5 <> 0.0):
        kk5 = aa5 / bb5
    else:
        kk5 = aa5


    ## determine parameter b of f(x) = m*x + b
    cc5 = min_prob - kk5 * ffmin

    # Debug:
    # print "npp = ", npp
    # print "Points = ", Points
    # print "num_par = ", num_par
    # print "abs(logZ) = ", abs(logZ)


    ## determine array Best_Points, containing all position vectors with chi^2 lower than abs(logZ)
    Best_Points = []
    for i in xrange(npp):
        if (nvar == 0):

            # Debug:
            # print "i, Points[i][num_par], (Points[i][num_par] <= abs(logZ)) = ", Points[i][num_par], (Points[i][num_par] <= abs(logZ))

            # if (Points[i][num_par] <= abs(logZ)):
            Best_Points.append(Points[i])
        else:
            # if (Points[i][num_par] <= abs(logZ) and Points[i][nvar] <> min_point[nvar]):
            if (Points[i][nvar] <> min_point[nvar]):
                Best_Points.append(Points[i])


    ## Form the second set of parameter vectors <Paramset2>, function values and weights for each point
    npp = len(Best_Points)
    npoint = npp + ni

    # Debug:
    # print "npp = ", npp
    # print "npoint = ", npoint
    # print "(npoint * (num_par + 2)) = ", (npoint * (num_par + 2))

    Paramset2 = numpy.array(range(npoint * (num_par + 2)), dtype = 'float')
    Paramset2.shape = (npoint, (num_par + 2))
    min_paramset = numpy.array(range(num_par + 1), dtype = 'float')
    min_paramset[0:num_par] = 0.0
    min_paramset[num_par] = 1e+10
    ii = 0
    for i in xrange(npoint):
        if (i < ni):
            Paramset2[i, :] = Paramset1[i + 1, :]
        else:
            Param = Best_Points[ii]
            for j in xrange(num_par + 1):
                Paramset2[i, j] = Param[j]
            Paramset2[i, j + 1] = kk5 * Param[num_par] + cc5                                ## determine proportional weight via linear interpolation
                                                                                            ## f(x) = m*x + b
            ii += 1                                                                         ## increase counter

            # Debug:
            # print "i,  Paramset2[i, :] = ", i,  Paramset2[i, :]


        ## check for NaN
        for j in xrange(num_par + 1):
            if (isnan(Paramset2[i, j])):
                Paramset2[i, j] = 0.0


        ## define new point of minimum after error estimation (if it exists!)
        if (min_paramset[num_par] > Paramset2[i, num_par]):
            min_paramset[0:(num_par + 1)] = Paramset2[i, 0:(num_par + 1)]


    ## number of parameter values for estimation of error for parameter <nvar>
    num_set2 = len(Paramset2[:, nvar])                                                      ## is equal to npoint ?????


    ## create <Param_one> for calling of subroutine <error_info>
    Param_one = numpy.array(range(num_set2 * 3), dtype = 'float')
    Param_one.shape = (num_set2, 3)
    Param_one[:, 0] = Paramset2[:, nvar]                                                    ## extract all values for parameter "param[nvar]"
    Param_one[:, 1] = Paramset2[:, num_par]                                                 ## get the corresponding chi^2 values
    Param_one[:, 2] = Paramset2[:, num_par + 1]                                             ## get the corresponding proportional weights


    ## calculate mean value and standard deviation for parameter <nvar> calling subroutine <error_info>
    minima = 0.0
    stdev = 0.0
    minima, stdev = error_info(min_point, nvar, Param_one)


    ## print some useful informations to screen
    if (printflag == "true"):
        print "\n"
        print "           Number of function evaluations with 3sigma boundary: ", len(Points)
        print "           Evidence = ", logZ
        print "\n"


    ## define return parameters
    return (Points, Paramset2[:,nvar], num_set2, minima, stdev, logZ, Paramset1, Bestinput)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Error estimator using INS
def ErrorEstimINS(printflag, modelClass, fit_log, paramchannel, logchannel, fitlogChi2, Chi2Channel, RenormalizedChi2, NumberExpFiles, \
                  NumberYColumns, LengthExpRange, num_par, chilm, max_iter, param_up_down, min_point, plotflag, ErrINSCounter):
    """

    There are not special algorithms setting; ONLY the maximum number of iteration (max_iter) in Error-Estimator_INS_Parameters.xml
    Number of function calls is probable: 4 x max_iter x num_par
    For possible parallelization use <def check_point_paral>: for parallel calculations use list <Points_paral> with parameter vectors


    input variables:            printflag:              flag indicating if messages are printed to screen or not
                                modelClass:             class for model function call
                                fit_log:                path and name of log file
                                paramchannel:           channel for parameter log file
                                logchannel:             channel for main log file
                                fitlogChi2:             path and name of chi2 log file
                                Chi2Channel:            channel for chi2 log file
                                RenormalizedChi2:       renormalization flag
                                NumberExpFiles:         number of obs. data files
                                NumberYColumns:         number of y columns for each data file
                                LengthExpRange:         number of data points for each data file
                                num_par:                number of parameter
                                chilm:                  lower limit for chi^2
                                max_iter:               max. number of iterations
                                param_up_down:          parameters with upper and lower limits
                                min_point:              parameter describing point of minimum
                                plotflag:               flag for plotting
                                ErrINSCounter:          counter for error estimation algorithm call

    output variables:           calstatus:              status flag of calculation (= 0: all ok)
                                Error_left:             list of all left errors
                                Error_right:            list of all right errors
                                error_tag_content:      line for xml file for each free parameter
    """


    ## initialize return parameter
    calstatus = 0
    error_tag_content = []


    ## lower limit of chi^2 for stopping condition normalized to the number of calculation points
    chilim = 0.0
    if (RenormalizedChi2 == 1):
        for i in xrange(NumberExpFiles):
            chilim = chilim + (NumberYColumns[i] * LengthExpRange[i] - num_par) * abs(chilm)
        chilim = abs(chilim)
        screen_out = "           Renormalized limit for chi^2 = %25.15e" % chilim
        logchannel.write(screen_out + "\n")
        logchannel.write("\n")
        if (printflag == "true"):
            print "\n"
            print screen_out
            print " "
    else:
        chilim = abs(chilm)
        screen_out = "           Limit for chi^2 = %25.15e" % chilim
        logchannel.write(screen_out + "\n")
        logchannel.write("\n")
        if (printflag == "true"):
            print "\n"
            print screen_out
            print " "
    logchannel.flush()


    ## Define output lists for return from main subroutine : Point, Paramset, Mean_values,  Err_mean_values
    Points = []
    Points0 = []


    ## Paramset0 contains samples of parameters values for each parameters
    h = (max_iter * 1000)
    Paramset0 = numpy.array(range(num_par * h), dtype = 'float')
    Paramset0.shape = (num_par, h)


    ## num_sets contains sizes of samples in Paramset0
    num_sets = numpy.array (range(num_par), dtype = 'float')
    Mean_values = numpy.array (range(num_par), dtype = 'float')
    Err_mean_values = numpy.array (range(num_par), dtype = 'float')
    all_logZ = numpy.array (range(num_par), dtype = 'float')


    ## uniform random number distrubution
    uniform = random.random


    ## reset arrays for chi2 distribution
    AllChi2DistributionParamValues = []
    AllChi2DistributionChi2Values = []


    ## Apply INS-algorithm for each parameters, other parameters are kept CONSTANTS ######################################################################
    for i_ins in xrange(num_par):
        modelClass.SetCurrentParameterIndex(i_ins)


        ## define working interval (parameter space) for parameter <i_ins>
        pud = numpy.array(range(2), dtype = 'float')
        pud[0] = param_up_down[i_ins, 0]
        pud[1] = param_up_down[i_ins, 1]


        ## call <Interval_NS> for error estimation for parameter <i_ins>
        Points11, Paramira, num_sets[i_ins], Mean_values[i_ins], Err_mean_values[i_ins], all_logZ[i_ins], Paramset1, \
        Bestinput = Interval_NS(min_point, i_ins, pud, max_iter, paramchannel, logchannel, printflag, modelClass, num_par, param_up_down)


        ## save chi2 distribution for the current parameter to global arrays
        Chi2DistributionParam, Chi2DistributionChi2 = modelClass.SaveChi2Distribution()
        AllChi2DistributionParamValues.append(Chi2DistributionParam)
        AllChi2DistributionChi2Values.append(Chi2DistributionChi2)


        ## devine the number of points <nset> for error estimation
        nset = int(num_sets[i_ins])

        # Debug:
        # print "i_ins = ", i_ins
        # print ">>>nset = ", nset
        # print "Paramira[0:nset] = ", Paramira[0:nset]


        Paramset0[i_ins, 0:nset] = Paramira[0:nset]
        Points0.append(Points11)


    ## in the following we are no longer interested in the chi2 distribution determined by the Interval Nested sampling algorithm
    modelClass.SetCurrentParameterIndex((-1))


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define vectors for errors from the left and right sides of the point of minimum
    Error_left = numpy.array(range(num_par), dtype='float')
    Error_right = numpy.array(range(num_par), dtype='float')


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## Building of histogramms for parameter values after INS-algorithm
    ## Calculate left and right errors from information about mean and stdev
    parud = numpy.array(range(2), dtype='float')
    for i in xrange(num_par):


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## calculate errors from the left and right sides of the point of minimum


        ## check if mean value is equal parameter value at the point of minimum
        if (Mean_values[i] == min_point[i]):
           Error_left[i] = Err_mean_values[i]
           Error_right[i] = Err_mean_values[i]


        ## ??
        if ((Mean_values[i] >= 0.0 and min_point[i] >= 0.0) or (Mean_values[i] < 0.0 and min_point[i] < 0.0)):
             rast = abs(Mean_values[i] - min_point[i])
        if ((Mean_values[i] >= 0.0 and min_point[i] < 0.0) or (Mean_values[i] < 0.0 and min_point[i] >= 0.0)):
             rast = abs(Mean_values[i]) + abs(min_point[i])


        ## if mean value is from right side of the parameter value at the point of minimum
        if (Mean_values[i] > min_point[i]):
            if (Err_mean_values[i] > rast):
                Error_left[i] = Err_mean_values[i] - rast
                Error_right[i] = Err_mean_values[i] + rast
            else:
                Error_left[i] = 0.0
                Error_right[i] = 2.0 * Err_mean_values[i]


        ## if mean value is from left side of the parameter value at the point of minimum
        if (Mean_values[i] < min_point[i]):
            if (Err_mean_values[i] > rast):
                Error_left[i] = Err_mean_values[i] + rast
                Error_right[i]= Err_mean_values[i] - rast
            else:
                Error_left[i] = 2.0 * Err_mean_values[i]
                Error_right[i] = 0.0


        ## check if (min-error_left; min+error_right) lies in the parameter space
        if ((min_point[i] - Error_left[i]) < param_up_down[i, 0]):                          ## check if Error_left is smaller than lower limit
            Error_left[i] = abs(min_point[i] - param_up_down[i, 0])
        if ((min_point[i] + Error_right[i]) > param_up_down[i, 1]):                         ## check if Error_right is larger than upper limit
            Error_right[i] = abs(param_up_down[i, 1] - min_point[i])


        ## ??
        nset = int(num_sets[i])
        parud[0] = param_up_down[i, 0]
        parud[1] = param_up_down[i, 1]

        # Debug:
        # print "==>nset = ", nset


        ## call subroutine for histogramms (with saving!)
        if (plotflag == "true" or plotflag == "saveonly"):
            HelpArray = Paramset0[i, 0:nset]
            my_hist1(HelpArray, parud, min_point[i], i, num_par, fit_log, ErrINSCounter, plotflag)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define best point from the list <Points>; maybe we find new minimum point
    PP = Points0[0]
    nnn = len(PP)
    new_min_point = numpy.array (range(num_par + 1), dtype = 'float')
    new_min_point[num_par] = 1e+3


    ## refine the point of minimum (from the list<Points0>
    for ii in xrange(nnn):
         if (new_min_point[num_par] > PP[ii][num_par]):
            new_min_point[:] = PP[ii][:]


    ## end of INS-process --------------------------------------------------------------------------------------------------------------------------------


    ## construct final value for error tag in xml-file
    ## tag_content = Error_left[j], Error_right[j], Mean_values[j], Err_mean_values[j], all_logZ[j]
    error_tag_content = []
    for j in xrange(num_par):                                                               ## loop over all free parameter
        tag_content =[abs(Error_left[j]), abs(Error_right[j]), Mean_values[j], Err_mean_values[j], all_logZ[j]]
        error_tag_content.append(tag_content)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## print results to screen and log-file
    fit_log = fit_log.strip()
    i = fit_log.rfind("/")
    if (i > (-1)):
        path_to_ErrorFile = fit_log[:i + 1]
    else:
        path_to_ErrorFile = ""
    path_to_ErrorFile = path_to_ErrorFile + "ErrorEstim_INS__error-values__call_" + str(abs(ErrINSCounter)) + ".dat"
    ErrorFile = open(path_to_ErrorFile, 'w')
    logline1 = "           "
    logline1 += "Table of results:"
    logline2 = "           | Free parameter "
    logline2 += "|    Minimum point  |   Error_left    |   Error_right   | Mean parameter value |  Standard deviation  |  log(Evidence)   |"
    HelpString = " Free parameter      Minimum point      Error_left        Error_right     Mean parameter value    Standard deviation    log(Evidence)"
    ErrorFile.write(HelpString + "\n")
    HelpString = "-" * 139
    logline3 = "           " + HelpString
    logchannel.write("\n")
    logchannel.write(logline1 + "\n")
    logchannel.write("\n")
    logchannel.write("\n")
    logchannel.write(logline2 + "\n")
    logchannel.write(logline3 + "\n")
    if (printflag == "true"):
        print logline1
        print "\n"
        print logline2
        print logline3
    for j1 in xrange(num_par):
        logline1 = "           |     % 6i     |% 18.8e |% 16.8e |% 16.8e |% 21.8e |% 21.8e |% 17.8e |"      \
                    % ((j1 + 1), min_point[j1], abs(Error_left[j1]),  abs(Error_right[j1]), Mean_values[j1], Err_mean_values[j1], all_logZ[j1])
        logline1a = "    % 6i      % 18.8e  % 16.8e  % 16.8e  % 21.8e  % 21.8e  % 17.8e"      \
                    % ((j1 + 1), min_point[j1], abs(Error_left[j1]),  abs(Error_right[j1]), Mean_values[j1], Err_mean_values[j1], all_logZ[j1])
        ErrorFile.write(logline1a + "\n")
        helpString = "-" * 140
        logline2 = "           " + helpString
        logchannel.write(logline1 + "\n")
        logchannel.write(logline2 + "\n")
        if (printflag == "true"):
            print logline1
            print logline2
    logchannel.write("\n")
    logchannel.write("\n")
    if (printflag == "true"):
        print "\n"
        print "\n"
    logchannel.flush()
    ErrorFile.close()


    ## define return parameters
    AddInfo = [AllChi2DistributionParamValues, AllChi2DistributionChi2Values, Bestinput]
    return (calstatus, Error_left, Error_right, error_tag_content, AddInfo)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## calculate mode(s) of list
## taken from http://stackoverflow.com/questions/10797819/finding-the-mode-of-a-list-in-python, and modified
def modeList(arr, LocalLeftLimit, LocalRightLimit, BestFitPoint):
    """
    input variables:            arr:                    list which is analyzed
                                LocalLeftLimit:         lower limit of list elements
                                LocalRightLimit:        upper limit of list elements
                                BestFitPoint:           parameter value of best fit

    output variables:           LocalMode:              calculated list of mode(s)
    """


    ## initialize return parameter
    LocalMode = []


    ## select list elements within given limits
    LocalList = []
    for a in arr:
        if (LocalLeftLimit <= a and a <= LocalRightLimit):
            LocalList.append(a)


    ## calculate mode(s)
    if (LocalList != []):
        m = max([LocalList.count(a) for a in LocalList])
        if (m > 1):
            LocalMode = [x for x in LocalList if LocalList.count(x) == m][0]

        # Debug:
        # print "old LocalMode = ", LocalMode


        ## compute histogram for selected list elements
        hist, bin_edges = numpy.histogram(LocalList, bins=20)
        BinDist = numpy.diff(bin_edges)[0]
        ii = numpy.argmax(hist)
        LocalMode = bin_edges[ii]
        if (abs(BestFitPoint - LocalMode) <= BinDist):                                      ## if mode is close to best fit value, use best fit value
            LocalMode = max(LocalLeftLimit, BestFitPoint)
            LocalMode = min(BestFitPoint, LocalRightLimit)

        # Debug:
        # print "new LocalMode = ", LocalMode
        # print "BestFitPoint = ", BestFitPoint


    ## define return parameters
    return LocalMode
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Error estimator using MCMC
def ErrorEstimMCMC(printflag, modelClass, fit_log, paramchannel, logchannel, fitlogChi2, Chi2Channel, RenormalizedChi2, NumberExpFiles, \
                   NumberYColumns, LengthExpRange, num_par, chilm, max_iter, param_up_down, min_point, plotflag, ErrINSCounter, NumberMCMCSampler, \
                   NumberBurnInIter, SigmaMultiplicity, UsePreviousResultsFlag, JobDir):
    """
    input variables:            printflag:              flag indicating if messages are printed to screen or not
                                modelClass:             class for model function call
                                fit_log:                path and name of log file
                                paramchannel:           channel for parameter log file
                                logchannel:             channel for main log file
                                fitlogChi2:             path and name of chi2 log file
                                Chi2Channel:            channel for chi2 log file
                                RenormalizedChi2:       renormalization flag
                                NumberExpFiles:         number of obs. data files
                                NumberYColumns:         number of y columns for each data file
                                LengthExpRange:         number of data points for each data file
                                num_par:                number of parameter
                                chilm:                  lower limit for chi^2
                                max_iter:               max. number of iterations
                                param_up_down:          parameters with upper and lower limits
                                min_point:              parameter describing point of minimum
                                plotflag:               flag for plotting
                                ErrINSCounter:          counter for error estimation algorithm call
                                NumberMCMCSampler:      number of samplers
                                NumberBurnInIter:       number of iterations for burn-in phase
                                SigmaMultiplicity:      multiplicity of standard deviation (1sigma, 2sigma, 3sigma, etc.)
                                UsePreviousResultsFlag: use previous calculated parameter vectors or not
                                JobDir:                 path and name of current job directory

    output variables:           calstatus:              status flag of calculation (= 0: all ok)
                                Error_left:             list of all left errors
                                Error_right:            list of all right errors
                                error_tag_content:      line for xml file for each free parameter
    """


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return parameter
    calstatus = 0
    Error_left = numpy.array(range(num_par), dtype='float')
    Error_right = numpy.array(range(num_par), dtype='float')
    error_tag_content = []
    AddInfo = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import modified emcee and MCMC packages
    import emcee__modified
    import MCMC


    ## start MCMC algorithm
    SampleMethod = "local"
    UsedFor = "error"
    ndim = int(num_par)                                                                     ## dimension of parameter space
    calstatus, AddInfo = MCMC.MCMC(printflag, modelClass, fit_log, paramchannel, logchannel, fitlogChi2, Chi2Channel, RenormalizedChi2, NumberExpFiles, \
                                   NumberYColumns, LengthExpRange, num_par, chilm, max_iter, param_up_down, min_point, plotflag, ErrINSCounter, \
                                   NumberMCMCSampler, NumberBurnInIter, JobDir, UsePreviousResultsFlag, SampleMethod, UsedFor)
    pos = AddInfo[0]
    prob = AddInfo[1]
    bestinput = AddInfo[2]
    sampler = AddInfo[3]
    bestchi = AddInfo[4]
    bestparam = AddInfo[5]
    FinalSamplesOrig = sampler.chain[:, :, :].reshape((-1, ndim))
    FinalSamples = numpy.atleast_1d(FinalSamplesOrig)
    if len(FinalSamples.shape) == 1:
        FinalSamples = numpy.atleast_2d(FinalSamples)
    else:
        assert len(FinalSamples.shape) == 2, "The input sample array must be 1- or 2-D."
        FinalSamples = FinalSamples.T


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## post processing


    ## define working directory
    LocalFitLog = fitlogChi2.strip()
    mm = LocalFitLog.rfind("/")
    if (mm > (-1)):
        path_to_save_png = LocalFitLog[:mm + 1]
    else:
        path_to_save_png = ""


    ## define error bounds using scipy.special.ndtr (Gaussian cumulative distribution function, returns the area under the standard Gaussian probability
    ## density function, integrated from minus infinity to x: 1/sqrt(2*pi) * integral(exp(-t**2 / 2), t=-inf..x)
    sigma = ndtr(SigmaMultiplicity) - ndtr(-SigmaMultiplicity)
    sigmaLimits = [50 - (sigma * 1.e2 / 2.0), 50, 50 + (sigma * 1.e2 / 2.0)]

    # Debug:
    #    print "sigmaLimits = ", sigmaLimits


    ## create plots for each parameter describing the parameter values for each iteration
    binVal = 400
    ErrorTyp = "HPD"
    ErrorTyp = ErrorTyp.lower()
    RefPointList = []
    LocalErrorLeftList = []
    LocalErrorRightList = []
    LabelList = []
    ErrorList = []
    for ParamIndex, xs in enumerate(FinalSamples):
        LabelList.append("free param. %i" % (ParamIndex + 1))
        LocalErrorList = []

        # Debug:
        #    print "xs[0] = ", xs[0]
        #    print "xs = ", xs


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create histogramm for each free parameter including different error bars
        fig = pylab.figure(figsize=(15, 10))
        fig.clear()
        layer = pylab.subplot(1, 1, 1)
        pylab.grid(True)
        pylab.xlabel("free parameter %i" % (ParamIndex + 1))
        if hasattr(xs, "compressed"):
            xs = xs.compressed()
        n, b, p = layer.hist(xs, histtype='step', normed=True, bins=binVal, lw=1, fill=True)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## compute quantile credible regions
        means = xs.mean()                                                                   ## compute mean
        stds = xs.std()                                                                     ## compute std.
        LocalErrorLeftGauss = means - (1.0 + sigma) * stds
        LocalErrorRightGauss = means + (1.0 + sigma) * stds
        RefGauss = means

        # Debug:
        #    print "LocalErrorLeftGauss = ", LocalErrorLeftGauss
        #    print "LocalErrorRightGauss = ", LocalErrorRightGauss


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## compute quantile credible regions (use numpy.percentile)
        perctiles = numpy.percentile(xs, sigmaLimits)                                       ## compute perctiles for corresponding sigma multiplicity
        LocalErrorLeftQuant = min(perctiles[0], perctiles[2])                               ## get lower limit of credible interval
        LocalErrorRightQuant = max(perctiles[0], perctiles[2])                              ## get upper limit of credible interval
        RefQuantile = perctiles[1]                                                          ## get median of xs

        # Debug:
        #    print "LocalErrorLeftQuant = ", LocalErrorLeftQuant
        #    print "LocalErrorRightQuant = ", LocalErrorRightQuant


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## compute HPD credible regions
        # hpds = emcee__modified.hpdPyMC(xs, (sigma * 1.e-2))                               ## use pymc routine to compute HPD interval
        hpds = emcee__modified.hpd(xs, sigma)                                               ## use caltech routine to compute HPD interval
        LocalErrorLeftHPD = min(hpds[0], hpds[1])                                           ## get lower limit of HPD interval
        LocalErrorRightHPD = max(hpds[0], hpds[1])                                          ## get upper limit of HPD interval
        RefHPD = modeList(xs, LocalErrorLeftHPD, LocalErrorRightHPD, bestparam[ParamIndex]) ## determine mode

        # Debug:
        #    print "hpds = ", hpds
        #    print "hpds2 = ", hpds2
        #    print "RefHPD = ", RefHPD
        #    print "LocalErrorLeftHPD = ", LocalErrorLeftHPD
        #    print "LocalErrorRightHPD = ", LocalErrorRightHPD


        ## check if mode lies within HPD interval
        # if (modes < LocalErrorLeftHPD or LocalErrorRightHPD < modes):
        #    print " "


        ## store calculated errors
        if (ErrorTyp == "gauss"):
            RefPoint = RefGauss
            LocalErrorLeft = LocalErrorLeftGauss
            LocalErrorRight = LocalErrorRightGauss
        elif (ErrorTyp == "quantile"):
            RefPoint = RefQuantile
            LocalErrorLeft = LocalErrorLeftQuant
            LocalErrorRight = LocalErrorRightQuant
        elif (ErrorTyp == "hpd"):
            RefPoint = RefHPD
            LocalErrorLeft = LocalErrorLeftHPD
            LocalErrorRight = LocalErrorRightHPD
        Error_left[ParamIndex] = RefPoint - abs(LocalErrorLeft)
        Error_right[ParamIndex] = abs(LocalErrorRight) - RefPoint
        tag_content =[abs(Error_left[ParamIndex]),  abs(Error_right[ParamIndex])]
        error_tag_content.append(tag_content)
        RefPointList.append(RefPoint)
        LocalErrorLeftList.append(LocalErrorLeft)
        LocalErrorRightList.append(LocalErrorRight)


        ## print errors
        print "\n\n\t Errors for free param. %i" % (ParamIndex + 1)
        print "\t Gaussian:  = ", LocalErrorLeftGauss, LocalErrorRightGauss
        print "\t Percentile = ", LocalErrorLeftQuant, LocalErrorRightQuant
        print "\t HPD        = ", LocalErrorLeftHPD, LocalErrorRightHPD


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## plot error levels
        layer.axvline(x = RefPoint, color = 'k', linewidth = 1, linestyle = '--')
        layer.axvline(x = LocalErrorLeft, color = 'r', linewidth = 1, linestyle = '--')
        layer.axvline(x = LocalErrorRight, color = 'r', linewidth = 1, linestyle = '--')


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## save plots
        ylims = layer.get_ylim()                                                            ## get y-limits
        layer.set_ylim(ylims[0], ylims[1] * 1.07)
        layer.set_xlim(min(param_up_down[ParamIndex, 0], param_up_down[ParamIndex, 1]), max(param_up_down[ParamIndex, 0], param_up_down[ParamIndex, 1]))
        if (plotflag == "true" or plotflag == "saveonly"):
            pylab.savefig(path_to_save_png + "ErrorEstim_MCMC__histogram_of_free-parameter_parm_" + str(ParamIndex + 1) + "__call_" \
                          + str(abs(ErrINSCounter)) + ".png", format='png')
        pylab.draw()
        matplotlib.pyplot.close(fig)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## do corner plots for models with a few parameters
    if (ndim < 1e99):
        print "\n\t Create corner plots ..",


        if (ErrorTyp == "gauss" or ErrorTyp == "hpd"):
            ErrorMarks = []
            for i, LocalRefPoint in enumerate(RefPointList):
                ErrorMarks.append([LocalErrorLeftList[i], LocalRefPoint, LocalErrorRightList[i]])


        ## define type of error
        if (ErrorTyp == "gauss"):
            LocalLevels = (1.0 - numpy.exp(-0.5 * SigmaMultiplicity**2),)
        elif (ErrorTyp == "quantile"):
            ErrorMarks = [[sigmaLimits[0] * 1.e-2, sigmaLimits[1] * 1.e-2, sigmaLimits[2] * 1.e-2]]
            LocalLevels = (1.0 - numpy.exp(-0.5 * SigmaMultiplicity**2),)
        elif (ErrorTyp == "hpd"):
            LocalLevels = (1.0 - numpy.exp(-0.5 * SigmaMultiplicity**2),)


        ## start corner package
        fig = emcee__modified.corner(FinalSamplesOrig, bin = binVal, labels = LabelList, truths = bestparam, quantiles = ErrorMarks, \
                                     show_titles = True, title_kwargs = {"fontsize": 12}, levels = LocalLevels, ErrorTyp = ErrorTyp)
        fig.savefig(path_to_save_png + "corner-triangle__call_" + str(abs(ErrINSCounter)) + ".png")
        print "done!"


    ## Print out the mean acceptance fraction. In general, acceptance_fraction has an entry for each walker so, in this case, it is a ndim-dimensional
    ## vector.
    LogMessage = ["\n\n\t Mean acceptance fraction:  " + str(numpy.mean(sampler.acceptance_fraction)) + "\n\n"]
    for line in LogMessage:
        print line
        logchannel.write(line + "\n")


    ## Estimate the integrated autocorrelation time for the time series in each parameter.
    AutoCorrTime = sampler.get_autocorr_time()
    LogMessage = ["\n\n\t Autocorrelation time for each free parameter:"]
    for i in xrange(ndim):
        NewLine = "\t\t Autocorrelation time for parameter %i:  %.4e" % ((i + 1), AutoCorrTime[i])
        LogMessage.append(NewLine)
    LogMessage.append("\n\n")
    for line in LogMessage:
        print line
        logchannel.write(line + "\n")

    # Debug:
    # sys.exit(0)


    ## define return parameters
    return (calstatus, Error_left, Error_right, error_tag_content, AddInfo)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## Error estimator using Fisher-matrix method
def ErrorEstimFisher(printflag, modelClass, fit_log, paramchannel, logchannel, fitlogChi2, Chi2Channel, RenormalizedChi2, NumberExpFiles, \
                     NumberYColumns, LengthExpRange, num_par, chilm, max_iter, param_up_down, min_point, plotflag, ErrINSCounter, VariationValue):
    """
    input variables:            printflag:              flag indicating if messages are printed to screen or not
                                modelClass:             class for model function call
                                fit_log:                path and name of log file
                                paramchannel:           channel for parameter log file
                                logchannel:             channel for main log file
                                fitlogChi2:             path and name of chi2 log file
                                Chi2Channel:            channel for chi2 log file
                                RenormalizedChi2:       renormalization flag
                                NumberExpFiles:         number of obs. data files
                                NumberYColumns:         number of y columns for each data file
                                LengthExpRange:         number of data points for each data file
                                num_par:                number of parameter
                                chilm:                  lower limit for chi^2
                                max_iter:               max. number of iterations
                                param_up_down:          parameters with upper and lower limits
                                min_point:              parameter describing point of minimum
                                plotflag:               flag for plotting
                                ErrINSCounter:          counter for error estimation algorithm call
                                VariationValue:         variation value

    output variables:           calstatus:              status flag of calculation (= 0: all ok)
                                Error_left:             list of all left errors
                                Error_right:            list of all right errors
                                error_tag_content:      line for xml file for each free parameter
    """


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize return parameter
    calstatus = 0
    Error_left = numpy.array(range(num_par), dtype='float')
    Error_right = numpy.array(range(num_par), dtype='float')
    error_tag_content = []
    AddInfo = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## calculate chi2 values for parameter vectors
    NewParamVectorGradient = min_point.copy()
    NumParamVec = 1
    ValuesOfChi2 = modelClass.logLhood(NewParamVectorGradient, NumParamVec, VariationValue)
    ValuesOfChi2[:] = [x * (-1) for x in ValuesOfChi2]                                      ## we want the normal chi^2 distribution

    # Debug:
    # print "ValuesOfChi2 = ", ValuesOfChi2
    # print "Listh = ", Listh


    ## save chi2 distribution for the current parameter to global arrays
    bestchi, bestparam, bestline, bestinput = modelClass.GetBestResult()
    AddInfo = [[], [], bestinput]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## calculate covariance (Fisher) matrix
    CovarianceMatrix = modelClass.GetCovarianceMatrix()
    # CovarianceMatrix = ValuesOfChi2[0] * CovarianceMatrix

    # Debug:
    # print "CovarianceMatrix = ", CovarianceMatrix


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## calculate eigenvalues of covariance (Fisher) matrix
    try:
        ew, ev = numpy.linalg.eig(CovarianceMatrix)
    except:
        ex = 0.0


    ## check, if at least on eigenvalue is less zero and write messages to screen and log file
    PosEWFlag = True
    for LocalEW in ew:
        if (LocalEW <= 0.0):
            PosEWFlag = False
            break
    if (not PosEWFlag):
        LogMessage = ["\n\n\t Error in subroutine ErrorEstimator.ErrorEstimFisher:", \
                      "\n\t\t The covariance matrix has one or more eigenvalues which are negative or zero.", \
                      "\t\t So, the covariance matrix is NOT positive definite and the likelihood is", \
                      "\t\t N O T Gaussian and the calculated errors are wrong!", \
                      "\n\t Eigenvalues of covariance matrix:", \
                      "\n\t\t " + str(numpy.sort(ew)) + "\n\n"]
    else:
        LogMessage = ["\n\n\t Eigenvalues of covariance matrix are all positive, i.e. likelihood might be Gaussian\n"
                      "\t and calculated errors might be useful!\n\n"]
    for line in LogMessage:
        print line
        logchannel.write(line + "\n")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## calculate determinant of covariance (Fisher) matrix
    try:
        det = numpy.linalg.det(CovarianceMatrix)
    except:
        det = 0.0


    ## check, if determinant is less than zero and write messages to screen and log file
    if (det <= 0.0):
        LogMessage = ["\n\n\t Error in subroutine ErrorEstimator.ErrorEstimFisher:", \
                      "\n\t\t The determinant of the covariance matrix is less or equal to zero.", \
                      "\t\t So, the covariance matrix is NOT positive definite and the likelihood is", \
                      "\t\t N O T Gaussian and the calculated errors are wrong!", \
                      "\n\t Eigenvalues of covariance matrix:", \
                      "\n\t\t " + str(numpy.sort(ew)) + "\n\n"]
    else:
        LogMessage = ["\n\n\t The determinant of the covariance matrix is positive, i.e. likelihood might be Gaussian\n"
                      "\t and calculated errors might be useful!\n\n"]
    for line in LogMessage:
        print line
        logchannel.write(line + "\n")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## determine errors
    try:
        InverseCovarianceMatrix = numpy.linalg.inv(CovarianceMatrix)
    except:
        InverseCovarianceMatrix = CovarianceMatrix * 0.0

    # Debug:
    # print "InverseCovarianceMatrix = ", InverseCovarianceMatrix


    for i in xrange(num_par):
        if (InverseCovarianceMatrix[i, i] != numpy.nan):
            LocalError = numpy.absolute(InverseCovarianceMatrix[i, i])**0.5
        else:
            LocalError = numpy.nan
        Error_left[i] = LocalError
        Error_right[i] = LocalError
        tag_content =[abs(Error_left[i]),  abs(Error_right[i])]
        error_tag_content.append(tag_content)

        # Debug:
        print "\t free-parameter: %i, Error = %.4f" % (i + 1, LocalError)


    ## write trace of inverse covariance matrix
    print "\n\n"
    TraceCovMatrix = [InverseCovarianceMatrix[i, i] for i in xrange(num_par)]
    LogMessage = ["\t Diagonal elements of inverse covariance matrix:\n\n\t\t " + str(TraceCovMatrix) + "\n\n\n\n"]
    for line in LogMessage:
        print line
        logchannel.write(line + "\n")


    ## define return parameters
    return (calstatus, Error_left, Error_right, error_tag_content, AddInfo)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##********************************************************************************************************************************************************
## main routinte for Error estimation
def ErrorEstimator(SpecialAlgorithmSettings, GeneralAlgorithmSettings, printflagNum, LastAlgorithmNum, ParamSetCounter, chilm, numiter, ErrINSCounter, \
                   DeterminationChi2, PlotIteration, PlotType, fit_log, NumberInputFiles, NumberOutputFiles, ParallelizationFlag, JobID, MaxInputLines, \
                   MaxParameter, RenormalizedChi2, currentpath, CalculationMethod, xAxisLabel, yAxisLabel, zAxisLabel, PathStartScript, \
                   ExeCommandStartScript, parameter_set, ExpDataX, ExpDataY, ExpDataError, NumberRanges, MinRange, MaxRange, NumberXColumns, \
                   NumberYColumns, LengthExpRange, MaxRangeNumber, NumberExpFiles, MaxLength, MaxColX, MaxColY, NumberParameter, plotflag, modelflag, \
                   MPIFlag, MAGIXrootDirectory, JobDir):
    """
    input variables:            SpecialAlgorithmSettings:   algorithm user setttings
                                GeneralAlgorithmSettings:   special algorithm settings
                                printflagNum:           flag for screen output 1 (=yes) or 0 (=no)
                                LastAlgorithmNum:       number of last algorithm
                                ParamSetCounter:        number of best sites
                                chilm:                  user defined abort criteria for chi**2
                                numiter:                max. number of iterations
                                ErrINSCounter:          counts number of calls
                                DeterminationChi2:      method being used for the determination of chi^2
                                PlotIteration:          plot model function for each iteration set 1(=yes) or 0(=no)
                                PlotType:               get type of plot
                                fit_log:                path for log-file containing the current values of chi**2
                                NumberInputFiles:       number of input files for the external model program
                                NumberOutputFiles:      number of output files for the external model program
                                ParallelizationFlag:    contains the number of processors used for parallelization
                                JobID:                  job identification number
                                MaxInputLines:          max number of lines in an input file
                                MaxParameter:           max number of parameters in a line of an input file
                                RenormalizedChi2:       flag for using renormalized chi**2
                                currentpath:            path of the working directory
                                CalculationMethod:      method of computation (at once or point-to-point)
                                xAxisLabel:             label of the x-axis (for plot)
                                yAxisLabel:             label of the y-axis (for plot)
                                zAxisLabel:             label of the z-axis (for plot)
                                PathStartScript:        path and name of the start script for calling model function
                                ExeCommandStartScript:  command for calling model function
                                parameter_set:          the complete set of paramters (incl. flags and limits)
                                ExpDataX:               array containing the experimental x side
                                ExpDataY:               array containing the experimental y side
                                ExpDataError:           array containing the experimental error of the y side
                                NumberRanges:           number of ranges
                                MinRange:               array containing lower limit of each range for each file
                                MaxRange:               array containing upper limit of each range for each file
                                NumberXColumns:         number x-columns
                                NumberYColumns:         number y-columns
                                LengthExpRange:         total number of data points
                                MaxRangeNumber:         max number of ranges over all files
                                NumberExpFiles:         number of exp. data files
                                MaxLength:              max. length of experimental data
                                MaxColX:                number of columns concerning to the experimental x side
                                MaxColY:                number of columns concerning to the experimental y side
                                NumberParameter:        number of model parameter
                                plotflag:               flag for plotting histograms
                                modelflag:              flag to use optimized MAGIX packages for a certain model or not
                                MPIFlag:                mpi flag
                                MAGIXrootDirectory:     MAGIX root directory
                                JobDir:                 job directory

    output variables:           error_tag_content:      complete description of the error for each free parameter
                                FitFunctionValues:      values of the model function of the last function call
                                Chi2Values:             corresponding chi^2 values
                                ErrorModelFunctionValues:   model function values with parameters modified by lower and upper errors
                                ErrorChi2Values:        chi2 function for model function values with parameters modified by lower and upper errors
    """

    ## initialize return parameters
    error_tag_content = []
    FitFunctionValues = []
    Chi2Values = []
    ErrorModelFunctionValues = []
    ErrorChi2Values = []
    NumberMCMCSampler = 100
    NumberBurnInIter = 10
    SigmaMultiplicity = 1.0
    UsePreviousResultsFlag = False


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get special settings
    ErrorMethod = SpecialAlgorithmSettings[0]                                               ## get method for error estimation
    max_iter = numiter                                                                      ## get max. number of iterations
    NumberMCMCSampler = SpecialAlgorithmSettings[1]                                         ## get number of samplers (only used for MCMC)
    NumberBurnInIter = SpecialAlgorithmSettings[2]                                          ## get number of iterations for burn-in (only used for MCMC)
    SigmaMultiplicity = float(SpecialAlgorithmSettings[3])                                  ## get multiplicity of standard deviation (only used for MCMC)
    UsePreviousResultsFlag = SpecialAlgorithmSettings[4]                                    ## get flag for usage of previous results (only used for MCMC)
    if (UsePreviousResultsFlag == 0):
        UsePreviousResultsFlag = True
    else:
        UsePreviousResultsFlag = False
    VariationValue = float(SpecialAlgorithmSettings[5])                                     ## get variation value (only used for Fisher-matrix method)
    if (ErrorMethod == 1):                                                                  ## for MCMC method
        ErrorString = "MCMC"
        ErrorLongString = "Markov chain Monte Carlo"
    elif (ErrorMethod == 2):                                                                ## for Fisher-matrix method
        ErrorString = "Fisher"
        ErrorLongString = "Fisher-matrix"
    else:
        ErrorString = "INS"
        ErrorLongString = "Interval-Nested-Sampling algorithm"

    # Debug:
    # print "ErrorMethod = ", ErrorMethod
    # print "NumberMCMCSampler = ", NumberMCMCSampler
    # print "max_iter = ", max_iter


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define num_par, min_point and param_up_down from parameter_set
    for i in range(NumberParameter):
        parameter_set[1][i] = float(parameter_set[1][i])
    num_par = int(sum(parameter_set[1]))                                                    ## determine num_par
    min_point = numpy.zeros(num_par, dtype = 'float')                                       ## determine minimum point
    param_up_down = numpy.zeros((num_par, 2), dtype = 'float')                              ## define param_up_down
    j = (-1)
    for i in range(NumberParameter):
        parameter_set[1][i] = float(parameter_set[1][i])
        if (parameter_set[1][i] == 1):
            j += 1
            min_point[j] = parameter_set[0][i]
            param_up_down[j, 0] = parameter_set[2][i]
            param_up_down[j, 1] = parameter_set[3][i]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## print what you do
    printflag = "true"
    if (printflagNum == 1):
        printflag = "true"
    else:
        printflag = "false"
    if (printflag == "true"):
        print "\n         temporary files are stored in: " + str(os.environ.get('MAGIXTempDirectory','')).strip() + "job_" + str(JobID).strip() + "/"
        print "\n\n         Start Error Estimator using " + ErrorLongString + " .."


    ## modify file names for log-files
    fit_log = fit_log.strip()
    i = fit_log.rfind("/")
    j = fit_log.rfind(".")
    if (j > i):
        fit_log = fit_log[:j] + "__ErrorEstim_" + ErrorString + "__call_" + str(abs(ErrINSCounter)).strip() + ".log"
    else:
        fit_log = fit_log + "__ErrorEstim_" + ErrorString + "__call_" + str(abs(ErrINSCounter)).strip() + ".log"

    # Debug:
    # print 'fit_log = ',fit_log


    ## open param-log file
    fitlogparam = fit_log.strip() + ".param"                                                ## define file name for param-log file
    paramchannel = open(fitlogparam.strip(), 'w')
    paramchannel.write("\n")
    LogString = "log-file containing the actual values of the parameters used in the error estimation algorithm using " + ErrorLongString + ":"
    paramchannel.write(LogString + "\n")
    LogString = "-" * len(LogString)
    paramchannel.write(LogString + "\n")
    paramchannel.write("\n")
    paramchannel.flush()


    ## open log file and get current time and date
    fitlogparam = fit_log.strip()                                                           ## define file name for param-log file
    logchannel = open(fitlogparam.strip(), 'w')
    logchannel.write("\n")
    LogString = "log-file for error estimation using " + ErrorLongString + ":"
    logchannel.write(LogString + "\n")
    LogString = "-" * len(LogString)
    logchannel.write(LogString + "\n")
    logchannel.write("\n")
    logchannel.write("\n")


    ## get local time
    lt = time.localtime()
    datestring = time.strftime("algorithm starts at Date: %d.%m.%Y", lt) + time.strftime(",     Time: %H:%M:%S", lt)
    logchannel.write(str(datestring) + "\n")
    logchannel.write("\n")
    logchannel.write("\n")
    logchannel.flush()


    ## open log file and get current time and date
    fitlogChi2 = fit_log.strip() + ".chi2"                                                  ## define file name for param-log file
    Chi2Channel = open(fitlogChi2.strip(), 'w')


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import MAGIX package ModelFunctionCall ..


    ## define name of MAGIX executable
    if (modelflag == "true" or modelflag.lower() == "myxclass"):                            ## .. optimized for myXCLASS model
        StarterExecutable = "Starter__myXCLASS.exe"
    else:                                                                                   ## .. default package
        StarterExecutable = "Starter.exe"


    ## define parameters for class
    LocalBestSiteCounter = 1
    modelClass = ModelFunctionCallClass(printflagNum, GeneralAlgorithmSettings, LastAlgorithmNum, DeterminationChi2, PlotIteration, PlotType, fit_log, \
                                        NumberInputFiles, NumberOutputFiles, ParallelizationFlag, JobID, MaxInputLines, MaxParameter, RenormalizedChi2, \
                                        currentpath, CalculationMethod, xAxisLabel, yAxisLabel, zAxisLabel, PathStartScript, ExeCommandStartScript, \
                                        parameter_set, ExpDataX, ExpDataY, ExpDataError, NumberRanges, MinRange, MaxRange, NumberXColumns, \
                                        NumberYColumns, LengthExpRange, MaxRangeNumber, NumberExpFiles, MaxLength, MaxColX, MaxColY, NumberParameter, \
                                        MPIFlag, MAGIXrootDirectory, JobDir, SpecialAlgorithmSettings, StarterExecutable, Chi2Channel, logchannel, \
                                        paramchannel, LocalBestSiteCounter)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start error estimation algorithm using different methods
    if (ErrorMethod == 1):                                                                  ## use MCMC method
        calstatus, Error_left, Error_right, error_tag_content, AddInfo = ErrorEstimMCMC(printflag, modelClass, fit_log, paramchannel, logchannel, \
                                                                                        fitlogChi2, Chi2Channel, RenormalizedChi2, NumberExpFiles, \
                                                                                        NumberYColumns, LengthExpRange, num_par, chilm, max_iter, \
                                                                                        param_up_down, min_point, plotflag, ErrINSCounter, \
                                                                                        NumberMCMCSampler, NumberBurnInIter, SigmaMultiplicity, \
                                                                                        UsePreviousResultsFlag, JobDir)


    elif (ErrorMethod == 2):                                                                ## use Fisher-matrix method
        calstatus, Error_left, Error_right, error_tag_content, AddInfo = ErrorEstimFisher(printflag, modelClass, fit_log, paramchannel, logchannel, \
                                                                                       fitlogChi2, Chi2Channel, RenormalizedChi2, NumberExpFiles, \
                                                                                       NumberYColumns, LengthExpRange, num_par, chilm, max_iter, \
                                                                                       param_up_down, min_point, plotflag, ErrINSCounter, VariationValue)

    elif (ErrorMethod == 3):                                                                ## use INS method
        calstatus, Error_left, Error_right, error_tag_content, AddInfo = ErrorEstimINS(printflag, modelClass, fit_log, paramchannel, logchannel, \
                                                                                       fitlogChi2, Chi2Channel, RenormalizedChi2, NumberExpFiles, \
                                                                                       NumberYColumns, LengthExpRange, num_par, chilm, max_iter, \
                                                                                       param_up_down, min_point, plotflag, ErrINSCounter)
    AllChi2DistributionParamValues = AddInfo[0]
    AllChi2DistributionChi2Values = AddInfo[1]
    Bestinput = AddInfo[2]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## close param-log file
    Chi2Channel.close()
    paramchannel.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read in contents of log file
    Chi2Channel = open(fitlogChi2.strip(), 'r')
    contens = Chi2Channel.readlines()                                                       ## read in contents of input file
    Chi2Channel.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## sort log.chi^2 file
    sortflag = "false"                                                                      ## define if log file is sorted or not
    if (sortflag == "true"):                                                                ## !!! sorting does not work properly !!!!
        Chi2Contents = []
        for inputline in contens:                                                           ## loop over all lines of the input file
            line = inputline.split()
            Currentline = []
            count = 0
            for elements in line:
                count += 1
                if (count == 1):
                    elements = float(elements)
                Currentline.append(elements)
            Chi2Contents.append(Currentline)


        ## sort belonging to chi^2 values
        sortchi2 = sorted(Chi2Contents)

        # Debug:
        # print Chi2Contents[0]
        # print sortchi2[0]


        ## write sorted chi^2 parameter vectors
        Chi2Channel = open(fitlogChi2.strip(), 'w')
        Chi2Channel.write("    Nr.:                   chi**2:  Parameters:\n")
        lastline = ""
        countLine = 0
        for inputline in sortchi2:
            count = 0
            formatline = ""
            for elements in inputline:
                count += 1
                if (count == 1):
                    formatline = '%27.15e        ' % float(elements)
                else:
                    k = len(elements)
                    if (k < 20):
                        formatelements = ' %20s ' % elements
                    else:
                        formatelements = " " + elements + " "
                    formatline += formatelements
            if (lastline != formatline):
                lastline = formatline
                countLine += 1
                countLineString = '%8d' % countLine
                Chi2Channel.write(countLineString + " " + str(formatline) + "\n")
        Chi2Channel.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## if log file is not sorted just write a header line at the beginning of the log file
    else:
        Chi2Channel = open(fitlogChi2.strip(), 'w')
        Chi2Channel.write("    Nr.:                   chi**2:  Parameters:\n")
        countLine = 0
        for line in contens:
            countLine += 1
            countLineString = '%8d' % countLine
            Chi2Channel.write(countLineString + " " + line)
        Chi2Channel.close()


    ## close log file
    logchannel.write("\n")
    lt = time.localtime()
    datestring = time.strftime("algorithm ends at Date: %d.%m.%Y", lt) + time.strftime(",     Time: %H:%M:%S", lt)
    logchannel.write(str(datestring) + "\n")
    logchannel.write("\n")
    helpString = "-" * 150
    logchannel.write(helpString + "\n")
    logchannel.flush()
    logchannel.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create final input files


    ## define base directory, i.e. the path where the final input file(s) are written to
    BaseDir = fit_log.strip()
    i = BaseDir.rfind("/")
    if (i > (-1)):
        BaseDir = BaseDir[:i + 1]
    else:
        BaseDir = ""


    ## extract informations from Bestinput variable
    NameInputFile = ""
    WriteFlag = False
    for line in Bestinput:
        StrippedLine = line.strip()


        ## get name of input file
        if (StrippedLine.startswith("Input-File ")):
            k = StrippedLine.find(":  , file:")
            if (k > (-1)):
                NameInputFile = StrippedLine[k + 10:].strip()


                ## modify input file name
                j = NameInputFile.rfind(".")
                if (j > (-1)):
                    NameInputFile = NameInputFile[:j] + "__ErrorEstim_" + ErrorString + "__error-values__call_" + str(abs(ErrINSCounter)) + ".out.input"
                else:
                    NameInputFile = NameInputFile + "__ErrorEstim_" + ErrorString + "__error-values__call_" + str(abs(ErrINSCounter)) + ".out.input"


                ## open channel for final input file
                NameInputFile = BaseDir + "/" + NameInputFile
                InputFile = open(NameInputFile, 'w')


        ## identify begin of contents of final input file(s)
        elif (StrippedLine.startswith("-start_input-file-")):
            WriteFlag = True


        ## identify end of contents of final input file(s)
        elif (StrippedLine.startswith("-end_input-file-")):
            try:
                InputFile.close()
            except AttributeError:
                NameInputFile = ""
            WriteFlag = False


        ## write contents of final input file(s) to current input file
        elif (WriteFlag == True and NameInputFile != ""):
            InputFile.write(line)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## determine chi2 distribution for each free parameter
    if (ErrorMethod == 3 and (plotflag == "true" or plotflag == "saveonly")):
        modelClass.extract_distribution(num_par, fitlogChi2, plotflag, ErrINSCounter, param_up_down[:, :], min_point[:], Error_left[:], Error_right[:], \
                                        AllChi2DistributionParamValues, AllChi2DistributionChi2Values)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get best function parameters
    FinalParameter, FitFunctionValues, Chi2Values = modelClass.GetBestFunctionValues()      ## get values of model function
    calstatus = 0


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## determine the model function values for the upper and the lower parameter error limits
    ErrorModelFunctionValues = []
    ErrorChi2Values = []


    ## determine function values for the upper error values
    parameter_vector = []
    for j1 in xrange(num_par):
        value = min_point[j1] + abs(Error_right[j1])
        parameter_vector.append(value)
    UpperErrorModelFunctionValues, UpperErrorChi2Values = modelClass.CalcErrorModelFunctionValues(1, parameter_vector)
    ErrorModelFunctionValues.append(UpperErrorModelFunctionValues)
    ErrorChi2Values.append(UpperErrorChi2Values)


    ## determine function values for the lower error values
    parameter_vector = []
    for j1 in xrange(num_par):
        value = min_point[j1] - abs(Error_left[j1])
        parameter_vector.append(value)

    # Debug:
    # print "lower error values: parameter_vector = ", parameter_vector

    LowerErrorModelFunctionValues, LowerErrorChi2Values = modelClass.CalcErrorModelFunctionValues(1, parameter_vector)
    ErrorModelFunctionValues.append(LowerErrorModelFunctionValues)
    ErrorChi2Values.append(LowerErrorChi2Values)

    # Debug:
    # print "lower error values: LowerErrorChi2Values = ", LowerErrorChi2Values


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define return variables for interval-nested-sampling algorithm
    return (error_tag_content, FitFunctionValues, Chi2Values, ErrorModelFunctionValues, ErrorChi2Values)
##--------------------------------------------------------------------------------------------------------------------------------------------------------
