!*********************************************************************************************************************************************************
!> Package: Bee algorithm
!>
!>
!>  This module contains the subroutines for bees algorithm
!>  Copyright (C) 2009 - 2016  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - Module BeeVariables:                      Module contains global variables and subroutines for Bees algorithm
!>      - subroutine InitalizeFloatbee:             initialize variables for bees algorithm
!>      - subroutine otherpatch:                    checks if the position of the vector is out of normalized parameter space
!>      - subroutine goto:                          defines a random vector within normalized parameter space
!>      - subroutine gotorandom:                    calculates new position in parameter space
!>      - subroutine sort_floatbee:                 sort bees with regards to their fitnesses (best fitness is first ?)
!>      - subroutine InitalizeHive:                 initialize class Hive: define initial positions and coresponding fitnesses
!>      - subroutine sendbees:                      calculates the bees (positions and corresponding fitnesses) for all bees within one iteration
!>      - subroutine nextstep:                      determines next step
!>      - subroutine InitializeLoglikebee:          initialize class loglikebee: define initial positions and coresponding fitnesses
!>      - subroutine calcfitness:                   determine the fitness of a given bee
!>      - subroutine CallBees:                      calls the different subroutines of bees algorithm
!>      - Module Algorithm:                         module contains the main subroutine used to start the different versions of the Bees algorithm
!>      - subroutine MainAlg:                       main subroutine which starts the Bees algorithm
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  I. Bernst     05.03.2010  Initial version (python)
!>  T. Moeller    14.06.2010  translation to fortran, parallelization using OpenMP
!>  T. Moeller    16.01.2012  Updated version
!>  T. Moeller    20.08.2014  myXCLASS (model) optimized version
!>  T. Moeller    26.08.2014  modified and restructured for GPU
!>
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!> Module: BeeVariables
!>
!>         Module contains global variables and subroutines for Bees algorithm
!>
!>
!> \author Thomas Moeller
!>
!> \date 09.06.2010
!>
Module BeeVariables

    use FunctionCalling
    use Model

    implicit none
    integer :: iterat, MaxIter                                                              !< current and max. iteration number
    integer :: selectedbeecount                                                             !< class hive:
    integer :: selsitescount                                                                !< class hive:
    integer :: bestbeecount                                                                 !< class hive:
    integer :: bestsitescount                                                               !< class hive:
    integer :: runcount                                                                     !< number of repeations
    integer :: max_func_counter                                                             !< counter for function

    real*8 :: minmaxTrans                                                                   !< value needed for transformation min-search to max search
    real*8 :: bestfitness                                                                   !< class hive:
    real*8 :: chilim                                                                        !< lower limit of chi**2 normalized to the total number of
                                                                                            !< all data points
    real*8, allocatable, dimension(:) :: bestsites                                          !< class hive:
    real*8, allocatable, dimension(:,:) :: bestsitesPosition                                !< positions of the best sites
    real*8, allocatable, dimension(:) :: selsites                                           !< class hive:
    real*8, allocatable, dimension(:,:) :: selsitesPosition                                 !< positions of the selected sites
    real*8, allocatable, dimension(:) :: HiveRange                                          !< class hive:
    real*8, allocatable, dimension(:) :: bestposition                                       !< class hive:
    real*8, allocatable, dimension(:) :: floatbee_fitness                                   !< class floatbee:
    real*8, allocatable, dimension(:,:) :: floatbee_position                                !< class floatbee:
    real*8, allocatable, dimension(:) :: a                                                  !< array containing all parameters
    real*8, allocatable, dimension(:) :: OptimizedParameterLowLimit                         !< lower limit for each free parameter
    real*8, allocatable, dimension(:) :: OptimizedParameterUpperLimit                       !< upper limit for each free parameter

    logical, allocatable, dimension(:) :: ia                                                !< array indicating the parameter which should be optimized


    contains


        !>------------------------------------------------------------------------------------------------------------------------------------------------
        !>
        !> class floatbee:
        !>
        !>------------------------------------------------------------------------------------------------------------------------------------------------


        !*************************************************************************************************************************************************
        !> subroutine: InitalizeFloatbee
        !>
        !> initialize variables for bees algorithm
        !>
        !> input variables:     nfit:               number of parameters which should be optimized
        !>                      number_bees:        number of bees
        !>
        !> output variables:    none
        !>
        !> \author Thomas Moeller
        !>
        !> \date 23.06.2010
        !>
        subroutine InitalizeFloatbee(nfit, number_bees)

            use Variables

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: number_bees                                                          !< total number of bees
            integer :: alloc_status, dealloc_status                                         !< variables for error handling used by allocate and deallocate


            !< (deallocate) allocate memory for several arrays
            if (allocated(floatbee_fitness)) then
                deallocate(floatbee_fitness, floatbee_position, stat = dealloc_status)
               if (dealloc_status /= 0) then                                                !< is all ok?
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine InitalizeFloatbee:")')
                    write(logchannel,'(2x,"Can not deallocate variables floatbee_fitness etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine InitalizeFloatbee:")'
                    print '(2x,"Can not deallocate variables floatbee_fitness etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("dealloc_status = ",I4)',dealloc_status
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif
            allocate(floatbee_fitness(number_bees), floatbee_position(number_bees, nfit), stat = alloc_status)
            if (alloc_status /= 0) then                                                     !< is all ok?
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine InitalizeFloatbee:")')
                write(logchannel,'(2x,"Can not allocate variables floatbee_fitness etc.")')
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("alloc_status = ",I4)') alloc_status
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine InitalizeFloatbee:")'
                print '(2x,"Can not allocate variables floatbee_fitness etc.")'
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("alloc_status = ",I4)',alloc_status
                print '(" ")'
                stop ' Program aborted!'
            endif
            floatbee_fitness = 0.d0
            floatbee_position = 0.d0

            return
        end subroutine InitalizeFloatbee


        !*************************************************************************************************************************************************
        !> subroutine: otherpatch
        !>
        !> checks if the position of the vector is out of normalized parameter space
        !>
        !> input variables:     curr_index:         index of the current bee
        !>                      nfit:               number of parameters which should be optimized
        !>                      sitesCounter:       counts the number of bees of the current site
        !>                      sitesDim:           dimension of the current site
        !>                      sitesPositions:     contains new positions
        !>                      range_list:         describes normalized parameter space
        !>
        !> output variables:    ReturnValue:        returns true if vector is out of normalized parameter space
        !>
        !> \author Irina Bernst, translation (python to fortran) by Thomas Moeller
        !>
        !> \date 05.03.2010
        !>
        subroutine otherpatch(ReturnValue, curr_index, nfit, sitesCounter, sitesDim, sitesPositions, range_list)
            !< is the position of the vector out of normalized parameter space?
            !<
            !< Note, the array range_list describes the length of the parameter space in each dimension. At the beginning of the whole algorithm each
            !< length is normalized to 1 and will be reduced for each iteration step.

            implicit none
            integer :: nfit                                                                 !< dimension of the parameter space = number of parameters
                                                                                            !< which should be optimized
            integer :: curr_index                                                           !< index of the current bee
            integer :: sitesCounter                                                         !< counts the number of bees of the current site
            integer :: sitesDim                                                             !< dimension of the current site
            integer :: i, n                                                                 !< working (loop) variables
            real*8, dimension(sitesDim, nfit) :: sitesPositions                             !< contains new positions
            real*8, dimension(nfit) :: range_list                                           !< describes normalized parameter space
            logical :: ReturnValue                                                          !< returns true if vector is out of normalized parameter space


            ReturnValue = .false.
            if (sitesCounter == 0) then                                                     !< if there are no bees return true
                ReturnValue = .true.
            endif
            Do i = 1, sitesCounter                                                          !< loop over all bees
                Do n = 1, nfit
                    if (dabs(floatbee_position(curr_index, n) - sitesPositions(i,n)) > range_list(n)) then  !< is vector out of parameter space ???
                        ReturnValue = .true.
                        exit
                    endif
                end Do
                if (ReturnValue) exit
            end Do
            return
        end subroutine otherpatch


        !*************************************************************************************************************************************************
        !> subroutine: goto
        !>
        !> defines a random vector within normalized parameter space
        !>
        !> input variables:     nfit:               number of parameters which should be optimized
        !>                      otherpos:           position of a bee
        !>                      otherFitness:       corresponding fitness
        !>                      range_list:         range_list
        !>                      ma:                 total number of all parameter
        !>                      NumFile:            number of input files
        !>                      MaxL:               max. total length
        !>                      MaxCol:             max. number of columns
        !>
        !> output variables:    otherpos:           changed position vector
        !>                      otherFitness:       fitness of changed position vector
        !>
        !> \author Irina Bernst, translation (python to fortran) by Thomas Moeller
        !>
        !> \date 05.03.2010
        !>
        subroutine goto(nfit, otherpos, otherFitness, range_list, ma, NumFile, MaxL, MaxCol)

            use Variables

            implicit none
            integer :: nfit                                                                 !< dimension of the parameter space = number of parameters
                                                                                            !< which should be optimized
            integer :: n                                                                    !< working (loop) variables
            integer :: ma                                                                   !< total number of all parameter
            integer :: NumFile                                                              !< number of input files
            integer :: MaxL                                                                 !< max. total length
            integer :: MaxCol                                                               !< max. number of columns
            real*8 :: otherFitness                                                          !< fitness for otherpos
            real*8 :: randomrange                                                           !< new range
            real*8 :: newvalue                                                              !< used for range modification
            real*8, dimension(nfit) :: range_list                                           !< describes normalized parameter space
            real*8, dimension(nfit) :: otherpos                                             !< position of a bee ???


            !< defines vector within parameter space
            Do n = 1, nfit
                randomrange = RandomWithLimits(-range_list(n), range_list(n))
                newvalue = (otherpos(n) + randomrange)


                !< check if component of the new vector is within the corresponding range
                if (newvalue >= OptimizedParameterLowLimit(n) .and. newvalue <= OptimizedParameterUpperLimit(n)) then
                    otherpos(n) = newvalue

                elseif (newvalue < OptimizedParameterLowLimit(n)) then                      !< if nth position is lower than lower limit
                    otherpos(n) = OptimizedParameterLowLimit(n)

                elseif (newvalue > OptimizedParameterUpperLimit(n)) then                    !< if nth position larger than upper limit
                    otherpos(n) = OptimizedParameterUpperLimit(n)

                endif
            end Do


            !< calculate corresponding fitness
            otherFitness = 0.d0
            if (ParallelizationFlag < 2) then


                !< if calculation reduction is chosen, read chi**2 file
                if (UseCalculationReduction .and. CurrentNumberLinesCalcReduction > 0) then
                    call CheckCalculatedParameterSets(nfit, 1)
                endif
                call calcfitness(otherFitness, ma, nfit, otherpos, NumFile, MaxL, MaxCol)
                CurrentNumberLinesCalcReduction = NumberLinesChi2                           !< update CurrentNumberLinesCalcReduction variable
            endif
            return
        end subroutine goto


        !*************************************************************************************************************************************************
        !> subroutine: gotorandom
        !>
        !> calculates new position in parameter space
        !>
        !> input variables:     nfit:               number of parameters which should be optimized
        !>                      ma:                 total number of all parameter
        !>                      a:                  values of all parameter
        !>                      NumFile:            number of input files
        !>                      MaxL:               max. total length
        !>                      MaxCol:             max. number of columns
        !>
        !> output variables:    PositionVector:     vector in parameter space
        !>
        !> \author Irina Bernst, translation (python to fortran) by Thomas Moeller
        !>
        !> \date 05.03.2010
        !>
        subroutine gotorandom(PositionVector, PositionFitness, nfit, ma, NumFile, MaxL, MaxCol)

            use Variables

            implicit none
            integer :: nfit                                                                 !< dimension of the parameter space = number of parameters
                                                                                            !< which should be optimized
            integer :: n                                                                    !< working (loop) variable
            integer :: ma                                                                   !< total number of all parameter
            integer :: NumFile                                                              !< number of input files
            integer :: MaxL                                                                 !< max. total length
            integer :: MaxCol                                                               !< max. number of columns
            real*8 :: PositionFitness                                                       !< contains the chi**2 value (fitness) for the new position
            real*8, dimension(nfit) :: PositionVector                                       !< position of vector in normalized parameter space


            PositionVector = 0.d0
            Do n = 1, nfit
                PositionVector(n) = RandomWithLimits(OptimizedParameterLowLimit(n), OptimizedParameterUpperLimit(n))
            end Do


            !< calculate value of the model function at the current position described by PositionVector
            PositionFitness = 0.d0
            if (ParallelizationFlag < 2) then


                !< if calculation reduction is chosen, read chi**2 file
                if (UseCalculationReduction .and. CurrentNumberLinesCalcReduction > 0) then
                    call CheckCalculatedParameterSets(nfit, 1)
                endif
                call calcfitness(PositionFitness, ma, nfit, PositionVector, NumFile, MaxL, MaxCol)
                CurrentNumberLinesCalcReduction = NumberLinesChi2                           !< update CurrentNumberLinesCalcReduction variable
            endif
            return
        end subroutine gotorandom


        !*************************************************************************************************************************************************
        !> subroutine: sort_floatbee
        !>
        !> sort bees with regards to their fitnesses (best fitness is first ?)
        !>
        !> input variables:     nfit:               number of parameters which should be optimized
        !>                      beecount:           total number of bees
        !>
        !> output variables:    floatbee_position:  the positions of each bee
        !>                      floatbee_fitness:   the corresponding fitness of the bees
        !>
        !> \author Thomas Moeller
        !>
        !> \date 28.06.2010
        !>
        subroutine sort_floatbee(beecount, nfit)

            implicit none
            integer :: i, j, k                                                              !< working (loop) variables
            integer :: nfit                                                                 !< number of parameters which should be optimized
            integer :: beecount                                                             !< number of bees
            real*8, dimension(beecount) :: fitness_copy                                     !< contains the chi**2 values for the current bees
            real*8, dimension(beecount,nfit) :: position_copy                               !< contains the corresponding position vectors


            ! Debug:
            !    Do i=1,beecount
            !        print*,'->',i,(minmaxTrans - floatbee_fitness(i)), floatbee_position(i,:)
            !    end Do
            !    print*,'------------------------------------------------------------------------'


            !< sort fitnesses in reversed order
            fitness_copy = (minmaxTrans - floatbee_fitness)
            position_copy = floatbee_position
            call sort(beecount, fitness_copy)


            !< sort positions of bees
            floatbee_position = 0.d0
            k = 0
            Do i = 1, beecount
                Do j = 1, beecount
                    if (dabs(fitness_copy(i) - (minmaxTrans - floatbee_fitness(j))) < 1.d-8) then
                        k = k + 1
                        floatbee_position(k,:) = position_copy(j,:)
                        exit
                    endif
                end Do
            end Do

            ! Debug:
            !    Do i=1,k
            !        print*,'>>',i,fitness_copy(i),floatbee_position(i,:)
            !    end Do
            !    stop

            floatbee_fitness = (minmaxTrans - fitness_copy)
            return
        end subroutine sort_floatbee


        !>------------------------------------------------------------------------------------------------------------------------------------------------
        !>
        !> class hive:
        !>
        !>------------------------------------------------------------------------------------------------------------------------------------------------


        !*************************************************************************************************************************************************
        !> subroutine: InitalizeHive
        !>
        !> initialize class Hive: define initial positions and coresponding fitnesses
        !>
        !> input variables:     nfit:               number of parameters which should be optimized
        !>                      beecount:           number of bees
        !>                      ma:                 total number of all parameter
        !>                      NumFile:            number of input files
        !>                      MaxL:               max. total length
        !>                      MaxCol:             max. number of columns
        !>                      ThreadNumber:       number of the current thread (used for parallelization)
        !>
        !> output variables:    floatbee_position:  initial position
        !>                      floatbee_fitness:   corresponding fitnesses
        !>
        !> \author Thomas Moeller
        !>
        !> \date 23.06.2010
        !>
        subroutine InitalizeHive(beecount, nfit, ma, NumFile, MaxL, MaxCol)

            use Variables

            implicit none
            integer :: i                                                                    !< loop variable
            integer :: nfit                                                                 !< number of parameters which should be optimized
            integer :: beecount                                                             !< number of bees
            integer :: alloc_status, dealloc_status                                         !< variables for error handling used by allocate and deallocate
            integer :: ma                                                                   !< total number of all parameter
            integer :: NumFile                                                              !< number of input files
            integer :: MaxL                                                                 !< max. total length
            integer :: MaxCol                                                               !< max. number of columns


            !< (deallocate) allocate memory for several arrays
            if (allocated(bestsites)) then
               deallocate(bestsites, bestsitesPosition, selsites, selsitesPosition, HiveRange, bestposition, stat = dealloc_status)
               if (dealloc_status /= 0) then                                                !< is all ok?
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine InitalizeHive:")')
                    write(logchannel,'(2x,"Can not deallocate variables bestsites etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("dealloc_status = ",I4)') dealloc_status
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine InitalizeHive:")'
                    print '(2x,"Can not deallocate variables bestsites etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("dealloc_status = ",I4)',dealloc_status
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif
            allocate(bestsites(bestsitescount), bestsitesPosition(bestsitescount, nfit), selsites(selsitescount), &
                     selsitesPosition(selsitescount, nfit), HiveRange(nfit), bestposition(nfit), stat = alloc_status)
            if (alloc_status /= 0) then                                                     !< is all ok?
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine InitalizeHive:")')
                write(logchannel,'(2x,"Can not allocate variables bestsites etc.")')
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("alloc_status = ",I4)') alloc_status
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine InitalizeHive:")'
                print '(2x,"Can not allocate variables bestsites etc.")'
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("alloc_status = ",I4)',alloc_status
                print '(" ")'
                stop ' Program aborted!'
            endif
            bestsites = 0.d0
            bestsitesPosition = 0.d0
            selsites = 0.d0
            selsitesPosition = 0.d0
            bestposition = 0
            HiveRange = 1.d0                                                                !< initialize range array with 1 representing normalized
                                                                                            !< parameter space

            !< initialize positions and corresponding fitness
            call InitializeLoglikebee(nfit, beecount, ma, NumFile, MaxL, MaxCol)


            !< determine best fitness
            bestfitness = -1.d99
            Do i = 1, beecount
                if (bestfitness < floatbee_fitness(i)) then
                    bestfitness = floatbee_fitness(i)
                    bestposition(:) = floatbee_position(i,:)
                endif
            end Do

            ! Debug:
            ! print*,'>>bestfitness = ',(minmaxTrans - bestfitness)
            ! print*,'bestposition(:) = ',bestposition(:)


            !< sort bees
            call sort_floatbee(beecount, nfit)

            return
        end subroutine InitalizeHive


        !*************************************************************************************************************************************************
        !> subroutine: sendbees
        !>
        !> calculates the bees (positions and corresponding fitnesses) for all bees within one iteration
        !>
        !> input variables:     nfit:               number of parameters which should be optimized
        !>                      beecount:           number of bees
        !>                      indexBee:           index of selected bee
        !>                      LocalCounter:       counter for bees of current site
        !>                      HelpPositionVector: position vector
        !>                      HelpFitness:        corresponding fitness
        !>                      ma:                 total number of all parameter
        !>                      NumFile:            number of input files
        !>                      MaxL:               max. total length
        !>                      MaxCol:             max. number of columns
        !>                      sitenumber:         number of the current site
        !>
        !> output variables:    indexBeeOut:        index of selected bee
        !>
        !> \author Irina Bernst, translation (python to fortran) by Thomas Moeller
        !>
        !> \date 05.03.2010
        !>
        subroutine sendbees(indexBeeOut, nfit, beecount, indexBee, LocalCounter, HelpPositionVector, HelpFitness, ma, NumFile, MaxL, MaxCol, &
                            sitenumber)

            use Variables
            use FunctionCalling

            implicit none
            integer :: nfit                                                                 !< number of free parameters
            integer :: beecount                                                             !< number of bees
            integer :: indexBee, indexBeeOut                                                !< index of the current bee within the swarm
            integer :: LocalCounter                                                         !< counter for bees of current site
            integer :: i, j, n                                                              !< working (loop) variables
            integer :: ma                                                                   !< total number of all parameter
            integer :: NumFile                                                              !< number of input files
            integer :: MaxL                                                                 !< max. total length
            integer :: MaxCol                                                               !< max. number of columns
            integer :: posCounter                                                           !< used for comparison of position vectors
            integer :: newcounter                                                           !< used for parallelization reorganization
            integer :: sitenumber                                                           !< number of the current site
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            integer, dimension(LocalCounter) :: newPosition_index                           !< used for parallelization reorganization
            real*8 :: HelpFitness                                                           !< corresponding fitness for position HelpPositionVector
            real*8, dimension(nfit) :: HelpPositionVector                                   !< help variable
            real*8, dimension(LocalCounter, nfit) :: newPosition                            !< new position vector
            real*8, allocatable, dimension(:) :: chi2ValuesVector                           !< chi2 value vector
            real*8, allocatable, dimension(:, :) :: LocalnewPosition                        !< local copy of parameter vector
            character(len=10) :: Number0, Number2                                           !< working variables
            logical :: NotIn_bestsites_flag, NotIn_selsites_flag                            !< flags indicating if certain bee correspond to best
                                                                                            !< (selected) site

            !< loop over all bees of given site (best and selected)
            newcounter = 0
            newPosition_index = 0
            newPosition = 0.d0
            Do n = 1, LocalCounter
                if (indexBee == beecount) then                                              !< if indexBee equal to last bee, abort subroutine
                    exit
                endif


                !< determine if curr_bee is within the selsites array
                NotIn_bestsites_flag = .true.
                Do i = 1, bestsitescount
                    posCounter = 0
                    Do j = 1, nfit
                        if (dabs(bestsitesPosition(i,j) - floatbee_position(indexBee,j)) < 1.d-8) then
                            posCounter = posCounter + 1
                        endif
                    end Do
                    if (posCounter == nfit .and. dabs(floatbee_fitness(indexBee) - bestsites(i)) < 1.d-8) then
                        NotIn_bestsites_flag = .false.
                        exit
                    endif
                end Do


                !< determine if curr_bee is within the selsites array
                NotIn_selsites_flag = .true.
                Do i = 1, selsitesCount
                    posCounter = 0
                    Do j = 1, nfit
                        if (dabs(selsitesPosition(i,j) - floatbee_position(indexBee,j)) < 1.d-8) then
                            posCounter = posCounter + 1
                        endif
                    end Do
                    if (posCounter == nfit.and.dabs(floatbee_fitness(indexBee) - selsites(i)) < 1.d-8) then
                        NotIn_selsites_flag = .false.
                        exit
                    endif
                end Do


                !< if curr_bee is neither in bestsites nor nin selsites define a new position for curr_bee
                if (NotIn_bestsites_flag .and. NotIn_selsites_flag) then
                    call goto(nfit, HelpPositionVector, HelpFitness, HiveRange, ma, NumFile, MaxL, MaxCol)
                    floatbee_position(indexBee,:) = HelpPositionVector
                    floatbee_fitness(indexBee) = HelpFitness
                    if (ParallelizationFlag > 1) then
                        newcounter = newcounter + 1
                        newPosition_index(newcounter) = indexBee
                        newPosition(newcounter,:) = HelpPositionVector
                    endif
                endif
                indexBee = indexBee + 1                                                     !< increase index of bee in swarm
            end Do
            indexBeeOut = indexBee

            ! Debug:
            !    Do i=1,beecount
            !        print*,'>>',i,floatbee_position(i,:)
            !    end Do
            !    print*,'######################################################################',indexBeeOut


            if (newcounter > 0) then


                !< clear screen massage and convert numbers to string
                if (printflag) then
                    print '(A,120(" "),A,$)',char(13),char(13)


                    !< convert numbers to string
                    write(Number2, '(I10)') newcounter
                    write(Number0, '(I10)') sitenumber
                    i = sitenumber - int(sitenumber * 0.1) * 10
                    if (i == 1) then
                        Number0 = trim(adjustl(Number0)) // "st"
                    elseif (i == 2) then
                        Number0 = trim(adjustl(Number0)) // "nd"
                    elseif (i == 3) then
                        Number0 = trim(adjustl(Number0)) // "rd"
                    else
                        Number0 = trim(adjustl(Number0)) // "th"
                    endif
                endif


                !< if calculation reduction is chosen, read chi**2 file
                if (UseCalculationReduction .and. CurrentNumberLinesCalcReduction > 0) then
                    call CheckCalculatedParameterSets(nfit, 1)
                endif


                !< calculate fitnesses for the new bees positions
                if (printflag) then
                    print '(A,11x,"Determine fitnesses for the modified bestsite position ..                       ",A,$)', char(13), char(13)
                endif


                !< allocate memory for variable chi2ValuesVector
                if (allocated(chi2ValuesVector)) then
                    deallocate(chi2ValuesVector, LocalnewPosition, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        print '(" ")'
                        print '(1x,"Error in subroutine sendbees!")'
                        print '(3x,"Can not deallocate variable chi2ValuesVector!")'
                        print '(" ")'
                        print '(3x,"Please restart the program!")'
                        return
                    endif
                endif
                allocate(chi2ValuesVector(newcounter), LocalnewPosition(newcounter, nfit), stat = allocstatus)
                if (allocstatus /= 0) then
                    print '(" ")'
                    print '(1x,"Error in subroutine sendbees!")'
                    print '(3x,"Can not allocate variable chi2ValuesVector!")'
                    print '(" ")'
                    print '(3x,"Please restart the program!")'
                    return
                endif
                chi2ValuesVector = 0.d0
                LocalnewPosition = 0.d0
                LocalnewPosition = newPosition(1:newcounter, :)
                call ModelCalcChiFunctionGeneral(ma, ia, a, newcounter, nfit, NumFile, MaxL, MaxCol, LocalnewPosition, chi2ValuesVector)
                Do n = 1, newcounter                                                        !< loop over all new bees positions
                    i = newPosition_index(n)
                    floatbee_fitness(i) = (minmaxTrans - chi2ValuesVector(n))
                end Do


                !< update CurrentNumberLinesCalcReduction variable
                CurrentNumberLinesCalcReduction = NumberLinesChi2


                !< clear screen massage
                if (printflag) then
                    print '(A,120(" "),A,$)',char(13),char(13)
                endif
            endif
            return
        end subroutine sendbees


        !*************************************************************************************************************************************************
        !> subroutine: nextstep
        !>
        !> determines next step
        !>
        !> input variables:     nfit:               number of parameters which should be optimized
        !>                      beecount_copy:      copy of number of bees
        !>                      ma:                 total number of all parameter
        !>                      NumFile:            number of input files
        !>                      MaxL:               max. total length
        !>                      MaxCol:             max. number of columns
        !>
        !> output variables:    bestsites:          fitness of bees
        !>
        !> \author Irina Bernst, translation (python to fortran) by Thomas Moeller
        !>
        !> \date 05.03.2010
        !>
        subroutine nextstep(nfit, beecount_copy, ma, NumFile, MaxL, MaxCol)

            use Variables
            use FunctionCalling

            implicit none
            integer :: nfit                                                                 !< number of parameters which should be optimized
            integer :: beecount_copy                                                        !< copy of number of bees
            integer :: bestsitesCounter, selsitesCounter                                    !< counter
            integer :: i, n                                                                 !< working (loop) variables
            integer :: curr_index                                                           !< current length of bestsites
            integer :: bee_index                                                            !< index for bee
            integer :: ma                                                                   !< total number of all parameter
            integer :: NumFile                                                              !< number of input files
            integer :: MaxL                                                                 !< max. total length
            integer :: MaxCol                                                               !< max. number of columns
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            real*8 :: PositionFitness                                                       !< fitness of the nth bee in the swarm
            real*8, dimension(nfit) :: PositionVector                                       !< vector in parameter space
            real*8, allocatable, dimension(:) :: chi2ValuesVector                           !< chi2 value vector
            real*8, allocatable, dimension(:, :) :: LocalCopyPosVector                      !< need for MPI version to avoid warning message
            character(len=10) :: Number2                                                    !< working variables
            logical :: ReturnBestSites, ReturnSelSites                                      !< flags indicating best and selected sites


            !< determine bestsites
            bestsites = 0.d0
            bestsites(1) = bestfitness                                                      !< add fitness of the current bee to array with bestsites
            bestsitesPosition = 0.d0
            bestsitesPosition(1, :) = bestposition(:)                                       !< add corresponding position
            bestsitesCounter = 1


            ! Debug:
            !    print*,'>> bestsitesCounter = ',bestsitesCounter
            !    print*,'>> bestsites = ',(minmaxTrans - bestsites(bestsitesCounter))
            !    print*,'-> position = ',bestsitesPosition(bestsitesCounter,:)
            !    print*,'###############################################################'

            curr_index = 2
            Do While (curr_index <= beecount_copy)
                call otherpatch(ReturnBestSites, curr_index, nfit, bestsitesCounter, bestsitescount, bestsitesPosition, HiveRange)  !< call otherpatch
                if (ReturnBestSites) then
                    if (curr_index > bestsitescount) then                                   !< if last bee is reached exit loop
                        exit
                    endif
                    bestsitesCounter = bestsitesCounter + 1
                    bestsites(bestsitesCounter) = floatbee_fitness(curr_index)              !< add fitness of the current bee to array with bestsites
                    bestsitesPosition(bestsitesCounter,:) = floatbee_position(curr_index,:) !< add corresponding position

                    ! Debug:
                    ! print*,'>> bestsitesCounter = ',bestsitesCounter
                    ! print*,'>> bestsites = ',(minmaxTrans - bestsites(bestsitesCounter))
                    ! print*,'-> position = ',bestsitesPosition(bestsitesCounter,:)
                    ! print*,'###############################################################'
                endif
                curr_index = curr_index + 1                                                 !< increase index
            end Do

            ! Debug:
            !    print*,' '
            !    print*,' '
            !    print*,'+++++++++++++++++++++++++++++++++++++++++++++++++++',curr_index,beecount_copy


            !< determine selsites
            selsites = 0.d0
            selsitesPosition = 0.d0
            selsitesCounter = 0
            Do n = curr_index, beecount_copy
                call otherpatch(ReturnBestSites, n, nfit, bestsitesCounter, bestsitescount, bestsitesPosition, HiveRange)   !< call otherpatch
                call otherpatch(ReturnSelSites, n, nfit, selsitesCounter, selsitescount, selsitesPosition, HiveRange)       !< call otherpatch
                if (ReturnBestSites .and. ReturnSelSites) then
                    selsitesCounter = selsitesCounter + 1
                    selsites(selsitesCounter) = floatbee_fitness(n)                         !< add fitness of the current bee to array with bestsites
                    selsitesPosition(selsitesCounter,:) = floatbee_position(n,:)            !< add corresponding position

                    ! Debug:
                    !    print*,'>> selsitesCounter = ',selsitesCounter
                    !    print*,'>> selsites = ',(minmaxTrans - selsites(selsitesCounter))
                    !    print*,'-> position = ',selsitesPosition(selsitesCounter,:)
                    !    print*,'=============================================================='

                    if (selsitesCounter == selsitescount) then
                        exit
                    endif
                endif
            end Do


            !< modify bestsides and selected sides
            bee_index = 1
            Do n = 1, bestsitescount
                PositionVector = bestsitesPosition(n,:)
                PositionFitness = bestsites(n)
                call sendbees(bee_index, nfit, beecount_copy, bee_index, bestbeecount, PositionVector, PositionFitness, ma, NumFile, MaxL, MaxCol, n)
            end Do
            if (selsitesCounter > 0) then
                bee_index = bee_index + 1
                Do n = 1, selsitescount
                    PositionVector = selsitesPosition(n,:)
                    PositionFitness = selsites(n)
                    call sendbees(bee_index, nfit, beecount_copy, bee_index, selectedbeecount, PositionVector, PositionFitness, ma, NumFile, MaxL, &
                                  MaxCol, n)
                end Do
            endif


            !< define bee_index
            Do n = bee_index, beecount_copy
                PositionFitness = floatbee_fitness(n)
                PositionVector = floatbee_position(n,:)
                call gotorandom(PositionVector, PositionFitness, nfit, ma, NumFile, MaxL, MaxCol)
                floatbee_position(n,:) = PositionVector
                if (ParallelizationFlag < 2) then
                    floatbee_fitness(n) = PositionFitness
                endif
            end Do


            !< determine new fitness
            if (ParallelizationFlag > 1) then


                !< clear screen massage
                if (printflag) then
                    print '(A,120(" "),A,$)',char(13),char(13)                              !< clear line on screen
                    write(Number2, '(I10)') (beecount_copy - bee_index + 1)                 !< convert number to string
                endif


                !< if calculation reduction is chosen, read chi**2 file
                if (UseCalculationReduction .and. CurrentNumberLinesCalcReduction > 0) then
                    call CheckCalculatedParameterSets(nfit, 1)
                endif


                !< print what you do ..
                if (printflag) then
                   print '(A,11x,"Determine fitnesses for new positions ..                       ",A,$)', char(13), char(13)
                endif
                n = beecount_copy - bee_index + 1
                if (allocated(chi2ValuesVector)) then
                    deallocate(chi2ValuesVector, LocalCopyPosVector, stat = deallocstatus)
                    if (deallocstatus /= 0) then
                        print '(" ")'
                        print '(1x,"Error in subroutine nextstep!")'
                        print '(3x,"Can not deallocate variable chi2ValuesVector!")'
                        print '(" ")'
                        print '(3x,"Please restart the program!")'
                        return
                    endif
                endif
                allocate(chi2ValuesVector(n), LocalCopyPosVector(n, nfit), stat = allocstatus)
                if (allocstatus /= 0) then
                    print '(" ")'
                    print '(1x,"Error in subroutine nextstep!")'
                    print '(3x,"Can not allocate variable chi2ValuesVector!")'
                    print '(" ")'
                    print '(3x,"Please restart the program!")'
                    return
                endif
                chi2ValuesVector = 0.d0
                LocalCopyPosVector(:, :) = floatbee_position(bee_index:beecount_copy, :)
                call ModelCalcChiFunctionGeneral(ma, ia, a, n, nfit, NumFile, MaxL, MaxCol, LocalCopyPosVector, chi2ValuesVector)
                Do n = bee_index, beecount_copy
                    floatbee_fitness(n) = (minmaxTrans - chi2ValuesVector(n - bee_index + 1))
                end Do


                !< update CurrentNumberLinesCalcReduction variable
                CurrentNumberLinesCalcReduction = NumberLinesChi2


                !< clear screen massage
                if (printflag) then
                    print '(A,120(" "),A,$)',char(13),char(13)
                endif
            endif


            !< determine best fitness
            bestfitness = -1.d99
            Do i = 1, beecount_copy
                if (bestfitness < floatbee_fitness(i)) then
                    bestfitness = floatbee_fitness(i)
                    bestposition(:) = floatbee_position(i,:)
                endif
            end Do


            !< sort bees
            call sort_floatbee(beecount_copy, nfit)


            return
        end subroutine nextstep


        !>------------------------------------------------------------------------------------------------------------------------------------------------
        !>
        !> class loglikebee:
        !>
        !>------------------------------------------------------------------------------------------------------------------------------------------------


        !*************************************************************************************************************************************************
        !> subroutine: InitializeLoglikebee
        !>
        !> initialize class loglikebee: define initial positions and coresponding fitnesses
        !>
        !> input variables:     nfit:               number of parameters which should be optimized
        !>                      number_bees:        number of bees
        !>                      ma:                 total number of all parameter
        !>                      NumFile:            number of input files
        !>                      MaxL:               max. total length
        !>                      MaxCol:             max. number of columns
        !>
        !> output variables:    floatbee_position:  initial position
        !>                      floatbee_fitness:   corresponding fitnesses
        !>
        !> \author Thomas Moeller
        !>
        !> \date 23.06.2010
        !>
        subroutine InitializeLoglikebee(nfit, number_bees, ma, NumFile, MaxL, MaxCol)

            use Variables
            use FunctionCalling

            implicit none
            integer :: i, n                                                                 !< working (loop) variables
            integer :: nfit                                                                 !< number of parameters which should be optimized
            integer :: number_bees                                                          !< number of bees
            integer :: ma                                                                   !< total number of all parameter
            integer :: NumFile                                                              !< number of input files
            integer :: MaxL                                                                 !< max. total length
            integer :: MaxCol                                                               !< max. number of columns
            integer :: allocstatus, deallocstatus                                           !< variables for (de)allocation
            real*8, allocatable, dimension(:) :: chi2ValuesVector                           !< chi2 value vector
            character(len=10) :: Number2                                                    !< working variable


            !< initialize class floatbee
            call InitalizeFloatbee(nfit, number_bees)


            !< determine initial positions and corresponding fitnesses
            floatbee_position = 0.d0
            Do i = 1, number_bees
                Do n = 1, nfit
                    floatbee_position(i,n) = RandomWithLimits(OptimizedParameterLowLimit(n), OptimizedParameterUpperLimit(n))
                end Do
            end Do


            !< determine corresponding fitnesses
            floatbee_fitness = 0.d0


            !< clear screen massage
            if (printflag) then
                print '(A,120(" "),A,$)',char(13),char(13)
                write(Number2, '(I10)') number_bees
            endif


            !< if calculation reduction is chosen, read chi**2 file
            if (UseCalculationReduction .and. CurrentNumberLinesCalcReduction > 0) then
                call CheckCalculatedParameterSets(nfit, 1)
            endif


            !< print what you do ..
            if (printflag) then
                print '(A,11x,"Initialize Loglike bees ..                      ",A,$)', char(13), char(13)
            endif


            !< allocate memory for variable chi2ValuesVector
            if (allocated(chi2ValuesVector)) then
                deallocate(chi2ValuesVector, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    print '(" ")'
                    print '(1x,"Error in subroutine InitializeLoglikebee!")'
                    print '(3x,"Can not deallocate variable chi2ValuesVector!")'
                    print '(" ")'
                    print '(3x,"Please restart the program!")'
                    return
                endif
            endif
            allocate(chi2ValuesVector(number_bees), stat = allocstatus)
            if (allocstatus /= 0) then
                print '(" ")'
                print '(1x,"Error in subroutine InitializeLoglikebee!")'
                print '(3x,"Can not allocate variable chi2ValuesVector!")'
                print '(" ")'
                print '(3x,"Please restart the program!")'
                return
            endif
            chi2ValuesVector = 0.d0
            call ModelCalcChiFunctionGeneral(ma, ia, a, number_bees, nfit, NumFile, MaxL, MaxCol, floatbee_position, chi2ValuesVector)
            Do i = 1, number_bees
                floatbee_fitness(i) = (minmaxTrans - chi2ValuesVector(i))
            end Do


            !< update CurrentNumberLinesCalcReduction variable
            CurrentNumberLinesCalcReduction = NumberLinesChi2


            !< clear screen massage
            if (printflag) then
                print '(A,120(" "),A,$)',char(13),char(13)
            endif
            return
        end subroutine InitializeLoglikebee


        !*************************************************************************************************************************************************
        !> subroutine: calcfitness
        !>
        !> determine the fitness of a given bee
        !>
        !> input variables:     ma:                 total number of all parameter
        !>                      nfit:               number of parameters which should be optimized
        !>                      OptimizedParamter:  optimized parameter
        !>                      NumFile:            number of input files
        !>                      MaxL:               max. total length
        !>                      MaxCol:             max. number of columns
        !>                      ThreadNumber:       number of the current thread (used for parallelization)
        !>
        !> output variables:    chisq:              return fitness of bee with position positions
        !>
        !> \author Thomas Moeller
        !>
        !> \date 22.06.2010
        !>
        subroutine calcfitness(chisq, ma, nfit, OptimizedParamter, NumFile, MaxL, MaxCol)

            use Variables
            use FunctionCalling

            implicit none
            integer :: n                                                                    !< loop variable
            integer :: nfit                                                                 !< number of parameters which should be optimized
            integer :: ma                                                                   !< total number of all parameter
            integer :: NumFile                                                              !< number of input files
            integer :: MaxL                                                                 !< max. total length
            integer :: MaxCol                                                               !< max. number of columns
            real*8 :: chisq                                                                 !< fitness of bee with position OptimizedParamter
            real*8, dimension(1) :: chi2ValuesVector                                        !< we have only one chi2 value
            real*8, dimension(nfit) :: OptimizedParamter                                    !< optimized parameters
            real*8, dimension(1, nfit) :: OptimizedParamterVec                              !< copy optimized parameters


            !< check if component of the new vector is within the corresponding range
            OptimizedParamterVec(1, :) = 0.d0
            Do n = 1, nfit
                if (OptimizedParamter(n) >= OptimizedParameterLowLimit(n) .and. OptimizedParamter(n) <= OptimizedParameterUpperLimit(n)) then
                    OptimizedParamterVec(1, n) = OptimizedParamter(n)
                elseif (OptimizedParamter(n) < OptimizedParameterLowLimit(n)) then          !< if nth position is lower than lower limit
                    OptimizedParamterVec(1, n) = OptimizedParameterLowLimit(n)
                elseif (OptimizedParamter(n) > OptimizedParameterUpperLimit(n)) then        !< if nth position larger than upper limit
                    OptimizedParamterVec(1, n) = OptimizedParameterUpperLimit(n)
                endif
            end Do


            !< determine chi2 values for parameter vectors
            chi2ValuesVector = 0.d0
            call ModelCalcChiFunctionGeneral(ma, ia, a, 1, nfit, NumFile, MaxL, MaxCol, OptimizedParamterVec, chi2ValuesVector)
            chisq = (minmaxTrans - chi2ValuesVector(1))
            return
        end subroutine calcfitness


        !*************************************************************************************************************************************************
        !> subroutine: CallBees
        !>
        !> calls the different subroutines of bees algorithm
        !>
        !> input variables:     NumberBees:         number of bees
        !>                      numiter:            max. number of iterations
        !>                      nfit:               number of parameters which should be optimized
        !>                      runcounter:         number of repeatitions
        !>                      ma:                 total number of all parameter
        !>                      counter:            number of best sites
        !>                      colx:               number of columns belonging to the x-column
        !>                      NumFile:            number of input files
        !>                      MaxL:               max. total length
        !>                      MaxCol:             max. number of columns
        !>                      PlotIteration:      dummy argument
        !>                      PlotType:           kind of plotting
        !>                      xAxisLabel:         label for x-axis
        !>                      yAxisLabel:         label for y-axis
        !>                      zAxisLabel:         label for z-axis
        !>
        !> output variables:    counter:            number of best sites
        !>                      Parameterset:       paramsets for the best sites
        !>
        !> \author Thomas Moeller
        !>
        !> \date 22.06.2010
        !>
        subroutine CallBees(NumberBees, nfit, runcounter, ma, counter, Parameterset, colx, NumFile, MaxL, MaxCol, PlotIteration, PlotType, &
                            xAxisLabel, yAxisLabel, zAxisLabel, fitlog)

            use Variables
            use FunctionCalling

            implicit none
            integer :: i, j, k, m, n, runnumber                                             !< loop variables
            integer :: NumberBees                                                           !< number of bees
            integer :: nfit                                                                 !< number of free parameters
            integer :: ma                                                                   !< total number of all parameters
            integer :: runcounter                                                           !< number of replications
            integer :: counter                                                              !< counts the number of parameter sets
            integer :: func_counter                                                         !< counter
            integer :: colx                                                                 !< max. number of columns belonging to "x-column"
            integer :: NumFile                                                              !< number of exp. files
            integer :: MaxL                                                                 !< max. number of lines
            integer :: MaxCol                                                               !< max. number of columns belonging to "y-column"
            integer :: NumInputFiles                                                        !< needed for loop over input files
            integer :: PlotIteration                                                        !< flag for plotting the model function for each step
            integer :: PlotType                                                             !< get type of plot
            integer :: NumInputFile_index                                                   !< contains index for input file
            integer :: i_index                                                              !< contains index for i
            integer :: j_index                                                              !< contains index for j
            integer :: number_bees                                                          !< total number of bees
            integer :: scoutbeecount                                                        !< class hive:

            real*8 :: best_func                                                             !< chisq for the current iteration
            real*8 :: dummy                                                                 !< dummy variable for random number generator
            real*8 :: koeff                                                                 !< coefficients for the range reduction
            real*8 :: chisq                                                                 !< single chi2 value
            real*8, dimension(1) :: chi2ValuesVector                                        !< here only one chi2 value
            real*8, dimension(nfit) :: xx                                                   !< help array
            real*8, dimension(colx) :: posdatexp                                            !< array for point within observation data
            real*8, dimension(ma) :: acopy                                                  !< copy of total parameter list
            real*8, dimension(counter, ma) :: Parameterset                                  !< array containing the parameter set which fullfil the
                                                                                            !< quality creteria
            character(len=25) :: LongNumber1, LongNumber2                                   !< working variables
            character(len=256) :: xAxisLabel                                                !< label of the x-axis (for plot)
            character(len=256) :: yAxisLabel                                                !< label of the y-axis (for plot)
            character(len=256) :: zAxisLabel                                                !< label of the z-axis (for plot)
            character(len=100) :: HelpString                                                !< used for number to string conversion
            character(len=5196) :: ListParamFormated                                        !< contains the formated list of all free parameters
            character(len=8192) :: fitlog                                                   !< path for log files
            character(len=512) :: LongHelpString1                                           !< help string
            character(len=512) :: WorkingDirectory1, WorkingDirectory2                      !< path and fi

            logical :: IntegerTrue                                                          !< flag is true, if number is an integer
            logical :: InitPlotFlag                                                         !< flag for plotting


            !< initialize ran1
            idum = (-1)
            call ran1(dummy, idum)


            !< define OptimizedParameter range and determine starting point
            OptimizedParameterLowLimit = 0.d0
            OptimizedParameterUpperLimit = 0.d0
            counter = 0
            Do i = 1, parameternumber
                if (ia(i)) then
                    counter = counter + 1
                    OptimizedParameterLowLimit(counter) = paramset(3, i)
                    OptimizedParameterUpperLimit(counter) = paramset(4, i)

                    ! Debug:
                    !    print*,'counter = ',counter
                    !    print*,'OptimizedParameterLowLimit(counter) = ',OptimizedParameterLowLimit(counter)
                    !    print*,'OptimizedParameterUpperLimit(counter) = ',OptimizedParameterUpperLimit(counter)
                    !    print*,'-----------------------------------------------------------------------------------'
                endif
            end Do


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< set relevant parameters
            koeff = 0.8d0 * nfit                                                            !< reduction factor for range


            !< set relevant parameters
            scoutbeecount = 5 * bestsitescount
            selectedbeecount = 4 * bestsitescount
            bestbeecount = 3 * bestsitescount
            selsitescount = 2 * bestsitescount
            number_bees = (scoutbeecount + selectedbeecount * selsitescount + bestbeecount * bestsitescount) * nfit
            if (NumberBees /= 0 .and. NumberBees > number_bees) then                        !< set user definition
                number_bees = NumberBees
            endif


            !< print what you do
            if (printflag) then
                print '(11x,"Number of Bees = ",I10)',number_bees
                print '(" ")'
                print '(11x,"Iteration:",20x,"chi^2:",5x,"Parameter:")'
                print '(11x,"Initialize model function ..",20(" "),A1,$ )',char(13)
            endif
            write(logchannel,'(11x,"Number of Bees = ",I10)') number_bees
            write(logchannel,'(" ")')
            write(logchannel,'(11x,"Iteration:",20x,"chi^2:",5x,"Parameter:")')


            !< start bees algorithm
            Do runnumber = 1, runcounter                                                    !< loop over runcount ?


                !< initialize class hive
                call InitalizeHive(number_bees, nfit, ma, NumFile, MaxL, MaxCol)


                !< start iteration loop
                best_func = -1.d99
                func_counter = 0
                Do n = 1, maxiter                                                           !< loop for iterations


                    !< determine next step
                    call nextstep(nfit, number_bees, ma, NumFile, MaxL, MaxCol)


                    !< update a array
                    k = 0
                    Do j = 1, parameternumber
                        if (ia(j)) then
                            k = k + 1
                            a(j) = BestSitesParamSet(1, k + 1)
                        endif
                    end Do


                    !< build list with fit parameters
                    k = 0
                    ListParamFormated = ""
                    Do j = 1, parameternumber
                        if (ia(j)) then
                            k = k + 1
                            HelpString = ""
                            call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, j)
                            if (IntegerTrue) then
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(a(j))
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) int(a(j))
                                endif
                            else
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) a(j)
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) a(j)
                                endif
                            endif
                            if (k == 1) then
                                ListParamFormated = trim(adjustl(ListParamFormated)) // trim(adjustl(HelpString))
                            else
                                ListParamFormated = trim(adjustl(ListParamFormated)) // ',  ' // trim(adjustl(HelpString))
                            endif
                        endif
                    end Do


                    !< print status of iteration process ..
                    if (printflag) then
                        print '(11x, I10, ES26.15, 5x, A)', n, (BestSitesParamSet(1, 1)), trim(adjustl(ListParamFormated))
                    endif
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("  ")')
                    write(paramchannel,'(123("*"))')
                    write(paramchannel,'("Iteration: ",I5,",  chi^2 = ",ES25.15)') n, (BestSitesParamSet(1, 1))
                    write(logchannel,'(11x, I10, ES26.15, 5x, A)') n, (BestSitesParamSet(1, 1)), trim(adjustl(ListParamFormated))


                    !< write actual parameters to files
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("  ")')
                    write(paramchannel,'("Parameters: ",A)') trim(adjustl(ListParamFormated))
                    write(paramchannel,'(123("-"))')
                    write(paramchannel,'("  ")')


                    !< save current experimental x point of the first experimental file to variable posdatexp
                    posdatexp(1:colx) = 0.d0


                    !< call subroutine to write current values of the parameter to file
                    write(paramchannel,'("-",61(" -"))')
                    Do NumInputFiles = 1, NumberInputFiles
                        write(paramchannel,'("Input-File ",I5,":  , file: ",A)') NumInputFiles, trim(adjustl(FitFktInput(NumInputFiles)))
                        write(paramchannel,'("  ")')
                        write(paramchannel,'("-start_input-file",106("-"))')
                        call WriteParameter(paramchannel, .true., colx, posdatexp, parameternumber, a, NumInputFiles)
                        write(paramchannel,'("-end_input-file",108("-"))')
                    end Do


                    !< plot experimental data, model function, and chi**2
                    if (PlotIteration == 0) then
                        if (n == 1) then
                            InitPlotFlag = .true.
                        else
                            InitPlotFlag = .false.
                        endif
                        call PlotFitFunction(InitPlotFlag, xAxisLabel, yAxisLabel, zAxisLabel)
                    endif


                    if (bestfitness /= (minmaxTrans - best_func)) then
                        best_func = (minmaxTrans - bestfitness)
                        func_counter = 0
                    else
                        func_counter = func_counter + 1
                        if (func_counter == max_func_counter) then
                            Do m = 1, nfit
                                HiveRange(m) = HiveRange(m) * koeff
                            end Do
                            func_counter = 0
                        endif
                    endif
                    if ((BestSitesParamSet(1, 1)) <= chilim) then


                        !< print seperator
                        write(paramchannel,'("  ")')
                        write(paramchannel,'(123("="))')
                        write(paramchannel,'("  ")')


                        !< print status of iteration process ..
                        write(LongNumber1,'(ES25.15)') (BestSitesParamSet(1, 1))
                        write(LongNumber2,'(ES25.15)') chilim
                        if (printflag) then
                            print '(" ")'
                            print '(11x,"Iteration stopped. chi^2 (=",A,") dropped below limit = ",A)', trim(adjustl(LongNumber1)), &
                                                                                                            trim(adjustl(LongNumber2))
                        endif
                        write(logchannel,'("  ")')
                        write(logchannel,'(11x,"Iteration stopped. chi^2 (=",A,") dropped below limit = ",A)') trim(adjustl(LongNumber1)), &
                                                                                                            trim(adjustl(LongNumber2))
                        exit
                    endif
                end Do
                if ((BestSitesParamSet(1, 1)) > chilim) then                                !< if limit of chi^2 is not reached, max. number of
                    write(paramchannel,'("  ")')                                            !< iterations is reached
                    write(paramchannel,'(123("="))')
                    write(paramchannel,'("  ")')
                    if (printflag) then
                        print '(" ")'
                        print '(11x,"Iteration stopped. Number of iterations is equal to max. number of iterations = ",I6)',maxiter
                    endif
                    write(logchannel,'("  ")')
                    write(logchannel,'(11x,"Iteration stopped. Number of iterations is equal to max. number of iterations = ",I6)') maxiter
                endif
            end Do


            !< determine the number of best sites
            j = 0
            Do i = 1, QualityLimit
                if (BestSitesParamSet(i, 1) /= -1.d99) then
                    j = j + 1
                endif
            end Do
            QualityLimit = j


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< select best sites
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(11x,"Best results:")'
            endif
            write(logchannel,'(" ")')
            write(logchannel,'(" ")')
            write(logchannel,'(11x,"Best results:")')


            !< write best results to screen and to log-file
            Do i = 1, QualityLimit
                if (dabs(BestSitesParamSet(i, 1)) /= 1.d99) then
                    acopy = a
                    chisq = (BestSitesParamSet(i, 1))
                    xx = BestSitesParamSet(i, 2:)


                    !< build list with fit parameters
                    k = 0
                    ListParamFormated = ""
                    Do j = 1, parameternumber
                        if (ia(j)) then
                            k = k + 1
                            acopy(j) = xx(k)

                            HelpString = ""
                            call IndexFormat(IntegerTrue, NumInputFile_index, i_index, j_index, j)
                            if (index(ParameterFormat(NumInputFile_index, i_index, j_index),'I') /= 0 &
                                .or.index(ParameterFormat(NumInputFile_index, i_index, j_index),'i') /= 0) then
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) int(acopy(j))
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) int(acopy(j))
                                endif
                            else
                                write(HelpString, ParameterFormat(NumInputFile_index, i_index, j_index)) acopy(j)
                                if (index(HelpString, "*") > 0) then                        !< search for bad real number
                                    write(HelpString, *) acopy(j)
                                endif
                            endif
                            if (k == 1) then
                                ListParamFormated = trim(adjustl(ListParamFormated)) // trim(adjustl(HelpString))
                            else
                                ListParamFormated = trim(adjustl(ListParamFormated)) // ',  ' // trim(adjustl(HelpString))
                            endif
                        endif
                    end Do
                    Parameterset(i, :) = acopy(:)


                    !< print best results to screen and to log-file
                    if (printflag) then
                        print '(13x,"Site number: ",I4,", chi^2 = ",ES25.15,",  Parameterset = ",A)', i, chisq, trim(adjustl(ListParamFormated))
                    endif
                    write(logchannel, '(13x,"Site number: ",I4,", chi^2 = ",ES25.15,",  Parameterset = ",A)') i, chisq, trim(adjustl(ListParamFormated))
                endif
            end Do


            !< clear screen massage
            if (printflag) then
                print '(A,120(" "),A,$)',char(13),char(13)
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< write model functions values and chi2 values for good and best sites to scratch file


            !< create path and na,es for scratch files used to exchange data between fortran and python
            write(LongHelpString1,'(I30)') JobID                                            !< write JobID to string
            WorkingDirectory1 = trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(LongHelpString1)) // "/" // "FitFunction.dat"
            WorkingDirectory2 = trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(LongHelpString1)) // "/" // "Chi2Values.dat"
            return
        end subroutine CallBees
end Module BeeVariables
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!> Module: Algorithm
!>
!>         Module contains the main subroutine used to start the different versions of the Bees algorithm
!>
!>
!> \author Thomas Moeller
!>
!> \date 26.08.2014
!>
Module Algorithm

    use Variables
    use BeeVariables

    implicit none

    contains


        !*************************************************************************************************************************************************
        !> subroutine: MainAlg
        !>
        !> main subroutine which starts the Bees algorithm
        !>
        !>
        !> input variables:         printflagNum:           flag for screen output 1 (=yes) or 0 (=no)
        !>                          LastAlgorithmNum:       number of last algorithm
        !>                          ParamSetCounter:        number of best sites
        !>                          FinalParameterSet:      array containing the parameter set with SampleslogL(i) >= logZ
        !>                          chilm:                  user defined abort criteria for chi**2
        !>                          numiter:                max. number of iterations
        !>                          BeesCounter:            counts number of calls
        !>                          NumberBees:             number of bees
        !>                          DeterminationChi2:      method being used for the determination of chi^2
        !>                          PlotIterationOrg:       plot model function for each iteration set 1(=yes) or 0(=no)
        !>                          PlotTypeOrg:            get type of plot
        !>                          fitlog:                 path for log-file containing the current values of chi**2
        !>                          NumberInputFilesorg:    number of input files for the external model program
        !>                          NumberOutputFilesOrg:   number of output files for the external model program
        !>                          ParallelizationFlagorg: contains the number of processors used for parallelization
        !>                          JobIDorg:               job identification number
        !>                          MaxInputLinesOrg:       max number of lines in an input file
        !>                          MaxParameterOrg:        max number of parameters in a line of an input file
        !>                          RenormalizedChi2Org:    flag for using renormalized chi**2
        !>                          currentpathorg:         path of the working directory
        !>                          FitParameterNameOrg:    array containing the names of the model parameters
        !>                          FitParameterValueLineOrg:   array containing the values of the model parameters as string
        !>                          CalculationMethodOrg:   method of computation (at once or point-to-point)
        !>                          xAxisLabel:             label of the x-axis (for plot)
        !>                          yAxisLabel:             label of the y-axis (for plot)
        !>                          zAxisLabel:             label of the z-axis (for plot)
        !>                          PathStartScriptOrg:     path and name of the start script for calling model function
        !>                          ExeCommandStartScriptOrg:   command for calling model function
        !>                          parametersetorg:        the complete set of paramters (incl. flags and limits)
        !>                          expdataxorg:            array containing the experimental x side
        !>                          expdatayorg:            array containing the experimental y side
        !>                          expdataerrororg:        array containing the experimental error of the y side
        !>                          NumberRangesOrg:        number of y-columns for each experimental file
        !>                          MinRangeOrg:            array containing the minimal exp. ranges
        !>                          MaxRangeOrg:            array containing the maximal exp. ranges
        !>                          NumberXColumnsOrg:      number of x-columns for each experimental file
        !>                          NumberYColumnsOrg:      number of y-columns for each experimental file
        !>                          lengthexpdataorg:       number of lines in experimental data
        !>                          MaxRangeNumber:         max. number of ranges
        !>                          NumFileOrg:             number of experimental files
        !>                          MaxLengthOrg:           max length of experimental data
        !>                          MaxColXOrg:             number of columns concerning to the experimental x side
        !>                          MaxColYOrg:             number of columns concerning to the experimental y side
        !>                          parameternum:           number of model parameter
        !>
        !> output variables:        calstatus:              status flag of calculation (= 0: all ok)
        !>                          FitFunctionOut:         values of the model function at the calculated points
        !>                          Chi2Values:             values of the chi^2 function at the calculated points
        !>                          FinalParameterSet:      the complete set of paramters (incl. flags and limits)
        !>
        subroutine MainAlg(printflagNum, LastAlgorithmNum, calstatus, FitFunctionOut, Chi2Values, chilm, NumberOfFitAlgorithms, numiter, BeesCounter, &
                           ParamSetCounter, GeneralAlgorithmSettings, DeterminationChi2, PlotIteration, PlotType, fitlog, NumberInputFilesorg, &
                           NumberOutputFilesOrg, ParallelizationFlagorg, JobIDorg, MaxInputLinesOrg, MaxParameterOrg, RenormalizedChi2Org, &
                           currentpathorg, FitParameterNameLocal, FitParameterValueLocal, CalculationMethodOrg, xAxisLabel, yAxisLabel, zAxisLabel, &
                           PathStartScriptOrg, ExeCommandStartScriptOrg, parametersetorg, FinalParameterSet, expdataxorg, expdatayorg, expdataerrororg, &
                           NumberRangesOrg, MinRangeOrg, MaxRangeOrg, NumberXColumnsOrg, NumberYColumnsOrg, lengthexpdataorg, MaxRangeNumber, &
                           NumFileOrg, MaxLengthOrg, MaxColXOrg, MaxColYOrg, parameternum)


            implicit none
            ! ********** input variables **********
            integer :: parameternum                                                         !< number of model parameter
            integer :: NumberOfFitAlgorithms                                                !< total number of all algorithms in the chain
            integer :: numiter                                                              !< max. number of iterations
            integer :: NumFileOrg                                                           !< number of experimental files
            integer, dimension(NumFileOrg) :: lengthexpdataorg                              !< number of lines in experimental data
            integer, dimension(NumFileOrg) :: NumberXColumnsOrg                             !< number of x-columns for each experimental file
            integer, dimension(NumFileOrg) :: NumberYColumnsOrg                             !< number of y-columns for each experimental file
            integer, dimension(NumFileOrg) :: NumberRangesOrg                               !< number of y-columns for each experimental file
            integer :: MaxColXOrg                                                           !< number of columns concerning to the experimental x side
            integer :: MaxColYOrg                                                           !< number of columns concerning to the experimental y side
            integer :: MaxLengthOrg                                                         !< max length of experimental data
            integer :: printflagNum                                                         !< flag for screen output 1 (=yes) or 0 (=no)
            integer :: LastAlgorithmNum                                                     !< flag for screen output 1 (=yes) or 0 (=no)
            integer :: DeterminationChi2                                                    !< method being used for the determination of chi^2
            integer :: PlotIteration                                                        !< plot model func. for each iteration set 1 (=yes) or 0 (=no)
            integer :: PlotTypeOrg                                                          !< get type of plot
            integer :: NumberInputFilesorg                                                  !< number of input files for the external model program
            integer :: NumberOutputFilesOrg                                                 !< number of output files for the external model program
            integer :: ParallelizationFlagorg                                               !< contains the number of processors used for parallelization
            integer :: JobIDorg                                                             !< job identification number
            integer :: MaxInputLinesOrg                                                     !< max number of lines in an input file
            integer :: MaxParameterOrg                                                      !< max number of parameters in a line of an input file
            integer :: RenormalizedChi2Org                                                  !< flag for using renormalized chi**2
            integer :: BeesCounter                                                          !< counts number of calls
            integer :: ParamSetCounter                                                      !< number of best sites
            integer :: NumberBees                                                           !< number of bees
            integer :: MaxRangeNumber                                                       !< max. number of ranges
            real*8 :: chilm                                                                 !< user defined abort criteria for chi**2
            real*8, dimension(15) :: GeneralAlgorithmSettings                               !< special algorithm settings
            real*8, dimension(NumFileOrg, MaxLengthOrg, MaxColXOrg) :: expdataxorg          !< array containing the experimental x side
            real*8, dimension(NumFileOrg, MaxLengthOrg, MaxColYOrg) :: expdatayorg          !< array containing the experimental y side
            real*8, dimension(NumFileOrg, MaxLengthOrg, MaxColYOrg) :: expdataerrororg      !< array containing the experimental error of the y side
            real*8, dimension(NumFileOrg, MaxRangeNumber, MaxColXOrg) :: MinRangeOrg        !< array containing the minimal exp. ranges
            real*8, dimension(NumFileOrg, MaxRangeNumber, MaxColXOrg) :: MaxRangeOrg        !< array containing the maximal exp. ranges
            character(len=8192) :: fitlog                                                   !< path for log-file containing the current values of chi**2
            character(len=256) :: xAxisLabel                                                !< label of the x-axis (for plot)
            character(len=256) :: yAxisLabel                                                !< label of the y-axis (for plot)
            character(len=256) :: zAxisLabel                                                !< label of the z-axis (for plot)
            character(len=20) :: CalculationMethodOrg                                       !< method of computation
            character(len=8192) :: PathStartScriptOrg                                       !< command for calling model function
            character(len=8192) :: ExeCommandStartScriptOrg                                 !< command for calling model function
            character(len=8192) :: currentpathorg                                           !< path of the working directory
            character(len=512), dimension(parameternum) :: FitParameterNameLocal            !< array containing the names of the model parameters
            character(len=512), dimension(parameternum) :: FitParameterValueLocal           !< array containing the values of the model parameters as


            ! ********** in/output variables **********
            real*8, dimension(4, parameternum) :: parametersetorg                           !< the non-optimized (initial parameter set)


            ! ********** output variables **********
            integer :: calstatus                                                            !< the following line is necessary for f2py
            real*8, dimension(ParamSetCounter, parameternum) :: FinalParameterSet           !< array containing the optimized parameter set
            real*8, dimension(ParamSetCounter, NumFileOrg, MaxLengthOrg, MaxColYOrg) :: FitFunctionOut !< values of the model func. at the calculated pts.
            real*8, dimension(ParamSetCounter, NumFileOrg, MaxLengthOrg, MaxColYOrg) :: Chi2Values     !< values of the model func. at the calculated pts.


            ! ********** working variabels **********
            integer :: i, j, k, ii, jj                                                      !< working variables
            integer :: nfit                                                                 !< number of free parameters
            integer :: ma                                                                   !< total number of parameters
            integer :: MaxLength                                                            !< max. number of lines
            integer :: PlotType                                                             !< used for plotting
            integer :: ok                                                                   !< status of calculation
            integer :: actualiteration                                                      !< contains the current iteration within the iteration loop
            integer :: flag                                                                 !< working variable used within the iteration loop
            integer :: NumInputFiles                                                        !< need for loop over input files
            integer :: allocstatus, deallocstatus                                           !< working variables for allocation/deallocation
            integer, dimension(8) :: VALUES                                                 !< value for the date_and_time subroutine
            real*8, dimension(parameternum) :: LocalParamVector                             !< needed for MPI version to avoid warning messages
            real*8, dimension(1) :: chi2ValuesVector                                        !< here only one chi2 value
            character(len=512) :: fitlogparam                                               !< path for log-file containing the current parameter values
            character(len=512) :: fitlogChi2                                                !< path for log-file containing chi**2 and the corresponding
                                                                                            !< parameter values
            character(len=8) :: DATE                                                        !< variable for the date_and_time subroutine
            character(len=10) :: TIME                                                       !< variable for the date_and_time subroutine
            character(len=5) :: ZONE                                                        !< variable for the date_and_time subroutine
            character(len=10) :: Number1                                                    !< variable for number to string converting
            character(len=100) :: helpString                                                !< used for number to string conversion
            character(len=8192) :: SiteExt, NumExt, BaseDir, FuncCallExt                    !< working variables for final input file name


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< set print flag and last algorithm flag
            calstatus = 0                                                                   !< set calculation status to 0 = everything is ok
            if (printflagNum == 1) then                                                     !< set printflag
                printflag = .true.
            else
                printflag = .false.
            endif
            if (LastAlgorithmNum == 1) then
                LastAlgorithmFlag = .true.
            else
                LastAlgorithmFlag = .false.
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< copy contents of some input variables to module variables
            NumberExpFiles = NumFileOrg                                                     !< copy number of experimental files to global variable
            currentpath = trim(adjustl(currentpathorg))                                     !< copy path of working directory to module variable without
                                                                                            !< trailing and leading blanks
            MaxColX = MaxColXOrg                                                            !< copy number of columns of the experimental x data to
                                                                                            !< module variable
            MaxColY = MaxColYOrg                                                            !< copy number of columns of the experimental y data to
                                                                                            !< module variable
            MaxLength = MaxLengthOrg                                                        !< copy max. number of lines in experimental data
            MaxExpLength = MaxLength                                                        !< copy max. number of exp. data points
            MaxNumberRanges = MaxRangeNumber                                                !< copy of max. number of data ranges in a exp. data file
            parameternumber = parameternum                                                  !< copy input variable containing the number of parameters
                                                                                            !< to module variable
            DetChi2 = DeterminationChi2                                                     !< copy method of chi**2 determination
            NumberInputFiles = NumberInputFilesorg                                          !< copy number of input files for the external program to
                                                                                            !< global variable
            NumberOutputFiles = NumberOutputFilesOrg                                        !< copy number of output files for the external program to
                                                                                            !< global variable
            ParallelizationFlag = ParallelizationFlagorg                                    !< copy number of used processors to global variable
            JobID = JobIDorg                                                                !< copy job-ID number to global variable
            PlotType = PlotTypeOrg                                                          !< copy type of plotting
            MaxInputLines = MaxInputLinesOrg                                                !< copy max number of input lines in an input file
            MaxParameter = MaxParameterOrg                                                  !< copy max number of parameters in a line of an input file
            MaxIter = numiter                                                               !< copy max. number of iterations
            bestsitescount = ParamSetCounter                                                !< copy number of best bee sites
            QualityLimit = ParamSetCounter                                                  !< which positions should be used
            RenormalizedChi2 = .true.                                                       !< define flag for using renormalized chi**2
            if (RenormalizedChi2Org /= 1) then
                RenormalizedChi2 = .false.
            endif
            PlotIterationFlag = .false.
            if (PlotIteration == 0) PlotIterationFlag = .true.


            !< get special algorithm settings
            NumberBees = int(GeneralAlgorithmSettings(3))
            minmaxTrans = 0                                                                 !< number to invert the model function
            runcount = 1                                                                    !< number of repetition of the iteration loop
            max_func_counter = 10                                                           !< used for internal operations within the algorithm
                                                                                            !< (value from the algorithm xml-file)
            ! Debug:
            !    print*,"runcount = ", runcount
            !    print*,"max_func_counter = ", max_func_counter
            !    print*,"minmaxTrans = ", minmaxTrans
            !    print*,"NumberBees = ", NumberBees
            !    print*,'PathStartScriptOrg = ',trim(PathStartScriptOrg)
            !    print*,'ExeCommandStartScriptOrg = ',trim(ExeCommandStartScriptOrg)
            !    print*,'FitFktInputOrg = ',trim(FitFktInputOrg)
            !    print*,'MaxNumberParameter = ',MaxNumberParameter
            !    print*,'NumFileOrg = ',NumFileOrg
            !    print*,'MaxColXOrg = ',MaxColXOrg
            !    print*,'MaxColYOrg = ',MaxColYOrg
            !    Do i=1,NumFileOrg
            !        print*,'    Experimental file: i = ',i
            !        print*,'    lengthexpdataorg(i) = ',lengthexpdataorg(i)
            !        print*,'    NumberYColumnsOrg(i) = ',NumberYColumnsOrg(i)
            !        print*,'    expdataxorg(i,1:5,1) = ',expdataxorg(i,1:5,1)
            !        print*,'    expdatayorg(i,1:5,1) = ',expdatayorg(i,1:5,1)
            !        print*,'    expdataerrororg(i,1:5,1) = ',expdataerrororg(i,1:5,1)
            !    end Do
            !    print*,'chilm = ',chilm
            !    print*,'numiter = ',numiter
            !    print*,'fitlog = ',trim(fitlog)
            !    print*,'currentpathorg = ',trim(adjustl(currentpathorg))
            !    print*,'len(FitParameterNameOrg) = ',len(FitParameterNameOrg
            !    print*,'FitParameterNameOrg = ',FitParameterNameOrg
            !    print*,'FitParameterValueLineOrg = ',FitParameterValueLineOrg
            !    print*,"parametersetorg(1,:) = ",parametersetorg(1,:)
            !    print*,"parametersetorg(2,:) = ",parametersetorg(2,:)
            !    print*,"parametersetorg(3,:) = ",parametersetorg(3,:)
            !    print*,"parametersetorg(4,:) = ",parametersetorg(4,:)
            !    print*,"bestsitescount = ",bestsitescount
            !    print*,"RenormalizedChi2 = ",RenormalizedChi2
            !    return


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< set temp-directory
            TempDirectory = " "
            CALL GetEnv('MAGIXTempDirectory', TempDirectory)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< modify name of log file
            i = index(fitlog, "/", back = .true.)
            j = index(fitlog, ".", back = .true.)
            Number1 = "          "
            write(Number1,'(I10)') BeesCounter
            if (j > i) then
                if (NumberOfFitAlgorithms > 1) then
                    fitlog = trim(adjustl(fitlog(:j-1))) // "__Bees__call_" // trim(adjustl(Number1)) // trim(adjustl(fitlog(j:)))
                else
                    fitlog = trim(adjustl(fitlog(:j-1))) // "__Bees" // trim(adjustl(fitlog(j:)))
                endif
            else
                if (NumberOfFitAlgorithms > 1) then
                    fitlog = trim(adjustl(fitlog)) // "__Bees__call_" // trim(adjustl(Number1)) // ".log"
                else
                    fitlog = trim(adjustl(fitlog)) // "__Bees.log"
                endif
            endif

            ! Debug:
            !   print*,'>',trim(adjustl(fitlog)),'<'


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< open log file and write header
            open(logchannel, file = trim(fitlog), status='replace')
            write(logchannel, '(" ")')
            write(logchannel, '("log-file for Bees algorithm:")')
            write(logchannel, '(28("-"))')
            write(logchannel, '(" ")')


            !< get current local time and date and write to log-file
            call date_and_time(DATE, TIME, ZONE, VALUES)
            write(logchannel, '(" ")')
            write(logchannel, '("algorithm starts at Date: ",A2,".",A2,".",A4,",     Time: ",A2,":",A2,":",A2)') DATE(7:8),DATE(5:6),DATE(1:4), &
                                                                                                                 TIME(1:2),TIME(3:4),TIME(5:6)
            write(logchannel, '(" ")')
            write(logchannel, '(" ")')


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< open log-file for the parameter and write header
            fitlogparam = trim(fitlog) // ".param"
            open(paramchannel, file = trim(fitlogparam), status = 'replace')
            write(paramchannel, '(" ")')
            write(paramchannel, '("log-file containing the actual values of the parameters used in the Particle-Swarm algorithm:")')
            write(paramchannel, '(93("-"))')
            write(paramchannel, '(" ")')


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< open file containing the values of chi**2 and the corresponding values of the parameters
            WriteChi2Flag = .true.
            NumberLinesChi2 = 0
            fitlogChi2 = trim(fitlog) // ".chi2"
            open(Chi2Channel, file = trim(fitlogChi2), status = 'replace')


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< deallocate if necessary and print error message if necessay
            if (allocated(expdatax)) then
                deallocate(expdatax, expdatay, expdatae, lengthexpdata, NumberXColumns, NumberYColumns, FirstPointExpData, LastPointExpData, &
                            NumberRanges, MinRange, MaxRange, ExpData_reversed_flag, stat = deallocstatus)
                if (deallocstatus /= 0) then                                                !< is all ok?
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine dofit:")')
                    write(logchannel,'(2x,"Can not deallocate variables expdatax etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine dofit:")'
                    print '(2x,"Can not deallocate variables expdatax etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< allocate memory for variables, clear content of the variables and print error message if necessay
            allocate(expdatax(NumberExpFiles, MaxLength, MaxColX), expdatay(NumberExpFiles, MaxLength, MaxColY), &
                     expdatae(NumberExpFiles, MaxLength, MaxColY), lengthexpdata(NumberExpFiles), NumberXColumns(NumberExpFiles), &
                     NumberYColumns(NumberExpFiles), FirstPointExpData(NumberExpFiles, MaxColX), LastPointExpData(NumberExpFiles, MaxColX), &
                     NumberRanges(NumberExpFiles), MinRange(NumberExpFiles, MaxRangeNumber, MaxColX), MaxRange(NumberExpFiles, MaxRangeNumber, MaxColX), &
                     ExpData_reversed_flag(NumberExpFiles), stat = allocstatus)
            if (allocstatus /= 0) then                                                      !< is all ok?
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine dofit:")')
                write(logchannel,'(2x,"Can not allocate variables expdatax etc.")')
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine dofit:")'
                print '(2x,"Can not allocate variables expdatax,expdatay etc.")'
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("allocstatus = ",I4)',allocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            MaxRangeNumber = MaxRangeNumber - 1                                             !< get real value
            expdatax = 0.d0
            expdatay = 0.d0
            expdatae = 0.d0
            lengthexpdata = 0
            NumberXColumns = 0
            NumberYColumns = 0
            FirstPointExpData = 1.d99
            LastPointExpData = -1.d99
            NumberRanges = 0
            MinRange = 0.d0
            MaxRange = 0.d0
            ExpData_reversed_flag = .false.


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< copy input variables to module variabels
            expdatax(:,:,:) = expdataxorg(:,:,:)
            expdatay(:,:,:) = expdatayorg(:,:,:)
            expdatae(:,:,:) = expdataerrororg(:,:,:)
            lengthexpdata = lengthexpdataorg
            NumberXColumns = NumberXColumnsOrg
            NumberYColumns = NumberYColumnsOrg
            CalculationMethod = CalculationMethodOrg
            PathStartScript = PathStartScriptOrg
            ExeCommandStartScript = ExeCommandStartScriptOrg
            NumberRanges = NumberRangesOrg
            MinRange = MinRangeOrg
            MaxRange = MaxRangeOrg


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine first and last point of each exp. data file
            Do i = 1, NumberExpFiles
                Do j = 1, lengthexpdata(i)
                    ii = 0
                    jj = 0
                    Do k = 1,NumberXColumns(i)
                        if (expdatax(i, j, k) <= FirstPointExpData(i, k)) then
                            ii = ii + 1
                        endif
                        if (expdatax(i, j, k) >= LastPointExpData(i, k)) then
                            jj = jj + 1
                        endif
                    end Do
                    if (ii == NumberXColumns(i)) then
                        FirstPointExpData(i, 1:NumberXColumns(i)) = expdatax(i, j, 1:NumberXColumns(i))
                    endif
                    if (jj == NumberXColumns(i)) then
                        LastPointExpData(i, 1:NumberXColumns(i)) = expdatax(i, j, 1:NumberXColumns(i))
                    endif
                end Do


                !< check output file starts with the highest x-column value interchange FirstPointOutputFile and LastPointOutputFile
                ii = 0
                Do k = 1, NumberXColumns(i)
                    if (expdatax(i, 1, k) >= expdatax(i, lengthexpdata(i), k)) then
                        ii = ii + 1
                    endif
                end Do
                if (ii == NumberXColumns(i)) then
                    ExpData_reversed_flag(i) = .true.
                endif

                ! Debug:
                ! print*,' '
                ! print*,'File = ',i
                ! print*,'FirstPointExpData(i, 1:NumberXColumns(i)) = ', FirstPointExpData(i, 1:NumberXColumns(i))
                ! print*,'LastPointExpData(i, 1:NumberXColumns(i)) = ', LastPointExpData(i, 1:NumberXColumns(i))
                ! print*,'##########################################'
            end Do


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< copy number of parameters required for the model function to working variable
            nfit = int(sum(parametersetorg(2, :)))                                          !< number of parameters which should be optimized
            ma = parameternumber                                                            !< copy total number of parameters
            Gradientflag = .false.                                                          !< we do not need the gradient of the function here
            UseCalculationReduction = .false.                                               !< activate calculation reduction
            CurrentNumberLinesCalcReduction = 0                                             !< reset number of lines


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< deallocate/allocate memory for some working variables, clear contents of the variables and print error message if necessary
            if (allocated(paramset)) then
                deallocate(paramset, FitParameterName, FitParameterValue, ia, a, OptimizedParameterLowLimit, OptimizedParameterUpperLimit, &
                           ModelFunction, chisqValues, BestSitesParamSet, BestSitesModelValues, BestSitesChi2Values, AtOnceFunction, &
                           ConverterInfit, stat = deallocstatus)
                if (deallocstatus /= 0) then
                    write(logchannel,*)
                    write(logchannel,'("Error in subroutine dofit:")')
                    write(logchannel,'(2x,"Can not deallocate variables paramset etc.")')
                    write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                    write(logchannel,*)
                    write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                    write(logchannel,'(" ")')
                    write(logchannel,'("Program aborted!")')

                    print '(" ")'
                    print '("Error in subroutine dofit:")'
                    print '(2x,"Can not deallocate variables paramset etc.")'
                    print '(2x,"Please close all other programs and restart the program!")'
                    print '(" ")'
                    print '("deallocstatus = ",I4)',deallocstatus
                    print '(" ")'
                    stop ' Program aborted!'
                endif
            endif
            allocate(paramset(4, parameternumber), OptimizedParameterLowLimit(nfit), OptimizedParameterUpperLimit(nfit), ConverterInfit(nfit), &
                     ia(parameternumber), FitParameterName(parameternumber), FitParameterValue(parameternumber), a(parameternumber), &
                     ModelFunction(1, NumberExpFiles, MaxColY, MaxLength), chisqValues(0:0), &
                     AtOnceFunction(0:ParallelizationFlag - 1, NumberExpFiles, MaxColY, MaxLength), &
                     BestSitesParamSet(bestsitescount, nfit + 1), BestSitesModelValues(bestsitescount, NumberExpFiles, MaxLength, MaxColY), &
                     BestSitesChi2Values(bestsitescount, NumberExpFiles, MaxLength, MaxColY), stat = allocstatus)
            if (allocstatus /= 0) then
                write(logchannel,'(" ")')
                write(logchannel,'("Error in subroutine dofit:")')
                write(logchannel,'(2x,"Can not allocate variables paramset etc.")')
                write(logchannel,'(2x,"Please close all other programs and restart the program!")')
                write(logchannel,'(" ")')
                write(logchannel,'("allocstatus = ",I4)') allocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine dofit:")'
                print '(2x,"Can not allocate variables paramset etc.")'
                print '(2x,"Please close all other programs and restart the program!")'
                print '(" ")'
                print '("allocstatus = ",I4)',allocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            ia = .false.
            FitParameterName = FitParameterNameLocal
            FitParameterValue = FitParameterValueLocal
            OptimizedParameterLowLimit = 0.d0
            OptimizedParameterUpperLimit = 0.d0
            ModelFunction = 0.d0
            AtOnceFunction = 0.d0
            chisqValues = 0.d0
            ConverterInfit = 0
            BestSitesParamSet = 0.d0
            BestSitesParamSet(:, 1) = 1.d99
            BestSitesModelValues = 0.d0
            BestSitesChi2Values = 0.d0


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< copy parameter set to module variable
            paramset = parametersetorg


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< write registration mask for the fit model
            if (printflag) print '(9x, "Writing registration mask for the fit model .. ", $)'
            call RegistrationMask(ok)
            if (ok /= 0) then
                return
            endif
            if (printflag) print '("done!")'


            !< define ia variable
            ConverterInfit = 0
            ia = .false.
            k = 0
            Do i = 1, parameternumber
                if (paramset(2, i) == 1) then
                    k = k + 1
                    ia(i) = .true.
                    ConverterInfit(k) = i
                endif
            end Do
            NumberFreeParameter = int(sum(paramset(2, :)))                                  !< determine number of free parameter


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< initialize model program
            call ModelInit


            !< write on the screen what you Do ..
            if (printflag) then
                print '(" ")'
                print '(" ")'
                write(Number1,'(I10)') JobID                                                !< write JobID to string
                print '(9x,"Temporary files are stored in: ",A)', trim(adjustl(TempDirectory)) // "job_" // trim(adjustl(Number1)) // "/"
                print '(" ")'
                print '(" ")'
                print '(9x,"Start Bees algorithm (", A, " version) ..")', trim(adjustl(ParallelizationMethod))
                print '(" ")'
            endif
            write(logchannel,'(11x,"Start Bees algorithm (", A, " version) ..")') trim(adjustl(ParallelizationMethod))
            write(logchannel,'(" ")')


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< lower limit of chi^2 for stopping condition normalized to the number of calculation points
            chilim = 0.d0
            if (RenormalizedChi2) then
                Do i = 1, NumberExpFiles
                    chilim = chilim + (NumberYColumns(i) * lengthexpdata(i) - parameternumber) * dabs(chilm)
                end Do
                chilim = dabs(chilim)
                write(logchannel,'(11x,"Renormalized limit for chi^2 = ", ES25.15)') chilim
                write(logchannel,'(" ")')
                if (printflag) then
                    print '(11x,"Renormalized limit for chi^2 = ",ES25.15)', chilim
                    print '(" ")'
                endif
            else
                chilim = dabs(chilm)
                write(logchannel,'(11x,"Limit for chi^2 = ", ES25.15)') chilim
                write(logchannel,'(" ")')
                if (printflag) then
                    print '(11x,"Limit for chi^2 = ",ES25.15)', chilim
                    print '(" ")'
                endif
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< start iteration ..
            Gradientflag = .false.                                                          !< we do not need the gradient of the function here
            flag = 0                                                                        !< set flag variable to 0
            actualiteration = 0                                                             !< set working variable containing the current iteration
                                                                                            !< number to 0
            a = paramset(1, 1:parameternumber)                                              !< copy the values of the parameters to another array


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< call subroutine to call bees algorithm
            FinalParameterSet = 0.d0
            call CallBees(NumberBees, nfit, runcount, ma, ParamSetCounter, FinalParameterSet, MaxColX, NumberExpFiles, MaxLength, MaxColY, &
                          PlotIteration, PlotType, xAxisLabel, yAxisLabel, zAxisLabel, fitlog)
            ParamSetCounter = QualityLimit
            FitFunctionOut = 0.d0
            Chi2Values = 0.d0
            Do i  = 1, QualityLimit


                !-----------------------------------------------------------------------------------------------------------------------------------------
                !< determine model function values if plot for each iteration option is not selected
                if (.not. PlotIterationFlag .and. DontStoreModelFuncValuesFlag) then
                    PlotIterationFlag = .true.
                    chi2ValuesVector = 0.d0
                    call ModelCalcChiFunctionGeneral(parameternumber, ia, paramset(1, :), 1, nfit, NumFileOrg, MaxColY, MaxLength, &
                                                     BestSitesParamSet(i, 2:), chi2ValuesVector)
                    PlotIterationFlag = .false.
                endif
                Do j = 1, NumFileOrg
                    Do k = 1, MaxLength
                        FitFunctionOut(i, j, k, 1:MaxColY) = BestSitesModelValues(i, j, k, 1:MaxColY)
                        Chi2Values(i, j, k, 1:MaxColY) = BestSitesChi2Values(i, j, k, 1:MaxColY)
                    end Do
                end Do
            end Do


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< close file
            call SortChi2File(nfit, parameternumber, ia, a)
            close(Chi2Channel)
            close(paramchannel)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< write final parameter sets for each site to final input files


            !< define base directory, i.e. path where final input file is written to
            k = index(fitlog, '/', back = .true.)
            if (k == 0) then
                BaseDir = ""
            else
                BaseDir = trim(adjustl(fitlog(:k)))
            endif


            !< define file name extension for number of algorithm call
            write(Number1,'(I10)') BeesCounter
            if (NumberOfFitAlgorithms > 1) then
                FuncCallExt = "__Bees__call_" //trim(adjustl(Number1))
            else
                FuncCallExt = "__Bees"
            endif

            ! Debug:
            !    print*,"NumberOfFitAlgorithms = ", NumberOfFitAlgorithms
            !    print*,"FuncCallExt = ", trim(adjustl(FuncCallExt))


            !< write files
            Do i  = 1, QualityLimit                                                         !< loop over all sites


                !< create site extension for file name
                write(helpString,'(I5)') i
                SiteExt = "__site_" // trim(adjustl(helpString)) // ".out"


                !< write parameter sets to file
                Do NumInputFiles = 1, NumberInputFiles                                      !< loop over all input files
                    NumExt = trim(adjustl(FitFktInput(NumInputFiles)))
                    k = index(NumExt, '/', back = .true.)
                    if (k > 0) then                                                         !< we only need the file name
                        NumExt = trim(adjustl(NumExt(k:)))
                    endif
                    j = index(NumExt, '.', back = .true.)
                    if (j > 1) then                                                         !< remove file name extension
                        NumExt = trim(adjustl(NumExt(:j - 1)))
                    endif

                    ! Debug:
                    ! print*,"Site = ", i
                    ! print*,"Nr. Final input file = ", NumInputFiles
                    ! print*,"Final input file = ", trim(adjustl(BaseDir)) // trim(adjustl(NumExt)) // trim(adjustl(FuncCallExt))  &
                    !                               // trim(adjustl(SiteExt)) // ".input"


                    !< write parameter sets to file
                    open(235,file = trim(adjustl(BaseDir)) // trim(adjustl(NumExt)) // trim(adjustl(FuncCallExt)) // trim(adjustl(SiteExt)) // ".input")
                    LocalParamVector(:) = FinalParameterSet(i, :)
                    call WriteParameter(235, .false., MaxColX, expdataxorg, parameternumber, LocalParamVector(:), NumInputFiles)
                    close(235)
                end Do
            end Do


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< write end of log-file and print message to screen
            call date_and_time(DATE, TIME, ZONE, VALUES)
            write(logchannel,*)
            write(logchannel,'("algorithm ends at Date: ",A2,".",A2,".",A4,",     Time: ",A2,":",A2,":",A2)') DATE(7:8),DATE(5:6),DATE(1:4), &
                                                                                                                  TIME(1:2),TIME(3:4),TIME(5:6)
            write(logchannel,'(" ")')
            write(logchannel,'(150("-"))')
            close(logchannel)
            if (printflag) then
                print '(" ")'
                print '(" ")'
                print '(9x,"Finished Bees algorithm!")'
                print '(" ")'
                print '(" ")'
            endif

            ! Debug:
            !    print*,'lengthexpdata = ',lengthexpdata
            !    print*,'MaxColX = ',MaxColX
            !    print*,'MaxColY = ',MaxColY
            !    print*,'expdataxorg(1:5,1) = ',expdataxorg(1:5,1)
            !    print*,'expdatayorg(1:5,1) = ',expdatayorg(1:5,1)
            !    print*,'expdataerrororg(1:5,1) = ',expdataerrororg(1:5,1)
            !    print*,'chilm = ',chilm
            !    print*,'numiter = ',numiter
            !    print*,'fitlog = ',trim(fitlog)
            !    print*,'model = ',trim(model)
            !    print*,'currentpathorg = ',currentpathorg
            !    print*,'parametersetorg(1,:) = ',parametersetorg(1,:)
            !    print*,'parametersetorg(2,:) = ',parametersetorg(2,:)
            !    print*,'parametersetorg(3,:) = ',parametersetorg(3,:)
            !    print*,'parametersetorg(4,:) = ',parametersetorg(4,:)


            !< send plot program the kill signal
            if (PlotIteration == 0) then
                call system("kill -9 " // trim(adjustl(PlotPID)) // " 1>/dev/null 2>&1")
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< free memory of general Bees variables
            if (allocated(ia)) deallocate(ia, stat = deallocstatus)
            if (allocated(a)) deallocate(a, stat = deallocstatus)
            if (allocated(OptimizedParameterLowLimit)) deallocate(OptimizedParameterLowLimit, stat = deallocstatus)
            if (allocated(OptimizedParameterUpperLimit)) deallocate(OptimizedParameterUpperLimit, stat = deallocstatus)
            if (allocated(floatbee_fitness)) deallocate(floatbee_fitness, stat = deallocstatus)
            if (allocated(floatbee_position)) deallocate(floatbee_position, stat = deallocstatus)
            if (allocated(bestsites)) deallocate(bestsites, stat = deallocstatus)
            if (allocated(bestsitesPosition)) deallocate(bestsitesPosition, stat = deallocstatus)
            if (allocated(selsites)) deallocate(selsites, stat = deallocstatus)
            if (allocated(selsitesPosition)) deallocate(selsitesPosition, stat = deallocstatus)
            if (allocated(HiveRange)) deallocate(HiveRange, stat = deallocstatus)
            if (allocated(bestposition)) deallocate(bestposition, stat = deallocstatus)


            !< free memory of model variables
            call ModelParamFree(deallocstatus)
            if (deallocstatus /= 0) then
                write(logchannel,*)
                write(logchannel,'("Error in subroutine MainAlg:")')
                write(logchannel,'(2x,"Can not deallocate expdatax etc.")')
                write(logchannel,*)
                write(logchannel,'("deallocstatus = ",I4)') deallocstatus
                write(logchannel,'(" ")')
                write(logchannel,'("Program aborted!")')

                print '(" ")'
                print '("Error in subroutine MainAlg:")'
                print '(2x,"Can not deallocate variables expdatax etc.")'
                print '(" ")'
                print '("deallocstatus = ",I4)',deallocstatus
                print '(" ")'
                stop ' Program aborted!'
            endif
            return
    end subroutine MainAlg
end Module Algorithm
!*********************************************************************************************************************************************************

