#!/usr/bin/env python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  Interface for using some optimization algorithms from the scipy package
##  Copyright (C) 2009 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  Input:
##  ------
##
##  Point of minimum (after any optimization procedure)
##  Parameter space
##
##  Output:
##  -------
##
##  Mean value for each parameter
##  Standard deviations for each parameter
##  Histogramms of parameter values
##
##
##
##  The following subroutines and functions are included in this module:
##
##      - subroutine start:                                                 main subroutine for package for making use of other optimization
##                                                                          packages like Scipy etc. error in point of minimum
##      - subroutine ModelFunctionCallClass.__init__:                       initialize class ModelFunctionCallClass
##      - subroutine ModelFunctionCallClass.setParameters:                  set class variables to parameter values
##      - subroutine ModelFunctionCallClass.DetermineModelFunction:         call of model-function
##      - subroutine ModelFunctionCallClass.DetermineModelFunctionGradient: determines the gradient of the model function
##      - subroutine ModelFunctionCallClass.DetermineModelFunctionAll:      determine model function value for current parameter vector
##      - subroutine ModelFunctionCallClass.GetBestResult:                  get best result
##      - subroutine ModelFunctionCallClass.GetBestFunctionValues:          get function values of best result
##      - subroutine ModelFunctionCallClass.PrintIteration:                 plot model function together with experimental data for each iteration
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    09.06.2011   initial version
##  T. Moeller    16.01.2012   improve documentation of source code
##  T. Moeller    05.11.2014   modifications for MPI and GPU
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##******************************************************************** load packages *********************************************************************
import numpy                                                                                ## load numpy package
import random                                                                               ## load random package
import sys                                                                                  ## load sys package
import time                                                                                 ## load time package
import os                                                                                   ## load os package
import FittingEngine                                                                        ## import package containing the fitting engine


##********************************************************************************************************************************************************
def start(SpecialAlgorithmSettings, GeneralAlgorithmSettings, printflagNum, LastAlgorithmNum, ParamSetCounter, chilm, numiter, \
          AdditionalPackagesCounter, DeterminationChi2, PlotIteration, PlotType, fit_log, NumberInputFiles, NumberOutputFiles, ParallelizationFlag, \
          JobID, MaxInputLines, MaxParameter, RenormalizedChi2, currentpath, CalculationMethod, xAxisLabel, yAxisLabel, zAxisLabel, PathStartScript, \
          ExeCommandStartScript, parameter_set, ExpDataX, ExpDataY, ExpDataError, NumberRanges, MinRange, MaxRange, NumberXColumns, NumberYColumns, \
          LengthExpRange, MaxRangeNumber, NumberExpFiles, MaxLength, MaxColX, MaxColY, NumberParameter, modelflag, MPIFlag, MAGIXrootDirectory, JobDir):
    """

    package for making use of other optimization packages like Scipy etc.


    input variables:            SpecialAlgorithmSettings:   algorithm user setttings
                                GeneralAlgorithmSettings:   special algorithm settings
                                printflagNum:           flag for screen output 1 (=yes) or 0 (=no)
                                LastAlgorithmNum:       number of last algorithm
                                ParamSetCounter:        number of best sites
                                chilm:                  user defined abort criteria for chi**2
                                numiter:                max. number of iterations
                                AdditionalPackagesCounter:  counts number of calls
                                DeterminationChi2:      method being used for the determination of chi^2
                                PlotIteration:          plot model function for each iteration set 1(=yes) or 0(=no)
                                PlotType:               get type of plot
                                fit_log:                path for log-file containing the current values of chi**2
                                NumberInputFiles:       number of input files for the external model program
                                NumberOutputFiles:      number of output files for the external model program
                                ParallelizationFlag:    contains the number of processors used for parallelization
                                JobID:                  job identification number
                                MaxInputLines:          max number of lines in an input file
                                MaxParameter:           max number of parameters in a line of an input file
                                RenormalizedChi2:       flag for using renormalized chi**2
                                currentpath:            path of the working directory
                                CalculationMethod:      method of computation (at once or point-to-point)
                                xAxisLabel:             label of the x-axis (for plot)
                                yAxisLabel:             label of the y-axis (for plot)
                                zAxisLabel:             label of the z-axis (for plot)
                                PathStartScript:        path and name of the start script for calling model function
                                ExeCommandStartScript:  command for calling model function
                                parameter_set:          the complete set of paramters (incl. flags and limits)
                                ExpDataX:               array containing the experimental x side
                                ExpDataY:               array containing the experimental y side
                                ExpDataError:           array containing the experimental error of the y side
                                NumberRanges:           number of ranges
                                MinRange:               array containing lower limit of each range for each file
                                MaxRange:               array containing upper limit of each range for each file
                                NumberXColumns:         number x-columns
                                NumberYColumns:         number y-columns
                                LengthExpRange:         total number of data points
                                MaxRangeNumber:         max number of ranges over all files
                                NumberExpFiles:         number of exp. data files
                                MaxLength:              max. length of experimental data
                                MaxColX:                number of columns concerning to the experimental x side
                                MaxColY:                number of columns concerning to the experimental y side
                                NumberParameter:        number of model parameter
                                plotflag:               flag for plotting histograms
                                modelflag:              flag to use optimized MAGIX packages for a certain model or not
                                MPIFlag:                mpi flag
                                MAGIXrootDirectory:     MAGIX root directory
                                JobDir:                 job directory
                                SpecialAlgorithmSettings:   user settings for algorithm


    output variables:           calstatus:              status flag of calculation (= 0: all ok)
                                FitFunctionValues:      values of the model function of the last function call
                                Chi2Values:             corresponding chi^2 values
                                parameter_set:          compelete parameter set

    """


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import MAGIX package ModelFunctionCall ..


    ## .. optimized for myXCLASS model
    if (modelflag == "true" or modelflag.lower() == "myxclass"):
        StarterExecutable = "Starter__myXCLASS.exe"


    ## .. default package
    else:
        StarterExecutable = "Starter.exe"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    class ModelFunctionCallClass:
        """
        This class contains all required parameters for the ModelFunctionCall package
        """


        ##************************************************************************************************************************************************
        def __init__(self, printflagNumIn, GeneralAlgorithmSettingsIn, LastAlgorithmNumIn, DeterminationChi2In, PlotIterationIn, \
                     PlotTypeIn, fit_logIn, NumberInputFilesIn, NumberOutputFilesIn, ParallelizationFlagIn, JobIDIn, MaxInputLinesIn, MaxParameterIn, \
                     RenormalizedChi2In, currentpathIn, CalculationMethodIn, xAxisLabelIn, yAxisLabelIn, zAxisLabelIn, PathStartScriptIn, \
                     ExeCommandStartScriptIn, parameter_setIn, ExpDataXIn, ExpDataYIn, ExpDataErrorIn, NumberRangesIn, MinRangeIn, MaxRangeIn, \
                     NumberXColumnsIn, NumberYColumnsIn, LengthExpRangeIn, MaxRangeNumberIn, NumberExpFilesIn, MaxLengthIn, MaxColXIn, MaxColYIn, \
                     NumberParameterIn, AdditionalPackagesCounterIn, MPIFlagIn, MAGIXrootDirectoryIn, JobDirIn, SpecialAlgorithmSettingsIn):
                self.printflagNum = printflagNumIn
                self.GeneralAlgorithmSettings = GeneralAlgorithmSettingsIn
                self.LastAlgorithmNum = LastAlgorithmNumIn
                self.DeterminationChi2 = DeterminationChi2In
                self.PlotIteration = PlotIterationIn
                self.PlotType = PlotTypeIn
                self.fitlog = fit_logIn
                self.NumberInputFiles = NumberInputFilesIn
                self.NumberOutputFiles = NumberOutputFilesIn
                self.ParallelizationFlag = ParallelizationFlagIn
                self.JobID = JobIDIn
                self.MaxInputLines = MaxInputLinesIn
                self.MaxParameter = MaxParameterIn
                self.RenormalizedChi2 = RenormalizedChi2In
                self.currentpath = currentpathIn
                self.CalculationMethod = CalculationMethodIn
                self.xAxisLabel = xAxisLabelIn
                self.yAxisLabel = yAxisLabelIn
                self.zAxisLabel = zAxisLabelIn
                self.PathStartScript = PathStartScriptIn
                self.ExeCommandStartScript = ExeCommandStartScriptIn
                self.parameter_set = parameter_setIn
                self.ExpDataX = ExpDataXIn
                self.ExpDataY = ExpDataYIn
                self.ExpDataError = ExpDataErrorIn
                self.NumberRanges = NumberRangesIn
                self.MinRange = MinRangeIn
                self.MaxRange = MaxRangeIn
                self.NumberXColumns = NumberXColumnsIn
                self.NumberYColumns = NumberYColumnsIn
                self.LengthExpRange = LengthExpRangeIn
                self.MaxRangeNumber = MaxRangeNumberIn
                self.NumberExpFiles = NumberExpFilesIn
                self.MaxLength = MaxLengthIn
                self.MaxColX = MaxColXIn
                self.MaxColY = MaxColYIn
                self.NumberParameter = NumberParameterIn
                self.MPIFlag = MPIFlagIn
                self.MAGIXrootDirectory = MAGIXrootDirectoryIn
                self.JobDir = JobDirIn
                self.SpecialAlgorithmSettings = SpecialAlgorithmSettingsIn
                self.BestChi2 = 1.e99
                self.BestFunctionValues = 0.0
                self.BestChi2Values = 0.0
                self.BestParameter = 0.0
                self.BestLogLine = ""
                self.BestInputFiles = ""
                self.FuncCallCounter = 0
                self.IterCounter = 0
                self.TempDir = str(os.environ.get('MAGIXTempDirectory',''))
                self.parmLog = 0
                self.Log = 0
                self.chi2Log = 0
                self.PlotFunctionCall = 0
                self.AdditionalPackagesCounter = AdditionalPackagesCounterIn
                self.CurrIter = (-1)                                                        ## turn initialization flag on


        ##************************************************************************************************************************************************
        def setParameters(self, PlotFuncCall, paramchannel, logchannel, chi2channel):
            """
            set class variables to parameter values
            """
            self.PlotFunctionCall = PlotFuncCall
            self.parmLog = paramchannel
            self.Log = logchannel
            self.chi2Log = chi2channel
            return


        ##************************************************************************************************************************************************
        def DetermineModelFunction(self, parameter_vector):
            """
            call of DetermineModelFunction

            input variables:    parameter_vector:       parameter vector

            output variables:   chi2value:              value of chi^2
            """
            number_param_set = 1

            # Debug:
            # print 'number_param_set = ',number_param_set
            # print 'parameter_vector = ',parameter_vector


            ## reset output value
            chi2value = 0.0


            ## update parameter_set array
            if (number_param_set == 1):
                dummyVector = parameter_vector
                parameter_vector = []
                parameter_vector.append(dummyVector)
            parameter_vector_set = self.parameter_set
            for k in xrange(number_param_set):
                j = (-1)
                for i in xrange(self.NumberParameter):
                    if (self.parameter_set[1][i] == 1):
                        j += 1
                        parameter_vector_set[k][i] = parameter_vector[k][j]


            ## write new parameter vectors to file
            WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"
            NewParamFile = open(WorkingDirectory.strip() + "new-parameters.dat", 'w')
            for k in xrange(number_param_set):
                j = (-1)
                line = ""
                for i in xrange(self.NumberParameter):
                    if (self.parameter_set[1][i] == 1):
                        j += 1
                        line += "   " + str(parameter_vector[k][j])
                    else:
                        line += "   " + str(self.parameter_set[0][i])
                NewParamFile.write(line + "\n")
            NewParamFile.close()

            # Debug:
            # print 'ModelFunctionCall.startcall.__doc__ = ',ModelFunctionCall.startcall.__doc__
            # print 'self.printflagNum = ',self.printflagNum
            # print 'number_param_set = ',number_param_set
            # print 'self.DeterminationChi2 = ',self.DeterminationChi2
            # print 'self.PlotIteration = ', self.PlotIteration
            # print 'self.PlotType = ',self.PlotType
            # print 'self.NumberInputFiles = ',self.NumberInputFiles
            # print 'self.NumberOutputFiles = ',self.NumberOutputFiles
            # print 'self.ParallelizationFlag = ',self.ParallelizationFlag
            # print 'self.JobID = ',self.JobID
            # print 'self.MaxInputLines = ',self.MaxInputLines
            # print 'self.MaxParameter = ',self.MaxParameter
            # print 'self.RenormalizedChi2 = ',self.RenormalizedChi2


            ## define dummy arguments
            chilm = 1.0
            NumberOfFitAlgorithms = 1
            AlgCounter = 1


            ## call model function package
            ParamSetCounter = number_param_set
            numiter = self.CurrIter
            NameOfAlgorithm = "Model-Function_Call"
            calstatus, FitFunctionOut, Chi2Values, parmCopy, FinalParameterSet = FittingEngine.StartAlgPackage(NameOfAlgorithm, StarterExecutable, \
                                                                                            self.MPIFlag, self.MAGIXrootDirectory, self.JobDir, \
                                                                                            self.JobID, self.NumberExpFiles, self.MaxLength, \
                                                                                            self.MaxColX, self.MaxColY, self.MaxRangeNumber, \
                                                                                            self.NumberParameter, ParamSetCounter, \
                                                                                            self.LastAlgorithmNum, numiter, self.ParallelizationFlag, \
                                                                                            self.DeterminationChi2, self.PlotIteration, self.PlotType, \
                                                                                            self.RenormalizedChi2, AlgCounter, chilm, \
                                                                                            self.GeneralAlgorithmSettings, \
                                                                                            self.SpecialAlgorithmSettings, parameter_vector_set, \
                                                                                            self.xAxisLabel, self.yAxisLabel, self.zAxisLabel)
            self.CurrIter = 0                                                               ## turn initialization flag off


            ## get calculated chi2 values from temp file
            WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"
            ResultFile = open(WorkingDirectory.strip() + "chi2Values.dat", 'r')
            ResultFileContents = ResultFile.readlines()
            ResultFile.close()
            ValueOfChi2 = []
            for val in ResultFileContents:
                ValueOfChi2.append(float(val))

            # Debug:
            # print "ValueOfChi2 = ", ValueOfChi2


            ## check if parameter vector is within given range
            for i in xrange(number_param_set):                                              ## loop over parameter sets
                k = (-1)
                for j in xrange(self.NumberParameter):                                      ## loop over all parameter
                    if (self.parameter_set[1][j] == 1):
                        k += 1
                        value = float(parameter_vector[i][k])
                        if (value < float(self.parameter_set[2][j]) or value > float(self.parameter_set[3][j])):
                            ValueOfChi2[i] = 1.e99
                            FitFunctionOut[i] = 0.e0
                            Chi2Values = 0.e0


            ## define output variable
            if (number_param_set == 1):
                chi2value = ValueOfChi2[0]
            else:
                chi2value = ValueOfChi2

            # Debug:
            # print 'parameter_vector = ',parameter_vector
            # print 'chi2value = ',chi2value
            # sys.exit(0)


            ## save best chi2
            WorkingDirectory = self.TempDir.strip() + "job_" + str(self.JobID).strip() + "/"
            for i in xrange(number_param_set):


                ## append content of chi2-log file to total list of all calculated chi^2 values
                f = open(WorkingDirectory.strip() + "log_chi2_single_call.dat", 'r')
                contentChi2LogFile = f.readlines()
                for line in contentChi2LogFile:
                    self.chi2Log.write(line)
                    self.chi2Log.flush()
                f.close()


                ## current value of chi^2 better than saved result ?
                if (ValueOfChi2[i] < self.BestChi2):
                    self.BestChi2 = ValueOfChi2[i]
                    self.BestParameter = parameter_vector[i]
                    self.BestFunctionValues = FitFunctionOut[i]
                    self.BestChi2Values = Chi2Values[i]


                    ## write best model function values to temp file
                    for j in xrange(NumberExpFiles):
                        f = open(WorkingDirectory.strip() + "best_model_function_values_call__" + str(i + 1) + "__" + str(j + 1) + ".dat", 'w')
                        for lines in self.BestFunctionValues[j]:
                            f.write(str(lines[0]) + "\n")
                        f.close()


                    ## write chi2 values to temp file
                    for j in xrange(NumberExpFiles):
                        f = open(WorkingDirectory.strip() + "best_chi2_values_call__" + str(i + 1) + "__" + str(j + 1) + ".dat", 'w')
                        for lines in self.BestFunctionValues[j]:
                            f.write(str(lines[0]) + "\n")
                        f.close()


                    ## get corresponding line with formatted parameters
                    f = open(WorkingDirectory.strip() + "log_lines__single_call__" + str(i + 1) + ".dat", 'r')
                    self.BestLogLine = f.readline()
                    self.BestLogLine = self.BestLogLine[0:len(self.BestLogLine)-1]
                    f.close()


                    ## get corresponding contents of the input file(s)
                    f = open(WorkingDirectory.strip() + "log_input-files__single_call__" + str(i + 1) + ".dat", 'r')
                    self.BestInputFiles = f.readlines()
                    f.close()


                    ## get
                    # Debug:
                    # print 'self.BestChi2 = ',self.BestChi2
                    # print 'self.BestLogLine = ',self.BestLogLine


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## print what you do
            if (self.BestChi2 < 1.e99):
                self.FuncCallCounter += 1                                                   ## increase counter for model function calls
                if (self.PlotFunctionCall == 1):


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## write informations to log-file
                    if (printflag == "true"):
                        print "                %5d%s" % (self.FuncCallCounter, self.BestLogLine)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## get channels for log-files
                    paramchannel = self.parmLog
                    logchannel = self.Log
                    Chi2Channel = self.chi2Log


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## wirte information to param-log file
                    paramchannel.write("\n")
                    paramchannel.write("\n")
                    paramchannel.write("**************************************************************************************************************************\n")
                    outputstring = "Iteration: %5d,  chi^2 = %25.15e" % (self.FuncCallCounter, self.BestChi2)
                    paramchannel.write(outputstring + "\n")
                    paramchannel.write("\n")
                    paramchannel.write("\n")
                    paramchannel.write("\n")
                    paramchannel.write("Parameters: " + str(self.BestParameter) + "\n")
                    paramchannel.write("--------------------------------------------------------------------------------------------------------------------------\n")
                    paramchannel.write("\n")
                    paramchannel.flush()


                    ## write contents of input file(s) to param-log file
                    for line in self.BestInputFiles:
                        paramchannel.write(line)
                        paramchannel.flush()


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## wirte information to log file
                    outputstring = "           %10d  %s" % (self.FuncCallCounter, self.BestLogLine)
                    logchannel.write(outputstring + "\n")
                    logchannel.flush()

                else:
                    if (printflag == "true"):
                        #if (self.FuncCallCounter % 11 == 1):
                        #    command_string = "echo -en \"\r                      \""
                        #    os.system(command_string)
                        #    command_string = "echo -en \"\r           \""
                        #else:
                        command_string = "echo -n \".\""
                        os.system(command_string)

            return chi2value


        ##************************************************************************************************************************************************
        def DetermineModelFunctionGradient(self, parameter_vector):
            """
            determines the gradient of the model function
            """


            ## determine model function value for current parameter vector
            chisqNormal = modelClass.DetermineModelFunction(parameter_vector)


            ## determine gradient of model function
            num_par = sum(self.parameter_set[1])
            ModelFunctionGradient = numpy.zeros(num_par)
            j = (-1)
            for i in xrange(self.NumberParameter):                                          ## loop over all parameters
                if (self.parameter_set[1][i] == 1):
                    j += 1
                    ModParameter = parameter_vector                                         ## save current parameter vector


                    ## define variation
                    variation = 1.e-6
                    if (float(parameter_vector[j]) == 0.0):
                        h = variation
                    else:
                        h = parameter_vector[j] + (parameter_vector[j] * variation)         ## determine h
                    h = float(h)


                    ## is modified parameter out of limits
                    if (h < float(self.parameter_set[2][i]) or h > float(self.parameter_set[3][i])):
                        if (h < float(self.parameter_set[2][i])):
                            h = float(self.parameter_set[2][i])
                        elif (h > float(self.parameter_set[3][i])):
                            h = float(self.parameter_set[3][i])
                    ModParameter[j] = h


                    ## determine model function value for modified parameter
                    chisq = modelClass.DetermineModelFunction(ModParameter)


                    ## determine fprime
                    value = (float(chisq) - float(chisqNormal))/h

                    # Debug:
                    # print 'value = ', value
                    # print 'chisqNormal = ', chisqNormal
                    # print 'chisq = ', chisq
                    # print 'h = ', h


                    ## add value of fprime to gradient
                    ModelFunctionGradient[j] = float(value)


            # Debug:
            # print 'ModelFunctionGradient = ', ModelFunctionGradient
            # print 'parameter_vector = ',parameter_vector

            return ModelFunctionGradient


        ##************************************************************************************************************************************************
        def DetermineModelFunctionAll(self, parameter_vector):
            """
            determine model function value for current parameter vector
            """


            ## determine model function value for current parameter vector
            chisq = modelClass.DetermineModelFunction(parameter_vector)
            ModelFunctionGradient = modelClass.DetermineModelFunctionGradient(parameter_vector)

            return chisq, ModelFunctionGradient


        ##************************************************************************************************************************************************
        def GetBestResult(self):
            """
            get best result
            """
            return self.BestChi2, self.BestParameter, self.BestLogLine, self.BestInputFiles


        ##************************************************************************************************************************************************
        def GetBestFunctionValues(self):
            """
            get function values of best result
            """
            return self.BestParameter, self.BestFunctionValues, self.BestChi2Values


        ##************************************************************************************************************************************************
        def PrintIteration(self, parameter_vector):
            """
            plot model function together with experimental data for each iteration
            """

            self.FuncCallCounter = 0
            self.IterCounter += 1

            ## write informations to log-file
            if (printflag == "true"):
                print "%s                %5d%s\r" % (chr(13), self.IterCounter, self.BestLogLine)


            ## get channels for log-files
            paramchannel = self.parmLog
            logchannel = self.Log
            Chi2Channel = self.chi2Log


            ## wirte information to param-log file
            paramchannel.write("\n")
            paramchannel.write("\n")
            paramchannel.write("**************************************************************************************************************************\n")
            outputstring = "Iteration: %5d,  chi^2 = %25.15e" % (self.IterCounter, self.BestChi2)
            paramchannel.write(outputstring + "\n")
            paramchannel.write("\n")
            paramchannel.write("\n")
            paramchannel.write("\n")
            paramchannel.write("Parameters: " + str(self.BestParameter) + "\n")
            paramchannel.write("--------------------------------------------------------------------------------------------------------------------------\n")
            paramchannel.write("\n")
            paramchannel.flush()


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## write contents of input files to log file and create final input file(s)


            ## define path where final input files are written to
            fit_log = self.fitlog.strip()
            i = fit_log.rfind("/")
            if (i > (-1)):
                BaseDir = fit_log[:i]
            else:
                BaseDir = ""


            ## extract informations from Bestinput variable
            NameInputFile = ""
            WriteFlag = False
            for line in self.BestInputFiles:
                StrippedLine = line.strip()


                ## write contents of input file(s) to param-log file
                paramchannel.write(line)
                paramchannel.flush()


                ## get name of input file
                if (StrippedLine.startswith("Input-File ")):
                    k = StrippedLine.find(":  , file:")
                    if (k > (-1)):
                        NameInputFile = StrippedLine[k + 10:].strip()


                        ## modify input file name
                        j = NameInputFile.rfind(".")
                        if (j > (-1)):
                            NameInputFile = NameInputFile[:j] + "__AP__call_" + str(abs(self.AdditionalPackagesCounter)) + ".out.input"
                        else:
                            NameInputFile = NameInputFile + "__AP__call_" + str(abs(self.AdditionalPackagesCounter)) + ".out.input"


                        ## open channel for final input file
                        NameInputFile = BaseDir + "/" + NameInputFile
                        InputFile = open(NameInputFile, 'w')


                ## identify begin of contents of final input file(s)
                elif (StrippedLine.startswith("-start_input-file-")):
                    WriteFlag = True


                ## identify end of contents of final input file(s)
                elif (StrippedLine.startswith("-end_input-file-")):
                    try:
                        InputFile.close()
                    except AttributeError:
                        NameInputFile = ""
                    WriteFlag = False


                ## write contents of final input file(s) to current input file
                elif (WriteFlag == True and NameInputFile != ""):
                    InputFile.write(line)


            ## wirte information to log file
            outputstring = "                %5d%s" % (self.IterCounter, self.BestLogLine)
            logchannel.write(outputstring + "\n")
            logchannel.flush()
            return
    ##----------------------------------------------------------------------------------------------------------------------------------------------------


    ##====================================================================================================================================================
    ##
    ## Main-Routine
    ##
    ##====================================================================================================================================================


    ## determine calstatus
    calstatus = 0


    ## define printflag
    printflag = "true"
    if (printflagNum == 1):
        printflag = "true"
    else:
        printflag = "false"


    ## define name of algorithm
    AvailAlgorithms=["fmin", "fmin_powell", "fmin_cg", "fmin_bfgs", "fmin_ncg", "fmin_l_bfgs_b", "fmin_tnc", "anneal", "brute"];
    if (SpecialAlgorithmSettings[0] == "fmin"):
        NameOfAlgorithm = "scipy.fmin"
    elif (SpecialAlgorithmSettings[0] == "fmin_powell"):
        NameOfAlgorithm = "scipy.fmin_powell"
    elif (SpecialAlgorithmSettings[0] == "fmin_cg"):
        NameOfAlgorithm = "scipy.fmin_cg"
    elif (SpecialAlgorithmSettings[0] == "fmin_bfgs"):
        NameOfAlgorithm = "scipy.fmin_bfgs"
    elif (SpecialAlgorithmSettings[0] == "fmin_ncg"):
        NameOfAlgorithm = "scipy.fmin_ncg"
    elif (SpecialAlgorithmSettings[0] == "fmin_l_bfgs_b"):
        NameOfAlgorithm = "scipy.fmin_l_bfgs_b"
    elif (SpecialAlgorithmSettings[0] == "fmin_tnc"):
        NameOfAlgorithm = "scipy.fmin_tnc"
    elif (SpecialAlgorithmSettings[0] == "anneal"):
        NameOfAlgorithm = "scipy.anneal"
    elif (SpecialAlgorithmSettings[0] == "brute"):
        NameOfAlgorithm = "scipy.brute"
    else:
        NameOfAlgorithm = "unknown"

    if (printflag == "true"):
        print "\n         available algorithms = ",AvailAlgorithms
        print " "

    # Debug:
    # print "parameter_set[1] = ",parameter_set[1]


    ## define num_par and param_up_down
    LowUpBounds = []
    j = (-1)
    for i in range(NumberParameter):
        parameter_set[1][i] = float(parameter_set[1][i])
        if (parameter_set[1][i] == 1):
            j += 1
            LowUpBounds.append((float(parameter_set[2][i]), float(parameter_set[3][i])))
    num_par = int(sum(parameter_set[1]))                                                    ## determine num_par

    # Debug:
    # print 'LowUpBounds = ',LowUpBounds


    ## define parameters for class
    modelClass = ModelFunctionCallClass(printflagNum, GeneralAlgorithmSettings, LastAlgorithmNum, DeterminationChi2, PlotIteration, PlotType, fit_log, \
                                        NumberInputFiles, NumberOutputFiles, ParallelizationFlag, JobID, MaxInputLines, MaxParameter, RenormalizedChi2, \
                                        currentpath, CalculationMethod, xAxisLabel, yAxisLabel, \
                                        zAxisLabel, PathStartScript, ExeCommandStartScript, parameter_set, ExpDataX, ExpDataY, ExpDataError, \
                                        NumberRanges, MinRange, MaxRange, NumberXColumns, NumberYColumns, LengthExpRange, MaxRangeNumber, \
                                        NumberExpFiles, MaxLength, MaxColX, MaxColY, NumberParameter, AdditionalPackagesCounter, MPIFlag, \
                                        MAGIXrootDirectory, JobDir, SpecialAlgorithmSettings)

    ## print what you do
    if (printflag == "true"):
        print "\n         temporary files are stored in: " + str(os.environ.get('MAGIXTempDirectory','')).strip() + "job_" + str(JobID).strip() + "/"
        print "\n\n         Start %s algorithm .." % (NameOfAlgorithm)


    ## modify file names for log-files
    fit_log = fit_log.strip()
    i = fit_log.rfind("/")
    j = fit_log.rfind(".")
    if (j > i):
        fit_log = fit_log[:j] + "__AP__call_" + str(abs(AdditionalPackagesCounter)).strip() + ".log"
    else:
        fit_log = fit_log + "__AP__call_" + str(abs(AdditionalPackagesCounter)).strip() + ".log"


    ## open param-log file
    fitlogparam = fit_log.strip() + ".param"                                                ## define file name for param-log file
    paramchannel = open(fitlogparam.strip(), 'w')
    paramchannel.write("\n")
    paramchannel.write("log-file containing the actual values of the parameters used in the " + NameOfAlgorithm + " algorithm:\n")
    paramchannel.write("\n")
    paramchannel.flush()


    ## open log file and get current time and date
    fitlogparam = fit_log.strip()                                                           ## define file name for param-log file
    logchannel = open(fitlogparam.strip(), 'w')
    logchannel.write("\n")
    logchannel.write("log-file for " + NameOfAlgorithm + " algorithm:\n")
    logchannel.write("\n")
    logchannel.write("\n")

    lt = time.localtime()
    datestring = time.strftime("algorithm starts at Date: %d.%m.%Y", lt) + time.strftime(",     Time: %H:%M:%S", lt)

    logchannel.write(str(datestring) + "\n")
    logchannel.write("\n")
    logchannel.write("\n")
    logchannel.flush()


    ## open log file and get current time and date
    fitlogChi2 = fit_log.strip() + ".chi2"                                                  ## define file name for param-log file
    Chi2Channel = open(fitlogChi2.strip(), 'w')


    ## set channels for log files
    modelClass.setParameters(0, paramchannel, logchannel, Chi2Channel)


    ## lower limit of chi^2 for stopping condition normalized to the number of calculation points
    chilim = 0.E0
    if (RenormalizedChi2 == 1):
        for i in xrange(NumberExpFiles):
            chilim = chilim + (NumberYColumns[i] * LengthExpRange[i] - num_par) * abs(chilm)
        chilim = abs(chilim)
        screen_out = "           Renormalized limit for chi^2 = %25.15e" % chilim
        logchannel.write(screen_out + "\n")
        logchannel.write("\n\n")
        if (printflag == "true"):
            print "\n"
            print screen_out
            print "\n"
    else:
        chilim = abs(chilm)
        screen_out = "           Limit for chi^2 = %25.15e" % chilim
        logchannel.write(screen_out + "\n")
        logchannel.write("\n\n")
        if (printflag == "true"):
            print "\n"
            print screen_out
            print "\n"


    ## print what you do
    if (printflag == "true"):
        print "           Iteration:                    chi^2:     Parameter:"
        logchannel.write("           Iteration:                    chi^2:     Parameter:\n")
    logchannel.flush()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create list with optimized parameters
    parameterList = []
    for i in xrange(NumberParameter):
        if (parameter_set[1][i] == 1):
            parameterList.append(parameter_set[0][i])


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ##
    ## Nelder-Mead Simplex algorithm (fmin)
    ##
    ## The simplex algorithm is probably the simplest way to minimize a fairly well-behaved function. The simplex algorithm requires only function
    ## evaluations and is a good choice for simple minimization problems. However, because it does not use any gradient evaluations, it may take
    ## longer to find the minimum.
    if (SpecialAlgorithmSettings[0] == "fmin"):
        from scipy.optimize import fmin
        xopt, fopt, iter, funcalls, warnflag, allvecs = fmin(modelClass.DetermineModelFunction, parameterList, maxiter=(numiter+1), ftol=chilim, \
                                                             callback=modelClass.PrintIteration, full_output=1, disp=1, retall=1)

        # Debug:
        # print 'xopt = ', xopt
        # print 'fopt = ', fopt
        # print 'iter = ', iter
        # print 'funcalls = ', funcalls
        # print 'warnflag = ', warnflag
        # print 'allvecs = ', allvecs


    ##
    ## modified Powell's (fmin_powell)
    ##
    ## Minimize a function using modified Powell's method. Uses a modification of Powell's method to find the minimum of a function of N variables.
    elif (SpecialAlgorithmSettings[0] == "fmin_powell"):
        from scipy.optimize import fmin_powell
        xopt, fopt, direc, iter, funcalls, warnflag, allvecs = fmin_powell(modelClass.DetermineModelFunction, parameterList, maxiter=numiter, \
                                                                           callback=modelClass.PrintIteration, full_output=1, disp=1, retall=1)

        # Debug:
        # print 'xopt = ', xopt
        # print 'fopt = ', fopt
        # print 'direc = ', direc
        # print 'iter = ', iter
        # print 'funcalls = ', funcalls
        # print 'warnflag = ', warnflag
        # print 'allvecs = ', allvecs


    ##
    ## nonlinear conjugate gradient algorithm.
    ##
    ## Minimize a function using a nonlinear conjugate gradient algorithm. Optimize the function, f, whose gradient is given by fprime using
    ## the nonlinear conjugate gradient algorithm of Polak and Ribiere See Wright, and Nocedal 'Numerical Optimization', 1999, pg. 120-122.
    elif (SpecialAlgorithmSettings[0] == "fmin_cg"):
        from scipy.optimize import fmin_cg
        xopt, fopt, func_calls, grad_calls, warnflag, allvecs = fmin_cg(modelClass.DetermineModelFunction, parameterList, maxiter=numiter, \
                                                                        callback=modelClass.PrintIteration, full_output=1, disp=1, retall=1)

        # Debug:
        # print 'xopt = ', xopt
        # print 'fopt = ', fopt
        # print 'func_calls = ', func_calls
        # print 'grad_calls = ', grad_calls
        # print 'warnflag = ', warnflag
        # print 'allvecs = ', allvecs


    ##
    ## BFGS algorithm.
    ##
    ## Minimize a function using the BFGS algorithm. Optimize the function, f, whose gradient is given by fprime using the quasi-Newton method
    ## of Broyden, Fletcher, Goldfarb, and Shanno (BFGS) See Wright, and Nocedal 'Numerical Optimization', 1999, pg. 198.
    elif (SpecialAlgorithmSettings[0] == "fmin_bfgs"):
        from scipy.optimize import fmin_bfgs
        xopt, fopt, gopt, Bopt, func_calls, grad_calls, warnflag, allvecs = fmin_bfgs(modelClass.DetermineModelFunction, parameterList, maxiter=numiter, \
                                                                                      callback=modelClass.PrintIteration, full_output=1, disp=1, retall=1)

        # Debug:
        # print 'xopt = ', xopt
        # print 'fopt = ', fopt
        # print 'gopt = ', gopt
        # print 'Bopt = ', Bopt
        # print 'func_calls = ', func_calls
        # print 'grad_calls = ', grad_calls
        # print 'warnflag = ', warnflag
        # print 'allvecs = ', allvecs


    ##
    ## Newton-CG method
    ##
    ## Minimize a function using the Newton-CG method. Only one of fhess_p or fhess need to be given. If fhess is provided, then fhess_p will be
    ## ignored. If neither fhess nor fhess_p is provided, then the hessian product will be approximated using finite differences on fprime.
    ## fhess_p must compute the hessian times an arbitrary vector. If it is not given, finite-differences on fprime are used to compute it.
    ## See Wright & Nocedal, 'Numerical Optimization', 1999, pg. 140.
    elif (SpecialAlgorithmSettings[0] == "fmin_ncg"):
        from scipy.optimize import fmin_ncg
        xopt, fopt, fcalls, gcalls, hcalls, warnflag, allvecs = fmin_ncg(modelClass.DetermineModelFunction, parameterList, \
                                                                         modelClass.DetermineModelFunctionGradient, maxiter=numiter, epsilon=1.e-03, \
                                                                         callback=modelClass.PrintIteration, full_output=1, disp=1, retall=1)

        # Debug:
        # print 'xopt = ', xopt
        # print 'fopt = ', fopt
        # print 'fcalls = ', fcalls
        # print 'gcalls = ', gcalls
        # print 'hcalls = ', hcalls
        # print 'warnflag = ', warnflag
        # print 'allvecs = ', allvecs


    ##
    ## L-BFGS-B algorithm.
    ##
    ## Minimize a function func using the L-BFGS-B algorithm. License of L-BFGS-B (Fortran code):
    ## The version included here (in fortran code) is 2.1 (released in 1997). It was written by Ciyou Zhu, Richard Byrd,
    ## and Jorge Nocedal <nocedal@ece.nwu.edu>. It carries the following condition for use:
    ## This software is freely available, but we expect that all publications describing work using this software , or all commercial products using
    ## it, quote at least one of the references given below.
    elif (SpecialAlgorithmSettings[0] == "fmin_l_bfgs_b"):
        from scipy.optimize import fmin_l_bfgs_b
        modelClass.setParameters(1, paramchannel, logchannel, Chi2Channel)
        x, f, d = fmin_l_bfgs_b(modelClass.DetermineModelFunctionAll, parameterList, bounds=LowUpBounds, maxfun=numiter)

        # Debug:
        # print 'x = ', x
        # print 'f = ', f
        # print 'd = ', d


    ##
    ## tnc algorithm.
    ##
    ## Minimize a function with variables subject to bounds, using gradient information.
    elif (SpecialAlgorithmSettings[0] == "fmin_tnc"):
        from scipy.optimize import fmin_tnc
        modelClass.setParameters(1, paramchannel, logchannel, Chi2Channel)
        x, nfeval, rc = fmin_tnc(modelClass.DetermineModelFunctionAll, parameterList, bounds=LowUpBounds, maxfun=numiter)

        # Debug:
        # print 'x = ', x
        # print 'nfeval = ', nfeval
        # print 'rc = ', rc


    ##
    ## simulated annealing.
    ##
    ## Minimize a function using simulated annealing. Schedule is a schedule class implementing the annealing schedule. Available ones are 'fast',
    ## 'cauchy', 'boltzmann'
    elif (SpecialAlgorithmSettings[0] == "anneal"):
        from scipy.optimize import anneal
        modelClass.setParameters(1, paramchannel, logchannel, Chi2Channel)
        xmin, retval, Jmin, T, feval, iters, accept = anneal(modelClass.DetermineModelFunction, parameterList, full_output=1, maxiter=numiter)

        # Debug:
        # print 'xmin = ', xmin
        # print 'retval = ', retval
        # print 'Jmin = ', Jmin
        # print 'T = ', T
        # print 'feval = ', feval
        # print 'iters = ', iters
        # print 'accept = ', accept


    ##
    ## brute force.
    ##
    ## Minimize a function over a given range by brute force. Find the minimum of a function evaluated on a grid given by the tuple ranges.
    elif (SpecialAlgorithmSettings[0] == "brute"):
        from scipy.optimize import brute
        modelClass.setParameters(1, paramchannel, logchannel, Chi2Channel)
        x0, fval, grid, Jout = brute(modelClass.DetermineModelFunction, LowUpBounds, finish=chilim, full_output=1)

        # Debug:
        # print 'x0 = ', x0
        # print 'fval = ', fval
        # print 'grid = ', grid
        # print 'Jout = ', Jout


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get best function parameters
    FinalParameter, FitFunctionValues, Chi2Values = modelClass.GetBestFunctionValues()  ## get values of model function


    ## define output parameter set
    k = (-1)
    for i in xrange(NumberParameter):
        if (parameter_set[1][i] == 1):
            k += 1
            parameter_set[0][i] = FinalParameter[k]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## close param-log file
    Chi2Channel.close()
    paramchannel.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## sort log.chi^2 file
    Chi2Channel = open(fitlogChi2.strip(), 'r')
    contens = Chi2Channel.readlines()                                                       ## read in contents of input file
    Chi2Contents = []
    for inputline in contens:                                                               ## loop over all lines of the input file
        line = inputline.split()
        Currentline = []
        count = 0
        for elements in line:
            count += 1
            if (count == 1):
                elements = float(elements)
            Currentline.append(elements)

        Chi2Contents.append(Currentline)
    Chi2Channel.close()


    ## sort belonging to chi^2 values
    sortchi2 = sorted(Chi2Contents)

    # Debug:
    # print Chi2Contents[0]
    # print sortchi2[0]


    ## write sorted chi^2 parameter vectors
    Chi2Channel = open(fitlogChi2.strip(), 'w')
    Chi2Channel.write("    Nr.:                     chi**2:         parameters:\n")
    lastline = ""
    countLine = 0
    for inputline in sortchi2:
        count = 0
        formatline = ""
        for elements in inputline:
            count += 1
            if (count == 1):
                formatline = '%27.15e        ' % float(elements)
            else:
                k = len(elements)
                if (k < 20):
                    formatelements = ' %20s ' % elements
                else:
                    formatelements = " " + elements + " "
                formatline += formatelements
        if (lastline != formatline):
            lastline = formatline
            countLine += 1
            countLineString = '%8d' % countLine
            Chi2Channel.write(countLineString + " " + str(formatline) + "\n")
    Chi2Channel.close()


    ## close log file
    logchannel.write("\n")
    lt = time.localtime()
    datestring = time.strftime("algorithm ends at Date: %d.%m.%Y", lt) + time.strftime(",     Time: %H:%M:%S", lt)
    logchannel.write(str(datestring) + "\n")
    logchannel.write("\n")
    logchannel.write("-----------------------------------------------------------------------------------------------------------------------------------------------------\n")
    logchannel.flush()
    logchannel.close()


    ## print what you do
    if (printflag == "true"):
        print " "


    ## define return variables for interval-nested-sampling algorithm
    return (calstatus, FitFunctionValues, Chi2Values, parameter_set)



