!*********************************************************************************************************************************************************
!> Module: OpticalProperties
!>
!>
!>  This module contains the subroutines for calculating the epsilon tensor using the generalized Drude-Lorentz model
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - function g:                               calculate g function defined by A.B. Kuzmenko, Rev. of Scien. Instr. 76, 083108 (2005), (eq. 13)
!>      - subroutine vde:                           calculate the real and imaginary part of the epsilon tensor elements
!>      - subroutine reflectance:                   calculate the reflectance within the ac-plane
!>      - subroutine InverseMatrix:                 calculate the inverse of a 2x2 complex matrix
!>
!>
!> Copyright (C) 2009 - 2013
!>
!> I. Physikalisches Institut, Universitaet zu Koeln
!>
!> Produced for the CATS project and MnWO4 Paper
!>
!>
!> fortran module containing the subroutine for "generalized Drude-Lorentz model"
!>
!> Who           When        What
!>
!> T. Moeller    07/07/2009  Initial version
!>
!*********************************************************************************************************************************************************
Module OpticalProperties

    implicit none
    integer :: NumberXValues                                                                !< number exp. data points
    integer :: number_osc                                                                   !< number of osc. for the generalized Drude-Lorentz model
    integer :: number_PolAngle                                                              !< number of polarization angles
    real*8 :: pi                                                                            !< pi
    real*8 :: eps_inf_xx                                                                    !< epsilon_\infty for epsilon_xx matrix element
    real*8 :: eps_inf_xz                                                                    !< epsilon_\infty for epsilon_xz matrix element
    real*8 :: eps_inf_zz                                                                    !< epsilon_\infty for epsilon_zz matrix element
    real*8, allocatable, dimension(:) :: omega                                              !< frequency points
    real*8, allocatable, dimension(:) :: PolarizationAngles                                 !< polarization angles
    real*8, allocatable, dimension(:) :: w0, wp, Gamma, theta                               !< eigenfrequencies, plasma frequencies, damping, and angle
                                                                                            !< for the generalized Drude-Lorentz model
    real*8, allocatable, dimension(:, :) :: Aj                                              !< oscillator strengths for var-dielectric function


    contains


        !>************************************************************************************************************************************************
        !> subroutine: g
        !>
        !> calculate g function defined by A.B. Kuzmenko, Rev. of Scien. Instr. 76, 083108 (2005), (eq. 13)
        !>
        !>
        !> input variables:     x:                  i
        !>
        !>                      y:                  i
        !>
        !> output variables:    g:                  calculated value for g
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        function g(x, y)

            implicit none
            real*8 :: x, y                                                                  !<
            real*8 :: g                                                                     !<

            if (dabs(x - y) < 1.d-12) then
                g = (x + y) * dlog(dabs(x + y))
            else
                g = (x + y) * dlog(dabs(x + y)) + (x - y) * dlog(dabs(x - y))
            endif
            return
        end function g


        !>************************************************************************************************************************************************
        !> subroutine: vde
        !>
        !> calculate the real and imaginary part of the epsilon tensor elements 
        !>
        !>
        !> input variables:     w_index:            index for frequency
        !>
        !>                      eps_index:          index for epsilon function
        !>
        !> output variables:    eps:                calculated epsilon
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine vde(w_index, eps_index, eps)

            implicit none
            integer :: i                                                                    !< loop variable
            integer :: w_index                                                              !< index of frequency
            integer :: eps_index                                                            !< index for epsilon function
            real*8 :: w, w_i, w_im1, w_ip1                                                  !< parameters for frequency points
            real*8 :: eps1, eps2                                                            !< real and imaginary part of epsilon function
            complex(8) :: eps                                                               !< final epsilon function


            Do i = 2, (NumberXValues - 1)
                w = omega(w_index)
                w_i = omega(i)
                w_im1 = omega(i - 1)
                w_ip1 = omega(i + 1)


                !< define epsilon2
                if (w_im1 < w .and. w <= w_i) then
                    eps2 = (w - w_im1) / (w_i - w_im1)
                elseif (w_i < w .and. w < w_ip1) then
                    eps2 = (w_ip1 - w) / (w_ip1 - w_i)
                else
                    eps2 = 0.d0
                endif


                !< define epsilon1
                eps1 = (1.d0/pi) * ( (g(w, w_im1)) / (w_i - w_im1) - ((w_ip1 - w_im1) * g(w, w_i)) / ((w_i - w_im1) * (w_ip1 - w_i)) &
                                   + (g(w, w_ip1)) / (w_ip1 - w_i))


                !< construct final epsilon
                eps = eps + Aj(eps_index, i) * dcmplx(eps1, eps2)
            end Do
            return
        end subroutine vde


        !>************************************************************************************************************************************************
        !> subroutine: reflectance
        !>
        !> calculate the reflectance within the ac-plane
        !>
        !>
        !> input variables:     w_index:            index for frequency
        !>
        !>                      angle_index:        index for polarization angle
        !>
        !>                      w:                  frequency
        !>
        !>                      angle:              polarization angle
        !>
        !> output variables:    value:              calculated reflectance
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine reflectance(w_index, angle_index, w, angle, value)

            implicit none
            integer :: w_index                                                              !< index of frequency
            integer :: angle_index                                                          !< index of polarization angle
            integer :: i, k                                                                 !< loop variables
            real*8 :: w                                                                     !< frequency
            real*8 :: value                                                                 !< calculated reflectance
            real*8 :: angle                                                                 !< polarization angle
            real*8 :: theta_rad                                                             !< theta is in radians
            real*8, dimension(2, 2) :: Unit_matrix                                          !< 2x2 unit matrix
            complex(8) :: eps11, eps12, eps21, eps22                                        !< working variables: components of the epsilon tensor
            complex(8) :: DrudeLorentz                                                      !< working variables: argument of the sum for one oscillator
            complex(8), dimension(2) :: dir_vector, help_vector                             !< working variables: used for calculation of reflectance
            complex(8), dimension(2) :: ev1, ev2                                            !< eigenvectors
            complex(8), dimension(2, 2) :: epsilon_tensor                                   !< epsilon tensor
            complex(8), dimension(2, 2) :: Inverse_epsilon_tensor                           !< inverse of epsilon tensor
            complex(8), dimension(2, 2) :: Eigenvectors_epsilon_tensor                      !< eigenvector matrix
            complex(8), dimension(2, 2) :: diagonal_matrix                                  !< diagonal (rotated) form of the epsilon tensor
            complex(8), dimension(2, 2) :: sqrt_diagonal_matrix                             !< square root of diagonal form
            complex(8), dimension(2, 2) :: new_diagonal_matrix                              !< working variable: new diagonal form of epsilon tensor
            complex(8), dimension(2, 2) :: upper_refl_matrix                                !< working variable: first matrix for Fresnel formula
            complex(8), dimension(2, 2) :: lower_refl_matrix                                !< working variable: second matrix for Fresnel formula
            complex(8), dimension(2, 2) :: Inverse_lower_refl_matrix                        !< working variable: inverse of second matrix


            !< reset output variable
            value = 0.d0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate matrix elements of the epsilon tensor
            eps11 = eps_inf_xx
            eps12 = eps_inf_xz
            eps22 = eps_inf_zz
            k = 4
            Do i = 1, number_osc                                                            !< loop over all oscillators
                DrudeLorentz = ((wp(i)**2) / (w0(i)**2 - w**2 - dcmplx(0.d0,1.d0) * Gamma(i) * w))
                theta_rad = theta(i) * (pi/180.d0)                                          !< convert degree to rad
                eps11 = eps11 + DrudeLorentz * (dcos(theta_rad)**2)
                eps12 = eps12 + DrudeLorentz * (dsin(theta_rad) * dcos(theta_rad))
                eps22 = eps22 + DrudeLorentz * (dsin(theta_rad)**2)
                k = k + 4
            end Do


            !< get contribution var. dielectric function
            call vde(w_index, 1, eps11)
            call vde(w_index, 2, eps12)
            call vde(w_index, 3, eps22)
            eps21 = eps12


            !< write epsilon tensor to file
            if (angle_index == 1) then
                write(80,*) w, dreal(eps11)
                write(81,*) w, dimag(eps11)
                write(82,*) w, dreal(eps12)
                write(83,*) w, dimag(eps12)
                write(84,*) w, dreal(eps22)
                write(85,*) w, dimag(eps22)
            endif


            !< define epsilon tensor
            epsilon_tensor = dcmplx(0.d0, 0.d0)
            epsilon_tensor(1, 1) = eps11
            epsilon_tensor(1, 2) = eps12
            epsilon_tensor(2, 1) = eps21
            epsilon_tensor(2, 2) = eps22

            ! Debug:
            !   print*,'frequency = ',w
            !   print*,epsilon_tensor(1, 1), epsilon_tensor(1, 2)
            !   print*,epsilon_tensor(2, 1), epsilon_tensor(2, 2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine eigenvectors of epsilon tensor (expression taken from Mathematica)


            !< EV1
            ev1(1) = -((-eps11 + eps22 + cdsqrt(eps11**2 + 4 * eps12**2 - 2 * eps11 * eps22 + eps22**2))/(2*eps12))
            ev1(2) = 1.d0


            !< EV2
            ev2(1) = -((-eps11 + eps22 - cdsqrt(eps11**2 + 4 * eps12**2 - 2 * eps11 * eps22 + eps22**2))/(2*eps12))
            ev2(2) = 1.d0


            !< construct eigenvector matrix
            Eigenvectors_epsilon_tensor = dcmplx(0.d0, 0.d0)
            Eigenvectors_epsilon_tensor(1, 1) = ev2(1)
            Eigenvectors_epsilon_tensor(2, 1) = ev2(2)
            Eigenvectors_epsilon_tensor(1, 2) = ev1(1)
            Eigenvectors_epsilon_tensor(2, 2) = ev1(2)

            ! Debug:
            !	print*,Eigenvectors_epsilon_tensor(1, 1), Eigenvectors_epsilon_tensor(1, 2)
            !	print*,Eigenvectors_epsilon_tensor(2, 1), Eigenvectors_epsilon_tensor(2, 2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine inverse of eigenvector matrix
            Inverse_epsilon_tensor = dcmplx(0.d0, 0.d0)
            call InverseMatrix(Eigenvectors_epsilon_tensor, Inverse_epsilon_tensor)

            ! Debug:
            !	print*,Inverse_epsilon_tensor(1, 1), Inverse_epsilon_tensor(1, 2)
            !	print*,Inverse_epsilon_tensor(2, 1), Inverse_epsilon_tensor(2, 2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< bring epsilon tensor on diagonal form
            diagonal_matrix = dcmplx(0.d0, 0.d0)
            diagonal_matrix = matmul(matmul(Inverse_epsilon_tensor, epsilon_tensor), Eigenvectors_epsilon_tensor)

            ! Debug:
            !	print*,diagonal_matrix(1, 1), diagonal_matrix(1, 2)
            !	print*,diagonal_matrix(2, 1), diagonal_matrix(2, 2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< take square root of diagonal elements
            sqrt_diagonal_matrix = dcmplx(0.d0, 0.d0)
            sqrt_diagonal_matrix = diagonal_matrix
            sqrt_diagonal_matrix(1, 1) = cdsqrt(sqrt_diagonal_matrix(1, 1))
            sqrt_diagonal_matrix(2, 2) = cdsqrt(sqrt_diagonal_matrix(2, 2))

            ! Debug:
            !	print*,sqrt_diagonal_matrix(1, 1), sqrt_diagonal_matrix(1, 2)
            !	print*,sqrt_diagonal_matrix(2, 1), sqrt_diagonal_matrix(2, 2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< rotate matrix back
            new_diagonal_matrix = dcmplx(0.d0, 0.d0)
            new_diagonal_matrix = matmul(matmul(Eigenvectors_epsilon_tensor, sqrt_diagonal_matrix), Inverse_epsilon_tensor)

            ! Debug:
            !	print*,new_diagonal_matrix(1, 1), new_diagonal_matrix(1, 2)
            !	print*,new_diagonal_matrix(2, 1), new_diagonal_matrix(2, 2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine upper and lower part of the reflectance matrix


            !< define unity matrix
            Unit_matrix = 0.d0
            Unit_matrix(1, 1) = 1.d0
            Unit_matrix(2, 2) = 1.d0


            !< define upper matrix
            upper_refl_matrix = dcmplx(0.d0, 0.d0)
            upper_refl_matrix = Unit_matrix - new_diagonal_matrix


            !< define lower matrix
            lower_refl_matrix = dcmplx(0.d0, 0.d0)
            lower_refl_matrix = Unit_matrix + new_diagonal_matrix


            !< determine inverse of lower_refl_matrix
            Inverse_lower_refl_matrix = dcmplx(0.d0, 0.d0)
            call InverseMatrix(lower_refl_matrix, Inverse_lower_refl_matrix)

            ! Debug:
            !	print*,upper_refl_matrix(1,1),upper_refl_matrix(1,2)
            !	print*,upper_refl_matrix(2,1),upper_refl_matrix(2,2)

            !	print*,lower_refl_matrix(1,1),lower_refl_matrix(1,2)
            !	print*,lower_refl_matrix(2,1),lower_refl_matrix(2,2)

            !	print*,Inverse_lower_refl_matrix(1,1),Inverse_lower_refl_matrix(1,2)
            !	print*,Inverse_lower_refl_matrix(2,1),Inverse_lower_refl_matrix(2,2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< calulate reflectance
            dir_vector = (/dcmplx(dcos(angle * (pi/180.d0)),0.d0),dcmplx(dsin(angle * (pi/180.d0)),0.d0)/)
            help_vector = matmul(matmul(upper_refl_matrix, Inverse_lower_refl_matrix), dir_vector)
            value = dreal(dot_product(help_vector, help_vector))
            return
        end subroutine reflectance


        !>************************************************************************************************************************************************
        !> subroutine: InverseMatrix
        !>
        !> calculate the inverse of a 2x2 symmetric complex matrix using analytic functions determined by Mathematica
        !>
        !>
        !> input variables:     matrix:             matrix which has to inverted
        !>
        !> output variables:    Inverse_tensor:     inverted matrix
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine InverseMatrix(matrix, Inverse_tensor)

            implicit none
            complex(8) :: eps11, eps12, eps21, eps22                                        !< working variables: matrix elements
            complex(8), dimension(2, 2) :: matrix                                           !< input variable: matrix
            complex(8), dimension(2, 2) :: Inverse_tensor                                   !< output variable: inverse of matrix

            eps11 = matrix(1, 1)
            eps12 = matrix(1, 2)
            eps21 = matrix(2, 1)
            eps22 = matrix(2, 2)

            Inverse_tensor = dcmplx(0.d0, 0.d0)
            Inverse_tensor(1, 1) =   eps22/(-eps12 * eps21 + eps11 * eps22)
            Inverse_tensor(1, 2) = -(eps12/(-eps12 * eps21 + eps11 * eps22))
            Inverse_tensor(2, 1) = -(eps21/(-eps12 * eps21 + eps11 * eps22))
            Inverse_tensor(2, 2) =   eps11/(-eps12 * eps21 + eps11 * eps22)
            return
        end subroutine InverseMatrix
end Module OpticalProperties
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!>
!>
!> Main Program
!>
!>
program DrudeLorentzGeneral
    !< calculation of the refelctivity using the generalized Drude-Lorentz model

    use OpticalProperties

    implicit none
    integer :: i, j                                                                         !< loop variables
    integer :: alloc_status, dealloc_status                                                 !< status variables for (de-)allocation
    integer :: dummy                                                                        !< dummy variable
    real*8 :: w                                                                             !< working variable: current frequency w
    real*8 :: angle_chi                                                                     !< working variable: current polarization angle
    real*8, allocatable, dimension(:) :: refl                                               !< working variable: final reflectance for frequency w


    pi = 4.d0 * datan(1.d0)                                                                 !< pi


    !< open files
    open(21,file = "in.txt")                                                                !< open parameter file
    open(22,file = "DataIn.dat")                                                            !< open experimental-point file
    open(23,file = "FitFunctionValues.dat")                                                 !< open file for fit function values


    !< read parameter from file
    read(21,*) NumberXValues                                                                !< read number of exp. data points
    read(21,*) number_PolAngle                                                              !< read number of polarization angles


    ! deallocate/allocate memory
    if (allocated(PolarizationAngles)) then
        deallocate(PolarizationAngles, Aj, refl, omega, stat = dealloc_status)
        if (dealloc_status /= 0) then
            print '(" ")'
            print '(2x,"Error in module DrudeLorentzGeneral:")'
            print '(4x,"Cannot deallocate variables omega, PolarizationAngles, refl and Aj.")'
            print '(" ")'
            stop '  Program aborted!'
        endif
    endif
    allocate(omega(NumberXValues), PolarizationAngles(number_PolAngle), Aj(number_PolAngle, NumberXValues), refl(number_PolAngle), stat = alloc_status)
    if (alloc_status /= 0) then
        print '(" ")'
        print '(2x,"Error in module DrudeLorentzGeneral:")'
        print '(4x,"Cannot allocate variables omega, PolarizationAngles, refl and Aj.")'
        print '(" ")'
        stop '  Program aborted!'
    endif
    omega = 0.d0
    PolarizationAngles = 0.d0
    Aj = 0.d0
    refl = 0.d0


    !< read in polarization angles
    Do i = 1, number_PolAngle
        read(21,*) PolarizationAngles(i)
    end Do


    !< read values for epsilon^\infty and number of oscillators
    read(21,*) eps_inf_xx                                                                   !< read epsilon_xx
    read(21,*) eps_inf_xz                                                                   !< read epsilon_xz
    read(21,*) eps_inf_zz                                                                   !< read epsilon_zz
    read(21,*) number_osc                                                                   !< read number of oscillators


    ! deallocate/allocate memory
    if (allocated(w0)) then
        deallocate(w0, wp, Gamma, theta, stat = dealloc_status)
        if (dealloc_status /= 0) then
            print '(" ")'
            print '(2x,"Error in module DrudeLorentzGeneral:")'
            print '(4x,"Cannot deallocate variable w0, wp, Gamma, theta. ")'
            print '(" ")'
            stop '  Program aborted!'
        endif
    endif
    allocate(w0(number_osc), wp(number_osc), Gamma(number_osc), theta(number_osc), stat = alloc_status)
    if (alloc_status /= 0) then
        print '(" ")'
        print '(2x,"Error in module DrudeLorentzGeneral:")'
        print '(4x,"Cannot allocate variable w0, wp, Gamma, theta. ")'
        print '(" ")'
        stop '  Program aborted!'
    endif
    w0 = 0.d0
    wp = 0.d0
    Gamma = 0.d0
    theta = 0.d0


    !< read parameters for each oscillators from file
    Do i = 1, number_osc                                                                    !< loop over all oscillators
        read(21,*) w0(i), wp(i), Gamma(i), theta(i)
    end Do


    !< read in parameters for var-dielectric function and frequencies from file
    read(21,*) dummy
    Do j = 1, NumberXValues
        read(22,*) omega(j)                                                                 !< read frequency form file
        Do i = 1, 3
            read(21,*) Aj(i, j)
        end Do
    end Do
    close(21)                                                                               !< close and delete parameter file

    ! Debug:
    !    print*,'DrudeLorentzGeneral:'
    !    print*,'NumberXValues = ', NumberXValues
    !    print*,'number_PolAngle = ', number_PolAngle
    !    print*,'PolarizationAngles = ', PolarizationAngles
    !    print*,'eps_inf_xx = ',eps_inf_xx
    !    print*,'eps_inf_xz = ',eps_inf_xz
    !    print*,'eps_inf_zz = ',eps_inf_zz
    !    print*,'number_osc = ',number_osc
    !    print*,'w0(1:number_osc) = ',w0(1:number_osc)
    !    print*,'wp(1:number_osc) = ',wp(1:number_osc)
    !    print*,'Gamma(1:number_osc) = ',Gamma(1:number_osc)
    !    print*,'theta(1:number_osc) = ',theta(1:number_osc)
    !    print*,' '


    !< open output files for final epsilon component functions
    open(80, file = "epsilon-tensor__Re-eps11.dat")
    open(81, file = "epsilon-tensor__Im-eps11.dat")
    open(82, file = "epsilon-tensor__Re-eps12.dat")
    open(83, file = "epsilon-tensor__Im-eps12.dat")
    open(84, file = "epsilon-tensor__Re-eps22.dat")
    open(85, file = "epsilon-tensor__Im-eps22.dat")
    Do i = 1, NumberXValues                                                                 !< loop over all frequency points
        refl = 0.d0
        Do j = 1, number_PolAngle                                                           !< loop over all polarization angles
            angle_chi = PolarizationAngles(j)
            w = omega(i)


            !< call subroutine for determine reflectance
            call reflectance(i, j, w, angle_chi, refl(j))                                   !< calculate reflectance
        end Do


        !< write fit function value to file
        write(23,*) refl(:)                                                                 !< write reflection to output file
    end Do


    !< close files
    close(22)                                                                               !< close and delete experimental-point file
    close(23)                                                                               !< close output file
    close(80)
    close(81)
    close(82)
    close(83)
    close(84)
    close(85)
end program DrudeLorentzGeneral
!---------------------------------------------------------------------------------------------------------------------------------------------------------

