!*********************************************************************************************************************************************************
!> Module: OpticalProperties
!>
!>
!>  This module contains the subroutines for calculating the epsilon tensor using the generalized Drude-Lorentz model
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine reflectance:                   calculate the reflectance within the ac-plane
!>      - subroutine CalcEigewnvectors:             calculate the eigenvectors of a sym. 2x2 complex matrix using analytic func. determined by Mathematica
!>      - subroutine DetermineRotationAngles:       calculate the rotation angles for real and imaginary part, determine the corresponding diagonal
!>                                                  matrix entries and calculate R_p using Koch formula
!>      - subroutine InverseMatrix:                 calculate the inverse of a 2x2 complex matrix
!>
!>
!> Copyright (C) 2009 - 2013
!>
!> I. Physikalisches Institut, Universitaet zu Koeln
!>
!> Produced for the CATS project and MnWO4 Paper
!>
!>
!> fortran module containing the subroutine for "generalized Drude-Lorentz model"
!>
!> Who           When        What
!>
!> T. Moeller    07/07/2009  Initial version
!> T. Moeller    04.11.2013  write epsilon tensor components, rotation angles and diagonal components
!> T. Moeller    09.11.2013  calculate R_p
!>
!*********************************************************************************************************************************************************
Module OpticalProperties

    implicit none
    integer :: number_osc                                                                   !< number of osc. for the generalized Drude-Lorentz model
    integer :: freq_counter                                                                 !< counter for frequencies
    integer :: switchflag                                                                   !< working variable: used for consistency
    real*8 :: eps_inf_xx                                                                    !< epsilon_\infty for epsilon_xx matrix element
    real*8 :: eps_inf_xz                                                                    !< epsilon_\infty for epsilon_xz matrix element
    real*8 :: eps_inf_zz                                                                    !< epsilon_\infty for epsilon_zz matrix element
    real*8 :: pi                                                                            !< pi
    real*8 :: LastReRotAngle, LastImRotAngle                                                !< rotation angles for the previous frequencies
    real*8, allocatable, dimension(:) :: sqrt_func                                          !< working variable: used for consistency
    real*8, allocatable, dimension(:) :: w0, wp, G, theta                                   !< eigenfrequencies, plasma frequencies, damping, and angle
                                                                                            !< for the generalized Drude-Lorentz model
    real*8 :: w2, w12, gamma2                                                               !< parameter for frequency dependent damping
    logical :: first_pol_angle_flag                                                         !< indicating first polarization angle
    ! real*8 :: Lorentzian_w0, Lorentzian_A, Lorentzian_gamma, Lorentzian_const


    contains


        !>************************************************************************************************************************************************
        !> subroutine: reflectance
        !>
        !> calculate the reflectance within the ac-plane
        !>
        !>
        !> input variables:     w:                  frequency
        !>
        !>                      angle:              polarization angle
        !>
        !> output variables:    value:              calculated reflectance
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine reflectance(w, angle, value)

            implicit none
            integer :: i, k                                                                 !< loop variables
            real*8 :: w                                                                     !< input variable: frequency
            real*8 :: angle                                                                 !< input variable: polarization angle
            real*8 :: value                                                                 !< output variable: calculated reflectance
            real*8 :: theta_rad                                                             !< theta angle in radians
            real*8 :: w1                                                                    !< working variable: eigenfrequency
            real*8 :: gamma1                                                                !< working variable: damping
            real*8, dimension(2, 2) :: Unit_matrix                                          !< unit matrix
            complex(8) :: eps11, eps12, eps21, eps22                                        !< components of the epsilon tensor
            complex(8) :: DrudeLorentz                                                      !< working variable: value of the Drude-Lorentz model
            complex(8) :: A1, A2                                                            !< working variables
            complex(8), dimension(2) :: dir_vector, help_vector                             !< working variables: help vectors
            complex(8), dimension(2) :: ev1, ev2                                            !< eigenvectors
            complex(8), dimension(2, 2) :: epsilon_tensor                                   !< epsilon tensor
            complex(8), dimension(2, 2) :: Inverse_epsilon_tensor                           !< inverse of epsilon tensor
            complex(8), dimension(2, 2) :: Eigenvectors_epsilon_tensor                      !< eigenvector matrix
            complex(8), dimension(2, 2) :: diagonal_matrix                                  !< diagonal form of the epsilon tensor
            complex(8), dimension(2, 2) :: sqrt_diagonal_matrix                             !< squart-root of diagonal form of the epsilon tensor
            complex(8), dimension(2, 2) :: new_diagonal_matrix                              !< new diagonal form of the epsilon tensor
            complex(8), dimension(2, 2) :: upper_refl_matrix                                !< upper part of the epsilon tensor
            complex(8), dimension(2, 2) :: lower_refl_matrix                                !< lower part of the epsilon tensor
            complex(8), dimension(2, 2) :: Inverse_lower_refl_matrix                        !< inverse of the lower part of the epsilon tensor


            !< reset output variable
            value = 0.d0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate matrix elements of the epsilon tensor
            eps11 = eps_inf_xx
            eps12 = eps_inf_xz
            eps22 = eps_inf_zz
            k = 4
            Do i = 1, number_osc    
                if (G(i) >= 0.d0) then                                                      !< loop over all oscillators
                    DrudeLorentz = ((wp(i)**2) / (w0(i)**2 - w**2 - dcmplx(0.d0, 1.d0) * w * G(i)))
                else

                    !< use a non-KK konsistent Ansatz
                    ! gamma = Lorentzian_const * ((Lorentzian_A**2) / ((w - Lorentzian_w0)**2 + Lorentzian_gamma**2))


                    !< apply KK konsistent ansatz (Martin)
                    ! gamma = ((Lorentzian_A**2) /  (Lorentzian_w0**2 - w**2 - dcmplx(0.d0, 1.d0) * w * Lorentzian_gamma))


                    !< apply correct derived ansatz eq. 18 (Grueninger) with additional parameters: w12, w2, gamma2
                    ! w1 = w0(i)
                    ! gamma1 = dabs(G(i))
                    ! A1 = (w1**2 + w12**2 - w**2 - dcmplx(0.d0, 1.d0) * w * gamma1)
                    ! A2 = (w2**2 + w12**2 - w**2 - dcmplx(0.d0, 1.d0) * w * gamma2)
                    ! DrudeLorentz = ((wp(i)**2) / (A1 - (w12**4) / A2))


                    !< apply modified ansatz eq. 12 (Grueninger) from paper with additional parameters: w12, w2, gamma2
                    w1 = w0(i)
                    gamma1 = dabs(G(i))
                    A1 = (w1**2 - w**2 - dcmplx(0.d0, 1.d0) * w * gamma1)
                    A2 = (w2**2 - w**2 - dcmplx(0.d0, 1.d0) * w * gamma2)
                    DrudeLorentz = ((wp(i)**2) / (A1 - (w12**4) / A2))
                endif
                theta_rad = theta(i) * (pi/180.d0)                                          !< convert degree to rad
                eps11 = eps11 + DrudeLorentz * (dcos(theta_rad)**2)
                eps12 = eps12 + DrudeLorentz * (dsin(theta_rad) * dcos(theta_rad))
                eps22 = eps22 + DrudeLorentz * (dsin(theta_rad)**2)
                k = k + 4
            end Do
            eps21 = eps12


            !< define epsilon tensor
            epsilon_tensor = dcmplx(0.d0, 0.d0)
            epsilon_tensor(1, 1) = eps11
            epsilon_tensor(1, 2) = eps12
            epsilon_tensor(2, 1) = eps21
            epsilon_tensor(2, 2) = eps22

            ! Debug:
            ! print*,'frequency = ', w
            ! print*,"epsilon_tensor(1,:) = ", epsilon_tensor(1, 1), epsilon_tensor(1, 2)
            ! print*,"epsilon_tensor(2,:) = ", epsilon_tensor(2, 1), epsilon_tensor(2, 2)


            !< write components of the epsilon tensor to output file
            if (first_pol_angle_flag) then
                write(1011, *) w, dreal(eps11)
                write(1111, *) w, dimag(eps11)
                write(1012, *) w, dreal(eps12)
                write(1112, *) w, dimag(eps12)
                write(1022, *) w, dreal(eps22)
                write(1122, *) w, dimag(eps22)
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine eigenvectors of epsilon tensor (expression taken from Mathematica)
            call CalcEigewnvectors(epsilon_tensor, ev1, ev2)


            !< construct eigenvector matrix
            Eigenvectors_epsilon_tensor = dcmplx(0.d0, 0.d0)
            Eigenvectors_epsilon_tensor(1, 1) = ev2(1)
            Eigenvectors_epsilon_tensor(2, 1) = ev2(2)
            Eigenvectors_epsilon_tensor(1, 2) = ev1(1)
            Eigenvectors_epsilon_tensor(2, 2) = ev1(2)

            ! Debug:
        	! print*,"Eigenvectors_epsilon_tensor(1,:) = ", Eigenvectors_epsilon_tensor(1, 1), Eigenvectors_epsilon_tensor(1, 2)
        	! print*,"Eigenvectors_epsilon_tensor(2,:) = ", Eigenvectors_epsilon_tensor(2, 1), Eigenvectors_epsilon_tensor(2, 2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine inverse of eigenvector matrix
            Inverse_epsilon_tensor = dcmplx(0.d0, 0.d0)
            call InverseMatrix(Eigenvectors_epsilon_tensor, Inverse_epsilon_tensor)

            ! Debug:
            !	print*,"Inverse_epsilon_tensor(1,:) = ", Inverse_epsilon_tensor(1, 1), Inverse_epsilon_tensor(1, 2)
            !	print*,"Inverse_epsilon_tensor(2,:) = ", Inverse_epsilon_tensor(2, 1), Inverse_epsilon_tensor(2, 2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< bring epsilon tensor on diagonal form
            diagonal_matrix = dcmplx(0.d0, 0.d0)
            diagonal_matrix = matmul(matmul(Inverse_epsilon_tensor, epsilon_tensor), Eigenvectors_epsilon_tensor)

            ! Debug:
        	! print*,"diagonal_matrix(1,:) = ", diagonal_matrix(1, 1), diagonal_matrix(1, 2)
        	! print*,"diagonal_matrix(2,:) = ", diagonal_matrix(2, 1), diagonal_matrix(2, 2)


            !< determine rotation angle
            if (first_pol_angle_flag) then
                call DetermineRotationAngles(w, epsilon_tensor)
            endif


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< take square root of diagonal elements
            sqrt_diagonal_matrix = dcmplx(0.d0, 0.d0)
            sqrt_diagonal_matrix = diagonal_matrix
            sqrt_diagonal_matrix(1, 1) = cdsqrt(sqrt_diagonal_matrix(1, 1))
            sqrt_diagonal_matrix(2, 2) = cdsqrt(sqrt_diagonal_matrix(2, 2))

            ! Debug:
            !	print*, sqrt_diagonal_matrix(1, 1), sqrt_diagonal_matrix(1, 2)
            !	print*, sqrt_diagonal_matrix(2, 1), sqrt_diagonal_matrix(2, 2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< rotate matrix back
            new_diagonal_matrix = dcmplx(0.d0, 0.d0)
            new_diagonal_matrix = matmul(matmul(Eigenvectors_epsilon_tensor, sqrt_diagonal_matrix), Inverse_epsilon_tensor)

            ! Debug:
            !	print*, new_diagonal_matrix(1, 1), new_diagonal_matrix(1, 2)
            !	print*, new_diagonal_matrix(2, 1), new_diagonal_matrix(2, 2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine upper and lower part of the reflectance matrix


            !< define unity matrix
            Unit_matrix = 0.d0
            Unit_matrix(1, 1) = 1.d0
            Unit_matrix(2, 2) = 1.d0


            !< define upper matrix
            upper_refl_matrix = dcmplx(0.d0, 0.d0)
            upper_refl_matrix = Unit_matrix - new_diagonal_matrix


            !< define lower matrix
            lower_refl_matrix = dcmplx(0.d0, 0.d0)
            lower_refl_matrix = Unit_matrix + new_diagonal_matrix


            !< determine inverse of lower_refl_matrix
            Inverse_lower_refl_matrix = dcmplx(0.d0, 0.d0)
            call InverseMatrix(lower_refl_matrix, Inverse_lower_refl_matrix)

            ! Debug:
            !	print*,upper_refl_matrix(1,1),upper_refl_matrix(1,2)
            !	print*,upper_refl_matrix(2,1),upper_refl_matrix(2,2)

            !	print*,lower_refl_matrix(1,1),lower_refl_matrix(1,2)
            !	print*,lower_refl_matrix(2,1),lower_refl_matrix(2,2)

            !	print*,Inverse_lower_refl_matrix(1,1),Inverse_lower_refl_matrix(1,2)
            !	print*,Inverse_lower_refl_matrix(2,1),Inverse_lower_refl_matrix(2,2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< calulate reflectance
            dir_vector = (/dcmplx(dcos(angle * (pi/180.d0)), 0.d0), dcmplx(dsin(angle * (pi/180.d0)), 0.d0)/)
            help_vector = matmul(matmul(upper_refl_matrix, Inverse_lower_refl_matrix), dir_vector)
            value = dreal(dot_product(help_vector, help_vector))
            return
        end subroutine reflectance


        !>************************************************************************************************************************************************
        !> subroutine: CalcEigewnvectors
        !>
        !> calculate the eigenvectors of a symmetric 2x2 complex matrix using analytic functions determined by Mathematica
        !>
        !>
        !> input variables:     matrix:             matrix which has to be diagonalized
        !>
        !> output variables:    ev1:                first eigenvector
        !>
        !>                      ev2:                second eigenvector
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine CalcEigewnvectors(matrix, ev1, ev2)

            implicit none
            complex(8) :: eps11, eps12, eps22                                               !< working variables: matrix elements
            complex(8), dimension(2) :: ev1, ev2                                            !< eigenvectors 1 and 2
            complex(8), dimension(2, 2) :: matrix                                           !< symmetric matrix


            !< get component of matirx
            eps11 = matrix(1, 1)
            eps12 = matrix(1, 2)
            eps22 = matrix(2, 2)


            !< EV1: Mathematica formula for non-normalized eigenvector 1 and a separat normalization
            ev1(1) = -((-eps11 + eps22 + cdsqrt(eps11**2 + 4 * eps12**2 - 2 * eps11 * eps22 + eps22**2))/(2*eps12))
            ev1(2) = 1.d0
            ev1(:) = 1.d0 / dsqrt(cdabs(dot_product(ev1(:), ev1(:)))) * ev1(:)


            !< EV1: Mathematica formula for already normalized eigenvector 1
            !ev1(1) = (eps11 - eps22 - cdsqrt(eps11**2 + dcmplx(4.d0, 0.d0) * eps12**2 - dcmplx(2.d0, 0.d0) * eps11 * eps22 + eps22**2)) &
            !         /(eps12 * cdsqrt(dcmplx(4.d0, 0.d0) + cdabs((-eps11 + eps22 + cdsqrt(eps11**2 + dcmplx(4.d0, 0.d0) * eps12**2 &
            !         - dcmplx(2.d0, 0.d0) * eps11 * eps22 + eps22**2))/eps12)**2))
            !ev1(2) = 2.d0 / cdsqrt(dcmplx(4.d0, 0.d0) + cdabs((-eps11 + eps22 + cdsqrt(eps11**2 + dcmplx(4.d0, 0.d0) * eps12**2 &
            !         - dcmplx(2.d0, 0.d0) * eps11 * eps22 + eps22**2)) / eps12)**2)


            !< EV2: Mathematica formula for non-normalized eigenvector 2 and a separat normalization
            ev2(1) = -((-eps11 + eps22 - cdsqrt(eps11**2 + 4 * eps12**2 - 2 * eps11 * eps22 + eps22**2))/(2*eps12))
            ev2(2) = 1.d0
            ev2(:) = 1 / dsqrt(cdabs(dot_product(ev2(:), ev2(:)))) * ev2(:)


            !< EV2: Mathematica formula for already normalized eigenvector 2
            !ev2(1) = (eps11 - eps22 + cdsqrt(eps11**2 + dcmplx(4.d0, 0.d0) * eps12**2 - dcmplx(2.d0, 0.d0) * eps11 * eps22 + eps22**2)) &
            !         /(eps12 * cdsqrt(dcmplx(4.d0, 0.d0) + cdabs((eps11 - eps22 + cdsqrt(eps11**2 + dcmplx(4.d0, 0.d0) * eps12**2 &
            !         - dcmplx(2.d0, 0.d0) * eps11 * eps22 + eps22**2))/eps12)**2))
            !ev2(2) = 2.d0 /cdsqrt(dcmplx(4.d0, 0.d0) + cdabs((eps11 - eps22 + cdsqrt(eps11**2 + dcmplx(4.d0, 0.d0) * eps12**2 &
            !         - dcmplx(2.d0, 0.d0) * eps11 * eps22 + eps22**2))/eps12)**2)
            return
        end subroutine CalcEigewnvectors


        !>************************************************************************************************************************************************
        !> subroutine: CalcEigewnvectorsLAPACK (NOT USED AT THE MOMENT)
        !>
        !> calculate the eigenvectors of a symmetric NxN complex matrix using LAPACK routine
        !>
        !>
        !> input variables:     sizematrix:         size of matrix
        !>
        !>                      matrix_org:         matrix which has to be diagonalized
        !>
        !> output variables:    ev1:                first eigenvector
        !>
        !>                      ev2:                second eigenvector
        !>
        !>                      ew1:                first eigenvalue
        !>
        !>                      ew2:                second eigenvalue
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine CalcEigewnvectorsLAPACK(sizematrix, matrix_org, ev1, ev2, ew1, ew2)

            implicit none
            integer :: sizematrix                                                           !< size (dimension) of matrix
            integer :: sizematrix_copy                                                      !< copy of sizematrix
            integer :: ok                                                                   !< status variable
            !real*8, dimension(2 * sizematrix) :: rwork                                      !< working array
            !complex(8) :: dummy                                                             !< dummy variable
            complex(8) :: ew1, ew2                                                          !< copy of eigenvalues
            complex(8), dimension(sizematrix) :: ev1, ev2                                   !< copy of eigenvectors
            !complex(8), dimension(2 * sizematrix) :: work                                   !< working array
            complex(8), dimension(sizematrix) :: ew                                         !< list containing eigenvalues
            complex(8), dimension(sizematrix, sizematrix) :: ev                             !< array containing the eigenvectors
            complex(8), dimension(sizematrix, sizematrix) :: matrix                         !< matrix which should be diagonalized
            complex(8), dimension(sizematrix, sizematrix) :: matrix_org                     !< copy of matrix

            matrix = matrix_org
            sizematrix_copy = sizematrix

            ew = dcmplx(0.d0, 0.d0)
            ev = dcmplx(0.d0, 0.d0)
            ! CALL zgeev('N', 'V', sizematrix_copy, matrix, sizematrix_copy, ew, dummy, 1, ev, sizematrix_copy, work, 2*sizematrix_copy, rwork, ok)
            if (ok /= 0) then
                print*,' '
                print*,' Error in subroutine CalcEigewnvectorsLAPAC:'
                print*,'  ok = ',ok
                print*,' '
                stop 'Program aborted!'
            endif
            ev1(:) = ev(:, 1)
            ev2(:) = ev(:, 2)
            ew1 = ew(1)
            ew2 = ew(2)
            return
        end subroutine CalcEigewnvectorsLAPACK


        !>************************************************************************************************************************************************
        !> subroutine: DetermineRotationAngles
        !>
        !> calculate the rotation angles for real and imaginary part, determine the corresponding diagonal matrix entries and calculate R_p using Koch
        !>
        !>
        !> input variables:     w:                  frequency
        !>
        !>                      matrix:             epsilon tensor
        !>
        !> output variables:    None
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine DetermineRotationAngles(w, matrix)

            implicit none
            real*8 :: w                                                                     !< frequency
            real*8 :: ReRotAngle                                                            !< value of current rotation angle of the real part
            real*8 :: ImRotAngle                                                            !< value of current rotation angle of the imaginary part
            real*8 :: h1                                                                    !< working variable for eigenvector check
            real*8, dimension(2, 2) :: test                                                 !< working variable for unit check of inverse matrix
            complex(8) :: ww                                                                !< working variable: used for sign of square root
            complex(8), dimension(2) :: ev1, ev2                                            !< eigenvectors for real and imaginary part
            complex(8), dimension(2, 2) :: ReMatrix, ImMatrix                               !< real and imaginary part of the epsilon tensor
            complex(8), dimension(2, 2) :: matrix                                           !< copy of epsilon tensor
            complex(8), dimension(2, 2) :: eigenvectormatrix, Inverse_eigenvectormatrix     !< eigenvector matrix and inverse of eigenvector matrix
            complex(8), dimension(2, 2) :: diagonal_matrix                                  !< diagonal (rotated) matrix for real and imaginary part


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< real part


            !< determine real part of the epsilon tensor
            ReMatrix = dcmplx(0.d0, 0.d0)
            ReMatrix(1, 1) = dcmplx(dreal(matrix(1, 1)), 0.d0)
            ReMatrix(1, 2) = dcmplx(dreal(matrix(1, 2)), 0.d0)
            ReMatrix(2, 1) = dcmplx(dreal(matrix(2, 1)), 0.d0)
            ReMatrix(2, 2) = dcmplx(dreal(matrix(2, 2)), 0.d0)


            !< determine eigenvectors
            ev1 = dcmplx(0.d0, 0.d0)
            ev2 = dcmplx(0.d0, 0.d0)
            call CalcEigewnvectors(ReMatrix, ev1, ev2)

            ! Debug:
            ! print*,"EV(Re[epsion_tensor])(1,1) = ", dreal(ev1(1))
            ! print*,"EV(Re[epsion_tensor])(1,2) = ", dreal(ev1(2))
            ! print*,"EV(Re[epsion_tensor])(2,1) = ", dreal(ev2(1))
            ! print*,"EV(Re[epsion_tensor])(2,2) = ", dreal(ev2(2))


            !< check orthonormality of eigenvectors
            h1 = dreal(dot_product(ev1, ev1))
            if (dabs(h1 - 1.d0) > 1.d-10) then
                print*,' '
                print*,' '
                print*,'Error in subroutine DetermineRotationAngles:'
                print*,'  The first calculated eigenvector of Re[epsilon] is not normalized to 1'
                print*,' '
                print*,' '
                print*,'  ev1 = ', ev1
                print*,'  ev1.ev1 = ', h1
                stop
            endif
            h1 = dreal(dot_product(ev2, ev2))
            if (dabs(h1 - 1.d0) > 1.d-10) then
                print*,' '
                print*,' '
                print*,'Error in subroutine DetermineRotationAngles:'
                print*,'  The second calculated eigenvector of Re[epsilon] is not normalized to 1'
                print*,' '
                print*,' '
                print*,'  ev2 = ', ev2
                print*,'  ev2.ev2 = ', h1
                stop
            endif
            h1 = dreal(dot_product(ev1, ev2))
            if (dabs(h1) > 1.d-10) then
                print*,' '
                print*,' '
                print*,'Error in subroutine DetermineRotationAngles:'
                print*,'  The two calculated eigenvector of Re[epsilon] are not orthogonal to each other!'
                print*,' '
                print*,' '
                print*,'  ev1 = ', ev1
                print*,'  ev2 = ', ev2
                print*,'  ev1.ev2 = ', h1
                stop
            endif


            !< check for consistency
            ww = cdsqrt(matrix(1, 1)**2 + 4 * matrix(1, 2)**2 - 2 * matrix(1, 1) * matrix(2, 2) + matrix(2, 2)**2)
            sqrt_func(freq_counter) = dimag(ww)
            if (sqrt_func(freq_counter - 1) > 0.d0 .and. sqrt_func(freq_counter) < 0.d0 &
                .or. sqrt_func(freq_counter - 1) < 0.d0 .and. sqrt_func(freq_counter) > 0.d0) then
                if (dabs(sqrt_func(freq_counter - 1) - sqrt_func(freq_counter)) > 1.d1) then
                    switchflag = 1 - switchflag

                    ! Debug:
                    ! print*,'w_jump = ', w
                endif
            endif

            ! Debug:
            ! write(12345,*) w, dreal(ww), dimag(ww)


            !< deterimine rotation angle for real part of the epsilon tensor
            if (switchflag == 1) then
                ReRotAngle = datan2(dreal(ev2(2)), dreal(ev2(1)))
            else
                ReRotAngle = datan2(dreal(ev1(2)), dreal(ev1(1)))
            endif


            !< check, if rotation angle function runs out of interval [0,pi)
            if (LastReRotAngle /= -1.d99) then

                ! Debug:
                ! print*,"dabs(ReRotAngle - LastReRotAngle) = ", dabs(ReRotAngle - LastReRotAngle)


                if (dabs(ReRotAngle - LastReRotAngle) >= (pi * 0.98d0)) then
                    ReRotAngle = ReRotAngle + pi
                endif
            endif


            !< write rotation angle to file
            write(3010, *) w, ReRotAngle * 180.d0/pi


            !< construct eigenvectormatrix
            if (switchflag == 1) then
                eigenvectormatrix(1, 1) = ev2(1)
                eigenvectormatrix(1, 2) = ev2(2)
                eigenvectormatrix(2, 1) = ev1(1)
                eigenvectormatrix(2, 2) = ev1(2)
            else
                eigenvectormatrix(1, 1) = ev1(1)
                eigenvectormatrix(1, 2) = ev1(2)
                eigenvectormatrix(2, 1) = ev2(1)
                eigenvectormatrix(2, 2) = ev2(2)
            endif


            !< calculate inverse of eigenvector matrix
            Inverse_eigenvectormatrix = dcmplx(0.d0, 0.d0)
            call InverseMatrix(eigenvectormatrix, Inverse_eigenvectormatrix)


            !< rotate real part of epsion tensor to diagonal form
            diagonal_matrix = dcmplx(0.d0, 0.d0)
            diagonal_matrix = matmul(matmul(eigenvectormatrix, ReMatrix), Inverse_eigenvectormatrix)
            if (dabs(dreal(diagonal_matrix(1, 2))) > 1.d-10 .or. dabs(dreal(diagonal_matrix(2, 1))) > 1.d-10) then
                diagonal_matrix = matmul(matmul(Inverse_eigenvectormatrix, ReMatrix), eigenvectormatrix)
            endif


            !< check off-diagonal elements
            if (dabs(dreal(diagonal_matrix(1, 2))) > 1.d-10 .or. dabs(dreal(diagonal_matrix(2, 1))) > 1.d-10) then
                test = dreal(matmul(Inverse_eigenvectormatrix, eigenvectormatrix))
                print*,' '
                print*,' '
                print*,'Error in subroutine DetermineRotationAngles:'
                print*,'  Off-diagonal elements of rotated Re[epsilon] tensor are not zero!'
                print*,' '
                print*,' '
                print*,'  w = ', w
                print*,'  ev1 = ', ev1
                print*,'  ev2 = ', ev2
                print*,' '
                print*,'  eigenvectormatrix(1, :) = ', dreal(eigenvectormatrix(1, :))
                print*,'  eigenvectormatrix(2, :) = ', dreal(eigenvectormatrix(2, :))
                print*,' '
                print*,'  Inverse_eigenvectormatrix(1, :) = ', dreal(Inverse_eigenvectormatrix(1, :))
                print*,'  Inverse_eigenvectormatrix(2, :) = ', dreal(Inverse_eigenvectormatrix(2, :))
                print*,' '
                print*,'  unity(1, :) = ', test(1, :)
                print*,'  unity(2, :) = ', test(2, :)
                print*,' '
                print*,'  diagonal_matrix(1, :) = ', dreal(diagonal_matrix(1, :))
                print*,'  diagonal_matrix(2, :) = ', dreal(diagonal_matrix(2, :))
                print*,' '
                stop
            endif


            !< write diagonal elements to file
            write(3001, *) w, dreal(diagonal_matrix(1, 1))
            write(3003, *) w, dreal(diagonal_matrix(2, 2))



            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< imaginary part


            !< determine real part of the epsilon tensor
            ImMatrix = dcmplx(0.d0, 0.d0)
            ImMatrix(1, 1) = dcmplx(dimag(matrix(1, 1)), 0.d0)
            ImMatrix(1, 2) = dcmplx(dimag(matrix(1, 2)), 0.d0)
            ImMatrix(2, 1) = dcmplx(dimag(matrix(2, 1)), 0.d0)
            ImMatrix(2, 2) = dcmplx(dimag(matrix(2, 2)), 0.d0)


            !< determine eigenvectors
            ev1 = dcmplx(0.d0, 0.d0)
            ev2 = dcmplx(0.d0, 0.d0)
            call CalcEigewnvectors(ImMatrix, ev1, ev2)

            ! Debug:
            ! print*,"EV(Im[epsion_tensor])(1,1) = ", dreal(ev1(1))
            ! print*,"EV(Im[epsion_tensor])(1,2) = ", dreal(ev1(2))
            ! print*,"EV(Im[epsion_tensor])(2,1) = ", dreal(ev2(1))
            ! print*,"EV(Im[epsion_tensor])(2,2) = ", dreal(ev2(2))


            !< check orthonormality of eigenvectors
            h1 = dreal(dot_product(ev1, ev1))
            if (dabs(h1 - 1.d0) > 1.d-10) then
                print*,' '
                print*,' '
                print*,'Error in subroutine DetermineRotationAngles:'
                print*,'  The first calculated eigenvector of Im[epsilon] is not normalized to 1'
                print*,' '
                print*,' '
                print*,'  ev1 = ', ev1
                print*,'  ev1.ev1 = ', h1
                stop
            endif
            h1 = dreal(dot_product(ev2, ev2))
            if (dabs(h1 - 1.d0) > 1.d-10) then
                print*,' '
                print*,' '
                print*,'Error in subroutine DetermineRotationAngles:'
                print*,'  The second calculated eigenvector of Im[epsilon] is not normalized to 1'
                print*,' '
                print*,' '
                print*,'  ev2 = ', ev2
                print*,'  ev2.ev2 = ', h1
                stop
            endif
            h1 = dreal(dot_product(ev1, ev2))
            if (dabs(h1) > 1.d-10) then
                print*,' '
                print*,' '
                print*,'Error in subroutine DetermineRotationAngles:'
                print*,'  The two calculated eigenvector of Im[epsilon] are not orthogonal to each other!'
                print*,' '
                print*,' '
                print*,'  ev1 = ', ev1
                print*,'  ev2 = ', ev2
                print*,'  ev1.ev2 = ', h1
                stop
            endif


            !< deterimine rotation angle for real part of the epsilon tensor
            ImRotAngle = datan2(dreal(ev2(2)), dreal(ev2(1)))
            if (LastImRotAngle /= -1.d99) then
                if (dabs(ImRotAngle - LastImRotAngle) >= (pi * 0.82d0)) then
                    ImRotAngle = ImRotAngle - pi
                endif
            endif


            !< write rotation angle to file
            write(3011, *) w, ImRotAngle * 180.d0/pi


            !< construct eigenvectormatrix
            eigenvectormatrix(1, 1) = ev1(1)
            eigenvectormatrix(1, 2) = ev1(2)
            eigenvectormatrix(2, 1) = ev2(1)
            eigenvectormatrix(2, 2) = ev2(2)


            !< calculate inverse of eigenvector matrix
            Inverse_eigenvectormatrix = dcmplx(0.d0, 0.d0)
            call InverseMatrix(eigenvectormatrix, Inverse_eigenvectormatrix)


            !< rotate real part of epsion tensor to diagonal form
            diagonal_matrix = dcmplx(0.d0, 0.d0)
            diagonal_matrix = matmul(matmul(eigenvectormatrix, ImMatrix), Inverse_eigenvectormatrix)
            if (dabs(dreal(diagonal_matrix(1, 2))) > 1.d-10 .or. dabs(dreal(diagonal_matrix(2, 1))) > 1.d-10) then
                diagonal_matrix = matmul(matmul(Inverse_eigenvectormatrix, ImMatrix), eigenvectormatrix)
            endif


            !< check off-diagonal elements
            if (dabs(dreal(diagonal_matrix(1, 2))) > 1.d-10 .or. dabs(dreal(diagonal_matrix(2, 1))) > 1.d-10) then
                print*,' '
                print*,' '
                print*,'Error in subroutine DetermineRotationAngles:'
                print*,'  Off-diagonal elements of rotated Im[epsilon] tensor are not zero!'
                print*,' '
                print*,' '
                print*,'  w = ', w
                print*,'  ev1 = ', ev1
                print*,'  ev2 = ', ev2
                print*,' '
                print*,'  eigenvectormatrix(1, :) = ', dreal(eigenvectormatrix(1, :))
                print*,'  eigenvectormatrix(2, :) = ', dreal(eigenvectormatrix(2, :))
                print*,' '
                print*,'  Inverse_eigenvectormatrix(1, :) = ', dreal(Inverse_eigenvectormatrix(1, :))
                print*,'  Inverse_eigenvectormatrix(2, :) = ', dreal(Inverse_eigenvectormatrix(2, :))
                print*,' '
                print*,'  unity(1, :) = ', test(1, :)
                print*,'  unity(2, :) = ', test(2, :)
                print*,' '
                print*,'  diagonal_matrix(1, :) = ', dreal(diagonal_matrix(1, :))
                print*,'  diagonal_matrix(2, :) = ', dreal(diagonal_matrix(2, :))
                print*,' '
                stop
            endif

            ! Debug:
        	! print*,"ReRotAngle = ", ReRotAngle
        	! print*,"ImRotAngle = ", ImRotAngle


            !< write diagonal elements
            write(3002, *) w, dreal(diagonal_matrix(1, 1))
            write(3004, *) w, dreal(diagonal_matrix(2, 2))


            !< save rotation angles
            LastReRotAngle = ReRotAngle
            LastImRotAngle = ImRotAngle
            return
        end subroutine DetermineRotationAngles


        !>************************************************************************************************************************************************
        !> subroutine: InverseMatrix
        !>
        !> calculate the inverse of a 2x2 symmetric complex matrix using analytic functions determined by Mathematica
        !>
        !>
        !> input variables:     matrix:             matrix which has to inverted
        !>
        !> output variables:    Inverse_tensor:     inverted matrix
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine InverseMatrix(matrix, Inverse_tensor)

            implicit none
            complex(8) :: eps11, eps12, eps21, eps22                                        !< working variables: matrix elements
            complex(8), dimension(2, 2) :: matrix                                           !< input variable: matrix
            complex(8), dimension(2, 2) :: Inverse_tensor                                   !< output variable: inverse of matrix

            eps11 = matrix(1, 1)
            eps12 = matrix(1, 2)
            eps21 = matrix(2, 1)
            eps22 = matrix(2, 2)

            Inverse_tensor = dcmplx(0.d0, 0.d0)
            Inverse_tensor(1, 1) =   eps22/(-eps12 * eps21 + eps11 * eps22)
            Inverse_tensor(1, 2) = -(eps12/(-eps12 * eps21 + eps11 * eps22))
            Inverse_tensor(2, 1) = -(eps21/(-eps12 * eps21 + eps11 * eps22))
            Inverse_tensor(2, 2) =   eps11/(-eps12 * eps21 + eps11 * eps22)
            return
        end subroutine InverseMatrix
end Module OpticalProperties
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!>
!>
!> Main Program
!>
!>
program DrudeLorentzGeneral
    !< calculation of the refelctivity using the generalized Drude-Lorentz model

    use OpticalProperties

    implicit none
    integer :: i                                                                            !< loop variables
    integer :: NumberXValues                                                                !< number exp. data points
    integer :: alloc_status, dealloc_status                                                 !< status variables for (de-)allocation
    real*8 :: w                                                                             !< working variable: current frequency w
    real*8 :: w_old                                                                         !< working variable: previous frequency w
    real*8 :: refl                                                                          !< working variable: final reflectance for frequency w
    real*8 :: angle_chi                                                                     !< working variable: current polarization angle


    !< define constants
    pi = 4.d0 * datan(1.d0)                                                                 !< determine pi


    !< open input files
    open(21, file = "in.txt")                                                               !< open parameter file
    open(22, file = "DataIn.dat")                                                           !< open experimental-point file


    !< open output files
    open(23, file = "FitFunctionValues.dat")                                                !< open file for fit function values
    open(1011, file = "Re-eps11.dat")                                                       !< open file for Re[eps_xx]
    open(1111, file = "Im-eps11.dat")                                                       !< open file for Im[eps_xx]
    open(1012, file = "Re-eps12.dat")                                                       !< open file for Re[eps_xz]
    open(1112, file = "Im-eps12.dat")                                                       !< open file for Im[eps_xz]
    open(1022, file = "Re-eps22.dat")                                                       !< open file for Re[eps_zz]
    open(1122, file = "Im-eps22.dat")                                                       !< open file for Im[eps_zz]
    open(3001, file = "Diag__Re-eps_xx.dat")                                                !< open file for real part of diagonal part of eps_xx
    open(3002, file = "Diag__Im-eps_xx.dat")                                                !< open file for imaginary part of diagonal part of eps_xx
    open(3003, file = "Diag__Re-eps_zz.dat")                                                !< open file for real part of diagonal part of eps_zz
    open(3004, file = "Diag__Im-eps_zz.dat")                                                !< open file for imaginary part of diagonal part of eps_zz
    open(3010, file = "Re-rotation-angle.dat")                                              !< open file for rotation angle of real part
    open(3011, file = "Im-rotation-angle.dat")                                              !< open file for rotation angle of imaginary part


    !< read parameter from file
    read(21,*) NumberXValues                                                                !< read number of exp. data points
    read(21,*) eps_inf_xx                                                                   !< read epsilon_xx
    read(21,*) eps_inf_xz                                                                   !< read epsilon_xz
    read(21,*) eps_inf_zz                                                                   !< read epsilon_zz
    read(21,*) number_osc                                                                   !< read number of oscillators


    ! deallocate/allocate memory
    if (allocated(w0)) then
        deallocate(w0, wp, G, theta, stat = dealloc_status)
        if (dealloc_status /= 0) then
            print '(" ")'
            print '(2x,"Error in module DrudeLorentzGeneral:")'
            print '(4x,"Cannot deallocate variable w0, wp, G, theta. ")'
            print '(" ")'
            stop '  Program aborted!'
        endif
    endif
    allocate(w0(number_osc), wp(number_osc), G(number_osc), theta(number_osc), sqrt_func(NumberXValues), stat = alloc_status)
    if (alloc_status /= 0) then
        print '(" ")'
        print '(2x,"Error in module DrudeLorentzGeneral:")'
        print '(4x,"Cannot allocate variable w0, wp, G, theta. ")'
        print '(" ")'
        stop '  Program aborted!'
    endif
    w0 = 0.d0
    wp = 0.d0
    G = 0.d0
    theta = 0.d0
    w2 = 0.d0
    w12 = 0.d0
    gamma2 = 0.d0
    sqrt_func = 0.d0


    !< read parameters for each oscillators from file
    Do i = 1, number_osc                                                                    !< loop over all oscillators
        read(21,*) w0(i), wp(i), G(i), theta(i)
    end Do
    read(21,*) w2, w12, gamma2


    !< read old models:
    ! Lorentzian_w0 = 0.d0
    ! Lorentzian_A = 0.d0
    ! Lorentzian_gamma = 0.d0
    ! Lorentzian_const = 0.d0
    ! read(21,*) Lorentzian_w0, Lorentzian_A, Lorentzian_gamma, Lorentzian_const
    close(21)                                                                               !< close and delete parameter file

    ! Debug:
    !    print*,'DrudeLorentzGeneral:'
    !    print*,'w = ',w
    !    print*,'angle_chi = ',angle_chi
    !    print*,'eps_inf_xx = ',eps_inf_xx
    !    print*,'eps_inf_xz = ',eps_inf_xz
    !    print*,'eps_inf_zz = ',eps_inf_zz
    !    print*,'number_osc = ',number_osc
    !    print*,'w0(1:number_osc) = ',w0(1:number_osc)
    !    print*,'wp(1:number_osc) = ',wp(1:number_osc)
    !    print*,'G(1:number_osc) = ',G(1:number_osc)
    !    print*,'theta(1:number_osc) = ',theta(1:number_osc)
    !    print*,' '


    !< calculate optical properties at any frequency w
    switchflag = 1
    LastReRotAngle = -1.d99
    LastImRotAngle = -1.d99
    first_pol_angle_flag = .true.
    w_old = 0.d0
    Do i = 1, NumberXValues                                                                 !< loop over all frequency points
        read(22,*) w, angle_chi                                                             !< read frequency and polarization angle form file
        freq_counter = i


        !< set first_pol_angle_flag
        if (w_old > w .and. first_pol_angle_flag) then
            first_pol_angle_flag = .false.
        endif
        w_old = w


        !< call subroutine for determine reflectance
        call reflectance(w, angle_chi, refl)                                                !< calculate reflectance


        !< write fit function value to file
        write(23,*) refl                                                                    !< write reflection to output file
    end Do


    !< close files
    close(22)                                                                               !< close and delete experimental-point file
    close(23)                                                                               !< close output file
    close(1011)                                                                             !< close file for Re[eps_xx]
    close(1111)                                                                             !< close file for Im[eps_xx]
    close(1012)                                                                             !< close file for Re[eps_xz]
    close(1112)                                                                             !< close file for Im[eps_xz]
    close(1022)                                                                             !< close file for Re[eps_zz]
    close(1122)                                                                             !< close file for Im[eps_zz]
    close(3001)                                                                             !< close file for real part of diagonal part of eps_xx
    close(3002)                                                                             !< close file for imaginary part of diagonal part of eps_xx
    close(3003)                                                                             !< close file for real part of diagonal part of eps_zz
    close(3004)                                                                             !< close file for imaginary part of diagonal part of eps_zz
    close(3010)                                                                             !< close file for rotation angle of real part
    close(3011)                                                                             !< close file for rotation angle of imaginary part
end program DrudeLorentzGeneral
!---------------------------------------------------------------------------------------------------------------------------------------------------------

