!*********************************************************************************************************************************************************
!> Module: OpticalProperties
!>
!>
!>  This module contains the subroutines for calculating the epsilon tensor using the generalized Drude-Lorentz model
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine eps_cDL:                       calculate epsilon for frequency w using conventional Drude-Lorentz model
!>      - subroutine InverseMatrix:                 calculate the inverse of a 3x3 complex matrix
!>      - subroutine InverseMatrix2x2:              calculate the inverse of a 2x2 symmetric complex matrix using analytic functions determined by Mathematica
!>      - subroutine Reflectance_All:               calculate the reflectance within the ac-plane
!>
!>
!> Copyright (C) 2009 - 2013
!>
!> I. Physikalisches Institut, Universitaet zu Koeln
!>
!> Produced for the CATS project and MnWO4 Paper
!>
!>
!> fortran module containing the subroutine for "generalized Drude-Lorentz model"
!>
!> Who           When        What
!>
!> T. Moeller    07/07/2009  Initial version
!> T. Moeller    04.11.2013  write epsilon tensor components, rotation angles and diagonal components
!> T. Moeller    09.11.2013  calculate R_p
!>
!*********************************************************************************************************************************************************
Module OpticalProperties

    implicit none
    integer :: num_col_y                                                                    !< number of y-columns
    integer :: number_osc                                                                   !< number of osc. for the generalized Drude-Lorentz model
    integer :: number_osc_cDL                                                               !< number of osc. for the conventional Drude-Lorentz model
    integer :: currentindex                                                                 !< current index
    integer :: switchflag                                                                   !< working variable: used for consistency
    real*8 :: RotAngle
    real*8 :: eps_inf_cDL                                                                   !< epsilon_\infty for conventional Drude-Lorentz model
    real*8 :: eps_inf_xx                                                                    !< epsilon_\infty for epsilon_xx matrix element
    real*8 :: eps_inf_xz                                                                    !< epsilon_\infty for epsilon_xz matrix element
    real*8 :: eps_inf_zz                                                                    !< epsilon_\infty for epsilon_zz matrix element
    real*8 :: Polarization_Offset                                                           !< offset of polarization angle \chi
    real*8 :: alpha, gamma                                                                  !< rotation angles around x-(alpha) and z-(gamma) axis
    real*8 :: AngleOfIncidence                                                              !< angle of incidence
    real*8 :: pi
    real*8, allocatable, dimension(:) :: sqrt_func                                          !< working variable: used for consistency
    real*8, allocatable, dimension(:) :: w0, wp, G, theta                                   !< eigenfrequencies, plasma frequencies, damping, and angle
                                                                                            !< for the generalized Drude-Lorentz model
    real*8, allocatable, dimension(:) :: w0_cDL, wp_cDL, G_cDL                              !< eigenfrequencies, plasma frequencies and damping for the
                                                                                            !< conventional Drude-Lorentz model
    contains


        !>************************************************************************************************************************************************
        !> subroutine: eps_cDL
        !>
        !> calculate epsilon for frequency w using conventional Drude-Lorentz model
        !>
        !>
        !> input variables:     w:                  frequency
        !>
        !> output variables:    epsilon_cDL:        epsilon value determine by conventional Drude-Lorentz model
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine eps_cDL(epsilon_cDL, w)

            implicit none
            integer :: i                                                                    !< loop variable
            real*8 :: w                                                                     !< input parameter: frequency
            complex(8) :: epsilon_cDL                                                       !< output parameter: epsilon


            epsilon_cDL = eps_inf_cDL                                                       !< add epsilon_\infty
            Do i = 1, number_osc_cDL                                                        !< loop over all oscillators
                epsilon_cDL = epsilon_cDL + ((wp_cDL(i)**2) / (w0_cDL(i)**2 - w**2 - dcmplx(0.d0,1.d0) * G_cDL(i) * w))
            end Do
            return
        end subroutine eps_cDL


        !>************************************************************************************************************************************************
        !> subroutine: InverseMatrix
        !>
        !> calculate the inverse of a 3x3 symmetric complex matrix using analytic functions determined by Mathematica
        !>
        !>
        !> input variables:     matrix:             matrix which has to inverted
        !>
        !> output variables:    Inverse_tensor:     inverted matrix
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine InverseMatrix(matrix, Inverse_tensor)

            implicit none
            complex(8) :: a11, a12, a13, a21, a22, a23, a31, a32, a33
            complex(8),dimension(3,3) :: matrix, Inverse_tensor

            a11 = matrix(1,1)
            a12 = matrix(1,2)
            a13 = matrix(1,3)
            a21 = matrix(2,1)
            a22 = matrix(2,2)
            a23 = matrix(2,3)
            a31 = matrix(3,1)
            a32 = matrix(3,2)
            a33 = matrix(3,3)

            Inverse_tensor = dcmplx(0.d0,0.d0)
            Inverse_tensor(1, 1) = (a23*a32 - a22*a33)/(a13*a22*a31 - a12*a23*a31 - a13*a21*a32 + a11*a23*a32 + a12*a21*a33 - a11*a22*a33)
            Inverse_tensor(1, 2) = (a13*a32 - a12*a33)/(-(a13*a22*a31) + a12*a23*a31 + a13*a21*a32 - a11*a23*a32 - a12*a21*a33 + a11*a22*a33)
            Inverse_tensor(1, 3) = (a13*a22 - a12*a23)/(a13*a22*a31 - a12*a23*a31 - a13*a21*a32 + a11*a23*a32 + a12*a21*a33 - a11*a22*a33)
            Inverse_tensor(2, 1) = (a23*a31 - a21*a33)/(-(a13*a22*a31) + a12*a23*a31 + a13*a21*a32 - a11*a23*a32 - a12*a21*a33 + a11*a22*a33)
            Inverse_tensor(2, 2) = (a13*a31 - a11*a33)/(a13*a22*a31 - a12*a23*a31 - a13*a21*a32 + a11*a23*a32 + a12*a21*a33 - a11*a22*a33)
            Inverse_tensor(2, 3) = (a13*a21 - a11*a23)/(-(a13*a22*a31) + a12*a23*a31 + a13*a21*a32 - a11*a23*a32 - a12*a21*a33 + a11*a22*a33)
            Inverse_tensor(3, 1) = (a22*a31 - a21*a32)/(a13*a22*a31 - a12*a23*a31 - a13*a21*a32 + a11*a23*a32 + a12*a21*a33 - a11*a22*a33)
            Inverse_tensor(3, 2) = (a12*a31 - a11*a32)/(-(a13*a22*a31) + a12*a23*a31 + a13*a21*a32 - a11*a23*a32 - a12*a21*a33 + a11*a22*a33)
            Inverse_tensor(3, 3) = (a12*a21 - a11*a22)/(a13*a22*a31 - a12*a23*a31 - a13*a21*a32 + a11*a23*a32 + a12*a21*a33 - a11*a22*a33)
            return
        end subroutine InverseMatrix


        !>************************************************************************************************************************************************
        !> subroutine: InverseMatrix2x2
        !>
        !> calculate the inverse of a 2x2 symmetric complex matrix using analytic functions determined by Mathematica
        !>
        !>
        !> input variables:     matrix:             matrix which has to inverted
        !>
        !> output variables:    Inverse_tensor:     inverted matrix
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine InverseMatrix2x2(matrix, Inverse_tensor)

            implicit none
            complex(8) :: eps11, eps12, eps21, eps22                                        !< working variables: matrix elements
            complex(8), dimension(2, 2) :: matrix                                           !< input variable: matrix
            complex(8), dimension(2, 2) :: Inverse_tensor                                   !< output variable: inverse of matrix

            eps11 = matrix(1, 1)
            eps12 = matrix(1, 2)
            eps21 = matrix(2, 1)
            eps22 = matrix(2, 2)

            Inverse_tensor = dcmplx(0.d0, 0.d0)
            Inverse_tensor(1, 1) =   eps22/(-eps12 * eps21 + eps11 * eps22)
            Inverse_tensor(1, 2) = -(eps12/(-eps12 * eps21 + eps11 * eps22))
            Inverse_tensor(2, 1) = -(eps21/(-eps12 * eps21 + eps11 * eps22))
            Inverse_tensor(2, 2) =   eps11/(-eps12 * eps21 + eps11 * eps22)
            return
        end subroutine InverseMatrix2x2


        !>************************************************************************************************************************************************
        !> subroutine: Reflectance_All
        !>
        !> calculate the reflectance within the ac-plane
        !>
        !>
        !> input variables:     w:                  frequency
        !>
        !>                      angle:              polarization angle
        !>
        !> output variables:    value:              calculated reflectance
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine Reflectance_All(w, angle, value)

            implicit none
            integer :: i                                                                    !< loop variable
            real*8 :: w                                                                     !< input parameter: current frequency w
            real*8 :: angle                                                                 !< input parameter: current corrected polarization angle
            real*8 :: value                                                                 !< output parameter: final value of reflectance
            real*8 :: theta_rad                                                             !< angle theta in radians not in degree
            real*8, dimension(2, 2) :: RotMatrix, InverseRotMatrix
            complex(8) :: epsxx                                                             !< tensor component eps_xx
            complex(8) :: epsxz                                                             !< tensor component eps_xz
            complex(8) :: epszx                                                             !< tensor component eps_zx
            complex(8) :: epszz                                                             !< tensor component eps_zz
            complex(8) :: epsyy                                                             !< tensor component eps_yy
            complex(8) :: eps_para, eps_perp, rs, rp, ryy, w1, w2                           !< working variables
            complex(8) :: DrudeLorentz                                                      !< working variable: pure Drude-Lorentz part
            complex(8) :: eps11, eps12, eps22                                               !< working variables
            complex(8), dimension(2) :: ev1, ev2                                            !< working variables for eigenvectors
            complex(8), dimension(2) :: help_2d_vector                                      !< working variables
            complex(8), dimension(3) :: help_3d_vector                                      !< working variables
            complex(8), dimension(2) :: pol_2d_vector                                       !< 2d polarization vector
            complex(8), dimension(3) :: pol_3d_vector                                       !< 3d polarization vector
            complex(8), dimension(2, 2) :: epsilon_tensor                                   !< rotated epsilon tensor
            complex(8), dimension(2, 2) :: refl_tensor                                      !< reflectance tensor
            complex(8), dimension(3, 3) :: refl_rot_tensor                                  !< rotated reflectance tensor
            complex(8), dimension(2, 2) :: EV                                               !< eigenvectors of rotated epsilon tensor
            complex(8), dimension(2, 2) :: InverseEV                                        !< inverse of eigenvectors of rotated epsilon tensor
            complex(8), dimension(2, 2) :: diagonal_matrix                                  !< working array
            complex(8), dimension(3, 3) :: rotation_matrix, Inverse_rotation_matrix         !< rotation matrix

            value = 0.d0                                                                    !< reset final output reflectance


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< determine all matrix elements of the epsilon tensor
            epsxx = eps_inf_xx
            epsxz = eps_inf_xz
            epszz = eps_inf_zz


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< determine the unmodified tensor elements which corresponds to the generalized Drude-Lorentz model


            ! Debug:
            ! if (w > 1.d1 .and. w < 1.d3) then
            !     write(643,'(F15.10,$)') w
            ! endif
            Do i = 1, number_osc                                                            !< loop over all oscillators
                DrudeLorentz = ((wp(i)**2) / (w0(i)**2 - w**2 - dcmplx(0.d0,1.d0) * G(i) * w))  !< determine pure Drude-Lorentz part
                theta_rad = theta(i) * (pi/180.d0)                                          !< convert degree to rad
                epsxx = epsxx + DrudeLorentz * (dcos(theta_rad)**2)
                epsxz = epsxz + DrudeLorentz * (dsin(theta_rad) * dcos(theta_rad))
                epszz = epszz + DrudeLorentz * (dsin(theta_rad)**2)

                ! Debug:
                ! if (w > 1.d1 .and. w < 1.d3) then
                !     write(643,'(F25.10,F25.10,$)') dreal(DrudeLorentz), dimag(DrudeLorentz)
                ! endif
            end Do
            epszx = epsxz

            ! Debug:
            ! if (w > 1.d1 .and. w < 1.d3) then
            !     write(643,*) w, dreal(epszz), dimag(epszz)
            ! endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define eigenvectors of the rotated epsilon tensor
            epsilon_tensor = 0.d0
            epsilon_tensor(1,1) = epsxx
            epsilon_tensor(1,2) = epsxz
            epsilon_tensor(2,1) = epsxz
            epsilon_tensor(2,2) = epszz

            ! Debug:
            !   print*,'frequency = ',w
            !   print*,epsilon_tensor(1,1),epsilon_tensor(1,2)
            !   print*,epsilon_tensor(2,1),epsilon_tensor(2,2)


            !< rotate basis of epsilon tensor
            RotMatrix(1, 1) = dcos(RotAngle)
            RotMatrix(1, 2) = -dsin(RotAngle)
            RotMatrix(2, 1) = dsin(RotAngle)
            RotMatrix(2, 2) = dcos(RotAngle)
            InverseRotMatrix(1, 1) = dcos(RotAngle)
            InverseRotMatrix(1, 2) = dsin(RotAngle)
            InverseRotMatrix(2, 1) = -dsin(RotAngle)
            InverseRotMatrix(2, 2) = dcos(RotAngle)
            epsilon_tensor = matmul(matmul(RotMatrix, epsilon_tensor), InverseRotMatrix)

            epsxx = epsilon_tensor(1, 1)
            epsxz = epsilon_tensor(1, 2)
            epszz = epsilon_tensor(2, 2)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define eigenvectors of the rotated epsilon tensor
            EV = dcmplx(0.d0, 0.d0)
            eps11 = epsxx
            eps12 = epsxz
            eps22 = epszz
            w1 = cdsqrt((eps11**2 + 4*eps12**2 - 2*eps11*eps22 + eps22**2))


            !< check for consistency
            sqrt_func(currentindex) = dimag(w1)
            if (sqrt_func(currentindex - 1) > 0.d0 .and. sqrt_func(currentindex) < 0.d0) then
                if (sqrt_func(currentindex - 1) - sqrt_func(currentindex) > 1.d1) then
                    switchflag = 1 - switchflag

                    ! Debug:
                    ! print*,'w_jump = ', w
                endif
            endif


            !< determine eigenvectors
            ev1(1) = -(-eps11 + eps22 + w1)/(2.*eps12)
            ev1(2) = dcmplx(1.d0, 0.d0)
            ev2(1) = -(-eps11 + eps22 - w1)/(2.*eps12)
            ev2(2) = dcmplx(1.d0, 0.d0)

            ! Debug:
            ! write(646,*) w, dreal(cdsqrt(w1)), dimag(cdsqrt(w1))


            !< check, if eigenvectors are orthogonal
            if (dabs(dreal(dot_product(conjg(ev1(:)), ev2(:)))) > 1.d-7 .or. dabs(dimag(dot_product(conjg(ev1(:)), ev2(:)))) > 1.d-7) then
                print*,'dot_product(conjg(ev1(:)), ev2(:)) = ',dot_product(conjg(ev1(:)), ev2(:))
                print*,'------------------------------------------------------------------'
            endif


            !< check, if eigenvectors are normalized
            w1 = dot_product(conjg(ev1(:)), ev1(:))
            ev1(:) = 1.d0 / cdsqrt(w1) * ev1(:)
            if (dabs(dreal(dot_product(conjg(ev1(:)), ev1(:))) - 1.d0) > 1.d-7 .or. dabs(dimag(dot_product(conjg(ev1(:)), ev1(:)))) > 1.d-7) then
                print*,'dot_product(conjg(ev1(:)), ev1(:)) = ',dot_product(conjg(ev1(:)), ev1(:))
                print*,'------------------------------------------------------------------'
            endif


            !< check, if eigenvectors are normalized
            w2 = dot_product(conjg(ev2(:)), ev2(:))
            ev2(:) = 1.d0 / cdsqrt(w2) * ev2(:)
            if (dabs(dreal(dot_product(conjg(ev2(:)), ev2(:))) - 1.d0) > 1.d-7 .or. dabs(dimag(dot_product(conjg(ev2(:)), ev2(:)))) > 1.d-7) then
                print*,'dot_product(conjg(ev2(:)), ev2(:)) = ',dot_product(conjg(ev2(:)), ev2(:))
                print*,'------------------------------------------------------------------'
            endif


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< construct eigenvector matrix
            if (switchflag == 100) then
                EV(1, 1) = ev2(1)
                EV(2, 1) = ev2(2)
                EV(1, 2) = ev1(1)
                EV(2, 2) = ev1(2)
            else
                EV(1, 1) = ev1(1)
                EV(2, 1) = ev1(2)
                EV(1, 2) = ev2(1)
                EV(2, 2) = ev2(2)
            endif

            ! Debug:
            !   print*,'Eigenvectors:'
            !   print*,'EV(1,:) = ', EV(1,:)
            !   print*,'EV(2,:) = ', EV(2,:)


            !< write angle of orientation
            !write(644,*) w, dacos(dreal(EV(1, 1))) * 180/pi , dacos(dimag(EV(1, 1))) * 180/pi
            !write(646,*) w, dreal(EV(1, 1)), dacos(dreal(EV(1, 1))), dacos(dimag(EV(1, 1))), dacos(dreal(EV(2, 2))), dacos(dimag(EV(2, 2))), &
            !             dasin(dreal(EV(2, 2))), dasin(dimag(EV(2, 2)))


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< define inverse of eigenvectors of the rotated epsilon tensor
            InverseEV = dcmplx(0.d0,0.d0)
            call InverseMatrix2x2(EV, InverseEV)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< bring epsilon tensor on diagonal form
            diagonal_matrix = dcmplx(0.d0, 0.d0)
            diagonal_matrix = matmul(matmul(InverseEV, epsilon_tensor), EV)

            ! Debug:
            !	print*,diagonal_matrix(1,1),diagonal_matrix(1,2)
            !	print*,diagonal_matrix(2,1),diagonal_matrix(2,2)


            !<--------------------------------------------------------------------------------------------------------------------------------------------
            !< using Fresnel formula and Jones matrix formalism
            !< take both diagonal matrix element of the rotated epsilon tensor
            eps_para = diagonal_matrix(1, 1)
            eps_perp = diagonal_matrix(2, 2)

            ! Debug:
            !   write(645,*) w, dreal(eps_para), dimag(eps_para), dreal(eps_perp), dimag(eps_perp)
            !    if (w > 1.d1 .and. w < 1.d3) then
            !       write(645,*) w, dreal(eps_para), dreal(eps_perp)
            !    endif


            !< calculate the square root carfully.
            w1 = cdsqrt(eps_para - (dsin(AngleOfIncidence))**2)
            w2 = cdsqrt(eps_perp - (dsin(AngleOfIncidence))**2)


            !< determine Fresnel coefficients using pure Fresnel formula
            rs = (dcos(AngleOfIncidence) - w1) / (dcos(AngleOfIncidence) + w1)
            rp = (dcos(AngleOfIncidence) - w2 / eps_perp) / (dcos(AngleOfIncidence) + w2 / eps_perp)


            !< use modified rp formula taken from E.E.Koch et al., Chemical Physics 3 (1974) 362-369
            ! rp = (dcos(AngleOfIncidence) - cdsqrt(epszz - (dsin(AngleOfIncidence))**2) / cdsqrt(epsxx * epszz - epsxz**2)) &
            !     / (dcos(AngleOfIncidence) + cdsqrt(epszz - (dsin(AngleOfIncidence))**2) / cdsqrt(epsxx * epszz - epsxz**2))


            !< determine Jones matrix
            !< Note, in the rotated coordinate system, the sample should be isotropic (?), i.e. r_sp and r_ps should be zero (?)
            refl_tensor = dcmplx(0.d0, 0.d0)
            refl_tensor(1, 1) = dcmplx(dabs(dreal(rp)), dabs(dimag(rp)))
            refl_tensor(2, 2) = dcmplx(dabs(dreal(rp)), dabs(dimag(rp)))


            !< rotate coordinate system back
            diagonal_matrix = matmul(matmul(EV, refl_tensor), InverseEV)

            ! Debug:
            !    print*,'w = ', w
            !    print*,'diagonal_matrix(1,:) = ',diagonal_matrix(1,:)
            !    print*,'diagonal_matrix(2,:) = ',diagonal_matrix(2,:)
            !    print*,'#################################################################'
            !    stop


            !< consider rotation of the sample around the x- and z- axes
            if (alpha /= 0.d0 .or. gamma /= 0.d0) then


                !< determine eps__yy
                epsyy = dcmplx(0.d0)                                                        !< reset eps_yy
                call eps_cDL(epsyy, w)


                !< determine reflectance
                ryy = (dcmplx(1.d0, 0.d0) - cdsqrt(epsyy)) / (dcmplx(1.d0, 0.d0) + cdsqrt(epsyy))


                !< construct rotated reflectance tensor
                refl_rot_tensor = dcmplx(0.d0, 0.d0)
                refl_rot_tensor(1, 1) = diagonal_matrix(1, 1)
                refl_rot_tensor(1, 3) = diagonal_matrix(1, 2)
                refl_rot_tensor(2, 2) = ryy
                refl_rot_tensor(3, 1) = diagonal_matrix(2, 1)
                refl_rot_tensor(3, 3) = diagonal_matrix(2, 2)


                !< construct rotation matrix
                rotation_matrix(1, 1) = dcmplx(dcos(gamma), 0.d0)
                rotation_matrix(1, 2) = dcmplx(-dsin(gamma), 0.d0)
                rotation_matrix(1, 3) = dcmplx(0.d0, 0.d0)
                rotation_matrix(2, 1) = dcmplx(dcos(alpha) * dsin(gamma), 0.d0)
                rotation_matrix(2, 2) = dcmplx(dcos(alpha) * dcos(gamma), 0.d0)
                rotation_matrix(2, 3) = dcmplx(-dsin(alpha), 0.d0)
                rotation_matrix(3, 1) = dcmplx(dsin(alpha) * dsin(gamma), 0.d0)
                rotation_matrix(3, 2) = dcmplx(dcos(gamma) * dsin(alpha), 0.d0)
                rotation_matrix(3, 3) = dcmplx(dcos(alpha), 0.d0)


                !< determine inverse of rotation matrix
                Inverse_rotation_matrix = dcmplx(0.d0, 0.d0)
                call InverseMatrix(rotation_matrix, Inverse_rotation_matrix)


                !< rotate coordinate system
                refl_rot_tensor = matmul(matmul(Inverse_rotation_matrix, refl_rot_tensor), rotation_matrix)


                !< multiply polarization vector and determine value of reflectance
                pol_3d_vector = (/dcmplx(dcos(angle * (pi/180.d0)), 0.d0), dcmplx(0.d0, 0.d0), dcmplx(dsin(angle * (pi/180.d0)), 0.d0)/)
                help_3d_vector = matmul(refl_rot_tensor, pol_3d_vector)
                value = cdabs(dot_product(help_3d_vector, help_3d_vector))
            else


                !< multiply polarization vector and determine value of reflectance
                pol_2d_vector = (/dcmplx(dcos(angle * (pi/180.d0)), 0.d0), dcmplx(dsin(angle * (pi/180.d0)), 0.d0)/)
                help_2d_vector = matmul(diagonal_matrix, pol_2d_vector)
                value = dreal(dot_product(help_2d_vector, help_2d_vector))
            endif

            ! Debug:
            ! print*,'> w, value = ', w, value
            return
        end subroutine Reflectance_All
end Module OpticalProperties
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!>
!>
!> Main Program
!>
!>
program DrudeLorentzGeneral
    !> calculation of the refelctivity using the generalized Drude-Lorentz model

    use OpticalProperties

    implicit none
    integer :: i                                                                            !< loop variables
    integer :: NumberXValues                                                                !< number exp. data points
    integer :: alloc_status, dealloc_status                                                 !< status variables for (de-)allocation
    real*8 :: w                                                                             !< working variable: current frequency w
    real*8 :: angle_chi                                                                     !< working variable: current polarization anlge
    real*8 :: polarization_angle                                                            !< working variable: corrected polarization angle
    real*8 :: refl                                                                          !< working variable: final reflectance for frequency w


    pi = 4.d0 * datan(1.d0)                                                                 !< determine pi


    !<----------------------------------------------------------------------------------------------------------------------------------------------------
    !< open files
    open(21,file = "in.txt")                                                                !< open input file for generalized Drude-Lorentz model
    open(22,file = "in_conventional-Drude-Lorentz.txt")                                     !< open input file for conventional Drude-Lorentz model
    open(23,file = "DataIn.dat")                                                            !< open experimental-point file
    open(24,file = "FitFunctionValues.dat")                                                 !< open file for fit function values


    !<----------------------------------------------------------------------------------------------------------------------------------------------------
    !< read parameter for the generalized Drude-Lorentz model
    !< read general parameter from file: number of exp. data points, polarization angle offset, and rotation angles alpha and gamma
    read(21,*) NumberXValues                                                                !< read number of exp. data points
    read(21,*) Polarization_Offset                                                          !< read polarization offset
    read(21,*) AngleOfIncidence                                                             !< read angle of incidence
    read(21,*) alpha                                                                        !< read rotation angle alpha
    read(21,*) gamma                                                                        !< read rotation angle gamma


    !< convert angle degree in rad
    AngleOfIncidence = AngleOfIncidence * (pi/180.d0)
    alpha = alpha * (pi/180.d0)
    gamma = gamma * (pi/180.d0)


    !< read epsilon_infty for each tensor element and number of oscillators
    read(21,*) eps_inf_xx                                                                   !< read epsilon_xx
    read(21,*) eps_inf_xz                                                                   !< read epsilon_xz
    read(21,*) eps_inf_zz                                                                   !< read epsilon_zz
    read(21,*) number_osc                                                                   !< read number of oscillators


    !< deallocate/allocate memory
    if (allocated(w0)) then
        deallocate(w0, wp, G, theta, stat = dealloc_status)
        if (dealloc_status /= 0) then
            print '(" ")'
            print '(2x,"Error in module DrudeLorentzGeneral:")'
            print '(4x,"Cannot deallocate variable w0, wp, G, theta. ")'
            print '(" ")'
            stop '  Program aborted!'
        endif
    endif
    allocate(w0(number_osc), wp(number_osc), G(number_osc), theta(number_osc), sqrt_func(NumberXValues), stat = alloc_status)
    if (alloc_status /= 0) then
        print '(" ")'
        print '(2x,"Error in module DrudeLorentzGeneral:")'
        print '(4x,"Cannot allocate variable w0, wp, G, theta. ")'
        print '(" ")'
        stop '  Program aborted!'
    endif
    w0 = 0.d0
    wp = 0.d0
    G = 0.d0
    theta = 0.d0
    sqrt_func = 0.d0


    !< read parameters for each oscillator
    Do i = 1, number_osc                                                                    !< loop over all oscillators
        read(21,*) w0(i), wp(i), G(i), theta(i)
    end Do
    read(21,*) RotAngle
    RotAngle = RotAngle * (pi/180.d0)
    angle_chi = 90 * (pi/180.d0)
    close(21, status = 'delete')                                                            !< close and delete parameter file

    ! Debug:
    !    print*,'generalied Drude-Lorentz model:'
    !    print*,'eps_inf_xx = ',eps_inf_xx
    !    print*,'eps_inf_xz = ',eps_inf_xz
    !    print*,'eps_inf_zz = ',eps_inf_zz
    !    print*,'number_osc = ',number_osc
    !    print*,'w0(1:number_osc) = ',w0(1:number_osc)
    !    print*,'wp(1:number_osc) = ',wp(1:number_osc)
    !    print*,'G(1:number_osc) = ',G(1:number_osc)
    !    print*,'theta(1:number_osc) = ',theta(1:number_osc)
    !    print*,' '


    !<----------------------------------------------------------------------------------------------------------------------------------------------------
    !< read parameter for the conventional Drude-Lorentz model
    read(22,*) eps_inf_cDL                                                                  !< read epsilon_\infty
    read(22,*) number_osc_cDL                                                               !< read number of oscillators


    !< deallocate/allocate memory
    if (allocated(w0_cDL)) then
        deallocate(w0_cDL, wp_cDL, G_cDL, stat = dealloc_status)
        if (dealloc_status /= 0) then
            print '(" ")'
            print '(2x,"Error in module DrudeLorentzGeneral:")'
            print '(4x,"Cannot deallocate variable w0_cDL, wp_cDL, G_cDL. ")'
            print '(" ")'
            stop '  Program aborted!'
        endif
    endif
    allocate(w0_cDL(number_osc_cDL), wp_cDL(number_osc_cDL), G_cDL(number_osc_cDL), stat = alloc_status)
    if (alloc_status /= 0) then
        print '(" ")'
        print '(2x,"Error in module DrudeLorentzGeneral:")'
        print '(4x,"Cannot allocate variable w0_cDL, wp_cDL, G_cDL. ")'
        print '(" ")'
        stop '  Program aborted!'
    endif
    w0_cDL = 0.d0
    wp_cDL = 0.d0
    G_cDL = 0.d0


    !< read parameters for each oscillator
    Do i = 1, number_osc_cDL                                                                !< read parameter describing the ith oscillator
        read(22,*) w0_cDL(i), wp_cDL(i), G_cDL(i)
    end Do
    close(22, status = 'delete')                                                            !< close and delete parameter file

    ! Debug:
    !    print*,'conventional Drude-Lorentz model:'
    !    print*,'eps_inf_cDL = ',eps_inf_cDL
    !    print*,'number_osc_cDL = ',number_osc_cDL
    !    print*,'w0_cDL(1:number_osc_cDL) = ',w0_cDL(1:number_osc_cDL)
    !    print*,'wp_cDL(1:number_osc_cDL) = ',wp_cDL(1:number_osc_cDL)
    !    print*,'G_cDL(1:number_osc_cDL) = ',G_cDL(1:number_osc_cDL)
    !    print*,' '


    !< calculate optical properties at any frequency w
    switchflag = 0
    Do i = 1, NumberXValues
        currentindex = i
        read(23,*) w                                                                        !< read frequency and polarization angle form exp. file
        polarization_angle = angle_chi + Polarization_Offset


        !< call subroutine for determine reflectance
        call Reflectance_All(w, polarization_angle, refl)                                   !< calculate reflectance for non-normal incidence


        !< write fit function value to file
        write(24,*) refl                                                                    !< write reflection to output file
    end Do


    !< close files
    close(23, status = 'delete')                                                            !< close and delete experimental-point file
    close(24)                                                                               !< close output file
end program DrudeLorentzGeneral
!---------------------------------------------------------------------------------------------------------------------------------------------------------

