!*********************************************************************************************************************************************************
!> Module: OpticalProperties
!>
!>
!>  This module contains the subroutines for calculating the epsilon tensor using the generalized Drude-Lorentz model
!>
!>
!>
!>  The following subroutines and functions are included in this module:
!>
!>      - subroutine reflectance:                   calculate the reflectance within the ac-plane
!>      - subroutine InverseMatrix:                 calculate the inverse of a 2x2 complex matrix
!>
!>
!> Copyright (C) 2009 - 2013
!>
!> I. Physikalisches Institut, Universitaet zu Koeln
!>
!> Produced for the CATS project and MnWO4 Paper
!>
!>
!> fortran module containing the subroutine for "generalized Drude-Lorentz model"
!>
!> Who           When        What
!>
!> T. Moeller    07/07/2009  Initial version
!>
!*********************************************************************************************************************************************************
Module OpticalProperties

    implicit none
    integer :: number_osc                                                                   !< number of osc. for the generalized Drude-Lorentz model
    real*8 :: eps_inf_xx                                                                    !< epsilon_\infty for epsilon_xx matrix element
    real*8 :: eps_inf_xz                                                                    !< epsilon_\infty for epsilon_xz matrix element
    real*8 :: eps_inf_zz                                                                    !< epsilon_\infty for epsilon_zz matrix element
    real*8, allocatable, dimension(:) :: w0, wp, G, theta, phi                              !< eigenfrequencies, plasma frequencies, damping, angle, phi
                                                                                            !< for the generalized Drude-Lorentz model

    contains


        !>************************************************************************************************************************************************
        !> subroutine: reflectance
        !>
        !> calculate the reflectance within the ac-plane
        !>
        !>
        !> input variables:     w:                  frequency
        !>
        !>                      angle:              polarization angle
        !>
        !> output variables:    value:              calculated reflectance
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine reflectance(w, angle, value)
            !< calculate the reflectance within the ac-plane

            implicit none
            integer :: i,k
            real*8 :: w, value, angle
            real*8 :: theta_rad,pi
            real*8, dimension(2,2) :: Unit_matrix
            complex(8) :: eps11,eps12,eps21,eps22,DrudeLorentz
            complex(8),dimension(2) :: dir_vector,help_vector
            complex(8),dimension(2) :: ev1,ev2
            complex(8),dimension(2,2) :: epsilon_tensor,Inverse_epsilon_tensor,Eigenvectors_epsilon_tensor
            complex(8),dimension(2,2) :: diagonal_matrix,sqrt_diagonal_matrix,new_diagonal_matrix
            complex(8),dimension(2,2) :: upper_refl_matrix,lower_refl_matrix,Inverse_lower_refl_matrix

            value = 0.d0


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< calculate matrix elements of the epsilon tensor
            eps11 = eps_inf_xx
            eps12 = eps_inf_xz
            eps22 = eps_inf_zz

            pi = 4.d0 * datan(1.d0)                                                                 !< pi
            k = 4
            Do i = 1, number_osc                                                                    !< loop over all oscillators
                DrudeLorentz = ((wp(i)**2) * cdexp(dcmplx(0.d0, 1.d0) * phi(i)) / (w0(i)**2 - w**2 - dcmplx(0.d0,1.d0) * G(i) * w))
                theta_rad = theta(i) * (pi/180.d0)                                                  !< convert degree to rad
                eps11 = eps11 + DrudeLorentz * (dcos(theta_rad)**2)
                eps12 = eps12 + DrudeLorentz * (dsin(theta_rad) * dcos(theta_rad))
                eps22 = eps22 + DrudeLorentz * (dsin(theta_rad)**2)
                k = k + 4
            end Do
            eps21 = eps12


            !< define epsilon tensor
            epsilon_tensor = 0.d0
            epsilon_tensor(1,1) = eps11
            epsilon_tensor(1,2) = eps12
            epsilon_tensor(2,1) = eps21
            epsilon_tensor(2,2) = eps22

            ! Debug:
            !   print*,'frequency = ',w
            !   print*,epsilon_tensor(1,1),epsilon_tensor(1,2)
            !   print*,epsilon_tensor(2,1),epsilon_tensor(2,2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine eigenvectors of epsilon tensor (expression taken from Mathematica)


            !< EV1
            ev1(1) = -((-eps11 + eps22 + cdsqrt(eps11**2 + 4 * eps12**2 - 2 * eps11 * eps22 + eps22**2))/(2*eps12))
            ev1(2) = 1.d0


            !< EV2
            ev2(1) = -((-eps11 + eps22 - cdsqrt(eps11**2 + 4 * eps12**2 - 2 * eps11 * eps22 + eps22**2))/(2*eps12))
            ev2(2) = 1.d0


            !< construct eigenvector matrix
            Eigenvectors_epsilon_tensor = dcmplx(0.d0,0.d0)
            Eigenvectors_epsilon_tensor(1,1) = ev2(1)
            Eigenvectors_epsilon_tensor(2,1) = ev2(2)
            Eigenvectors_epsilon_tensor(1,2) = ev1(1)
            Eigenvectors_epsilon_tensor(2,2) = ev1(2)

            ! Debug:
            !	print*,Eigenvectors_epsilon_tensor(1,1),Eigenvectors_epsilon_tensor(1,2)
            !	print*,Eigenvectors_epsilon_tensor(2,1),Eigenvectors_epsilon_tensor(2,2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine inverse of eigenvector matrix
            Inverse_epsilon_tensor = dcmplx(0.d0,0.d0)
            call InverseMatrix(Eigenvectors_epsilon_tensor,Inverse_epsilon_tensor)

            ! Debug:
            !	print*,Inverse_epsilon_tensor(1,1),Inverse_epsilon_tensor(1,2)
            !	print*,Inverse_epsilon_tensor(2,1),Inverse_epsilon_tensor(2,2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< bring epsilon tensor on diagonal form
            diagonal_matrix = dcmplx(0.d0,0.d0)
            diagonal_matrix = matmul(matmul(Inverse_epsilon_tensor,epsilon_tensor),Eigenvectors_epsilon_tensor)

            ! Debug:
            !	print*,diagonal_matrix(1,1),diagonal_matrix(1,2)
            !	print*,diagonal_matrix(2,1),diagonal_matrix(2,2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< take square root of diagonal elements
            sqrt_diagonal_matrix = dcmplx(0.d0,0.d0)
            sqrt_diagonal_matrix = diagonal_matrix
            sqrt_diagonal_matrix(1,1) = cdsqrt(sqrt_diagonal_matrix(1,1))
            sqrt_diagonal_matrix(2,2) = cdsqrt(sqrt_diagonal_matrix(2,2))

            ! Debug:
            !	print*,sqrt_diagonal_matrix(1,1),sqrt_diagonal_matrix(1,2)
            !	print*,sqrt_diagonal_matrix(2,1),sqrt_diagonal_matrix(2,2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< rotate matrix back
            new_diagonal_matrix = dcmplx(0.d0,0.d0)
            new_diagonal_matrix = matmul(matmul(Eigenvectors_epsilon_tensor,sqrt_diagonal_matrix),Inverse_epsilon_tensor)

            ! Debug:
            !	print*,new_diagonal_matrix(1,1),new_diagonal_matrix(1,2)
            !	print*,new_diagonal_matrix(2,1),new_diagonal_matrix(2,2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< determine upper and lower part of the reflectance matrix


            !< define unity matrix
            Unit_matrix = 0.d0
            Unit_matrix(1,1) = 1.d0
            Unit_matrix(2,2) = 1.d0


            !< define upper matrix
            upper_refl_matrix = dcmplx(0.d0,0.d0)
            upper_refl_matrix = Unit_matrix - new_diagonal_matrix


            !< define lower matrix
            lower_refl_matrix = dcmplx(0.d0,0.d0)
            lower_refl_matrix = Unit_matrix + new_diagonal_matrix


            !< determine inverse of lower_refl_matrix
            Inverse_lower_refl_matrix = dcmplx(0.d0,0.d0)
            call InverseMatrix(lower_refl_matrix,Inverse_lower_refl_matrix)

            ! Debug:
            !	print*,upper_refl_matrix(1,1),upper_refl_matrix(1,2)
            !	print*,upper_refl_matrix(2,1),upper_refl_matrix(2,2)

            !	print*,lower_refl_matrix(1,1),lower_refl_matrix(1,2)
            !	print*,lower_refl_matrix(2,1),lower_refl_matrix(2,2)

            !	print*,Inverse_lower_refl_matrix(1,1),Inverse_lower_refl_matrix(1,2)
            !	print*,Inverse_lower_refl_matrix(2,1),Inverse_lower_refl_matrix(2,2)


            !---------------------------------------------------------------------------------------------------------------------------------------------
            !< calulate reflectance
            dir_vector = (/dcmplx(dcos(angle * (pi/180.d0)),0.d0),dcmplx(dsin(angle * (pi/180.d0)),0.d0)/)
            help_vector = matmul(matmul(upper_refl_matrix, Inverse_lower_refl_matrix), dir_vector)
            value = dreal(dot_product(help_vector, help_vector))
            return
        end subroutine reflectance


        !>************************************************************************************************************************************************
        !> subroutine: InverseMatrix
        !>
        !> calculate the inverse of a 2x2 symmetric complex matrix using analytic functions determined by Mathematica
        !>
        !>
        !> input variables:     matrix:             matrix which has to inverted
        !>
        !> output variables:    Inverse_tensor:     inverted matrix
        !>
        !>
        !> \author Thomas Moeller
        !>
        !> \date 07.07.2009
        !>
        subroutine InverseMatrix(matrix, Inverse_tensor)
            !< calculate the inverse of a 2x2 complex matrix

            implicit none
            complex(8) :: eps11,eps12,eps21,eps22
            complex(8),dimension(2,2) :: matrix,Inverse_tensor

            eps11 = matrix(1,1)
            eps12 = matrix(1,2)
            eps21 = matrix(2,1)
            eps22 = matrix(2,2)

            Inverse_tensor = dcmplx(0.d0,0.d0)
            Inverse_tensor(1,1) =   eps22/(-eps12 * eps21 + eps11 * eps22)
            Inverse_tensor(1,2) = -(eps12/(-eps12 * eps21 + eps11 * eps22))
            Inverse_tensor(2,1) = -(eps21/(-eps12 * eps21 + eps11 * eps22))
            Inverse_tensor(2,2) =   eps11/(-eps12 * eps21 + eps11 * eps22)
            return
        end subroutine InverseMatrix
end Module OpticalProperties
!*********************************************************************************************************************************************************


!*********************************************************************************************************************************************************
!>
!>
!> Main Program
!>
!>
program DrudeLorentzGeneral
    !< calculation of the refelctivity using the generalized Drude-Lorentz model

    use OpticalProperties

    implicit none
    integer :: i                                                                            !< loop variables
    integer :: NumberXValues                                                                !< number exp. data points
    integer :: alloc_status, dealloc_status                                                 !< status variables for (de-)allocation
    real*8 :: w                                                                             !< working variable: current frequency w
    real*8 :: refl                                                                          !< working variable: final reflectance for frequency w
    real*8 :: angle_chi                                                                     !< working variable: current polarization angle


    !< open files
    open(21,file = "in.txt")                                                                !< open parameter file
    open(22,file = "DataIn.dat")                                                            !< open experimental-point file
    open(23,file = "FitFunctionValues.dat")                                                 !< open file for fit function values


    !< read parameter from file
    read(21,*) NumberXValues                                                                !< read number of exp. data points
    read(21,*) eps_inf_xx                                                                   !< read epsilon_xx
    read(21,*) eps_inf_xz                                                                   !< read epsilon_xz
    read(21,*) eps_inf_zz                                                                   !< read epsilon_zz
    read(21,*) number_osc                                                                   !< read number of oscillators


    ! deallocate/allocate memory
    if (allocated(w0)) then
        deallocate(w0, wp, G, theta, phi, stat = dealloc_status)
        if (dealloc_status /= 0) then
            print '(" ")'
            print '(2x,"Error in module DrudeLorentzGeneral:")'
            print '(4x,"Cannot deallocate variable w0, wp, G, theta, phi.")'
            print '(" ")'
            stop '  Program aborted!'
        endif
    endif
    allocate(w0(number_osc), wp(number_osc), G(number_osc), theta(number_osc), phi(number_osc), stat = alloc_status)
    if (alloc_status /= 0) then
        print '(" ")'
        print '(2x,"Error in module DrudeLorentzGeneral:")'
        print '(4x,"Cannot allocate variable w0, wp, G, theta, phi.")'
        print '(" ")'
        stop '  Program aborted!'
    endif
    w0 = 0.d0
    wp = 0.d0
    G = 0.d0
    theta = 0.d0
    phi = 0.d0


    !< read parameters for each oscillators from file
    Do i = 1, number_osc                                                                    !< loop over all oscillators
        read(21,*) w0(i), wp(i), G(i), theta(i), phi(i)
    end Do
    close(21, status = 'delete')                                                            !< close and delete parameter file

    ! Debug:
    !    print*,'DrudeLorentzGeneral:'
    !    print*,'w = ',w
    !    print*,'angle_chi = ',angle_chi
    !    print*,'eps_inf_xx = ',eps_inf_xx
    !    print*,'eps_inf_xz = ',eps_inf_xz
    !    print*,'eps_inf_zz = ',eps_inf_zz
    !    print*,'number_osc = ',number_osc
    !    print*,'w0(1:number_osc) = ',w0(1:number_osc)
    !    print*,'wp(1:number_osc) = ',wp(1:number_osc)
    !    print*,'G(1:number_osc) = ',G(1:number_osc)
    !    print*,'theta(1:number_osc) = ',theta(1:number_osc)
    !    print*,'phi(1:number_osc) = ',phi(1:number_osc)
    !    print*,' '


    !< calculate optical properties at any frequency w
    Do i = 1, NumberXValues                                                                 !< loop over all frequency points
        read(22,*) w, angle_chi                                                             !< read frequency and polarization angle form file


        !< call subroutine for determine reflectance
        call reflectance(w, angle_chi, refl)                                                !< calculate reflectance

        !< write fit function value to file
        write(23,*) refl                                                                    !< write reflection to output file
    end Do

    !< close files
    close(22, status = 'delete')                                                            !< close and delete experimental-point file
    close(23)                                                                               !< close output file
end program DrudeLorentzGeneral
!---------------------------------------------------------------------------------------------------------------------------------------------------------

