!*********************************************************************************************************************************************************
!>
!>
!>
!>  Fortran module containing the subroutine for "conventional Drude-Lorentz model"
!>  Copyright (C) 2009 - 2014  Thomas Moeller
!>
!>  I. Physikalisches Institut, University of Cologne
!>
!>
!>
!>  Versions of the program:
!>
!>  Who           When        What
!>
!>  T. Moeller    07/06/2009  Initial version
!>
!>
!>  License:
!>
!>    GNU GENERAL PUBLIC LICENSE
!>    Version 3, 29 June 2007
!>    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
!>
!>
!>    This program is free software: you can redistribute it and/or modify
!>    it under the terms of the GNU General Public License as published by
!>    the Free Software Foundation, either version 3 of the License, or
!>    (at your option) any later version.
!>
!>    This program is distributed in the hope that it will be useful,
!>    but WITHOUT ANY WARRANTY; without even the implied warranty of
!>    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!>    GNU General Public License for more details.
!>
!>    You should have received a copy of the GNU General Public License
!>    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!>
!---------------------------------------------------------------------------------------------------------------------------------------------------------
program DrudeLorentzConv
    !< calculation of the refelctivity using the conventional Drude-Lorentz model

    implicit none
    integer :: i, j                                                                         !< loop variables
    integer :: NumberXValues                                                                !< number of frequency points
    integer :: alloc_status, dealloc_status                                                 !< status variables
    integer :: number_osc                                                                   !< number of oscillators
    real*8 :: pi                                                                            !< pi
    real*8 :: Einf                                                                          !< value of epsilon infinity
    real*8 :: w                                                                             !< frequency of the experimental data file
    real*8 :: refl                                                                          !< calculated reflectance
    real*8, allocatable, dimension(:) :: w0                                                 !< Eigenfrequency for each oscillator
    real*8, allocatable, dimension(:) :: wp                                                 !< plasma frequency for each oscillator
    real*8, allocatable, dimension(:) :: G                                                  !< damping for each oscillator
    complex*16 :: eps                                                                       !< value of dielectric function
    ! complex*16 :: sigma                                                                     !< value of optical conductivity function
    ! complex*16 :: n_ref                                                                     !< value of the complex refraction index


    pi = 4.d0 * datan(1.d0)                                                                 !< get value of pi


    !< open files
    open(21,file = "in.txt")                                                                !< open parameter file
    open(22,file = "DataIn.dat")                                                            !< open experimental-point file
    open(23,file = "FitFunctionValues.dat")                                                 !< open file for fit function values


    !< read parameter from file
    read(21,*) NumberXValues                                                                !< read number of frequency points
    read(21,*) Einf                                                                         !< value of epsilon infinity
    read(21,*) number_osc                                                                   !< read number of oscillators


    !< deallocate/allocate memory
    if (allocated(w0)) then
        deallocate(w0, wp, G, stat = dealloc_status)
        if (dealloc_status /= 0) then
            print '(" ")'
            print '(2x,"Error in module DrudeLorentzConv:")'
            print '(4x,"Cannot deallocate variable w0, wp, G. ")'
            print '(" ")'
            stop '  Program aborted!'
        endif
    endif
    allocate(w0(number_osc), wp(number_osc), G(number_osc), stat = alloc_status)
    if (alloc_status /= 0) then
        print '(" ")'
        print '(2x,"Error in module DrudeLorentzConv:")'
        print '(4x,"Cannot allocate variable w0, wp, G. ")'
        print '(" ")'
        stop '  Program aborted!'
    endif
    w0 = 0.d0
    wp = 0.d0
    G = 0.d0


    !< read parameters for each oscillators from file
    Do i = 1, number_osc                                                                    !< loop over all oscillators
        read(21,*) w0(i), wp(i), G(i)
    end Do
    close(21, status = 'delete')                                                            !< close and delete parameter file

    ! Debug:
    !    print*,'DrudeLorentzConv:'
    !    print*,'w = ',w
    !    print*,'Einf = ',Einf
    !    print*,'number_osc = ',number_osc
    !    print*,'w0(1:number_osc) = ',w0(1:number_osc)
    !    print*,'wp(1:number_osc) = ',wp(1:number_osc)
    !    print*,'G(1:number_osc) = ',G(1:number_osc)
    !    print*,' '


    !< calculate optical properties at any frequency w
    Do i = 1, NumberXValues                                                                 !< loop over all frequency points
        read(22,*) w                                                                        !< read frequency form file

        eps = dcmplx(0.d0, 0.d0)                                                            !< reset sum of all oscillators
        Do j = 1, number_osc                                                                !< loop over all oscillators
            if (w == 0.d0 .and. w0(j) == 0.d0) then
                eps = eps + (wp(j)**2)/(w0(j)**2 - (w + 0.00001)**2 - dcmplx(0.d0, 1.d0) * G(j) * (w + 0.00001))
            else
                eps = eps + (wp(j)**2)/(w0(j)**2 - w**2 - dcmplx(0.d0, 1.d0) * G(j) * w)
            endif
        end Do
        eps = dcmplx(Einf, 0.d0) + eps                                                       !< calculate epsilon


        ! Debug:
        ! print*,'i,w,eps = ', i,w,eps

        refl = cdabs( (1.d0 - cdsqrt(eps)) / (1.d0 + cdsqrt(eps)) )**2                      !< calculate reflectance


        !< NOT USED: determine values of optical conductivity function and complex refraction index
    !   sigma = dcmplx(0.,-1.) * (eps - 1.) * w  * (1./(4. * pi)) * (pi/15.)                !< sigma
    !   n_ref = cdsqrt(eps)                                                                 !< complex refraction index


        !< write fit function value to file
        write(23,*) refl                                                                    !< write reflection to output file
    end Do

    !< close files
    close(22, status = 'delete')                                                            !< close and delete experimental-point file
    close(23)                                                                               !< close output file
end program DrudeLorentzConv
!------------------------------------------------------------------------------------------------------------------------------------------------------------------------

