#!/usr/bin/python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module creates a plot showing the (observational) data, the calculated model function, and the transition energies from the database
##  Copyright (C) 2012 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function myXCLASSPlot:                        creates a plot showing the (observational) data, the calculated model function, and the
##                                                      transition energies from the database
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    25.07.2013   initial version
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import string                                                                               ## import string package
import random                                                                               ## import random package
import datetime                                                                             ## import datetime package
import warnings                                                                             ## import warnings package
import task_myXCLASS                                                                        ## import package MAGIX
import matplotlib                                                                           ## import matplotlib package
import pylab                                                                                ## load python package for plotting pylab
from matplotlib.font_manager import fontManager, FontProperties                             ## import fontManager and FontProperties packages
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## A ploting routine
##
def myXCLASSPlot(expdata, modeldata, TransEnergies, RestFreq, vLSR, MinIntensity, xLowerLimit, xUpperLimit, yLowerLimit, yUpperLimit, PlotTitle, \
                 LegendFlag, SaveFigureFile):
    """

This function creates a plot showing the (observational) data stored in the input parameter
"expdata" as black dots, the calculated model function stored in the parameter "modeldata"
as green line and the transition energies from the table "Transitions" stored in the parameter
"TransEnergies" as blue vertical lines.


input parameters:
-----------------

    - expdata:              2D numpy array containing the observational data

                            Note, the data has to be given as a function of frequency (in MHz)

    - modeldata:            2D numpy array containing the values of a model function

                            Note, the data has to be given as a function of frequency (in MHz)

    - TransEnergies:        python list containing the transition frequencies (in MHz) from the last myXCLASS run, the Doppler-shifted transition
                            frequencies (in MHz), the corresponding intensities (in K), the energy of the lower level (in K), the upper state
                            degeneracy, the Einstein A coefficient (in s^{-1}), and the molecule names.

    - RestFreq:             rest frequency in MHz (default: 0). (If this parameter is set to zero, the intensity is plotted against frequency (in MHz)
                            otherwise against velocity (in km/s).

    - vLSR:                 velocity (local standard of rest) in km/s, only used, if RestFreq /= 0. (velocity(Frequency = RestFreq) = vLSR)

    - MinIntensity:         minimal intensity (in K) of a transition for plotting molecule names (default: 0, i.e. plot all names)

    - xLowerLimit:          lower limit (in MHz / km/s) of the frequency/velocity (default: 0). (Depending on the value of the rest frequency: If
                            rest frequency is set to zero, the lower limit has to be given as frequency) If parameter is not given, all data will
                            be plotted (default: 0).

    - xUpperLimit:          upper limit (in MHz / km/s) of the frequency/velocity (default: 1.e8). (Depending on the value of the rest frequency:
                            If rest frequency is set to zero, the upper limit has to be given as frequency) If parameter is not given, all data will
                            be plotted (default: 1.e8).

    - yUpperLimit:          upper limit of the intensity (y-axis) (default: 0). (If value is not given or if yUpperLimit is equal to yLowerLimit then
                            the y-axis range is set automatically.)

    - yLowerLimit:          lower limit of the intensity (y-axis) (default: 0). (If value is not given or if yUpperLimit is equal to yLowerLimit then
                            the y-axis range is set automatically.)

    - PlotTitle:            defines the title of the plot (default: "")

    - LegendFlag:           defines, if legend is plottet (true) or not (false) (default: "T")

    - SaveFigureFile:       defines the path and name of the file to which the current figure is stored. If no file is defined, i.e. "", the figure is
                            shown in a GUI and not saved (default: "").



output parameters:
------------------

    - None


Example:
--------

FreqMin = 580102.0
FreqMax = 580546.5
FreqStep = 5.0000000000E-01
TelescopeSize = 3.5
Inter_Flag = F
t_back_flag = T
tBack = 1.1
tslope = 0.0000000000E+00
nH_flag = T
N_H = 3.0000000000E+24
beta_dust = 2.0
kappa_1300 = 0.02
MolfitsFileName = "demo/myXCLASS/CH3OH__pure.molfit"
iso_flag = T
IsoTableFileName = "demo/myXCLASS/iso_names.txt"
RestFreq = 0.0
vLSR = 0.0
modeldata, log, TransEnergies, IntOptical, jobDir = myXCLASS()

FileName = "demo/myXCLASS/band1b.dat"
NumHeaderLines = 1
expdata = LoadASCIIFile()

MinIntensity = 0.0
xLowerLimit = 580102.0
xUpperLimit = 580546.5
yLowerLimit = 0.8
yUpperLimit = 2.5
PlotTitle = "Example for myXCLASSPlot function"
LegendFlag = T
SaveFigureFile = ""
myXCLASSPlot()

    """

    # Debug:
    # print "len(expdata) = ", len(expdata)
    # print "len(modeldata) = ", len(modeldata)
    # print "len(TransEnergies) = ", len(TransEnergies)
    # print "TransEnergies = ", TransEnergies
    # print "MinIntensity = ", MinIntensity
    # print "RestFreq = ", RestFreq
    # print "vLSR = ", vLSR
    # print "xUpperLimit = ", xUpperLimit
    # print "xLowerLimit = ", xLowerLimit
    # print "yUpperLimit = ", yUpperLimit
    # print "yLowerLimit = ", yLowerLimit


    ##====================================================================================================================================================
    ## analyze expdata parameter
    if (type(expdata).__name__ == 'list'):                                                  ## is expdata a list ?
        expdata_flag = "list"
    else:                                                                                   ## is expdata an array ?
        expdata_flag = "array"


    ##====================================================================================================================================================
    ## analyze modeldata parameter
    if (type(modeldata).__name__ == 'list'):                                                ## is modeldata a list ?
        modeldata_flag = "list"
    else:                                                                                   ## is modeldata an array ?
        modeldata_flag = "array"


    ##====================================================================================================================================================
    ## analyze xlimit parameter
    xlimits_flag = "false"
    if (xUpperLimit != xLowerLimit):
        if (xUpperLimit < xLowerLimit):
            print "Error in XCLASS package, function myXCLASSPlot:"
            print " "
            print "  The given upper limit of x-axis is smaller than the lower limit of x-axis!"
            print " "
            print "  Please correct the limits of x-axis parameter and start myXCLASSPlot again."
            print " "
            print " "
            return
        else:
            xlimits_flag = "true"

    # Debug:
    # print "xlimits_flag = ", xlimits_flag


    ##====================================================================================================================================================
    ## analyze ylimit parameter
    ylimits_flag = "false"
    if (yUpperLimit != yLowerLimit):
        if (yUpperLimit < yLowerLimit):
            print "Error in XCLASS package, function myXCLASSPlot:"
            print " "
            print "  The given upper limit of y-axis is smaller than the lower limit of y-axis!"
            print " "
            print "  Please correct the limits of y-axis parameter and start myXCLASSPlot again."
            print " "
            print " "
            return
        else:
            ylimits_flag = "true"

    # Debug:
    # print "ylimits_flag = ", ylimits_flag


    ##====================================================================================================================================================
    ## check input variables


    ## check data array
    PlotExpDataFlag = "true"
    if (expdata == [] or expdata == "" or len(expdata[0]) < 2 or len(expdata) < 3):
        PlotExpDataFlag = "false"


    ## check model array
    PlotModelDataFlag = "true"
    if (modeldata == [] or modeldata == "" or len(modeldata[0]) < 2 or len(modeldata) < 3):
        PlotModelDataFlag = "false"


    ## check transition frequency array
    PlotTransEnergiesFlag = "true"
    if (TransEnergies == [] or TransEnergies == "" or len(TransEnergies) < 1 or RestFreq != 0.0):
        PlotTransEnergiesFlag = "false"
    elif (len(TransEnergies) == 1):
        if (TransEnergies == 0.0 or TransEnergies == ["0.0"] or TransEnergies == [["0.0"]] or TransEnergies == [["0.0", ""]]):
            PlotTransEnergiesFlag = "false"
    elif (len(TransEnergies) < 1):
        PlotTransEnergiesFlag = "false"


    ## extract transition frequencies which fulfill the creteria
    FinalTransitionArray = []
    MolFreqName = []
    if (PlotTransEnergiesFlag == "true"):
        TransEn = 0.0
        MaxTransInt = 0.0
        TransAddedFlag = False
        for line in TransEnergies:
            if (line[0].find("%") == (-1)):
                FirstIndex = 0
                if (len(line) == 7):
                    TransEn = float(line[0])
                    MaxTransInt = 0.0
                    FirstIndex = 1
                    TransAddedFlag = False
                if (not TransAddedFlag):
                    LocalInt = float(line[FirstIndex + 1])                                  ## get intensity
                    if (LocalInt > MaxTransInt):
                        MaxTransInt = LocalInt
                    if (MaxTransInt >= MinIntensity):
                        TransAddedFlag = True
                        FinalTransitionArray.append(TransEn)
                        MolFreqName.append(line[-1])

    # Debug:
    # print "PlotExpDataFlag = ", PlotExpDataFlag
    # print "PlotModelDataFlag = ", PlotModelDataFlag
    # print "PlotTransEnergiesFlag = ", PlotTransEnergiesFlag
    # print "FinalTransitionArray = ", FinalTransitionArray
    # print "MolFreqName = ", MolFreqName


    ##====================================================================================================================================================
    ## test if there is something to plot
    if (PlotExpDataFlag == "false" and PlotModelDataFlag == "false" and PlotTransEnergiesFlag == "false"):
        print "Error in XCLASS package, function myXCLASSPlot:"
        print " "
        print "  No data to plot!"
        print " "
        print " "
        return


    ##====================================================================================================================================================
    ## clear plot
    if (SaveFigureFile != ""):
        fig = pylab.figure(figsize=(17, 10))
    else:
        fig = pylab.figure(1)
    fig.clear()


    ##====================================================================================================================================================
    ## create plot
    # pylab.subplots_adjust(hspace=0.45, wspace=0.2)
    # layer = pylab.subplot(1, 1, 1)
    # tl = pylab.title(r'%s' % PlotTitle)
    # layer.grid(True)
    # layer.set_ylabel("Intensity")

    ax1 = fig.add_subplot(111)
    if (RestFreq != 0.0):
        ax2 = ax1.twiny()
    if (PlotTitle != ""):
        ax1.set_title(r'%s' % PlotTitle)
    ax1.grid(True)
    ax1.set_ylabel(r"T$_{\rm MB}$ [K]")

    # Debug:
    # print "RestFreq = ", RestFreq
    # print "PlotExpDataFlag = ", PlotExpDataFlag
    # print "PlotModelDataFlag = ", PlotModelDataFlag
    # print "PlotTransEnergiesFlag = ", PlotTransEnergiesFlag


    ## if RestFreq != 0.0, plot data against velocity
    if (RestFreq != 0.0):
        ax1.set_xlabel("velocity [km/s]")


        ## convert data points for expdata array
        if (PlotExpDataFlag == "true"):
            if (expdata_flag == "list"):
                FreqData = [(li[0]) for li in expdata]
            else:
                FreqData = expdata[:,0]
            expX = task_myXCLASS.ChangeToVelocitiy(FreqData, RestFreq, vLSR)


        ## convert data points for modeldata array
        if (PlotModelDataFlag == "true"):
            if (modeldata_flag == "list"):
                FreqData = [(li[0]) for li in modeldata]
            else:
                FreqData = modeldata[:,0]
            modX = task_myXCLASS.ChangeToVelocitiy(FreqData, RestFreq, vLSR)


        ## convert data points for TransEn array
        if (PlotTransEnergiesFlag == "true"):
            FreqData = FinalTransitionArray[:]
            transX = task_myXCLASS.ChangeToVelocitiy(FreqData, RestFreq, vLSR)


    ## plot data against frequency
    else:
        ax1.set_xlabel("Frequency [MHz]")
        if (PlotExpDataFlag == "true"):
            if (expdata_flag == "list"):
                expX = [(li[0]) for li in expdata]
            else:
                expX = expdata[:,0]
        if (PlotModelDataFlag == "true"):
            if (modeldata_flag == "list"):
                modX = [(li[0]) for li in modeldata]
            else:
                modX = modeldata[:,0]
        if (PlotTransEnergiesFlag == "true"):
            transX = FinalTransitionArray[:]


    ## check limits for exp data
    LowerLimitExpX = 0
    UpperLimitExpX = 0
    if (PlotExpDataFlag == "true"):


        ## check Limits
        LowestVal = min(s for s in expX)
        if (xlimits_flag == "false"):
            LowerLimitExpX = LowestVal
        else:
            if (xLowerLimit < LowestVal):
                LowerLimitExpX = LowestVal
            else:
                LowerLimitExpX = xLowerLimit

        HighestVal = max(s for s in expX)
        if (xlimits_flag == "false"):
            UpperLimitExpX = HighestVal
        else:
            if (xUpperLimit > HighestVal):
                UpperLimitExpX = HighestVal
            else:
                UpperLimitExpX = xUpperLimit

        # Debug:
        # print "LowerLimitExpX = ", LowerLimitExpX
        # print "UpperLimitExpX = ", UpperLimitExpX


    ## check limits for model data
    LowerLimitModX = 0
    UpperLimitModX = 0
    if (PlotModelDataFlag == "true"):


        ## check Limits
        LowestVal = min(s for s in modX)
        if (xlimits_flag == "false"):
            LowerLimitModX = LowestVal
        else:
            if (xLowerLimit < LowestVal):
                LowerLimitModX = LowestVal
            else:
                LowerLimitModX = xLowerLimit

        HighestVal = max(s for s in modX)
        if (xlimits_flag == "false"):
            UpperLimitModX = HighestVal
        else:
            if (xUpperLimit > HighestVal):
                UpperLimitModX = HighestVal
            else:
                UpperLimitModX = xUpperLimit


    ## redefine limit values
    if (PlotExpDataFlag == "true" and PlotModelDataFlag == "true"):
        xLowerLimit = max(LowerLimitExpX, LowerLimitModX)
        xUpperLimit = min(UpperLimitExpX, UpperLimitModX)

    elif (PlotExpDataFlag == "true" and PlotModelDataFlag == "false"):
        xLowerLimit = LowerLimitExpX
        xUpperLimit = UpperLimitExpX

    elif (PlotExpDataFlag == "false" and PlotModelDataFlag == "true"):
        xLowerLimit = LowerLimitModX
        xUpperLimit = UpperLimitModX
    ax1.set_xlim(xLowerLimit, xUpperLimit)

    # Debug:
    # print 'xLowerLimit, xUpperLimit = ', xLowerLimit, xUpperLimit


    ##====================================================================================================================================================
    ## add experimental to plot
    if (PlotExpDataFlag == "true"):
        if (expdata_flag == "list"):
            expY = [(li[-1]) for li in expdata]
        else:
            expY = expdata[:,-1]


        ## plot experimental data and set ranges
        ax1.plot(expX, expY, '-', color='black', linewidth=2.0, label = 'exp. data', drawstyle='steps-mid')
        ax1.set_xlim(xLowerLimit, xUpperLimit)
        if (xlimits_flag == "true"):


            ## get index in expX where expX = xLowerLimit and expX = xUpperLimit
            # val = min(enumerate(expX), key=lambda x: abs(x[1]-xLowerLimit))[0]
            # minexp = expY[val]

            f1 = max(0, (numpy.abs(expX - xLowerLimit)).argmin() - 1)
            f2 = min(len(expX) - 1, (numpy.abs(expX - xUpperLimit)).argmin() + 1)
            ObsFreqMinIndex = min(f1, f2)
            ObsFreqMaxIndex = max(f1, f2)
            minexp = min(expY[ObsFreqMinIndex:ObsFreqMaxIndex])
            maxexp = max(expY[ObsFreqMinIndex:ObsFreqMaxIndex])

            # Debug:
            # print "val = ", val
            # print "minexp = ", minexp
            # print "expY = ", expY
        else:
            #    minexp = expY[0]
            minexp = min(expY)
            maxexp = max(expY)


    ##====================================================================================================================================================
    ## add model data to plot
    if (PlotModelDataFlag == "true"):
        if (modeldata_flag == "list"):
            modY = [(li[-1]) for li in modeldata]
        else:
            modY = modeldata[:,-1]


        ## plot experimental data and set ranges
        ax1.plot(modX, modY, '-', color='red', label = 'model data', drawstyle='steps-mid')
        ax1.set_xlim(xLowerLimit, xUpperLimit)
        if (xlimits_flag == "true"):


            ## get index in expX where expX = xLowerLimit and expX = xUpperLimit
            #    val = min(enumerate(modX), key=lambda x: abs(x[1]-xLowerLimit))[0]
            #    minmodel = modY[val]

            f1 = max(0, (numpy.abs(modX - xLowerLimit)).argmin() - 1)
            f2 = min(len(modX) - 1, (numpy.abs(modX - xUpperLimit)).argmin() + 1)
            ModFreqMinIndex = min(f1, f2)
            ModFreqMaxIndex = max(f1, f2)
            minmodel = min(modY[ModFreqMinIndex:ModFreqMaxIndex])
            maxmodel = max(modY[ModFreqMinIndex:ModFreqMaxIndex])

            # Debug:
            # print "val = ", val
            # print "minmodel = ", minmodel
            # print "maxmodel = ", maxmodel
            # print "modY = ", modY
        else:
            # minmodel = modY[0]
            minmodel = min(modY)
            maxmodel = max(modY)


    ## set y ranges
    if (ylimits_flag == "false"):                                                           ## continue here, if no user limits are given
        if (PlotExpDataFlag == "true" and PlotModelDataFlag == "true" and PlotTransEnergiesFlag == "true"):         ## T T T
            yLowLim = min(minexp, minmodel)
            yUpLim = max(maxexp, maxmodel)
        elif (PlotExpDataFlag == "true" and PlotModelDataFlag == "true" and PlotTransEnergiesFlag == "false"):      ## T T F
            yLowLim = min(minexp, minmodel)
            yUpLim = max(maxexp, maxmodel)
        elif (PlotExpDataFlag == "true" and PlotModelDataFlag == "false" and PlotTransEnergiesFlag == "false"):     ## T F F
            yLowLim = minexp
            yUpLim = maxexp
        elif (PlotExpDataFlag == "false" and PlotModelDataFlag == "true" and PlotTransEnergiesFlag == "false"):     ## F T F
            yLowLim = minmodel
            yUpLim = maxmodel
        elif (PlotExpDataFlag == "false" and PlotModelDataFlag == "false" and PlotTransEnergiesFlag == "true"):     ## F F T
            yLowLim = 0.0
            yUpLim = 1.0
        elif (PlotExpDataFlag == "true" and PlotModelDataFlag == "false" and PlotTransEnergiesFlag == "true"):      ## T F T
            yLowLim = minexp
            yUpLim = maxexp
        elif (PlotExpDataFlag == "false" and PlotModelDataFlag == "true" and PlotTransEnergiesFlag == "true"):      ## F T T
            yLowLim = minmodel
            yUpLim = maxmodel
    else:
        yLowLim = yLowerLimit
        yUpLim = yUpperLimit
    if (PlotExpDataFlag == "true" or PlotModelDataFlag == "true" or PlotTransEnergiesFlag == "true"):
        ax1.set_ylim(yLowLim, yUpLim)


    ##====================================================================================================================================================
    ## add transition energies to plot
    if (PlotTransEnergiesFlag == "true"):


        ## construct array for y positions of ticks
        CounterTransFreq = (-1)
        for lines in transX:
            CounterTransFreq += 1

            #    ## add ticks to plot
            #    if (PlotExpDataFlag == "true" and PlotModelDataFlag == "true"):
            #        val = min(minexp, minmodel)
            #    elif (PlotExpDataFlag == "true" and PlotModelDataFlag == "false"):
            #        val = minexp
            #    elif (PlotExpDataFlag == "false" and PlotModelDataFlag == "true"):
            #        val = minmodel
            #    else:
            #        val = 1.0
            #    yPos.append(val)


            #    ## add vertical ticks at transition frequencies (velocities) to graph
            #    ax1.plot(transX, yPos, '|', markersize= 10, color='blue', label = 'transition energies')


            ## add vertical ticks at transition frequencies (velocities) to graph
            if (xLowerLimit <= lines and lines <= xUpperLimit):
                ax1.axvline(x=lines, color='blue', linewidth=1, linestyle='--', zorder=0)
                y = yUpLim - (yUpLim * 0.02)
                ax1.text(lines, y, MolFreqName[CounterTransFreq], fontsize=10, rotation=90, color = 'g', zorder=1)


    ## if rest freuqency is unequal zero, add frequency axis on top of the graph
    if (RestFreq != 0.0):
        cLight = 299792458.0                                                                ## speed of light in m/s
        cLight = cLight * 1.e-3                                                             ## speed of light in km/s
        fMin = RestFreq - (xLowerLimit - vLSR)/cLight * RestFreq
        fMax = RestFreq - (xUpperLimit - vLSR)/cLight * RestFreq
        ax2.set_xlim(fMin, fMax)
        ax2.set_xlabel("Rest Frequency [MHz]")
        step = (fMax - fMin)/5                                                              ## define only 3 ticks
        xtic = numpy.arange(fMin, fMax, step)
        ax2.set_xticks(xtic)
        ax2.xaxis.set_major_formatter(matplotlib.ticker.FormatStrFormatter('%.3e'))


    ## show legend
    if (LegendFlag):
        if (PlotTitle != ""):
            ax1.legend()
        else:
            ax1.legend(bbox_to_anchor=(0., 1.02, 1., .102), loc=3, ncol=4, mode="expand", borderaxespad=0.)


    ##====================================================================================================================================================
    ## show or save figure
    SaveFigureFile = SaveFigureFile.strip()
    if (SaveFigureFile != ""):
        pylab.savefig(SaveFigureFile, dpi=300)
        pylab.draw()
    else:
        pylab.show(block=True)


    ## close figure environment
    pylab.close(fig)


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------

