#!/usr/bin/python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module contains all subroutines to perform the myXCLASSMapRedoFit function which enables to improve a previous myXCLASSMapFit function run.
##  Copyright (C) 2012 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function myXCLASSMapRedoFit:                  redo one or more pixel fits of a previous myXCLASSMapFit function run
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    01.10.2014   initial version
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import string                                                                               ## import string package
import shutil                                                                               ## import shutil package
import random                                                                               ## import random package
import datetime                                                                             ## import datetime package
import time                                                                                 ## import package date and time
import warnings                                                                             ## import warnings package
from math import *                                                                          ## import math package
import sqlite3                                                                              ## import sqlite3 package
import logging                                                                              ## import logging package
import matplotlib                                                                           ## import matplotlib package
import pylab                                                                                ## import pylab package
import task_MAGIX                                                                           ## import package MAGIX
import task_myXCLASS                                                                        ## import package myXCLASS
import task_myXCLASSFit                                                                     ## import package myXCLASSMapFit
import task_myXCLASSMapFit                                                                  ## import package myXCLASSFit


## import pyfits package via astropy
try:
    import astropy.io.fits as pyfits
except ImportError:
    import pyfits
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## Simplified CASA interface for MAGIX with myXCLASS program to fit a complete data cube instead of a single spectrum
##
def myXCLASSMapRedoFit(JobNumber, PixelList, NumberIteration, AlgorithmXMLFile, MolfitsFileName, experimentalData, Threshold, FreqMin, \
                       FreqMax, t_back_flag, tBack, tslope, N_H, beta_dust, kappa_1300, iso_flag, IsoTableFileName):
    """

This function offers the possibility to redo one or more so called pixel fits of a
previous myXCLASSMapFit run. The function performs fits for selected pixels and
recreates the different parameter maps using the new optimized paramater values. The
user has to define the pixel coordinates, which should be re-fitted, the job number
of the previous myXCLASSMapFit and the new molfit and algorithm XML file.


input parameters:
-----------------

    - JobNumber:            Job number of a previous myXCLASSMapFit run which should
                            be improved.

    - PixelList:            list of all pixel coordinates which should be re-fitted.
                            Each coordinate consists of two numbers separated by a
                            komma, where the first number corresponds to the right
                            ascension and second to the declination. Different pixel
                            coordinates are enclosed by squard brackets and separated
                            by kommas.

    - NumberIteration:      max. number of iterations (default: 50).

    - AlgorithmXMLFile:     only necessary, if the user wants to use another fit
                            algorithm (than Levenberg-Marquardt) for fitting. Therefore, 
                            the path and name of a MAGXI xml-file defining settings
                            for an algorithm or algorithm chain has to be given. (A
                            relative path has to be defined relative to the current
                            working directory!)

                            NOTE, if the user specify a xml file, the number of
                            iterations given by the parameter "NumberIteration" is
                            ignored. The number of iteration is then given by the xml
                            file. In order to use the implemented fit algorithm
                            (Levenberg-Marquardt) clear the AlgorithmXMLFile parameter,
                            i.e. AlgorithmXMLFile = "", and define the max. number of
                            iterations by using parameter "NumberIteration".

    - MolfitsFileName:      path and name of the extended molfit file, including the
                            source size, the rotation temperature the column density,
                            the velocity width, the velocity offset and the flag
                            indicating if the current component is considered for
                            core "c" or foreground "f".

                            In contrast to the format of the molfit file the extended
                            molfit file required by the myXCLASSMapFit function contains
                            one (three) additional column(s) for each parameter of
                            each component.

                            A detailed description of the extended molfit file required
                            by the myXCLASSMapFit function can be found in the manual.

                            NOTE, a relative path has to be defined relative to the
                            current working directory!

    - experimentalData:     This parameter offers the possibility to use a new
                            experimental xml-file suitable for MAGIX for the selected
                            pixel(s), where new settings for each frequency range can be
                            defined. For that purpose the parameter experimentalData
                            has to define the path and name of the new xml file.

                            Please note, the myXCLASSMapRedoFit function does not
                            consider the tags <FileNamesExpFiles> and <ImportFilter>
                            because these tags has been defined by the previous
                            myXCLASSMapFit run.

                            NOTE, if the parameter experimentalData defines a relative
                            path, the path has to be defined relative to the current
                            working directory!

    - Threshold:            defines a threshold for a pixel. If the spectrum of a pixel
                            has a intensity lower than the value defined by this parameter
                            the pixel is not fitted (ignored).

    The following parameters are needed, if the parameter experimentalData does NOT
    describe the path and name of a MAGIX xml-file:


    - FreqMin:              start frequency of simulated spectrum (in MHz), (default: 1.e99).

                            Please note, if no start frequency is given, or if start frequency is
                            lower or equal to the end frequency, the myXCLASSMapRedoFit  function will
                            use the max. frequency range defined in the FITS header. Additionally,
                            the stepsize, i.e. the difference between two frequencies is taken
                            from the FITS header.

    - FreqMax:              end frequency of simulated spectrum (in MHz), (default: 1.e99).

    - t_back_flag:          ("T"/"F"), defines, if the user defined background 
                            temperature Tbg and temperature slope Tslope given
                            by the input parameters tBack and tslope describe the
                            continuum contribution completely (t_back_flag = "T")
                            or not (t_back_flag = "F") (default: "T").

    - tBack:                background temperature (in K), (default: 1.e99).

    - tslope:               temperature slope (dimensionless), (default: 1.e99).

    - N_H:                  Hydrogen column density (cm^{-2}), (default: 1.e99).

    - beta_dust:            beta for dust (dimensionless), (default: 1.e99).

    - kappa_1300:           kappa (cm^2 g^{-1}), (default: 1.e99).

    - iso_flag:             use isotopologues ("T"/"F"). If iso_flag is set to
                            "F" the ratios are all set to 1 (default: "F").

    - IsoTableFileName:     path and name of an ASCII file including the iso ratios
                            between certain molecules. The so-called "iso ratio file"
                            defines the iso ratios between molecules. The ASCII file
                            consists of three columns, where the first two columns
                            indicates the molecules, respectively. The third column
                            defines the ratio for both molecules. Here, the first
                            41 characters belong to the first column, the next
                            41 characters to the second column. The further character
                            belongs to the third column.

                            If no path and name is specified (default), the so-called iso-flag
                            is set to "false".

                            NOTE, if the parameter "experimentalData" defines a relative
                            path, the path has to be defined relative to the current
                            working directory!



Example using a molfit file in the old format and an ASCII file containing the experimental data:
-------------------------------------------------------------------------------------------------

JobNumber = 1234
PixelList = [[83.201, -15.1345], [83.111, -15.221]]
NumberIteration = 10
AlgorithmXMLFile = ""
MolfitsFileName = "demo/myXCLASSMapFit/CH3OH.molfit"
experimentalData = ""
Threshold = 0.0
FreqMin = 0.0
FreqMax = 0.0
t_back_flag = T
tBack = 9.5000000000E-01
tslope = 0.0000000000E+00
nH_flag = T
N_H = 3.0000000000E+24
beta_dust = 2.0
kappa_1300 = 0.3095
iso_flag = T
IsoTableFileName = "demo/myXCLASSMapFit/iso_names.txt"
myXCLASSMapRedoFit()


    """

    # Debug:
    # print "JobNumber = ", JobNumber
    # print "PixelList = ", PixelList
    # print "NumberIteration = ", NumberIteration
    # print "AlgorithmXMLFile = ", AlgorithmXMLFile
    # print "MolfitsFileName = ", MolfitsFileName
    # print "UsePreviousResults = ", UsePreviousResults
    # print "Threshold = ", Threshold
    # print "FreqMin = ", FreqMin
    # print "FreqMax = ", FreqMax
    # print "t_back_flag = ", t_back_flag
    # print "tBack = ", tBack
    # print "tslope = ", tslope
    # print "nH_flag = ", nH_flag
    # print "N_H = ", N_H
    # print "beta_dust = ", beta_dust
    # print "kappa_1300 = ", kappa_1300
    # print "IsoTableFileName = ", IsoTableFileName


    ## define some internal variables
    MultiprocessingMethod = "pp"
    cms = 299792458.0                                                                       ## speed of light in m/s
    ckms = cms * 1.e-3                                                                      ## speed of light in km/s
    NameOfFunction = "myXCLASSMapRedoFit"


    ##====================================================================================================================================================
    ## initialize working variables
    CurrentDir = os.getcwd() + "/"

    # Debug:
    # print "CurrentDir = ", CurrentDir


    ######################################################################################################################################################
    ## DO NOT EDIT OR REMOVE THE FOLLOWING LINE !!!!
    MAGIXrootDir = "/home/moeller/ALMA/CASA/myXCLASS-CASA-Interface/No-NR-version/Linux/XCLASS-Interface/programs/MAGIX/"
    ######################################################################################################################################################


    ##====================================================================================================================================================
    ## define myXCLASS root directory
    myXCLASSrootDir = os.path.normpath(MAGIXrootDir + "../../")

    # Debug:
    # print "myXCLASSrootDir = ", myXCLASSrootDir


    ##====================================================================================================================================================
    ## analyze input parameters


    ## determine XCLASS working directory
    myXCLASSRunDirectory = str(os.environ.get('myXCLASSRunDirectory','')).strip()           ## get value of environment variable
    if (myXCLASSRunDirectory == ""):                                                        ## is environment variable defined?
        RunDir = os.path.abspath(myXCLASSrootDir) + "/run/"
    else:
        if (myXCLASSRunDirectory[0] != "/"):                                                ## make path absolute
            myXCLASSRunDirectory = myXCLASSrootDir + myXCLASSRunDirectory
        RunDir = myXCLASSRunDirectory + "/"                                                 ## if environment variable is defined, apply user setting
    MAGIXjobDir = os.path.normpath(RunDir + "/myXCLASSMapFit/") + "/"                       ## define path of directory for myXCLASSMapFit function runs

    # Debug:
    #    print "RunDir = ", RunDir
    #    print "MAGIXjobDir = ", MAGIXjobDir


    ## set naem and path of default database file
    dbFilename = os.path.abspath(myXCLASSrootDir) + "/Database/cdms_sqlite.db"


    ## read in all job directories in the myXCLASSMapFit run diretory
    listing = os.listdir(MAGIXjobDir)


    ## check, if a job directory with the given job ID exists
    FoundFlag = "false"
    SelectedJobDir = ""
    WorkDir = ""
    for JobDir in listing:

        # Debug:
        # print "JobNumber, JobDir = ", JobNumber, JobDir

        if (JobDir.endswith("_" + str(JobNumber))):
            SelectedJobDir = MAGIXjobDir + "/" + JobDir + "/"
            WorkDir = SelectedJobDir
            FoundFlag = "true"
            break
    if (FoundFlag == "false"):
        print "\n\n\nError in XCLASS package, function task_myXCLASSMapRedoFit.myXCLASSMapRedoFit:"
        print "\tCan not find a directory with the give job number."
        print "\n\tJobNumber = ", JobNumber
        print "\tcurrent myXCLASS working directory = ", MAGIXjobDir
        print "\n\tPlease enter an exsisting job number and redo myXCLASSMapRedoFit  function call!"
        return


    ## check, if the user defined pixels exists
    print "\n\nCheck user input ..",
    sys.stdout.flush()
    if (PixelList == []):
        print "\n\n\nError in XCLASS package, function task_myXCLASSMapRedoFit.myXCLASSMapRedoFit:"
        print "\tNo pixel is defined."
        print "\n\tPlease enter a list of exsisting pixel coordinates for job number ", JobNumber
        return
    AllPixelOKFlag = "true"
    FoundPixelList = []
    for pixel in PixelList:
        RACoord = pixel[0]
        xString = "%.10f" % RACoord
        DecCoord = pixel[1]
        yString = "%.10f" % DecCoord
        PixelDirectory = xString + "__" + yString + "___"

        # Debug:
        # print "PixelDirectory = ", PixelDirectory


        ## read in all files and directories in selected job directory
        listing = os.listdir(SelectedJobDir + "Pixel-Fits/")
        FoundDirFlag = False
        for FileDir in listing:


            ## analyze user input
            if (FileDir.startswith(PixelDirectory)):
                RACoordFlag = True
                DecCoordFlag = True
            else:
                RACoordFlag = False
                DecCoordFlag = False
                if (float(RACoord).is_integer()):
                    if (FileDir.find("__" + str(RACoord) + "_-_") > (-1)):
                        RACoordFlag = True
                elif (FileDir.find(str(RACoord)) > (-1)):
                    RACoordFlag = True
                if (float(DecCoord).is_integer()):
                    if (FileDir.endswith("_-_" + str(DecCoord))):
                        DecCoordFlag = True
                elif (FileDir.find("__" + str(DecCoord)) > (-1)):
                    DecCoordFlag = True

            # Debug:
            # print "RACoordFlag = ", RACoordFlag
            # print "DecCoordFlag = ", DecCoordFlag


            ## is user defined pixel found?
            if (RACoordFlag and DecCoordFlag):
                FoundDirFlag = True
                h = FileDir.split("__")
                FoundPixelList.append([h[0], h[1]])
                # break
        if (not FoundDirFlag):
            print "\n\n\nError in XCLASS package, function task_myXCLASSMapRedoFit.myXCLASSMapRedoFit:"
            print "\tCan not find the directory for the give pixel coordinate."
            print "\n\tpixel coordinate = ", pixel
            print "\n"
            AllPixelOKFlag = "false"
    if (AllPixelOKFlag == "false"):
        print "\n"
        print "\tJobNumber = ", JobNumber
        print "\tcurrent myXCLASS working directory = ", MAGIXjobDir
        print " "
        print "\tPlease enter a pixel coordinates and redo myXCLASSMapRedoFit  function call!"
        return


    ## we've done
    print "done!"

    # Debug:
    # print "FoundPixelList = ", FoundPixelList
    # sys.exit(0)


    ##====================================================================================================================================================
    ## set up environment for MAGIX
    task_MAGIX.SetMAGIXEnvironment(MAGIXrootDir)


    ##====================================================================================================================================================
    ## prepare new pixel fits
    os.environ["OMP_STACKSIZE"] = '6999M'
    counterPixels = 0
    PixelListNew = []
    for pixel in FoundPixelList:                                                                 ## loop over all user defined pixel
        counterPixels += 1


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define current pixel directory
        RACoord = pixel[0]
        DecCoord = pixel[1]
        PixelDirectory = RACoord + "__" + DecCoord + "__"
        PixelListNew.append([RACoord, DecCoord])


        ## read in all files and directories in selected job directory
        listing = os.listdir(SelectedJobDir + "Pixel-Fits/")
        FoundDirFlag = False
        for FileDir in listing:
            if (FileDir.startswith(PixelDirectory)):
                FoundDirFlag = True
                PixelDirectory = os.path.normpath(SelectedJobDir + "Pixel-Fits/" + FileDir) + "/"
                break

        # Debug:
        # print "\nPixelDirectory = ", PixelDirectory


        ## print current pixel
        print_string = "\rPrepare files for fit (" + str(counterPixels) + "/" + str(len(FoundPixelList))
        print_string += ") for pixel (" + str(RACoord) + "/" + str(DecCoord) + ") ..                                                            "
        sys.stdout.write(print_string)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## make a backup of the existing directory
        lt = time.localtime()
        CurrentBackupDir = os.path.normpath(PixelDirectory + "/backup_for_RedoFit__" + time.strftime("%d-%m-%Y", lt) + "__" \
                                            + time.strftime("%H-%M-%S", lt)) + "/"
        cmdString = "mkdir " + CurrentBackupDir + "; "
        cmdString += "mv " + PixelDirectory + "/*.* " + CurrentBackupDir
        os.system(cmdString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## copy new algorithm control xml file to each pixel directory
        AlgorithmXMLFile = AlgorithmXMLFile.strip()
        if (AlgorithmXMLFile != ""):                                                        ## is a new algorithm xml file defined?


            ## if neccessary, make relative path absolute
            if (AlgorithmXMLFile[0] != "/"):
                AlgorithmXMLFile = CurrentDir + AlgorithmXMLFile
            filename = AlgorithmXMLFile


            ## check if path and name of the experimental file exsits
            if not(os.path.exists(filename) or os.path.isfile(filename)):
                print " "
                print " "
                print "Error in XCLASS package, function task_myXCLASSMapRedoFit.myXCLASSMapRedoFit:"
                print "  The given path and name of the algorithm xml-file ", filename
                print "  does not exsist!"
                print " "
                print "  Please enter a valid path and name of an algorithm xml-file and redo myXCLASSMapRedoFit  function call!"
                return


            ## copy new algorithm control xml file to current pixel directory
            cmdString = "cp " + filename + " " + PixelDirectory + "algorithm_control.xml"
            os.system(cmdString)


        else:
            cmdString = "cp " + CurrentBackupDir + "/algorithm_control.xml " + PixelDirectory
            task_myXCLASSFit.CreateLMControlXMLFile(PixelDirectory, NumberIteration)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get paths for exp. data files from myXCLASSMapFit run
        ExpFiles = []
        ExpFiles = task_MAGIX.GetXMLtag(CurrentBackupDir + "/exp.xml", "FileNamesExpFiles")


        ## copy spectrum data files from backup directory to current pixel directory
        for SpecFile in ExpFiles:
            SpecFileName = os.path.basename(SpecFile)
            SourceDir = CurrentBackupDir + "/" + SpecFileName
            cmdString = "cp " + SourceDir + " " + PixelDirectory
            os.system(cmdString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## analyze parameter "experimentalData"
        experimentalData = experimentalData.strip()
        if (experimentalData != ""):                                                        ## check, if a new xml file was defined


            ## copy new experimental xml file to current pixel directory
            MAGIXExpXML = PixelDirectory + "/exp.xml"
            cmdString = "cp " + experimentalData + " " + PixelDirectory + "/exp.xml"
            os.system(cmdString)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## check/adjust paths defined in xml file


            ## get flag for iso file
            IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "Isotopologues")
            if (len(IsoFlag) == 0):
                IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "iso_flag")
                if (len(IsoFlag) == 0):
                    IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "ISO_FLAG")
                    if (len(IsoFlag) == 0):
                        IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "Iso_Flag")
                        if (len(IsoFlag) == 0):
                            IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "Iso_flag")
            if (len(IsoFlag) > 0):
                IsoFlag = IsoFlag[0].lower()
                if (IsoFlag == "y" or IsoFlag == "t" or IsoFlag == "true"):
                    IsoFlag = "y"
                else:
                    IsoFlag = "n"
            else:
                IsoFlag = "n"

            # Debug:
            # print 'IsoFlag = ', IsoFlag


            ## get name of iso file
            if (IsoFlag == "y"):
                IsoFile = task_MAGIX.GetXMLtag(MAGIXExpXML, "IsoTableFileName")

                # Debug:
                # print 'IsoFile = ', IsoFile

                if (len(IsoFile) > 0):
                    IsoFile = IsoFile[0]
                    if (IsoFile.strip() != ""):
                        if (IsoFile[0] != "/"):                                             ## make relative path absolute
                            IsoTableFileName = CurrentDir + IsoFile
                        else:
                            IsoTableFileName = IsoFile
                        if not(os.path.exists(IsoTableFileName) or os.path.isfile(IsoTableFileName)):
                            print " "
                            print " "
                            print "Error in XCLASS package, function task_myXCLASSMapRedoFit.myXCLASSMapRedoFit :"
                            print "  The given path and name of the iso file " + IsoTableFileName
                            print "  does not exsist!"
                            print " "
                            print "  Please note, if a relative path is specified, the path has to be defined relative to the"
                            print "  current working directory!"
                            print " "
                            print "  Please enter an exsisting path and name of iso file and redo myXCLASSMapRedoFit  function call!"
                            return


                    ## copy iso table file to current pixel directory
                    task_MAGIX.WriteXMLtag(MAGIXExpXML, "IsoTableFileName", [PixelDirectory + "/isotable.dat"])
                    cmdString = "cp " + IsoTableFileName + " " + PixelDirectory + "/isotable.dat"
                    os.system(cmdString)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get name of database file
            PathFileNameDB = task_MAGIX.GetXMLtag(MAGIXExpXML, "dbFilename")

            # Debug:
            # print 'PathFileNameDB = ', PathFileNameDB

            if (len(PathFileNameDB) > 0):
                PathFileNameDB = PathFileNameDB[0].strip()
                if (PathFileNameDB.strip() != ""):
                    if (PathFileNameDB[0] != "/"):                                          ## make relative path absolute
                        PathFileNameDB = CurrentDir + PathFileNameDB
                    if not(os.path.exists(PathFileNameDB) or os.path.isfile(PathFileNameDB)):
                        print " "
                        print " "
                        print "Error in XCLASS package, function task_myXCLASSMapRedoFit.myXCLASSMapRedoFit :"
                        print "  The given path and name of the database file " + PathFileNameDB
                        print "  does not exsist!"
                        print " "
                        print "  Please note, if a relative path is specified, the path has to be defined relative to the"
                        print "  current working directory!"
                        print " "
                        print "  Please enter an exsisting path and name of database file and redo myXCLASSMapRedoFit  function call!"
                        return

                    # Debug:
                    # print 'PathFileNameDB = ', PathFileNameDB


                    ## modify experimental xml-file in MAGIX temp directory
                    PathFileNameDB = [PathFileNameDB]
                else:
                    PathFileNameDB = [dbFilename]
            else:
                PathFileNameDB = [dbFilename]


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## update path of exp data files for current pixel directory


            ## update paths for exp. data files
            task_MAGIX.WriteXMLtag(MAGIXExpXML, "FileNamesExpFiles", ExpFiles)


            ## set all import filter in xml file to "xclassASCII"
            NumberSpectra = len(ExpFiles)
            ImportFilter = []
            for i in xrange(NumberSpectra):
                ImportFilter.append("xclassASCII")
            task_MAGIX.WriteXMLtag(MAGIXExpXML, "ImportFilter", ImportFilter)


            ## determine the number of freq. ranges
            FreqMinList = task_MAGIX.GetXMLtag(MAGIXExpXML, "MinExpRange")
            NumFreqRanges = len(FreqMinList)


            ## check, if user defined a XML file without a threshold, but specifies a general threshold using the threshold input parameter
            if (Threshold != -1.e99):                                                       ## user defined a XML file and a general threshold value
                helplist=[str(Threshold) for x in xrange(NumFreqRanges)]
                task_MAGIX.WriteXMLtag(MAGIXExpXML, "Threshold", helplist)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## use old experimental xml file and update tags with new user settings
        else:
            print "\nUpdate parametes for import of experimental data .."


            ## copy experimental xml file to current pixel directory
            MAGIXExpXML = PixelDirectory + "/exp.xml"
            cmdString = "cp " + CurrentBackupDir + "/exp.xml" + " " + MAGIXExpXML
            os.system(cmdString)


            ## get number of spectra
            NumberSpectra = int(task_MAGIX.GetXMLtag(MAGIXExpXML, "NumberExpFiles")[0])

            # Debug:
            # print "NumberSpectra = ", NumberSpectra
            # print "t_back_flag = ", t_back_flag
            # print "tBack = ", tBack
            # print "tslope = ", tslope
            # print "nH_flag = ", nH_flag
            # print "N_H = ", N_H
            # print "beta_dust = ", beta_dust
            # print "kappa_1300 = ", kappa_1300


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## update settings


            ## determine the number of freq. ranges
            FreqMinList = task_MAGIX.GetXMLtag(MAGIXExpXML, "MinExpRange")
            NumFreqRanges = len(FreqMinList)


            ## update with new iso table file
            if (iso_flag or iso_flag == "true" or iso_flag == "y" or iso_flag == "TRUE" or iso_flag == "True" or iso_flag == 1):
                if (len(IsoTableFileName) > 0):
                    if (IsoTableFileName.strip() != ""):
                        if (IsoTableFileName[0] != "/"):                                    ## make relative path absolute
                            IsoTableFileName = CurrentDir + IsoTableFileName
                        if not(os.path.exists(IsoTableFileName) or os.path.isfile(IsoTableFileName)):
                            print " "
                            print " "
                            print "Error in XCLASS package, function task_myXCLASSMapRedoFit.myXCLASSMapRedoFit :"
                            print "  The given path and name of the iso file " + IsoTableFileName
                            print "  does not exsist!"
                            print " "
                            print "  Please note, if a relative path is specified, the path has to be defined relative to the"
                            print "  current working directory!"
                            print " "
                            print "  Please enter an exsisting path and name of iso file and redo myXCLASSMapRedoFit  function call!"
                            return


                    ## copy iso table file to current pixel directory
                    task_MAGIX.WriteXMLtag(MAGIXExpXML, "IsoTableFileName", [PixelDirectory + "/isotable.dat"])
                    cmdString = "cp " + IsoTableFileName + " " + PixelDirectory + "/isotable.dat"
                    os.system(cmdString)


                ## update path and name of iso table file
                IsoFileName = PixelDirectory + "/isotable.dat"
                task_MAGIX.WriteXMLtag(MAGIXExpXML, "iso_flag", ["True"])
                task_MAGIX.WriteXMLtag(MAGIXExpXML, "IsoTableFileName", [IsoFileName])


            ## deactivate iso ratio file
            else:
                task_MAGIX.WriteXMLtag(MAGIXExpXML, "iso_flag", ["False"])
                task_MAGIX.WriteXMLtag(MAGIXExpXML, "IsoTableFileName", [""])


            ## update xml file with new max. frequency
            if (FreqMin != 1.e99 and FreqMin != FreqMax):
                helplist=[str(FreqMin) for x in xrange(NumFreqRanges)]
                task_MAGIX.WriteXMLtag(MAGIXExpXML, "MinExpRange", helplist)


            ## update xml file with new max. frequency
            if (FreqMax != 1.e99 and FreqMin != FreqMax):
                helplist=[str(FreqMax) for x in xrange(NumFreqRanges)]
                task_MAGIX.WriteXMLtag(MAGIXExpXML, "MaxExpRange", helplist)


            ## update xml file with new background temperature
            if (tBack != 1.e99):
                helplist=[str(tBack) for x in xrange(NumFreqRanges)]
                task_MAGIX.WriteXMLtag(MAGIXExpXML, "BackgroundTemperature", helplist)


            ## update xml file with new temperature slope
            if (tslope != 1.e99):
                helplist=[str(tslope) for x in xrange(NumFreqRanges)]
                task_MAGIX.WriteXMLtag(MAGIXExpXML, "TemperatureSlope", helplist)


            ## update xml file with new hydrogen column density
            if (N_H != 1.e99):
                helplist=[str(N_H) for x in xrange(NumFreqRanges)]
                task_MAGIX.WriteXMLtag(MAGIXExpXML, "HydrogenColumnDensity", helplist)


            ## update xml file with new beta
            if (beta_dust != 1.e99):
                helplist=[str(beta_dust) for x in xrange(NumFreqRanges)]
                task_MAGIX.WriteXMLtag(MAGIXExpXML, "DustBeta", helplist)


            ## update xml file with new kappa
            if (kappa_1300 != 1.e99):
                helplist=[str(kappa_1300) for x in xrange(NumFreqRanges)]
                task_MAGIX.WriteXMLtag(MAGIXExpXML, "Kappa", helplist)


            ## check, if user defined a XML file without a threshold, but specifies a general threshold using the threshold input parameter
            if (Threshold != -1.e99):                                                       ## user defined a XML file and a general threshold value
                helplist=[str(Threshold) for x in xrange(NumFreqRanges)]
                task_MAGIX.WriteXMLtag(MAGIXExpXML, "Threshold", helplist)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get all min. and max. frequencies for each range and path and name of user defined database file
        FreqMinList = task_MAGIX.GetXMLtag(MAGIXExpXML, "MinExpRange")
        FreqMaxList = task_MAGIX.GetXMLtag(MAGIXExpXML, "MaxExpRange")
        dbList = task_MAGIX.GetXMLtag(MAGIXExpXML, "dbFilename")
        if (len(dbList) > 0):
            PathFileNameDB = dbList[0].strip()
            if (PathFileNameDB.strip() != ""):
                if (PathFileNameDB[0] != "/"):                                              ## make relative path absolute
                    PathFileNameDB = CurrentDir + PathFileNameDB
                if not(os.path.exists(PathFileNameDB) or os.path.isfile(PathFileNameDB)):
                    print " "
                    print " "
                    print "Error in XCLASS package, function task_myXCLASSMapRedoFit.myXCLASSMapRedoFit:"
                    print "  The given path and name of the database file " + PathFileNameDB
                    print "  does not exsist!"
                    print " "
                    print "  Please note, if a relative path is specified, the path has to be defined relative to the"
                    print "  current working directory!"
                    print " "
                    print "  Please enter an exsisting path and name of a database file and redo myXCLASSMapRedoFit function call!"
                    return

                # Debug:
                # print "PathFileNameDB = ", PathFileNameDB


                ## modify experimental xml-file in MAGIX temp directory
                dbList = [PathFileNameDB]
            else:
                dbList = []
        else:
            dbList = []


        ## analyze molfit parameter: check path and name of the molfit xml-file and copy molfit file to temp directory
        ok, dbFilename, MolfitsFile, PureNameMolfitsFile = task_myXCLASSFit.CheckMolfitFile(MolfitsFileName, NameOfFunction, CurrentDir, MAGIXjobDir, \
                                                                                            FreqMinList, FreqMaxList, dbList)
        if (ok == 1):                                                                       ## an error occurred
            return

        # Debug:
        # print "ok = ", ok
        # print "MolfitsFile = ", MolfitsFile
        # print "PureNameMolfitsFile = ", PureNameMolfitsFile


        ## update path of database file
        if (dbList == [] or dbList == [""]):
            dbList = [dbFilename]
        task_MAGIX.WriteXMLtag(MAGIXExpXML, "dbFilename", dbList)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## copy new molfit file to current pixel directory
        cmdString = "cp " + MolfitsFile + " " + PixelDirectory + PureNameMolfitsFile
        os.system(cmdString)


        ##----------------------------------------------------------------------------------------------------------------------------------------------------
        ## get flag for iso file
        IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "Isotopologues")
        if (len(IsoFlag) == 0):
            IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "iso_flag")
            if (len(IsoFlag) == 0):
                IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "ISO_FLAG")
                if (len(IsoFlag) == 0):
                    IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "Iso_Flag")
                    if (len(IsoFlag) == 0):
                        IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "Iso_flag")
        if (len(IsoFlag) > 0):
            IsoFlag = IsoFlag[0].lower()
            if (IsoFlag == "y" or IsoFlag == "t" or IsoFlag == "true"):
                IsoFlag = "y"
            else:
                IsoFlag = "n"
        else:
            IsoFlag = "n"

        # Debug:
        # print 'IsoFlag = ', IsoFlag


        ## get name of iso file
        if (IsoFlag == "y"):
            IsoFile = task_MAGIX.GetXMLtag(MAGIXExpXML, "IsoTableFileName")

            # Debug:
            # print 'IsoFile = ', IsoFile

            if (len(IsoFile) > 0):
                IsoFile = IsoFile[0]
                if (IsoFile.strip() != ""):
                    if (IsoFile[0] != "/"):                                                 ## make relative path absolute
                        IsoTableFileName = CurrentDir + IsoFile
                    else:
                        IsoTableFileName = IsoFile
                    if not(os.path.exists(IsoTableFileName) or os.path.isfile(IsoTableFileName)):
                        print " "
                        print " "
                        print "Error in XCLASS package, function task_myXCLASSMapRedoFit.myXCLASSMapRedoFit:"
                        print "  The given path and name of the iso file " + IsoTableFileName
                        print "  does not exsist!"
                        print " "
                        print "  Please note, if a relative path is specified, the path has to be defined relative to the"
                        print "  current working directory!"
                        print " "
                        print "  Please enter an exsisting path and name of iso file and redo myXCLASSMapRedoFit function call!"
                        return


                    ## copy iso ratio file to current pixel directory
                    i = IsoTableFileName.rfind("/")
                    if (i < 0):
                        i = 0
                    IsoTableFileName = PixelDirectory + IsoTableFileName[i + 1:]            ## define new iso file name


                    command_string = "cp " + IsoTableFileName + " " + PixelDirectory        ## copy experimetnal dat files to temp directory
                    os.system(command_string)
                    IsoTableFileNameList = [IsoTableFileName]

                    # Debug:
                    # print 'IsoFileList = ', IsoFileList


                    ## modify experimental xml-file in MAGIX temp directory
                    task_MAGIX.WriteXMLtag(MAGIXExpXML, "IsoTableFileName", IsoTableFileNameList)

                    # Debug:
                    #    print "MolfitsFile = ", MolfitsFile
                    #    print "IsoTableFileName = ", IsoTableFileName


                    ## check molecules
                    SelectIsoRatioTable = task_myXCLASSFit.CheckMolecules(MolfitsFile, IsoTableFileName, FreqMinList, FreqMaxList, dbList)


                    ## write new iso ratio file
                    NewIsoRatioFile = open(IsoTableFileName, 'w')
                    NewLine = "%-45s %-45s %25s %25s %25s" % ("% isotopologue:", "Iso-master:", "Iso-ratio:", "Lower-limit:", "upper-limit:")
                    NewIsoRatioFile.write(NewLine + "\n")
                    for IsoLine in SelectIsoRatioTable:
                        if (len(IsoLine) > 3):
                            NewLine = "%-45s %-45s %25s %25s %25s" % (IsoLine[0], IsoLine[1], IsoLine[2], IsoLine[3], IsoLine[4])
                        else:
                            NewLine = "%-45s %-45s %25s" % (IsoLine[0], IsoLine[1], IsoLine[2])
                        NewIsoRatioFile.write(NewLine + "\n")
                    NewIsoRatioFile.close()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create new xml io-control file
        task_myXCLASSFit.CreateIOControlXMLFile(PixelDirectory, PixelDirectory + "/exp.xml", PixelDirectory + PureNameMolfitsFile, MAGIXrootDir)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine max intensity and check if max. intensity is above threshold
        ListOfThresholds = task_MAGIX.GetXMLtag(MAGIXExpXML, "Threshold")                   ## get threshold values for each freq. range
        ListOfFreqMin = task_MAGIX.GetXMLtag(MAGIXExpXML, "MinExpRange")                    ## get min. freq. values for each freq. range
        ListOfFreqMax = task_MAGIX.GetXMLtag(MAGIXExpXML, "MaxExpRange")                    ## get max. freq. values for each freq. range
        MaxSpec = -1.e99
        DataFileID = 0
        FitPixelFlag = "true"
        if (Threshold != -1.e99 or len(ListOfThresholds) > 0):
            FitPixelFlag = "false"
            for DataFileName in ExpFiles:
                DataFileID += 1


                ## read in spectrum
                SpectrumData = numpy.loadtxt(DataFileName, skiprows = 0)


                ## analyze spectrum and determine max intensity
                for line in SpectrumData:
                    freq = line[0]
                    intensity = line[1]


                    ## consider only frequency within a given frequency range
                    for i in xrange(len(ListOfFreqMin)):                                    ## loop over all freq. ranges
                        freqMinLocal = ListOfFreqMin[i]                                     ## get min. freq. of ith range
                        freqMaxLocal = ListOfFreqMax[i]                                     ## get max. freq. of ith range

                        # Debug:
                        #    print "\n\n\n\nfreqMinLocal = ", freqMinLocal
                        #    print "freqMaxLocal = ", freqMaxLocal
                        #    print "DataFileID = ", DataFileID

                        if (freqMinLocal[0] == DataFileID and freqMaxLocal[0] == DataFileID):       ## check, if current range definition belongs to
                                                                                            ## the current data file
                            if (float(freqMinLocal[1]) <= freq and freq <= float(freqMaxLocal[1])): ## check, if current intensity is the highest
                                thresLocal = ListOfThresholds[i]                            ## get threshold value for current freq. range
                                if (thresLocal[0] == DataFileID):                           ## check, if threshold value corresponds to current data file
                                    Thres = float(thresLocal[1])

                                    # Debug
                                    #    print "thresLocal = ", thresLocal
                                    #    print "Thres = ", Thres
                                    #    print "intensity = ", intensity

                                    if (intensity >= Thres):
                                        FitPixelFlag = "true"
                                        break


                    ## exit loop if intensity is at least in one freq. range above threshold
                    if (FitPixelFlag == "true"):
                        break


                ## exit loop if intensity is at least in one freq. range above threshold
                if (FitPixelFlag == "true"):
                    break

        # Debug:
        #    print "FitPixelFlag = ", FitPixelFlag


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define if current pixel is fitted or not
        if (FitPixelFlag == "true"):


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## start new MAGIX fit of current pixel


            ## print what you do
            print_string = "\rStart MAGIX fit (" + str(counterPixels) + "/" + str(len(FoundPixelList))
            print_string += ") for pixel (" + str(RACoord) + "/" + str(DecCoord) + ") ..                                                            \n\n"
            sys.stdout.write(print_string)


            ## call MAGIX starter
            MAGIXOption = "plotsaveonly, model=myxclass"
            ok = 0
            ok = task_MAGIX.StartMAGIX(MAGIXrootDir, MAGIXOption, PixelDirectory + "io_control.xml")
            if (ok != 0):
                print "\n\n\n\t Program MAGIX aborted!\n\n"


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## analyze results of MAGIX fit (only if MAGIX finished successfully)
            else:


                ## determine all files in the current pixel directory
                listing = os.listdir(PixelDirectory)


                ## determine number and name(s) of algorithm(s)
                NumberFittedParameter = 0
                NumOutFiles = 0
                AlgNameList = []
                for files in listing:
                    if (files.endswith(".out.dat")):
                        NumOutFiles += 1
                        AlgName = files.strip()                                             ## remove leading and tailing blanks
                        AlgName = AlgName.replace(".out.dat", '')                           ## remove ending of file name
                        i = AlgName.find("Spectrum__00001")                                 ## determine the different algorithms only for the first cube
                        if (i > (-1)):
                            AlgName = AlgName[i + 15:].strip()                              ## extract name of algorithm
                            if (AlgName == ""):
                                AlgName = "LM"
                            AlgNameList.append(AlgName)


                    ## determine the number of free parameters
                    elif (files.startswith("parameters") and files.endswith(".out.xml") and NumberFittedParameter == 0):


                        ## read in xml file
                        output_file = open(PixelDirectory + files)
                        output_file_contents = output_file.readlines()
                        output_file.close()


                        ## get value of fitted parameter
                        for line in output_file_contents:
                            StrippedLine = line.strip()


                            ## check, if current line describes a fitted parameter
                            i = StrippedLine.find("Parameter fit=" + chr(34) + "true" + chr(34))
                            if (i > (-1)):
                                FittedParameter = "true"
                                NumberFittedParameter += 1

                # Debug:
                # print "AlgNameList = ", AlgNameList
                # print "NumberFittedParameter = ", NumberFittedParameter


    ##====================================================================================================================================================
    ## create parameter maps


    ## remove molfit copy
    cmdString = "rm -rf " + MolfitsFile
    os.system(cmdString)


    ## get a list of all files and directories in the current job directory
    SelectedJobDir = os.path.normpath(SelectedJobDir) + "/"
    listing = os.listdir(SelectedJobDir)                                                    ## get a list of everything in the job directory
    PixelDirectories = [d for d in os.listdir(SelectedJobDir + "Pixel-Fits/") if os.path.isdir(os.path.join(SelectedJobDir + "Pixel-Fits/", d))]
                                                                                            ## create a list containing only directories in the current
                                                                                            ## job directory
    # Debug:
    # print "PixelDirectories = ", PixelDirectories
    # print "listing = ", listing
    # print "SelectedJobDir = ", SelectedJobDir


    ## find fits file to get fits header
    NameOfCubeAll = []
    NumFrequenciesALL = []
    NumDimensionsALL = []
    lengthALL = []
    lengthFITSDATAALL = []
    CRPIXListALL = []
    CRVALListALL = []
    CDELTListALL = []
    CROTListALL = []
    CUNITListALL = []
    BSCALEALL = []
    BZEROALL = []
    BUNITALL = []
    RESTFRQALL = []
    UnitConversionFactorALL = []
    ValueOfFirstPointListALL = []
    fitsdataALL = []
    for files in listing:                                                                   ## loop over all files in the current job directory
        if (files.endswith(".fits")):                                                       ## is file a fits file?
            if (files.find("___model-function-values") > (-1)):                             ## does the fits file contains the model function values?

                # Debug:
                # print "files = ", files


                ## get fits header informations
                HDU = 0
                print_flag = "false"
                DataCubeFile = SelectedJobDir + files
                NameOfCubeAll.append(DataCubeFile)
                ok, NumDimensionsLocal, lengthLocal, lengthFITSDATALocal, CParameterLocal, BParameterLocal, ValueOfFirstPointListLocal, \
                                                            fitsdataLocal = task_myXCLASSMapFit.LoadFits(DataCubeFile, HDU, print_flag)


                ## append parameters for each FITS data cube to global lists
                NumDimensionsALL.append(NumDimensionsLocal)
                lengthALL.append(lengthLocal)
                lengthFITSDATAALL.append(lengthFITSDATALocal)
                CRPIXListALL.append(CParameterLocal[0])
                CRVALListALL.append(CParameterLocal[1])
                CDELTListALL.append(CParameterLocal[2])
                CROTListALL.append(CParameterLocal[3])
                CUNITListLOCAL = CParameterLocal[4]
                CUNITListALL.append(CParameterLocal[4])
                BSCALEALL.append(BParameterLocal[0])
                BZEROALL.append(BParameterLocal[1])
                BUNITALL.append(BParameterLocal[2])
                RESTFRQALL.append(BParameterLocal[3])
                ValueOfFirstPointListALL.append(ValueOfFirstPointListLocal)
                fitsdataALL.append(fitsdataLocal)
                NumFrequenciesALL.append(lengthLocal[2])


                # determine conversion factor to convert the third axis to MHz
                unit = CUNITListLOCAL[2].strip()                                            ## get unit description from FITS header
                UnitConversionFactor = task_myXCLASSMapFit.ConvertUnit(unit)
                UnitConversionFactorALL.append(UnitConversionFactor)

                # Debug:
                # print "UnitConversionFactor = ", UnitConversionFactor


                ## use ReadRegion subroutine to get a list of all pixel coordinates
                if (len(lengthALL) == 1):
                    regionFileName = ""
                    TotalNumberPixels, RegionPixelsX, RegionPixelsY, CentralPixelOfRegionX, CentralPixelOfRegionY, \
                    RegionMask = task_myXCLASSMapFit.ReadRegionFile(regionFileName, lengthALL[0], ValueOfFirstPointListALL[0], CDELTListALL[0], \
                                                                    CRVALListALL[0], CRPIXListALL[0])
                ## exit loop
                # break


    ## print what you do
    sys.stdout.write("\n\n\rRead in results of all pixel fits and create FITS images and plots ..")
    sys.stdout.flush()


    ## get default settings from first cube
    NumDimensions = NumDimensionsALL[0]
    length = lengthALL[0]
    lengthFITSDATA = lengthFITSDATAALL[0]
    CRPIXList = CRPIXListALL[0]
    CRVALList = CRVALListALL[0]
    CDELTList = CDELTListALL[0]
    CROTList = CROTListALL[0]
    CUNITList = CUNITListALL[0]
    BSCALE = BSCALEALL[0]
    BZERO = BZEROALL[0]
    BUNIT = BUNITALL[0]
    RESTFRQ = RESTFRQALL[0]
    UnitConversionFactor = 1.0
    ValueOfFirstPointList = ValueOfFirstPointListALL[0]
    fitsdata = fitsdataALL[0]


    ## get size of telescope and interferrometric flag
    SizeTelescopePerSpectrum = task_MAGIX.GetXMLtag(MAGIXExpXML, "TelescopeSize")
    InterFlagPerSpectrum = task_MAGIX.GetXMLtag(MAGIXExpXML, "Inter_Flag")


    ## construct body of FITS files
    NumXPixels = len(RegionPixelsX)
    NumYPixels = len(RegionPixelsY)
    MaxNumFreq = max(NumFrequenciesALL[:])
    ModelValues = numpy.zeros((NumXPixels, NumYPixels, MaxNumFreq, NumberSpectra), dtype=numpy.float32)
    if (lengthFITSDATA == length):
        chi2Values = numpy.zeros((NumXPixels, NumYPixels), dtype=numpy.float32)
    else:
        chi2Values = numpy.zeros((NumYPixels, NumXPixels), dtype=numpy.float32)

    # Debug:
    #    print "RegionPixelsY = ", RegionPixelsY
    #    print "RegionPixelsX = ", RegionPixelsX
    #    print "Number pixel directories = ", len(PixelDirectories)


    ## determine the best result for each pixel in the current job directory
    kk = 0
    ListOfKnownParameters = []
    ParamMapsList = []
    ModelFuncList = []
    ParamMapsCoordList = []
    for PixelDir in PixelDirectories:                                                       ## loop over all pixel directories
        PixelDirName = PixelDir.split("/")[-1]                                              ## split path into names and get name of file
        PixelDirName = PixelDirName.strip()                                                 ## remove blanks from the file name
        if (PixelDirName.find("__") > (-1)):                                                ## check, if it is really a pixel directory
            PixelDirNameComp = PixelDirName.split("__")                                     ## split name of pixel directory
            RACoord = PixelDirNameComp[0]                                                   ## get y coordinate
            DecCoord = PixelDirNameComp[1]                                                  ## get x coordinate
            RACoordString = "%.10f" % float(RACoord)
            DecCoordString = "%.10f" % float(DecCoord)

            # Debug:
            # print "\n\nRACoord, DecCoord = ", RACoord, DecCoord


            ## check, if current pixel is a new fitted pixel or an old pixel
            NewPixelFlag = "false"
            if ([RACoordString, DecCoordString] in PixelListNew):
                NewPixelFlag = "true"


            ## determine the x and y coordinates of the current pixel within the original fits cube
            yyy = (-1)
            xxx = (-1)
            counterX = (-1)                                                                 ## reset counter for x pixels
            for xPoints in RegionPixelsX:                                                   ## loop over all columns of the selected region
                counterX += 1                                                               ## increase counter for x pixels
                xPixels = xPoints[0] * BSCALE + BZERO                                       ## apply scale factor and zero offset ((FITS) for x pixel
                xString = "%.10f" % xPixels
                x = xPoints[1]
                xIndex = counterX

                # Debug:
                # print ">>xPixels = ", xPixels
                # print "xIndex, abs(xPixels - float(RACoord)) = ", xIndex, abs(xPixels - float(RACoord))


                if (xString.strip() == RACoord.strip()):                                    ## check, if x coordinate is reached
                    counterY = (-1)                                                         ## reset counter for y pixels
                    for yPoints in RegionPixelsY:                                           ## loop over all rows of the selected region
                        counterY += 1                                                       ## increase counter for y pixels
                        yPixels = yPoints[0] * BSCALE + BZERO                               ## apply scale factor and zero offset ((FITS) for y pixel
                        yString = "%.10f" % yPixels
                        y = yPoints[1]
                        yIndex = counterY
                        if (yString.strip() == DecCoord.strip()):                          ## check, if y coordinate is reached
                            yyy = counterY
                            xxx = counterX
                            break
                if (xxx != (-1) and yyy != (-1)):
                    break

            # Debug:
            #    print "DecCoord = ", DecCoord
            #    print "RACoord = ", RACoord
            #    print "yIndex, xIndex = ", yIndex, xIndex
            #    print "yPixels, xPixels = ", yPixels, xPixels
            #    print "yyy, xxx = ", yyy, xxx
            #    sys.exit(0)


            ## check, if pixel indices were found
            if (xxx != (-1) and yyy != (-1)):
                xString = "%.10f" % float(RACoord)
                yString = "%.10f" % float(DecCoord)
                PixelDirectory = xString + "__" + yString + "__"


                ## read in all files and directories in selected job directory
                listing = os.listdir(SelectedJobDir + "/Pixel-Fits/")
                FoundDirFlag = False
                for FileDir in listing:
                    if (FileDir.startswith(PixelDirectory)):
                        FoundDirFlag = True
                        PixelDirectory = os.path.normpath(SelectedJobDir + "/Pixel-Fits/" + FileDir) + "/"
                        break

                # Debug:
                # print "\n\n>>PixelDirectory = ", PixelDirectory


                ## get name of exp. data files from xml file
                ExpFiles = task_MAGIX.GetXMLtag(PixelDirectory + "/exp.xml", "FileNamesExpFiles")


                ## determine the best result for the current pixel, i.e. get best molfit with name(s) of corresponding xml and model data file(s)
                ConvertLogLinFlag = "false"
                BestChi2Value, InputFile, ListOfParamXMLFiles, ListOfModelFuncFiles = task_myXCLASSMapFit.GetMolfitFileOfBestResult(PixelDirectory, \
                                                                                                                         ConvertLogLinFlag, ExpFiles)
                # Debug:
                # print "BestChi2Value = ", BestChi2Value


                ## check, if fit of current pixel was done successfully
                if (InputFile == []):
                    BestChi2Value = -1.e99      ## NaN
                else:


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## create png file and store in png directory
                    if ([RACoordString, DecCoordString] in PixelListNew):
                        NumPlots = len(ListOfModelFuncFiles)
                        fig = pylab.figure(figsize=(15, NumPlots * 10))
                        fig.clear()
                        pylab.subplots_adjust(hspace=0.1, wspace=0.2, bottom=0.1, top=0.9)
                        for PlotIndex in xrange(NumPlots):
                            LocalModelData = numpy.loadtxt(PixelDirectory + ListOfModelFuncFiles[PlotIndex], skiprows = 0)
                            LocalExpData = numpy.loadtxt(ExpFiles[PlotIndex], skiprows = 0)
                            layer = pylab.subplot(NumPlots, 1, (PlotIndex + 1))
                            if (PlotIndex == 0):
                                layer.set_title("Best fit for pixel " + xString + ", " + yString)
                            if (PlotIndex == (NumPlots - 1)):
                                layer.set_xlabel("Rest Frequency [MHz]")
                                layer.xaxis.set_major_formatter(matplotlib.ticker.FormatStrFormatter('%.5e'))
                            layer.set_ylabel(r"T$_{mb}$ [K]")
                            layer.grid(True)
                            layer.plot(LocalExpData[:, 0], LocalExpData[:, 1], '-', color='black', linewidth=3.0, label = 'exp. data', \
                                                                                    drawstyle='steps-mid')
                            layer.plot(LocalModelData[:, 0], LocalModelData[:, 1], '-', color='red', linewidth=2.0, label = 'fit')
                            layer.set_xlim(LocalModelData[0, 0], LocalModelData[-1, 0])
                            layer.legend()
                        # pngFile = SelectedJobDir + "/Pixel-Plots/" + FileDir + ".png"
                        pngFile = SelectedJobDir + "/Pixel-Plots/" + FileDir + "___RedoFit.png"
                        pylab.savefig(pngFile, dpi=300)
                        pylab.draw()
                        matplotlib.pyplot.close(fig)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## read parameter values from xml file
                    ParameterList = task_myXCLASSMapFit.ReadParmXMLFile(PixelDirectory, ListOfParamXMLFiles)


                    ## determine the parameter ID for each parameter and store parameter values for current pixel in variable
                    ## ParamMapsList = .. [x-coord., y-coord., [[paramID_1, val_1], [paramID_2, val_2], .. ] ]
                    NewParameterList = []
                    for param in ParameterList:
                        paramMol = param[0]
                        paramComp = param[1]
                        paramName = param[2]
                        paramVal = float(param[3])
                        if (not ([paramName, paramMol, paramComp] in ListOfKnownParameters)):
                            ListOfKnownParameters.append([paramName, paramMol, paramComp])
                        paramID = ListOfKnownParameters.index([paramName, paramMol, paramComp])
                        NewParameterList.append([paramID, paramVal])
                    ParamMapsCoordList.append([xIndex, yIndex])
                    ParamMapsList.append([xIndex, yIndex, NewParameterList])


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## read model function values for different cubes

                    # Debug:
                    #    print "PixelDirectory = ", PixelDirectory
                    #    print "NumberSpectra = ", NumberSpectra
                    #    print "NumFrequenciesALL = ", NumFrequenciesALL
                    #    print "ListOfModelFuncFiles = ", ListOfModelFuncFiles
                    #    print "CDELTListALL = ", CDELTListALL
                    #    print "ValueOfFirstPointListALL = ", ValueOfFirstPointListALL
                    #    print "BSCALEALL = ", BSCALEALL
                    #    print "BZEROALL = ", BZEROALL
                    #    print "BUNITALL = ", BUNITALL
                    #    print "UnitConversionFactorALL = ", UnitConversionFactorALL
                    #    print "RESTFRQALL = ", RESTFRQALL
                    #    print "SizeTelescopePerSpectrum = ", SizeTelescopePerSpectrum


                    PixelSpectrumList = task_myXCLASSMapFit.ReadModelFiles(PixelDirectory, NumberSpectra, NumFrequenciesALL, ListOfModelFuncFiles, \
                                                                           CDELTListALL, ValueOfFirstPointListALL, BSCALEALL, BZEROALL, BUNITALL, \
                                                                           UnitConversionFactorALL, RESTFRQALL, SizeTelescopePerSpectrum, \
                                                                           InterFlagPerSpectrum)
                    for CubeNumLocal in xrange(NumberSpectra):                              ## loop over all data cubes
                        for freq_index in xrange(NumFrequenciesALL[CubeNumLocal]):          ## loop over all freq. points
                            ModelValues[xIndex, yIndex, freq_index, CubeNumLocal] = PixelSpectrumList[CubeNumLocal, freq_index]


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## store chi^2 value
                    if (lengthFITSDATA == length):
                        chi2Values[xIndex, yIndex] = numpy.float32(BestChi2Value)
                    else:
                        chi2Values[yIndex, xIndex] = numpy.float32(BestChi2Value)
            else:
                if (lengthFITSDATA == length):
                    chi2Values[xIndex, yIndex] = numpy.nan
                else:
                    chi2Values[yIndex, xIndex] = numpy.nan

                # Debug:
                #    kk += 1
                #    print "PixelDirectory = ", PixelDirectory
                #    print "BestChi2Value = ", BestChi2Value
                #    print "RACoord, DecCoord = ", RACoord, DecCoord
                #    print "yyy, xxx = ", yyy, xxx
                #    print "BSCALE = ", BSCALE
                #    print "BZERO = ", BZERO

    # Debug:
    # print "kk = ",kk
    # return


    ##====================================================================================================================================================
    ## an application of the myXCLASSMapRedoFit function might introduce new paramters which requires a second loop over all pixel coordinates
    NumberFittedParameter = len(ListOfKnownParameters)
    if (lengthFITSDATA == length):
        paramValues = numpy.zeros((NumXPixels, NumYPixels, NumberFittedParameter), dtype=numpy.float32)
    else:
        paramValues = numpy.zeros((NumberFittedParameter, NumYPixels, NumXPixels), dtype=numpy.float32)
    counterX = (-1)                                                                         ## reset counter for x pixels
    for xPoints in RegionPixelsX:                                                           ## loop over all columns of the selected region
        counterX += 1                                                                       ## increase counter for x pixels
        xPixels = xPoints[0] * BSCALE + BZERO                                               ## apply scale factor and zero offset ((FITS) for x pixel
        xIndex = counterX
        counterY = (-1)                                                                     ## reset counter for y pixels
        for yPoints in RegionPixelsY:                                                       ## loop over all rows of the selected region
            counterY += 1                                                                   ## increase counter for y pixels
            yPixels = yPoints[0] * BSCALE + BZERO                                           ## apply scale factor and zero offset ((FITS) for y pixel
            yIndex = counterY


            ## index of current pixel in list ParamMapsList
            try:
                ListIndex = ParamMapsCoordList.index([xIndex, yIndex])
            except ValueError:
                ListIndex = (-1)

            # Debug:
            #    print "counterY, counterX = ", counterY, counterX
            #    print "len(ParamMapsCoordList) = ", len(ParamMapsCoordList)
            #    print "ListIndex = ", ListIndex
            #    print "len(RegionPixelsY), len(RegionPixelsX) = ", len(RegionPixelsY), len(RegionPixelsX)
            #    print "yIndex, xIndex = ", yIndex, xIndex


            ## get parameter list of current coordinate
            if (ListIndex != (-1)):
                NewParameterList = ParamMapsList[ListIndex][2]


                ## store parameter value in array 
                for i in xrange(len(NewParameterList)):
                    paramID = NewParameterList[i][0]
                    if (lengthFITSDATA == length):
                        try:
                            paramValues[xIndex, yIndex, paramID] = NewParameterList[i][1]
                        except IndexError:
                            paramValues[xIndex, yIndex, paramID] = numpy.nan
                    else:
                        try:
                            paramValues[paramID, yIndex, xIndex] = NewParameterList[i][1]
                        except IndexError:
                            paramValues[paramID, yIndex, xIndex] = numpy.nan
#        else:
#            if (lengthFITSDATA == length):
#                paramValues[xIndex, yIndex, :] = numpy.nan
#            else:
#                paramValues[:, yIndex, xIndex] = numpy.nan


    ##====================================================================================================================================================
    ## write fits files
    CubeNumber = 0


    ## get rest frequency
    RestFreqLocal = RESTFRQALL[CubeNumber]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write new header and new data to new fits file for chi2 values
    Chi2FileName = SelectedJobDir + "BestResult___chi2values__RedoFit.fits"
    hdu = pyfits.PrimaryHDU(chi2Values)
    FrequencyFlag = "false"
    CentralPixelOfFrequency = 0
    bunitinfo = [' K      ', 'Brightness (pixel) unit']
    task_myXCLASSMapFit.WriteFitsHeader(hdu, BSCALEALL[CubeNumber], BZEROALL[CubeNumber], bunitinfo, RestFreqLocal, lengthALL[CubeNumber], \
                                        CentralPixelOfRegionX, CentralPixelOfRegionY, NumXPixels, NumYPixels, CRVALListALL[CubeNumber], \
                                        CRPIXListALL[CubeNumber], CDELTListALL[CubeNumber], CROTListALL[CubeNumber], CUNITListALL[CubeNumber], \
                                        FrequencyFlag, CentralPixelOfFrequency, NameOfFunction)
    pyfits.writeto(Chi2FileName, numpy.float32(hdu.data), hdu.header, clobber=True)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write new header and new data to new fits file for parameter values


    ## write parameter values to fits files
    for paramNum in xrange(NumberFittedParameter):
        NameParam = ListOfKnownParameters[paramNum][0]                                      ## get name of current parameter
        MoleculeName = ListOfKnownParameters[paramNum][1]                                   ## get name of molecule for current parameter
        NumberComp = ListOfKnownParameters[paramNum][2]                                     ## get index of component for current parameter

        # Debug:
        # print "\nMoleculeName = ", MoleculeName
        # print "NumberComp = ", NumberComp
        # print "NameParam = ", NameParam


        paramFileName = SelectedJobDir + "BestResult___parameter__" + NameParam
        if (NameParam == "source_size" or NameParam == "T_rot" or NameParam == "N_tot" or NameParam == "V_width" or NameParam == "V_off" \
            or NameParam == "T_Back" or NameParam == "T_Slope" or NameParam == "nHcolumn" or NameParam == "kappa" or NameParam == "beta"):
            paramFileName += "___molecule__" + MoleculeName + "___component__" + str(NumberComp)
        paramFileName += "__RedoFit.fits"
        if (lengthFITSDATAALL[CubeNumber] == lengthALL[CubeNumber]):
            hdu = pyfits.PrimaryHDU(paramValues[:, :, paramNum])
        else:
            hdu = pyfits.PrimaryHDU(paramValues[paramNum, :, :])


        ## write header
        FrequencyFlag = "false"
        CentralPixelOfFrequency = 0
        UnitParm = task_myXCLASS.GetUnitParameter(NameParam)
        bunitinfo = [UnitParm, 'parameter ' + NameParam]
        task_myXCLASSMapFit.WriteFitsHeader(hdu, BSCALEALL[CubeNumber], BZEROALL[CubeNumber], bunitinfo, RestFreqLocal, lengthALL[CubeNumber], \
                                            CentralPixelOfRegionX, CentralPixelOfRegionY, NumXPixels, NumYPixels, CRVALListALL[CubeNumber], \
                                            CRPIXListALL[CubeNumber], CDELTListALL[CubeNumber], CROTListALL[CubeNumber], CUNITListALL[CubeNumber], \
                                            FrequencyFlag, CentralPixelOfFrequency, NameOfFunction)
        pyfits.writeto(paramFileName, numpy.float32(hdu.data), hdu.header, clobber=True)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write new header and new data to new fits file for model function values
    for CubeNumber in xrange(NumberSpectra):                                                ## loop over all data cubes


        ## prepare array for fits file
        NumFrequencies = NumFrequenciesALL[CubeNumber]
        if (lengthFITSDATAALL[CubeNumber] == lengthALL[CubeNumber]):
            ModelValuesAll = numpy.zeros((NumXPixels, NumYPixels, NumFrequencies), dtype=numpy.float32)
        else:
            ModelValuesAll = numpy.zeros((NumFrequencies, NumYPixels, NumXPixels), dtype=numpy.float32)


        ## the following part take the different order for each cube into account
        counterX = (-1)                                                                     ## reset counter for x pixels
        for xPoints in RegionPixelsX:                                                       ## loop over all columns of the selected region
            counterX += 1                                                                   ## increase counter for x pixels
            xPixels = xPoints[0] * BSCALE + BZERO                                           ## apply scale factor and zero offset ((FITS) for x pixel
            xIndex = counterX
            counterY = (-1)                                                                 ## reset counter for y pixels
            for yPoints in RegionPixelsY:                                                   ## loop over all rows of the selected region
                counterY += 1                                                               ## increase counter for y pixels
                yPixels = yPoints[0] * BSCALE + BZERO                                       ## apply scale factor and zero offset ((FITS) for y pixel
                yIndex = counterY
                for freq_index in xrange(NumFrequenciesALL[CubeNumber]):                    ## loop over all freq. points
                    if (lengthFITSDATAALL[CubeNumber] == lengthALL[CubeNumber]):
                        ModelValuesAll[xIndex, yIndex, freq_index] = ModelValues[xIndex, yIndex, freq_index, CubeNumber]
                    else:
                        ModelValuesAll[freq_index, yIndex, xIndex] = ModelValues[xIndex, yIndex, freq_index, CubeNumber]


        ## write new header and new data to new fits file for model values
        ModelFunctionFileName = NameOfCubeAll[CubeNumber]
        hdu = pyfits.PrimaryHDU(ModelValuesAll)
        FrequencyFlag = "true"
        if (CDELTListALL[CubeNumber][2] > 0):
            FreqLowestRaw = ValueOfFirstPointListALL[CubeNumber][2]
            FreqHighestRaw = ValueOfFirstPointListALL[CubeNumber][2] + ((lengthALL[CubeNumber][2] - 1) * CDELTListALL[CubeNumber][2])
        else:
            FreqLowestRaw = ValueOfFirstPointListALL[CubeNumber][2] + ((lengthALL[CubeNumber][2] - 1) * CDELTListALL[CubeNumber][2])
            FreqHighestRaw = ValueOfFirstPointListALL[CubeNumber][2]


        if (CDELTListALL[CubeNumber][2] < 0.0):
            CentralPixelOfFrequency = max(FreqLowestRaw, FreqHighestRaw)
        else:
            CentralPixelOfFrequency = min(FreqLowestRaw, FreqHighestRaw)
        bunitinfo = [BUNITALL[CubeNumber], 'Brightness (pixel) unit']
        task_myXCLASSMapFit.WriteFitsHeader(hdu, BSCALEALL[CubeNumber], BZEROALL[CubeNumber], bunitinfo, RestFreqLocal, \
                                            lengthALL[CubeNumber], CentralPixelOfRegionX, CentralPixelOfRegionY, NumXPixels, NumYPixels, \
                                            CRVALListALL[CubeNumber], CRPIXListALL[CubeNumber], CDELTListALL[CubeNumber], CROTListALL[CubeNumber], \
                                            CUNITListALL[CubeNumber], FrequencyFlag, CentralPixelOfFrequency, NameOfFunction)
        pyfits.writeto(ModelFunctionFileName.replace(".fits", "") + "__RedoFit.fits", numpy.float32(hdu.data), hdu.header, clobber=True)


    ## we've done
    sys.stdout.write("\rAnalyze results of fits and create FITS files .. done!                                                                  \n")
    sys.stdout.flush()


    ##====================================================================================================================================================
    ## print some informations to screen
    print " "
    print " "
    print "Please note, all files created by the current fit process are stored in the"
    print "myXCLASSMapRedoFit  working directory " + chr(34) + MAGIXjobDir + chr(34) + "!"
    print " "
    print " "


    ##====================================================================================================================================================
    ## define return variables
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------

