#!/usr/bin/python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module contains all subroutines to perform the myXCLASSMapFit function providing a simplified interface for MAGIX with myXCLASS program to
##  fit a complete data cube instead of a single spectrum
##  Copyright (C) 2012 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function CheckHourRange:                      Checks that the given value is in the range (-24, 24).
##      - function CheckMinuteRange:                    Checks that the given value is in the range [0,60].
##      - function CheckSecondRange:                    Checks that the given value is in the range [0,60].
##      - function check_hms_ranges:                    Checks that the given hour, minute and second are all within reasonable range.
##      - function t_SIGN:                              get sign of value
##      - function RAinDegree:                          converts right ascension in (hour, minutes, second) to decimal degree (sexagesimal system)
##      - function DECinDegree:                         converts declination to decimal degrees
##      - function JanskyPerBeamInKelvin:               converts Jansky in Kelvin and vis versa
##      - function ConvertUnit:                         determine unit conversion factor
##      - function LoadFits:                            load contents of FITS-file
##      - function WriteFitsHeader:                     write new header for fits file
##      - function ReadRegionFile:                      get pixels which are located within the given region
##      - function ReadClusterdefFile:                  reads in file that contains cluster definition
##      - function SSHStarter:                          starts worker via command line
##      - function StartClusterParallelization:         start different MAGIX instances via the command line on the cluster
##      - function GetMolfitFileOfBestResult:           determines the best fit result from a previous pixel fit
##      - function ReadParmXMLFile:                     read parameter values from xml file
##      - function ReadModelFiles:                      read model function values from file(s)
##      - function myXCLASSMapFit:                      Simplified CASA interface for MAGIX with myXCLASS program to fit a complete data cube instead of
##                                                      a single spectrum
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    25.07.2013   initial version
##  T. Moeller    09.04.2014   improved version
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import string                                                                               ## import string package
import shutil                                                                               ## import shutil package
import random                                                                               ## import random package
import datetime                                                                             ## import datetime package
import time                                                                                 ## import package date and time
import warnings                                                                             ## import warnings package
from math import *                                                                          ## import math package
import sqlite3                                                                              ## import sqlite3 package
import logging                                                                              ## import logging package
import copy                                                                                 ## import copy package
import Queue                                                                                ## import Queue package
import threading                                                                            ## import threading package
import multiprocessing                                                                      ## import multiprocessing package
import subprocess                                                                           ## import subprocess package
import matplotlib                                                                           ## import matplotlib package
matplotlib.use("Agg")                                                                       ## avoid display error
import pylab                                                                                ## import pylab package
import task_MAGIX                                                                           ## import package MAGIX
import task_myXCLASS                                                                        ## import package myXCLASS
import task_myXCLASSFit                                                                     ## import package myXCLASSFit


## import pyfits package via astropy
try:
    import astropy.io.fits as pyfits
except ImportError:
    import pyfits
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## load contents of FITS-file
##
"""
Collection of subroutines and functions to convert sky coordinates to pixel


Partly taken from the astropy project!

https://github.com/astropy/astropy/blob/master/astropy/coordinates/angle_utilities.py

call 10.10.2013
"""
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check hour range (Checks that the given value is in the range (-24, 24).)
##
def CheckHourRange(hrs):
    """

input parameters:
-----------------

    - hrs:                  hour



output parameters:
------------------

    - None


    """
    if (abs(hrs) == 24.):
        print "Treating hour as 24 hr"
    elif (hrs < -24. or hrs > 24.):
        print "Error: Hours not in range (-24, 24): Hours = ", hrs


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check minute range (Checks that the given value is in the range [0,60]. If the value is equal to 60, then a warning is raised.)
##
def CheckMinuteRange(m):
    """

input parameters:
-----------------

    - m:                    minute



output parameters:
------------------

    - None


    """
    if (m == 60.):
        print "Treating minute as 0 min, +1 hr/deg"
    elif (m < 0. or m > 60.):
        print "Error: Minutes not in range [0,60): Minute = ", m


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check second range (Checks that the given value is in the range [0,60]. If the value is equal to 60, then a warning is raised.)
##
def CheckSecondRange(sec):
    """

input parameters:
-----------------

    - sec:                  second



output parameters:
------------------

    - None


    """
    if (sec == 60.):
        print "Treating as 0 sec, +1 min"
    elif (sec < 0. or sec > 60.):
        print "Error: Seconds not in range [0,60): Second = ", sec


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check hours, minutes and seconds ranges (Checks that the given hour, minute and second are all within reasonable range.)
##
def check_hms_ranges(h, m, s):
    """

input parameters:
-----------------

    - h:                    hour

    - m:                    minutes

    - s:                    seconds



output parameters:
------------------

    - None


    """
    CheckHourRange(h)
    CheckMinuteRange(m)
    CheckSecondRange(s)


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get sign of value
##
def t_SIGN(value):
    """

input parameters:
-----------------

    - value:                value



output parameters:
------------------

    - t:                    sign of value


    """
    if (value < 0):
        t = -1.0
    else:
        t = 1.0


    ## define return parameter
    return t
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## convert declination to decimal degrees
## http://en.wikipedia.org/wiki/Right_ascension, call 15.10.2013
## 1 h = 15\circ = pi/12 rad
## 1 minute = 15' = 1/4\circ = pi/720 rad
## 1 second = 15" = 1/4' = 1/240\circ = pi/43200 rad
##
def RAinDegree(h, m, s):
    """

input parameters:
-----------------

    - h:                    hour

    - m:                    minutes

    - s:                    seconds



output parameters:
------------------

    - DecDegree:            decimal degrees


    """
    sign = t_SIGN(h)
    DecDegree = sign * (h * 15 + m * 15 / 60.0 + s * 15 / 3600.0)


    ## define return parameter
    return DecDegree
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## convert declination to decimal degrees
##
def DECinDegree(degree, minutes, seconds):
    """

input parameters:
-----------------

    - degree:               degree

    - minutes:              minutes

    - seconds:              seconds



output parameters:
------------------

    - DecDegree:            decimal degrees


    """
    sign = t_SIGN(degree)
    DecDegree = sign * (abs(degree) + minutes / 60.0 + seconds / 3600.0)


    ## define return parameter
    return DecDegree
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## converts Jansky/Beam in Kelvin and vise versa
##
def JanskyPerBeamInKelvin(diameter, InterFlag, frequency):
    """

taken from Peter Schilke (<schilke@mpifr-bonn.mpg.de>),  Time-stamp: <2007-08-27 11:59:11 schilke>


input parameters:
-----------------

    - diameter:             telescope diameter (in m)

    - InterFlag:            flag indicating single dish (False) or interferometric (True) data

    - frequency:            frequency (in MHz)



output parameters:
------------------

    - ConvertJyPerBeam2Kelvin:     Jansky/Beam to Kelvin prefactor

    - ConvertKelvin2JyPerBeam:     Kelvin to Jansky/Beam prefactor
    """

    # Debug:
    #    print "diameter = ", diameter
    #    print "frequency = ", frequency

    c = 2.99792458e8                                                                        ## speed of light (in m/s)
    k = 1.3806488e-23                                                                       ## Boltzmann constan (in J/K)
    wavelength = c / (frequency * 1.e6)                                                     ## determine wavelength (in m)


    ## calculate beam size
    if (InterFlag == "y" or InterFlag == "t" or InterFlag == "true" or InterFlag):
        beam_arcsec = diameter                                                              ## for interferometric data, diameter describes FWHM in arcsec
        beam_radian = beam_arcsec / ((180.0 / pi) * 3600.0)                                 ## convert to radian
    else:
        beam_radian = 1.22 * wavelength / diameter                                          ## determine telescope beam FWHM size
                                                                                            ## using diffraction limit (in radian)
        beam_arcsec = beam_radian * (180.0 / pi) * 3600.0                                   ## convert beam size to arcsec


    ## calculate solid angle of the beam Omega
    Omega = pi * beam_radian**2 / (4.0 * log(2.0))                                          ## calculate solid angle of the beam omega assuming a
                                                                                            ## circular beam with Gaussian shape:
                                                                                            ## f(r) = e^{-4 \ln(2) \frac{r^2}{\theta^2}},
                                                                                            ## \Omega = \int_0^\infty 2 \pi r f(r) dr
                                                                                            ##        = \frac{\pi}{4 \ln 2} \theta^2 (\theta in radian)
    ## calculate conversion factors
    ## Dividing the flux F (in Jy) by Omega gives the flux density or intensity, which is converted to brightness temperature by applying the
    ## Rayleigh-Jeans approximation of the Planck function
    ConvertJyPerBeam2Kelvin = (wavelength**2 / (2.0 * k)) * 1.0 / Omega * 1.e-26            ## convert Jansky/Beam to Kelvin prefactor
                                                                                            ## Rayleigh-Jeans:  (wavelength**2 / (2.0 * k))
                                                                                            ## Intensity = flux F (in Jy) / Omega
    ConvertKelvin2JyPerBeam = 1.0 / ConvertJyPerBeam2Kelvin                                 ## convert Kelvin to Jansky/Beam prefactor

    # Debug:
    #    print "ConvertJyPerBeam2Kelvin = %12.6f" % (ConvertJyPerBeam2Kelvin)
    #    print "ConvertKelvin2JyPerBeam = %12.6f" % (ConvertKelvin2JyPerBeam)


    ## define return variable
    return (ConvertJyPerBeam2Kelvin, ConvertKelvin2JyPerBeam)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## determine unit conversion factor
##
def ConvertUnit(unit):
    """


input parameters:
-----------------

    - unit:                     given unit



output parameters:
------------------

    - UnitConversionFactor:     conversion factor

    """

    # Debug:
    #    print "unit = ", unit


    ## initialize return parameter
    UnitConversionFactor = 1.0
    UnitConversionFactor = 1.e-6


    ## determine conversion factor
    unit = unit.lower()                                                                     ## convert all characters to lower case
    VelocityUnitKMSList = ["kms", "km/s", "kms-1", "km s-1", "kms1", "km s1", "kms^-1", "km s^-1", "kms^(-1)", "km s^(-1)", "kms$^{-1}$", \
                           "km s$^{-1}$", "kms^{-1}", "km s^{-1}"]
    VelocityUnitMSList = ["ms", "m/s", "ms-1", "m s-1", "ms1", "m s1", "ms^-1", "m s^-1", "ms^(-1)", "m s^(-1)", "ms$^{-1}$", "m s$^{-1}$", \
                          "ms^{-1}", "m s^{-1}"]
    if (unit == "hz"):                                                                      ## Hz
        UnitConversionFactor = 1.e-6
    elif (unit == "khz"):                                                                   ## kHz
        UnitConversionFactor = 1.e-3
    elif (unit == "mhz"):                                                                   ## MHz
        UnitConversionFactor = 1.0
    elif (unit == "ghz"):                                                                   ## GHz
        UnitConversionFactor = 1.e+3
    elif (unit == "thz"):                                                                   ## THz
        UnitConversionFactor = 1.e+6
    elif (unit == "cm1" or unit == "cm-1" or unit == "cm^-1" or unit == "cm^(-1)" or unit == "cm^{-1}" or unit == "cm$^{-1}$" or unit == "1/cm"):
        UnitConversionFactor = 1.0 / 3.33565e-11 * 1e-6                                     ## from cm-1 to Hz to MHz
    elif (unit == "k"):                                                                     ## Kelvin
        UnitConversionFactor = 2.08364e+10
    elif (unit == "j"):                                                                     ## Joule
        UnitConversionFactor = 1.50930e+33
    elif (unit == "ev"):                                                                    ## eV
        UnitConversionFactor = 2.41804e+14
    elif (unit in VelocityUnitKMSList):
        UnitConversionFactor = "velocityKMS"
    elif (unit in VelocityUnitMSList or 1==1):
        UnitConversionFactor = "velocityMS"


    ## define return variable
    return UnitConversionFactor
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## load contents of FITS-file
##
def LoadFits(filename, HDU, print_flag):
    """

    function fits:          load FITS file containing experimental data

    input parameters:       filename:                   path and name of FITS file
                            HDU:                        number of HDU
                            print_flag:                 flag for printing message to screen

    output parameters:      ok:                         status of subroutine
                            NumDimensions:              dimension of data cube
                            length:                     length of data array
                            lengthFITSDATA:             lenfth of data array (fitsdata.shape version)
                            CRPIXList:                  list containing the crpix parameter
                            CRVALList:                  list containing the crval parameter
                            CDELTList:                  list containing the crdelt parameter
                            CROTList:                   list containing the crot parameter
                            CUNITList:                  list containing the cunit parameter
                            BSCALE:                     scale parameter
                            BZERO:                      zero parameter
                            BUNIT:                      unit parameter
                            RESTFRQ:                    rest frequency
                            ValueOfFirstPointList:      list containing the first pixel parameter
                            fitsdata:                   data of FITS file

    working variables:      k:                          loop counter
                            filename:                   temp variable for file name of the ith fits-file
                            hdulist:                    temp variable containing the whole ith fits-file
                            fitsdata:                   temp variable containing the data of the ith fits-file
                            HDU:                        temp variable for number of HDU in the ith fits-file
                            number_columns:             temp variable containing the total number of columns in the ith fits-file
                            dimension:                  temp variable containing the dimension of the data of the ith fits-file
                            NumDimensions:              number of dimensions
                            fitshead:                   header of the ith fits file


    CLASS understands RA--, RA ; DEC-, DEC ; GLON ; GLAT ; TIME, UT.
    For the projection system, CLASS understands
        : P_NONE      = 0 ! Unprojected data
    -TAN: P_GNOMONIC  = 1 ! Radial Tangent plane
    -SIN: P_ORTHO     = 2 ! Dixon Tangent plane
    -ARC: P_AZIMUTHAL = 3 ! Schmidt Tangent plane
    -STG: P_STEREO    = 4 ! Stereographic
    lamb: P_LAMBERT   = 5 ! Lambert equal area
    -ATF: P_AITOFF    = 6 ! Aitoff equal area
    -GLS: P_RADIO     = 7 ! Classic Single dish radio mapping
    Read Representations of celestial coordinates in FITS
    Authors: Mark R. Calabretta, Eric W. Greisen
    (Submitted on 19 Jul 2002)
    arXiv:astro-ph/0207413v1
    http://arxiv.org/abs/astro-ph/0207413

    Be careful:
    RA becomes  RA---GLS with three hyphens
    Dec becomes DEC--GLS with two hyphens.
    That's why we can also write 'RA--' and 'DEC-'
    for 'RA' and 'DEC': it's just easier to add the


    """
    ## initialize output parameters
    ok = 0
    length = []
    CRPIXList = []
    CRVALList = []
    CDELTList = []
    CROTList = []
    CUNITList = []
    ValueOfFirstPointList = []


    ## print what you do
    if (print_flag == "true"):
        print "\n\n\t Reading fits file " + filename + "..",


    ## open fits-file using pyfits
    hdulist = pyfits.open(filename)
    if (print_flag == "true"):
        print "done!"

    # Debug:
    # print "\t HDU = ",HDU
    # print hdulist.info()


    ## save data
    ## number of columns in scidata is given by len(scidata)
    fitsdata = hdulist[HDU].data


    ## verify header, fix if necessary and save header
    # hdulist.verify('fix')
    fitshead = hdulist[HDU].header

    # Debug:
    # print "\t fitsdata.shape = ",fitsdata.shape


    ## set array containing dimension of each axis
    dimension = fitsdata.shape
    NumDimensions = len(dimension)


    ## print what you do
    if (print_flag == "true"):
        print "\t Number of columns = ",NumDimensions
        print "\t dimension = ",dimension


    ## check dimension of fits file
    if (NumDimensions > 4):
        print " "
        print "\t Error in subroutine fits:"
        print "\t   Can not import fits-file with more than 4 dimensions."
        print "\t   Please select other file and restart program."
        print " "
        print "\t   Program aborted"
        ok = 1
        CParameter = [0, 0, 0, 0, ""]
        BParameter = [0, 0, ""]
        return ok, NumDimensions, length, lengthFITSDATA, CParameter, BParameter, ValueOfFirstPointList, fitsdata


    ## read header information
    ## get number of axes
    if (print_flag == "true"):
        print "\t Reading data ..",
    try:
        NAXIS = pyfits.getval(filename, 'NAXIS', 0)
    except KeyError, err:
        NAXIS = 1

    # Debug:
    # print '\n\nNAXIS = ',NAXIS


    ## get scale parameter BSCALE
    try:
        BSCALE = pyfits.getval(filename, 'BSCALE', 0)
    except KeyError, err:
        BSCALE = 1

    # Debug:
    # print '\n\nBSCALE = ',BSCALE


    ## get zero parameter BZERO
    try:
        BZERO = pyfits.getval(filename, 'BZERO', 0)
    except KeyError, err:
        BZERO = 0.0

    # Debug:
    # print '\n\nBZERO = ',BZERO

    # Debug:
    # print '\n\nBSCALE = ',BSCALE


    ## get unit parameter BUNIT
    try:
        BUNIT = pyfits.getval(filename, 'BUNIT', 0)
    except KeyError, err:
        BUNIT = ""

    # Debug:
    # print '\n\nBUNIT = ',BUNIT


    ## get rest frequency RESTFRQ
    try:
        RESTFRQ = pyfits.getval(filename, 'RESTFRQ', 0)
    except KeyError, err:
        try:
            RESTFRQ = pyfits.getval(filename, 'RESTFREQ', 0)
        except KeyError, err:
            RESTFRQ = 0
    # RESTFRQ = RESTFRQ * 1.e-6

    # Debug:
    # print '\n\nRESTFRQ = ',RESTFRQ


    ## analyze header
    length = []
    NumDimensions = 0
    for i in xrange(NAXIS):                                                                 ## loop over all dimensions
        try:
            lenAxis = pyfits.getval(filename, 'NAXIS' + str(i + 1), 0)                      ## read CRPIX
        except KeyError, err:
            lenAxis = 1
        if (lenAxis > 0):                                                                   ## read only those entries with more than 1 point
            length.append(lenAxis)
            NumDimensions += 1


            ## get header parameters
            try:
                CRPIX = pyfits.getval(filename, 'CRPIX' + str(i + 1), 0)                    ## location of a reference point along axis 1
            except KeyError, err:
                CRPIX = 1
            CRPIXList.append(CRPIX)
            try:
                CRVAL = pyfits.getval(filename, 'CRVAL' + str(i + 1), 0)                    ## the value of the coordinate at the reference point CRPIX1
            except KeyError, err:
                CRVAL = 0
            CRVALList.append(CRVAL)
            try:
                CDELT = pyfits.getval(filename, 'CDELT' + str(i + 1), 0)                    ## partial derivative of the coordinate with respect to the
            except KeyError, err:                                                           ## pixel index, evaluated at the reference point CRPIX
                CDELT = 1
            CDELTList.append(CDELT)
            try:
                CROT = pyfits.getval(filename, 'CROT' + str(i + 1), 0)                      ## indicate a rotation from a standard coordinate system
            except KeyError, err:
                CROT = 0
            CROTList.append(CROT)
            try:
                CUNIT = pyfits.getval(filename, 'CUNIT' + str(i + 1), 0)                    ## indicate the unit of the current axis
            except KeyError, err:
                CUNIT = ""
            CUNITList.append(CUNIT)


            ## calculate value at first pixel
            ValueOfFirstPoint = CRVAL - ((CRPIX - 1) * CDELT)
            ValueOfFirstPointList.append(ValueOfFirstPoint)

            # Debug:
            # print "CRPIX = ", CRPIX
            # print "CRVAL = ", CRVAL
            # print "CDELT = ", CDELT
            # print "CROT = ", CROT
            # print "CUNIT = ", CUNIT
            # print "ValueOfFirstPoint = ", ValueOfFirstPoint


    ## get length of data
    lengthFITSDATA = fitsdata.shape

    # Debug:
    # print "\n\n\nlength = ", length


    ## close fits-file
    hdulist.close()


    ## everything is fine
    if (print_flag == "true"):
        print "done!"


    ## define return parameters
    CParameter = [CRPIXList, CRVALList, CDELTList, CROTList, CUNITList]
    BParameter = [BSCALE, BZERO, BUNIT, RESTFRQ]
    return ok, NumDimensions, length, lengthFITSDATA, CParameter, BParameter, ValueOfFirstPointList, fitsdata
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## write new header for fits file
##
def WriteFitsHeader(hdu, BSCALE, BZERO, BUNIT, RestFreq, length, CentralPixelOfRegionX, CentralPixelOfRegionY, NumXPixels, NumYPixels, \
                    CRVALList, CRPIXList, CDELTList, CROTList, CUNITList, FrequencyFlag, CentralPixelOfFrequency, NameOfFunction):
    """

input parameters:
-----------------

    - hdu:                  hdu

    - BSCALE:               bscale

    - BZERO:                bzero

    - RestFreq:             rest frequency

    - length:               number of pixel for each direction of the region

    - CentralPixelOfRegionX:x coordinate of region center

    - CentralPixelOfRegionY:y coordinate of region center

    - NumXPixels:           number of pixels in x direction

    - NumYPixels:           number of pixels in y direction

    - CRVALList:            value of first pixel

    - CRPIXList:            index of first pixel

    - CDELTList:            stepsize for each axis

    - CROTList:             rotation angle

    - CUNITList:            unit for each axis

    - FrequencyFlag:        flag indicates if frequency is considered or not

    - CentralPixelOfFrequency: central frequency point (only used for FrequencyFlag == true)

    - NameOfFunction:       name of function


output parameters:
------------------

    - None
    """

    # Debug:
    #    print "\n\n\n\n\n\n\n\n\n\n\n"
    #    print "BSCALE = ", BSCALE
    #    print "BZERO = ", BZERO
    #    print "BUNIT = ", BUNIT
    #    print "RestFreq = ", RestFreq
    #    print "\n\n\n\n\n\n\n\n\n\n\n"


    ## select FITS command for header update
    try:
        import astropy.io.fits as pyfits
        OldFITSCommandFlag = False
    except ImportError:
        OldFITSCommandFlag = True


    ## select kind of header
    UseCentralPixelFlag = True

    # Debug:
    # print "\n\n\nhdu.header = ", hdu.header


    # hdu.header.update('BTYPE', ' K      ', 'Beam temperature')
    if (OldFITSCommandFlag):
        hdu.header.update('OBJECT', '        ', '')
    else:
        hdu.header['OBJECT'] = ('        ', '')

    if (OldFITSCommandFlag):
        hdu.header.update('BSCALE', BSCALE, 'PHYSICAL = PIXEL * BSCALE + BZERO')
    else:
        hdu.header['BSCALE'] = (BSCALE, 'PHYSICAL = PIXEL * BSCALE + BZERO')

    if (OldFITSCommandFlag):
        hdu.header.update('BZERO', BZERO, '')
    else:
        hdu.header['BZERO'] = (BZERO, '')

    if (OldFITSCommandFlag):
        hdu.header.update('BUNIT', BUNIT[0], BUNIT[1])
    else:
        hdu.header['BUNIT'] = (BUNIT[0], BUNIT[1])

    if (OldFITSCommandFlag):
        hdu.header.update('RESTFRQ', RestFreq, '')
    else:
        hdu.header['RESTFRQ'] = (RestFreq, '')

    if (OldFITSCommandFlag):
        hdu.header.update('software', NameOfFunction, 'Produced with XCLASS interface for CASA')
    else:
        hdu.header['software'] = (NameOfFunction, 'Produced with XCLASS interface for CASA')

    ii = len(length)
    if (FrequencyFlag == "true"):
        ii = 3
    else:
        ii = 2
    for i in xrange(ii):
        if (UseCentralPixelFlag):
            if (i == 0):
                if (OldFITSCommandFlag):
                    hdu.header.update('CTYPE1', 'RA---SIN', '')
                else:
                    hdu.header['CTYPE1'] = ('RA---SIN', '')

                crval = (CentralPixelOfRegionX + CDELTList[i] / 2.0)
                # crval = CentralPixelOfRegionX
                if (OldFITSCommandFlag):
                    hdu.header.update('CRVAL' + str(i + 1), crval, '')
                else:
                    hdu.header['CRVAL' + str(i + 1)] = (crval, '')

                crpix = int(NumXPixels / 2.0 + 1.0)
                if (OldFITSCommandFlag):
                    hdu.header.update('CRPIX' + str(i + 1), crpix, '')
                else:
                    hdu.header['CRPIX' + str(i + 1)] = (crpix, '')

            elif (i == 1):
                if (OldFITSCommandFlag):
                    hdu.header.update('CTYPE2', 'DEC--SIN', '')
                else:
                    hdu.header['CTYPE2'] = ('DEC--SIN', '')

                crval = (CentralPixelOfRegionY + CDELTList[i] / 2.0)
                # crval = CentralPixelOfRegionY
                if (OldFITSCommandFlag):
                    hdu.header.update('CRVAL' + str(i + 1), crval, '')
                else:
                    hdu.header['CRVAL' + str(i + 1)] = (crval, '')

                crpix = int(NumYPixels / 2.0 + 1.0)
                if (OldFITSCommandFlag):
                    hdu.header.update('CRPIX' + str(i + 1), crpix, '')
                else:
                    hdu.header['CRPIX' + str(i + 1)] = (crpix, '')

            elif (i == 2):
                if (OldFITSCommandFlag):
                    hdu.header.update('CTYPE3', 'FREQ    ', '')
                else:
                    hdu.header['CTYPE3'] = ('FREQ    ', '')

                crval = (CentralPixelOfFrequency)
                # crval = (CentralPixelOfFrequency + CDELTList[i] / 2.0)
                if (OldFITSCommandFlag):
                    hdu.header.update('CRVAL' + str(i + 1), crval, '')
                else:
                    hdu.header['CRVAL' + str(i + 1)] = (crval, '')

                crpix = (i - 1)
                # crpix = (NumFrequencies / 2.0 + 1.0)
                if (OldFITSCommandFlag):
                    hdu.header.update('CRPIX' + str(i + 1), crpix, '')
                else:
                    hdu.header['CRPIX' + str(i + 1)] = (crpix, '')

            else:
                crval = (CRVALList[i])
                if (OldFITSCommandFlag):
                    hdu.header.update('CRVAL' + str(i + 1), crval, '')
                else:
                    hdu.header['CRVAL' + str(i + 1)] = (crval, '')

                crpix = (CRPIXList[i])
                if (OldFITSCommandFlag):
                    hdu.header.update('CRPIX' + str(i + 1), crpix, '')
                else:
                    hdu.header['CRPIX' + str(i + 1)] = (crpix, '')

            cdelt = (CDELTList[i])
            if (OldFITSCommandFlag):
                hdu.header.update('CDELT' + str(i + 1), cdelt, '')
            else:
                hdu.header['CDELT' + str(i + 1)] = (cdelt, '')


        ## the header uses always the first pixel as reference point
        else:
            if (i < 3):
                crval = CRVALList[i] - ((CRPIXList[i] - 1) * CDELTList[i])
                crpix = 1
                cdelt = abs(CDELTList[i])
            else:
                crval = CRVALList[i]
                crpix = CRPIXList[i]
                cdelt = CDELTList[i]
            if (i == 0):
                if (OldFITSCommandFlag):
                    hdu.header.update('CTYPE1', 'RA---SIN', '')
                else:
                    hdu.header['CTYPE1'] = ('RA---SIN', '')
            elif (i == 1):
                if (OldFITSCommandFlag):
                    hdu.header.update('CTYPE2', 'DEC--SIN', '')
                else:
                    hdu.header['CTYPE2'] = ('DEC--SIN', '')
            elif (i == 2):
                if (OldFITSCommandFlag):
                    hdu.header.update('CTYPE3', 'FREQ    ', '')
                else:
                    hdu.header['CTYPE3'] = ('FREQ    ', '')
            if (OldFITSCommandFlag):
                hdu.header.update('CRVAL' + str(i + 1), crval, '')
            else:
                hdu.header['CRVAL' + str(i + 1)] = (crval, '')
            if (OldFITSCommandFlag):
                hdu.header.update('CRPIX' + str(i + 1), crpix, '')
            else:
                hdu.header['CRPIX' + str(i + 1)] = (crpix, '')
            if (OldFITSCommandFlag):
                hdu.header.update('CDELT' + str(i + 1), cdelt, '')
            else:
                hdu.header['CDELT' + str(i + 1)] = (cdelt, '')

        crot = (CROTList[i])
        if (OldFITSCommandFlag):
            hdu.header.update('CROT' + str(i + 1), crot, '')
        else:
                hdu.header['CROT' + str(i + 1)] = (crot, '')
        if (OldFITSCommandFlag):
            hdu.header.update('CUNIT' + str(i + 1), CUNITList[i], '')
        else:
            hdu.header['CUNIT' + str(i + 1)] = (CUNITList[i], '')
    lt = time.localtime()
    TimeStamp = time.strftime("%d-%m-%Y", lt) + "__" + time.strftime("%H-%M-%S", lt)
    if (OldFITSCommandFlag):
        hdu.header.update('DATE', TimeStamp, 'Date FITS file was written')
    else:
        hdu.header['DATE'] = (TimeStamp, 'Date FITS file was written')


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## analyze the region file and determine the pixels within the given region
##
def ReadRegionFile(regionFileName, length, ValueOfFirstPointList, CDELTList, CRVALList, CRPIXList):
    """

input parameters:
-----------------

    - regionFileName:       path and name of the region file

    - length:               number of pixels for each direction

    - ValueOfFirstPointList:    list of first pixel for each direction

    - CDELTList:            stepsize for each direction

    - CRVALList:            value of reference pixel

    - CRPIXList:            number of reference pixel


output parameters:
------------------

    - RegionPixel:          list containing all pixel which are located within the given region

    - TotalNumberPixels:    total number of pixels within the given region

    - RealLength:           number of pixel for each direction of the region

    - RegionMask:           indicates for each pixel if pixel is considered or not (necessarry for regions with shapes different from box)

    - CentralPixelOfRegionX:x coordinate of region center (not used at the moment)

    - CentralPixelOfRegionY:y coordinate of region center (not used at the moment)

    """

    # Debug:
    # print "\n\n\n"
    # print "regionFileName = ", regionFileName
    # print "length = ", length


    ## analyze input parameter regionFileName
    CoordinateSystem = "fk5"
    RegionPixel = []
    RealLength = []
    TotalNumberPixels = 0
    regionFileName = regionFileName.strip()
    if (regionFileName != None and regionFileName != ""):                                   ## check if parameter regionFileName is defined


        ## read in contents of regionFileName
        regionFile = open(regionFileName)
        regionFileContents = regionFile.readlines()
        regionFile.close()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## analyze contents of regionFile
        FileFormat = "ds9"
        RegionShape = "box"
        for line in regionFileContents:
            StrippedLine = line.strip()

            # Debug:
            # print "StrippedLine = ", StrippedLine


            ## check if line is a comment line
            if (StrippedLine[0] != "#"):


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## determine format of the coordinates
                if (StrippedLine[:8] == "physical"):
                    CoordinateSystem = "physical"
                elif (StrippedLine[:3] == "fk4"):
                    CoordinateSystem = "fk4"
                elif (StrippedLine[:3] == "fk5"):
                    CoordinateSystem = "fk5"
                elif (StrippedLine[:4] == "icrs"):
                    CoordinateSystem = "icrs"
                elif (StrippedLine[:8] == "galactic"):
                    CoordinateSystem = "galactic"
                elif (StrippedLine[:8] == "ecliptic"):
                    CoordinateSystem = "ecliptic"

                # Debug:
                # print "CoordinateSystem = ", CoordinateSystem
                # print "StrippedLine = ", StrippedLine


                ## a box is defined in a ds9 file
                elif (StrippedLine[:4]  == "box("):                                         ## a box is defined in a ds9 file
                    leftBracket = StrippedLine.find("(")                                    ## get index of left bracket
                    rightBracket = StrippedLine.find(")")                                   ## get index of right bracket
                    Coordinates = StrippedLine[(leftBracket + 1):rightBracket]              ## get pure coordinates
                    CoordinatesSplittedLine = Coordinates.split(",")                        ## split string

                    # Debug:
                    # print "StrippedLine = ", StrippedLine
                    # print "leftBracket = ", leftBracket
                    # print "rightBracket = ", rightBracket
                    # print "Coordinates = ", Coordinates


                    ## check, if coordinates are given in degrees or sexagesimal
                    i = StrippedLine.find(":")                                              ## the ":" indicates the sexagesimal system
                    if (i > (-1)):
                        degree_flag = "false"
                    else:
                        degree_flag = "true"


                        ## one more bug fix of CASA: In contrast to the real ds9 region format, the CASA viewer replaces the "." by "," as well.
                        ## So, check, if CoordinatesSplittedLine[0] contains the "." character. If yes, the region file comes from ds9 itself and no
                        ## further bug-fix is needed
                        #    NeedBugFix = "true"
                        #    i = CoordinatesSplittedLine[0].find(".")
                        #    if (i > (-1)):
                        #        NeedBugFix = "false"

                        """
                        else:
                            for inNumbers in CoordinatesSplittedLine:                       ## loop over all integer numbers
                                if (counterNumbers == 1):
                                    NumberString = inNumbers
                                elif (counterNumbers == 2):
                                    NumberString += "." + inNumbers
                                    Values.append(float(NumberString))
                                    counterNumbers = 0
                                    NumberString = ""
                        """
                    # Debug:
                    # print "degree_flag = ", degree_flag


                    ## get box center, height and width
                    counterNumbers = 0
                    Values = []
                    for inNumbers in CoordinatesSplittedLine:                               ## loop over all
                        counterNumbers += 1

                        # Debug:
                        # print "inNumbers = ", inNumbers


                        ## check, if value is given in minutes, i.e. string 'inNumbers' contains the character '''
                        if (inNumbers.find(chr(39)) > (-1)):                                ## is "'" contained in the current string?
                            minutes = float(inNumbers.replace(chr(39), ''))

                            # Debug:
                            # print "minutes = ", minutes

                            check_hms_ranges(0, minutes, 0)
                            degree = minutes / 60.0
                            Values.append(degree)


                        ## check, if value is given in seconds, i.e. string 'inNumbers' contains the character '"'
                        elif (inNumbers.find(chr(34)) > (-1)):                              ## is '"' contained in the current string?
                            seconds = float(inNumbers.replace(chr(34), ''))

                            # Debug:
                            # print "seconds = ", seconds

                            check_hms_ranges(0, 0, seconds)
                            degree = seconds / 3600.0
                            Values.append(degree)


                        ## convert value to degree, if values are given in sexagesimal system
                        elif (degree_flag == "false"):                                      ## values are given in sexagesimal system
                            if (counterNumbers < 3):
                                SplittedLine = inNumbers.split(":")
                                hours = int(SplittedLine[0])
                                minutes = int(SplittedLine[1])
                                seconds = float(SplittedLine[2])
                                check_hms_ranges(hours, minutes, seconds)

                                # Debug:
                                # print "hours, minutes, seconds = ", hours, minutes, seconds


                                ## convert values to degrees for coordinate system: fk5
                                if (counterNumbers == 1):
                                    inNumbers = RAinDegree(hours, minutes, seconds)
                                elif (counterNumbers == 2):
                                    degree = hours
                                    inNumbers = DECinDegree(degree, minutes, seconds)
                                Values.append(inNumbers)
                            else:
                                Values.append(float(inNumbers))


                        ## the coordinates are given in degree, nothing to do
                        elif (degree_flag == "true"):                                       ## values are given in degrees
                            Values.append(float(inNumbers))

                        # Debug:
                        # print "Values = ", Values


                    ## determine upper left and lower right corner of box
                    ## Values[0] = x coord. of center pixel
                    ## Values[1] = y coord. of center pixel
                    ## Values[2] = width of box
                    ## Values[3] = height of box
                    CentralPixelOfRegionX = Values[0]
                    CentralPixelOfRegionY = Values[1]
                    xULCorner = (Values[0] - Values[2]/2.0)
                    yULCorner = (Values[1] - Values[3]/2.0)
                    xLRCorner = (Values[0] + Values[2]/2.0)
                    yLRCorner = (Values[1] + Values[3]/2.0)
                    FileFormat = "ds9"
                    RegionShape = "box"

                    # Debug:
                    # print "\n"
                    # print "xULCorner = ", xULCorner
                    # print "yULCorner = ", yULCorner
                    # print "xLRCorner = ", xLRCorner
                    # print "yLRCorner = ", yLRCorner
                    # print "FileFormat = ", FileFormat
                    # print "RegionShape = ", RegionShape
                    # print "CentralPixelOfRegionX = ", CentralPixelOfRegionX
                    # print "CentralPixelOfRegionY = ", CentralPixelOfRegionY
                    break                                                                   ## if more than one region is defined in the region file,


                ## a box is defined in a CASA file
                elif (StrippedLine[:6] == "box [["):                                        ## a box is defined in a CASA region file
                    leftsqBracket = StrippedLine.find("[[")                                 ## get index of left double squared brackets
                    rightsqBracket = StrippedLine.find("]]")                                ## get index of right double squared brackets
                    KommaInBetween = StrippedLine.find("], [")                              ## get index of the phrase "], ["
                    FirstCoordinates = StrippedLine[(leftsqBracket + 2):KommaInBetween]     ## get first coordinates
                    SecondCoordinates = StrippedLine[(KommaInBetween + 4):rightsqBracket]   ## get first coordinates
                    FirstCoordinatesSplittedLine = FirstCoordinates.split(",")              ## split string
                    SecondCoordinatesSplittedLine = SecondCoordinates.split(",")            ## split string


                    ## check, if coordinates are given in degrees or sexagesimal
                    i = StrippedLine.find(":")                                              ## the ":" indicates the sexagesimal system
                    if (i > (-1)):
                        degree_flag = "false"
                    else:
                        degree_flag = "true"

                    # Debug:
                    # print "StrippedLine = ", StrippedLine


                    ## get x-coordinate or upper left corner
                    SplittedLine = FirstCoordinatesSplittedLine[0].split(":")
                    hours = int(SplittedLine[0])
                    minutes = int(SplittedLine[1])
                    seconds = float(SplittedLine[2])
                    check_hms_ranges(hours, minutes, seconds)
                    xULCorner = RAinDegree(hours, minutes, seconds)                         ## convert values to degrees for coordinate system: fk5

                    # Debug:
                    # print "xULCorner = ", xULCorner


                    ## get y-coordinate or upper left corner
                    SplittedLine = FirstCoordinatesSplittedLine[1].split(".")
                    degree = int(SplittedLine[0])
                    minutes = int(SplittedLine[1])
                    seconds = int(SplittedLine[2])
                    if (len(SplittedLine) > 2):
                        seconds = float(SplittedLine[2] + "." + SplittedLine[3])
                    yULCorner = DECinDegree(degree, minutes, seconds)                       ## convert values to degrees for coordinate system: fk5

                    # Debug:
                    # print "degree, minutes, seconds = ", degree, minutes, seconds
                    # print "yULCorner = ", yULCorner


                    ## get x-coordinate or lower right corner
                    SplittedLine = SecondCoordinatesSplittedLine[0].split(":")
                    hours = int(SplittedLine[0])
                    minutes = int(SplittedLine[1])
                    seconds = float(SplittedLine[2])
                    check_hms_ranges(hours, minutes, seconds)
                    xLRCorner = RAinDegree(hours, minutes, seconds)                         ## convert values to degrees for coordinate system: fk5

                    # Debug:
                    # print "xLRCorner = ", xLRCorner


                    ## get y-coordinate or lower right corner
                    SplittedLine = SecondCoordinatesSplittedLine[1].split(".")
                    degree = int(SplittedLine[0])
                    minutes = int(SplittedLine[1])
                    seconds = int(SplittedLine[2])
                    if (len(SplittedLine) > 2):
                        seconds = float(SplittedLine[2] + "." + SplittedLine[3])
                    yLRCorner = DECinDegree(degree, minutes, seconds)                       ## convert values to degrees for coordinate system: fk5

                    # Debug:
                    # print "yLRCorner = ", yLRCorner


                    ## determine coordinates of central pixel of region
                    CentralPixelOfRegionX = 0
                    CentralPixelOfRegionY = 0


                    ## define file format and region shape
                    FileFormat = "CASA"
                    RegionShape = "box"

                    # Debug:
                    # print "xULCorner = ", xULCorner
                    # print "yULCorner = ", yULCorner
                    # print "xLRCorner = ", xLRCorner
                    # print "yLRCorner = ", yLRCorner
                    # print "FileFormat = ", FileFormat
                    # print "RegionShape = ", RegionShape
                    # print "CentralPixelOfRegionX = ", CentralPixelOfRegionX
                    # print "CentralPixelOfRegionY = ", CentralPixelOfRegionY
                    break


                ## a polygon defined in a ds9 file
                elif (StrippedLine[:8] == "polygon("):                                      ## a polygon is defined in a ds9 file
                    leftBracket = StrippedLine.find("(")                                    ## get index of left bracket
                    rightBracket = StrippedLine.find(")")                                   ## get index of right bracket
                    Coordinates = StrippedLine[(leftBracket + 1):rightBracket]              ## get pure coordinates

                    # Debug:
                    # print "StrippedLine = ", StrippedLine
                    # print "leftBracket = ", leftBracket
                    # print "rightBracket = ", rightBracket
                    # print "Coordinates = ", Coordinates

                    ## has to be filled ...
                    print "\n\n\t\tError in myXCLASSMapFit function:"
                    print "\t\tA region file defining a polygon is not supported at the moment!"


                    ## determine coordinates of central pixel of region
                    CentralPixelOfRegionX = 0
                    CentralPixelOfRegionY = 0


                    ## define file format and region shape
                    FileFormat = "ds9"
                    RegionShape = "polygon"
                    break


                ## a circle defined in a ds9 file
                elif (StrippedLine[:7] == "circle("):                                       ## a circle is defined in a ds9 file
                    leftBracket = StrippedLine.find("(")                                    ## get index of left bracket
                    rightBracket = StrippedLine.find(")")                                   ## get index of right bracket
                    Coordinates = StrippedLine[(leftBracket + 1):rightBracket]              ## get pure coordinates

                    # Debug:
                    # print "StrippedLine = ", StrippedLine
                    # print "leftBracket = ", leftBracket
                    # print "rightBracket = ", rightBracket
                    # print "Coordinates = ", Coordinates

                    ## has to be filled ...
                    print "\n\n\t\tError in myXCLASSMapFit function:"
                    print "\t\tA region file defining a circle is not supported at the moment!"


                    ## determine coordinates of central pixel of region
                    CentralPixelOfRegionX = 0
                    CentralPixelOfRegionY = 0


                    ## define file format and region shape
                    FileFormat = "ds9"
                    RegionShape = "polygon"
                    break


                ## a ellipse defined in a ds9 file
                elif (StrippedLine[:8] == "ellipse("):                                      ## a ellipse is defined in a ds9 file
                    leftBracket = StrippedLine.find("(")                                    ## get index of left bracket
                    rightBracket = StrippedLine.find(")")                                   ## get index of right bracket
                    Coordinates = StrippedLine[(leftBracket + 1):rightBracket]              ## get pure coordinates

                    # Debug:
                    # print "StrippedLine = ", StrippedLine
                    # print "leftBracket = ", leftBracket
                    # print "rightBracket = ", rightBracket
                    # print "Coordinates = ", Coordinates

                    ## has to be filled ...
                    print "\n\n\t\tError in myXCLASSMapFit function:"
                    print "\t\tA region file defining an ellipse is not supported at the moment!"


                    ## determine coordinates of central pixel of region
                    CentralPixelOfRegionX = 0
                    CentralPixelOfRegionY = 0


                    ## define file format and region shape
                    FileFormat = "ds9"
                    RegionShape = "ellipse"
                    break


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## depending on the file format and the region shape determine relevant pixel


        ## for box shaped regions
        if (RegionShape == "box"):
            RegionMask = numpy.ones((length[1], length[0]), dtype=numpy.int)                ## create array for all pixels of the selected region


            ## collect all x coordinates of region
            RegionPixelsX = []
            for xx in xrange(length[0]):                                                    ## loop over all pixel along the x direction
                x = xx
                ValueOfFirstPoint1 = ValueOfFirstPointList[0]
                xPixels = (ValueOfFirstPoint1 + (x * CDELTList[0]))


                ## determine if current x coordinate is located within the given region
                region_flag = "false"
                if (CoordinateSystem == "physical"):
                    if (x >= (xULCorner - 1) and x <= (xLRCorner - 1)):
                        region_flag = "true"
                else:
                    if (xPixels >= xULCorner and xPixels <= xLRCorner):
                        region_flag = "true"


                ## identify if current x position is located within box
                if (region_flag == "true"):
                    RegionPixelsX.append([xPixels, x])


            ## collect all y coordinates of region
            RegionPixelsY = []
            for yy in xrange(length[1]):                                                    ## loop over all pixel along the y direction
                y = yy
                ValueOfFirstPoint2 = ValueOfFirstPointList[1]
                yPixels = (ValueOfFirstPoint2 + (y * CDELTList[1]))


                ## determine if current x coordinate is located within the given region
                region_flag = "false"
                if (CoordinateSystem == "physical"):
                    if (y >= (yULCorner - 1) and y <= (yLRCorner - 1)):
                        region_flag = "true"
                else:
                    if (yPixels >= yULCorner and yPixels <= yLRCorner):
                        region_flag = "true"


                ## identify if current x position is located within box
                if (region_flag == "true"):
                    RegionPixelsY.append([yPixels, y])


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## for circle shaped regions
        if (RegionShape == "circle"):
            RegionMask = numpy.zeros((length[1], length[0]), dtype=numpy.int)               ## create array for all pixels of the selected region



    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## if parameter regionFileName is not defined include all pixel
    else:
        RegionMask = numpy.ones((length[1], length[0]), dtype=numpy.int)                    ## create array for all pixels of the selected region
                                                                                            ## (here, no region is given, i.e. use all pixels)

        ## define x coordinate of central pixel
        xx = int(length[0] / 2.0 + 1.0)
        CentralPixelOfRegionX = ValueOfFirstPointList[0] + ((xx - 1) * CDELTList[0]) - (CDELTList[0] / 2.0)

        # Debug:
        #    print "\n\n\n\n\n\n\n"
        #    print "xx = ", xx
        #    print "ValueOfFirstPointList[0] = ", ValueOfFirstPointList[0]
        #    print "corrected CentralPixelOfRegionX = ", CentralPixelOfRegionX
        #    print "CentralPixelOfRegionX = ", CentralPixelOfRegionX + (CDELTList[0] / 2.0)
        #    print "\n\n\n\n\n\n\n"


        ## define y coordinate of central pixel
        yy = int(length[1] / 2.0 + 1.0)
        CentralPixelOfRegionY = ValueOfFirstPointList[1] + ((yy - 1) * CDELTList[1]) - (CDELTList[1] / 2.0)

        # Debug:
        # print "CentralPixelOfRegionX = ", CentralPixelOfRegionX
        # print "CentralPixelOfRegionY = ", CentralPixelOfRegionY
        # print "CRVALList[0] = ", CRVALList[0]
        # print "CRVALList[1] = ", CRVALList[1]


        ## collect all x coordinates of region
        RegionPixelsX = []
        for xx in xrange(length[0]):                                                        ## loop over all pixel along the x direction
            ValueOfFirstPoint1 = ValueOfFirstPointList[0]                                   #- (CDELTList[0] / 2.0)
            xPixels = (ValueOfFirstPoint1 + (xx * CDELTList[0]))
            RegionPixelsX.append([xPixels, xx])


        ## collect all y coordinates of region
        RegionPixelsY = []
        for yy in xrange(length[1]):                                                        ## loop over all pixel along the y direction
            ValueOfFirstPoint2 = ValueOfFirstPointList[1]                                   #- (CDELTList[1] / 2.0)
            yPixels = (ValueOfFirstPoint2 + (yy * CDELTList[1]))
            RegionPixelsY.append([yPixels, yy])


    ## determine total number of pixels
    TotalNumberPixels = len(RegionPixelsX) * len(RegionPixelsY)


    ## convert coordinates of the central pixel to degrees, if coordinate system is physical
    if (CoordinateSystem == "physical" and CentralPixelOfRegionX != 0 and CentralPixelOfRegionY != 0):

        # Debug:
        #    print "\n\n>Pixel: CentralPixelOfRegionX = ", CentralPixelOfRegionX
        #    print ">Pixel: CentralPixelOfRegionY = ", CentralPixelOfRegionY
        #    print ">length = ", length
        #    print ">ValueOfFirstPointList = ", ValueOfFirstPointList


        ## find the pixel which is closest to given central pixel (x-direction)
        #    MinDist = 1.e99
        #    CenterX = 0
        #    for xx in xrange(length[0]):
        #        xPixels = (ValueOfFirstPointList[0] + (xx * CDELTList[0]))
        #        if (abs(CentralPixelOfRegionX - xPixels) < MinDist):
        #            MinDist = abs(CentralPixelOfRegionX - xPixels)
        #            CenterX = xx
        #    CentralPixelOfRegionX = (ValueOfFirstPointList[0] + (CenterX * CDELTList[0]))


        ## find the pixel which is closest to given central pixel (y-direction)
        #    MinDist = 1.e99
        #    CenterY = 0
        #    for yy in xrange(length[1]):
        #        yPixels = (ValueOfFirstPointList[1] + (yy * CDELTList[1]))
        #        if (abs(CentralPixelOfRegionY - yPixels) < MinDist):
        #            MinDist = abs(CentralPixelOfRegionY - yPixels)
        #            CenterY = yy
        #    CentralPixelOfRegionY = (ValueOfFirstPointList[1] + (CenterY * CDELTList[1]))

        # Debug:
        #    print "Pixel: CentralPixelOfRegionX = ", CentralPixelOfRegionX
        #    print "Pixel: CentralPixelOfRegionY = ", CentralPixelOfRegionY


        ## get pixel value for physical x-coordinate
        x = CentralPixelOfRegionX - 1
        ValueOfFirstPoint1 = ValueOfFirstPointList[0]
        CentralPixelOfRegionX = (ValueOfFirstPoint1 + (x * CDELTList[0]))


        ## get pixel value for physical y-coordinate
        y = CentralPixelOfRegionY - 1
        ValueOfFirstPoint2 = ValueOfFirstPointList[1]
        CentralPixelOfRegionY = (ValueOfFirstPoint2 + (y * CDELTList[1]))

    # Debug:
    #    print "\n\nRealLength, TotalNumberPixels = ", RealLength, TotalNumberPixels
    #    print "RegionPixelsX = ", RegionPixelsX
    #    print "RegionPixelsY = ", RegionPixelsY
    #    print "CDELTList = ", CDELTList
    #    print "length = ", length
    #    print "ValueOfFirstPointList = ", ValueOfFirstPointList
    #    print "CentralPixelOfRegionX = ", CentralPixelOfRegionX
    #    print "CentralPixelOfRegionY = ", CentralPixelOfRegionY


    ## define return values
    return (TotalNumberPixels, RegionPixelsX, RegionPixelsY, CentralPixelOfRegionX, CentralPixelOfRegionY, RegionMask)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## reads in file that contains cluster definition
##
def ReadClusterdefFile(clusterdefFileName, MAGIXjobDir):
    """

input parameters:
-----------------

    - clusterdefFileName:   path and name of the clusterdef file

    - MAGIXjobDir:          current job directory


output parameters:
------------------

    - TotalNumberCores:     total number of cores of all machines defined in the clusterdef file

    - ServerList:           list of server parameters

    - ClusterFlag:          flag indicating a cluster or smp calculation

    """

    # Debug:
    # print "clusterdefFileName = ", clusterdefFileName, os.path.exists(clusterdefFileName)


    ## initialize output variables
    ClusterFlag = "false"
    TotalNumberCores = 1
    Server = "localhost"
    NumCores = 1
    ServerPath = ""
    ServerList = [[Server, NumCores, ServerPath]]
    ppservers = ()


    ## continue here, if a clusterdef file is defined
    if (clusterdefFileName != "" and os.path.exists(clusterdefFileName)):
        print "Analyze clusterdef file ..",
        ErrorFlag = "false"


        ## readin clusterdef file
        clusterdefFile = open(clusterdefFileName)
        clusterdefFileContents = clusterdefFile.readlines()
        clusterdefFile.close()


        ## analyze clusterdef file
        ServerList = []
        TotalNumberCores = 0
        for lines in clusterdefFileContents:                                                ## loop over all lines of clusterdef file
            pureline = lines.strip()                                                        ## remove leading and trailing blanks

            # Debug:
            # print "pureline = ", pureline


            ## remove comments
            w = pureline.find("#")                                                          ## are there comments in the current line ?
            if (w == 0):                                                                    ## ignore lines which contains only comments
                pureline = ""                                                               ## if line is only a comment clear line
            elif (w > 0):                                                                   ## is there a comment in the current line ?
                pureline = pureline[:w]                                                     ## remove comments
            pureline = pureline.strip()

            # Debug:
            # print "pureline = ", pureline


            ## if remaining line (without comments) is not empty, get server parameters
            if (pureline != ""):                                                            ## ignore empty lines


                ## split current line
                SplittedLine = pureline.split(",")

                # Debug:
                # print "SplittedLine = ", SplittedLine


                ## import socket package
                import socket


                ## get name of current machine (system hostname)
                try:
                    SystemHostname = socket.gethostname()
                except ImportError:
                    SystemHostname = "localhost"


                ## get server
                Server = "localhost"
                NumCores = 1
                ServerPath = ""
                i = len(SplittedLine)
                ServerErrorFlag = "false"
                if (i > 0):
                    Server = SplittedLine[0].strip()
                    Server = Server.replace(chr(34), "")
                    if (Server.lower() != "localhost" and Server != SystemHostname.strip()):


                        ## test, if defined server is reachable
                        s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
                        try:
                            s.connect((Server, 22))
                            ClusterFlag = "true"
                        except socket.error as e:
                            if (ErrorFlag == "false"):
                                print "\n\n"
                            print "\tWARNING:    Server " + Server + " is not reachable!"
                            ErrorFlag = "true"
                            ServerErrorFlag = "true"
                        s.close()


                    ## check, defined number of cores
                    if (i > 1):
                        NumCores = int(SplittedLine[1].strip())
                        TotalNumberCores += NumCores
                        if (i > 2):
                            ServerPath = SplittedLine[2].strip()
                if (ServerErrorFlag == "false"):
                    ServerList.append([Server, NumCores, ServerPath])


        ## check ServerList
        if (ServerList == []):
            Server = "localhost"
            NumCores = 1
            TotalNumberCores = 1
            ServerPath = ""
            ServerList.append([Server, NumCores, ServerPath])


        ## done
        if (ErrorFlag == "false"):
            print "done!"


    # Debug:
    print "\tTotalNumberCores = ", TotalNumberCores
    print "\n\n"
    # print "\tServerList = ", ServerList


    ## define return variables
    return (TotalNumberCores, ServerList, ClusterFlag)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## starts worker via command line
##
def SSHStarter(command):
    """

input parameters:
-----------------

    - command:              command line argument


output parameters:
------------------

    - None
    """


    ## start worker
    os.system(command)


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## start different MAGIX instances via the command line on the cluster
##
def StartClusterParallelization(CurrentDir, myXCLASSrootDir, MAGIXjobDir, MAGIXDir, ClusterFlag, NumberWorkers, InputPathList, ServerList):
    """

input parameters:
-----------------

    - CurrentDir:           current working directory

    - myXCLASSrootDir:      myXCLASS (absolute) root directory

    - MAGIXjobDir:          (absolute) job directory

    - RelativeMAGIXDir:     MAGIX directory relative to myXCLASS root directory

    - ClusterFlag:          flag indicating a cluster or smp calculation

    - NumberWorkers:        number of worker

    - InputPathList:        list with all pixel directories relative to the current job directory

    - ServerList:           list of server parameters


output parameters:
------------------

    - ok:                   status variable
    """

    # Debug:
    # print "CurrentDir = ", CurrentDir
    # print "myXCLASSrootDir = ", myXCLASSrootDir
    # print "MAGIXjobDir = ", MAGIXjobDir
    # print "RelativeMAGIXDir = ", RelativeMAGIXDir
    # print "MAGIXOption = ", MAGIXOption
    # print "ClusterFlag = ", ClusterFlag
    # print "NumberWorkers = ", NumberWorkers
    # print "InputPathList = ", InputPathList


    ## define absolute MAGIX directory
    AbsMAGIXDir = os.path.normpath(myXCLASSrootDir + "/" + MAGIXDir)


    ## construct command line arguments
    CommandList = []
    CounterInstances = (-1)                                                                 ## reset counter for instances
    for server in ServerList:                                                               ## loop over all server in the cluster
        Node = server[0]                                                                    ## get name of node
        NumCores = server[1]                                                                ## get number of cores
        RootPath = server[2].strip()                                                        ## get path of xclass interface
        if (RootPath != ""):                                                                ## if path of xclass interface is defined ..
            RootPath = RootPath + "/programs/MAGIX/"                                        ## .. define path of MAGIX
        for core in xrange(NumCores):                                                       ## loop over all cores of the current node
            CounterInstances += 1                                                           ## increase counter for instances

            # Debug:
            # print "CounterInstances = ", CounterInstances
            # print "NumberWorkers = ", NumberWorkers

            if (CounterInstances < NumberWorkers and CounterInstances < len(InputPathList)):


                ## define path of the xclass interface on the different nodes
                if (RootPath.strip() == ""):
                    PathXCLASSInterface = AbsMAGIXDir
                else:
                    PathXCLASSInterface = RootPath


                ## ssh charger 'cd ~/magma; nohup nice /usr/bin/magma magma.script > /var/tmp/magma.out' &
                if (len(ServerList) == 1 and Node == "localhost"):
                    cmdString = "cd " + PathXCLASSInterface + "; python magix_start.py --quiet --model=myxclass"
                    cmdString +=  " " + os.path.normpath(MAGIXjobDir + "/" + InputPathList[CounterInstances] \
                                      + "/io_control.xml > " + MAGIXjobDir + "/" + InputPathList[CounterInstances] + "/screen.out")
                                      #    + "/io_control.xml >/dev/null")
                else:
                    cmdString = "ssh " + Node + " " + chr(34) + "cd " + PathXCLASSInterface + "; "
                    cmdString += "nohup python magix_start.py --noplot --model=myxclass"
                    cmdString +=  " " + os.path.normpath(MAGIXjobDir + "/" + InputPathList[CounterInstances]) \
                                      + "/io_control.xml > " + MAGIXjobDir + "/" + InputPathList[CounterInstances] + "/screen.out" + chr(34)
                                      #    + "/io_control.xml >/dev/null" + chr(34)
                # Debug:
                # print "\n\n\n\ncmdString = ", cmdString


                ## append to list of commands
                CommandList.append(cmdString)


    ## prepare start of workers
    thread_list = []
    for command in CommandList:                                                             ## loop over all workers


        ## Instatiates the thread
        t = threading.Thread(target=SSHStarter, args=(command,))


        ## Sticks the thread in a list so that it remains accessible
        thread_list.append(t)


    # Start all threads (workers)
    for thread in thread_list:
       thread.start()


    # This blocks the calling thread until the thread whose join() method is called is terminated.
    # From http://docs.python.org/2/library/threading.html#thread-objects
    for thread in thread_list:
       thread.join()


    # Debug:
    # sys.exit(0)
    ok = 0


    ## declare return variable
    return ok
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## determines the best fit result from a previous pixel fit
##
def GetMolfitFileOfBestResult(JobDir, ConvertLogLinFlag, ListOfSpectraNames):
    """

input parameters:
-----------------

    - JobDir:               path of job directory

    - ConvertLogLinFlag:    defines if column densities in the fitted molfit file(s) are converted back to linear values

    - ListOfSpectraNames:   list of exp. data file names


output parameters:
------------------

    - input_file:           list of all molfit files

    - ListOfParamXMLFiles:  list of all parameter xml files which corresponds to molfit file

    - ListOfModelFuncFiles: list of all model function files which corresponds to molfit file

    - BestChi2Value:        best chi^2 value

    """


    ## initialize return parameters
    input_file = []
    ListOfParamXMLFiles = []
    ListOfModelFuncFiles = []
    BestChi2Value = 1.e99


    ## get all filenames in the current job directory
    listing = os.listdir(JobDir)

    # Debug:
    # print "\n\n\nJobDir = ", JobDir
    # print "\n\n\nlisting = ", listing


    ## get number of chi2 log files
    NumOutFiles = 0                                                                         ## determine number of output files
    NumMolfitFiles = 0
    for files in listing:
        if (files.endswith(".log.chi2")):
            NumOutFiles += 1
        if (files.endswith(".out.molfit")):
            NumMolfitFiles += 1
    if (NumOutFiles == 0):                                                                  ## check if new input files exsist
        print " "
        print " "
        print "Error in XCLASS package, function task_myXCLASSMapFit.myXCLASSMapFit:"
        print "  Can not find a chi2 log file!"
        print " "
        print "JobDir = ", JobDir
        print "listing = ", listing
        print "\n\n\n"
        return BestChi2Value, input_file, ListOfParamXMLFiles, ListOfModelFuncFiles


    ## determine the chi2 log file, which corresponds to the best fit
    FileNameBestChi2 = ""
    for files in listing:                                                                   ## loop over all files in the current job directory
        if (files.endswith(".log.chi2")):                                                   ## we're interested only in the log.chi2 files
            Chi2LogFile = open(JobDir + files)                                              ## open log.chi2 file
            dummyline = Chi2LogFile.readline()                                              ## read in first line as dummy
            LogLine = Chi2LogFile.readline()                                                ## read second line with best chi2 value
            Chi2LogFile.close()                                                             ## close log file
            LogLine = LogLine.strip()                                                       ## remove blanks
            SplittedLine = LogLine.split()                                                  ## split line into columns
            try:
                chi2Value = float(SplittedLine[1])                                          ## get chi2 value
            except IndexError:
                chi2Value = numpy.nan
            if (not numpy.isnan(chi2Value)):
                if (chi2Value <= BestChi2Value):                                            ## check, if current chi2 value is the best one
                    BestChi2Value = chi2Value                                               ## if yes, save chi2 value
                    FileNameBestChi2 = files                                                ## and corresponding file name

                    # Debug:
                    # print "chi2Value = ", chi2Value


    ## get phrase identifying the used algorithm for the best chi2 value
    BestAlgorithm = ""
    BestAlgorithm = FileNameBestChi2.replace(".log.chi2", "")
    # BestAlgorithm = BestAlgorithm.replace("fit_", "")
    BestAlgorithm = BestAlgorithm.replace("fit__", "")                                      ## bug fix???


    ## find molfit file
    for files in listing:                                                                   ## loop over all files in the current job directory
        if (files.endswith(".out.molfit")):                                                 ## we're interested only in the molfit files
            i = files.find(BestAlgorithm)
            if (i > (-1)):


                ## convert column and hydrogen column density (if given for each component) back to linear scale
                if (ConvertLogLinFlag == "true"):
                    LogLinearFlag = "linear"
                    task_myXCLASS.ConvertNtotToLogScale(JobDir + files, LogLinearFlag)


                ## read in new output files
                MolfitFile = open(JobDir + files)
                input_file = MolfitFile.readlines()
                MolfitFile.close()

                # Debug:
                # print "\n\nfiles = ", files
                # print "input_file = ", input_file
                break


    ## special handling for LM
    if (NumMolfitFiles == 1 and BestAlgorithm.find("LM") > (-1)):
        BestAlgorithm = "LM.out"

    # Debug:
    # print "BestAlgorithm = ", BestAlgorithm


    ## find xml file which corresponds to molfit file
    for files in listing:                                                                   ## loop over all files in the current job directory
        if (files.endswith(".out.xml")):                                                    ## we're interested only in the molfit files
            i = files.find(BestAlgorithm)
            if (i > (-1)):
                ListOfParamXMLFiles.append(files)


    ## find model data file(s)
    if (ListOfSpectraNames != []):
        for CounterRange in xrange(len(ListOfSpectraNames)):                                ## loop over all frequency ranges
            ExpDataFileName = ListOfSpectraNames[CounterRange]                              ## get path and name of corresponding exp. data file


            ## analyze exp. data file name, we only want the name, not the path
            i = ExpDataFileName.rfind("/")
            if (i > (-1)):
                ExpDataFileName = ExpDataFileName[i + 1:]
                if (ExpDataFileName.endswith(".dat")):
                    ExpDataFileName = ExpDataFileName.replace(".dat", "")
                elif (ExpDataFileName.endswith(".fits")):
                    ExpDataFileName = ExpDataFileName.replace(".fits", "")

            # Debug:
            # print "\n\n\n\n\n\n\n\n\n\n\n"
            # print "ExpDataFileName = ", ExpDataFileName
            # print "BestAlgorithm = ", BestAlgorithm


            ## find exp. data file in the current single molecule fit directory
            for files in listing:                                                           ## loop over all files in the current job directory

                # Debug:
                # print "files = ", files


                if (files.startswith(ExpDataFileName) and (files.endswith(BestAlgorithm) or files.endswith(".out.dat"))):
                    i = files.find(BestAlgorithm)
                    if (i > (-1)):
                        ListOfModelFuncFiles.append(files)

    # Debug:
    #    print "\nBestChi2Value = ", BestChi2Value
    #    print "input_file = ", input_file
    #    print "ListOfParamXMLFiles = ", ListOfParamXMLFiles
    #    print "ListOfModelFuncFiles = ", ListOfModelFuncFiles
    #    sys.exit(0)


    ## define return variable
    return BestChi2Value, input_file, ListOfParamXMLFiles, ListOfModelFuncFiles
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## read parameter values from xml file
##
def ReadParmXMLFile(PixelDirectory, ListOfParamXMLFiles):
    """

input parameters:
-----------------

    - PixelDirectory:       current pixel directory

    - ListOfParamXMLFiles:  list of xml files


output parameters:
------------------

    - ParameterList:        list of fitted parameters with param. names

    """
    ParameterList = []
    LocalKnownParameterNames = []
    IsoMolecule = ""
    IsoMaster = ""


    ## check, if a xml file is given
    if (ListOfParamXMLFiles == []):
        return ParameterList
    else:


        ## read total contents of xml file
        output_file = open(PixelDirectory + ListOfParamXMLFiles[0])
        output_file_contents = output_file.readlines()
        output_file.close()


        ## get value of fitted parameter
        FittedParameter = "false"
        LogLinearFlag = "false"
        CurrentNumberFittedParameter = 0
        CurrentMolecule = ""
        ComponentCounter = 0
        NameFlag = "false"
        for line in output_file_contents:
            StrippedLine = line.strip()

            # Debug:
            # print "StrippedLine = ", StrippedLine


            ## determine name of molecule (only for the first pixel)
            i = StrippedLine.find("<name>")
            if (i > (-1)):
                j = StrippedLine.find("</name>")
                ParameterName = StrippedLine[i + 6:j].strip()
                if (ParameterName.strip() == "Molecule_Name"):
                    ComponentCounter = 0
                    NameFlag = "true"
                else:
                    NameFlag = "false"
                if (ParameterName.strip() == "source_size"):                                ## the occurrence of parameter source_size is used to
                    ComponentCounter += 1                                                   ## determine the index of the current component and molecule


            i = StrippedLine.find("<value>")
            if (i > (-1)):
                j = StrippedLine.find("</value>")
                if (NameFlag == "true"):
                    CurrentMolecule = StrippedLine[i + 7:j].strip()
                    CurrentMolecule = CurrentMolecule.replace(";", "_")
                    CurrentMolecule = CurrentMolecule.replace(",", "_")
                    CurrentMolecule = CurrentMolecule.replace("'", "_")
                    CurrentMolecule = CurrentMolecule.replace("(", "_")
                    CurrentMolecule = CurrentMolecule.replace(")", "_")
                elif (ParameterName.strip() == "IsoMolecule" or ParameterName.strip() == "IsoMasters"):
                    IsoName = StrippedLine[i + 7:j].strip()
                    IsoName = IsoName.replace(";", "_")
                    IsoName = IsoName.replace(",", "_")
                    IsoName = IsoName.replace("'", "_")
                    IsoName = IsoName.replace("(", "_")
                    IsoName = IsoName.replace(")", "_")
                    if (ParameterName.strip() == "IsoMolecule"):
                        IsoMolecule = IsoName.strip()
                    else:
                        IsoMaster = IsoName.strip()


            ## check, if current line describes a fitted parameter
            i = StrippedLine.find("Parameter fit=" + chr(34) + "true" + chr(34))
            if (i > (-1)):
                FittedParameter = "true"
                CurrentNumberFittedParameter += 1
            else:


                ## continue here, if current parameter block descibes a fitted parameter
                if (FittedParameter == "true"):


                    ## convert column densities to linear values
                    ## check, if parameter value is given
                    i = StrippedLine.find("<name>")
                    if (i > (-1)):
                        j = StrippedLine.find("</name>")
                        ParameterName = StrippedLine[i + 6:j]
                        ParameterName = ParameterName.strip()


                        ## modify name of non-standard XCLASS parameters
                        if (ParameterName == "IsoRatio"):
                            countOccurrence = 1
                            for name in LocalKnownParameterNames:
                                if (name.strip() == ParameterName):
                                    countOccurrence += 1
                            LocalKnownParameterNames.append(ParameterName)
                            ParameterName = ParameterName + "__" + IsoMolecule + "__" + IsoMaster + "__" + str(countOccurrence)
                        else:
                            LocalKnownParameterNames.append(ParameterName)


                        ## check new parameter name
                        if (ParameterName == "N_tot" or ParameterName == "nHcolumn"):
                            LogLinearFlag = "true"
                        else:
                            LogLinearFlag = "false"


                    ## check, if parameter value is given
                    i = StrippedLine.find("<value>")
                    if (i > (-1)):
                        j = StrippedLine.find("</value>")
                        value = StrippedLine[i + 7:j].strip()
                        if (LogLinearFlag == "true"):
                            value = 10.0**float(value)

                        # Debug:
                        # print "value = ", value


                        ## append parameter values of current pixel to parameter list
                        ParameterList.append([CurrentMolecule, ComponentCounter, ParameterName, value])


                    ## check, if parameter block is closed
                    i = StrippedLine.find("</Parameter>")
                    if (i > (-1)):
                        FittedParameter = "false"


    ## define return variable
    return ParameterList
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## read model function values from file(s)
##
def ReadModelFiles(PixelDirectory, NumberSpectra, NumFrequenciesALL, ListOfModelFuncFiles, CDELTListALL, ValueOfFirstPointListALL, \
                   BSCALEALL, BZEROALL, BUNIT, UnitConversionFactorALL, RESTFRQALL, SizeTelescopePerSpectrum, InterFlagPerSpectrum):
    """

input parameters:
-----------------

    - PixelDirectory:       current pixel directory

    - NumberSpectra:        number of data cubes

    - NumFrequenciesALL:    number of frequencies for each data cube

    - ListOfModelFuncFiles: list of model files

    - CDELTListALL:         delta values for each cube

    - ValueOfFirstPointListALL: values of first freq. point for each cube

    - BSCALEALL:            bscale values for each cube

    - BZEROALL:             bzero values for each cube

    - BUNIT:                bunit values for each cube

    - UnitConversionFactorALL:  unit conversion factors for each cube

    - RESTFRQALL:           rest freq. values for each cube

    - SizeTelescopePerSpectrum: size of telescope for each cube

    ⁻ InterFlagPerSpectrum: flag indicating interferrometric data


output parameters:
------------------

    - PixelSpectrumList:    list containing the modelled spectrum of the current pixel

    """

    # Debug:
    #    print "PixelDirectory = ", PixelDirectory
    #    print "NumberSpectra = ", NumberSpectra
    #    print "NumFrequenciesALL = ", NumFrequenciesALL
    #    print "ListOfModelFuncFiles = ", ListOfModelFuncFiles
    #    print "CDELTListALL = ", CDELTListALL
    #    print "ValueOfFirstPointListALL = ", ValueOfFirstPointListALL
    #    print "BSCALEALL = ", BSCALEALL
    #    print "BZEROALL = ", BZEROALL
    #    print "BUNIT = ", BUNIT
    #    print "UnitConversionFactorALL = ", UnitConversionFactorALL
    #    print "RESTFRQALL = ", RESTFRQALL
    #    print "SizeTelescopePerSpectrum = ", SizeTelescopePerSpectrum
    #    print "InterFlagPerSpectrum = ", InterFlagPerSpectrum


    ## initialize return parameters
    cms = 299792458.0                                                                       ## speed of light in m/s
    ckms = cms * 1.e-3                                                                      ## speed of light in km/s
    maxFreqPoints = max(NumFrequenciesALL[:])
    PixelSpectrumList = numpy.zeros((NumberSpectra, maxFreqPoints), dtype=numpy.float32)    ## create array for all pixels of the selected region

    # Debug:
    # print "-->PixelSpectrumList = ", PixelSpectrumList


    ## check, if list of model function data files is not empty
    if (ListOfModelFuncFiles == []):
        return PixelSpectrumList


    for CubeNumber in xrange(NumberSpectra):                                                ## loop over all data cubes
        NumFrequencies = NumFrequenciesALL[CubeNumber]                                      ## get total number of frequencies


        ## create the name of the spectrum file for each data cube
        NumString = "%05i" % (CubeNumber + 1)
        SpecFile = "Spectrum__" + NumString                                                 ## define name of ASCII file containing spectra


        ## loop over all model function data files
        for files in ListOfModelFuncFiles:                                                  ## loop over all model data files
            if (files.rfind(SpecFile) > (-1)):                                              ## current model function data file corresponds to the
                                                                                            ## current data cube

                ## read in data from ASCII file
                ModelData = numpy.loadtxt(PixelDirectory + files, skiprows = 0)


                ## extract data
                MinFileFreq = min(ModelData[:, 0])                                          ## determine lowest frequency in model data file
                MaxFileFreq = max(ModelData[:, 0])                                          ## determine highest frequency in model data file
                for nu_index in xrange(NumFrequencies):                                     ## loop over all frequencies
                    nu = nu_index                                                           ## get frequency
                    ValueOfFirstPoint3 = ValueOfFirstPointListALL[CubeNumber][2]


                    ## determine the current frequency index (in MHz)
                    point = (ValueOfFirstPoint3 + (nu * CDELTListALL[CubeNumber][2])) * BSCALEALL[CubeNumber] + BZEROALL[CubeNumber]
                    if (UnitConversionFactorALL[CubeNumber] == "velocityKMS"):              ## if necessary change to velocities
                        # Freq = ckms * (RESTFRQALL[CubeNumber] - point) / point
                        Freq = float(RESTFRQALL[CubeNumber]) / ckms * (ckms - point)
                    elif (UnitConversionFactorALL[CubeNumber] == "velocityMS"):
                        # Freq = ckms * (RESTFRQALL[CubeNumber] - point) / point
                        Freq = float(RESTFRQALL[CubeNumber]) / cms * (cms - point)
                    else:
                        Freq = point * UnitConversionFactorALL[CubeNumber]

                    # Debug:
                    #    print "MinFileFreq = ", MinFileFreq
                    #    print "MaxFileFreq = ", MaxFileFreq
                    #    print "Freq = ", Freq
                    #    print "CubeNumber, UnitConversionFactorALL[CubeNumber] = ", CubeNumber, UnitConversionFactorALL[CubeNumber]


                    ## find current frequency in array ModelData
                    i = (-1)
                    if (abs(MinFileFreq - Freq) < 1.e-6 or abs(MaxFileFreq - Freq) < 1.e-6 or (MinFileFreq <= Freq and Freq <= MaxFileFreq)):
                        ii = (-1)

                        i = max(0, (numpy.abs(ModelData[:, 0] - Freq)).argmin() - 1)

                        # for x in ModelData[:, 0]:                                           ## loop over all frequency points in array ModelData
                        #     ii += 1
                        #     if (abs(x - Freq) < 1.e-6):                                     ## find frequency with 1.e-6

                        #         # Debug:
                        #         # print "ii, x, Freq = ", ii, x, Freq

                        #         i = ii
                        #         break
                    if (i != (-1)):                                                         ## was frequency found in array ModelData
                        value = ModelData[i, 1]


                        ## If spectra in the original data cube are given in Jy/Beam, convert modelled data to Jy/Beam as well                    
                        UnitIntensity = BUNIT[CubeNumber].lower()
                        UnitIntensity = UnitIntensity.replace(" ", "")
                        if (UnitIntensity.strip() == "jy/beam"):
                            ConvertJyPerBeam2Kelvin, ConvertKelvin2JyPerBeam = JanskyPerBeamInKelvin(float(SizeTelescopePerSpectrum[CubeNumber]), \
                                                                                                     InterFlagPerSpectrum[CubeNumber], Freq)
                            value = ConvertKelvin2JyPerBeam * value
                    else:                                                                   ## set intensity to zero, if frequency was not found in
                        value = 0.0                                                         ## 'ModelData'

                        # Debug:
                        #    print "\n\n\n\n\n\n\n\n\n"
                        #    print "NumFrequencies = ", NumFrequencies
                        #    print "ValueOfFirstPoint3 = ", ValueOfFirstPoint3
                        #    print "nu = ", nu
                        #    print "CDELTListALL[CubeNumber][2] = ", CDELTListALL[CubeNumber][2]
                        #    print "BSCALEALL[CubeNumber], BZEROALL[CubeNumber] = ", BSCALEALL[CubeNumber], BZEROALL[CubeNumber]
                        #    print "MinFileFreq = ", MinFileFreq
                        #    print "MaxFileFreq = ", MaxFileFreq
                        #    print "Freq = ", Freq
                        #    print "ModelData[:, 0] = ", ModelData[:, 0]
                        #    print "MinFileFreq <= Freq = ", MinFileFreq <= Freq
                        #    print "Freq <= MaxFileFreq = ", Freq <= MaxFileFreq
                        #    print "Freq - MaxFileFreq = ", Freq - MaxFileFreq
                        #    print "\n\n\n\n\n\n\n\n\n"

                    # Debug:
                    # print "NumberSpectra, nu_index, value = ", NumberSpectra, nu_index, value
                    # print "NumberSpectra, NumFrequencies, NumYPixels, NumXPixels = ", NumberSpectra, NumFrequencies, NumYPixels, \
                    #                                                                   NumXPixels
                    # print "------------------------------------------------------------------------------------------------------------"


                    ## append model function values to total output values array
                    PixelSpectrumList[CubeNumber, nu_index] = numpy.float32(value)

    # Debug:
    # print ">>PixelSpectrumList = ", PixelSpectrumList


    ## define return variable
    return PixelSpectrumList
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## Simplified CASA interface for MAGIX with myXCLASS program to fit a complete data cube instead of a single spectrum
##
def myXCLASSMapFit(NumberIteration, AlgorithmXMLFile, MolfitsFileName, experimentalData, regionFileName, UsePreviousResults, Threshold, FreqMin, \
                   FreqMax, TelescopeSize, Inter_Flag, t_back_flag, tBack, tslope, nH_flag, N_H, beta_dust, kappa_1300, iso_flag, IsoTableFileName, \
                   RestFreq, vLSR, clusterdef):
    """

This function provides a simplified interface for MAGIX using the myXCLASS program for a
map. The function starts MAGIX using the Levenberg-Marquardt algorithm to fit
experimental data with the myXCLASS program. The user has to specify the max. number of
iterations, the experimental data (i.e. the path and the name of a FITS file
containing the data cube), and the path and name of the molfit file.

Please note, the FITS data cube has to be given in the World Coordinate System (WCS)
described in Greisen, E. W., and Calabretta, M. R., A & A, 395, 1061-1075, 2002,
Calabretta, M. R., and Greisen, E. W., A & A, 395, 1077-1122, 2002, and
Greisen, E. W., Calabretta, M. R., Valdes, F. G., and Allen, S. L., A & A, 446,
747-771, 2006.

For each run the myXCLASSMapFit function creates a so-called job directory located in
the run directory ("path-of-myXCLASS-CASA-Interface/run/myXCLASSMapFit/") where all
files created by the myXCLASSMapFit function are stored in. The name of this job
directory is made up of four components: The first part of the name consists of the
phrase "job_" whereas the second part describes the date (day, month, year), the third
part the time stamp (hours, minutes, seconds) of the function execution. The last part
describes a so-called job ID which is composed of the so-called PID followed by a
four digit random integer number to create a really unambiguous job number, e.g.
"path-to-myXCLASS-Interface/run/myXCLASSMapFit/job__25-07-2013__12-02-03__189644932/"

Additionally, the myXCLASSMapFit function creates a subdirectory within this job
directory called "Pixel-Fits/" which contains furhter subdirectories for each pixel,
where the name of the subdirectory is made up of the values (in degrees) for the right
ascension and for the declination, separately, e.g. "83.65498__-5.24258". All output
files, created by MAGIX for this pixel are stored in this subdirectory.

At the end of the whole fit procedure, the myXCLASSMapFit function creates FITS images
for each free parameter, where each pixel correspond to the value of the free parameter
taken from the best fit for this pixel. Additionally, the myXCLASSMapFit function
creates one FITS image, where each pixel corresponds to the chi^2 value of the best
fit for this pixel.



input parameters:
-----------------

    - NumberIteration:      max. number of iterations (default: 50).

    - AlgorithmXMLFile:     only necessary, if the user wants to use another fit
                            algorithm (than Levenberg-Marquardt) for fitting. Therefore,
                            the path and name of a MAGXI xml-file defining settings
                            for an algorithm or algorithm chain has to be given. (A
                            relative path has to be defined relative to the current
                            working directory!)

                            NOTE, if the user specify a xml file, the number of
                            iterations given by the parameter "NumberIteration" is
                            ignored. The number of iteration is then given by the xml
                            file. In order to use the implemented fit algorithm
                            (Levenberg-Marquardt) clear the AlgorithmXMLFile parameter,
                            i.e. AlgorithmXMLFile = "", and define the max. number of
                            iterations by using parameter "NumberIteration".

    - MolfitsFileName:      path and name of the extended molfit file, including the
                            source size, the rotation temperature the column density,
                            the velocity width, the velocity offset and the flag
                            indicating if the current component is considered for
                            core "c" or foreground "f". Depending on the values of
                            "t_back_flag" and "nH_flag", the molfit file has to
                            include the columns for the background temperature, the
                            temperature slope, the hydrogen column density, for beta
                            and for kappa for each component as well.

                            In contrast to the format of the molfit file the extended
                            molfit file required by the myXCLASSMapFit function contains
                            one (three) additional column(s) for each parameter of
                            each component.

                            For a detailed description of the extended molfit file
                            required by the myXCLASSMapFit function, see the manual.

                            NOTE, a relative path has to be defined relative to the
                            current working directory!

    - experimentalData:     This parameter offers two different possibility to send
                            the experimental data to the myXCLASSMapFit function:

                            - the parameter experimentalData defines the path and
                              name of and experimental xml-file suitable for MAGIX.
                              Here, the XML file contains the path and the name of
                              each FITS or Measurement Set (MS) file. The tag
                              <ImportFilter> has to be included in the XML file. The
                              content will be automatically set by the myXCLASSMapFit
                              function.

                              Please note, if more than one data cube is specified
                              in the xml file, the data cubes must describe the same
                              map, i.e. the right ascension and the declination has
                              to be identical!  The different data cubes must be differ
                              only in the frequency/velocity axis, i.e. it is possible
                              to specify different units for the frequency axis for
                              different data cubes.

                            - the parameter experimentalData defines the path and
                              name of a FITS or Measurement Set (MS) file called
                              "experimental data file" containing the data cube. Please
                              note, the myXCLASSMapFit function assumes, that the first
                              three axes describe the right ascension, the declination,
                              and the frequency, respectively. If the frequencies are
                              not given in MHz, the FITS header has to contain the
                              definition of the CUNIT3 command word. For velocities, the
                              header has to contain the command word RESTFRQ as well.

                            NOTE, if the parameter experimentalData defines a relative
                            path, the path has to be defined relative to the current
                            working directory!

    - regionFileName:       the so-called region file (defined by the ds9 program or the casa
                            viewer). At the moment the shape of the region has to be
                            rectangular. (Other formats and other shapes will be available soon).

    - UsePreviousResults:   (T/F) defines if the molfit file (described by the paramter
                            MolfitsFileName) is used as initial guess for all pixels (F)
                            or if the result for a previous pixel is used.

                            Please note, that this parameter is valid only for so-called
                            local optimization algorithm like Levenberg-Marquardt or
                            Simulated annealing.

    The following parameters are needed, if the parameter experimentalData does NOT
    describe the path and name of a MAGIX xml-file:

    - Threshold:            defines a threshold for a pixel. If the spectrum of a pixel
                            has a intensity lower than the value defined by this parameter
                            the pixel is not fitted (ignored), (default: 0)

    - FreqMin:              start frequency of simulated spectrum (in MHz), (default: 0).

                            Please note, if no start frequency is given, or if start frequency is
                            lower or equal to the end frequency, the myXCLASSMapFit function will
                            use the max. frequency range defined in the FITS header. Additionally,
                            the stepsize, i.e. the difference between two frequencies is taken
                            from the FITS header.

    - FreqMax:              end frequency of simulated spectrum (in MHz), (default: 0).

    - vLSR:                 local standard of rest velocity (in km/s), (default: 0).

    - TelescopeSize:        for single dish observations (Inter_Flag = F): TelescopeSize
                            describes the size of telescope (in m), (default: 1); for
                            interferometric observations (Inter_Flag = T): TelescopeSize
                            describes the interferometric beam FWHM size (in arcsec),
                            (default: 1).

    - Inter_Flag:           defines, if single dish ("F") or interferometric
                            observations ("T") are described, (default: "F").

    - t_back_flag:          ("T"/"F"), defines, if the user defined background 
                            temperature Tbg and temperature slope Tslope given
                            by the input parameters tBack and tslope describe the
                            continuum contribution completely (t_back_flag = "T")
                            or not (t_back_flag = "F") (default: "T").

    - tBack:                background temperature (in K), (default: 0).

    - tslope:               temperature slope (dimensionless), (default: 0).

    - nH_flag:              ("T"/"F"), defines, if column density, beta for dust
                            and kappa are given by the molfit file ("F")
                            or if nH_flag is set to "T", the following
                            three parameters define the hydrogen column density,
                            beta for dust and kappa for all components (default: "F"):

    - N_H:                  (has to be given only if nH_flag is set to "T")
                            Hydrogen column density (cm^{-2}), (default: 0).

    - beta_dust:            (has to be given only if nH_flag is set to "T")
                            beta for dust (dimensionless), (default: 0).

    - kappa_1300:           (has to be given only if nH_flag is set to "T")
                            kappa (cm^2 g^{-1}), (default: 0).

    - iso_flag:             use isotopologues ("T"/"F"). If iso_flag is set to
                            "F" the ratios are all set to 1 (default: "F").

    - IsoTableFileName:     path and name of an ASCII file including the iso ratios
                            between certain molecules. The so-called "iso ratio file"
                            defines the iso ratios between molecules. The ASCII file
                            consists of three columns, where the first two columns
                            indicates the molecules, respectively. The third column
                            defines the ratio for both molecules. The columns are
                            separated by blanks or tabs. So, the names of the
                            molecules must not contain blanks.

                            The myXCLASSFit function offers the possibility to
                            optimize the ratios between isotopologues as well. For
                            that purpose, the user has to add two additional columns
                            on the right indicating the lower and the upper limit
                            of a certain ratio, respectively. For more informations
                            please look at the description of the iso ratio file given
                            for the myXCLASS function.

                            If the lower and upper limit are equal or if the lower
                            limit is higher than the upper limit, the ratio is kept
                            constant and is not optimized by the myXCLASSFit function.

                            If no path and name is specified (default), the so-called
                            iso-flag is set to "false".

                            NOTE, if the parameter "IsoTableFileName" defines a relative
                            path, the path has to be defined relative to the current
                            working directory!

    - RestFreq:             rest frequency in MHz (default: 0). (If this parameter is
                            set to zero, the intensity is plotted against frequency (in MHz)
                            otherwise against velocity (in km/s).

    - clusterdef:           file that contains cluster definition.


output parameters:
------------------

    - JobDir:               absolute path of the job directory created for the current run.



Example using a molfit file in the old format and an ASCII file containing the experimental data:
-------------------------------------------------------------------------------------------------

NumberIteration = 10
MolfitsFileName = "demo/myXCLASSMapFit/CH3OH.molfit"
AlgorithmXMLFile = ""
experimentalData = "demo/myXCLASSMapFit/Orion.methanol.cbc.contsub.image.fits"
regionFileName = "demo/myXCLASSMapFit/ds9__test.reg"
UsePreviousResults = T
Threshold = 0.0
vLSR = 0.0
TelescopeSize = 3.5
Inter_Flag = F
t_back_flag = T
tBack = 9.5000000000E-01
tslope = 0.0000000000E+00
nH_flag = T
N_H = 3.0000000000E+24
beta_dust = 2.0
kappa_1300 = 0.02
iso_flag = T
IsoTableFileName = "demo/myXCLASSMapFit/iso_names.txt"
RestFreq = 0.0
clusterdef = "demo/myXCLASSMapFit/clusterdef.txt"
JobDir = myXCLASSMapFit()

    """

    # Debug:
    # print "NumberIteration = ", NumberIteration
    # print "AlgorithmXMLFile = ", AlgorithmXMLFile
    # print "MolfitsFileName = ", MolfitsFileName
    # print "experimentalData = ", experimentalData
    # print "regionFileName = ", regionFileName
    # print "UsePreviousResults = ", UsePreviousResults
    # print "Threshold = ", Threshold
    # print "FreqMin = ", FreqMin
    # print "FreqMax = ", FreqMax
    # print "vLSR = ", vLSR
    # print "TelescopeSize = ", TelescopeSize
    # print "Inter_Flag = ", Inter_Flag
    # print "t_back_flag = ", t_back_flag
    # print "tBack = ", tBack
    # print "tslope = ", tslope
    # print "nH_flag = ", nH_flag
    # print "N_H = ", N_H
    # print "beta_dust = ", beta_dust
    # print "kappa_1300 = ", kappa_1300
    # print "IsoTableFileName = ", IsoTableFileName
    # print "RestFreq = ", RestFreq


    ## define some internal variables
    MultiprocessingMethod = "pp"
    cms = 299792458.0                                                                       ## speed of light in m/s
    ckms = cms * 1.e-3                                                                      ## speed of light in km/s
    NameOfFunction = "myXCLASSMapFit"


    ##====================================================================================================================================================
    ## initialize working variables
    CurrentDir = os.getcwd() + "/"

    # Debug:
    # print "CurrentDir = ", CurrentDir


    ######################################################################################################################################################
    ## DO NOT EDIT OR REMOVE THE FOLLOWING LINE !!!!
    MAGIXrootDir = "/home/moeller/ALMA/CASA/myXCLASS-CASA-Interface/No-NR-version/Linux/XCLASS-Interface/programs/MAGIX/"
    ######################################################################################################################################################


    ##====================================================================================================================================================
    ## create run directory for the current function call
    LocalPrintFlag = True
    myXCLASSrootDir = os.path.normpath(MAGIXrootDir + "../../")
    MAGIXjobDir = task_MAGIX.CreateRunDirectory("myXCLASSMapFit", MAGIXrootDir, LocalPrintFlag)
    RelativejobDir = MAGIXjobDir.replace(myXCLASSrootDir, "")
    RelativeMAGIXDir = MAGIXrootDir.replace(myXCLASSrootDir, "")

    # Debug:
    # print "myXCLASSrootDir = ", myXCLASSrootDir
    # print "MAGIXjobDir = ", MAGIXjobDir
    # print "RelativejobDir = ", RelativejobDir
    # print "RelativeMAGIXDir = ", RelativeMAGIXDir


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze clusterdef file
    NumberProcessors, ServerList, ClusterFlag = ReadClusterdefFile(clusterdef, MAGIXjobDir)


    ##====================================================================================================================================================
    ## set up environment for MAGIX
    task_MAGIX.SetMAGIXEnvironment(MAGIXrootDir)


    ##====================================================================================================================================================
    ## analyze experimentalData parameter: check if input parameter defines path and name of a file
    print "Analyze experimentalData parameter .."
    NumberSpectra = 1                                                                       ## set number of spectra to 1 (default)

    # Debug:
    # print 'type(experimentalData).__name__ = ', type(experimentalData).__name__


    ## is experimentalData a string
    ParamIsXMLFile = "false"
    if (type(experimentalData).__name__ == 'str'):                                          ## is experimentalData a string ?
        filename = experimentalData.strip()                                                 ## remove leading and tailing blanks

        # Debug:
        # print 'filename = ', filename


        ## if neccessary, make relative path absolute
        if (filename[0] != "/"):
            filename = CurrentDir + filename


        ## check if path and name of the experimental file exsits
        if not(os.path.exists(filename) or os.path.isfile(filename)):
            print " "
            print " "
            print "Error in XCLASS package, function task_myXCLASSMapFit.myXCLASSMapFit:"
            print "  The given path and name of the experimental xml- or ASCII file " + filename.strip()
            print "  does not exsist!"
            print " "
            print "  Please enter an exsisting path and name of an experimental xml- or ASCII file and redo myXCLASSMapFit function call!"
            return MAGIXjobDir


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if file is a xml-file, i.e. the filename ends with '.xml'
        if (filename.endswith(".xml")):
            ParamIsXMLFile = "true"


            ## copy iso ratio file to current job directory
            command_string = "cp " + filename + " " + MAGIXjobDir                           ## copy xml file to job directory
            os.system(command_string)
            i = filename.rfind("/")
            if (i < 0):
                i = 0
            MAGIXExpXML = MAGIXjobDir + filename[i + 1:]                                    ## define new xml file name


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get all min. and max. frequencies for each range and path and name of user defined database file
            FreqMinList = task_MAGIX.GetXMLtag(MAGIXExpXML, "MinExpRange")
            FreqMaxList = task_MAGIX.GetXMLtag(MAGIXExpXML, "MaxExpRange")
            dbList = task_MAGIX.GetXMLtag(MAGIXExpXML, "dbFilename")
            if (len(dbList) > 0):
                PathFileNameDB = dbList[0].strip()
                if (PathFileNameDB.strip() != ""):
                    if (PathFileNameDB[0] != "/"):                                          ## make relative path absolute
                        PathFileNameDB = CurrentDir + PathFileNameDB
                    if not(os.path.exists(PathFileNameDB) or os.path.isfile(PathFileNameDB)):
                        print " "
                        print " "
                        print "Error in XCLASS package, function task_myXCLASSMapFit.myXCLASSMapFit:"
                        print "  The given path and name of the database file " + PathFileNameDB
                        print "  does not exsist!"
                        print " "
                        print "  Please note, if a relative path is specified, the path has to be defined relative to the"
                        print "  current working directory!"
                        print " "
                        print "  Please enter an exsisting path and name of a database file and redo myXCLASSMapFit function call!"
                        return MAGIXjobDir

                    # Debug:
                    # print "PathFileNameDB = ", PathFileNameDB


                    ## modify experimental xml-file in MAGIX temp directory
                    dbList = [PathFileNameDB]
                else:
                    dbList = []
            else:
                dbList = []


            ## analyze molfit parameter: check path and name of the molfit xml-file and copy molfit file to temp directory
            ok, dbFilename, MolfitsFile, PureNameMolfitsFile = task_myXCLASSFit.CheckMolfitFile(MolfitsFileName, NameOfFunction, CurrentDir, \
                                                                                                MAGIXjobDir, FreqMinList, FreqMaxList, dbList)
            if (ok == 1):                                                                   ## an error occurred
                return MAGIXjobDir

            # Debug:
            # print "ok = ", ok
            # print "MolfitsFile = ", MolfitsFile
            # print "PureNameMolfitsFile = ", PureNameMolfitsFile


            ## update path of database file
            if (dbList == [] or dbList == [""]):
                dbList = [dbFilename]
            task_MAGIX.WriteXMLtag(MAGIXExpXML, "dbFilename", dbList)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get flag for iso file
            IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "Isotopologues")
            if (len(IsoFlag) == 0):
                IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "iso_flag")
                if (len(IsoFlag) == 0):
                    IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "ISO_FLAG")
                    if (len(IsoFlag) == 0):
                        IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "Iso_Flag")
                        if (len(IsoFlag) == 0):
                            IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "Iso_flag")
            if (len(IsoFlag) > 0):
                IsoFlag = IsoFlag[0].lower()
                if (IsoFlag == "y" or IsoFlag == "t" or IsoFlag == "true"):
                    IsoFlag = "y"
                else:
                    IsoFlag = "n"
            else:
                IsoFlag = "n"

            # Debug:
            # print 'IsoFlag = ', IsoFlag


            ## get name of iso file
            if (IsoFlag == "y"):
                IsoFile = task_MAGIX.GetXMLtag(MAGIXExpXML, "IsoTableFileName")

                # Debug:
                # print 'IsoFile = ', IsoFile

                if (len(IsoFile) > 0):
                    IsoFile = IsoFile[0]
                    if (IsoFile.strip() != ""):
                        if (IsoFile[0] != "/"):                                             ## make relative path absolute
                            IsoTableFileName = CurrentDir + IsoFile
                        else:
                            IsoTableFileName = IsoFile
                        if not(os.path.exists(IsoTableFileName) or os.path.isfile(IsoTableFileName)):
                            print " "
                            print " "
                            print "Error in XCLASS package, function task_myXCLASSMapFit.myXCLASSMapFit:"
                            print "  The given path and name of the iso file " + IsoTableFileName
                            print "  does not exsist!"
                            print " "
                            print "  Please note, if a relative path is specified, the path has to be defined relative to the"
                            print "  current working directory!"
                            print " "
                            print "  Please enter an exsisting path and name of iso file and redo myXCLASSMapFit function call!"
                            return MAGIXjobDir


                        ## copy iso ratio file to current job directory
                        command_string = "cp " + IsoTableFileName + " " + MAGIXjobDir               ## copy experimetnal dat files to temp directory
                        os.system(command_string)
                        i = IsoTableFileName.rfind("/")
                        if (i < 0):
                            i = 0
                        IsoTableFileName = MAGIXjobDir + IsoTableFileName[i + 1:]                            ## define new iso file name
                        IsoTableFileNameList = [IsoTableFileName]

                        # Debug:
                        # print 'IsoTableFileNameList = ', IsoTableFileNameList


                        ## modify experimental xml-file in MAGIX temp directory
                        task_MAGIX.WriteXMLtag(MAGIXExpXML, "IsoTableFileName", IsoTableFileNameList)

                        # Debug:
                        #    print "MolfitsFile = ", MolfitsFile
                        #    print "IsoTableFileName = ", IsoTableFileName


                        ## check molecules
                        SelectIsoRatioTable = task_myXCLASSFit.CheckMolecules(MolfitsFile, IsoTableFileName, FreqMinList, FreqMaxList, dbList)


                        ## write new iso ratio file
                        NewIsoRatioFile = open(IsoTableFileName, 'w')
                        NewLine = "%-45s %-45s %25s %25s %25s" % ("% isotopologue:", "Iso-master:", "Iso-ratio:", "Lower-limit:", "upper-limit:")
                        NewIsoRatioFile.write(NewLine + "\n")
                        for IsoLine in SelectIsoRatioTable:
                            if (len(IsoLine) > 3):
                                NewLine = "%-45s %-45s %25s %25s %25s" % (IsoLine[0], IsoLine[1], IsoLine[2], IsoLine[3], IsoLine[4])
                            else:
                                NewLine = "%-45s %-45s %25s" % (IsoLine[0], IsoLine[1], IsoLine[2])
                            NewIsoRatioFile.write(NewLine + "\n")
                        NewIsoRatioFile.close()


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get size of telescope for each spectrum
            GlobalvLSRList = task_MAGIX.GetXMLtag(MAGIXExpXML, "GlobalvLSR")
            SizeTelescopePerSpectrum = task_MAGIX.GetXMLtag(MAGIXExpXML, "TelescopeSize")
            InterFlagPerSpectrum = task_MAGIX.GetXMLtag(MAGIXExpXML, "Inter_Flag")


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get number of experimental data files
            NumberSpectra = int(task_MAGIX.GetXMLtag(MAGIXExpXML, "NumberExpFiles")[0])

            # Debug:
            # print "NumberSpectra = ", NumberSpectra


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get value of threshold for each freqeuncy range (continuum subtracted)
            ListOfThresholds = task_MAGIX.GetXMLtag(MAGIXExpXML, "Threshold")               ## get threshold values for each freq. range
            ListOfFreqMin = task_MAGIX.GetXMLtag(MAGIXExpXML, "MinExpRange")                ## get min. freq. values for each freq. range
            ListOfFreqMax = task_MAGIX.GetXMLtag(MAGIXExpXML, "MaxExpRange")                ## get max. freq. values for each freq. range
            ListOfTBack = task_MAGIX.GetXMLtag(MAGIXExpXML, "BackgroundTemperature")        ## get background temperature values for each freq. range
            ListOfTSlope = task_MAGIX.GetXMLtag(MAGIXExpXML, "TemperatureSlope")            ## get temperature slope values for each freq. range
            if (len(ListOfThresholds) > 0 and len(ListOfThresholds) != len(ListOfFreqMin)): ## Is threshold for each freq. range defined?
                print " "                                                                   ## (Does not work, if one range does not contain a value
                print " "                                                                   ## and another range contains two definitions!!!)
                print "Error in XCLASS package, function task_myXCLASSMapFit.myXCLASSMapFit:"
                print "  Using parameter 'experimentalData' for defining path and name of a xml file requires"
                print "  the definition of the threshold for each frequency range."
                print " "
                print "  Please check the definitions in the defined xml file and restart myXCLASSMapFit function!"
                print " "
                return MAGIXjobDir


            ## check, if user defined a XML file without a threshold, but specifies a general threshold using the threshold input parameter
            if (len(ListOfThresholds) == 0 and Threshold != -1.e99):                        ## user defined a XML file and a general threshold value
                ListOfThresholds = []                                                       ## create a list containing the general threshold value for
                for i in xrange(len(ListOfFreqMin)):                                        ## each freq. range
                    j = ListOfFreqMin[i][0]
                    ListOfThresholds.append([j, Threshold])


            ## check, if user defined a XML file without a background temperature, but specifies a general background temperature using input parameter
            if (len(ListOfTBack) == 0):                                                     ## user defined a XML file and a general tBack value
                ListOfTBack = []                                                            ## create a list containing the general tBack value for
                for i in xrange(len(ListOfFreqMin)):                                        ## each freq. range
                    j = ListOfFreqMin[i][0]
                    ListOfTBack.append([j, tBack])


            ## check, if user defined a XML file without a temperature slope, but specifies a general temperature slope using input parameter
            if (len(ListOfTSlope) == 0):                                                    ## user defined a XML file and a general tslope value
                ListOfTSlope = []                                                           ## create a list containing the general tslope value for
                for i in xrange(len(ListOfFreqMin)):                                        ## each freq. range
                    j = ListOfFreqMin[i][0]
                    ListOfTSlope.append([j, tslope])


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## copy the files defined in the xml-file to the current MAGIX working directory
            ExpFiles = []
            ExpFilesOrig = task_MAGIX.GetXMLtag(MAGIXExpXML, "FileNamesExpFiles")           ## get path and names of all exp. data files
            for filename in ExpFilesOrig:                                                   ## loop over all files
                if (filename[0] != "/"):                                                    ## make relative path absolute
                    filename = CurrentDir + filename                                        ## define absolute path of current exp. data file
                if not(os.path.exists(filename) or os.path.isfile(filename)):               ## check, if current exp. data file exists
                    print " "
                    print " "
                    print "Error in XCLASS package, function task_myXCLASSMapFit.myXCLASSMapFit:"
                    print "  The given path and name of the experimental data file " + filename
                    print "  defined in the experimental data xml-file does not exsist!"
                    print " "
                    print "  Please note, if a relative path is specified, the path has to be defined relative to the"
                    print "  current working directory!"
                    print " "
                    print "  Please enter an exsisting path and name and redo myXCLASSMapFit function call!"
                    return MAGIXjobDir
                ExpFiles.append(filename)                                                   ## save name(s) in list

            # Debug:
            # print 'ExpFiles = ', ExpFiles


            ## get path and name of first experimental data file
            if (ExpFiles != []):
                DataFileName = ExpFiles
            else:
                print " "
                print " "
                print "Error in XCLASS package, function task_myXCLASSMapFit.myXCLASSMapFit:"
                print "  Can not find the experimental data files defined in the xml file!"
                print " "
                print "  Please enter an exsisting path and name and redo myXCLASSMapFit function call!"
                return MAGIXjobDir
        else:                                                                               ## file is not an xml-file


            ## define name of spectrum
            DataFileName = [filename]


            ## define size of telescope
            GlobalvLSRList = [vLSR]
            SizeTelescopePerSpectrum = [TelescopeSize]
            InterFlagPerSpectrum = [Inter_Flag]


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## if FreqMin and FreqMax is not defined or equal, get min and max. frequency for current data file
            if (FreqMin == FreqMax):


                ## convert measurment set file to fits format
                if (filename.endswith(".ms")):
                    print "Convert measurment set to fits format .."
                    try:
                        import casa                                                             ## import casa package
                        casa.exportfits(imagename=filename, fitsimage=filename + '.fits')
                        filename = filename + '.fits'
                        print "done!"
                    except ImportError:
                        print " "
                        print " "
                        print "Error in XCLASS package, function task_myXCLASSMapFit.myXCLASSMapFit:"
                        print "  Can not import CASA python package to convert measurment set file to fits format!"
                        print " "
                        print "  Please enter name of an experimental data file which ends with .fits and redo myXCLASSMapFit function call!"
                        return MAGIXjobDir

                elif (not filename.endswith("fits")):
                    print " "
                    print " "
                    print "Error in XCLASS package, function task_myXCLASSMapFit.myXCLASSMapFit:"
                    print "  The name of the experimental data file does not end with .fits!"
                    print " "
                    print "  Please enter name of an experimental data file which ends with .fits or .ms and redo myXCLASSMapFit function call!"
                    return MAGIXjobDir


                ## determine frequency ranges for each data file
                HDU = 0
                print_flag = "false"
                ok, NumDimensionsLocal, lengthLocal, lengthFITSDATALocal, CParameterLocal, BParameterLocal, ValueOfFirstPointListLocal, \
                                                                                    fitsdataLocal = LoadFits(filename, HDU, print_flag)

                ## append parameters for each FITS data cube to global lists
                CDELTListALL = CParameterLocal[2]
                CUNITListLOCAL = CParameterLocal[4]
                BSCALEALL = BParameterLocal[0]
                BZEROALL = BParameterLocal[1]
                RESTFRQALL = BParameterLocal[3]


                # determine conversion factor to convert the third axis to MHz
                unit = CUNITListLOCAL[2].strip()                                                ## get unit description from FITS header
                UnitConversionFactor = ConvertUnit(unit)


                ## determine FreqMin and FreqMax
                FreqLowest = 1.e99
                FreqHighest = -1.e99
                MaxIntSpec = -1.e99
                for zz in xrange(lengthLocal[2]):                                           ## loop over all frequencies
                    z = zz
                    ValueOfFirstPoint3 = ValueOfFirstPointListLocal[2]


                    ## determine the current data point in MHz
                    point = (ValueOfFirstPoint3 + (z * CDELTListALL[2])) * BSCALEALL + BZEROALL
                    if (UnitConversionFactor == "velocityKMS"):                             ## if necessary change to velocities
                        # Freq = ckms * (float(RESTFRQALL) - point) / point
                        Freq = float(RESTFRQALL) / ckms * (ckms - point)
                    elif (UnitConversionFactor == "velocityMS"):
                        # Freq = ckms * (float(RESTFRQALL) - point) / point
                        Freq = float(RESTFRQALL) / cms * (cms - point)
                    else:
                        Freq = point * UnitConversionFactor


                    ## determine min. and max. frequency of spectrum
                    if (Freq < FreqLowest):
                        FreqLowest = Freq
                        FreqLowestRaw = (ValueOfFirstPoint3 + (z * CDELTListALL[2]))
                    if (Freq > FreqHighest):
                        FreqHighest = Freq
                        FreqHighestRaw = (ValueOfFirstPoint3 + (z * CDELTListALL[2]))


                ## store min and max frequency
                FreqMin = FreqLowest
                FreqMax = FreqHighest


            ## store user input parameters into working variabes
            ListOfThresholds = [[1, Threshold]]
            ListOfFreqMin = [[1, FreqMin]]
            ListOfFreqMax = [[1, FreqMax]]
            ListOfTBack = [[1, tBack]]
            ListOfTSlope = [[1, tslope]]
            dbList = []


            ## analyze molfit parameter: check path and name of the molfit xml-file and copy molfit file to temp directory
            ok, dbFilename, MolfitsFile, PureNameMolfitsFile = task_myXCLASSFit.CheckMolfitFile(MolfitsFileName, NameOfFunction, CurrentDir, \
                                                                                                MAGIXjobDir, ListOfFreqMin, ListOfFreqMax, dbList)
            if (ok == 1):                                                                   ## an error occurred
                return MAGIXjobDir

            # Debug:
            # print "ok = ", ok
            # print "MolfitsFile = ", MolfitsFile
            # print "PureNameMolfitsFile = ", PureNameMolfitsFile


            ## update path of database file
            if (dbList == [] or dbList == [""]):
                dbList = [dbFilename]


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## define iso flag
            IsoFlag = "y"
            if (IsoTableFileName.strip() == ""):
                IsoFlag = "n"
                IsoTableFileName = ""
            else:


                ## check path and name of iso table file
                IsoTableFileName = IsoTableFileName.strip()
                if (IsoTableFileName[0] != "/"):
                    IsoTableFileName = CurrentDir + IsoTableFileName
                if not(os.path.exists(IsoTableFileName) or os.path.isfile(IsoTableFileName)):
                    print " "
                    print " "
                    print "Error in XCLASS package, function task_myXCLASSMapFit.myXCLASSMapFit:"
                    print "  The given path and name of the iso table file " + IsoTableFileName + " does not exsist!"
                    print " "
                    print "  Please enter an exsisting path and name of the iso table file and redo myXCLASSMapFit function call!"
                    return MAGIXjobDir


                ## check molecules
                SelectIsoRatioTable = task_myXCLASSFit.CheckMolecules(MolfitsFile, IsoTableFileName, ListOfFreqMin, ListOfFreqMax, dbList)


                ## write new iso ratio file
                IsoTableFileName = MAGIXjobDir + "New-Iso-Ratio-File.dat"
                NewIsoRatioFile = open(IsoTableFileName, 'w')
                NewLine = "%-45s %-45s %25s %25s %25s" % ("% isotopologue:", "Iso-master:", "Iso-ratio:", "Lower-limit:", "upper-limit:")
                NewIsoRatioFile.write(NewLine + "\n")
                for IsoLine in SelectIsoRatioTable:
                    if (len(IsoLine) > 3):
                        NewLine = "%-45s %-45s %25s %25s %25s" % (IsoLine[0], IsoLine[1], IsoLine[2], IsoLine[3], IsoLine[4])
                    else:
                        NewLine = "%-45s %-45s %25s" % (IsoLine[0], IsoLine[1], IsoLine[2])
                    NewIsoRatioFile.write(NewLine + "\n")
                NewIsoRatioFile.close()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check if conversion from measurment set to fits format neccessary
        for files in DataFileName:
            if (files.endswith(".ms")):
                print "Convert measurment set to fits format .."
                try:
                    import casa                                                             ## import casa package
                    casa.exportfits(imagename=files, fitsimage=files + '.fits')
                    filename = filename + '.fits'
                    print "done!"
                except ImportError:
                    print " "
                    print " "
                    print "Error in XCLASS package, function task_myXCLASSMapFit.myXCLASSMapFit:"
                    print "  Can not import CASA python package to convert measurment set file to fits format!"
                    print " "
                    print "  Please enter name of an experimental data file which ends with .fits and redo myXCLASSMapFit function call!"
                    return MAGIXjobDir

            elif (not files.endswith("fits")):
                print " "
                print " "
                print "Error in XCLASS package, function task_myXCLASSMapFit.myXCLASSMapFit:"
                print "  The name of the experimental data file does not end with .fits!"
                print " "
                print "  Please enter name of an experimental data file which ends with .fits or .ms and redo myXCLASSMapFit function call!"
                return MAGIXjobDir

        # Debug:
        #    print "NumberSpectra = ", NumberSpectra
        #    print "DataFileName = ", DataFileName
        #    print "ListOfThresholds = ", ListOfThresholds
        #    print "ListOfFreqMin = ", ListOfFreqMin
        #    print "ListOfFreqMax = ", ListOfFreqMax
        #    print "ListOfTBack = ", ListOfTBack
        #    print "ListOfTSlope = ", ListOfTSlope


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## Import data from FITS file
        UnitConversionFactorALL = []
        NumDimensionsALL = []
        lengthALL = []
        lengthFITSDATAALL = []
        CRPIXListALL = []
        CRVALListALL = []
        CDELTListALL = []
        CROTListALL = []
        CUNITListALL = []
        BSCALEALL = []
        BZEROALL = []
        BUNITALL = []
        RESTFRQALL = []
        ValueOfFirstPointListALL = []
        fitsdataALL = []
        counterDataCubes = (-1)                                                             ## reset counter for data cubes
        for DataCubeFile in DataFileName:                                                   ## loop over all data files
            counterDataCubes += 1
            if ((counterDataCubes + 1) == len(DataFileName)):
                print "\rImport data form FITS file ", DataCubeFile, "..",
            else:
                # print "\rImport data form FITS file ", DataCubeFile, "..                                                   ",
                print "\rImport data form FITS file ", DataCubeFile, "..",
            HDU = 0
            print_flag = "false"
            ok, NumDimensionsLocal, lengthLocal, lengthFITSDATALocal, CParameterLocal, BParameterLocal, ValueOfFirstPointListLocal, \
                                                                                fitsdataLocal = LoadFits(DataCubeFile, HDU, print_flag)


            ## check, if each data cube describe the same map, i.e. right ascension and declination are identical
            ## has to be done ............................................................................................................................


            ## append parameters for each FITS data cube to global lists
            NumDimensionsALL.append(NumDimensionsLocal)
            lengthALL.append(lengthLocal)
            lengthFITSDATAALL.append(lengthFITSDATALocal)
            CRPIXListALL.append(CParameterLocal[0])
            CRVALListALL.append(CParameterLocal[1])
            CDELTListALL.append(CParameterLocal[2])
            CROTListALL.append(CParameterLocal[3])
            CUNITListLOCAL = CParameterLocal[4]
            CUNITListALL.append(CParameterLocal[4])
            BSCALEALL.append(BParameterLocal[0])
            BZEROALL.append(BParameterLocal[1])
            BUNITALL.append(BParameterLocal[2])
            RESTFRQALL.append(BParameterLocal[3])
            ValueOfFirstPointListALL.append(ValueOfFirstPointListLocal)
            fitsdataALL.append(fitsdataLocal)


            # determine conversion factor to convert the third axis to MHz
            unit = CUNITListLOCAL[2].strip()                                                ## get unit description from FITS header
            UnitConversionFactor = ConvertUnit(unit)
            UnitConversionFactorALL.append(UnitConversionFactor)

            # Debug:
            # print "UnitConversionFactor = ", UnitConversionFactor


            ## we've done
            print "done!                                                                      "
    else:
        print " "
        print " "
        print "Error in XCLASS package, function task_myXCLASSMapFit.myXCLASSMapFit:"
        print "  The parameter experimentalData has to define a path and name of an FITS file!"
        print " "
        print "  Please correct this input parameter and redo myXCLASSMapFit function call!"
        return MAGIXjobDir

    # Debug:
    # print "\n\n\nlength = ", length
    # print "lengthFITSDATA = ", lengthFITSDATA
    # return MAGIXjobDir


    ## get parameter for the first data cube
    NumDimensions = NumDimensionsALL[0]
    length = lengthALL[0]
    lengthFITSDATA = lengthFITSDATAALL[0]
    CRPIXList = CRPIXListALL[0]
    CRVALList = CRVALListALL[0]
    CDELTList = CDELTListALL[0]
    CROTList = CROTListALL[0]
    CUNITList = CUNITListALL[0]
    BSCALE = BSCALEALL[0]
    BZERO = BZEROALL[0]
    BUNIT = BUNITALL[0]
    RESTFRQ = RESTFRQALL[0]
    UnitConversionFactor = UnitConversionFactorALL[0]
    ValueOfFirstPointList = ValueOfFirstPointListALL[0]
    fitsdata = fitsdataALL[0]


    ## check if datacube contains enough data
    if (NumDimensions < 3):
        print " "
        print " "
        print "Error in XCLASS package, function task_myXCLASSMapFit.myXCLASSMapFit:"
        print "  The dimension of the FITS data file is less than 3!"
        print " "
        print "  Please correct this input parameter and redo myXCLASSMapFit function call!"
        return MAGIXjobDir


    ##====================================================================================================================================================
    ## analyze parameter regionFileName
    print "Analyze region file ..",
    TotalNumberPixels, RegionPixelsX, RegionPixelsY, CentralPixelOfRegionX, CentralPixelOfRegionY, RegionMask = ReadRegionFile(regionFileName, length, \
                                                                                                                   ValueOfFirstPointList, CDELTList, \
                                                                                                                   CRVALList, CRPIXList)
    print "done!"


    ## check, if selected region contains a pixel
    if (TotalNumberPixels == 0):
        print "\n\n\tError in myXCLASSMapFit function:"
        print "\tThe selected region does not contain a pixel. Please use another region file"
        print "\tand execute the myXCLASSMapFit function again!"
        return MAGIXjobDir


    ## print some status message to screen
    print "\n\nTotal number of pixels = ", TotalNumberPixels
    print "Number of columns of region = ", len(RegionPixelsX)
    print "Number of rows of region = ", len(RegionPixelsY)

    # Debug:
    # print "length = ", length
    # RegionPixelsX = [RegionPixelsX[0]]
    # RegionPixelsY = [RegionPixelsY[0]]
    # print "FreqMin, FreqMax = ", FreqMin, FreqMax


    ## get number of processors used by the algorithms
    AlgorithmXMLFile = AlgorithmXMLFile.strip()
    if (AlgorithmXMLFile != ""):


        ## if neccessary, make relative path absolute
        if (AlgorithmXMLFile[0] != "/"):
            AlgorithmXMLFile = CurrentDir + AlgorithmXMLFile
        filename = AlgorithmXMLFile


        ## check if path and name of the experimental file exsits
        if not(os.path.exists(filename) or os.path.isfile(filename)):
            print " "
            print " "
            print "Error in XCLASS package, function task_myXCLASSMapFit.myXCLASSMapFit:"
            print "  The given path and name of the algorithm xml-file does not exsist!"
            print " "
            print "  Please enter a valid path and name of an algorithm xml-file and redo myXCLASSMapFit function call!"
            return MAGIXjobDir


    ##====================================================================================================================================================
    ## prepare files for parallel computation
    if (NumberProcessors > 1):                                                              ## is parallel computation selected


        ## get max. number of processors from algorithm xml file
        if (AlgorithmXMLFile != ""):
            NumProcList = task_MAGIX.GetXMLtag(filename, "NumberProcessors")
            NumProc = max(NumProcList)
        else:
            NumProc = 4


        ## get total number of processors available on the current machine
        if (MultiprocessingMethod == "multiprocessing"):
            NumberAvailProc = multiprocessing.cpu_count()

            # Debug:
            # print "NumberAvailProc = ", NumberAvailProc


            ## check, if total number of processors is larger than number of available processors
            if (NumberProcessors * NumProc > NumberAvailProc):
                if (printflag != "false"):
                    print "\t WARNING:"
                    print "\t The required number of processors (%i) " % NumberProcessors * NumProc
                    print "\t is larger than the number of available processors (%i). " % NumberAvailProc
                    print "\t Reduce number of processors to the number of available processors."
                    print " "
                NumProcessors = int(NumberAvailProc / NumProc)
                if (NumProcessors == 0):
                    NumProcessors = 1
            else:
                NumProcessors = NumberProcessors
        else:
            NumProcessors = NumberProcessors
    else:
        NumProcessors = NumberProcessors


    ## settings for multiprocessing package
    if (len(RegionPixelsX) < NumProcessors):
        NumProcessors = len(RegionPixelsX)

    # Debug:
    # print "NumProcessors = ", NumProcessors


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create a subdirectory for all png files of all pixel directory
    PNGDirectory = MAGIXjobDir + "Pixel-Plots/"
    command_string = "mkdir -p " + PNGDirectory
    os.system(command_string)


    ##====================================================================================================================================================
    ## start loop over all pixels of map
    if (FreqMin > FreqMax):                                                                 ## check frequency range
        FreqMin = 0                                                                         ## use all freqeuncy (min. freq. to 0)
        FreqMax = 1e99                                                                      ## use all freqeuncy (max. freq. to 1.e99)
    FitFlagTable = numpy.zeros((len(RegionPixelsY), len(RegionPixelsX)), dtype=numpy.int)   ## create array for all pixels of the selected region
    counterPixels = 0                                                                       ## reset counter for total number of pixel
    NumFrequenciesALL = []                                                                  ## initialize list for number of frequency points
    counterY = (-1)                                                                         ## reset counter for y pixels
    for yPoints in RegionPixelsY:                                                           ## loop over all rows of the selected region
        yPixels = yPoints[0] * BSCALE + BZERO                                               ## apply scale factor and zero offset ((FITS) for y pixel
        y = yPoints[1]                                                                      ## get y number
        counterY += 1                                                                       ## increase counter for y pixels
        counterX = (-1)                                                                     ## reset counter for x pixels
        ListPixelPath = []                                                                  ## initialize list for paths of pixel directories
        print "\n\n"                                                                        ## print something to screen
        NumFittedXPixels = 0
        for xPoints in RegionPixelsX:                                                       ## loop over all columns of the selected region
            xPixels = xPoints[0] * BSCALE + BZERO                                           ## apply scale factor and zero offset ((FITS) for x pixel
            x = xPoints[1]                                                                  ## get x number
            counterX += 1                                                                   ## increase counter for x pixels
            counterPixels += 1                                                              ## increase counter for total number of pixels


            ## define pixel directory
            xString = "%.10f" % xPixels
            yString = "%.10f" % yPixels
            # PixelDirectory = MAGIXjobDir + "Pixel-Fits/" + xString + "__" + yString + "/"
            PixelDirectory = MAGIXjobDir + "Pixel-Fits/" + xString + "__" + yString + "___" + str(counterX) + "_-_" + str(counterY) + "/"
            ListPixelPath.append(PixelDirectory)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## check, if current pixel is contained in the selected region
            FitFlagTable[counterY, counterX] = 0                                            ## set FitFlag for current pixel first to no (default)
            if (RegionMask[counterY, counterX] == 1):                                       ## if RegionMask[counterY, counterX] == 1, pixel is included
                FitFlagTable[counterY, counterX] = 1


                ## print current pixel
                print_string = "\rPrepare files for fit (" + str(counterPixels) + "/" + str(TotalNumberPixels)
                print_string += ") for pixel (" + str(xPixels) + "/" + str(yPixels) + ") ..                                                            "
                # sys.stdout.write(print_string)                                              ## write message immediately to screen


                ## define name of current pixel directory and create it
                command_string = "mkdir -p " + PixelDirectory
                os.system(command_string)


                ## copy molfit, molecule file to current pixel directory
                if (counterY == 0 or UsePreviousResults == False or UsePreviousResults == "f" or UsePreviousResults == "false" \
                    or UsePreviousResults == "False"):
                    cmd_string = "cp " + MolfitsFile + " " + PixelDirectory                 ## for the first row, use user molfit file
                    os.system(cmd_string)                                                   ## copy molfit file to current pixel directory
                else:                                                                       ## copy final molfit file of previous row to current dir.
                    PrevPixelY = RegionPixelsY[counterY - 1][0] * BSCALE + BZERO            ## define previous y pixel

                    # Debug:
                    # print "\nPrevPixelY = ", PrevPixelY


                    ## define previous pixel directory
                    xString = "%.10f" % xPixels
                    yString = "%.10f" % PrevPixelY
                    # PreviousPixelDirectory = MAGIXjobDir + "Pixel-Fits/" + xString + "__" + yString + "/"
                    PreviousPixelDirectory = MAGIXjobDir + "Pixel-Fits/" + xString + "__" + yString
                    PreviousPixelDirectory += "___" + str(counterX) + "_-_" + str(counterY - 1) + "/"


                    ## determine the best result from the previous pixel fit
                    ConvertLogLinFlag = "false"
                    ExpFilesOrig = []
                    BestChi2Value, InputFile, ListOfParamXMLFiles, ListOfModelFuncFiles = GetMolfitFileOfBestResult(PreviousPixelDirectory, \
                                                                                                                    ConvertLogLinFlag, ExpFilesOrig)
                    if (InputFile != []):


                        ## write new contents of previous molfit file to new input file
                        NewInputFile = open(PixelDirectory + PureNameMolfitsFile, 'w')
                        for line in InputFile:
                            NewInputFile.write(line)
                        NewInputFile.close()
                    else:
                        print "\n\nWARNING:  Can not determine the optimized molfit file from the pixel fit of the previous line."
                        print "          Use user defined molfit file instead.\n"


                        ## copy user defined molfit file to current pixel directory
                        cmd_string = "cp " + MolfitsFile + " " + PixelDirectory             ## for the first row, use user molfit file
                        os.system(cmd_string)                                               ## copy molfit file to current pixel directory


                ## copy iso table file to current pixel directory
                IsoTableLocalFileName = ""
                if (os.path.exists(IsoTableFileName) and os.path.isfile(IsoTableFileName)): ## is a iso file defined?
                    cmd_string = "cp " + IsoTableFileName + " " + PixelDirectory            ## copy iso file to current pixel directory
                    os.system(cmd_string)


                    ## determine path and name of iso file
                    i = IsoTableFileName.rfind("/")
                    if (i > 0):
                        CurrentIsoTableFileName = IsoTableFileName[i + 1:]
                    else:
                        CurrentIsoTableFileName = IsoTableFileName
                    IsoTableLocalFileName = PixelDirectory + CurrentIsoTableFileName

                # Debug:
                # print "PixelDirectory = ", PixelDirectory
                # print "length = ", length
                # print "ValueOfFirstPointList = ", ValueOfFirstPointList
                # print "IsoTableLocalFileName = ", IsoTableLocalFileName


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## construct spectra file(s) for current pixel
                NewSpectraFileNames = []
                PixelIncludeFlag = "false"
                for specindex in xrange(NumberSpectra):                                     ## loop over all spectra
                    NumString = "%05i" % (specindex + 1)
                    SpecFile = PixelDirectory + "Spectrum__" + NumString + ".dat"           ## define name of ASCII file containing spectra
                    NewSpectraFileNames.append(SpecFile)                                    ## save name
                    DatFile = open(SpecFile, "w")                                           ## open ASCII file
                    CountFreq = 0
                    FreqLowest = 1.e99
                    FreqHighest = -1.e99
                    MaxIntSpec = -1.e99
                    for zz in xrange(lengthALL[specindex][2]):                              ## loop over all frequencies
                        z = zz
                        ValueOfFirstPoint3 = ValueOfFirstPointListALL[specindex][2]


                        ## determine the current data point in MHz
                        point = (ValueOfFirstPoint3 + (z * CDELTListALL[specindex][2])) * BSCALEALL[specindex] + BZEROALL[specindex]
                        if (UnitConversionFactorALL[specindex] == "velocityKMS"):           ## if necessary change to velocities
                            # Freq = ckms * (float(RESTFRQALL[specindex]) - point) / point
                            Freq = float(RESTFRQALL[specindex]) / ckms * (ckms - point)
                        elif (UnitConversionFactorALL[specindex] == "velocityMS"):
                            # Freq = cms * (float(RESTFRQALL[specindex]) - point) / point
                            Freq = float(RESTFRQALL[specindex]) / cms * (cms - point)
                        else:
                            Freq = point * UnitConversionFactorALL[specindex]


                        ## determine min. and max. frequency of spectrum
                        if (Freq < FreqLowest):
                            FreqLowest = Freq
                            FreqLowestRaw = (ValueOfFirstPoint3 + (z * CDELTListALL[specindex][2]))
                        if (Freq > FreqHighest):
                            FreqHighest = Freq
                            FreqHighestRaw = (ValueOfFirstPoint3 + (z * CDELTListALL[specindex][2]))


                        ## check, if current frequency is within given frequency range
                        if (FreqMin == FreqMax or (Freq >= FreqMin and Freq <= FreqMax)):   ## check, if frequencies are within given frequency range
                            CountFreq += 1
                            if (NumDimensionsALL[specindex] == 3):
                                if (lengthFITSDATAALL[specindex] == lengthALL[specindex]):
                                    intensity = fitsdataALL[specindex][x, y, z]
                                    # line = "%.15e %.16e" % (Freq, intensity)
                                else:
                                    intensity = fitsdataALL[specindex][z, y, x]
                                    # line = "%.15e %.16e" % (Freq, intensity)
                            else:
                                if (lengthFITSDATAALL[specindex] == lengthALL[specindex]):
                                    intensity = fitsdataALL[specindex][0, z, y, x]
                                    # line = "%.15e %.16e" % (Freq, intensity)
                                else:
                                    intensity = fitsdataALL[specindex][0, z, y, x]
                                    # line = "%.15e %.16e" % (Freq, intensity)


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## check, if spectra are not given in Kelvin
                            UnitIntensity = BUNITALL[specindex].lower()
                            UnitIntensity = UnitIntensity.replace(" ", "")
                            OrigIntensity = intensity


                            ## check, if spectra are given in Jy/Beam
                            if (UnitIntensity.strip() == "jy/beam"):
                                ConvertJyPerBeam2Kelvin, ConvertKelvin2JyPerBeam = JanskyPerBeamInKelvin(float(SizeTelescopePerSpectrum[specindex]), \
                                                                                                         InterFlagPerSpectrum[specindex], Freq)
                                intensity = ConvertJyPerBeam2Kelvin * intensity


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## write frequency and intensity to ASCII file
                            line = str(Freq) + "\t\t" + str(intensity)
                            DatFile.write(line + "\n")

                            # Debug:
                            #    print "intensity, MaxIntSpec = ", intensity, MaxIntSpec
                            #    print "Freq = ", Freq
                            #    print "ListOfFreqMin = ", ListOfFreqMin
                            #    print "ListOfFreqMax = ", ListOfFreqMax


                            ## determine max. intensity
                            for i in xrange(len(ListOfFreqMin)):                            ## loop over all freq. ranges
                                freqMinLocal = ListOfFreqMin[i]
                                freqMaxLocal = ListOfFreqMax[i]

                                # Debug:
                                #    print "specindex = ", specindex
                                #    print "freqMinLocal = ", freqMinLocal
                                #    print "freqMaxLocal = ", freqMaxLocal

                                if (freqMinLocal[0] == (specindex + 1) and freqMaxLocal[0] == (specindex + 1)): ## check, if current range definition
                                                                                            ## belongs to the current data file

                                    if ((Freq >= float(freqMinLocal[1]) and Freq <= float(freqMaxLocal[1])) \
                                        or float(freqMinLocal[1]) == float(freqMaxLocal[1])):


                                        ## subtract continuum offset
                                        #    tBackLocal = ListOfTBack[i]
                                        #    tSlopeLocal = ListOfTSlope[i]
                                        #    if (tBackLocal[0] == (specindex + 1) and tSlopeLocal[0] == (specindex + 1)):
                                        #        if (tSlopeLocal[1] != 0.0):                ## subtract continuum if background temp. is defined
                                        #            if (freqMinLocal[1] == freqMaxLocal[1]):
                                        #                intensity = intensity - (abs(tBackLocal[1]) * (Freq / FreqLowest)**tSlopeLocal[1])
                                        #            else:
                                        #                intensity = intensity - (abs(tBackLocal[1])
                                        #                                      * (Freq / freqMinLocal[1])**tSlopeLocal[1])


                                        ## determine max. intensity of current freq. range
                                        if (OrigIntensity >= MaxIntSpec):
                                            MaxIntSpec = OrigIntensity


                    ## close dat file
                    DatFile.close()


                    ## add new method to check, if intensity is above threshold
                    if (len(ListOfThresholds) > 0):                                         ## there are thresholds defined for each freq. ranges
                        for thresLocal in ListOfThresholds:                                 ## loop over all freq. ranges
                            if (thresLocal[0] == (specindex + 1)):                          ## check, if current range definition belongs to the current
                                                                                            ## data file
                                Thres = float(thresLocal[1])
                                if (PixelIncludeFlag == "false" and (Thres == -1.e99 or MaxIntSpec >= Thres)):
                                    PixelIncludeFlag = "true"
                                    break
                    else:
                        Thres = Threshold


                        ## check if intensity is above threshold
                        if (PixelIncludeFlag == "false" and (Thres == -1.e99 or MaxIntSpec >= Thres)):
                            PixelIncludeFlag = "true"

                    # Debug:
                    #    print "Thres = ", Thres
                    #    print "Threshold = ", Threshold
                    #    print "PixelIncludeFlag = ", PixelIncludeFlag
                    #    print "specindex = ", specindex
                    #    print "MaxIntSpec = ", MaxIntSpec
                    #    sys.exit(0)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## include current pixel or not, i.e. is spectrum above threshold or not
                    if (PixelIncludeFlag != "true"):
                        FitFlagTable[counterY, counterX] = 0

                        # Debug:
                        print "\n\n\n\t\tExclude pixel (" + str(xPixels) + ", " + str(yPixels) + ") from fitting because intensity is below given threshold."
                        print "\t\tMax. intensity of current spectrum = ", MaxIntSpec
                        print "\t\tThreshold = ", Thres
                        print "\n\n"


                    ## for the first pixel: save number of frequency points
                    if (counterPixels == 1):
                        NumFrequenciesALL.append(CountFreq)
                        CentralPixelOfFrequency = min(FreqLowestRaw, FreqHighestRaw)

                        # Debug:
                        # print "\n\n\n\n\n\nCentralPixelOfFrequency = ", CentralPixelOfFrequency
                        # print "FreqLowestRaw = ", FreqLowestRaw
                        # print "FreqHighestRaw = ", FreqHighestRaw
                        # print "FreqMin, FreqMax = ", FreqMin, FreqMax


                ##========================================================================================================================================
                ## construct experimental xml-file
                MAGIXExpLocalXML = PixelDirectory + "/exp.xml"

                # Debug:
                # print 'MAGIXExpLocalXML = ', MAGIXExpLocalXML


                ## if experimental data parameter describes path and name of a xml file ..
                if (ParamIsXMLFile == "true"):


                    ## copy xml file to temp directory
                    command_string = "cp " + MAGIXExpXML + " " + MAGIXExpLocalXML
                    os.system(command_string)


                    ## modify path and names of spectra files in copy of experimental XML file
                    task_MAGIX.WriteXMLtag(MAGIXExpLocalXML, "FileNamesExpFiles", NewSpectraFileNames)


                    ## set all import filter in xml file to "xclassASCII"
                    ImportFilter = []
                    for i in xrange(NumberSpectra):
                        ImportFilter.append("xclassASCII")
                    task_MAGIX.WriteXMLtag(MAGIXExpLocalXML, "ImportFilter", ImportFilter)


                    ## update iso flag and path and name of iso table in MAGIX xml file
                    task_MAGIX.WriteXMLtag(MAGIXExpLocalXML, "Isotopologues", [IsoFlag])
                    task_MAGIX.WriteXMLtag(MAGIXExpLocalXML, "IsoTableFileName", [IsoTableLocalFileName])


                    ## update path and name of db-file if path and name is not given in the xml file
                    PathFileNameDBXML = task_MAGIX.GetXMLtag(MAGIXExpXML, "dbFilename")
                    if (PathFileNameDBXML == []):                                           ## add tag dbFilename if it not exists
                        task_MAGIX.WriteXMLtag(MAGIXExpLocalXML, "dbFilename", PathFileNameDB)


                ## continue here, if experimental data file is a
                else:


                    ## determine lowest and highest frequency and stepsize
                    if (FreqMin == FreqMax):
                        FreqMinXML = FreqLowest
                        FreqMaxXML = FreqHighest
                    else:
                        FreqMinXML = FreqMin
                        FreqMaxXML = FreqMax
                    FreqStepXML = abs(CDELTList[2]) * UnitConversionFactor


                    ## create xml file for import of observational data
                    task_myXCLASSFit.CreateExpXMLFile(PixelDirectory, SpecFile, FreqMinXML, FreqMaxXML, FreqStepXML, t_back_flag, tBack, tslope, \
                                                      nH_flag, N_H, beta_dust, kappa_1300, vLSR, TelescopeSize, Inter_Flag, iso_flag, \
                                                      IsoTableLocalFileName, dbFilename)

                ##========================================================================================================================================
                ## construct io_control.xml file


                ## determine path and name of iso file
                MolfitsFileNameXML = PixelDirectory + PureNameMolfitsFile


                ## create xml file for io-control file
                task_myXCLASSFit.CreateIOControlXMLFile(PixelDirectory, MAGIXExpLocalXML, MolfitsFileNameXML, MAGIXrootDir)


                ##========================================================================================================================================
                ## construct fit.xml file
                AlgorithmXMLFile = AlgorithmXMLFile.strip()
                if (AlgorithmXMLFile != ""):


                    ## copy algorithm file to working directory
                    cmd_string = "cp " + filename + " " + PixelDirectory + "algorithm_control.xml"
                    os.system(cmd_string)
                else:
                    task_myXCLASSFit.CreateLMControlXMLFile(PixelDirectory, NumberIteration)


                ## print that prepartion for current pixel is finished
                FirstPixel = counterY * len(ListPixelPath) + 1
                LastPixel = (counterY + 1) * len(ListPixelPath)
                print_string = "\rPrepare files for pixels (" + str(FirstPixel) + " - " + str(LastPixel)
                print_string += ") of " + str(TotalNumberPixels) + " pixel .. done                                                                    \n"
                #sys.stdout.write(print_string)


        ##================================================================================================================================================
        ## start MAGIX for all pixel in the current row simultaneously using the multiprocessing package


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## prepare parameters for parallel MAGIX run
        os.environ["OMP_STACKSIZE"] = '6999M'
        NumxPixel = len(ListPixelPath)
        InputPaths = []
        PixelList = []
        LastLastPixel = counterY * len(ListPixelPath)
        PixelCounter = 0
        for c in xrange(NumxPixel):
            PixelCounter += 1

            # Debug:
            # print "PixelCounter = ", PixelCounter


            ## check, if current pixel should be fitted or not
            if (FitFlagTable[counterY, PixelCounter - 1] == 1):
                CurrentXPixelPath = ListPixelPath[PixelCounter - 1]
                FirstPixel = counterY * len(ListPixelPath) + PixelCounter


                ## save path and name of io-control file
                RelativeCurrentXPixelPath = CurrentXPixelPath.replace(MAGIXjobDir, "")
                InputPaths.append(RelativeCurrentXPixelPath)
                PixelList.append(FirstPixel)

                # Debug:
                # print "CurrentXPixelPath = ", CurrentXPixelPath


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## start MAGIX instances


        ## in parallel
        if (NumProcessors > 1):
            MAGIXOption = "quiet"
            for i in xrange(0, len(InputPaths), NumProcessors):


                ## number of workers
                NumberWorkers = 0
                if ((i + NumProcessors) <= len(InputPaths)):
                    NumberWorkers = NumProcessors
                else:
                    NumberWorkers = len(InputPaths) % NumProcessors

                # Debug:
                # print "NumberWorkers = ", NumberWorkers


                ## print what you do
                FirstPixel = PixelList[i]
                LastPixel = FirstPixel + NumberWorkers - 1
                print_string = "\rStart MAGIX for pixels (" + str(FirstPixel) + " - " + str(LastPixel)
                print_string += ") of " + str(TotalNumberPixels) + " pixel ..                                                         \n\n"
                sys.stdout.write(print_string)


                ## call MAGIX starter
                ok = StartClusterParallelization(CurrentDir, myXCLASSrootDir, MAGIXjobDir, RelativeMAGIXDir, ClusterFlag, NumberWorkers, \
                                                 InputPaths[i:(i + NumberWorkers)], ServerList)


        ## start serial MAGIX instance
        else:
            MAGIXOption = "plotsaveonly, model=myxclass"
            for i in xrange(0, len(InputPaths)):


                ## print what you do
                FirstPixel = PixelList[i]
                print_string = "\rStart MAGIX for pixel (" + str(FirstPixel) + "/" + str(TotalNumberPixels) + ") ..                       \n\n"
                sys.stdout.write(print_string)

                # Debug:
                # print "path = ", MAGIXjobDir + "/" + InputPaths[i] + "/io_control.xml"


                ## call MAGIX starter
                ok = 0
                ok = task_MAGIX.StartMAGIX(MAGIXrootDir, MAGIXOption, MAGIXjobDir + "/" + InputPaths[i] + "/io_control.xml")
                if (ok != 0):
                    print "\n\n\n\t Program MAGIX aborted!\n\n"
        # Debug:
        # sys.exit(0)

    # Debug:
    # return MAGIXjobDir


    ## print what you do
    sys.stdout.write("\rRead in results of all pixel fits and create FITS images and plots ..")
    sys.stdout.flush()


    ##====================================================================================================================================================
    ## get default settings from first cube
    NumDimensions = NumDimensionsALL[0]
    length = lengthALL[0]
    lengthFITSDATA = lengthFITSDATAALL[0]
    CRPIXList = CRPIXListALL[0]
    CRVALList = CRVALListALL[0]
    CDELTList = CDELTListALL[0]
    CROTList = CROTListALL[0]
    CUNITList = CUNITListALL[0]
    BSCALE = BSCALEALL[0]
    BZERO = BZEROALL[0]
    BUNIT = BUNITALL[0]
    RESTFRQ = RESTFRQALL[0]
    UnitConversionFactor = 1.0
    ValueOfFirstPointList = ValueOfFirstPointListALL[0]
    fitsdata = fitsdataALL[0]


    ## construct body of FITS files
    NumXPixels = len(RegionPixelsX)
    NumYPixels = len(RegionPixelsY)
    NumFrequencies = max(NumFrequenciesALL[:])
    ModelValues = numpy.zeros((NumXPixels, NumYPixels, NumFrequencies, NumberSpectra), dtype=numpy.float32)
    if (lengthFITSDATA == length):
        chi2Values = numpy.zeros((NumXPixels, NumYPixels), dtype=numpy.float32)
    else:
        chi2Values = numpy.zeros((NumYPixels, NumXPixels), dtype=numpy.float32)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## determine total number of output files for all algorithms
    ListOfKnownParameters = []
    ModelFuncList = []
    ParamMapsList = []
    ParamMapsCoordList = []
    NumberFittedParameter = 0
    counterPixels = 0
    AlgNameList = ["LM"]
    counterX = (-1)
    DoneFlag = "false"
    TotalPixelCounter = (-1)
    for xPoints in RegionPixelsX:
        counterX += 1
        xPixels = xPoints[0] * BSCALEALL[0] + BZEROALL[0]
        x = xPoints[1]
        xIndex = counterX
        counterY = (-1)
        for yPoints in RegionPixelsY:
            counterY += 1
            counterPixels += 1
            yPixels = yPoints[0] * BSCALEALL[0] + BZEROALL[0]
            y = yPoints[1]
            yIndex = counterY


            ## check if pixel is included within region
            if (FitFlagTable[counterY, counterX] == 1):                                     ## if FitFlagTable[counterY, counterX] == 1, pixel is included
                TotalPixelCounter += 1


                ## define pixel directory
                xString = "%.10f" % xPixels
                yString = "%.10f" % yPixels
                # PixelDirectory = MAGIXjobDir + "Pixel-Fits/" + xString + "__" + yString + "/"
                PixelDirectory = MAGIXjobDir + "Pixel-Fits/" + xString + "__" + yString + "___" + str(counterX) + "_-_" + str(counterY) + "/"


                ## get name of exp. data files from xml file
                ExpFiles = task_MAGIX.GetXMLtag(PixelDirectory + "/exp.xml", "FileNamesExpFiles")


                ## determine the best result for the current pixel, i.e. get best molfit with name(s) of corresponding xml and model data file(s)
                ConvertLogLinFlag = "true"
                BestChi2Value, InputFile, ListOfParamXMLFiles, ListOfModelFuncFiles = GetMolfitFileOfBestResult(PixelDirectory, ConvertLogLinFlag, \
                                                                                                                ExpFiles)

                ## check, if fit of current pixel was done successfully
                if (InputFile == []):
                    BestChi2Value = -1.e99      ## NaN
                else:


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## create png file and store in png directory
                    NumPlots = len(ListOfModelFuncFiles)
                    if (NumPlots > 0):
                        fig = pylab.figure(figsize=(15, NumPlots * 10))
                        fig.clear()
                        pylab.subplots_adjust(hspace=0.1, wspace=0.2, bottom=0.1, top=0.9)
                        for PlotIndex in xrange(NumPlots):
                            LocalModelData = numpy.loadtxt(PixelDirectory + ListOfModelFuncFiles[PlotIndex], skiprows = 0)
                            LocalExpData = numpy.loadtxt(ExpFiles[PlotIndex], skiprows = 0)
                            layer = pylab.subplot(NumPlots, 1, (PlotIndex + 1))
                            if (PlotIndex == 0):
                                layer.set_title("Best fit for pixel " + xString + ", " + yString)
                            if (PlotIndex == (NumPlots - 1)):
                                layer.set_xlabel("Rest Frequency [MHz]")
                                layer.xaxis.set_major_formatter(matplotlib.ticker.FormatStrFormatter('%.5e'))
                            layer.set_ylabel(r"T$_{mb}$ [K]")
                            layer.grid(True)
                            layer.plot(LocalExpData[:, 0], LocalExpData[:, 1], '-', color='black', linewidth=3.0, label = 'exp. data', \
                                       drawstyle='steps-mid')
                            layer.plot(LocalModelData[:, 0], LocalModelData[:, 1], '-', color='red', linewidth=2.0, label = 'fit')
                            layer.set_xlim(LocalModelData[0, 0], LocalModelData[-1, 0])
                            layer.legend()
                        # pngFile = PNGDirectory + xString + "__" + yString + ".png"
                        pngFile = PNGDirectory + xString + "__" + yString + "___" + str(counterX) + "_-_" + str(counterY) + ".png"
                        pylab.savefig(pngFile, dpi=300)
                        pylab.draw()
                        matplotlib.pyplot.close(fig)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## read parameter values from xml file
                    ParameterList = ReadParmXMLFile(PixelDirectory, ListOfParamXMLFiles)


                    ## determine the parameter ID for each parameter and store parameter values for current pixel in variable
                    ## ParamMapsList = .. [x-coord., y-coord., [[paramID_1, val_1], [paramID_2, val_2], .. ] ]
                    NewParameterList = []
                    for param in ParameterList:
                        paramMol = param[0]
                        paramComp = param[1]
                        paramName = param[2]
                        paramVal = float(param[3])
                        if (TotalPixelCounter == 0):
                            if (not ([paramName, paramMol, paramComp] in ListOfKnownParameters)):
                                ListOfKnownParameters.append([paramName, paramMol, paramComp])
                        paramID = ListOfKnownParameters.index([paramName, paramMol, paramComp])
                        NewParameterList.append([paramID, paramVal])
                    ParamMapsCoordList.append([xIndex, yIndex])
                    ParamMapsList.append([xIndex, yIndex, NewParameterList])


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## read model function values for different cubes
                    PixelSpectrumList = ReadModelFiles(PixelDirectory, NumberSpectra, NumFrequenciesALL, ListOfModelFuncFiles, CDELTListALL, \
                                                       ValueOfFirstPointListALL, BSCALEALL, BZEROALL, BUNITALL, UnitConversionFactorALL, RESTFRQALL, \
                                                       SizeTelescopePerSpectrum, InterFlagPerSpectrum)
                    for CubeNumLocal in xrange(NumberSpectra):                              ## loop over all data cubes
                        for freq_index in xrange(NumFrequenciesALL[CubeNumLocal]):          ## loop over all freq. points
                            ModelValues[xIndex, yIndex, freq_index, CubeNumLocal] = PixelSpectrumList[CubeNumLocal, freq_index]


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## store chi^2 value
                    if (lengthFITSDATA == length):
                        chi2Values[xIndex, yIndex] = numpy.float32(BestChi2Value)
                    else:
                        chi2Values[yIndex, xIndex] = numpy.float32(BestChi2Value)
            else:
                if (lengthFITSDATA == length):
                    chi2Values[xIndex, yIndex] = numpy.nan
                else:
                    chi2Values[yIndex, xIndex] = numpy.nan

    # Debug:
    # print "ParamMapsList = ", ParamMapsList


    ##====================================================================================================================================================
    ## store parameter values
    NumberFittedParameter = len(ListOfKnownParameters)
    if (lengthFITSDATA == length):
        paramValues = numpy.zeros((NumXPixels, NumYPixels, NumberFittedParameter), dtype=numpy.float32)
    else:
        paramValues = numpy.zeros((NumberFittedParameter, NumYPixels, NumXPixels), dtype=numpy.float32)
    counterX = (-1)                                                                         ## reset counter for x pixels
    for xPoints in RegionPixelsX:                                                           ## loop over all columns of the selected region
        counterX += 1                                                                       ## increase counter for x pixels
        xPixels = xPoints[0] * BSCALE + BZERO                                               ## apply scale factor and zero offset ((FITS) for x pixel
        xIndex = counterX
        counterY = (-1)                                                                     ## reset counter for y pixels
        for yPoints in RegionPixelsY:                                                       ## loop over all rows of the selected region
            counterY += 1                                                                   ## increase counter for y pixels
            yPixels = yPoints[0] * BSCALE + BZERO                                           ## apply scale factor and zero offset ((FITS) for y pixel)
            yIndex = counterY


            ## check if pixel is included within region
            if (FitFlagTable[counterY, counterX] == 1):                                     ## if FitFlagTable[counterY, counterX] == 1, pixel is included


                ## index of current pixel in list ParamMapsList
                try:
                    ListIndex = ParamMapsCoordList.index([xIndex, yIndex])
                except ValueError:
                    ListIndex = (-1)

                # Debug:
                #    print "counterY, counterX = ", counterY, counterX
                #    print "len(ParamMapsCoordList) = ", len(ParamMapsCoordList)
                #    print "ListIndex = ", ListIndex


                ## get parameter list of current coordinate
                if (ListIndex != (-1)):
                    NewParameterList = ParamMapsList[ListIndex][2]


                    ## store parameter value in array
                    for i in xrange(len(NewParameterList)):
                        paramID = NewParameterList[i][0]
                        if (lengthFITSDATA == length):
                            try:
                                paramValues[xIndex, yIndex, paramID] = NewParameterList[i][1]
                            except IndexError:
                                paramValues[xIndex, yIndex, paramID] = numpy.nan
                        else:
                            try:
                                paramValues[paramID, yIndex, xIndex] = NewParameterList[i][1]
                            except IndexError:
                                paramValues[paramID, yIndex, xIndex] = numpy.nan
            else:
                if (lengthFITSDATA == length):
                    paramValues[xIndex, yIndex, :] = numpy.nan
                else:
                    paramValues[:, yIndex, xIndex] = numpy.nan


    ##====================================================================================================================================================
    ## write fits files
    CubeNumber = 0


    ## get rest frequency
    RestFreqLocal = RESTFRQALL[CubeNumber]
    if (RestFreq != 0.0):
        RestFreqLocal = RestFreq


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write new header and new data to new fits file for chi2 values
    Chi2FileName = MAGIXjobDir + "BestResult___chi2values.fits"
    hdu = pyfits.PrimaryHDU(chi2Values)
    FrequencyFlag = "false"
    CentralPixelOfFrequency = 0
    bunitinfo = [' K^2    ', ' ']
    WriteFitsHeader(hdu, BSCALEALL[CubeNumber], BZEROALL[CubeNumber], bunitinfo, RestFreqLocal, lengthALL[CubeNumber], CentralPixelOfRegionX, \
                    CentralPixelOfRegionY, NumXPixels, NumYPixels, CRVALListALL[CubeNumber], CRPIXListALL[CubeNumber], CDELTListALL[CubeNumber], \
                    CROTListALL[CubeNumber], CUNITListALL[CubeNumber], FrequencyFlag, CentralPixelOfFrequency, NameOfFunction)
    pyfits.writeto(Chi2FileName, numpy.float32(hdu.data), hdu.header, clobber=True)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write new header and new data to new fits file for parameter values

    # Debug:
    # print "ListOfKnownParameters = ", ListOfKnownParameters


    ## write parameter values to fits files
    for paramNum in xrange(NumberFittedParameter):
        NameParam = ListOfKnownParameters[paramNum][0]                                      ## get name of current parameter
        MoleculeName = ListOfKnownParameters[paramNum][1]                                   ## get name of molecule for current parameter
        NumberComp = ListOfKnownParameters[paramNum][2]                                     ## get index of component for current parameter

        # Debug:
        # print "\nMoleculeName = ", MoleculeName
        # print "NumberComp = ", NumberComp
        # print "NameParam = ", NameParam


        paramFileName = MAGIXjobDir + "BestResult___parameter__" + NameParam
        if (NameParam == "source_size" or NameParam == "T_rot" or NameParam == "N_tot" or NameParam == "V_width" or NameParam == "V_off" \
            or NameParam == "T_Back" or NameParam == "T_Slope" or NameParam == "nHcolumn" or NameParam == "kappa" or NameParam == "beta"):
            paramFileName += "___molecule__" + MoleculeName + "___component__" + str(NumberComp)
        paramFileName += ".fits"
        if (lengthFITSDATAALL[CubeNumber] == lengthALL[CubeNumber]):
            hdu = pyfits.PrimaryHDU(paramValues[:, :, paramNum])
        else:
            hdu = pyfits.PrimaryHDU(paramValues[paramNum, :, :])


        ## write header
        FrequencyFlag = "false"
        CentralPixelOfFrequency = 0
        UnitParm = task_myXCLASS.GetUnitParameter(NameParam)
        bunitinfo = [UnitParm, 'parameter ' + NameParam]
        WriteFitsHeader(hdu, BSCALEALL[CubeNumber], BZEROALL[CubeNumber], bunitinfo, RestFreqLocal, lengthALL[CubeNumber], CentralPixelOfRegionX, \
                        CentralPixelOfRegionY, NumXPixels, NumYPixels, CRVALListALL[CubeNumber], CRPIXListALL[CubeNumber], CDELTListALL[CubeNumber], \
                        CROTListALL[CubeNumber], CUNITListALL[CubeNumber], FrequencyFlag, CentralPixelOfFrequency, NameOfFunction)
        pyfits.writeto(paramFileName, numpy.float32(hdu.data), hdu.header, clobber=True)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write new header and new data to new fits file for model function values
    for CubeNumber in xrange(NumberSpectra):                                                ## loop over all data cubes


        ## prepare array for fits file
        if (lengthFITSDATAALL[CubeNumber] == lengthALL[CubeNumber]):
            ModelValuesAll = numpy.zeros((NumXPixels, NumYPixels, NumFrequencies), dtype=numpy.float32)
        else:
            ModelValuesAll = numpy.zeros((NumFrequencies, NumYPixels, NumXPixels), dtype=numpy.float32)


        ## the following part take the different order for each cube into account
        counterX = (-1)                                                                     ## reset counter for x pixels
        for xPoints in RegionPixelsX:                                                       ## loop over all columns of the selected region
            counterX += 1                                                                   ## increase counter for x pixels
            xPixels = xPoints[0] * BSCALE + BZERO                                           ## apply scale factor and zero offset ((FITS) for x pixel
            xIndex = counterX
            counterY = (-1)                                                                 ## reset counter for y pixels
            for yPoints in RegionPixelsY:                                                   ## loop over all rows of the selected region
                counterY += 1                                                               ## increase counter for y pixels
                yPixels = yPoints[0] * BSCALE + BZERO                                       ## apply scale factor and zero offset ((FITS) for y pixel)
                yIndex = counterY
                if (FitFlagTable[counterY, counterX] == 1):                                 ## if FitFlagTable[counterY, counterX] == 1, pixel is included
                    for freq_index in xrange(NumFrequenciesALL[CubeNumber]):                ## loop over all freq. points
                        if (lengthFITSDATAALL[CubeNumber] == lengthALL[CubeNumber]):
                            ModelValuesAll[xIndex, yIndex, freq_index] = ModelValues[xIndex, yIndex, freq_index, CubeNumber]
                        else:
                            ModelValuesAll[freq_index, yIndex, xIndex] = ModelValues[xIndex, yIndex, freq_index, CubeNumber]


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## write new header and new data to new fits file for model values
        NameOfCube = DataFileName[CubeNumber]                                               ## get name of current cube
        i = NameOfCube.rfind("/")
        if (i > (-1)):
            NameOfCube = NameOfCube[i + 1:]
            i = NameOfCube.rfind(".")
            if (i > (-1)):
                NameOfCube = NameOfCube[:i]


        ## define name of model function cube
        ModelFunctionFileName = MAGIXjobDir + NameOfCube + "___model-function-values.fits"


        ## get hdu
        hdu = pyfits.PrimaryHDU(ModelValuesAll)


        ## define lowest and highest frequency
        FrequencyFlag = "true"
        if (CDELTListALL[CubeNumber][2] > 0):
            FreqLowestRaw = ValueOfFirstPointListALL[CubeNumber][2]
            FreqHighestRaw = ValueOfFirstPointListALL[CubeNumber][2] + ((lengthALL[CubeNumber][2] - 1) * CDELTListALL[CubeNumber][2])
        else:
            FreqLowestRaw = ValueOfFirstPointListALL[CubeNumber][2] + ((lengthALL[CubeNumber][2] - 1) * CDELTListALL[CubeNumber][2])
            FreqHighestRaw = ValueOfFirstPointListALL[CubeNumber][2]


        ## define frequency of central pixel
        if (CDELTListALL[CubeNumber][2] < 0.0):
            CentralPixelOfFrequency = max(FreqLowestRaw, FreqHighestRaw)
        else:
            CentralPixelOfFrequency = min(FreqLowestRaw, FreqHighestRaw)
        bunitinfo = [BUNITALL[CubeNumber], 'Brightness (pixel) unit']
        # bunitinfo = [' K      ', 'Brightness (pixel) unit']                                 ## use this if spectral intensities are given in Kelvin only


        ## write header
        WriteFitsHeader(hdu, BSCALEALL[CubeNumber], BZEROALL[CubeNumber], bunitinfo, RestFreqLocal, lengthALL[CubeNumber], CentralPixelOfRegionX, \
                        CentralPixelOfRegionY, NumXPixels, NumYPixels, CRVALListALL[CubeNumber], CRPIXListALL[CubeNumber], CDELTListALL[CubeNumber], \
                        CROTListALL[CubeNumber], CUNITListALL[CubeNumber], FrequencyFlag, CentralPixelOfFrequency, NameOfFunction)


        ## write header with model function values to file
        pyfits.writeto(ModelFunctionFileName, numpy.float32(hdu.data), hdu.header, clobber=True)


    ## we've done
    sys.stdout.write("\rRead in results of all pixel fits and create FITS images and plots .. done!                                              \n")
    sys.stdout.flush()


    ## copy iso ratio file to current job directory
    if (ParamIsXMLFile == "true"):
        command_string = "rm -rf " + MAGIXExpXML
        os.system(command_string)


    ##====================================================================================================================================================
    ## print some informations to screen
    print " "
    print " "
    print "Please note, all files created by the current fit process are stored in the"
    print "myXCLASSMapFit working directory " + chr(34) + MAGIXjobDir + chr(34) + "!"
    print " "
    print " "


    ##====================================================================================================================================================
    ## define return variables
    return MAGIXjobDir
##--------------------------------------------------------------------------------------------------------------------------------------------------------

