#!/usr/bin/python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module contains all subroutines to perform the myXCLASSFit function, a simplified interface to MAGIX using myXCLASS.
##  Copyright (C) 2013 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function CheckMoleculesInDatabase:            determine molecules in database
##      - function CheckMolecules:                      check molecules used in molfit and iso ratio file
##      - function CheckMolfitFile:                     check given molfit file
##      - function AnalyzeExpXMLFile:                   analyze given experimental xml file
##      - function CreateExpXMLFile:                    creates the experimental xml file in the current job directory
##      - function CreateLMControlXMLFile:              creates the LM control xml file in the current job directory
##      - function CreateIOControlXMLFile:              creates the io-control xml file in the current job directory
##      - function ReadInOutputFiles:                   reads in ouput files
##      - function myXCLASSFit:                         Simplified CASA interface for MAGIX with myXCLASS program
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    25.07.2013   initial version
##  T. Moeller    09.04.2014   improved version
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import copy                                                                                 ## import copy package
import string                                                                               ## import string package
import shutil                                                                               ## import shutil package
import random                                                                               ## import random package
import datetime                                                                             ## import datetime package
import time                                                                                 ## import package date and time
import math                                                                                 ## import math package
import warnings                                                                             ## import warnings package
import task_MAGIX                                                                           ## import package MAGIX
import task_myXCLASS                                                                        ## import package myXCLASS
import sqlite3                                                                              ## import sqlite3 package
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## determine molecules in database
##
def CheckMoleculesInDatabase(ListOfMolecules, PrintFlag, FreqMinList, FreqMaxList, dbList):
    """

input parameters:
-----------------

    - ListOfMolecules:      list of molecules which are analyzed

    - PrintFlag:            flag indicating if messages are printed to screen

    - FreqMinList:          list of min. frequencies of all ranges (if list is empty, i.e. FreqMinList == [], table "partitionfunctions" is used
                            without taken frequency ranges into account, for FreqMinList != [], table "transitions" is used.)

    - FreqMaxList:          list of max. frequencies of all ranges

    - dbList:               path and name of user defined database file


output parameters:
------------------

    - ListOfCheckedMolecules:   list of identified molecules

    - dbFile:               path and name of used database file

    """

    # Debug:
    #    print "\n\nListOfMolecules = ", ListOfMolecules
    #    print "PrintFlag = ", PrintFlag
    #    print "FreqMinList = ", FreqMinList
    #    print "FreqMaxList = ", FreqMaxList
    #    print "dbList = ", dbList


    ## initialize return parameter
    ListOfCheckedMolecules = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check, if molecule names are included in database
    if (PrintFlag):
        print "\n\nCheck, if names of molecules in molfit file are included in myXCLASS database ..",
        sys.stdout.flush()


    ######################################################################################################################################################
    ## DO NOT EDIT OR REMOVE THE FOLLOWING LINE !!!!
    dbFilename = "/home/moeller/ALMA/CASA/myXCLASS-CASA-Interface/No-NR-version/Linux/XCLASS-Interface/Database/cdms_sqlite.db"
    ######################################################################################################################################################


    ## get path and name of database file
    if (dbList == [] or dbList == [""]):
        dbFile = dbFilename
    else:
        dbFile = dbList[0]

    # Debug:
    #    print "dbList = ", dbList
    #    print "dbFile = ", dbFile


    ## connect to database
    connOK = "true"
    try:
        conn = sqlite3.connect(dbFile)
    except sqlite3.Error, e:
        print "\n\n"
        print "Error in function task_myXCLASSFit.CheckMoleculesInDatabase:"
        print "\t Can not connect to sqlite3 databse %s." % dbFile
        print "\t Error: %d: %s" % (e.args[0], e.args[1])
        print "\n\t Can not check names of molecules."
        print "\n\n"
        connOK = "false"


    ## send query string to database
    if (connOK == "true" and len(ListOfMolecules) > 0):
        if (PrintFlag):
            print "done!"


        ## start loop over both tables
        # TableList = ["partitionfunctions", "transitions"]
        if (FreqMinList != []):
            TableList = ["transitions"]
        else:
            TableList = ["partitionfunctions"]
        MoleculeNames = sorted(set(ListOfMolecules))
        for table in TableList:
            if (table == "partitionfunctions"):
                ColumnNameForNameTransitions = "PF_Name"
            else:
                ColumnNameForNameTransitions = "T_Name"


            ## define query string
            query_string = "SELECT " + ColumnNameForNameTransitions + " FROM " + table + " where ("
            if (table == "transitions"):
                query_string += "("
            counter = 0
            for molecule in MoleculeNames:
                counter += 1
                if (counter > 1):
                    query_string += " or "
                query_string += ColumnNameForNameTransitions + " = " + chr(34) + molecule.strip() + chr(34)
            if (table == "transitions"):
                query_string += ") and ("
                for RangeIndex in xrange(len(FreqMinList)):
                    FreqMin = FreqMinList[RangeIndex][1]
                    FreqMax = FreqMaxList[RangeIndex][1]
                    if (RangeIndex > 0):
                        query_string += " or "
                    query_string += "(T_Frequency >= " + str(FreqMin) + " and T_Frequency <= " + str(FreqMax) + ")"
                query_string += ")"
            query_string += ")"

            # Debug:
            # print "query_string = ", query_string


            ## read informations from database
            cursor = conn.cursor()
            cursor.execute(query_string)
            rows = cursor.fetchall()
            dbContents = []
            for name in rows:
                FormatName = name[0].encode('ascii', 'ignore')
                FormatName = FormatName.strip()
                FormatName = '%s' % FormatName
                dbContents.append(FormatName)


            ## remove double entries
            dbContents = sorted(set(dbContents))

            # Debug:
            # print "MoleculeNames = ", MoleculeNames
            # print "dbContents = ", dbContents


        ## close connection to database
        ListOfCheckedMolecules = dbContents
        if (PrintFlag):
            print "\n\n"
        conn.close()

    # Debug:
    #    print "ListOfCheckedMolecules = ", ListOfCheckedMolecules, "\n\n"


    ## define return parameter
    return (ListOfCheckedMolecules, dbFile)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check molecules used in molfit and iso ratio file
##
def CheckMolecules(MolfitsFileName, IsoRatioFileName, FreqMinList, FreqMaxList, dbList):
    """

input parameters:
-----------------

    - MolfitsFileName:      path and name of molfit file

    - IsoRatioFileName:     path and name of iso ratio file

    - FreqMinList:          list of min. frequencies of all ranges

    - FreqMaxList:          list of max. frequencies of all ranges

    - dbList:               path and name of database file


output parameters:
------------------

    - FoundIsoRatioMolecules:   list of iso ratio molecules which were found in database
"""

    # Debug:
    #    print "\n\nMolfitsFileName = ", MolfitsFileName
    #    print "IsoRatioFileName = ", IsoRatioFileName
    #    print "FreqMinList = ", FreqMinList
    #    print "FreqMaxList = ", FreqMaxList
    #    print "dbList = ", dbList


    ## define return parameters
    FoundIsoRatioMolecules = []


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze given molfit file
    MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(MolfitsFileName)

    # Debug:
    #    print "MoleculesInMolfitFile = ", MoleculesInMolfitFile


    ## check, if molecules defined in the molfit file are included in the current database
    PrintFlag = False
    FoundMolfitMolecules, dbFile = CheckMoleculesInDatabase(MoleculesInMolfitFile, PrintFlag, FreqMinList, FreqMaxList, dbList)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze given iso-ratio file
    NewLocalIsoRatioFileName = ""
    IsoRatioTable, Isotopologues, IsoMolecule = task_myXCLASS.ImportIsoRatioFile(IsoRatioFileName, NewLocalIsoRatioFileName)

    # Debug:
    #    print "len(IsoRatioTable) = ", len(IsoRatioTable)
    #    print "IsoRatioTable = ", IsoRatioTable
    #    print "len(Isotopologues) = ", len(Isotopologues)
    #    print "Isotopologues = ", Isotopologues
    #    print "IsoMolecule = ", IsoMolecule


    ## select only those isotopologues whose iso master molecule is included in the molfit file
    SelectIsoRatioTable = []
    SelectedIsotopologues = []
    SelectedIsoMolecule = []
    GlobalIsoList = []
    for LineIndex in xrange(len(Isotopologues)):
        LocalIsoMolecule = IsoMolecule[LineIndex]
        LocalIsotopologues = Isotopologues[LineIndex]
        if (LocalIsoMolecule in FoundMolfitMolecules or LocalIsotopologues.find("GLOBAL__") > (-1)):


            ## check, if molecules defined in the molfit file are included in the current database
            if (LocalIsotopologues.find("GLOBAL__") == (-1)):
                FoundIsoRatioMolecules, dbFile = CheckMoleculesInDatabase([LocalIsotopologues], PrintFlag, FreqMinList, FreqMaxList, dbList)
                if (FoundIsoRatioMolecules != []):
                    SelectIsoRatioTable.append(IsoRatioTable[LineIndex])
                    SelectedIsotopologues.append(Isotopologues[LineIndex])
                    SelectedIsoMolecule.append(LocalIsoMolecule)
            else:
                SelectIsoRatioTable.append(IsoRatioTable[LineIndex])
                GlobalIsoList.append(LocalIsotopologues.replace("GLOBAL__", ""))

    # Debug:
    #    print "SelectIsoRatioTable = ", SelectIsoRatioTable
    #    print "SelectedIsotopologues = ", SelectedIsotopologues
    #    print "SelectedIsoMolecule = ", SelectedIsoMolecule


    ## check, if globally defined isotopologue ratios are really needed
    if (GlobalIsoList != []):
        UnusedGlobalIsoList = []
        for GlobalIso in GlobalIsoList:                                                     ## loop over all global iso definitions
            IncludedFlag = False
            for LocalIsotopologue in SelectedIsotopologues:                                 ## loop over all user defined isotopologues
                if (LocalIsotopologue.find(GlobalIso) > (-1)):                              ## is globally defined iso used
                    IncludedFlag = True
                    break
            if (not IncludedFlag):                                                          ## if globally defined isotopologue not used?
                UnusedGlobalIsoList.append(GlobalIso)


        ## remove unused globally defined isotopologues from iso ratio file
        if (UnusedGlobalIsoList != []):
            OldSelectIsoRatioTable = copy.deepcopy(SelectIsoRatioTable)
            SelectIsoRatioTable = []
            for IsoLine in OldSelectIsoRatioTable:
                PureIsoLine = IsoLine[0].strip()
                FoundFlag = False
                for UnusedGlobalIsoRatio in UnusedGlobalIsoList:
                    if (PureIsoLine == "GLOBAL__" + UnusedGlobalIsoRatio):
                        FoundFlag = True
                        break
                if (not FoundFlag):
                    SelectIsoRatioTable.append(IsoLine)

    # Debug:
    # print "SelectIsoRatioTable = ", SelectIsoRatioTable


    ## define return parameter
    return SelectIsoRatioTable
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check given molfit file
##
def CheckMolfitFile(MolfitsFileName, NameFunction, CurrentDir, MAGIXjobDir, FreqMinList, FreqMaxList, dbList):
    """

input parameters:
-----------------

    - MolfitsFileName:      path and name of molfit file

    - NameFunction:         name of the calling function

    - CurrentDir:           current directory

    - MAGIXjobDir:          MAGIX job directory

    - FreqMinList:          list of min. frequencies of all ranges

    - FreqMaxList:          list of max. frequencies of all ranges

    - dbList:               path and name of user defined database file


output parameters:
------------------

    - ok                    status variable

    - dbFile:               path and name of database file

    - MolfitsFile:          the absolute path and name of the molfit file within the MAGIX job directory

    - PureNameMolfitsFile:  the name of the molfit file without path

    """


    ## initialize return variables
    ok = 0
    dbFile = ""
    MolfitsFile = ""
    PureNameMolfitsFile = ""


    ##====================================================================================================================================================
    ## analyze molfit parameter: check path and name of the molfit xml-file and copy molfit file to temp directory
    print "Analyze molfit parameter ..",
    if (MolfitsFileName == None):
        print " "
        print " "
        print " "
        print "Error in XCLASS package, function " + NameFunction + ":"
        print "  No path and name of the molfit file is defined!"
        print " "
        print "  Please enter an exsisting path and name of a molfit file and redo " + NameFunction + " function call!"
        ok = 1
        return (ok, dbFile, MolfitsFile, PureNameMolfitsFile)
    MolfitsFileName = MolfitsFileName.strip()

    # Debug:
    # print " "
    # print 'MolfitsFileName = ', MolfitsFileName


    ## make path of molfit file absolute path
    if (MolfitsFileName[0] != "/"):
        MolfitsFileName = CurrentDir + MolfitsFileName


    ## does path and file exists?
    if not(os.path.exists(MolfitsFileName) or os.path.isfile(MolfitsFileName)):
        print " "
        print " "
        print " "
        print "Error in XCLASS package, function " + NameFunction + ":"
        print "  The given path and name of the molfit file " + MolfitsFileName + " does not exsist!"
        print " "
        print "  Please enter an exsisting path and name of a molfit file and redo " + NameFunction + " function call!"
        ok = 1
        return (ok, dbFile, MolfitsFile, PureNameMolfitsFile)
    print "done!"


    ## copy molfit file to temp directory
    command_string = "cp " + MolfitsFileName + " " + MAGIXjobDir
    os.system(command_string)


    ## get name of molfit file
    i = MolfitsFileName.rfind("/")
    if (i > 0):
        PureNameMolfitsFile = MolfitsFileName[i + 1:]
    else:
        PureNameMolfitsFile = MolfitsFileName

    # Debug:
    # print 'MolfitsFileName = ', MolfitsFileName

    MolfitsFile = MAGIXjobDir + PureNameMolfitsFile

    # Debug:
    # print 'MolfitsFile = ', MolfitsFile


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze molfit file using myXCLASS routine
    print "Analyze molfit file ..",
    MoleculeNames, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(MolfitsFile)
    print "done!"


    ## check molecules in database
    PrintFlag = True
    FoundMolfitMolecules, dbFile = CheckMoleculesInDatabase(MoleculeNames, PrintFlag, FreqMinList, FreqMaxList, dbList)


    ## print error message, if both lists are not identical
    SortedListOfMolecules = sorted(set(MoleculeNames))
    if (len(FoundMolfitMolecules) != len(SortedListOfMolecules)):
        print "\n\n"
        print "Error in function task_myXCLASSFit.CheckMolfitFile:"
        print "\n\t There are some molecules given in the molfit file, which are not included in the database"
        print "\t or which do no show a transition within the given frequency range(s)."
        print "\n\t Molecule names in molfit file:    ", SortedListOfMolecules
        print "\t Molecule names in database table: ", FoundMolfitMolecules
        print "\t Minimal Frequencies = ", FreqMinList
        print "\t Maximal Frequencies = ", FreqMaxList
        print "\n\n\t Please correct the molfit file and restart function!"
        print "\n\n"
        ok = 1


    ## define return variables
    return (ok, dbFile, MolfitsFile, PureNameMolfitsFile)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## creates the experimental xml file in the current job directory
##
def CreateExpXMLFile(JobDir, ASCIIDataFile, FreqMin, FreqMax, FreqStep, t_back_flag, tBack, tslope, nH_flag, N_H, beta_dust, kappa_1300, \
                     vLSR, TelescopeSize, Inter_Flag, iso_flag, IsoTableFileName, dbFilename):
    """

input parameters:
-----------------

    - JobDir:               path of job directory

    - ASCIIDataFile:        path and name of experimental ASCII file

    - FreqMin:              first frequency

    - FreqMax:              last frequency

    - FreqStep:             stepsize

    - t_back_flag:          defines, if background temperature and slope describe continuum completely

    - tBack:                background temperature

    - tslope:               temperature slope

    - nH_flag:              defines, if Hydrogen column density etc. is given by the xml file

    - N_H:                  Hydrogen column density

    - beta_dust:            beta for dust

    - kappa_1300:           kappa

    - vLSR:                 local standard of rest v_LSR

    - TelescopeSize:        size of the telescope

    - Inter_Flag:           defines if observational data is single dish ("F") or interferrometric data ("T")

    - iso_flag:             use isotopologues ("T"/"F"). If iso_flag is set to "F" the ratios are all set to 1 (default: "F")

    - IsoTableFileName:     name of iso ratio file

    - dbFilename:           path and name of the sqlite3 database file



output parameters:
------------------

    - None

    """

    EXPcontrol = open(JobDir + "exp.xml", 'w')
    EXPcontrol.write("<?xml version=" + chr(34) + "1.0" + chr(34) + " encoding=" + chr(34) + "UTF-8" + chr(34) + "?>\n")
    EXPcontrol.write("<ExpFiles>\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("    <!-- set number of exp. data files -->\n")
    EXPcontrol.write("    <NumberExpFiles>1</NumberExpFiles>\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("    <!-- define setting for import of exp. data files -->\n")
    EXPcontrol.write("    <file>\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("        <!-- define path and name of exp. data file and set import filter -->\n")
    EXPcontrol.write("        <FileNamesExpFiles>" + ASCIIDataFile + "</FileNamesExpFiles>\n")
    EXPcontrol.write("        <ImportFilter>xclassASCII</ImportFilter>\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("        <!-- define number of frequency ranges -->\n")
    EXPcontrol.write("        <NumberExpRanges>1</NumberExpRanges>\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("        <!-- define parameters for 1st frequency range -->\n")
    EXPcontrol.write("        <FrequencyRange>\n")
    EXPcontrol.write("            <MinExpRange>" + str(FreqMin) + "</MinExpRange>\n")
    EXPcontrol.write("            <MaxExpRange>" + str(FreqMax) + "</MaxExpRange>\n")
    EXPcontrol.write("            <StepFrequency>" + str(FreqStep) + "</StepFrequency>\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("            <!-- define background temperature and slope for 1st frequency range -->\n")
    if (t_back_flag == "y" or t_back_flag == "Y" or t_back_flag == "T" or t_back_flag == "t" or t_back_flag == True):
        EXPcontrol.write("            <t_back_flag>True</t_back_flag>\n")
    else:
        EXPcontrol.write("            <t_back_flag>False</t_back_flag>\n")
    EXPcontrol.write("            <BackgroundTemperature>" + str(tBack) + "</BackgroundTemperature>\n")
    EXPcontrol.write("            <TemperatureSlope>" + str(tslope) + "</TemperatureSlope>\n")
    if (nH_flag == "y" or nH_flag == "Y" or nH_flag == "T" or nH_flag == "t" or nH_flag == True):
        EXPcontrol.write("\n")
        EXPcontrol.write("\n")
        EXPcontrol.write("            <!-- define hydrogen column density etc. for 1st frequency range -->\n")
        EXPcontrol.write("            <HydrogenColumnDensity>" + str(N_H) + "</HydrogenColumnDensity>\n")
        EXPcontrol.write("            <DustBeta>" + str(beta_dust) + "</DustBeta>\n")
        EXPcontrol.write("            <Kappa>" + str(kappa_1300) + "</Kappa>\n")
    EXPcontrol.write("        </FrequencyRange>\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("        <!-- define local standard of rest (vLSR) -->\n")
    EXPcontrol.write("        <GlobalvLSR>" + str(vLSR) + "</GlobalvLSR>\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("        <!-- define size of telescope -->\n")
    EXPcontrol.write("        <TelescopeSize>" + str(TelescopeSize) + "</TelescopeSize>\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("        <!-- define if interferrometric observation is modeled -->\n")
    if (Inter_Flag == "y" or Inter_Flag == "Y" or Inter_Flag == "T" or Inter_Flag == "t" or Inter_Flag == True):
        EXPcontrol.write("        <Inter_Flag>True</Inter_Flag>\n")
    else:
        EXPcontrol.write("        <Inter_Flag>False</Inter_Flag>\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("        <!-- define parameters for ASCII file import -->\n")
    EXPcontrol.write("        <ErrorY>no</ErrorY>\n")
    EXPcontrol.write("        <NumberHeaderLines>0</NumberHeaderLines>\n")
    EXPcontrol.write("        <SeparatorColumns> </SeparatorColumns>\n")
    EXPcontrol.write("    </file>\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("    <!-- parameters for isotopologues -->\n")
    if (iso_flag == "y" or iso_flag == "Y" or iso_flag == "T" or iso_flag == "t" or iso_flag == True):
        EXPcontrol.write("    <iso_flag>True</iso_flag>\n")
        EXPcontrol.write("    <IsoTableFileName>" + IsoTableFileName + "</IsoTableFileName>\n")
    else:
        EXPcontrol.write("    <iso_flag>False</iso_flag>\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("\n")
    EXPcontrol.write("    <!-- define path and name of database file -->\n")
    EXPcontrol.write("    <dbFilename>" + dbFilename + "</dbFilename>\n")
    EXPcontrol.write("</ExpFiles>\n")
    EXPcontrol.write("\n")
    EXPcontrol.close()


    ## we've done
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## creates the LM control xml file in the current job directory
##
def CreateLMControlXMLFile(JobDir, NumberIteration):
    """

input parameters:
-----------------

    - JobDir:               path of job directory

    - NumberIteration:      max. number of iterations


output parameters:
------------------

    - None

    """

    LMcontrol = open(JobDir + "algorithm_control.xml", 'w')
    LMcontrol.write("<?xml version=" + chr(34) + "1.0" + chr(34) + " encoding=" + chr(34) + "UTF-8" + chr(34) + "?>\n")
    LMcontrol.write("<FitControl>\n")
    LMcontrol.write("    <!-- settings for fit process -->\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    LMcontrol.write("    <!-- set number of used algorithms -->\n")
    LMcontrol.write("    <NumberOfFitAlgorithms>1</NumberOfFitAlgorithms>\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    LMcontrol.write("    <algorithm>\n")
    LMcontrol.write("        <FitAlgorithm>Levenberg-Marquardt</FitAlgorithm>\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    LMcontrol.write("        <!-- define method used for Levenberg-Marquardt -->\n")
    LMcontrol.write("        <MethodLM>nr</MethodLM>\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    LMcontrol.write("        <!-- define value of the variation (used for the calculation of the gradient (of chi^2 function)) in percent/100 -->\n")
    LMcontrol.write("        <VariationValue>1.e-3</VariationValue>\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    LMcontrol.write("        <!-- set max. number of iterations -->\n")
    LMcontrol.write("        <number_iterations>" + str(NumberIteration) + "</number_iterations>\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    LMcontrol.write("        <!-- set max. number of processors -->\n")
    LMcontrol.write("        <NumberProcessors>8</NumberProcessors>\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    LMcontrol.write("        <!-- define path and name of host file -->\n")
    LMcontrol.write("        <MPIHostFileName> </MPIHostFileName>\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    LMcontrol.write("        <!-- settings for chi^2 -->\n")
    LMcontrol.write("        <limit_of_chi2>1e-7</limit_of_chi2>\n")
    LMcontrol.write("        <RenormalizedChi2>yes</RenormalizedChi2>\n")
    LMcontrol.write("        <DeterminationChi2>default</DeterminationChi2>\n")
    LMcontrol.write("        <SaveChi2>yes</SaveChi2>\n")
    LMcontrol.write("\n")
    LMcontrol.write("\n")
    LMcontrol.write("        <!-- set plot options -->\n")
    LMcontrol.write("        <PlotAxisX>Frequency [MHz]</PlotAxisX>\n")
    LMcontrol.write("        <PlotAxisY>T_mb [K]</PlotAxisY>\n")
    LMcontrol.write("        <PlotIteration>yes</PlotIteration>\n")
    LMcontrol.write("    </algorithm>\n")
    LMcontrol.write("</FitControl>\n")
    LMcontrol.close()


    ## we've done
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## creates the io-control xml file in the current job directory
##
def CreateIOControlXMLFile(JobDir, MAGIXExpXML, MolfitsFileName, MAGIXrootDir):
    """

input parameters:
-----------------

    - JobDir:               path of job directory

    - MAGIXExpXML:          path and name of an experimental xml file

    - MolfitsFileName:      path and name of a molfits file

    - MAGIXrootDir:         path of MAGIX root directory


output parameters:
------------------

    - None

    """

    iocontrol = open(JobDir + "io_control.xml", 'w')
    iocontrol.write("<?xml version=" + chr(34) + "1.0" + chr(34) + " encoding=" + chr(34) + "UTF-8" + chr(34) + "?>\n")
    iocontrol.write("<ioControl>\n")
    iocontrol.write("    <title>io control</title>\n")
    iocontrol.write("    <description>This xml-file contains the paths and the names of all files which are used by MAGIX</description>\n")
    iocontrol.write("    <PathFilename>\n")
    iocontrol.write("\n")
    iocontrol.write("        <experimental_data>\n")
    if (MAGIXExpXML != ""):
        iocontrol.write("            <filename>" + MAGIXExpXML + "</filename>\n")
    else:
        iocontrol.write("            <filename>" + JobDir + "exp.xml</filename>\n")
    iocontrol.write("            <description>path and name of the experimental file</description>\n")
    iocontrol.write("        </experimental_data>\n")
    iocontrol.write("\n")
    iocontrol.write("        <parameter_file>\n")
    iocontrol.write("            <filename>" + MolfitsFileName + "</filename>\n")
    iocontrol.write("            <description>path and name of the xml-file including the start values of each parameter</description>\n")
    iocontrol.write("        </parameter_file>\n")
    iocontrol.write("\n")
    iocontrol.write("        <fit_control>\n")
    iocontrol.write("            <filename>" + JobDir + "algorithm_control.xml</filename>\n")
    iocontrol.write("            <description>path and name of the xml file controlling the fitting process</description>\n")
    iocontrol.write("        </fit_control>\n")
    iocontrol.write("\n")
    iocontrol.write("        <fit_log>\n")
    iocontrol.write("            <filename>" + JobDir + "fit.log</filename>\n")
    iocontrol.write("            <description>path and name of log file describing the fitting process</description>\n")
    iocontrol.write("        </fit_log>\n")
    iocontrol.write("\n")
    iocontrol.write("        <model_description>\n")
    iocontrol.write("            <filename>" + MAGIXrootDir + "Fit-Functions/myXCLASS/xml/myNewXCLASS.xml</filename>\n")
    iocontrol.write("            <description>path and name of the xml-description of the input/output file of the fit function module</description>\n")
    iocontrol.write("        </model_description>\n")
    iocontrol.write("\n")
    iocontrol.write("    </PathFilename>\n")
    iocontrol.write("</ioControl>\n")
    iocontrol.write("\n")
    iocontrol.close()


    ## we've done
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## reads in ouput files
##
def ReadInOutputFiles(JobDir, NameOfFunction, ConvertLogLinFlag, RestFreq, vLSR, printInfoFlag):
    """

input parameters:
-----------------

    - JobDir:               path of job directory

    - NameOfFunction:       name of current CASA function

    - ConvertLogLinFlag:    defines if column densities in the fitted molfit file(s) are converted back to linear values

    - RestFreq:             rest frequency in MHz

    - vLSR:                 velocity (local standard of rest) in km/s

    - printInfoFlag:        (T/F) print information to screen


output parameters:
------------------

    - input_file:           list of all molfit files

    - ListOfMolfitFileNames: list of molfit file names

    - model_values:         list of all function values for all molfit files

    - ListOfModelFileNames: list of model file names

    """

    input_file = []                                                                         ## reset list of input files
    ListOfMolfitFileNames = []                                                              ## reset list of molfit file names
    ListOfModelFileNames = []                                                               ## reset list of model file names
    model_values = []                                                                       ## reset list of model values


    ## determine number of new output files
    listing = os.listdir(JobDir)
    listing.sort()
    NumOutFiles = 0                                                                         ## determine number of output files
    for files in listing:
        if (files.endswith(".out.molfit")):
            NumOutFiles += 1
    if (NumOutFiles == 0):                                                                  ## check if new input files exsist
        print " "
        print " "
        print "Error in XCLASS package, function " + NameOfFunction + ":"
        print "  No path and name of the molfit file is defined!"
        print " "
        print "  Please enter an exsisting path and name of the molfit file and redo MAGIX call!"
        return input_file, ListOfMolfitFileNames, model_values, ListOfModelFileNames


    ## change column densities back to linear scale and reformat iso ratio files
    count_files = 0
    ErrorInstanceXMLFileAll = []
    ErrorMolfitFileAll = []
    for files in listing:
        if (files.endswith(".out.molfit")):
            NewInputFile = JobDir + files
            ListOfMolfitFileNames.append(files)
            count_files += 1

            # Debug:
            # print "NewInputFile = ", NewInputFile


            ## convert column and hydrogen column density (if given for each component) back to linear scale
            if (ConvertLogLinFlag == "true"):
                LogLinearFlag = "linear"
                task_myXCLASS.ConvertNtotToLogScale(NewInputFile, LogLinearFlag)


            ## read in new output files
            f = open(NewInputFile)
            NewInputFileContent = f.readlines()
            f.close()
            input_file.append(NewInputFileContent)


            ## save molfit files corresponding to error estimation algorithm
            if (files.find("ErrorEstim_INS") > (-1)):
                ErrorMolfitFileAll.append(files)


        ## reformat iso ratio files
        elif (files.startswith("myXClass_isoratios") and files.endswith(".out.input")):
            NewLocalIsoRatioFileName = JobDir + files
            IsoRatioTable, Isotopologues, IsoMolecule = task_myXCLASS.ImportIsoRatioFile(JobDir + files, NewLocalIsoRatioFileName)



        ## get all fitted instance xml-file for error estimation
        if (files.startswith("parameters.ErrorEstim_INS") and files.endswith(".out.xml")):
            ErrorInstanceXMLFileAll.append(files)


    ## create error molfit file
    if (ErrorInstanceXMLFileAll != []):

        # Debug:
        # print "\n\n\n\n\n\n\n\nErrorMolfitFileAll = ", ErrorMolfitFileAll

        for InstanceXMLFile in ErrorInstanceXMLFileAll:


            ## determine file name addon for current xml-file
            AddOn = ""
            i = InstanceXMLFile.find(".")
            if (i > (-1)):
                AddOn = InstanceXMLFile[i + 1:]
                i = AddOn.rfind(".out.xml")
                if (i > (-1)):
                    AddOn = AddOn[:i].strip()

            # Debug:
            # print "InstanceXMLFile = ", InstanceXMLFile
            # print "AddOn = ", AddOn


            ## find molfit file with this add on
            if (AddOn.strip() != ""):
                for mol in ErrorMolfitFileAll:
                    molMod = mol.replace("__error-values","")
                    i = molMod.find(AddOn)
                    if (i > (-1)):

                        # Debug:
                        # print "mol = ", mol


                        ## create error molfit file
                        task_myXCLASS.CreateErrorMolfitFile(JobDir + InstanceXMLFile, JobDir + mol, printInfoFlag)
                        break


    ##------------------------------------------------------------------------------------------------------------------------------------------------
    ## read in output file(s)
    if (printInfoFlag == "true"):
        print "\n\t Read in model function values from file(s) .."
    # listing = os.listdir(JobDir)


     ## determine number of output files
    NumOutFiles = 0
    for files in listing:
        if (files.endswith(".out.dat")):
            NumOutFiles += 1


    ## read in model function values
    count_files = 0
    for files in listing:
        if (files.endswith(".out.dat")):
            count_files += 1
            ListOfModelFileNames.append(files)


            ## read in model function values
            output_file = numpy.loadtxt(JobDir + files, skiprows = 0)


            ## add velocity axis if RestFreq != 0.0
            if (RestFreq != 0.0):
                if (printInfoFlag == "true"):
                    print "\tAdd velocity axis ..",
                FreqData = output_file[:, 0]
                VelocityData = task_myXCLASS.ChangeToVelocitiy(FreqData, RestFreq, vLSR)


                ## add velocity axis to output_file array
                NewmyXCLASSspectrum = numpy.zeros((len(output_file), 3), dtype=numpy.float32)
                NewmyXCLASSspectrum[:, 0] = output_file[:, 0]
                NewmyXCLASSspectrum[:, 1] = VelocityData[:]
                NewmyXCLASSspectrum[:, 2] = output_file[:, 1]
                output_file = NewmyXCLASSspectrum


            ## append current array with model function values to array model_values
            model_values.append(output_file)


            ## print some informations to the screen
            if (printInfoFlag == "true"):
                if (NumOutFiles > 1):
                    if (count_files == 1):
                        print "\t Assuming, that the model function values for the best fit are stored to the variable " \
                               + chr(34) + "model_values" + chr(34) + ", "
                    if (count_files == NumOutFiles):
                        helpstring = "."
                    else:
                        helpstring = ","
                    print "\t\t the output array " + chr(34) + "model_values[" + str(count_files - 1) + "]" + chr(34) \
                            + " contains the model function values for the experimental data file " + chr(34) + files + chr(34) + helpstring
                else:
                    print "\t Assuming, that the model function values for the best fit are stored to the variable " + chr(34) + "model_values" \
                            + chr(34) + ", "
                    print "\t\t the output array " + chr(34) + "model_values" + chr(34) \
                            + " contains the model function values for the experimental data file " + files


    ## reduce dimension of output array if there is only one output file
    if (count_files == 1):
        model_values = model_values[0]


    ## we've done
    return input_file, ListOfMolfitFileNames, model_values, ListOfModelFileNames
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## Simplified CASA interface for MAGIX with myXCLASS program
##
def myXCLASSFit(NumberIteration, AlgorithmXMLFile, MolfitsFileName, experimentalData, TelescopeSize, Inter_Flag, t_back_flag, tBack, tslope, \
                nH_flag, N_H, beta_dust, kappa_1300, iso_flag, IsoTableFileName, RestFreq, vLSR):
    """

Simplified CASA interface for MAGIX with myXCLASS program:
----------------------------------------------------------


This function provides a simplified interface for MAGIX using the myXCLASS program. The
function starts MAGIX using the Levenberg-Marquardt algorithm to fit experimental data
with the myXCLASS program. The user has to specify the max. number of iterations, the
experimental data (or path and name of the experimental xml-file), and the path and
name of the molfit file.

For each run the myXCLASSFit function creates a so-called job directory located in
the run directory ("path-of-myXCLASS-CASA-Interface/run/myXCLASSFit/") where all
files created by the myXCLASSFit function are stored in. The name of this job
directory is made up of four components: The first part of the name consists of the
phrase "job_" whereas the second part describes the date (day, month, year), the third
part the time stamp (hours, minutes, seconds) of the function execution. The last
part describes a so-called job ID which is composed of the so-called PID followed
by a four digit random integer number to create a really unambiguous job number, e.g.
"path-to-myXCLASS-Interface/run/myXCLASSFit/job__25-07-2013__12-02-03__189644932/"

Before the fit procedure starts, the function copies the molfits, the experimental data,
and xml-files to the myXCLASSFit job directory. The path(s) defined in the exerimental
xml-file are adjusted so that these paths point to the current myXCLASSFit job directory.

Please note, that the original experimental xml- and data files are not modified. Only the
copy of the experimental xml file located in the myXCLASSFit job directory is modified!

If no experimental xml-file is defined, the myXCLASSFit function creates an experimental
xml-file containing the settings for the different parameters in the myXCLASSFit job
directory. Additionally, the function creates an ASCII file located in the myXCLASSFit
job directory contining the experimental data given by the parameter "experimentalData".
If the parameter "experimentalData" defines the path and the name of an experimental
data file, then the data file is copied to the myXCLASSFit job directory as well.
The path and the name of this ASCII file is pasted in the created experimental xml file.
Furthermore, the function creates automatically the algorithm and i/o control xml files,
need by MAGIX, so that the user does not need to edit any xml-file.


input parameters:
-----------------

    - NumberIteration:      max. number of iterations (default: 50).

    - AlgorithmXMLFile:     only necessary, if the user wants to use another fit
                            algorithm (than Levenberg-Marquardt) for fitting. Therefore, 
                            the path and name of a MAGXI xml-file defining settings
                            for an algorithm or algorithm chain has to be given. (A
                            relative path has to be defined relative to the current
                            working directory!)

                            NOTE, if the user specify a xml file, the number of
                            iterations given by the parameter "NumberIteration" is
                            ignored. The number of iteration is then given by the xml
                            file. In order to use the implemented fit algorithm
                            (Levenberg-Marquardt) clear the AlgorithmXMLFile parameter,
                            i.e. AlgorithmXMLFile = "", and define the max. number of
                            iterations by using parameter "NumberIteration".

    - MolfitsFileName:      path and name of the extended molfit file, including the
                            source size, the rotation temperature the column density,
                            the velocity width, the velocity offset and the flag
                            indicating if the current component is considered for
                            core "c" or foreground "f".

                            In contrast to the format of the molfit file the extended
                            molfit file required by the myXCLASSFit function contains
                            one (three) additional column(s) for each parameter of
                            each component.

                            For a detailed description of the extended molfit file
                            required by the myXCLASSFit function, see the manual.

                            NOTE, a relative path has to be defined relative to the
                            current working directory!

    - experimentalData:     This parameter offers two different possibility to send
                            the experimental data to the myXCLASSFit function:

                            - the parameter experimentalData defines the path and
                              name of and experimental xml-file suitable for MAGIX.

                            - the parameter experimentalData defines the path and
                              name of and ASCII file called experimental data file,
                              where the first column describe the frequency (in MHz)
                              and the second column the beam temperature (intensity).

                            NOTE, if the parameter experimentalData defines a relative
                            path, the path has to be defined relative to the current
                            working directory!

    The following parameters are needed, if the parameter experimentalData does NOT
    describe the path and name of a MAGIX xml-file:

    - vLSR:                 local standard of rest velocity (in km/s), (default 0).

    - TelescopeSize:        for single dish observations (Inter_Flag = F): TelescopeSize
                            describes the size of telescope (in m), (default: 1); for
                            interferometric observations (Inter_Flag = T): TelescopeSize
                            describes the interferometric beam FWHM size (in arcsec),
                            (default: 1).

    - Inter_Flag:           defines, if single dish ("F") or interferometric
                            observations ("T") are described, (default: "F").

    - t_back_flag:          ("T"/"F"), defines, if the user defined background 
                            temperature Tbg and temperature slope Tslope given
                            by the input parameters tBack and tslope describe the
                            continuum contribution completely (t_back_flag = "T")
                            or not (t_back_flag = "F") (default: "T").

    - tBack:                background temperature (in K), (default: 0).

    - tslope:               temperature slope (dimensionless), (default: 0).

    - nH_flag:              ("T"/"F"), defines, if column density, beta for dust
                            and kappa are given by the molfit file ("F") or, if
                            "nH_flag" is set to "T", the following three parameters
                            define the hydrogen column density, beta for dust and
                            kappa for all components (default: "F"):

    - N_H:                  (has to be given only if "nH_flag" is set to "T") 
                            Hydrogen column density (in cm^{-2}), (default: 0).

    - beta_dust:            (has to be given only if "nH_flag" is set to "T") beta
                            for dust (dimensionless), (default: 0).

    - kappa_1300:           (has to be given only if "nH_flag" is set to "T") kappa
                            (cm^2 g^{-1}), (default: 0).

    - iso_flag:             use isotopologues ("T"/"F"). If iso_flag is set to
                            "F" the ratios are all set to 1 (default: "F").

    - IsoTableFileName:     path and name of an ASCII file including the iso ratios
                            between certain molecules. The so-called "iso ratio file"
                            defines the iso ratios between molecules. The ASCII file
                            consists of three columns, where the first two columns
                            indicates the molecules, respectively. The third column
                            defines the ratio for both molecules. The columns are
                            separated by blanks or tabs. So, the names of the
                            molecules must not contain blanks.

                            The myXCLASSFit function offers the possibility to
                            optimize the ratios between isotopologues as well. For
                            that purpose, the user has to add two additional columns
                            on the right indicating the lower and the upper limit
                            of a certain ratio, respectively. For more informations
                            please look at the description of the iso ratio file given
                            for the myXCLASS function.

                            If the lower and upper limit are equal or if the lower
                            limit is higher than the upper limit, the ratio is kept
                            constant and is not optimized by the myXCLASSFit function.

                            NOTE, if the parameter "IsoTableFileName" defines a relative
                            path, the path has to be defined relative to the current
                            working directory!

    The following parameter is needed, to add a velocity axis to the output
    array model_values, see below:

    - RestFreq:             rest frequency in MHz (default: 0). (If this parameter is
                            set to zero, the intensity is plotted against frequency (in MHz)
                            otherwise against velocity (in km/s).


Output parameters:
------------------

    - input_file :          the contents of the molfit file containing the
                            parameters for the best fit.

    - model_values:         the model function values for each data point, which
                            correspond to the best fit.

    - JobDir:               absolute path of the job directory created for the current run.


Note, the user is free to define different names for the output parameters.



Example using a molfit file in the old format and an ASCII file containing the experimental data:
-------------------------------------------------------------------------------------------------

NumberIteration = 10
MolfitsFileName = "demo/myXCLASSFit/CH3OH__old.molfit"
experimentalData = "demo/myXCLASSFit/band1b.dat"
vLSR = 0.0
TelescopeSize = 3.5
Inter_Flag = F
t_back_flag = T
tBack = 1.1
tslope = 0.0000000000E+00
nH_flag = T
N_H = 3.0000000000E+24
beta_dust = 2.0
kappa_1300 = 0.02
iso_flag = F
IsoTableFileName = ""
RestFreq = 0.0
newmolfit, modeldata, JobDir = myXCLASSFit()


Example NGC6334 with MAGIX xml files (old format):
--------------------------------------------------

NumberIteration = 10
MolfitsFileName = "demo/myXCLASSFit/CH3OH__old.molfit"
experimentalData = "demo/myXCLASSFit/observation.xml"
RestFreq = 0.0
vLSR = 0.0
newmolfit, modeldata, JobDir = myXCLASSFit()


Example NGC6334 with MAGIX xml files (new format):
--------------------------------------------------

NumberIteration = 10
MolfitsFileName = "demo/myXCLASSFit/CH3OH__new.molfit"
experimentalData = "demo/myXCLASSFit/observation.xml"
RestFreq = 0.0
vLSR = 0.0
newmolfit, modeldata, JobDir = myXCLASSFit()


Example NGC6334 with MAGIX xml files (other algorithm):
-------------------------------------------------------

MolfitsFileName = "demo/myXCLASSFit/CH3OH__new.molfit"
experimentalData = "demo/myXCLASSFit/observation.xml"
AlgorithmXMLFile = "demo/myXCLASSFit/algorithm-settings.xml"
RestFreq = 0.0
vLSR = 0.0
newmolfit, modeldata, JobDir = myXCLASSFit()

    """


    ##====================================================================================================================================================
    ## initialize working variables
    input_file = []
    model_values = []
    MAGIXjobDir = ""
    CurrentDir = os.getcwd() + "/"
    NameOfFunction = "myXCLASSFit"


    ######################################################################################################################################################
    ## DO NOT EDIT OR REMOVE THE FOLLOWING LINE !!!!
    MAGIXrootDir = "/home/moeller/ALMA/CASA/myXCLASS-CASA-Interface/No-NR-version/Linux/XCLASS-Interface/programs/MAGIX/"
    ######################################################################################################################################################


    ##====================================================================================================================================================
    ## create run directory for the current function call
    LocalPrintFlag = True
    MAGIXjobDir = task_MAGIX.CreateRunDirectory("myXCLASSFit", MAGIXrootDir, LocalPrintFlag)

    # Debug:
    # print "command_string = ", command_string


    ##====================================================================================================================================================
    ## analyze experimentalData parameter: check if input parameter defines path and name of a file
    print "Analyze experimentalData parameter .."
    ParamIsXMLFile = "false"
    MAGIXExpXML = ""

    # Debug:
    # print 'type(experimentalData).__name__ = ', type(experimentalData).__name__


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## is experimentalData a string
    if (type(experimentalData).__name__ == 'str'):                                          ## is experimentalData a string ?
        filename = experimentalData.strip()                                                 ## remove leading and tailing blanks

        # Debug:
        # print 'filename = ', filename


        ## if neccessary, make relative path absolute
        if (filename[0] != "/"):
            filename = CurrentDir + filename


        ## check if path and name of the experimental file exsits
        if not(os.path.exists(filename) or os.path.isfile(filename)):
            print " "
            print " "
            print "Error in XCLASS package, function task_myXCLASSFit.myXCLASSFit:"
            print "  The given path and name of the experimental xml- or ASCII file does not exsist!"
            print " "
            print "  Please enter an exsisting path and name of an experimental xml- or ASCII file and redo myXCLASSFit function call!"
            return (input_file, model_values, MAGIXjobDir)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check, if file is a xml-file, i.e. the filename ends with '.xml'
        if (filename.endswith(".xml")):
            MAGIXExpXML = filename
            ParamIsXMLFile = "true"


            ## copy xml file to temp directory
            command_string = "cp " + MAGIXExpXML + " " + MAGIXjobDir
            os.system(command_string)


            ## get name of experimental xml-file
            i = MAGIXExpXML.rfind("/")
            if (i > 0):
                MAGIXExpXML = MAGIXExpXML[i + 1:]
            MAGIXExpXML = MAGIXjobDir + MAGIXExpXML

            # Debug:
            # print "MAGIXjobDir = ", MAGIXjobDir
            # print "MAGIXExpXML = ", MAGIXExpXML


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get all min. and max. frequencies for each range and path and name of user defined database file
            FreqMinList = task_MAGIX.GetXMLtag(MAGIXExpXML, "MinExpRange")
            FreqMaxList = task_MAGIX.GetXMLtag(MAGIXExpXML, "MaxExpRange")
            dbList = task_MAGIX.GetXMLtag(MAGIXExpXML, "dbFilename")
            if (len(dbList) > 0):
                PathFileNameDB = dbList[0].strip()
                if (PathFileNameDB.strip() != ""):
                    if (PathFileNameDB[0] != "/"):                                          ## make relative path absolute
                        PathFileNameDB = CurrentDir + PathFileNameDB
                    if not(os.path.exists(PathFileNameDB) or os.path.isfile(PathFileNameDB)):
                        print " "
                        print " "
                        print "Error in XCLASS package, function task_myXCLASSFit.myXCLASSFit:"
                        print "  The given path and name of the database file " + PathFileNameDB
                        print "  does not exsist!"
                        print " "
                        print "  Please note, if a relative path is specified, the path has to be defined relative to the"
                        print "  current working directory!"
                        print " "
                        print "  Please enter an exsisting path and name of a database file and restart function call!"
                        return MAGIXjobDir

                    # Debug:
                    # print "PathFileNameDB = ", PathFileNameDB


                    ## modify experimental xml-file in MAGIX temp directory
                    dbList = [PathFileNameDB]
                else:
                    dbList = []
            else:
                dbList = []


            ## analyze molfit parameter: check path and name of the molfit xml-file and copy molfit file to temp directory
            ok, dbFilename, MolfitsFile, PureNameMolfitsFile = CheckMolfitFile(MolfitsFileName, NameOfFunction, CurrentDir, MAGIXjobDir, \
                                                                               FreqMinList, FreqMaxList, dbList)
            if (ok == 1):                                                                   ## an error occurred
                return (input_file, model_values, MAGIXjobDir)

            # Debug:
            # print "ok = ", ok
            # print "MolfitsFile = ", MolfitsFile
            # print "PureNameMolfitsFile = ", PureNameMolfitsFile


            ## update path of database file
            if (dbList == [] or dbList == [""]):
                dbList = [dbFilename]
            task_MAGIX.WriteXMLtag(MAGIXExpXML, "dbFilename", dbList)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get flag for iso file
            IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "Isotopologues")
            if (len(IsoFlag) == 0):
                IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "iso_flag")
                if (len(IsoFlag) == 0):
                    IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "ISO_FLAG")
                    if (len(IsoFlag) == 0):
                        IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "Iso_Flag")
                        if (len(IsoFlag) == 0):
                            IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "Iso_flag")
            if (len(IsoFlag) > 0):
                IsoFlag = IsoFlag[0].lower()
                if (IsoFlag == "y" or IsoFlag == "t" or IsoFlag == "true"):
                    IsoFlag = "y"
                else:
                    IsoFlag = "n"
            else:
                IsoFlag = "n"

            # Debug:
            # print 'IsoFlag = ', IsoFlag


            ## get name of iso file
            if (IsoFlag == "y"):
                IsoFile = task_MAGIX.GetXMLtag(MAGIXExpXML, "IsoTableFileName")

                # Debug:
                # print 'IsoFile = ', IsoFile

                if (len(IsoFile) > 0):
                    IsoFile = IsoFile[0]
                    if (IsoFile.strip() != ""):
                        if (IsoFile[0] != "/"):                                             ## make relative path absolute
                            IsoFile = CurrentDir + IsoFile
                        if not(os.path.exists(IsoFile) or os.path.isfile(IsoFile)):
                            print " "
                            print " "
                            print "Error in XCLASS package, function myXCLASSFit:"
                            print "  The given path and name of the iso file " + IsoFile
                            print "  does not exsist!"
                            print " "
                            print "  Please note, if a relative path is specified, the path has to be defined relative to the"
                            print "  current working directory!"
                            print " "
                            print "  Please enter an exsisting path and name of iso file and redo myXCLASSFit function call!"
                            return (input_file, model_values, MAGIXjobDir)
                        command_string = "cp " + IsoFile + " " + MAGIXjobDir               ## copy experimetnal dat files to temp directory
                        os.system(command_string)
                        i = IsoFile.rfind("/")
                        if (i < 0):
                            i = 0
                        IsoFile = MAGIXjobDir + IsoFile[i + 1:]                            ## define new iso file name
                        IsoFileList = [IsoFile]

                        # Debug:
                        # print 'IsoFileList = ', IsoFileList


                        ## modify experimental xml-file in MAGIX temp directory
                        task_MAGIX.WriteXMLtag(MAGIXExpXML, "IsoTableFileName", IsoFileList)

                        # Debug:
                        #    print "MolfitsFile = ", MolfitsFile
                        #    print "IsoFile = ", IsoFile


                        ## check molecules
                        SelectIsoRatioTable = CheckMolecules(MolfitsFile, IsoFile, FreqMinList, FreqMaxList, dbList)


                        ## write new iso ratio file
                        NewIsoRatioFile = open(IsoFile, 'w')
                        NewLine = "%-45s %-45s %25s %25s %25s" % ("% isotopologue:", "Iso-master:", "Iso-ratio:", "Lower-limit:", "upper-limit:")
                        NewIsoRatioFile.write(NewLine + "\n")
                        for IsoLine in SelectIsoRatioTable:
                            if (len(IsoLine) > 3):
                                NewLine = "%-45s %-45s %25s %25s %25s" % (IsoLine[0], IsoLine[1], IsoLine[2], IsoLine[3], IsoLine[4])
                            else:
                                NewLine = "%-45s %-45s %25s" % (IsoLine[0], IsoLine[1], IsoLine[2])
                            NewIsoRatioFile.write(NewLine + "\n")
                        NewIsoRatioFile.close()


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## redefine paths of observational data files
            ExpFiles = []
            ExpFilesOrig = task_MAGIX.GetXMLtag(MAGIXExpXML, "FileNamesExpFiles")
            for filename in ExpFilesOrig:
                if (filename[0] != "/"):                                                    ## make relative path absolute
                    filename = CurrentDir + filename
                if not(os.path.exists(filename) or os.path.isfile(filename)):
                    print " "
                    print " "
                    print "Error in XCLASS package, function myXCLASSFit:"
                    print "  The given path and name of the experimental data file " + filename
                    print "  defined in the experimental data xml-file does not exsist!"
                    print " "
                    print "  Please note, if a relative path is specified, the path has to be defined relative to the"
                    print "  current working directory!"
                    print " "
                    print "  Please enter an exsisting path and name and redo myXCLASSFit function call!"
                    return (input_file, model_values, MAGIXjobDir)
                ExpFiles.append(filename)                                                   ## save name(s) in list

            # Debug:
            # print 'ExpFiles = ', ExpFiles


            ## copy observational data files defined in the xml-file to the current MAGIX working directory
            NewExpFiles = []                                                                ## list with modified data file names
            for files in ExpFiles:
                files = files.strip()
                if (files[0] != "/"):
                    files = CurrentDir + files

                command_string = "cp " + files + " " + MAGIXjobDir                         ## copy experimetnal dat files to temp directory
                os.system(command_string)
                i = files.rfind("/")
                if (i < 0):
                    i = 0
                NewExpFiles.append(MAGIXjobDir + files[i + 1:])                            ## define new data file names


            ## modify experimental xml-file in MAGIX temp directory
            task_MAGIX.WriteXMLtag(MAGIXExpXML, "FileNamesExpFiles", NewExpFiles)

            # Debug:
            # print "NewExpFiles = ", NewExpFiles
        else:                                                                               ## file is not an xml-file
            print "Import ASCII data from ", filename, "..",
            ASCIIDataFile = filename
            experimentalData = numpy.loadtxt(filename, skiprows = 0)
            print "done!"


    ## if parameter is not the file name of a xml-file, analyze parameter
    if (ParamIsXMLFile == "false"):


        ## is experimentalData an array or a list?
        if (type(experimentalData).__name__ == 'ndarray'):                                  ## is experimentalData an array ?
            if (len(experimentalData[0]) != 2):                                             ## check, if array is a 2D array
                print " "
                print " "
                print "Error in XCLASS package, function myXCLASSFit:"
                print "  The given parameter for the experimental data is an array, but it is not a 2D array containing frequencies and intensities!"
                print " "
                print "  Please correct your input and redo myXCLASSFit function call!"
                return (input_file, model_values, MAGIXjobDir)
            else:                                                                           ## write exp. data to temp. file
                print "\rWrite ASCII data to temp file ..",
                ASCIIDataFile = MAGIXjobDir + "ASCIIdata.dat"
                numpy.savetxt(ASCIIDataFile, experimentalData, delimiter='\t')
                print "done!"


            ## determine min. and max. frequency and stepsize of frequency
            FreqMin = numpy.min(experimentalData[:,0])
            FreqMax = numpy.max(experimentalData[:,0])
            number_points = len(experimentalData[:,0])
            if (number_points == 1):                                                        ## data has just one data point
                print " "
                print " "
                print "Error in XCLASS package, function myXCLASSFit:"
                print "  The given experimental data is an array, but it contains only one data point!"
                print " "
                print "  Please correct your input and redo myXCLASSFit function call!"
                return (input_file, model_values, MAGIXjobDir)


            ## define equidistant stepsize
            FreqStep = (FreqMax - FreqMin)/(number_points - 1)


        ## is experimentalData a list?
        elif (type(experimentalData).__name__ == 'list'):                                   ## is experimentalData a list ?
            if (len(experimentalData[0]) != 2):                                             ## check, if array is a 2D array
                print " "
                print " "
                print "Error in XCLASS package, function myXCLASSFit:"
                print "  The given parameter for the experimental data is a list, but it is not a 2D list containing frequencies and intensities!"
                print " "
                print "  Please correct your input and redo myXCLASSFit function call!"
                return (input_file, model_values, MAGIXjobDir)
            else:                                                                           ## write exp. data to temp. file
                print "Write ASCII data to temp file ..",
                ASCIIDataFile = MAGIXjobDir + "ASCIIdata.dat"
                datafile = open(ASCIIDataFile, 'w')
                datafile.write("\n".join(experimentalData))
                datafile.closer()
                print " done!"


            ## determine min. and max. frequency and stepsize of frequency
            frequencies = []
            for line in experimentalData:
                frequencies.append(line[0])
            FreqMin = min(frequencies)
            FreqMax = max(frequencies)
            number_points = len(frequencies)
            if (number_points == 1):                                                        ## data has just one data point
                print " "
                print " "
                print "Error in XCLASS package, function myXCLASSFit:"
                print "  The given experimental data is an array, but it contains only one data point!"
                print " "
                print "  Please correct your input and redo myXCLASSFit function call!"
                return (input_file, model_values, MAGIXjobDir)


            ## define equidistant stepsize
            FreqStep = (FreqMax - FreqMin)/(number_points - 1)


        ## the parameter is neither an array nor a list
        else:
            print " "
            print " "
            print "Error in XCLASS package, function myXCLASSFit:"
            print "  The given parameter for the experimental data is neither a string describing the path and name of"
            print "  an exsisting xml- or ASCII file, nor a 2D array, nor a 2D list."
            print " "
            print "  Please correct your input and redo myXCLASSFit function call!"
            return (input_file, model_values, MAGIXjobDir)


        ## define FreqMinList and FreqMaxList, clear dbList to use default database file
        FreqMinList = [[1, FreqMin]]
        FreqMaxList = [[1, FreqMax]]
        dbList = []


        ## analyze molfit parameter: check path and name of the molfit xml-file and copy molfit file to temp directory
        ok, dbFilename, MolfitsFile, PureNameMolfitsFile = CheckMolfitFile(MolfitsFileName, NameOfFunction, CurrentDir, MAGIXjobDir, \
                                                                           FreqMinList, FreqMaxList, dbList)
        if (ok == 1):                                                                   ## an error occurred
            return (input_file, model_values, MAGIXjobDir)

        # Debug:
        # print "ok = ", ok
        # print "MolfitsFile = ", MolfitsFile
        # print "PureNameMolfitsFile = ", PureNameMolfitsFile


        ## update path of database file
        if (dbList == [] or dbList == [""]):
            dbList = [dbFilename]


        ## check path and name of iso table file
        iso_flag = "false"
        IsoTableFileName = IsoTableFileName.strip()
        if (IsoTableFileName != ""):
            iso_flag = "true"
            if (IsoTableFileName[0] != "/"):
                IsoTableFileName = CurrentDir + IsoTableFileName
            if not(os.path.exists(IsoTableFileName) or os.path.isfile(IsoTableFileName)):
                print " "
                print " "
                print "Error in XCLASS package, function myXCLASSFit:"
                print "  The given path and name of the iso table file " + IsoTableFileName + " does not exsist!"
                print " "
                print "  Please enter an exsisting path and name of the iso table file and redo myXCLASSFit function call!"
                return (input_file, model_values, MAGIXjobDir)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## check molecules
            SelectIsoRatioTable = CheckMolecules(MolfitsFile, IsoTableFileName, FreqMinList, FreqMaxList, dbList)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## write new iso ratio file
            IsoTableFileName = MAGIXjobDir + "New-Iso-Ratio-File.dat"
            NewIsoRatioFile = open(NewIsoFile, 'w')
            NewLine = "%-45s %-45s %25s %25s %25s" % ("% isotopologue:", "Iso-master:", "Iso-ratio:", "Lower-limit:", "upper-limit:")
            NewIsoRatioFile.write(NewLine + "\n")
            for IsoLine in SelectIsoRatioTable:
                if (len(IsoLine) > 3):
                    NewLine = "%-45s %-45s %25s %25s %25s" % (IsoLine[0], IsoLine[1], IsoLine[2], IsoLine[3], IsoLine[4])
                else:
                    NewLine = "%-45s %-45s %25s" % (IsoLine[0], IsoLine[1], IsoLine[2])
                NewIsoRatioFile.write(NewLine + "\n")
            NewIsoRatioFile.close()


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## construct xml-file
        print "Creating experimental xml file ..",
        CreateExpXMLFile(MAGIXjobDir, ASCIIDataFile, FreqMin, FreqMax, FreqStep, t_back_flag, tBack, tslope, nH_flag, N_H, beta_dust, kappa_1300, \
                         vLSR, TelescopeSize, Inter_Flag, iso_flag, IsoTableFileName, dbFilename)
        print "done!"


    ##====================================================================================================================================================
    ## construct io_control.xml file
    print "Creating io_control file ..",
    CreateIOControlXMLFile(MAGIXjobDir, MAGIXExpXML, MolfitsFile, MAGIXrootDir)
    print "done!"


    ##====================================================================================================================================================
    ## construct fit.xml file, if no path for another algorithm xml file is given
    AlgorithmXMLFile = AlgorithmXMLFile.strip()
    if (AlgorithmXMLFile != ""):


        ## if neccessary, make relative path absolute
        if (AlgorithmXMLFile[0] != "/"):
            filename = CurrentDir + AlgorithmXMLFile
        else:
            filename = AlgorithmXMLFile


        ## check if path and name of the experimental file exsits
        if not(os.path.exists(filename) or os.path.isfile(filename)):
            print " "
            print " "
            print "Error in XCLASS package, function myXCLASSFit:"
            print "  The given path and name of the algorithm xml-file does not exsist!"
            print " "
            print "  Please enter a valid path and name of an algorithm xml-file and redo myXCLASSFit function call!"
            return (input_file, model_values, MAGIXjobDir)


        ## copy algorithm file to working directory
        print "Copy algorithm file to working directory ..",
        cmd_string = "cp " + filename + " " + MAGIXjobDir + "algorithm_control.xml"
        os.system(cmd_string)
        print "done!"
    else:
        print "Creating fit.xml file ..",
        CreateLMControlXMLFile(MAGIXjobDir, NumberIteration)
        print "done!"


    ##====================================================================================================================================================
    ## start MAGIX
    os.environ["OMP_STACKSIZE"] = '6999M'
    print "\n\nStart MAGIX ..\n"
    task_MAGIX.SetMAGIXEnvironment(MAGIXrootDir)
    MAGIXOption = "model=myxclass"
    ok = task_MAGIX.StartMAGIX(MAGIXrootDir, MAGIXOption, MAGIXjobDir + "io_control.xml")
    if (ok != 0):
        print "\n\n\n\t Program MAGIX aborted!\n\n"


    ##====================================================================================================================================================
    ## read in ouput files
    else:


        ## read in input file
        print " "
        print "\nRead in optimized input file(s) .."
        ConvertLogLinFlag = "true"
        printInfoFlag = "true"
        input_file, ListOfMolfitFileNames, model_values, ListOfModelFileNames = ReadInOutputFiles(MAGIXjobDir, NameOfFunction, ConvertLogLinFlag, \
                                                                                                  RestFreq, vLSR, printInfoFlag)

        ##================================================================================================================================================
        ## remove temp directory
        # command_string = "cd " + MAGIXrootDir + "run/; rm -rf job_" + jobID + "/"
        # os.system(command_string)


        ##================================================================================================================================================
        ## print some informations to screen
        print " "
        print " "
        print "Please note, all files created by the current fit process (new molfit file, log-files etc.) are stored in the"
        print "myXCLASSFit working directory " + chr(34) + MAGIXjobDir + chr(34) + "!"
        print " "
        print " "


    ##====================================================================================================================================================
    ## define return variables
    return (input_file, model_values, MAGIXjobDir)
##--------------------------------------------------------------------------------------------------------------------------------------------------------

