#!/usr/bin/python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module contains the interface to the myXCLASS program.
##  Copyright (C) 2012 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function ChangeToVelocitiy:                   convert frequency points (in MHz) into velocities (in km/s)
##      - function GetParameterLimits:                  get parameter limits from parameter fitflag (valid only for OLD format)
##      - function AnalyzeMolfitFile:                   read in an analyze molfit file
##      - function ConvertParamValue:                   convert parameter value from linear to log value and vise versa
##      - function ConvertNtotToLogScale:               convert column and hydrogen column density (if given for each component) to log scale
##      - function GetUnitParameter:                    determine unit of myXCLASS parameter
##      - function WriteMolfitFile:                     write molfit parameters to new molfit file
##      - function CreateErrorMolfitFile:               creates a machinereadable molfit file containing the best fit values, the left and right errors,
##                                                      the mean value and standard deviation for each free parameter
##      - function ConversionMolfit2XML:                converts a molfit file used by myXCLASS to a xml-file used by MAGIX.
##      - function ImportIsoRatioFile:                  import the iso ratio file
##      - function GetIsoTableParameters:               read iso table parameters from iso table file and include parameters to xml file
##      - function myXCLASSCore:                        calls myXCLASS program
##      - function myXCLASS:                            function defines the interface for the myXCLASS program
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    25.07.2013   initial version
##  T. Moeller    09.04.2014   improved version
##  T. Moeller    23.04.2015   add function ImportIsoRatioFile
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
import numpy                                                                                ## import numpy package
import scipy.integrate                                                                      ## import scipy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import copy                                                                                 ## import copy package
import string                                                                               ## import string package
import random                                                                               ## import random package
import datetime                                                                             ## import datetime package
import math                                                                                 ## import math package
import time                                                                                 ## import package date and time
import copy                                                                                 ## import package copy
import warnings                                                                             ## import warnings package
import matplotlib                                                                           ## import matplotlib package
import pylab                                                                                ## load python package for plotting pylab
import task_MAGIX                                                                           ## import package MAGIX
import task_myXCLASSFit                                                                     ## import package myXCLASSFit
from matplotlib.font_manager import fontManager, FontProperties                             ## import some special packages from matplotlib
import sqlite3                                                                              ## import sqlite3 package
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## convert frequency points (in MHz) into velocities (in km/s)
##
def ChangeToVelocitiy(FreqData, RestFreq, vLSR):
    """

input parameters:
-----------------

    - FreqData:             observational data in MHz

    - RestFreq:             rest frequency in MHz (If this parameter is set to zero, the intensity is plotted against frequency (in MHz)
                            otherwise against velocity (in km/s).

    - vLSR:                 velocity (local standard of rest) in km/s, only used, if RestFreq /= 0. (velocity(Frequency = RestFreq) = vLSR)


output parameters:
------------------

    - VelData:              observational data in km/s


    """


    ## initialize some variables
    VelData = []
    cms = 299792458.0
    ckms = cms * 1.e-3


    ## calculate new velocity scale
    for freq in FreqData[:]:
        vel = vLSR + (ckms / RestFreq) * (RestFreq - freq)
        VelData.append(vel)


    ## define return parameter
    return VelData
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## convert frequency points (in MHz) into velocities (in km/s)
##
def ChangeToVelocitiyOLD(FreqData, RestFreq, vLSR):
    """

input parameters:
-----------------

    - FreqData:             observational data in MHz

    - RestFreq:             rest frequency in MHz (If this parameter is set to zero, the intensity is plotted against frequency (in MHz)
                            otherwise against velocity (in km/s).

    - vLSR:                 velocity (local standard of rest) in km/s, only used, if RestFreq /= 0. (velocity(Frequency = RestFreq) = vLSR)


output parameters:
------------------

    - VelData:              observational data in km/s


    """


    ## initialize some variables
    VelData = []
    cLight = 299792458.0                                                                    ## speed of light in m/s
    cLight = cLight * 1.e-3                                                                 ## speed of light in km/s
    if (len(FreqData) == 1):
        deltaf = 1.0
    else:
        deltaf = abs(FreqData[1] - FreqData[0])                                             ## stepsize of frequency (in MHz)


    ## check, if deltaf is not zero
    if (deltaf == 0):
        print "\n\nError in XCLASS package, function task_myXCLASS.ChangeToVelocitiyOLD:"
        print "\t The stepsize of frequency is zero!"
        print "\t deltaf = ", deltaf
        print "\t FreqData[0] = ", FreqData[0]
        print "\t FreqData[1] = ", FreqData[1]
        print "\n\nSet stepsize to 1!"
        deltaf = 1.0


    ## define delta v
    deltav = -cLight * deltaf
    deltav = deltav / RestFreq


    ## find frequency index, which correspond to rest frequency
    minFreq = min(FreqData[:])
    maxFreq = max(FreqData[:])
    RestFreqIndex = 0
    if (minFreq <= RestFreq and RestFreq <= maxFreq):
        RestFreqIndex = 0
        for freq in FreqData[:]:
            RestFreqIndex += 1
            if (freq > RestFreq):
                break


        ## calculate new velocity scale
        counter = 0
        for freq in FreqData[:]:
            counter += 1
            vel = vLSR + ((counter - RestFreqIndex + 1) - 1) * deltav
            VelData.append(vel)


    ## continue here, if rest frequency is lower than lowest frequency point in FreqData array
    elif (RestFreq < minFreq):
        NumFreqPoints = (RestFreq - minFreq) / deltaf                                       ## determine number of frequency points between lowest
                                                                                            ## frequency and given rest frequency

        ## calculate new velocity scale
        counter = 0
        for freq in FreqData[:]:
            counter += 1
            vel = vLSR + (NumFreqPoints * deltav) + (counter - 1) * deltav
            VelData.append(vel)


    ## continue here, if rest frequency is higher than highest frequency point in FreqData array
    elif (RestFreq > maxFreq):
        NumFreqPoints = (maxFreq - RestFreq) / deltaf                                       ## determine number of frequency points between highest
                                                                                            ## frequency and given rest frequency

        ## calculate new velocity scale
        counter = 0
        for freq in FreqData[:]:
            counter += 1
            vel = vLSR - (NumFreqPoints * deltav) - (counter - 1) * deltav
            VelData.append(vel)


    ## define return parameter
    return VelData
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get parameter limits from parameter fitflag (valid only for OLD format)
##
def GetParameterLimits(ParamName, ParameterValue, LimitValues):
    """

input parameters:
-----------------

    - ParamName:            name of parameter

    - ParameterValue:       value of parameter

    - LimitValues:          range of parameter (fitflag)


output parameters:
------------------

    - lowlimit:             lower limit of current parameter

    - uplimit:              upper limit of current parameter

    """


    ## determine limits for current parameter
    if (ParamName == "source_size" or ParamName == "N_tot"):
        lowlimit = ParameterValue / LimitValues                 ## modification by Tom Bell
        if (lowlimit < 0):
            lowlimit = 0
        uplimit = ParameterValue * LimitValues                  ## modification by Tom Bell
    else:
        lowlimit = ParameterValue - LimitValues
        if (lowlimit < 0 and ParamName != "V_off" and ParamName != "T_Back"):   ## V_off, T_Back could be negativ !!!
            lowlimit = 0
        uplimit = ParameterValue + LimitValues


    ## define return parameter
    return (lowlimit, uplimit)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## read in an analyze molfit file
##
def AnalyzeMolfitFile(MolfitsFile):
    """

input parameters:
-----------------

    - MolfitsFile:          path and name of molfit file or content of molfit file


output parameters:
------------------

    - MoleculesInMolfitFile:    list of molecule names in molfit file

    - AllParameters:            list molfit files for each molecule

    - MolfitFileForEachMolecule:    molfit files for each molecule

    """

    # Debug:
    # print "MolfitsFile = ", MolfitsFile


    ## analyze molfit file
    if (type(MolfitsFile) is str):
        f = open(MolfitsFile)
        MolfitFileContent = f.readlines()
        f.close()
    else:
        MolfitFileContent = MolfitsFile


    ##====================================================================================================================================================
    ## Molfit file format line coding
    MolfitFileFormats = {\


        ## PUREtbFnhF and PURAtbFnhF
        "PUREtbFnhF":["source_size", \
                      "T_rot", \
                      "N_tot", \
                      "V_width", \
                      "V_off", \
                      "CFFlag"], \
        "PURAtbFnhF":["T_rot", \
                      "N_tot", \
                      "V_width", \
                      "V_off", \
                      "CFFlag"], \


        ## PUREtbTnhF and PURAtbTnhF
        "PUREtbTnhF":["source_size", \
                      "T_rot", \
                      "N_tot", \
                      "V_width", \
                      "V_off", \
                      "T_dOff", \
                      "T_dSlope", \
                      "CFFlag"], \
        "PURAtbTnhF":["T_rot", \
                      "N_tot", \
                      "V_width", \
                      "V_off", \
                      "T_dOff", \
                      "T_dSlope", \
                      "CFFlag"], \


        ## PUREtbFnhT and PURAtbFnhT
        "PUREtbFnhT":["source_size", \
                      "T_rot", \
                      "N_tot", \
                      "V_width", \
                      "V_off", \
                      "nHcolumn", \
                      "kappa", \
                      "beta", \
                      "CFFlag"], \
        "PURAtbFnhT":["T_rot", \
                      "N_tot", \
                      "V_width", \
                      "V_off", \
                      "nHcolumn", \
                      "kappa", \
                      "beta", \
                      "CFFlag"], \


        ## PUREtbTnhT and PURAtbTnhT
        "PUREtbTnhT":["source_size", \
                      "T_rot", \
                      "N_tot", \
                      "V_width", \
                      "V_off", \
                      "T_dOff", \
                      "T_dSlope", \
                      "nHcolumn", \
                      "kappa", \
                      "beta", \
                      "CFFlag"], \
        "PURAtbTnhT":["T_rot", \
                      "N_tot", \
                      "V_width", \
                      "V_off", \
                      "T_dOff", \
                      "T_dSlope", \
                      "nHcolumn", \
                      "kappa", \
                      "beta", \
                      "CFFlag"], \


        ## OLDEtbFnhF and OLDAtbFnhF
        "OLDEtbFnhF":["source_size_FitFlag", "source_size", \
                      "T_rot_FitFlag", "T_rot", \
                      "N_tot_FitFlag", "N_tot", \
                      "V_width_FitFlag", "V_width", \
                      "V_off_FitFlag", "V_off", \
                      "CFFlag"], \
        "OLDAtbFnhF":["T_rot_FitFlag", "T_rot", \
                      "N_tot_FitFlag", "N_tot", \
                      "V_width_FitFlag", "V_width", \
                      "V_off_FitFlag", "V_off", \
                      "CFFlag"], \


        ## OLDEtbTnhF and OLDAtbTnhF
        "OLDEtbTnhF":["source_size_FitFlag", "source_size", \
                      "T_rot_FitFlag", "T_rot", \
                      "N_tot_FitFlag", "N_tot", \
                      "V_width_FitFlag", "V_width", \
                      "V_off_FitFlag", "V_off", \
                      "T_dOff_FitFlag", "T_dOff", \
                      "T_dSlope_FitFlag", "T_dSlope", \
                      "CFFlag"], \
        "OLDAtbTnhF":["T_rot_FitFlag", "T_rot", \
                      "N_tot_FitFlag", "N_tot", \
                      "V_width_FitFlag", "V_width", \
                      "V_off_FitFlag", "V_off", \
                      "T_dOff_FitFlag", "T_dOff", \
                      "T_dSlope_FitFlag", "T_dSlope", \
                      "CFFlag"], \


        ## OLDEtbFnhT and OLDAtbFnhT
        "OLDEtbFnhT":["source_size_FitFlag", "source_size", \
                      "T_rot_FitFlag", "T_rot", \
                      "N_tot_FitFlag", "N_tot", \
                      "V_width_FitFlag", "V_width", \
                      "V_off_FitFlag", "V_off", \
                      "nHcolumn_FitFlag", "nHcolumn", \
                      "kappa_FitFlag", "kappa", \
                      "beta_FitFlag", "beta", \
                      "CFFlag"], \
        "OLDAtbFnhT":["T_rot_FitFlag", "T_rot", \
                      "N_tot_FitFlag", "N_tot", \
                      "V_width_FitFlag", "V_width", \
                      "V_off_FitFlag", "V_off", \
                      "nHcolumn_FitFlag", "nHcolumn", \
                      "kappa_FitFlag", "kappa", \
                      "beta_FitFlag", "beta", \
                      "CFFlag"], \


        ## OLDEtbTnhT and OLDAtbTnhT
        "OLDEtbTnhT":["source_size_FitFlag", "source_size", \
                      "T_rot_FitFlag", "T_rot", \
                      "N_tot_FitFlag", "N_tot", \
                      "V_width_FitFlag", "V_width", \
                      "V_off_FitFlag", "V_off", \
                      "T_dOff_FitFlag", "T_dOff", \
                      "T_dSlope_FitFlag", "T_dSlope", \
                      "nHcolumn_FitFlag", "nHcolumn", \
                      "kappa_FitFlag", "kappa", \
                      "beta_FitFlag", "beta", \
                      "CFFlag"], \
        "OLDAtbTnhT":["T_rot_FitFlag", "T_rot", \
                      "N_tot_FitFlag", "N_tot", \
                      "V_width_FitFlag", "V_width", \
                      "V_off_FitFlag", "V_off", \
                      "T_dOff_FitFlag", "T_dOff", \
                      "T_dSlope_FitFlag", "T_dSlope", \
                      "nHcolumn_FitFlag", "nHcolumn", \
                      "kappa_FitFlag", "kappa", \
                      "beta_FitFlag", "beta", \
                      "CFFlag"], \


        ## NEWEtbFnhF and NEWAtbFnhF
        "NEWEtbFnhF":["source_size_flag", "source_size_uplimit", "source_size_lowlimit", "source_size", \
                      "T_rot_flag", "T_rot_uplimit", "T_rot_lowlimit", "T_rot", \
                      "N_tot_flag", "N_tot_uplimit", "N_tot_lowlimit", "N_tot", \
                      "V_width_flag", "V_width_uplimit", "V_width_lowlimit", "V_width", \
                      "V_off_flag", "V_off_uplimit", "V_off_lowlimit", "V_off", \
                      "CFFlag"], \
        "NEWAtbFnhF":["T_rot_flag", "T_rot_uplimit", "T_rot_lowlimit", "T_rot", \
                      "N_tot_flag", "N_tot_uplimit", "N_tot_lowlimit", "N_tot", \
                      "V_width_flag", "V_width_uplimit", "V_width_lowlimit", "V_width", \
                      "V_off_flag", "V_off_uplimit", "V_off_lowlimit", "V_off", \
                      "CFFlag"], \


        ## NEWEtbTnhF and NEWAtbTnhF
        "NEWEtbTnhF":["source_size_flag", "source_size_uplimit", "source_size_lowlimit", "source_size", \
                      "T_rot_flag", "T_rot_uplimit", "T_rot_lowlimit", "T_rot", \
                      "N_tot_flag", "N_tot_uplimit", "N_tot_lowlimit", "N_tot", \
                      "V_width_flag", "V_width_uplimit", "V_width_lowlimit", "V_width", \
                      "V_off_flag", "V_off_uplimit", "V_off_lowlimit", "V_off", \
                      "T_dOff_flag", "T_dOff_uplimit", "T_dOff_lowlimit", "T_dOff", \
                      "T_dSlope_flag", "T_dSlope_uplimit", "T_dSlope_lowlimit", "T_dSlope", \
                      "CFFlag"], \
        "NEWAtbTnhF":["T_rot_flag", "T_rot_uplimit", "T_rot_lowlimit", "T_rot", \
                      "N_tot_flag", "N_tot_uplimit", "N_tot_lowlimit", "N_tot", \
                      "V_width_flag", "V_width_uplimit", "V_width_lowlimit", "V_width", \
                      "V_off_flag", "V_off_uplimit", "V_off_lowlimit", "V_off", \
                      "T_dOff_flag", "T_dOff_uplimit", "T_dOff_lowlimit", "T_dOff", \
                      "T_dSlope_flag", "T_dSlope_uplimit", "T_dSlope_lowlimit", "T_dSlope", \
                      "CFFlag"], \


        ## NEWEtbFnhT and NEWAtbFnhT
        "NEWEtbFnhT":["source_size_flag", "source_size_uplimit", "source_size_lowlimit", "source_size", \
                      "T_rot_flag", "T_rot_uplimit", "T_rot_lowlimit", "T_rot", \
                      "N_tot_flag", "N_tot_uplimit", "N_tot_lowlimit", "N_tot", \
                      "V_width_flag", "V_width_uplimit", "V_width_lowlimit", "V_width", \
                      "V_off_flag", "V_off_uplimit", "V_off_lowlimit", "V_off", \
                      "nHcolumn_flag", "nHcolumn_uplimit", "nHcolumn_lowlimit", "nHcolumn", \
                      "kappa_flag", "kappa_uplimit", "kappa_lowlimit", "kappa", \
                      "beta_flag", "beta_uplimit", "beta_lowlimit", "beta", \
                      "CFFlag"], \
        "NEWAtbFnhT":["T_rot_flag", "T_rot_uplimit", "T_rot_lowlimit", "T_rot", \
                      "N_tot_flag", "N_tot_uplimit", "N_tot_lowlimit", "N_tot", \
                      "V_width_flag", "V_width_uplimit", "V_width_lowlimit", "V_width", \
                      "V_off_flag", "V_off_uplimit", "V_off_lowlimit", "V_off", \
                      "nHcolumn_flag", "nHcolumn_uplimit", "nHcolumn_lowlimit", "nHcolumn", \
                      "kappa_flag", "kappa_uplimit", "kappa_lowlimit", "kappa", \
                      "beta_flag", "beta_uplimit", "beta_lowlimit", "beta", \
                      "CFFlag"], \


        ## NEWEtbTnhT and NEWAtbTnhT
        "NEWEtbTnhT":["source_size_flag", "source_size_uplimit", "source_size_lowlimit", "source_size", \
                      "T_rot_flag", "T_rot_uplimit", "T_rot_lowlimit", "T_rot", \
                      "N_tot_flag", "N_tot_uplimit", "N_tot_lowlimit", "N_tot", \
                      "V_width_flag", "V_width_uplimit", "V_width_lowlimit", "V_width", \
                      "V_off_flag", "V_off_uplimit", "V_off_lowlimit", "V_off", \
                      "T_dOff_flag", "T_dOff_uplimit", "T_dOff_lowlimit", "T_dOff", \
                      "T_dSlope_flag", "T_dSlope_uplimit", "T_dSlope_lowlimit", "T_dSlope", \
                      "nHcolumn_flag", "nHcolumn_uplimit", "nHcolumn_lowlimit", "nHcolumn", \
                      "kappa_flag", "kappa_uplimit", "kappa_lowlimit", "kappa", \
                      "beta_flag", "beta_uplimit", "beta_lowlimit", "beta", \
                      "CFFlag"], \
        "NEWAtbTnhT":["T_rot_flag", "T_rot_uplimit", "T_rot_lowlimit", "T_rot", \
                      "N_tot_flag", "N_tot_uplimit", "N_tot_lowlimit", "N_tot", \
                      "V_width_flag", "V_width_uplimit", "V_width_lowlimit", "V_width", \
                      "V_off_flag", "V_off_uplimit", "V_off_lowlimit", "V_off", \
                      "T_dOff_flag", "T_dOff_uplimit", "T_dOff_lowlimit", "T_dOff", \
                      "T_dSlope_flag", "T_dSlope_uplimit", "T_dSlope_lowlimit", "T_dSlope", \
                      "nHcolumn_flag", "nHcolumn_uplimit", "nHcolumn_lowlimit", "nHcolumn", \
                      "kappa_flag", "kappa_uplimit", "kappa_lowlimit", "kappa", \
                      "beta_flag", "beta_uplimit", "beta_lowlimit", "beta", \
                      "CFFlag"]}


    ##====================================================================================================================================================
    ## split final molfit file in N small molfit files containing only one molecule
    MoleculesInMolfitFile = []
    MolfitFileForEachMolecule = []
    CounterMolecules = 0
    CounterLines = (-1)
    LinesOfMolfitFile = []
    AllParameters = []
    ParametersPerMolecule = []
    for line in MolfitFileContent:                                                          ## loop over all lines of the molfit file
        CounterLines += 1                                                                   ## increase counter for lines
        StrippedLine = line.strip()                                                         ## remove leading and trailing blanks


        ## remove comments
        i = StrippedLine.find("%") 
        if (i > (-1)):
            StrippedLine = StrippedLine[:i].strip()


        ## analyze current line of molfit file
        if (StrippedLine != ""):
            SplittedLine = StrippedLine.split()                                             ## split current line into columns
            NumberElementsLine = len(SplittedLine)                                          ## determine number of columns
            if (NumberElementsLine == 2):                                                   ## two columns current line: a molecule name is defined
                if (CounterMolecules > 0):
                    MolfitFileForEachMolecule.append(LinesOfMolfitFile)                     ## add molecule name to output list
                    AllParameters.append(ParametersPerMolecule)
                    ParametersPerMolecule = []
                CounterMolecules += 1                                                       ## increase counter for molecules
                LinesOfMolfitFile = []
                LinesOfMolfitFile.append(line)
                MoleculesInMolfitFile.append(SplittedLine[0])
            else:
                LinesOfMolfitFile.append(line)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## get parameters from current line


                ## analyze last element of current line and add CFFlag if necessary
                LastElement = SplittedLine[-1]                                              ## get the last element
                LastElement = LastElement.lower()
                if (LastElement != "a" and LastElement != "f" and LastElement != "e" and LastElement != "c"):
                    LastElement = " c"
                    NumberElementsLine += 1
                    SplittedLine.append(LastElement)
                    LastElement = "c"


                ## convert new label of compounds ('c' (emission) for core, 'f' (absorption) for foreground to known label
                if (LastElement == "a"):
                    LastElement = "f"
                if (LastElement == "e"):
                    LastElement = "c"


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## determine format version of current line
                MolfitFileFormatVersion = ""
                lowline = StrippedLine.lower()                                              ## convert line to lower case
                NewOldMolfitFileFormat = "OLD"
                if (lowline[0] == "y" or lowline[0] == "n" or lowline[0] == "t" or lowline[0] == "f"):
                    NewOldMolfitFileFormat = "NEW"
                elif (NumberElementsLine <= 11):
                    NewOldMolfitFileFormat = "PUR"
                    if (NumberElementsLine == 9 and LastElement == "f"):
                        NewOldMolfitFileFormat = "OLD"
                    elif (NumberElementsLine == 11 and SplittedLine[2].find("e") == (-1) and SplittedLine[2].find("E") == (-1)):
                        NewOldMolfitFileFormat = "OLD"

                # Debug:
                # print "lowline = ", lowline.strip()
                # print "NumberElementsLine = ", NumberElementsLine
                # print "NewOldMolfitFileFormat = ", NewOldMolfitFileFormat



                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## determine format version of myXCLASS
                ErrorFlag = "false"                                                         ## initialize error flag, i.e. no error occurred


                ## pure molfit file without limit definition
                if (NewOldMolfitFileFormat == "PUR"):                                       ## line is in PURE format
                    if (NumberElementsLine == 6):
                        MolfitFileFormatVersion = "PUREtbFnhF"                              ## ss, Tex, N, dv, v, ae
                    elif (NumberElementsLine == 5):
                        MolfitFileFormatVersion = "PURAtbFnhF"                              ##     Tex, N, dv, v, ae
                    elif (NumberElementsLine == 7):
                        MolfitFileFormatVersion = "PURAtbTnhF"                              ##     Tex, N, dv, v, Td, TdSlope, ae
                    elif (NumberElementsLine == 9):
                        MolfitFileFormatVersion = "PUREtbFnhT"                              ## ss, Tex, N, dv, v, nH, kappa, beta, ae
                    elif (NumberElementsLine == 11):
                        MolfitFileFormatVersion = "PUREtbTnhT"                              ## ss, Tex, N, dv, v, Td, TdSlope, nH, kappa, beta, ae
                    elif (NumberElementsLine == 10):
                        MolfitFileFormatVersion = "PURAtbTnhT"                              ##     Tex, N, dv, v, Td, TdSlope, nH, kappa, beta, ae
                    elif (NumberElementsLine == 8):
                        SecondColumnString = SplittedLine[1].lower()
                        SecondColumnValue = float(SecondColumnString)
                        FifthColumnString = SplittedLine[4].lower()
                        FifthColumnValue = float(FifthColumnString)
                        # if (SecondColumnString.find("e") == (-1) and FifthColumnString.find("e") == (-1)):
                        if (SecondColumnValue < 1.e5 and FifthColumnValue < 1.e5):
                            MolfitFileFormatVersion = "PUREtbTnhF"                          ## ss, Tex, N, dv, v, Td, TdSlope, ae
                        # elif (SecondColumnString.find("e") > (-1) and FifthColumnString.find("e") > (-1)):
                        elif (SecondColumnValue > 1.e5 or FifthColumnValue > 1.e5):
                            MolfitFileFormatVersion = "PURAtbFnhT"                          ##     Tex, N, dv, v, nH, kappa, beta, ae
                        else:
                            ErrorFlag = "true"
                    else:
                        ErrorFlag = "true"

                ## old version
                elif (NewOldMolfitFileFormat == "OLD"):                                     ## line is in old format
                    if (NumberElementsLine == 9):
                        MolfitFileFormatVersion = "OLDAtbFnhF"                              ##     Tex, N, dv, v, ae
                    elif (NumberElementsLine == 11):
                        MolfitFileFormatVersion = "OLDEtbFnhF"                              ## ss, Tex, N, dv, v, ae
                    elif (NumberElementsLine == 13):
                        MolfitFileFormatVersion = "OLDAtbTnhF"                              ##     Tex, N, dv, v, Td, TdSlope, ae
                    elif (NumberElementsLine == 17):
                        MolfitFileFormatVersion = "OLDEtbFnhT"                              ## ss, Tex, N, dv, v, nH, kappa, beta, ae
                    elif (NumberElementsLine == 19):
                        MolfitFileFormatVersion = "OLDAtbTnhT"                              ##     Tex, N, dv, v, Td, TdSlope, nH, kappa, beta, ae
                    elif (NumberElementsLine == 21):
                        MolfitFileFormatVersion = "OLDEtbTnhT"                              ## ss, Tex, N, dv, v, Td, TdSlope, nH, kappa, beta, ae
                    elif (NumberElementsLine == 15):
                        FourthColumnString = SplittedLine[3].lower()
                        FourthColumnValue = float(FourthColumnString)
                        TenthColumnString = SplittedLine[9].lower()
                        TenthColumnValue = float(TenthColumnString)
                        # if (FourthColumnString.find("e") == (-1) and TenthColumnString.find("e") == (-1)):
                        if (FourthColumnValue < 1.e5 and TenthColumnValue < 1.e5):
                            MolfitFileFormatVersion = "OLDEtbTnhF"                          ## ss, Tex, N, dv, v, Td, TdSlope, ae
                        # elif (FourthColumnString.find("e") > (-1) and TenthColumnString.find("e") > (-1)):
                        elif (FourthColumnValue > 1.e5 or TenthColumnValue > 1.e5):
                            MolfitFileFormatVersion = "OLDAtbFnhT"                          ##     Tex, N, dv, v, nH, kappa, beta, ae
                        else:
                            ErrorFlag = "true"
                    else:
                        ErrorFlag = "true"


                ## new version
                elif (NewOldMolfitFileFormat == "NEW"):                                     ## line is in new format
                    if (NumberElementsLine == 17):
                        MolfitFileFormatVersion = "NEWAtbFnhF"                              ##     Tex, N, dv, v, ae
                    elif (NumberElementsLine == 21):
                        MolfitFileFormatVersion = "NEWEtbFnhF"                              ## ss, Tex, N, dv, v, ae
                    elif (NumberElementsLine == 25):
                        MolfitFileFormatVersion = "NEWAtbTnhF"                              ##     Tex, N, dv, v, Td, TdSlope, ae
                    elif (NumberElementsLine == 33):
                        MolfitFileFormatVersion = "NEWEtbFnhT"                              ## ss, Tex, N, dv, v, nH, kappa, beta, ae
                    elif (NumberElementsLine == 37):
                        MolfitFileFormatVersion = "NEWAtbTnhT"                              ##     Tex, N, dv, v, Td, TdSlope, nH, kappa, beta, ae
                    elif (NumberElementsLine == 41):
                        MolfitFileFormatVersion = "NEWEtbTnhT"                              ## ss, Tex, N, dv, v, Td, TdSlope, nH, kappa, beta, ae
                    elif (NumberElementsLine == 29):
                        EigthColumnString = SplittedLine[7].lower()
                        EigthColumnValue = float(EigthColumnString)
                        TwentythColumnString = SplittedLine[19].lower()
                        TwentythColumnValue = float(TwentythColumnString)
                        # if (EigthColumnString.find("e") == (-1) and TwentythColumnString.find("e") == (-1)):
                        if (EigthColumnValue < 1.e5 and TwentythColumnValue < 1.e5):
                            MolfitFileFormatVersion = "NEWEtbTnhF"                          ## ss, Tex, N, dv, v, Td, TdSlope, ae
                        # elif (EigthColumnString.find("e") > (-1) and TwentythColumnString.find("e") > (-1)):
                        elif (EigthColumnValue > 1.e5 or TwentythColumnValue > 1.e5):
                            MolfitFileFormatVersion = "NEWAtbFnhT"                          ##     Tex, N, dv, v, nH, kappa, beta, ae
                        else:
                            ErrorFlag = "true"
                    else:
                        ErrorFlag = "true"


                ## other version of molfit file format
                else:
                    ErrorFlag = "true"


                ## does an error occur?
                if (ErrorFlag == "true"):
                    print "\n\nError in XCLASS package, function task_myXCLASS.AnalyzeMolfitFile:"
                    print "\t\t Can not identify the format of the current line"
                    print "\t\t in molfit file " + MolfitsFile
                    print "\n\t\t MolfitFileFormatVersion = ", MolfitFileFormatVersion
                    print "\t\t NumberElementsLine = ", NumberElementsLine
                    print "\t\t line number: " + str(CounterLines)
                    print "\t\t line = >>" + line.replace("\n", "") + "<<"
                    print "\n\n"
                    return (MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule)

                # Debug:
                # print "CounterLines, NumberElementsLine, MolfitFileFormatVersion = ", CounterLines, NumberElementsLine, MolfitFileFormatVersion


                ## add molfit file format version to output list
                ParameterPerLineList = []
                entry = ["MolfitFileFormatVersion", 0, 0, 0, MolfitFileFormatVersion]
                ParameterPerLineList.append(entry)


                ##========================================================================================================================================
                ## analyse current line
                flag = "0"
                uplimit = 0
                lowlimit = 0
                counter = (-1)
                NumberParameter = 0
                for element in SplittedLine:                                                ## loop over all elements of the current line
                    counter += 1                                                            ## increase counter for elements in the current line
                    NumberParameter += 1


                    ## get name of parameter which correspond to the current column
                    ParamName = MolfitFileFormats[MolfitFileFormatVersion][counter]
                    lowlimit = 0.0
                    uplimit = 0.0


                    ## special treatment of CFFlag
                    if (ParamName == "CFFlag"):
                        element = "  " + element
                        flag = "0"


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## determine lower and upper limits for the old format
                    if (NewOldMolfitFileFormat == "PUR"):                                   ## line is in PURE format
                        if (ParamName != "CFFlag"):                                         ## determine lower and upper limits except for e/a flag
                            ParameterValue = (float(element))                               ## get parameter value


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## determine lower and upper limits for the old format
                    elif (NewOldMolfitFileFormat == "OLD"):                                 ## line is in old format
                        if (ParamName.endswith("_FitFlag")):                                ## get limit range definition
                            LimitValues = abs(float(element))
                            flag = "0"
                            lowlimit = 0.0
                            uplimit = 1.0
                        elif (ParamName != "CFFlag"):                                       ## determine lower and upper limits except for e/a flag
                            ParameterValue = (float(element))                               ## get parameter value
                            if (LimitValues > 0.0):                                         ## a limit specification unequal zero indicates a free param.
                                flag = "1"
                                lowlimit, uplimit = GetParameterLimits(ParamName, ParameterValue, LimitValues)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## determine lower and upper limits for the old format
                    elif (NewOldMolfitFileFormat == "NEW"):                                 ## current line is in new format
                        if (ParamName.endswith("_flag")):                                   ## get flag defining if a parameter is fitted or not
                            flag = element
                            if (flag.find("y") > (-1) or flag.find("t") > (-1)):
                                flag = "1"
                            else:
                                flag = "0"
                        elif (ParamName.endswith("_uplimit")):                              ## get upper limit for parameter
                            ParamUplimit = float(element)
                            lowlimit = 0.0
                            uplimit = 1.0
                        elif (ParamName.endswith("_lowlimit")):                             ## get lower limit for parameter
                            ParamLowlimit = float(element)
                            lowlimit = 0.0
                            uplimit = 1.0
                        elif (ParamName != "CFFlag"):                                       ## determine lower and upper limits except for CFFlag
                            lowlimit = ParamLowlimit
                            uplimit = ParamUplimit


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## add parameter name, limits and value to output list
                    entry = [ParamName, flag, lowlimit, uplimit, element]
                    ParameterPerLineList.append(entry)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## add parameters of current line to ParametersPerMolecule array
                ParametersPerMolecule.append(ParameterPerLineList)


    ## add parameter list for last line to global list
    if (CounterMolecules > 0):
        MolfitFileForEachMolecule.append(LinesOfMolfitFile)
        AllParameters.append(ParametersPerMolecule)


    ## define return parameter
    return (MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## convert parameter value from linear to log value and vise versa
##
def ConvertParamValue(LogLinearFlag, OrigValue):
    """

input parameters:
-----------------

    - LogLinearFlag:        flag indicating linear or log conversion

    - OrigValue:            original parameter value



output parameters:
------------------

    - NewParamValue:        converted parameter value

    """

    # Debug:
    # print "OrigValue = ", OrigValue
    # print "LogLinearFlag = ", LogLinearFlag


    ## initialize return parameter
    NewParamValue = 0.0


    ## to from linear to log conversion
    if (LogLinearFlag == "log"):
        if (OrigValue == 0.0):
            NewParamValue = math.log10(1.e-300)
        else:
            if (OrigValue < 0.E0):
                OrigValue = abs(OrigValue)
            NewParamValue = math.log10(OrigValue)
            if (NewParamValue > 307.0):                                                     ## check for overflow
                NewParamValue = 307.0


    ## to from log to linear conversion
    elif (LogLinearFlag == "linear"):
        NewParamValue = 10.E0**OrigValue

    # Debug:
    # print "NewParamValue = ", NewParamValue


    ## define return value
    return NewParamValue
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## convert column and hydrogen column density (if given for each component) to log scale
##
def ConvertNtotToLogScale(MolfitsFile, LogLinearFlag):
    """

input parameters:
-----------------

    - MolfitsFile:          path and name of the molfit file located in the job directory

    - LogLinearFlag:        indicates, if column density values have to be converted from linear to log. scale (=log) or from log t linear (linear)



output parameters:
------------------

    - None

    """


    ## print what you do
    #    if (LogLinearFlag == "log"):
    #        print "Convert values for N_tot and hydrogen column density to log scale for file " + MolfitsFile + " ..",
    #    elif (LogLinearFlag == "linear"):
    #        print "Convert log values for N_tot and hydrogen column density to linear scale for file " + MolfitsFile + " ..",


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze molfit file using myXCLASS routine
    MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = AnalyzeMolfitFile(MolfitsFile)


    ## write new molfit file
    NEWFormatFlag = ""
    NewMolfitFile = WriteMolfitFile(MolfitsFile, MoleculesInMolfitFile, AllParameters, LogLinearFlag, NEWFormatFlag)


    ## define return value ?
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## determine unit of a myXCLASS parameter
##
def GetUnitParameter(NameParam):
    """

input parameters:
-----------------

    - NameParam:            name of myXCLASS parameter



output parameters:
------------------

    - UnitParm:             unit of selected myXCLASS parameter

    """

    # Debug:
    # print "NameParam = ", NameParam


    ## define unit for each parameter
    if (NameParam == "source_size"):
        UnitParm = "arcsec"
    elif (NameParam == "T_rot" or NameParam == "T_ex" or NameParam == "T_Back" or NameParam == "T_dOff"):
        UnitParm = "K"
    elif (NameParam == "N_tot" or NameParam == "nHcolumn"):
        UnitParm = "cm^(-2)"
    elif (NameParam == "V_width" or NameParam == "V_off"):
        UnitParm = "km/s"
    elif (NameParam == "T_Slope" or NameParam == "T_dSlope" or NameParam == "beta" or NameParam == "CFFlag"):
        UnitParm = " "
    elif (NameParam == "kappa"):
        UnitParm = "cm^2/g"
    else:
        UnitParm = ""


    ## define return value ?
    return UnitParm
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## write molfit parameters to new molfit file
##
def WriteMolfitFile(MolfitsFile, MoleculesInMolfitFile, AllParameters, LogLinearFlag, NEWFormatFlag):
    """

input parameters:
-----------------

    - MolfitsFile:          path and name of the molfit file located in the job directory

    - MoleculesInMolfitFile:    list of all molcules included in the new molfit file

    - AllParameters:        all parameters for the new molfit file

    - LogLinearFlag:        indicates, if column density values have to be converted from linear to log. scale (=log) or from log t linear (linear)

    - NEWFormatFlag:        indicates, if molfit file is written in new format (True) or if current format is used (False)



output parameters:
------------------

    - NewMolfitFile:        array containing the new molfit file, if variable does not define path and name of molfit file
    """


    ## check, if result is written to file or to python array
    NewMolfitFile = []
    if (type(MolfitsFile) is str):
        WriteToFileFlag = "true"
        MolFile = open(MolfitsFile, 'w')
    else:
        WriteToFileFlag = "false"


    ## write parameter to new molfit file
    for MoleculeIndex in xrange(len(MoleculesInMolfitFile)):                                ## loop over all molecules in the molfit file
        LocalMolfitParameters = AllParameters[MoleculeIndex]                                ## get the parameters describing the current molecule


        ## write name of molecule and number of components
        NewLine = MoleculesInMolfitFile[MoleculeIndex] + "   " + str(len(LocalMolfitParameters)) + "\n"
        if (WriteToFileFlag == "false"):
            NewMolfitFile.append(NewLine)
        else:
            MolFile.write(NewLine)                                                          ## write new line to file


        ## write parameter for current molecule to xml file
        for line in LocalMolfitParameters:                                                  ## loop over all components of the current molecule


            ## write each parameter for current molecule to xml file
            MolfitFileFormatVersionLocal = ""
            EmissionFlag = "T"                                                              ## define default setting for EmissionFlag
            tdFlag = "F"                                                                    ## define default setting for tdFlag
            nHFlag = "F"                                                                    ## define default setting for nHFlag
            NewLine = ""                                                                    ## initialize new line
            for param in line:                                                              ## loop over all parameters

                # Debug:
                # print "param = ", param


                ParameterName = param[0].strip()                                            ## get name of parameter
                LimitValues = param[1]                                                      ## get fit-flag of parameter
                LimitValuesTrueFalse = "false"                                              ## define LimitValuesTrueFalse variable
                if (float(LimitValues) != 0):
                    LimitValuesTrueFalse = "true"
                lowlimit = param[2]                                                         ## get lower limit
                uplimit = param[3]                                                          ## get upper limit
                ParameterValue = param[4]                                                   ## get value of parameter

                # Debug:
                # print "param = ", param


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## define molfit file format for each line
                if (ParameterName == "MolfitFileFormatVersion"):
                    MolfitFileFormatVersionLocal = ParameterValue[:3]                       ## get parameter indicating molfit format
                    EmissionFlag = ParameterValue[3:4]                                      ## get parameter indicating if a absorption component
                                                                                            ## definiton contains no source size setting
                    tdFlag = ParameterValue[6:7]                                            ## get parameter tbFlag indicating if background temperature
                                                                                            ## etc is given in molfit file
                    nHFlag = ParameterValue[9:10]                                           ## write parameter nHFlag indicating if column density etc
                                                                                            ## is given in molfit file
                else:


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## if LogLinearFlag is defined, convert column densities from linear to log scale or vis versa
                    if (LogLinearFlag != ""):
                        if (ParameterName == "N_tot" or ParameterName == "N_tot_uplimit" or ParameterName == "N_tot_lowlimit" or \
                            ParameterName == "nHcolumn" or ParameterName == "nHcolumn_uplimit" or ParameterName == "nHcolumn_lowlimit"):
                            ParameterValue = float(ParameterValue)
                            NewParamValue = ConvertParamValue(LogLinearFlag, ParameterValue)
                            ParameterValue = "%25.15e" % NewParamValue

                        # Debug:
                        # print "ParameterValue = ", ParameterValue


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## convert OLD format to NEW format
                    if (MolfitFileFormatVersionLocal == "OLD" and NEWFormatFlag == "true"):
                        if (ParameterName.endswith("_FitFlag")):
                            ParameterValue = float(ParameterValue)
                            if (ParameterValue == 0.0):                                     ## no range is defined, parameter is kept constant
                                element = " n "
                                lowlimit = 0.0
                                uplimit = 0.0
                            else:                                                           ## a range is defined, determine limits for parameter
                                element = " y "
                                lowlimit, uplimit = GetParameterLimits(ParamName, ParameterValue, LimitValues)
                            element += "%12.3f %12.3f" % (lowlimit, uplimit)                ## create new part of current line
                        elif (ParameterName != "CFFlag"):
                            element = "%20.10e" % float(ParameterValue)
                        else:
                            element = ParameterValue


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## write parameter value with different formats to line
                    else:
                        if (MolfitFileFormatVersionLocal == "PUR"):
                            if (ParameterName != "CFFlag"):
                                element = "%20.10e" % float(ParameterValue)
                            else:
                                element = ParameterValue

                        elif (MolfitFileFormatVersionLocal == "OLD"):
                            if (ParameterName.endswith("_FitFlag")):
                                element = "%12.2f" % float(ParameterValue)
                            elif (ParameterName != "CFFlag"):
                                element = "%20.10e" % float(ParameterValue)
                            else:
                                element = ParameterValue

                        elif (MolfitFileFormatVersionLocal == "NEW"):
                            if (ParameterName.endswith("_uplimit") or ParameterName.endswith("_lowlimit")):
                                if (ParameterName.startswith("N_tot") or ParameterName.startswith("nHcolumn")):
                                    element = "%12.3e" % float(ParameterValue)
                                else:
                                    element = "%12.3f" % float(ParameterValue)
                            elif (ParameterName == "CFFlag" or ParameterName.endswith("_flag") ):
                                element = ParameterValue
                            else:
                                element = "%20.10e" % float(ParameterValue)


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## create new line
                    NewLine += " " + element


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## write new line to file
            NewLine = NewLine + "\n"
            if (WriteToFileFlag == "false"):
                NewMolfitFile.append(NewLine)
            else:
                MolFile.write(NewLine)


    ## close xml-file
    if (WriteToFileFlag == "true"):
        MolFile.close()


    ## we've finished
    return NewMolfitFile
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## creates a machinereadable molfit file containing the best fit values, the left and right errors, the mean value and standard deviation for each free
## parameter
##
def CreateErrorMolfitFile(InstanceXMLFile, MolfitFile, printInfoFlag):
    """

input parameters:
-----------------

    - InstanceXMLFile:      path and name of instance xml-file located in the job directory

    - MolfitFile:           path and name of the molfit file located in the job directory

    - printInfoFlag:        (T/F) print information to screen



output parameters:
------------------

    - None

    """

    # Debug:
    # print "InstanceXMLFile = ", InstanceXMLFile
    # print "MolfitFile = ", MolfitFile


    ## define name of error molfit file
    i = MolfitFile.rfind("/")                                                               ## get path of molfit file
    if (i > 0):                                                                             ## is a path defined?
        i += 1                                                                              ## include "/" character
        MolfitFileName = MolfitFile[i:]                                                     ## get name of molfit file
        j = MolfitFileName.rfind(".")                                                       ## get name without .molfit extension
        NewErrorMolfitFileName = MolfitFile[:i] + MolfitFileName[:j] + "__error.molfit"     ## define new name

    # Debug:
    # print "NewErrorMolfitFileName = ", NewErrorMolfitFileName


    ## print what you do
    if (printInfoFlag == "true"):
        print "Create further molfit file " + NewErrorMolfitFileName + " containing error informations for each free parameter ..",


    ## get total number of parameters in the instance xml file
    TotalParamNumber = task_MAGIX.GetXMLtag(InstanceXMLFile, "NumberParameters")
    TotalParamNumber = int(float(TotalParamNumber[0]))


    ## get names of each parameter
    AllParamNames = task_MAGIX.GetXMLtag(InstanceXMLFile, "name")


    ## get values of each parameter
    AllParamValues = task_MAGIX.GetXMLtag(InstanceXMLFile, "value")


    ## get low limits of each parameter
    AllParamLowLimit = task_MAGIX.GetXMLtag(InstanceXMLFile, "lowlimit")


    ## get upper limits of each parameter
    AllParamUpLimit = task_MAGIX.GetXMLtag(InstanceXMLFile, "uplimit")


    ## get error informations of each parameter
    AllParamErrors = task_MAGIX.GetXMLtag(InstanceXMLFile, "error")
    AllParamErrorsCopy = copy.deepcopy(AllParamErrors)
    AllParamErrors = []
    for ParamError in AllParamErrorsCopy:
        ParamErrorSplitted = ParamError.split(";")
        if (len(ParamError) < 2):
            NewContent = [0.0, 0.0, 0.0, 0.0, 0.0]
        else:
            NewContent = []
            counterElement = 0
            for element in ParamErrorSplitted:
                counterElement += 1
                if (counterElement > 1):                                                    ## ignore first entry
                    i = element.find("=")
                    NewContent.append(float(element[i + 1:].strip()))                       ## add error values
        AllParamErrors.append(NewContent)

    # Debug:
    #    print "TotalParamNumber = ", TotalParamNumber
    #    print "AllParamNames = ", AllParamNames
    #    print "AllParamValues = ", AllParamValues
    #    print "AllParamErrors = ", AllParamErrors
    #    print "AllParamLowLimit = ", AllParamLowLimit
    #    print "AllParamUpLimit = ", AllParamUpLimit


    ## check consitency of instance xml file
    if (TotalParamNumber != len(AllParamNames) or TotalParamNumber != len(AllParamValues) or TotalParamNumber != len(AllParamErrors)):
        print "\n\nError in XCLASS package, function task_myXCLASS.CreateErrorMolfitFile:"
        print "\t The number of parameters does not coincide with the number of parameter names, values, errors etc."
        print "\n\t Please check instance xml-file " + InstanceXMLFile
        print " "


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write header of error molfit file


    ## open channel for new error molfit file
    NewErrorMolfitFile = open(NewErrorMolfitFileName, 'w')                                  ## open channel for new error molfit file


    ## write first header line
    NumberOfMolecules = AllParamNames.count("Molecule_Name")
    FirstHeaderLine = "% Number of molecules = " + str(NumberOfMolecules) + "\n"
    NewErrorMolfitFile.write(FirstHeaderLine)
    AllNamesList = []


    ## determine source size is defined at least in one component
    SecondHeaderLine = ""
    SourceSizeFlag = False
    countSourceSizeFlag = AllParamNames.count("source_size")
    if (countSourceSizeFlag > 0):
        SourceSizeFlag = True
        AllNamesList.append("source size")


    ## define list of parameter names
    AllNamesList.append("T_rot")
    AllNamesList.append("N_tot")
    AllNamesList.append("V_width")
    AllNamesList.append("V_off")


    ## determine if T_doff and T_dslope are defined for at least one component
    tdFlag = False
    countnT_dOff = AllParamNames.count("T_dOff")
    if (countnT_dOff > 0):
        tdFlag = True
        AllNamesList.append("T_back")
        AllNamesList.append("T_slope")


    ## determine if nHcolumn, kappa, and beta are defined for at least one component
    nHFlag = False
    countnHcolumn = AllParamNames.count("nHcolumn")
    if (countnHcolumn > 0):
        nHFlag = True
        AllNamesList.append("nH")
        AllNamesList.append("beta")
        AllNamesList.append("kappa")


    ## determine method which was used for error estimation
    ErrorMethod = "INS"
    if (len(AllParamErrors) > 0):
        ParamError = AllParamErrors[0]
        if (len(ParamError) == 3):
            ErrorMethod = "MCMC"


    ## write second header line
    SecondHeaderLine = ""
    counter = (-1)
    for ParamName in AllNamesList:
        counter += 1
        if (counter == 0):
            SecondHeaderLine += "%         flag:"
        else:
            SecondHeaderLine += "          flag:"
        SecondHeaderLine += "     low limit:"
        SecondHeaderLine += "      up limit:"
        SecondHeaderLine += "%22s:" % ParamName
        SecondHeaderLine += "    error left:"
        SecondHeaderLine += "   error right:"
        if (ErrorMethod == "INS"):
            SecondHeaderLine += "     mean val.:"
            SecondHeaderLine += "      std-dev.:"
            SecondHeaderLine += "    log(evid.):"
    SecondHeaderLine += "   a/e-flag:\n"
    NewErrorMolfitFile.write(SecondHeaderLine)


    ## write paramters with error informations for each line to error molfit file
    InitialitionFlag = True
    for i in xrange(len(AllParamNames)):
        ParamName = AllParamNames[i]
        ParamValue = AllParamValues[i]
        ParamError = AllParamErrors[i]


        ParamLowLimit = float(AllParamLowLimit[i])
        ParamUpLimit = float(AllParamUpLimit[i])
        if (ParamName == "MolfitFileFormatVersion"):
            NewErrorMolfitFile.write(NewLine + "\n")
            FitFlag = ""
            NewLine = ""


        ## write name of molecule and number of components to file
        elif (ParamName == "Molecule_Name"):                                                ## does current name describes a molecule name?
            if (not InitialitionFlag):
                NewErrorMolfitFile.write(NewLine + "\n")
            else:
                InitialitionFlag = False
            NewLine = ParamValue
        elif (ParamName == "Number_MolLine"):                                               ## is the number of components defined?
            NewLine += "%5i " % float(ParamValue)


        ## get fit flag
        elif (ParamName.endswith("_FitFlag")):
            ParamValue = float(ParamValue)                                                  ## convert string to float
            if (ParamValue != 0.0):
                FitFlag = "y"
            else:
                FitFlag = "n"
        elif (ParamName.endswith("_flag")):
            FitFlag = ParamValue


        ## write parameters for each component with flag, lower and upper limit, parameter value
        elif (ParamName == "source_size" or ParamName == "T_rot" or ParamName == "N_tot" or ParamName == "V_width" or ParamName == "V_off" \
            or ParamName == "T_dOff" or ParamName == "T_slope" or ParamName == "nH" or ParamName == "beta" or ParamName == "kappa"):
            ParamValue = float(ParamValue)                                                  ## convert string to float
            NewLine += "%15s" % FitFlag                                                     ## add fit flag
            if (FitFlag == "n"):                                                            ## if parameter is not fitted, set limits to zero
                ParamLowLimit = 0.0
                ParamUpLimit = 0.0
            if (ParamName == "N_tot" or ParamName == "nH"):                                 ## special handling for N_tot and nH: 
                ParamValue = 10.E0**ParamValue                                              ## convert to linear value
                NewLine += "%15.3e" % (10.E0**ParamLowLimit)                                ## add lower limit to line
                NewLine += "%15.3e" % (10.E0**ParamUpLimit)                                 ## add upper limit to line
            else:
                NewLine += "%15.2f" % ParamLowLimit                                         ## add lower limit to line
                NewLine += "%15.2f" % ParamUpLimit                                          ## add upper limit to line
            NewLine += "%23.10e" % ParamValue                                               ## add parameter value to new line of molfit file


            ## add error informations of current parameter to line
            x = float(ParamError[0])                                                        ## add left error
            NewLine += "{:15.5e}".format(x)
            x = float(ParamError[1])                                                        ## add right error
            NewLine += "{:15.5e}".format(x)
            if (ErrorMethod == "INS"):
                x = float(ParamError[2])                                                    ## add mean value
                NewLine += "{:15.5e}".format(x)
                x = float(ParamError[3])                                                    ## add standard dev.
                NewLine += "{:15.5e}".format(x)
                x = float(ParamError[4])                                                    ## add log(evidence)
                NewLine += "{:15.5e}".format(x)


        ## write parameters for each component with flag, lower and upper limit, parameter value
        elif (ParamName == "CFFlag"):
            NewLine += "%12s" % ParamValue                                                  ## add parameter value to new line of molfit file


    ## write last line to new molfit file
    if (NewLine != ""):
        NewErrorMolfitFile.write(NewLine + "\n")


    ## close channel for new molfit file
    NewErrorMolfitFile.close()


    ## we've done
    if (printInfoFlag == "true"):
        print "done!"
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## converts a molfit file used by myXCLASS to a xml-file used by MAGIX.
##
def ConversionMolfit2XML(print_flag, molfitFilename, xmlFileName):
    """

input parameters:
-----------------

    - print_flag:               flag if screen output is permitted (=true) or not

    - molfitFilename:           path and file name of the molfit file

    - xmlFileName:              path and file name of the xml description of the model


output parameters:
------------------

    - ok:                       status; if everything is ok: ok = 0

    """


    ## initialize return parameter
    ok = 0


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get molfit and xml filename
    if not molfitFilename.endswith(".molfit"):
        print "\n\nError in XCLASS package, function task_myXCLASS.ConversionMolfit2XML:"
        print "\t The selected file ", molfitFilename
        print "\t is not a molfit file!"
        print "\n\t Please correct input!\n"
        ok = 1
        return

    if not xmlFileName.endswith(".xml"):
        print "\n\nError in XCLASS package, function task_myXCLASS.ConversionMolfit2XML:"
        print "\t The selected file ", xmlFileName
        print "\t is not a xml file!"
        print "\n\t Please correct input!\n"
        ok = 1
        return

    # Debug:
    # print 'molfitFilename = ',molfitFilename
    # print 'xmlFileName = ',xmlFileName


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze molfit file using myXCLASS routine
    MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = AnalyzeMolfitFile(molfitFilename)


    ## get special (secret) commands from molfit file
    MaxNumTransitionsSQL = 0
    LocalMolfitFile = open(molfitFilename)
    LocalMolfitFileContents = LocalMolfitFile.readlines()
    LocalMolfitFile.close()
    CommandWordList = ["%maxnumtransitionssql"]
    for line in LocalMolfitFileContents:
        StrippedLine = line.strip()
        i = StrippedLine.find("%")
        if (i > (-1)):
            LocalComment = StrippedLine[(i + 1):].strip()
            LocalComment = LocalComment.lower()
            for LocalCommand in CommandWordList:
                j = LocalComment.find(LocalCommand)                                         ## search for command word in molfit file
                if (j > (-1)):
                    LocalComment = LocalComment[(i + len(LocalCommand)):].strip()
                    i = LocalComment.find("=")
                    if (i > (-1)):
                        LocalComment = LocalComment[(i + 1):].strip()
                        i = LocalComment.find("%")
                        if (i > (-1)):
                            LocalComment = LocalComment[:i].strip()
                        if (LocalCommand == CommandWordList[0]):                            ## get parameter for first command word in molfit file
                            MaxNumTransitionsSQL = int(LocalComment)

    # Debug:
    # print "MaxNumTransitionsSQL = ", MaxNumTransitionsSQL


    ## determine number of parameters
    NumberParameter = 0


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write parameter.xml file
    file = open(xmlFileName, 'w')
    line="<?xml version=" + chr(34) + "1.0" + chr(34) + " ?>\n"
    file.write(line)
    file.write("<ModelParameters>\n")


    ## write number of parameters
    file.write("\n")
    file.write("\n")
    file.write("    <!-- define number of parameters -->\n")
    file.write("    <NumberParameters>" + str(NumberParameter + 5) + "</NumberParameters>\n")
    file.write("\n")
    file.write("\n")
    star = "=" * 141
    file.write("    <!-- " + star + " -->\n")
    file.write("    <!-- first input file -->\n")
    file.write("\n")
    file.write("\n")


    ## write max. number of transitions to xml-file
    file.write("    <!-- define max. number of transitions used for SQL query -->\n")
    file.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
    file.write("        <name>MaxNumTransitionsSQL</name>\n")
    file.write("        <value>" + str(MaxNumTransitionsSQL) + "</value>\n")
    file.write("        <error> </error>\n")
    file.write("        <lowlimit>0</lowlimit>\n")
    file.write("        <uplimit>1</uplimit>\n")
    file.write("    </Parameter>\n")
    file.write("\n")
    file.write("\n")


    ## write number of molecules to xml-file
    file.write("    <!-- define number of molecules -->\n")
    file.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
    file.write("        <name>Number_Molecules</name>\n")
    file.write("        <value>" + str(len(MoleculesInMolfitFile)) + "</value>\n")
    file.write("        <error> </error>\n")
    file.write("        <lowlimit>0</lowlimit>\n")
    file.write("        <uplimit>1</uplimit>\n")
    file.write("    </Parameter>\n")
    file.write("\n")
    file.write("\n")
    NumberParameter = 2

    # Debug:
    # print "AllParameters = ", AllParameters


    ## write parameter to xml-file
    MolCounter = 0
    for MoleculeIndex in xrange(len(MoleculesInMolfitFile)):                                ## loop over all molecules in the molfit file
        LocalMolfitParameters = AllParameters[MoleculeIndex]                                ## get the parameters describing the current molecule
        MolCounter += 1


        ## write name of molecule
        star = "=" * 59
        file.write("    <!-- " + star + " describe new molecule " + star + " -->\n")
        file.write("    <!-- define Molecule_Name -->\n")
        file.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
        file.write("        <name>Molecule_Name</name>\n")
        file.write("        <value>" + MoleculesInMolfitFile[MoleculeIndex] + "</value>\n")
        file.write("        <error> </error>\n")
        file.write("        <lowlimit>0</lowlimit>\n")
        file.write("        <uplimit>1</uplimit>\n")
        file.write("    </Parameter>\n")
        file.write("\n")
        file.write("\n")


        ## write index of molecule
        file.write("    <!-- define Molecule_Inex -->\n")
        file.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
        file.write("        <name>Molecule_Inex</name>\n")
        file.write("        <value>" + str(MolCounter) + "</value>\n")
        file.write("        <error> </error>\n")
        file.write("        <lowlimit>0</lowlimit>\n")
        file.write("        <uplimit>1</uplimit>\n")
        file.write("    </Parameter>\n")
        file.write("\n")
        file.write("\n")


        ## write number of components
        file.write("    <!-- define Molecule_Name -->\n")
        file.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
        file.write("        <name>Number_MolLine</name>\n")
        file.write("        <value>" + str(len(LocalMolfitParameters)) + "</value>\n")
        file.write("        <error> </error>\n")
        file.write("        <lowlimit>0</lowlimit>\n")
        file.write("        <uplimit>1</uplimit>\n")
        file.write("    </Parameter>\n")
        file.write("\n")
        file.write("\n")
        NumberParameter += 3


        ## write parameter for current molecule to xml file
        NumComp = 0
        for line in LocalMolfitParameters:                                                  ## loop over all components of the current molecule
            NumComp += 1


            ## write number of components
            if (NumComp < 10):
                j = 53
            elif (NumComp < 100):
                j = 52
            elif (NumComp < 1000):
                j = 51
            else:
                j = 50
            star = "*" * j
            file.write("    <!-- " + star + " define parameters for component " + str(NumComp) + " " + star + " -->\n")
            file.write("\n")
            file.write("\n")


            ## write each parameter for current molecule to xml file
            MolfitFileFormatVersionLocal = ""
            for param in line:                                                              ## loop over all parameters

                # Debug:
                # print "param = ", param


                ParameterName = param[0].strip()                                            ## get name of parameter
                LimitValues = param[1]                                                      ## get fit-flag of parameter
                lowlimit = param[2]                                                         ## get lower limit
                uplimit = param[3]                                                          ## get upper limit
                ParameterValue = param[4]                                                   ## get value of parameter
                LimitValuesTrueFalse = "true"

                # Debug:
                # print "param = ", param


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## define molfit file format for each line
                if (ParameterName.strip() == "MolfitFileFormatVersion"):
                    MolfitFileFormatVersionLocal = ParameterValue


                    ## write parameter indicating molfit format
                    file.write("    <!-- define MolfitFileFormatVersion -->\n")
                    file.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
                    file.write("        <name>MolfitFileFormatVersion</name>\n")
                    file.write("        <value>" + ParameterValue[:3] + "</value>\n")
                    file.write("        <error> </error>\n")
                    file.write("        <lowlimit>0</lowlimit>\n")
                    file.write("        <uplimit>1</uplimit>\n")
                    file.write("    </Parameter>\n")
                    file.write("\n")
                    file.write("\n")


                    ## write parameter indicating if a absorption component definiton contains no source size setting
                    file.write("    <!-- define EmissionFlag -->\n")
                    file.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
                    file.write("        <name>EmissionFlag</name>\n")
                    file.write("        <value>" + ParameterValue[3:4] + "</value>\n")
                    file.write("        <error> </error>\n")
                    file.write("        <lowlimit>0</lowlimit>\n")
                    file.write("        <uplimit>1</uplimit>\n")
                    file.write("    </Parameter>\n")
                    file.write("\n")
                    file.write("\n")


                    ## write parameter tbFlag indicating if background temperature etc is given in molfit file
                    file.write("    <!-- define tdFlag -->\n")
                    file.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
                    file.write("        <name>tdFlag</name>\n")
                    file.write("        <value>" + ParameterValue[6:7] + "</value>\n")
                    file.write("        <error> </error>\n")
                    file.write("        <lowlimit>0</lowlimit>\n")
                    file.write("        <uplimit>1</uplimit>\n")
                    file.write("    </Parameter>\n")
                    file.write("\n")
                    file.write("\n")


                    ## write parameter nHFlag indicating if column density etc is given in molfit file
                    file.write("    <!-- define nHFlag -->\n")
                    file.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
                    file.write("        <name>nHFlag</name>\n")
                    file.write("        <value>" + ParameterValue[9:10] + "</value>\n")
                    file.write("        <error> </error>\n")
                    file.write("        <lowlimit>0</lowlimit>\n")
                    file.write("        <uplimit>1</uplimit>\n")
                    file.write("    </Parameter>\n")
                    file.write("\n")
                    file.write("\n")
                    NumberParameter += 4
                else:


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## for column and hydrogen column density use a log scale for fit
                    if (ParameterName.strip() == "N_tot" or ParameterName.strip() == "nHcolumn"):
                        ParameterValue = float(ParameterValue)

                        # Debug:
                        # print "ParameterValue = ", ParameterValue
                        # print "lowlimit = ", lowlimit
                        # print "uplimit = ", uplimit


                        if (ParameterValue != 0.E0):
                            if (ParameterValue < 0.E0):
                                ParameterValue = abs(ParameterValue)
                            ParameterValue = math.log10(ParameterValue)                     ## convert value to log scale
                            if (lowlimit != 0.E0 or uplimit != 0.E0):                       ## continue here, only if limit unequal zero


                                ## check and convert lower limit
                                if (lowlimit == 0.E0):                                      ## is lower limit exactly zero?
                                    lowlimit = 1.e-99                                       ## set to a very small number
                                elif (lowlimit < 0.E0):                                     ## is lower limit less than zero?
                                    lowlimit = abs(lowlimit)                                ## take the absolute value
                                lowlimit = math.log10(lowlimit)                             ## convert value to log scale
                                if (lowlimit > 307.0):                                      ## check for overflow
                                    lowlimit = 307.0

                                # Debug:
                                # print ">>lowlimit = ", lowlimit


                                ## check and convert upper limit
                                if (uplimit == 0.E0):                                       ## is upper limit exactly zero?
                                    uplimit = 1.e-99                                        ## set to a very small number
                                elif (uplimit < 0.E0):                                      ## is upper limit less than zero?
                                    uplimit = abs(uplimit)                                  ## take the absolute value
                                uplimit = math.log10(uplimit)                               ## convert value to log scale
                                if (uplimit > 307.0):                                       ## check for overflow
                                    uplimit = 307.0

                                # Debug:
                                # print ">>uplimit = ", uplimit
                    ##------------------------------------------------------------------------------------------------------------------------------------


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## for column and hydrogen column density use a log scale for fit
                    if (MolfitFileFormatVersionLocal[:3] == "NEW"):
                        if (ParameterName.strip() == "N_tot_uplimit" or ParameterName.strip() == "N_tot_lowlimit"):
                            LimitValuesTrueFalse = "false"
                            ParameterValue = float(ParameterValue)
                            if (ParameterValue != 0.E0):
                                if (ParameterValue < 0.E0):
                                    ParameterValue = abs(ParameterValue)
                                ParameterValue = math.log10(ParameterValue)                 ## convert value to log scale

                        elif (ParameterName.strip() == "nHcolumn_uplimit" or ParameterName.strip() == "nHcolumn_lowlimit"):
                            LimitValuesTrueFalse = "false"
                            ParameterValue = float(ParameterValue)
                            if (ParameterValue != 0.E0):
                                if (ParameterValue < 0.E0):
                                    ParameterValue = abs(ParameterValue)
                                ParameterValue = math.log10(ParameterValue)                 ## convert value to log scale
                        elif (ParameterName.endswith("_flag") or ParameterName.endswith("_uplimit") or ParameterName.endswith("_lowlimit") \
                            or ParameterName == "CFFlag" or lowlimit == uplimit):
                            LimitValuesTrueFalse = "false"
                        elif (LimitValues == "0"):
                            LimitValuesTrueFalse = "false"


                    ## for column and hydrogen column density use a log scale for fit
                    if (MolfitFileFormatVersionLocal[:3] == "OLD"):
                        if (ParameterName.endswith("_FitFlag") or ParameterName == "CFFlag" or lowlimit == uplimit):
                            LimitValuesTrueFalse = "false"


                    ##------------------------------------------------------------------------------------------------------------------------------------


                    ## write name, value and optimization status for each parameter to xml file
                    NumberParameter += 1
                    file.write("    <!-- define " + ParameterName + " -->\n")
                    file.write("    <Parameter fit=" + chr(34) + LimitValuesTrueFalse + chr(34) + ">\n")
                    file.write("        <name>" + ParameterName + "</name>\n")
                    file.write("        <value>" + str(ParameterValue).strip() + "</value>\n")
                    file.write("        <error> </error>\n")


                    ## check order of limit values
                    if (lowlimit > uplimit):                                                ## check, if upper and lower limit values have still
                        h = uplimit                                                         ## correct order
                        uplimit = lowlimit
                        lowlimit = h


                    ## write limits for each parameter to xml-file
                    if (LimitValuesTrueFalse=="false"):
                        file.write("        <lowlimit>-1e99</lowlimit>\n")
                        file.write("        <uplimit>1e99</uplimit>\n")
                    else:
                        file.write("        <lowlimit>" + str(lowlimit) + "</lowlimit>\n")
                        file.write("        <uplimit>" + str(uplimit) + "</uplimit>\n")
                    file.write("    </Parameter>\n")
                    file.write("\n")
                    file.write("\n")


    ## write end of xml-file
    file.write("</ModelParameters>\n")


    ## close xml-file
    file.close()


    ## update number of parameters
    task_MAGIX.WriteXMLtag(xmlFileName, "NumberParameters", [str(NumberParameter)])


    ## define return parameter
    return ok
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## import the iso ratio file
##
def ImportIsoRatioFile(IsoRatioFileName, NewIsoTableFileName):
    """

input parameters:
-----------------

    - IsoRatioFileName:     path and name of iso ratio file which is imported

    - NewIsoTableFileName:  path and name of new iso ratio file


output parameters:
------------------

    - IsoRatioTable:        the final table containing molecules, isotopologues and the corresponding ratios

    - Isotopologues:        list containing the isotopologues

    - IsoMolecule:          list containing the corresponding molecules

    """

    # Debug:
    # print "IsoRatioFileName = ", IsoRatioFileName


    ## initialize ouptut parameter
    IsoRatioTable = []
    Isotopologues = []
    IsoMolecule = []


    ## read in whole iso ratio file
    if (type(IsoRatioFileName) is str):
        IsoRatioFile = open(IsoRatioFileName)
        IsoRatioFileContents = IsoRatioFile.readlines()
        IsoRatioFile.close()
    else:
        IsoRatioFileContents = IsoRatioFileName


    ## analyze iso ratio table
    TotalGlobalIsoList = []
    for line in IsoRatioFileContents:
        StrippedLine = line.strip()


        ## remove comments
        i = StrippedLine.find("%")
        if (i > (-1)):
            StrippedLine = StrippedLine[:i].strip()
        if (StrippedLine != ""):


            ## check, if global ratios are defined in the current line
            LowerLine = StrippedLine.lower()
            if (LowerLine.startswith("global")):
                StrippedLine = StrippedLine.replace("GLOBAL__", "global ")
                StrippedLine = StrippedLine[6:].strip()
                SplittedLine = StrippedLine.split()
                TotalGlobalIsoList.append(SplittedLine[0])
                StrippedLine = "GLOBAL__" + StrippedLine


            ## get isotopologues, iso master, ratio, and lower and upper limits
            SplittedLine = StrippedLine.split()
            if (len(SplittedLine) > 2):
                LocalIsotopologue = SplittedLine[0]
                LocalIsoMaster = SplittedLine[1]


                ## store local isotopologue, isomaster, iso ratio, and lower and upper limits
                Isotopologues.append(LocalIsotopologue)
                IsoMolecule.append(LocalIsoMaster)
                if (len(SplittedLine) == 3):
                    IsoRatioTable.append([LocalIsotopologue, LocalIsoMaster, SplittedLine[2]])
                if (len(SplittedLine) == 5):
                    IsoRatioTable.append([LocalIsotopologue, LocalIsoMaster, SplittedLine[2], SplittedLine[3], SplittedLine[4]])

    # Debug:
    # print "TotalGlobalIsoList = ", TotalGlobalIsoList


    ## if global defined iso ratios are used, modify limits for iso ratios, whose isotopologues contain the globally defined iso ratios
    if (TotalGlobalIsoList != []):
        NewIsoRatioTable = []
        for LocalLine in IsoRatioTable:
            if (len(LocalLine) == 3):                                                       ## current line of the iso ratio file contians no limit def.
                NewIsoRatioTable.append([LocalLine[0], LocalLine[1], LocalLine[2]])
            if (len(LocalLine) == 5):                                                       ## limits are defined
                LocalIsotopologue = LocalLine[0]
                if (LocalIsotopologue.find("GLOBAL__") > (-1)):
                    NewIsoRatioTable.append([LocalLine[0], LocalLine[1], LocalLine[2], LocalLine[3], LocalLine[4]])
                else:
                    IsoLowerLimit = LocalLine[3]
                    IsoUpperLimit = LocalLine[4]
                    #    for TotalGlobalIso in TotalGlobalIsoList:
                    #        if (LocalIsotopologue.find(TotalGlobalIso) > (-1)):                 ## the current isotopologue contains a globally defined ratio
                    #            IsoLowerLimit = "-1.e9"
                    #            IsoUpperLimit = "-1.e9"
                    #            break
                    NewIsoRatioTable.append([LocalLine[0], LocalLine[1], LocalLine[2], IsoLowerLimit, IsoUpperLimit])
        IsoRatioTable = copy.deepcopy(NewIsoRatioTable)

        # Debug:
        # print "IsoRatioTable = ", IsoRatioTable


    ## if new file name is defined
    if (NewIsoTableFileName != ""):
        NewIsoRatioFile = open(NewIsoTableFileName, 'w')
        NewLine = "%-45s %-45s %25s %25s %25s" % ("% isotopologue:", "Iso-master:", "Iso-ratio:", "Lower-limit:", "upper-limit:")
        NewIsoRatioFile.write(NewLine + "\n")
        for IsoLine in IsoRatioTable:
            if (len(IsoLine) > 3):
                NewLine = "%-45s %-45s %25s %25s %25s" % (IsoLine[0], IsoLine[1], IsoLine[2], IsoLine[3], IsoLine[4])
            else:
                NewLine = "%-45s %-45s %25s" % (IsoLine[0], IsoLine[1], IsoLine[2])
            NewIsoRatioFile.write(NewLine + "\n")
        NewIsoRatioFile.close()


    ## define output parameters
    return (IsoRatioTable, Isotopologues, IsoMolecule)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## read iso table parameters from iso table file and include parameters to xml file
##
def GetIsoTableParameters(printflag, IsoTableFileName, parameter_file):
    """

input parameters:
-----------------

    - print_flag:               flag if screen output is permitted (=true) or not

    - IsoTableFileName:         path and file name of the iso file

    - parameter_file:           path and file name of the instance XML file


output parameters:
------------------

    - ok:                       status; if everything is ok: ok = 0

    """


    ## initialize return parameter
    ok = 0                                                                                  ## reset status variable


    ## read in contents of instance XML file
    ParameterFile = open(parameter_file)
    ParameterFileContents = ParameterFile.readlines()
    ParameterFile.close()


    ## read in contents of iso file
    counterIsoRatios = 0
    GlobalIsoRatioParameterList = []
    NumberOfGlobalISORatios = 0
    if (IsoTableFileName.strip() != ""):
        NewLocalIsoRatioFileName = ""
        IsoRatioTable, Isotopologues, IsoMolecule = ImportIsoRatioFile(IsoTableFileName, NewLocalIsoRatioFileName)
        counterIsoRatios = len(Isotopologues)


        ## analyze, if global iso ratios are defined
        GlobalIsoRatioList = []
        IsotopologuesIndex = (-1)
        for IsoRatioTableLine in IsoRatioTable:
            IsotopologuesIndex += 1
            LocalIsotopologue = IsoRatioTableLine[0]
            LocalIsoRatio = IsoRatioTableLine[2]
            if (LocalIsotopologue.find("GLOBAL__") > (-1)):
                LocalIsotopologueNew = LocalIsotopologue.replace("GLOBAL__", "")
                GlobalIsoRatioList.append([LocalIsotopologueNew, IsotopologuesIndex, LocalIsoRatio])

        # Debug:
        # print "GlobalIsoRatioList = ", GlobalIsoRatioList


        ## analyze each isotopologue
        NumberOfGlobalISORatios = len(GlobalIsoRatioList)
        if (len(GlobalIsoRatioList) > 0):
            for LocalIsotopologue in Isotopologues:
                GlobalIsoRatioParameter = []
                for GlobalIsoRatio in GlobalIsoRatioList:
                    iii = LocalIsotopologue.count(GlobalIsoRatio[0])
                    if (iii > 0):
                        IsotopologuesIndex = GlobalIsoRatio[1] + 1
                        GlobalIsoRatioParameter.append([IsotopologuesIndex, iii])
                GlobalIsoRatioParameterList.append(GlobalIsoRatioParameter)

        # Debug:
        # print "\nGlobalIsoRatioParameterList = ", GlobalIsoRatioParameterList


    ## write first part of instance XML file back to instance file
    NewParameterFile = open(parameter_file, 'w')
    counterLine = 0
    for line in ParameterFileContents:
        counterLine += 1
        if (counterLine == 6):
            Newline = line.replace("<NumberParameters>", " ")
            Newline = Newline.replace("</NumberParameters>", " ")
            NumParam = int(Newline.strip())

            # Debug:
            # print "Newline = >>", Newline.strip(), "<<"
            # print "NumParam = ", NumParam

            NumParam += (counterIsoRatios * 6) + 2
            NewParameterFile.write("    <NumberParameters>" + str(NumParam) + "</NumberParameters>\n")
        else:
            if (counterLine < len(ParameterFileContents)):
                NewParameterFile.write(line)


    ## write separator
    NewParameterFile.write("\n")
    NewParameterFile.write("\n")
    star = "*" * 141
    NewParameterFile.write("    <!-- " + star + " -->\n")
    NewParameterFile.write("    <!-- third input file -->\n")
    NewParameterFile.write("\n")
    NewParameterFile.write("\n")
    NewParameterFile.write("    <!-- define number of iso ratios -->\n")
    NewParameterFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
    NewParameterFile.write("        <name>NumberOfISOLines</name>\n")
    NewParameterFile.write("        <value>" + str(counterIsoRatios) + "</value>\n")
    NewParameterFile.write("        <error> </error>\n")
    NewParameterFile.write("        <lowlimit>0</lowlimit>\n")
    NewParameterFile.write("        <uplimit>10000</uplimit>\n")
    NewParameterFile.write("    </Parameter>\n")


    ## define number of global iso ratio definitions
    NewParameterFile.write("\n")
    NewParameterFile.write("\n")
    NewParameterFile.write("    <!-- define number of global iso ratio definitions -->\n")
    NewParameterFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
    NewParameterFile.write("        <name>NumberOfGlobalISORatios</name>\n")
    NewParameterFile.write("        <value>" + str(NumberOfGlobalISORatios) + "</value>\n")
    NewParameterFile.write("        <error> </error>\n")
    NewParameterFile.write("        <lowlimit>0</lowlimit>\n")
    NewParameterFile.write("        <uplimit>10000</uplimit>\n")
    NewParameterFile.write("    </Parameter>\n")


    ## analyze iso file
    if (counterIsoRatios > 0):
        counterLines = 0
        for LineIndex in xrange(len(IsoRatioTable)):
            IsoRatioTableLine = IsoRatioTable[LineIndex]
            counterLines += 1                                                               ## increase counter for lines in iso file


            ## write comment to xml file
            NewParameterFile.write("\n")
            NewParameterFile.write("\n")
            NewParameterFile.write("    <!-- line " + str(counterLines) + ":  ratio: " + IsoRatioTableLine[0] + " / " + IsoRatioTableLine[1] + " -->\n")


            ## write name of iso molecule
            NewParameterFile.write("\n")
            NewParameterFile.write("\n")
            NewParameterFile.write("    <!-- define IsoMolecule -->\n")
            NewParameterFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
            NewParameterFile.write("        <name>IsoMolecule</name>\n")
            NewParameterFile.write("        <value>" + IsoRatioTableLine[0].strip() + "</value>\n")
            NewParameterFile.write("        <error> </error>\n")
            NewParameterFile.write("        <lowlimit>-1e9</lowlimit>\n")
            NewParameterFile.write("        <uplimit>1e9</uplimit>\n")
            NewParameterFile.write("    </Parameter>\n")


            ## write name of iso molecule
            NewParameterFile.write("\n")
            NewParameterFile.write("\n")
            NewParameterFile.write("    <!-- define IsoMasters -->\n")
            NewParameterFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
            NewParameterFile.write("        <name>IsoMasters</name>\n")
            NewParameterFile.write("        <value>" + IsoRatioTableLine[1].strip() + "</value>\n")
            NewParameterFile.write("        <error> </error>\n")
            NewParameterFile.write("        <lowlimit>-1e9</lowlimit>\n")
            NewParameterFile.write("        <uplimit>1e9</uplimit>\n")
            NewParameterFile.write("    </Parameter>\n")


            ## check limit specification
            fitFlag = "false"
            if (len(IsoRatioTableLine) == 5):
                lowerlimit = float(IsoRatioTableLine[3].strip())
                upperlimit = float(IsoRatioTableLine[4].strip())
                if (lowerlimit < upperlimit):
                    fitFlag = "true"


            ## write name of iso ratio
            NewParameterFile.write("\n")
            NewParameterFile.write("\n")
            NewParameterFile.write("    <!-- define IsoMasters -->\n")
            if (fitFlag == "true"):
                NewParameterFile.write("    <Parameter fit=" + chr(34) + "true" + chr(34) + ">\n")
            else:
                NewParameterFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
            NewParameterFile.write("        <name>IsoRatio</name>\n")
            NewParameterFile.write("        <value>" + IsoRatioTableLine[2].strip() + "</value>\n")
            NewParameterFile.write("        <error> </error>\n")
            if (fitFlag == "true"):
                NewParameterFile.write("        <lowlimit>" + IsoRatioTableLine[3].strip() + "</lowlimit>\n")
                NewParameterFile.write("        <uplimit>" + IsoRatioTableLine[4].strip() + "</uplimit>\n")
            else:
                NewParameterFile.write("        <lowlimit>-1e9</lowlimit>\n")
                NewParameterFile.write("        <uplimit>1e9</uplimit>\n")
            NewParameterFile.write("    </Parameter>\n")


            ## write value of lower limit
            NewParameterFile.write("\n")
            NewParameterFile.write("\n")
            NewParameterFile.write("    <!-- define IsoLowerLimit -->\n")
            NewParameterFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
            NewParameterFile.write("        <name>IsoLowerLimit</name>\n")
            if (fitFlag == "true"):
                NewParameterFile.write("        <value>" + IsoRatioTableLine[3].strip() + "</value>\n")
            else:
                NewParameterFile.write("        <value>-1e9</value>\n")
            NewParameterFile.write("        <error> </error>\n")
            NewParameterFile.write("        <lowlimit>-1e9</lowlimit>\n")
            NewParameterFile.write("        <uplimit>1e9</uplimit>\n")
            NewParameterFile.write("    </Parameter>\n")


            ## write value of upper limit
            NewParameterFile.write("\n")
            NewParameterFile.write("\n")
            NewParameterFile.write("    <!-- define IsoUpperLimit -->\n")
            NewParameterFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
            NewParameterFile.write("        <name>IsoUpperLimit</name>\n")
            if (fitFlag == "true"):
                NewParameterFile.write("        <value>" + IsoRatioTableLine[4].strip() + "</value>\n")
            else:
                NewParameterFile.write("        <value>-1e9</value>\n")
            NewParameterFile.write("        <error> </error>\n")
            NewParameterFile.write("        <lowlimit>-1e9</lowlimit>\n")
            NewParameterFile.write("        <uplimit>1e9</uplimit>\n")
            NewParameterFile.write("    </Parameter>\n")


            ## write parameters for global iso ratio definitions
            NewParameterFile.write("\n")
            NewParameterFile.write("\n")
            NewParameterFile.write("    <!-- define IsoGlobalParameters -->\n")
            NewParameterFile.write("    <Parameter fit=" + chr(34) + "false" + chr(34) + ">\n")
            NewParameterFile.write("        <name>IsoGlobalParameters</name>\n")
            if (GlobalIsoRatioParameterList == []):
                NewParameterFile.write("        <value> </value>\n")
            else:
                LocalGlobalIsoRatioParameter = GlobalIsoRatioParameterList[LineIndex]
                LocalGlobalIsoRatioParameterString = ""
                iii = 0
                for LocalParam in LocalGlobalIsoRatioParameter:
                    LocalParamString = str(LocalParam[0]) + ", " + str(LocalParam[1])
                    iii += 1
                    if (iii == 1):
                        LocalGlobalIsoRatioParameterString = LocalParamString
                    else:
                        LocalGlobalIsoRatioParameterString += ", " + LocalParamString
                NewParameterFile.write("        <value> " + LocalGlobalIsoRatioParameterString.strip() + "</value>\n")
            NewParameterFile.write("        <error> </error>\n")
            NewParameterFile.write("        <lowlimit>-1e9</lowlimit>\n")
            NewParameterFile.write("        <uplimit>1e9</uplimit>\n")
            NewParameterFile.write("    </Parameter>\n")


    ## write end of xml-file
    NewParameterFile.write("</ModelParameters>\n")


    ## close new parameter file
    NewParameterFile.close()


    ## define return parameter
    return ok
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
##
## The function defines the interface for the myXCLASS program
##
##
def myXCLASSCore(FreqMin, FreqMax, FreqStep, TelescopeSize, Inter_Flag, t_back_flag, tBack, tslope, nH_flag, N_H, beta_dust, kappa_1300, \
                 MolfitsFileName, iso_flag, IsoTableFileName, RestFreq, vLSR, printflag):
    """

input parameters:
-----------------

    - FreqMin:              start frequency of simulated spectrum (in K), (default: 1).

    - FreqMax:              end frequency of simulated spectrum (in K), (default: 1.e8).

    - FreqStep:             step frequency of simulated spectrum (in K), (default: 1).

    - TelescopeSize:        for single dish observations (Inter_Flag = F): TelescopeSize
                            describes the size of telescope (in m), (default: 1); for
                            interferometric observations (Inter_Flag = T): TelescopeSize
                            describes the interferometric beam FWHM size (in arcsec),
                            (default: 1).

    - Inter_Flag:           defines, if single dish ("F") or interferometric
                            observations ("T") are described, (default: "F").

    - t_back_flag:          ("T"/"F"), defines, if the user defined background 
                            temperature Tbg and temperature slope Tslope given
                            by the input parameters tBack and tslope describe the
                            continuum contribution completely (t_back_flag = "T")
                            or not (t_back_flag = "F") (default: "T").

    - tBack:                background temperature (in K), (default: 0.0).

    - tslope:               temperature slope (dimensionless), (default: 0.0).

    - nH_flag:              ("T"/"F"), defines, if column density, beta for dust
                            and kappa are given by the molfit file ("F")
                            or if nH_flag is set to "T", the following
                            three parameters define the H density, beta for
                            dust and kappa for all components (default: "T").

    - N_H:                  (has to be given only if nH_flag is set to "T")
                            Hydrogen column density (in cm^{-2}), (default: 1.e24).

    - beta_dust:            (has to be given only if nH_flag is set to "T")
                            beta for dust (dimensionless), (default: 0.1).

    - kappa_1300:           (has to be given only if nH_flag is set to "T")
                            kappa (cm^2 g^{-1}), (default: 0.01).

    - MolfitsFileName:      ABSOLUTE path and file name of the molfits file,
                            including the source_size, T_rot (rotation temperature),
                            N_tot (total column density), V_width (velocity
                            witdth), V_off (velocity offset), CFFlag
                            (core or foreground)

    - iso_flag:             use isotopologues ("T"/"F"). If iso_flag is set to "F"
                            the ratios are all set to 1 (default: "F").

    - IsoTableFileName:     (has to be given only if iso_flag is set to "T")
                            ABSOLUTE path and file name of an ASCII file including
                            the iso ratios between certain molecules. The so-called
                            "iso ratio file" defines the iso ratios between
                            molecules. The ASCII file consists of three columns,
                            where the first two columns indicates the molecules,
                            respectively. The third column defines the ratio for
                            both molecules. The columns are separated by blanks or
                            tabs. So, the names of the molecules must not contain
                            blanks. Comments have to be marked with the "%" sign,
                            i.e. all characters on the right side of a "%" sign
                            are ignored.

                            The myXCLASSFit function offers the possibility to
                            optimize the ratios between isotopologues as well. For
                            that purpose, the user has to add two additional columns
                            on the right indicating the lower and the upper limit
                            of a certain ratio, respectively.

    - RestFreq:             rest frequency in MHz (default: 0). (If this parameter
                            is set to zero, the intensity is plotted against
                            frequency (in MHz) otherwise against velocity (in km/s).

    - vLSR:                 velocity (local standard of rest) in km/s, only used,
                            if RestFreq /= 0. (velocity(Frequency = RestFreq) = vLSR)

    - printflag:            flag for printing messages to screen (True/False)


output parameters:
------------------

    - myXCLASSspectrum:     this variable contains the calculated myXCLASS spectrum

    - myXCLASSlog:          this variable contains the corrsponding log file

    - myXCLASSTrans:        list of transition energies with the corrsponding molecule names within the defined range

    - myXCLASSIntOptical    contains intensities and optical depths for each molecule and component


                            The intensities and optical depths are stored as follows:


                            myXCLASSIntOptical[0], contains all informations about the intensities

                                myXCLASSIntOptical[0][i][0]    contains the name of the ith molecule

                                myXCLASSIntOptical[0][i][1]    contains the index of the component of the ith molecule

                                myXCLASSIntOptical[0][i][2]    contains the intensities of the ith molecule as a 2D python list

                                myXCLASSIntOptical[0][i][3]    contains the integrated intensities of the ith molecule


                            myXCLASSIntOptical[1], contains all informations about the optical depths.

                                myXCLASSIntOptical[1][i][0]    contains the name of the ith molecule

                                myXCLASSIntOptical[1][i][1]    contains the index of the component of the ith molecule

                                myXCLASSIntOptical[1][i][2]    contains the optical depath of the ith molecule as a 2D python list

    - myXCLASSJobDir:       absolute path of the myXCLASS job directory created for the current run


    """


    ## get current working directory
    CurrentDir = os.getcwd() + "/"


    ######################################################################################################################################################
    ## DO NOT EDIT OR REMOVE THE FOLLOWING LINE !!!!
    myXCLASSrootDir = "/home/moeller/ALMA/CASA/myXCLASS-CASA-Interface/No-NR-version/Linux/XCLASS-Interface/programs/myXCLASS/"
    ######################################################################################################################################################

    myXCLASSsrcDir = myXCLASSrootDir + "src/"
    myXCLASSexe = "myNewXCLASS.exe"


    ##====================================================================================================================================================
    ## create run directory for the current function call
    LocalPrintFlag = printflag
    myXCLASStempDir = task_MAGIX.CreateRunDirectory("myXCLASS", myXCLASSrootDir, LocalPrintFlag)

    # Debug:
    # print "command_string = ", command_string


    ##====================================================================================================================================================
    ## copy all required files to temp directory
    command_string = "cp " + myXCLASSsrcDir + myXCLASSexe + " " + myXCLASStempDir           ## copy myXCLASS exe file to temp directory
    os.system(command_string)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## molfits file:


    ## check path and file name of the molfits file
    if (MolfitsFileName == None):
        print "\n\nError in XCLASS package, function task_myXCLASS.myXCLASS:"
        print "  No path and file name for molfits file is defined!"
        print "\n  Please enter the path and file name of the molfits file and redo myXCLASS call!"
        return ([], [], [], [], "")
    MolfitsFileName = MolfitsFileName.strip()
    if (MolfitsFileName[0] != "/"):
        MolfitsFileName = CurrentDir + MolfitsFileName
    if not(os.path.exists(MolfitsFileName) or os.path.isfile(MolfitsFileName)):
        print "\n\nError in XCLASS package, function task_myXCLASS.myXCLASS:"
        print "  The given path and file name of the molfits file " + chr(34) + MolfitsFileName + chr(34) + " does not exsist!"
        print "\n  Please enter an exsisting path and file name of the molfits file and redo myXCLASS call!"
        return ([], [], [], [], "")

    # Debug:
    # print "MolfitsFileName = ", MolfitsFileName


    ## copy molfits file to temp directory
    if (printflag):
        print "Copy molfits file to temp directory " + myXCLASStempDir + " .. ",
    command_string = "cp " + MolfitsFileName + " " + myXCLASStempDir
    os.system(command_string)


    ## get name of molfits file
    i = MolfitsFileName.rfind("/")
    if (i > 0):
        MolfitsFileName = MolfitsFileName[i + 1:]
    MolfitsFileName = myXCLASStempDir + MolfitsFileName

    # Debug:
    # print ' '
    # print 'MolfitsFileName = ', MolfitsFileName


    ## analyze molfit file using myXCLASS routine
    MolfitMoleculeNames, AllParameters, MolfitFileForEachMolecule = AnalyzeMolfitFile(MolfitsFileName)
    SortedListOfMolfitMolecules = sorted(set(MolfitMoleculeNames))


    ## check molecules in database
    LocalPrintFlag = False
    FreqMinList = []
    FreqMaxList = []
    dbList = []
    MoleculeNames, dbFile = task_myXCLASSFit.CheckMoleculesInDatabase(MolfitMoleculeNames, LocalPrintFlag, FreqMinList, FreqMaxList, dbList)

    # Debug:
    # print "dbFile = ", dbFile


    ## print error message, if both lists are not identical
    SortedListOfFoundMolecules = sorted(set(MoleculeNames))
    if (len(SortedListOfMolfitMolecules) != len(SortedListOfFoundMolecules)):
        print "\n\nError in XCLASS package, function task_myXCLASS.myXCLASS:"
        print "\n\t There are some molecules given in the molfit file, which are not included in the database."
        print "\n\t Molecule names in molfit file:    ", SortedListOfMolfitMolecules
        print "\t Molecule names in database table: ", SortedListOfFoundMolecules
        print "\n\n\t Please correct the molfit file and restart function!"
        print "\n\n"
        return ([], [], [], [], "")
    if (printflag):
        print "done!"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## IsoTableFileName file:
    if (str(iso_flag).lower() == "true"):


        ## check path and file name of the iso file
        if (IsoTableFileName == None):
            print "\n\nError in XCLASS package, function task_myXCLASS.myXCLASS:"
            print "  No path and file name for iso file is defined!"
            print "\n  Please enter the path and file name of the iso file and redo myXCLASS call!"
            return ([], [], [], [], "")
        IsoTableFileName = IsoTableFileName.strip()
        if (IsoTableFileName[0] != "/"):
            IsoTableFileName = CurrentDir + IsoTableFileName
        if not(os.path.exists(IsoTableFileName) or os.path.isfile(IsoTableFileName)):
            print "\n\nError in XCLASS package, function task_myXCLASS.myXCLASS:"
            print "  The given path and file name of the iso file " + chr(34) + IsoTableFileName + chr(34) + " does not exsist!"
            print "\n  Please enter an exsisting path and file name of the iso file and redo myXCLASS call!"
            return ([], [], [], [], "")

        # Debug:
        # print "IsoTableFileName = ", IsoTableFileName


        ## get name of iso file
        OldIsoTableFileName = IsoTableFileName
        i = IsoTableFileName.rfind("/")
        if (i > 0):
            IsoTableFileName = IsoTableFileName[i + 1:]
        IsoTableFileName = myXCLASStempDir + IsoTableFileName


        ## analyze iso ratio file
        IsoRatioTable, Isotopologues, IsoMolecule = ImportIsoRatioFile(OldIsoTableFileName, IsoTableFileName)


        ## check, if molecules described in iso ratio file are contained in used database file
        LocalPrintFlag = False
        FreqMinList = []
        FreqMaxList = []
        dbList = []
        NotFoundIsotopologuesList = []
        for LineIndex in xrange(len(Isotopologues)):
            LocalIsoMolecule = IsoMolecule[LineIndex]
            if (LocalIsoMolecule in SortedListOfFoundMolecules):


                ## check, if molecules defined in the molfit file are included in the current database
                LocalIsotopologues = Isotopologues[LineIndex]
                if (LocalIsotopologues.find("GLOBAL__") == (-1)):
                    FoundIsoRatioMolecules, dbFile = task_myXCLASSFit.CheckMoleculesInDatabase([LocalIsotopologues], LocalPrintFlag, FreqMinList, \
                                                                                               FreqMaxList, dbList)
                    if (FoundIsoRatioMolecules == []):
                        NotFoundIsotopologuesList.append(LocalIsotopologues)


        ## check, if isotopologues are defined in iso ratio file which are not contained in used database file
        if (NotFoundIsotopologuesList != []):
            print "\n\nError in XCLASS package, function task_myXCLASS.myXCLASS:"
            print "\n\t There are some molecules given in the iso ratio file, which are not included in the database."
            print "\n\t Not included molecule(s):    ", NotFoundIsotopologuesList
            print "\n\n\t Please correct the iso-ratio file and restart function!"
            print "\n\n"
            return ([], [], [], [], "")

        # Debug:
        # print ' '
        # print 'IsoTableFileName = ', IsoTableFileName
        # print "IsoMolecule = ", IsoMolecule


    ##====================================================================================================================================================
    ## convert molfit and iso file to MAGIX xml instance file


    ## extend sys.path
    MAGIXrootDir = myXCLASSrootDir + "../MAGIX/"
    MAGIXrootDir = os.path.normpath(MAGIXrootDir) + "/"
    task_MAGIX.SetMAGIXEnvironment(MAGIXrootDir)

    # Debug:
    #    print "\n\nMAGIXrootDir = ", MAGIXrootDir
    #    print "\n\nsys.path = ", sys.path


    ## convert molfit to xml file
    LocalPrintflag = "false"
    xmlFileName = myXCLASStempDir + "parameter.xml"
    ok = ConversionMolfit2XML(LocalPrintflag, MolfitsFileName, xmlFileName)
    if (str(iso_flag).lower() == "true"):
        ok = GetIsoTableParameters(LocalPrintflag, IsoTableFileName, xmlFileName)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## convert column and hydrogen column density (if given for each component) to log scale.
    # LogLinearFlag = "log"
    # ConvertNtotToLogScale(MolfitsFileName, LogLinearFlag)


    ##====================================================================================================================================================
    ## construct myXCLASS call
    ## sh myNewXCLASS.exe FreqMin, FreqMax, FreqStep, TelescopeSize, Inter_Flag,  t_back_flag, tBack, tslope, nH_flag, N_H, beta_dust, kappa_1300, 
    ##                    MolfitsFileName, iso_flag, IsoTableFileName
    ## FreqMin:            start frequency of simulated spectrum
    ## FreqMax:            end frequency of simulated spectrum
    ## FreqStep:           step frequency of simulated spectrum
    ## TelescopeSize:      size of telescope
    ## Inter_Flag:         defines, if single dish ("F") or interferometric
    ##                     observations ("T") are described, (default: "F").
    ## t_back_flag:        ("T"/"F"), defines, if the user defined background
    ##                     temperature Tbg and temperature slope Tslope given
    ##                     by the input parameters tBack and tslope describe the
    ##                     continuum contribution completely (t_back_flag = "T")
    ##                     or not (t_back_flag = "F") (default: "T").
    ## tBack:              (has to be given only if t_back_flag is set to "T")
    ##                     background temperature
    ## tslope:             (has to be given only if t_back_flag is set to "T")
    ##                     temperature slope
    ## nH_flag:            ("T"/"F"), defines, if column density, beta for dust
    ##                     and kappa are given by the molfit file ("F")
    ##                     or if nH_flag is set to "T", the following
    ##                     three parameters define the H density, beta for
    ##                     dust and kappa for all components
    ## N_H:                (has to be given only if nH_flag is set to "T")
    ##                     H density
    ## beta_dust:          (has to be given only if nH_flag is set to "T")
    ##                     beta for dust
    ## kappa_1300:         (has to be given only if nH_flag is set to "T")
    ##                     kappa
    ## MolfitsFileName:    path and file name of the molfits file, including
    ##                     the source_size, T_rot (rotation temperature),
    ##                     N_tot (total column density), V_width (velocity
    ##                     witdth), V_off (velocity offset), CFFlag
    ##                     (core or foreground)
    ## iso_flag:           use isotopologues ("T"/"F"). If iso_flag is set to "F"
    ##                     the ratios are all set to 1.
    ## IsoTableFileName:   (has to be given only if iso_flag is set to "T")
    ##                     path and file name of a ASCII file including the
    ##                     iso ratios between certain molecules:
    ##                     First column (A25) and second column (A33) indicates
    ##                     the molecules, respectively. The thrid column
    ##                     defines the ratio for both molecules.
    ##
    ##
    ##
    myXCLASScommand = "cd " + myXCLASStempDir + "; ./" + myXCLASSexe + " " + str(FreqMin) + " " + str(FreqMax) + " " + str(FreqStep)
    myXCLASScommand += " " + str(TelescopeSize) + " " + str(Inter_Flag) + " " + str(t_back_flag) + " " + str(tBack) + " " + str(tslope)


    ## check for nH_flag
    myXCLASScommand += " " + str(nH_flag)
    if (str(nH_flag).lower() == "true"):
        myXCLASScommand += " " + str(N_H) + " " + str(beta_dust) + " " + str(kappa_1300)


    ## add global v_lsr
    myXCLASScommand += " " + str(vLSR)


    ## check for iso_flag
    myXCLASScommand += " " + xmlFileName.strip() + " " + str(iso_flag)


    ## add rest of parameters to command string
    i = (-1)                                                                                ## define negative number of processors to get output files
    myXCLASScommand += " " + str(i) + " " + dbFile.strip()
    if (not printflag):
        myXCLASScommand += " > screen.out"
    myXCLASScommand += "; rm -rf " + myXCLASSexe + "; cd ../../../"

    # Debug:
    # print "myXCLASScommand = ", myXCLASScommand


    ##====================================================================================================================================================
    ## start myXCLASS
    os.system(myXCLASScommand)


    ##====================================================================================================================================================
    ## read in calculated spectrum
    myXCLASSoutputFilename = myXCLASStempDir + "xclass_spectrum_output.dat"

    # Debug:
    # print "myXCLASSoutputFilename = ", myXCLASSoutputFilename

    if (printflag):
        print "\nImport calculated myXCLASS spectrum ..",
    try:
        myXCLASSspectrum = numpy.loadtxt(myXCLASSoutputFilename, skiprows = 0)
        if (printflag):
            print "done!"
    except IOError, err:
        myXCLASSspectrum = []
        print "\n\nError in XCLASS package, function task_myXCLASS.myXCLASS:"
        print "\t An error occurred while importing the modeled spectrum!"
        print "\n\n\n\n"
        return ([], [], [], [], "")



    ## add velocity axis if RestFreq != 0.0
    if (RestFreq != 0.0):
        if (printflag):
            print "Add velocity axis ..",
        FreqData = myXCLASSspectrum[:, 0]
        VelocityData = ChangeToVelocitiy(FreqData, RestFreq, vLSR)


        ## add velocity axis to ASCIIdata array
        NewmyXCLASSspectrum = numpy.zeros((len(myXCLASSspectrum), 3), dtype=numpy.float32)
        NewmyXCLASSspectrum[:, 0] = myXCLASSspectrum[:, 0]
        NewmyXCLASSspectrum[:, 1] = VelocityData[:]
        NewmyXCLASSspectrum[:, 2] = myXCLASSspectrum[:, 1]
        myXCLASSspectrum = NewmyXCLASSspectrum


        ## print some informations to screen
        if (printflag):
            print "done!"


    ##====================================================================================================================================================
    ## read in log file
    myXCLASSlogFilename = myXCLASStempDir + "xclass_spectrum.log"

    # Debug:
    # print "myXCLASSlogFilename = ", myXCLASSlogFilename

    try:
        f = open(myXCLASSlogFilename)                                                       ## open log file
        myXCLASSlog = f.readlines()                                                         ## read contents
        f.close()                                                                           ## close file
    except IOError, err:
        myXCLASSlog = []


    ##====================================================================================================================================================
    ## read in transition energies from file
    myXCLASSTransFilename = myXCLASStempDir + "transition_energies.dat"

    # Debug:
    # print "myXCLASSTransFilename = ", myXCLASSTransFilename

    try:
        f = open(myXCLASSTransFilename)
        contents = f.readlines()
        f.close()                                                                           ## close file
        myXCLASSTrans = []
        for line in contents:
            StrippedLine = line.strip()
            SplittedLine = StrippedLine.split()
            myXCLASSTrans.append(SplittedLine)
    except IOError, err:
        myXCLASSTrans = []


    ##====================================================================================================================================================
    ## read in intensity from output files
    myXCLASSIntOptical = []
    if (printflag):
        print "\nRead in intensities for each molecule and for each component ..",
    sys.stdout.flush()
    listing = os.listdir(myXCLASStempDir)


    ## get all optical depths files
    IntensityFiles = []
    for files in listing:
        if (files.startswith("intensity__")):
            IntensityFiles.append(files)
    IntensityFiles = sorted(IntensityFiles)


    ## read in model function values
    IntensitiesPerComponent = []
    for files in IntensityFiles:


        ## read header informations
        f = open(myXCLASStempDir + files, 'r')
        line = f.readline()
        i = line.find("=")
        molname = line[i + 1:].strip()
        line = f.readline()
        i = line.find("=")
        ncomp = int(line[i + 1:].strip())
        f.close()

        # Debug:
        # print "\nmolname = ", molname
        # print "ncomp = ", ncomp


        ## read in tntensity from file
        CurrentIntensity = numpy.loadtxt(myXCLASStempDir + files, skiprows = 4)
        if (RestFreq != 0.0):                                                               ## add velocity axis if RestFreq != 0.0
            FreqData = CurrentIntensity[:, 0]
            VelocityData = ChangeToVelocitiy(FreqData, RestFreq, vLSR)


            ## add velocity axis to ASCIIdata array
            NewmCurrentIntensity = numpy.zeros((len(CurrentIntensity), 3), dtype=numpy.float32)
            NewmCurrentIntensity[:, 0] = CurrentIntensity[:, 0]
            NewmCurrentIntensity[:, 1] = VelocityData[:]
            NewmCurrentIntensity[:, 2] = CurrentIntensity[:, 1]
            CurrentIntensity = NewmCurrentIntensity


        ## integrate intensity
        IntSpectra = scipy.integrate.simps(CurrentIntensity[:, 1], CurrentIntensity[:, 0])


        ## add to output array
        IntensitiesPerComponent.append([molname, ncomp, CurrentIntensity, IntSpectra])


    ## sort output array
    # IntensitiesPerComponent.sort()
    myXCLASSIntOptical.append(IntensitiesPerComponent)


    ## we've done
    if (printflag):
        print "done!"


    ##====================================================================================================================================================
    ## read in optical depth from output files
    if (printflag):
        print "\nRead in optical depths for each molecule and for each component ..",
    sys.stdout.flush()


    ## get all optical depths files
    OpticalDepthsFiles = []
    for files in listing:
        if (files.startswith("optical_depth__")):
            OpticalDepthsFiles.append(files)
    OpticalDepthsFiles = sorted(OpticalDepthsFiles)


    ## read in optical depth from output files
    OpticalDepthsPerComponent = []
    for files in OpticalDepthsFiles:


        ## read header informations
        f = open(myXCLASStempDir + files, 'r')
        line = f.readline()
        i = line.find("=")
        molname = line[i + 1:].strip()
        line = f.readline()
        i = line.find("=")
        ncomp = int(line[i + 1:].strip())
        f.close()

        # Debug:
        # print "\nmolname = ", molname
        # print "ncomp = ", ncomp


        ## read in tntensity
        CurrentOpticalDepth = numpy.loadtxt(myXCLASStempDir + files, skiprows = 4)
        if (RestFreq != 0.0):                                                               ## add velocity axis if RestFreq != 0.0
            FreqData = CurrentOpticalDepth[:, 0]
            VelocityData = ChangeToVelocitiy(FreqData, RestFreq, vLSR)


            ## add velocity axis to ASCIIdata array
            NewCurrentOpticalDepth = numpy.zeros((len(CurrentOpticalDepth), 3), dtype=numpy.float32)
            NewCurrentOpticalDepth[:, 0] = CurrentOpticalDepth[:, 0]
            NewCurrentOpticalDepth[:, 1] = VelocityData[:]
            NewCurrentOpticalDepth[:, 2] = CurrentOpticalDepth[:, 1]
            CurrentOpticalDepth = NewCurrentOpticalDepth


        ## add to output array
        OpticalDepthsPerComponent.append([molname, ncomp, CurrentOpticalDepth])


    ## sort output array
    # OpticalDepthsPerComponent.sort()
    myXCLASSIntOptical.append(OpticalDepthsPerComponent)


    ## we've done
    if (printflag):
        print "done!"


    ##====================================================================================================================================================
    ## print a message to screen
    if (printflag):
        print " "
        print "All files of the current myXCLASS run are stored in the job directory " + myXCLASStempDir + "!"
        print " "
        print " "


    ##====================================================================================================================================================
    ## define return variables
    myXCLASSJobDir = myXCLASStempDir
    return (myXCLASSspectrum, myXCLASSlog, myXCLASSTrans, myXCLASSIntOptical, myXCLASSJobDir)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
##
## The function defines the interface for the myXCLASS program
##
##
def myXCLASS(FreqMin, FreqMax, FreqStep, TelescopeSize, Inter_Flag, t_back_flag, tBack, tslope, nH_flag, N_H, beta_dust, kappa_1300, MolfitsFileName, \
             iso_flag, IsoTableFileName, RestFreq, vLSR):
    """

The files produced by a myXCLASS function call are copied to a so-called "job-directory" located in the myXCLASS working directory
"path-of-myXCLASS-CASA-Interface/run/myXCLASS/"! The name of a job directory for a myXCLASS run is made up of four components: The first part
consists of the phrase "job_" whereas the second and third part describes the date (day, month, year) and the time stamp (hours, minutes,
seconds) of the function execution, respectively. The last part indicates a so-called job ID which is composed of the so-called PID followed
by a four digit random integer number to create a really unambiguous job number, e.g.
"path-of-myXCLASS-CASA-Interface/run/myXCLASS/job__25-07-2013__12-02-03__189644932/"


input parameters:
-----------------

    - FreqMin:              start frequency of simulated spectrum (in K), (default: 1).

    - FreqMax:              end frequency of simulated spectrum (in K), (default: 1.e8).

    - FreqStep:             step frequency of simulated spectrum (in K), (default: 1).

    - vLSR:                 local standard of rest velocity (in km/s), (default 0).

    - TelescopeSize:        for single dish observations (Inter_Flag = F): TelescopeSize
                            describes the size of telescope (in m), (default: 1); for
                            interferometric observations (Inter_Flag = T): TelescopeSize
                            describes the interferometric beam FWHM size (in arcsec),
                            (default: 1).

    - Inter_Flag:           defines, if single dish ("F") or interferometric
                            observations ("T") are described, (default: "F").

    - t_back_flag:          ("T"/"F"), defines, if the user defined background 
                            temperature Tbg and temperature slope Tslope given
                            by the input parameters tBack and tslope describe the
                            continuum contribution completely (t_back_flag = "T")
                            or not (t_back_flag = "F") (default: "T").

    - tBack:                background temperature (in K), (default: 0.0).

    - tslope:               temperature slope (dimensionless), (default: 0.0).

    - nH_flag:              ("T"/"F"), defines, if column density, beta for dust
                            and kappa are given by the molfit file ("F")
                            or if nH_flag is set to "T", the following
                            three parameters define the H density, beta for
                            dust and kappa for all components (default: "T").

    - N_H:                  (has to be given only if nH_flag is set to "T")
                            Hydrogen column density (in cm^{-2}), (default: 1.e24).

    - beta_dust:            (has to be given only if nH_flag is set to "T")
                            beta for dust (dimensionless), (default: 0.1).

    - kappa_1300:           (has to be given only if nH_flag is set to "T")
                            kappa (cm^2 g^{-1}), (default: 0.01).

    - MolfitsFileName:      ABSOLUTE path and file name of the molfits file,
                            including the source_size, T_rot (rotation temperature),
                            N_tot (total column density), V_width (velocity
                            witdth), V_off (velocity offset), CFFlag
                            (core or foreground)

    - iso_flag:             use isotopologues ("T"/"F"). If iso_flag is set to "F"
                            the ratios are all set to 1 (default: "F").

    - IsoTableFileName:     (has to be given only if iso_flag is set to "T")
                            ABSOLUTE path and file name of an ASCII file including
                            the iso ratios between certain molecules. The so-called
                            "iso ratio file" defines the iso ratios between
                            molecules. The ASCII file consists of three columns,
                            where the first two columns indicates the molecules,
                            respectively. The third column defines the ratio for
                            both molecules. The columns are separated by blanks or
                            tabs. So, the names of the molecules must not contain
                            blanks. Comments have to be marked with the "%" sign,
                            i.e. all characters on the right side of a "%" sign
                            are ignored.

                            The myXCLASSFit function offers the possibility to
                            optimize the ratios between isotopologues as well. For
                            that purpose, the user has to add two additional columns
                            on the right indicating the lower and the upper limit
                            of a certain ratio, respectively.

    - RestFreq:             rest frequency in MHz (default: 0). (If this parameter
                            is set to zero, the intensity is plotted against
                            frequency (in MHz) otherwise against velocity (in km/s).


output parameters:
------------------

    - myXCLASSspectrum:     contains the calculated myXCLASS spectrum

    - myXCLASSlog:          contains the corrsponding log file

    - myXCLASSTrans:        python list containing the transition frequencies (in MHz) from the last myXCLASS run, the Doppler-shifted transition
                            frequencies (in MHz), the corresponding intensities (in K), the energy of the lower level (in K), the upper state
                            degeneracy, the Einstein A coefficient (in s^{-1}), and the molecule names within the defined range

    - myXCLASSIntOptical    contains intensities and optical depths for each molecule and component


                            The intensities and optical depths are stored as follows:


                            myXCLASSIntOptical[0], contains all informations about the intensities

                                myXCLASSIntOptical[0][i][0]    contains the name of the ith molecule

                                myXCLASSIntOptical[0][i][1]    contains the index of the component of the ith molecule

                                myXCLASSIntOptical[0][i][2]    contains the intensities of the ith molecule as a 2D python list

                                myXCLASSIntOptical[0][i][3]    contains the integrated intensities of the ith molecule


                            myXCLASSIntOptical[1], contains all informations about the optical depths.

                                myXCLASSIntOptical[1][i][0]    contains the name of the ith molecule

                                myXCLASSIntOptical[1][i][1]    contains the index of the component of the ith molecule

                                myXCLASSIntOptical[1][i][2]    contains the optical depath of the ith molecule as a 2D python list

    - myXCLASSJobDir:       absolute path of the myXCLASS job directory created for the current run



Note, the user is free to define a different name for the output parameter.


myXCLASS:
---------

The function models a spectrum by solving the radiative transfer equation for an isothermal object in one dimension,
called detection equation (Stahler &amp; Palla 2005). Furthermore, non-extended sources and dust attenuation
is considered. 

Overall, there are five input parameters for one component of a molecule:

$\\theta_{m,c}$:      source size in arcsec (only for core components)
$T_{ex}$:           excitation temperature in K
$N_{tot}$:          total column density in cm$^{-2}$
$\\Delta \\nu$:       velocity width (FWHM) in km s$^{-1}$
$\\nu_{LSR}$:        source velocity in km s$^{-1}$.

Components can be identified as spatially distinct sources such as clumps, hot dense cores, colder envelopes
or outflows, and can usually be distinguished by different radial velocities. They do not interact with each
other radiatively but superimpose in the model. However, for absorption lines the emission by other
components is considered by first calculating the emission lines and then use this emission as new continuum
for absorption lines. By fitting all species and their components at once, line blending and optical depth
effects are taken into account. With the LTE condition, every population of a level i is known and given by the
Boltzmann distribution $n_i = \\frac{N}{Q} g_i e^{-E_i/(kT)}$. The modeling can be done together with
isotopologues (and higher vibrational states) of a molecule assuming a conversion factor stored in a separate
database. All parameters are expected to be the same except the abundance.


The molfit file:

Lines starting with the "%" character are ignored by the myXCLASS program. The first line defines
the name of the molecule followed by the number of components. Here, we consider HDO-18, v=0 with three components
which are defined in the next three lines. Each component is defined by the source size (in "),
("s_size"), the rotation temperature (in K), ("T_rot"), the column density (in cm^{-2}),
("N_tot"), the velocity width (in km s^{-1}), ("V_width"), the velocity offset (in km s^{-1}),
("V_off") and the flag ("AFlag") indicating if the current component is considered for
core "c" or foreground "f".

Example of a molfit file:

% Number of molecules =     2
HDO-18;v=0;    3
% s_size:  T_rot:   N_tot:  V_width:   V_off:  AFlag:
 48.47039  50.000  3.9E+16  2.862117  -2.5643  e
 40.10603  56.531  2.3E+18  4.210278  -7.3161  e
 29.09758  68.441  2.0E+17  4.024519   0.2130  e
           6.0857  2.2E+17  1.000000  -1.9000  a
HN-15-C;v=0;   1
% s_size:  T_rot:   N_tot:  V_width:   V_off:  AFlag:
 32.43532  43.234  2.1E+16  1.354217  -9.5211  e


In order to define the dust temperature for each component the molfit file has to contain two additional
columns: the dust offset temperature ("T_dOff") (in K), and the dust temperature slope
(dimensionless), ("T_dSlope") for each component for all molecules. If these parameters are
not defined for a component, myXCLASS sets the dust temperature equal to the excitation temperture
of the corresponding component.

Example of an extended molfit file, if "t_back_flag" is set to "F" (false):

% Number of molecules =     2
HDO-18;v=0;    3
% s_size:  T_rot:   N_tot:  V_width:   V_off:  T_dOff:  T_dSlope:  AFlag:
 48.47039  50.000  3.9E+16  2.862117  -2.5643      3.0        0.0  e
 40.10603  56.531  2.3E+18  4.210278  -7.3161      3.0        0.0  e
 29.09758  68.441  2.0E+17  4.024519   0.2130      2.5        1.0  e
           6.0857  2.2E+17  1.000000  -1.9000      3.0        0.0  a
HN-15-C;v=0;   1
% s_size:  T_rot:   N_tot:  V_width:   V_off:  T_dOff:  T_dslope:  AFlag:
 32.43532  43.234  2.1E+16  1.354217  -9.5211      4.0        0.1  e


If "nH_flag" is set to "F" (false) myXCLASS expects three additional columns within the
molfit file: the hydrogen column density (cm^{-2}), ("n_H"), beta (dimensionless), ("beta")
for dust and kappa (dimensionless), ("kappa").

Example of an extended molfit file, if "nH_flag" is set to "F" (false):

% Number of molecules =     2
HDO-18;v=0;    3
% s_size:  T_rot:   N_tot:  V_width:   V_off:  n_H:  beta:  kappa:  AFlag:
 48.47039  50.000  3.9E+16  2.862117  -2.5643  3e24    2.0  0.3095  e
 40.10603  56.531  2.3E+18  4.210278  -7.3161  3e24    2.0  0.0005  e
 29.09758  68.441  2.0E+17  4.024519   0.2130  3e24    2.0  0.0015  e
500.00000  6.0857  2.2E+17  1.000000  -1.9000  3e24    2.0  0.0005  e
HN-15-C;v=0;   1
% s_size:  T_rot:   N_tot:  V_width:   V_off:  n_H:  beta:  kappa:  AFlag:
 32.43532  43.234  2.1E+16  1.354217  -9.5211  3e24    2.0  0.3095  e


If the dust parameters "T_dOff", "T_dOff", "n_H", "beta",
and "kappa" has to be defined for each component, the columns defining the dust temperature
"T_dOff", "T_dOff", have to be given before the hydrogen column density,
beta and kappa.


The iso ratio file:

The so-called "iso ratio file" defines the iso ratios between molecules. The ASCII file consists
of three columns, where the first two columns indicates the molecules, respectively. The third column defines
the ratio for both molecules. The columns are separated by blanks or tabs. So, the names of the molecules
must not contain blanks.

Example for an iso ratio file:

% isotopologue:   molecule:   ratio:
HO-18;v=0;        OH;v=0;     500
HDO-18;v=0;       HDO;v=0;    500
HCN;v2=1;         HCN;v=0;    1
HNC-13;v=0;       HNC;v=0;    60
HN-15-C;v=0;      HNC;v=0;    300

Here, the first line sets the iso ratio between the "HO-18, v=0" and the "OH, v=0"
molecule to "500".


The myXCLASSFit function offers the possibility to optimize the ratios between
isotopologues as well. For that purpose, the user has to add two additional
columns on the right indicating the lower and the upper limit of a certain
ratio.

Example for an iso ratio file where the ratios are optimized by the myXCLASSFit function:

% isotopologue:   molecule:   ratio:   lower:   upper:
HO-18;v=0;        OH;v=0;     500      0.1      1000
HDO-18;v=0;       HDO;v=0;    500      0.1      1000
HCN;v2=1;         HCN;v=0;    1        0.1      1000
HNC-13;v=0;       HNC;v=0;    60       0.1      1000
HN-15-C;v=0;      HNC;v=0;    300      0.1      1000

If the lower and upper limit are equal or if the lower limit is higher than
the upper limit, the ratio is kept constant and is not optimized by the
myXCLASSFit function.


Example:
--------

FreqMin = 578000.5
FreqMax = 582000.5
FreqStep = 5.0000000000E-01
vLSR = 0.0
TelescopeSize = 3.5
Inter_Flag = F
t_back_flag = T
tBack = 1.1
tslope = 0.0000000000E+00
nH_flag = T
N_H = 3.0000000000E+24
beta_dust = 2.0
kappa_1300 = 0.42
MolfitsFileName = "demo/myXCLASS/CH3OH__pure.molfit"
iso_flag = T
IsoTableFileName = "demo/myXCLASS/iso_names.txt"
RestFreq = 0.0
modeldata, log, TransEnergies, IntOptical, jobDir = myXCLASS()

FileName = "demo/myXCLASS/band1b.dat"
NumHeaderLines = 0
expdata = LoadASCIIFile()

MinIntensity = 0.0
xLowerLimit = 578000.5
xUpperLimit = 582000.5
yLowerLimit = 0.8
yUpperLimit = 2.5
PlotTitle = "Example for myXCLASSPlot function"
LegendFlag = T
SaveFigureFile = ""
myXCLASSPlot()

Please note, the bottom half in the example reads the experimental data from file and plots
the calculated spectrum using the myXCLASSPlot function. Note, the user is free to define
different names for the output parameters. In the example describe above, we use the
name "modeldata" for output parameter myXCLASSspectrum, "log" for output parameter myXCLASSlog,
"TransEnergies" for output parameter myXCLASSTrans, "IntOptical" for output parameter
myXCLASSIntOptical, and "jobDir" for output parameter myXCLASSJobDir.

    """


    ## call myXCLASS core function
    printflag = True
    myXCLASSspectrum, myXCLASSlog, myXCLASSTrans, myXCLASSIntOptical, myXCLASSJobDir = myXCLASSCore(FreqMin, FreqMax, FreqStep, TelescopeSize, \
                                                                                                    Inter_Flag, t_back_flag, tBack, tslope, nH_flag, \
                                                                                                    N_H, beta_dust, kappa_1300, MolfitsFileName, \
                                                                                                    iso_flag, IsoTableFileName, RestFreq, vLSR, printflag)

    ##====================================================================================================================================================
    ## define return variables
    return (myXCLASSspectrum, myXCLASSlog, myXCLASSTrans, myXCLASSIntOptical, myXCLASSJobDir)
##--------------------------------------------------------------------------------------------------------------------------------------------------------

