#!/usr/bin/python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module function updates the myXCLASS sqlite3 database
##  Copyright (C) 2012 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function dlProgress:                          function updates the myXCLASS sqlite3 database
##      - function UpdateDatabase:                      function updates the myXCLASS sqlite3 database
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    25.07.2013   initial version
##  C. Endres     06.09.2013   add Christians download package
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import math                                                                                 ## import math package
import sys                                                                                  ## import sys package
import string                                                                               ## import string package
import urllib                                                                               ## import package for handling url's
import time                                                                                 ## import package date and time
import warnings                                                                             ## import package warnings
import sqlite3                                                                              ## import sqlite3 package
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## This function updates the myXCLASS sqlite3 database
##
def dlProgress(count, blockSize, totalSize):
    """

input parameters:
-----------------

    - count:                counter

    - blockSize:            size of block

    - totalSize:            total size



output parameters:
------------------

    - None

    """

    percent = int(count * blockSize * 100 / totalSize)
    sys.stdout.write("\rDownloading sqlite3 database file for myXCLASS from CDMS Server .. %2d%%" % percent)
    sys.stdout.flush()
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## This function updates the myXCLASS sqlite3 database
##
def UpdateDatabase(DBUpdateNew):
    """

This function updates the myXCLASS database file located in the subdirectory "Database".

The XCLASS interface uses a sqlite3 database which contains two tables:

The values for the partition function for many molecules are saved in table "partitionfunctions". For each molecule the following data
are saved:

column 1:       name of molecule
column 6 - 115: the partition function for the temperatures (all in K):
                1.072, 1.148, 1.23, 1.318, 1.413, 1.514, 1.622, 1.738, 1.862, 1.995, 2.138, 2.291, 2.455, 2.63, 2.725, 2.818, 3.02, 3.236,
                3.467, 3.715, 3.981, 4.266, 4.571, 4.898, 5, 5.248, 5.623, 6.026, 6.457, 6.918, 7.413, 7.943, 8.511, 9.12, 9.375, 9.772,
                10.471, 11.22, 12.023, 12.882, 13.804, 14.791, 15.849, 16.982, 18.197, 18.75, 19.498, 20.893, 22.387, 23.988, 25.704, 27.542,
                29.512, 31.623, 33.884, 36.308, 37.5, 38.905, 41.687, 44.668, 47.863, 51.286, 54.954, 58.884, 63.096, 67.608, 72.444, 75, 77.625,
                83.176, 89.125, 95.499, 102.329, 109.648, 117.49, 125.893, 134.896, 144.544, 150, 154.882, 165.959, 177.828, 190.546, 204.174,
                218.776, 225, 234.423, 251.189, 269.153, 288.403, 300, 309.03, 331.131, 354.813, 380.189, 407.38, 436.516, 467.735, 500, 501.187,
                537.032, 575.44, 616.595, 660.693, 707.946, 758.578, 812.831, 870.964, 933.254, 1000

Additionally, the data for many radiative transitions are stored in table "transitions". Here, the
following data are saved for each radiative transition:

column 1*:   name of molecule
column 2*:   Frequency (in MHz)
column 3:    Intensity (in nm2 MHz)
column 4*:   Einstein A coefficient
column 5*:   Uncertainty of frequency (in MHz)
column 6*:   Energy_Lower (in cm^(-1))
column 7*:   upper state degeneracy
column 8:    nuclear spin isomer
column 9:    HFS
column 10:   not used at the moment
column 11:   upper state quantum numbers
column 12:   lower state quantum numbers

Please note, if you add private entries to the database, please make sure, that the partition function is given for all temperatures described above
and that all entries/columns in table "transitions" which are marked with a "*" sign are defined as well!

During the update process, the old sqlite database file is renamed whereat the file name "cdms_sqlite.db" is extended by the current date and time. For
example, the XCLASS interface renames this file to "cdms_sqlite__16-09-2013__10-40-15.db" and then downloads/upates the latest version of the database from
the CDMS server. So, the database which is used for a previous simulated spectra is not removed.



input parameters:
-----------------

    - DBUpdateNew:          flag indicating, if a complete new database file is downloaded from the CDMS server ("new") or the exsisting database
                            file is updated ("update").



output parameters:
------------------

    - None


Example 1:
----------

DBUpdateNew = "new"
UpdateDatabase()


Example 2:
----------

DBUpdateNew = "update"
UpdateDatabase()


    """


    ## get current directory
    CurrentDir = os.getcwd() + "/"


    ######################################################################################################################################################
    ## DO NOT EDIT OR REMOVE THE FOLLOWING LINE !!!!
    dbFilename = "/home/moeller/ALMA/CASA/myXCLASS-CASA-Interface/No-NR-version/Linux/XCLASS-Interface/Database/cdms_sqlite.db"
    ######################################################################################################################################################


    ## get path of database directory
    i = dbFilename.rfind("/")
    if (i > (-1)):
        dbPath = dbFilename[:i]
        dbFile = dbFilename[i + 1:]
    else:
        dbPath = "Database/"
        dbFile = dbFilename


    ##====================================================================================================================================================
    ## download the sqlite file from the CDMS server and store the file within the directory "Database"
    if (DBUpdateNew == "new"):


        ## rename the current sqlite3 file and download the latest version of the sqlite3 database file
        urlDB = "http://cdms.ph1.uni-koeln.de/cdms/portal/cdms_lite.db.gz"                  ## define url
        dbFilenameNEW = dbPath + "/cdms_lite.db"                                            ## define name of new database file


        ## move old database file
        print "\nRename the current sqlite3 file ..",
        sys.stdout.flush()
        lt = time.localtime()
        cmd_string = "cd " + dbPath                                                         ## change to database directory
        cmd_string += "; mv " + dbFilename + " " + dbPath + "/cdms_sqlite__" + time.strftime("%d-%m-%Y", lt) + "__" + time.strftime("%H-%M-%S", lt) + ".db"
        os.system(cmd_string)                                                               ## execute command string
        print "done!"


        ## download latest version of database
        global ZipFileName                                                                  ## global variable to be used in dlProgress
        ZipFileName = dbPath + "/cdms_lite.db.gz"
        urllib.urlretrieve(urlDB, ZipFileName, reporthook=dlProgress)
        sys.stdout.write("\rDownloading sqlite3 database file for myXCLASS from CDMS Server .. done!                        ")


        ## unzip file
        print "\nUnzip the downloaded database file ..",
        sys.stdout.flush()
        cmd_string = "cd " + dbPath                                                         ## change to database directory
        cmd_string += "; gunzip cdms_lite.db.gz"                                            ## unzip database file
        cmd_string += "; mv cdms_lite.db " + dbFilename                                     ## make a copy of the old database file
        os.system(cmd_string)                                                               ## execute command string
        print "done!"



    ##====================================================================================================================================================
    ## add path of update package to sys.path variable and check if sys.path variable is already updated and modify sys path variable if neccessary
    else:


        ## print what you do
        print " "
        print "Updating the sqlite3 database file %s." % dbFile
        print " "

        lt = time.localtime()
        datestring = time.strftime("Start updating at Date: %d.%m.%Y", lt) + time.strftime(",     Time: %H:%M:%S", lt)
        print str(datestring)
        print " "
        sys.stdout.flush()


        ## make a copy of the old database file
        print "\nMake a copy of the old sqlite3 file ..",
        sys.stdout.flush()
        lt = time.localtime()
        cmd_string = "cd " + dbPath                                                         ## change to database directory
        cmd_string += "; cp " + dbFilename + " " + dbPath + "/cdms_sqlite__" + time.strftime("%d-%m-%Y", lt) + "__" + time.strftime("%H-%M-%S", lt) + ".db"
        os.system(cmd_string)                                                               ## execute command string
        print "done!"


        ## define path of program package
        i = dbPath.rfind("/")
        if (i > (-1)):
            UpadatePackagePath = dbPath[:i]
        else:
            UpadatePackagePath = dbPath
        UpadatePackagePath = UpadatePackagePath + "/programs/Update_VAMDC/"


        ## check if sys.path variable is already updated and modify sys path variable if neccessary
        modules_path = [UpadatePackagePath]
        for NewModulesPath in modules_path:                                                 ## loop over all new module paths
            already_included_flag = "false"
            for entries in sys.path:
                if (entries == NewModulesPath):
                    already_included_flag = "true"
                    break
            if (already_included_flag == "false"):
                sys.path.append(NewModulesPath)

        # Debug:
        # print "sys.path = ", sys.path


        ## try to import update package
        try:
            import database
        except ImportError:
            print " "
            print "Error in subroutine UpdateDatabase:"
            print " "
            print "Can not import update package. Please use another update option!"
            print " "
            return


        ## update database
        db = database.Database(dbFilename)
        db.update_database()


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------

