#!/usr/bin/python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module contains the interface to MAGIX
##  Copyright (C) 2012 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function GetXMLtag:                           read xml tags with name tagName in xml file GetXMLtag and return contents in ContentsTag
##      - function WriteXMLtag:                         write contents in ContentsTag to xml file GetXMLtag xml at tags with name tagName
##      - function CreateRunDirectory:                  create a run directory for a function of the XCLASS interface
##      - function SetMAGIXEnvironment:                 set up environment variables for MAGIX call
##      - function StartMAGIX:                          start MAGIX without command line call
##      - function MAGIX:                               function defines the interface to MAGIX
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    25.07.2013   initial version
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import string                                                                               ## import string package
import random                                                                               ## import random package
import datetime                                                                             ## import datetime package
import time                                                                                 ## import package date and time
import warnings                                                                             ## import warnings package
import matplotlib                                                                           ## import matplotlib package
import pylab                                                                                ## load python package for plotting pylab
from matplotlib.font_manager import fontManager, FontProperties                             ## import fontManager and FontProperties packages
import sqlite3                                                                              ## import sqlite3 package
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## read xml tags with name tagName in xml file GetXMLtag and return contents in ContentsTag
##
def GetXMLtag(xmlFileName, tagName):
    """

input parameters:
-----------------

    - xmlFileName:          path and name of the xml-file

    - tagName:              name of the tag, whose contents is read


output parameters:
------------------

    - ContentsTag:          the contents of the selected tag, always a list, including an entry for each occurance

    """
    ContentsTag = []


    ## read in whole xml-file
    xmlFile = open(xmlFileName)
    ContentsXMLFile = xmlFile.readlines()
    xmlFile.close()


    ## analyze contents of xml file
    tagName = tagName.strip()
    CurrentDataFileID = 0
    for line in ContentsXMLFile:                                                            ## loop over all lines in the xml file
        striplines = line.strip()
        if (not striplines.startswith("<!--")):                                             ## check if current line is not a comment


            ## special handling for some tags
            if (striplines.find("<FileNamesExpFiles>") > (-1)):
                CurrentDataFileID += 1


            ## get contents of defined tag
            if (striplines.find("<" + tagName + ">") > (-1)):                               ## does the current line described selected tag?
                i = striplines.find(">")
                j = striplines.rfind("<")
                if (i < j):


                    ## special handling for some tags
                    if (tagName == "MinExpRange" or tagName == "MaxExpRange" or tagName == "StepFrequency" or tagName == "BackgroundTemperature" \
                          or tagName == "TemperatureSlope" or tagName == "HydrogenColumnDensity" or tagName == "DustBeta" or tagName == "Kappa" \
                          or tagName == "Threshold" or tagName == "ErrorY" or tagName == "NumberHeaderLines" or tagName == "SeparatorColumns" \
                          or tagName == "NoiseLevel"):
                        ContentsTag.append([CurrentDataFileID, striplines[i + 1:j]])
                    else:
                        ContentsTag.append(striplines[i + 1:j])                             ## save name(s) in list


    ## define return value
    return ContentsTag
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## write contents in ContentsTag to xml file
##
def WriteXMLtag(xmlFileName, tagName, ContentsTag):
    """

input parameters:
-----------------

    - xmlFileName:          path and name of the xml-file

    - tagName:              name of the tag, whose contents has to be modified

    - ContentsTag:          the new contents of the tag, which has to be always a list, i.e. a content for each occurance of the tag



output parameters:
------------------

    - None

    """


    ## read in whole xml-file
    xmlFile = open(xmlFileName)
    ContentsXMLFile = xmlFile.readlines()
    xmlFile.close()


    ## analyze contents of xml file
    NXF = open(xmlFileName, 'w')                                                            ## open xml file
    counter = (-1)                                                                          ## reset counter
    for line in ContentsXMLFile:                                                            ## loop over all lines in the xml file
        striplines = line.strip()                                                           ## remove leading and trailing blanks
        if (not striplines.startswith("<!--")):                                             ## ignore comments
            if (striplines.find("<" + tagName + ">") > (-1)):                               ## get name of experimental data file
                counter += 1
                i = line.find("<")
                if (i > (-1)):
                    space = line[:i]
                else:
                    space = ""
                if ((len(ContentsTag) - 1) >= counter):
                    NXF.write(space + "<" + tagName + ">" + ContentsTag[counter] + "</" + tagName + ">\n")

                # Debug:
                #    print "tagName = >>", tagName, '<<'
                #    print "counter = ", counter
                #    print "(len(ContentsTag) - 1) = ", (len(ContentsTag) - 1)
                #    print "ContentsTag[counter] = ", ContentsTag[counter]


            ## add tag at the end of the xml file if tag is not found
#            elif (striplines == "</ExpFiles>" and counter == (-1)):
#                if ((len(ContentsTag) - 1) >= counter):
#                    NXF.write("\n\n")
#                    NXF.write("    <!-- " + tagName + " -->\n")
#                    NXF.write("    <" + tagName + ">" + ContentsTag[counter] + "</" + tagName + ">\n")
#                    NXF.write(line)
            else:
                NXF.write(line)
        else:
            NXF.write(line)
    NXF.close()


    ## special handling for experimental data files for myXCLASS
    if (counter == (-1) or len(ContentsTag) > counter):                                     ## the selected tag was not found or the given tag has more
                                                                                            ## entries than tags in xml file
        if (tagName == "BackgroundTemperature" or tagName == "TemperatureSlope" or tagName == "HydrogenColumnDensity" or tagName == "DustBeta" \
            or tagName == "Kappa" or tagName == "Threshold" or tagName == "dbFilename" or tagName == "ErrorY"):    ## insert only these tags to xml file


            ## read in contents of xml file
            xmlFile = open(xmlFileName)
            ContentsXMLFile = xmlFile.readlines()
            xmlFile.close()


            ## insert new tag to xml file
            NXF = open(xmlFileName, 'w')                                                    ## open xml file
            for line in ContentsXMLFile:                                                    ## loop over all lines in the xml file
                striplines = line.strip()                                                   ## remove leading and trailing blanks
                if (not striplines.startswith("<!--")):                                     ## ignore comments


                    ## add tag at the end of FrequencyRange in xml file if tag is not found
                    if (striplines == "</FrequencyRange>" and counter == (-1) and tagName != "dbFilename" and tagName != "ErrorY"):
                        counter += 1
                        NXF.write("\n\n")
                        NXF.write("            <!-- " + tagName + " -->\n")
                        NXF.write("            <" + tagName + ">" + ContentsTag[counter] + "</" + tagName + ">\n")
                        NXF.write(line)


                    ## add tag at the end of ExpFiles in xml file if tag is not found
                    elif (striplines == "</ExpFiles>" and counter == (-1) and (tagName == "dbFilename" or tagName == "ErrorY")):
                        counter += 1
                        NXF.write("\n\n")
                        NXF.write("    <!-- " + tagName + " -->\n")
                        NXF.write("    <" + tagName + ">" + ContentsTag[counter] + "</" + tagName + ">\n")
                        NXF.write(line)
                    else:
                        NXF.write(line)
                else:
                    NXF.write(line)
            NXF.close()


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## write contents in ContentsTag to xml file GetXMLtag xml at tags with name tagName
##
def CreateRunDirectory(NameOfFunction, myXCLASSrootDir, PrintFlag):
    """

input parameters:
-----------------

    - NameOfFunction:       name of function of the XCLASS interface

    - myXCLASSrootDir:      path and name of root directory

    - PrintFlag:            flag for writing messages to screen



output parameters:
------------------

    - RunDir:               name of run directory for the current function

    """


    ## determine job id, i.e. create a number to identify job unambiguous
    pid = os.getpid()
    ranNum = int(random.uniform (1000, 9999))
    jobID = str(pid) + str(ranNum)
    lt = time.localtime()
    jobID = "_" + time.strftime("%d-%m-%Y", lt) + "__" + time.strftime("%H-%M-%S", lt) + "__" + jobID


    ##====================================================================================================================================================
    ## create temp directory
    myXCLASSRunDirectory = str(os.environ.get('myXCLASSRunDirectory','')).strip()           ## get value of environment variable
    if (myXCLASSRunDirectory == ""):                                                        ## is environment variable defined?
        RunDir = myXCLASSrootDir + "../../run/" + NameOfFunction + "/job_" + jobID + "/"    ## if no, use default setting
        RunDir = os.path.abspath(RunDir) + "/"
    else:
        if (myXCLASSRunDirectory[0] != "/"):                                                ## make path absolute
            myXCLASSRunDirectory = myXCLASSrootDir + "../../" + myXCLASSRunDirectory
        RunDir = myXCLASSRunDirectory + "/" + NameOfFunction + "/job_" + jobID + "/"        ## if environment variable is defined, apply user setting
    if (PrintFlag):
        print " "
        print "Start function " + NameOfFunction + ":"
        print " "
        print " "
        print "Creating job directory for current myXCLASS run: " + RunDir + " ..",
    command_string = "mkdir -p " + RunDir
    os.system(command_string)
    if (PrintFlag):
        print "done!"
        print " "
        print "All files of the current " + NameOfFunction + " run are stored here!"
        print " "
        print " "


    ## define return variable
    return RunDir
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## set up environment variables for MAGIX call
##
def SetMAGIXEnvironment(MAGIXrootDir):
    """

input parameters:
-----------------

    - MAGIXrootDir:         MAGIX root directory



output parameters:
------------------

    - None


    """


    ## check if sys.path variable is already updated and modify sys path variable if neccessary
    modules_path = [MAGIXrootDir + "Modules/magix-parts_python",
                    MAGIXrootDir + "Modules/Additional_Packages/bin",
                    MAGIXrootDir + "Modules/Interval_Nested-Sampling/bin",
                    MAGIXrootDir + "Modules/ErrEstim_INS/bin",
                    MAGIXrootDir + "Modules/MCMC/bin",
                    MAGIXrootDir + "Modules/MCMC/emcee"]
    for NewModulesPath in modules_path:                                                     ## loop over all new module paths
        already_included_flag = "false"
        if (os.path.exists(NewModulesPath)):
            for entries in sys.path:
                if (entries == NewModulesPath):
                    already_included_flag = "true"
                    break
            if (already_included_flag == "false"):
                sys.path.append(NewModulesPath)

    # Debug:
    # os.system("export PYTHONPATH=$PYTHONPATH:" + modules_path)
    # os.system("echo Pythonpath = $PYTHONPATH")
    # print "sys.path = ", sys.path


    ## check MAGIX temp directory
    MAGIXTempDirectory = str(os.environ.get('MAGIXTempDirectory',''))
    if (MAGIXTempDirectory.strip() == ""):
        os.environ["MAGIXTempDirectory"] = MAGIXrootDir + 'temp/'
        MAGIXTempDirectory = str(os.environ.get('MAGIXTempDirectory','')).strip()
        print "Set environment variable MAGIXTempDirectory = ", MAGIXTempDirectory


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## start MAGIX without command line call
##
def StartMAGIX(MAGIXrootDir, option, filename_control):
    """

input parameters:
-----------------

    - MAGIXrootDir:         MAGIX root directory

    - option:               run option

    - filename_control:     path and name of io control file



output parameters:
------------------

    - ok:                   status variable


    """

    # Debug:
    # print "option = ", option
    # print "len(option) = ", len(option)
    # print "filename_control = ", filename_control
    #    print 'module name:', __name__
    #    if hasattr(os, 'getppid'):  # only available on Unix
    #        print 'parent process:', os.getppid()
    #    print 'process id:', os.getpid()



    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read commands from command line
    ok = 0
    printflag = "true"
    plotflag = "true"
    debugflag = "false"
    modelflag = "false"
    if (len(option) > 1):
        OptionList = option.split(",")
        for SingleOption in OptionList:                                                     ## loop over all run flags
            StrippedSingleOption = SingleOption.strip()                                     ## remove leading and tailing blanks
            StrippedSingleOption = StrippedSingleOption.lower()                             ## convert all characters to lower casa


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get start flags
            if (StrippedSingleOption == "debug"):
                debugflag = "true"

            elif (StrippedSingleOption.startswith("model")):
                modelflag = "true"
                SplitModel = StrippedSingleOption.split("=")
                if (len(SplitModel) > 1):
                    modelflag = SplitModel[1]

            elif (StrippedSingleOption == "quiet"):
                plotflag = "saveonly"
                printflag = "false"

            elif (StrippedSingleOption == "noplot"):
                plotflag = "false"

            elif (StrippedSingleOption == "plotsaveonly"):
                plotflag = "saveonly"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check if io-control file exsists
    if not(os.path.exists(filename_control)):
        ok = 1
        print " "
        print "\t  Error in io_control.xml file!"
        print "\t      The file ",filename_control
        print "\t      does not exists!"
        print " "
        print "\t      Enter the path and the name of the"
        print "\t      io_control xml-file at the command line:"
        print " "
        print "\t      For example:"
        print "\t      ./magix_start.py run/examples/Levenberg-Marquardt/io_control.xml"
        print " "
        return ok

    # Debug:
    # print "printflag = ", printflag
    # print "plotflag = ", plotflag
    # print "debugflag = ", debugflag
    # print "modelflag = ", modelflag
    # print "filename_control = ", filename_control


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## determine job id, i.e. create a number to identify job unambiguous
    pid = os.getpid()
    ranNum = int(random.uniform (1000, 9999))
    jobID = str(pid) + str(ranNum)

    # Debug:
    # print "jobID = ", jobID
    # print "\n\n\n\n\n>>>>>>>>>>>>>>>>>>", sys.path, "\n\n\n"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## import first MAGIX module and start main program
    import MainProgram
    ok = MainProgram.main(printflag, plotflag, debugflag, modelflag, filename_control, jobID, MAGIXrootDir)


    ## define return value
    return ok
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## The function defines the interface to MAGIX
##
def MAGIX(MAGIXExpXML, MAGIXInstanceXML, MAGIXFitXML, MAGIXRegXML, MAGIXOption):
    """

    The function starts the MAGIX program.

    This function copies the different xml-files (except the registration xml-file)
    to a so-called "job-directory" located in the run working directory
    "path-of-myXCLASS-CASA-Interface/run/MAGIX/"! The name of a job directory for a
    MAGIX run is made up of four components: The first part of the name consists of the
    phrase "job_" whereas the second part describes the date (day, month, year), the
    third part the time stamp (hours, minutes, seconds) of the function execution. The
    last part describes a so-called job ID which is composed of the so-called PID
    followed by a four digit random integer number to create a really unambiguous job
    number, e.g.
    "path-of-myXCLASS-CASA-Interface/run/MAGIX/job__25-07-2013__12-02-03__189644932/"

    All file(s), which are created by the current MAGIX run, are stored in such a job
    directory!

    Additionally, the function copies all xml-files (except the so-called
    registration xml-file) to the current job directory.
    Furthermore, the function copies the experimental data file(s), i.e., the files
    which contains the experimental/observational data, to the current MAGIX job
    directory as well. The path(s) of the experimental data file(s) defined in the
    experimental xml-file are adjusted, so that these path(s) become absolute
    and point to the current job directory. Please note, that all modifications are
    applied to the copies of the xml-files. The original xml-files are not modified.

    As mentioned above, the registration xml-file is neither copied to the job
    directory nor modified! But, if the path of the registration xml-file is
    relative, the path has to be defined relative to the myXCLASS working directory
    "path-of-myXCLASS-CASA-Interface/"

    The io-control file (which is required for each MAGIX run) is created
    automatically!


input parameters:
-----------------

    - MAGIXExpXML:          path and name of the experimental xml-file.

                            Note, if the paramter defines a relative path, this path has to be defined relative to the
                            current working directory!


    - MAGIXInstanceXML:     path and name of the instance xml-file.

                            Note, if the paramter defines a relative path, this path has to be defined relative to the
                            current working directory!


    - MAGIXFitXML:          path and name of the xml file controlling the fitting process.

                            Note, if the paramter defines a relative path, this path has to be defined relative to the
                            current working directory!


    - MAGIXRegXML:          path and name of the so-called registration xml file containing the description of
                            the input and output files of the external model program.

                            Note, if the paramter defines a relative path, this path has to be defined relative to the
                            current working directory!


    - MAGIXOption:          option for the MAGIX run (default is None):

                            - None (default): all informations are printed out to the screen
                            - quiet:          no informations are printed to the screen except warning and error messages
                            - plotsaveonly:   MAGIX disables the interactive GUI of matplotlib but creates all plots and
                                              saves them into files
                            - debug:          Stop MAGIX after the first function call. This flag can be very helpful to
                                              analyze problems occurring with the call of the external model program.





output parameters:
------------------

    - myXCLASSJobDir:       absolute path of the myXCLASS job directory created for the current run.




Example:
--------

MAGIXExpXML = "demo/MAGIX/TwoOscillators_RefFit_R.xml"
MAGIXInstanceXML = "demo/MAGIX/parameters.xml"
MAGIXFitXML = "demo/MAGIX/Levenberg-Marquardt_Parameters.xml"
MAGIXRegXML = "Fit-Functions/Drude-Lorentz_conv/xml/Conventional_Drude-Lorentz.xml"
MAGIXOption = None

myXCLASSJobDir = MAGIX()

    """


    ##====================================================================================================================================================
    ## initialize working variables
    CurrentDir = os.getcwd() + "/"
    MAGIXjobDir = ""


    ######################################################################################################################################################
    ## DO NOT EDIT OR REMOVE THE FOLLOWING LINE !!!!
    MAGIXrootDir = "/home/moeller/ALMA/CASA/myXCLASS-CASA-Interface/No-NR-version/Linux/XCLASS-Interface/programs/MAGIX/"
    ######################################################################################################################################################


    ##====================================================================================================================================================
    ## check path and file name of the experimental xml-file
    if (MAGIXExpXML == None):
        print " "
        print " "
        print "Error in XCLASS package, function MAGIX:"
        print "  No path and file name of the experimental xml-file is defined!"
        print " "
        print "  Please enter the path and file name of the experimental xml-file and redo MAGIX call!"
        return MAGIXjobDir
    MAGIXExpXML = MAGIXExpXML.strip()
    if (MAGIXExpXML[0] != "/"):
        MAGIXExpXML = CurrentDir + MAGIXExpXML
    if not(os.path.exists(MAGIXExpXML) or os.path.isfile(MAGIXExpXML)):
        print " "
        print " "
        print "Error in XCLASS package, function MAGIX:"
        print "  The given path and file name of the experimental xml-file " + chr(34) + MAGIXExpXML + chr(34) + " does not exsist!"
        print " "
        print "  Please enter an exsisting path and file name of the experimental xml-file and redo MAGIX call!"
        return MAGIXjobDir

    # Debug:
    # print "MAGIXExpXML = ", testpath


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check path and file name of the instance xml-file
    if (MAGIXInstanceXML == None):
        print " "
        print " "
        print "Error in XCLASS package, function MAGIX:"
        print "  No path and file name of the instance xml-file is defined!"
        print " "
        print "  Please enter the path and file name of the instance xml-file and redo MAGIX call!"
        return MAGIXjobDir

    MAGIXInstanceXML = MAGIXInstanceXML.strip()
    if (MAGIXInstanceXML[0] != "/"):
        MAGIXInstanceXML = CurrentDir + MAGIXInstanceXML
    if not(os.path.exists(MAGIXInstanceXML) or os.path.isfile(MAGIXInstanceXML)):
        print " "
        print " "
        print "Error in XCLASS package, function MAGIX:"
        print "  The given path and file name of the instance xml-file " + chr(34) + MAGIXInstanceXML + chr(34) + " does not exsist!"
        print " "
        print "  Please enter an exsisting path and file name of the experimental xml-file and redo MAGIX call!"
        return MAGIXjobDir

    # Debug:
    # print "MAGIXInstanceXML = ", MAGIXInstanceXML


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check path and file name of the fit xml-file
    if (MAGIXFitXML == None):
        print " "
        print " "
        print "Error in XCLASS package, function MAGIX:"
        print "  No path and file name of the fit xml-file is defined!"
        print " "
        print "  Please enter the path and file name of the fit xml-file and redo MAGIX call!"
        return MAGIXjobDir
    MAGIXFitXML = MAGIXFitXML.strip()
    if (MAGIXFitXML[0] != "/"):
        MAGIXFitXML = CurrentDir + MAGIXFitXML
    if not(os.path.exists(MAGIXFitXML) or os.path.isfile(MAGIXFitXML)):
        print " "
        print " "
        print "Error in XCLASS package, function MAGIX:"
        print "  The given path and file name of the fit xml-file " + chr(34) + MAGIXFitXML + chr(34) + " does not exsist!"
        print " "
        print "  Please enter an exsisting path and file name of the fit xml-file and redo MAGIX call!"
        return MAGIXjobDir

    # Debug:
    # print "MAGIXFitXML = ", testpath


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check path and file name of registration xml-file
    if (MAGIXRegXML == None):
        print " "
        print " "
        print "Error in XCLASS package, function MAGIX:"
        print "  No path and file name of the registration xml-file is defined!"
        print " "
        print "  Please enter the path and file name of the registration xml-file and redo MAGIX call!"
        return MAGIXjobDir
    MAGIXRegXML = MAGIXRegXML.strip()
    if (MAGIXRegXML[0] != "/"):
        MAGIXRegXML = MAGIXrootDir + MAGIXRegXML
    if not(os.path.exists(MAGIXRegXML) or os.path.isfile(MAGIXRegXML)):
        print " "
        print " "
        print "Error in XCLASS package, function MAGIX:"
        print "  The given path and file name of the registration xml- file " + chr(34) + MAGIXRegXML + chr(34) + " does not exsist!"
        print " "
        print "  Please note, that the path of the start script defined in the registration xml-file has to be"
        print "  given absolute or relative to the MAGIX WORKING DIRECTORY " + MAGIXrootDir + " !"
        print " "
        print "  Please enter an exsisting path and file name of the registration xml-file and redo MAGIX call!"
        return MAGIXjobDir

    # Debug:
    # print "MAGIXRegXML = ", MAGIXRegXML


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check option for MAGIX run
    if (MAGIXOption == None or MAGIXOption.strip() == ""):
        MAGIXOption = ""
    elif (MAGIXOption.lower() != "noplot" and MAGIXOption.lower() != "plotsaveonly" and MAGIXOption.lower() != "quiet" and MAGIXOption.lower() != "debug"):
        print " "
        print " "
        print "Error in XCLASS package, function MAGIX:"
        print "  The given option for the MAGIX run is neither 'noplot' nor 'plotsaveonly' nor 'quiet' nor 'None'"
        print " "
        print "  Please enter one of the above given options and redo MAGIX call!"
        return MAGIXjobDir
    else:
        MAGIXOption = MAGIXOption.lower()


    ##====================================================================================================================================================
    ## define path to MAGIX modules, check if sys.path variable is already updated and modify sys path variable if neccessary


    ## set MAGIX environment variables
    SetMAGIXEnvironment(MAGIXrootDir)


    ##====================================================================================================================================================
    ## create run directory for the current function call
    LocalPrintFlag = True
    MAGIXjobDir = CreateRunDirectory("MAGIX", MAGIXrootDir, LocalPrintFlag)

    # Debug:
    # print "command_string = ", command_string


    ## set path for log-files
    MAGIXLogFile = MAGIXjobDir + "fit.log"


    ##====================================================================================================================================================
    ## copy xml-files to current temp directory and modify experimental xml-file


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## copy instance xml file to temp directory
    print "\n\nCopy instance xml-file to MAGIX temp directory " + MAGIXjobDir + " ..",
    command_string = "cp " + MAGIXInstanceXML + " " + MAGIXjobDir
    os.system(command_string)


    ## get name of experimental xml-file
    i = MAGIXInstanceXML.rfind("/")
    if (i > 0):
        MAGIXInstanceXML = MAGIXInstanceXML[i + 1:]
    MAGIXInstanceXML = MAGIXjobDir + MAGIXInstanceXML

    # Debug:
    # print ' '
    # print 'MAGIXInstanceXML = ', MAGIXInstanceXML
    print "done!"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## copy algorithm xml file to temp directory
    print "Copy algorithm xml-file to MAGIX temp directory " + MAGIXjobDir + " ..",
    command_string = "cp " + MAGIXFitXML + " " + MAGIXjobDir
    os.system(command_string)


    ## get name of experimental xml-file
    i = MAGIXFitXML.rfind("/")
    if (i > 0):
        MAGIXFitXML = MAGIXFitXML[i + 1:]
    MAGIXFitXML = MAGIXjobDir + MAGIXFitXML

    # Debug:
    # print ' '
    # print 'MAGIXFitXML = ', MAGIXFitXML
    print "done!"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## copy experimental xml file to temp directory
    print "Copy experimental xml-file to MAGIX temp directory " + MAGIXjobDir + " ..",
    command_string = "cp " + MAGIXExpXML + " " + MAGIXjobDir
    os.system(command_string)


    ## get name of experimental xml-file
    i = MAGIXExpXML.rfind("/")
    if (i > 0):
        MAGIXExpXML = MAGIXExpXML[i + 1:]
    MAGIXExpXML = MAGIXjobDir + MAGIXExpXML

    # Debug:
    # print ' '
    # print 'MAGIXExpXML = ', MAGIXExpXML
    print "done!"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## make paths of all data files defined in the experimental xml-file absolute
    print "Make paths of all data files defined in the experimental xml-file absolute ..",
    ExpFiles = []

    # Debug:
    # print "CurrentDir = ", CurrentDir
    # print "MAGIXExpXML = ", MAGIXExpXML

    ExpFilesOrig = GetXMLtag(MAGIXExpXML, "FileNamesExpFiles")
    for filename in ExpFilesOrig:
        if (filename[0] != "/"):                                                            ## make relative path absolute
            filename = CurrentDir + filename
        if not(os.path.exists(filename) or os.path.isfile(filename)):
            print "\n\n\n"
            print "Error in XCLASS package, function MAGIX:"
            print "  The given path and file name of the experimental data file " + filename
            print "  defined in the experimental data xml-file does not exsist!"
            print " "
            print "  Please note, if a relative path is specified, the path has to be defined relative to the"
            print "  current working directory!"
            print " "
            print "  Please enter an exsisting path and file name and redo MAGIX function call!"
            return MAGIXjobDir
        ExpFiles.append(filename)                                                           ## save name(s) in list

    # Debug:
    # print ' '
    # print 'ExpFiles = ', ExpFiles
    print "done!"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## copy experimental data file(s) to MAGIX temp directory
    print "Copy all experimental data file(s) defined in the experimental data file to MAGIX temp directory " + MAGIXjobDir + " and modify xml-file ..",
    NewExpFiles = []                                                                        ## list with modified data file names
    for files in ExpFiles:
        files = files.strip()
        if (files[0] != "/"):
            files = CurrentDir + files

        command_string = "cp " + files + " " + MAGIXjobDir                                 ## copy experimetnal dat files to temp directory
        os.system(command_string)
        i = files.rfind("/")
        if (i < 0):
            i = 0
        NewExpFiles.append(MAGIXjobDir + files[i + 1:])                                    ## define new data file names


    ## modify experimental xml-file in MAGIX temp directory
    WriteXMLtag(MAGIXExpXML, "FileNamesExpFiles", NewExpFiles)
    print "done!\n\n"


    ##====================================================================================================================================================
    ## construct io_control.xml file
    print "Create io-control xml-file ..",
    iocontrol = open(MAGIXjobDir + "io_control.xml", 'w')
    iocontrol.write("<?xml version=" + chr(34) + "1.0" + chr(34) + " encoding=" + chr(34) + "ISO-8859-1" + chr(34) + "?>\n")
    iocontrol.write("<ioControl>\n")
    iocontrol.write("    <title>io control</title>\n")
    iocontrol.write("    <description>This xml-file contains the paths and the file names of all working files which are used by MAGIX during the fit process</description>\n")
    iocontrol.write("    <PathFilename>\n")
    iocontrol.write("\n")
    iocontrol.write("        <experimental_data>\n")
    iocontrol.write("            <filename>" + MAGIXExpXML + "</filename>\n")
    iocontrol.write("            <description>path and file name of the experimental file</description>\n")
    iocontrol.write("        </experimental_data>\n")
    iocontrol.write("\n")
    iocontrol.write("        <parameter_file>\n")
    iocontrol.write("            <filename>" + MAGIXInstanceXML + "</filename>\n")
    iocontrol.write("            <description>path and file name of the xml-file including the start values of each parameter</description>\n")
    iocontrol.write("        </parameter_file>\n")
    iocontrol.write("\n")
    iocontrol.write("        <fit_control>\n")
    iocontrol.write("            <filename>" + MAGIXFitXML + "</filename>\n")
    iocontrol.write("            <description>path and file name of the xml file controlling the fitting process</description>\n")
    iocontrol.write("        </fit_control>\n")
    iocontrol.write("\n")
    iocontrol.write("        <fit_log>\n")
    iocontrol.write("            <filename>" + MAGIXLogFile + "</filename>\n")
    iocontrol.write("            <description>path and file name of log file describing the fitting process</description>\n")
    iocontrol.write("        </fit_log>\n")
    iocontrol.write("\n")
    iocontrol.write("        <model_description>\n")
    iocontrol.write("            <filename>" + MAGIXRegXML + "</filename>\n")
    iocontrol.write("            <description>path and file name of the xml-description of the input/output file of the fit function module</description>\n")
    iocontrol.write("        </model_description>\n")
    iocontrol.write("\n")
    iocontrol.write("    </PathFilename>\n")
    iocontrol.write("</ioControl>\n")
    iocontrol.write("\n")
    iocontrol.close()
    print "done!"


    ##====================================================================================================================================================
    ## start MAGIX
    ok = StartMAGIX(MAGIXrootDir, MAGIXOption, MAGIXjobDir + "io_control.xml")
    if (ok != 0):
        print "\n\n\n\t Program MAGIX aborted!\n\n"


    ## import warning:
    else:
        print "\n\nIMPORTANT:"
        print "----------"
        print "\nAll files of the current MAGIX run are stored in the MAGIX job directory: " + MAGIXjobDir
        print " "


    ##====================================================================================================================================================
    ## define return variables
    return MAGIXjobDir
##--------------------------------------------------------------------------------------------------------------------------------------------------------

