#!/usr/bin/python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module import data from an ASCII file using the numpy.loadtxt function
##  Copyright (C) 2012 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function LoadASCIIFile:                       primitive routine to import data from an ASCII file using the numpy.loadtxt function
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    25.07.2013   initial version
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import string                                                                               ## import string package
import random                                                                               ## import random package
import datetime                                                                             ## import datetime package
import task_myXCLASS                                                                        ## import package MAGIX
import warnings                                                                             ## import warnings package
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## A simple ASCII import routine
##
def LoadASCIIFile(FileName, NumHeaderLines, RestFreq, vLSR):
    """

A very primitive routine to import data from an ASCII file using the numpy.loadtxt function.


input parameters:
-----------------

    - FileName:             path and file name of an ASCII file.

                            Note, a relative path has to be defined relative to the current working directory!

    - NumHeaderLines:       number of header lines, which are ignored (default: 0).

    - RestFreq:             rest frequency in MHz (default: 0). (If this parameter is set to zero, the intensity is plotted against frequency (in MHz)
                            otherwise against velocity (in km/s).

    - vLSR:                 velocity (local standard of rest) in km/s, only used, if RestFreq /= 0. (velocity(Frequency = RestFreq) = vLSR)


output parameters:
------------------

    - ASCIIdata:            contents of the ASCII file

                            Please note, the user is free to define a different name for
                            the output parameter.


Example:
--------

FileName = "demo/LoadASCIIFile/ASCII.dat"
NumHeaderLines = 0
RestFreq = 0.0
vLSR = 0.0
ASCIIdata = LoadASCIIFile()

    """


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check input parameters
    ASCIIdata = 0

    # Debug:
    # print "NumHeaderLines = ", NumHeaderLines
    # print "FileName = ", FileName


    ##====================================================================================================================================================
    ## check number of header lines
    if (NumHeaderLines < 0):
        print "Error in XCLASS package, function LoadASCIIFile:"
        print "  The given number of header lines is less than zero!"
        print " "
        print "  Please correct input setting!"
        return


    ##====================================================================================================================================================
    ## check if file exists
    FileName = FileName.strip()
    if not(os.path.exists(FileName)):
        print "Error in XCLASS package, function LoadASCIIFile:"
        print "  The specified file does not exist!"
        print " "
        print "  Please correct input setting!"
        return


    ##====================================================================================================================================================
    ## import ASCII data
    print "Import ASCII data from ", FileName, "..",
    ASCIIdata = numpy.loadtxt(FileName, skiprows = NumHeaderLines)
    print "done!"
    print " "
    print "Number of data points = ", len(ASCIIdata)
    print "First data point = ", ASCIIdata[0]
    print "Last data point = ", ASCIIdata[-1]
    print " "


    ##====================================================================================================================================================
    ## add velocity axis if RestFreq != 0.0
    if (RestFreq != 0.0):
        print "Add velocity axis ..",
        FreqData = ASCIIdata[:, 0]
        VelocityData = task_myXCLASS.ChangeToVelocitiy(FreqData, RestFreq, vLSR)


        ## add velocity axis to ASCIIdata array
        NewASCIIdata = numpy.zeros((len(ASCIIdata), 3), dtype=numpy.float32)
        NewASCIIdata[:, 0] = ASCIIdata[:, 0]
        NewASCIIdata[:, 1] = VelocityData[:]
        NewASCIIdata[:, 2] = ASCIIdata[:, 1]
        ASCIIdata = NewASCIIdata


        ## print some informations to screen
        print "done!"
        print " "
        print "First velocity point = ", ASCIIdata[0, 1]
        print "Last velocity point = ", ASCIIdata[-1, 1]
        print " "


    ##====================================================================================================================================================
    ## define return value
    return ASCIIdata
##--------------------------------------------------------------------------------------------------------------------------------------------------------

