#!/usr/bin/python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module reads in entries from the sqlite3 database
##  Copyright (C) 2012 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function ListDatabase:                        function reads in entries from the sqlite3 database
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    25.07.2013   initial version
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import string                                                                               ## import string package
import random                                                                               ## import random package
import datetime                                                                             ## import datetime package
from math import *                                                                          ## import math package
import sqlite3                                                                              ## import sqlite3 package
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## This function reads in entries from the sqlite3 database
##
def ListDatabase(FreqMin, FreqMax, ElowMin, ElowMax, SelectMolecule, OutputDevice):
    """

This function reads in entries from the sqlite database table "Transitions" and prints out the contents
to the screen or file (defined by the input parameter OutputDevice).

The user can limit the output by defining a minimum and maximum for the frequency (or for
the lower energy) for the transitions.


input parameters:
-----------------


    - FreqMin:              minimum frequency (in MHz) (default: 0) for transitions in the table "Transitions"

    - FreqMax:              maximum frequency (in MHz) (default: 1.e8) for transitions in the table "Transitions"

    - ElowMin:              minimum for the lower energy (in K) (default: 0) in the "Transitions" table

    - ElowMax:              maximum for the lower energy (in K) (default: 1.e6) in the "Transitions" table

    - SelectMolecule:       a (python) list containing the names of all molecules which should be considered
                            or a file name including the molecules which should be selected.

                            Note, if the paramter defines a relative path, this path has to be defined
                            relative to the current working directory!

    - OutputDevice:         path and name of the file where the output is written to. If no file name is
                            defined (None), the contents of the database is written to the screen.

                            Note, if the paramter defines a relative path, this path has to be defined
                            relative to the current working directory!


output parameters:
------------------

    - Contents:             contents of the database (as an python array, e.g. contents[0] contains
                            the entries for the first molecule within the frequency/energy range)

                            Note, the user is free to define a different name for the output parameter.


Example 1:
----------

FreqMin = 20000.0
FreqMax = 20100.0
ElowMin = 100.0
ElowMax = 1000.0
SelectMolecule = "demo/ListDatabase/molecules.txt"
OutputDevice = ""
Contents = ListDatabase()


Example 2:
----------

FreqMin = 20000.0
FreqMax = 20100.0
ElowMin = 100.0
ElowMax = 2000.0
SelectMolecule = []
OutputDevice = ""
Contents = ListDatabase() 

    """

    ##====================================================================================================================================================
    ## check input parameters
    printFlag = "true"
    if (OutputDevice.strip() == "quiet"):
        printFlag = "false"

    # Debug:
    # print "FreqMin = ", FreqMin
    # print "FreqMax = ", FreqMax
    # print "SelectMolecule = ", SelectMolecule


    ## test, if SelectMolecule is a list, otherwise try to use SelectMolecule as filename
    if (printFlag != "false"):
        print " "
        print " "
        print "Analyze selected molecules ..",
        sys.stdout.flush()
    ListOfSelectedMolecules = []
    if (len(SelectMolecule) > 0 and SelectMolecule != ["None"]):
        for entry in SelectMolecule:                                                        ## loop over all entries of SelectMolecule list
            entryStrip = entry.strip()                                                      ## remove leading and trailing blanks


            ## check, if current entry is a file
            if (os.path.isfile(entryStrip)):


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## if file is a molfit file, get names of molecules from molfit file
                if (entryStrip.endswith("molfit")):

                    # Debug:
                    # print "Selection string is a molfit file!"


                    ## read in contents of molfit file
                    f = open(entryStrip)
                    MolfitContents = f.readlines()                                          ## read in contents of molfit file
                    f.close()


                    ## get names of molecules
                    for lines in MolfitContents:                                            ## loop over all lines of molfit file
                        CurrentLine = lines.strip()                                         ## remove leading and trailing blanks

                        # Debug:
                        # print "CurrentLine = ", CurrentLine

                        ## check for comments
                        w = CurrentLine.find("%")                                           ## are there comments in the current line ?
                        if (w == 0):                                                        ## ignore lines which contains only comments
                            CurrentLine = ""                                                ## if line is only a comment clear line
                        elif (w > 0):                                                       ## is there a comment in the current line ?
                            CurrentLine = CurrentLine[:w]                                   ## remove comments
                        CurrentLine = CurrentLine.strip()

                        # Debug:
                        # print "CurrentLine = ", CurrentLine


                        ## analyze, if the current line contains the name of a molecule
                        if (CurrentLine != ""):                                             ## ignore empty lines
                            SplitLines = CurrentLine.split()                                ## split current line into columns

                            # Debug:
                            # print "SplitLines = ", SplitLines


                            ## contains the last column an integer number?
                            helpstring = SplitLines[-1].strip()
                            is_int = "false"
                            if (helpstring.isdigit()):                                      ## continue, if string is an integer number
                                is_int = "true"


                            ## If yes, then the current line contains the name of a molecule
                            if (is_int == "true"):                                          ## if last entry in a line is an integer number, then
                                MolName = ""                                                ## the current line contains the name of a molecule
                                for col in SplitLines[:-1]:                                 ## get whole line without integer number at the end
                                    MolName += col                                          ## construct entry for SelectMolecule

                                # Debug:
                                # print "MolName = ", MolName

                                ListOfSelectedMolecules.append(MolName)                     ## construct final array containing all molecule names


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## read names of molecules from ASCII file
                else:

                    # Debug:
                    # print "Selection string is a ASCII file!"


                    ## read in whole ASCII file
                    f = open(entryStrip)
                    AllLines = f.readlines()
                    f.close()


                    ## append lines in ASCII file to selection list
                    for line in AllLines:
                        ListOfSelectedMolecules.append(line)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## continue here, if current entry is not a filename
            else:
                ListOfSelectedMolecules.append(entryStrip)


    ##====================================================================================================================================================
    ## connect to the sqlite3 database
    if (printFlag != "false"):
        print "done!"
        print "Reading data from sqlite3 database ..",
        sys.stdout.flush()


    ######################################################################################################################################################
    ## DO NOT EDIT OR REMOVE THE FOLLOWING LINE !!!!
    dbFilename = "/home/moeller/ALMA/CASA/myXCLASS-CASA-Interface/No-NR-version/Linux/XCLASS-Interface/Database/cdms_sqlite.db"
    ######################################################################################################################################################

    try:
        conn = sqlite3.connect(dbFilename)

        # Debug:
        # print " "
        # print "Connection to sqlite3 database %s established." % dbFilename
    except sqlite3.Error, e:
        print " "
        print "Can not connect to sqlite3 databse %s." % dbFilename
        print "Error: %d: %s" % (e.args[0], e.args[1])
        sys.exit(1)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get entries from database


    ## define query string
    ##
    ## get names of columns with
    ##      import sqlite3
    ##      connection = sqlite3.connect('cdms_sqlite.db')
    ##      cursor = connection.execute('select * from partitionfunctions')
    ##      names = list(map(lambda x: x[0], cursor.description))
    ##
    NameOfRadTransTable = "transitions"
    ColumnNameForNameTransitions = "T_Name"
    ColumnNameForFreqTransitions = "T_Frequency"
    ColumnNameForIntTransitions = "T_Intensity"
    ColumnNameForEinsteinATransitions = "T_EinsteinA"
    ColumnNameForFreqErrTransitions = "T_Uncertainty"
    ColumnNameForELowTransitions = "T_EnergyLower"
    ColumnNameForgUpTransitions = "T_UpperStateDegeneracy"
    ColumnNameForQNUpLabelTransitions = "T_UpperStateQuantumNumbers"
    ColumnNameForQNLowLabelTransitions = "T_LowerStateQuantumNumbers"
    ColumnNameForURLTransitions = "T_URL"

    query_string = "SELECT "
    query_string += ColumnNameForNameTransitions + ", "                                     ## position 1: molecule name
    query_string += ColumnNameForFreqTransitions + ", "                                     ## position 2: frequency
    query_string += ColumnNameForIntTransitions + ", "                                      ## position 3: intensity
    query_string += ColumnNameForEinsteinATransitions + ", "                                ## position 4: Einstein A
    query_string += ColumnNameForFreqErrTransitions + ", "                                  ## position 5: Error frequency
    query_string += ColumnNameForELowTransitions + ", "                                     ## position 6: E_low
    query_string += ColumnNameForgUpTransitions + ", "                                      ## position 7: upper state degeneracy
    query_string += ColumnNameForQNUpLabelTransitions + ", "                                ## position 8: quantum number label for upper state
    query_string += ColumnNameForQNLowLabelTransitions                                      ## position 9: quantum number label for lower state
    # query_string += ColumnNameForURLTransitions                                             ## position 10: URL
    query_string += " FROM " + NameOfRadTransTable
    query_string += " WHERE (" + ColumnNameForFreqTransitions + " >= " + str(FreqMin) + " and " + ColumnNameForFreqTransitions + " <= " + str(FreqMax)
    h = ElowMin / 1.42879
    query_string += " and " + ColumnNameForELowTransitions + " >= " + str(h)
    h = ElowMax / 1.42879
    query_string += " and " + ColumnNameForELowTransitions + " <= " + str(h)
    if (len(ListOfSelectedMolecules) > 0 and SelectMolecule != ["None"] and SelectMolecule != [""]):
        query_string += " and ("
        counter = 0
        for molecule in ListOfSelectedMolecules:


            ## check, if name of molecule contains invalid character
            i = molecule.find(chr(34))
            if (i > (-1)):
                molecule.translate(None, chr(34))

            counter += 1
            if (counter > 1):
                query_string += " or "
            query_string += ColumnNameForNameTransitions + " = " + chr(34) + molecule.strip() + chr(34)
        query_string += ")"
    query_string += ")"
    query_string += " ORDER by " + ColumnNameForFreqTransitions

    # Debug:
    # print " "
    # print "SelectMolecule = ", SelectMolecule
    # print "ListOfSelectedMolecules = ", ListOfSelectedMolecules
    # print "query_string = ", query_string
    # print "len(query_string) = ", len(query_string)


    ## read data from database
    cursor = conn.cursor()
    cursor.execute(query_string)
    rows = cursor.fetchall()
    conn.close()


    ## finished
    if (printFlag != "false"):
        print "done!"


    ## open file if file name is defined
    file_flag = "false"
    formatted_line = "%25s" % "Name:"
    formatted_line += "%37s" % "quantum number upper state:"
    formatted_line += "%37s" % "quantum number lower state:"
    formatted_line += "%18s" % "Frequency [MHz]:"
    formatted_line += "%24s" % "Error Frequency [MHz]:"
    formatted_line += "%27s" % "Einstein A coefficient:"
    #formatted_line += "%27s" % "log(Intensity) [nm2 MHz]:"
    formatted_line += "%20s" % "Energy_low [K]:"
    formatted_line += "  %23s" % "upper state degeneracy:"
    if (OutputDevice.strip() != "" and OutputDevice.strip() != None and printFlag == "true"):
        file_flag = "true"
        out = open(OutputDevice.strip(), 'w')
        out.write(formatted_line + "\n")
        if (printFlag != "false"):
            print "Reading contents of database ..",
    else:
        if (printFlag != "false"):
            print "\n\nContents of the database between %s MHz and %s MHz:\n" % (str(FreqMin), str(FreqMax))
            print formatted_line


    ## store entries in output variable Contents
    Contents = []
    col_counter = 0
    for row in rows:                                                                        ## loop over all entries of the database
        col_counter += 1
        # Contents.append(row)

        # Debug:
        # print "col_counter, row = ", col_counter, row


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define formatted line
        ## position 1: molecule name
        ## position 2: frequency
        ## position 3: intensity
        ## position 4: Einstein A
        ## position 5: Error frequency
        ## position 6: E_low
        ## position 7: upper state degeneracy
        ## position 8: quantum number label for upper state
        ## position 9: quantum number label for lower state
        ## position 10: URL
        formatted_line = ""
        name = ""
        freq = ""
        unfreq = ""
        intensity = ""
        EinsteinA = ""
        elow = ""
        gUp = ""
        QNUpLabel = ""
        QNLowLabel = ""
        DBurl = ""
        element_counter = 0
        for elements in row:
            element_counter += 1
            if (element_counter == 1):                                                      ## get name of molecule
                name = elements.strip()
                if (OutputDevice.strip() == "quiet"):
                    name = '%-45s' % name                                                   ## name of molecule
                else:
                    name = '%25s' % name[:25]                                               ## name of molecule has max. 25 characters

            elif (element_counter == 2):                                                    ## get frequency (in MHz)
                freqVal = '%17s' % elements
                if (freqVal.strip() != "None"):
                    freqVal = float(freqVal)
                    freq = '%17.5f' % freqVal
                else:
                    freq = freqVal

                # Debug:
                # print "freq = ", freq

            elif (element_counter == 3):                                                    ## get intensity
                intensityVal = '%26s' % elements
                if (intensityVal.strip() != "None"):
                    intensityVal = float(intensityVal)
                    intensity = '%26.9e' % intensityVal
                else:
                    intensity = intensityVal

                # Debug:
                # print "intensity = ", intensity

            elif (element_counter == 4):                                                    ## get Einstein A coefficient
                EinsteinAVal = '%26s' % elements
                if (EinsteinAVal.strip() != "None"):
                    EinsteinAVal = float(EinsteinAVal)
                    EinsteinA = '%26.3e' % EinsteinAVal
                else:
                    EinsteinA = EinsteinAVal

                # Debug:
                # print "EinsteinA = ", EinsteinA

            elif (element_counter == 5):                                                    ## get uncertainty of frequency (in MHz)
                unfreqVal = '%23s' % elements
                if (unfreqVal.strip() != "None"):
                    unfreqVal = float(unfreqVal)
                    unfreq = '%23.4e' % unfreqVal
                else:
                    unfreq = unfreqVal

                # Debug:
                # print "unfreq = ", unfreq

            elif (element_counter == 6):                                                    ## get E_lower (in cm-1)
                elowVal = '%19s' % elements
                if (elowVal.strip() != "None"):
                    elowVal = float(elowVal) * 1.438769                                     ## convert cm-1 to K
                    if (abs(elowVal) > 0.01 and abs(elowVal) < 10000):
                        elow = '%19.3f' % elowVal
                    else:
                        elow = '%19.3e' % elowVal
                else:
                    elow = elowVal

                # Debug:
                # print "elow = ", elow

            elif (element_counter == 7):                                                    ## get upper state degeneracy
                gUpVal = '%23s' % elements
                if (gUpVal.strip() != "None"):
                    gUpVal = float(gUpVal)
                    gUp = '%25.1e' % gUpVal
                else:
                    gUp = gUpVal

                # Debug:
                # print "gUp = ", gUp

            elif (element_counter == 8):                                                    ## get quantum number label for upper state
                try:
                    lab = elements
                    lab = lab.replace("ElecStateLabel = ", "").strip()
                    lab = lab.replace(" ", "").strip()                                      ## remove blank characters
                    if (len(lab) > 34):
                        lab = lab[:31] + "..."
                    QNUpLabel = '  %34s' % lab
                except TypeError:
                    QNUpLabel = "     "

                # Debug:
                # print "QNUpLabel = ", QNUpLabel

            elif (element_counter == 9):                                                    ## get quantum number label for lower state
                try:
                    lab = elements
                    lab = lab.replace("ElecStateLabel = ", "").strip()
                    lab = lab.replace(" ", "").strip()                                      ## remove blank characters
                    if (len(lab) > 34):
                        lab = lab[:31] + "..."
                    QNLowLabel = '  %34s' % lab
                except TypeError:
                    QNLowLabel = "     "

                # Debug:
                # print "QNLowLabel = ", QNLowLabel

            #elif (element_counter == 10):                                                   ## get URL
            #    try:
            #        DBurl = '%s' % elements
            #    except TypeError:
            #        DBurl = "     "

                # Debug:
                # print "DBurl = ", DBurl


        ## construct line for output
        formatted_line = name + " " + QNUpLabel + " " + QNLowLabel + " " + freq + " " + unfreq + " " + EinsteinA + " " + elow + " " + gUp + " " + DBurl
        Contents.append(formatted_line)

        # Debug:
        # print "formatted_line = ", formatted_line


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## if file name is defined
        if (file_flag == "true"):
            out.write(formatted_line + "\n")
        else:
            if (printFlag != "false"):
                print formatted_line


    ## close file, if file name is defined
    if (file_flag == "true"):
        out.close()
        if (printFlag != "false"):
            print "done!"


    ##====================================================================================================================================================
    ## define return variables
    return Contents
##--------------------------------------------------------------------------------------------------------------------------------------------------------

