#!/usr/bin/python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module identifies lines in the given frequency range
##  Copyright (C) 2012 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function MoleculeFileName:                    get file name of a molecule
##      - function GetTransFreq:                        get all transition energies from database
##      - function DefineNewFreqRanges:                 define new frequency ranges
##      - function CreateXMLFile:                       create new observational xml file for each molecule
##      - function ShrinkFreqRanges:                    shrink freq ranges defined in the exp. xml file to reduce computational effort
##      - function AnalyzeExperimentalDataParameter:    analyze the experimentalData parameter
##      - function AdjustExpXMLFile:                    adjust paths in an MAGIX xml file
##      - function PreprocessSpectrum:                  preprocess spectrum, i.e. normalize spectrum etc.
##      - function ReadDatabase:                        get information for all molecules between FreqMin and FreqMax from sqlite3 database
##      - function CreateMolfitFile:                    creates molfit file for the current molecule, and writes create molfits file to current molecule
##                                                      subdirectory
##      - function PrepareMAGIXExpXMLFile:              copy user defined MAGIX exp. xml file to current single molecule fit directory and adopt paths
##      - function GetBestResult:                       determines the best fit result for a molecule
##      - function CheckContribution:                   check, if molecule contributes to current spectrum or not
##      - function SingleMoleculeFit:                   start a single molecule fit
##      - function StartWorker:                         starts worker for cluster fits via command line
##      - function StrongLinesFits:                     start single molecule fits related to strong lines
##      - function StartClusterFits:                    start different molecule fits on the cluster
##      - function CreateMolfitFileInNewFormat:         create a new molfit file in the new format
##      - function LineIdentification:                  function identifies the lines in the given frequency range
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    08.05.2014   initial version
##  T. Moeller    14.04.2015   improved version (add new method: redefine freq. ranges for single molecule fits)
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import string                                                                               ## import string package
import random                                                                               ## import random package
import copy                                                                                 ## import random copy
import datetime                                                                             ## import datetime package
import time                                                                                 ## import time package
import math                                                                                 ## import math package
import warnings                                                                             ## import warnings package
import matplotlib                                                                           ## import matplotlib package
matplotlib.use("Agg")                                                                       ## switch backend
import pylab                                                                                ## import pylab package
import task_LoadASCIIFile                                                                   ## import package LoadASCIIFile
import task_myXCLASSPlot                                                                    ## import package myXCLASSPlot
import task_ListDatabase                                                                    ## import package ListDatabase
import task_MAGIX                                                                           ## import package MAGIX
import task_myXCLASS                                                                        ## import package myXCLASS
import task_myXCLASSFit                                                                     ## import package myXCLASSFit
import task_myXCLASSMapFit                                                                  ## import package myXCLASSMapFit
import sqlite3                                                                              ## import sqlite3 package
import threading                                                                            ## import threading package
import multiprocessing                                                                      ## import multiprocessing package
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get file name of a molecule
##
def MoleculeFileName(molecule):
    """

input parameters:
-----------------

    - molecule:             name of molecule


output parameters:
------------------

    - MoleculeFileName:     corresponding molecule file name

    """

    ## initialize output parameter
    MoleculeFileName = ""


    ## remove not allowed characters from the molecule name
    MoleculeFileName = molecule
    MoleculeFileName = MoleculeFileName.replace(";", "_")
    MoleculeFileName = MoleculeFileName.replace(",", "_")
    MoleculeFileName = MoleculeFileName.replace("'", "_")
    MoleculeFileName = MoleculeFileName.replace("(", "_")
    MoleculeFileName = MoleculeFileName.replace(")", "_")


    ## define return parameter
    return MoleculeFileName
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get all transition energies from database
##
def GetTransFreq(MolName, FreqMin, FreqMax, ElowMax, dbFile, MaxNumTransitionsSQL):
    """

input parameters:
-----------------

    - MolName:                  name of molecule

    - FreqMin:                  min. freq. of range

    - FreqMax:                  max. freq. of range

    - ElowMax:                  max. lower energy

    - dbFile:                   path and name of database file

    - MaxNumTransitionsSQL:     max. number of transitions


output parameters:
------------------

    - TransFreqList:            list containing the transition frequencies

    - DBParamList:              list containing the all important transition parameters for a each transition within FreqMin and FreqMax

    """

    # Debug:
    #    print "MolName = ", MolName
    #    print "FreqMin = ", FreqMin
    #    print "FreqMax = ", FreqMax


    ## check, if database file is defined
    if (dbFile.strip() == ""):
        print "\n\nError in function LineIdentification.GetTransFreq:"
        print "\t\tThere is no database file defined!"
        print "\n\tP A N I C!!!!"
        sys.exit(1) 


    ## establish connection to database
    try:
        conn = sqlite3.connect(dbFile)
    except sqlite3.Error, e:
        print "\nErrot in function LineIdentification.GetTransFreq:"
        print "\t\tCan not connect to sqlite3 databse %s." % dbFile
        print "\t\tError: %d: %s" % (e.args[0], e.args[1])
        sys.exit(1)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get entries from database


    ## check, if name of molecule contains invalid character
    i = MolName.find(chr(34))
    if (i > (-1)):
        MolName.translate(None, chr(34))


    ## define query string
    ##
    ## get names of columns with
    ##      import sqlite3
    ##      connection = sqlite3.connect('cdms_sqlite.db')
    ##      cursor = connection.execute('select * from partitionfunctions')
    ##      names = list(map(lambda x: x[0], cursor.description))
    ##
    NameOfRadTransTable = "transitions"
    ColumnNameForNameTransitions = "T_Name"
    ColumnNameForFreqTransitions = "T_Frequency"
    ColumnNameForIntTransitions = "T_Intensity"
    ColumnNameForEinsteinATransitions = "T_EinsteinA"
    ColumnNameForFreqErrTransitions = "T_Uncertainty"
    ColumnNameForELowTransitions = "T_EnergyLower"
    ColumnNameForgUpTransitions = "T_UpperStateDegeneracy"
    ColumnNameForQNUpLabelTransitions = "T_UpperStateQuantumNumbers"
    ColumnNameForQNLowLabelTransitions = "T_LowerStateQuantumNumbers"
    ColumnNameForURLTransitions = "T_URL"

    # MaxNumTransitionsSQL = 0
    if (MaxNumTransitionsSQL > 0 and MaxNumTransitionsSQL != 1.e99):
        query_string = "SELECT * FROM (SELECT "
    else:
        query_string = "SELECT "
    query_string += ColumnNameForNameTransitions + ", "                                     ## position 1: molecule name
    query_string += ColumnNameForFreqTransitions + ", "                                     ## position 2: frequency
    query_string += ColumnNameForIntTransitions + ", "                                      ## position 3: intensity
    query_string += ColumnNameForEinsteinATransitions + ", "                                ## position 4: Einstein A
    query_string += ColumnNameForFreqErrTransitions + ", "                                  ## position 5: Error frequency
    query_string += ColumnNameForELowTransitions + ", "                                     ## position 6: E_low
    query_string += ColumnNameForgUpTransitions + ", "                                      ## position 7: upper state degeneracy
    query_string += ColumnNameForQNUpLabelTransitions + ", "                                ## position 8: quantum number label for upper state
    query_string += ColumnNameForQNLowLabelTransitions                                      ## position 9: quantum number label for lower state
    ElowMin = 0.0
    query_string += " FROM " + NameOfRadTransTable
    query_string += " WHERE (" + ColumnNameForFreqTransitions + " >= " + str(FreqMin) + " and " + ColumnNameForFreqTransitions + " <= " + str(FreqMax)
    h = ElowMin / 1.42879
    query_string += " and " + ColumnNameForELowTransitions + " >= " + str(h)
    h = ElowMax / 1.42879
    query_string += " and " + ColumnNameForELowTransitions + " <= " + str(h) + " and ("
    query_string += ColumnNameForNameTransitions + " = " + chr(34) + MolName.strip() + chr(34) + "))"
    if (MaxNumTransitionsSQL > 0 and MaxNumTransitionsSQL != 1.e99):
        query_string += " ORDER BY " + ColumnNameForEinsteinATransitions + " * " + ColumnNameForgUpTransitions + " DESC limit " \
                                     + str(MaxNumTransitionsSQL) + ")"
    query_string += " ORDER BY " + ColumnNameForFreqTransitions

    # Debug:
    #    print "\n\n\n\ndbFile = ", dbFile
    #    print "query_string = ", query_string
    #    print "len(query_string) = ", len(query_string)
    #    sys.exit(0)

    ## read data from database
    cursor = conn.cursor()
    cursor.execute(query_string)
    rows = cursor.fetchall()
    conn.close()


    ## store entries in output variable Contents
    TransFreqList = []
    DBParamList = []
    for row in rows:                                                                        ## loop over all entries of the database


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define formatted line
        ## position 1: molecule name
        ## position 2: frequency
        ## position 3: intensity
        ## position 4: Einstein A
        ## position 5: Error frequency
        ## position 6: E_low
        ## position 7: upper state degeneracy
        ## position 8: quantum number label for upper state
        ## position 9: quantum number label for lower state
        ## position 10: URL
        FreqString = ""
        EinsteinAString = ""
        ElowString = ""
        gUpString = ""
        element_counter = 0
        for elements in row:
            element_counter += 1
            if (element_counter == 2):                                                      ## get frequency (in MHz)
                freqVal = '%17s' % elements
                if (freqVal.strip() != "None"):
                    freqVal = float(freqVal)
                    FreqString = '%17.5f' % freqVal
                else:
                    FreqString = freqVal
                TransFreq = float(FreqString)

                # Debug:
                # print "FreqString = ", FreqString

            elif (element_counter == 4):                                                    ## get Einstein A coefficient
                EinsteinAVal = '%26s' % elements
                if (EinsteinAVal.strip() != "None"):
                    EinsteinAVal = float(EinsteinAVal)
                    EinsteinAString = '%26.3e' % EinsteinAVal
                else:
                    EinsteinAString = EinsteinAVal
                EinsteinA = float(EinsteinAString)

                # Debug:
                # print "EinsteinAString = ", EinsteinAString

            elif (element_counter == 6):                                                    ## get E_lower (in cm-1)
                elowVal = '%19s' % elements
                if (elowVal.strip() != "None"):
                    elowVal = float(elowVal) * 1.438769                                     ## convert cm-1 to K
                    if (abs(elowVal) > 0.01 and abs(elowVal) < 10000):
                        ElowString = '%19.3f' % elowVal
                    else:
                        ElowString = '%19.3e' % elowVal
                else:
                    ElowString = elowVal
                ElowMin = float(ElowString)

                # Debug:
                # print "ElowString = ", ElowString

            elif (element_counter == 7):                                                    ## get upper state degeneracy
                gUpVal = '%23s' % elements
                if (gUpVal.strip() != "None"):
                    gUpVal = float(gUpVal)
                    gUpString = '%25.1e' % gUpVal
                else:
                    gUpString = gUpVal
                gup = float(gUpString)

                # Debug:
                # print "gUpString = ", gUpString


        ## check, if transition frequencies have enough distance
        IncludeFlag = False
        if (TransFreqList == []):
            IncludeFlag = True
        else:
            NPTransFreqList = numpy.asarray(TransFreqList)
            TransFreqDistance = numpy.min(numpy.abs(NPTransFreqList - TransFreq))
            if (TransFreqDistance > 2.0):
                IncludeFlag = True
            ## to do: if two transition frequencies are located to close together, special handling is necessary
            # else:

        if (IncludeFlag):
            TransFreqList.append(TransFreq)
            DBParamList.append([TransFreq, EinsteinA, ElowMin, gup])

    # Debug:
    #    print "\n\nMolName = ", MolName
    #    print "FreqMin, FreqMax = ", FreqMin, FreqMax
    #    print "ElowMin, ElowMax = ", ElowMin, ElowMax
    #    print "SelectMolecule = ", SelectMolecule
    #    print "OutputDevice = ", OutputDevice
    #    print "TransFreqList = ", TransFreqList
    #    print "\n\n\n"


    ## define return parameter
    return (TransFreqList, DBParamList)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## define new frequency ranges
##
##
## to do: extend function to handle list of molecules MolName -> [Mol1, Mol2, ...]
##
##
def DefineNewFreqRanges(LowestDataPoint, HighestDataPoint, MolName, FreqMin, FreqMax, FreqStep, t_back_flag, tBack, tSlope, nH_flag, N_H, beta_dust, \
                        kappa_1300, Noise, velLowLimit, velUpLimit, MaxWidth, ElowMax, dbFile, LocalMaxNumTransInFit):
    """

input parameters:
-----------------

    - LowestDataPoint:          lowest frequency of the current exp. data file

    - HighestDataPoint:         highest frequency of the current exp. data file

    - MolName:                  name of current molecule

    - FreqMin:                  lowest frequency for current frequency range

    - FreqMax:                  highest frequency for current frequency range

    - FreqStep:                 step size for current frequency range

    - t_back_flag:              flag indicating if continuum is described completely by tBack and tSlope

    - tBack:                    background temperature for current frequency range

    - tSlope:                   temperature slope for current frequency range

    - nH_flag:                  flag indicating if dust parameters are defined globally or not

    - N_H:                      hydrogen column density for current frequency range

    - beta_dust:                spectral index for current frequency range

    - kappa_1300:               kappa for current frequency range

    - Noise:                    noise level for current frequency range

    - velLowLimit:              mini. lower limit of velocity offset for current molecule

    - velUpLimit:               max. upper limit of velocity offset for current molecule

    - MaxWidth:                 max. velocity width for current molecule

    - ElowMax:                  highest allowed lower energy of a transition

    - dbFile:                   path and name of database file

    - LocalMaxNumTransInFit:    local max. number of transitions which are taken into account


output parameters:
------------------

    - NewFreqRanges:            list containing all parameters suitable for new freq. ranges

    """

    # Debug:
    #    print "LowestDataPoint = ", LowestDataPoint
    #    print "HighestDataPoint = ", HighestDataPoint
    #    print "MolName = ", MolName
    #    print "FreqMin = ", FreqMin
    #    print "FreqMax = ", FreqMax
    #    print "FreqStep = ", FreqStep
    #    print "t_back_flag = ", t_back_flag
    #    print "tBack = ", tBack
    #    print "tSlope = ", tSlope
    #    print "nH_flag = ", nH_flag
    #    print "N_H = ", N_H
    #    print "beta_dust = ", beta_dust
    #    print "kappa_1300 = ", kappa_1300
    #    print "Noise = ", Noise
    #    print "velLowLimit = ", velLowLimit
    #    print "velUpLimit, ", velUpLimit
    #    print "MaxWidth = ", MaxWidth


    ## reset output parameter
    NewFreqRanges = []


    ## define speed of light (m/s) and (km/s)
    cms = 299792458.0
    ckms = cms * 1.e-3
    LineWidthNr = 1.0
    SimSize = int((FreqMax - FreqMin)/FreqStep) + 1


    ## get transition frequencies in current frequency range
    TransFreqList, DBParamList = GetTransFreq(MolName, FreqMin, FreqMax, ElowMax, dbFile, LocalMaxNumTransInFit)

    # Debug:
    #    print "TransFreqList = ", TransFreqList
    #    print "MaxWidth = ", MaxWidth
    #    print "MolName = ", MolName
    #    print "FreqMin = ", FreqMin
    #    print "FreqMax = ", FreqMax
    #    print "ElowMax = ", ElowMax, "\n\n"


    ## determine ranges
    if (TransFreqList != []):


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## determine new frequency ranges
        FreqRangeList = []
        MultiplicityOfWidth = 1.0                                                           ## set multiplity of half of velocity width
        for TransIndex in xrange(len(TransFreqList)):                                       ## loop over all transition frequencies
            if (TransIndex <= LocalMaxNumTransInFit):
                TransFreq = TransFreqList[TransIndex]


                ## get min/max of lower/upper limits
                v1 = velLowLimit - MultiplicityOfWidth * (MaxWidth / 2.0)
                v2 = velUpLimit + MultiplicityOfWidth * (MaxWidth / 2.0)
                r1 = TransFreq * (1.0 - v1 / ckms)
                r2 = TransFreq * (1.0 - v2 / ckms)
                LowFreq = max(min(r1, r2), FreqMin)
                HighFreq = min(max(r1, r2), FreqMax)

                # Debug:
                #    print "\nTransFreq = ", TransFreq
                #    print "velLowLimit = ", velLowLimit
                #    print "velUpLimit = ", velUpLimit
                #    print "v1 = ", v1
                #    print "v2 = ", v2
                #    print "LowFreq = ", LowFreq
                #    print "HighFreq = ", HighFreq


                ## make sure, that non-shifted transition frequency is always included in new freq. range
                if (TransFreq < LowFreq):
                    LowFreq = max((TransFreq - 10.0), FreqMin)
                if (TransFreq > HighFreq):
                    HighFreq = min((TransFreq + 10.0), FreqMax)


                ## check, if new frequency ranges are within frequency range defined in exp. data file
                if (LowestDataPoint > LowFreq):
                    LowFreq = LowestDataPoint
                if (HighestDataPoint < HighFreq):
                    HighFreq = HighestDataPoint
                if (LowFreq >= HighFreq):
                    AddNewRangeFlag = "false"
                else:

                    # Debug:
                    #    print "MaxWidth = ", MaxWidth
                    #    print "LowFreq = ", LowFreq
                    #    print "HighFreq = ", HighFreq
                    #    print "\n\n\n"


                    ## check, if part of current freq. range is already covered by a previous range
                    AddNewRangeFlag = "true"
                    for i in xrange(len(FreqRangeList)):
                        lf = FreqRangeList[i][0]
                        uf = FreqRangeList[i][1]


                        ## check case:  overlapping range definitons
                        ##  lf|                 |uf
                        ##          LowFreq|            |HighFreq
                        if (LowFreq <= uf and uf <= HighFreq):
                            FreqRangeList[i][1] = HighFreq
                            AddNewRangeFlag = "false"
                            break


                        ## check case:  overlapping range definitons
                        ##  lf|                             |uf
                        ##          LowFreq|    |HighFreq
                        elif (lf <= LowFreq and HighFreq <= uf):
                            AddNewRangeFlag = "false"
                            break


                        ## check case:  overlapping range definitons
                        ##          lf|                     |uf
                        ##  LowFreq|            |HighFreq
                        elif (LowFreq <= lf and HighFreq <= uf):
                            FreqRangeList[i][0] = LowFreq
                            AddNewRangeFlag = "false"
                            break


                ## add new range if range is not covered before
                if (AddNewRangeFlag == "true"):
                    if (abs(HighFreq - LowFreq) >= 2.0 * FreqStep):                             ## new range contains at least two data points
                        FreqRangeList.append([LowFreq, HighFreq, TransFreq])

                        # Debug:
                        # print "TransFreq, LowFreq, HighFreq = ", TransFreq, LowFreq, HighFreq


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## construct new ranges
        for NewRange in FreqRangeList:
            LowFreq = NewRange[0]
            HighFreq = NewRange[1]
            TransFreq = NewRange[2]

            # Debug:
            # print "(HighFreq - LowFreq) / FreqStep = ", (HighFreq - LowFreq) / FreqStep


            ## determine new background temperature
            NewTBack = tBack * (LowFreq / FreqMin)**tSlope


            ## add parameters to NewFreqRanges list
            NewFreqRanges.append([str(LowFreq), str(HighFreq), str(FreqStep), str(t_back_flag), str(NewTBack), str(tSlope), str(nH_flag), str(N_H), \
                                  str(beta_dust), str(kappa_1300), str(Noise), str(TransFreq)])

    # Debug:
    #    print "len(NewFreqRanges) = ", len(NewFreqRanges)
    #    print "NewFreqRanges = ", NewFreqRanges


    ## define return parameter
    return NewFreqRanges
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## create new observational xml file for each molecule
##
##
## to do: extend function to handle list of molecules MolName -> [Mol1, Mol2, ...]
##
##
def CreateXMLFile(MolName, velLowLimit, velUpLimit, MaxWidth, ExpXML, ExpFileList, FreqMinList, FreqMaxList, FreqStepList, t_back_flagList, tBackList, \
                  tSlopeList, N_HList, beta_dustList, kappa_1300List, NoiseList, GlobalvLSRList, TelescopeSizeList, InterFlagList, ErrorYList, \
                  NumberHeaderLinesList, SeparatorColumnsList, IsotopologuesList, IsoTableFileNameList, dbList, ElowMax, LocalMaxNumTransInFit):
    """

input parameters:
-----------------

    - MolName:                  name of current molecule

    - velLowLimit:              lower velocity limit

    - velUpLimit:               upper velocity limit

    - MaxWidth:                 max. velocity width

    - ExpXML:                   path and name of exp xml file

    - ExpFileList:              list of observational data files

    - FreqMinList:              list of lowest frequency for each range

    - FreqMaxList:              list of highest frequency for each range

    - FreqStepList:             list of step sizes for each range

    - t_back_flagList:          list of all T_back flags

    - tBackList:                list of background temperatures

    - tSlopeList:               list of temperature slopes

    - N_HList:                  list of hydrogen column densitites

    - beta_dustList:            list of spectral indices

    - kappa_1300List:           list of kappa values

    - NoiseList:                list of noise levels

    - GlobalvLSRList:           list of vLSR values

    - TelescopeSizeList:        list of telescope sizes

    - InterFlagList:            list of interferometer flags

    - ErrorYList:               list error flags

    - NumberHeaderLinesList:    list of numbers of header lines

    - SeparatorColumnsList:     list of separator characters

    - IsotopologuesList:        iso flag

    - IsoTableFileNameList:     path and name of iso flag

    - dbList:                   path and name of database file

    - ElowMax:                  highest allowed lower energy of a transition

    - LocalMaxNumTransInFit:    local max. number of transitions which are taken into account


output parameters:
------------------

    - None

    """


    # Debug:
    #    print "MolName = ", MolName


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create new xml file for current xml file
    NumExpDataFiles = len(ExpFileList)
    NewExpXMLFile = open(ExpXML, 'w')

    # Debug:
    #    print "\n\nExpXML = ", ExpXML
    #    print "LocalMaxNumTransInFit = ", LocalMaxNumTransInFit


    ## write header of xml file
    NewExpXMLFile.write("<?xml version=" + chr(34) + "1.0" + chr(34) + " encoding=" + chr(34) + "UTF-8" + chr(34) + "?>\n")
    NewExpXMLFile.write("<ExpFiles>\n\n\n")
    NewExpXMLFile.write("    <!-- define number of observation files -->\n")


    ## write parameters for each observational data file
    CounterTransitions = abs(LocalMaxNumTransInFit)
    EffectiveObsDataFileCounter = 0
    LinesForXMLFile = []
    for i in xrange(NumExpDataFiles):                                                       ## loop over all exp. data files
        helpString = "*" * 141
        LinesForObsFile = ["    <!-- " + helpString + " -->\n"]
        LinesForObsFile.append("    <!-- define parameters for observation file " + chr(34) + os.path.basename(ExpFileList[i]) + chr(34) + " -->\n")
        LinesForObsFile.append("    <file>\n")
        LinesForObsFile.append("        <FileNamesExpFiles>" + ExpFileList[i] + "</FileNamesExpFiles>\n")
        LinesForObsFile.append("        <ImportFilter>xclassASCII</ImportFilter>\n\n\n")

        data = numpy.loadtxt(ExpFileList[i])
        LowestDataPoint = min(data[:,0])
        HighestDataPoint = max(data[:,0])

        # Debug:
        # print "LowestDataPoint = ", LowestDataPoint
        # print "HighestDataPoint = ", HighestDataPoint


        ## determine new frequency range definitions and write to xml file
        TotalNumberFreqRanges = 0
        for j in xrange(len(FreqMinList)):
            DataFileIndex = int(FreqMinList[j][0])
            if (DataFileIndex == (i + 1)):
                FreqMin = float(FreqMinList[j][1])
                FreqMax = float(FreqMaxList[j][1])
                FreqStep = float(FreqStepList[j][1])
                if (tBackList[j][1] != []):
                    t_back_flag = t_back_flagList[0].lower()
                    if (t_back_flag == "true"):
                        t_back_flag = True
                    else:
                        t_back_flag = False
                    tBack = float(tBackList[j][1])
                    tSlope = float(tSlopeList[j][1])
                else:
                    t_back_flag = False
                    tBack = 0.0
                    tSlope = 0.0
                if (N_HList[j][1] != []):
                    nH_flag = True
                    N_H = float(N_HList[j][1])
                    beta_dust = float(beta_dustList[j][1])
                    kappa_1300 = float(kappa_1300List[j][1])
                else:
                    nH_flag = False
                    N_H = 0.0
                    beta_dust = 0.0
                    kappa_1300 = 0.0
                try:
                    Noise = float(NoiseList[j][1])
                except IndexError:
                    Noise = 0.0


                ## define new parameter ranges
                dbFile = dbList[0]
                NewFreqRanges = DefineNewFreqRanges(LowestDataPoint, HighestDataPoint, MolName, FreqMin, FreqMax, FreqStep, t_back_flag, tBack, tSlope, \
                                                    nH_flag, N_H, beta_dust, kappa_1300, Noise, velLowLimit, velUpLimit, MaxWidth, ElowMax, dbFile, \
                                                    CounterTransitions)
                NumberFreqRanges = len(NewFreqRanges)
                TotalNumberFreqRanges += NumberFreqRanges
                CounterTransitions = CounterTransitions - NumberFreqRanges

                # Debug:
                #    print "\n\n>>>FreqMin = ", FreqMin
                #    print ">>>FreqMax = ", FreqMax
                #    print ">>>ElowMax = ", ElowMax
                #    print ">>>NumberFreqRanges = ", NumberFreqRanges
                #    print ">>>NewFreqRanges = ", NewFreqRanges


                ## write parameters for new freq range(s) to xml file
                for FreqRange in xrange(NumberFreqRanges):
                    MinExpRange = NewFreqRanges[FreqRange][0]
                    MaxExpRange = NewFreqRanges[FreqRange][1]
                    StepFrequency = NewFreqRanges[FreqRange][2]
                    t_back_flag = NewFreqRanges[FreqRange][3]
                    BackgroundTemperature = NewFreqRanges[FreqRange][4]
                    TemperatureSlope = NewFreqRanges[FreqRange][5]
                    nH_flag = NewFreqRanges[FreqRange][6]
                    HydrogenColumnDensity = NewFreqRanges[FreqRange][7]
                    DustBeta = NewFreqRanges[FreqRange][8]
                    Kappa = NewFreqRanges[FreqRange][9]
                    NoiseLevel = NewFreqRanges[FreqRange][10]


                    ## write to xml file
                    LinesForObsFile.append("        <!-- define parameters for each data ranges -->\n")
                    LinesForObsFile.append("        <FrequencyRange>\n")
                    LinesForObsFile.append("            <MinExpRange>" + MinExpRange + "</MinExpRange>\n")
                    LinesForObsFile.append("            <MaxExpRange>" + MaxExpRange + "</MaxExpRange>\n")
                    LinesForObsFile.append("            <StepFrequency>" + StepFrequency + "</StepFrequency>\n\n\n")
                    LinesForObsFile.append("            <!-- define background temperature and temperature slope -->\n")
                    LinesForObsFile.append("            <t_back_flag>" + t_back_flag + "</t_back_flag>\n")
                    LinesForObsFile.append("            <BackgroundTemperature>" + BackgroundTemperature + "</BackgroundTemperature>\n")
                    LinesForObsFile.append("            <TemperatureSlope>" + TemperatureSlope + "</TemperatureSlope>\n\n\n")
                    if (nH_flag == "True"):
                        LinesForObsFile.append("            <!-- define hydrogen column density, beta for dust, and kappa -->\n")
                        LinesForObsFile.append("            <HydrogenColumnDensity>" + HydrogenColumnDensity + "</HydrogenColumnDensity>\n")
                        LinesForObsFile.append("            <DustBeta>" + DustBeta + "</DustBeta>\n")
                        LinesForObsFile.append("            <Kappa>" + Kappa + "</Kappa>\n")
                    if (NoiseList != []):
                        LinesForObsFile.append("\n\n            <!-- define noise level for current frequency range -->\n")
                        LinesForObsFile.append("            <NoiseLevel>" + NoiseLevel + "</NoiseLevel>\n")
                    LinesForObsFile.append("        </FrequencyRange>\n\n\n")


        ## add number of frequency ranges
        LinesForObsFile.append("        <!-- define number of data ranges -->\n")
        LinesForObsFile.append("        <NumberExpRanges>" + str(TotalNumberFreqRanges) + "</NumberExpRanges>\n\n\n")


        ## write rest settings of frequency range definition
        LinesForObsFile.append("        <!-- define if interferrometric observation is modeled -->\n")
        LinesForObsFile.append("        <Inter_Flag>" + InterFlagList[i] + "</Inter_Flag>\n\n\n")
        LinesForObsFile.append("        <!-- define size of telescope -->\n")
        LinesForObsFile.append("        <TelescopeSize>" + TelescopeSizeList[i] + "</TelescopeSize>\n\n\n")
        LinesForObsFile.append("        <!-- define local standard of rest (vLSR) -->\n")
        if (GlobalvLSRList != []):
            LinesForObsFile.append("        <GlobalvLSR>" + GlobalvLSRList[i] + "</GlobalvLSR>\n\n\n")
        else:
            LinesForObsFile.append("        <GlobalvLSR>0.0</GlobalvLSR>\n\n\n")


        if (ErrorYList != [] or NumberHeaderLinesList != [] or SeparatorColumnsList != []):
            LinesForObsFile.append("        <!-- define parameters for ASCII file import -->\n")
        if (ErrorYList != []):
            DataFileIndex = int(ErrorYList[i][0])
            if (DataFileIndex == (i + 1)):
                LinesForObsFile.append("        <ErrorY>" + ErrorYList[i][1] + "</ErrorY>\n")
        if (NumberHeaderLinesList != []):
            DataFileIndex = int(NumberHeaderLinesList[i][0])
            if (DataFileIndex == (i + 1)):
                LinesForObsFile.append("        <NumberHeaderLines>" + NumberHeaderLinesList[i][1] + "</NumberHeaderLines>\n")
        if (SeparatorColumnsList != []):
            DataFileIndex = int(SeparatorColumnsList[i][0])
            if (DataFileIndex == (i + 1)):
                LinesForObsFile.append("        <SeparatorColumns>" + SeparatorColumnsList[i][1] + "</SeparatorColumns>\n")
        LinesForObsFile.append("    </file>\n\n\n")


        ## write parameters for current obs. data file to global output array, if at least one frequency array is defined
        if (TotalNumberFreqRanges > 0):
            EffectiveObsDataFileCounter += 1
            for line in LinesForObsFile:
                LinesForXMLFile.append(line)


    ## write effective total number of obs. data file and parameters for obs. data file(s) to xml file
    NewExpXMLFile.write("    <NumberExpFiles>" + str(EffectiveObsDataFileCounter)  + "</NumberExpFiles>\n\n\n")
    for line in LinesForXMLFile:
        NewExpXMLFile.write(line)


    ## write rest of xml file
    helpString = "*" * 141
    NewExpXMLFile.write("    <!-- " + helpString + " -->\n")
    NewExpXMLFile.write("    <!-- parameters for isotopologues -->\n")
    NewExpXMLFile.write("    <iso_flag>" + IsotopologuesList[0] + "</iso_flag>\n")
    if (IsoTableFileNameList != []):
        NewExpXMLFile.write("    <IsoTableFileName>" + IsoTableFileNameList[0] + "</IsoTableFileName>\n\n\n")
    if (dbList != []):
        NewExpXMLFile.write("    <!-- dbFilename -->\n")
        NewExpXMLFile.write("    <dbFilename>" + dbList[0] + "</dbFilename>\n")
    NewExpXMLFile.write("</ExpFiles>\n")
    NewExpXMLFile.close()


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## shrink freq ranges defined in the exp. xml file to reduce computational effort
##
def ShrinkFreqRanges(molecule, MolDirName, ExpXML, MolfitsFileName, ElowMax):
    """

input parameters:
-----------------

    - molecule:                 name of the current molecule

    - MolDirName:               the corresponding file name

    - ExpXML:                   path and name of exp. xml file

    - MolfitsFileName:          path and name of molfits file

    - ElowMax:                  highest allowed lower energy of a transition


output parameters:
------------------

    - None

    """


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read in some parameters from observational xml file
    ExpFileList = task_MAGIX.GetXMLtag(ExpXML, "FileNamesExpFiles")
    FreqMinList = task_MAGIX.GetXMLtag(ExpXML, "MinExpRange")
    FreqMaxList = task_MAGIX.GetXMLtag(ExpXML, "MaxExpRange")
    FreqStepList = task_MAGIX.GetXMLtag(ExpXML, "StepFrequency")
    t_back_flagList = task_MAGIX.GetXMLtag(ExpXML, "t_back_flag")
    tBackList = task_MAGIX.GetXMLtag(ExpXML, "BackgroundTemperature")
    tSlopeList = task_MAGIX.GetXMLtag(ExpXML, "TemperatureSlope")
    N_HList = task_MAGIX.GetXMLtag(ExpXML, "HydrogenColumnDensity")
    beta_dustList = task_MAGIX.GetXMLtag(ExpXML, "DustBeta")
    kappa_1300List = task_MAGIX.GetXMLtag(ExpXML, "Kappa")
    NoiseList = task_MAGIX.GetXMLtag(ExpXML, "NoiseLevel")
    GlobalvLSRList = task_MAGIX.GetXMLtag(ExpXML, "GlobalvLSR")
    TelescopeSizeList = task_MAGIX.GetXMLtag(ExpXML, "TelescopeSize")
    InterFlagList = task_MAGIX.GetXMLtag(ExpXML, "Inter_Flag")
    ErrorYList = task_MAGIX.GetXMLtag(ExpXML, "ErrorY")
    NumberHeaderLinesList = task_MAGIX.GetXMLtag(ExpXML, "NumberHeaderLines")
    SeparatorColumnsList = task_MAGIX.GetXMLtag(ExpXML, "SeparatorColumns")
    IsotopologuesList = task_MAGIX.GetXMLtag(ExpXML, "Isotopologues")
    if (IsotopologuesList == []):
        IsotopologuesList = task_MAGIX.GetXMLtag(ExpXML, "iso_flag")
    IsoTableFileNameList = task_MAGIX.GetXMLtag(ExpXML, "IsoTableFileName")
    dbList = task_MAGIX.GetXMLtag(ExpXML, "dbFilename")

    # Debug:
    #    print "ExpFileList = ", ExpFileList
    #    print "FreqMinList = ", FreqMinList
    #    print "FreqMaxList = ", FreqMaxList
    #    print "FreqStepList = ", FreqStepList
    #    print "t_back_flagList = ", t_back_flagList
    #    print "tBackList = ", tBackList
    #    print "tSlopeList = ", tSlopeList
    #    print "N_HList = ", N_HList
    #    print "beta_dustList = ", beta_dustList
    #    print "kappa_1300List = ", kappa_1300List
    #    print "NoiseList = ", NoiseList
    #    print "GlobalvLSRList = ", GlobalvLSRList
    #    print "TelescopeSizeList = ", TelescopeSizeList
    #    print "InterFlagList = ", InterFlagList
    #    print "ErrorYList = ", ErrorYList
    #    print "NumberHeaderLinesList = ", NumberHeaderLinesList
    #    print "SeparatorColumnsList = ", SeparatorColumnsList
    #    print "IsotopologuesList = ", IsotopologuesList
    #    print "IsoTableFileNameList = ", IsoTableFileNameList
    #    print "dbList = ", dbList


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get parameters from molfit file
    MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(MolfitsFileName)
    MolName = MoleculesInMolfitFile[0]
    LocalMolfitParameters = AllParameters[0]


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get velocitiy offset and width
    MaxWidth = -1.e99
    velLowLimit = 1.e99
    velUpLimit = -1.e99
    for line in LocalMolfitParameters:

        # Debug:
        # print "line = ", line

        for col in line:
            Name = col[0]
            flag = col[1]
            lowlimit = col[2]
            uplimit = col[3]
            element = col[4]

            # Debug:
            # print "col = ", col


            ## determine minimal lower and max. upper range for velocity
            if (Name == "V_off"):
                lowlimit = float(lowlimit)
                uplimit = float(uplimit)
                if (lowlimit < velLowLimit):
                    velLowLimit = lowlimit
                if (uplimit > velUpLimit):
                    velUpLimit = uplimit


            ## determine minimal an max. velocity offset
            elif (Name == "V_width"):
                element = float(element)
                if (MaxWidth < element):
                    MaxWidth = element

    # Debug:
    # print "velLowLimit = ", velLowLimit
    # print "velUpLimit = ", velUpLimit
    # print "MaxWidth = ", MaxWidth


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create new xml file for current molecule with shrinked frequency ranges
    LocalMaxNumTransInFit = 1.e99
    CreateXMLFile(molecule, velLowLimit, velUpLimit, MaxWidth, ExpXML, ExpFileList, FreqMinList, FreqMaxList, FreqStepList, t_back_flagList, tBackList, \
                  tSlopeList, N_HList, beta_dustList, kappa_1300List, NoiseList, GlobalvLSRList, TelescopeSizeList, InterFlagList, ErrorYList, \
                  NumberHeaderLinesList, SeparatorColumnsList, IsotopologuesList, IsoTableFileNameList, dbList, ElowMax, LocalMaxNumTransInFit)


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## analyze the experimentalData parameter
##
def AnalyzeExperimentalDataParameter(CurrentDir, experimentalData, Noise, vLSR, TelescopeSize, tBack, tSlope, N_H, beta_dust, kappa_1300):
    """

input parameters:
-----------------

    - CurrentDir:               current directory

    - experimentalData:         path and name of the MAGIX xml file or of an ASCII file containing the spectrum

    - Noise:                    noise level (user input)

    - vLSR:                     local standard of rest vLSR (user input)

    - TelescopeSize:            size of the telescope (user input)

    - tBack:                    background temperature (user input)

    - tSlope:                   temperature slope (user input)

    - N_H:                      hydrogen column density (user input)

    - beta_dust:                beta for dust (user input)

    - kappa_1300:               kappa (user input)


output parameters:
------------------

    - ListOfSpectraNames:       list containing the names for each spectra

    - ListOfSpectra:            list containing the different spectra

    - ListFreqMin:              list containing min. frequencies

    - ListFreqMax:              list containing max. frequencies

    - ListFreqStep:             list containing step sizes

    - GlobalvLSRList:           list containing vLSR values

    - TelescopeSizeList:        list containing the different telescope sizes

    - InterFlagList:            list containing the different interferometric flags

    - tBackList:                list containing the different background temperatures

    - tSlopeList:               list containing the different temperature slopes

    - nHList:                   list containing the different hydrogen column densities 

    - betaList:                 list containing the different betas 

    - kappaList:                list containing the different kappas

    - IsoRatioFile:             list containing iso flag and path and name of iso ratio file

    - PathFileNameDB:           contains the path and name of the sqlite3 database file

    """


    ## reset output variables
    ListOfSpectraNames = []
    ListOfSpectra = []
    ListFreqMin = []
    ListFreqMax = []
    ListFreqStep = []
    GlobalvLSRList = []
    TelescopeSizeList = []
    InterFlagList = []
    tBackList = []
    tSlopeList = []
    nHList = []
    betaList = []
    kappaList = []
    NoiseLevelList = []
    IsoRatioFile = []
    PathFileNameDB = ""


    ## analyze experimentalData parameter: check if input parameter defines path and name of a file
    print "Analyze experimentalData parameter .."
    ParamIsXMLFile = "false"
    MAGIXExpXML = ""

    # Debug:
    # print 'type(experimentalData).__name__ = ', type(experimentalData).__name__


    ## is experimentalData a string
    filename = experimentalData.strip()                                                     ## remove leading and tailing blanks

    # Debug:
    # print 'filename = ', filename


    ## if neccessary, make relative path absolute
    if (filename[0] != "/"):
        filename = CurrentDir + filename


    ## check if path and name of the experimental file exsits
    if (filename.strip() != ""):
        if not(os.path.exists(filename) or os.path.isfile(filename)):
            print " "
            print " "
            print "Error in XCLASS package, function LineID, subroutine AnalyzeExperimentalDataParameter:"
            print "  The given path and name of the experimental xml- or ASCII file does not exsist!"
            print " "
            print "  Please enter an exsisting path and name of an experimental xml- or ASCII file and redo LineIdentification function call!"
            return (ListOfSpectraNames, ListOfSpectra, ListFreqMin, ListFreqMax, ListFreqStep, GlobalvLSRList, TelescopeSizeList, InterFlagList, \
                    tBackList, tSlopeList, nHList, betaList, kappaList, NoiseLevelList, IsoRatioFile, PathFileNameDB)


    ## check, if file is a xml-file, i.e. the filename ends with '.xml'
    if (filename.endswith(".xml")):
        MAGIXExpXML = filename
        ParamIsXMLFile = "true"

        # Debug:
        # print 'MAGIXExpXML = ', MAGIXExpXML


        ## get parameters from experimental xml file
        NumberExpFiles = task_MAGIX.GetXMLtag(MAGIXExpXML, "NumberExpFiles")
        ListOfSpectraNames = task_MAGIX.GetXMLtag(MAGIXExpXML, "FileNamesExpFiles")
        NumberExpRanges = task_MAGIX.GetXMLtag(MAGIXExpXML, "NumberExpRanges")
        ListFreqMin = task_MAGIX.GetXMLtag(MAGIXExpXML, "MinExpRange")
        ListFreqMax = task_MAGIX.GetXMLtag(MAGIXExpXML, "MaxExpRange")
        ListFreqStep = task_MAGIX.GetXMLtag(MAGIXExpXML, "StepFrequency")
        tBackList = task_MAGIX.GetXMLtag(MAGIXExpXML, "BackgroundTemperature")
        tSlopeList = task_MAGIX.GetXMLtag(MAGIXExpXML, "TemperatureSlope")
        nHList = task_MAGIX.GetXMLtag(MAGIXExpXML, "HydrogenColumnDensity")
        betaList = task_MAGIX.GetXMLtag(MAGIXExpXML, "DustBeta")
        kappaList = task_MAGIX.GetXMLtag(MAGIXExpXML, "Kappa")
        NoiseLevel = task_MAGIX.GetXMLtag(MAGIXExpXML, "NoiseLevel")
        GlobalvLSRList = task_MAGIX.GetXMLtag(MAGIXExpXML, "GlobalvLSR")
        if (GlobalvLSRList == []):
            GlobalvLSRList = [vLSR]
        TelescopeSizeList = task_MAGIX.GetXMLtag(MAGIXExpXML, "TelescopeSize")
        InterFlagList = task_MAGIX.GetXMLtag(MAGIXExpXML, "Inter_Flag")
        NumberHeaderLinesXML = task_MAGIX.GetXMLtag(MAGIXExpXML, "NumberHeaderLines")
        SeparatorColumnsXML = task_MAGIX.GetXMLtag(MAGIXExpXML, "SeparatorColumns")


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get settings for iso ratio file


        ## get flag for iso file
        IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "Isotopologues")
        if (IsoFlag == []):
            IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "iso_flag")
        if (len(IsoFlag) > 0):
            IsoFlag = IsoFlag[0].lower()
            if (IsoFlag == "y" or IsoFlag == "t" or IsoFlag == "true" or IsoFlag == "True"):
                IsoFlag = "y"
            else:
                IsoFlag = "n"
        else:
            IsoFlag = "n"


        ## get name of iso ratio file
        IsoFile = ""
        if (IsoFlag == "y" or IsoFlag == "t" or IsoFlag == "true" or IsoFlag == "True"):
            IsoFile = task_MAGIX.GetXMLtag(MAGIXExpXML, "IsoTableFileName")

            # Debug:
            # print 'IsoFile = ', IsoFile

            if (len(IsoFile) > 0):
                IsoFile = IsoFile[0]
                if (IsoFile.strip() != ""):
                    if (IsoFile[0] != "/"):                                                     ## make relative path absolute
                        IsoFile = CurrentDir + IsoFile
                    if not(os.path.exists(IsoFile) or os.path.isfile(IsoFile)):
                        print " "
                        print " "
                        print "Error in XCLASS package, function LineID, subroutine AnalyzeExperimentalDataParameter:"
                        print "  The given path and name of the iso file " + IsoFile
                        print "  does not exsist!"
                        print " "
                        print "  Please note, if a relative path is specified, the path has to be defined relative to the"
                        print "  current working directory!"
                        print " "
                        print "  Please enter an exsisting path and name of iso file and redo LineIdentification function call!"
                        return (ListOfSpectraNames, ListOfSpectra, ListFreqMin, ListFreqMax, ListFreqStep, GlobalvLSRList, TelescopeSizeList, \
                                InterFlagList, tBackList, tSlopeList, nHList, betaList, kappaList, NoiseLevelList, IsoRatioFile, PathFileNameDB)


        ## define return parameter
        IsoRatioFile = [IsoFlag, IsoFile]


        ## check, if tag "NoiseLevel" is defined for each freq. range if not use global definition for all freq. range
        if (len(NoiseLevel) == len(ListFreqMin)):
            NoiseLevelList = NoiseLevel

        # Debug:
        #    print "NumberHeaderLinesXML = ", NumberHeaderLinesXML
        #    print "SeparatorColumnsXML = ", SeparatorColumnsXML


        ## check path and name of each exp. data file read in spectra from file and select only the defined ranges
        CounterExpDataFiles = (-1)
        for filename in ListOfSpectraNames:                                                 ## loop over all exp. data files
            CounterExpDataFiles += 1


            ## make path absolute (if necessary)
            if (filename[0] != "/"):                                                        ## make relative path absolute
                filename = CurrentDir + filename


            ## check, if path and file exists
            if not(os.path.exists(filename) or os.path.isfile(filename)):
                print " "
                print " "
                print "Error in XCLASS package, function LineID, subroutine AnalyzeExperimentalDataParameter:"
                print "  The given path and name of the experimental data file " + filename
                print "  defined in the experimental data xml-file does not exsist!"
                print " "
                print "  Please note, if a relative path is specified, the path has to be defined relative to the"
                print "  current working directory!"
                print " "
                print "  Please enter an exsisting path and name and redo LineIdentification function call!"
                return (ListOfSpectraNames, ListOfSpectra, ListFreqMin, ListFreqMax, ListFreqStep, GlobalvLSRList, TelescopeSizeList, InterFlagList, \
                        tBackList, tSlopeList, nHList, betaList, kappaList, NoiseLevelList, IsoRatioFile, PathFileNameDB)
            else:


                ## read in entire exp. data file
                if (NumberHeaderLinesXML != [] and SeparatorColumnsXML != []):
                    sepString = SeparatorColumnsXML[CounterExpDataFiles][1]
                    if (sepString.strip() != "" and sepString != "\t"):
                        SpectrumAll = numpy.loadtxt(filename, skiprows = int(NumberHeaderLinesXML[CounterExpDataFiles][1]), delimiter = sepString)
                    else:
                        SpectrumAll = numpy.loadtxt(filename, skiprows = int(NumberHeaderLinesXML[CounterExpDataFiles][1]))
                elif (NumberHeaderLinesXML != []):
                    SpectrumAll = numpy.loadtxt(filename, skiprows = int(NumberHeaderLinesXML[CounterExpDataFiles][1]))
                elif (SeparatorColumnsXML != []):
                    sepString = SeparatorColumnsXML[CounterExpDataFiles][1]
                    if (sepString.strip() != "" and sepString != "\t"):
                        SpectrumAll = numpy.loadtxt(filename, delimiter = sepString)
                    else:
                        SpectrumAll = numpy.loadtxt(filename)
                else:
                    SpectrumAll = numpy.loadtxt(filename)
                ListOfSpectra.append(SpectrumAll)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get name of database file
        ListPathFileNameDB = task_MAGIX.GetXMLtag(MAGIXExpXML, "dbFilename")

        # Debug:
        # print 'ListPathFileNameDB = ', ListPathFileNameDB
        # print "len(ListPathFileNameDB) = ", len(ListPathFileNameDB)


        if (len(ListPathFileNameDB) > 0):
            PathFileNameDB = ListPathFileNameDB[0].strip()
            if (PathFileNameDB.strip() != ""):
                if (PathFileNameDB[0] != "/"):                                              ## make relative path absolute
                    PathFileNameDB = CurrentDir + PathFileNameDB
                if not(os.path.exists(PathFileNameDB) or os.path.isfile(PathFileNameDB)):
                    print "\n\n\tError in XCLASS package, function LineID, subroutine AnalyzeExperimentalDataParameter:"
                    print "\t\tThe given path and name of the database file " + PathFileNameDB
                    print "\t\tdoes not exsist!"
                    print "\n\t\tPlease note, if a relative path is specified, the path has to be defined relative to the"
                    print "\t\tcurrent working directory!"
                    print "\n\t\tPlease enter an exsisting path and name of iso file and redo LineIdentification function call!"
                    return (ListOfSpectraNames, ListOfSpectra, ListFreqMin, ListFreqMax, ListFreqStep, GlobalvLSRList, TelescopeSizeList, InterFlagList, \
                            tBackList, tSlopeList, nHList, betaList, kappaList, NoiseLevelList, IsoRatioFile, PathFileNameDB)
        else:
            PathFileNameDB = ""

        # Debug:
        # print 'PathFileNameDB = ', PathFileNameDB


    ##====================================================================================================================================================
    ## if parameter is not the file name of a xml-file ..
    else:                                                                                   ## file is not an xml-file
        ASCIIDataFile = filename
        Spectrum = numpy.loadtxt(filename, skiprows = 0)
        ListFreqMin = [numpy.min(Spectrum[:, 0])]
        ListFreqMax = [numpy.max(Spectrum[:, 0])]
        ListFreqStep = [Spectrum[1, 0] - Spectrum[0, 0]]
        ListOfSpectra = [Spectrum]
        ListOfSpectraNames = [filename]
        GlobalvLSRList = [vLSR]
        TelescopeSizeList = [str(TelescopeSize)]
        if (InterFlag[0] == "true" or InterFlag[0] == "True" or InterFlag[0] == "TRUE" or InterFlag[0] == "y" or InterFlag[0] == "t" \
            or InterFlag[0] == True):
            InterFlagList = [True]
        else:
            InterFlagList = [False]
        tBackList = [str(tBack)]
        tSlopeList = [str(tSlope)]
        nHList = [str(N_H)]
        betaList = [str(beta_dust)]
        kappaList = [str(kappa_1300)]
        NoiseLevelList = [str(Noise)]
        IsoRatioFile = ["n", ""]


    ## define return values
    return (ListOfSpectraNames, ListOfSpectra, ListFreqMin, ListFreqMax, ListFreqStep, GlobalvLSRList, TelescopeSizeList, InterFlagList, tBackList, \
            tSlopeList, nHList, betaList, kappaList, NoiseLevelList, IsoRatioFile, PathFileNameDB)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## adjust paths in an MAGIX xml file
##
def AdjustExpXMLFile(MAGIXExpXML, JobDir, CurrentDir, dbDefaultFilename):
    """

input parameters:
-----------------

    - MAGIXExpXML:              path and name of MAGIX xml file

    - JobDir:                   path of the current job directory

    - CurrentDir:               current root directory

    - dbDefaultFilename:        path and name of default database file


output parameters:
------------------

    - None

    """


    ##====================================================================================================================================================
    ## analyze experimentalData parameter: check if input parameter defines path and name of a file
    print "Adjust MAGIX XML file .."


    ## copy xml file to temp directory
    command_string = "cp " + MAGIXExpXML + " " + JobDir + "exp.xml"
    os.system(command_string)
    MAGIXExpXML = JobDir + "exp.xml"

    # Debug:
    # print 'MAGIXExpXML = ', MAGIXExpXML


    ## copy the files defined in the xml-file to the current MAGIX working directory
    ExpFiles = []
    ExpFilesOrig = task_MAGIX.GetXMLtag(MAGIXExpXML, "FileNamesExpFiles")
    for filename in ExpFilesOrig:
        if (filename[0] != "/"):                                                            ## make relative path absolute
            filename = CurrentDir + filename
        if not(os.path.exists(filename) or os.path.isfile(filename)):
            print " "
            print " "
            print "Error in XCLASS package, function LineID, subroutine AdjustExpXMLFile:"
            print "  The given path and name of the experimental data file " + filename
            print "  defined in the experimental data xml-file does not exsist!"
            print " "
            print "  Please note, if a relative path is specified, the path has to be defined relative to the"
            print "  current working directory!"
            print " "
            print "  Please enter an exsisting path and name and redo LineIdentification function call!"
            return
        ExpFiles.append(filename)                                                           ## save name(s) in list

    # Debug:
    # print 'ExpFiles = ', ExpFiles


    ## get flag for iso file
    IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "Isotopologues")
    if (IsoFlag == []):
        IsoFlag = task_MAGIX.GetXMLtag(MAGIXExpXML, "iso_flag")
    if (len(IsoFlag) > 0):
        IsoFlag = IsoFlag[0].lower()
        if (IsoFlag == "y" or IsoFlag == "t" or IsoFlag == "True" or IsoFlag == "true"):
            IsoFlag = "y"
        else:
            IsoFlag = "n"
    else:
        IsoFlag = "n"

    # Debug:
    # print 'IsoFlag = ', IsoFlag


    ## get name of iso file
    if (IsoFlag == "y" or IsoFlag == "t" or IsoFlag == "True" or IsoFlag == "true"):
        IsoFile = task_MAGIX.GetXMLtag(MAGIXExpXML, "IsoTableFileName")

        # Debug:
        # print 'IsoFile = ', IsoFile

        if (len(IsoFile) > 0):
            IsoFile = IsoFile[0]
            if (IsoFile.strip() != ""):
                if (IsoFile[0] != "/"):                                                     ## make relative path absolute
                    IsoFile = CurrentDir + IsoFile
                if not(os.path.exists(IsoFile) or os.path.isfile(IsoFile)):
                    print " "
                    print " "
                    print "Error in XCLASS package, function LineID, subroutine AdjustExpXMLFile:"
                    print "  The given path and name of the iso file " + IsoFile
                    print "  does not exsist!"
                    print " "
                    print "  Please note, if a relative path is specified, the path has to be defined relative to the"
                    print "  current working directory!"
                    print " "
                    print "  Please enter an exsisting path and name of iso file and redo LineIdentification function call!"
                    return
                command_string = "cp " + IsoFile + " " + JobDir                             ## copy experimetnal dat files to temp directory
                os.system(command_string)
                i = IsoFile.rfind("/")
                if (i < 0):
                    i = 0
                IsoFile = JobDir + IsoFile[i + 1:]                                          ## define new iso file name
                IsoFile = [IsoFile]

                # Debug:
                # print 'IsoFile = ', IsoFile


                ## modify experimental xml-file in MAGIX temp directory
                task_MAGIX.WriteXMLtag(MAGIXExpXML, "IsoTableFileName", IsoFile)


    ## copy experimental data file(s) to MAGIX temp directory
    NewExpFiles = []                                                                        ## list with modified data file names
    for files in ExpFiles:
        files = files.strip()
        if (files[0] != "/"):
            files = CurrentDir + files

        command_string = "cp " + files + " " + JobDir                                       ## copy experimetnal dat files to temp directory
        os.system(command_string)
        i = files.rfind("/")
        if (i < 0):
            i = 0
        NewExpFiles.append(JobDir + files[i + 1:])                                          ## define new data file names


    ## modify experimental xml-file in MAGIX temp directory
    task_MAGIX.WriteXMLtag(MAGIXExpXML, "FileNamesExpFiles", NewExpFiles)

    # Debug:
    # print "NewExpFiles = ", NewExpFiles


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get name of database file
    PathFileNameDB = task_MAGIX.GetXMLtag(MAGIXExpXML, "dbFilename")

    # Debug:
    # print 'PathFileNameDB = ', PathFileNameDB

    if (len(PathFileNameDB) > 0):
        PathFileNameDB = PathFileNameDB[0].strip()
        if (PathFileNameDB.strip() != ""):
            if (PathFileNameDB[0] != "/"):                                                  ## make relative path absolute
                PathFileNameDB = CurrentDir + PathFileNameDB
            if not(os.path.exists(PathFileNameDB) or os.path.isfile(PathFileNameDB)):
                print " "
                print " "
                print "Error in XCLASS package, function LineID, subroutine AdjustExpXMLFile:"
                print "  The given path and name of the database file " + PathFileNameDB
                print "  does not exsist!"
                print " "
                print "  Please note, if a relative path is specified, the path has to be defined relative to the"
                print "  current working directory!"
                print " "
                print "  Please enter an exsisting path and name of iso file and redo LineIdentification function call!"
                return

            # Debug:
            # print 'PathFileNameDB = ', PathFileNameDB


            ## modify experimental xml-file in MAGIX temp directory
            PathFileNameDB = [PathFileNameDB]
        else:
            PathFileNameDB = [dbDefaultFilename]
    else:
        PathFileNameDB = [dbDefaultFilename]
    task_MAGIX.WriteXMLtag(MAGIXExpXML, "dbFilename", PathFileNameDB)


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## subtract continuum from obs. spectra
##
def PreprocessSpectrum(Spectrum, FreqMin, FreqMax, tBack, tSlope):
    """

input parameters:
-----------------

    - Spectrum:                 the full uncorrected spectrum

    - FreqMin:                  the minimum frequency

    - FreqMax:                  the maximum frequency

    - tBack:                    background temperature (used for baseline substraction)

    - tSlope:                   temperature slope (used for baseline substraction)


output parameters:
------------------

    - SubtractedSpectrum:       the normalized (baseline corrected) spectrum

    """


    ## if necessary, substruct baseline
    if (tBack != 0.0 or tSlope != 0.0):
        FreqArray = Spectrum[:, 0]
        FreqMinIndex = max(0, (numpy.abs(FreqArray - FreqMin)).argmin() - 1)
        FreqMaxIndex = min(len(FreqArray) - 1, (numpy.abs(FreqArray - FreqMax)).argmin() + 1)
        for i in xrange(FreqMinIndex, FreqMaxIndex):
            if (Spectrum[i, 2] == 0.0):
                Spectrum[i, 2] = 1.0                                                        ## avoid multiple continuum subtraction
                Spectrum[i, 1] = Spectrum[i, 1] - (abs(tBack) * (Spectrum[i, 0] / FreqMin)**tSlope)


    ## define return values
    return Spectrum
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## get information for all molecules between FreqMin and FreqMax from sqlite3 database
##
def ReadDatabase(PathFileNameDB, ListOfAllFreqMin, ListOfAllFreqMax, SelectedMolecules, IsotopologuesList, IsoMoleculeList, ElowMax):
    """

input parameters:
-----------------

    - PathFileNameDB:           path and name of a user defined sqlite3 database file (if emtpy, use default file)

    - ListOfAllFreqMin:         list of minimum frequencies

    - ListOfAllFreqMax:         list of maximum frequencies

    - SelectedMolecules:        (optional) python list containing names of molecules

    - IsotopologuesList:        list containing the isotopologues

    - IsoMoleculeList:          list containing the corresponding molecules

    - ElowMax:                  highest allowed lower energy of a transition


output parameters:
------------------

    - ok:                       status variable

    - CounterMolecules:         number of molecules in given frequency range

    - MoleculeData:             all informations about molecules between FreqMin and FreqMax from sqlite3 database

    - Sqlite3DBFile:            path and name of used sqlite3 database file

    - dbFilename:               path and name of default sqlite3 database file

    - ListTotalNumTransPerMol:  updated list containing the names and the number of transitions for each molecule

    """


    ## reset output variables
    ok = 0
    CounterMolecules = 0
    ListTotalNumTransPerMol = []
    MoleculeData = []


    ## define some parameters for database
    NameOfRadTransTable = "transitions"
    ColumnNameForNameTransitions = "T_Name"
    ColumnNameForFreqTransitions = "T_Frequency"
    ColumnNameForIntTransitions = "T_Intensity"
    ColumnNameForEinsteinATransitions = "T_EinsteinA"
    ColumnNameForFreqErrTransitions = "T_Uncertainty"
    ColumnNameForELowTransitions = "T_EnergyLower"
    ColumnNameForgUpTransitions = "T_UpperStateDegeneracy"
    NameOfPartFuncTable = "partitionfunctions"
    NumberOfTemperatures = 110
    ColumnNameForNamePartFunc = 'PF_Name'
    ColumnNamesPartFunc = ['PF_1_072', 'PF_1_148', 'PF_1_230', 'PF_1_318', 'PF_1_413', 'PF_1_514', 'PF_1_622', 'PF_1_738', 'PF_1_862', 'PF_1_995', \
                           'PF_2_138', 'PF_2_291', 'PF_2_455', 'PF_2_630', 'PF_2_725', 'PF_2_818', 'PF_3_020', 'PF_3_236', 'PF_3_467', 'PF_3_715', \
                           'PF_3_981', 'PF_4_266', 'PF_4_571', 'PF_4_898', 'PF_5_000', 'PF_5_248', 'PF_5_623', 'PF_6_026', 'PF_6_457', 'PF_6_918', \
                           'PF_7_413', 'PF_7_943', 'PF_8_511', 'PF_9_120', 'PF_9_375', 'PF_9_772', 'PF_10_471', 'PF_11_220', 'PF_12_023', 'PF_12_882', \
                           'PF_13_804', 'PF_14_791', 'PF_15_849', 'PF_16_982', 'PF_18_197', 'PF_18_750', 'PF_19_498', 'PF_20_893', 'PF_22_387', \
                           'PF_23_988', 'PF_25_704', 'PF_27_542', 'PF_29_512', 'PF_31_623', 'PF_33_884', 'PF_36_308', 'PF_37_500', 'PF_38_905', \
                           'PF_41_687', 'PF_44_668', 'PF_47_863', 'PF_51_286', 'PF_54_954', 'PF_58_884', 'PF_63_096', 'PF_67_608', 'PF_72_444', \
                           'PF_75_000', 'PF_77_625', 'PF_83_176', 'PF_89_125', 'PF_95_499', 'PF_102_329', 'PF_109_648', 'PF_117_490', 'PF_125_893', \
                           'PF_134_896', 'PF_144_544', 'PF_150_000', 'PF_154_882', 'PF_165_959', 'PF_177_828', 'PF_190_546', 'PF_204_174', 'PF_218_776', \
                           'PF_225_000', 'PF_234_423', 'PF_251_189', 'PF_269_153', 'PF_288_403', 'PF_300_000', 'PF_309_030', 'PF_331_131', 'PF_354_813', \
                           'PF_380_189', 'PF_407_380', 'PF_436_516', 'PF_467_735', 'PF_500_000', 'PF_501_187', 'PF_537_032', 'PF_575_440', 'PF_616_595', \
                           'PF_660_693', 'PF_707_946', 'PF_758_578', 'PF_812_831', 'PF_870_964', 'PF_933_254', 'PF_1000_000']


    ## print what you do
    print " "
    print " "
    print "Reading data from sqlite3 database ..",
    sys.stdout.flush()


    ######################################################################################################################################################
    ## DO NOT EDIT OR REMOVE THE FOLLOWING LINE !!!!
    dbFilename = "/home/moeller/ALMA/CASA/myXCLASS-CASA-Interface/No-NR-version/Linux/XCLASS-Interface/Database/cdms_sqlite.db"
    ######################################################################################################################################################


    ## use default file or user setting
    if (PathFileNameDB.strip() != ""):
        Sqlite3DBFile = PathFileNameDB.strip()
    else:
        Sqlite3DBFile = dbFilename


    ## connect to the sqlite3 database
    try:
        conn = sqlite3.connect(Sqlite3DBFile)
    except sqlite3.Error, e:
        print "\nErrot in function LineIdentification.ReadDatabase:"
        print "\t\tCan not connect to sqlite3 databse %s." % Sqlite3DBFile
        print "\t\tError: %d: %s" % (e.args[0], e.args[1])
        ok = 1
        return (ok, CounterMolecules, MoleculeData, Sqlite3DBFile, dbFilename, ListTotalNumTransPerMol)


    ## get cursor
    cursor = conn.cursor()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get all molecule names within frequency range


    ## construct query string for transition table
    query_string = "SELECT " + ColumnNameForNameTransitions + " FROM " + NameOfRadTransTable + " WHERE ("
    NumberRanges = len(ListOfAllFreqMin)
    for RangeIndex in xrange(NumberRanges):
        FreqMin = ListOfAllFreqMin[RangeIndex]
        FreqMax = ListOfAllFreqMax[RangeIndex]
        query_string += "(" + ColumnNameForFreqTransitions + " >= " + str(FreqMin)
        query_string += " AND " + ColumnNameForFreqTransitions + " <= " + str(FreqMax)
        query_string += " AND " + ColumnNameForELowTransitions + " <= " + str(ElowMax / 1.42879) + ")"
        if (RangeIndex < (NumberRanges - 1)):
            query_string += " OR "
    query_string += ") ORDER by " + ColumnNameForNameTransitions

    # Debug:
    # print "\n\nquery_string = ", query_string


    ## read data for all molecules from table RadTrans
    cursor.execute(query_string)
    rows = cursor.fetchall()

    # Debug:
    #    print "\n\nrows = ", rows
    #    print "rows[0] = ", rows[0]
    #    print "rows[1] = ", rows[1]


    ## check, if list is empty, i.e. no molecule is within frequency range
    if (len(rows) == []):
        conn.close()
        ok = 1
        print "\n\n"
        print " "
        print "Error in XCLASS package, function LineID, subroutine ReadDatabase:"
        print "  There is no molecule data in the database within the given frequency range!"
        print " "
        print "  Please enlarge the frequency range and redo LineIdentification function call!"
        return (ok, CounterMolecules, MoleculeData, Sqlite3DBFile, dbFilename, ListTotalNumTransPerMol)


    ## remove dublicates in list with names of molecules
    ListMoleculeNames = []
    for MolName in rows:
        name = MolName[0].encode('ascii', 'ignore')
        if not name in ListMoleculeNames:
            if (name[-2:] == "#1"):
                namered = name[:-2]
                if not name in ListMoleculeNames:
                    if not namered in ListMoleculeNames:
                        ListMoleculeNames.append(name)
            elif (name[-2:] == "#2"):
                namered = name[:-2]
                if not name in ListMoleculeNames:
                    if not namered in ListMoleculeNames:
                        ListMoleculeNames.append(name)
            else:
                ListMoleculeNames.append(name)
    print "done!"
    sys.stdout.flush()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get data for each molecule separatly
    print "Analyze molecular data ..",
    sys.stdout.flush()


    ## exclude / consider only molecules
    MoleculesInFreqeuncyRange = []
    FinalListMoleculeNames = []
    ExcludedMolecules = []
    for MolName in ListMoleculeNames:                                                       ## loop over all molecules within the frequency range

        # Debug:
        # print "MolName = ", MolName


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## check molecules and read in data
        checkPartitionFuncTableFlag = "true"
        NoNoneEntry = "true"
        ConsiderMoleculeFlag = "true"
        PrintWarningFlag = "minimal"


        ## check, if some molecules should be excluded or included
        if (SelectedMolecules != [] and SelectedMolecules != [None] and SelectedMolecules != None):
            OnlyExcludeMoleculesFlag = "true"
            for mol in SelectedMolecules:
                StrippedMoleculeName = mol.strip()


                ## should molecule NOT be considered
                if (StrippedMoleculeName[:2] == "--"):
                    StrippedMoleculeName = StrippedMoleculeName[2:]
                    if (StrippedMoleculeName == MolName):
                        ConsiderMoleculeFlag = "false"
                        break
                else:
                    OnlyExcludeMoleculesFlag = "false"
                    break


            ## consider only those molecules defined in list SelectedMolecules
            if (ConsiderMoleculeFlag == "true" and OnlyExcludeMoleculesFlag == "false"):
                if not MolName in SelectedMolecules:
                    ConsiderMoleculeFlag = "false"

        # Debug:
        #    print "\n\nMolName = ", MolName
        #    print "type(MolName).__name__ = ", type(MolName).__name__
        #    print "checkPartitionFuncTableFlag = ", checkPartitionFuncTableFlag
        #    print "ConsiderMoleculeFlag = ", ConsiderMoleculeFlag


        ## check, if table partitionfunctions contain data for all molecules as well
        if (checkPartitionFuncTableFlag == "true" and type(MolName).__name__ != 'float' and ConsiderMoleculeFlag == "true"):


            ## construct query string for transition table
            query_string = "SELECT * FROM " + NameOfPartFuncTable + " WHERE " + ColumnNameForNamePartFunc + " = " + chr(34) + MolName + chr(34)

            # Debug:
            # print "\n\nquery_string = ", query_string


            ## read data for all molecules from table RadTrans
            cursor.execute(query_string)
            rows = cursor.fetchall()
            DoNotIncludeFlag = "false"
            try:
                i = rows[0]                                                                 ## check, if rows variable is not empty
            except IndexError:
                NoNoneEntry = "false"
                DoNotIncludeFlag = "true"


            ## do further checks
            if (DoNotIncludeFlag != "true"):
                if (rows == []):
                    NoNoneEntry = "false"
                    DoNotIncludeFlag = "true"

                elif (len(rows[0]) < 110):
                    NoNoneEntry = "false"
                    DoNotIncludeFlag = "true"

                else:
                    HelpList = rows[0][5:5+110]


                ## continue here, if rows is not empty and has (more than) 110 columns
                if (DoNotIncludeFlag != "true"):

                    # Debug:
                    # print "\n\nrows = ", rows
                    # print "rows[0] = ", rows[0]
                    # print "rows[0][5:5+110] = ", rows[0][5:5+110]


                    ## check if partition function is given for the current molecule MolName
                    if (HelpList.count(None) > 0):
                        NoNoneEntry = "false"
                        if (PrintWarningFlag == "true"):
                            print " "
                            print " "
                            print "WARNING in XCLASS package, subroutine LineIdentification:"
                            print "  There is no partition function given for molecule " + MolName
                            print "  Ignore molecule " + MolName + " in the current line identification."
                            print " "
                            print "Partition function values = ", HelpList
                        DoNotIncludeFlag = "true"


        ## continue here, if partition function for the current molecule is complete
        if (NoNoneEntry == "true" and type(MolName).__name__ != 'float' and ConsiderMoleculeFlag == "true"):
            CounterMolecules += 1
            MoleculesInFreqeuncyRange.append(MolName)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## get all molecule names within frequency range


            ## create entry for MoleculeData
            FinalListMoleculeNames.append(MolName)
            helpList = [MolName]
            MoleculeData.append(helpList)


        ## if not included, add current molecule to list of excluded molecules
        if (NoNoneEntry != "true" or type(MolName).__name__ == 'float' or ConsiderMoleculeFlag != "true"):
            ExcludedMolecules.append(MolName)


    ## we've done
    conn.close()
    print "done!"

    # Debug:
    # print "MoleculeData[0] = ", MoleculeData[0]
    # print "MoleculesInFreqeuncyRange = ", MoleculesInFreqeuncyRange
    #    print "\n\n\n"
    #    print "IsotopologuesList = ", IsotopologuesList
    #    print "IsoMoleculeList = ", IsoMoleculeList
    #    for mol in FinalListMoleculeNames:
    #        print "===>>>" + mol + "<<<==="


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## is an iso ratio file defined? Extend SelectedMolecules list
    if (IsotopologuesList != []):                                                           ## is an iso ratio file defined?
        for LocalIsotopologue in IsotopologuesList:                                         ## loop over all isotopologues in the iso ratio file
            Isotopologue = LocalIsotopologue.strip()
            if (Isotopologue in FinalListMoleculeNames):                                    ## is an isotopologue located in the given spectra?
                i = IsotopologuesList.index(Isotopologue)                                   ## get index of current isotopologue
                                                                                            ## (we consider only the first occurrance of an isotopologue
                                                                                            ##  in the iso ratio file)
                IsoMolecule = IsoMoleculeList[i]                                            ## to get the corresponding iso molecule
                if (IsoMolecule in FinalListMoleculeNames):                                 ## check, if iso molecule is already considered
                    i = FinalListMoleculeNames.index(Isotopologue)                          ## get position of isotopologue in molecule list
                                                                                            ## delete isotopologue from list of mol. in the given spectra
                    FinalListMoleculeNames = FinalListMoleculeNames[:i] + FinalListMoleculeNames[i + 1:]
                    MoleculeData = MoleculeData[:i] + MoleculeData[i + 1:]
                    # ListTotalNumTransPerMol = ListTotalNumTransPerMol[:i] + ListTotalNumTransPerMol[i + 1:]
                    ExcludedMolecules.append(Isotopologue)


        ## define new numbers of molecules
        CounterMolecules = len(FinalListMoleculeNames)

        # Debug:
        #    print "CounterMolecules = ", CounterMolecules
        #    print "FinalListMoleculeNames = ", FinalListMoleculeNames
        #    print "\n\n\n"
        #    for mol in FinalListMoleculeNames:
        #        print "+++>>>", mol, "<<<+++"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## print list of identified molecules
    print "\nNumber of molecules in the freq. range = ", CounterMolecules
    if (ExcludedMolecules != []):
        for mol in ExcludedMolecules:
            print "excluded molecule = ", mol


    ## define return values
    return (ok, CounterMolecules, MoleculeData, Sqlite3DBFile, dbFilename, ListTotalNumTransPerMol)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## creates molfit file for the current molecule, and writes created molfits file to current molecule subdirectory
##
def CreateMolfitFile(CurrentDir, myXCLASSRootDirectory, MoleculejobDir, MolfitsFileName, MoleculeName, SourceName, DefaultMolfitFile):
    """

input parameters:
-----------------

    - CurrentDir:               current direcotry

    - myXCLASSRootDirectory:    XCLASS interface root directory

    - MoleculejobDir:           working directory for current molecule

    - MolfitsFileName:          path and name of molfits file

    - MoleculeName:             full name of current molecule

    - SourceName:               name of source

    - DefaultMolfitFile:        path and name of a default molfit file used for molecules which are not described by source templates


output parameters:
------------------

    - DefaultMolfitUsedFlag:    flag indicating if default molfit file was used


    """


    ## define some internal parameters
    MolfFitFileCreatedFlag = "false"                                                        ## no molfit file is created
    DefaultMolfitUsedFlag = True
    SourceName = SourceName.strip()
    DefaultMolfitFile = DefaultMolfitFile.strip()

    # Debug:
    # print "SourceName = ", SourceName
    # print "DefaultMolfitFile = ", DefaultMolfitFile


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## test if template directory already exists
    MolfitFileNotCreatedFlag = True
    if (SourceName != ""):


        ## define template directory
        # TemplateDirectory = myXCLASSRootDirectory + "programs/LineIdentification/Templates/"
        # SourceMolfitFile = TemplateDirectory + SourceName + ".molfit"

        # Debug:
        # print "TemplateDirectory = ", TemplateDirectory


        ## check if template for the current source already exists
        if (SourceName[0] != "/"):                                                          ## make relative path absolute
            SourceMolfitFile = CurrentDir + SourceName
        else:
            SourceMolfitFile = SourceName
        if (not SourceMolfitFile.endswith(".molfit")):
            SourceMolfitFile = SourceMolfitFile + ".molfit"
        if (os.path.isfile(SourceMolfitFile)):                                              ## check, if template for current molecule and source exists

            # Debug:
            # print "SourceMolfitFile = ", SourceMolfitFile


            ## get all parameters from source molfit file
            MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(SourceMolfitFile)


            ## create new molfit file in new format
            CurrentMoleculeFlag = "false"
            NewMolfitFileContent = []
            for MoleculeIndex in xrange(len(MoleculesInMolfitFile)):
                LocalMolName = MoleculesInMolfitFile[MoleculeIndex]
                if (LocalMolName.strip() == MoleculeName.strip()):
                    CurrentMoleculeFlag = "true"
                    LocalMolfitParameters = AllParameters[MoleculeIndex]


                    ## create final contribution to all further molfit files of identified strong molecule
                    NewLine = MoleculeName.strip() + "    " + str(len(LocalMolfitParameters))
                    NewMolfitFileContent.append(NewLine)
                    for line in LocalMolfitParameters:

                        # Debug:
                        # print "line = ", line

                        NewLine = ""
                        for col in line:
                            Name = col[0]
                            flag = col[1]
                            lowlimit = col[2]
                            uplimit = col[3]
                            element = col[4]

                            # Debug:
                            # print "col = ", col


                            ## create new line with fixed parameters
                            if (Name == "source_size" or Name == "T_rot" or Name == "N_tot" or Name == "V_width" or Name == "V_off" or Name == "T_Back" \
                                or Name == "T_Slope" or Name == "nHcolumn" or Name == "kappa" or Name == "beta"):
                                if (flag != "0"):
                                    NewLine += "   y  %12.3e  %12.3e  %25.15e" % (float(uplimit), float(lowlimit), float(element))
                                else:
                                    NewLine += "   n  %12.3e  %12.3e  %25.15e" % (float(uplimit), float(lowlimit), float(element))
                            elif (Name == "CFFlag"):
                                NewLine += "  " + element


                        ## append new line to NewMolfitFileContent list
                        NewMolfitFileContent.append(NewLine)


            ## write part of template which describes the current molecule to current working directory
            if (NewMolfitFileContent != []):
                MolfitFileNotCreatedFlag = False
                NewMolfitFile = open(MolfitsFileName, 'w')
                NewMolfitFile.write("% Number of molecules =     1\n")                      ## write first line
                for line in NewMolfitFileContent:
                    NewMolfitFile.write(line + "\n")
                NewMolfitFile.close()
                MolfFitFileCreatedFlag = "true"
                DefaultMolfitUsedFlag = False
                return DefaultMolfitUsedFlag


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check, if default molfit file is defined
    if (MolfFitFileCreatedFlag == "false" and DefaultMolfitFile != "" and MolfitFileNotCreatedFlag):
        DefaultMolfitUsedFlag = False
        if (DefaultMolfitFile.endswith(".molfit")):                                         ## check, if variable 'DefaultMolfitFile' ends with ".molfit"
            MolfFitFileCreatedFlag = "true"


            ## if neccessary, make relative path absolute
            if (DefaultMolfitFile[0] != "/"):
                filename = CurrentDir + DefaultMolfitFile
            else:
                filename = DefaultMolfitFile


            ## copy default molfit file to current working directory
            command_string = "cp " + filename + " " + MolfitsFileName
            os.system(command_string)


            ## get all parameters from source molfit file
            MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(MolfitsFileName)

            # Debug:
            #    print "MoleculesInMolfitFile = ", MoleculesInMolfitFile
            #    print "AllParameters = ", AllParameters


            ## create new molfit file in new format
            NewMolfitFileContent = []
            for MoleculeIndex in xrange(len(MoleculesInMolfitFile)):
                LocalMolName = MoleculesInMolfitFile[MoleculeIndex]
                LocalMolfitParameters = AllParameters[MoleculeIndex]


                ## create final contribution to all further molfit files of identified strong molecule
                if (MoleculeIndex == 0):
                    NewLine = MoleculeName.strip() + "    " + str(len(LocalMolfitParameters))
                else:
                    NewLine = LocalMolName.strip() + "    " + str(len(LocalMolfitParameters))
                NewMolfitFileContent.append(NewLine)
                for line in LocalMolfitParameters:

                    # Debug:
                    # print "line = ", line

                    NewLine = ""
                    for col in line:
                        Name = col[0]
                        flag = col[1]
                        lowlimit = col[2]
                        uplimit = col[3]
                        element = col[4]

                        # Debug:
                        # print "col = ", col


                        ## create new line with fixed parameters
                        if (Name == "source_size" or Name == "T_rot" or Name == "N_tot" or Name == "V_width" or Name == "V_off" or Name == "T_Back" \
                            or Name == "T_Slope" or Name == "nHcolumn" or Name == "kappa" or Name == "beta"):
                            if (float(flag) != 0):
                                NewLine += "   y  %12.3e  %12.3e  %25.15e" % (float(uplimit), float(lowlimit), float(element))
                            else:
                                NewLine += "   n  %12.3e  %12.3e  %25.15e" % (float(uplimit), float(lowlimit), float(element))
                        elif (Name == "CFFlag"):
                            NewLine += "  " + element


                    ## append new line to NewMolfitFileContent list
                    NewMolfitFileContent.append(NewLine)


            ## write part of template which describes the current molecule to current working directory
            if (NewMolfitFileContent != []):
                NewMolfitFile = open(MolfitsFileName, 'w')
                # NewMolfitFile.write("% Number of molecules =     " + str(len(LocalMolfitParameters)) + "\n")
                for line in NewMolfitFileContent:
                    NewMolfitFile.write(line + "\n")
                NewMolfitFile.close()
                MolfFitFileCreatedFlag = "true"
                DefaultMolfitUsedFlag = True
                return DefaultMolfitUsedFlag
            return DefaultMolfitUsedFlag


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## if no default molfit file is defined, create a new one (has to be done!!!!!!!!)
    if (MolfFitFileCreatedFlag == "false"):
        print ""

    # Debug:
    #    print "DefaultMolfitUsedFlag = ", DefaultMolfitUsedFlag


    ## finished
    return DefaultMolfitUsedFlag
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## copy user defined MAGIX exp. xml file to current single molecule fit directory and adopt paths
##
def PrepareMAGIXExpXMLFile(CurrentDir, MAGIXExpXML, MoleculejobDir, dbDefaultFilename):
    """

input parameters:
-----------------

    - CurrentDir:           current working directory

    - MAGIXExpXML:          path and name of MAGIX exp. xml file

    - MoleculejobDir:       path of current single molecule fit directory

    - dbDefaultFilename:    path and name of default database file


output parameters:
------------------

    - ok:                   status variable

    - MAGIXExpXML:          modified path and name of MAGIX exp. xml file

    """
    ok = 0


    ## copy xml file to temp directory
    command_string = "cp " + MAGIXExpXML + " " + MoleculejobDir
    os.system(command_string)


    ## get name of experimental xml-file
    i = MAGIXExpXML.rfind("/")
    if (i > 0):
        MAGIXExpXML = MAGIXExpXML[i + 1:]

    # Debug:
    # print 'MAGIXExpXML = ', MAGIXExpXML

    MAGIXExpXML = MoleculejobDir + MAGIXExpXML

    # Debug:
    # print 'MAGIXExpXML = ', MAGIXExpXML


    ## copy the files defined in the xml-file to the current MAGIX working directory
    ExpFiles = []
    ExpFilesOrig = task_MAGIX.GetXMLtag(MAGIXExpXML, "FileNamesExpFiles")
    for filename in ExpFilesOrig:
        if (filename[0] != "/"):                                                            ## make relative path absolute
            filename = CurrentDir + filename
        if not(os.path.exists(filename) or os.path.isfile(filename)):
            print " "
            print " "
            print "Error in XCLASS package, function LineID, subroutine PrepareMAGIXExpXMLFile:"
            print "  The given path and name of the experimental data file " + filename
            print "  defined in the experimental data xml-file does not exsist!"
            print " "
            print "  Please note, if a relative path is specified, the path has to be defined relative to the"
            print "  current working directory!"
            print " "
            print "  Please enter an exsisting path and name and redo LineIdentification function call!"
            ok = 1
            return (ok, MAGIXExpXML)
        ExpFiles.append(filename)                                                           ## save name(s) in list

    # Debug:
    # print 'ExpFiles = ', ExpFiles


    ## deactivate iso flag
    #    task_MAGIX.WriteXMLtag(MAGIXExpXML, "Isotopologues", ["n"])
    #    task_MAGIX.WriteXMLtag(MAGIXExpXML, "IsoTableFileName", [" "])


    ## copy experimental data file(s) to MAGIX temp directory
    NewExpFiles = []                                                                        ## list with modified data file names
    for files in ExpFiles:
        files = files.strip()
        if (files[0] != "/"):
            files = CurrentDir + files

        command_string = "cp " + files + " " + MoleculejobDir                               ## copy experimetnal dat files to temp directory
        os.system(command_string)
        i = files.rfind("/")
        if (i < 0):
            i = 0
        NewExpFiles.append(MoleculejobDir + files[i + 1:])                                  ## define new data file names


    ## modify experimental xml-file in MAGIX temp directory
    task_MAGIX.WriteXMLtag(MAGIXExpXML, "FileNamesExpFiles", NewExpFiles)

    # Debug:
    # print "NewExpFiles = ", NewExpFiles


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## get name of database file
    PathFileNameDB = task_MAGIX.GetXMLtag(MAGIXExpXML, "dbFilename")

    # Debug:
    # print 'PathFileNameDB = ', PathFileNameDB

    if (len(PathFileNameDB) > 0):
        PathFileNameDB = PathFileNameDB[0].strip()
        if (PathFileNameDB.strip() != ""):
            if (PathFileNameDB[0] != "/"):                                                  ## make relative path absolute
                PathFileNameDB = CurrentDir + PathFileNameDB
            if not(os.path.exists(PathFileNameDB) or os.path.isfile(PathFileNameDB)):
                print " "
                print " "
                print "Error in XCLASS package, function LineID, subroutine PrepareMAGIXExpXMLFile:"
                print "  The given path and name of the database file " + PathFileNameDB
                print "  does not exsist!"
                print " "
                print "  Please note, if a relative path is specified, the path has to be defined relative to the"
                print "  current working directory!"
                print " "
                print "  Please enter an exsisting path and name of iso file and redo LineIdentification function call!"
                ok = 1
                return (ok, MAGIXExpXML)

            # Debug:
            # print 'PathFileNameDB = ', PathFileNameDB


            ## modify experimental xml-file in MAGIX temp directory
            PathFileNameDB = [PathFileNameDB]
        else:
            PathFileNameDB = [dbDefaultFilename]
    else:
        PathFileNameDB = [dbDefaultFilename]
    task_MAGIX.WriteXMLtag(MAGIXExpXML, "dbFilename", PathFileNameDB)


    ## return to main function
    return (ok, MAGIXExpXML)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## determines the best fit result for a molecule
##
def GetBestResult(JobDir, ConvertLogLinFlag, ListOfSpectraNames, ListOfAllFreqMin, ListOfAllFreqMax, tBackList, tSlopeList):
    """

input parameters:
-----------------

    - JobDir:                       path of job directory

    - ConvertLogLinFlag:            defines if column densities in the fitted molfit file(s) are converted back to linear values

    - ListOfSpectraNames:           list of exp. data file names

    - ListOfAllFreqMin:             list containing all min. frequencies of all frequency ranges

    - ListOfAllFreqMax:             list containing all max. frequencies of all frequency ranges

    - tBackList:                    list containing the background temperatures for all frequency ranges

    - tSlopeList:                   list containing the temperature slopes for all frequency ranges


output parameters:
------------------

    - BestMolfitFileContent:        list of all molfit files

    - IsoRatioFileContent:          optimitzed iso ratio file

    - ListSubtractedModelSpectra:   list of all function values for all molfit files

    """


    ## initialize return parameters
    BestMolfitFileContent = []
    IsoRatioFileContent = []
    ListSubtractedModelSpectra = []


    ## get all filenames in the current job directory
    listing = os.listdir(JobDir)

    # Debug:
    # print "\n\n\nlisting = ", listing


    ## get number of chi2 log files
    NumOutFiles = 0                                                                         ## determine number of output files
    NumMolfitFiles = 0
    for files in listing:
        if (files.endswith(".log.chi2")):
            NumOutFiles += 1
        if (files.endswith(".out.molfit")):
            NumMolfitFiles += 1
    if (NumOutFiles == 0):                                                                  ## check if new input files exsist
        print " "
        print " "
        print "Error in XCLASS package, function LineID, subroutine GetBestResult:"
        print "  Can not find a chi2 log file!"
        print " "
        print "JobDir = ", JobDir
        print "listing = ", listing
        print "\n\n\n"
        return (BestMolfitFileContent, IsoRatioFileContent, ListSubtractedModelSpectra)


    ## determine the chi2 log file, which corresponds to the best fit
    BestChi2Value = 1.e99
    FileNameBestChi2 = ""
    for files in listing:                                                                   ## loop over all files in the current job directory
        if (files.endswith(".log.chi2")):                                                   ## we're interested only in the log.chi2 files
            Chi2LogFile = open(JobDir + files)                                              ## open log.chi2 file
            dummyline = Chi2LogFile.readline()                                              ## read in first line as dummy
            LogLine = Chi2LogFile.readline()                                                ## read second line with best chi2 value
            Chi2LogFile.close()                                                             ## close log file
            LogLine = LogLine.strip()                                                       ## remove blanks
            SplittedLine = LogLine.split()                                                  ## split line into columns
            chi2Value = float(SplittedLine[1])                                              ## get chi2 value
            if (chi2Value <= BestChi2Value):                                                ## check, if current chi2 value is the best one
                BestChi2Value = chi2Value                                                   ## if yes, save chi2 value
                FileNameBestChi2 = files                                                    ## and corresponding file name

                # Debug:
                # print "chi2Value = ", chi2Value


    ## get phrase identifying the used algorithm for the best chi2 value
    BestAlgorithm = ""
    BestAlgorithm = FileNameBestChi2.replace(".log.chi2", "")
    BestAlgorithm = BestAlgorithm.replace("fit__", "")


    ## find molfit file 
    for files in listing:                                                                   ## loop over all files in the current job directory
        if (files.endswith(".out.molfit")):                                                 ## we're interested only in the molfit files
            i = files.find(BestAlgorithm)
            if (i > (-1)):


                ## convert column and hydrogen column density (if given for each component) back to linear scale
                if (ConvertLogLinFlag == "true"):
                    LogLinearFlag = "linear"
                    task_myXCLASS.ConvertNtotToLogScale(JobDir + files, LogLinearFlag)


                ## read in new output files
                MolfitFile = open(JobDir + files)
                BestMolfitFileContent = MolfitFile.readlines()
                MolfitFile.close()

                # Debug:
                # print "\n\nfiles = ", files
                # print "BestMolfitFileContent = ", BestMolfitFileContent
                break


    ## find optimized iso ratio file 
    for files in listing:                                                                   ## loop over all files in the current job directory
        if (files.startswith("myXClass_isoratios") and files.endswith(".out.input")):
            i = files.find(BestAlgorithm)
            if (i > (-1)):
                IsoRatioFile = open(JobDir + files)
                IsoRatioFileContent = IsoRatioFile.readlines()
                IsoRatioFile.close()
                break


    ## special handling for LM
    if (NumMolfitFiles == 1 and BestAlgorithm.find("LM") > (-1)):
        BestAlgorithm = "LM.out"

    # Debug:
    # print "BestAlgorithm = ", BestAlgorithm


    ## find model data file
    if (ListOfAllFreqMin != []):
        for CounterRange in xrange(len(ListOfSpectraNames)):                                ## loop over all frequency ranges


            ## analyze exp. data file name, we only want the name, not the path
            ExpDataFileName = ListOfSpectraNames[CounterRange]                              ## get path and name of corresponding exp. data file
            i = ExpDataFileName.rfind("/")
            ExpDataFileName = ExpDataFileName[i + 1:]
            if (ExpDataFileName.endswith(".dat")):
                ExpDataFileName = ExpDataFileName.replace(".dat", "")
            elif (ExpDataFileName.endswith(".fits")):
                ExpDataFileName = ExpDataFileName.replace(".fits", "")

            # Debug:
            #    print "\n\n\n\n\n\n\n\n\n\n\n"
            #    print "ExpDataFileName = ", ExpDataFileName
            #    print "BestAlgorithm = ", BestAlgorithm


            for j in xrange(len(ListOfAllFreqMin)):                                         ## loop over all range definitions in the whole xml file
                DataFileIndex = int(ListOfAllFreqMin[j][0])                                 ## get exp data file index for each range definition
                if (DataFileIndex == (CounterRange + 1)):                                   ## if current range definiton belongs to current range index
                    FreqMin = float(ListOfAllFreqMin[j][1])                                 ## continue and get min. freq.
                    FreqMax = float(ListOfAllFreqMax[j][1])                                 ## and max. freq.
                    if (tBackList != []):
                        tBack = float(tBackList[j][1])                                      ## and background temperature
                    else:
                        tBack = 0.0
                    if (tSlopeList != []):
                        tSlope = float(tSlopeList[j][1])                                    ## and temperature slope
                    else:
                        tSlope = 0.0

                    # Debug:
                    #    print "\n\n\n\n\n\n\n\n\n\n\n"
                    #    print "DataFileIndex, CounterRange = ", DataFileIndex, CounterRange
                    #    print "FreqMin = ", FreqMin
                    #    print "FreqMax = ", FreqMax
                    #    print "tBack = ", tBack
                    #    print "tSlope = ", tSlope


                    ## find exp. data file in the current single molecule fit directory
                    for files in listing:                                                   ## loop over all files in the current job directory

                        # Debug:
                        #    print "files = ", files


                        if (files.startswith(ExpDataFileName) and (files.endswith(BestAlgorithm) or files.endswith(".out.dat"))):
                            i = files.find(BestAlgorithm)
                            if (i > (-1)):

                                # Debug:
                                #    print "JobDir + files = ", JobDir + files


                                ## read in model function from file
                                SpectrumAll = numpy.loadtxt(JobDir + files, skiprows = 0)


                                ## save only data within current frequency range
                                FreqMinRange = FreqMin
                                FreqMaxRange = FreqMax
                                FreqMinIndex = -1e99
                                FreqMaxIndex = len(SpectrumAll[:, 0])


                                ## get range indices for exp. data
                                LocalCopyArray = SpectrumAll[:, 0]
                                FreqMinIndex = max(0, (numpy.abs(LocalCopyArray - FreqMin)).argmin() - 1)
                                FreqMaxIndex = min(len(LocalCopyArray) - 1, (numpy.abs(LocalCopyArray - FreqMax)).argmin() + 1)


                                ## check, thath indices describe frequencies within given frequency range
                                nuMin = SpectrumAll[FreqMinIndex, 0]
                                if (nuMin < FreqMin):
                                    FreqMinIndex += 1
                                    nuMin = SpectrumAll[FreqMinIndex, 0]
                                    if (nuMin < FreqMin):
                                        FreqMinIndex = FreqMinIndex - 2
                                        FreqMinIndex = max(0, FreqMinIndex)
                                    FreqMinIndex = max(0, FreqMinIndex)
                                nuMax = SpectrumAll[FreqMaxIndex, 0]
                                if (nuMax > FreqMax):
                                    FreqMaxIndex -= 1
                                    nuMax = SpectrumAll[FreqMaxIndex, 0]
                                    if (nuMax > FreqMax):
                                        FreqMaxIndex = FreqMaxIndex + 2
                                    FreqMaxIndex = min(len(LocalCopyArray), FreqMaxIndex)


                                ## take reversed order into account                                
                                FreqMinIndexCopy = FreqMinIndex
                                FreqMaxIndexCopy = FreqMaxIndex
                                FreqMinIndex = min(FreqMinIndexCopy, FreqMaxIndexCopy)
                                FreqMaxIndex = max(FreqMinIndexCopy, FreqMaxIndexCopy)

                                # Debug:
                                # print "FreqMinRange = ", FreqMinRange
                                # print "FreqMaxRange = ", FreqMaxRange
                                # print "FreqMinIndex = ", FreqMinIndex
                                # print "FreqMaxIndex = ", FreqMaxIndex


                                ## save part of the exp. data file, which corresponds to the current frequency range to ListOfSpectra variable
                                if (FreqMinIndex < FreqMaxIndex and FreqMinIndex != -1e99):


                                    ## subtract continuum
                                    SubtractedModelSpectra = SpectrumAll[FreqMinIndex:FreqMaxIndex, :]
                                    if (abs(numpy.min(SubtractedModelSpectra[:, 1])) != 0.0 or abs(numpy.max(SubtractedModelSpectra[:, 1])) != 0.0):
                                        if (tBack != 0.0 or tSlope != 0.0):
                                            for i in xrange(len(SubtractedModelSpectra[:, 1])):     ## loop over all points in the spectrum
                                               SubtractedModelSpectra[i, 1] = SubtractedModelSpectra[i, 1] \
                                                                              - (abs(tBack) * (SubtractedModelSpectra[i, 0] / FreqMin)**tSlope)

                                    # Debug:
                                    # print "len(SubtractedModelSpectra) = ", len(SubtractedModelSpectra)
                                    # print "SubtractedModelSpectra = ", SubtractedModelSpectra


                                    ListSubtractedModelSpectra.append(SubtractedModelSpectra)
                                else:
                                    ListSubtractedModelSpectra.append([])

                                # Debug:
                                # print "files = ", files
                                # print "ListSubtractedModelSpectra = ", ListSubtractedModelSpectra
                                break


    ## define output variables
    return (BestMolfitFileContent, IsoRatioFileContent, ListSubtractedModelSpectra)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## check, if molecule contributes to current spectrum or not
##
def CheckContribution(molecule, CurrentMolfitFileName, NewExpFileList, NewFreqMinList, NewFreqMaxList, ListSubtractedSpectra, \
                      ListSubtractedModelSpectra, MaxOverestimationHeight, LineIDjobDir, NewtBackList, NewtSlopeList, NewNoiseList, ListOfSpectraNames, \
                      ListOfSpectra, StrongMolContribution, CheckEmissionAbsorptionFlag, DecisionMethod, Tolerance, dbFile):
    """

input parameters:
-----------------

    - molecule:                     name of current molecule

    - CurrentMolfitFileName:        path and name of current molfit file

    - NewExpFileList:               list containing all paths and names of all observational data files

    - NewFreqMinList:               min. freq. for each frequency range

    - NewFreqMaxList:               max. freq. for each frequency range

    - ListSubtractedSpectra:        the observed spectrum (contiuum substracted)

    - ListSubtractedModelSpectra:   the modeled spectrum (contiuum substracted)

    - MaxOverestimationHeight:      overestimation limit

    - LineIDjobDir:                 current line ID job directory

    - NewtBackList:                 list containing the background temperatures for all frequency ranges

    - NewtSlopeList:                list containing the temperature slopes for all frequency ranges

    - NewNoiseList:                 list with all noise levels for all frequency ranges

    - ListOfSpectraNames:           list containing the names of the exp. data file names for all frequency range

    - ListOfSpectra:                list of all observed spectra

    - StrongMolContribution:        contribution of strong molecule(s)

    - CheckEmissionAbsorptionFlag:  check (true/false), if a peak in modeled spectrum has the same type as corresponding peak in observational data

    - DecisionMethod:               method which is used

    - Tolerance:                    tolerance factor for method "significant"

    - dbFile:                       path and file of database file


output parameters:
------------------

    - ContributionFlag:             flag indicating if current molecule contributes or not

    - LocalResultsFile:             contents for result file

    """


    ## define speed of light (m/s) and (km/s)
    cms = 299792458.0
    ckms = cms * 1.e-3


    ## define numerical zero
    NumericalZero = 1.e-6


    ## initalize output variable
    ContributionFlag = "true"

    # Debug:
    #    print "len(ListSubtractedModelSpectra) = ", len(ListSubtractedModelSpectra)
    #    print "ListSubtractedModelSpectra = ", ListSubtractedModelSpectra


    ## read all velocity offsets from molfit file
    MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(CurrentMolfitFileName)
    VelocityOffsets = []
    for MoleculeIndex in xrange(len(MoleculesInMolfitFile)):
        LocalMolName = MoleculesInMolfitFile[MoleculeIndex]
        if (LocalMolName.strip() == molecule.strip()):
            LocalMolfitParameters = AllParameters[MoleculeIndex]


            ## create final contribution to all further molfit files of identified strong molecule
            for line in LocalMolfitParameters:
                for col in line:
                    Name = col[0]
                    element = col[4]
                    if (Name == "V_off"):
                        VelocityOffsets.append(float(element))


    ## open local result file
    LocalResultsFile = []


    ## loop over all modeled spectra
    EmptySpectrumCounter = 0
    CounterPeak = 0                                                                         ## reset counter for overall peaks
    CounterAcceptedPeaks = 0                                                                ## reset counter for accepted peaks
    CounterOverestimatedPeaks = 0                                                           ## reset counter for overestimated peaks
    CounterFreqRange = (-1)
    for ExpDataFileIndex in xrange(len(NewExpFileList)):                                    ## loop over all obs. data files
        NewExpFileName = NewExpFileList[ExpDataFileIndex]
        NewSpectraName = os.path.basename(NewExpFileName)                                   ## get name without path

        # Debug:
        #    print "\n\nNewExpFileName = ", NewExpFileName
        #    print "NewSpectraName = ", NewSpectraName


        ## write name of observational data file to log file
        LocalResultsFile.append("\n")
        LocalResultsFile.append("\n")
        LocalResultsFile.append("\n")
        helpstring = "-" * 150
        LocalResultsFile.append("\t" + helpstring + "\n")
        LocalResultsFile.append("\tSpectrum file = " + NewExpFileName + "\n")
        LocalResultsFile.append("\t" + helpstring + "\n\n")
        helpString = "-" * 150


        ## determine corresponding observed unmodified spectrum
        SpectraName = ""
        NewASCIIdata = []
        OldSpectraIndex = 0
        for LocalSpectraIndex in xrange(len(ListOfSpectraNames)):
            if (NewSpectraName == os.path.basename(ListOfSpectraNames[LocalSpectraIndex])):
                OldSpectraIndex = LocalSpectraIndex
                NewASCIIdata = ListOfSpectra[LocalSpectraIndex]
                SubtractedSpectrumLocal = ListSubtractedSpectra[LocalSpectraIndex]


                ## get name of exp. data file of current frequency range
                SpectraName = ListOfSpectraNames[LocalSpectraIndex]                                 
                i = SpectraName.rfind("/")
                if (i > (-1)):
                    SpectraName = SpectraName[i + 1:]
                OrigSpectraName = os.path.basename(SpectraName)
                SpectraName = os.path.splitext(OrigSpectraName)[0]
                break

        # Debug:
        #    print "OldSpectraIndex = ", OldSpectraIndex
        #    print "SpectraName = ", SpectraName


        ## subtract strong molecule contribution
        FreqArray = NewASCIIdata[:, 0]
        for RangeIndex in xrange(len(NewFreqMinList)):
            DataFileIndex = int(NewFreqMinList[RangeIndex][0])
            if (DataFileIndex == (ExpDataFileIndex + 1)):
                CounterFreqRange += 1
                FreqMin = float(NewFreqMinList[RangeIndex][1])
                FreqMax = float(NewFreqMaxList[RangeIndex][1])
                Noise = float(NewNoiseList[RangeIndex][1])
                tBack = float(NewtBackList[RangeIndex][1])
                tSlope = float(NewtSlopeList[RangeIndex][1])

                # Debug:
                #    print "CounterFreqRange = ", CounterFreqRange
                #    print "ListSubtractedModelSpectra = ", ListSubtractedModelSpectra


                ## get modeled spectrum
                CurrentFreqRangeModelValues = ListSubtractedModelSpectra[CounterFreqRange]
                ModelFreqMinIndex = max(0, (numpy.abs(CurrentFreqRangeModelValues[:, 0] - FreqMin)).argmin() - 1)
                ModelFreqMaxIndex = min(len(CurrentFreqRangeModelValues[:, 0]) - 1, (numpy.abs(CurrentFreqRangeModelValues[:, 0] - FreqMax)).argmin() + 1)
                In1 = min(ModelFreqMinIndex, ModelFreqMaxIndex)
                In2 = max(ModelFreqMinIndex, ModelFreqMaxIndex)
                CurrentFreqRangeModelValues = CurrentFreqRangeModelValues[In1:In2, :]


                ## add strong molecule contribution to modeled spectra
                if (StrongMolContribution != []):
                    StrongMolModeledSpectra = StrongMolContribution[1]


                    ## get index for FreqMin and FreqMax
                    AllLocalStrongMolModeledSpectra = StrongMolModeledSpectra[OldSpectraIndex]
                    StrongFreqMinIndex = max(0, (numpy.abs(AllLocalStrongMolModeledSpectra[:, 0] - FreqMin)).argmin() - 1)
                    StrongFreqMaxIndex = min(len(AllLocalStrongMolModeledSpectra[:, 0]) - 1, \
                                             (numpy.abs(AllLocalStrongMolModeledSpectra[:, 0] - FreqMax)).argmin() + 1)
                    In1 = min(StrongFreqMinIndex, StrongFreqMaxIndex)
                    In2 = max(StrongFreqMinIndex, StrongFreqMaxIndex)
                    LocalStrongMolModeledSpectra = AllLocalStrongMolModeledSpectra[In1:In2, :]


                    ## add strong molecule contribution and remove continuum contribution
                    for i in xrange(min(len(LocalStrongMolModeledSpectra[:, 1]), len(CurrentFreqRangeModelValues[:, 1]))):
                        CurrentFreqRangeModelValues[i, 1] = CurrentFreqRangeModelValues[i, 1] + LocalStrongMolModeledSpectra[i, 1]

                # Debug:
                #    print "RangeIndex = ", RangeIndex
                #    print "CurrentFreqRangeModelValues = ", CurrentFreqRangeModelValues


                ## get transition frequencies in current frequency range from database
                LocalElowMax = 1.e6
                LocalMaxNumTransInFit = 0
                TransFreqList, DBParamList = GetTransFreq(molecule, FreqMin, FreqMax, LocalElowMax, dbFile, LocalMaxNumTransInFit)


                ## define all doppler-shifted transition frequencies
                DopplerShiftedTransFreq = []
                if (TransFreqList != []):
                    for TransFreq in TransFreqList:
                        for VelOff in VelocityOffsets:
                            vt = TransFreq * (1.0 - VelOff / ckms)
                            if vt not in DopplerShiftedTransFreq:
                                DopplerShiftedTransFreq.append(vt)
                    DopplerShiftedTransFreq.sort()


                ## write some informations to log file
                LocalResultsFile.append("\n")
                LocalResultsFile.append("\t" + helpstring + "\n")
                LocalResultsFile.append("\tFrequency range:\n\n")
                LocalResultsFile.append("\t\tMin. Frequency = " + str(FreqMin) + "\n")
                LocalResultsFile.append("\t\tMax. Frequency = " + str(FreqMax) + "\n")
                LocalResultsFile.append("\t\tNoise level = " + str(Noise) + "\n\n")


                ## check, if modeled spectrum is identical to zero
                if (numpy.max(abs(CurrentFreqRangeModelValues[:, 1])) <= Noise):
                    EmptySpectrumCounter += 1
                    LocalResultsFile.append("\n")
                    LocalResultsFile.append("\t\tFound no peak above the noise level!\n")
                    LocalResultsFile.append("\t" + helpstring + "\n")
                    LocalResultsFile.append("\n")

                    if (EmptySpectrumCounter == len(ListSubtractedModelSpectra)):
                        LocalResultsFile.append("\n")
                        LocalResultsFile.append("\n")
                        LocalResultsFile.append("\t\tModeled spectrum is zero!\n")
                        LocalResultsFile.append("\n")
                        LocalResultsFile.append("\t\t\tMolecule " + molecule + " is not included!\n")
                        LocalResultsFile.append("\t" + helpstring + "\n")
                        LocalResultsFile.append("\n")
                        ContributionFlag = "false"
                        return (ContributionFlag, LocalResultsFile)
                else:


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## determine the intensities of observed and modeled spectra at doppler-shifted transition frequencies
                    number_points = len(CurrentFreqRangeModelValues[:, 1])                  ## define number of data points
                    AllPeakIndices = []
                    for vt in DopplerShiftedTransFreq:                                      ## loop over all doppler-shifted transition freq.


                        ## get index of central peak in modeled spectrum
                        LocalFreqArray = CurrentFreqRangeModelValues[:, 0]
                        ModelPeakIndex = min(len(LocalFreqArray) - 1, max(0, (numpy.abs(LocalFreqArray - vt)).argmin()))


                        ## get index of central peak in modeled spectrum
                        LocalFreqArray = SubtractedSpectrumLocal[:, 0]
                        ObsPeakIndex = min(len(LocalFreqArray) - 1, max(0, (numpy.abs(LocalFreqArray - vt)).argmin()))


                        ## if current index is not already analyzed before, continue ..
                        if ModelPeakIndex not in AllPeakIndices:
                            AllPeakIndices.append(ModelPeakIndex)


                            ## print some status, why molecule is not included
                            if (abs(CurrentFreqRangeModelValues[ModelPeakIndex, 1]) >= Noise):
                                CounterPeak += 1                                            ## increase counter for all peaks
                                LocalResultsFile.append("\n")
                                LocalResultsFile.append("\t\tPeak of modeled spectrum at: " + str(CurrentFreqRangeModelValues[ModelPeakIndex, 0]) \
                                                        + " MHz\n")
                                LocalResultsFile.append("\t\t\tHeight of modeled spectrum at this frequency (continuum subtracted): " \
                                                        + str(CurrentFreqRangeModelValues[ModelPeakIndex, 1]) + " K\n")
                                LocalResultsFile.append("\t\t\tHeight of observed data at this frequency (continuum subtracted): " \
                                                        + str(SubtractedSpectrumLocal[ObsPeakIndex, 1]) + " K\n")
                                LocalResultsFile.append("\t\t\tDifference between model and exp. data at this frequency: " \
                                                        + str(CurrentFreqRangeModelValues[ModelPeakIndex, 1] - SubtractedSpectrumLocal[ObsPeakIndex, 1]) \
                                                        + " K\n")
                                if (abs(SubtractedSpectrumLocal[ObsPeakIndex, 1]) < NumericalZero):
                                    LocalResultsFile.append("\t\t\tFraction of height: infinity\n")
                                else:
                                    FractionOfHeight = (CurrentFreqRangeModelValues[ModelPeakIndex, 1] / SubtractedSpectrumLocal[ObsPeakIndex, 1]) * 100.0
                                    LocalResultsFile.append("\t\t\tFraction of height: " + str(abs(FractionOfHeight)) + " %\n")


                                ## check, if modeled and observed spectrum describe same type of peak (both emission or both absorption)
                                if ((CurrentFreqRangeModelValues[ModelPeakIndex, 1] * SubtractedSpectrumLocal[ObsPeakIndex, 1]) < 0.0 \
                                    and CheckEmissionAbsorptionFlag):
                                    LocalResultsFile.append("\n")
                                    if (CurrentFreqRangeModelValues[ModelPeakIndex, 1] < 0.0):  ## model spectrum describes absorption at current freq.
                                        line = "\t\t\tModeled spectrum at this frequency describes absorption whereas obs. spectrum describes emission "
                                        line += "or vis versa!"
                                    else:
                                        line = "\t\t\tModeled spectrum at this frequency describes emission whereas obs. spectrum describes absorption "
                                        line += "or vis versa!"
                                    LocalResultsFile.append(line + "\n")
                                    if (DecisionMethod == "strict"):
                                        LocalResultsFile.append("\n\t\t\t\tMolecule " + molecule + " is not included!\n")
                                        LocalResultsFile.append("\t" + helpstring + "\n\n")
                                        ContributionFlag = "false"
                                        break
                                    elif (DecisionMethod == "significant"):
                                        LocalResultsFile.append("\t" + helpstring + "\n\n")
                                        CounterOverestimatedPeaks += 1


                                ##------------------------------------------------------------------------------------------------------------------------
                                ## check, if max. peak height of modeled peak is within user defined range above observed peak height
                                elif (abs((CurrentFreqRangeModelValues[ModelPeakIndex, 1] / SubtractedSpectrumLocal[ObsPeakIndex, 1])) \
                                      > (1.0 + MaxOverestimationHeight / 100.0)):
                                    LocalResultsFile.append("\n")
                                    LocalResultsFile.append("\t\t\tFraction of heights is   O U T   O F   allowed range of " \
                                                            + str(100.0 + MaxOverestimationHeight) + " %\n")
                                    if (DecisionMethod == "strict"):
                                        LocalResultsFile.append("\n")
                                        LocalResultsFile.append("\t\t\t\tMolecule " + molecule + " is not included!\n")
                                        LocalResultsFile.append("\t" + helpstring + "\n")
                                        LocalResultsFile.append("\n")
                                        ContributionFlag = "false"
                                        break
                                    elif (DecisionMethod == "significant"):
                                        LocalResultsFile.append("\t" + helpstring + "\n")
                                        CounterOverestimatedPeaks += 1


                                ##------------------------------------------------------------------------------------------------------------------------
                                ## peak is within user defined range
                                else:
                                    CounterAcceptedPeaks += 1
                                    LocalResultsFile.append("\n")
                                    LocalResultsFile.append("\t\t\tFraction of heights is   W I T H I N   allowed range of " \
                                                            + str(100.0 + MaxOverestimationHeight) + " %\n")
                                    LocalResultsFile.append("\t" + helpstring + "\n")
                                    LocalResultsFile.append("\n")


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## abort loop if mismatch is observed
                if (ContributionFlag == "false"):                                           ## abort if an absolute mismatch is observed
                    break


    ## if no absolute abort creterium is specified ..
    if (ContributionFlag != "false" and CounterPeak > 0):
        if (DecisionMethod == "significant"):                                               ## the number of accepted peaks has to be significant
                                                                                            ## greater than the number of overestimated peaks
            LocalResultsFile.append("\n")
            LocalResultsFile.append("\t" + helpstring + "\n")
            LocalResultsFile.append("\tSummary:\n\n")
            fraction = (float(CounterOverestimatedPeaks) / float(CounterPeak)) * 100.0      ## define fraction of overestimated peaks
            line = "\t\tFraction (number_overestimated_peaks / total_number_all_peaks) =  (" + str(CounterOverestimatedPeaks) + "/" + str(CounterPeak)
            line += ") = " + str(fraction)
            if (fraction > float(Tolerance)):                                               ## is fraction greater than Tolerance % abort!
                line += "% > " + str(Tolerance) + "% !\n"
                LocalResultsFile.append(line)
                LocalResultsFile.append("\n")
                LocalResultsFile.append("\tTo many overestimated peaks!\n")
                LocalResultsFile.append("\n")
                LocalResultsFile.append("\n")
                LocalResultsFile.append("\t\tMolecule " + molecule + " is not included!\n")
                LocalResultsFile.append("\n")
                LocalResultsFile.append("\t" + helpstring + "\n")
                LocalResultsFile.append("\n")
                ContributionFlag = "false"
            else:
                line += "% <= " + str(Tolerance) + "% !\n"
                LocalResultsFile.append(line)
                LocalResultsFile.append("\n")
                LocalResultsFile.append("\n")
                LocalResultsFile.append("\t\tMolecule " + molecule + " is identified!\n")
                LocalResultsFile.append("\n")
                LocalResultsFile.append("\t" + helpstring + "-\n")
                LocalResultsFile.append("\n")


    ## check, if at least one peak was found
    if (CounterPeak == 0):
        LocalResultsFile.append("\n")
        LocalResultsFile.append("\n")
        LocalResultsFile.append("\t" + helpstring + "\n")
        LocalResultsFile.append("\tFound no peak in all modeled spectra!\n")
        LocalResultsFile.append("\n")
        LocalResultsFile.append("\n")
        LocalResultsFile.append("\t\tMolecule " + molecule + " is not included!\n")
        LocalResultsFile.append("\t" + helpstring + "\n")
        LocalResultsFile.append("\n")
        ContributionFlag = "false"


    ## define return variables
    return (ContributionFlag, LocalResultsFile)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## start a single molecule fit
##
def SingleMoleculeFit(ClusterFlag, lock, Node, molecule, StrongMolContribution, PathXCLASSInterface, LineIDjobDir, MoleculejobDirList, \
                      LocalMoleculeIndex, ListOfSpectraNames, ListOfSpectra, ListSubtractedSpectra, MaxOverestimationHeight, ResultsFile, \
                      IdentifiedMoleculeFile, ListNoiseLevels, CheckEmissionAbsorptionFlag, IdentifiedMoleculesDir, NotIdentifiedMoleculesDir, \
                      ListMolDirName, MAGIXExpXMLFilename, DefaultMolfitUsedFlagList, AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, VelBin, \
                      ElowMax, DecisionMethod, Tolerance, TypeSelection, SmoothValue, MaxNumTransInFit):
    """

input parameters:
-----------------

    - ClusterFlag:                  flag indicating if a fit is done on a cluster or not

    - lock:                         lock object used for threading package

    - Node:                         current node

    - molecule:                     current molecule

    - StrongMolContribution:        contribution of strong (highly abundant) molecules

    - PathXCLASSInterface:          path of XCLASS interface

    - LineIDjobDir:                 job directory for current LineID run

    - MoleculejobDirList:           current single molecule fit directory

    - LocalMoleculeIndex:           local molecule index

    - ListOfSpectraNames:           list of all spectrum (observational data) files

    - ListOfSpectra:                list of all observed spectra

    - ListSubtractedSpectra:        list of all continuum subtracted observational spectra

    - MaxOverestimationHeight:      overestimation factor

    - ResultsFile:                  final result file

    - IdentifiedMoleculeFile:       file for intermediate identified molecules

    - ListNoiseLevels:              list of all noise levels for all frequency ranges of all spectra

    - CheckEmissionAbsorptionFlag:  flag to consider emission and absorption separatley

    - IdentifiedMoleculesDir:       path of the directory containing the intermediate identified molecules of the current LineID run

    - NotIdentifiedMoleculesDir:    path of the directory containing the non-identified molecules of the current LineID run

    - ListMolDirName:               list of modified molecule names (used for naming of direcotries)

    - MAGIXExpXMLFilename:          name of MAGIX exp. xml file

    - DefaultMolfitUsedFlagList:    defines, if a default molfit file was used or not

    - AlgorithmXMLFileEstParam:     path and name of algorithm xml file (used for parameter estimation algorithm)

    - AlgorithmXMLFileISO:          path and name of algorithm xml file (used for fit of iso ratios)

    - VelBin:                       velocity resolution for histogram analysis

    - ElowMax:                      highest allowed lower energy of a transition

    - DecisionMethod:               method which is used

    - Tolerance:                    tolerance factor for method "significant"

    - TypeSelection:                exclude type of lines (only emission or only absorption or all types)

    - SmoothValue:                  smooth factor (=0, no smoothing is performed)

    - MaxNumTransInFit:             max. number of transitions which are fitted in paramter estimation algorithm


output parameters:
------------------

    - ContributionFlag:             flag indicting if current molecule contributes or not

    - MoleculeResultName:           path and name of corresponding molfit file

    """

    # Debug:
    #    print "ClusterFlag = ", ClusterFlag
    #    print "Node = ", Node
    #    print "molecule = ", molecule
    #    print "PathXCLASSInterface = ", PathXCLASSInterface
    #    print "LocalMoleculeIndex = ", LocalMoleculeIndex
    #    print "LineIDjobDir + MoleculejobDirList[LocalMoleculeIndex] = ", LineIDjobDir + MoleculejobDirList[LocalMoleculeIndex]
    #    print "ConvertLogLinFlag = ", ConvertLogLinFlag
    #    print "ListOfSpectraNames = ", ListOfSpectraNames
    #    print "ListSubtractedSpectra = ", ListSubtractedSpectra
    #    print "MaxOverestimationHeight = ", MaxOverestimationHeight
    #    print "ResultsFile = ", ResultsFile
    #    print "ListNoiseLevels = ", ListNoiseLevels
    #    print "CheckEmissionAbsorptionFlag = ", CheckEmissionAbsorptionFlag
    #    print "IdentifiedMoleculesDir = ", IdentifiedMoleculesDir
    #    print "ListMolDirName = ", ListMolDirName


    ## initialize output parameters
    ContributionFlag = "false"
    MoleculeResultName = ""


    ## for testing
    DoFinalSMFitFlag = True


    ## define name of current molecule where all not allowed characters were removed from the molecule name
    MolDirName = MoleculeFileName(molecule)


    ## deactivate iso ratio file for single molecule file
    CurrentExpXMLFile = os.path.normpath(LineIDjobDir + "/" + MoleculejobDirList[LocalMoleculeIndex]) + "/" + MAGIXExpXMLFilename
    OrigIsotopologuesList = task_MAGIX.GetXMLtag(CurrentExpXMLFile, "iso_flag")
    if (OrigIsotopologuesList == []):
        OrigIsotopologuesList = task_MAGIX.GetXMLtag(CurrentExpXMLFile, "Isotopologues")
    OrigIsoTableFileNameList = task_MAGIX.GetXMLtag(CurrentExpXMLFile, "IsoTableFileName")
    task_MAGIX.WriteXMLtag(CurrentExpXMLFile, "iso_flag", ["n"])
    task_MAGIX.WriteXMLtag(CurrentExpXMLFile, "Isotopologues", ["n"])

    # Debug:
    #    print "\n\n\n\n"
    #    print "OrigIsotopologuesList = ", OrigIsotopologuesList
    #    print "OrigIsoTableFileNameList = ", OrigIsoTableFileNameList
    #    sys.exit(0)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## use parameter estimation algorithm to create molfit file
    MoleculejobDir = LineIDjobDir + "/" + MoleculejobDirList[LocalMoleculeIndex] + "/"
    MoleculejobDir = os.path.normpath(MoleculejobDir) + "/"
    CurrentMolfitFileName = os.path.normpath(MoleculejobDir + MolDirName + ".molfit")
    MAGIXrootDir = os.path.normpath(PathXCLASSInterface) + "/"


    ## read in some parameters from observational xml file
    UnmodifiedExpXML = CurrentExpXMLFile
    ExpFileList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "FileNamesExpFiles")
    FreqMinList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "MinExpRange")
    FreqMaxList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "MaxExpRange")
    FreqStepList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "StepFrequency")
    t_back_flagList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "t_back_flag")
    tBackList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "BackgroundTemperature")
    tSlopeList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "TemperatureSlope")
    N_HList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "HydrogenColumnDensity")
    beta_dustList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "DustBeta")
    kappa_1300List = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "Kappa")
    NoiseList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "NoiseLevel")
    GlobalvLSRList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "GlobalvLSR")
    TelescopeSizeList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "TelescopeSize")
    Inter_FlagList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "Inter_Flag")
    ErrorYList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "ErrorY")
    NumberHeaderLinesList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "NumberHeaderLines")
    SeparatorColumnsList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "SeparatorColumns")
    IsotopologuesList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "iso_flag")
    if (IsotopologuesList == []):
        IsotopologuesList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "Isotopologues")
    IsoTableFileNameList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "IsoTableFileName")
    dbList = task_MAGIX.GetXMLtag(UnmodifiedExpXML, "dbFilename")
    if (dbList == []):
        dbFile = MAGIXrootDir + "../../"
        dbFile = os.path.normpath(dbFile) + "/Database/cdms_sqlite.db"
        dbList = [dbFile]
    dbFile = dbList[0]

    # Debug:
    #    print "\n\nExpFileList = ", ExpFileList
    #    print "len(ExpFileList) = ", len(ExpFileList)
    #    print "FreqMinList = ", FreqMinList
    #    print "FreqMaxList = ", FreqMaxList
    #    print "FreqStepList = ", FreqStepList
    #    print "t_back_flagList = ", t_back_flagList
    #    print "tBackList = ", tBackList
    #    print "tSlopeList = ", tSlopeList
    #    print "N_HList = ", N_HList
    #    print "beta_dustList = ", beta_dustList
    #    print "kappa_1300List = ", kappa_1300List
    #    print "NoiseList = ", NoiseList
    #    print "GlobalvLSRList = ", GlobalvLSRList
    #    print "TelescopeSizeList = ", TelescopeSizeList
    #    print "Inter_FlagList = ", Inter_FlagList
    #    print "ErrorYList = ", ErrorYList
    #    print "NumberHeaderLinesList = ", NumberHeaderLinesList
    #    print "SeparatorColumnsList = ", SeparatorColumnsList
    #    print "IsotopologuesList = ", IsotopologuesList
    #    print "IsoTableFileNameList = ", IsoTableFileNameList
    #    print "dbList = ", dbList


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## shrink freqeuncy ranges


    ## create xml file for each molecule
    NewExpXMLFileName = CurrentExpXMLFile
    velLowLimit = -100.0
    velUpLimit = 100.0
    MaxWidth = 100.0
    ElowMaxLocal = 1.e6
    LocalMaxNumTransInFit = 1.e99
    CreateXMLFile(molecule, velLowLimit, velUpLimit, MaxWidth, NewExpXMLFileName, ExpFileList, FreqMinList, FreqMaxList, FreqStepList, \
                  t_back_flagList, tBackList, tSlopeList, N_HList, beta_dustList, kappa_1300List, NoiseList, GlobalvLSRList, TelescopeSizeList, \
                  Inter_FlagList, ErrorYList, NumberHeaderLinesList, SeparatorColumnsList, IsotopologuesList, IsoTableFileNameList, dbList, \
                  ElowMaxLocal, LocalMaxNumTransInFit)


    ## get path and names of observational data files from shrinked xml file
    NewExpFileList = task_MAGIX.GetXMLtag(NewExpXMLFileName, "FileNamesExpFiles")
    NewFreqMinList = task_MAGIX.GetXMLtag(NewExpXMLFileName, "MinExpRange")
    NewFreqMaxList = task_MAGIX.GetXMLtag(NewExpXMLFileName, "MaxExpRange")
    NewtBackList = task_MAGIX.GetXMLtag(NewExpXMLFileName, "BackgroundTemperature")
    NewtSlopeList = task_MAGIX.GetXMLtag(NewExpXMLFileName, "TemperatureSlope")
    NewNoiseList = task_MAGIX.GetXMLtag(NewExpXMLFileName, "NoiseLevel")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## subtract strong molecule contribution from observational data
    if (StrongMolContribution != []):
        StrongMolModeledSpectra = StrongMolContribution[1]                                  ## get modeled spectra of strong molecules


        ## modify ListSubtractedSpectra or add strong molecule spectra to modeled spectra
        NewListNumberHeaderLines = []
        NewListSeparatorColumns = []
        ListSubtractedStrongMolModelSpectra = []
        for ExpDataFileIndex in xrange(len(NewExpFileList)):
            NewExpFileName = NewExpFileList[ExpDataFileIndex]
            NewSpectraName = os.path.basename(NewExpFileName)
            NewListNumberHeaderLines.append("0")
            NewListSeparatorColumns.append(" ")


            ## determine corresponding observed spectrum
            OldSpectraIndex = 0
            for LocalSpectraIndex in xrange(len(ListOfSpectraNames)):
                if (NewSpectraName == os.path.basename(ListOfSpectraNames[LocalSpectraIndex])):
                    NewASCIIdata = ListOfSpectra[LocalSpectraIndex]
                    OldSpectraIndex = LocalSpectraIndex
                    break


            ## subtract strong molecule contribution
            FreqArray = NewASCIIdata[:, 0]
            IntArray = NewASCIIdata[:, 1]
            for RangeIndex in xrange(len(NewFreqMinList)):
                DataFileIndex = int(NewFreqMinList[RangeIndex][0])
                if (DataFileIndex == (ExpDataFileIndex + 1)):
                    FreqMin = float(NewFreqMinList[RangeIndex][1])
                    FreqMax = float(NewFreqMaxList[RangeIndex][1])
                    tBack = float(NewtBackList[RangeIndex][1])
                    tSlope = float(NewtSlopeList[RangeIndex][1])


                    ## get index for FreqMin and FreqMax of observational data spectrum
                    FreqMinIndex = max(0, (numpy.abs(FreqArray - FreqMin)).argmin() - 1)
                    FreqMaxIndex = min(len(FreqArray) - 1, (numpy.abs(FreqArray - FreqMax)).argmin() + 1)


                    ## get index for FreqMin and FreqMax of modeled (strong molecule) spectrum
                    AllLocalStrongMolModeledSpectra = StrongMolModeledSpectra[OldSpectraIndex]
                    StrongFreqMinIndex = max(0, (numpy.abs(AllLocalStrongMolModeledSpectra[:, 0] - FreqMin)).argmin() - 1)
                    StrongFreqMaxIndex = min(len(AllLocalStrongMolModeledSpectra[:, 0]) - 1, \
                                             (numpy.abs(AllLocalStrongMolModeledSpectra[:, 0] - FreqMax)).argmin() + 1)
                    In1 = min(StrongFreqMinIndex, StrongFreqMaxIndex)
                    In2 = max(StrongFreqMinIndex, StrongFreqMaxIndex)
                    LocalStrongMolModeledSpectra = AllLocalStrongMolModeledSpectra[In1:In2, :]


                    ## subtract strong molecule contribution from observational data
                    for kkk in xrange(min(len(IntArray), len(LocalStrongMolModeledSpectra[:, 0]))):
                        NewASCIIdata[FreqMinIndex + kkk, 1] = IntArray[FreqMinIndex + kkk] - LocalStrongMolModeledSpectra[kkk, 1]


            ## save modified observational data array to file
            numpy.savetxt(NewExpFileName, NewASCIIdata, delimiter='\t')


            ## update number of header lines and separator characters
            task_MAGIX.WriteXMLtag(NewExpXMLFileName, "NumberHeaderLines", NewListNumberHeaderLines)
            task_MAGIX.WriteXMLtag(NewExpXMLFileName, "SeparatorColumns", NewListSeparatorColumns)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## print what you do
    print "\rStart single molecule fit for " + chr(34) + molecule + chr(34) + " on node: " + Node + "                "


    ## ssh charger 'cd ~/magma; nohup nice /usr/bin/magma magma.script > /var/tmp/magma.out' &
    if (ClusterFlag == "false"):
        cmdString = "cd " + PathXCLASSInterface + "; python magix_start.py --quiet --model=myxclass"
        cmdString +=  " " + MoleculejobDir + "io_control.xml >/dev/null"
    else:
        cmdString = "ssh " + Node + " " + chr(34) + "cd " + PathXCLASSInterface + "; "
        cmdString += "nohup python magix_start.py --plotsaveonly --model=myxclass"
        cmdString +=  " " + MoleculejobDir + "io_control.xml >/dev/null" + chr(34)

    # Debug:
    # print "\n\n\n\ncmdString = ", cmdString


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start MAGIX run
    os.system(cmdString)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read in MAGIX result
    NameOfFunction = "LineIdentification"
    ConvertLogLinFlag = "true"
    printInfoFlag = "false"
    LocalJobDir = LineIDjobDir + MoleculejobDirList[LocalMoleculeIndex]
    MolfitFile, IsoRatioFileContent, ListSubtractedModelSpectra = GetBestResult(LocalJobDir, ConvertLogLinFlag, NewExpFileList, NewFreqMinList, \
                                                                                NewFreqMaxList, NewtBackList, NewtSlopeList)

    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## check, if molecule contributes to current spectrum or not
    ContributionFlag, LocalResultsFile = CheckContribution(molecule, MolfitFile, NewExpFileList, NewFreqMinList, NewFreqMaxList, \
                                                           ListSubtractedSpectra, ListSubtractedModelSpectra, MaxOverestimationHeight, LineIDjobDir, \
                                                           NewtBackList, NewtSlopeList, NewNoiseList, ListOfSpectraNames, ListOfSpectra, \
                                                           StrongMolContribution, CheckEmissionAbsorptionFlag, DecisionMethod, Tolerance, dbFile)
    ## read in intermediate result file
    helpString = "=" * 154
    ResultsFile.write(helpString + "\n")
    ResultsFile.write("Results of single-molecule fit for molecule: " + chr(34) + molecule + chr(34) + ":\n")
    for line in LocalResultsFile:
        ResultsFile.write(line)
        ResultsFile.flush()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## if molecule contributes, save molfit file, add molecule to the list of identified molecules and fit iso ratios (if required)
    if (ContributionFlag == "true"):

        # Debug:
        # print "\n\n\nMolfitFile = ", MolfitFile


        ## write best molfit file to subdirectory IdentifiedMoleculesDir and create corresponding plot
        MoleculeResultName = IdentifiedMoleculesDir + ListMolDirName[LocalMoleculeIndex]
        BestMolfitFile = open(MoleculeResultName + ".out.molfit", 'w')
        for line in MolfitFile:
            BestMolfitFile.write(line)
        BestMolfitFile.close()


        ## write message to output files
        helpString = "-" * 154
        IdentifiedMoleculeFile.write(helpString + "\n")
        IdentifiedMoleculeFile.write("Molecule " + chr(34) + molecule + chr(34) + " was identified!\n")
        IdentifiedMoleculeFile.flush()

        # Debug:
        #    print "\n\n\n\n\n"
        #    print "LocalMoleculeIndex = ", LocalMoleculeIndex
        #    print "DefaultMolfitUsedFlagList[LocalMoleculeIndex] = ", DefaultMolfitUsedFlagList[LocalMoleculeIndex]
        #    print "OrigIsotopologuesList = ", OrigIsotopologuesList


    ##------------------------------------------------------------------------------------------------------------------------------------------------
    ## create a plot of the current molecule and store the plot into the identified or not-identified directory
    CounterFreqRange = (-1)
    for ExpDataFileIndex in xrange(len(NewExpFileList)):
        NewExpFileName = NewExpFileList[ExpDataFileIndex]
        NewSpectraName = os.path.basename(NewExpFileName)

        # Debug:
        #    print "\n\nNewExpFileName = ", NewExpFileName
        #    print "NewSpectraName = ", NewSpectraName


        ## determine corresponding observed unmodified spectrum
        # SpectraName = ""
        OldSpectraIndex = 0
        for LocalSpectraIndex in xrange(len(ListOfSpectraNames)):
            if (NewSpectraName == os.path.basename(ListOfSpectraNames[LocalSpectraIndex])):
                OldSpectraIndex = LocalSpectraIndex
                NewASCIIdata = ListOfSpectra[LocalSpectraIndex]
                SubtractedSpectrumLocal = ListSubtractedSpectra[LocalSpectraIndex]


                ## get name of exp. data file of current frequency range
                SpectraName = ListOfSpectraNames[LocalSpectraIndex]                                 
                i = SpectraName.rfind("/")
                if (i > (-1)):
                    SpectraName = SpectraName[i + 1:]
                SpectraName = os.path.basename(SpectraName)
                break

        # Debug:
        #    print "OldSpectraIndex = ", OldSpectraIndex
        #    print "SpectraName = ", SpectraName
        #    print "SubtractedSpectrumLocal = ", SubtractedSpectrumLocal


        ## subtract strong molecule contribution
        FreqArray = NewASCIIdata[:, 0]
        for RangeIndex in xrange(len(NewFreqMinList)):
            DataFileIndex = int(NewFreqMinList[RangeIndex][0])
            if (DataFileIndex == (ExpDataFileIndex + 1)):
                CounterFreqRange += 1
                FreqMin = float(NewFreqMinList[RangeIndex][1])
                FreqMax = float(NewFreqMaxList[RangeIndex][1])
                Noise = float(NewNoiseList[RangeIndex][1])

                # Debug:
                #    print "\n\n\nRangeIndex = ", RangeIndex
                #    print "FreqMin = ", FreqMin
                #    print "FreqMax = ", FreqMax


                ## get modeled spectrum
                CurrentFreqRangeModelValues = ListSubtractedModelSpectra[CounterFreqRange]
                ModelFreqMinIndex = max(0, (numpy.abs(CurrentFreqRangeModelValues[:, 0] - FreqMin)).argmin() - 1)
                ModelFreqMaxIndex = min(len(CurrentFreqRangeModelValues[:, 0]) - 1, \
                                        (numpy.abs(CurrentFreqRangeModelValues[:, 0] - FreqMax)).argmin() + 1)
                In1 = min(ModelFreqMinIndex, ModelFreqMaxIndex)
                In2 = max(ModelFreqMinIndex, ModelFreqMaxIndex)
                CurrentFreqRangeModelValues = CurrentFreqRangeModelValues[In1:In2, :]

                # Debug:
                #    print "SpectraName = ", SpectraName
                #    print "CurrentFreqRangeModelValues = ", CurrentFreqRangeModelValues


                ## add strong molecule contribution to modeled spectra
                if (StrongMolContribution != []):
                    AllLocalStrongMolModeledSpectra = StrongMolModeledSpectra[OldSpectraIndex]
                    StrongFreqMinIndex = max(0, (numpy.abs(AllLocalStrongMolModeledSpectra[:, 0] - FreqMin)).argmin() - 1)
                    StrongFreqMaxIndex = min(len(AllLocalStrongMolModeledSpectra[:, 0]) - 1, \
                                             (numpy.abs(AllLocalStrongMolModeledSpectra[:, 0] - FreqMax)).argmin() + 1)
                    In1 = min(StrongFreqMinIndex, StrongFreqMaxIndex)
                    In2 = max(StrongFreqMinIndex, StrongFreqMaxIndex)
                    LocalStrongMolModeledSpectra = AllLocalStrongMolModeledSpectra[In1:In2, :]


                    ## subtract strong molecule contribution from observational data
                    for kkk in xrange(min(len(CurrentFreqRangeModelValues[:, 1]), len(LocalStrongMolModeledSpectra[:, 1]))):
                        CurrentFreqRangeModelValues[kkk, 1] = CurrentFreqRangeModelValues[kkk, 1] + LocalStrongMolModeledSpectra[kkk, 1]

                # Debug:
                #    print "ExpDataFileIndex = ", ExpDataFileIndex
                #    print "RangeIndex = ", RangeIndex
                #    print "CurrentFreqRangeModelValues = ", CurrentFreqRangeModelValues
                #    print "ListSubtractedModelSpectra[CounterFreqRange] = ", ListSubtractedModelSpectra[CounterFreqRange]


                ## plot window only, if modeled spectrum has a length > 1
                if (len(CurrentFreqRangeModelValues[:, 0]) > 1):


                    ## get observation data for current freq. range
                    ymin1 = numpy.min(CurrentFreqRangeModelValues[:, 1])
                    ymax1 = numpy.max(CurrentFreqRangeModelValues[:, 1])


                    ## get range indices for exp. data
                    LocalCopyArray = SubtractedSpectrumLocal[:, 0]
                    FreqMinIndexLocal = max(0, (numpy.abs(LocalCopyArray - FreqMin)).argmin() - 1)
                    FreqMaxIndexLocal = min(len(LocalCopyArray) - 1, (numpy.abs(LocalCopyArray - FreqMax)).argmin() + 1)


                    ## cut out data within frequency range
                    xminIndex = min(FreqMinIndexLocal, FreqMaxIndexLocal)
                    xmaxIndex = max(FreqMinIndexLocal, FreqMaxIndexLocal)
                    SubtractedSpectrum = SubtractedSpectrumLocal[xminIndex:xmaxIndex, :]

                    # Debug:
                    #    print "FreqMin = ", FreqMin
                    #    print "FreqMax = ", FreqMax
                    #    print "FreqMinIndexLocal = ", FreqMinIndexLocal
                    #    print "FreqMaxIndexLocal = ", FreqMaxIndexLocal
                    #    print "xminIndex = ", xminIndex
                    #    print "xmaxIndex = ", xmaxIndex
                    #    print "LocalCopyArray = ", LocalCopyArray
                    #    print "SubtractedSpectrum = ", SubtractedSpectrum


                    ## determine ymin and ymax
                    ymax2 = numpy.max(SubtractedSpectrum[:, 1])
                    ymax = max(ymax1, ymax2)
                    ymin2 = numpy.min(SubtractedSpectrum[:, 1])
                    ymin = min(ymin1, ymin2)
                    if (ymin == 0.0 and ymax != 0.0):
                        ymin = ymax * 1.e-2


                    ## define extended spectral name
                    FreqMinString = "%.3f" % FreqMin
                    FreqMaxString = "%.3f" % FreqMax
                    ExtendedSpectraName = SpectraName + "__" + FreqMinString + "_-_" + FreqMaxString + "_MHz"


                    ##************************************************************************************************************************************
                    ## create a plot if current molecule does not contribute to spectra
                    fig, layer = matplotlib.pyplot.subplots()
                    fig.set_size_inches(15.0, 10.0)
                    layer.grid(True)
                    layer.set_ylabel(r"T$_{mb}$ [K]")
                    layer.set_xlabel("Rest Frequency [MHz]")
                    layer.axhline(y=Noise, color='blue', linewidth=1, linestyle='--')
                    if (numpy.min(CurrentFreqRangeModelValues[:, 1]) < 0 or numpy.min(SubtractedSpectrum[:, 1]) < 0):
                        layer.axhline(y=-Noise, color='blue', linewidth=1, linestyle='--')

                    # Debug:
                    #    print "molecule = ", molecule
                    #    print "FreqMin, FreqMax = ", FreqMin, FreqMax


                    ## mark non-doppler shifted transition frequencies
                    LocalElowMax = 1.e6
                    LocalMaxNumTransInFit = 0
                    TransFreqList, DBParamList = GetTransFreq(molecule, FreqMin, FreqMax, LocalElowMax, dbFile, LocalMaxNumTransInFit)
                    if (TransFreqList != []):
                        for TransFreq in TransFreqList:
                            layer.axvline(x=TransFreq, color='green', linewidth=2, linestyle='--')

                        # Debug:
                        #    print "TransFreqList = ", TransFreqList


                        ##********************************************************************************************************************************
                        ## add frequency axis to plot
                        RestFreq = TransFreqList[0]
                        vLSR = 0.0
                        ckms = 299792458.0 * 1.e-3                                      ## define speed of light in km/s
                        f1 = FreqMin
                        v1 = vLSR + (ckms / RestFreq) * (RestFreq - f1)
                        f2 = FreqMax
                        v2 = vLSR + (ckms / RestFreq) * (RestFreq - f2)


                        ## add velociy axis to plot
                        ax2 = layer.twiny()
                        ax2.set_xlim(v1, v2)
                        ax2.set_xlabel("velocity [km/s]")
                        ax2.xaxis.set_major_formatter(matplotlib.ticker.FormatStrFormatter('%.2f'))


                    ##************************************************************************************************************************************
                    ## add observational and modeled spectra to plot
                    layer.plot(SubtractedSpectrum[:, 0], SubtractedSpectrum[:, 1], '-', color='black', linewidth=3.0, label='exp. data', \
                               drawstyle='steps-mid')
                    layer.plot(CurrentFreqRangeModelValues[:, 0], CurrentFreqRangeModelValues[:, 1], '-', color='red', linewidth=2.0, label = 'fit')
                    if (abs(ymin - ymax) > 1.e-9):
                        layer.set_ylim(ymin * 1.007, ymax * 1.007)
                    else:
                        layer.set_ylim(-1.0, 1.0)
                    layer.set_xlim(FreqMin, FreqMax)
                    xres = (FreqMax - FreqMin) / 3.0
                    ticksX = numpy.arange(FreqMin, (FreqMax + xres * 0.999), xres)
                    layer.set_xticks(ticksX)
                    layer.xaxis.set_major_formatter(matplotlib.ticker.FormatStrFormatter('%.5e'))
                    layer.legend()
                    if (ContributionFlag == "true"):
                        fig.savefig(MoleculeResultName + "__" + ExtendedSpectraName + ".png", dpi=100)
                    else:
                        fig.savefig(NotIdentifiedMoleculesDir + ListMolDirName[LocalMoleculeIndex] + "__" + ExtendedSpectraName + ".png", dpi=100)
                    matplotlib.pyplot.close(fig)


    ## define return variables
    return (ContributionFlag, MoleculeResultName)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## starts worker via command line and analyze results
##
def StartWorker(ThreadID, ServerList, ListOfMolecules, StrongMolContribution, myXCLASSrootDir, RelativeMAGIXDir, LineIDjobDir, MoleculejobDirList, \
                ListOfSpectraNames, ListOfSpectra, ListSubtractedSpectra, MaxOverestimationHeight, ResultsFile, IdentifiedMoleculeFile, ListNoiseLevels, \
                CheckEmissionAbsorptionFlag, IdentifiedMoleculesDir, NotIdentifiedMoleculesDir, ListMolDirName, MAGIXExpXMLFilename, \
                DefaultMolfitUsedFlagList, AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, VelBin, ElowMax, DecisionMethod, Tolerance, TypeSelection, \
                SmoothValue, MaxNumTransInFit):
    """

input parameters:
-----------------

    - ThreadID:                     current thread id

    - ServerList:                   list of all server in the cluster and the corresponding settings 

    - ListOfMolecules:              list of molecules

    - StrongMolContribution:        contribution of strong (highly abundant) molecules

    - myXCLASSrootDir:              root directory of the XCLASS package

    - RelativeMAGIXDir:             relative MAGIX directory

    - LineIDjobDir:                 job directory for current LineID run

    - MoleculejobDirList:           current single molecule fit directory

    - ListOfSpectraNames:           list of all spectrum (observational data) files

    - ListOfSpectra:                list of all observed spectra

    - ListSubtractedSpectra:        list of all continuum subtracted observational spectra

    - MaxOverestimationHeight:      overestimation factor

    - ResultsFile:                  final result file

    - IdentifiedMoleculeFile:       file for intermediate identified molecules

    - ListNoiseLevels:              list of all noise levels for all frequency ranges of all spectra

    - CheckEmissionAbsorptionFlag:  flag to consider emission and absorption separatley

    - IdentifiedMoleculesDir:       path of the directory containing the intermediate identified molecules of the current LineID run

    - NotIdentifiedMoleculesDir:    path of the directory containing the non-identified molecules of the current LineID run

    - ListMolDirName:               list of modified molecule names (used for naming of direcotries)

    - MAGIXExpXMLFilename:          name of MAGIX exp. xml file

    - DefaultMolfitUsedFlagList:    defines, if a default molfit file was used or not

    - AlgorithmXMLFileEstParam:     path and name of algorithm xml file (used for parameter estimation algorithm)

    - AlgorithmXMLFileISO:          path and name of algorithm xml file (used for fit of iso ratios)

    - VelBin:                       velocity resolution for histogram analysis

    - ElowMax:                      highest allowed lower energy of a transition

    - DecisionMethod:               method which is used

    - Tolerance:                    tolerance factor for method "significant"

    - TypeSelection:                exclude type of lines (only emission or only absorption or all types)


output parameters:
------------------

    - None
    """

    # Debug:
    #    print "ThreadID = ", ThreadID
    #    print "ThreadID, ServerList = ", ThreadID, ServerList
    #    print "ListOfMolecules = ", ListOfMolecules
    #    print "RelativeMAGIXDir = ", RelativeMAGIXDir
    #    print "ConvertLogLinFlag = ", ConvertLogLinFlag
    #    print "ListSubtractedSpectra = ", ListSubtractedSpectra
    #    print "ListSubtractedModelSpectra = ", ListSubtractedModelSpectra
    #    print "MaxOverestimationHeight = ", MaxOverestimationHeight
    #    print "ResultsFile = ", ResultsFile
    #    print "ListNoiseLevels = ", ListNoiseLevels
    #    print "ListOfSpectraNames = ", ListOfSpectraNames
    #    print "CheckEmissionAbsorptionFlag = ", CheckEmissionAbsorptionFlag
    #    print "IdentifiedMoleculesDir = ", IdentifiedMoleculesDir
    #    print "ListMolDirName = ", ListMolDirName


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## increase global molecule counter for all threads
    # lock = threading.Lock()
    lock = ""
    InitFlag = "true"
    NumMol = len(ListOfMolecules)
    NumWorkers = len(ServerList)
    for LocalMoleculeIndex in xrange(ThreadID, NumMol, NumWorkers):
        molecule = ListOfMolecules[LocalMoleculeIndex]

        # Debug:
        # print "\n\n  LocalMoleculeIndex = ", LocalMoleculeIndex


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## get some parameters for each node in the cluster
        server = ServerList[ThreadID]                                                       ## get cluster settings for the server ThreadID
        Node = server[0]                                                                    ## get name of node
        NumCores = server[1]                                                                ## get number of cores
        RootPath = server[2].strip()                                                        ## get path of xclass interface
        if (RootPath != ""):                                                                ## if path of xclass interface is defined ..
            RootPath = RootPath + "/programs/MAGIX/"                                        ## .. define path of MAGIX


        ## define path of the xclass interface on the different nodes
        AbsMAGIXDir = os.path.normpath(myXCLASSrootDir + "/" + RelativeMAGIXDir)
        if (RootPath.strip() == ""):
            PathXCLASSInterface = AbsMAGIXDir
        else:
            PathXCLASSInterface = RootPath


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## define cluster flag
        ClusterFlag = "true"
        if (len(ServerList) == 1 and Node == "localhost"):
            ClusterFlag = "false"


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## do single molecule fit
        ContributionFlag, MoleculeResultName = SingleMoleculeFit(ClusterFlag, lock, Node, molecule, StrongMolContribution, PathXCLASSInterface, \
                                                                 LineIDjobDir, MoleculejobDirList, LocalMoleculeIndex, ListOfSpectraNames, \
                                                                 ListOfSpectra, ListSubtractedSpectra, MaxOverestimationHeight, ResultsFile, \
                                                                 IdentifiedMoleculeFile, ListNoiseLevels, CheckEmissionAbsorptionFlag, \
                                                                 IdentifiedMoleculesDir, NotIdentifiedMoleculesDir, ListMolDirName, MAGIXExpXMLFilename, \
                                                                 DefaultMolfitUsedFlagList, AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, VelBin, \
                                                                 ElowMax, DecisionMethod, Tolerance, TypeSelection, SmoothValue, MaxNumTransInFit)
    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## start single molecule fits related to strong lines
##
def StrongLinesFits(StrongMoleculeList, ListOfMolecules, myXCLASSRootDirectory, MAGIXrootDir, LineIDjobDir, MoleculejobDirList, \
                    ListOfSpectraNames, ListOfSpectra, ListSubtractedSpectra, MaxOverestimationHeight, ResultsFile, IdentifiedMoleculeFile, \
                    ListNoiseLevels, CheckEmissionAbsorptionFlag, IdentifiedMoleculesDir, NotIdentifiedMoleculesDir, ListMolDirName, \
                    MAGIXExpXMLFilename, UnmodifiedExpXMLFilename, DefaultMolfitUsedFlagList, AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, \
                    VelBin, ElowMax, DecisionMethod, Tolerance, TypeSelection, SmoothValue, MaxNumTransInFit):
    """

input parameters:
-----------------

    - StrongMoleculeList:           list of all strong molecules, i.e. molecules with a high abundance

    - ListOfMolecules:              list of all molecules

    - myXCLASSRootDirectory:        root directory of the XCLASS package

    - MAGIXrootDir:                 MAGIX root directory

    - LineIDjobDir:                 job directory for current LineID run

    - MoleculejobDirList:           current single molecule fit directory

    - ListOfSpectraNames:           list of all spectrum (observational data) files

    - ListOfSpectra:                list of all observed spectra

    - ListSubtractedSpectra:        list of all continuum subtracted observational spectra

    - MaxOverestimationHeight:      overestimation factor

    - ResultsFile:                  final result file

    - IdentifiedMoleculeFile:       file for intermediate molecules

    - ListNoiseLevels:              list of all noise levels for all frequency ranges of all spectra

    - CheckEmissionAbsorptionFlag:  flag to consider emission and absorption separatley

    - IdentifiedMoleculesDir:       path of the directory containing the intermediate identified molecules of the current LineID run

    - NotIdentifiedMoleculesDir:    path of the directory containing the non-identified molecules of the current LineID run

    - ListMolDirName:               list of modified molecule names (used for naming of direcotries)

    - MAGIXExpXMLFilename:          name of MAGIX exp. xml file

    - UnmodifiedExpXMLFilename:     path and name of unmodified observational xml file

    - DefaultMolfitUsedFlagList:    defines, if a default molfit file was used or not

    - AlgorithmXMLFileEstParam:     path and name of algorithm xml file (used for parameter estimation algorithm)

    - AlgorithmXMLFileISO:          path and name of algorithm xml file (used for fit of iso ratios)

    - VelBin:                       velocity resolution for histogram analysis

    - ElowMax:                      highest allowed lower energy of a transition

    - DecisionMethod:               method which is used

    - Tolerance:                    tolerance factor for method "significant"

    - TypeSelection:                exclude type of lines (only emission or only absorption or all types)

    - SmoothValue:                  smooth factor (=0, no smoothing is performed)

    - MaxNumTransInFit:             max. number of transitions which are fitted in paramter estimation algorithm


output parameters:
------------------

    - StrongMolContribution:        contribution of all identified strong molecules to each single molecule fit:

                                    - StrongMolContribution[0]:   molfit parameters

                                    - StrongMolContribution[1]:   corresponding modeled spectra

    - NewListOfMolecules:           correct list of molecules
    """


    ## initialize output parameters
    StrongMolContribution = []
    NewListOfMolecules = []
    ModelSpectra = []


    ## check, if no strong molecule is defined
    if (StrongMoleculeList == []):
        return (StrongMolContribution, NewListOfMolecules)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## do single molecule fit for each strong molecule
    ClusterFlag = "false"
    RelativeMAGIXDir = MAGIXrootDir.replace(myXCLASSRootDirectory, "")
    PathXCLASSInterface = os.path.normpath(myXCLASSRootDirectory + "/" + RelativeMAGIXDir)
    Node = "localhost"
    lock = ""
    LocalMoleculeIndex = 1
    StrongMolMolfit = []
    StrongMolContributionLocal = []
    CopyListOfMolecules = ListOfMolecules
    for StrongMolecule in StrongMoleculeList:

        # Debug:
        #    print "StrongMolecule = ", StrongMolecule
        #    print "ClusterFlag = ", ClusterFlag
        #    print "Node = ", Node
        #    print "PathXCLASSInterface = ", PathXCLASSInterface
        #    print "LineIDjobDir = ", LineIDjobDir
        #    print "MoleculejobDirList = ", MoleculejobDirList
        #    print "LocalMoleculeIndex = ", LocalMoleculeIndex
        #    print "ListOfSpectraNames = ", ListOfSpectraNames
        #    print "ListSubtractedSpectra = ", ListSubtractedSpectra
        #    print "CheckEmissionAbsorptionFlag = ", CheckEmissionAbsorptionFlag
        #    print "IdentifiedMoleculesDir = ", IdentifiedMoleculesDir
        #    print "NotIdentifiedMoleculesDir = ", NotIdentifiedMoleculesDir
        #    print "ListMolDirName = ", ListMolDirName
        #    print "MAGIXExpXMLFilename = ", MAGIXExpXMLFilename


        ## determine LocalMoleculeIndex
        for i in xrange(len(CopyListOfMolecules)):
            if (CopyListOfMolecules[i].strip() == StrongMolecule.strip()):
                LocalMoleculeIndex = i
                break


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## do single molecule fit
        ContributionFlag, MolfitsFileName = SingleMoleculeFit(ClusterFlag, lock, Node, StrongMolecule, StrongMolContributionLocal, PathXCLASSInterface, \
                                                              LineIDjobDir, MoleculejobDirList, LocalMoleculeIndex, ListOfSpectraNames, ListOfSpectra, \
                                                              ListSubtractedSpectra, MaxOverestimationHeight, ResultsFile, IdentifiedMoleculeFile, \
                                                              ListNoiseLevels, CheckEmissionAbsorptionFlag, IdentifiedMoleculesDir, \
                                                              NotIdentifiedMoleculesDir, ListMolDirName, MAGIXExpXMLFilename, DefaultMolfitUsedFlagList, \
                                                              AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, VelBin, ElowMax, DecisionMethod, \
                                                              Tolerance, TypeSelection, SmoothValue, MaxNumTransInFit)
        MolfitsFileName += ".out.molfit"
        StrongMolMolfit.append(MolfitsFileName)

        # Debug:
        #    print "ContributionFlag = ", ContributionFlag
        #    print "MolfitsFileName = ", MolfitsFileName


        ## remove an identified molecule from list of all molecules
        CopyListOfMolecules = ListOfMolecules
        ListOfMolecules = []
        for mol in CopyListOfMolecules:
            if (mol.strip() != StrongMolecule.strip()):
                ListOfMolecules.append(mol)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## if a strong molecule was identified
        if (ContributionFlag == "true"):


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## determine modeled spectra


            ## prepare molfit file
            LocalMolfitFile = open(MolfitsFileName)
            LocalMolfitFileContent = LocalMolfitFile.readlines()
            LocalMolfitFile.close()
            LocalMolfitsFileName = LineIDjobDir + "/strong_molecule__all.molfit"
            # LocalMolfitsFileName = MolfitsFileName
            NewLocalMolfitFile = open(LocalMolfitsFileName, 'w')
            for line in LocalMolfitFileContent:
                StrippedLine = line.strip()
                SplittedLine = StrippedLine.split()
                if (len(SplittedLine) > 2):
                    line = line.lower()
                    line = line.replace("y", "n")                                           ## we don't want to fit, we just use the model values
                NewLocalMolfitFile.write(line)
            NewLocalMolfitFile.close()


            ## prepare call of myXCLASSFit function
            LocalNumberIteration = 1
            LocalexperimentalData = UnmodifiedExpXMLFilename
            LocalRestFreq = 0.0
            LocalvLSR = 0.0


            ## define dummy arguments
            LocalTelescopeSize = 3.5
            LocalInter_Flag = False
            Localt_back_flag = True
            LocaltBack = 1.1
            LocaltSlope = 0.0
            LocalnH_flag = True
            LocalN_H = 3.e+24
            Localbeta_dust = 0.0
            Localkappa_1300 = 0.002
            Localiso_flag = False
            LocalIsoTableFileName = ""
            LocalAlgorithmXMLFile = ""


            ## start myXCLASSFit function
            newmolfit, modeldata, JobDir = task_myXCLASSFit.myXCLASSFit(LocalNumberIteration, LocalAlgorithmXMLFile, LocalMolfitsFileName, \
                                                                        LocalexperimentalData, LocalTelescopeSize, LocalInter_Flag, Localt_back_flag, \
                                                                        LocaltBack, LocaltSlope, LocalnH_flag, LocalN_H, Localbeta_dust, \
                                                                        Localkappa_1300, Localiso_flag, LocalIsoTableFileName, LocalRestFreq, LocalvLSR)
            ## get names of obs. data files and some other parameters
            ExpFileList = task_MAGIX.GetXMLtag(UnmodifiedExpXMLFilename, "FileNamesExpFiles")
            NumberHeaderLinesList = task_MAGIX.GetXMLtag(UnmodifiedExpXMLFilename, "NumberHeaderLines")
            SeparatorColumnsList = task_MAGIX.GetXMLtag(UnmodifiedExpXMLFilename, "SeparatorColumns")
            UnmodifiedFreqMinList = task_MAGIX.GetXMLtag(UnmodifiedExpXMLFilename, "MinExpRange")
            UnmodifiedFreqMaxList = task_MAGIX.GetXMLtag(UnmodifiedExpXMLFilename, "MaxExpRange")
            UnmodifiedtBackList = task_MAGIX.GetXMLtag(UnmodifiedExpXMLFilename, "BackgroundTemperature")
            UnmodifiedtSlopeList = task_MAGIX.GetXMLtag(UnmodifiedExpXMLFilename, "TemperatureSlope")


            ## read in modeled spectra
            modeldata = []
            for ExpDataFileIndex in xrange(len(ExpFileList)):
                if (NumberHeaderLinesList[ExpDataFileIndex] == []):
                    NumberHeaderLines = (-1)
                else:
                    NumberHeaderLines = int(NumberHeaderLinesList[ExpDataFileIndex][1])
                if (SeparatorColumnsList[ExpDataFileIndex] == []):
                    SeparatorColumns = ""
                else:
                    SeparatorColumns = SeparatorColumnsList[ExpDataFileIndex][1]


                ## define name of modeled spectra 
                LocalObsFileName = ExpFileList[ExpDataFileIndex]
                LocalObsFileName = os.path.basename(LocalObsFileName)
                if (LocalObsFileName.endswith(".dat")):
                    LocalObsFileName = LocalObsFileName.replace(".dat", ".LM.out.dat")
                elif (LocalObsFileName.endswith(".fits")):
                    LocalObsFileName = LocalObsFileName.replace(".fits", ".LM.out.fits")


                ## import exp data
                if (SeparatorColumns.strip() == ""):
                    ImportExpData = numpy.loadtxt(JobDir + LocalObsFileName, skiprows = NumberHeaderLines)
                else:
                    ImportExpData = numpy.loadtxt(JobDir + LocalObsFileName, skiprows = NumberHeaderLines, delimiter = SeparatorColumns)


                ## remove continuum from modeled (strong molecule) spectra
                FreqArray = ImportExpData[:, 0]
                for RangeIndex in xrange(len(UnmodifiedFreqMinList)):
                    DataFileIndex = int(UnmodifiedFreqMinList[RangeIndex][0])
                    if (DataFileIndex == (ExpDataFileIndex + 1)):
                        FreqMin = float(UnmodifiedFreqMinList[RangeIndex][1])
                        FreqMax = float(UnmodifiedFreqMaxList[RangeIndex][1])
                        tBack = float(UnmodifiedtBackList[RangeIndex][1])
                        tSlope = float(UnmodifiedtSlopeList[RangeIndex][1])


                        ## remove continuum, if defined
                        if (tBack != 0.0 or tSlope != 0.0):
                            LowFreq = numpy.min(FreqArray)
                            NewTBack = tBack * (LowFreq / FreqMin)**tSlope                  ## determine new background temperature
                            for i in xrange(len(ImportExpData[:, 1])):
                               ImportExpData[i, 1] = ImportExpData[i, 1] - (abs(NewTBack) * (ImportExpData[i, 0] / LowFreq)**tSlope)


                ## store modeled spectra
                modeldata.append(ImportExpData)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## remove job directory for last myXCLASSFit run and modified molfit file
            cmdString = "rm -rf " + JobDir + " " + LocalMolfitsFileName
            os.system(cmdString)


            ## add modeled spectra to modeled spectra of other strong molecules
            if (ModelSpectra == []):
                ModelSpectra = modeldata
            else:
                for SpectraIndex in xrange(len(modeldata)):
                    ModelSpectra[SpectraIndex][:, 1] = ModelSpectra[SpectraIndex][:, 1] + modeldata[SpectraIndex][:, 1]


            ## store results to return parameter
            StrongMolContributionLocal = ModelSpectra


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define further return parameters
    NewListOfMolecules = ListOfMolecules
    StrongMolContribution = [StrongMolMolfit, StrongMolContributionLocal]
    return (StrongMolContribution, NewListOfMolecules)
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## start different molecule fits on the cluster
##
def StartClusterFits(ServerList, ListOfMolecules, StrongMolContribution, myXCLASSRootDirectory, MAGIXrootDir, LineIDjobDir, MoleculejobDirList, \
                     ListOfSpectraNames, ListOfSpectra, ListSubtractedSpectra, MaxOverestimationHeight, ResultsFile, IdentifiedMoleculeFile, \
                     ListNoiseLevels, CheckEmissionAbsorptionFlag, IdentifiedMoleculesDir, NotIdentifiedMoleculesDir, ListMolDirName, \
                     MAGIXExpXMLFilename, DefaultMolfitUsedFlagList, AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, VelBin, ElowMax, DecisionMethod, \
                     Tolerance, TypeSelection, SmoothValue, MaxNumTransInFit):

    """

input parameters:
-----------------

    - ServerList:                   list of all server in the cluster and the corresponding settings 

    - ListOfMolecules:              list of molecules

    - StrongMolContribution:        contribution of strong (highly abundant) molecules

    - myXCLASSRootDirectory:        root directory of the XCLASS package

    - MAGIXrootDir:                 MAGIX root directory

    - LineIDjobDir:                 job directory for current LineID run

    - MoleculejobDirList:           current single molecule fit directory

    - ListOfSpectraNames:           list of all spectrum (observational data) files

    - ListOfSpectra:                list of all observed spectra

    - ListSubtractedSpectra:        list of all continuum subtracted observational spectra

    - MaxOverestimationHeight:      overestimation factor

    - ResultsFile:                  final result file

    - IdentifiedMoleculeFile:       file for intermediate molecules

    - ListNoiseLevels:              list of all noise levels for all frequency ranges of all spectra

    - CheckEmissionAbsorptionFlag:  flag to consider emission and absorption separatley

    - IdentifiedMoleculesDir:       path of the directory containing the intermediate identified molecules of the current LineID run

    - NotIdentifiedMoleculesDir:    path of the directory containing the non-identified molecules of the current LineID run

    - ListMolDirName:               list of modified molecule names (used for naming of direcotries)

    - MAGIXExpXMLFilename:          name of MAGIX exp. xml file

    - DefaultMolfitUsedFlagList:    defines, if a default molfit file was used or not

    - AlgorithmXMLFileEstParam:     path and name of algorithm xml file (used for parameter estimation algorithm)

    - AlgorithmXMLFileISO:          path and name of algorithm xml file (used for fit of iso ratios)

    - VelBin:                       velocity resolution for histogram analysis

    - ElowMax:                      highest allowed lower energy of a transition

    - DecisionMethod:               method which is used

    - Tolerance:                    tolerance factor for method "significant"

    - TypeSelection:                exclude type of lines (only emission or only absorption or all types)

    - SmoothValue:                  smooth factor (=0, no smoothing is performed)

    - MaxNumTransInFit:             max. number of transitions which are fitted in paramter estimation algorithm


output parameters:
------------------

    - None
    """

    # Debug:
    # print "CurrentDir = ", CurrentDir


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start single molecule fits
    print "\r                                                                                                                                           ",
    RelativeMAGIXDir = MAGIXrootDir.replace(myXCLASSRootDirectory, "")


    ## print what you do
    NumWorkers = len(ServerList)
    if (NumWorkers == 1):
        print "\rPerform single molecule fit for ",
    else:
        print "\rPerform single molecule fits for ",


    # Debug:
    #    print "\n\nServerList = ", ServerList
    #    print "ListOfMolecules = ", ListOfMolecules
    #    print "StrongMolContribution = ", StrongMolContribution
    #    print "myXCLASSRootDirectory = ", myXCLASSRootDirectory
    #    print "RelativeMAGIXDir = ", RelativeMAGIXDir
    #    print "LineIDjobDir = ", LineIDjobDir
    #    print "MoleculejobDirList = ", MoleculejobDirList
    #    print "ListOfSpectraNames = ", ListOfSpectraNames
    #    print "ListSubtractedSpectra = ", ListSubtractedSpectra
    #    print "MaxOverestimationHeight = ", MaxOverestimationHeight
    #    print "ResultsFile = ", ResultsFile
    #    print "IdentifiedMoleculeFile = ", IdentifiedMoleculeFile
    #    print "ListNoiseLevels = ", ListNoiseLevels
    #    print "CheckEmissionAbsorptionFlag = ", CheckEmissionAbsorptionFlag
    #    print "IdentifiedMoleculesDir = ", IdentifiedMoleculesDir
    #    print "NotIdentifiedMoleculesDir = ", NotIdentifiedMoleculesDir
    #    print "ListMolDirName = ", ListMolDirName
    #    print "MAGIXExpXMLFilename = ", MAGIXExpXMLFilename


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## prepare start of workers
    thread_list = []
    for i in xrange(len(ServerList)):                                                       ## loop over all workers


        ## Instatiates the thread
        t = multiprocessing.Process(target=StartWorker, args=(i, ServerList, ListOfMolecules, StrongMolContribution, myXCLASSRootDirectory, \
                                                              RelativeMAGIXDir, LineIDjobDir, MoleculejobDirList, ListOfSpectraNames, ListOfSpectra, \
                                                              ListSubtractedSpectra, MaxOverestimationHeight, ResultsFile, IdentifiedMoleculeFile, \
                                                              ListNoiseLevels, CheckEmissionAbsorptionFlag, IdentifiedMoleculesDir, \
                                                              NotIdentifiedMoleculesDir, ListMolDirName, MAGIXExpXMLFilename, DefaultMolfitUsedFlagList, \
                                                              AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, VelBin, ElowMax, DecisionMethod, Tolerance, \
                                                              TypeSelection, SmoothValue, MaxNumTransInFit))


        ## Sticks the thread in a list so that it remains accessible
        thread_list.append(t)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    # Start all threads (workers)
    for thread in thread_list:
       thread.start()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    # This blocks the calling thread until the thread whose join() method is called is terminated.
    # From http://docs.python.org/2/library/threading.html#thread-objects
    for thread in thread_list:
       thread.join()


    # Debug:
    # sys.exit(0)
    ok = 0


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## create a new molfit file in the new format
##
def CreateMolfitFileInNewFormat(MolfitsFileName, SingleMolMolfitFiles, MinColumnDensityAbs, MinColumnDensityEmis):
    """

input parameters:
-----------------

    - MolfitsFileName:          path and name of molfit file

    - SingleMolMolfitFiles:     list containing all molfit files of all intermediate identified molecules

    - MinColumnDensityAbs:      minimal column density for an absorption component

    - MinColumnDensityEmis:     minimal column density for an emission component


output parameters:
------------------

    - None

    """


    ## analyze single molecule files and remove components with column densities lower than user defined threshold
    AllMolfitsFile = open(MolfitsFileName, "w")
    AllMolfitsFileContens = []
    MoleculeDefFlag = "false"
    EmptyFlag = "false"
    CurrentCompMol = []
    for molfitFile in SingleMolMolfitFiles:


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## analyze single molecule molfit file using myXCLASS routine
        MoleculesInMolfitFile, AllParameters, MolfitFileForEachMolecule = task_myXCLASS.AnalyzeMolfitFile(molfitFile)


        ## write parameter to new molfit file
        for MoleculeIndex in xrange(len(MoleculesInMolfitFile)):                            ## loop over all molecules in the molfit file
            LocalMolfitParameters = AllParameters[MoleculeIndex]                            ## get the parameters describing the current molecule


            ## check, if molecule is already included in the overall molfit file
            AlreadyIncludedFlag = "false"
            if (MoleculeIndex > 0):
                for ii in xrange(0, MoleculeIndex):
                    if (MoleculesInMolfitFile[ii] == MoleculesInMolfitFile[MoleculeIndex]):
                        AlreadyIncludedFlag = "true"
                        break
            if (AlreadyIncludedFlag == "false"):                                            ## continue here, if molecule is not already included


                ## write parameter for current molecule to xml file
                NewSingleMolMolfitFileContent = []
                for line in LocalMolfitParameters:                                          ## loop over all components of the current molecule


                    ## write each parameter for current molecule to xml file
                    MolfitFileFormatVersionLocal = ""
                    EmissionFlag = "T"                                                      ## define default setting for EmissionFlag
                    tdFlag = "F"                                                            ## define default setting for tdFlag
                    nHFlag = "F"                                                            ## define default setting for nHFlag
                    EAFlag = "f"
                    ColumnDensity = 0.e0
                    NewLine = ""                                                            ## initialize new line
                    for param in line:                                                      ## loop over all parameters

                        # Debug:
                        # print "param = ", param


                        ParameterName = param[0].strip()                                    ## get name of parameter
                        LimitValues = param[1]                                              ## get fit-flag of parameter
                        LimitValuesTrueFalse = "false"                                      ## define LimitValuesTrueFalse variable
                        if (float(LimitValues) != 0):
                            LimitValuesTrueFalse = "true"
                        lowlimit = param[2]                                                 ## get lower limit
                        uplimit = param[3]                                                  ## get upper limit
                        ParameterValue = param[4]                                           ## get value of parameter

                        # Debug:
                        # print "param = ", param


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## define molfit file format for each line
                        if (ParameterName == "MolfitFileFormatVersion"):
                            MolfitFileFormatVersionLocal = ParameterValue[:3]               ## get parameter indicating molfit format
                            EmissionFlag = ParameterValue[3:4]                              ## get parameter indicating if a absorption component
                                                                                            ## definiton contains no source size setting
                            tdFlag = ParameterValue[6:7]                                    ## get parameter tbFlag indicating if background temperature
                                                                                            ## etc is given in molfit file
                            nHFlag = ParameterValue[9:10]                                   ## write parameter nHFlag indicating if column density etc
                                                                                            ## is given in molfit file
                        else:


                            ## get column density
                            if (ParameterName == "N_tot"):
                                ColumnDensity = float(ParameterValue)


                            ## get absorption/emission flag
                            elif (ParameterName == "CFFlag"):
                                EAFlag = ParameterValue


                            ##============================================================================================================================
                            ## construct new line of current single molecule molfit file


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## molfit file is in OLD format
                            if (MolfitFileFormatVersionLocal == "OLD"):                     ## check, if current line has to be converted from OLD to NEW
                                                                                            ## format
                                if (ParameterName.endswith("_FitFlag")):
                                    ParameterValue = float(ParameterValue)
                                    if (ParameterValue == 0.0):                             ## no range is defined, parameter is kept constant
                                        element = " n "
                                        lowlimit = 0.0
                                        uplimit = 0.0
                                    else:                                                   ## a range is defined, determine limits for parameter
                                        element = " y "
                                        lowlimit, uplimit = GetParameterLimits(ParamName, ParameterValue, LimitValues)
                                elif (ParameterName != "CFFlag"):


                                    ## define lower and upper limit
                                    l1 = lowlimit
                                    u1 = uplimit
                                    lowlimit = min(l1, u1)
                                    uplimit = max(l1, u1)


                                    ## check, if parameter value is located within parameter ranges defined by lower and upper limits
                                    value = float(ParameterValue)
                                    if (value <= lowlimit):
                                        if (value < 0.0):
                                            lowlimit = value * 1.15
                                        elif (value == 0.0):
                                            lowlimit = 0.0
                                        else:
                                            lowlimit = value * 0.85
                                    elif (value >= uplimit):
                                        if (value < 0.0):
                                            uplimit = value * 0.85
                                        elif (value == 0.0):
                                            uplimit = 0.0
                                        else:
                                            uplimit = value * 1.15


                                    ## define part of molfit file
                                    if (ParameterName.startswith("N_tot") or ParameterName.startswith("nHcolumn")):
                                        element += "%12.3e %12.3e %20.10e" % (lowlimit, uplimit, value)
                                    else:
                                        element += "%12.3f %12.3f %20.10e" % (lowlimit, uplimit, value)
                                else:
                                    element = ParameterValue


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## molfit file is in NEW format
                            else:
                                if (ParameterName == "CFFlag" or ParameterName.endswith("_flag") ):
                                    element = ParameterValue
                                elif (ParameterName.endswith("_uplimit")):
                                    UpperLimit = float(ParameterValue)
                                    element = ""
                                elif (ParameterName.endswith("_lowlimit")):
                                    LowerLimit = float(ParameterValue)
                                    element = ""
                                else:


                                    ## define lower and upper limit
                                    l1 = LowerLimit
                                    u1 = UpperLimit
                                    LowerLimit = min(l1, u1)
                                    UpperLimit = max(l1, u1)


                                    ## check, if parameter value is located within parameter ranges defined by lower and upper limits
                                    value = float(ParameterValue)
                                    if (value <= LowerLimit):
                                        if (value < 0.0):
                                            LowerLimit = value * 1.15
                                        elif (value == 0.0):
                                            LowerLimit = 0.0
                                        else:
                                            LowerLimit = value * 0.85
                                    elif (value >= UpperLimit):
                                        if (value < 0.0):
                                            UpperLimit = value * 0.85
                                        elif (value == 0.0):
                                            UpperLimit = 0.0
                                        else:
                                            UpperLimit = value * 1.15


                                    ## define part of molfit file
                                    if (ParameterName.startswith("N_tot") or ParameterName.startswith("nHcolumn")):
                                        element = "%12.3e %12.3e %20.10e" % (LowerLimit, UpperLimit, value)
                                    else:
                                        element = "%12.3f %12.3f %20.10e" % (LowerLimit, UpperLimit, value)


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## create new line
                            NewLine += " " + element


                    ##------------------------------------------------------------------------------------------------------------------------------------
                    ## if column density is above user defined limit
                    if (EAFlag == "f"):
                        if (ColumnDensity >= MinColumnDensityAbs):
                            NewSingleMolMolfitFileContent.append(NewLine)
                    else:
                        if (ColumnDensity >= MinColumnDensityEmis):
                            NewSingleMolMolfitFileContent.append(NewLine)


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## consider only molecules with at least one component
                if (len(NewSingleMolMolfitFileContent) > 0):


                    ## write modified contents of single molecule molfit file to overall molfit file
                    NewLine = MoleculesInMolfitFile[MoleculeIndex] + "   " + str(len(NewSingleMolMolfitFileContent)) + "\n"
                    AllMolfitsFile.write(NewLine)                                           ## write new line to file
                    for line in NewSingleMolMolfitFileContent:                              ## loop over all lines of the current single molecule file
                        AllMolfitsFile.write(line + "\n")                                   ## write line to new overall molfit file


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## close channel for overall molfit file
    AllMolfitsFile.close()


    ## finished
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## Line identification routine
##
def LineIdentification(Noise, MaxOverestimationHeight, SourceName, DefaultMolfitFile, Tolerance, SelectedMolecules, StrongMoleculeList, \
                       MinColumnDensityEmis, MinColumnDensityAbs, NumberIteration, AlgorithmXMLFileSMF, AlgorithmXMLFileOverAll, experimentalData, \
                       vLSR, TelescopeSize, Inter_Flag, tBack, tSlope, N_H, beta_dust, kappa_1300, clusterdef):
    """

The function starts the line identification routine and returns the identified lines with the corresponding best fit.


input parameters:
-----------------

    - Noise:                defines noise level, (default: 0.1).

    - MaxOverestimationHeight:    defines the overestimation limit

    - SourceName:           name of source molfit file, (default: "").

    - DefaultMolfitFile:    path and name of a default molfit file used for molecules
                            which are not described by the soure templates, (default: "").

    - Tolerance:            fraction (%) of overestimated lines (default: 10.0).

    - SelectedMolecules:    a (python) list containing the names of all molecules which
                            should be considered or a file name including the molecules
                            which should be selected.

                            Note, if the paramter defines a relative path, this path has
                            to be defined relative to the current working directory!

    - StrongMoleculeList:   list of strong (highly abundant) molecules which has to be
                            fitted before the other single molecule fits are started

    - MinColumnDensityEmis: min. column densitiy (for emission lines) of an optimized
                            component to be included in the overall molfit file (default: 0).

    - MinColumnDensityAbs:  min. column densitiy (for absorption lines) of an optimized
                            component to be included in the overall molfit file (default: 0).

    - NumberIteration:      max. number of iterations (default: 50).

    - AlgorithmXMLFileSMF:  path and name of a MAGXI xml-file including definitions
                            for an algorithm or algorithm chain has to be given. This
                            xml file is used for the each single molecule fit. (A
                            relative path has to be defined relative to the current
                            working directory!)

                            NOTE, if the user specify a xml file, the number of
                            iterations given by the parameter "NumberIteration" is
                            ignored. The number of iteration is then given by the xml
                            file. In order to use the implemented fit algorithm
                            (Levenberg-Marquardt) clear the AlgorithmXMLFileSMF parameter,
                            i.e. AlgorithmXMLFileSMF = "", and define the max. number of
                            iterations by using parameter "NumberIteration".

    - AlgorithmXMLFileOverAll:  only necessary, if the user wants to use another fit
                            algorithm (than Levenberg-Marquardt) for fitting the final
                            molfit file including all identified molecules. Therefore, 
                            the path and name of a MAGXI xml-file defining settings
                            for an algorithm or algorithm chain has to be given. (A
                            relative path has to be defined relative to the current
                            working directory!)

                            NOTE, if the user specify a xml file, the number of
                            iterations given by the parameter "NumberIteration"
                            is ignored. The number of iteration is then given by the
                            xml file. In order to use the implemented fit algorithm
                            (Levenberg-Marquardt) clear the AlgorithmXMLFileOverAll
                            parameter, i.e. AlgorithmXMLFileOverAll = "", and define
                            the max. number of iterations by using parameter
                            "NumberIteration".

    - experimentalData:     This parameter offers two different possibility to send
                            the experimental data to the line identification function:

                            - the parameter experimentalData defines the path and
                              name of and experimental xml-file suitable for MAGIX.

                            - the parameter experimentalData defines the path and
                              name of and ASCII file called experimental data file,
                              where the first column describe the frequency (in MHz)
                              and the second column the beam temperature (intensity).

                            NOTE, if the parameter experimentalData defines a relative
                            path, the path has to be defined relative to the current
                            working directory!

    The following parameters are needed, if the parameter experimentalData does NOT
    describe the path and name of a MAGIX xml-file:

    - vLSR:                 local standard of rest v_LSR (in km/s), (default: 0).

    - TelescopeSize:        size of telescope (in m), (default: 1).

    - Inter_Flag:           defines, if single dish ("F") or interferometric
                            observations ("T") are described, (default: "F").

    - tBack:                background temperature (in K), (default: 0).

    - tSlope:               temperature slope (dimensionless), (default: 0).

    - N_H:                  Hydrogen column density (in cm^{-2}), (default: 0).

    - beta_dust:            beta for dust (dimensionless), (default: 0).

    - kappa_1300:           kappa (cm^2 g^{-1}), (default: 0).

    - clusterdef:           file that contains cluster definition.


output parameters:
------------------

    - IdentifiedLines:      contains all informations about the identified lines
                            (as an python array, e.g. IdentifiedLines[0] contains
                            the entries for the first molecule within the
                            frequency range)

    - JobDir:               absolute path of the job directory created for the current run.



Example:
--------

Noise = 2.0
MaxOverestimationHeight = 200.0
SourceName = ""
DefaultMolfitFile = "demo/LineIdentification/default.molfit"
Tolerance = 10.0
MaxNumTransInFit = 2000.0
SelectedMolecules = []
MinColumnDensityEmis = 0.0
MinColumnDensityAbs = 0.0
NumberIteration = 10
AlgorithmXMLFileSMF = ""
AlgorithmXMLFileOverAll = ""
experimentalData = "demo/LineIdentification/sma1_part-A.dat"
vLSR = 0.0
TelescopeSize = 200
Inter_Flag = F
tBack = 0.0
tSlope = 0.0
N_H = 3.0000000000E+24
beta_dust = 2.0
kappa_1300 = 0.42
clusterdef = "demo/LineIdentification/clusterdef.txt"
IdentifiedLines, JobDir = LineIdentification()
    """


    ##====================================================================================================================================================
    ## reset output variable
    BestMolfitFile = []
    JobDir = ""


    ## this function is currently not available
    #    print " "
    #    print " "
    #    print "Please note, this function is currently not available!"
    #    sys.stdout.flush()
    #    return (BestMolfitFile, JobDir)


    ##====================================================================================================================================================
    ## define some internal parameters (maybe user input parameters?)
    ##====================================================================================================================================================


    ## define method for decision
    # DecisionMethod = "strict"
    DecisionMethod = "significant"


    ## define internal parameter
    CheckEmissionAbsorptionFlag = True                                                      ## check for emission/absorption flag
    ElowMax = 1.e6
    AlgorithmXMLFileEstParam = ""
    AlgorithmXMLFileISO = ""
    VelBin = 1.0
    TypeSelection = ""
    SmoothValue = 0.0
    MaxNumTransInFit = 1e6


    ##====================================================================================================================================================
    ## get current directory
    CurrentDir = os.getcwd() + "/"


    ##====================================================================================================================================================
    ## check input parameters

    # Debug:
    # print "FreqMin = ", FreqMin
    # print "FreqMax = ", FreqMax
    # print "SelectMolecule = ", SelectMolecule


    ## define default values, maybe used user input parameters instead
    RestFreq = 0.0                                                                          ## set frequency of rest to zero
    LineIDMethod = ""


    ##====================================================================================================================================================
    ## analyze spectrum input parameter, (read in data, if input parameter specify a path and name of an ASCII file), normalize spectrum
    experimentalDataOrig = experimentalData
    tSlope = tSlope
    ListOfSpectraNames, ListOfSpectra, ListFreqMin, ListFreqMax, ListFreqStep, GlobalvLSRList, TelescopeSizeList, InterFlagList, tBackList, tSlopeList, \
    nHList, betaList, kappaList, NoiseLevelList, IsoRatioFile, PathFileNameDB = AnalyzeExperimentalDataParameter(CurrentDir, experimentalData, Noise, \
                                                                                                                 vLSR, TelescopeSize, tBack, tSlope, \
                                                                                                                 N_H, beta_dust, kappa_1300)
    # Debug:
    #    print "ListOfSpectraNames = ", ListOfSpectraNames
    #    print "ListOfSpectra = ", ListOfSpectra
    #    print "ListFreqMin = ", ListFreqMin
    #    print "ListFreqMax = ", ListFreqMax
    #    print "ListFreqStep = ", ListFreqStep
    #    print "GlobalvLSRList = ", GlobalvLSRList
    #    print "TelescopeSizeList = ", TelescopeSizeList
    #    print "InterFlagList = ", InterFlagList
    #    print "tBackList = ", tBackList
    #    print "tSlopeList = ", tSlopeList
    #    print "nHList = ", nHList
    #    print "betaList = ", betaList
    #    print "kappaList = ", kappaList
    #    print "NoiseLevelList = ", NoiseLevelList
    #    print "IsoRatioFile = ", IsoRatioFile
    #    print "PathFileNameDB = ", PathFileNameDB


    ##====================================================================================================================================================
    ## read in iso ratio file
    IsoRatioTable = []
    Isotopologues  = []
    IsoMolecule = []
    if (IsoRatioFile != []):
        if (IsoRatioFile[0] != "n" and IsoRatioFile[0] != "f" and IsoRatioFile[0] != "false" and IsoRatioFile[0] != "False"):
            NewLocalIsoRatioFileName = ""
            IsoRatioTable, Isotopologues, IsoMolecule = task_myXCLASS.ImportIsoRatioFile(IsoRatioFile[1], NewLocalIsoRatioFileName)


    ##====================================================================================================================================================
    ## get informations for each frequency range and create list with all molecules which occur within at least one frequency range
    ListSubtractedSpectra = []
    ListNoiseLevels = []
    ListOfAllMolecules = []
    ListOfAllFreqMin = []
    ListOfAllFreqMax = []
    ListOfAllFreqStep = []
    TotalCounterMol = 0
    for CounterSpectra in xrange(len(ListOfSpectra)):                                       ## loop over all spectra in the experimentalData directory
        ObsSpectrum = ListOfSpectra[CounterSpectra]                                         ## get spectrum


        ## make a copy of obs. spectrum array and add further column to avoid multiple continuum subtraction
        ObsSpectrumLocal = numpy.zeros((len(ObsSpectrum), 3))
        ObsSpectrumLocal[:, 0] = ObsSpectrum[:, 0]
        ObsSpectrumLocal[:, 1] = ObsSpectrum[:, 1]
        ObsSpectrumLocal[:, 2] = 0.0


        ## check length of ObsSpectrum
        number_points = len(ObsSpectrum[:,0])
        if (number_points == 1):                                                            ## data has just one data point
            print " "
            print " "
            print "Error in XCLASS package, function LineID, subroutine LineIdentification:"
            print "  The given experimental data is an array, but it contains only one data point!"
            print " "
            print "  Please correct your input and redo LineIdentification function call!"
            return (BestMolfitFile, JobDir)

        # Debug:
        #    print "number_points = ", number_points
        #    print "len(ListFreqMin) = ", len(ListFreqMin)


        ## get parameter for each frequency range
        for j in xrange(len(ListFreqMin)):                                                  ## loop over all range definitions in the whole xml file
            DataFileIndex = int(ListFreqMin[j][0])                                          ## get exp data file index for each range definition
            if (DataFileIndex == (CounterSpectra + 1)):                                     ## if current range definiton belongs to current range index
                FreqMin = float(ListFreqMin[j][1])                                          ## continue and get min. freq.
                ListOfAllFreqMin.append(FreqMin)
                FreqMax = float(ListFreqMax[j][1])                                          ## and max. freq.
                ListOfAllFreqMax.append(FreqMax)
                FreqStep = float(ListFreqStep[j][1])                                        ## and step freq.
                ListOfAllFreqStep.append(FreqStep)
                tBack = float(tBackList[j][1])                                              ## and background temperature
                tSlope = float(tSlopeList[j][1])                                            ## and temperature slope
                N_H = float(nHList[j][1])                                                   ## and n_H
                beta_dust = float(betaList[j][1])                                           ## and beta
                kappa_1300 = float(kappaList[j][1])                                         ## and kappa
                if (NoiseLevelList != []):
                    NoiseLevel = float(NoiseLevelList[CounterSpectra][1])
                else:
                    NoiseLevel = 0.0
                ListNoiseLevels.append(NoiseLevel)

                # Debug:
                #    print "\n\n\nCounterSpectra = ", CounterSpectra
                #    print "j = ", j
                #    print "DataFileIndex = ", DataFileIndex
                #    print "FreqMin = ", FreqMin
                #    print "FreqMax = ", FreqMax
                #    print "FreqStep = ", FreqStep
                #    print "tBack = ", tBack
                #    print "tSlope = ", tSlope
                #    print "N_H = ", N_H
                #    print "beta_dust = ", beta_dust
                #    print "kappa_1300 = ", kappa_1300
                #    print "NoiseLevel = ", NoiseLevel


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## subtract continuum
                ObsSpectrumLocal = PreprocessSpectrum(ObsSpectrumLocal, FreqMin, FreqMax, tBack, tSlope)


        ## store continuum subtracted spectra
        ObsSpectrum[:, 0] = ObsSpectrumLocal[:, 0]
        ObsSpectrum[:, 1] = ObsSpectrumLocal[:, 1]
        ListSubtractedSpectra.append(ObsSpectrum)


    ##====================================================================================================================================================
    ## get informations about all molecules which are stored in the sqlite3 database between FreqMin and FreqMax
    ok, TotalCounterMol, MoleculeData, FilenameDB, dbDefaultFilename, ListTotalNumTransPerMol = ReadDatabase(PathFileNameDB, ListOfAllFreqMin, \
                                                                                                             ListOfAllFreqMax, SelectedMolecules, \
                                                                                                             Isotopologues, IsoMolecule, ElowMax)
    if (ok == 1):                                                                       ## an error occured
        return (BestMolfitFile, JobDir)
    if (TotalCounterMol == 0):                                                          ## no molecule is found within range
        print "\n\n\nError in XCLASS package, function LineID, subroutine LineIdentification:"
        print "\tFound no molecule with transitions within the given frequency range(s)!\n\n"
        print "\n\tAbort LineID function!\n\n\n"
        return (BestMolfitFile, JobDir)
    else:


        ## create list with molecules
        for molecule in MoleculeData:
            ListOfAllMolecules.append(molecule[0])


    ## check, if molecules were found
    if (TotalCounterMol == 0):
        print " "
        print " "
        print "Error in XCLASS package, function LineID, subroutine LineIdentification:"
        print "  No molecule is found in the database within the given frequency range!"
        print " "
        return (BestMolfitFile, JobDir)


    ## remove dublicates in list ListOfAllMolecules and sort list alphabetical
    ListOfMolecules = []
    for mol in ListOfAllMolecules:
        if mol not in ListOfMolecules:
            ListOfMolecules.append(mol)
    ListOfMolecules.sort()

    # Debug:
    #    print "\n\n\n\n\n\n\n\n\n\n"
    #    for mol in ListOfMolecules:
    #        print mol
    #    sys.exit(0)


    ## print list of all molecules with the total number of transitions within the selected range(s)
    print "\n\nConsider the following molecules (with total number of transitions):"
    ListTotalNumTransPerMol = sorted(ListTotalNumTransPerMol, key=lambda l:l[1], reverse=True)
    print "\n\n%-40s  Total number of transitions:" % "Molecule:"
    for entry in ListTotalNumTransPerMol:
        print "%-40s  %i5" % (entry[0], entry[1])
    print "\n\n"


    ##====================================================================================================================================================
    ## prepare job and other directories


    ## determine base directory of the XCLASS interface
    myXCLASSRootDirectory = ""
    i = dbDefaultFilename.rfind("/")
    if (i == (-1)):
        print " "
        print " "
        print "Error in XCLASS package, function LineID, subroutine LineIdentification:"
        print "  Can not determine the root directory of the XCLASS interface!"
        print " "
        return (BestMolfitFile, JobDir)
    else:
        helpString = dbDefaultFilename[:i]
        i = helpString.rfind("/")
        if (i == (-1)):
            print " "
            print " "
            print "Error in XCLASS package, function LineID, subroutine LineIdentification:"
            print "  Can not determine the root directory of the XCLASS interface!"
            print " "
            return (BestMolfitFile, JobDir)
        else:
            myXCLASSRootDirectory = helpString[:i + 1]

            # Debug:
            # print "myXCLASSRootDirectory = ", myXCLASSRootDirectory


    ## set up environment for MAGIX
    MAGIXrootDir = myXCLASSRootDirectory + "programs/MAGIX/"
    task_MAGIX.SetMAGIXEnvironment(MAGIXrootDir)


    ##====================================================================================================================================================
    ## create run directory for the current function call
    LocalPrintFlag = True
    LineIDjobDir = task_MAGIX.CreateRunDirectory("LineIdentification", MAGIXrootDir, LocalPrintFlag)
    JobDir = LineIDjobDir


    ## create directory for single molecule fits
    print "\n\nCreating directory for single molecule fits ..",
    LineIDSingleMoleculeFitsDir = LineIDjobDir + "single-molecule_fits/"
    command_string = "mkdir -p " + LineIDSingleMoleculeFitsDir
    os.system(command_string)
    print "done!"


    ## create subdirectory for identified molecules
    IdentifiedMoleculesDir = LineIDjobDir + "Intermediate_identified_molecules/"
    command_string = "mkdir -p " + IdentifiedMoleculesDir
    os.system(command_string)


    ## create subdirectory for non-identified molecules
    NotIdentifiedMoleculesDir = LineIDjobDir + "Not_identified_molecules/"
    command_string = "mkdir -p " + NotIdentifiedMoleculesDir
    os.system(command_string)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze clusterdef file
    NumberProcessors, ServerList, ClusterFlag = task_myXCLASSMapFit.ReadClusterdefFile(clusterdef, LineIDjobDir)
    if (NumberProcessors > 1 and len(ListOfMolecules) == 1):
        NumberProcessors = 1
        ClusterFlag = "false"


    ## modifiy ServerList, so that ServerList has as many entries as workers, i.e. an entry with settings for each worker
    OldServerList = ServerList
    ServerList = []
    for server in OldServerList:                                                            ## loop over all server in the cluster
        Node = server[0]                                                                    ## get name of node
        NumCores = server[1]                                                                ## get number of cores
        RootPath = server[2].strip()                                                        ## get path of xclass interface
        if (RootPath != ""):                                                                ## if path of xclass interface is defined ..
            RootPath = RootPath + "/programs/MAGIX/"                                        ## .. define path of MAGIX
        for core in xrange(NumCores):                                                       ## loop over all cores of the current node
            ServerList.append([Node, NumCores, RootPath])


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## analyze MinColumnDensity parameter
    if (MinColumnDensityEmis < 0.0 or MinColumnDensityAbs < 0.0):
        print " "
        print " "
        print "Error in XCLASS package, function LineID, subroutine LineIdentification:"
        print "  The given min. column density is less than zero."
        print " "
        print "  MinColumnDensityEmis = ", MinColumnDensityEmis
        print "  MinColumnDensityAbs = ", MinColumnDensityAbs
        print " "
        print "  Please correct the definition of this parameter and restart LineIdentification function!"
        print " "
        return (BestMolfitFile, JobDir)


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## open file for including the results of the single molecule fits
    ResultsFile = open(LineIDjobDir + "results.dat", 'w')
    ResultsFile.write("Results of the line identification:\n")
    lt = time.localtime()
    ResultsFile.write("\n\nStart LineID at date: " + time.strftime("%d-%m-%Y", lt) + ",  time: " + time.strftime("%H:%M:%S", lt) + "\n")
    ResultsFile.write("\n\n\nCurrent directory = " + CurrentDir + "\n")
    ResultsFile.write("\n\n\nInput parameter:\n\n")
    ResultsFile.write("SourceName = " + SourceName + "\n")
    ResultsFile.write("DefaultMolfitFile = " + DefaultMolfitFile + "\n")
    if (AlgorithmXMLFileSMF == ""):
        ResultsFile.write("NumberIteration = " + str(NumberIteration) + "\n")
    else:
        ResultsFile.write("AlgorithmXMLFileSMF = " + AlgorithmXMLFileSMF + "\n")
        ResultsFile.write("AlgorithmXMLFileOverAll = " + AlgorithmXMLFileOverAll + "\n")
    ResultsFile.write("\n\n\nConsider the following molecules:\n\n")
    i = 0
    for molecule in ListOfMolecules:
        i += 1
        ResultsFile.write("Molecule %i:   %s\n" % (i, molecule))
    ResultsFile.write("\n\n")
    ResultsFile.flush()

    # Debug:
    #    ResultsFile.write("Noise = " + str(Noise) + "\n")
    #    ResultsFile.write("FreqMin = " + str(FreqMin) + "\n")
    #    ResultsFile.write("FreqMax = " + str(FreqMax) + "\n")
    #    ResultsFile.write("FreqStep = " + str(FreqStep) + "\n")
    #    ResultsFile.write("TelescopeSize = " + str(TelescopeSize) + "\n")
    #    ResultsFile.write("InterFlag = " + str(InterFlag) + "\n")
    #    ResultsFile.write("N_H = " + str(N_H) + "\n")
    #    ResultsFile.write("beta_dust = " + str(beta_dust) + "\n")
    #    ResultsFile.write("kappa_1300 = " + str(kappa_1300) + "\n")
    #    ResultsFile.write("tBack = " + str(tBack) + "\n")
    #    ResultsFile.write("tSlope = " + str(tSlope) + "\n")


    ## open file for identified molecules
    IdentifiedMoleculeFile = open(LineIDjobDir + "Identified_Molecules.dat", 'w')
    IdentifiedMoleculeFile.write("Identified molecules:\n")
    lt = time.localtime()
    IdentifiedMoleculeFile.write("\n\nStart LineID at date: " + time.strftime("%d-%m-%Y", lt) + ",  time: " + time.strftime("%H:%M:%S", lt) \
                                 + "\n\n\n\n")
    IdentifiedMoleculeFile.flush()
    print "\n"


    ##====================================================================================================================================================
    ## start single molecule fits
    ListMolDirName = []
    MoleculejobDirList = []
    DefaultMolfitUsedFlagList = []
    counterMolecule = 0                                                                     ## reset counter for molecules
    for molecule in ListOfMolecules:                                                        ## loop over all molecules in the freq. range
        counterMolecule += 1                                                                ## increase counter for molecules

        # Debug:
        # print "counterMolecule, molecule = ", counterMolecule, molecule


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## prepare MAGIX call
        print "\rPrepare single molecule fit for molecule:  " + molecule + " ..                                                                         ",


        ## define name of subdirectory for current molecule, remove not allowed characters from the molecule name
        MolDirName = MoleculeFileName(molecule)
        ListMolDirName.append(MolDirName)


        ## create single molecule fit directory
        MoleculejobDir = LineIDSingleMoleculeFitsDir + MolDirName + "/"
        RelativeMoleculejobDir = MoleculejobDir.replace(LineIDjobDir, "")
        MoleculejobDirList.append(RelativeMoleculejobDir)
        command_string = "mkdir -p " + MoleculejobDir
        os.system(command_string)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## prepare experimental xml file for MAGIX
        filename = experimentalDataOrig.strip()                                             ## remove leading and tailing blanks
        if (filename[0] != "/"):                                                            ## if neccessary, make relative path absolute
            filename = CurrentDir + filename
        if (filename.endswith(".xml")):                                                     ## if a MAGIX xml file is used adjust paths
            ok, MAGIXExpXML = PrepareMAGIXExpXMLFile(CurrentDir, filename, MoleculejobDir, dbDefaultFilename)
            LocalExpXMLFile = MAGIXExpXML
            if (ok == 1):
                return (BestMolfitFile, JobDir)
        else:                                                                               ## .. if not, create new MAGIX xml file


            ## copy exp. data file to current single molecule fit directory
            NewExpDataFile = MoleculejobDir + os.path.basename(filename)
            cmd_string = "cp " + filename + " " + NewExpDataFile
            os.system(cmd_string)
            experimentalData = NewExpDataFile


            ## creates experimental xml file
            LocalExpXMLFile = MoleculejobDir + "exp.xml"
            MAGIXExpXML = ""
            t_back_flag = "T"
            nH_flag = "T"
            IsoTableFileName = ""
            ObsSpectrum = ListOfSpectra[0]
            GlobalvLSR = float(GlobalvLSRList[0])
            TelescopeSize = float(TelescopeSizeList[0])
            InterFlag = InterFlagList[0]
            tBack = float(tBackList[0])
            tSlope = float(tSlopeList[0])
            N_H = float(nHList[0])
            beta_dust = float(betaList[0])
            kappa_1300 = float(kappaList[0])
            FreqMin = ListOfAllFreqMin[0]
            FreqMax = ListOfAllFreqMax[0]
            FreqStep = ListOfAllFreqStep[0]
            task_myXCLASSFit.CreateExpXMLFile(MoleculejobDir, experimentalData, FreqMin, FreqMax, FreqStep, t_back_flag, tBack, tSlope, nH_flag, \
                                              N_H, beta_dust, kappa_1300, GlobalvLSR, TelescopeSize, InterFlag, iso_flag, IsoTableFileName, FilenameDB)

        ## copy created xml file
        UnmodifiedExpXMLFilename = LineIDjobDir + "Unmodified_exp.xml"
        cmdString = "cp " + LocalExpXMLFile + " " + UnmodifiedExpXMLFilename
        os.system(cmdString)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## create molfit file for the current molecule
        MolfitsFileName = MoleculejobDir + MolDirName + ".molfit"
        DefaultMolfitUsedFlag = CreateMolfitFile(CurrentDir, myXCLASSRootDirectory, MoleculejobDir, MolfitsFileName, molecule, SourceName, \
                                                 DefaultMolfitFile)
        DefaultMolfitUsedFlag = False
        DefaultMolfitUsedFlagList.append(DefaultMolfitUsedFlag)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## shrink freq. ranges to reduce computational effort
        if (not DefaultMolfitUsedFlag):
            ShrinkFreqRanges(molecule, MolDirName, MAGIXExpXML, MolfitsFileName, ElowMax)


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## creates fit control xml-file
        AlgorithmXMLFileSMF = AlgorithmXMLFileSMF.strip()
        if (AlgorithmXMLFileSMF != ""):


            ## if neccessary, make relative path absolute
            if (AlgorithmXMLFileSMF[0] != "/"):
                filename = CurrentDir + AlgorithmXMLFileSMF
            else:
                filename = AlgorithmXMLFileSMF


            ## check if path and name of the experimental file exsits
            if not(os.path.exists(filename) or os.path.isfile(filename)):
                print "\n\nError in XCLASS package, function LineID, subroutine LineIdentification:"
                print "  The given path and name of the algorithm xml-file does not exsist!"
                print " "
                print "  Please enter a valid path and name of an algorithm xml-file and redo line identification function call!"
                return (BestMolfitFile, JobDir)


            ## copy algorithm file to working directory
            cmd_string = "cp " + filename + " " + MoleculejobDir + "algorithm_control.xml"
            os.system(cmd_string)
        else:
            task_myXCLASSFit.CreateLMControlXMLFile(MoleculejobDir, NumberIteration)
        AlgorithmXMLFileLocal = MoleculejobDir + "algorithm_control.xml"


        ##------------------------------------------------------------------------------------------------------------------------------------------------
        ## creates i/o control xml file
        task_myXCLASSFit.CreateIOControlXMLFile(MoleculejobDir, MAGIXExpXML, MolfitsFileName, MAGIXrootDir)
        if (MAGIXExpXML.strip() == ""):
            MAGIXExpXMLFilename = "exp.xml"
        else:
            MAGIXExpXMLFilename = os.path.basename(MAGIXExpXML)

    # Debug:
    #    print "DefaultMolfitUsedFlagList = ", DefaultMolfitUsedFlagList
    #    print "\n\nStrongMoleculeList = ", StrongMoleculeList
    #    print "ListOfMolecules = ", ListOfMolecules
    #    print "\n\n\n\n\n\n"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## do single molecule fits for strong line first
    StrongMolContribution, NewListOfMolecules = StrongLinesFits(StrongMoleculeList, ListOfMolecules, myXCLASSRootDirectory, MAGIXrootDir, LineIDjobDir, \
                                                                MoleculejobDirList, ListOfSpectraNames, ListOfSpectra, ListSubtractedSpectra, \
                                                                MaxOverestimationHeight, ResultsFile, IdentifiedMoleculeFile, ListNoiseLevels, \
                                                                CheckEmissionAbsorptionFlag, IdentifiedMoleculesDir, NotIdentifiedMoleculesDir, \
                                                                ListMolDirName, MAGIXExpXMLFilename, UnmodifiedExpXMLFilename, \
                                                                DefaultMolfitUsedFlagList, AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, VelBin, \
                                                                ElowMax, DecisionMethod, Tolerance, TypeSelection, SmoothValue, MaxNumTransInFit)
    if (NewListOfMolecules != []):
        ListOfMolecules = NewListOfMolecules

    # Debug:
    #    print "\n\n\n\n\n\nListOfMolecules = ", ListOfMolecules
    #    print "NewListOfMolecules = ", NewListOfMolecules
    #    print "\n\n\n\n\n\n"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start different molecule fits on the cluster
    StartClusterFits(ServerList, ListOfMolecules, StrongMolContribution, myXCLASSRootDirectory, MAGIXrootDir, LineIDjobDir, MoleculejobDirList, \
                     ListOfSpectraNames, ListOfSpectra, ListSubtractedSpectra, MaxOverestimationHeight, ResultsFile, IdentifiedMoleculeFile, \
                     ListNoiseLevels, CheckEmissionAbsorptionFlag, IdentifiedMoleculesDir, NotIdentifiedMoleculesDir, ListMolDirName, \
                     MAGIXExpXMLFilename, DefaultMolfitUsedFlagList, AlgorithmXMLFileEstParam, AlgorithmXMLFileISO, VelBin, ElowMax, DecisionMethod, \
                     Tolerance, TypeSelection, SmoothValue, MaxNumTransInFit)


    ## close output files
    lt = time.localtime()
    ResultsFile.write("\n\nEnd LineID at date: " + time.strftime("%d-%m-%Y", lt) + ",  time: " + time.strftime("%H:%M:%S", lt) + "\n")
    ResultsFile.close()
    IdentifiedMoleculeFile.write("\n\nEnd LineID at date: " + time.strftime("%d-%m-%Y", lt) + ",  time: " + time.strftime("%H:%M:%S", lt) + "\n")
    IdentifiedMoleculeFile.close()


    ## remove copy of unmodified xml file
    os.system("rm -rf " + LineIDjobDir + "Unmodified_exp.xml")


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## read in all molfit files of intermediate identified molecules
    listing = os.listdir(IdentifiedMoleculesDir)
    SingleMolMolfitFiles = []
    SingleMolIsoRatioFiles = []
    StrongMolFileList = []
    StrongIsoFileList = []


    ## make sure, that strong molecules (if defined) are read first
    if (StrongMoleculeList != []):
        for StrongMol in StrongMoleculeList:
            StrongMolFileName = MoleculeFileName(StrongMol)                                 ## get file name of current molecule
            for files in listing:


                ## molfit files
                if (files == StrongMolFileName + ".out.molfit"):
                    StrongMolFileList.append(files)


                    ## read in molfit file and add to list of all single molecule
                    LocalMolfitFile = open(IdentifiedMoleculesDir + "/" + files)
                    LocalMolfitFileContent = LocalMolfitFile.readlines()
                    LocalMolfitFile.close()
                    SingleMolMolfitFiles.append(LocalMolfitFileContent)


                ## iso ratio files
                if (files == StrongMolFileName + ".out.iso"):
                    StrongIsoFileList.append(files)


                    ## read in iso ratio file and add to list of all single molecule
                    LocalIsoRatioFile = open(IdentifiedMoleculesDir + "/" + files)
                    LocalIsoRatioFileContent = LocalIsoRatioFile.readlines()
                    LocalIsoRatioFile.close()
                    for line in LocalIsoRatioFileContent:
                        SingleMolIsoRatioFiles.append(line)



    ## read molfit files for all other molecules
    for files in listing:
        if (files.endswith(".out.molfit") or files.endswith(".out.iso")):


            ## do not add strong molecules again
            AddFlag = "true"
            if (StrongMoleculeList != []):
                if (files in StrongMolFileList):
                    AddFlag = "false"


            ## read in molfit/iso file and add to list of all single molecule
            if (AddFlag == "true"):
                if (files.endswith(".out.molfit")):
                    LocalMolfitFile = open(IdentifiedMoleculesDir + "/" + files)
                    LocalMolfitFileContent = LocalMolfitFile.readlines()
                    LocalMolfitFile.close()
                    SingleMolMolfitFiles.append(LocalMolfitFileContent)
                elif (files.endswith(".out.iso")):
                    LocalIsoRatioFile = open(IdentifiedMoleculesDir + "/" + files)
                    LocalIsoRatioFileContent = LocalIsoRatioFile.readlines()
                    LocalIsoRatioFile.close()
                    for line in LocalIsoRatioFileContent:
                        SingleMolIsoRatioFiles.append(line)


    ## check, if molecules are identified
    if (len(SingleMolMolfitFiles) == 0):
        print "\n\nError in XCLASS package, function LineID, subroutine LineIdentification:"
        print "  Can not find a molecules in the given spectrum/spectra!"
        print " "
        print "  Modify list of included/excluded molecules and redo call of function LineIdentification!"
        return (BestMolfitFile, JobDir)


    ##====================================================================================================================================================
    ## start final fit with all identified molecules
    ## prepare MAGIX call
    LineSting = "=" * 173
    print "\n\n\n\n" + LineSting
    print "Prepare final fit with all identified molecules .."


    ## define name of subdirectory for current molecule, remove not allowed characters from the molecule name
    MolDirName = "all"


    ## create working directory
    print "\tCreate working directory ..",
    MoleculejobDir = LineIDjobDir + MolDirName + "/"
    command_string = "mkdir -p " + MoleculejobDir
    os.system(command_string)
    print "done!"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## create molfit file for all identified molecules
    print "\tCreate molfit file ..",
    MolfitsFileName = MoleculejobDir + MolDirName + ".molfit"
    CreateMolfitFileInNewFormat(MolfitsFileName, SingleMolMolfitFiles, MinColumnDensityAbs, MinColumnDensityEmis)
    print "done!"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## write merged iso ratio files to working directory
    if (SingleMolIsoRatioFiles != []):
        IsoFileName = MoleculejobDir + MolDirName + ".iso"
        NewIsoFile = open(IsoFileName, 'w')
        for line in SingleMolIsoRatioFiles:
            NewIsoFile.write(line)
        NewIsoFile.close()


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## if experimentalData defines path and name of a MAGIX xml file, adjust paths in xml file
    MAGIXExpXMLFilename = MoleculejobDir + "exp.xml"
    if (experimentalDataOrig.endswith(".xml")):


        ## copy obs. xml file to all subdirectory
        cmdString = "cp " + experimentalDataOrig + " " + MAGIXExpXMLFilename
        os.system(cmdString)


        ## adjust path
        AdjustExpXMLFile(MAGIXExpXMLFilename, MoleculejobDir, CurrentDir, dbDefaultFilename)
    else:


        ## write original spectrum to file in the molecule subdirectory
        print "\tWrite spectrum to file ..",
        experimentalData = MoleculejobDir + os.path.basename(experimentalDataOrig)
        numpy.savetxt(experimentalData, experimentalDataOrig, delimiter='\t')
        print "done!"


        ## creates experimental xml file
        print "\tCreating experimental xml file ..",
        t_back_flag = "T"
        nH_flag = "T"
        IsoTableFileName = ""
        task_myXCLASSFit.CreateExpXMLFile(MoleculejobDir, experimentalData, FreqMin, FreqMax, FreqStep, t_back_flag, tBack, tSlope, nH_flag, N_H, \
                                          beta_dust, kappa_1300, vLSR, TelescopeSize, Inter_Flag, iso_flag, IsoTableFileName, FilenameDB)
        print "done!"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## creates fit control xml-file
    AlgorithmXMLFileOverAll = AlgorithmXMLFileOverAll.strip()
    if (AlgorithmXMLFileOverAll != ""):


        ## if neccessary, make relative path absolute
        if (AlgorithmXMLFileOverAll[0] != "/"):
            filename = CurrentDir + AlgorithmXMLFileOverAll
        else:
            filename = AlgorithmXMLFileOverAll


        ## check if path and name of the experimental file exsits
        if not(os.path.exists(filename) or os.path.isfile(filename)):
            print "\n\nError in XCLASS package, function LineID, subroutine LineIdentification:"
            print "  The given path and name of the algorithm xml-file does not exsist!"
            print " "
            print "  Please enter a valid path and name of an algorithm xml-file and redo line identification function call!"
            return (BestMolfitFile, JobDir)


        ## copy algorithm file to working directory
        print "\tCopy algorithm file to current working directory ..",
        cmd_string = "cp " + filename + " " + MoleculejobDir + "algorithm_control.xml"
        os.system(cmd_string)
        print "done!"
    else:
        print "\tCreating fit control xml file ..",
        task_myXCLASSFit.CreateLMControlXMLFile(MoleculejobDir, NumberIteration)
        print "done!"


    ## creates i/o control xml file
    print "\tCreating i/o control xml file ..",
    MAGIXExpXML = ""
    task_myXCLASSFit.CreateIOControlXMLFile(MoleculejobDir, MAGIXExpXML, MolfitsFileName, MAGIXrootDir)
    print "done!"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## update path and name of iso ratio file and set iso_flag in experimental xml file
    #    if (SingleMolIsoRatioFiles != []):
    #        task_MAGIX.WriteXMLtag(MAGIXExpXMLFilename, "iso_flag", ["y"])
    #        task_MAGIX.WriteXMLtag(MAGIXExpXMLFilename, "Isotopologues", ["y"])
    #        task_MAGIX.WriteXMLtag(MAGIXExpXMLFilename, "IsoTableFileName", [IsoFileName])


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## start myXCLASSFit
    print "\n\nStart MAGIX ..\n"
    MAGIXOption = "plotsaveonly, model=myxclass"
    ok = task_MAGIX.StartMAGIX(MAGIXrootDir, MAGIXOption, MoleculejobDir + "io_control.xml")


    ##------------------------------------------------------------------------------------------------------------------------------------------------
    ## read in parameters for each freq. range for the current molecule
    ListOfAllFreqMin = task_MAGIX.GetXMLtag(MAGIXExpXMLFilename, "MinExpRange")
    ListOfAllFreqMax = task_MAGIX.GetXMLtag(MAGIXExpXMLFilename, "MaxExpRange")
    FreqStepList = task_MAGIX.GetXMLtag(MAGIXExpXMLFilename, "StepFrequency")
    t_back_flagList = task_MAGIX.GetXMLtag(MAGIXExpXMLFilename, "t_back_flag")
    tBackList = task_MAGIX.GetXMLtag(MAGIXExpXMLFilename, "BackgroundTemperature")
    tSlopeList = task_MAGIX.GetXMLtag(MAGIXExpXMLFilename, "TemperatureSlope")
    N_HList = task_MAGIX.GetXMLtag(MAGIXExpXMLFilename, "HydrogenColumnDensity")
    beta_dustList = task_MAGIX.GetXMLtag(MAGIXExpXMLFilename, "DustBeta")
    kappa_1300List = task_MAGIX.GetXMLtag(MAGIXExpXMLFilename, "Kappa")
    NoiseList = task_MAGIX.GetXMLtag(MAGIXExpXMLFilename, "NoiseLevel")


    ## read in molfit file which correspond to the best fit (variable 'ListSubtractedModelSpectra' is not used)
    print "Read in molfit file which correspond to the best fit ..",
    ConvertLogLinFlag = "true"
    ListOfAllFreqMinCopy = ListOfAllFreqMin
    ListOfAllFreqMin = []                                                                   ## clear list, because we don't want the model values
    BestMolfitFile, IsoRatioFileContent, ListSubtractedModelSpectra = GetBestResult(MoleculejobDir, ConvertLogLinFlag, ListOfSpectraNames, \
                                                                                    ListOfAllFreqMin, ListOfAllFreqMax, tBackList, tSlopeList)
    ListOfAllFreqMin = ListOfAllFreqMinCopy
    print "done!"


    ##====================================================================================================================================================
    ## determine the spectra of each identified molecule


    ## split final molfit file in N small molfit files containing only one molecule
    MoleculesInOverallMolfitFile = []
    MolfitFileForEachMolecule = []
    CounterMolecules = 0
    CounterLines = (-1)
    LinesOfMolfitFile = []
    for line in BestMolfitFile:
        CounterLines += 1
        StrippedLine = line.strip()
        i = StrippedLine.find("%")
        if (i > (-1)):
            StrippedLine = StrippedLine[:i].strip()
        if (StrippedLine != ""):
            SplittedLine = StrippedLine.split()
            if (len(SplittedLine) == 2):
                if (CounterMolecules > 0):
                    MolfitFileForEachMolecule.append(LinesOfMolfitFile)
                CounterMolecules += 1
                LinesOfMolfitFile = []
                LinesOfMolfitFile.append(line)
                MoleculesInOverallMolfitFile.append(SplittedLine[0])
            else:
                LinesOfMolfitFile.append(line)
    if (CounterMolecules > 0):
        MolfitFileForEachMolecule.append(LinesOfMolfitFile)


    ## add complete molfit file to MolfitFileForEachMolecule
    MolfitFileForEachMolecule.append(BestMolfitFile)
    MoleculesInOverallMolfitFile.append("all identified molecules")

    # Debug:
    #    print "\n\n\n\n\n\n\n\n\n"
    #    print "BestMolfitFile = ", BestMolfitFile
    #    print "MolfitFileForEachMolecule = ", MolfitFileForEachMolecule
    #    print "MoleculesInOverallMolfitFile = ", MoleculesInOverallMolfitFile
    #    print "\n\n\n\n\n\n\n\n\n"


    ## create new subdirectories for each molecule and exp. data file within the directory where the final fit was done
    print "\nCreate working directory for determine contributions of each identified molecule ..",
    IdentifiedMoleculesSubDirectories = MoleculejobDir + "final_fit/"
    command_string = "mkdir -p " + IdentifiedMoleculesSubDirectories
    os.system(command_string)
    print "done!"


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## extend sys.path
    MAGIXrootDir = myXCLASSRootDirectory + "../MAGIX/"
    MAGIXrootDir = os.path.normpath(MAGIXrootDir) + "/"

    # Debug:
    #    print "\n\nMAGIXrootDir = ", MAGIXrootDir


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## determine the intensities for each molecule
    for CounterSpectra in xrange(len(ListOfSpectra)):                                       ## loop over all spectra in the experimentalData directory
        TelescopeSize = float(TelescopeSizeList[CounterSpectra])
        InterFlag = InterFlagList[CounterSpectra]
        if (GlobalvLSRList == []):
            GlobalvLSR = 0.0
        else:
            GlobalvLSR = GlobalvLSRList[CounterSpectra]


        ## get path and name of corresponding data file
        SpectraName = ListOfSpectraNames[CounterSpectra].strip()
        i = SpectraName.rfind("/")
        if (i > (-1)):
            SpectraName = SpectraName[i+1:]


        ## get parameter for current spectrum
        print "\n\n\n\nAnalyze spectrum " + SpectraName                                     ## print what you do
        t_back_flag = True
        nH_flag = True
        if (SingleMolIsoRatioFiles != []):
            iso_flag = True
        else:
            iso_flag = False
            IsoTableFileName = ""
        NumberProcessors = 1
        myXCLASSProgramDirectory = myXCLASSRootDirectory + "programs/myXCLASS/src/"
        myXCLASSexe = "myNewXCLASS.exe"


        ## get obs. data
        LocalObsSpectrum = ListSubtractedSpectra[CounterSpectra]
        LowestDataPoint = min(LocalObsSpectrum[:,0])
        HighestDataPoint = max(LocalObsSpectrum[:,0])
        FreqArray = LocalObsSpectrum[:, 0]

        # Debug:
        # print "LowestDataPoint = ", LowestDataPoint
        # print "HighestDataPoint = ", HighestDataPoint


        for j in xrange(len(ListOfAllFreqMin)):                                             ## loop over all range definitions in the whole xml file
            DataFileIndex = int(ListOfAllFreqMin[j][0])                                     ## get exp data file index for each range definition
            if (DataFileIndex == (CounterSpectra + 1)):                                     ## if current range definiton belongs to current range index
                FreqMin = float(ListOfAllFreqMin[j][1])                                     ## continue and get min. freq.
                FreqMax = float(ListOfAllFreqMax[j][1])                                     ## and max. freq.
                FreqStep = float(FreqStepList[j][1])
                if (tBackList[j][1] != []):
                    t_back_flag = t_back_flagList[0].lower()
                    if (t_back_flag == "true"):
                        t_back_flag = True
                    else:
                        t_back_flag = False
                    tBack = float(tBackList[j][1])
                    tSlope = float(tSlopeList[j][1])
                else:
                    t_back_flag = False
                    tBack = 0.0
                    tSlope = 0.0
                if (N_HList[j][1] != []):
                    nH_flag = True
                    N_H = float(N_HList[j][1])
                    beta_dust = float(beta_dustList[j][1])
                    kappa_1300 = float(kappa_1300List[j][1])
                else:
                    nH_flag = False
                    N_H = 0.0
                    beta_dust = 0.0
                    kappa_1300 = 0.0
                Noise = float(NoiseList[j][1])                                              ## and noise

                # Debug:
                #    print "\n\n\n\n\n\n\n\n\n\n\n"
                #    print "j, CounterSpectra = ", j, CounterSpectra
                #    print "FreqMin = ", FreqMin
                #    print "FreqMax = ", FreqMax
                #    print "tBack = ", tBack
                #    print "tSlope = ", tSlope
                #    print "N_H = ", N_H
                #    print "beta_dust = ", beta_dust
                #    print "kappa_1300 = ", kappa_1300
                #    print "Noise = ", Noise


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## loop over all molecules
                counterMolecules = (-1)
                for molfitFile in MolfitFileForEachMolecule:
                    counterMolecules += 1


                    ## get name of current molecule
                    TransFreqList = []
                    if (counterMolecules < len(MolfitFileForEachMolecule) - 1):
                        CurrentMolecule = MoleculeFileName(MoleculesInOverallMolfitFile[counterMolecules])
                        LocalElowMax = 1.e6
                        LocalMaxNumTransInFit = 0
                        TransFreqList, DBParamList = GetTransFreq(MoleculesInOverallMolfitFile[counterMolecules], FreqMin, FreqMax, LocalElowMax, \
                                                                  FilenameDB, LocalMaxNumTransInFit)
                    else:
                        CurrentMolecule = "all"

                    # Debug:
                    #    print "molfitFile = ", molfitFile
                    #    print "CurrentMolecule = ", CurrentMolecule
                    #    print "MoleculesInOverallMolfitFile[counterMolecules] = ", MoleculesInOverallMolfitFile[counterMolecules]
                    #    print "TransFreqList = ", TransFreqList


                    ## consider only molecules which have transitions
                    if (TransFreqList != [] or CurrentMolecule == "all"):


                        ## create working directory for current molecule
                        SubDirectoryForCurrentMolecule = IdentifiedMoleculesSubDirectories + CurrentMolecule + "/"
                        command_string = "mkdir -p " + SubDirectoryForCurrentMolecule
                        os.system(command_string)


                        ## copy myXCLASS exe file to temp directory
                        command_string = "cp " + myXCLASSProgramDirectory + myXCLASSexe + " " + SubDirectoryForCurrentMolecule
                        os.system(command_string)


                        ## write molfit file to the subdirectory of the current molecule
                        NameOfCurrentMolfitFile = SubDirectoryForCurrentMolecule + CurrentMolecule + ".molfit"
                        CurrentMolfitFile = open(NameOfCurrentMolfitFile, 'w')
                        for line in molfitFile:
                            CurrentMolfitFile.write(line)
                        CurrentMolfitFile.close()


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## write iso ratio file to current subdirectory
                        if (iso_flag):
                            IsoTableFileName = SubDirectoryForCurrentMolecule + CurrentMolecule + ".iso"
                            IsoTableFile = open(IsoTableFileName, 'w')
                            for line in IsoRatioFileContent:
                                IsoTableFile.write(line)
                            IsoTableFile.close()


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## convert molfit to xml file
                        printflag = "false"
                        xmlFileName = SubDirectoryForCurrentMolecule + "parameter.xml"
                        ok = task_myXCLASS.ConversionMolfit2XML(printflag, NameOfCurrentMolfitFile, xmlFileName)
                        if (str(iso_flag).lower() == "true"):
                            ok = task_myXCLASS.GetIsoTableParameters(printflag, IsoTableFileName, xmlFileName)


                        ##--------------------------------------------------------------------------------------------------------------------------------
                        ## prepare call of myXCLASS


                        ## create command line call of myXCLASS program
                        myXCLASScommand = "cd " + SubDirectoryForCurrentMolecule + "; "
                        myXCLASScommand += "./" + myXCLASSexe + " " + str(FreqMin) + " " + str(FreqMax) + " " + str(FreqStep) + " " \
                                                              + str(TelescopeSize) + " " + str(InterFlag) + " " + str(t_back_flag) + " " \
                                                              + str(tBack) + " " + str(tSlope)

                        ## check for nH_flag
                        myXCLASScommand += " " + str(nH_flag)
                        if (str(nH_flag).lower() == "true"):
                            myXCLASScommand += " " + str(N_H) + " " + str(beta_dust) + " " + str(kappa_1300)


                        ## add global v_lsr
                        myXCLASScommand += " " + str(GlobalvLSR)


                        ## check for iso_flag
                        myXCLASScommand += " " + xmlFileName.strip() + " " + str(iso_flag)


                        ## add number of processors
                        i = 1                                                               ## do not produce output files for intensities of each comp.
                        myXCLASScommand += " " + str(i) + " " + FilenameDB.strip() + "; "


                        ## clean up directory
                        if (CounterSpectra == len(ListOfSpectra) - 1):
                            myXCLASScommand += "rm -rf " + myXCLASSexe

                        # Debug:
                        #    print "\n\n\nmyXCLASScommand = ", myXCLASScommand
                        #    print "SubDirectoryForCurrentMolecule = ", SubDirectoryForCurrentMolecule


                        ##============================================================================================================================
                        ## start myXCLASS
                        os.system(myXCLASScommand)


                        ##============================================================================================================================
                        ## read in calculated spectrum
                        myXCLASSoutputFilename = SubDirectoryForCurrentMolecule + "xclass_spectrum_output.dat"
                        ImportModeledSpectrumFlag = "true"
                        try:
                            modeldata = numpy.loadtxt(myXCLASSoutputFilename, skiprows = 0)
                            if (abs(numpy.min(modeldata[:, 1])) != 0.0 or abs(numpy.max(modeldata[:, 1])) != 0.0):
                                if (tBack != 0 or tSlope != 0):                             ## apply a baseline correction if one of the input parameter
                                                                                            ## tBack or tSlope are unequal to zero.
                                    for i in xrange(len(modeldata[:, 1])):                  ## loop over all points in the spectrum
                                        modeldata[i, 1] = modeldata[i, 1] - (abs(tBack) * (modeldata[i, 0] / FreqMin)**tSlope)
                            ModFreqArray = modeldata[:, 0]
                        except IOError, err:
                            ImportModeledSpectrumFlag = "false"

                        # Debug:
                        # print "CurrentMolecule, ImportModeledSpectrumFlag = ", CurrentMolecule, ImportModeledSpectrumFlag


                        if (ImportModeledSpectrumFlag == "true"):


                            ## define extended spectral name
                            ExtendedSpectraName = SpectraName + "__" + str(FreqMin) + "_-_" + str(FreqMax) + "_MHz"


                            ## read in transition energies from file
                            myXCLASSTransFilename = SubDirectoryForCurrentMolecule + "transition_energies.dat"
                            try:
                                f = open(myXCLASSTransFilename)
                                contents = f.readlines()
                                f.close()
                                myXCLASSTrans = []
                                for line in contents:
                                    StrippedLine = line.strip()
                                    i = StrippedLine.find("%")
                                    if (i > (-1)):
                                        StrippedLine = StrippedLine[:i].strip()
                                    if (StrippedLine != ""):
                                        SplittedLine = StrippedLine.split()
                                        myXCLASSTrans.append(float(SplittedLine[0]))

                                cmdString = "mv " + myXCLASSTransFilename + " " + SubDirectoryForCurrentMolecule \
                                                                                    + "transition_energies___" + ExtendedSpectraName + ".dat; "
                                cmdString += "mv " + SubDirectoryForCurrentMolecule + "xclass_spectrum.log " + SubDirectoryForCurrentMolecule \
                                                                                    + "xclass___" + ExtendedSpectraName + ".log; "
                                cmdString += "mv " + SubDirectoryForCurrentMolecule + "xclass_spectrum_output.dat " + SubDirectoryForCurrentMolecule \
                                                                                    + "modeled_spectrum___" + ExtendedSpectraName + ".dat"
                                os.system(cmdString)
                            except IOError, err:
                                myXCLASSTrans = []


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## define new frequency ranges
                            MolName = CurrentMolecule
                            if (CurrentMolecule == "all" and len(MoleculesInOverallMolfitFile) > 2):
                                NewFreqRanges = [[str(FreqMin), str(FreqMax), str(FreqStep), str(t_back_flag), str(tBack), str(tSlope)]]
                            else:
                                MaxWidth = 2.0
                                #    if (CurrentMolecule == "all" and 1==2):
                                #        MolName = MoleculesInOverallMolfitFile[0]
                                #    else:
                                #        MolName = MoleculesInOverallMolfitFile[counterMolecules]

                                if (CurrentMolecule != "all"):
                                    MolName = MoleculesInOverallMolfitFile[counterMolecules]
                                LocalvelLowLimit = -100.0
                                LocalvelUpLimit = 100.0
                                LocalMaxNumTransInFit = 1.e99
                                NewFreqRanges = DefineNewFreqRanges(LowestDataPoint, HighestDataPoint, MolName, FreqMin, FreqMax, FreqStep, t_back_flag, \
                                                                    tBack, tSlope, nH_flag, N_H, beta_dust, kappa_1300, Noise, LocalvelLowLimit, \
                                                                    LocalvelUpLimit, MaxWidth, ElowMax, FilenameDB, LocalMaxNumTransInFit)
                                if (CurrentMolecule == "all"):
                                    NewFreqRanges.insert(0, [str(FreqMin), str(FreqMax), str(FreqStep), str(t_back_flag), str(tBack), str(tSlope)])

                            # Debug:
                            #    print "\nNewFreqRanges = ", NewFreqRanges
                            #    print "CurrentMolecule = ", CurrentMolecule
                            #    print "MolName = ", MolName
                            #    print "len(MolfitFileForEachMolecule) = ", len(MolfitFileForEachMolecule)
                            #    print "MoleculesInOverallMolfitFile[0] = ", MoleculesInOverallMolfitFile[0]
                            #    print "counterMolecules = ", counterMolecules
                            #    print "MoleculesInOverallMolfitFile = ", MoleculesInOverallMolfitFile
                            #    print "MoleculeFileName = ", MoleculeFileName
                            #    print "MoleculeFileName(MoleculesInOverallMolfitFile[counterMolecules]) = ", \
                            #          MoleculeFileName(MoleculesInOverallMolfitFile[counterMolecules])
                            #    print "FreqMin = ", FreqMin
                            #    print "FreqMax = ", FreqMax
                            #    print "FreqStep = ", FreqStep
                            #    print "t_back_flag = ", t_back_flag
                            #    print "tBack = ", tBack
                            #    print "tSlope = ", tSlope


                            ##----------------------------------------------------------------------------------------------------------------------------
                            ## define new frequency ranges
                            for FreqRange in NewFreqRanges:
                                LocalFreqMin = float(FreqRange[0])
                                LocalFreqMax = float(FreqRange[1])
                                tBackLocal = float(FreqRange[4])
                                tSlopeLocal = float(FreqRange[5])

                                # Debug:
                                #    print "LocalFreqMin = ", LocalFreqMin
                                #    print "LocalFreqMax = ", LocalFreqMax


                                ## get index for LocalFreqMin and LocalFreqMax
                                ObsFreqMinIndex = max(0, (numpy.abs(FreqArray - LocalFreqMin)).argmin() - 1)
                                ObsFreqMaxIndex = min(len(FreqArray) - 1, (numpy.abs(FreqArray - LocalFreqMax)).argmin() + 1)
                                In1 = min(ObsFreqMinIndex, ObsFreqMaxIndex)
                                In2 = max(ObsFreqMinIndex, ObsFreqMaxIndex)
                                ObsSpectrum = LocalObsSpectrum[In1:In2, :]

                                # Debug:
                                #    print "ObsFreqMinIndex = ", ObsFreqMinIndex
                                #    print "ObsFreqMaxIndex = ", ObsFreqMaxIndex


                                ## get index for LocalFreqMin and LocalFreqMax

                                ModFreqMinIndex = max(0, (numpy.abs(ModFreqArray - LocalFreqMin)).argmin() - 1)
                                ModFreqMaxIndex = min(len(ModFreqArray) - 1, (numpy.abs(ModFreqArray - LocalFreqMax)).argmin() + 1)
                                In1 = min(ModFreqMinIndex, ModFreqMaxIndex)
                                In2 = max(ModFreqMinIndex, ModFreqMaxIndex)
                                ModSpectrum = modeldata[In1:In2, :]

                                # Debug:
                                #    print "ModFreqMinIndex = ", ModFreqMinIndex
                                #    print "ModFreqMaxIndex = ", ModFreqMaxIndex


                                ## produce plots for each range
                                fig = pylab.figure(figsize=(15, 10))
                                fig.clear()
                                pylab.subplots_adjust(hspace=0.45, wspace=0.2)
                                layer = pylab.subplot(1, 1, 1)
                                tl = pylab.title("Contribution of " + MoleculesInOverallMolfitFile[counterMolecules])
                                layer.grid(True)
                                layer.set_ylabel(r"T$_{mb}$ [K]")
                                layer.set_xlabel("Rest Frequency [MHz]")
                                if (myXCLASSTrans != [] and CurrentMolecule != "all"):
                                    for lines in myXCLASSTrans:
                                        layer.axvline(x=lines, color='green', linewidth=1, linestyle='--')
                                layer.axhline(y=Noise, color='blue', linewidth=1, linestyle='--')
                                if (numpy.min(ModSpectrum[:, 1]) < 0 or numpy.min(ObsSpectrum[:, 1]) < 0):
                                    layer.axhline(y=-Noise, color='blue', linewidth=1, linestyle='--')
                                layer.plot(ObsSpectrum[:, 0], ObsSpectrum[:, 1], '-', color='black', linewidth=3.0, label = 'exp. data', \
                                           drawstyle='steps-mid')
                                layer.set_xlim(ObsSpectrum[0, 0], ObsSpectrum[-1, 0])


                                ## determine min. and max. values for current plot
                                y1Min = numpy.min(ObsSpectrum[:, 1])
                                y2Min = numpy.min(ModSpectrum[:, 1])
                                yMin = min(y1Min, y2Min)
                                y1Max = numpy.max(ObsSpectrum[:, 1])
                                y2Max = numpy.max(ModSpectrum[:, 1])
                                yMax = max(y1Max, y2Max)
                                if (yMin < -10.0):
                                    layer.set_ylim(-2.0, yMax * 1.07)
                                layer.plot(ModSpectrum[:, 0], ModSpectrum[:, 1], '-', color='red', linewidth=2.0, label = 'fit')
                                layer.legend()
                                pngFile = SubDirectoryForCurrentMolecule + MolName + "__" + SpectraName + "__" + str(LocalFreqMin) + "_-_"
                                pngFile += str(LocalFreqMax) + "_MHz"
                                pylab.savefig(pngFile + ".png", dpi=100)
                                pylab.draw()
                                matplotlib.pyplot.close(fig)


    ##====================================================================================================================================================
    ## define return variable
    return (BestMolfitFile, JobDir)
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
##--------------------------------------------------------------------------------------------------------------------------------------------------------
