#!/usr/bin/python
# -*- coding: utf-8 -*-
##********************************************************************************************************************************************************
##
##  This module reads in entries from the sqlite3 database around a selected frequency
##  Copyright (C) 2012 - 2016  Thomas Moeller
##
##  I. Physikalisches Institut, University of Cologne
##
##
##
##  The following functions are included in this module:
##
##      - function ListDatabase:                        function reads in entries from the sqlite3 database
##      - function Cursor.__init__:                     initialize cursor class
##      - function Cursor.connect:                      connect all the stored connection ids
##      - function Cursor.disconnect:                   disconnect all the stored connection ids
##      - function Cursor.on_key:                       (only for debugging) on key event
##      - function Cursor.mouse_move:                   define what to do when mouse is moved
##      - function Cursor.onclick:                      print out information from database on mouse click
##      - function GetTransitions:                      function reads in entries from the sqlite3 database around a selected frequency
##
##
##
##  Versions of the program:
##
##  Who           When         What
##
##  T. Moeller    25.07.2013   initial version
##
##
##
##  License:
##
##    GNU GENERAL PUBLIC LICENSE
##    Version 3, 29 June 2007
##    (Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>)
##
##
##    This program is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program.  If not, see <http://www.gnu.org/licenses/>.
##
##********************************************************************************************************************************************************


##********************************************************************* load packages ********************************************************************
from __future__ import print_function                                                       ## import print_function package
import numpy                                                                                ## import numpy package
import os                                                                                   ## import os package
import sys                                                                                  ## import sys package
import string                                                                               ## import string package
import random                                                                               ## import random package
import datetime                                                                             ## import datetime package
import sqlite3                                                                              ## import sqlite3 package
from pylab import *                                                                         ## import pylab package
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## define routines for mouse handling
##
class Cursor:


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## initialize
    def __init__(self, ax, conn, ListOfSelectedMolecules, FrequencyWidth, ElowMin, ElowMax, LowerLimitExpX, UpperLimitExpX):
        self.ax = ax
        self.lx = ax.axhline(color='k')                                                     ## the horiz line
        self.ly = ax.axvline(color='k')                                                     ## the vert line
        self.conn = conn
        self.FrequencyWidth = FrequencyWidth
        self.ElowMin = ElowMin / 1.42879                                                    ## convert K to cm-1
        self.ElowMax = ElowMax / 1.42879                                                    ## convert K to cm-1
        self.ListOfSelectedMolecules = ListOfSelectedMolecules
        self.LowerLimitExpX = LowerLimitExpX
        self.UpperLimitExpX = UpperLimitExpX


        ## text location in axes coords
        self.txt = ax.text( 0.5, 0.9, '', transform=ax.transAxes)
        self.ax.set_xlim(self.LowerLimitExpX, self.UpperLimitExpX)
        # self.ax.xaxis.set_major_formatter(matplotlib.ticker.FormatStrFormatter('%.4e'))
        return


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## connect all the stored connection ids
    def connect(self):
        self.cidpress = self.ax.figure.canvas.mpl_connect('button_press_event', self.onclick)
        self.cidmotion = self.ax.figure.canvas.mpl_connect('motion_notify_event', self.mouse_move)
        # self.cidkey = self.ax.figure.canvas.mpl_connect('key_press_event', self.on_key)
        return


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## disconnect all the stored connection ids
    def disconnect(self):
        self.ax.figure.canvas.mpl_disconnect(self.cidpress)
        self.ax.figure.canvas.mpl_disconnect(self.cidmotion)
        # self.ax.figure.canvas.mpl_disconnect(self.cidkey)
        return


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## (only for debugging): on key event
    def on_key(self, event):
        print('you pressed', event.key, event.xdata, event.ydata)
        return


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## define what to do when mouse is moved
    def mouse_move(self, event):
        if not event.inaxes: return
        x, y = event.xdata, event.ydata


        ## update the line positions
        self.lx.set_ydata(y )
        self.ly.set_xdata(x )
        # self.ax.set_xlim(self.LowerLimitExpX, self.UpperLimitExpX)
        self.txt.set_text( 'Freq.=%1.2f MHz, Int.=%1.2f'%(x,y) )
        self.ax.patch.figure.canvas.draw()
        return


    ##----------------------------------------------------------------------------------------------------------------------------------------------------
    ## print out information from database on mouse click
    def onclick(self, event):

        # Debug:
        # print('you pressed', event.key, event.xdata, event.ydata)

        if (event.key != "escape"): return
        print ("\n\nPlease wait ..")


        ## define frequency interval form mouse position
        FreqIntMin = event.xdata - self.FrequencyWidth
        FreqIntMax = event.xdata + self.FrequencyWidth


        ## initialize some parameter
        NameOfRadTransTable = "transitions"
        ColumnNameForNameTransitions = "T_Name"
        ColumnNameForFreqTransitions = "T_Frequency"
        ColumnNameForIntTransitions = "T_Intensity"
        ColumnNameForEinsteinATransitions = "T_EinsteinA"
        ColumnNameForFreqErrTransitions = "T_Uncertainty"
        ColumnNameForELowTransitions = "T_EnergyLower"
        ColumnNameForgUpTransitions = "T_UpperStateDegeneracy"
        ColumnNameForQNUpLabelTransitions = "T_UpperStateQuantumNumbers"
        ColumnNameForQNLowLabelTransitions = "T_LowerStateQuantumNumbers"
        ColumnNameForURLTransitions = "T_URL"


        ## define query string
        ##
        ## get names of columns with
        ##      import sqlite3
        ##      connection = sqlite3.connect('cdms_sqlite.db')
        ##      cursor = connection.execute('select * from partitionfunctions')
        ##      names = list(map(lambda x: x[0], cursor.description))
        ##
        query_string = "SELECT "
        query_string += ColumnNameForNameTransitions + ", "                                 ## position 1: molecule name
        query_string += ColumnNameForFreqTransitions + ", "                                 ## position 2: frequency
        query_string += ColumnNameForIntTransitions + ", "                                  ## position 3: intensity
        query_string += ColumnNameForEinsteinATransitions + ", "                            ## position 4: Einstein A
        query_string += ColumnNameForFreqErrTransitions + ", "                              ## position 5: Error frequency
        query_string += ColumnNameForELowTransitions + ", "                                 ## position 6: E_low
        query_string += ColumnNameForgUpTransitions + ", "                                  ## position 7: upper state degeneracy
        query_string += ColumnNameForQNUpLabelTransitions + ", "                            ## position 8: quantum number label for upper state
        query_string += ColumnNameForQNLowLabelTransitions                                  ## position 9: quantum number label for lower state
        # query_string += ColumnNameForURLTransitions                                         ## position 10: URL
        query_string += " FROM " + NameOfRadTransTable
        query_string += " WHERE (" + ColumnNameForFreqTransitions + " >= " + str(FreqIntMin)
        query_string += " and " + ColumnNameForFreqTransitions + " <= " + str(FreqIntMax)
        if (self.ElowMin < self.ElowMax):
            query_string += " and " + ColumnNameForELowTransitions + " >= " + str(self.ElowMin)
            query_string += " and " + ColumnNameForELowTransitions + " <= " + str(self.ElowMax)
        if (len(self.ListOfSelectedMolecules) > 0):
            if (self.ListOfSelectedMolecules[0] != ["None"]):
                query_string += " and ("
                counter = 0
                for molecule in self.ListOfSelectedMolecules:
                    counter += 1
                    if (counter > 1):
                        query_string += " or "
                    query_string += ColumnNameForNameTransitions + " = " + chr(34) + molecule.strip() + chr(34)
                query_string += ")"
        query_string += ")"
        query_string += " ORDER by " + ColumnNameForFreqTransitions

        # Debug:
        # print " "
        # print ("query_string = ", query_string)
        # print "len(query_string) = ", len(query_string)


        ## read data from database
        cursor = self.conn.cursor()
        cursor.execute(query_string)
        rows = cursor.fetchall()


        ## open file if file name is defined
        file_flag = "false"
        OutputDevice = ""
        formatted_line = "%25s" % "Name:"
        formatted_line += "%37s" % "quantum number upper state:"
        formatted_line += "%37s" % "quantum number lower state:"
        formatted_line += "%18s" % "Frequency [MHz]:"
        formatted_line += "%24s" % "Error Frequency [MHz]:"
        formatted_line += "%27s" % "Einstein A coefficient:"
        # formatted_line += "%27s" % "log(Intensity) [nm2 MHz]:"
        formatted_line += "%20s" % "Energy_low [K]:"
        if (OutputDevice.strip() != "" and OutputDevice != None):
            file_flag = "true"
            out = open(OutputDevice.strip(), 'w')
            out.write(formatted_line + "\n")
        else:
            if (len(rows) > 0):
                print ('\n\nContents of the database between %s MHz and %s MHz:\n' % (str(FreqIntMin), str(FreqIntMax)))
                print (formatted_line)
            else:
                print ('\n\nNo entries in the database between %s MHz and %s MHz:\n' % (str(FreqIntMin), str(FreqIntMax)))
                return


        ## store entries in output variable Contents
        Contents = []
        col_counter = 0
        for row in rows:                                                                    ## loop over all entries of the database
            col_counter += 1
            Contents.append(row)

            # Debug:
            # print "col_counter, row = ", col_counter, row


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## define formatted line
            ## position 1: molecule name
            ## position 2: frequency
            ## position 3: intensity
            ## position 4: Einstein A
            ## position 5: Error frequency
            ## position 6: E_low
            ## position 7: upper state degeneracy
            ## position 8: quantum number label for upper state
            ## position 9: quantum number label for lower state
            ## position 10: URL
            formatted_line = ""
            name = ""
            freq = ""
            unfreq = ""
            intensity = ""
            EinsteinA = ""
            elow = ""
            QNUpLabel = ""
            QNLowLabel = ""
            DBurl = ""
            element_counter = 0
            for elements in row:
                element_counter += 1

                if (element_counter == 1):                                                  ## get name of molecule
                    name = elements.strip()
                    name = '%25s' % name[:25]                                               ## name of molecule has max. 25 characters

                elif (element_counter == 2):                                                ## get frequency (in MHz)
                    freqVal = '%17s' % elements
                    if (freqVal.strip() != "None"):
                        freqVal = float(freqVal)
                        freq = '%17.5f' % freqVal
                    else:
                        freq = freqVal

                    # Debug:
                    # print "freq = ", freq

                elif (element_counter == 3):                                                ## get intensity
                    intensityVal = '%26s' % elements
                    if (intensityVal.strip() != "None"):
                        intensityVal = float(intensityVal)
                        intensity = '%26.9e' % intensityVal
                    else:
                        intensity = intensityVal

                    # Debug:
                    # print "intensity = ", intensity

                elif (element_counter == 4):                                                ## get Einstein A coefficient
                    EinsteinAVal = '%26s' % elements
                    if (EinsteinAVal.strip() != "None"):
                        EinsteinAVal = float(EinsteinAVal)
                        EinsteinA = '%26.3e' % EinsteinAVal
                    else:
                        EinsteinA = EinsteinAVal

                    # Debug:
                    # print "EinsteinA = ", EinsteinA

                elif (element_counter == 5):                                                ## get uncertainty of frequency (in MHz)
                    unfreqVal = '%23s' % elements
                    if (unfreqVal.strip() != "None"):
                        unfreqVal = float(unfreqVal)
                        unfreq = '%23.4e' % unfreqVal
                    else:
                        unfreq = unfreqVal

                    # Debug:
                    # print "unfreq = ", unfreq

                elif (element_counter == 6):                                                ## get E_lower (in cm-1)
                    elowVal = '%19s' % elements
                    if (elowVal.strip() != "None"):
                        elowVal = float(elowVal) * 1.42879                                  ## convert cm-1 to K
                        if (abs(elowVal) > 0.01 and abs(elowVal) < 10000):
                            elow = '%19.3f' % elowVal
                        else:
                            elow = '%19.3e' % elowVal
                    else:
                        elow = elowVal

                    # Debug:
                    # print "elow = ", elow

                elif (element_counter == 8):                                                ## get quantum number label for upper state
                    try:
                        lab = elements
                        lab = lab.replace("ElecStateLabel = ", "").strip()
                        if (len(lab) > 34):
                            lab = lab[:31] + "..."
                        QNUpLabel = '  %34s' % lab
                    except TypeError:
                        QNUpLabel = "     "

                    # Debug:
                    # print "QNUpLabel = ", QNUpLabel

                elif (element_counter == 9):                                                ## get quantum number label for lower state
                    try:
                        lab = elements
                        lab = lab.replace("ElecStateLabel = ", "").strip()
                        if (len(lab) > 34):
                            lab = lab[:31] + "..."
                        QNLowLabel = '  %34s' % lab
                    except TypeError:
                        QNLowLabel = "     "

                    # Debug:
                    # print "QNLowLabel = ", QNLowLabel

                #elif (element_counter == 10):                                              ## get URL
                #    try:
                #        DBurl = '%s' % elements
                #    except TypeError:
                #        DBurl = "     "

                    # Debug:
                    # print "DBurl = ", DBurl


            ## construct line for output
            formatted_line = name + " " + QNUpLabel + " " + QNLowLabel + " " + freq + " " + unfreq + " " + EinsteinA + " " + elow + " " + DBurl

            # Debug:
            # print "formatted_line = ", formatted_line


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## if file name is defined
            if (file_flag == "true"):
                out.write(formatted_line + "\n")
            else:
                print (formatted_line)


        ## close file, if file name is defined
        if (file_flag == "true"):
            out.close()


        ## do some screen formating
        print (" ")
        print (" ")
        print (" ")


        ## finished
        return
##--------------------------------------------------------------------------------------------------------------------------------------------------------


##--------------------------------------------------------------------------------------------------------------------------------------------------------
##
## This function reads in entries from the sqlite3 database around a selected frequency
##
def GetTransitions(expdata, FreqMin, FreqMax, SelectMolecule, FrequencyWidth, ElowMin, ElowMax):
    """

Therefore, the function plots the observational data (contained in the input parameter "expdata") from frequency
"FreqMin" to "FreqMax". By using the mouse cursor and pressing the mouse button and the "Escape-key"
simultaneously, the user selects a frequency and gets all entries for the corresponding transitions around
the selected frequency from the table "Transitions". The frequency interval is defined by the selected
frequency +/- the input parameter "FrequencyWidth".

The informations are printed out to the screen.


Bug:

Please note, due to a bug in the matplotlib package version 0.99 and lower, the following error message is printed out to the screen

    SEVERE	GetTransitions::::casa	An error occurred running task GetTransitions.

You can ignore this message. The function still works!


Please note, you have to close the plotting window, to stop the function.



input parameters:
-----------------


    - expdata:              2D numpy array containing the observational data

                            Note, the data has to be given as a function of
                            frequency (in MHz).

    - FreqMin:              minimum frequency (in MHz) (default: 0) for plot window

    - FreqMax:              maximum frequency (in MHz) (default: 1.e8) for plot window

    - SelectMolecule:       a (python) list containing the names of all molecules which should be considered
                            or an (ASCII) file name including the molecules which should be considered.

                            If this parameter is left empty, all molecules in the database within the given
                            range are considered!

                            Note, if the paramter defines a relative path, this path has to be defined
                            relative to the current working directory!

    - FrequencyWidth:       defines the width of the frequency interval in MHz (selected frequency
                            +/- "FrequencyWidth" where the line informations are printed out to the screen.
                            (default: 5).

    - ElowMin:              minimum for lower energy (in K) (default: 0) in the "Transitions" table

    - ElowMax:              maximum for lower energy (in K) (default: 1.e6) in the "Transitions" table



Example 1:
----------

FileName = "demo/LoadASCIIFile/ASCII.dat"
NumHeaderLines = 0
expdata = LoadASCIIFile()

FreqMin = 4.92100000e+05
FreqMax = 4.92200000e+05
SelectMolecule = "demo/ListDatabase/molecules.txt"
FrequencyWidth = 5.0
ElowMin = 100.0
ElowMax = 1000.0
GetTransitions()


Example 2:
----------

FileName = "demo/LoadASCIIFile/ASCII.dat"
NumHeaderLines = 0
expdata = LoadASCIIFile()

FreqMin = 4.92100000e+05
FreqMax = 4.92200000e+05
SelectMolecule = ["HNO3;v5=1;", "C2H5CN-15;v=0;"]
FrequencyWidth = 5.0
ElowMin = 100.0
ElowMax = 1000.0
GetTransitions()

    """

    ##====================================================================================================================================================
    ## check input parameters

    # Debug:
    # print "FreqMin = ", FreqMin
    # print "FreqMax = ", FreqMax
    # print "SelectMolecule = ", SelectMolecule
    # print "FrequencyWidth = ", FrequencyWidth
    # print "ElowMin = ", ElowMin
    # print "ElowMax = ", ElowMax


    ##====================================================================================================================================================
    ## analyze expdata parameter
    if (type(expdata).__name__ == 'list'):                                                  ## is expdata a list ?
        expdata_flag = "list"
    else:                                                                                   ## is expdata an array ?
        expdata_flag = "array"
    PlotExpDataFlag = "true"
    if (expdata == None or expdata == [] or expdata == "" or len(expdata[0]) < 2 or len(expdata) < 3):
        PlotExpDataFlag = "false"
    if (PlotExpDataFlag == "false"):
        print ("Error in XCLASS package, function GetTransitions:")
        print (" ")
        print ("  Please specify a input parameter named 'expdata' containing observational data as a 2D python list or array!")
        print (" ")
        print ("  Please correct parameter 'expdata' and start myXCLASSPlot again.")
        print (" ")
        print (" ")
        return


    ##====================================================================================================================================================
    ## analyze xlimit parameter
    xlimits_flag = "false"
    if (FreqMin != 0.0 or FreqMax != 0.0):
        if (FreqMin != FreqMax):
            if (FreqMin < FreqMax):
                xlimits_flag = "true"
            else:
                if (xlimits_flag == "true"):
                    print ("Error in XCLASS package, function GetTransitions:")
                    print (" ")
                    print ("  The given upper limit of the frequency is smaller than the lower limit!")
                    print (" ")
                    print ("  Please correct the frequency range definition and start myXCLASSPlot again.")
                    print (" ")
                    print (" ")
                    return

    # Debug:
    # print "xlimits_flag = ", xlimits_flag


    ##====================================================================================================================================================
    ## test, if SelectMolecule is a list, otherwise try to use SelectMolecule as filename
    print (" ")
    print (" ")
    print ("Analyze selected molecules ..")
    sys.stdout.flush()
    ListOfSelectedMolecules = []
    if (len(SelectMolecule) > 0 and SelectMolecule != ["None"]):
        for entry in SelectMolecule:                                                        ## loop over all entries of SelectMolecule list
            entryStrip = entry.strip()                                                      ## remove leading and trailing blanks


            ## check, if current entry is a file
            if (os.path.isfile(entryStrip)):


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## if file is a molfit file, get names of molecules from molfit file
                if (entryStrip.endswith("molfit")):

                    # Debug:
                    # print "Selection string is a molfit file!"


                    ## read in contents of molfit file
                    f = open(entryStrip)
                    MolfitContents = f.readlines()                                          ## read in contents of molfit file
                    f.close()


                    ## get names of molecules
                    for lines in MolfitContents:                                            ## loop over all lines of molfit file
                        CurrentLine = lines.strip()                                         ## remove leading and trailing blanks

                        # Debug:
                        # print "CurrentLine = ", CurrentLine


                        ## check for comments
                        w = CurrentLine.find("%")                                           ## are there comments in the current line ?
                        if (w == 0):                                                        ## ignore lines which contains only comments
                            CurrentLine = ""                                                ## if line is only a comment clear line
                        elif (w > 0):                                                       ## is there a comment in the current line ?
                            CurrentLine = CurrentLine[:w]                                   ## remove comments

                        # Debug:
                        # print "CurrentLine = ", CurrentLine


                        ## analyze, if the current line contains the name of a molecule
                        if (CurrentLine != ""):                                             ## ignore empty lines
                            SplitLines = CurrentLine.split()                                ## split current line into columns

                            # Debug:
                            # print "SplitLines = ", SplitLines


                            ## contains the last column an integer number?
                            helpstring = SplitLines[-1].strip()
                            is_int = "false"
                            if (helpstring.isdigit()):                                      ## continue, if string is an integer number
                                is_int = "true"


                            ## If yes, then the current line contains the name of a molecule
                            if (is_int == "true"):                                          ## if last entry in a line is an integer number, then
                                MolName = ""                                                ## the current line contains the name of a molecule
                                for col in SplitLines[:-1]:                                 ## get whole line without integer number at the end
                                    MolName += col                                          ## construct entry for SelectMolecule

                                # Debug:
                                # print "MolName = ", MolName

                                ListOfSelectedMolecules.append(MolName)                     ## construct final array containing all molecule names


                ##----------------------------------------------------------------------------------------------------------------------------------------
                ## read names of molecules from ASCII file
                else:

                    # Debug:
                    # print "Selection string is a ASCII file!"


                    ## read in whole ASCII file
                    f = open(entryStrip)
                    AllLines = f.readlines()
                    f.close()


                    ## append lines in ASCII file to selection list
                    for line in AllLines:
                        ListOfSelectedMolecules.append(line)


            ##--------------------------------------------------------------------------------------------------------------------------------------------
            ## continue here, if current entry is not a filename
            else:
                ListOfSelectedMolecules.append(entryStrip)


    ##====================================================================================================================================================
    ## connect to the sqlite3 database


    ######################################################################################################################################################
    ## DO NOT EDIT OR REMOVE THE FOLLOWING LINE !!!!
    dbFilename = "/home/moeller/ALMA/CASA/myXCLASS-CASA-Interface/No-NR-version/Linux/XCLASS-Interface/Database/cdms_sqlite.db"
    ######################################################################################################################################################

    try:
        conn = sqlite3.connect(dbFilename)
        print (" ")
        print ("Connection to sqlite3 database %s established." % dbFilename)
        print (" ")
        print (" ")
    except sqlite3.Error, e:
        print (" ")
        print ("Can not connect to sqlite3 databse %s." % dbFilename)
        print ("Error: %d: %s" % (e.args[0], e.args[1]))
        sys.exit(1)


    ## print some help to screen
    print ("Select frequency by using the mouse cursor.")
    print ("To get the transitions around the selected frequency press the 'Escape'-key and the mouse button simultaneously!\n")
    print ("Close plot window to exit myXCLASSPlot function.\n")


    ##====================================================================================================================================================
    ## define plotting variables, clear and create plot


    ## define plotting variables
    if (expdata_flag == "list"):
        expX = [(li[0]) for li in expdata]
    else:
        expX = expdata[:,0]
    if (expdata_flag == "list"):
        expY = [(li[1]) for li in expdata]
    else:
        expY = expdata[:,1]


    ## clear plot window
    fig = figure(1)
    fig.clear()


    ## create plot window and define axes labels
    subplots_adjust(hspace=0.45, wspace=0.2)
    ax = subplot(1,1,1)
    ax.set_title("Select frequency, press 'Escape'-key and the mouse button simultaneously")
    ax.grid(True)
    ax.set_ylabel(r"T$_b$ [K]")
    ax.set_xlabel("Frequency [MHz]")


    ## set Limits
    LowestVal = min(s for s in expX)
    if (xlimits_flag == "false"):
        LowerLimitExpX = LowestVal
    else:
        if (FreqMin < LowestVal):
            LowerLimitExpX = LowestVal
        else:
            LowerLimitExpX = FreqMin
    HighestVal = max(s for s in expX)
    if (xlimits_flag == "false"):
        UpperLimitExpX = HighestVal
    else:
        if (FreqMax > HighestVal):
            UpperLimitExpX = HighestVal
        else:
            UpperLimitExpX = FreqMax

    # Debug:
    # print ('LowerLimitExpX=%19.2f, UpperLimitExpX=%19.2f' % (LowerLimitExpX, UpperLimitExpX))
    # print ('expX[0]=%19.2f, expX[-1]=%19.2f' % (expX[0], expX[-1]))
    # print ('LowestVal=%19.2f, HighestVal=%19.2f' % (LowestVal, HighestVal))


    ## add data to plot window
    ax.plot(expX, expY, '-', color='black', linewidth=2.0, label = 'obs. data', drawstyle='steps-mid')
    ax.set_xlim(LowerLimitExpX, UpperLimitExpX)


    ## handle mouse input
    cursor = Cursor(ax, conn, ListOfSelectedMolecules, FrequencyWidth, ElowMin, ElowMax, LowerLimitExpX, UpperLimitExpX)
    cursor.connect()
    show(block=True)
    cursor.disconnect()
    conn.close()                                                                            ## close connection to database


    ##====================================================================================================================================================
    ## define return variables
    return
##--------------------------------------------------------------------------------------------------------------------------------------------------------

