function readmodatm,parameter

;+
; NAME:
;	READMODATM
;
; PURPOSE:
;	This procedures reads the requested parameter (e.g. pressure, abundance)
;	from the MIPAS model atmosphere.
;
; CALLING SEQUENCE:
;	Result = READMODATM(Parameter)
;
; INPUTS:
;	Parameter:   Parameter to be read (scalar string) in format of MIPAS model
;		     atmosphere file.
;
; OUTPUTS:
;	This function returns the requested parameter in a float array.
;
; COMMON BLOCKS:
;       TELLREM_INFO:  This common block contains relevant folder names and strings
;	              for running tellrem. It has to be initialised by running
;		      LOADTELLREMINFO.
;
; RESTRICTIONS:
;	This function only works for the parameters available parameters in the
;	MIPAS model atmosphere. These are:
;	HGT, PRE, TEM, N2, O2, CO2, O3, H2O, CH4, N2O, HNO3, CO, NO2,
;	N2O5, ClO, HOCl, ClONO2, NO, HNO4, HCN, NH3, F11, F12, F14,
;	F22, CCl4, COF2, H2O2, C2H2, C2H6, OCS, SO2, SF6.
;
; EXAMPLE:
;	height = READMODATM('HGT')
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
common tellrem_info

; check the input parameters
if n_params() lt 1 then begin
    message,/info,'Syntax: Result = READMODATM(Parameter)'
    retall
endif

; how many lines has the file
n=numlines(modelatmosphere)
cont=strarr(n)

; read it
openr,unit,modelatmosphere,/get_lun
readf,unit,cont

; find the number of profile levels
therepl=where(strmatch(cont,'*Profile Levels*') eq 1,count)
if count eq 0 then begin
    print,'Could not determine number of profile levels! Check file!'
    retall
endif
reads,cont[therepl],npl

; create array to take data
array=fltarr(npl)

; find where the individual data portions lie
inddat=where(strmid(cont,0,1) eq '*')
continddat=cont[inddat]

; find the parameter asked for
there=where(strmatch(continddat,'\*'+parameter+' *') eq 1,count)
if count eq 0 then begin
    print,'Could not find parameter asked for! Check input and/or file!'
    retall
endif
reads,cont[(inddat[there]+1):(inddat[there+1]-1)],array
close,unit
free_lun,unit
return,array
end