///////////////////////////////////////////////////////////////////////////////////////////////
//
//	The KEPLERH program solves the Kepler equation for hyperbolic Keplerian orbits. The code is 
//	collected in the following modules: Keplerh.c, polinomios25.c polinomios25Q.c and hyperk.h.
//  The module main.c is a driver to check good operation of the program
//	Copyright (C) 2018 by the UNIVERSIDAD POLITECNICA DE MADRID (UPM)  
//	AUthors: Virginia Raposo-Pulido and Jesus Pelaez
//
//	This program is free software: you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation, either version 3 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program.  If not, see <https://www.gnu.org/licenses/>.
//
//////////////////////////////////////////////////////////////////////////////////////////////////

#include "hyperk.h"

/*************************************************************************************/
//
//This file contains the 24 polynomials QQHi(e, M) that provide the seed (S = sinh(H)) 
//of the modified Newton-Rapson method that solves the hyperbolic Kepler equation.
//The polynomials in this module are grade 5.
//
/*************************************************************************************/


double QQH0_ke (double e, double x)
{
  double t1, t11, t112, t118, t129, t149, t2, t20, t24, t29, t31, t32, t37, t4, t42, t47, t5, t58, t59, t6, t68, t74, t8, t86, t88, t99;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t20 = 0.1e1 / (0.208801853556464060172846674001845583e36 * t4 - 0.408707086854597186269411766966380406e36 * e + 0.199999999999999999999999999999999996e36);
  t24 = 0.1e1 / (0.250000000000000000000000000000000000e36 * t4 - 0.478313955950124144801090431097121357e36 * e + 0.228784240456657300784340395355441668e36);
  t29 = 0.1e1 / (0.125000000000000000000000000000000000e36 * e - 0.124377593619153311454313314074966573e36);
  t31 = 0.204353543427298593134705883483190205e36 * e - 0.199999999999999999999999999999999998e36;
  t32 = 0.1e1 / t31;
  t37 = 0.1e1 / (0.510883858568246482836764708707975510e36 * t4 - 0.988725758845683403670657945627647635e36 * e + 0.478313955950124144801090431097121352e36);
  t42 = 0.1e1 / (0.319302411605154051772977942942484695e36 * t11 - 0.947925049475863565670394745824782426e36 * t4 + 0.938018522412323939105498295127845890e36 * e - 0.309395715889824930733689240596732683e36);
  t47 = t5 * t11;
  t58 = 0.500000000000000000000000000000000000e36 * e - 0.478313955950124144801090431097121357e36;
  t59 = 0.1e1 / t58;
  t68 = 0.1e1 / (0.407816120227468867525091160159854655e36 * t8 - 0.201561253316469052611631096609806397e37 * t5 + 0.398476391603869935214387049153088988e37 * t11 - 0.393877364378860395595382160093819904e37 * t4 + 0.194662508689449242536874502317774301e37 * e - 0.384818946184516601190110018760320066e36);
  t74 = t5 * t5;
  t86 = t31 * t31;
  t88 = 0.1e1 / t86 / t31;
  t99 = t74 * e;
  t112 = t58 * t58;
  t118 = t74 * t4;
  t129 = 0.111368119099203973906410815518435742e36 * t118 - 0.108566894978128606702088932581405881e37 * t99 + 0.476252330516090458895187964793543494e37 * t74 - 0.123800902705659364370784950085084396e38 * t47 + 0.211188739881252782984744726717170052e38 * t6 - 0.247031314394892645957552275679872109e38 * t8 + 0.200660581504807867487299901076951739e38 * t5 - 0.111765260989567323576560669801052104e38 * t11 + 0.408519393704081499692248404857515478e37 * t4 - 0.884844100726072082447454653377228525e36 * e + 0.86243359612302934717272564075793715e35;
  t149 = 0.197753334362221370285862583084933442e144 * t74 * t11 - 0.129218182306798386659067444654879594e174 * t118 + 0.100365454761543724925028760609297998e175 * t99 - 0.337674127087304644525667248034505114e175 * t74 + 0.639052138015217374251831541622267100e175 * t47 - 0.735525430039195088406650381966203910e175 * t6 + 0.513062591169164154329577580190574955e175 * t8 - 0.194532872753458068543465347482048820e175 * t5 + 0.208604542929372860544226196637502148e174 * t11 + 0.997752162657495525840900326209474345e173 * t4 - 0.260815068397315239177919780208811877e173 * e - 0.557610708267290903141942619491996825e171;
  return(0.488281250000000000000000000000000000e145 * t2 * x * (0.437327486851215633006750731529872377e36 * t6 + 0.187337031705988070499015111203523559e67 * t8 - 0.819861240197865846349421365089801035e67 * t5 + 0.130349499121658934476819803971928170e68 * t11 - 0.867350408380737367129377540915733235e67 * t4 + 0.167034755764033960859332403734727968e67 * e + 0.293456059201043190338130138023153198e66) * t20 * t24 * t29 * t32 * t37 * t42 - 0.976562500000000000000000000000000000e110 * t2 * (0.221225959600832414040675711597376807e36 * t47 - 0.449803013236251426777843291314882300e66 * t6 + 0.167679475751243520148369878652322221e67 * t8 - 0.223037193877381166956931895412246883e67 * t5 + 0.113048124038641183096280344793397780e67 * t11 - 0.298860625089794573963071110598266798e65 * t4 - 0.953307260026526464500726175652499120e65 * e - 0.188420489277721539668499474012215890e64) * t59 * t32 * t24 * t20 * t68 + 0.976562500000000000000000000000000000e144 * t1 * x * (0.327670854827456931546589950237299581e36 * t74 - 0.153218270976929729870761448158310029e68 * t47 + 0.102451977227752983708404213404784271e69 * t6 - 0.293966953818646115428633197880805262e69 * t8 + 0.469286087348627556919990490092858536e69 * t5 - 0.450250002597466793402649526105220127e69 * t11 + 0.259681757141567599953643051067567930e69 * t4 - 0.833821601883903467617061775658684625e68 * e + 0.115011219852951041547580246881963353e68) * t42 * t88 * t29 / (0.255441929284123241418382354353987756e36 * t5 - 0.983088638268525105505986918441471456e36 * t11 + 0.141876398718980183866091410747032721e37 * t4 - 0.909978224428198183648327072224018993e36 * e + 0.218861390223736380851262018786895901e36) - 0.122070312500000000000000000000000000e114 * t1 * (0.53984103956688868539305144294553e32 * t99 - 0.624148617591256394352820626404909136e62 * t74 + 0.359355919006056505169696587589985356e63 * t47 - 0.845916959025179842639304341637757536e63 * t6 + 0.102430925519158418318550975034493867e64 * t8 - 0.640923212405337909774045602555959064e63 * t5 + 0.156724714601482223569039225330326971e63 * t11 + 0.221689362729940339299300579095155320e62 * t4 - 0.130362266158971774643946788970235000e62 * e - 0.267565264513054973086680985430145724e60) * t68 * t20 / t112 * t37 + 0.390625000000000000000000000000000000e213 * x * t129 * t29 * t37 * t88 * t42 * t24 - 0.390625000000000000000000000000000000e-2 * t149 * t29 * t24 * t59 / (0.156250000000000000000000000000000000e36 * t4 - 0.310943984047883278635783285187416432e36 * e + 0.154697857944912465366844620298366343e36) / (0.178188990558726358250257304829497180e36 * t8 - 0.871964280972301261872800437939881800e36 * t5 + 0.170677594593805278068329549404876181e37 * t11 - 0.167041482845171248138277339201476462e37 * t4 + 0.817414173709194372538823533932760785e36 * e - 0.159999999999999999999999999999999992e36));
}
double QQH1_ke (double e, double x)
{
  double t1, t103, t11, t116, t122, t133, t153, t2, t24, t29, t31, t32, t37, t4, t42, t47, t5, t58, t59, t6, t65, t72, t78, t8, t90, t92;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t24 = 0.1e1 / (0.312500000000000000000000000000000000e36 * t4 - 0.554261802481296192911811123614630350e36 * e + 0.245764916551852315671780916982386283e36);
  t29 = 0.1e1 / (0.500000000000000000000000000000000000e36 * e - 0.478313955950124144801090431097121357e36);
  t31 = 0.216575012046604742663990359406396330e36 * e - 0.199999999999999999999999999999999999e36;
  t32 = 0.1e1 / t31;
  t37 = 0.1e1 / (0.541437530116511856659975898515990825e36 * t4 - 0.980157026197438325714200285478408599e36 * e + 0.443409441985036954329448898891704280e36);
  t42 = 0.1e1 / (0.541437530116511856659975898515990825e36 * t11 - 0.153590850771957306891436653884787845e37 * t4 + 0.145211740822998700009533222488419657e37 * e - 0.457568480913314601568680790710883333e36);
  t47 = t5 * t11;
  t58 = 0.250000000000000000000000000000000000e36 * e - 0.221704720992518477164724449445852141e36;
  t59 = 0.1e1 / t58;
  t65 = 0.1e1 / (0.469047358429869894055932501514259593e36 * t4 - 0.866300048186418970655961437625585316e36 * e + 0.399999999999999999999999999999999996e36);
  t72 = 0.1e1 / (0.586309198037337367569915626892824491e36 * t8 - 0.276551429177212530927531229980555432e37 * t5 + 0.521738517770740837150924637512988468e37 * t11 - 0.492116063056246720213806897062267528e37 * t4 + 0.232070360722479116746922340001955575e37 * e - 0.437722780447472761702524037573791801e36);
  t78 = t5 * t5;
  t90 = t31 * t31;
  t92 = 0.1e1 / t90 / t31;
  t103 = t78 * e;
  t116 = t58 * t58;
  t122 = t78 * t4;
  t133 = 0.148898992033079848262307815545767496e36 * t122 - 0.136243351128648936456327575295466450e37 * t103 + 0.560934665474421405301814432391492190e37 * t78 - 0.136842550120420502189121059587604240e38 * t47 + 0.219055150059904510713891786357775647e38 * t6 - 0.240428519213544155449523768043646850e38 * t8 + 0.183240454036109727392704499473764397e38 * t5 - 0.957591303298951102753957684524669805e37 * t11 + 0.328406150924954349073276389885621766e37 * t4 - 0.667472435201749530547842471406390905e36 * e + 0.61058347248187116509890124551204253e35;
  t153 = 0.62972244425115803342566015984261743e35 * t78 * t11 - 0.111312006378887701334693258463378829e68 * t122 + 0.889879640335499838352319397953557330e68 * t103 - 0.307005652725472355147594768744165010e69 * t78 + 0.594040862299370798831266990175121610e69 * t47 - 0.697803419231693428497559390022902290e69 * t6 + 0.497032243981596989023674530143299050e69 * t8 - 0.194069504993611656296070797486612703e69 * t5 + 0.236653489310690062223137760499170400e68 * t11 + 0.878731009038949748989547661807841790e67 * t4 - 0.245383610383807345392753087127845045e67 * e - 0.501156469893622391506043640846245755e65;
  return(0.156250000000000000000000000000000000e145 * t2 * x * (0.74107030605945005332402639566047631e35 * t6 - 0.728841893739751016941937994134555910e67 * t8 - 0.102145007268838953628302725762916348e68 * t5 + 0.695372339008046313053873682702828940e68 * t11 - 0.760790071995290224735282330795140280e68 * t4 + 0.213161103328099721999160167593037652e68 * e + 0.275224857004605872251933301124699582e67) / (0.117261839607467473513983125378564898e36 * t4 - 0.216575012046604742663990359406396329e36 * e + 0.99999999999999999999999999999999999e35) * t24 * t29 * t32 * t37 * t42 - 0.156250000000000000000000000000000000e109 * t2 * (0.611839204594581505203576539966246449e36 * t47 - 0.292576210420264959926782552598091187e69 * t6 + 0.108165090435101161433864285182409867e70 * t8 - 0.142500441466002290804502960761812676e70 * t5 + 0.716259430375427679294000018721468098e69 * t11 - 0.219440898561459958336470868589223137e68 * t4 - 0.572803065347548441229075347316199722e68 * e - 0.109773423896387384604825416059609676e67) * t59 * t32 * t24 * t65 * t72 + 0.312500000000000000000000000000000000e107 * t1 * x * (0.493557422374606459228050558765690745e73 * t78 - 0.654221158394994742804265917682538506e106 * t47 + 0.381166517982181941267099052046107983e107 * t6 - 0.946481976077158906027976825184012926e107 * t8 + 0.130362731086094039469073554759882690e108 * t5 - 0.108268344510870790937515529466320614e108 * t11 + 0.547531450607499743104282561930669087e107 * t4 - 0.158154177307982952780628505340676239e107 * e + 0.204164588207982892836123417241377715e106) * t42 * t92 * t29 / (0.338398456322819910412484936572494266e36 * t5 - 0.121279442412019686071412553527201613e37 * t11 + 0.162979080016469765807641491874541513e37 * t4 - 0.973306252605954939670269526221825559e36 * e + 0.217948969015512015375695370210048360e36) - 0.312500000000000000000000000000000000e71 * t1 * (0.194178301352802223695581751844190034e74 * t103 - 0.461641992991528379021327330426611410e106 * t78 + 0.272282138288192458232609784693200871e107 * t47 - 0.653916868535586961900453324156877017e107 * t6 + 0.805778040148976331979598819210335744e107 * t8 - 0.513563909621079633816197692548425792e107 * t5 + 0.130627802639161417137626592722501337e107 * t11 + 0.151678725576331754370812869524380937e106 * t4 - 0.100133959516179653167261910036532176e106 * e - 0.197476310975815289503582926168476488e104) * t72 * t65 / t116 * t37 + 0.312500000000000000000000000000000000e214 * x * t133 * t29 * t37 * t92 * t42 * t24 - 0.312500000000000000000000000000000000e107 * t153 * t29 * t24 * t59 / (0.250000000000000000000000000000000000e36 * t4 - 0.478313955950124144801090431097121357e36 * e + 0.228784240456657300784340395355441668e36) / (0.238238387252927757219692504873227990e36 * t8 - 0.110002712225019420183069106969423778e37 * t5 + 0.203167874604754409800901691590309938e37 * t11 - 0.187618943371947957622373000605703834e37 * t4 + 0.866300048186418970655961437625585305e36 * e - 0.159999999999999999999999999999999996e36));
}
double QQH2_ke (double e, double x)
{
  double t1, t103, t11, t116, t122, t133, t153, t2, t24, t29, t31, t32, t37, t4, t42, t47, t5, t58, t59, t6, t65, t72, t78, t8, t90, t92;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t24 = 0.1e1 / (0.312500000000000000000000000000000000e36 * t4 - 0.554261802481296192911811123614630350e36 * e + 0.245764916551852315671780916982386283e36);
  t29 = 0.1e1 / (0.500000000000000000000000000000000000e36 * e - 0.398352729996437512605651024591095913e36);
  t31 = 0.1181534310178040506653975229757595e34 * e - 0.995024875621890547263681592039801e33;
  t32 = 0.1e1 / t31;
  t37 = 0.1e1 / (0.59076715508902025332698761487879750e35 * t4 - 0.102141590697316578349616012892853559e36 * e + 0.44120342486073328788999890436985501e35);
  t42 = 0.1e1 / (0.295383577544510126663493807439398750e36 * t11 - 0.719423636949352657606114514516660707e36 * t4 + 0.583862526516499903017733431314689438e36 * e - 0.157895420393646414767914118868413211e36);
  t47 = t5 * t11;
  t58 = 0.250000000000000000000000000000000000e36 * e - 0.221704720992518477164724449445852141e36;
  t59 = 0.1e1 / t58;
  t65 = 0.1e1 / (0.698011663063949017197304242853533575e36 * t4 - 0.117565603002790100164574649727123881e37 * e + 0.495037251553179376748100294547164679e36);
  t72 = 0.1e1 / (0.349005831531974508598652121426766788e36 * t8 - 0.142199256966658329431464088189399873e37 * t5 + 0.231707944829603147806722697989370262e37 * t11 - 0.188744388397493633273030505162430242e37 * t4 + 0.768594532465884656036310629882320230e36 * e - 0.125170292075112882652205190383237749e36);
  t78 = t5 * t5;
  t90 = t31 * t31;
  t92 = 0.1e1 / t90 / t31;
  t103 = t78 * e;
  t116 = t58 * t58;
  t122 = t78 * t4;
  t133 = 0.983673007955003757650327915027605395e36 * t122 - 0.813410545974343780923741307505868220e37 * t103 + 0.302822167027002167213568543583107188e38 * t78 - 0.668136502236541300720952541259754760e38 * t47 + 0.967173044862230701091346781175161105e38 * t6 - 0.959537964386546173942624383055945610e38 * t8 + 0.660667542103798221852266300309498670e38 * t5 - 0.311765549269867544494936888138338530e38 * t11 + 0.965468622405621172645826877848370225e37 * t4 - 0.177353055415031226917366137336033800e37 * e + 0.147002995080605360383066907263893681e36;
  t153 = 0.364612353606360555720369827337732146e156 * t78 * t11 - 0.675352045057338920095628721884643410e187 * t122 + 0.664834444457671268042287036544246175e188 * t103 - 0.263012570629197760571994814954255323e189 * t78 + 0.559522497181815227652075102370712075e189 * t47 - 0.703736811469004211138847440201372570e189 * t6 + 0.528564642445946902704242198585040545e189 * t8 - 0.217738662703993014980659007197498273e189 * t5 + 0.312577368425560619396529844834124152e188 * t11 + 0.794583877741799100307467583054481610e187 * t4 - 0.248543440408247511249596757726061816e187 * e - 0.471628905764745968584061873766408032e185;
  return(0.285753125495948924039824162114335430e143 * t2 * x * (0.15315197376732811503451664212020817e35 * t6 - 0.113120092167666218323300214232286892e67 * t8 + 0.243319065472368634167334147962491424e67 * t5 - 0.587717197959642309626481774949184988e66 * t11 - 0.140105362670748907994198802131401536e67 * t4 + 0.642691592639916305511113861513712005e66 * e + 0.581118475633464498471941859732926745e65) / (0.349005831531974508598652121426766788e36 * t4 - 0.587828015013950500822873248635619405e36 * e + 0.247518625776589688374050147273582340e36) * t24 * t29 * t32 * t37 * t42 - 0.142876562747974462019912081057167715e108 * t2 * (0.195985365761707113596788899636229877e36 * t47 - 0.227355681111035600784100283063508462e68 * t6 + 0.843309608826037746482453398230719862e68 * t8 - 0.110355084684940496516276313796872281e69 * t5 + 0.548533700094057801348365621086873962e68 * t11 - 0.200053494656042561318118281220274660e67 * t4 - 0.400606426852219874300615834189793516e67 * e - 0.729850108185235411324779585503884344e65) * t59 * t32 * t24 * t65 * t72 + 0.285753125495948924039824162114335430e96 * t1 * x * (0.247919063507216253937457695247698938e78 * t78 - 0.416809981637033871779156489863235611e110 * t47 + 0.219289469740655982670217348051632054e111 * t6 - 0.472711832111143663187098848611120576e111 * t8 + 0.540108121049581542849200748867170235e111 * t5 - 0.354853436765339889823601203343877693e111 * t11 + 0.137913154442551493183595034025302372e111 * t4 - 0.320469972264364712225227698422940504e110 * e + 0.398398033833447571475160550211844073e109) * t42 * t92 * t29 / (0.184614735965318829164683629649624219e36 * t5 - 0.646632139155502126007749623358064302e36 * t11 + 0.849197732523022230316609345105409737e36 * t4 - 0.495570400666225796434560698384690607e36 * e + 0.108432322893289559888405656820919583e36) - 0.571506250991897848079648324228670861e109 * t1 * (0.154943837429176408327176466524337883e36 * t103 - 0.592032078780604548792066143742768650e67 * t78 + 0.402726003443990984148024015713527986e68 * t47 - 0.105880485176160583972851924148336104e69 * t6 + 0.138462533614174993334933979465029995e69 * t8 - 0.921640278298834100474516441756524056e68 * t5 + 0.248977437446401777519274704918563542e68 * t11 + 0.202953790273695024449310059121331050e67 * t4 - 0.166668700544159014708006970124565369e67 * e - 0.308111416691689902740778969837416966e65) * t72 * t65 / t116 * t37 + 0.457205000793518278463718659382936688e205 * x * t133 * t29 * t37 * t92 * t42 * t24 - 0.228602500396759139231859329691468344e-11 * t153 * t29 * t24 * t59 / (0.250000000000000000000000000000000000e36 * t4 - 0.398352729996437512605651024591095913e36 * e + 0.158684897495614646841753689462755277e36) / (0.230266991811906562488422700116218307e37 * t8 - 0.969592600543880375495127053020503061e37 * t5 + 0.163307785216106499948354611726592911e38 * t11 - 0.137529064770245259866318612201936006e38 * t4 + 0.579098039678541527478444759086791152e37 * e - 0.975370668359505660722599806683598293e36));
}
double QQH3_ke (double e, double x)
{
  double t1, t106, t11, t119, t125, t136, t156, t2, t21, t24, t29, t31, t32, t37, t4, t42, t47, t5, t58, t59, t6, t68, t75, t8, t81, t93, t95;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t21 = 0.250000000000000000000000000000000000e36 * t4;
  t24 = 0.1e1 / (t21 - 0.398352729996437512605651024591095913e36 * e + 0.158684897495614646841753689462755277e36);
  t29 = 0.1e1 / (0.125000000000000000000000000000000000e36 * e - 0.87224330137541537412355338902241397e35);
  t31 = 0.133966511934320886249230757192827098e36 * e - 0.99999999999999999999999999999999999e35;
  t32 = 0.1e1 / t31;
  t37 = 0.1e1 / (0.669832559671604431246153785964135490e36 * t4 - 0.103365925757137051723369612796234284e37 * e + 0.398352729996437512605651024591095909e36);
  t42 = 0.1e1 / (0.209322674897376384764423058113792341e36 * t11 - 0.448378481608352583032812749692161088e36 * t4 + 0.319983669833396551295513329064479757e36 * e - 0.76080837679428369148429308290962260e35);
  t47 = t5 * t11;
  t58 = 0.500000000000000000000000000000000000e36 * e - 0.398352729996437512605651024591095913e36;
  t59 = 0.1e1 / t58;
  t68 = 0.1e1 / (0.179470263198485404434301604315704009e36 * t4 - 0.267933023868641772498461514385654193e36 * e + 0.99999999999999999999999999999999998e35);
  t75 = 0.1e1 / (0.350527857809541805535745320929109393e36 * t8 - 0.125709606945107357144296668855364147e37 * t5 + 0.180282860463213174651922854374182690e37 * t11 - 0.129238364439903369413179018671903591e37 * t4 + 0.463106178024119972700356411571020570e36 * e - 0.66361001028911696640128477473880784e35);
  t81 = t5 * t5;
  t93 = t31 * t31;
  t95 = 0.1e1 / t93 / t31;
  t106 = t81 * e;
  t119 = t58 * t58;
  t125 = t81 * t4;
  t136 = 0.539375557942991739201877211935130607e214 * t125 - 0.388662344880936249093105964621255054e215 * t106 + 0.126625974555357556283064270646371119e216 * t81 - 0.245277039975061658128001503073574972e216 * t47 + 0.312181737889782211677091664197004709e216 * t6 - 0.272178187252787726130125793412878740e216 * t8 + 0.164287576988711648298110890980125394e216 * t5 - 0.677255410848779069022743502207626659e215 * t11 + 0.182747054648328606668816766616126309e215 * t4 - 0.293191985791347346738894143932680168e214 * e + 0.215186656478257708724126858826732954e213;
  t156 = 0.279983494413102835776284088052057359e36 * t81 * t11 + 0.723305249074495806252678992085131170e65 * t125 - 0.106512538839927735789704815532578209e66 * t106 - 0.843633092799483382483364689347430960e66 * t81 + 0.334516817822379209122610769881515341e67 * t47 - 0.537832811793507893559345091962160784e67 * t6 + 0.458637357399210508667164727844992196e67 * t8 - 0.205451467042308262955105357818919931e67 * t5 + 0.345177606413770539468471399929369662e66 * t11 + 0.544527368044046595809951945833690932e65 * t4 - 0.201698376285795949625156023652387984e65 * e - 0.347085135071314996992047307868807871e63;
  return(0.122070312500000000000000000000000000e147 * t2 * x * (0.34116532977354627203849902895557013e35 * t6 - 0.426663931605505950162725571313373362e65 * t8 + 0.113166889802756015664177064469670388e66 * t5 - 0.662217035348889810502451815195852380e65 * t11 - 0.176160718666677098707213460496022668e65 * t4 + 0.148931594502597449216587830278746001e65 * e + 0.958181719215580867025620560584881978e63) / (0.448675657996213511085754010789260022e36 * t4 - 0.669832559671604431246153785964135482e36 * e + 0.249999999999999999999999999999999995e36) * t24 * t29 * t32 * t37 * t42 - 0.488281250000000000000000000000000000e111 * t2 * (0.152253594010803127467565795237024679e36 * t47 - 0.234971181970550105705761677531395188e66 * t6 + 0.918632276253821974400279419374855265e66 * t8 - 0.121500034431186998061866601722840782e67 * t5 + 0.597096344916783765162480267086290765e66 * t11 - 0.253728324884482460415294820318842830e65 * t4 - 0.382609547177193423031922693915736332e65 * e - 0.650478596180736313239388634069842935e63) * t59 * t32 / (t21 - 0.398352729996437512605651024591095912e36 * e + 0.158684897495614646841753689462755277e36) * t68 * t75 + 0.244140625000000000000000000000000000e146 * t1 * x * (0.134873494023369027490748574859986849e36 * t81 - 0.224843266586029637855999789618353980e66 * t47 + 0.120579747590957479577876604320438108e67 * t6 - 0.247532041626850070515142051079441114e67 * t8 + 0.250924600816880073949567834127703249e67 * t5 - 0.132589664186464930500397706263989043e67 * t11 + 0.357629715548600602902955570996939116e66 * t4 - 0.539810817818837331236599364211584085e65 * e + 0.754955685498117895547217272000111435e64) * t42 * t95 * t29 / (0.418645349794752769528846116227584680e36 * t5 - 0.131311110794631971981318023992939283e37 * t11 + 0.154410370177367331489901789642200690e37 * t4 - 0.806777527071801788534408991998518160e36 * e + 0.158031405316457361625991934271037131e36) - 0.976562500000000000000000000000000000e113 * t1 * (0.11855969975031575264908362498522181e35 * t106 + 0.774278728371421214475284960198101380e63 * t81 + 0.181152258149746763869691581253042566e65 * t47 - 0.790074153713721539921540849856080697e65 * t6 + 0.124971526933261190842003353223205087e66 * t8 - 0.909693154190865871055947881888588909e65 * t5 + 0.264010587744638505001045586335766320e65 * t11 + 0.120862052153332630042591649591379924e64 * t4 - 0.146660105506741603666962657984816320e64 * e - 0.248570189313843549006703297685845936e62) * t75 * t68 / t119 * t37 + 0.1953125000000000000000000000000000e34 * x * t136 * t29 * t37 * t95 * t42 * t24 - 0.390625000000000000000000000000000000e110 * t156 * t29 * t24 * t59 / (0.195312500000000000000000000000000000e36 * t4 - 0.272576031679817304413610434069504365e36 * e + 0.95101047099285461435536635363702826e35) / (0.431500446354393391361501769548083799e36 * t8 - 0.161047876862668122551837030345854709e37 * t5 + 0.240430051566356054834789011715569005e37 * t11 - 0.179470263198485404434301604315704004e37 * t4 + 0.669832559671604431246153785964135463e36 * e - 0.99999999999999999999999999999999995e35));
}
double QQH4_ke (double e, double x)
{
  double t1, t11, t112, t118, t129, t149, t2, t20, t24, t29, t31, t32, t37, t4, t42, t47, t5, t58, t59, t6, t68, t74, t8, t86, t88, t99;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t20 = 0.1e1 / (0.238904542857978111729319601013115978e36 * t4 - 0.309130744416001503426444361600256093e36 * e + 0.100000000000000000000000000000000002e36);
  t24 = 0.1e1 / (0.195312500000000000000000000000000000e36 * t4 - 0.272576031679817304413610434069504365e36 * e + 0.95101047099285461435536635363702826e35);
  t29 = 0.1e1 / (0.500000000000000000000000000000000000e36 * e - 0.299667030285396464904836487084422473e36);
  t31 = 0.154565372208000751713222180800128045e36 * e - 0.100000000000000000000000000000000001e36;
  t32 = 0.1e1 / t31;
  t37 = 0.1e1 / (0.193206715260000939641527726000160056e37 * t4 - 0.259818610533026451318835203886032284e37 * e + 0.872243301375415374123553389022413979e36);
  t42 = 0.1e1 / (0.386413430520001879283055452000320112e36 * t11 - 0.713181460745285383285323781123765720e36 * t4 + 0.438467243110188073722523846046062874e36 * e - 0.89800329040068722102514307620622049e35);
  t47 = t5 * t11;
  t58 = 0.125000000000000000000000000000000000e36 * e - 0.87224330137541537412355338902241397e35;
  t59 = 0.1e1 / t58;
  t68 = 0.1e1 / (0.298630678572472639661649501266394973e36 * t8 - 0.923352042119555813245352877061217334e36 * t5 + 0.114157778948513259160142872203465529e37 * t11 - 0.705440596516365646514824074562463980e36 * t4 + 0.217888188720475686856876773432878071e36 * e - 0.26910197922088841403682559969875436e35);
  t74 = t5 * t5;
  t86 = t31 * t31;
  t88 = 0.1e1 / t86 / t31;
  t99 = t74 * e;
  t112 = t58 * t58;
  t118 = t74 * t4;
  t129 = 0.861511469342863190414521017764590372e214 * t118 - 0.521022876039717072972489949260601100e215 * t99 + 0.143863551820274004580850598505913711e216 * t74 - 0.238753691721793610814972180695668958e216 * t47 + 0.262601986627062913933128495589739514e216 * t6 - 0.198444571292207581611253209229439946e216 * t8 + 0.103331306539996931963953119116986609e216 * t5 - 0.363026352851606859539755158980408179e215 * t11 + 0.822902465954053103760277225152477450e214 * t4 - 0.110917494433405346766893242729150864e214 * e + 0.719814312519221329117580290079388905e212;
  t149 = 0.395376347524173450627198521458595472e148 * t74 * t11 - 0.175589914345450439424120742722350774e178 * t118 + 0.771642533350523288892936182240633086e178 * t99 - 0.109761877076739548231563943623225784e179 * t74 + 0.337783324780084296936371201588680669e177 * t47 + 0.159142131609283862468516434427296389e179 * t6 - 0.188857053464795412029909817708829748e179 * t8 + 0.955509135948953735661652459599845212e178 * t5 - 0.182113749002921427155517211924891285e178 * t11 - 0.161910981680517873943460494198194502e177 * t4 + 0.765985035710423170812743101539383635e176 * e + 0.117133302630576438411176507989168042e175;
  return(-0.122070312500000000000000000000000000e147 * t2 * x * (0.373075017792718232406230522214144111e36 * t6 + 0.164819408583267429883883986133246779e66 * t8 - 0.568662693291349004903737444681436112e66 * t5 + 0.392619242919396134427484829418966489e66 * t11 + 0.224492589680124257223937337198131951e65 * t4 - 0.497890146625441994392774335577690819e65 * e - 0.235313767881583587696972662889979868e64) * t20 * t24 * t29 * t32 * t37 * t42 + 0.122070312500000000000000000000000000e111 * t2 * (0.220316112347369174542656320709722699e36 * t47 + 0.101357630190706570643351171420433139e66 * t6 - 0.491894831017402159753301699084554030e66 * t8 + 0.684903307758960629402745162062380270e66 * t5 - 0.333060568398332006278445058574339139e66 * t11 + 0.157471775031267174494553562105468877e65 * t4 + 0.179687920301217788586562949073992210e65 * e + 0.280244908447734365802490618089531758e63) * t59 * t32 * t24 * t20 * t68 - 0.488281250000000000000000000000000000e105 * t1 * x * (0.103490371026536677767466275213888596e78 * t74 + 0.349605536773786881784692157928386653e107 * t47 - 0.264423338075633765974604283942457059e108 * t6 + 0.586034962278152813702133832796831994e108 * t8 - 0.570270853597620349499507261025473735e108 * t5 + 0.258431483126904417601948766536822571e108 * t11 - 0.469263870126466814973445818427342326e107 * t4 + 0.243002836719314124526383961307644001e106 * e - 0.694151094794044029246022879545730329e105) * t42 * t88 * t29 / (0.301885492593751468189887071875250088e36 * t5 - 0.827274736873561490557040018215776325e36 * t11 + 0.849843909787875983610654585557865822e36 * t4 - 0.387874269539148956754491423454056933e36 * e + 0.66361001028911696640128477473880786e35) + 0.390625000000000000000000000000000000e70 * t1 * (0.151033796654195217831138173646275348e78 * t99 - 0.772098623548663690071989478108847645e107 * t74 + 0.135454916826359269109645307083822378e108 * t47 + 0.127379617563789673678389842194477160e108 * t6 - 0.468662250450980200829335805991498858e108 * t8 + 0.405291303603439978157513715973487059e108 * t5 - 0.126121221789936883939887736571769394e108 * t11 - 0.193289872453981123380870286054111663e106 * t4 + 0.555672092074140666597743195826954417e106 * e + 0.847013533799272757319127607398925268e104) * t68 * t20 / t112 * t37 + 0.31250000000000000000000000000000000e35 * x * t129 * t29 * t37 * t88 * t42 * t24 + 0.250000000000000000000000000000000000e0 * t149 * t29 * t24 * t59 / (0.250000000000000000000000000000000000e37 * t4 - 0.299667030285396464904836487084422473e37 * e + 0.898003290400687221025143076206220481e36) / (0.882187744607091906984469522191080804e36 * t8 - 0.285376902990897501967811473549656234e37 * t5 + 0.369263695890256545094630000789690386e37 * t11 - 0.238904542857978111729319601013115985e37 * t4 + 0.772826861040003758566110904000640256e36 * e - 0.100000000000000000000000000000000005e36));
}
double QQH5_ke (double e, double x)
{
  double t1, t103, t11, t116, t122, t133, t153, t2, t24, t29, t31, t32, t37, t4, t42, t47, t5, t58, t59, t6, t65, t72, t78, t8, t90, t92;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t24 = 0.1e1 / (0.500000000000000000000000000000000000e36 * t4 - 0.507378750740602083160994650247236515e36 * e + 0.128716598351547009475867237190332471e36);
  t29 = 0.1e1 / (0.500000000000000000000000000000000000e36 * e - 0.299667030285396464904836487084422473e36);
  t31 = 0.36273496716843902928279785064538991e35 * e - 0.20000000000000000000000000000000000e35;
  t32 = 0.1e1 / t31;
  t37 = 0.1e1 / (0.362734967168439029282797850645389910e36 * t4 - 0.384044014491855906449000316080208371e36 * e + 0.101475750148120416632198930049447303e36);
  t42 = 0.1e1 / (0.906837417921097573206994626613474775e36 * t11 - 0.158699710392036910968503486785564232e37 * t4 + 0.925071254631436428233400130926772549e36 * e - 0.179600658080137444205028615241244096e36);
  t47 = t5 * t11;
  t58 = 0.100000000000000000000000000000000000e37 * e - 0.507378750740602083160994650247236513e36;
  t59 = 0.1e1 / t58;
  t65 = 0.1e1 / (0.131576656406688540485187130628784107e36 * t4 - 0.145093986867375611713119140258155964e36 * e + 0.40000000000000000000000000000000000e35);
  t72 = 0.1e1 / (0.164470820508360675606483913285980134e37 * t8 - 0.477086377686277062126633935814333553e37 * t5 + 0.553333536765801191158068998209977562e37 * t11 - 0.320749964180311712564814477195901652e37 * t4 + 0.929252764631016015552591257712865751e36 * e - 0.107640791688355365614730239879501742e36);
  t78 = t5 * t5;
  t90 = t31 * t31;
  t92 = 0.1e1 / t90 / t31;
  t103 = t78 * e;
  t116 = t58 * t58;
  t122 = t78 * t4;
  t133 = 0.784977354347916599742106989369631000e211 * t122 - 0.384908911858807812493364353233258897e212 * t103 + 0.869672870685040296622504096255765050e212 * t78 - 0.120556978938529814065339739051498628e213 * t47 + 0.113460533574440099131138099539166460e213 * t6 - 0.745006746337256834575952687119869215e212 * t8 + 0.335889971573886586289426587173970416e212 * t5 - 0.994712052155327508072996654108945430e211 * t11 + 0.181633185383642324282747246696350888e211 * t4 - 0.193092692895593775684566759717636544e210 * e + 0.115217793333724045944266687391096556e209;
  t153 = 0.190725842038255280284633818414640759e36 * t78 * t11 - 0.326115168599859105082399072410692784e67 * t122 + 0.154778491249029585540923583343702456e68 * t103 - 0.271906421476435730657382677988986394e68 * t78 + 0.190771496425442290455306162837431317e68 * t47 + 0.184850513233441910345993463783403686e67 * t6 - 0.113044407822120757035938074786597445e68 * t8 + 0.678243208541434589245969786157313940e67 * t5 - 0.140542457923750547647111436067434041e67 * t11 - 0.613449472918819981614023219696097450e65 * t4 + 0.421299665596223132870486416875768072e65 * e + 0.563208916333676646433370475034106100e63;
  return(-0.125000000000000000000000000000000000e146 * t2 * x * (0.15590114676277737493934397329271971e35 * t6 + 0.604635852326679998602061722538187475e65 * t8 - 0.469366079242827028794636143626543834e66 * t5 + 0.352736992185046433909157320966131531e66 * t11 - 0.409600992582036562413017888878475400e64 * t4 - 0.296742584094894326810243139533049477e65 * e - 0.106147188621829482539838869653840077e64) / (0.205588525635450844508104891607475167e36 * t4 - 0.226709354480274393301748656653368694e36 * e + 0.62500000000000000000000000000000000e35) * t24 * t29 * t32 * t37 * t42 + 0.312500000000000000000000000000000000e111 * t2 * (0.30272559480245838872864674707828459e35 * t47 + 0.609585610623536861800717912961845088e65 * t6 - 0.105754250009552270719297857687586695e67 * t8 + 0.168289662940909927048308979543365524e67 * t5 - 0.809332539194521488223883656860115456e66 * t11 + 0.403429453913589370330432022161098868e65 * t4 + 0.353071755021417218868276879914636284e65 * e + 0.497424059630993326812729296354730440e63) * t59 * t32 * t24 * t65 * t72 - 0.125000000000000000000000000000000000e145 * t1 * x * (0.114262801503172478392206757973133947e36 * t78 - 0.458973786156155367329477134255972332e66 * t47 - 0.288222225528571032029455226117179358e67 * t6 + 0.937478212356553573950606729919707060e67 * t8 - 0.943011105444954353572315891171257530e67 * t5 + 0.385873575303275312916623689251366795e67 * t11 - 0.484825085007855911052846610017617955e66 * t4 - 0.227684908807400447643361152970781459e65 * e - 0.488184922696910540471728191164018750e64) * t42 * t92 * t29 / (0.906837417921097573206994626613474775e36 * t5 - 0.188033010868891929836750237060156278e37 * t11 + 0.146141829226712649114292816325268572e37 * t4 - 0.504597392516413597881935194848777260e36 * e + 0.65308066871187763110656439194453746e35) + 0.625000000000000000000000000000000000e71 * t1 * (0.137342982107247047908295207710317106e77 * t103 - 0.356063420341553599773531894772627621e108 * t78 + 0.831921328681731359607277209981844988e108 * t47 - 0.392579118267225689645205588722088148e108 * t6 - 0.487719585476435473602314858296566590e108 * t8 + 0.587328564960717546509344485294458720e108 * t5 - 0.192747937551171754645053771176333732e108 * t11 + 0.130602791190332927666012901043674038e106 * t4 + 0.648841299133916720331361490336023313e106 * e + 0.875231310529347544808455833064733436e104) * t72 * t65 / t116 * t37 + 0.500000000000000000000000000000000000e36 * x * t133 * t29 * t37 * t92 * t42 * t24 + 0.500000000000000000000000000000000000e112 * t153 * t29 * t24 * t59 / (0.250000000000000000000000000000000000e37 * t4 - 0.299667030285396464904836487084422473e37 * e + 0.898003290400687221025143076206220481e36) / (0.627981883478333279793685591495709933e36 * t8 - 0.173124165111637725465186493267128721e37 * t5 + 0.190909816567252602374365441236509171e37 * t11 - 0.105261325125350832388149704503027286e37 * t4 + 0.290187973734751223426238280516311928e36 * e - 0.32000000000000000000000000000000000e35));
}
double QQH6_ke (double e, double x)
{
  double t1, t100, t11, t113, t116, t127, t144, t2, t20, t22, t23, t24, t29, t34, t39, t4, t41, t42, t47, t5, t58, t6, t65, t69, t75, t8, t87;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t20 = 0.1e1 / (0.92836119975415459285206022660673343e35 * t4 - 0.86179403560440337633361824090712660e35 * e + 0.20000000000000000000000000000000000e35);
  t22 = 0.100000000000000000000000000000000000e37 * e - 0.507378750740602083160994650247236513e36;
  t23 = t22 * t22;
  t24 = 0.1e1 / t23;
  t29 = 0.1e1 / (0.50000000000000000000000000000000000e35 * e - 0.21254801747114023046234045005356217e35);
  t34 = 0.1e1 / (0.269310636126376055104255700283477062e37 * t11 - 0.353965767170211468316930406941563665e37 * t4 + 0.154940228616357248605152126721024482e37 * e - 0.225883298654560663246800142178258191e36);
  t39 = 0.1e1 / (0.430897017802201688166809120453563300e36 * t4 - 0.418627990590332068802136169940426372e36 * e + 0.101475750148120416632198930049447303e36);
  t41 = 0.4308970178022016881668091204535633e34 * e - 0.2000000000000000000000000000000000e34;
  t42 = 0.1e1 / t41;
  t47 = t5 * t11;
  t58 = t29 * t20;
  t65 = 0.1e1 / (0.185672239950830918570412045321346686e38 * t8 - 0.454977254581266377121806299418356633e38 * t5 + 0.445748183517097465416555689539384770e38 * t11 - 0.218249876305717372444705379493453448e38 * t4 + 0.534048246069485397672178840347598368e37 * e - 0.522464534969502104885251513555629967e36);
  t69 = 0.1e1 / (0.125000000000000000000000000000000000e37 * t4 - 0.106274008735570115231170225026781085e37 * e + 0.225883298654560663246800142178258191e36);
  t75 = t5 * t5;
  t87 = t41 * t41;
  t100 = t75 * e;
  t113 = t58 * t34;
  t116 = t75 * t4;
  t127 = 0.148548216497881168129500385310388870e111 * t116 - 0.576049126364971172581075939757982295e111 * t100 + 0.102441695968502936176048113908940682e112 * t75 - 0.114608623821109027132442925700063788e112 * t47 + 0.915381570992560028482758744474953235e111 * t6 - 0.532324735611105752161087862685590050e111 * t8 + 0.213392515685542889943954094305417866e111 * t5 - 0.536851119319743036687221906795592300e110 * t11 + 0.751404186276048537764374200634647915e109 * t4 - 0.549172515217645646543801739382837900e108 * e + 0.342112898729163012624504798495087372e107;
  t144 = 0.196668776510467402693314530553424249e36 * t75 * t11 + 0.379166438205109256252349143556741002e66 * t116 - 0.189927884469612702983285728758438734e67 * t100 + 0.345955756264613804042987528787866760e67 * t75 - 0.279419293080853707442890482836230424e67 * t47 + 0.649449793282225062212296410089814236e66 * t6 + 0.502893067555138103526479366905375764e66 * t8 - 0.383318086465243211320305467987949644e66 * t5 + 0.826012447802999307786758615174437424e65 * t11 + 0.128794025168029219953129403367587407e64 * t4 - 0.175530020385210830140441973509054589e64 * e - 0.202519464261448203634768354317412476e62;
  return(0.250000000000000000000000000000000000e180 * t2 * x * (0.124847905514050829826879892055562447e36 * t6 + 0.483356677218383917095228936784670344e65 * t8 + 0.278451725219893577588289706904753136e66 * t5 - 0.222918426998442233554741325749698076e66 * t11 + 0.863311175553249403134131010816814230e64 * t4 + 0.129874660655775346776008701598179108e65 * e + 0.359945569159784195327050609022334318e63) * t20 * t24 * t29 * t34 * t39 * t42 - 0.625000000000000000000000000000000000e110 * t2 * (0.95575114056103853719526188878377689e35 * t47 + 0.718934224451220115860211017247080500e65 * t6 + 0.168614080039166627225915745537897364e66 * t8 - 0.366612720519519549344947366679382420e66 * t5 + 0.173590516163005824243919766334301892e66 * t11 - 0.865957281524014155063238304236189775e64 * t4 - 0.589967409791855600578101648656006750e64 * e - 0.740617429526005713157630474831114030e62) * t42 * t58 * t65 * t69 + 0.625000000000000000000000000000000000e178 * t1 * x * (0.182141917957566842879018891937274013e36 * t75 + 0.287768215884258590327227739845353445e66 * t47 - 0.658439442619044299006048970022018290e65 * t6 - 0.796927957047642930632914142059272400e66 * t8 + 0.906365147762075898956382757133013450e66 * t5 - 0.344089109412319349013325234367622795e66 * t11 + 0.315155388531553670430782546826881872e65 * t4 + 0.392174113691978175321481536169111350e64 * e + 0.212926313135602492439972573762206854e63) * t39 / t87 / t41 * t24 / (0.673276590315940137760639250708692655e36 * t5 - 0.117112162688829300618848902603086374e37 * t11 + 0.763524181864291432421752606557254545e36 * t4 - 0.221132016758299975675696892633431750e36 * e + 0.24005523654434173182507769602944788e35) - 0.312500000000000000000000000000000000e78 * t1 * (0.69126246320241927366651683264432157e35 * t100 + 0.270145301684900709817356382396377856e66 * t75 - 0.623519785811780264168208573892427000e66 * t47 + 0.408901547012466675064313373241330880e66 * t6 + 0.711046115588797147659255795384007748e65 * t8 - 0.181049992354115689007867302013269612e66 * t5 + 0.608836798181912557268388961176421650e65 * t11 - 0.124712312122826316175831140537872576e64 * t4 - 0.152133089982109127962912672008127427e64 * e - 0.179204521939691225657641200079483174e62) * t65 * t113 + 0.1562500000000000000000000000000000e34 * x * t127 * t65 * t113 - 0.781250000000000000000000000000000000e111 * t144 * t69 / t22 * t29 / (0.500000000000000000000000000000000000e36 * t4 - 0.507378750740602083160994650247236515e36 * e + 0.128716598351547009475867237190332471e36) / (0.148548216497881168129500385310375808e36 * t8 - 0.344741806883589330268696029242213691e36 * t5 + 0.320022457933871415270519510833812983e36 * t11 - 0.148537791960664734856329636257077349e36 * t4 + 0.34471761424176135053344729636285064e35 * e - 0.3200000000000000000000000000000000e34));
}
double QQH7_ke (double e, double x)
{
  double t1, t103, t11, t116, t122, t133, t153, t2, t24, t29, t31, t32, t37, t4, t42, t47, t5, t58, t59, t6, t65, t72, t78, t8, t90, t92;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t24 = 0.1e1 / (0.781250000000000000000000000000000000e36 * t4 - 0.552448983595940699157918322096806110e36 * e + 0.97663961432380145827631098813351181e35);
  t29 = 0.1e1 / (0.50000000000000000000000000000000000e35 * e - 0.21254801747114023046234045005356217e35);
  t31 = 0.12908811968560376461227335448686169e35 * e - 0.5000000000000000000000000000000000e34;
  t32 = 0.1e1 / t31;
  t37 = 0.1e1 / (0.322720299214009411530683386217154224e36 * t4 - 0.239103360823396710079106481580347372e36 * e + 0.44195918687675255932633465767744489e35);
  t42 = 0.1e1 / (0.322720299214009411530683386217154225e37 * t11 - 0.399374239182523500746990200584229767e37 * t4 + 0.164591709318967928652224454228485125e37 * e - 0.225883298654560663246800142178258191e36);
  t47 = t5 * t11;
  t58 = 0.125000000000000000000000000000000000e36 * e - 0.44195918687675255932633465767744489e35;
  t59 = 0.1e1 / t58;
  t65 = 0.1e1 / (0.83318713219823810881433926697899504e35 * t4 - 0.64544059842801882306136677243430845e35 * e + 0.12500000000000000000000000000000000e35);
  t72 = 0.1e1 / (0.208296783049559527203584816744748760e37 * t8 - 0.426998559303805213372020374442390132e37 * t5 + 0.349952514077375546961428996402843317e37 * t11 - 0.143330178879379495300932480556909269e37 * t4 + 0.293365590415688130011868850601646577e36 * e - 0.24005523654434173182507769602944788e35);
  t78 = t5 * t5;
  t90 = t31 * t31;
  t92 = 0.1e1 / t90 / t31;
  t103 = t78 * e;
  t116 = t58 * t58;
  t122 = t78 * t4;
  t133 = 0.179226151205267866659533051953146886e40 * t122 - 0.542864588250335323715782979198669090e40 * t103 + 0.728923788802592846342251371877812190e40 * t78 - 0.623216793383243756757335432841465855e40 * t47 + 0.417151250191555470799325361669493912e40 * t6 - 0.224090382601308160135849747401496182e40 * t8 + 0.840301079700665532101292512788588630e39 * t5 - 0.184368690120926448222571620667860848e39 * t11 + 0.187802183719667144695287492145307861e38 * t4 - 0.623303983269446652792640584325357700e36 * e + 0.61364211032961787249997924497413219e35;
  t153 = 0.957139066306318627708339582992890800e150 * t78 * t11 - 0.580360213279136938237361204878939740e181 * t122 + 0.340291767964575988193786421194745792e182 * t103 - 0.633857830776712151749853310020532395e182 * t78 + 0.518730035898204097409709038698554590e182 * t47 - 0.159662144139841110103922931797115107e182 * t6 - 0.331015550091680051391623556503243512e181 * t8 + 0.360427955956485129504038410164299202e181 * t5 - 0.775169326100345972389550174927593420e180 * t11 + 0.654706162705903897847889796716125270e177 * t4 + 0.116345589527240339602827928824021011e179 * e + 0.114710762034019939540162559499566252e177;
  return(-0.610351562500000000000000000000000000e143 * t2 * x * (0.32082071216067603459318413297603897e35 * t6 - 0.134648288120605740050859841030618439e66 * t8 - 0.190611672579673098590853056009156076e66 * t5 + 0.160561603925276706334257489282615545e66 * t11 - 0.771131097493163440417538499864573831e64 * t4 - 0.652531811167854982818323573870540350e64 * e - 0.142537549195753705409622389993595952e63) / (0.26037097881194940900448102093093595e35 * t4 - 0.20170018700875588220667711638572139e35 * e + 0.3906250000000000000000000000000000e34) * t24 * t29 * t32 * t37 * t42 + 0.122070312500000000000000000000000000e110 * t2 * (0.30637809446026060896950532409960519e35 * t47 - 0.186530969360737189840220598666195506e66 * t6 - 0.542356913639457408811581610915034136e65 * t8 + 0.295510032747068242045162404041460536e66 * t5 - 0.136447792113393858885622933482499663e66 * t11 + 0.650767448585088275022393195491336016e64 * t4 + 0.349779427215712502441204268267772220e64 * e + 0.386600691155460283927966331125697458e62) * t59 * t32 * t24 * t65 * t72 - 0.781250000000000000000000000000000000e145 * t1 * x * (0.4553830543301496497163613724563073e34 * t78 - 0.448348245709696563012492841784522854e65 * t47 + 0.400793121005834044572865176633415051e65 * t6 + 0.341652649023752396995110692457918972e65 * t8 - 0.507347254768903430025951860252748276e65 * t5 + 0.180106882307191341533448421110805086e65 * t11 - 0.126455996588542789106517616027300236e64 * t4 - 0.215128440738144678005465934535427202e63 * e - 0.573139718448119303966876192804986398e61) * t42 * t92 * t29 / (0.126062616880472426379173197741075869e38 * t5 - 0.182543000964918019623952908101969576e38 * t11 + 0.990693064959442343316384583541641605e37 * t4 - 0.238838858839518633291494515900040427e37 * e + 0.215817424909086257161644352382309739e36) + 0.976562500000000000000000000000000000e110 * t1 * (0.172591657071059763555548297909574801e36 * t103 - 0.329809277250342290064891858272219043e67 * t78 + 0.704704745737299533156914511285099530e67 * t47 - 0.467188001158045696511489495477681770e67 * t6 + 0.142398990218536713125028480067760167e66 * t8 + 0.100450493043237910661364976124198646e67 * t5 - 0.334056338238320878156947505921554188e66 * t11 + 0.916492034564609922963606458038053345e64 * t4 + 0.606561833672886079265872538591418915e64 * e + 0.617271631652988767858400867499994175e62) * t72 * t65 / t116 * t37 + 0.488281250000000000000000000000000000e206 * x * t133 * t29 * t37 * t92 * t42 * t24 + 0.244140625000000000000000000000000000e-3 * t153 * t29 * t24 * t59 / (0.125000000000000000000000000000000000e37 * t4 - 0.106274008735570115231170225026781085e37 * e + 0.225883298654560663246800142178258191e36) / (0.358452302410535733319066103906301917e37 * t8 - 0.694200797260724306711544856444708178e37 * t5 + 0.537772801208555635212881490599066979e37 * t11 - 0.208296783049559527203584816744748760e37 * t4 + 0.403400374017511764413354232771442781e36 * e - 0.31250000000000000000000000000000000e35));
}
double QQH8_ke (double e, double x)
{
  double t1, t102, t11, t115, t121, t132, t152, t18, t2, t20, t24, t29, t30, t31, t36, t4, t41, t46, t5, t57, t58, t6, t71, t77, t8, t89, t91;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t18 = 0.62253097769108462643227841666948487e35 * e;
  t20 = 0.1e1 / (0.96886204546254426210682643755394204e35 * t4 - t18 + 0.10000000000000000000000000000000000e35);
  t24 = 0.1e1 / (0.781250000000000000000000000000000000e36 * t4 - 0.552448983595940699157918322096806110e36 * e + 0.97663961432380145827631098813351181e35);
  t29 = 0.1e1 / (0.250000000000000000000000000000000000e36 * e - 0.73147933870940821270720357584048163e35);
  t30 = t18 - 0.20000000000000000000000000000000000e35;
  t31 = 0.1e1 / t30;
  t36 = 0.1e1 / (0.778163722113855783040348020836856088e36 * t4 - 0.525133284705941548905707644365098689e36 * e + 0.88391837375350511865266931535488978e35);
  t41 = 0.1e1 / (0.389081861056927891520174010418428044e37 * t11 - 0.352684273943797969641680342364849836e37 * t4 + 0.106457202298729037646569852157865397e37 * e - 0.107012404591750628821783386246097117e36);
  t46 = t5 * t11;
  t57 = 0.125000000000000000000000000000000000e36 * e - 0.44195918687675255932633465767744489e35;
  t58 = 0.1e1 / t57;
  t71 = 0.1e1 / (0.605538778414090163816766523471213776e38 * t8 - 0.920608787293617746024945632035043200e38 * t5 + 0.559546796719640271595236691544940876e38 * t11 - 0.169956748881730330992385214861052106e38 * t4 + 0.257978773457436718835670563021598541e37 * e - 0.156554725888954777821151580159209727e36);
  t77 = t5 * t5;
  t89 = t30 * t30;
  t91 = 0.1e1 / t89 / t30;
  t102 = t77 * e;
  t115 = t57 * t57;
  t121 = t77 * t4;
  t132 = 0.114133961692507980191570191053936818e41 * t121 - 0.269559821109789444142496838005107987e41 * t102 + 0.261775237657665874782326596815957464e41 * t77 - 0.154623661093811335517547162640279876e41 * t46 + 0.842538295201204608085236905420100130e40 * t6 - 0.459960900204495326027011724823706304e40 * t8 + 0.174294684632500461855527389945915438e40 * t5 - 0.345368218603416776381793979662940345e39 * t11 + 0.246639549904851550357576228983658222e38 * t4 + 0.305698588042683440685535773341228896e36 * e + 0.58090016211792488043527534027774649e35;
  t152 = 0.69649933835993433764441320055165659e35 * t77 * t11 - 0.575390835973639752436553860833407930e67 * t121 + 0.640959830224482903786073558341828060e68 * t102 - 0.126314469069612211843606188719199324e69 * t77 + 0.101186098790892007801344879836355029e69 * t46 - 0.324819080421481686272342651774034565e68 * t6 - 0.191767143691093304990420657475736972e67 * t8 + 0.394522393991366107952485726917902555e67 * t5 - 0.817385233758360996489352143262676130e66 * t11 + 0.799915021701272761894418422675780260e64 * t4 + 0.863783701506832914163652122499666795e64 * e + 0.722318878482352249546812030399686700e62;
  return(-0.125000000000000000000000000000000000e144 * t2 * x * (0.51429341458064892482661190878690393e35 * t6 - 0.103258353071024710316543090378946046e68 * t8 - 0.653007460803126431637212485940042290e67 * t5 + 0.569746401801098066383828662106501080e67 * t11 - 0.277628897772577348956856145339080468e66 * t4 - 0.160787533543935524709967767731571246e66 * e - 0.280181202533585191336339280599898770e64) * t20 * t24 * t29 * t31 * t36 * t41 + 0.488281250000000000000000000000000000e109 * t2 * (0.30414257441524892708348074661819007e35 * t46 - 0.824381732803589442745064522664967498e67 * t6 + 0.136420740126066076771501814379320730e67 * t8 + 0.593946533013011558422876619639313438e67 * t5 - 0.263578081807171237493764526149963604e67 * t11 + 0.115755959452815077752807633403770200e66 * t4 + 0.496255581560619305024394210604199131e65 * e + 0.477946271687068081464332620273264731e63) * t58 * t31 / (0.122070312500000000000000000000000000e36 * t4 - 0.86320153686865734243424737827625955e35 * e + 0.15259993973809397785567359189586122e35) * t20 * t71 - 0.156250000000000000000000000000000000e105 * t1 * x * (0.286718890224777639748798907380085198e76 * t77 - 0.115160517531272777328084788664898249e109 * t46 + 0.116741128198950847739145639882571100e109 * t6 + 0.213192622002135853531552875588356337e108 * t8 - 0.579937826819457459495348381124696745e108 * t5 + 0.191638932777959481048639462867589570e108 * t11 - 0.106959609017921204817057930807493623e107 * t4 - 0.199837407808465355644709655055639624e106 * e - 0.327986120721697662462983023452712107e104) * t41 * t91 * t29 / (0.121588081580289966100054378255758764e38 * t5 - 0.168031227205910101049550458296140011e38 * t11 + 0.870328048164585478767467802992343645e37 * t4 - 0.200237155161140502069040200370735419e37 * e + 0.172653939927269005729315481905847791e36) + 0.781250000000000000000000000000000000e69 * t1 * (0.420756003671279738354769718377629242e77 * t102 - 0.294614037574358401454136140480925240e110 * t77 + 0.567038058670335494130381747400781175e110 * t46 - 0.350896654965295111529829384070001745e110 * t6 + 0.325730088365196356375534429112965720e109 * t8 + 0.434721298823497651060310464210780025e109 * t5 - 0.137879718044647433146557328841508396e109 * t11 + 0.409448139346968570500929600487920482e107 * t4 + 0.179172283440310601106979060784775707e107 * e + 0.156177611316068534960971886776028969e105) * t71 * t20 / t115 * t36 + 0.125000000000000000000000000000000000e209 * x * t132 * t29 * t36 * t91 * t41 * t24 + 0.100000000000000000000000000000000000e112 * t152 * t29 * t24 * t58 / (0.312500000000000000000000000000000000e37 * t4 - 0.182869834677352053176800893960120408e37 * e + 0.267531011479376572054458465615242793e36) / (0.233746353546256343432335751278462760e37 * t8 - 0.375477465255146073835847574362041090e37 * t5 + 0.241258654563832704459200915964053700e37 * t11 - 0.775089636370035409685461150043153630e36 * t4 + 0.124506195538216925286455683333896974e36 * e - 0.8000000000000000000000000000000000e34));
}
double QQH9_ke (double e, double x)
{
  double t1, t11, t112, t115, t126, t143, t18, t2, t20, t22, t23, t24, t29, t34, t39, t4, t40, t41, t46, t5, t57, t6, t64, t68, t74, t8, t86, t99;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t18 = 0.37684691531450844994989034395421961e35 * e;
  t20 = 0.1e1 / (0.71006798791030151654505863252377901e35 * t4 - t18 + 0.5000000000000000000000000000000000e34);
  t22 = 0.250000000000000000000000000000000000e36 * e - 0.73147933870940821270720357584048163e35;
  t23 = t22 * t22;
  t24 = 0.1e1 / t23;
  t29 = 0.1e1 / (0.62500000000000000000000000000000000e35 * e - 0.15080906637615917147310526458612547e35);
  t34 = 0.1e1 / (0.368014565736824658154189789017792588e37 * t11 - 0.275256035829114184446723974869175275e37 * t4 + 0.685547595541453683293795570717808995e36 * e - 0.56858436253121956939703551067914817e35);
  t39 = 0.1e1 / (0.942117288286271124874725859885549025e36 * t4 - 0.525655732408936999422240265160943596e36 * e + 0.73147933870940821270720357584048163e35);
  t40 = t18 - 0.10000000000000000000000000000000000e35;
  t41 = 0.1e1 / t40;
  t46 = t5 * t11;
  t57 = t29 * t20;
  t64 = 0.1e1 / (0.221896246221969223920330822663680941e39 * t8 - 0.312539684374039172699399271634250646e39 * t5 + 0.175985461760781984785194027616923857e39 * t11 - 0.495189334893843727500563879417215766e38 * t4 + 0.696282344864869981922070351204670192e37 * e - 0.391386814722386944552878950398024317e36);
  t68 = 0.1e1 / (0.976562500000000000000000000000000000e36 * t4 - 0.471278332425497410853453951831642095e36 * e + 0.56858436253121956939703551067914817e35);
  t74 = t5 * t5;
  t86 = t40 * t40;
  t99 = t74 * e;
  t112 = t57 * t34;
  t115 = t74 * t4;
  t126 = 0.185551673457652279607731270708945516e42 * t115 - 0.343430644697656077607605548467383438e42 * t99 + 0.230651596603212294819362003553974936e42 * t74 - 0.734098061948186240142139282018408476e41 * t46 + 0.289454166668239821180846436334158740e41 * t6 - 0.219098057695000657273675117486260982e41 * t8 + 0.937835486735962605086442053445037478e40 * t5 - 0.172543606218292861249134271930696705e40 * t11 + 0.860471311369313751424687423330272368e38 * t4 + 0.489928130686096197321654102402342308e37 * e + 0.146239256316429050240908533618876171e36;
  t143 = 0.180682923181232649031147287667492924e149 * t74 * t11 - 0.427038904213694157035614248579874785e179 * t115 - 0.320289390824707243232768119355140945e180 * t99 + 0.726858188807138111927148339294742058e180 * t74 - 0.562956744316642713310128700074159875e180 * t46 + 0.175003865528286743955761536336208149e180 * t6 + 0.100783653589956266130664142350429604e177 * t8 - 0.129048446916649590951879153852118614e179 * t5 + 0.250156280221117916425057990002588590e178 * t11 - 0.351796746984964775968585620339728312e176 * t4 - 0.185573038623787208914333666150624621e176 * e - 0.130869136648515902587277267588393358e174;
  return(0.305175781250000000000000000000000000e171 * t2 * x * (0.297974368797119607213356145768650406e43 * t6 + 0.209200660901308427170250645339611111e74 * t8 + 0.716067175194400608055183255415504310e73 * t5 - 0.632734169931999855395519627857611915e73 * t11 + 0.286793508184080225090454283829165089e72 * t4 + 0.123067690973687253763212457919848490e72 * e + 0.172511802975423020132050970121793505e70) * t20 * t24 * t29 * t34 * t39 * t41 - 0.122070312500000000000000000000000000e111 * t2 * (0.135764937801598650548943651865325387e36 * t46 + 0.124635343627066623845278631896667408e67 * t6 - 0.381797826262512933988504249803130232e66 * t8 - 0.473303619088587127744995622625908060e66 * t5 + 0.198285263630741735161648028867405442e66 * t11 - 0.778540887346753928988451124229360212e64 * t4 - 0.268495514502993733955767425262488845e64 * e - 0.223337451869197470833188889174112220e62) * t41 * t57 * t64 * t68 + 0.488281250000000000000000000000000000e182 * t1 * x * (0.19724053943601079285230199391600413e35 * t74 + 0.257268448124139843849429526715483990e66 * t46 - 0.253749172849444153464219648306620462e66 * t6 + 0.292755354623955270721952256101546862e64 * t8 + 0.638927658891747285805804015326183488e65 * t5 - 0.194309255395328888056074124484744116e65 * t11 + 0.880960200367602970160070808100586102e63 * t4 + 0.161924803146057314364291353471320899e63 * e + 0.184704889477248920468828368708879335e61) * t39 / t86 / t40 * t24 / (0.920036414342061645385474472544481470e38 * t5 - 0.910139821859178191675214905759407282e38 * t11 + 0.337431321996528733400301446101490960e38 * t4 - 0.555693261992914531565458093000195002e37 * e + 0.342990707477667367295985514610349313e36) - 0.244140625000000000000000000000000000e78 * t1 * (0.178450561641162493267874878291126469e36 * t99 + 0.383272893024134888170277315569555144e67 * t74 - 0.654443427884477403584371145393983142e67 * t46 + 0.365408300847316130349515457949729710e67 * t6 - 0.402000277586564715205764181018065648e66 * t8 - 0.284944730677588148793696508177744642e66 * t5 + 0.834405936086467286053091534904050622e65 * t11 - 0.242539140920310307770872761853165522e64 * t4 - 0.767649074413087990211504092991181992e63 * e - 0.569237505942342916363407897270538988e61) * t64 * t112 + 0.195312500000000000000000000000000000e104 * x * t126 * t64 * t112 - 0.625000000000000000000000000000000000e-1 * t143 * t68 / t22 * t29 / (0.312500000000000000000000000000000000e37 * t4 - 0.182869834677352053176800893960120408e37 * e + 0.267531011479376572054458465615242793e36) / (0.152003930896508747454653456964764872e37 * t8 - 0.201678618981993640264417122804199456e37 * t5 + 0.107034772363030722186613284207454389e37 * t11 - 0.284027195164120606618023453009511604e36 * t4 + t18 - 0.2000000000000000000000000000000000e34));
}
double QQH10_ke (double e, double x)
{
  double t1, t104, t11, t118, t121, t132, t149, t2, t20, t22, t23, t24, t29, t34, t39, t4, t41, t42, t47, t5, t6, t69, t73, t79, t8, t91;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t20 = 0.1e1 / (0.261692899074843225183782685765719199e36 * t4 - 0.114388132923578052687287151794054115e36 * e + 0.12500000000000000000000000000000000e35);
  t22 = 0.50000000000000000000000000000000000e35 * e - 0.9926108757466946691059784515905583e34;
  t23 = t22 * t22;
  t24 = 0.1e1 / t23;
  t29 = 0.1e1 / (0.62500000000000000000000000000000000e35 * e - 0.15080906637615917147310526458612547e35);
  t34 = 0.1e1 / (0.446828644232726768309715436695523886e37 * t11 - 0.313290844133960010134463999760350842e37 * t4 + 0.731435546983453557094244130115508020e36 * e - 0.56858436253121956939703551067914817e35);
  t39 = 0.1e1 / (0.228776265847156105374574303588108230e36 * t4 - 0.95417161918520851228698397869251232e35 * e + 0.9926108757466946691059784515905583e34);
  t41 = 0.22877626584715610537457430358810823e35 * e - 0.5000000000000000000000000000000000e34;
  t42 = 0.1e1 / t41;
  t47 = t5 * t11;
  t69 = 0.1e1 / (0.654232247687108062959456714414297998e39 * t8 - 0.675609158900002407097574145514315969e39 * t5 + 0.278916304571945964903952994065906198e39 * t11 - 0.575413228514605247473820693020351003e38 * t4 + 0.593220909253925252147271522757453247e37 * e - 0.244499005317954864621101235321214039e36);
  t73 = 0.1e1 / (0.976562500000000000000000000000000000e36 * t4 - 0.471278332425497410853453951831642095e36 * e + 0.56858436253121956939703551067914817e35);
  t79 = t5 * t5;
  t91 = t41 * t41;
  t104 = t79 * e;
  t118 = t20 * t29 * t34;
  t121 = t79 * t4;
  t132 = 0.313346493796166626276146253514832686e42 * t121 - 0.458134356865104864971265832302399866e42 * t104 + 0.202492495851125243892745522566957712e42 * t79 - 0.449951006259321771458932235790582356e40 * t47 - 0.669859147063160028351978960124730000e40 * t6 - 0.932047599971732889384750521611980120e40 * t8 + 0.537339858100385429839041218625109640e40 * t5 - 0.926958371602247073076398077780739410e39 * t11 + 0.333115807734110455470107651420820098e38 * t4 + 0.300434420701178052386574626076664468e37 * e + 0.40542607540780120403690713619785917e35;
  t149 = 0.10653015995996516818764240725876837e35 * t79 * t11 - 0.272701351093770305568765828329863390e65 * t121 - 0.314329067178832673032641425899263448e65 * t104 + 0.100188011112964196330442111713604309e66 * t79 - 0.746026606829733185504658818586237698e65 * t47 + 0.216578509823659999327160916922625091e65 * t6 - 0.544229152337064570383926576800381688e63 * t8 - 0.102664976457015346762547393797485975e64 * t5 + 0.181887893697518551637757294531186072e63 * t11 - 0.282638783849461033990430411335243990e61 * t4 - 0.944419489372735688453789429362406038e60 * e - 0.559266306708361097912129083039501275e58;
  return(0.381469726562500000000000000000000000e178 * t2 * x * (0.7829106584252734039349627437800887e34 * t6 + 0.193091487489646769228675946201297168e65 * t8 + 0.398726037890645002933597175939526228e64 * t5 - 0.346760967112342863650646055203200888e64 * t11 + 0.139462250355282050151949829782037141e63 * t4 + 0.461508763386789322737797798593795332e62 * e + 0.523390347885338447943889677681597030e60) * t20 * t24 * t29 * t34 * t39 * t42 - 0.762939453125000000000000000000000000e112 * t2 * (0.27398259874589595489431469281081431e35 * t47 + 0.868504117439355606170874979973999694e65 * t6 - 0.296999823398169205912126797653176902e65 * t8 - 0.188792810936237071366110168504569972e65 * t5 + 0.732442045467530783903653294455819894e64 * t11 - 0.251403971746228512859737633011041368e63 * t4 - 0.701932289470342622461630791671316788e62 * e - 0.500483077737453400619433361569656125e60) * t42 * t29 / (0.327116123843554031479728357207148999e36 * t4 - 0.142985166154472565859108939742567644e36 * e + 0.15625000000000000000000000000000000e35) * t69 * t73 + 0.122070312500000000000000000000000000e179 * t1 * x * (0.195681890458348176968417975837302789e36 * t79 + 0.860956055567590402019947233933433565e66 * t47 - 0.782709701585163609591126332416111050e66 * t6 + 0.644757854961889078866695110337690040e65 * t8 + 0.112025675026413081874352745314679472e66 * t5 - 0.309161337122677307596523221634470692e65 * t11 + 0.115124499471762299300826478037148953e64 * t4 + 0.195660713404016518866442684607622066e63 * e + 0.166443044733159625321834355985759627e61) * t39 / t91 / t41 * t24 / (0.558535805290908460387144295869404858e38 * t5 - 0.526385176501175019002119999550657828e38 * t11 + 0.185923642789014319375318932824882615e38 * t4 - 0.291687269226222321652697598786050481e37 * e + 0.171495353738833683647992757305174657e36) - 0.305175781250000000000000000000000000e67 * t1 * (0.435210467264666818226787746888801988e48 * t104 + 0.305007983716296976853711644645819902e79 * t79 - 0.457499502826614954695889215060996075e79 * t47 + 0.226316277168649449700154186199406948e79 * t6 - 0.248054252891074988212242121590455294e78 * t8 - 0.116931056892685251437545013356924428e78 * t5 + 0.307877487798058231392686621797122442e77 * t11 - 0.827207509924907953373443865818576930e75 * t4 - 0.198978030456042796258952873230062798e75 * e - 0.124843622977950948288095173592099788e73) * t69 * t118 + 0.152587890625000000000000000000000000e105 * x * t132 * t69 * t118 - 0.122070312500000000000000000000000000e116 * t149 * t73 / t22 * t29 / (0.125000000000000000000000000000000000e38 * t4 - 0.496305437873347334552989225795279150e37 * e + 0.492638175325310061637897203000092541e36) / (0.626692987592333252552292507029626788e38 * t8 - 0.684831734261960821433093060039845869e38 * t5 + 0.299345621245296628512627334891968231e38 * t11 - 0.654232247687108062959456714414297998e37 * t4 + 0.714925830772362829295544698712838219e36 * e - 0.31250000000000000000000000000000000e35));
}
double QQH11_ke (double e, double x)
{
  double t1, t100, t11, t113, t116, t127, t144, t2, t20, t22, t23, t24, t29, t34, t39, t4, t40, t41, t42, t47, t5, t58, t6, t65, t69, t75, t8, t87;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t20 = 0.1e1 / (0.619654627270329127338932628511373345e36 * t4 - 0.222648535098765763667297281358993554e36 * e + 0.20000000000000000000000000000000000e35);
  t22 = 0.100000000000000000000000000000000000e37 * e - 0.163071231929977826043195139320894767e36;
  t23 = t22 * t22;
  t24 = 0.1e1 / t23;
  t29 = 0.1e1 / (0.50000000000000000000000000000000000e35 * e - 0.9926108757466946691059784515905583e34);
  t34 = 0.1e1 / (0.347888336091821505730152002123427428e37 * t11 - 0.200627098380065356460214309089768636e37 * t4 + 0.385259179024060191329126674480198050e36 * e - 0.24631908766265503081894860150004627e35);
  t39 = 0.1e1 / (0.556621337746914409168243203397483885e37 * t4 - 0.190768927264901600761833641964729151e37 * e + 0.163071231929977826043195139320894767e36);
  t40 = 0.111324267549382881833648640679496777e36 * e;
  t41 = t40 - 0.20000000000000000000000000000000000e35;
  t42 = 0.1e1 / t41;
  t47 = t5 * t11;
  t58 = t29 * t20;
  t65 = 0.1e1 / (0.123930925454065825467786525702274669e40 * t8 - 0.105158413083803434975528993507997949e40 * t5 + 0.356713403317615808152091571356908072e39 * t11 - 0.604670440449239420123120403069208253e38 * t4 + 0.512206551367173982353568946751147280e37 * e - 0.173457086599369215139172777122933431e36);
  t69 = 0.1e1 / (0.125000000000000000000000000000000000e38 * t4 - 0.496305437873347334552989225795279150e37 * e + 0.492638175325310061637897203000092541e36);
  t75 = t5 * t5;
  t87 = t41 * t41;
  t100 = t75 * e;
  t113 = t58 * t34;
  t116 = t75 * t4;
  t127 = 0.133579330471746555765979187503661418e43 * t116 - 0.155561118901093836712695113762350249e43 * t100 + 0.424921885756103410433769897996259498e42 * t75 + 0.141621802052097352441257425914054196e42 * t47 - 0.735552981851333811304725630227111695e41 * t6 - 0.687342287203536152651250419804306740e40 * t8 + 0.797016511080267011759734192494316510e40 * t5 - 0.127990893272760103721257810490904157e40 * t11 + 0.347638896056699017375312684164515010e38 * t4 + 0.369104811411895343219639397135099612e37 * e + 0.29879022004306029151784056768740771e35;
  t144 = 0.104541070539094620426674134274964634e153 * t75 * t11 + 0.359218308355186925321883364205200260e183 * t116 + 0.527297569579848580559927404651363832e182 * t100 - 0.529352856863505576339353782392450735e183 * t75 + 0.377263116485565970099222685099987252e183 * t47 - 0.999697102253449027976544153057703180e182 * t6 + 0.360111960666850129255324347491091935e181 * t8 + 0.313972298923698010908316251263377652e181 * t5 - 0.499034553957314895839851554204539195e180 * t11 + 0.768924995975553754190240509944923362e178 * t4 + 0.180852437350357571405194414693331902e178 * e + 0.896381367445127225333793536957609550e175;
  return(-0.250000000000000000000000000000000000e174 * t2 * x * (0.432437834756317104717893618426085460e43 * t6 - 0.845448247897769024830816914281825955e73 * t8 - 0.113703562958193965532862075007712411e73 * t5 + 0.942675167793210292288480120690048925e72 * t11 - 0.327013559551611879448686151157741630e71 * t4 - 0.853475456659976688202595910764029140e70 * e - 0.786129951250323997547051243614588690e68) * t20 * t24 * t29 * t34 * t39 * t42 + 0.250000000000000000000000000000000000e114 * t2 * (0.95025327981574899491473032241004651e35 * t47 - 0.236371194745757672715100609922873740e66 * t6 + 0.793419058636993265110210473821190370e65 * t8 + 0.310285294370732584223162478076907528e65 * t5 - 0.109416125903857283713133111391226580e65 * t11 + 0.322877635461420098340402777177290170e63 * t4 + 0.733241574866149456137554577898363360e62 * e + 0.445277883338423483120930763987370398e60) * t42 * t58 * t65 * t69 - 0.125000000000000000000000000000000000e105 * t1 * x * (0.416194500760026401033622323264262642e116 * t75 - 0.141694526067929255374033570728570256e147 * t47 + 0.115666966483224192077950177008168044e147 * t6 - 0.123983076612961075323348972776039296e146 * t8 - 0.100413807507551666701743759003131901e146 * t5 + 0.247940177937047470609991308286348539e145 * t11 - 0.761905506824589519994782970061853515e143 * t4 - 0.115404200604231873191212107196396384e143 * e - 0.763636118171703814825304135167912615e140) * t39 / t87 / t41 * t24 / (0.139155334436728602292060800849370971e39 * t5 - 0.107876259028039213876128585453861182e39 * t11 + 0.313419383466868029229055241637653915e38 * t4 - 0.404457595929482018543006144757000961e37 * e + 0.195599204254363891696880988256971231e36) + 0.125000000000000000000000000000000000e80 * t1 * (0.283837123906920441851486564953290951e36 * t100 - 0.150823721056587481249963380152275876e67 * t75 + 0.197335161775721762223078868569149195e67 * t47 - 0.855245491333762129219789909420336245e66 * t6 + 0.872853603151816966681103041572990280e65 * t8 + 0.301792438539316812885667795912906696e65 * t5 - 0.700161234504131264846858620861865840e64 * t11 + 0.167790764115620813591407623774524626e63 * t4 + 0.316070780593139118358873044314748142e62 * e + 0.167072299338978182678383492243933850e60) * t65 * t113 + 0.100000000000000000000000000000000000e105 * x * t127 * t65 * t113 + 0.100000000000000000000000000000000000e0 * t144 * t69 / t22 * t29 / (0.100000000000000000000000000000000000e38 * t4 - 0.326142463859955652086390278641789534e37 * e + 0.265922266831606196559941380165769469e36) / (0.213726928754794489225566700005873702e38 * t8 - 0.191985928548765259104980719042259040e38 * t5 + 0.689825975144552458210851549417607210e37 * t11 - 0.123930925454065825467786525702274668e37 * t4 + t40 - 0.4000000000000000000000000000000000e34));
}
double QQH12_ke (double e, double x)
{
  double t1, t103, t11, t117, t120, t131, t148, t18, t2, t20, t22, t23, t24, t29, t34, t39, t4, t40, t41, t46, t5, t6, t68, t72, t78, t8, t90;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t18 = 0.67802931251337742596044851047790593e35 * e;
  t20 = 0.1e1 / (0.229861874313681615052072033513647245e36 * t4 - t18 + 0.5000000000000000000000000000000000e34);
  t22 = 0.100000000000000000000000000000000000e37 * e - 0.163071231929977826043195139320894767e36;
  t23 = t22 * t22;
  t24 = 0.1e1 / t23;
  t29 = 0.1e1 / (0.100000000000000000000000000000000000e36 * e - 0.13380667679310147538570226469208713e35);
  t34 = 0.1e1 / (0.678029312513377425960448510477905930e37 * t11 - 0.281449698151452574020633515562326667e37 * t4 + 0.389009259159989134839166006807230247e36 * e - 0.17904226754413520933123226187218380e35);
  t39 = 0.1e1 / (0.678029312513377425960448510477905930e37 * t4 - 0.210567075276192386863734601045588222e37 * e + 0.163071231929977826043195139320894767e36);
  t40 = t18 - 0.10000000000000000000000000000000000e35;
  t41 = 0.1e1 / t40;
  t46 = t5 * t11;
  t68 = 0.1e1 / (0.459723748627363230104144067027294490e40 * t8 - 0.360509016611070208830457512377061292e40 * t5 + 0.113015479571109013463836814795229233e40 * t11 - 0.177038810155453859747247700965922262e39 * t4 + 0.138581174638253699571870452584937206e38 * e - 0.433642716498423037847931942807333579e36);
  t72 = 0.1e1 / (0.50000000000000000000000000000000000e35 * t4 - 0.13380667679310147538570226469208713e35 * e + 0.895211337720676046656161309360919e33);
  t78 = t5 * t5;
  t90 = t40 * t40;
  t103 = t78 * e;
  t117 = t20 * t29 * t34;
  t120 = t78 * t4;
  t131 = 0.114638985812406398902006539476574396e114 * t120 - 0.107112828132203744602149724028570014e114 * t103 + 0.164084937471260055023624763960510532e113 * t78 + 0.138913990966069930308640783616585760e113 * t46 - 0.536894406152562532537165003281742450e112 * t6 + 0.555927348086494662522211353129993800e110 * t8 + 0.242837524196914840532880611875097743e111 * t5 - 0.356995701079850985158414909994788420e110 * t11 + 0.764803105776677181724377343305016640e108 * t4 + 0.818140685745886537277691017523128480e107 * e + 0.457861423975593602169912564565835654e105;
  t148 = 0.100639311463995506565117594499891568e150 * t78 * t11 - 0.961583159487830079285592086691557345e181 * t120 + 0.158527681920947702705689425116009498e181 * t103 + 0.681246299833488497103238540905699475e181 * t78 - 0.461975991979341495338020693250554109e181 * t46 + 0.109888728030287405981816471652729632e181 * t6 - 0.434167777427524915054570300649767813e179 * t8 - 0.232528192462426609750557897031600307e179 * t5 + 0.326754109493969576037317001605373184e178 * t11 - 0.471073093613540520672929473389298316e176 * t4 - 0.824294000270069811427354002239256690e175 * e - 0.341099441724095053957330110434233907e173;
  return(0.125000000000000000000000000000000000e180 * t2 * x * (0.188672510932882138433819027954120737e36 * t6 + 0.717080537909446453032416075589779330e67 * t8 + 0.666165531284022189616083203962369630e66 * t5 - 0.510026269889101991381681351939013820e66 * t11 + 0.149903951863534131747542139792065320e65 * t4 + 0.312736040760157268299083734104124451e64 * e + 0.234573651430591280664826079210616202e62) * t20 * t24 * t29 * t34 * t39 * t41 - 0.625000000000000000000000000000000000e111 * t2 * (0.7933954774832165878437932329722637e34 * t46 + 0.381251819896922012106390657685976199e66 * t6 - 0.118955105218450525514610046167926166e66 * t8 - 0.313592806150149648998057612304820762e65 * t5 + 0.988370619563842725198306227619822169e64 * t11 - 0.247696984650391878835287144219318472e63 * t4 - 0.459034294571947012557438596169571060e62 * e - 0.236140905931749258759924635983634288e60) * t41 * t29 / (0.287327342892102018815090041892059056e36 * t4 - 0.84753664064172178245056063809738241e35 * e + 0.6250000000000000000000000000000000e34) * t68 * t72 + 0.125000000000000000000000000000000000e183 * t1 * x * (0.212803200246951392580923249813515801e36 * t78 + 0.138741458259275360068730542427857506e68 * t46 - 0.100177168677125606847852019895429822e68 * t6 + 0.112295128798559529208830082457117903e67 * t8 + 0.549309161461161140065867671896823328e66 * t5 - 0.119814941902342159453315290049448698e66 * t11 + 0.304269812590547061407813261515692420e64 * t4 + 0.401477525834230369532583726755514075e63 * e + 0.211946657513803258279454889813258504e61) * t39 / t90 / t40 * t24 / (0.847536640641721782450560638097382410e38 * t5 - 0.465218184033973576288687841679362501e38 * t11 + 0.957009683875828722950489032931786690e37 * t4 - 0.874453287054475458006927752992531610e36 * e + 0.29946313531977627751039552665615051e35) - 0.375000000000000000000000000000000000e79 * t1 * (0.189608428017297427244600069375555199e36 * t103 + 0.190475127630159896699881082368872528e68 * t78 - 0.216225043439384422321901893691429072e68 * t46 + 0.814904922318137501664560899710878322e67 * t6 - 0.749055742492768740952801052216026112e66 * t8 - 0.199829506681899405757120956379741819e66 * t5 + 0.402483414391694837932404837978075216e65 * t11 - 0.840386657485815469885306936828545956e63 * t4 - 0.126357297733966822687657658895422156e63 * e - 0.560766164098264484842641528981130232e60) * t68 * t117 + 0.6250000000000000000000000000000000e34 * x * t131 * t68 * t117 - 0.625000000000000000000000000000000000e-1 * t148 * t72 / t22 * t29 / (0.100000000000000000000000000000000000e38 * t4 - 0.326142463859955652086390278641789534e37 * e + 0.265922266831606196559941380165769469e36) / (0.286597464531015997255016348691435044e38 * t8 - 0.211345925051995055395311704138207300e38 * t5 + 0.623412354455767660009628998652833192e37 * t11 - 0.919447497254726460208288134054588980e36 * t4 + t18 - 0.2000000000000000000000000000000000e34));
}
double QQH13_ke (double e, double x)
{
  double t1, t102, t11, t115, t121, t132, t152, t18, t2, t24, t29, t30, t31, t32, t37, t4, t42, t47, t5, t58, t59, t6, t64, t71, t77, t8, t89, t91;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t18 = 0.66131918745071387338077374493629233e35 * e;
  t24 = 0.1e1 / (0.50000000000000000000000000000000000e35 * t4 - 0.13380667679310147538570226469208713e35 * e + 0.895211337720676046656161309360919e33);
  t29 = 0.1e1 / (0.20000000000000000000000000000000000e35 * e - 0.2194285482830037819323248450175033e34);
  t30 = 0.165329796862678468345193436234073083e36 * e;
  t31 = t30 - 0.20000000000000000000000000000000000e35;
  t32 = 0.1e1 / t31;
  t37 = 0.1e1 / (0.275549661437797447241989060390121805e36 * t4 - 0.70203717821789233538417161743051346e35 * e + 0.4460222559770049179523408823069571e34);
  t42 = 0.1e1 / (0.165329796862678468345193436234073083e39 * t11 - 0.562780773135014494780907800577238068e38 * t4 + 0.637868242551012487260324766044434098e37 * e - 0.240744439007932609879642453005152209e36);
  t47 = t5 * t11;
  t58 = 0.100000000000000000000000000000000000e36 * e - 0.13380667679310147538570226469208713e35;
  t59 = 0.1e1 / t58;
  t64 = 0.1e1 / (0.683348543266363177860517913316138390e36 * t4 - t30 + 0.10000000000000000000000000000000000e35);
  t71 = 0.1e1 / (0.218671533845236216915365732261164286e40 * t8 - 0.124879700829773976181305845064215804e40 * t5 + 0.285100704718670270245159729599808866e39 * t11 - 0.325255370282616670527053485198578193e38 * t4 + 0.185427293692810160060638953235041308e37 * e - 0.42260962206973439828928990078489681e35);
  t77 = t5 * t5;
  t89 = t31 * t31;
  t91 = 0.1e1 / t89 / t31;
  t102 = t77 * e;
  t115 = t58 * t58;
  t121 = t77 * t4;
  t132 = 0.772032668797594172033157989577275795e216 * t121 - 0.582107578644100663830866142709071570e216 * t102 + 0.409980802885644503689020699569438942e215 * t77 + 0.781326579667410386004728859710226045e215 * t47 - 0.244563676493402897390788220748380218e215 * t6 + 0.881852912127744767277756147833822725e213 * t8 + 0.588249326419109518967514902325800400e213 * t5 - 0.776768413311592350877876415610349650e212 * t11 + 0.135063566585149339565268566243024114e211 * t4 + 0.133265920401203947686148657644603984e210 * e + 0.557825195226148802972043173231033080e207;
  t152 = 0.143425843785267387514924068215231309e36 * t77 * t11 - 0.361359516970418231839773130309303644e67 * t121 + 0.967259374991088396597387396335443510e66 * t102 + 0.135038796372047507584704779603438938e67 * t77 - 0.864196252452083157395547355355145536e66 * t47 + 0.182108433650319409607616474514555691e66 * t6 - 0.713083391022235114977358831019159540e64 * t8 - 0.262116820619898528005844311922278812e64 * t5 + 0.321919274844244348260152648424057516e63 * t11 - 0.419145278166315026514087231621191582e61 * t4 - 0.563874739749492791021225242358286736e60 * e - 0.194433594828743673035321986196553496e58;
  return(0.133333333333333333333333333333333334e143 * t2 * x * (0.98037173677725636642501610580628301e35 * t6 + 0.745084190984804388439339358805686170e66 * t8 + 0.500581301454098158912770233288855850e65 * t5 - 0.343865608807005608786053946709894916e65 * t11 + 0.846453105227060937681053608091295976e63 * t4 + 0.142344625105767149692230634254841152e63 * e + 0.870982123346516439181554673572303570e60) / (0.273339417306545271144207165326455357e36 * t4 - t18 + 0.4000000000000000000000000000000000e34) * t24 * t29 * t32 * t37 * t42 - 0.200000000000000000000000000000000000e112 * t2 * (0.16142628649560497273194879897014547e35 * t47 + 0.154494221905368724760370313316193870e66 * t6 - 0.435483998318111114038055766331917628e65 * t8 - 0.817256845966615747668268070377900190e64 * t5 + 0.226945842571997736316592844917837280e64 * t11 - 0.478746677586298837304369040028144458e62 * t4 - 0.725653953135305148497875886544230035e61 * e - 0.314787731405964832084805825830162877e59) * t59 * t32 * t24 * t64 * t71 + 0.100000000000000000000000000000000000e149 * t1 * x * (0.66227109423677982292689238317151189e35 * t77 + 0.855163723383798836249097711307886564e66 * t47 - 0.540900591612513196005668939257622320e66 * t6 + 0.583847152454450824533254580061106044e65 * t8 + 0.192342150853688433420843500378384869e65 * t5 - 0.366609077859678144461769796848503056e64 * t11 + 0.769317591676396367336700637043222504e62 * t4 + 0.871370975991162413952421612218181764e61 * e + 0.372627653986701763809794254893862210e59) * t42 * t91 * t29 / (0.103331123039174042715745897646295677e39 * t5 - 0.539791825495128877307193069609327642e38 * t11 + 0.105679419527860311311841149170858489e38 * t4 - 0.918959399941904063141856372986927170e36 * e + 0.29946313531977627751039552665615051e35) - 0.100000000000000000000000000000000000e114 * t1 * (0.144417321589615381946690627350648943e36 * t102 + 0.284955319018386066645721736122907194e67 * t77 - 0.279470806313269456905121936106095782e67 * t47 + 0.911052577704872900796870652267581490e66 * t6 - 0.740458649765754484030421078979476924e65 * t8 - 0.156672854876594091785719297206022956e65 * t5 + 0.271064482213596804997349335215316898e64 * t11 - 0.485352827426296633144636236281247470e62 * t4 - 0.589779847444133145604589999508888080e61 * e - 0.219141531169291305245219051452939802e59) * t71 * t64 / t115 * t37 + 0.266666666666666666666666666666666666e33 * x * t132 * t29 * t37 * t91 * t42 * t24 - 0.200000000000000000000000000000000000e113 * t152 * t29 * t24 * t59 / (0.200000000000000000000000000000000000e38 * t4 - 0.438857096566007563864649690035006600e37 * e + 0.240744439007932609879642453005152209e36) / (0.617626135038075337626526391661811810e38 * t8 - 0.373572185267408502456495111279936670e38 * t5 + 0.903823006757080580068291816535120190e37 * t11 - 0.109335766922618108457682866130582143e37 * t4 + t18 - 0.1600000000000000000000000000000000e34));
}
double QQH14_ke (double e, double x)
{
  double t1, t103, t11, t116, t122, t133, t153, t2, t20, t24, t29, t31, t32, t37, t4, t42, t47, t5, t58, t59, t6, t72, t78, t8, t90, t92;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t20 = 0.1e1 / (0.635600801599191628553924285708575439e36 * t4 - 0.126055622801919433909791538930781945e36 * e + 0.6250000000000000000000000000000000e34);
  t24 = 0.1e1 / (0.200000000000000000000000000000000000e38 * t4 - 0.438857096566007563864649690035006600e37 * e + 0.240744439007932609879642453005152209e36);
  t29 = 0.1e1 / (0.100000000000000000000000000000000000e38 * e - 0.899159266508793973819556425024500269e36);
  t31 = 0.25211124560383886781958307786156389e35 * e - 0.2500000000000000000000000000000000e34;
  t32 = 0.1e1 / t31;
  t37 = 0.1e1 / (0.100844498241535547127833231144625556e37 * t4 - 0.210640809257340363930341292866007858e36 * e + 0.10971427414150189096616242250875165e35);
  t42 = 0.1e1 / (0.126055622801919433909791538930781945e39 * t11 - 0.351688162675766165666302693481328100e38 * t4 + 0.326704197666082529761791230881500854e37 * e - 0.101060923318579048543562806367040357e36);
  t47 = t5 * t11;
  t58 = 0.20000000000000000000000000000000000e35 * e - 0.2194285482830037819323248450175033e34;
  t59 = 0.1e1 / t58;
  t72 = 0.1e1 / (0.317800400799595814276962142854287720e41 * t8 - 0.148753763984709311234761677236230252e41 * t5 + 0.278347406652796561044761874862796306e40 * t11 - 0.260270536581468052728390720014858794e39 * t4 + 0.121614322539744641639154968382115724e38 * e - 0.227174664209587525561354024613942797e36);
  t78 = t5 * t5;
  t90 = t31 * t31;
  t92 = 0.1e1 / t90 / t31;
  t103 = t78 * e;
  t116 = t58 * t58;
  t122 = t78 * t4;
  t133 = 0.130368017200045863641213312780988799e46 * t122 - 0.796676655662485774869545244525150970e45 * t103 + 0.119198628677470484448754265154457052e44 * t78 + 0.973861935974225225355443259159410964e44 * t47 - 0.252114415958836339823776994422264870e44 * t6 + 0.105622760881360318050974898542520633e43 * t8 + 0.359616519150116357910134239805973008e42 * t5 - 0.419266978211294044593807235612906842e41 * t11 + 0.601119892646096431155576969751209636e39 * t4 + 0.520610836198403441314809319078618060e38 * e + 0.170358530872455019514172021823298767e36;
  t153 = 0.78553822130357281056165162002003011e35 * t78 * t11 + 0.124102584094393420154615001429720400e66 * t122 - 0.359527587888433335205598150070025756e65 * t103 - 0.259812510447249412202292220517215234e65 * t78 + 0.155238780449981979209160738036840413e65 * t47 - 0.286704137304237687756917322054017468e64 * t6 + 0.105637726814450088560275969785764957e63 * t8 + 0.282238588240852281887841199510469002e62 * t5 - 0.300212110812524673646349481582314528e61 * t11 + 0.344899299402749475756653647020734748e59 * t4 + 0.364270803154852260629033255256251942e58 * e + 0.104505833508402568406680205770619619e56;
  return(-0.195312500000000000000000000000000000e149 * t2 * x * (0.100512219171930470128083160801799209e36 * t6 - 0.382089540490242838870413161937535610e65 * t8 - 0.192298274029292870355197205972803431e64 * t5 + 0.115689261087262612768594934951727866e64 * t11 - 0.236702791950487887138415966465725830e62 * t4 - 0.322577754444236528871617490115437730e61 * e - 0.161208091872439991668977673258022360e59) * t20 * t24 * t29 * t32 * t37 * t42 + 0.390625000000000000000000000000000000e113 * t2 * (0.79059189955971203094894623559915041e35 * t47 - 0.377479588522543578852193088546203655e65 * t6 + 0.945169592371662103998272509566675960e64 * t8 + 0.130804298681345701079899713743217791e64 * t5 - 0.316291718452151046385647381292663782e63 * t11 + 0.558029457209293967161385842289407775e61 * t4 + 0.692845910078644694539903565215859755e60 * e + 0.252474666453430120993674595930954335e58) * t59 * t32 / (0.125000000000000000000000000000000000e37 * t4 - 0.274285685353754727415406056271879125e36 * e + 0.15046527437995788117477653312822013e35) * t20 * t72 - 0.625000000000000000000000000000000000e99 * t1 * x * (0.158661300277063189214634159502708746e87 * t78 - 0.101837202470712131053044828138674628e117 * t47 + 0.560465470803184646706767714058932085e116 * t6 - 0.559131857667737963098766203032293730e115 * t8 - 0.131514237765146803298140993277348940e115 * t5 + 0.217113720437837259273256733720929999e114 * t11 - 0.376224651232970111635456846452359628e112 * t4 - 0.362195871259740715033229543155572802e111 * e - 0.126612263025454312159893350716392698e109) * t42 * t92 * t29 / (0.100844498241535547127833231144625556e40 * t5 - 0.431922427772021091791023878598023574e39 * t11 + 0.693309104767429500722087218191671975e38 * t4 - 0.494297456292109329434824999378432790e37 * e + 0.132065506896791999465403093995280253e36) + 0.250000000000000000000000000000000000e67 * t1 * (0.158668291418945472802198433222711688e84 * t103 - 0.154778457785852683906285172077997062e114 * t78 + 0.130693874417872013730171300498134085e114 * t47 - 0.367065564207805819086945274785144357e113 * t6 + 0.260917288678758423286686576564333781e112 * t8 + 0.444441390690885873949398526776435667e111 * t5 - 0.655576544441324732244985502817761485e110 * t11 + 0.995531337700378782327356341164641060e108 * t4 + 0.985757089913468289766491575390825480e107 * e + 0.305966104119202217650875559859547271e105) * t72 * t20 / t116 * t37 + 0.312500000000000000000000000000000000e207 * x * t133 * t29 * t37 * t92 * t42 * t24 + 0.250000000000000000000000000000000000e121 * t153 * t29 * t24 * t59 / (0.100000000000000000000000000000000000e39 * t4 - 0.179831853301758794763911285004900054e38 * e + 0.808487386548632388348502450936322857e36) / (0.101850013437535830969697900610172209e40 * t8 - 0.504985473741918699352210128885688470e39 * t5 + 0.100151318623731663056989202295971153e39 * t11 - 0.993126252498736919615506696419649123e37 * t4 + 0.492404776569997788710123198948366973e36 * e - 0.9765625000000000000000000000000000e34));
}
double QQH15_ke (double e, double x)
{
  double t1, t101, t11, t124, t135, t160, t2, t20, t23, t24, t30, t31, t33, t34, t35, t4, t41, t5, t54, t56, t58, t6, t65, t67, t69, t74, t8;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t20 = 0.1e1 / (0.250000000000000000000000000000000000e38 * t4 - 0.368330608824904098874536200283162635e37 * e + 0.135667437397324521240766841076624727e36);
  t23 = 0.6153567092760714778266411395513491e34 * e - 0.500000000000000000000000000000000e33;
  t24 = 0.1e1 / t23;
  t30 = 0.582533397259490543901337960446744520e42 * t6;
  t31 = 0.299136045528169155755466164315592420e42 * t8;
  t33 = 0.729244201133241236297048959775833310e40 * t11;
  t34 = 0.467371916328588638383160114878779295e39 * t4;
  t35 = 0.159671764768279273157672092302628833e38 * e;
  t41 = t5 * t11;
  t54 = 0.1e1 / (0.100000000000000000000000000000000000e39 * t4 - 0.179831853301758794763911285004900054e38 * e + 0.808487386548632388348502450936322857e36);
  t56 = t23 * t23;
  t58 = 0.1e1 / t56 / t23;
  t65 = 0.1e1 / (0.189331939825537776582276094593020724e42 * t8 - 0.726098846633678357226146978670316680e41 * t5 + 0.111319828323957667280173072296736251e41 * t11 - 0.852839211486523818346898626348814615e39 * t4 + 0.326499811559543298022856839675904912e38 * e - 0.499704698142711036747781019709991983e36);
  t67 = 0.100000000000000000000000000000000000e38 * e;
  t69 = 0.1e1 / (t67 - 0.899159266508793973819556425024500269e36);
  t74 = t5 * t5;
  t101 = t74 * e;
  t124 = t74 * t4;
  t135 = 0.564695951955455776704125597579853025e81 * t124 - 0.280563664330305081458675357826294118e81 * t101 - 0.588722876556904759253854214738252970e79 * t74 + 0.291521182616237357873892167546237772e80 * t41 - 0.629106485885475386496195111354066520e79 * t6 + 0.257661163096880613363603575008434750e78 * t8 + 0.564656903466393322924914101654418685e77 * t5 - 0.573194513223368007650112016210203050e76 * t11 + 0.682638412276730534728863219108228970e74 * t4 + 0.504915074187740758673951941226021170e73 * e + 0.132409897861127064923829600727528582e71;
  t160 = -0.100499580995934285685983124434653527e36 * t74 * t11 + 0.166857574049684115276728975966009485e66 * t124 - 0.463513907267117002583782878287897216e65 * t101 - 0.204630319500087919397886134461397974e65 * t74 + 0.112716525554448105284835343618530117e65 * t41 - 0.180841594239499942369697079656323083e64 * t6 + 0.607975105184919225287618546617130384e62 * t8 + 0.122137639091855332083572880062895583e62 * t5 - 0.111709083427266992638928060965690790e61 * t11 + 0.111431157176385013780340373443016882e59 * t4 + 0.936937314891805890161396560145663272e57 * e + 0.223368802577996962602947918187565096e55;
  return(0.244140625000000000000000000000000000e113 * t2 * x * (0.471057320111715430757243186490089907e36 * t6 + 0.155425180800357849050255950294986300e66 * t8 + 0.601573029127235713529694948197410335e64 * t5 - 0.310976537187142380679250499281907490e64 * t11 + 0.526201559763960402646126183492482895e62 * t4 + 0.583151229006532439555615747489905320e61 * e + 0.238213062311022715007916444160113592e59) * t20 * t24 / (0.615356709276071477826641139551349100e71 * t4 - 0.953309422744289838062245886555839999e70 * e + 0.368330608824904098874536200283162636e69) / (t30 - t31 + 0.639709682483039677745052574369925210e41 * t5 - t33 + t34 - t35 + 0.227174664209587525561354024613942797e36) - 0.125000000000000000000000000000000000e150 * t2 * (0.45292030529652486258872526876055887e35 * t41 + 0.187380212511377806888602738707857512e65 * t6 - 0.412243663460416117548997522146350178e64 * t8 - 0.430872014031464751058513643428120150e63 * t5 + 0.898725997980974599793948259936474750e62 * t11 - 0.131994304838301358584084712371958340e61 * t4 - 0.134372678551405986044112168795611000e60 * e - 0.409993131986744163164888040098567892e57) * t54 * t58 * t65 * t69 + 0.976562500000000000000000000000000000e50 * t1 * x * (0.88889347618363225370117292275356073e35 * t74 + 0.493181531357694738801417643995210260e65 * t41 - 0.234988035195577710366428596700652536e65 * t6 + 0.211626707221356161362376808142351988e64 * t8 + 0.368075213326082829532351454912224064e63 * t5 - 0.522579068148209710308879795538478470e62 * t11 + 0.747149674432745995448130948517239510e60 * t4 + 0.607139798342777033980830479582292544e59 * e + 0.174430972227104314326182397429035315e57) / (0.307678354638035738913320569775674550e39 * t11 - 0.703309422744289838062245886555840000e38 * t4 + 0.535298344490374204336439363160422315e37 * e - 0.135667437397324521240766841076624727e36) / (t30 - t31 + 0.639709682483039677745052574369925208e41 * t5 - t33 + t34 - t35 + 0.227174664209587525561354024613942797e36) / (0.153839177319017869456660284887837275e38 * t4 - 0.238327355686072459515561471638960000e37 * e + 0.92082652206226024718634050070790659e35) - 0.625000000000000000000000000000000000e69 * t1 * (0.679161432386234379794205705841581080e152 * t101 + 0.570636722809158745987935205997943380e182 * t74 - 0.413641968053913744610301299475667674e182 * t41 + 0.997816400643826373330536140090495620e181 * t6 - 0.616103737906969715992311075036635010e180 * t8 - 0.851321268887381900887574651087475180e179 * t5 + 0.106493965189820033152847830141876370e179 * t11 - 0.136079280067641948392890349240715589e177 * t4 - 0.110395197178159455510578482776508560e176 * e - 0.285700002050450181286142984191130001e173) * t69 / (0.189331939825537776582276094593020724e39 * t11 - 0.447151851969916611684262706073888593e38 * t4 + 0.351654711372144919031122943277920000e37 * e - 0.92082652206226024718634050070790659e35) * t58 * t54 * t20 + 0.39062500000000000000000000000000000e35 * x * t135 / (0.189331939825537776582276094593020724e41 * t8 - 0.818397059058688734758696155112258285e40 * t5 + 0.141417298669010231230404219953761494e40 * t11 - 0.122108347499031654654765409379260132e39 * t4 + 0.526852296033990443630688385840717245e37 * e - 0.90869865683835010224541609845577119e35) * t24 * t65 + 0.200000000000000000000000000000000000e118 * t160 / (0.246142683710428591130656455820539640e39 * t5 - 0.863964424824725128154282993706716792e38 * t11 + 0.113650532511860950735608292333166634e38 * t4 - 0.664028052791782071409101443953470652e36 * e + 0.14539178509413601635926657575292339e35) * t24 * t20 / (0.932053435615184870242140736714791232e36 * t11 - 0.227198327790645331898731313511624868e36 * t4 + 0.18460701278282144334799234186540473e35 * e - 0.500000000000000000000000000000000e33) / (t67 - 0.736661217649808197749072400566325271e36));
}
double QQH16_ke (double e, double x)
{
  double t1, t11, t112, t118, t129, t149, t2, t20, t24, t29, t31, t32, t37, t4, t42, t47, t5, t58, t59, t6, t68, t74, t8, t86, t88, t99;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t20 = 0.1e1 / (0.881691310834468144623615799596479397e36 * t4 - 0.117373023867448199776185442432329875e36 * e + 0.3906250000000000000000000000000000e34);
  t24 = 0.1e1 / (0.250000000000000000000000000000000000e38 * t4 - 0.368330608824904098874536200283162635e37 * e + 0.135667437397324521240766841076624727e36);
  t29 = 0.1e1 / (0.250000000000000000000000000000000000e37 * e - 0.150849360300419116109756246910592487e36);
  t31 = 0.938984190939585598209483539458639e33 * e - 0.62500000000000000000000000000000e32;
  t32 = 0.1e1 / t31;
  t37 = 0.1e1 / (0.469492095469792799104741769729319500e38 * t4 - 0.658356618725737562540997080309841785e37 * e + 0.230206630515565061796585125176976647e36);
  t42 = 0.1e1 / (0.586865119337240998880927212161649375e39 * t11 - 0.109885082267721543419111316692101373e39 * t4 + 0.685075075537001758812279394117754387e37 * e - 0.142222059394035393025603028306855707e36);
  t47 = t5 * t11;
  t58 = 0.100000000000000000000000000000000000e38 * e - 0.736661217649808197749072400566325271e36;
  t59 = 0.1e1 / t58;
  t68 = 0.1e1 / (0.137764267317885647597439968686949906e42 * t8 - 0.432775168958795643099170670932782594e41 * t5 + 0.543491155401990314645323336928817455e40 * t11 - 0.341067208481223260769471850533576404e39 * t4 + 0.106956724347829495211446113776570250e38 * e - 0.134088166751240326435787802166249953e36);
  t74 = t5 * t5;
  t86 = t31 * t31;
  t88 = 0.1e1 / t86 / t31;
  t99 = t74 * e;
  t112 = t58 * t58;
  t118 = t74 * t4;
  t129 = 0.298986342128350180114594087576427822e47 * t118 - 0.121041404814335984895878610782115116e47 * t99 - 0.522932502676729535700014387744491880e45 * t74 + 0.102426466210675679864212193455259743e46 * t47 - 0.184902388357316995554361138319337118e45 * t6 + 0.694637351558124511471512028498162285e43 * t8 + 0.108129271289551320212651150257346837e43 * t5 - 0.945529061752217537689333184664519600e41 * t11 + 0.937814626306283872564480999583322615e39 * t4 + 0.583571155049415144397721179457611860e38 * e + 0.124366057789623874481076633002545147e36;
  t149 = 0.541606088664820669621154573912048740e164 * t74 * t11 - 0.169630217092839551540600452349816601e195 * t118 + 0.427561197371132415303759593391816068e194 * t99 + 0.126266475572294858099342969800590306e194 * t74 - 0.632375704258342084738398831537102988e193 * t47 + 0.875003057635909805489914744767648985e192 * t6 - 0.263103495402804173301136463769980952e191 * t8 - 0.406137363844855600006792959474473750e190 * t5 + 0.317567073910782561749803456618307652e189 * t11 - 0.271885253791573087415421647838371275e187 * t4 - 0.183749233381752828015385418206744648e186 * e - 0.363661760586761162271660406989325265e183;
  return(0.596046447753906250000000000000000000e148 * t2 * x * (0.116415606176044091622646490598959971e36 * t6 + 0.613961777331862240199069822735455290e65 * t8 + 0.186280293068911460561103773842745665e64 * t5 - 0.815633933373875395583488955191525060e63 * t11 + 0.113758330502098528216684720696998977e62 * t4 + 0.102753743960653154653430254737598038e61 * e + 0.343279171587869334200025355312873755e58) * t20 * t24 * t29 * t32 * t37 * t42 - 0.298023223876953125000000000000000000e116 * t2 * (0.35018402427793393040546857827873127e35 * t47 + 0.231308372589457822095526612198388936e65 * t6 - 0.443764001059427692639700358484307518e64 * t8 - 0.356138143338979837580039035516919004e63 * t5 + 0.636037605940687601731738571893700545e62 * t11 - 0.774985664696393067898773515604726620e60 * t4 - 0.647261903746592897952323044403917640e59 * e - 0.164907197341420697298754385030368771e57) * t59 * t32 * t24 * t20 * t68 + 0.238418579101562500000000000000000000e97 * t1 * x * (0.524933850011865057604527650731738840e85 * t74 + 0.464266347219521004264295987300621695e115 * t47 - 0.190766933377663265177445912450242737e115 * t6 + 0.152800940690938757020667066000670536e114 * t8 + 0.200977680687523787394193245263730862e113 * t5 - 0.244012402616946637637321397375490709e112 * t11 + 0.287619894314658125972615651823676247e110 * t4 + 0.196283778164778302663768616134354333e109 * e + 0.464921079856674208468628299124731440e106) * t42 * t88 * t29 / (0.938984190939585598209483539458639000e40 * t5 - 0.270013971235442537524598248185905071e40 * t11 + 0.290991473038987409960645554441231203e39 * t4 - 0.139287762983532860128749846227042190e38 * e + 0.249852349071355518373890509854995991e36) - 0.238418579101562500000000000000000000e123 * t1 * (0.39212218205052607853442485540735977e35 * t99 + 0.523941220149429090518863673548475236e65 * t74 - 0.325225092489889224972210466790694072e65 * t47 + 0.672141901693584543890992772901209510e64 * t6 - 0.358905649655361716982478247456951884e63 * t8 - 0.403663856563237480129986928637292080e62 * t5 + 0.426604297419038775543899155656273832e61 * t11 - 0.456107319372396168767691913390990788e59 * t4 - 0.304237975159725614477419388119503248e58 * e - 0.655468850419188535370266391958600918e55) * t68 * t20 / t112 * t37 + 0.476837158203125000000000000000000000e203 * x * t129 * t29 * t37 * t88 * t42 * t24 - 0.190734863281250000000000000000000000e-5 * t149 * t29 * t24 * t59 / (0.625000000000000000000000000000000000e38 * t4 - 0.754246801502095580548781234552962435e37 * e + 0.227555295030456628840964845290969131e36) / (0.729947124336792431920395721622049147e40 * t8 - 0.242931114875313351209146133424192049e40 * t5 + 0.323396172719674122104186009233189266e39 * t11 - 0.215256667684196324370999951073359228e38 * t4 + 0.716388085128468016212069350783263397e36 * e - 0.9536743164062500000000000000000000e34));
}
double QQH17_ke (double e, double x)
{
  double t1, t106, t11, t126, t132, t143, t163, t2, t24, t29, t30, t31, t32, t37, t4, t42, t47, t5, t59, t6, t60, t68, t8, t81, t93, t95;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t24 = 0.1e1 / (0.100000000000000000000000000000000000e39 * t4 - 0.988336935133049232522018833033737898e37 * e + 0.244202474337047291591882731295095191e36);
  t29 = 0.1e1 / (0.250000000000000000000000000000000000e37 * e - 0.150849360300419116109756246910592487e36);
  t30 = 0.36686631291635279438889843990153671e35 * e;
  t31 = t30 - 0.2000000000000000000000000000000000e34;
  t32 = 0.1e1 / t31;
  t37 = 0.1e1 / (0.183433156458176397194449219950768355e39 * t4 - 0.190646881827827578389178156550261988e38 * e + 0.494168467566524616261009416516868949e36);
  t42 = 0.1e1 / (0.229291445572720496493061524938460444e40 * t11 - 0.401707743096026030122764394111069041e39 * t4 + 0.234331732372835597687743000387058509e38 * e - 0.455110590060913257681929690581938263e36);
  t47 = t5 * t11;
  t59 = 0.100000000000000000000000000000000000e38 * e - 0.494168467566524616261009416516868949e36;
  t60 = 0.1e1 / t59;
  t68 = 0.1e1 / (0.336477228882098212941003431407009030e36 * t4 - t30 + 0.1000000000000000000000000000000000e34);
  t81 = t5 * t5;
  t93 = t31 * t31;
  t95 = 0.1e1 / t93 / t31;
  t106 = t81 * e;
  t126 = t59 * t59;
  t132 = t81 * t4;
  t143 = 0.519193450481493127821264964678213480e221 * t132 - 0.171531237335793567974069413966030174e221 * t106 - 0.958901102061158079637749113783462925e219 * t81 + 0.114849104679684774735790187945651972e220 * t47 - 0.173728727818441855139046942951491199e219 * t6 + 0.580693277882914468852572742153557110e217 * t8 + 0.676094947952978477515556551137576480e216 * t5 - 0.505204564775170203372670384956824135e215 * t11 + 0.417404685218720990659858081658697534e213 * t4 + 0.216719704824175958859732089316988935e212 * e + 0.378278663242205974474713403181690686e209;
  t163 = 0.726186792727270797390985441224577332e158 * t81 * t11 + 0.172292492548859533198716623687481699e189 * t132 - 0.382240584745854954011555461479053460e188 * t106 - 0.801425327956537666601879463587124116e187 * t81 + 0.359835679709993858656535006997638861e187 * t47 - 0.426857705284356326759724736238394445e186 * t6 + 0.113230951659801218526859762593340481e185 * t8 + 0.136280267491835254485312371898188296e184 * t5 - 0.906779865135579570315441996299752544e182 * t11 + 0.660650803878088552361309172726619889e180 * t4 + 0.361305076287505896399718647232338569e179 * e + 0.593118726882580515879645898505329098e176;
  return(-0.500000000000000000000000000000000000e154 * t2 * x * (0.279839616197906492439478117322179e33 * t6 - 0.966507878157820861631141613749818579e62 * t8 - 0.233020031490298462597167773525918980e61 * t5 + 0.855225614677884594880497149608649370e60 * t11 - 0.980990166995463145520069704071075801e58 * t4 - 0.723297274978259017582061808836251123e57 * e - 0.197691832085852013113143743553745512e55) / (0.131436417532069614430079465393362902e37 * t4 - 0.143307153482950310308163453086537777e36 * e + 0.3906250000000000000000000000000000e34) * t24 * t29 * t32 * t37 * t42 + 0.625000000000000000000000000000000000e118 * t2 * (0.3291264711736997620599822225818617e34 * t47 - 0.142265273576821926637183636362811524e64 * t6 + 0.236692443908753024315427739177926097e63 * t8 + 0.147576148011591253318673233377483334e62 * t5 - 0.224352227364443898368308141560741736e61 * t11 + 0.226225780918556272260160948287881569e59 * t4 + 0.155059663808868055195484582261362715e58 * e + 0.329111898165052575398572496282062068e55) * t32 * t60 / (0.125000000000000000000000000000000000e39 * t4 - 0.123542116891631154065252354129217237e38 * e + 0.305253092921309114489853414118868989e36) * t68 / (0.525745670128278457720317861573451610e41 * t8 - 0.152492939016659291444104462276070240e41 * t5 + 0.176815784977982017149732619260359040e40 * t11 - 0.102446143492871484428465102951734388e39 * t4 + 0.296599095507698474913046144685765188e37 * e - 0.34326570688317523567561677354559988e35) - 0.250000000000000000000000000000000000e152 * t1 * x * (0.385808398342522687861369049234942089e36 * t81 - 0.223058347187576078469784864642725070e66 * t47 + 0.787900434447473826921057025057845366e65 * t6 - 0.555654199864069744193931852163888756e64 * t8 - 0.561052216210469013994074948959398221e63 * t5 + 0.579915959921917557848597556988484411e62 * t11 - 0.563151856553551888549791724038726916e60 * t4 - 0.321521594955107485199442251044634469e59 * e - 0.628950605769256064018096255220437816e56) * t42 * t95 * t29 / (0.366866312916352794388898439901536710e41 * t5 - 0.743881290966965470335068939301571927e40 * t11 + 0.565270064635163183530623164160658625e39 * t4 - 0.190793870268815220353899595511325599e38 * e + 0.241354325038184428912494791498979771e36) + 0.156250000000000000000000000000000000e122 * t1 * (0.88034180928900302884697219073171841e35 * t106 - 0.767733669613245219677231205043039990e65 * t81 + 0.407197006118396512525191714189471490e65 * t47 - 0.719488828749458385798178622771622740e64 * t6 + 0.331235022954508019909774213492371362e63 * t8 + 0.303526919855894331731675904849443000e62 * t5 - 0.270257103085728500953936522058566130e61 * t11 + 0.240764257954863346006815365431404442e59 * t4 + 0.132347416851342214516746735215036568e58 * e + 0.237060417134513268390601578612146502e55) / (0.131436417532069614430079465393362902e41 * t8 - 0.381232347541648228610261155690175600e40 * t5 + 0.442039462444955042874331548150897600e39 * t11 - 0.256115358732178711071162757379335970e38 * t4 + 0.741497738769246187282615361714412970e36 * e - 0.8581642672079380891890419338639997e34) * t68 / t126 * t37 + 0.100000000000000000000000000000000000e36 * x * t143 * t29 * t37 * t95 * t42 * t24 + 0.1e1 * t163 * t29 * t24 * t60 / (0.625000000000000000000000000000000000e38 * t4 - 0.754246801502095580548781234552962435e37 * e + 0.227555295030456628840964845290969131e36) / (0.830709520770389004514023943485269704e39 * t8 - 0.226433851112351815612764763100697820e39 * t5 + 0.246884320680574206275600531690340219e38 * t11 - 0.134590891552839285176401372562803612e37 * t4 + t30 - 0.400000000000000000000000000000000e33));
}
double QQH18_ke (double e, double x)
{
  double t1, t11, t111, t117, t128, t148, t18, t2, t20, t21, t24, t29, t30, t31, t36, t4, t41, t46, t5, t57, t58, t6, t67, t73, t8, t85, t87, t98;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t18 = 0.22399065901218295883995067853552399e35 * e;
  t20 = 0.1e1 / (0.250859076623560094792046402557171396e36 * t4 - t18 + 0.500000000000000000000000000000000e33);
  t21 = 0.100000000000000000000000000000000000e39 * t4;
  t24 = 0.1e1 / (t21 - 0.988336935133049232522018833033737898e37 * e + 0.244202474337047291591882731295095191e36);
  t29 = 0.1e1 / (0.100000000000000000000000000000000000e37 * e - 0.40467242047039263370626985088284183e35);
  t30 = t18 - 0.1000000000000000000000000000000000e34;
  t31 = 0.1e1 / t30;
  t36 = 0.1e1 / (0.223990659012182958839950678535523990e39 * t4 - 0.210689120713266409236390086460055386e38 * e + 0.494168467566524616261009416516868949e36);
  t41 = 0.1e1 / (0.223990659012182958839950678535523990e40 * t11 - 0.281285684290436886909734008379503008e39 * t4 + 0.117615142423299790926079140122582154e38 * e - 0.163759767889366251085448809304718439e36);
  t46 = t5 * t11;
  t57 = 0.100000000000000000000000000000000000e38 * e - 0.494168467566524616261009416516868949e36;
  t58 = 0.1e1 / t57;
  t67 = 0.1e1 / (0.501718153247120189584092805114342792e43 * t8 - 0.105707581642970844358193855669661326e43 * t5 + 0.890340797836200326895780554229240204e41 * t11 - 0.374734067343628326620978787244659116e40 * t4 + 0.788152319478767496670241291652490350e38 * e - 0.662690616474595216056826268447137933e36);
  t73 = t5 * t5;
  t85 = t30 * t30;
  t87 = 0.1e1 / t85 / t30;
  t98 = t73 * e;
  t111 = t57 * t57;
  t117 = t73 * t4;
  t128 = 0.180426164041302709090330800696346600e49 * t117 - 0.486968877787996705466108820897220791e48 * t98 - 0.302353806084802568070588619618104972e47 * t73 + 0.252715472035570756892400261651208692e47 * t46 - 0.320592228882979905295029307804927435e46 * t6 + 0.937630737410998982031073376637111518e44 * t8 + 0.841190359322091670049670966892539438e43 * t5 - 0.533935188654625695230525574012301832e42 * t11 + 0.367152325585994163313988749837185050e40 * t4 + 0.158331395873707418335428830488118484e39 * e + 0.227344118531717443271432817116462879e36;
  t148 = 0.84925023863165766375475422400662627e35 * t73 * t11 + 0.858410710891622917190271612604037690e66 * t117 - 0.164523317458471092674137170035328882e66 * t98 - 0.255600854514717880469080196442130555e65 * t73 + 0.101503140188485027158392637350265537e65 * t46 - 0.102730954468396273462845248005799146e64 * t6 + 0.238051722056836283164468539092003234e62 * t8 + 0.225674481811553677966229578441581235e61 * t5 - 0.127296913774280185204804181842892198e60 * t11 + 0.784149346361625988203234544216730368e57 * t4 + 0.348700524346613018750961408011327435e56 * e + 0.474464874226330988749769998977739108e53;
  return(-0.375000000000000000000000000000000000e149 * t2 * x * (0.205229541311164012540834455644750491e36 * t6 - 0.265242160325349246559391298570989250e66 * t8 - 0.513001750008980186221783784837194558e64 * t5 + 0.156658545897462125538176176977480776e64 * t11 - 0.147566517665966125330359493172479991e62 * t4 - 0.889018527425157907487696419243631848e60 * e - 0.198842819215918256657549722840257105e58) * t20 * t24 * t29 * t31 * t36 * t41 + 0.375000000000000000000000000000000000e118 * t2 * (0.23038244318084214297482651344572627e35 * t46 - 0.372456406440779041210163012199406548e65 * t6 + 0.535101893812994611827587500791185315e64 * t8 + 0.261401144955770306471040984129451976e63 * t5 - 0.336725498653520036662969530072699634e62 * t11 + 0.280470158464858447557277365491716825e60 * t4 + 0.157788863380256466980718733044385532e59 * e + 0.278451035104330817748721098818616524e56) * t58 * t31 * t24 * t20 * t67 - 0.500000000000000000000000000000000000e150 * t1 * x * (0.30931202326273940670002621501356637e35 * t73 - 0.668388688939253835779741938155855120e65 * t46 + 0.202414852043810592782957125875116566e65 * t6 - 0.124774736166322150240678499270775884e64 * t8 - 0.977277882963188904582466560560410795e62 * t5 + 0.856868338064182482975135916701150722e61 * t11 - 0.685141590906958461804012025451462900e59 * t4 - 0.326275412035847481113255486032207410e58 * e - 0.527570095001565071059537283857891768e55) * t41 * t87 * t29 / (0.447981318024365917679901357071047980e40 * t5 - 0.864134724279598455418340518760332318e39 * t11 + 0.624695519494880287664365636212777644e38 * t4 - 0.200582598922971915158562278835982520e37 * e + 0.24135432503818442891249479149897977e35) + 0.600000000000000000000000000000000000e125 * t1 * (0.1437122989439081989098380453754259e34 * t98 - 0.467930181083985668144638528383569046e64 * t73 + 0.211657270539264905374474010692991228e64 * t46 - 0.319164076913566549473223683358603610e63 * t6 + 0.126401417919808840780910008081209472e62 * t8 + 0.943368559249903316677511403183844506e60 * t5 - 0.706222146008433478725855737949115138e59 * t11 + 0.522626222432450772923200282305740290e57 * t4 + 0.237082242052060286466253073304580752e56 * e + 0.352659854912974490523858606255405512e53) * t67 * t20 / t111 * t36 + 0.312500000000000000000000000000000000e207 * x * t128 * t29 * t36 * t87 * t41 * t24 + 0.200000000000000000000000000000000000e123 * t148 * t29 * t24 * t58 / (t21 - 0.809344840940785267412539701765683660e37 * e + 0.163759767889366251085448809304718439e36) / (0.112766352525814193181456750435220458e40 * t8 - 0.251721105297700779235350148825192010e39 * t5 + 0.224760359568395705454823230957767730e38 * t11 - 0.100343630649424037916818561022868558e37 * t4 + t18 - 0.200000000000000000000000000000000e33));
}
double QQH19_ke (double e, double x)
{
  double t1, t102, t11, t115, t121, t132, t152, t18, t2, t20, t24, t29, t30, t31, t36, t4, t41, t46, t5, t57, t58, t6, t71, t77, t8, t89, t91;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t18 = 0.3419221164961199612079930316426341e34 * e;
  t20 = 0.1e1 / (0.46764293499674492039313921157611730e35 * t4 - t18 + 0.62500000000000000000000000000000e32);
  t24 = 0.1e1 / (0.625000000000000000000000000000000000e39 * t4 - 0.414203122677694150626641828931608845e38 * e + 0.686256907343811801442623862528360479e36);
  t29 = 0.1e1 / (0.100000000000000000000000000000000000e37 * e - 0.40467242047039263370626985088284183e35);
  t30 = t18 - 0.125000000000000000000000000000000e33;
  t31 = 0.1e1 / t30;
  t36 = 0.1e1 / (0.854805291240299903019982579106585250e38 * t4 - 0.595750416730518414276342601805605757e37 * e + 0.103550780669423537656660457232902211e36);
  t41 = 0.1e1 / (0.213701322810074975754995644776646312e39 * t11 - 0.251083063118555537766720685657134171e38 * t4 + 0.982257447195272566183710898721829475e36 * e - 0.12793731866356738366050688226931128e35);
  t46 = t5 * t11;
  t57 = 0.250000000000000000000000000000000000e37 * e - 0.82840624535538830125328365786321769e35;
  t58 = 0.1e1 / t57;
  t71 = 0.1e1 / (0.584553668745931150491424014470146625e42 * t8 - 0.113706088969894214233042050822230168e42 * t5 + 0.884178308693800130300156874652067650e40 * t11 - 0.343556995784291462768309163336793113e39 * t4 + 0.667047678699592275301401122300941140e37 * e - 0.51772704412077751254439552222432651e35);
  t77 = t5 * t5;
  t89 = t30 * t30;
  t91 = 0.1e1 / t89 / t30;
  t102 = t77 * e;
  t115 = t57 * t57;
  t121 = t77 * t4;
  t132 = 0.478559478235482346929845461281710525e49 * t121 - 0.105593155063985522597044675884326631e49 * t102 - 0.674672535419219329756064779268117244e47 * t77 + 0.418230253831793572374104919579450276e47 * t46 - 0.445124335245348496761856917566972923e46 * t6 + 0.112776434877213518328566310637685956e45 * t8 + 0.793822437750595495733021007216520481e43 * t5 - 0.426104483088916747157943580476836449e42 * t11 + 0.243544905284165090825657631395936405e40 * t4 + 0.870089837005035088182156368887752575e38 * e + 0.103020662665027616228095805725257749e36;
  t152 = 0.200963523495412254010382401140921291e36 * t77 * t11 + 0.658171006029352877640829122614024428e66 * t121 - 0.107638835817538869661761569411141768e66 * t102 - 0.127958237252101206573866585353688133e65 * t77 + 0.443887726976314811956202672099045563e64 * t46 - 0.381793736710355802892590845287256996e63 * t6 + 0.767221720795126753361653728947724536e61 * t8 + 0.577085326881443954049266608802829292e60 * t5 - 0.275103557835437665622225232114381450e59 * t11 + 0.142567121878855647412142216408876385e57 * t4 + 0.517290208223578915298323191581336879e55 * e + 0.583039764293910773973550841389383199e52;
  return(-0.457763671875000000000000000000000000e147 * t2 * x * (0.139457439000587316233110504564851189e36 * t6 - 0.519784654494839297666314336060562680e65 * t8 - 0.811774586896562341032684714263967275e63 * t5 + 0.205202140077557012215241620578422934e63 * t11 - 0.158574439763947372250978269448657374e61 * t4 - 0.781115774466959392076326813259801890e59 * e - 0.142992151479353087343338578208379652e57) * t20 * t24 * t29 * t31 * t36 * t41 + 0.476837158203125000000000000000000000e113 * t2 * (0.183591683298017636286673259509663077e36 * t46 - 0.855684446388785748454953439643921080e65 * t6 + 0.105796087535453694714417474360584781e65 * t8 + 0.407454871065819759188626755931715180e63 * t5 - 0.443117998009730548521528567438636852e62 * t11 + 0.304457729755011872524894529739798748e60 * t4 + 0.140592330315399693955362615371729940e59 * e + 0.205935138379780873846495886958726518e56) * t58 * t31 / (0.781250000000000000000000000000000000e37 * t4 - 0.517753903347117688283302286164511055e36 * e + 0.8578211341797647518032798281604506e34) * t20 * t71 - 0.190734863281250000000000000000000000e148 * t1 * x * (0.156830573688269750092253692572651551e36 * t77 - 0.976549513690875580871445298735157188e65 * t46 + 0.252972849426971104150959221200253941e65 * t6 - 0.135556283671662649970923321827106264e64 * t8 - 0.829714748240705889381484758754383100e62 * t5 + 0.615305779004488586357660186639053624e61 * t11 - 0.404911490703885301068164451497918919e59 * t4 - 0.160451113893171832629974863682201492e58 * e - 0.214523684722527420572477412610593178e55) * t41 * t91 * t29 / (0.133563326756296859846872277985403945e42 * t5 - 0.181601757842430506692035594596377699e41 * t11 + 0.925356311048693746962776486094820525e39 * t4 - 0.209437101406038754843945212374957208e38 * e + 0.177656096238089917170005929877780833e36) + 0.572204589843750000000000000000000000e64 * t1 * (0.182301534908109590368608414415980374e92 * t102 - 0.170923960811421177237734581534068146e122 * t77 + 0.658213716404174420207960554370705545e121 * t46 - 0.845708760847173160446053773890432895e120 * t6 + 0.287624245142189087017839498904953572e119 * t8 + 0.174656234008413529211996816085423024e118 * t5 - 0.109757948055087856700565693209462547e117 * t11 + 0.673151765017552527025790794868898160e114 * t4 + 0.252203057318530210049883552558123742e113 * e + 0.311239296923861025397561621586404040e110) * t71 * t20 / t115 * t36 + 0.953674316406250000000000000000000000e203 * x * t132 * t29 * t36 * t91 * t41 * t24 + 0.190734863281250000000000000000000000e125 * t152 * t29 * t24 * t58 / (0.100000000000000000000000000000000000e39 * t4 - 0.809344840940785267412539701765683660e37 * e + 0.163759767889366251085448809304718439e36) / (0.467343240464338229423677208282965755e41 * t8 - 0.854257479110819511867142754192695084e40 * t5 + 0.624599461322439340089260968411337111e39 * t11 - 0.228341276853879355660712505652401026e38 * t4 + 0.417385396113427687021475868704387329e36 * e - 0.3051757812500000000000000000000000e34));
}
double QQH20_ke (double e, double x)
{
  double t1, t102, t11, t115, t121, t132, t152, t18, t2, t20, t24, t29, t30, t31, t36, t4, t41, t46, t5, t57, t58, t6, t71, t77, t8, t89, t91;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t18 = 0.66812551093116938650297422930428957e35 * e;
  t20 = 0.1e1 / (0.111597924589259035843612493325474058e37 * t4 - t18 + 0.1000000000000000000000000000000000e34);
  t24 = 0.1e1 / (0.625000000000000000000000000000000000e39 * t4 - 0.414203122677694150626641828931608845e38 * e + 0.686256907343811801442623862528360479e36);
  t29 = 0.1e1 / (0.500000000000000000000000000000000000e37 * e - 0.135660614216959472980354049826638771e36);
  t30 = t18 - 0.2000000000000000000000000000000000e34;
  t31 = 0.1e1 / t30;
  t36 = 0.1e1 / (0.835156888663961733128717786630361960e38 * t4 - 0.526739672968320237642846526081308825e37 * e + 0.82840624535538830125328365786321769e35);
  t41 = 0.1e1 / (0.167031377732792346625743557326072392e41 * t11 - 0.140638317186942309447353085645710051e40 * t4 + 0.394281726245640826069819003146680335e38 * e - 0.368076044994454133646677871792995949e36);
  t46 = t5 * t11;
  t57 = 0.250000000000000000000000000000000000e37 * e - 0.82840624535538830125328365786321769e35;
  t58 = 0.1e1 / t57;
  t71 = 0.1e1 / (0.557989622946295179218062466627370291e44 * t8 - 0.788246045319088858616869517665901264e43 * t5 + 0.445144519698038352998309176490299451e42 * t11 - 0.125619415554244143476457839345049483e41 * t4 + 0.177146400128581405915881476256902271e39 * e - 0.998668446849937182080724645326739979e36);
  t77 = t5 * t5;
  t89 = t30 * t30;
  t91 = 0.1e1 / t89 / t30;
  t102 = t77 * e;
  t115 = t57 * t57;
  t121 = t77 * t4;
  t132 = 0.650070294484344588062789551385631610e223 * t121 - 0.117319052556748271764669183868876355e223 * t102 - 0.737002446193816798056794780269244095e221 * t77 + 0.350435064009781913224207649501159284e221 * t46 - 0.312950361960136966197759419652247662e220 * t6 + 0.682486195585561587633784294103480030e218 * t8 + 0.381190202900642759051048020937407238e217 * t5 - 0.172453249423209969935246891215374274e216 * t11 + 0.818159192044370942526512563280946705e213 * t4 + 0.241792812323511685435275280482014472e212 * e + 0.236370237772147243580950046536238222e209;
  t152 = -0.136618157149440724508434594582233227e36 * t77 * t11 + 0.249226706552093687820689775323760150e66 * t121 - 0.344815513315006539144205925909596782e65 * t102 - 0.321478312341195317086967463530178596e64 * t77 + 0.963591297473035064232070229328246696e63 * t46 - 0.702088504496335889292015375359996556e62 * t6 + 0.121675084868324330688596733611782534e61 * t8 + 0.730024195499686482561964328855744216e59 * t5 - 0.293358613865769867758315129761184962e58 * t11 + 0.127397491673229007790556773154904317e56 * t4 + 0.378125242113318210288159910375716446e54 * e + 0.352834809796917840242764030082894643e51;
  return(0.125000000000000000000000000000000000e153 * t2 * x * (0.66921010251090442706148146261262261e35 * t6 + 0.125027636921155916877391918876346172e65 * t8 + 0.158380140222990675531441338834425136e63 * t5 - 0.330232292243942258025280609585295642e62 * t11 + 0.209218028294006621938936585607760456e60 * t4 + 0.843010673095312974473335506164588025e58 * e + 0.126320787444650160209548409085012300e56) * t20 * t24 * t29 * t31 * t36 * t41 - 0.625000000000000000000000000000000000e118 * t2 * (0.56050503579209435785243909685929337e35 * t46 + 0.130919793995708198882542129882905588e65 * t6 - 0.138908775763823808435450741447935021e64 * t8 - 0.423742930430877583976177153979610080e62 * t5 + 0.387920739010573933156397892815597935e61 * t11 - 0.219628511776507004778840689765014835e59 * t4 - 0.832431050242915233574770460606068380e57 * e - 0.101061812046251810577010037101158886e55) * t58 * t31 / (0.781250000000000000000000000000000000e37 * t4 - 0.517753903347117688283302286164511055e36 * e + 0.8578211341797647518032798281604506e34) * t20 * t71 + 0.125000000000000000000000000000000000e154 * t1 * x * (0.598914437781903417556876511205890743e36 * t77 + 0.186834702211688521490022473875993964e66 * t46 - 0.413172970453183473205083357891385202e65 * t6 + 0.191626093635066761648129556125559648e64 * t8 + 0.921624322362318099299391898490016520e62 * t5 - 0.576684359641473266099256800714312726e61 * t11 + 0.312207751936758893366380227373184852e59 * t4 + 0.102743447560056229423397339431836610e58 * e + 0.113587143951780930124554256370503942e55) * t41 * t91 * t29 / (0.104394611082995216641089723328795245e42 * t5 - 0.135027377363120089116067447280490809e41 * t11 + 0.654531652145116934403209473459815415e39 * t4 - 0.140921438523687234369226700874666491e38 * e + 0.113699901592377546988803795121779733e36) - 0.500000000000000000000000000000000000e69 * t1 * (0.622864981880467720146529056837621960e90 * t102 + 0.292436291562089785789925156015067145e120 * t77 - 0.957266343712675015579563002625966700e119 * t46 + 0.104650543275720291619893539400113542e119 * t6 - 0.305192372296052522334323193568375420e117 * t8 - 0.150593801649145989947597006437606289e116 * t5 + 0.793419854626610836971224096422207290e114 * t11 - 0.402566259379772618097444449751431435e112 * t4 - 0.124614869990865392905279194960525483e111 * e - 0.127475572219807798114405216514392489e108) * t71 * t20 / t115 * t36 + 0.200000000000000000000000000000000000e36 * x * t132 * t29 * t36 * t91 * t41 * t24 + 0.100000000000000000000000000000000000e125 * t152 * t29 * t24 * t58 / (0.100000000000000000000000000000000000e38 * t4 - 0.542642456867837891921416199306555084e36 * e + 0.7361520899889082672933557435859919e34) / (0.166417995387992214544074125154695248e41 * t8 - 0.249081935452598930880073943090346312e40 * t5 + 0.149122840770113609639814792118286756e39 * t11 - 0.446391698357036143374449973301896232e37 * t4 + t18 - 0.400000000000000000000000000000000e33));
}
double QQH21_ke (double e, double x)
{
  double t1, t103, t11, t117, t120, t131, t148, t18, t2, t20, t22, t23, t24, t29, t34, t39, t4, t40, t41, t46, t5, t6, t68, t72, t78, t8, t90;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t18 = 0.10199873227124984539759120634475411e35 * e;
  t20 = 0.1e1 / (0.208074827698842092901543128624953590e36 * t4 - t18 + 0.125000000000000000000000000000000e33);
  t22 = 0.500000000000000000000000000000000000e37 * e - 0.135660614216959472980354049826638771e36;
  t23 = t22 * t22;
  t24 = 0.1e1 / t23;
  t29 = 0.1e1 / (0.78125000000000000000000000000000000e35 * e - 0.1735566523175755870408493716575681e34);
  t34 = 0.1e1 / (0.622550856147765169663032265287805847e41 * t11 - 0.429190367400543535956064619324766047e40 * t4 + 0.985195352423774664204540817884874592e38 * e - 0.753047789092095384708576546168062879e36);
  t39 = 0.1e1 / (0.159373019173827883433736259913678297e38 * t4 - 0.823037833414654992883652958455497355e36 * e + 0.10598485485699958826590160142706154e35);
  t40 = t18 - 0.250000000000000000000000000000000e33;
  t41 = 0.1e1 / t40;
  t46 = t5 * t11;
  t68 = 0.1e1 / (0.650233836558881540317322276952979970e43 * t8 - 0.848012768261023211306606354806903650e42 * t5 + 0.442111499378091045946718515629449286e41 * t11 - 0.115176485461604007138290698310241370e40 * t4 + 0.149932145256819967988055768004682812e38 * e - 0.78020972410151342350056612916151561e35);
  t72 = 0.1e1 / (0.953674316406250000000000000000000000e37 * t4 - 0.423722295697206023048948661273359620e36 * e + 0.4706548681825596154428603413550393e34);
  t78 = t5 * t5;
  t90 = t40 * t40;
  t103 = t78 * e;
  t117 = t20 * t29 * t34;
  t120 = t78 * t4;
  t131 = 0.134767113450563767221124518123777986e123 * t120 - 0.198995111954878909896756294849673243e122 * t103 - 0.119251212840999345334583316440570803e121 * t78 + 0.444013662372143747014898846476385508e120 * t46 - 0.332703602512209156878558569517224956e119 * t6 + 0.621849567152544794780149346987549190e117 * t8 + 0.277583150561294958588796423004397838e116 * t5 - 0.105569592966002515212386014629511674e115 * t11 + 0.415176732837448973853114188091312042e112 * t4 + 0.101407576413113746987990516358203633e111 * e + 0.818967548216069309601250396173154800e107;
  t148 = 0.258786581557413495184485282134325155e159 * t78 * t11 + 0.556816551762825425296398286216544420e189 * t120 - 0.647680101859613745528728176218485488e188 * t103 - 0.482671249910892035481187495687140928e187 * t78 + 0.123842150275082200861933235286335431e187 * t46 - 0.762388921783674425909445017862671000e185 * t6 + 0.113458636194092993086980991574117232e184 * t8 + 0.545127038302455005354091750501148350e182 * t5 - 0.184260020433688895188715011243221342e181 * t11 + 0.668456477691223530271619492740229405e178 * t4 + 0.162596601566282538676016254441443160e177 * e + 0.125545902194023977919055103002959247e174;
  return(-0.298023223876953125000000000000000000e183 * t2 * x * (0.122204738216942730843518097855203307e36 * t6 - 0.244560672140003828668233006766486590e65 * t8 - 0.252050154781265203356387105948705788e63 * t5 + 0.432442452426125613144053162646278275e62 * t11 - 0.224513480833864956472452663130855565e60 * t4 - 0.740213163511742437631779662882465225e58 * e - 0.907977747619433549372329637654963625e55) * t20 * t24 * t29 * t34 * t39 * t41 + 0.149011611938476562500000000000000000e115 * t2 * (0.78140500965259153798837736854547539e35 * t46 - 0.195475525480262862732972005308542222e65 * t6 + 0.177558086487383878561625554006722754e64 * t8 + 0.430553142444014608605082483134188905e62 * t5 - 0.331003589452847078197438188163233025e61 * t11 + 0.154298172636722257839650964434434519e59 * t4 + 0.479961078844368704234440435823434150e57 * e + 0.482365425366505285509612362086506425e54) * t41 * t29 / (0.130046767311776308063464455390595994e37 * t4 - 0.63749207669531153373494503965471319e35 * e + 0.781250000000000000000000000000000e33) * t68 * t72 - 0.953674316406250000000000000000000000e97 * t1 * x * (0.637430733485009343926590660824891780e123 * t78 - 0.212924287018065815584402823852097334e153 * t46 + 0.401252599185039312102205151761182875e152 * t6 - 0.160501279146429604346984356334689280e151 * t8 - 0.609066958455145505445916631648595110e149 * t5 + 0.320920370997387372353065946813613348e148 * t11 - 0.142889744845592603912259410210096768e146 * t4 - 0.389881154721124709779991108610173100e144 * e - 0.356344683267665947279409689446944578e141) * t39 / t90 / t40 * t24 / (0.972735712730883077598487914512196636e41 * t5 - 0.886705634044242662397026451542420422e40 * t11 + 0.302914460562192115202522760406025108e39 * t4 - 0.459638131536648720496037226131545926e37 * e + 0.26139290661995157665081691782000481e35) + 0.610351562500000000000000000000000000e88 * t1 * (0.6478046811250945981003141173780903e34 * t103 - 0.325569716756863970799601295830323771e64 * t78 + 0.904625078934167215732148488062662775e63 * t46 - 0.840365196333629211712983418464415838e62 * t6 + 0.209872968886673102002373761942004605e61 * t8 + 0.840415956409223393391259295642348538e59 * t5 - 0.370836387491933454768610460300736795e58 * t11 + 0.155445221838379648783899897043311702e56 * t4 + 0.397602051718042406263970755866310698e54 * e + 0.336889236609193756321748447407038050e51) * t68 * t117 + 0.48828125000000000000000000000000e32 * x * t131 * t68 * t117 + 0.156250000000000000000000000000000000e-1 * t148 * t72 / t22 * t29 / (0.100000000000000000000000000000000000e38 * t4 - 0.542642456867837891921416199306555084e36 * e + 0.7361520899889082672933557435859919e34) / (0.110401219338701838107545205247012364e42 * t8 - 0.135297293505946327031229062044878028e41 * t5 + 0.663230270088769880894529565526150881e39 * t11 - 0.162558459139720385079330569238244992e38 * t4 + 0.199216273967284854292170324892097871e36 * e - 0.976562500000000000000000000000000e33));
}
double QQH22_ke (double e, double x)
{
  double t1, t104, t11, t118, t121, t132, t149, t2, t20, t22, t23, t24, t29, t34, t39, t4, t40, t41, t42, t47, t5, t6, t69, t73, t79, t8, t91;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t20 = 0.1e1 / (0.496608514227598630087430872258664551e36 * t4 - 0.19932054871038231819017332769787614e35 * e + 0.200000000000000000000000000000000e33);
  t22 = 0.100000000000000000000000000000000000e38 * e - 0.181890495312657969209938980210165787e36;
  t23 = t22 * t22;
  t24 = 0.1e1 / t23;
  t29 = 0.1e1 / (0.78125000000000000000000000000000000e35 * e - 0.1735566523175755870408493716575681e34);
  t34 = 0.1e1 / (0.608278041718696039398722313531116150e41 * t11 - 0.392331306033407914461633272827094812e40 * t4 + 0.842560335546377126845877759061336568e38 * e - 0.602438231273676307766861236934450303e36);
  t39 = 0.1e1 / (0.498301371775955795475433319244690350e39 * t4 - 0.190636283327305523691142470652008973e38 * e + 0.181890495312657969209938980210165787e36);
  t40 = 0.9966027435519115909508666384893807e34 * e;
  t41 = t40 - 0.200000000000000000000000000000000e33;
  t42 = 0.1e1 / t41;
  t47 = t5 * t11;
  t69 = 0.1e1 / (0.124152128556899657521857718064666138e44 * t8 - 0.117576413649601379826561410252736302e43 * t5 + 0.445132887812730589862368806437062690e41 * t11 - 0.842123034584955322404588476330280430e39 * t4 + 0.796124744291720808495901415283509810e37 * e - 0.30088464230666714795777401054798639e35);
  t73 = 0.1e1 / (0.953674316406250000000000000000000000e37 * t4 - 0.423722295697206023048948661273359620e36 * e + 0.4706548681825596154428603413550393e34);
  t79 = t5 * t5;
  t91 = t41 * t41;
  t104 = t79 * e;
  t118 = t20 * t29 * t34;
  t121 = t79 * t4;
  t132 = 0.187516925783072742219641536915780532e127 * t121 - 0.226592090263344727020446412150438532e126 * t104 - 0.126783995484440062622653862012721778e125 * t79 + 0.374730248628242254511221810101729880e124 * t47 - 0.235571545259422286091095494966872895e123 * t6 + 0.376214410747696535679077107527545018e121 * t8 + 0.134835831747658836557953153848042358e120 * t5 - 0.430220632619027437602410530783438310e118 * t11 + 0.140084275199905215723849911527508158e116 * t4 + 0.282622532383247126100622483686795734e114 * e + 0.188598965725014262053766753137765752e111;
  t149 = 0.6851914210607869207006667398051207e34 * t79 * t11 - 0.211911112537433763061403231425310231e66 * t121 + 0.206270930584259813474291523390819262e65 * t104 + 0.124688660760338628018066149014827649e64 * t79 - 0.271728745348544050372245593928458125e63 * t47 + 0.141033813182609516769422348574719191e62 * t6 - 0.179621282908679335490579980326288688e60 * t8 - 0.693138916219254428076975986082274444e58 * t5 + 0.196714406845938770060339895913762654e57 * t11 - 0.594646680478857671085815870395532508e54 * t4 - 0.118699585712055985715261186808369172e53 * e - 0.758050802027940044346289281521969169e49;
  return(0.610351562500000000000000000000000000e161 * t2 * x * (0.475759123346485144760110842298518628e58 * t6 + 0.125340787375768155033736473236460248e89 * t8 + 0.105315770354468630648170022461154874e87 * t5 - 0.148438016664483975592616215715846594e86 * t11 + 0.631341015462396972048903975208683270e83 * t4 + 0.170351075096352679922004556403012520e82 * e + 0.171065005694740692468044960708131230e79) * t20 * t24 * t29 * t34 * t39 * t42 - 0.762939453125000000000000000000000000e127 * t2 * (0.29726634214510629388795077781366259e35 * t47 + 0.978870822028404945471004344631637192e65 * t6 - 0.759604038296769875172287455503860864e64 * t8 - 0.146825918044686601939309983215294170e63 * t5 + 0.946055716824938210091591923566778048e61 * t11 - 0.362866020114967374979576906332593434e59 * t4 - 0.926248154475967340474570160275662812e57 * e - 0.769774659418291099789885171494680520e54) * t42 * t29 / (0.397286811382078904069944697806931641e50 * t4 - 0.159456438968305854552138662158300912e49 * e + 0.160000000000000000000000000000000000e47) * t69 * t73 + 0.122070312500000000000000000000000000e186 * t1 * x * (0.296198813630708724977670039762643749e36 * t79 + 0.130220110698086506763292198788968532e67 * t47 - 0.208779974190311705106961061557356508e66 * t6 + 0.718114196147015712699156048477617990e64 * t8 + 0.215709253600832080090951748250255124e63 * t5 - 0.955443085408394047174252196152958160e61 * t11 + 0.349776188148144685658705707983748575e59 * t4 + 0.790217641809520369175045451835126260e57 * e + 0.596922897528938905396487291305617570e54) * t39 / t91 / t41 * t24 / (0.118804305023182820195062951861546123e42 * t5 - 0.103019883359396849939931622898562933e41 * t11 + 0.334791835722753089117900751472492038e39 * t4 - 0.483243595078145898968779100178911252e37 * e + 0.26139290661995157665081691782000481e35) - 0.610351562500000000000000000000000000e86 * t1 * (0.147073846002004134373543566640546451e36 * t104 + 0.972626847758123602193113218467213800e66 * t79 - 0.229102640188952950621038404176800186e66 * t47 + 0.180634030789011109821574820886364712e65 * t6 - 0.385843603847501245435689131007603840e63 * t8 - 0.125233540402979825319306805877529497e62 * t5 + 0.462397118797831798303442744466192808e60 * t11 - 0.159956460391395229795493067392940302e58 * t4 - 0.338044289820081342467644459165636480e56 * e - 0.237081047831486910118267542210316362e53) * t69 * t118 + 0.15625000000000000000000000000e29 * x * t132 * t69 * t118 - 0.500000000000000000000000000000000000e123 * t149 * t73 / t22 * t29 / (0.500000000000000000000000000000000000e39 * t4 - 0.181890495312657969209938980210165787e38 * e + 0.165420761425420253341680628854638347e36) / (0.122891092481194552341064277633124998e41 * t8 - 0.123310008201671515502055303758936823e40 * t5 + 0.494921407750463316215275987498046008e38 * t11 - 0.993217028455197260174861744517329100e36 * t4 + t40 - 0.40000000000000000000000000000000e32));
}
double QQH23_ke (double e, double x)
{
  double t1, t11, t112, t115, t126, t143, t18, t2, t20, t22, t23, t24, t29, t34, t39, t4, t40, t41, t46, t5, t57, t6, t64, t68, t74, t8, t86, t99;
  t1 = x * x;
  t2 = t1 * t1;
  t4 = e * e;
  t5 = t4 * t4;
  t6 = t5 * t4;
  t8 = t5 * e;
  t11 = t4 * e;
  t18 = 0.60860639817106893592259363367141759e35 * e;
  t20 = 0.1e1 / (0.185200873947380850916174808625337198e37 * t4 - t18 + 0.500000000000000000000000000000000e33);
  t22 = 0.100000000000000000000000000000000000e38 * e - 0.181890495312657969209938980210165787e36;
  t23 = t22 * t22;
  t24 = 0.1e1 / t23;
  t29 = 0.1e1 / (0.500000000000000000000000000000000000e37 * e - 0.74461701688788320236134632942099993e35);
  t34 = 0.1e1 / (0.152151599542767233980648408417854398e42 * t11 - 0.703178680665020606382710197115944675e40 * t4 + 0.108206157420185748009529898861146549e39 * e - 0.554454501839010138185996296147475427e36);
  t39 = 0.1e1 / (0.608606398171068935922593633671417590e39 * t4 - 0.210699719213788463934425772358308400e38 * e + 0.181890495312657969209938980210165787e36);
  t40 = t18 - 0.1000000000000000000000000000000000e34;
  t41 = 0.1e1 / t40;
  t46 = t5 * t11;
  t57 = t29 * t20;
  t64 = 0.1e1 / (0.370401747894761701832349617250674397e45 * t8 - 0.323838951801971203610268117383566244e44 * t5 + 0.113183115030307898795667389907845091e43 * t11 - 0.197668148631422189545196836966329992e41 * t4 + 0.172500584222952599964354635120735254e39 * e - 0.601769284613334295915548021095972781e36);
  t68 = 0.1e1 / (0.250000000000000000000000000000000000e40 * t4 - 0.744617016887883202361346329420999930e38 * e + 0.554454501839010138185996296147475427e36);
  t74 = t5 * t5;
  t86 = t40 * t40;
  t99 = t74 * e;
  t112 = t57 * t34;
  t115 = t74 * t4;
  t126 = 0.104374061038171409725615312534085318e123 * t115 - 0.103230224984350167437671913024342253e122 * t99 - 0.530721535979636663276045113223606680e120 * t74 + 0.125707033053403870802423503804492751e120 * t46 - 0.662870679647507439582421856164788040e118 * t6 + 0.902494089884687932443643166119552855e116 * t8 + 0.260475901353560731823809397785673123e115 * t5 - 0.696137116222555577293073093950584820e113 * t11 + 0.187470217516278118385545454967703062e111 * t4 + 0.312284392956933473902036729435894946e109 * e + 0.172190813361691745432832129586054372e106;
  t143 = 0.138110854490145902004205460029644168e164 * t74 * t11 - 0.262370648116898713848726018256931703e194 * t115 + 0.212955784094820217947751664642764471e193 * t99 + 0.105601274949867109342249474957638654e192 * t74 - 0.194224266682767700598619042568755466e191 * t46 + 0.848406655878987733351059844080807435e189 * t6 - 0.922191436304394518823007070341311115e187 * t8 - 0.286450608216217689971597773727549087e186 * t5 + 0.681513183921686903653019423663343260e184 * t11 - 0.171307955949782178837439146282130193e182 * t4 - 0.280898331219015985245689147815130286e180 * e - 0.148311445181869892265632569157793805e177;
  return(0.187500000000000000000000000000000000e170 * t2 * x * (0.448738231636501965389877024736142346e54 * t6 + 0.668933873623486829957518382638078685e83 * t8 + 0.458868757031389417304297351083605768e81 * t5 - 0.530724810630687922662501389204435475e80 * t11 + 0.184886865225537373776309536392681702e78 * t4 + 0.408329778734112654480844959093956888e76 * e + 0.335690839758062019099768370860091216e73) * t20 * t24 * t29 * t34 * t39 * t41 - 0.937500000000000000000000000000000000e120 * t2 * (0.273977587592977445256186185043694587e36 * t46 + 0.510446842747753066463644073179984380e65 * t6 - 0.337768947083237277443696585307255100e64 * t8 - 0.521602867624818883315487305758074185e62 * t5 + 0.281215931936706167037660322557581628e61 * t11 - 0.887043952308561019098274748646344555e58 * t4 - 0.185783522166219405816255185933531766e57 * e - 0.127557477637458092278343205583882500e54) * t41 * t57 * t64 * t68 + 0.125000000000000000000000000000000000e191 * t1 * x * (0.500169757111260272647661176615783321e36 * t74 + 0.124401959588423244121573114690978535e66 * t46 - 0.169441013199608698995069351491250387e65 * t6 + 0.499866302914302693747849270435534324e63 * t8 + 0.119163122628301390083907394780549824e62 * t5 - 0.443034651939640986350859636053249465e60 * t11 + 0.133322765675617103034087844059899045e58 * t4 + 0.249103220656743446561134518104280486e56 * e + 0.155463089877107942108205744232263640e53) * t39 / t86 / t40 * t24 / (0.760757997713836169903242042089271988e43 * t5 - 0.464884010498765454787032647836958506e42 * t11 + 0.106462959863687381837193873585144829e41 * t4 - 0.108294871239069906323334386842144062e39 * e + 0.412856257159421080146289524097905681e36) - 0.750000000000000000000000000000000000e70 * t1 * (0.252791268898281567557277031367907580e57 * t99 + 0.945641496296649191659887550705504240e86 * t74 - 0.188603183263186525766293753030601279e86 * t46 + 0.126067631549795966288337272949307436e85 * t6 - 0.230052944490506692287403313981938580e83 * t8 - 0.604560607232157652591778138489742425e81 * t5 + 0.186638691720734157705704436402161145e80 * t11 - 0.532360271891581934206471188713478145e77 * t4 - 0.929364925701439918356107343954658050e75 * e - 0.539171081320007380343992969408693150e72) * t64 * t112 + 0.500000000000000000000000000000000000e37 * x * t126 * t64 * t112 - 0.1e1 * t143 * t68 / t22 * t29 / (0.500000000000000000000000000000000000e39 * t4 - 0.181890495312657969209938980210165787e38 * e + 0.165420761425420253341680628854638347e36) / (0.166998497661074255560984500054518173e42 * t8 - 0.137197454843494604815389301754784787e41 * t5 + 0.450857747324998470593523942702663508e39 * t11 - 0.740803495789523403664699234501348794e37 * t4 + t18 - 0.200000000000000000000000000000000e33));
}
