;------------------------------------------------------------------------------;
pro DopTomogPlot, sdir = sdir, mdir = mdir, inf = inf, suf = suf, pht = pht, $
	ctb = ctb, eps = eps, psc = psc, mif = mif, smr = smr, nax = nax, amf = amf, $
	grd = grd, ovr = ovr, rlb = rlb, bal = bal, mag = mag, dsc = dsc, kep = kep
;------------------------------------------------------------------------------;
; plot doppler tomography output
;------------------------------------------------------------------------------;
; sdir	:	map source directory; default ''
; mdir	:	model overlay source directory; default ''
; inf	:	map source file; default 'doptomog.out'
; suf	:	output file suffix; default ''
; pht	:	phase range title; default ''
; ctb	:	color table; default 15 (stern-special)
; eps	:	output encapsulated postscript; default 1 (yes)
; psc	:	output postscript in color; default 1 (yes)
; mif	:	map interpolation factor to reduce pixelation; default 3
; smr	:	pixel radius around origin to 'black-out'; default 1 (only if proj=1)
; nax	:	non-axisymmetric part of map; default 0 (no)
; amf	:	amplitude filter; default 0.1
; grd	:	polar-grid=0, xy-grid=1; default 0
; ovr	:	model velocity overlay; default 1 (yes)
; rlb	:	roche lobe velocity overlay; default 1 (yes)
; bal	:	ballistic stream velocity overlay; default 1 (yes)
; mag	:	magnetic stream velocity overlay; default 1 (yes)
; dsc	:	disc circle of radius dsc around centre of mass; default 0 (no)
; kep	:	keplerian profile velocity overlay; default 0 (no)
;------------------------------------------------------------------------------;
	common inputdata, mdirs, rlbs, bals, mags, dscs, keps
;------------------------------------------------------------------------------;
	common griddata, grds, gs, ds
;------------------------------------------------------------------------------;
	common procdata, pmod, proj
;------------------------------------------------------------------------------;
	common pardata, vm, vf, gam, w0, abs, atm, src, ih, nrm, ns, wd, am, al, ac
;------------------------------------------------------------------------------;
	common specdata, npb, nvp, phase, dphase, vp, spi, sp, spri, spr, spf, sprf
;------------------------------------------------------------------------------;
	common mapdata, np, mapi, map, dmapi, dmap, mmapi, mmap
;------------------------------------------------------------------------------;
	common workdata, vxy, vmax, dv
;------------------------------------------------------------------------------;
	; check keywords, set defaults
	if n_elements(sdir) eq 0 then sdir = ''
	if n_elements(mdir) eq 0 then mdir = ''
	if n_elements(inf) eq 0 then inf = 'doptomog.out'
	if n_elements(suf) eq 0 then suf = ''
	if n_elements(pht) eq 0 then pht = ''
	if n_elements(ctb) eq 0 then ctb = 15
	if n_elements(eps) eq 0 then eps = 1
	if n_elements(psc) eq 0 then psc = 1
	if n_elements(mif) eq 0 then mif = 3
	if n_elements(smr) eq 0 then smr = 1
	if n_elements(nax) eq 0 then nax = 0
	if n_elements(amf) eq 0 then amf = 0.1
	if n_elements(grd) eq 0 then grd = 0
	if n_elements(ovr) eq 0 then ovr = 1
	if n_elements(rlb) eq 0 then rlb = 1
	if n_elements(bal) eq 0 then bal = 1
	if n_elements(mag) eq 0 then mag = 1
	if n_elements(dsc) eq 0 then dsc = 0
	if n_elements(kep) eq 0 then kep = 0
;------------------------------------------------------------------------------;
	; overrides
	if mif eq 0 then mif = 1
;------------------------------------------------------------------------------;
	; input data
	mdirs = mdir
	rlbs = rlb
	bals = bal
	mags = mag
	dscs = dsc
	keps = kep
;------------------------------------------------------------------------------;
	; grid data
	grds = grd
	; check if grid parameter file exists
	exist = findfile('pGrid')
	if exist(0) ne '' then begin
		; load parameter values from pGrid file
		@pGrid
	endif else begin
		print, 'Parameter file <pGrid> not found!'
		read, 'Press <Enter> to exit... ', key
		return
	endelse
;------------------------------------------------------------------------------;
	; read input
	DopTomogRead, sdir + inf
;------------------------------------------------------------------------------;
	; set plotting to postscript
	set_plot, 'ps'
	; set up device
	device, font_size = 12, bits_per_pixel = 8
	; set encapsulated postscript if necessary
	if eps eq 1 then device, /encaps else device, encaps = 0
	; set color postscript if necessary
	if psc eq 1 then device, /color else device, color = 0
	; load default color table
	loadct, ctb, /silent
	; max no of colors of display
	mcol = !d.table_size - 1
;------------------------------------------------------------------------------;
	; scale maximum velocity with velocity units
	vmax = vm / vf
	; override velocity grid scale
	if vmax le 0.5 then gs = 0.1
	; XY-velocity coordinates of centres of bins
	vxy = (2 * findgen(np) / (np - 1) - 1) * vmax
	dv = vxy(1) - vxy(0)
	vxy = [vxy(0) - dv / 2, vxy(np - 1) + dv / 2]
;------------------------------------------------------------------------------;
	; increase plot definition
	!p.thick = 3
	!p.charthick = 3
	!p.charsize = 1.25
	!x.charsize = 1
	!y.charsize = 1
;------------------------------------------------------------------------------;
	; output dopmap stuff to ps files
	if mapi gt 0 then begin
		; offsets and sizes
		xof = 0.125
		yof = 0.125
		vsi = 0.75
		pos = [xof, yof, xof + vsi, yof + vsi]
		; set up device for dopmap(s)
		device, /portrait, xsize = 15, ysize = 15
		; determine nr of loops for dopmaps
		m = 1 + nax
		n = 1 + ovr
		; initialise loop variables with default values
		sfx = ''
		mapav = 0
		; loop for full and non-axisymmetric dopmap
		for i = 1, m do begin
			; calculate axisymmetric average (around com!) map, if necessary
			if i eq 2 then begin
				sfx = '.nax'
				mapav = DopMapAxAv()
			endif
			; determine dopmap
			ma = max(map - mapav)
			map0 = (map - mapav) / ma > 0
			map0 = (mcol * (1 - map0) > 0) < mcol
			; interpolate to higher resolution
			if mif gt 1 then map0 = rebin(map0, mif * np, mif * np)
			if smr gt 0 and proj gt 0 then begin
				smt = mif * smr
				fv = fix(mif * np / 2)
				for ix = fv - smt, fv + (smt - 1) do begin
					for iy = fv - smt, fv + (smt - 1) do begin
						map0(iy, ix) = 0.0 ; min(map0)
					endfor
				endfor
			endif
			; loop for dopmap without and with model velocity overlay
			for j = 1, n do begin
				; dopmap without model velocity overlay
				if j eq 1 then sfx = sfx
				; dopmap with model velocity overlay (second round)
				if j eq 2 then sfx = sfx + '.vmod'
				; separate ps file for dopmap
				file = sdir + 'dopmap' + suf + sfx + '.ps'
				device, filename = file
				result = DopMapPlot(pos, vxy, map0, 0)
				if result ne 0 then return
				; add model velocity overlay (second round)
				if j eq 2 then VModelPlot
				if pht ne '' then $
					xyouts, vxy(0), vxy(1) * 1.25, 'Phase range = ' + pht
				device, /close
				if eps eq 1 then spawn, "sed -i '$ i\showpage' " + file
			endfor
		endfor
	endif
;------------------------------------------------------------------------------;
;------------------------------------------------------------------------------;
	; output modumap stuff to ps files
	if mmapi gt 0 then begin
		; offsets and sizes
		xof = 0.125
		yof = 0.125
		vsi = 0.75
		pos = [xof, yof, xof + vsi, yof + vsi]
		; set up device for modumap(s)
		device, /portrait, xsize = 15, ysize = 15
		; determine nr of loops for dopmaps
		n = 1 + ovr
		; initialise loop variables with default values
		sfx = ''
		; loop for map without and with model velocity overlay
		for i = 1, n do begin
			; map without model velocity overlay
			if i eq 1 then sfx = sfx
			; map with model velocity overlay
			if i eq 2 then sfx = sfx + '.vmod'
			; determine modumap
			for j = 1, mmapi do begin
				if j lt mmapi then begin
					map0 = mmap(*, *, j)
					; interpolate to higher resolution
					if mif gt 1 then map0 = rebin(map0, mif * np, mif * np)
					; 'clean' central pixels
					if smr gt 0 and proj gt 0 then begin
						smt = mif * smr
						fv = fix(mif * np / 2)
						for ix = fv - smt, fv + (smt - 1) do begin
							for iy = fv - smt, fv + (smt - 1) do begin
								map0(iy, ix) = max(map0)
								if j ge 2 then map0(iy, ix) = 0.0
							endfor
						endfor
					endif
					; store amplitude map
					if j eq 1 then mapamp = map0
				endif else begin
					; phase of maximum flux for amplitudes > specified amp
					for ix = 0, mif * np - 1 do begin
						for iy = 0, mif * np - 1 do begin
							if mapamp(iy, ix) lt amf then map0(iy, ix) = 99.9
						endfor
					endfor
				endelse
				; amplitude map
				if j eq 1 then begin
					map1 = map0 / max(map0)
					map1 = (mcol * (1 - map0) > 0) < mcol
				; phase maps
				endif else begin
					map1 = (mcol * map0 > 0) < mcol
				endelse
				; set specifics for each modumap
				if j eq 1 then typ = 'amp'
				if j eq 2 then typ = 'pha'
				if j eq 3 then typ = 'pham'
				if j eq 4 then typ = 'phap'
				; load prism color table
				if j ge 2 then loadct, 6, ncolors = 254, bottom = 0, /silent
				; separate ps file for modumap
				file = sdir + 'modu' + typ + suf + sfx + '.ps'
				device, filename = file
				result = DopMapPlot(pos, vxy, map1, 0)
				if result ne 0 then return
				; load default color table
				loadct, ctb, /silent
				; add model velocity overlay (second round)
				if i eq 2 then VModelPlot
				device, /close
				if eps eq 1 then spawn, "sed -i '$ i\showpage' " + file
			endfor
		endfor
	endif
;------------------------------------------------------------------------------;
	return
end
;------------------------------------------------------------------------------;
