;+
; NAME:
;         p3d_misc_imcombine_wrapper
;
;         $Id: p3d_misc_imcombine_wrapper.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         This routine combines several images into one image. Either because
;         an image was read out in several blocks, or if several images are to
;         be averaged (using some filter) in order to remove cosmic ray hits.
;         This routine also handles the case that both combinations are to be
;         done.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_imcombine_wrapper,filename,ofilename,ofsfx,odata,detsec=, $
;             /mbias,/dmask,ostr=,mostr=,parfile=,userparfile=,detector=, $
;             opath=,sfx=,xstr=,/dsh,/compress,title=,/dflip,stawid=,topwid=, $
;             logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         filename        - A scalar or a one-dimensional array of filenames,
;                           which contents are to be combined.
;
; KEYWORD PARAMETERS:
;         detsec          - This output variable is set to a four-element
;                           (columns) -by- number of blocks (rows) integer
;                           array that specifies the detector region to use on
;                           the CCD for each block. For each row the first two
;                           elements are used with the x-axis, and the second
;                           two elements with the y-axis.
;         mbias [0]       - If this keyword is set then the filename is
;                           appended with the string '_mbias' instead of
;                           '_imcmb'.
;         dmask [0]       - If this keyword is set then it is possible to call
;                           this routine also with a two-element array for
;                           FILENAME.
;         ostr ['_imcmb'] - A scalar string with the image-combination specific
;                           string that is used to create the output filename.
;         mostr ['_mbias'] - A scalar string with the master-bias specific
;                           string that is used to create the output filename.
;         parfile         - A scalar string that specifies the name of a file
;                           with instrument-specific setup parameters.
;         userparfile     - A scalar string with the name of a file with user-
;                           defined parameters.
;         detector [0]    - A scalar integer that specifies the currently
;                           selected detector; DETECTOR is a zero-based value.
;         opath ['.']     - A scalar string that specifies the path, where the
;                           output data is saved.
;         sfx [.fits]     - A scalar string specifying the file ending (without
;                           a trailing compression suffix, such as .gz or .Z).
;         xstr ['']       - To be filled in...
;         dsh             - This flag upon exit indicates whether anything was
;                           done by this routine (1) or not (0).
;         compress [0]    - If this keyword is set then the output data file is
;                           compressed (using gzip).
;         title [' ']     - This scalar string is used to mark in the output
;                           file which kind of data this is.
;         dflip [0]       - This keyword is set if the data is flipped on the
;                           dispersion axis before the routine is exited.
;         stawid          - If set to a valid ID then a log message is written
;                           using this ID for relevant actions.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         ofilename       - The name of the output file(s), without the file
;                           suffix (OFSFX). If FILENAME is a two-element array
;                           then OFILENAME is also a two-element array,
;                           otherwise it is a scalar.
;         ofsfx           - The file suffix(es) corresponding to OFILENAME.
;         odata           - Contains the combined output array.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
;-
pro p3d_misc_imcombine_wrapper_flipsave,filename,ofilename,ofsfx,daxis=daxis, $
        compress=compress,sfx=sfx,topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug
  compile_opt hidden,IDL2

  error=0 & rname='p3d_misc_imcombine_wrapper_flipsave: '
  if ~n_elements(verbose) then verbose=0
  if ~n_elements(topwid) then topwid=0L
  debug=keyword_set(debug)

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Creating a new output filename and writing the output file:

  pos=strpos(filename,sfx,/reverse_search)

  data=readfits(filename,hdr,/no_signed,silent=verbose lt 3)
  data_=reverse(data,daxis)
  ofilename=(pos eq -1L?filename:strmid(filename,0L,pos))+'_dflip'
  fxaddpar,hdr,'P3DDFLIP',1,'  Was the raw data disp. axis flipped?'
  writefits,ofilename+ofsfx,data_,hdr,compress=compress

  return
end ;;; procedure: p3d_misc_imcombine_wrapper_flipsave

pro p3d_misc_imcombine_wrapper,filename_,ofilename,ofsfx,odata, $
        detsec=detsec,mbias=mbias,dmask=dmask,ostr=ostr,mostr=mostr, $
        parfile=parfile,userparfile=userparfile,detector=d,opath=opath, $
        sfx=sfx,xstr=xstr,dsh=dsh,compress=compress,title=title,dflip=dflip, $
        stawid=stawid,topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_misc_imcombine_wrapper: '
  if ~n_elements(verbose) then verbose=0
  if ~n_elements(topwid) then topwid=0L
  usestawid=~n_elements(stawid)?0L:widget_info(stawid,/valid_id)
  debug=keyword_set(debug)
  dsh=0L & xstr=''

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_misc_imcombine_wrapper'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking some input arguments:

  if size(parfile,/type) ne 7L then begin
    errmsg='PARFILE [2] must be specified; as the scalar filename of an in' + $
           'strument-specific parameter file.'
    goto,error_handler
  endif
  if ~file_test(parfile,/read,/regular) then begin
    errmsg='The file PARFILE "'+parfile+'" does not exist.'
    goto,error_handler
  endif

  if n_elements(opath) ne 0L then begin
    if size(opath,/type) ne 7L then begin
      errmsg='OPATH must, if specified, be of string type.'
      goto,error_handler
    endif
    if ~file_test(opath,/directory,/read) then begin
      errmsg='The directory OPATH "'+opath+'" does not exist.'
      goto,error_handler
    endif
  endif else begin
    cd,current=opath
  endelse
  idx=strpos(opath,path_sep(),/reverse_search)
  if idx ne strlen(opath)-1L then opath+=path_sep()

  if ~n_elements(ostr) then ostr='_imcmb'
  if size(ostr,/type) ne 7L or n_elements(ostr) ne 1L then begin
    errmsg='OSTR must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  if ~n_elements(mostr) then mostr='_mbias'
  if size(mostr,/type) ne 7L or n_elements(mostr) ne 1L then begin
    errmsg='MOSTR must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  if n_elements(userparfile) ne 0L then begin
    s=size(userparfile)
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='USERPARFILE, if set, must be a scalar string.'
      goto,error_handler
    endif
    if userparfile ne '' then begin
      if ~file_test(userparfile,/regular,/read) then begin
        errmsg='USERPARFILE, cannot read the file "'+userparfile+'".'
        goto,error_handler
      endif

      readcol,userparfile,uparname,uparvalue,format='a,a',comment=';', $
          silent=verbose lt 3,delimiter=' '
    endif
  endif ;; n_elements(userparfile) ne 0L

  if ~n_elements(d) then d=0L
  s=size(d)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='DETECTOR must be a scalar integer; DETECTOR>=0.'
    goto,error_handler
  endif
  if d lt 0L then begin
    errmsg='DETECTOR must be a scalar integer; DETECTOR>=0.'
    goto,error_handler
  endif

  if ~n_elements(sfx) then sfx='.fits'
  if size(sfx,/type) ne 7L or n_elements(sfx) ne 1L then begin
    errmsg='SFX must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  compress=keyword_set(compress)

  if ~n_elements(title) then title=' '
  s=size(title)
  if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
    errmsg='TITLE, if set, must be a scalar string.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Reading various instrument-specific parameters:

  ppath=strmid(parfile,0L,strpos(parfile,path_sep(),/reverse_search)+1L)
  parname='' & parvalue=''
  readcol,parfile,parname,parvalue,format='a,a',delimiter=' ', $
      silent=verbose lt 3,comment=';'

  p3d_misc_read_params,parname,parvalue,'daxis',daxis,/must_exist,/nou, $
      type='integer',topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2 (not '+strtrim(daxis,2L)+').'
    goto,error_handler
  endif

  p3d_misc_read_params,parname,parvalue,'ndetector',ndetector,/nou, $
      type='integer',topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(ndetector) then ndetector=1L ;; default is one detector

  p3d_misc_read_params,parname,parvalue,'nblocks',nblocks,type='integer', $
      /nou,topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(nblocks) then nblocks=1L

  bsfx=['']
  if nblocks gt 1L then begin
    bsfx=strarr(nblocks)
    for j=0L,nblocks-1L do begin
      p3d_misc_read_params,parname,parvalue,'blocksfx_'+strtrim(j+1L,2L),tmp, $
          /must_exist,uparname=uparname,uparvalue=uparvalue, $
          topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then return
      bsfx[j]=strtrim(tmp,2L)
    endfor
  endif ;; nblocks gt 1L

  xstr=ndetector gt 1L?'_'+strtrim(d+1L,2L):''
  p3d_misc_read_params,parname,parvalue,'dflip'+xstr,dflip, $
      uparname=uparname,uparvalue=uparvalue, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  dflip=n_elements(dflip) ne 0L?(strlowcase(dflip) eq 'yes'?1L:0L):0L
  if dflip ne 1L and dflip ne 0L then begin
    errmsg='DFLIP must be a scalar string; yes||no.'
    goto,error_handler
  endif

  p3d_misc_read_params,parname,parvalue,'keywordsfile',kwrdlist,/must_exist, $
      uparname=uparname,uparvalue=uparvalue,topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  kwrdlist=ppath+kwrdlist
  if ~file_test(kwrdlist,/read,/regular) then begin
    errmsg='The file KWRDLIST "'+kwrdlist+'" does not exist.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Checking some more input arguments:

  s=size(filename_) & n=s[s[0L]+2L]
  if n lt 1L or s[s[0L]+1L] ne 7L then begin
    errmsg='FILENAME [1] must be specified, and be of string type.'
    goto,error_handler
  endif
  if n eq 2L and ~keyword_set(dmask) then begin
    errmsg='FILENAME [1] can only be an array of two elements if /DMAS' + $
           'K is set.'
    goto,error_handler
  endif
  if n eq 2L and nblocks eq 1L then begin
    msg=['Two input files are specified, and NBLOCKS==1', $
         'Returning the input file to the caller.']
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

    ofilename=strarr(2L) & ofsfx=strarr(2)

    for i=0,1 do begin
      pos=strpos(filename_[i],sfx,/reverse_search)
      ofsfx[i]=pos eq -1L?'':strmid(filename_[i],pos)
      tmp=strpos(ofsfx[i],'.gz')
      if compress then begin
        if tmp eq -1L then ofsfx[i]+='.gz'
      endif else begin
        if tmp ne -1L then ofsfx[i]=strmid(ofsfx[i],0L,tmp)
      endelse ;; compress
    endfor ;; i=0,1

    ;;========================================------------------------------
    ;; Acquiring information on the detector section:

    hdr1=headfits(filename_[0L])
    hdr2=headfits(filename_[1L])
    hdr=[[hdr1],[hdr2]]

    p3d_misc_detsec,hdr[0L,*],kwrdlist,parfile,detsec, $
        userparfile=userparfile, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return

    ;;========================================------------------------------
    ;; Flipping the data, if required:

    if dflip then begin
      for i=0,1 do begin
        p3d_misc_imcombine_wrapper_flipsave,filename_[i],ofilename__, $
            ofsfx[i],daxis=daxis,compress=compress,sfx=sfx,topwid=topwid, $
            logunit=logunit,verbose=verbose,error=error,debug=debug
        if error ne 0 then return
        ofilename[i]=ofilename__
      endfor ;; i=0,1
    endif else begin ;; dflip
      for i=0,1 do begin
        pos=strpos(filename_[i],sfx,/reverse_search)
        ofilename[i]=pos eq -1L?filename_[i]:strmid(filename_[i],0L,pos)
      endfor ;; i=0,1
    endelse ;; dflip

    return
  endif ;; n eq 2L and nblocks eq 1L

  loop=n eq 2L?1L:0L

  anycomb=0L & blccomb=0L
  for i=0L,n-1L do begin
    if ~file_test(filename_[i],/read,/regular) then begin
      errmsg='Cannot read or find the file "'+filename_[i]+'" ['+ $
          strtrim(i+1L,2L)+'/'+strtrim(n,2L)+']'
      goto,error_handler
    endif

    ;; Checking if any file has already been combined, if yes then exit:
    tmp=headfits(filename_[i])
    if long(fxpar(tmp,'NCOMBINE')) gt 1L then anycomb++
    if fxpar(tmp,'CETSEC1') ne '0' then blccomb++
  endfor ;; i=0L,n-1L

  if anycomb ge 1L or blccomb ge 1L then begin
    pos=strpos(filename_[0L],sfx,/reverse_search)
    ofsfx=pos eq -1L?'':strmid(filename_[0L],pos)
    ofilename=pos eq -1L?filename_[0L]:strmid(filename_[0L],0L,pos)

    msg=['The input file has already been combined out of several other' + $
         ' files.','Returning the input file to the caller.']
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

    odata=readfits(filename_[0L],/no_unsigned,silent=verbose lt 3)

    ;;========================================------------------------------
    ;; Acquiring information on the detector section:

    hdr=headfits(filename_[0L])
    p3d_misc_detsec,hdr,kwrdlist,parfile,detsec,userparfile=userparfile, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return

    return
  endif ;; anycomb ge 1L or blccomb ge 1L

  ;;========================================------------------------------
  ;; Checking the filenames: 

  n=n_elements(filename_)
  fsfx=strarr(n) & filename__=strarr(n)
  for i=0L,n-1L do begin
    pos=strpos(filename_[0L],sfx,/reverse_search)
    fsfx[i]=pos eq -1L?'':strmid(filename_[0L],pos)
    filename__[i]=pos eq -1L?filename_[i]:strmid(filename_[i],0L,pos)
  endfor
  filename=filename__

  ;; Checking if the DETECTOR of the file is the same as the used one:
  for i=0L,n-1L do begin
    p3d_misc_checkfile,filename[i]+fsfx[i],ndetectors,d,kwrdlist, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return
  endfor

  ;;========================================------------------------------
  ;; Finding and checking the existence of all filenames when there are several
  ;; blocks:

  p3d_misc_fileallbl,filename,fsfx,nblocks,bsfx,nfilename,nfsfx, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  filename=nfilename & fsfx=nfsfx

  n=size(filename,/n_dimensions) eq 2L? $
      (size(filename,/dimensions))[0L]:n_elements(filename)

  ;; TEMPORARY SOL.
  if ndetectors gt 1L then begin
    quadrant=d+1L
    block=nblocks eq 1L?'':'b'
    xstr='_'+strtrim(quadrant,2L)+strupcase(block)
  endif else begin
    xstr=''
  endelse

  ;; Return here if there is nothing to do:
  if nblocks eq 1L and n lt 3L then begin
    pos=strpos(filename_[0L],sfx,/reverse_search)
    ofsfx=pos eq -1L?'':strmid(filename_[0L],pos)

    hdr=headfits(filename_[0L])
    p3d_misc_detsec,hdr,kwrdlist,parfile,detsec,userparfile=userparfile, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return

    ;; Flipping the data, if required:
    if dflip then begin
      tmp=strpos(ofsfx,'.gz')
      if compress then begin
        if tmp eq -1L then ofsfx+='.gz'
      endif else begin
        if tmp ne -1L then ofsfx=strmid(ofsfx,0L,tmp)
      endelse ;; compress

      p3d_misc_imcombine_wrapper_flipsave,filename_[0L],ofilename,ofsfx, $
          daxis=daxis,compress=compress,sfx=sfx,topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then return
    endif else begin ;; dflip
      ofilename=pos eq -1L?filename_[0L]:strmid(filename_[0L],0L,pos)
    endelse  ;; dflip

    return
  endif ;; nblocks eq 1L and n lt 3L

  ;;========================================------------------------------
  ;; Looping two times, if n==2:

  ofilename=n eq 2L?strarr(2L):''
      ofsfx=n eq 2L?strarr(2L):''

  for ii=0L,loop do begin

    ;;========================================------------------------------
    ;; Acquiring information on the detector section of every block, in case
    ;; there are several blocks with different sections:

    if nblocks gt 1L then begin
      for j=0L,nblocks-1L do begin
        tmp=headfits(filename[ii,j]+fsfx[ii,j])
        hdr=~j?[tmp]:[[hdr],[tmp]]
      endfor
      p3d_misc_detsec,hdr,kwrdlist,parfile,detsec,odetsec, $
          userparfile=userparfile,topwid=topwid,logunit=logunit, $
          verbose=verbose,error=error,debug=debug
      if error ne 0 then return
    endif else begin
      hdr=headfits(filename[ii]+fsfx[ii])
      p3d_misc_detsec,hdr,kwrdlist,parfile,detsec,odetsec, $
          userparfile=userparfile,topwid=topwid,logunit=logunit, $
          verbose=verbose,error=error,debug=debug
      if error ne 0 then return
    endelse ;; nblocks gt 1L

    ;;========================================------------------------------
    ;; Combining the data:

    if n eq 2L then begin
      filen=filename[ii,*]+fsfx[ii,*]
      fjlen=filename[ii,0L]+fsfx[ii,0L]
      fklen=filen
    endif else begin
      filen=filename+fsfx
      fjlen=filename[0L]+fsfx[0L]
      fklen=filename[0L,*]+fsfx[0L,*]
    endelse ;; n eq 2L

    p3d_misc_imcombine,filen,comim,nval,rstr,detsec=odetsec, $
        daxis=daxis,userparfile=userparfile,stawid=stawid,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return

    ;; Flipping the data on the dispersion axis, if required to do so:
    if dflip then begin
       comim_=reverse(comim,daxis)
       comim=temporary(comim_)
    endif

    ;;========================================------------------------------
    ;; Creating an output filename (using the first file):

    tmppos=strpos(filename[ii,0L],'.',/reverse_search)
    tmpfile=tmppos eq -1L?filename[ii,0L]:strmid(filename[ii,0L],0L,tmppos)
    if bsfx[0L] ne '' then begin
      tmppos=strpos(tmpfile,bsfx[0L],/reverse_search)
      tmpfile=strmid(tmpfile,0L,tmppos)
    endif
    filetype=keyword_set(mbias) and n ge 3L?mostr:ostr
    ofilename[ii]=tmpfile+filetype+xstr+fsfx[ii,0L]

    tmp=strpos(ofilename[ii],'.gz')
    if compress then begin
      if tmp eq -1L then ofilename[ii]+='.gz'
    endif else begin
      if tmp ne -1L then ofilename[ii]=strmid(ofilename[ii],0L,tmp)
    endelse
    if n_elements(opath) ne 0L then $
       ofilename[ii]=opath+file_basename(ofilename[ii])

    ;;========================================------------------------------
    ;; Writing the combined data file:

    hdr=headfits(fjlen)
    fxhmake,tmphdr,comim
    fxaddpar,hdr,'BITPIX',fxpar(tmphdr,'BITPIX')
    fxaddpar,hdr,'BZERO',0
    if n ne 2L then begin
      for i=0L,n-1L do begin
        istr=strtrim(i,2L)
        if bsfx[0L] ne '' then begin
          tmppos=strpos(filename[i,0L]+fsfx[i,0L],path_sep(),/reverse_search)
          tmp=strmid(filename[i,0L]+fsfx[i,0L],1L+tmppos)
        endif else begin
          tmppos=strpos(filename[i]+fsfx[i],path_sep(),/reverse_search)
          tmp=strmid(filename[i]+fsfx[i],1L+tmppos)
        endelse

        tmpstr='000'
        tmpstr=strmid(tmpstr,0L,strlen(tmpstr)-strlen(istr))+istr
        fxaddpar,hdr,'IMCMB'+tmpstr,tmp
      endfor ;; i=0L,n-1L
    endif ;; n ne 2L

    tmp=file_dirname(fjlen,/mark_directory)
    p3d_misc_fitsparsplit,'IMPATH',tmp,hdr,' Raw data path', $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return

    fxaddpar,hdr,'NCOMBINE',nval,rstr
    fxaddpar,hdr,'P3DDFLIP',dflip,'  Was the raw data disp. axis flipped?'
    if nblocks gt 1L then begin
      fxaddpar,hdr,'NAXIS1',fxpar(tmphdr,'NAXIS1')
      fxaddpar,hdr,'NAXIS2',fxpar(tmphdr,'NAXIS2')

      ;; Normalizing the data to use the same gain:
      p3d_misc_mmodhdr,nblocks,fklen,kwrdlist,hdr,comim, $
          detsec=detsec,userparfile=userparfile,topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then return
    endif ;; nblocks gt 1L

    writefits,ofilename[ii],comim,hdr,compress=compress

    msg='Wrote the '+title+'file "'+file_basename(ofilename[ii])+'".'
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
  endfor ;; ii=0L,loop

  ;; Separating the output filename into a main part and a suffix:
  for ii=0L,loop do begin
    pos=strpos(ofilename[ii],sfx,/reverse_search)
    ofsfx[ii]=pos eq -1L?'':strmid(ofilename[ii],pos)
    ofilename[ii]=pos eq -1L?ofilename[ii]:strmid(ofilename[ii],0L,pos)
  endfor ;; ii=0L,loop

  odata=temporary(comim)
  dsh=1L

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
            verbose=verbose,/error)
  return
END ;; procedure: p3d_misc_imcombine_wrapper
