;+
; NAME:
;         p3d_misc_detsec
;
;         $Id: p3d_misc_detsec.pro 115 2010-03-19 12:17:32Z christersandin $
;
; PURPOSE:
;         This routine retrieves a four-element array (DETSEC) that contains
;         the array indices of the CCD data-area (i.e. to remove prescan- and
;         overscan-regions). There are four ways to set DETSEC, in order of
;         priority they are:
;
;          1.:: DETSEC is specified in the USERPARFILE (see below).
;          2.:: DETSEC is read from the object data header, if it exists. 
;               In order to use this function the proper DETSEC must be 
;               defined in the instrument keywords file.
;          3.:: DETSEC is read from the object data header as separate 
;               numbers, if they exist. In order to use this function the 
;               proper keywords must be specified in the instrument 
;               keywords file (i.e. DETSEC_PX,DETSEC_OX,DETSEC_PY,DETSEC_OY).
;          4.:: DETSEC is read from the instrument parameter file.
;
;         The format of DETSEC is: [x_start:x_end,y_start:y_end]
;             x_start :: used region, the first pixel in x
;             x_end   :: used region, the  last pixel in x
;             y_start :: used region, the first pixel in y
;             y_end   :: used region, the  last pixel in y
;         The first pixel is specified as 1 (despite IDL using 0-based
;         indices).
;
;         If the input parameter HDR contains headers of several files (number
;         of rows>1) or if HDR contains the keywords CETSECx (where x is the
;         block number) then DETSEC is an array with 4 columns and as many rows
;         as there are file headers.
;
;         In the case that HDR contains several rows DETSEC must be specified
;         as DETSEC_1, DETSEC_2, ..., for each case described above.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_detsec,hdr,kwrdlist,parfile,detsec,odetsec,userparfile=, $
;             detector=,topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         hdr             - An string array with the fits-file header of the
;                           file that should be parsed for the detector area.
;         kwrdlist        - A scalar string with the name of a file, that
;                           contains a two-column list of p3d-names and
;                           instrument-specific names for fits-header keywords.
;         parfile         - A scalar string with the name of a file, that
;                           contains a two-column list of instrument-specific
;                           parameters.
;
; KEYWORD PARAMETERS:
;         userparfile     - A scalar string specifying the name of an optional
;                           user parameter file, that could contain the keyword
;                           'detsec'. If it does then the value of that keyword
;                           is used to trim the data. If there are several
;                           'detsec'-lines in the file then the first is used.
;         detector [0]    - A scalar integer that specifies the currently
;                           selected detector; DETECTOR is a zero-based value.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         detsec          - A four-element integer array that specifies the
;                           detector region to use on the CCD. If HDR contains
;                           several rows then DETSEC will have as many rows.
;         odetsec         - Like DETSEC, but this array has not been flipped on
;                           the dispersion axis.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         09.07.2009 - New routine (extracted from
;                      p3d_extracted_prepare_extraction). /CS
;-
FUNCTION p3d_misc_detsec_str2val,str,nhdr,string
  compile_opt hidden,IDL2

  if ~n_elements(string) then string='DETSEC'

  ;;========================================------------------------------
  ;; Removing leading and trailing brackets in DETSRC, if they exist:

  dstr=str
  for i=0L,nhdr-1L do begin
    removedlead=0L & tmp=strpos(str[i],'[')
    if tmp ne -1L then begin
      str[i]=strmid(str[i],tmp+1L)
      removedlead=1L
    endif

    removedtrai=0L & tmp=strpos(str[i],']',/reverse_search)
    if tmp ne -1L then begin
      str[i]=strmid(str[i],0L,tmp)
      removedtrai=1L
    endif

    if ~removedlead or ~removedtrai then begin
      tmpstr=(nhdr eq 1L)?'':('_'+strtrim(i+1L,2L))
      errmsg=string+tmpstr+' {'+dstr+'} is improperly formatted; should' + $
             ' be '+string+tmpstr+'=[x_start:x_end,y_start:y_end].'
      goto,error_handler
    endif
  endfor ;; i=0L,nhdr-1L

  ;;========================================------------------------------
  ;; Extracting the numbers:

  out=lonarr(4L,nhdr)
  for i=0L,nhdr-1L do begin
    tmpstr=nhdr eq 1L?'':'_'+strtrim(i+1L,2L)
    tmp=strsplit(str[i],':,',/extract)

    on_ioerror,format_error
    tmp=long(tmp)
    on_ioerror,NULL

    ;;========================================------------------------------
    ;; Checking the numbers for consistency:

    if n_elements(tmp) ne 4L then begin
      errmsg=string+tmpstr+' must have the following format: [x_start:x_' + $
             'end,y_start:y_end].'
      goto,error_handler
    endif

    if ~i and nhdr eq 1L then begin
      if tmp[1L] le tmp[0L] or tmp[3L] le tmp[2L] then begin
        errmsg=[string+tmpstr+' must have the following format: [x_start' + $
                ':x_end,y_start:y_end]',' where x_end>x_start, y_end>y_sta' + $
                'rt. '+string+'=['+str+'].']
        goto,error_handler
      endif
    endif

    out[*,i]=tmp
  endfor ;; i=0L,nhdr-1L

  return,reform(out)

format_error:
  errmsg=detsrc+' Error converting '+string+' into an array of longs; ' + $
         string+tmpstr+'='+tmp+'.'

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return,error
END ;;; function: p3d_misc_detsec_str2val


PRO p3d_misc_detsec,hdr,kwrdlist,parfile,detsec,odetsec, $
        userparfile=userparfile,detector=d,topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_misc_detsec: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_misc_detsec'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(hdr)
  if s[s[0L]+1L] ne 7L or ~s[s[0L]+2L] then begin
    errmsg='HDR [1] must be set to a string array.'
    goto,error_handler
  endif
  nhdr=s[0L] eq 1L?1L:s[2L]

  s=size(kwrdlist)
  if s[s[0L]+1L] ne 7L or ~s[s[0L]+2L] then begin
    errmsg='KWRDLIST [2] must be set to the name of an instrument-specific' + $
           ' keywords-file.'
    goto,error_handler
  endif
  if ~file_test(kwrdlist,/read,/regular) then begin
    errmsg='KWRDLIST [2], cannot read the file "'+kwrdlist+'".'
    goto,error_handler
  endif

  s=size(parfile)
  if s[s[0L]+1L] ne 7L or ~s[s[0L]+2L] then begin
    errmsg='PARFILE [2] must be set to the name of an instrument-specific' + $
           ' parameter file.'
    goto,error_handler
  endif
  if ~file_test(kwrdlist,/read,/regular) then begin
    errmsg='PARFILE [2], cannot read the file "'+parfile+'".'
    goto,error_handler
  endif

  useuserpar=0L
  if n_elements(userparfile) ne 0L then begin
    s=size(userparfile)
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='USERPARFILE, if set, must be a scalar string.'
      goto,error_handler
    endif
    if userparfile ne '' then begin
      if ~file_test(userparfile,/regular,/read) then begin
        errmsg='USERPARFILE, cannot read the file "'+userparfile+'".'
        goto,error_handler
      endif

      readcol,userparfile,uparname,uparvalue,format='a,a',comment=';', $
          silent=verbose lt 3,delimiter=' '

      useuserpar=1L
    endif
  endif ;; n_elements(userparfile) ne 0L

  if ~n_elements(d) then d=0L
  s=size(d)
  if s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L then begin
    errmsg='DETECTOR must be a scalar integer; DETECTOR>=0.'
    goto,error_handler
  endif
  if d lt 0L then begin
    errmsg='DETECTOR must be a scalar integer; DETECTOR>=0.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Reading various instrument-specific parameters:

  ppath=strmid(parfile,0L,strpos(parfile,path_sep(),/reverse_search)+1L)
  parname='' & parvalue=''
  readcol,parfile,parname,parvalue,format='a,a',delimiter=' ', $
      silent=verbose lt 3,comment=';'

  p3d_misc_read_params,parname,parvalue,'daxis',daxis,/must_exist,/nou, $
      type='integer',topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,error_handler
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2 (not '+strtrim(daxis,2L)+').'
    goto,error_handler
  endif

  p3d_misc_read_params,parname,parvalue,'ndetector',ndetector,/nou, $
      type='integer',topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(ndetector) then ndetector=1L ;; default is one detector

  xstr=ndetector gt 1L?'_'+strtrim(d+1L,2L):''
  p3d_misc_read_params,parname,parvalue,'dflip'+xstr,dflip, $
      uparname=uparname,uparvalue=uparvalue, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,error_handler
  if ~n_elements(dflip) then dflip='no'
  dflip=strlowcase(dflip)
  if dflip ne 'yes' and dflip ne 'no' then begin
    errmsg='DFLIP must be a scalar string; yes||no.'
    goto,error_handler
  endif
  dflip=dflip eq 'yes'?1L:0L

  ;;========================================------------------------------
  ;; Retreiving the CCD binning parameters:

  thdr=nhdr eq 1L?hdr:hdr[*,0L]
  p3d_misc_binextract,kwrdlist,thdr,xbin=xbin,ybin=ybin,topwid=topwid, $
      logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return

  ;;========================================------------------------------
  ;; Reading the value of DETSEC:

  gotstr=0L
  bindivide=1L
  flipdata=dflip

  ;; 0) Attempting to read CETSEC (from a file that has been combined from
  ;;    several files):

  if nhdr eq 1L then begin
    str=fxpar(hdr,'CETSEC1')
    if str ne '0' then begin
      notready=1L & i=1L
      str=[strtrim(str,2L)]
      while notready do begin
        tmp=fxpar(hdr,'CETSEC'+strtrim(i+1L,2L))
        if tmp ne '0' then begin
          str=[str,strtrim(tmp,2L)]
          i++
        endif else notready=0L
      endwhile ;; notready
      gotstr=1L
      detsrc='[combined data file]'
      nhdr=i
      flipdata=0L
      bindivide=0L
    endif ;; str ne '0'
    ;; Should maybe check this better...accounting for non-multi-block CCDs

  endif ;; nhdr eq 1L

  ;; i) Using information in the user parameter file, if it exists:
  if ~gotstr and useuserpar then begin
    readcol,userparfile,uparname,uparvalue,format='a,a',comment=';', $
        silent=verbose lt 3,delimiter=' '
    if n_elements(uparname) ne 0L then begin
      idx=strpos(strlowcase(uparname),'detsec')
      pos=where(idx ge 0L,count)
      if count ge 1L then begin
        case count of
          1L: begin
            str=strtrim(uparvalue[pos[0L]],2L)
            gotstr=1L
            detsrc='[userparfile]'
          end
          else: begin
            str=strarr(nhdr)
            for i=1L,nhdr do begin
              tmp=where(uparname[pos] eq 'detsec_'+strtrim(i,2L),count)
              if ~count then begin
                errmsg='If there are several detector blocks then DETSEC ' + $
                       'must be specified for all blocks; cannot find DET' + $
                       'SEC_'+strtrim(i,2L)+'.'
                goto,error_handler
              endif
              str[i-1L]=strtrim(uparvalue[tmp],2L)
              gotstr=1L
              detsrc='[userparfile]'
            endfor ;; i=1L,nhdr
          end
        endcase ;; count
      endif ;; count ge 1L
    endif ;; n_elements(uparname) ne 0L
  endif ;; useuserpar

  ;; ii) Attempting to read the file header:
  if ~gotstr then begin
    tmp=p3d_misc_get_hdr_kwrd(kwrdlist,'DETSEC',topwid=topwid, $
            logunit=logunit,verbose=verbose,error=error,debug=debug)
    if error ne 0 then return

    if tmp ne '-1' then begin
      case nhdr of
        1L: begin
          str=fxpar(hdr,tmp)
          if str ne '0' then begin
            gotstr=1L
            detsrc='[data header]'
          endif
        end
        else: begin
          str=strarr(nhdr) & gotstr=lonarr(nhdr)
          for i=0L,nhdr-1L do begin
            str[i]=strtrim(fxpar(reform(hdr[*,i]),tmp),2L)
            if str[i] ne '0' then gotstr[i]=1L
          endfor
          gotstr=total(gotstr) eq nhdr?1L:0L
          if gotstr then detsrc='[data header]'
        end
      endcase ;; nhdr
    endif
  endif ;; ~gotstr

  if ~gotstr then begin
    ;; iii) Attempting to read DETSEC with all keywords provided separately:
    tmp=p3d_misc_get_hdr_kwrd(kwrdlist,'DETSEC_PX',topwid=topwid, $
            logunit=logunit,verbose=verbose,error=error,debug=debug)
    if error ne 0 then return
    str1=lonarr(nhdr)
    if tmp ne '-1' then begin
      case nhdr of
        1L: begin
          str1[0L]=p3d_misc_fitsstrex(hdr,tmp,topwid=topwid, $
              logunit=logunit,verbose=verbose,error=error,debug=debug)
          if error ne 0 then return
        end
        else: begin 
          for i=0L,nhdr-1L do begin
            str1[i]=p3d_misc_fitsstrex(hdr[i],tmp,topwid=topwid, $
                logunit=logunit,verbose=verbose,error=error,debug=debug)
            if error ne 0 then return
          endfor
        end
      endcase ;; nhdr
    endif ;; tmp ne '-1'

    tmp=p3d_misc_get_hdr_kwrd(kwrdlist,'DETSEC_OX',topwid=topwid, $
            logunit=logunit,verbose=verbose,error=error,debug=debug)
    if error ne 0 then return
    str2=lonarr(nhdr)
    if tmp ne '-1' then begin
      case nhdr of
        1L: begin
          str2[0L]=p3d_misc_fitsstrex(hdr,tmp,topwid=topwid, $
              logunit=logunit,verbose=verbose,error=error,debug=debug)
          if error ne 0 then return
        end
        else: begin 
          for i=0L,nhdr-1L do begin
            str2[i]=p3d_misc_fitsstrex(hdr[i],tmp,topwid=topwid, $
                logunit=logunit,verbose=verbose,error=error,debug=debug)
            if error ne 0 then return
          endfor
        end
      endcase ;; nhdr
    endif ;; tmp ne '-1'

    tmp=p3d_misc_get_hdr_kwrd(kwrdlist,'DETSEC_PY',topwid=topwid, $
            logunit=logunit,verbose=verbose,error=error,debug=debug)
    if error ne 0 then return
    str3=lonarr(nhdr)
    if tmp ne '-1' then begin
      case nhdr of
        1L: begin
          str3[0L]=p3d_misc_fitsstrex(hdr,tmp,topwid=topwid, $
              logunit=logunit,verbose=verbose,error=error,debug=debug)
          if error ne 0 then return
        end
        else: begin 
          for i=0L,nhdr-1L do begin
            str3[i]=p3d_misc_fitsstrex(hdr[i],tmp,topwid=topwid, $
                logunit=logunit,verbose=verbose,error=error,debug=debug)
            if error ne 0 then return
          endfor
        end
      endcase ;; nhdr
    endif ;; tmp ne '-1'

    tmp=p3d_misc_get_hdr_kwrd(kwrdlist,'DETSEC_OY',topwid=topwid, $
            logunit=logunit,verbose=verbose,error=error,debug=debug)
    if error ne 0 then return
    str4=lonarr(nhdr)
    if tmp ne '-1' then begin
      case nhdr of
        1L: begin
          str4[0L]=p3d_misc_fitsstrex(hdr,tmp,topwid=topwid, $
              logunit=logunit,verbose=verbose,error=error,debug=debug)
          if error ne 0 then return
        end
        else: begin 
          for i=0L,nhdr-1L do begin
            str4[i]=p3d_misc_fitsstrex(hdr[i],tmp,topwid=topwid, $
                logunit=logunit,verbose=verbose,error=error,debug=debug)
            if error ne 0 then return
          endfor
        end
      endcase ;; nhdr
    endif ;; tmp ne '-1'

    case nhdr of
      1: begin
        if str1 ne 0L and str2 ne 0L and str3 ne 0L and str4 ne 0L then begin
          tmpx=long(fxpar(hdr,'NAXIS1'))
          tmpy=long(fxpar(hdr,'NAXIS2'))

          gotstr=1L
          detsrc='[data header {sep}]'
          detsec='['+strtrim(str1,2L)+':'+strtrim(tmpx-str2,2L)+','+ $
                     strtrim(str3,2L)+':'+strtrim(tmpy-str4,2L)+']'
        endif
      end
      else: begin
        gotstr=lonarr(nhdr)
        detsec=strarr(nhdr)
        for i=0L,nhdr-1L do begin
          if str1[i] ne 0L and str2[i] ne 0L and $
             str3[i] ne 0L and str4[i] ne 0L then begin
            tmpx=long(fxpar(hdr[i],'NAXIS1'))
            tmpy=long(fxpar(hdr[i],'NAXIS2'))

            gotstr[i]=1L
            detsec[i]='['+strtrim(str1[i],2L)+':'+strtrim(tmpx-str2[i],2L)+ $
                      ','+strtrim(str3[i],2L)+':'+strtrim(tmpy-str4[i],2L)+']'
          endif
        endfor ;; i=0L,nhdr-1L
        if total(gotstr) eq nhdr then begin
          detsrc='[data header {sep}]'
          gotstr=1L
        endif else gotstr=0L
      end
    endcase ;; nhdr
  endif ;; ~gotstr

  ;; iv) Reading the default value from the instrument parameter file:
  if ~gotstr then begin
    if n_elements(parname) ne 0L then begin
      idx=strpos(strlowcase(parname),'detsec')
      pos=where(idx ge 0L,count)
      if count ge 1L then begin
        case count of
          1L: begin
            str=strtrim(parvalue[pos[0L]],2L)
            detsrc='[instrument default]'
            gotstr=1L
          end
          else: begin
            if count ne nhdr then begin
              errmsg='If there are more than one DETSEC entries in the par' + $
                     'ameter file then the number must equal '+ $
                     strtrim(nhdr,2L)+', '+strtrim(count,2L)+ $
                     ' entries is not ok.'
              goto,error_handler
            endif
            str=strarr(nhdr)
            for i=1L,nhdr do begin
              tmp=where(parname[pos] eq 'detsec_'+strtrim(i,2L),count)
              if ~count then begin
                errmsg='If there are several detector blocks then DETSEC ' + $
                       'must be specified for all blocks; cannot find DET' + $
                       'SEC_'+strtrim(i,2L)+'.'
                goto,error_handler
              endif
              str[i-1L]=strtrim(parvalue[pos[tmp]],2L)
              detsrc='[instrument default]'
              gotstr=1L
            endfor ;; i=1L,nhdr
          end
        endcase ;; count
      endif ;; count ge 1L
    endif ;; n_elements(parname) ne 0L
  endif ;; ~gotstr

  ;; v) Error:
  if ~gotstr then begin
    errmsg='Could not determine a value for DETSEC, cannot continue.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Removing leading and trailing brackets in DETSRC, if they exist.
  ;; Thereafter extracting the numbers in the string:

  detsec=p3d_misc_detsec_str2val(str,nhdr)
  odetsec=detsec

  ;;========================================------------------------------
  ;; Flipping the detsec values on the dispersion axis; if requested to do so:

  if flipdata then begin
    if n_elements(parname) ne 0L then begin
      idx=strpos(strlowcase(parname),'detsize')
      pos=where(idx ge 0L,count)
      case count of
        1L: begin
          str=strtrim(parvalue[pos[0L]],2L)
          detsrc='[instrument default]'
          gotstr=1L
        end
        else: begin
          errmsg='Could not find the detsize parameter in the instrument' + $
                 ' parameter file. It is required.'
          goto,error_handler
        end ;; else
      endcase ;; count
    endif ;; n_elements(parname) ne 0L
    detsize=p3d_misc_detsec_str2val(str,1L,'DETSIZE')
    detsec_=detsec
    nblocks=(size(detsec,/dimensions))[1L]

    ;; Swapping the x_start and x_end values if there is only one
    ;; block on the dispersion axis:
    if daxis eq 1L then begin
      if min(detsec[0L:1L,*]) eq min(detsec[0L:1L,0L]) and $
         max(detsec[0L:1L,*]) eq max(detsec[0L:1L,0L]) then begin
        ;; One block on the dispersion axis:
        detsec_[0L:1L,*]=detsize[1L]-detsec[0L:1L,*]+1L
        detsec=detsec_

        tmp=reverse(detsec[0L:1L,*],1L)
        detsec[0L:1L,*]=temporary(tmp)
      endif else begin
        ;; Two blocks on the dispersion axis:
        xmin1=min(detsec[0L:1L,*])
        a=detsec[0L:1L,*] & a=a[sort(a)] & a=a[uniq(a)] & xmin2=a[2L]-a[1L]
        for j=0L,nblocks-1L do begin
          if min(detsec[0L:1L,j]) eq xmin1 then begin ;; lower column:
            detsec_[0L:1L,j]=detsize[1L]-detsec[0L:1L,j]+xmin1
          endif else begin                            ;; upper column:
            detsec_[0L:1L,j]=detsize[1L]-detsec[0L:1L,j]+xmin2
          endelse ;; min(detsec[0L:1L,j]) eq xmin
        endfor ;; j=0L,nblocks-1L
        detsec=detsec_
      endelse
    endif else begin
      if min(detsec[2L:3L,*]) eq min(detsec[2L:3L,0L]) and $
         max(detsec[2L:3L,*]) eq max(detsec[2L:3L,0L]) then begin
        ;; One block on the dispersion axis:
        detsec_[2L:3L,*]=detsize[3L]-detsec[2L:3L,*]+1L
        detsec=detsec_

        tmp=reverse(detsec[2L:3L,*],1L)
        detsec[2L:3L,*]=temporary(tmp)
      endif else begin
        ;; Two blocks on the dispersion axis:
        xmin1=min(detsec[2L:3L,*])
        a=detsec[2L:3L,*] & a=a[sort(a)] & a=a[uniq(a)] & xmin2=a[2L]-a[1L]
        for j=0L,nblocks-1L do begin
          if min(detsec[2L:3L,j]) eq xmin1 then begin ;; lower column:
            detsec_[2L:3L,j]=detsize[3L]-detsec[2L:3L,j]+xmin1
          endif else begin                            ;; upper column:
            detsec_[2L:3L,j]=detsize[3L]-detsec[2L:3L,j]+xmin2
          endelse ;; min(detsec[0L:1L,j]) eq xmin
        endfor ;; j=0L,nblocks-1L
        detsec=detsec_
      endelse
    endelse
  endif ;; flipdata

  ;;========================================------------------------------
  ;; Dividing by the binning factors:

   detsec=double( detsec)
  odetsec=double(odetsec)
  if bindivide then begin
     detsec[0L:1L,*]/=double(xbin) &  detsec[2L:3L,*]/=double(ybin)
    odetsec[0L:1L,*]/=double(xbin) & odetsec[2L:3L,*]/=double(ybin)
  endif
   detsec=ceil( detsec)-1L
  odetsec=ceil(odetsec)-1L

  ;; Logging the result:
  case nhdr of
    1L: begin
      msg='Using DETSEC=['+ $
          strtrim(detsec[0L]+1L,2L)+':'+strtrim(detsec[1L]+1L,2L)+','+ $
          strtrim(detsec[2L]+1L,2L)+':'+strtrim(detsec[3L]+1L,2L)+ $
          '] '+detsrc+', xbin,ybin='+strtrim(xbin,2L)+','+strtrim(ybin,2L)+'.'
      error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
                            verbose=verbose ge 1L)
    end
    else: begin
      msg='Using '
      for i=0L,nhdr-1L do begin
        tmpstr='_'+strtrim(i+1L,2L)
        msg=[msg,'  DETSEC'+tmpstr+'=['+ $
             strtrim(detsec[0L,i]+1L,2L)+':'+ $
             strtrim(detsec[1L,i]+1L,2L)+','+ $
             strtrim(detsec[2L,i]+1L,2L)+':'+ $
             strtrim(detsec[3L,i]+1L,2L)+']']
      endfor
      msg=[msg,'  '+detsrc+', xbin,ybin='+strtrim(xbin,2L)+','+ $
           strtrim(ybin,2L)+'.']
      error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
          verbose=verbose ge 1L)
    end
  endcase ;; nhdr

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;; procedure: p3d_misc_detsec
