;+
; NAME:
;         p3d_wavecal_set_linelist
;
;         $Id: p3d_wavecal_set_linelist.pro 135 2010-03-27 09:52:39Z christersandin $
;
; PURPOSE:
;         The purpose of this routine is to provide linelist filenames for
;         wavelength calibration. Since different instruments have different
;         ways to setup calibration lines this routine allows the use
;         of four different methods:
;           0) In the first method a user parameter file is used to define the
;              linelist files to use (see below). Any number of linelist files
;              can be specified with rows which begin with the string
;              'arclinefile'. If 0 or fewer such filenames are specified then
;              the routine attempts to use method 1. If the p3d option to save
;              output data in a separate directory is not active the full path
;              must be specified for every entry. Otherwise it is assumed that
;              the linelist file is found in the output data directory.
;           1) In the second method the instrument data keywords list is
;              searched for the three keywords DETECTOR, FILTERNAME, and
;              GRATNAME. If all these keywords are present then a string is
;              created as follows:
;                GNAME=h[GRATNAME]+'_'+h[FILTERNAME]+'_'+h[DETECTOR]
;              where h[] denotes the value as is found in the data header. In a
;              last step GNAME is searched for in the instrument lamps file;
;              the linelist <filename> is extracted from the second column in
;              that file.
;           2) The third method is fairly specific to PMAS. In this approach
;              the data header is searched for information on which shutters
;              and power supplies are used, the header keywords are taken from
;              the instrument lamps file. The linelists which will be used
;              depend on which power supplies are switched on and which
;              shutters are open (several can be chosen). The last column in
;              the instruments lamps file contains the <filename> that will be
;              used.
;           3) In the fourth, and final, method linelist files are selected
;              manually, in a graphical widget. Among the options these
;              linelist <filenames> are found:
;                HG (Mercury), NE (Neon), CD (Cadmium), AR (Argon), 
;                ThAr (Thorium Argon), user defined.
;
;           For methods 1-3 the final filenames are created as follows:
;            linelist file(s) = !p3d_path/data/tables/<filename>.dat
;
;           Current instruments tend to use the following setups:
;             Method 0: Any instrument
;             Method 1: VIMOS
;             Method 2: PMAS/LARR
;             Method 3: PMAS/PPAK, VIRUS-P
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: wavelength calibration
;
; CALLING SEQUENCE:
;         p3d_wavecal_set_linelist,filename,lfilename,linelists,lamp, $
;             kwrdlist=,userparfile=,prompt=,/header,/no_header,continuum=, $
;             /wsep,topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         filename        - If HEADER==1 then this should be the filename of
;                           the input PMAS fits file. If HEADER==0 then this is
;                           a FITS header string array.
;         lfilename       - A scalar string with the filename of a file that
;                           contains the calibration lamp header keywords which
;                           are used with the current instrument and data.
;
;               The calibration lamp keyword file format is: 
;                         o Lines which begin with the character ';' are
;                             considered comments. Columns are separated by
;                             white space.
;              [Method 1] o For every combination of grisms/gratings, filters,
;                           and detector-id, the first column specifies such a
;                           string. The second column contains the linelist
;                           filename to use with that combination.
;              [Method 2] o For every calibration line the first column speci-
;                           fies the shutter keyword. The second column
;                           specifies the power supply. The third and final
;                           column specifies the name of the calibration lamp
;                           (and this string is also used to create the
;                           filename). All data is read as strings.
;
; KEYWORD PARAMETERS:
;         kwrdlist        - A two-dimensional string array holding the
;                           instrument-specific keywords for a defined set of
;                           keywords which are used to determine the
;                           grating/grism setup.
;                           KWRDLIST must have two columns.
;         userparfile     - A scalar string specifying the name of an optional
;                           user parameter file, which contains keywords, be-
;                           ginning with the string 'arclinefile'. The keywords
;                           must point at existing files which, if present,
;                           will be used instead of any header keywords. As in
;                           the other parameter files lines preceeded by ';'
;                           are considered to be comments. Entries of multiple
;                           files are concatenated to create one linelist.
;         prompt          - Prompt the user for the lamps to use.
;         header          - Set this keyword if FILENAME is actually a variable
;                           that contains the header keywords.
;         no_header       - Specified if the necessary information is
;                           not contained in the FITS header. In this
;                           case the information has to be entered at
;                           the prompt.
;         continuum ['HL'] - A scalar string that specifies the name of the
;                           continuum lamp of the current instrument.
;         wsep            - If this keyword is set,then the directory pointed
;                           at by this variable is used to load data.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         linelists       - Filenames of emission line lists.
;         lamp            - Used emission line lamps.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         05.09.2008 - Converted from original routine set_linelist.pro of
;                      Thomas Becker. /CS
;         11.05.2009 - Restructured code to be more general.
;-
PRO p3d_wavecal_set_linelist_event,event
  compile_opt hidden,IDL2

  rname='p3d_wavecal_set_linelist_event: '

  ;;========================================------------------------------
  ;; Get the state information:

  widget_control,event.top,get_uvalue=state

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=event.top
      catch,/cancel
      return
    endif
  endif ;; ~(*state).debug

  ;;========================================------------------------------
  ;; Handling tracking events first:

  tmpstrname=strlowcase(tag_names(event,/structure_name))
  if tmpstrname eq 'widget_tracking' then begin
    if event.enter then begin  
      infostr=widget_info(event.id,/uname)
      widget_control,(*state).lstatwid,set_value=infostr
    endif else widget_control,(*state).lstatwid,set_value=''

    ;; Nothing more to do, returning:
    return
 endif

  ;;========================================------------------------------
  ;; Get the user value of the event structure:

  widget_control,event.id,get_uvalue=uval;,get_value=value
  s=size(uval)

  index=''
  on_ioerror,noindexed_event
  tmppos=strpos(uval,'_')
  if tmppos ne -1L then begin
    index=strmid(uval,tmppos+1L)
    uval=strmid(uval,0L,tmppos)
  endif
  noindexed_event:
  on_ioerror,NULL

  case uval of
    'buttonsel': begin
      case strlowcase(index) of
        'hg':     (*state).hg    =event.select
        'ne':     (*state).neon  =event.select
        'cd':     (*state).cadmium=event.select
        'ar':     (*state).argon =event.select
        'thar':   (*state).thar  =event.select
        'spiral_ar': (*state).aao_ar=event.select
        'spiral_cu': (*state).aao_cu=event.select
        'spiral_fe': (*state).aao_fe=event.select
        'spiral_he': (*state).aao_he=event.select
        'spiral_ne': (*state).aao_ne=event.select
      endcase
    end

    'userselect': begin
      widget_control,(*state).ttextwid,get_value=tmp
      nfiles=1L
      if strpos(tmp,' ') ne -1L or strpos(tmp,';') ne -1L or $
         strpos(tmp,',') ne -1L or strpos(tmp,':') ne -1L then $
        tmp=strsplit(tmp,', ;:',count=nfiles)
      rfiles=lonarr(nfiles)

      for i=0L,nfiles-1L do $
         if file_test(tmp[i],/read,/regular) then rfiles[i]=1L

      if total(rfiles) ge 1L then begin
        (*state).nfile=total(rfiles)
        ptr_free,(*state).ufile
        idx=where(rfiles)
        (*state).ufile=ptr_new(tmp[rfiles])
        (*state).cancel=0L
      endif else begin
        msg='Cannot find the specified file(s).'
        widget_control,(*state).lstatwid,set_value=tmp
      endelse
    end ;; case: 'userselect'

    'userbrowse': begin
      widget_control,(*state).ttextwid,get_value=tmp

      if tmp[0] eq '' then begin
         path=(*state).wsepdir ne ''?(*state).wsepdir:!p3d_data_path
      endif else begin
         path=file_dirname((*(*state).ufile)[0L],/mark_directory)
      endelse
      ret=dialog_pickfile(title='Please select (a) linelist file(s).', $
              file=tmp,path=path,/must_exist,dialog_parent=event.top, $
              /multiple_files)

      if ret[0L] ne '' then begin
         nfiles=1L
         (*state).nfile=n_elements(ret)

        if ptr_valid((*state).ufile) then ptr_free,(*state).ufile
        (*state).ufile=ptr_new(ret)
        widget_control,(*state).ttextwid,set_value=strjoin(ret,' ')
      endif else (*state).nfile=0L
    end ;; case: 'userbrowse'

    'ok': begin
       (*state).cancel=0L
       widget_control,event.top,/destroy
    end

    'cancel': begin
       (*state).cancel=1L
       widget_control,event.top,/destroy
    end

  endcase ;; uval

  return
END ;;; procedure: p3d_wavecal_set_linelist_event


PRO p3d_wavecal_set_linelist,filename,lfilename,linelists,lamp, $
        kwrdlist=kwrdlist,userparfile=userparfile,prompt=prompt, $
        header=header,no_header=no_header,continuum=continuum_lamp, $
        wsep=wsep,topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_wavecal_set_linelist: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)
  if ~n_elements(topwid) then topwid=0L

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_wavecal_set_linelist'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  useuserpar=0
  if n_elements(userparfile) ne 0L then begin
    s=size(userparfile)
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='USERPARFILE, if set, must be a scalar string.'
      goto,error_handler
    endif
    if userparfile ne '' then begin
      if ~file_test(userparfile,/regular,/read) then begin
        errmsg='The file USERPARFILE "'+userparfile+'" does not exist.'
        goto,error_handler
      endif
      useuserpar=1
    endif
  endif ;; n_elements(userparfile) ne 0L

  if ~n_elements(continuum_lamp) then continuum_lamp='hl'
  s=size(continuum_lamp)
  if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
    errmsg='CONTINUUM, if set, must be a scalar string.'
    goto,error_handler
  endif
  continuum_lamp=strlowcase(continuum_lamp)

  wsepdir=''
  if size(wsep,/type) eq 7L then begin
    if file_test(wsep,/directory) then wsepdir=wsep
  endif else if n_elements(wsep) ne 0L then begin
    errmsg='WSEP must, if set, be a scalar string.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;;========================================------------------------------
  ;;========================================------------------------------
  ;; Determining the arc line list filenames:

  ;;========================================------------------------------
  ;; Method 0: Using information in the user parameter file:

  if useuserpar then begin
    readcol,userparfile,parname,parvalue,format='a,a',comment=';', $
            silent=verbose lt 3,delimiter=' '
    if n_elements(parname) ne 0L then begin
       idx=strpos(strlowcase(parname),'arclinefile')
       tmp=where(idx ge 0L,count)
       if count ge 1L then begin
         method='0'
         idx=idx[tmp]
         linelists=strtrim(parvalue[tmp],2L)
         path=file_dirname(userparfile,/mark_directory)
         for i=0L,n_elements(linelists)-1L do begin
           if ~file_test(linelists[i],/regular,/read) then begin
             if ~file_test(path+linelists[i],/regular,/read) then begin
               errmsg='Cannot find the ARCLINEFILE "'+linelists[i]+'".'
               goto,error_handler
             endif ;; ~file_test(path+linelists[i],/regular,/read)
             linelists[i]=path+linelists[i]
           endif ;; ~file_test(linelists[i],/regular,/read)
         endfor ;; i=0L,n_elements(linelists)-1L

         lamp=linelists
       endif ;; idx[0L] ne -1L
    endif ;; n_elements(parname) ne 0L
  endif ;; useuserpar

  ;;====================----------
  ;; Using information in the data header:

  if ~n_elements(linelists) then begin
    if ~keyword_set(no_header) then begin

      s=size(kwrdlist)
      if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
        errmsg='KWRDLIST must be se to a scalar string pointing to an' + $
               ' existing file with instrument header keywords.'
        goto,error_handler
      endif
      if ~file_test(kwrdlist,/regular,/read) then begin
        errmsg='The file "'+kwrdlist+'" cannot be found.'
        goto,error_handler
      endif

      ;; Reading the data header that is used to extract information about
      ;; which arc lamps are in use:
      if keyword_set(header) then begin
        hdr=filename
      endif else begin
        s=size(filename)
        if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
          errmsg='FILENAME must be present; and be a scalar of string type.'
          goto,error_handler
        endif
        if ~file_test(filename,/regular,/read) then begin
          errmsg='The file FILENAME "'+filename+'" does not exist.'
          goto,error_handler
        endif
        hdr=headfits(filename)
      endelse

      ;;========================================------------------------------
      ;; Method 1 :: Attempting to read grism and filter names in the data
      ;;             header:

      usemethod1=0L
      tmp2=p3d_misc_get_hdr_kwrd(kwrdlist,  'DETECTOR',topwid=topwid, $
               logunit=logunit,verbose=verbose,error=error,debug=debug)
      if error ne 0 then return
      tmp1=p3d_misc_get_hdr_kwrd(kwrdlist,'FILTERNAME',topwid=topwid, $
               logunit=logunit,verbose=verbose,error=error,debug=debug)
      if error ne 0 then return
      tmp3=p3d_misc_get_hdr_kwrd(kwrdlist,  'GRATNAME',topwid=topwid, $
               logunit=logunit,verbose=verbose,error=error,debug=debug)
      if error ne 0 then return

      if tmp1 ne '-1' and tmp2 ne '-1' and tmp3 ne '-1' then begin
        method='1'
        quadrant=p3d_misc_fitsstrex(hdr,tmp2,topwid=topwid, $
            logunit=logunit,verbose=verbose,error=error,debug=debug)
        if error ne 0 then return

        ;; Setting up the current filter name keyword (if required):
        tmp=strpos(tmp1,'@')
        if tmp ne -1L then tmp1=strmid(tmp1,0L,tmp)+ $
                                    strtrim(quadrant,2L)+strmid(tmp1,1L+tmp)
        tmp=strpos(tmp3,'@')
        if tmp ne -1L then tmp3=strmid(tmp3,0L,tmp)+ $
                                    strtrim(quadrant,2L)+strmid(tmp3,1L+tmp)

        filtname=p3d_misc_fitsstrex(hdr,tmp1,topwid=topwid, $
                     logunit=logunit,verbose=verbose,error=error,debug=debug)
        if error ne 0 then return

        gratname=p3d_misc_fitsstrex(hdr,tmp3,topwid=topwid, $
                     logunit=logunit,verbose=verbose,error=error,debug=debug)
        if error ne 0 then return

        ;; Reading the linelist filenames:
        s=size(lfilename)
        if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
          errmsg='LFILENAME must be present; as a scalar string.'
          goto,error_handler
        endif
        if ~file_test(lfilename,/regular,/read) then begin
          errmsg='The file with the name LFILENAME does not exist.'
          goto,error_handler
        endif

        ;; Reading the strings containing names for: shutters, power supplies,
        ;; and lamps:
        defstr='[Method 1] Error reading the lamp settings file "'+ $
               lfilename+'".'
        on_ioerror,file_error
        readcol,lfilename,gsetup,lamp,format='a,a',comment=';', $
            silent=verbose lt 3,delimiter=' '
        on_ioerror,NULL ;; turning off the IO error handler

        gname=gratname+'_'+filtname+'_'+strtrim(quadrant,2L)
        idx=where(gsetup eq gname,count)
        if count eq 1L then lamp=lamp[idx]

        usemethod1=1L
      endif

      ;;========================================------------------------------
      ;; Method 2 :: Assuming a PMAS-like setup, with header keywords
      ;;             for used shutters, power supplies and filenames.

      if ~usemethod1 then begin
        method='2'

        s=size(lfilename)
        if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
          errmsg='LFILENAME must be present; as a scalar string.'
          goto,error_handler
        endif
        fexists=file_test(lfilename,/regular,/read)
        if fexists then begin

          ;; Reading the strings containing names for: shutters, power
          ;; supplies, and lamps:
          defstr='[Method 2] Error reading the lamp settings file "'+ $
                 lfilename+'".'
          on_ioerror,file_error

          readcol,lfilename,shutter,power,lamp,format='a,a,a',comment=';', $
              silent=verbose lt 3,delimiter=' '
          on_ioerror,NULL ;; turning off the IO error handler
          n=n_elements(lamp)
          if ~n then begin
            msg='Could not read any lamp header keywords in "'+lfilename+'".'
            error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
                      verbose=verbose,/error)
            return
          endif

          ;; Reading the header keywords and determining the on/off-state of
          ;; each lamp:
          sh=bytarr(n)
          pow=bytarr(n)
          for k=0L,n-1L do begin
            sh[k]=p3d_misc_fitsstrex(hdr,shutter[k],topwid=topwid, $
                      logunit=logunit,verbose=verbose,error=error,debug=debug)
            if error ne 0 then return

            pow[k]=p3d_misc_fitsstrex(hdr,  power[k],topwid=topwid, $
                       logunit=logunit,verbose=verbose,error=error,debug=debug)
            if error ne 0 then return
          endfor
          on=sh*pow

          ison=where(on and (strlowcase(lamp) ne continuum_lamp),count)

          if ~count then begin
            msg='No arc lamp is switched on according to the FITS header!'
            error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
                      verbose=verbose)
          endif
        endif else count=0 ;; fexists
      endif  ;; ~usemethod1

    endif else begin ;; ~keyword_set(no_header)
      ison=-1
      count=0L
    endelse ;; ~keyword_set(no_header)

    ;;========================================------------------------------
    ;; Method 3: Defining the line list filenames by hand:

    if ~count then begin
      if keyword_set(prompt) or keyword_set(no_header) then begin
        method='3'
        name=''
        
        ;; Creating a widget tool to query the user for which linelist
        ;; files should be used:
        bbasewid=widget_base(title=p3d_misc_pathify(filename),/column, $
            /base_align_center,group_leader=topwid,xpad=0L,ypad=0L,space=0L, $
            event_pro='p3d_wavecal_set_linelist_event')

        lbasewid=widget_label(bbasewid, $
            value='Please select the required linelist files')
        value=['Hg','Ne','Cd','Ar','ThAr', $
               'spiral_Ar','spiral_Cu','spiral_Fe','spiral_He','spiral_Ne']
        uvalue='buttonsel_'+value
        uname='Select the '+['Mercury','Neon','Cadmium','Argon', $
              'Thorium Argon','AAO Ar','AAO Cu','AAO Fe','AAO He','AAO Ne']+ $
              ' linelist file.'

        bbutowid=widget_base(bbasewid,column=5L,/base_align_center, $
            /nonexclusive,xpad=0L,ypad=0L,space=0L)
        bmercwid=lonarr(n_elements(value))
        for i=0,n_elements(bmercwid)-1L do begin
          bmercwid[i]=widget_button(bbutowid,value=value[i], $
              uvalue=uvalue[i],/tracking_events,uname=uname[i])
        endfor

        buserwid=widget_base(bbasewid,/row,/base_align_center, $
            xpad=0L,ypad=0L,space=0L)
        lbas2wid=widget_label(buserwid,value='Linelist file(s):')
        ttextwid=widget_text(buserwid,value='',uvalue='userselect',/editable, $
            uname='Select any number of linelist files ('';, :''-separated).',$
            /tracking_events,xsize=60L)
        bmofile=filepath(subdir=['resource','bitmaps'],'open.bmp')
        btexowid=widget_button(buserwid,value=bmofile,uvalue='userbrowse', $
            uname='Click to browse for linelist file(s).',/tracking_events, $
            /bitmap)

        lstatwid=widget_label(bbasewid,/sunken_frame,value='t', $
            uname='Widget tool information line.',/tracking_events)

        bcommwid=widget_base(bbasewid,/row,/base_align_center, $
            xpad=0L,ypad=0L,space=20L)
        bcancwid=widget_button(bcommwid,value=' Cancel ',uvalue='cancel', $
            uname='Press this button when ready.',/tracking_events)
        bdonewid=widget_button(bcommwid,value='  OK  ',uvalue='ok', $
            uname='Press this button when ready.',/tracking_events)

        widget_control,bbasewid,/realize

        ;; Re-sizing the status line widget:
        tmp=widget_info(bbasewid,/geometry)
        widget_control,lstatwid,xsize=tmp.xsize-2L*tmp.xpad
        widget_control,lstatwid,set_value=widget_info(lstatwid,/uname)

        ;; Centering the tool:
        if widget_info(topwid,/valid_id) then begin
          ttp=widget_info(topwid,/geometry)
          xpos=ttp.xoffset+ttp.xsize/2L-tmp.xsize/2L
          ypos=ttp.yoffset+ttp.ysize/2L-tmp.ysize/2L
          widget_control,bbasewid,tlb_set_xoffset=xpos
          widget_control,bbasewid,tlb_set_yoffset=ypos
        endif

        sstate={lstatwid:lstatwid,ttextwid:ttextwid,verbose:verbose, $
                debug:debug,hg:0L,neon:0L,cadmium:0L,argon:0L,thar:0L, $
                aao_ar:0L,aao_cu:0L,aao_fe:0L,aao_he:0L,aao_ne:0L, $
                utext:'',ufile:ptr_new(''),nfile:0L,cancel:1L, $
                wsepdir:wsepdir}

        ;; Storing the state information in the widget tree:
        pstate=ptr_new(sstate,/no_copy)
        widget_control,bbasewid,set_uvalue=pstate

        xmanager,'p3d_wavecal_set_linelist',bbasewid

        ;; Saving the data...
        if (*pstate).cancel then begin
           error=-1
           return
        endif

        lamp=['']
        if (*pstate).hg      then lamp=[lamp,'hg']
        if (*pstate).neon    then lamp=[lamp,'ne']
        if (*pstate).cadmium then lamp=[lamp,'cd']
        if (*pstate).argon   then lamp=[lamp,'ar']
        if (*pstate).thar    then lamp=[lamp,'thar']
        if (*pstate).aao_ar  then lamp=[lamp,'spiral_ar']
        if (*pstate).aao_cu  then lamp=[lamp,'spiral_cu']
        if (*pstate).aao_fe  then lamp=[lamp,'spiral_fe']
        if (*pstate).aao_he  then lamp=[lamp,'spiral_he']
        if (*pstate).aao_ne  then lamp=[lamp,'spiral_ne']
        if n_elements(lamp) ge 2L then lamp=lamp[1L:n_elements(lamp)-1L]

        if (*pstate).nfile ge 1L then begin
          ulamp=[(*(*pstate).ufile)[0L]]
          for i=1L,(*pstate).nfile-1L do ulamp=[ulamp,(*(*pstate).ufile)[i]]
        endif

        ;; Freeing the memory of the state pointer:
        if ptr_valid((*pstate).ufile) then ptr_free,(*pstate).ufile
        ptr_free,pstate

      endif else begin ;; keyword_set(prompt) or no_header
        lamp=[''] & linelists=['']
        error=-1
        return
      endelse ;; keyword_set(prompt) or no_header
    endif else begin ;; ~count

      lampcollection=strtrim(lamp[ison],2L)
      lamp=strlowcase(lampcollection)

    endelse ;; ~count

  endif ;; ~n_elements(linelists)

;    ;; Treating the case of a switched on Thorium-Argon lamp specially (PMAS):
;    pos=where(strupcase(lampcollection) eq 'THAR',count)
;    if count ne 0L then begin
;
;      grat_id=long(strtrim(fxpar(hdr,'GRAT_ID'),2L))
;
;      p3d_wavecal_read_gratname,grat_id,name,topwid=topwid, $
;          logunit=logunit,verbose=verbose,error=error,debug=debug
;      if error ne 0 then return
;
;      name=strtrim(name,2L)
;      dispersion=strmid(name,1L,strlen(name)-1L)
;    endif

    ;;========================================------------------------------
    ;; Putting together the filenames of the lamps which will be used:

  if ~n_elements(linelists) and lamp[0L] ne '' then $
     linelists=!p3d_path+'data'+path_sep()+'tables'+path_sep()+ $
                    'linelists'+path_sep()+lamp+'.dat'
  if n_elements(ulamp) ge 1L then $
     linelists=~n_elements(linelists)?ulamp:[linelists,ulamp]

  if ~n_elements(linelists) then begin
    errmsg='No linelist selected, cannot continue with creating a' + $
           ' dispersion mask.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Checking to see that all linelist files are present:

  for k=0L,n_elements(linelists)-1L do begin
    if ~file_test(linelists[k],/regular,/read) then begin
       errmsg='[Method '+method+'] The linelist file "'+linelists[k]+ $
              '" does not exist.'
      goto,error_handler
    endif
  endfor

  ;; Logging the information about the used linelists:
  msg=['Concatenating the contents of the following files to create one' + $
       ' linelist:']
  for k=0L,n_elements(linelists)-1L do begin
    tmp=p3d_misc_pathify(linelists[k])
    msg=[msg,'  '+strtrim(k+1,2L)+'/'+strtrim(n_elements(linelists),2L)+ $
         ' "'+tmp+'"']
  endfor
  error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
            verbose=verbose ge 1L)

  return

file_error:
  errmsg=defstr
  on_ioerror,NULL

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
                        verbose=verbose,/error)
  return
END ;;; procedure: p3d_wavecal_set_linelist
