;+
; NAME:
;         p3d_masterbias_smooth
;
;         $Id: p3d_masterbias_smooth.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         Using a raw bias image, which can be noisy with bad pixels, this
;         routine models a "master bias" image. The input image is first
;         smoothed using a median filter, and is thereafter smoothed again
;         using a mean filter.
;
;         The bias level can optionally be assumed to be constant in the x-
;         and y-directions.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: master bias creation
;
; CALLING SEQUENCE:
;         p3d_masterbias_smooth,in,out,xclean=,yclean=,xsmo=,ysmo=,const=, $
;             xconst=,yconst=,topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         in              - A two-dimensional array.
;
; KEYWORD PARAMETERS:
;         xclean          - The x-size (2*xclean+1) of the median filter box.
;         yclean          - The y-size (2*yclean+1) of the median filter box.
;         xsmo            - The x-size (2*xsmo+1)   of the mean filter box.
;         ysmo            - The y-size (2*ysmo+1)   of the mean filter box.
;         const [0]       - This parameter is set to 1 if the bias is constant.
;         xconst [0]      - This parameter is set to 1 if the output bias is
;                           constant in the x-direction.
;         yconst [0]      - This parameter is set to 1 if the output bias is
;                           constant in the y-direction.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - The 'cleaned' two-dimensional array.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         07.10.2008 - Converted from the original routine p3d_bias of
;                      Thomas Becker. /CS
;-
PRO p3d_masterbias_smooth,in,out, $
        xclean=xclean,yclean=yclean,xsmo=xsmo,ysmo=ysmo, $
        const=const,xconst=xconst,yconst=yconst,logunit=logunit, $
        topwid=topwid,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_masterbias_smooth: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(in)
  if s[0L] ne 2L or (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='IN must be set; to a two-dimensional array of floating point type.'
    goto,error_handler
  endif
  array=float(in)
  s=size(array)

  if ~n_elements(xclean) then xclean=0L
  if ~n_elements(yclean) then yclean=0L
  if ~n_elements(xsmo) then xsmo=0L
  if ~n_elements(ysmo) then ysmo=0L
   const=keyword_set( const)
  xconst=keyword_set(xconst)
  yconst=keyword_set(yconst)

  cut=8

  ;;========================================------------------------------
  ;; Smoothing the data:

  if const or (xconst and yconst) then begin

    median=median(array)
    noise=abs(array-median)
    clean=where(noise le cut*median(noise))
    out=fltarr(s[1],s[2])+float(mean(array[clean],/double))

  endif else begin

    if xconst then begin
      msg='xconst==1; a median is taken over the columns of the input array.'
      error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
      if error ne 0 then return

      median=rebin(reform( $
          float(median(array,dimension=1L,/double)),1L,s[2L]),s[1L],s[2L])
      noise=abs(in-median)
      cosmic=where(noise gt cut*median(noise),count)
      mask=bytarr(s[1L],s[2L])+1b
      if count ne 0L then mask[cosmic]=0b

      column=total(mask*array,1L)/(total(mask,1L)>1.0)
      out=rebin(reform(column,1L,s[2L]),s[1L],s[2L])

    endif else if yconst then begin
      msg='yconst==1; a median is taken over the rows of the input array.'
      error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
      if error ne 0 then return

      median=rebin(float(median(array,dimension=2L,/double)),s[1L],s[2L])

      noise=abs(array-median)
      cosmic=where(noise gt cut*median(noise),count)
      mask=bytarr(s[1L],s[2L])+1b
      if count ne 0L then mask[cosmic]=0b

      row=total(mask*array,2L)/(total(mask,2L)>1.0)
      out=rebin(row,s[1L],s[2L])

    endif else begin
      msg='xclean,yclean [median] = ' + $
          string(format='(i3,",",i3)',long(xclean),long(yclean)) + $
          ' :: xsmo,ysmo [mean] = ' + $
          string(format='(i3,",",i3)',long(xsmo),long(ysmo))
      error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
      if error ne 0 then return

      p3d_misc_median_smooth,in,med,xclean,yclean, $
          topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then return

      p3d_misc_mean_smooth,med,out,xsmo,ysmo, $
          topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then return
    endelse

  endelse ;; const or (xconst and yconst)

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_masterbias_smooth
